/*!
    \file    gd25qxx.c
    \brief   SPI flash gd25qxx driver

    \version 2025-08-08, V1.3.0, firmware for GD32E51x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd25qxx.h"
#include "gd32e51x.h"
#include "drv_usb_hw.h"
#include <string.h>

#define WREN             0x06U     /* write enable instruction */
#define WRDI             0x04U     /* write disable instruction */
#define WRSR             0x01U     /* write status register instruction */
#define RDSR             0x05U     /* read status register instruction */
#define READ             0x03U     /* read from memory instruction */
#define FASTREAD         0x0BU     /* read data at higher speed */
#define QUADREAD         0x6BU     /* read data at quad fast read */
#define WRITE            0x02U     /* write to memory instruction */
#define QUADWRITE        0x32U     /* quad write to memory instruction */
#define SE               0x20U     /* sector erase instruction */
#define CE               0xC7U     /* chip erase instruction */
#define BE               0x52U     /* block erase instruction */
#define RDID             0x9FU     /* read identification */
#define WIP_FLAG         0x01U     /* write in progress(wip) flag */
#define WEL_FLAG         0x02U     /* write enable latch(wel) flag */

#define SPI_FLASH_PAGE_SIZE       0x100U

#define SQPI_ADDR_0BIT       (SQPI_INIT & (~(uint32_t)0x1F000000U))
#define SQPI_ADDR_16BITS     (SQPI_ADDR_0BIT |0x10000000U)
#define SQPI_ADDR_24BITS     (SQPI_ADDR_0BIT |0x18000000U)
#define SQPI_LOGIC_ADDR      ((uint32_t)0xB0000000U)

static void sqpi_state_check(uint8_t mask, uint8_t check);
static void sqpi_flash_page_write(uint32_t write_addr, uint8_t *write_buff, uint32_t size);
static void sqpi_flash_write_enable(void);
/*!
    \brief      check SQPI status register
    \param[in]  mask: mask value
    \param[in]  check: check value
    \param[out] none
    \retval     none
*/
static void sqpi_state_check(uint8_t mask, uint8_t check)
{
    static volatile uint32_t temp = 0U;
    SQPI_INIT = SQPI_ADDR_0BIT;
    sqpi_read_command_config(SQPI_MODE_SSS, 0U, RDSR);

    do {
        temp = *(uint8_t *)SQPI_LOGIC_ADDR;
    } while((temp & mask) != check);
}

/*!
    \brief      write more than one byte to the flash
    \param[in]  write_addr: flash's internal address to write
    \param[in]  write_buff: pointer to the buffer
    \param[in]  size: number of bytes to write to the flash
    \param[out] none
    \retval     none
*/
static void sqpi_flash_page_write(uint32_t write_addr, uint8_t *write_buff, uint32_t size)
{
    uint32_t i = 0U;

    for(i = 0U; i < size; i++) {
        sqpi_flash_write_enable();
        SQPI_INIT = SQPI_ADDR_24BITS;
        sqpi_write_command_config(SQPI_MODE_SSQ, 0U, QUADWRITE);
        /* read a byte from the flash */
        SQPI_INIT = SQPI_ADDR_24BITS;
        *(uint8_t *)(SQPI_LOGIC_ADDR + write_addr + i) = *write_buff;
        /* point to the next location where the byte read will be saved */
        write_buff++;
        sqpi_state_check(0x01U, 0x00U);
    }
}

/*!
    \brief      enable the write access to the flash
    \param[in]  none
    \param[out] none
    \retval     none
*/
static void sqpi_flash_write_enable(void)
{
    SQPI_INIT = SQPI_ADDR_0BIT;
    sqpi_write_command_config(SQPI_MODE_SSS, 0U, WREN);
    sqpi_special_command();
    sqpi_state_check(WEL_FLAG, 0x02U);
}

/*!
    \brief      initialize SQPI parameter
    \param[in]  none
    \param[out] none
    \retval     none
*/
void sqpi_flash_init(void)
{
    sqpi_parameter_struct  sqpi_para;

    rcu_periph_clock_enable(RCU_GPIOF);
    rcu_periph_clock_enable(RCU_SQPI);
    rcu_periph_clock_enable(RCU_AF);
    gpio_init(GPIOF, GPIO_MODE_AF_PP, GPIO_OSPEED_50MHZ, GPIO_PIN_0 | GPIO_PIN_4 | GPIO_PIN_6 | GPIO_PIN_8);
    /* configure for standard SPI mode */
    gpio_init(GPIOF, GPIO_MODE_OUT_PP, GPIO_OSPEED_50MHZ, GPIO_PIN_2 | GPIO_PIN_10);
    gpio_bit_set(GPIOF, GPIO_PIN_10);
    gpio_bit_reset(GPIOF, GPIO_PIN_2);

    /* SQPI parameter config */
    sqpi_para.addr_bit = 0U;
    sqpi_para.clk_div = 20U;
    sqpi_para.cmd_bit = SQPI_CMDBIT_8_BITS;
    sqpi_para.id_length = SQPI_ID_LENGTH_32_BITS;
    sqpi_para.polarity = SQPI_SAMPLE_POLARITY_RISING;

    sqpi_init(&sqpi_para);

    sqpi_flash_quad_enable();

    gpio_init(GPIOF, GPIO_MODE_AF_PP, GPIO_OSPEED_50MHZ, GPIO_PIN_2 | GPIO_PIN_10);
}

/*!
    \brief      enable flash quad operation
    \param[in]  none
    \param[out] none
    \retval     none
*/
void sqpi_flash_quad_enable(void)
{
    sqpi_flash_write_enable();
    SQPI_INIT = SQPI_ADDR_0BIT;
    sqpi_write_command_config(SQPI_MODE_SSS, 0U, WRSR);
    *(uint16_t *)SQPI_LOGIC_ADDR = 0x0200U;
}

/*!
    \brief      read flash identification
    \param[in]  none
    \param[out] none
    \retval     flash identification
*/
uint32_t sqpi_flash_read_id(void)
{
    uint32_t temp = 0U;

    SQPI_INIT = SQPI_ADDR_0BIT;
    sqpi_read_command_config(SQPI_MODE_SSS, 0U, RDID);
    sqpi_read_id_command();
    usb_mdelay(1U);
    temp = sqpi_high_id_receive();
    temp = sqpi_low_id_receive();
    return (temp & 0xFFFFFF00U);
}

/*!
    \brief      erase the specified flash block(unit: 32KB)
    \param[in]  block_addr: address of the block to erase
    \param[out] none
    \retval     none
*/
void sqpi_flash_block_erase(uint32_t block_addr)
{
    sqpi_flash_write_enable();
    SQPI_INIT = SQPI_ADDR_16BITS;
    sqpi_write_command_config(SQPI_MODE_SSS, 0U, BE);
    *(uint8_t *)(SQPI_LOGIC_ADDR + ((block_addr & 0x00FFFF00U) >> 8)) = (uint8_t)(block_addr & 0xFFU);

    sqpi_state_check(WIP_FLAG, 0x00U);
}

/*!
    \brief      erase the specified flash sector
    \param[in]  sector_addr: address of the sector to erase
    \param[out] none
    \retval     none
*/
void sqpi_flash_sector_erase(uint32_t sector_addr)
{
    sqpi_flash_write_enable();
    SQPI_INIT = SQPI_ADDR_16BITS;
    sqpi_write_command_config(SQPI_MODE_SSS, 0U, SE);
    *(uint8_t *)(SQPI_LOGIC_ADDR + ((sector_addr & 0x00FFFF00U) >> 8U)) = (uint8_t)(sector_addr & 0xFFU);

    sqpi_state_check(WIP_FLAG, 0x00U);
}

/*!
    \brief      erase the entire flash
    \param[in]  none
    \param[out] none
    \retval     none
*/
void sqpi_flash_chip_erase(void)
{
    sqpi_flash_write_enable();

    SQPI_INIT = SQPI_ADDR_0BIT;
    sqpi_write_command_config(SQPI_MODE_SSS, 0U, CE);
    sqpi_special_command();

    sqpi_state_check(WIP_FLAG, 0x00U);
}

/*!
    \brief      read a block of data from the flash
    \param[in]  read_addr: flash's internal address to read from
    \param[in]  read_buff: pointer to the buffer that receives the data read from the flash
    \param[in]  size: number of bytes to read from the flash
    \param[out] none
    \retval     none
*/
void sqpi_flash_buffer_read(uint32_t read_addr, uint8_t *read_buff, uint32_t size)
{
    uint32_t i = 0U;
    SQPI_INIT = SQPI_ADDR_24BITS;
    sqpi_read_command_config(SQPI_MODE_SSQ, 8U, QUADREAD);
    for(i = 0U; i < size; i++) {
        /* read a byte from the flash */
        *read_buff = *(uint8_t *)(SQPI_LOGIC_ADDR + read_addr + i);
        /* point to the next location where the byte read will be saved */
        read_buff++;
    }
}

/*!
    \brief      write block of data to the flash
    \param[in]  pbuffer: pointer to the buffer
    \param[in]  write_addr: flash's internal address to write
    \param[in]  num_byte_to_write: number of bytes to write to the flash
    \param[out] none
    \retval     none
*/
void sqpi_flash_buffer_write(uint32_t write_addr, uint8_t *pbuffer, uint16_t num_byte_to_write)
{
    uint8_t num_of_page = 0U, num_of_single = 0U, addr = 0U, count = 0U, temp = 0U;

    addr          = write_addr % SPI_FLASH_PAGE_SIZE;
    count         = SPI_FLASH_PAGE_SIZE - addr;
    num_of_page   = num_byte_to_write / SPI_FLASH_PAGE_SIZE;
    num_of_single = num_byte_to_write % SPI_FLASH_PAGE_SIZE;

    /* write_addr is SPI_FLASH_PAGE_SIZE aligned */
    if(0U == addr) {
        /* num_byte_to_write < SPI_FLASH_PAGE_SIZE */
        if(0U == num_of_page) {
            sqpi_flash_page_write(write_addr, pbuffer, num_byte_to_write);
        } else {
            /* num_byte_to_write >= SPI_FLASH_PAGE_SIZE */
            while(num_of_page--) {
                sqpi_flash_page_write(write_addr, pbuffer, SPI_FLASH_PAGE_SIZE);
                write_addr += SPI_FLASH_PAGE_SIZE;
                pbuffer += SPI_FLASH_PAGE_SIZE;
            }
            sqpi_flash_page_write(write_addr, pbuffer, num_of_single);
        }
    } else {
        /* write_addr is not SPI_FLASH_PAGE_SIZE aligned */
        if(0U == num_of_page) {
            /* (num_byte_to_write + write_addr) > SPI_FLASH_PAGE_SIZE */
            if(num_of_single > count) {
                temp = num_of_single - count;
                sqpi_flash_page_write(write_addr, pbuffer, count);
                write_addr += count;
                pbuffer += count;
                sqpi_flash_page_write(write_addr, pbuffer, temp);
            } else {
                sqpi_flash_page_write(write_addr, pbuffer, num_byte_to_write);
            }
        } else {
            /* num_byte_to_write >= SPI_FLASH_PAGE_SIZE */
            num_byte_to_write -= count;
            num_of_page = num_byte_to_write / SPI_FLASH_PAGE_SIZE;
            num_of_single = num_byte_to_write % SPI_FLASH_PAGE_SIZE;

            sqpi_flash_page_write(write_addr, pbuffer, count);
            write_addr += count;
            pbuffer += count;

            while(num_of_page--) {
                sqpi_flash_page_write(write_addr, pbuffer, SPI_FLASH_PAGE_SIZE);
                write_addr += SPI_FLASH_PAGE_SIZE;
                pbuffer += SPI_FLASH_PAGE_SIZE;
            }

            if(0 != num_of_single) {
                sqpi_flash_page_write(write_addr, pbuffer, num_of_single);
            }
        }
    }
}
