/*!
    \file    dual_cdc_acm_core.c
    \brief   CDC ACM driver

    \version 2025-08-08, V1.3.0, firmware for GD32E51x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this 
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice, 
       this list of conditions and the following disclaimer in the documentation 
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors 
       may be used to endorse or promote products derived from this software without 
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
*/

#include "usbd_transc.h"
#include "dual_cdc_acm_core.h"

#define USBD_VID                          0x28E9
#define USBD_PID                          0x029B

/* note:it should use the C99 standard when compiling the below codes */
/* USB standard device descriptor */
const usb_desc_dev dual_cdc_dev_desc =
{
    .header = 
     {
         .bLength          = USB_DEV_DESC_LEN, 
         .bDescriptorType  = USB_DESCTYPE_DEV,
     },
    .bcdUSB                = 0x0200,
    .bDeviceClass          = 0xEF,
    .bDeviceSubClass       = 0x02,
    .bDeviceProtocol       = 0x01,
    .bMaxPacketSize0       = USBD_EP0_MAX_SIZE,
    .idVendor              = USBD_VID,
    .idProduct             = USBD_PID,
    .bcdDevice             = 0x0100,
    .iManufacturer         = STR_IDX_MFC,
    .iProduct              = STR_IDX_PRODUCT,
    .iSerialNumber         = STR_IDX_SERIAL,
    .bNumberConfigurations = USBD_CFG_MAX_NUM,
};

/* USB device configuration descriptor */
usb_dual_cdc_desc_config_set dual_cdc_config_desc = 
{
    .config = 
    {
        .header = 
         {
             .bLength         = sizeof(usb_desc_config), 
             .bDescriptorType = USB_DESCTYPE_CONFIG,
         },
        .wTotalLength         = sizeof(usb_dual_cdc_desc_config_set),
        .bNumInterfaces       = 0x04,
        .bConfigurationValue  = 0x01,
        .iConfiguration       = 0x00,
        .bmAttributes         = 0x80,
        .bMaxPower            = 0x32
    },

    .cdc_iad0 = 
    {
         .header = 
         {
             .bLength         = sizeof(usb_desc_IAD), 
             .bDescriptorType = 0x0BU
         },
         .bFirstInterface     = 0x00,
         .bInterfaceCount     = 0x02,
         .bFunctionClass      = 0x02,
         .bFunctionSubClass   = 0x02,
         .bFunctionProtocol   = 0x01,
         .iFunction           = 0x00
    },
    
    .cdc_loopback_interface0 = 
    {
        .header = 
         {
             .bLength         = sizeof(usb_desc_itf), 
             .bDescriptorType = USB_DESCTYPE_ITF 
         },
        .bInterfaceNumber     = 0x00,
        .bAlternateSetting    = 0x00,
        .bNumEndpoints        = 0x01,
        .bInterfaceClass      = 0x02,
        .bInterfaceSubClass   = 0x02,
        .bInterfaceProtocol   = 0x01,
        .iInterface           = 0x00
    },

    .cdc_loopback_header0 = 
    {
        .header =
         {
            .bLength         = sizeof(usb_desc_header_func), 
            .bDescriptorType = USB_DESCTYPE_CS_INTERFACE
         },
        .bDescriptorSubtype  = 0x00,
        .bcdCDC              = 0x0110
    },

    .cdc_loopback_call_managment0 = 
    {
        .header = 
         {
            .bLength         = sizeof(usb_desc_call_managment_func), 
            .bDescriptorType = USB_DESCTYPE_CS_INTERFACE
         },
        .bDescriptorSubtype  = 0x01,
        .bmCapabilities      = 0x00,
        .bDataInterface      = 0x01
    },

    .cdc_loopback_acm0 = 
    {
        .header = 
         {
            .bLength         = sizeof(usb_desc_acm_func), 
            .bDescriptorType = USB_DESCTYPE_CS_INTERFACE
         },
        .bDescriptorSubtype  = 0x02,
        .bmCapabilities      = 0x02,
    },

    .cdc_loopback_union0 = 
    {
        .header = 
         {
            .bLength         = sizeof(usb_desc_union_func), 
            .bDescriptorType = USB_DESCTYPE_CS_INTERFACE
         },
        .bDescriptorSubtype  = 0x06,
        .bMasterInterface    = 0x00,
        .bSlaveInterface0    = 0x01,
    },

    .cdc_loopback_cmd_endpoint0 = 
    {
        .header = 
         {
            .bLength         = sizeof(usb_desc_ep), 
            .bDescriptorType = USB_DESCTYPE_EP
         },
        .bEndpointAddress    = CDC_ACM0_CMD_EP,
        .bmAttributes        = 0x03,
        .wMaxPacketSize      = CDC_ACM_CMD_PACKET_SIZE,
        .bInterval           = 0x0A
    },

    .cdc_loopback_data_interface0 = 
    {
        .header = 
         {
            .bLength         = sizeof(usb_desc_itf), 
            .bDescriptorType = USB_DESCTYPE_ITF
         },
        .bInterfaceNumber    = 0x01,
        .bAlternateSetting   = 0x00,
        .bNumEndpoints       = 0x02,
        .bInterfaceClass     = 0x0A,
        .bInterfaceSubClass  = 0x00,
        .bInterfaceProtocol  = 0x00,
        .iInterface          = 0x00
    },

    .cdc_loopback_out_endpoint0 = 
    {
        .header = 
         {
             .bLength         = sizeof(usb_desc_ep), 
             .bDescriptorType = USB_DESCTYPE_EP 
         },
        .bEndpointAddress     = CDC_ACM0_DATA_OUT_EP,
        .bmAttributes         = 0x02,
        .wMaxPacketSize       = CDC_ACM_DATA_PACKET_SIZE,
        .bInterval            = 0x00
    },

    .cdc_loopback_in_endpoint0 = 
    {
        .header = 
         {
             .bLength         = sizeof(usb_desc_ep), 
             .bDescriptorType = USB_DESCTYPE_EP 
         },
        .bEndpointAddress     = CDC_ACM0_DATA_IN_EP,
        .bmAttributes         = 0x02,
        .wMaxPacketSize       = CDC_ACM_DATA_PACKET_SIZE,
        .bInterval            = 0x00
    },
    
    .cdc_iad1 = 
    {
         .header = 
         {
             .bLength         = sizeof(usb_desc_IAD), 
             .bDescriptorType = 0x0B 
         },
         .bFirstInterface     = 0x02,
         .bInterfaceCount     = 0x02,
         .bFunctionClass      = 0x02,
         .bFunctionSubClass   = 0x02,
         .bFunctionProtocol   = 0x01,
         .iFunction           = 0x00
    },

    .cdc_loopback_interface1 = 
    {
        .header = 
         {
             .bLength         = sizeof(usb_desc_itf), 
             .bDescriptorType = USB_DESCTYPE_ITF 
         },
        .bInterfaceNumber     = 0x02,
        .bAlternateSetting    = 0x00,
        .bNumEndpoints        = 0x01,
        .bInterfaceClass      = 0x02,
        .bInterfaceSubClass   = 0x02,
        .bInterfaceProtocol   = 0x01,
        .iInterface           = 0x00
    },

    .cdc_loopback_header1 = 
    {
        .header =
         {
            .bLength         = sizeof(usb_desc_header_func), 
            .bDescriptorType = USB_DESCTYPE_CS_INTERFACE
         },
        .bDescriptorSubtype  = 0x00,
        .bcdCDC              = 0x0110
    },

    .cdc_loopback_call_managment1 = 
    {
        .header = 
         {
            .bLength         = sizeof(usb_desc_call_managment_func), 
            .bDescriptorType = USB_DESCTYPE_CS_INTERFACE
         },
        .bDescriptorSubtype  = 0x01,
        .bmCapabilities      = 0x00,
        .bDataInterface      = 0x03
    },

    .cdc_loopback_acm1 = 
    {
        .header = 
         {
            .bLength         = sizeof(usb_desc_acm_func), 
            .bDescriptorType = USB_DESCTYPE_CS_INTERFACE
         },
        .bDescriptorSubtype  = 0x02,
        .bmCapabilities      = 0x02,
    },

    .cdc_loopback_union1 = 
    {
        .header = 
         {
            .bLength         = sizeof(usb_desc_union_func), 
            .bDescriptorType = USB_DESCTYPE_CS_INTERFACE
         },
        .bDescriptorSubtype  = 0x06,
        .bMasterInterface    = 0x02,
        .bSlaveInterface0    = 0x03,
    },

    .cdc_loopback_cmd_endpoint1 = 
    {
        .header = 
         {
            .bLength         = sizeof(usb_desc_ep), 
            .bDescriptorType = USB_DESCTYPE_EP
         },
        .bEndpointAddress    = CDC_ACM1_CMD_EP,
        .bmAttributes        = 0x03,
        .wMaxPacketSize      = CDC_ACM_CMD_PACKET_SIZE,
        .bInterval           = 0x0A
    },

    .cdc_loopback_data_interface1 = 
    {
        .header = 
         {
            .bLength         = sizeof(usb_desc_itf), 
            .bDescriptorType = USB_DESCTYPE_ITF
         },
        .bInterfaceNumber    = 0x03,
        .bAlternateSetting   = 0x00,
        .bNumEndpoints       = 0x02,
        .bInterfaceClass     = 0x0A,
        .bInterfaceSubClass  = 0x00,
        .bInterfaceProtocol  = 0x00,
        .iInterface          = 0x00
    },

    .cdc_loopback_out_endpoint1 = 
    {
        .header = 
         {
             .bLength         = sizeof(usb_desc_ep), 
             .bDescriptorType = USB_DESCTYPE_EP 
         },
        .bEndpointAddress     = CDC_ACM1_DATA_OUT_EP,
        .bmAttributes         = 0x02,
        .wMaxPacketSize       = CDC_ACM_DATA_PACKET_SIZE,
        .bInterval            = 0x00
    },

    .cdc_loopback_in_endpoint1 = 
    {
        .header = 
         {
             .bLength         = sizeof(usb_desc_ep), 
             .bDescriptorType = USB_DESCTYPE_EP 
         },
        .bEndpointAddress     = CDC_ACM1_DATA_IN_EP,
        .bmAttributes         = 0x02,
        .wMaxPacketSize       = CDC_ACM_DATA_PACKET_SIZE,
        .bInterval            = 0x00
    }
};

/* USB language ID Descriptor */
static usb_desc_LANGID usbd_language_id_desc = 
{
    .header = 
     {
         .bLength         = sizeof(usb_desc_LANGID), 
         .bDescriptorType = USB_DESCTYPE_STR,
     },
    .wLANGID              = ENG_LANGID
};

/* USB manufacture string */
static usb_desc_str manufacturer_string = 
{
    .header = 
     {
         .bLength         = USB_STRING_LEN(10U), 
         .bDescriptorType = USB_DESCTYPE_STR,
     },
    .unicode_string = {'G', 'i', 'g', 'a', 'D', 'e', 'v', 'i', 'c', 'e'}
};

/* USB product string */
static usb_desc_str product_string = 
{
    .header = 
     {
         .bLength         = USB_STRING_LEN(13U), 
         .bDescriptorType = USB_DESCTYPE_STR,
     },
    .unicode_string = {'G', 'D', '3', '2', '-', 'D', 'U', 'A', 'L', '_', 'C', 'D', 'C'}
};

/* USBD serial string */
static usb_desc_str serial_string = 
{
    .header = 
     {
         .bLength         = USB_STRING_LEN(12U), 
         .bDescriptorType = USB_DESCTYPE_STR,
     }
};

/* USB string descriptor set */
static uint8_t* usbd_dual_cdc_strings[] = 
{
    [STR_IDX_LANGID]  = (uint8_t *)&usbd_language_id_desc,
    [STR_IDX_MFC]     = (uint8_t *)&manufacturer_string,
    [STR_IDX_PRODUCT] = (uint8_t *)&product_string,
    [STR_IDX_SERIAL]  = (uint8_t *)&serial_string
};

usb_desc dual_cdc_desc = {
    .dev_desc    = (uint8_t *)&dual_cdc_dev_desc,
    .config_desc = (uint8_t *)&dual_cdc_config_desc,
    .strings     = usbd_dual_cdc_strings
};

static uint8_t dual_cdc_init         (usb_dev *udev, uint8_t config_index);
static uint8_t dual_cdc_deinit       (usb_dev *udev, uint8_t config_index);
static uint8_t dual_cdc_req_handler  (usb_dev *udev, usb_req *req);
static uint8_t dual_cdc_ctlx_out     (usb_dev *udev);

static void dual_cdc_data_in         (usb_dev *udev, uint8_t ep_num);
static void dual_cdc_data_out        (usb_dev *udev, uint8_t ep_num);

usb_class dual_cdc_class = {
    .req_cmd       = 0xFFU,

    .init          = dual_cdc_init,
    .deinit        = dual_cdc_deinit,
    .req_process   = dual_cdc_req_handler,
    .ctlx_out      = dual_cdc_ctlx_out,
    .data_in       = dual_cdc_data_in,
    .data_out      = dual_cdc_data_out
};

/*!
    \brief      receive CDC ACM 0 data
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     USB device operation status
*/
void cdc_data_receive0(usb_dev *udev)
{
    usb_cdc_handler *cdc = (usb_cdc_handler *)udev->class_data[0];

    cdc->packet_receive = 0U;
    cdc->pre_packet_send = 0U;

    usbd_ep_recev(udev, CDC_ACM0_DATA_OUT_EP, (uint8_t*)(cdc->data), USB_CDC_RX_LEN);
}

/*!
    \brief      receive CDC ACM 1 data
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     USB device operation status
*/
void cdc_data_receive1(usb_dev *udev)
{
    usb_cdc_handler *cdc = (usb_cdc_handler *)udev->class_data[1];

    cdc->packet_receive = 0U;
    cdc->pre_packet_send = 0U;

    usbd_ep_recev(udev, CDC_ACM1_DATA_OUT_EP, (uint8_t*)(cdc->data), USB_CDC_RX_LEN);
}

/*!
    \brief      send CDC ACM 0 data
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     USB device operation status
*/
void cdc_acm_data_send0(usb_dev *udev)
{
    usb_cdc_handler *cdc = (usb_cdc_handler *)udev->class_data[0];
    uint32_t data_len = cdc->receive_length;

    if ((0U != data_len) && (1U == cdc->packet_sent)) {
        cdc->packet_sent = 0U;
        usbd_ep_send(udev, CDC_ACM0_DATA_IN_EP, (uint8_t*)(cdc->data), (uint16_t)data_len);
        cdc->receive_length = 0U;
    }
}

/*!
    \brief      send CDC ACM 1 data
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     USB device operation status
*/
void cdc_acm_data_send1(usb_dev *udev)
{
    usb_cdc_handler *cdc = (usb_cdc_handler *)udev->class_data[1];
    uint32_t data_len = cdc->receive_length;

    if ((0U != data_len) && (1U == cdc->packet_sent)) {
        cdc->packet_sent = 0U;
        usbd_ep_send(udev, CDC_ACM1_DATA_IN_EP, (uint8_t*)(cdc->data), (uint16_t)data_len);
        cdc->receive_length = 0U;
    }
}

/*!
    \brief      check CDC ACM 0 is ready for data transfer
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     0 if cdc is ready, 5 else
*/
uint8_t cdc_check_ready0(usb_dev *udev)
{
    if (udev->class_data[0] != NULL) {
        usb_cdc_handler *cdc = (usb_cdc_handler *)udev->class_data[0];

        if ((1U == cdc->packet_receive) && (1U == cdc->pre_packet_send)) {
            return 0U;
        }
    }

    return 5U;
}

/*!
    \brief      check CDC ACM 1 is ready for data transfer
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     0 if cdc is ready, 5 else
*/
uint8_t cdc_check_ready1(usb_dev *udev)
{
    if (udev->class_data[1] != NULL) {
        usb_cdc_handler *cdc = (usb_cdc_handler *)udev->class_data[1];

        if ((1U == cdc->packet_receive) && (1U == cdc->pre_packet_send)) {
            return 0U;
        }
    }

    return 5U;
}

/*!
    \brief      initialize the CDC ACM device
    \param[in]  udev: pointer to USB device instance
    \param[in]  config_index: configuration index
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t dual_cdc_init (usb_dev *udev, uint8_t config_index)
{
    static usb_cdc_handler cdc_handler0, cdc_handler1;

    /* initialize the data Tx/Rx endpoint */
    usbd_ep_init(udev, EP_BUF_SNG, BULK0_TX_ADDR, &(dual_cdc_config_desc.cdc_loopback_in_endpoint0));
    usbd_ep_init(udev, EP_BUF_SNG, BULK0_RX_ADDR, &(dual_cdc_config_desc.cdc_loopback_out_endpoint0));

    usbd_ep_init(udev, EP_BUF_SNG, BULK1_TX_ADDR, &(dual_cdc_config_desc.cdc_loopback_in_endpoint1));
    usbd_ep_init(udev, EP_BUF_SNG, BULK1_RX_ADDR, &(dual_cdc_config_desc.cdc_loopback_out_endpoint1));

    /* initialize the command Tx endpoint */
    usbd_ep_init(udev, EP_BUF_SNG, INT0_TX_ADDR, &(dual_cdc_config_desc.cdc_loopback_cmd_endpoint0));
    usbd_ep_init(udev, EP_BUF_SNG, INT1_TX_ADDR, &(dual_cdc_config_desc.cdc_loopback_cmd_endpoint1));

    udev->ep_transc[EP_ID(CDC_ACM0_DATA_IN_EP)][TRANSC_IN] = dual_cdc_class.data_in;
    udev->ep_transc[CDC_ACM0_DATA_OUT_EP][TRANSC_OUT] = dual_cdc_class.data_out;

    udev->ep_transc[EP_ID(CDC_ACM1_DATA_IN_EP)][TRANSC_IN] = dual_cdc_class.data_in;
    udev->ep_transc[CDC_ACM1_DATA_OUT_EP][TRANSC_OUT] = dual_cdc_class.data_out;

    /* initialize cdc handler 0 structure */
    cdc_handler0.packet_receive = 0U;
    cdc_handler0.packet_sent = 1U;
    cdc_handler0.pre_packet_send = 1U;
    cdc_handler0.receive_length = 0U;

    cdc_handler0.line_coding = (acm_line){
        .dwDTERate   = 115200,
        .bCharFormat = 0,
        .bParityType = 0,
        .bDataBits   = 0x08
    };

    udev->class_data[0] = (void *)&cdc_handler0;

    /* initialize cdc handler 1 structure */
    cdc_handler1.packet_receive = 0U;
    cdc_handler1.packet_sent = 1U;
    cdc_handler1.pre_packet_send = 1U;
    cdc_handler1.receive_length = 0U;

    cdc_handler1.line_coding = (acm_line){
        .dwDTERate   = 115200,
        .bCharFormat = 0,
        .bParityType = 0,
        .bDataBits   = 0x08
    };

    udev->class_data[1] = (void *)&cdc_handler1;

    return USBD_OK;
}

/*!
    \brief      deinitialize the CDC ACM device
    \param[in]  udev: pointer to USB device instance
    \param[in]  config_index: configuration index
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t dual_cdc_deinit (usb_dev *udev, uint8_t config_index)
{
    /* deinitialize the data Tx/Rx endpoint */
    usbd_ep_deinit(udev, CDC_ACM0_DATA_IN_EP);
    usbd_ep_deinit(udev, CDC_ACM0_DATA_OUT_EP);

    usbd_ep_deinit(udev, CDC_ACM1_DATA_IN_EP);
    usbd_ep_deinit(udev, CDC_ACM1_DATA_OUT_EP);

    /* deinitialize the command Tx endpoint */
    usbd_ep_deinit(udev, CDC_ACM0_CMD_EP);
    usbd_ep_deinit(udev, CDC_ACM1_CMD_EP);

    return USBD_OK;
}

/*!
    \brief      command data received on control endpoint
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t dual_cdc_ctlx_out (usb_dev *udev)
{
    usb_cdc_handler *cdc0 = (usb_cdc_handler *)udev->class_data[0];
    usb_cdc_handler *cdc1 = (usb_cdc_handler *)udev->class_data[1];
    
    if (NO_CMD != udev->class_core->req_cmd) {
        cdc0->packet_receive = 1U;
        cdc0->pre_packet_send = 1U;

        cdc1->packet_receive = 1U;
        cdc1->pre_packet_send = 1U;

        udev->class_core->req_cmd = NO_CMD;
    }

    return USBD_OK;
}

/*!
    \brief      handle CDC ACM data in DATA IN transaction
    \param[in]  udev: pointer to USB device instance
    \param[in]  ep_num: endpoint number
    \param[out] none
    \retval     USB device operation status
*/
static void dual_cdc_data_in (usb_dev *udev, uint8_t ep_num)
{
    usb_cdc_handler *cdc = NULL;
    usb_transc *transc = &udev->transc_in[ep_num];

   if ((CDC_ACM0_DATA_IN_EP & 0x7F) == ep_num) {
        cdc = (usb_cdc_handler *)udev->class_data[0];
    } else {
        cdc = (usb_cdc_handler *)udev->class_data[1];
    }

    if (transc->xfer_count == transc->max_len) {
        usbd_ep_send(udev, EP_ID(ep_num), NULL, 0U);
    } else {
        cdc->packet_sent = 1U;
        cdc->pre_packet_send = 1U;
    }
}

/*!
    \brief      handle CDC ACM data in DATA OUT transaction
    \param[in]  udev: pointer to USB device instance
    \param[in]  ep_num: endpoint number
    \param[out] none
    \retval     USB device operation status
*/
static void dual_cdc_data_out (usb_dev *udev, uint8_t ep_num)
{
    usb_cdc_handler *cdc = NULL;

   if (ep_num == (CDC_ACM0_DATA_OUT_EP & 0x7F)) {
        cdc = (usb_cdc_handler *)udev->class_data[0];
    } else {
        cdc = (usb_cdc_handler *)udev->class_data[1];
    }

    cdc->packet_receive = 1U;
    cdc->receive_length = udev->transc_out[ep_num].xfer_count;
}

/*!
    \brief      handle the CDC ACM class-specific requests
    \param[in]  udev: pointer to USB device instance
    \param[in]  req: device class-specific request
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t dual_cdc_req_handler (usb_dev *udev, usb_req *req)
{
    usb_cdc_handler *cdc = NULL;
    uint8_t status = REQ_NOTSUPP, noti_buf[10] = {0U};

    if(0x00 == (req->wIndex & 0xFF)){
        cdc = (usb_cdc_handler *)udev->class_data[0];
    }else {
        cdc = (usb_cdc_handler *)udev->class_data[1];
    }

    acm_notification *notif = (void *)noti_buf;

    switch (req->bRequest) {
    case SEND_ENCAPSULATED_COMMAND:
        break;

    case GET_ENCAPSULATED_RESPONSE:
        break;

    case SET_COMM_FEATURE:
        break;

    case GET_COMM_FEATURE:
        break;

    case CLEAR_COMM_FEATURE:
        break;

    case SET_LINE_CODING:
        /* set the value of the current command to be processed */
        udev->class_core->req_cmd = req->bRequest;

        usb_transc_config(&udev->transc_out[0U], (uint8_t *)&cdc->line_coding, req->wLength, 0U);

        status = REQ_SUPP;
        break;

    case GET_LINE_CODING:
        usb_transc_config(&udev->transc_in[0U], (uint8_t *)&cdc->line_coding, 7U, 0U);

        status = REQ_SUPP;
        break;

    case SET_CONTROL_LINE_STATE:
        notif->bmRequestType = 0xA1U;
        notif->bNotification = USB_CDC_NOTIFY_SERIAL_STATE;
        notif->wIndex = 0U;
        notif->wValue = 0U;
        notif->wLength = 2U;
        noti_buf[8] = (uint8_t)req->wValue & 3U;
        noti_buf[9] = 0U;

        status = REQ_SUPP;
        break;

    case SEND_BREAK:
        break;

    default:
        break;
    }

    return status;
}

