/*!
    \file    main.c
    \brief   CAN loopback and silent communication mode

    \version 2025-08-09, V1.7.0, firmware for GD32A50x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32a50x.h"
#include <stdio.h>
#include "gd32a503v_eval.h"
#include "string.h"

/* select CAN */
#define DEV_CAN0_USED
//#define DEV_CAN1_USED

#ifdef DEV_CAN0_USED
#define CANX    CAN0
#elif defined DEV_CAN1_USED
#define CANX    CAN1
#else
#error ("DEV_CANx_USED should be defined");
#endif /* DEV_CAN0_USED */

can_mailbox_descriptor_struct transmit_message;
can_mailbox_descriptor_struct receive_message;

const uint8_t tx_data[8] = {0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7, 0x8};
uint8_t rx_data[8];

/*!
    \brief      board configuration
    \param[in]  none
    \param[out] none
    \retval     none
*/
void bsp_board_config(void)
{
    /* enable CAN port clock */
    rcu_periph_clock_enable(RCU_GPIOB);
    rcu_periph_clock_enable(RCU_GPIOD);

    /* configure leds */
    gd_eval_led_init(LED1);
    gd_eval_led_init(LED2);

    gd_eval_led_off(LED1);
    gd_eval_led_off(LED2);

    /* configure CAN0 GPIO */
    gpio_output_options_set(GPIOB, GPIO_OTYPE_PP, GPIO_OSPEED_50MHZ, GPIO_PIN_13);
    gpio_mode_set(GPIOB, GPIO_MODE_AF, GPIO_PUPD_PULLUP, GPIO_PIN_13);
    gpio_af_set(GPIOB, GPIO_AF_6, GPIO_PIN_13);

    gpio_output_options_set(GPIOB, GPIO_OTYPE_PP, GPIO_OSPEED_50MHZ, GPIO_PIN_14);
    gpio_mode_set(GPIOB, GPIO_MODE_AF, GPIO_PUPD_NONE, GPIO_PIN_14);
    gpio_af_set(GPIOB, GPIO_AF_6, GPIO_PIN_14);


    /* configure CAN1 GPIO */
    gpio_output_options_set(GPIOD, GPIO_OTYPE_PP, GPIO_OSPEED_50MHZ, GPIO_PIN_6);
    gpio_mode_set(GPIOD, GPIO_MODE_AF, GPIO_PUPD_PULLUP, GPIO_PIN_6);
    gpio_af_set(GPIOD, GPIO_AF_6, GPIO_PIN_6);

    gpio_output_options_set(GPIOD, GPIO_OTYPE_PP, GPIO_OSPEED_50MHZ, GPIO_PIN_7);
    gpio_mode_set(GPIOD, GPIO_MODE_AF, GPIO_PUPD_NONE, GPIO_PIN_7);
    gpio_af_set(GPIOD, GPIO_AF_6, GPIO_PIN_7);
}

/*!
    \brief      initialize CAN and filter
    \param[in]  none
    \param[out] none
    \retval     none
*/
void can_loopback_init(void)
{
    can_parameter_struct can_parameter;

    /* enable CAN clock */
    rcu_can_clock_config(CAN0, RCU_CANSRC_PCLK2);
    rcu_can_clock_config(CAN1, RCU_CANSRC_PCLK2);
    rcu_periph_clock_enable(RCU_CAN0);
    rcu_periph_clock_enable(RCU_CAN1);

    /* initialize CAN register */
    can_deinit(CANX);
    /* initialize CAN */
    can_struct_para_init(CAN_INIT_STRUCT, &can_parameter);

    /* initialize CAN parameters */
    can_parameter.internal_counter_source = CAN_TIMER_SOURCE_BIT_CLOCK;            //internal time counter increase 1 when send or receive a bit
    can_parameter.self_reception = DISABLE;                                        //receive the frame sent by itself or not
    can_parameter.mb_tx_order = CAN_TX_HIGH_PRIORITY_MB_FIRST;                     //transmit frame priority by MB priority or MB number
    can_parameter.mb_tx_abort_enable = ENABLE;                                     //support transmission abort function or not
    can_parameter.local_priority_enable = DISABLE;                                 //MB priority structure includes local priority value or not
    can_parameter.mb_rx_ide_rtr_type = CAN_IDE_RTR_FILTERED;                       //when receive, (filter IDE && RTR bit), or (always compare IDE && not compare RTR bit)
    can_parameter.mb_remote_frame = CAN_STORE_REMOTE_REQUEST_FRAME;                //when receive remote request frame, store it as data frame or generate a remote response frame
    can_parameter.rx_private_filter_queue_enable = DISABLE;                        //use separate filters / a same filter for all received frames
    can_parameter.edge_filter_enable = DISABLE;                                    //used for bus intergration state, enable to detect two continuous norminal dominant bit for hardware synchronous edge
    can_parameter.protocol_exception_enable = DISABLE;                             //enable to detect protocol exception event(when not in FD mode, but receive a FD frame)
    can_parameter.rx_filter_order = CAN_RX_FILTER_ORDER_MAILBOX_FIRST;             //if RX FIFO is enabled, received frame to match mailbox/FIFO first
    can_parameter.memory_size = CAN_MEMSIZE_32_UNIT;                               //32*4 words specific RAM memory for mailbox and RX FIFO
    /* filter configuration */
    can_parameter.mb_public_filter = 0x5FFC0000;                                   //configure CAN_RMPUBF register (refer to can_private_filter_config() to configure CAN_RFIFOMPFx registers if separate filters are used)
    can_parameter.resync_jump_width = 1;                                           //SJW
    can_parameter.prop_time_segment = 2;                                           //PTS segment
    can_parameter.time_segment_1 = 3;                                              //PBS1 segment
    can_parameter.time_segment_2 = 4;                                              //PBS2 segment
    /* 125Kbps */
    can_parameter.prescaler = 80;                                                  //baudrate = fCANCLK/prescaler/(1+PTS+PBS1+PBS2)

    /* initialize CAN */
    can_init(CANX, &can_parameter);
    /* enter loopback silent mode */
    can_operation_mode_enter(CANX, CAN_LOOPBACK_SILENT_MODE);
}

/*!
    \brief      function for CAN loopback and silent mode
    \param[in]  none
    \param[out] none
    \retval     ErrStatus
*/
void can_loopback_test(void)
{
    uint32_t timeout = 0xFFFF;

    can_struct_para_init(CAN_MDSC_STRUCT, &transmit_message);
    can_struct_para_init(CAN_MDSC_STRUCT, &receive_message);
    /* initialize transmit message */
    transmit_message.rtr = 0;
    transmit_message.ide = 0;
    transmit_message.code = CAN_MB_TX_STATUS_DATA;
    transmit_message.brs = 0;
    transmit_message.fdf = 0;
    transmit_message.prio = 0;
    transmit_message.data_bytes = 8;
    /* tx message content */
    transmit_message.data = (uint32_t *)(tx_data);
    transmit_message.id = 0x55;

    receive_message.rtr = 0;
    receive_message.ide = 0;
    receive_message.code = CAN_MB_RX_STATUS_EMPTY;
    /* rx mailbox */
    receive_message.id = 0x55;
    receive_message.data = (uint32_t *)(rx_data);
    can_mailbox_config(CANX, 1, &receive_message);

    /* transmit message */
    can_mailbox_config(CANX, 0, &transmit_message);
    /* waiting for transmit completed */
    while((RESET == can_flag_get(CANX, CAN_FLAG_MB0)) && (0 != timeout)) {
        timeout--;
    }
    timeout = 0xFFFF;
    /* waiting for receive completed */
    while((RESET == can_flag_get(CANX, CAN_FLAG_MB1)) && (0 != timeout)) {
        timeout--;
    }

    /* check the receive message */
    can_mailbox_receive_data_read(CANX, 1, &receive_message);

    /* check the receive message */
    if(0 == memcmp(rx_data, tx_data, 8)) {
        /* loopback test is success */
        gd_eval_led_on(LED1);
    } else {
        /* loopback test is failed */
        gd_eval_led_on(LED2);
    }
}

/*!
    \brief      main function
    \param[in]  none
    \param[out] none
    \retval     none
*/
int main(void)
{
    bsp_board_config();
    /* initialize CAN */
    can_loopback_init();
    /* loopback of polling */
    can_loopback_test();

    while(1) {
    }
}
