/*
    \file  main.c
*/
/*
    Copyright (c) 2023, GigaDevice Semiconductor Inc.

    All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/
#include "gd32f3x0_hal.h"
#include "gd32f3x0_hal_init.h"
/* user code [global 0] begin */
#include "gd32f3x0r_hal_eval.h"

int keystatus[3] = {0, 0, 0};

/* the current cycle number array of the channel pin */
uint16_t samplenum[3] = {0, 0, 0};

/* reference value sample array of TSI group5 */
uint16_t sample_refnum_array1[20] = {0};
uint16_t sample_refnum_array2[20] = {0};
uint16_t sample_refnum_array3[20] = {0};

/* average value of cycles */
uint16_t sample_refnum[3] = {0};
uint32_t sample_addnum[3] = {0};

uint16_t threshold1 = 0;
uint16_t threshold2 = 0;

void delay(uint32_t nCount);

void tsi_transfer_pin(uint32_t TSI_groupx_pin);
/* user code [global 0] end */

/*!
    \brief      main function
    \param[in]  none
    \param[out] none
    \retval     none
*/
int main(void)
{
    /* user code [local 0] begin */
    int m = 0;
    /* user code [local 0] end */

    msd_system_init();
    msd_clock_init();
    /* user code [local 1] begin */

    /* user code [local 1] end */
    msd_gpio_init();
    msd_tsi_init();

    /* user code [local 2] begin */
    gd_eval_led_off(LED1);
    gd_eval_led_off(LED2);
    gd_eval_led_off(LED3);
    gd_eval_led_off(LED4);
    hals_tsi_channel_pin_disable(TSI_CHCFG_G5P1 | TSI_CHCFG_G5P2 | TSI_CHCFG_G5P3);
    /* reference cycle value acquisition and processing */
    for(m = 0; m < 20; m++) {
        /* get charge transfer complete cycle number of group5 pin1 */
        tsi_transfer_pin(TSI_CHCFG_G5P1);

        /* check the TSI flag:end of acquisition interrupt */
        if(SET == hals_tsi_flag_get(TSI_FLAG_CTCF)) {
            /* get charge transfer complete cycle number */
            sample_refnum_array1[m] = hal_tsi_group_cycle_get(&tsi_info, TSI_GROUP_IDX5);
        }

        /* disable the selected pin as channel pin */
        hals_tsi_channel_pin_disable(TSI_CHCFG_G5P1);

        /* get charge transfer complete cycle number of group5 pin2 */
        tsi_transfer_pin(TSI_CHCFG_G5P2);

        if(SET == hals_tsi_flag_get(TSI_FLAG_CTCF)) {
            sample_refnum_array2[m] = hal_tsi_group_cycle_get(&tsi_info, TSI_GROUP_IDX5);
        }

        hals_tsi_channel_pin_disable(TSI_CHCFG_G5P2);


        /* get charge transfer complete cycle number of group5 pin3 */
        tsi_transfer_pin(TSI_CHCFG_G5P3);

        if(SET == hals_tsi_flag_get(TSI_FLAG_CTCF)) {
            sample_refnum_array3[m] = hal_tsi_group_cycle_get(&tsi_info, TSI_GROUP_IDX5);
        }

        hals_tsi_channel_pin_disable(TSI_CHCFG_G5P3);

        /* delay for a period of time while all banks have been acquired */
        delay(0x1000);
    }

    for(m = 1; m < 20; m++) {
        sample_addnum[0] += sample_refnum_array1[m];
        sample_addnum[1] += sample_refnum_array2[m];
        sample_addnum[2] += sample_refnum_array3[m];
    }

    /* average channel cycle value are obtained */
    sample_refnum[0] = sample_addnum[0] / 19;
    sample_refnum[1] = sample_addnum[1] / 19;
    sample_refnum[2] = sample_addnum[2] / 19;

    /* set threshold use for determine touch location of TSI_CHCFG_G5P1 */
    threshold1 = sample_refnum[2] - sample_refnum[1];
    threshold2 = sample_refnum[2] - sample_refnum[1] + 15;
    /* user code [local 2] end */

    while(1) {
        /* user code [local 3] begin */
        /* acquisition pin1 of group5 */
        tsi_transfer_pin(TSI_CHCFG_G5P1);

        /* check the TSI flag--end of acquisition interrupt */
        if(SET == hals_tsi_flag_get(TSI_FLAG_CTCF)) {
            /* get charge transfer complete cycle number */
            samplenum[0] = hal_tsi_group_cycle_get(&tsi_info, TSI_GROUP_IDX5);
        }

        /* channel 1 touch */
        if((sample_refnum[0] - samplenum[0]) > 0x15) {
            /* pin1 of group5 is touched */
            keystatus[0] = 1;
        } else {
            keystatus[0] = 0;
            gd_eval_led_off(LED1);

            gd_eval_led_off(LED4);
        }
        hals_tsi_channel_pin_disable(TSI_CHCFG_G5P1);

        /* acquisition pin2 of group5 */
        tsi_transfer_pin(TSI_CHCFG_G5P2);

        /* check the TSI flag--end of acquisition interrupt */
        if(SET == hals_tsi_flag_get(TSI_FLAG_CTCF)) {
            samplenum[1] = hal_tsi_group_cycle_get(&tsi_info, TSI_GROUP_IDX5);
        }

        /* light LED2 */
        if((sample_refnum[1] - samplenum[1]) > 0x20) {
            /* TSI_GROUP6_PIN3 is touched */
            keystatus[1] = 1;
            gd_eval_led_on(LED2);
        } else {
            keystatus[1] = 0;
            gd_eval_led_off(LED2);
        }
        hals_tsi_channel_pin_disable(TSI_CHCFG_G5P2);

        /* acquisition pin3 of group5 */
        tsi_transfer_pin(TSI_CHCFG_G5P3);

        /* check the TSI flag--end of acquisition interrupt */
        if(SET == hals_tsi_flag_get(TSI_FLAG_CTCF)) {
            samplenum[2] =  hal_tsi_group_cycle_get(&tsi_info, TSI_GROUP_IDX5);
        }
        /* light LED3 */
        if((sample_refnum[2] - samplenum[2]) > 0x20) {
            /* pin3 of group5 is touched */
            keystatus[2] = 1;
            gd_eval_led_on(LED3);
        } else {
            keystatus[2] = 0;
            gd_eval_led_off(LED3);
        }
        hals_tsi_channel_pin_disable(TSI_CHCFG_G5P3);

        /* judge specific location of channel1 */
        if(1 == keystatus[0]) {
            if((samplenum[1] - samplenum[0] + threshold1) < (samplenum[2] - samplenum[0])) {
                /* light LED1 when touch the left location */
                gd_eval_led_on(LED1);
            } else if((samplenum[1] - samplenum[0] + threshold2) > (samplenum[2] - samplenum[0])) {
                /* light LED4 when touch the right location */
                gd_eval_led_on(LED4);
            }
        }

        /* delay for a period of time while all banks have been acquired */
        delay(0x1FFFFF);
        /* user code [local 3] end */
    }
}
/* user code [global 1] begin */
/*!
    \brief      insert a delay time
    \param[in]  nCount: stall Count
    \param[out] none
    \retval     none
*/
void delay(uint32_t nCount)
{
    for(; nCount != 0; nCount--);
}

/*!
    \brief      acquisition pin y of group x,x=0..5,y=0..3
    \param[in]  tsi_groupx_piny: TSI_CHCFG_GxPy,pin y of group x
    \param[out] none
    \retval     none
*/
void tsi_transfer_pin(uint32_t tsi_groupx_piny)
{
    /* configure the TSI pin channel mode */
    hals_tsi_channel_pin_enable(tsi_groupx_piny);

    /* wait capacitors discharge */
    delay(0xD00);

    /* clear both MNERR and CTCF flags */
    hals_tsi_flag_clear(TSI_FLAG_CTCF | TSI_FLAG_MNERR);

    /* start a new acquisition */
    hals_tsi_software_start();

    /* wait the specified TSI flag state: MNERR or CTCF */
    while(RESET == hals_tsi_flag_get(TSI_FLAG_CTCF | TSI_FLAG_MNERR));
}
/* user code [global 1] end */
