/*
    \file  main.c
*/
/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32c2x1.h"
#include "gd32c2x1_init.h"

/* External Includes*/
/* user code [External Includes] begin */
#include <stdio.h>
/* user code [External Includes] end */

/* Private Type Definitions */
/* user code [Private Type Definitions] begin */
typedef enum
{
    LED1 = 0U,
    LED2 = 1U,
    LED3 = 2U
} led_typedef_enum;
/* user code [Private Type Definitions] end */

/* Private Macros */
/* user code [Private Macros] begin */
#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
    #define __CLANG_ARM
#endif

#if defined(__CC_ARM) || defined(__CLANG_ARM) /* For ARM compiler */
    #define PUTCHAR_PROTOTYPE int fputc(int ch, FILE *f)
#elif defined(__GNUC__) && !defined(__clang__) /* For GNU GCC compiler */
    #define PUTCHAR_PROTOTYPE int __io_putchar(int ch)
#else
    #error Not supported toolchain
#endif

#define FMC_WRITE_START_ADDR ((uint32_t)0x08004000U)
#define FMC_WRITE_END_ADDR   ((uint32_t)0x08005000U)
/* user code [Private Macros] end */

/* Private Constants */
/* user code [Private Constants] begin */

/* user code [Private Constants] end */

/* Private Variables */
/* user code [Private Variables] begin */
uint64_t *ptrd = NULL;
uint64_t data  = 0x01234567AABBCCDDU;

led_typedef_enum led_num = LED3;

/* calculate the count of doubleword to be programmed/erased */
uint32_t dword_cnt = ((FMC_WRITE_END_ADDR - FMC_WRITE_START_ADDR) >> 3U);
/* user code [Private Variables] end */

/* Private Function Declaration */
/* user code [Private Function Declaration] begin */
static void fmc_erase_pages(void);
static void fmc_program(void);
static void fmc_erase_pages_check(void);
static void fmc_program_check(void);
/* user code [Private Function Declaration] end */

/* Extern Variables */
/* user code [Extern Variables] begin */

/* user code [Extern Variables] end */

/*!
    \brief      main function
    \param[in]  none
    \param[out] none
    \retval     none
*/
int main(void)
{
    /* user code [local 0] begin */

    /* user code [local 0] end */

    msd_system_init();
    msd_clock_init();

    /* user code [local 1] begin */

    /* user code [local 1] end */


    msd_gpio_init();

    /* user code [local 2] begin */
    /* step1: erase pages and check if it is successful. If not, light the LED1. */
    fmc_erase_pages();
    fmc_erase_pages_check();

    /* step2: program and check if it is successful. If not, light the LED2. */
    fmc_program();
    fmc_program_check();

    /* if all the operations are successful, light the LED3. */
    if(LED3 == led_num) {
        gpio_bit_set(GPIOD, GPIO_PIN_3);
    }
    /* user code [local 2] end */

    while(1){
        /* user code [local 3] begin */

        /* user code [local 3] end */
    }
}

/* user code [Public Functions Implementations] begin */

/* user code [Public Functions Implementations] end */

/* user code [Private Function Implementations] begin */
/*!
    \brief retarget the C library printf function to the USART
    \param[in] ch: The character to be transmitted via USART.
    \param[in] f: A standard C library file pointer, not actually used in this function.
    \param[out] none
    \retval ch: The character that was transmitted via USART.
*/
PUTCHAR_PROTOTYPE
{
    usart_data_transmit(USART0, (uint8_t)ch);

    while(RESET == usart_flag_get(USART0, USART_FLAG_TBE));

    return ch;
}

/*!
    \brief      erase fmc pages from FMC_WRITE_START_ADDR to FMC_WRITE_END_ADDR
    \param[in]  none
    \param[out] none
    \retval     none
*/
static void fmc_erase_pages(void)
{
    uint32_t erase_count;
    uint32_t erase_page_num_start;
    uint32_t page_cnt;
    uint32_t address = FMC_WRITE_START_ADDR;

    /* calculate the count of page to be programmed/erased */
    page_cnt = (FMC_WRITE_END_ADDR - FMC_WRITE_START_ADDR) / MAIN_FLASH_PAGE_SIZE + 1;

    /* calculate the start page number to be programmed/erased */
    erase_page_num_start = (address - MAIN_FLASH_BASE_ADDRESS) / MAIN_FLASH_PAGE_SIZE;

    /* unlock register FMC_CTL */
    fmc_unlock();

    /* clear all pending flags */
    fmc_flag_clear(FMC_FLAG_ENDF | FMC_FLAG_PGERR | FMC_FLAG_WPERR | FMC_FLAG_OPRERR | FMC_FLAG_PGSERR |
                   FMC_FLAG_PGMERR | FMC_FLAG_PGAERR);

    /* erase the flash pages */
    for(erase_count = 0U; erase_count < page_cnt; erase_count++) {
        fmc_page_erase(erase_page_num_start);
        fmc_flag_clear(FMC_FLAG_ENDF | FMC_FLAG_PGERR | FMC_FLAG_WPERR | FMC_FLAG_OPRERR | FMC_FLAG_PGSERR |
                       FMC_FLAG_PGMERR | FMC_FLAG_PGAERR);
        erase_page_num_start++;
    }

    /* lock register FMC_CTL after the erase operation */
    fmc_lock();
}

/*!
    \brief      program fmc word by word from FMC_WRITE_START_ADDR to FMC_WRITE_END_ADDR
    \param[in]  none
    \param[out] none
    \retval     none
*/
static void fmc_program(void)
{
    /* unlock register FMC_CTL */
    fmc_unlock();

    uint32_t address = FMC_WRITE_START_ADDR;

    /* program flash */
    while(address < FMC_WRITE_END_ADDR) {
        fmc_doubleword_program(address, data);
        address += sizeof(uint64_t);
        fmc_flag_clear(FMC_FLAG_ENDF | FMC_FLAG_PGERR | FMC_FLAG_WPERR | FMC_FLAG_OPRERR | FMC_FLAG_PGSERR |
                       FMC_FLAG_PGMERR | FMC_FLAG_PGAERR);
    }

    /* lock register FMC_CTL after the program operation */
    fmc_lock();
}

/*!
    \brief      check fmc erase result
    \param[in]  none
    \param[out] none
    \retval     none
*/
static void fmc_erase_pages_check(void)
{
    uint32_t i;

    ptrd = (uint64_t *)FMC_WRITE_START_ADDR;

    /* check flash whether has been erased */
    for(i = 0U; i < dword_cnt; i++) {
        if(0xFFFFFFFFFFFFFFFFU != (*ptrd)) {
            led_num = LED1;
            gpio_bit_set(GPIOA, GPIO_PIN_15);
            break;
        } else {
            ptrd++;
        }
    }
}

/*!
    \brief      check fmc program result
    \param[in]  none
    \param[out] none
    \retval     none
*/
static void fmc_program_check(void)
{
    uint32_t i;

    ptrd = (uint64_t *)FMC_WRITE_START_ADDR;

    /* check flash whether has been programmed */
    for(i = 0U; i < dword_cnt; i++) {
        if((*ptrd) != data) {
            led_num = LED2;
            gpio_bit_set(GPIOD, GPIO_PIN_1);
            break;
        } else {
            ptrd++;
        }
    }
}
/* user code [Private Function Implementations] end */
