/*!
    \file    gd32h77x_pmu.c
    \brief   PMU driver

    \version 2025-08-10, V0.0.0, firmware for GD32H77x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h77x_pmu.h"

/* PMU register bit offset */
#define PAR_TSW_HSICNT_OFFSET               ((uint32_t)0x00000010U)               /*!< bit offset of TSW_HSICNT in PMU_PAR */

/*!
    \brief      reset PMU register
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_deinit(void)
{
    /* reset PMU */
    rcu_periph_reset_enable(RCU_PMURST);
    rcu_periph_reset_disable(RCU_PMURST);
}

/*!
    \brief      select low voltage detector threshold
    \param[in]  lvdt_n:
      \arg        PMU_LVDT_0: voltage threshold is 2.1V
      \arg        PMU_LVDT_1: voltage threshold is 2.3V
      \arg        PMU_LVDT_2: voltage threshold is 2.4V
      \arg        PMU_LVDT_3: voltage threshold is 2.6V
      \arg        PMU_LVDT_4: voltage threshold is 2.7V
      \arg        PMU_LVDT_5: voltage threshold is 2.9V
      \arg        PMU_LVDT_6: voltage threshold is 3.0V
      \arg        PMU_LVDT_7: voltage threshold is 3.1V
    \param[out] none
    \retval     none
*/
void pmu_lvd_select(uint32_t lvdt_n)
{
    uint32_t temp;
    temp = PMU_CTL0;
    /* clear LVDT bits */
    temp &= ~PMU_CTL0_LVDT;
    /* set LVDT bits according to lvdt_n */
    temp |= lvdt_n;
    PMU_CTL0 = temp;
}

/*!
    \brief      enable PMU lvd
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_lvd_enable(void)
{
    PMU_CTL0 |= PMU_CTL0_LVDEN;
}

/*!
    \brief      disable PMU lvd
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_lvd_disable(void)
{
    PMU_CTL0 &= ~PMU_CTL0_LVDEN;
}

/*!
    \brief      select analog voltage detector threshold
    \param[in]  avdt_n:
      \arg        PMU_AVDT_0: voltage threshold is 1.7V
      \arg        PMU_AVDT_1: voltage threshold is 2.1V
      \arg        PMU_AVDT_2: voltage threshold is 2.5V
      \arg        PMU_AVDT_3: voltage threshold is 2.8V
    \param[out] none
    \retval     none
*/
void pmu_avd_select(uint32_t avdt_n)
{
    uint32_t temp;
    temp = PMU_CTL0;
    /* clear VAVDVC bits */
    temp &= ~PMU_CTL0_VAVDVC;
    /* set VAVDVC bits according to avdt_n */
    temp |= avdt_n;
    PMU_CTL0 = temp;
}

/*!
    \brief      enable PMU analog voltage detector
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_avd_enable(void)
{
    PMU_CTL0 |= PMU_CTL0_VAVDEN;
}

/*!
    \brief      disable PMU analog voltage detector
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_avd_disable(void)
{
    PMU_CTL0 &= ~PMU_CTL0_VAVDEN;
}

/*!
    \brief      select PMU V0.9V over voltage detector threshold
    \param[in]  ovdt_n:
      \arg        PMU_OVDT_0: voltage threshold is 0.9V
      \arg        PMU_OVDT_1: voltage threshold is 1.0V
      \arg        PMU_OVDT_2: voltage threshold is 1.1V
      \arg        PMU_OVDT_3: voltage threshold is 1.2V
    \param[out] none
    \retval     none
*/
void pmu_ovd_select(uint32_t ovdt_n)
{
    uint32_t temp;
    temp = PMU_CTL0;
    /* clear VOVDVC bits */
    temp &= ~PMU_CTL0_VOVDVC;
    /* set VAVDVC bits according to avdt_n */
    temp |= ovdt_n;
    PMU_CTL0 = temp;
}

/*!
    \brief      enable PMU V0.9V over voltage detector
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_ovd_enable(void)
{
    PMU_CTL0 |= PMU_CTL0_VOVDEN;
}

/*!
    \brief      disable PMU V0.9V over voltage detector
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_ovd_disable(void)
{
    PMU_CTL0 &= ~PMU_CTL0_VOVDEN;
}

/*!
    \brief      select PMU V0.9V under voltage detector threshold
    \param[in]  uvdt_n:
      \arg        PMU_UVDT_0: voltage threshold is 0.8V
      \arg        PMU_UVDT_1: voltage threshold is 0.7V
      \arg        PMU_UVDT_2: voltage threshold is 0.6V
    \param[out] none
    \retval     none
*/
void pmu_uvd_select(uint32_t uvdt_n)
{
    uint32_t temp;
    temp = PMU_CTL0;
    /* clear VAVDVC bits */
    temp &= ~PMU_CTL0_VUVDVC;
    /* set VAVDVC bits according to avdt_n */
    temp |= uvdt_n;
    PMU_CTL0 = temp;
}

/*!
    \brief      enable PMU V0.9V under voltage detector
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_uvd_enable(void)
{
    PMU_CTL0 |= PMU_CTL0_VUVDEN;
}

/*!
    \brief      disable PMU V0.9V under voltage detector
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_uvd_disable(void)
{
    PMU_CTL0 &= ~PMU_CTL0_VUVDEN;
}


/*!
    \brief      Deep-sleep mode V0.9V core voltage select
    \param[in]  sldo:
      \arg        PMU_SLDOVS_0:Deep-sleep mode LDO voltage 0.6V
      \arg        PMU_SLDOVS_1:Deep-sleep mode LDO voltage 0.7V
      \arg        PMU_SLDOVS_2:Deep-sleep mode LDO voltage 0.8V
      \arg        PMU_SLDOVS_3:Deep-sleep mode LDO voltage 0.9V
    \param[out] none
    \retval     none
*/
void pmu_sldo_output_select(uint32_t sldo)
{
    uint32_t temp;
    temp = PMU_CTL0;
    temp &= ~PMU_CTL0_SLDOVS;
    temp |= sldo;
    PMU_CTL0 = temp;
}

/*!
    \brief      PMU VBAT battery charging resistor selection
    \param[in]  resistor:
      \arg        PMU_VCRSEL_5K: 5 kOhms resistor is selected for charing VBAT battery
      \arg        PMU_VCRSEL_1P5K: 1.5 kOhms resistor is selected for charing VBAT battery
    \param[out] none
    \retval     none
*/
void pmu_vbat_charging_select(uint32_t resistor)
{
    PMU_CTL3 &= ~PMU_CTL3_VCRSEL;
    PMU_CTL3 |= resistor;
}

/*!
    \brief      enable VBAT battery charging
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_vbat_charging_enable(void)
{
    PMU_CTL3 |= PMU_CTL3_VCEN;
}

/*!
    \brief      disable VBAT battery charging
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_vbat_charging_disable(void)
{
    PMU_CTL3 &= ~PMU_CTL3_VCEN;
}

/*!
    \brief      VBAT and temperature monitoring enable
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_vbat_temp_moniter_enable(void)
{
    PMU_CTL3 |= PMU_CTL1_VBTMEN;
}

/*!
    \brief      VBAT and temperature monitoring disable
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_vbat_temp_moniter_disable(void)
{
    PMU_CTL3 &= ~PMU_CTL1_VBTMEN;
}

/*!
    \brief      USB regulator enable
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_usb_regulator_enable(void)
{
    PMU_CTL3 |= PMU_CTL3_USBSEN;
}

/*!
    \brief      USB regulator disable
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_usb_regulator_disable(void)
{
    PMU_CTL3 &= ~PMU_CTL3_USBSEN;
}

/*!
    \brief      VDD33USB voltage level detector enable
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_usb_voltage_detector_enable(void)
{
    PMU_CTL3 |= PMU_CTL3_VUSB33DEN;
}

/*!
    \brief      VDD33USB voltage level detector disable
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_usb_voltage_detector_disable(void)
{
    PMU_CTL3 &= ~PMU_CTL3_VUSB33DEN;
}

/*!
    \brief      power supply configurations
    \param[in]  smpsmode:
      \arg        PMU_DIRECT_SMPS_SUPPLY:
      \arg        PMU_BYPASS:
    \param[out] none
    \retval     none
*/
void pmu_smps_ldo_supply_config(uint32_t smpsmode)
{
    uint32_t temp;
    temp = PMU_CTL3;
    temp &= ~(PMU_CTL3_DVSEN |  PMU_CTL3_BYPASS);
    temp |= smpsmode;
    PMU_CTL3 = temp;
}

/*!
    \brief      enter sleep mode
    \param[in]  sleepmodecmd:
      \arg        WFI_CMD: use WFI command
      \arg        WFE_CMD: use WFE command
    \param[out] none
    \retval     none
*/
void pmu_to_sleepmode(uint8_t sleepmodecmd)
{
    /* clear sleepdeep bit of Cortex-M7 system control register */
    SCB->SCR &= ~((uint32_t)SCB_SCR_SLEEPDEEP_Msk);

    /* select WFI or WFE command to enter sleep mode */
    if(WFI_CMD == sleepmodecmd) {
        __WFI();
    } else {
        __SEV();
        __WFE();
        __WFE();
    }
}

/*!
    \brief      enter deepsleep mode
    \param[in]  deepsleepmodecmd:
      \arg        WFI_CMD: use WFI command
      \arg        WFE_CMD: use WFE command
    \param[out] none
    \retval     none
*/
void pmu_to_deepsleepmode(uint8_t deepsleepmodecmd)
{
    /* clear standby mode */
    PMU_CTL0 &= ~((uint32_t)(PMU_CTL0_STBMOD));

    /* set sleepdeep bit of Cortex-M7 system control register */
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;

    /* select WFI or WFE command to enter deepsleep mode */
    if(WFI_CMD == deepsleepmodecmd) {
        __WFI();
    } else {
        __SEV();
        __WFE();
        __WFE();
    }
    /* reset sleepdeep bit of Cortex-M7 system control register */
    SCB->SCR &= ~((uint32_t)SCB_SCR_SLEEPDEEP_Msk);
}

/*!
    \brief      enter standby mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_to_standbymode(void)
{
    /* set sleepdeep bit of Cortex-M7 system control register */
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;

    /* set stbmod bit */
    PMU_CTL0 |= PMU_CTL0_STBMOD;

    /* reset wakeup flag */
    if(0U != (PMU_CS & PMU_CS_WUF)) {
        PMU_CTL0 |= PMU_CTL0_WURST;
        while(0U != (PMU_CS & PMU_CS_WUF)) {}
    }

    /* enter standby mode */
    __WFI();
}

/*!
    \brief      enable wakeup pin
    \param[in]  wakeup_pin:
                one or more parameters can be selected which are shown as below:
      \arg        PMU_WAKEUP_PIN0: WKUP Pin 0
      \arg        PMU_WAKEUP_PIN1: WKUP Pin 1
      \arg        PMU_WAKEUP_PIN3: WKUP Pin 3
      \arg        PMU_WAKEUP_PIN5: WKUP Pin 5
    \param[out] none
    \retval     none
*/
void pmu_wakeup_pin_enable(uint32_t wakeup_pin)
{
    PMU_CS |= wakeup_pin;
}

/*!
    \brief      disable wakeup pin
    \param[in]  wakeup_pin:
                one or more parameters can be selected which are shown as below:
      \arg        PMU_WAKEUP_PIN0: WKUP Pin 0
      \arg        PMU_WAKEUP_PIN1: WKUP Pin 1
      \arg        PMU_WAKEUP_PIN3: WKUP Pin 3
      \arg        PMU_WAKEUP_PIN5: WKUP Pin 5
    \param[out] none
    \retval     none
*/
void pmu_wakeup_pin_disable(uint32_t wakeup_pin)
{
    PMU_CS &= ~(wakeup_pin);
}

/*!
    \brief      enable backup domain write
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_backup_write_enable(void)
{
    PMU_CTL0 |= PMU_CTL0_BKPWEN;
}

/*!
    \brief      disable backup domain write
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_backup_write_disable(void)
{
    PMU_CTL0 &= ~PMU_CTL0_BKPWEN;
}

/*!
    \brief      enable backup voltage stabilizer
    \param[in]  none
    \param[out] none
    \retval     none
    \note       This function contains scenarios leading to an infinite loop.
                Modify according to the user's actual usage scenarios.
*/
void pmu_backup_voltage_stabilizer_enable(void)
{
    PMU_CTL1 |= PMU_CTL1_BKPVSEN;
    while(RESET == (PMU_CTL1 & PMU_CTL1_BKPVSRF)) {
    }
}

/*!
    \brief      disable backup voltage stabilizer
    \param[in]  none
    \param[out] none
    \retval     none
*/
void pmu_backup_voltage_stabilizer_disable(void)
{
    PMU_CTL1 &= ~PMU_CTL1_BKPVSEN;
}

/*!
    \brief      configure IRC counter before enter Deep-sleep mode
    \param[in]  wait_time: 0x0~0x1F, IRC counter before enter Deep-sleep mode
    \param[out] none
    \retval     none
*/
void pmu_enter_deepsleep_wait_time_config(uint32_t wait_time)
{
    /* check parameter */
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_PMU_ENTER_WAITTIME(wait_time)) {
        fw_debug_report_err(PMU_MODULE_ID, API_ID(0x0021U), ERR_PARAM_OUT_OF_RANGE);
    } else
#endif
    {
        uint32_t temp;
        temp = PMU_PAR;
        temp &= ~PMU_PAR_TSW_HSICNT;
        temp |= (uint32_t)(wait_time << PAR_TSW_HSICNT_OFFSET);
        PMU_PAR = temp;
    }
}

/*!
    \brief      configure IRC counter before exit Deep-sleep mode
    \param[in]  wait_time: 0x0~0xFFF, IRC counter before exit Deep-sleep mode
    \param[out] none
    \retval     none
*/
void pmu_exit_deepsleep_wait_time_config(uint32_t wait_time)
{
    /* check parameter */
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_PMU_EXIT_WAITTIME(wait_time)) {
        fw_debug_report_err(PMU_MODULE_ID, API_ID(0x0022U), ERR_PARAM_OUT_OF_RANGE);
    } else
#endif
    {
        uint32_t temp;
        temp = PMU_PAR;
        temp &= ~PMU_PAR_PWR_CNT;
        temp |= (uint32_t)(wait_time);
        PMU_PAR = temp;
    }
}

/*!
    \brief      clear flag bit
    \param[in]  flag_reset:
      \arg        PMU_FLAG_WAKEUP: wakeup flag
      \arg        PMU_FLAG_STANDBY: standby flag
    \param[out] none
    \retval     none
*/
void pmu_flag_clear(uint32_t flag_reset)
{
    switch(flag_reset) {
    case PMU_FLAG_WAKEUP:
        PMU_CTL0 |= PMU_CTL0_WURST;
        break;
    case PMU_FLAG_STANDBY:
        PMU_CTL0 |= PMU_CTL0_STBRST;
        break;
    default:
        /* Do nothing or add error handling */
        break;
    }
}

/*!
    \brief      get flag state
    \param[in]  flag:
      \arg        PMU_FLAG_WAKEUP: wakeup flag
      \arg        PMU_FLAG_STANDBY: standby flag
      \arg        PMU_FLAG_LVDF: low voltage detector status flag
      \arg        PMU_FLAG_UVDF: record the undervoltage reset of the V0.9V
      \arg        PMU_FLAG_VUVDF0: V0.9V under voltage detector flag bit
      \arg        PMU_FLAG_VOVDF1: V0.9V over voltage detector flag bit after digital filter
      \arg        PMU_FLAG_VUVDF1: V0.9V under voltage detector flag bit after digital filter
      \arg        PMU_FLAG_VAVDF: VDDA analog voltage detector voltage output on VDDA flag
      \arg        PMU_FLAG_VOVDF0: peripheral voltage on V0.9V detector flag
      \arg        PMU_FLAG_BKPVSRF: backup voltage stabilizer ready flag
      \arg        PMU_FLAG_VBATLF: VBAT level monitoring versus low threshold
      \arg        PMU_FLAG_VBATHF: VBAT level monitoring versus high threshold
      \arg        PMU_FLAG_TEMPLF: temperature level monitoring versus low threshold
      \arg        PMU_FLAG_TEMPHF: temperature level monitoring versus high threshold
      \arg        PMU_FLAG_MIPIHTF: MIPI power switch ack flag
      \arg        PMU_FLAG_USB33RF: USB supply ready flag bit
    \param[out] none
    \retval     FlagStatus SET or RESET
*/
FlagStatus pmu_flag_get(uint32_t flag)
{
    FlagStatus reval = RESET;
    if(PMU_REG_VAL(flag) & BIT(PMU_BIT_POS(flag))) {
        reval = SET;
    } else {
        reval = RESET;
    }
    return reval;
}