/*!
    \file    gd32h77x_ospi.c
    \brief   OSPI driver

    \version 2025-08-10, V0.0.0, firmware for GD32H77x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h77x_ospi.h"

static void ospi_config(uint32_t ospi_periph, ospi_parameter_struct *ospi_struct, ospi_regular_cmd_struct *cmd_struct);

/*!
    \brief      reset the OSPI peripheral  (API_ID: 0x0001U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_deinit(uint32_t ospi_periph)
{
    switch(ospi_periph) {
    case OSPI0:
        /* reset OSPI0 */
        rcu_periph_reset_enable(RCU_OSPI0RST);
        rcu_periph_reset_disable(RCU_OSPI0RST);
        break;
    case OSPI1:
        /* reset OSPI1 */
        rcu_periph_reset_enable(RCU_OSPI1RST);
        rcu_periph_reset_disable(RCU_OSPI1RST);
        break;
    default:
        break;
    }
}

/*!
    \brief      initialize the parameters of OSPI struct with default values (API_ID: 0x0002U)
    \param[in]  none
    \param[out] ospi_struct: the initialized struct ospi_parameter_struct pointer
    \retval     none
*/
void ospi_struct_init(ospi_parameter_struct *ospi_struct)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_VALID_POINTER(ospi_struct)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x0002U), ERR_PARAM_POINTER);
    } else
#endif /* FW_DEBUG_ERR_REPORT */
    {
        /* configure the structure with default value */
        ospi_struct->prescaler          = 2U;
        ospi_struct->sample_shift       = OSPI_SAMPLE_SHIFTING_NONE;
        ospi_struct->clock_mode         = OSPI_CLOCK_MODE_LOW;
        ospi_struct->fifo_threshold     = OSPI_FIFO_THRESHOLD_4;
        ospi_struct->dual_quad          = OSPI_DUAL_QUAD_MODE_DISABLE;
        ospi_struct->device_size        = OSPI_MESZ_512_MBS;
        ospi_struct->free_clock_run     = OSPI_CLOCK_FREE_RUN_DISABLE;
        ospi_struct->cs_hightime        = OSPI_CS_HIGH_TIME_3_CYCLE;
        ospi_struct->memory_type        = OSPI_MICRON_MODE;
        ospi_struct->delay_hold_cycle   = OSPI_DELAY_HOLD_NONE;
        ospi_struct->delay_block_bypass = OSPI_DELAY_CLOCK_BYPASS_DISABLE;
        ospi_struct->refrate            = 0U;
        ospi_struct->clk_cs_hightime    = 0U;
    }
}

/*!
    \brief      initialize OSPI parameter (API_ID: 0x0003U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  ospi_struct: OSPI parameter initialization stuct members of the structure
                             and the member values are shown as below:
      \arg        prescaler: between 0 and 255
      \arg        fifo_threshold: OSPI_FIFO_THRESHOLD_x (x = 1, 2, ..., 31, 32)
      \arg        sample_shift: OSPI_SAMPLE_SHIFTING_NONE, OSPI_SAMPLE_SHIFTING_HALF_CYCLE
      \arg        device_size: OSPI_MESZ_x_BYTES (x = 2, 4, 8, ..., 512, 1024)
                               OSPI_MESZ_x_KBS (x = 2, 4, 8, ..., 512, 1024)
                               OSPI_MESZ_x_MBS (x = 2, 4, 8, ..., 2048, 4096)
      \arg        cs_hightime: OSPI_CS_HIGH_TIME_1_CYCLE, OSPI_CS_HIGH_TIME_2_CYCLE
                               OSPI_CS_HIGH_TIME_3_CYCLE, OSPI_CS_HIGH_TIME_4_CYCLE
                               OSPI_CS_HIGH_TIME_5_CYCLE, OSPI_CS_HIGH_TIME_6_CYCLE
                               OSPI_CS_HIGH_TIME_7_CYCLE, OSPI_CS_HIGH_TIME_8_CYCLE
      \arg        clock_mode: OSPI_CLOCK_MODE_LOW, OSPI_CLOCK_MODE_HIGH
      \arg        dual_quad: OSPI_DUAL_QUAD_MODE_DISABLE, OSPI_DUAL_QUAD_MODE_ENABLE
      \arg        memory_type: OSPI_MICRON_MODE, OSPI_MACRONIX_MODE, OSPI_STANDARD_MODE
                               OSPI_MACRONIX_RAM_MODE, OSPI_HYPERBUS_MEMORY_MODE, OSPI_HYPERBUS_REGISTER_MODE
                  free_clock_run: OSPI_CLOCK_FREE_RUN_DISABLE, OSPI_CLOCK_FREE_RUN_ENABLE
      \arg        delay_hold_cycle: OSPI_DELAY_HOLD_NONE, OSPI_DELAY_HOLD_QUARTER_CYCLE
      \arg        clk_cs_hightime: between 0 and 7
      \arg        delay_block_bypass: OSPI_DELAY_CLOCK_BYPASS_DISABLE, OSPI_DELAY_CLOCK_BYPASS_ENABLE
      \arg        refrate: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void ospi_init(uint32_t ospi_periph, ospi_parameter_struct *ospi_struct)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_VALID_POINTER(ospi_struct)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x0003U), ERR_PARAM_POINTER);
    } else
#endif /* FW_DEBUG_ERR_REPORT */
    {
        uint32_t reg = 0U;

        /* configure memory type, device size, chip select high time, delay block bypass, free running clock, clock mode */
        reg = OSPI_DCFG0(ospi_periph);

        reg &= ~(OSPI_DCFG0_DTYSEL | OSPI_DCFG0_MESZ | OSPI_DCFG0_CSHC | OSPI_DCFG0_RCKADD |
                 OSPI_DCFG0_RSCKDEN | OSPI_DCFG0_RXSFT | OSPI_DCFG0_FRCK);


        reg |= (ospi_struct->memory_type | OSPI_MESZ(ospi_struct->device_size) |
                ospi_struct->cs_hightime | ospi_struct->delay_block_bypass | ospi_struct->clock_mode);

        OSPI_DCFG0(ospi_periph) = reg;

        /* configure refresh */
        OSPI_DCFG3(ospi_periph) = (OSPI_DCFG3(ospi_periph) & ~OSPI_DCFG3_REFRATE) | ospi_struct->refrate;

        /* configure FIFO threshold */
        OSPI_CTL(ospi_periph) = (OSPI_CTL(ospi_periph) & ~OSPI_CTL_FTL) | ospi_struct->fifo_threshold;

        /* wait till BUSY flag reset */
        while(RESET != (OSPI_STAT(ospi_periph) & OSPI_FLAG_BUSY)) {
        }

        /* configure clock prescaler */
        OSPI_DCFG1(ospi_periph) = (OSPI_DCFG1(ospi_periph) & ~OSPI_DCFG1_PSC) | OSPI_PSC(ospi_struct->prescaler);

        /* configure Dual Quad mode */
        OSPI_CTL(ospi_periph) = (OSPI_CTL(ospi_periph) & ~OSPI_CTL_DQMOD) | ospi_struct->dual_quad;

        /* configure sample shifting and delay hold quarter cycle */
        OSPI_TIMCFG(ospi_periph) = (OSPI_TIMCFG(ospi_periph) & ~(OSPI_TIMCFG_SSAMPLE | OSPI_TIMCFG_DEHQC)) | (ospi_struct->sample_shift |
                                   ospi_struct->delay_hold_cycle);
    }
}

/*!
    \brief      enable OSPI (API_ID: 0x0004U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_enable(uint32_t ospi_periph)
{
    OSPI_CTL(ospi_periph) |= (uint32_t)OSPI_CTL_OSPIEN;
}

/*!
    \brief      disable OSPI (API_ID: 0x0005U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_disable(uint32_t ospi_periph)
{
    OSPI_CTL(ospi_periph) &= (uint32_t)(~OSPI_CTL_OSPIEN);
}

/*!
    \brief      select device (API_ID: 0x0006U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  deviceid: OSPI device select
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DEVICE0_SELECT: OSPI select flash 0
      \arg        OSPI_DEVICE1_SELECT: OSPI select flash 1
    \param[out] none
    \retval     none
*/
void ospi_device_select(uint32_t ospi_periph, uint32_t deviceid)
{
    OSPI_CTL(ospi_periph) &= (uint32_t)(~OSPI_CTL_DEVS);
    OSPI_CTL(ospi_periph) |= (uint32_t)(deviceid & OSPI_CTL_DEVS);
}

/*!
    \brief      configure device memory type (API_ID: 0x0007U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  dtysel: OSPI device type select
                only one parameter can be selected which is shown as below:
      \arg        OSPI_MICRON_MODE: micron mode
      \arg        OSPI_MACRONIX_MODE: micronix mode
      \arg        OSPI_STANDARD_MODE: standard mode
      \arg        OSPI_MACRONIX_RAM_MODE: micronix ram mode
      \arg        OSPI_HYPERBUS_MEMORY_MODE: hyperbus memory mode
      \arg        OSPI_HYPERBUS_REGISTER_MODE: hyperbus register mode
    \param[out] none
    \retval     none
*/
void ospi_device_memory_type_config(uint32_t ospi_periph, uint32_t dtysel)
{
    OSPI_DCFG0(ospi_periph) &= (uint32_t)(~OSPI_RESERVE_MODE);
    OSPI_DCFG0(ospi_periph) |= (uint32_t)(dtysel & OSPI_RESERVE_MODE);
}

/*!
    \brief      configure device memory size (API_ID: 0x0008U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  mesz: device memory size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_MESZ_x_BYTES (x = 2, 4, 8, ..., 512, 1024)
      \arg        OSPI_MESZ_x_KBS (x = 2, 4, 8, ..., 512, 1024)
      \arg        OSPI_MESZ_x_MBS (x = 2, 4, 8, ..., 2048, 4096)
    \param[out] none
    \retval     none
*/
void ospi_device_memory_size_config(uint32_t ospi_periph, uint32_t mesz)
{
    OSPI_DCFG0(ospi_periph) &= (uint32_t)(~OSPI_DCFG0_MESZ);
    OSPI_DCFG0(ospi_periph) |= (uint32_t)(mesz & OSPI_DCFG0_MESZ);
}

/*!
    \brief      select functional mode (API_ID: 0x0009U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  fmod: OSPI functional mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INDIRECT_WRITE: OSPI indirect write mode
      \arg        OSPI_INDIRECT_READ: OSPI indirect read mode
      \arg        OSPI_STATUS_POLLING: OSPI status polling mode
      \arg        OSPI_MEMORY_MAPPED: OSPI memory mapped mode
    \param[out] none
    \retval     none
*/
void ospi_functional_mode_config(uint32_t ospi_periph, uint32_t fmod)
{
    OSPI_CTL(ospi_periph) &= (uint32_t)(~OSPI_CTL_FMOD);
    OSPI_CTL(ospi_periph) |= (uint32_t)(fmod & OSPI_CTL_FMOD);
}

/*!
    \brief      configure status polling mode (API_ID: 0x000AU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  stop: OSPI automatic stop
                only one parameter can be selected which is shown as below:
      \arg        OSPI_AUTOMATIC_STOP_ABORT: status polling mode stop in abort or disable OSPI
      \arg        OSPI_AUTOMATIC_STOP_MATCH: status polling mode stop in match
    \param[in]  mode: OSPI match mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_MATCH_MODE_AND: status polling match mode and
      \arg        OSPI_MATCH_MODE_OR: status polling match mode or
    \param[out] none
    \retval     none
*/
void ospi_status_polling_config(uint32_t ospi_periph, uint32_t stop, uint32_t mode)
{
    OSPI_CTL(ospi_periph) &= (uint32_t)(~(OSPI_CTL_SPS | OSPI_CTL_SPMOD));
    OSPI_CTL(ospi_periph) |= (uint32_t)((stop & OSPI_CTL_SPS) | (mode & OSPI_CTL_SPMOD));
}

/*!
    \brief      configure status mask (API_ID: 0x000BU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  mask: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void ospi_status_mask_config(uint32_t ospi_periph, uint32_t mask)
{
    OSPI_STATMK(ospi_periph) &= (uint32_t)(~OSPI_STATMK_MASK);
    OSPI_STATMK(ospi_periph) |= (uint32_t)(mask & OSPI_STATMK_MASK);
}

/*!
    \brief      configure status match (API_ID: 0x000CU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  match: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void ospi_status_match_config(uint32_t ospi_periph, uint32_t match)
{
    OSPI_STATMATCH(ospi_periph) &= (uint32_t)(~OSPI_STATMATCH_MATCH);
    OSPI_STATMATCH(ospi_periph) |= (uint32_t)(match & OSPI_STATMATCH_MATCH);
}

/*!
    \brief      configure interval cycle (API_ID: 0x000DU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  interval: between 0 and 0xFFFF
    \param[out] none
    \retval     none
*/
void ospi_interval_cycle_config(uint32_t ospi_periph, uint16_t interval)
{
    OSPI_INTERVAL(ospi_periph) &= (uint32_t)(~OSPI_INTERVAL_INTERVAL);
    OSPI_INTERVAL(ospi_periph) |= (uint32_t)(interval & OSPI_INTERVAL_INTERVAL);
}

/*!
    \brief      configure timeout period (API_ID: 0x000EU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  timeout: between 0 and 0xFFFF
    \param[out] none
    \retval     none
*/
void ospi_timeout_cycle_config(uint32_t ospi_periph, uint16_t timeout)
{
    OSPI_LPOUT(ospi_periph) &= (uint32_t)(~OSPI_LPOUT_TIMEOUT);
    OSPI_LPOUT(ospi_periph) |= (uint32_t)(timeout & OSPI_LPOUT_TIMEOUT);
}

/*!
    \brief      enable OSPI timeout counter (API_ID: 0x000FU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_timeout_enable(uint32_t ospi_periph)
{
    OSPI_CTL(ospi_periph) |= (uint32_t)OSPI_CTL_TMOUTEN;
}

/*!
    \brief      disable OSPI timeout counter (API_ID: 0x0010U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_timeout_disable(uint32_t ospi_periph)
{
    OSPI_CTL(ospi_periph) &= (uint32_t)(~OSPI_CTL_TMOUTEN);
}

/*!
    \brief      enable OSPI dual-quad mod (API_ID: 0x0011U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_dual_quad_mod_enable(uint32_t ospi_periph)
{
    OSPI_CTL(ospi_periph) |= (uint32_t)OSPI_CTL_DQMOD;
}

/*!
    \brief      disable OSPI dual-quad mod (API_ID: 0x0012U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_dual_quad_mod_disable(uint32_t ospi_periph)
{
    OSPI_CTL(ospi_periph) &= (uint32_t)(~OSPI_CTL_DQMOD);
}

/*!
    \brief      configure OSPI fifo threshold level (API_ID: 0x0013U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  ftl: FIFO threshold level
                only one parameter can be selected which is shown as below:
      \arg        OSPI_FIFO_THRESHOLD_x (x = 1, 2, ..., 31, 32)
    \param[out] none
    \retval     none
*/
void ospi_fifo_level_config(uint32_t ospi_periph, uint32_t ftl)
{
    OSPI_CTL(ospi_periph) &= (uint32_t)(~OSPI_CTL_FTL);
    OSPI_CTL(ospi_periph) |= (uint32_t)(ftl & OSPI_CTL_FTL);
}

/*!
    \brief      enable clk free running when OSPI is free (API_ID: 0x0015U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_free_clk_run_enable(uint32_t ospi_periph)
{
    OSPI_DCFG0(ospi_periph) |= (uint32_t)OSPI_DCFG0_FRCK;
}

/*!
    \brief      disable clk free running when OSPI is free (API_ID: 0x0016U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_free_clk_run_disable(uint32_t ospi_periph)
{
    OSPI_DCFG0(ospi_periph) &= (uint32_t)(~OSPI_DCFG0_FRCK);
}

/*!
    \brief      configure chip select high cycle (API_ID: 0x0017U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  cshc: OSPI chip select high cycle
                only one parameter can be selected which is shown as below:
      \arg        OSPI_CS_HIGH_TIME_x_CYCLE (x = 1, 2, ..., 63, 64)
    \param[out] none
    \retval     none
*/
void ospi_chip_select_high_cycle_config(uint32_t ospi_periph, uint32_t cshc)
{
    OSPI_DCFG0(ospi_periph) &= (uint32_t)(~OSPI_DCFG0_CSHC);
    OSPI_DCFG0(ospi_periph) |= (uint32_t)(cshc & OSPI_DCFG0_CSHC);
}

/*!
    \brief      configure OSPI prescaler (API_ID: 0x0018U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  psc: between 0 and 0xFF
    \param[out] none
    \retval     none
*/
void ospi_prescaler_config(uint32_t ospi_periph, uint32_t psc)
{
    OSPI_DCFG1(ospi_periph) &= (uint32_t)(~OSPI_DCFG1_PSC);
    OSPI_DCFG1(ospi_periph) |= (uint32_t)(psc & OSPI_DCFG1_PSC);
}

/*!
    \brief      configure refresh rate (API_ID: 0x0019U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  refrate: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void ospi_refresh_config(uint32_t ospi_periph, uint32_t refrate)
{
    OSPI_DCFG3(ospi_periph) &= (uint32_t)(~OSPI_DCFG3_REFRATE);
    OSPI_DCFG3(ospi_periph) |= (uint32_t)(refrate & OSPI_DCFG3_REFRATE);
}

/*!
    \brief      enable OSPI dqs mode (API_ID: 0x001AU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_dqs_mode_enable(uint32_t ospi_periph)
{
    OSPI_TCFG(ospi_periph) |= (uint32_t)OSPI_TCFG_DQSEN;
}

/*!
    \brief      disable OSPI dqs mode (API_ID: 0x001BU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_dqs_mode_disable(uint32_t ospi_periph)
{
    OSPI_TCFG(ospi_periph) &= (uint32_t)(~OSPI_TCFG_DQSEN);
}

/*!
    \brief      send instruction only once mode (API_ID: 0x001CU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  sioo: OSPI sioo mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_SIOO_INST_EVERY_CMD: OSPI send instruction on every commmand sequence
      \arg        OSPI_SIOO_INST_ONLY_FIRST_CMD: OSPI send instruction only for the first sequence
    \param[out] none
    \retval     none
*/
void ospi_send_instruction_mode_config(uint32_t ospi_periph, uint32_t sioo)
{
    OSPI_TCFG(ospi_periph) &= (uint32_t)(~OSPI_TCFG_SIOO);
    OSPI_TCFG(ospi_periph) |= (uint32_t)(sioo & OSPI_TCFG_SIOO);
}

/*!
    \brief      configure dummy cycles number (API_ID: 0x001DU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  dumyc: between 0 and 0x1F
    \param[out] none
    \retval     none
*/
void ospi_dummy_cycles_config(uint32_t ospi_periph, uint32_t dumyc)
{
    OSPI_TIMCFG(ospi_periph) &= (uint32_t)(~OSPI_TIMCFG_DUMYC);
    OSPI_TIMCFG(ospi_periph) |= (uint32_t)(dumyc & OSPI_TIMCFG_DUMYC);
}

/*!
    \brief      delay hold 1/4 cycle (API_ID: 0x001EU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  dehqc: OSPI delay hold quarter cycle
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DELAY_HOLD_NONE: OSPI no delay hold cycle
      \arg        OSPI_DELAY_HOLD_QUARTER_CYCLE: OSPI delay hold 1/4 cycle
    \param[out] none
    \retval     none
*/
void ospi_delay_hold_cycle_config(uint32_t ospi_periph, uint32_t dehqc)
{
    OSPI_TIMCFG(ospi_periph) &= (uint32_t)(~OSPI_TIMCFG_DEHQC);
    OSPI_TIMCFG(ospi_periph) |= (uint32_t)(dehqc & OSPI_TIMCFG_DEHQC);
}

/*!
    \brief      configure sample shift (API_ID: 0x001FU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  ssample: OSPI sample shift
                only one parameter can be selected which is shown as below:
      \arg        OSPI_SAMPLE_SHIFTING_NONE: OSPI no sample shift
      \arg        OSPI_SAMPLE_SHIFTING_HALF_CYCLE: OSPI have 1/2 cycle sample shift
    \param[out] none
    \retval     none
*/
void ospi_sample_shift_config(uint32_t ospi_periph, uint32_t ssample)
{
    OSPI_TIMCFG(ospi_periph) &= (uint32_t)(~OSPI_TIMCFG_SSAMPLE);
    OSPI_TIMCFG(ospi_periph) |= (uint32_t)(ssample & OSPI_TIMCFG_SSAMPLE);
}

/*!
    \brief      configure sample shift (API_ID: 0x0020U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  sample_shift2: OSPI sample shift2, support use with
                only one parameter can be selected which are shown as below:
      \arg        OSPI_SHIFTING_NONE: no shift when sample data
      \arg        OSPI_SHIFTING_1_CYCLE: 1 sck cycle shift
      \arg        OSPI_SHIFTING_2_CYCLE: 2 sck cycle shift
      \arg        OSPI_SHIFTING_3_CYCLE: 3 sck cycle shift
      \arg        OSPI_SHIFTING_4_CYCLE: 4 sck cycle shift
      \arg        OSPI_SHIFTING_5_CYCLE: 5 sck cycle shift
      \arg        OSPI_SHIFTING_6_CYCLE: 6 sck cycle shift
      \arg        OSPI_SHIFTING_7_CYCLE: 7 sck cycle shift
    \param[out] none
    \retval     none
*/
void ospi_sample_shift2_config(uint32_t ospi_periph, uint32_t sample_shift2)
{
    OSPI_DCFG0(ospi_periph) &= (uint32_t)(~OSPI_DCFG0_RXSFT);
    OSPI_DCFG0(ospi_periph) |= (uint32_t)(sample_shift2 & OSPI_DCFG0_RXSFT);
}

/*!
    \brief      enable OSPI receive delay block (API_ID: 0x0023U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_receive_delay_block_enable(uint32_t ospi_periph)
{
    OSPI_DCFG0(ospi_periph) |= (uint32_t)OSPI_DCFG0_RSCKDEN;
}

/*!
    \brief      disable OSPI receive delay block (API_ID: 0x0024U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_receive_delay_block_disable(uint32_t ospi_periph)
{
    OSPI_DCFG0(ospi_periph) &= (uint32_t)(~OSPI_DCFG0_RSCKDEN);
}

/*!
    \brief      enable OSPI csn add 1/2 cycle mode (API_ID: 0x0025U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_csn_add_enable(uint32_t ospi_periph)
{
    OSPI_DCFG1(ospi_periph) |= (uint32_t)OSPI_DCFG1_CSNADD;
}

/*!
    \brief      disable OSPI csn add 1/2 cycle mode (API_ID: 0x0026U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_csn_add_disable(uint32_t ospi_periph)
{
    OSPI_DCFG1(ospi_periph) &= (uint32_t)(~OSPI_DCFG1_CSNADD);
}

/*!
    \brief      enable OSPI dqs rck add 1 cycle mode (API_ID: 0x0027U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_dqs_rck_add_enable(uint32_t ospi_periph)
{
    OSPI_DCFG0(ospi_periph) |= (uint32_t)OSPI_DCFG0_RCKADD;
}

/*!
    \brief      disable OSPI dqs rck add 1 cycle mode (API_ID: 0x0028U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_dqs_rck_add_disable(uint32_t ospi_periph)
{
    OSPI_DCFG0(ospi_periph) &= (uint32_t)(~OSPI_DCFG0_RCKADD);
}

/*!
    \brief      enable OSPI dqs output mode (API_ID: 0x0029U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_dqs_output_enable(uint32_t ospi_periph)
{
    OSPI_CTL(ospi_periph) |= (uint32_t)OSPI_CTL_DQSOEN;
}

/*!
    \brief      disable OSPI dqs output mode (API_ID: 0x002AU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_dqs_output_disable(uint32_t ospi_periph)
{
    OSPI_CTL(ospi_periph) &= (uint32_t)(~OSPI_CTL_DQSOEN);
}

/*!
    \brief      OSPI dqs output delay chain (API_ID: 0x002BU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  delay: between 1 and 16
    \param[out] none
    \retval     none
*/
void ospi_dqs_output_delay_chain_config(uint32_t ospi_periph, uint32_t delay)
{
    OSPI_DCFG1(ospi_periph) &= (uint32_t)(~OSPI_DCFG1_DQSODV);
    OSPI_DCFG1(ospi_periph) |= (uint32_t)(((delay & OSPI_DCFG1_DQSODV) - 1U) << 28U);
}

/*!
    \brief      enable OSPI dqs output delay chain (API_ID: 0x002CU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_dqs_output_delay_chain_enable(uint32_t ospi_periph)
{
    OSPI_DCFG1(ospi_periph) |= (uint32_t)OSPI_DCFG1_DQSODVEN;
}

/*!
    \brief      disable OSPI dqs output delay chain (API_ID: 0x002DU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_dqs_output_delay_chain_disable(uint32_t ospi_periph)
{
    OSPI_DCFG1(ospi_periph) &= (uint32_t)(~OSPI_DCFG1_DQSODVEN);
}

/*!
    \brief      OSPI transmit clk delay chain (API_ID: 0x002EU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  delay: between 1 and 16
    \param[out] none
    \retval     none
*/
void ospi_tck_delay_chain_config(uint32_t ospi_periph, uint32_t delay)
{
    OSPI_DCFG1(ospi_periph) &= (uint32_t)(~OSPI_DCFG1_TCKDV);
    OSPI_DCFG1(ospi_periph) |= (uint32_t)(((delay & OSPI_DCFG1_TCKDV) - 1U) << 12U);
}

/*!
    \brief      enable OSPI transmit clk delay chain (API_ID: 0x002FU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_tck_delay_chain_enable(uint32_t ospi_periph)
{
    OSPI_DCFG1(ospi_periph) |= (uint32_t)OSPI_DCFG1_TCKDVEN;
}

/*!
    \brief      disable OSPI transmit clk delay chain (API_ID: 0x0030U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_tck_delay_chain_disable(uint32_t ospi_periph)
{
    OSPI_DCFG1(ospi_periph) &= (uint32_t)(~OSPI_DCFG1_TCKDVEN);
}

/*!
    \brief      OSPI receive clk delay chain (API_ID: 0x0031U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  delay: between 1 and 16
    \param[out] none
    \retval     none
*/
void ospi_rck_delay_chain_config(uint32_t ospi_periph, uint32_t delay)
{
    OSPI_CTL(ospi_periph) &= (uint32_t)(~OSPI_CTL_RCKDV);
    OSPI_CTL(ospi_periph) |= (uint32_t)(((delay & OSPI_CTL_RCKDV) - 1U) << 24U);
}

/*!
    \brief      enable OSPI receive clk delay chain (API_ID: 0x0032U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_rck_delay_chain_enable(uint32_t ospi_periph)
{
    OSPI_CTL(ospi_periph) |= (uint32_t)OSPI_CTL_RCKDVMUX;
}

/*!
    \brief      disable OSPI receive clk delay chain (API_ID: 0x0033U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_rck_delay_chain_disable(uint32_t ospi_periph)
{
    OSPI_CTL(ospi_periph) &= (uint32_t)(~OSPI_CTL_RCKDVMUX);
}

/*!
    \brief      configure data length (API_ID: 0x0034U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  dtlen: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void ospi_data_length_config(uint32_t ospi_periph, uint32_t dtlen)
{
    OSPI_DTLEN(ospi_periph) &= (uint32_t)(~OSPI_DTLEN_DTLEN);
    OSPI_DTLEN(ospi_periph) |= (uint32_t)(dtlen & OSPI_DTLEN_DTLEN);
}

/*!
    \brief      configure instruction mode (API_ID: 0x0035U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  imod: OSPI instruction mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INSTRUCTION_NONE: no instruction mode
      \arg        OSPI_INSTRUCTION_1_LINE: instruction mode on a single line
      \arg        OSPI_INSTRUCTION_2_LINES: instruction mode on two lines
      \arg        OSPI_INSTRUCTION_4_LINES: instruction mode on four lines
      \arg        OSPI_INSTRUCTION_8_LINES: instruction mode on eight lines
    \param[in]  insdtr: OSPI instruction double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INSDTR_MODE_DISABLE: instruction double transfer rate mode disable
      \arg        OSPI_INSDTR_MODE_ENABLE: instruction double transfer rate mode enable
    \param[in]  inssz: OSPI instruction size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INSTRUCTION_8_BITS: instruction size on 8-bit address
      \arg        OSPI_INSTRUCTION_16_BITS: instruction size on 16-bit address
      \arg        OSPI_INSTRUCTION_24_BITS: instruction size on 24-bit address
      \arg        OSPI_INSTRUCTION_32_BITS: instruction size on 32-bit address
    \param[in]  instruction: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void ospi_instruction_config(uint32_t ospi_periph, uint32_t imod, uint32_t insdtr, uint32_t inssz, uint32_t instruction)
{
    OSPI_TCFG(ospi_periph) &= (uint32_t)~(OSPI_TCFG_IMOD | OSPI_TCFG_INSSZ | OSPI_TCFG_INSDTR);
    OSPI_TCFG(ospi_periph) |= (uint32_t)((imod & OSPI_TCFG_IMOD) | (inssz & OSPI_TCFG_INSSZ) | (insdtr & OSPI_TCFG_INSDTR));
    OSPI_INS(ospi_periph) &= (uint32_t)(~OSPI_INS_INSTRUCTION);
    OSPI_INS(ospi_periph) |= (uint32_t)(instruction & OSPI_INS_INSTRUCTION);
}

/*!
    \brief      configure address mode (API_ID: 0x0036U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  addrmod: OSPI address mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRESS_NONE: no address mode
      \arg        OSPI_ADDRESS_1_LINE: address mode on a single line
      \arg        OSPI_ADDRESS_2_LINES: address mode on two lines
      \arg        OSPI_ADDRESS_4_LINES: address mode on four lines
      \arg        OSPI_ADDRESS_8_LINES: address mode on eight lines
    \param[in]  addrdtr: OSPI address double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRDTR_MODE_DISABLE: address double transfer rate mode disable
      \arg        OSPI_ADDTR_MODE_ENABLE: address double transfer rate mode enable
    \param[in]  addrsz: OSPI address size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRESS_8_BITS: address size on 8-bit address
      \arg        OSPI_ADDRESS_16_BITS: address size on 16-bit address
      \arg        OSPI_ADDRESS_24_BITS: address size on 24-bit address
      \arg        OSPI_ADDRESS_32_BITS: address size on 32-bit address
    \param[in]  addr: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void ospi_address_config(uint32_t ospi_periph, uint32_t addrmod, uint32_t addrdtr, uint32_t addrsz, uint32_t addr)
{
    OSPI_TCFG(ospi_periph) &= (uint32_t)~(OSPI_TCFG_ADDRMOD | OSPI_TCFG_ADDRSZ | OSPI_TCFG_ADDRDTR);
    OSPI_TCFG(ospi_periph) |= (uint32_t)((addrmod & OSPI_TCFG_ADDRMOD) | (addrsz & OSPI_TCFG_ADDRSZ) | (addrdtr & OSPI_TCFG_ADDRDTR));
    OSPI_ADDR(ospi_periph) &= (uint32_t)(~OSPI_ADDR_ADDR);
    OSPI_ADDR(ospi_periph) |= (uint32_t)(addr & OSPI_ADDR_ADDR);
}

/*!
    \brief      configure alternate bytes mode (API_ID: 0x0037U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  atlemod: OSPI alternate bytes mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ALTERNATE_BYTES_NONE: no alternate bytes mode
      \arg        OSPI_ALTERNATE_BYTES_1_LINE: alternate mode on a single line
      \arg        OSPI_ALTERNATE_BYTES_2_LINES: alternate bytes mode on two lines
      \arg        OSPI_ALTERNATE_BYTES_4_LINES: alternate bytes mode on four lines
      \arg        OSPI_ALTERNATE_BYTES_8_LINES: alternate bytes mode on eight lines
    \param[in]  abdtr: OSPI alternate bytes double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ABDTR_MODE_DISABLE: alternate bytes double transfer rate mode disable
      \arg        OSPI_ABDTR_MODE_ENABLE: alternate bytes double transfer rate mode enable
    \param[in]  altesz: OSPI alternate bytes size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ALTERNATE_BYTES_8_BITS: alternate bytes size on 8-bit address
      \arg        OSPI_ALTERNATE_BYTES_16_BITS: alternate bytes size on 16-bit address
      \arg        OSPI_ALTERNATE_BYTES_24_BITS: alternate bytes size on 24-bit address
      \arg        OSPI_ALTERNATE_BYTES_32_BITS: alternate bytes size on 32-bit address
    \param[in]  alte: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void ospi_alternate_bytes_config(uint32_t ospi_periph, uint32_t atlemod, uint32_t abdtr, uint32_t altesz, uint32_t alte)
{
    OSPI_TCFG(ospi_periph) &= (uint32_t)~(OSPI_TCFG_ALTEMOD | OSPI_TCFG_ALTESZ | OSPI_TCFG_ABDTR);
    OSPI_TCFG(ospi_periph) |= (uint32_t)((atlemod & OSPI_TCFG_ALTEMOD) | (altesz & OSPI_TCFG_ALTESZ) | (abdtr & OSPI_TCFG_ABDTR));
    OSPI_ALTE(ospi_periph) &= (uint32_t)(~OSPI_ALTE_ALTE);
    OSPI_ALTE(ospi_periph) |= (uint32_t)(alte & OSPI_ALTE_ALTE);
}

/*!
    \brief      configure data mode (API_ID: 0x0038U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  datamod: OSPI data mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DATA_NONE: no data mode
      \arg        OSPI_DATA_1_LINE: data mode on a single line
      \arg        OSPI_DATA_2_LINES: data mode on two lines
      \arg        OSPI_DATA_4_LINES: data mode on four lines
      \arg        OSPI_DATA_8_LINES: data mode on eight lines
    \param[in]  dadtr: OSPI data double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DADTR_MODE_DISABLE: data double transfer rate mode disable
      \arg        OSPI_DADTR_MODE_ENABLE: data double transfer rate mode enable
    \param[out] none
    \retval     none
*/
void ospi_data_config(uint32_t ospi_periph, uint32_t datamod, uint32_t dadtr)
{
    OSPI_TCFG(ospi_periph) &= (uint32_t)~(OSPI_TCFG_DATAMOD | OSPI_TCFG_DADTR);
    OSPI_TCFG(ospi_periph) |= (uint32_t)((datamod & OSPI_TCFG_DATAMOD) | (dadtr & OSPI_TCFG_DADTR));
}

/*!
    \brief      OSPI transmit data (API_ID: 0x0039U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  data: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void ospi_data_transmit(uint32_t ospi_periph, uint32_t data)
{
    OSPI_DATA(ospi_periph) = (uint32_t)data;
}

/*!
    \brief      OSPI receive data (API_ID: 0x003AU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     between 0 and 0xFFFFFFFF
*/
uint32_t ospi_data_receive(uint32_t ospi_periph)
{
    return ((uint32_t)OSPI_DATA(ospi_periph));
}

/*!
    \brief      enable OSPI MDMA (API_ID: 0x003BU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_mdma_enable(uint32_t ospi_periph)
{
    OSPI_CTL(ospi_periph) |= (uint32_t)OSPI_CTL_MDMAEN;
}

/*!
    \brief      disable OSPI MDMA (API_ID: 0x003CU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_mdma_disable(uint32_t ospi_periph)
{
    OSPI_CTL(ospi_periph) &= (uint32_t)(~OSPI_CTL_MDMAEN);
}

/*!
    \brief      abort the current transmission (API_ID: 0x003DU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_transmission_abort(uint32_t ospi_periph)
{
    /* check if the MDMA is enabled */
    if(RESET != (OSPI_CTL(ospi_periph) & OSPI_CTL_MDMAEN)) {
        /* disable the MDMA transfer on the OSPI side */
        OSPI_CTL(ospi_periph) &= ~OSPI_CTL_MDMAEN;
    }

    if(RESET != (OSPI_STAT(ospi_periph) & OSPI_FLAG_BUSY)) {
        /* perform an abort of the OSPI */
        OSPI_CTL(ospi_periph) |= OSPI_CTL_ABORT;

        /* wait until the transfer complete flag is set to go back in idle state */
        while(RESET == (OSPI_STAT(ospi_periph) & OSPI_FLAG_TC));

        /* clear transfer complete flag */
        OSPI_STATC(ospi_periph) = OSPI_STATC_TCC;

        /* wait until the busy flag is reset to go back in idle state */
        while(RESET != (OSPI_STAT(ospi_periph) & OSPI_FLAG_BUSY));
    }
}

/*!
    \brief      abort the current transmission  (API_ID: 0x003EU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_interrupt_transmission_abort(uint32_t ospi_periph)
{
    /* disable all interrupts */
    OSPI_CTL(ospi_periph) &= ~(OSPI_CTL_TERRIE | OSPI_CTL_TCIE | OSPI_CTL_FTIE | OSPI_CTL_SMIE | OSPI_CTL_TMOUTIE);

    /* check if the DMA is enabled */
    if(RESET != (OSPI_CTL(ospi_periph) & OSPI_CTL_MDMAEN)) {
        /* disable the DMA transfer on the OctoSPI side */
        OSPI_CTL(ospi_periph) &= ~OSPI_CTL_MDMAEN;
    } else {
        if(RESET != (OSPI_STAT(ospi_periph) & OSPI_FLAG_BUSY)) {
            /* clear transfer complete flag */
            OSPI_STATC(ospi_periph) = OSPI_STATC_TCC;

            /* enable the transfer complete interrupts */
            OSPI_CTL(ospi_periph) |= OSPI_CTL_TCIE;

            /* perform an abort of the OctoSPI */
            OSPI_CTL(ospi_periph) |= OSPI_CTL_ABORT;
        }
    }
}

/*!
    \brief      configure wrap instruction mode (API_ID: 0x0040U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  imod: OSPI instruction mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INSTRUCTION_NONE: no instruction mode
      \arg        OSPI_INSTRUCTION_1_LINE: instruction mode on a single line
      \arg        OSPI_INSTRUCTION_2_LINES: instruction mode on two lines
      \arg        OSPI_INSTRUCTION_4_LINES: instruction mode on four lines
      \arg        OSPI_INSTRUCTION_8_LINES: instruction mode on eight lines
    \param[in]  insdtr: OSPI instruction double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INSDTR_MODE_DISABLE: instruction double transfer rate mode disable
      \arg        OSPI_INSDTR_MODE_ENABLE: instruction double transfer rate mode enable
    \param[in]  inssz: OSPI instruction size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INSTRUCTION_8_BITS: instruction size on 8-bit address
      \arg        OSPI_INSTRUCTION_16_BITS: instruction size on 16-bit address
      \arg        OSPI_INSTRUCTION_24_BITS: instruction size on 24-bit address
      \arg        OSPI_INSTRUCTION_32_BITS: instruction size on 32-bit address
    \param[in]  instruction: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void ospi_wrap_instruction_config(uint32_t ospi_periph, uint32_t imod, uint32_t insdtr, uint32_t inssz, uint32_t instruction)
{
    OSPI_WPTCFG(ospi_periph) &= (uint32_t)~(OSPI_WPTCFG_IMOD | OSPI_WPTCFG_INSSZ | OSPI_WPTCFG_INSDTR);
    OSPI_WPTCFG(ospi_periph) |= (uint32_t)((imod & OSPI_WPTCFG_IMOD) | (inssz & OSPI_WPTCFG_INSSZ) | (insdtr & OSPI_WPTCFG_INSDTR));
    OSPI_WPINS(ospi_periph) &= (uint32_t)(~OSPI_INS_INSTRUCTION);
    OSPI_WPINS(ospi_periph) |= (uint32_t)(instruction & OSPI_INS_INSTRUCTION);
}

/*!
    \brief      configure wrap address mode (API_ID: 0x0041U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  addrmod: OSPI address mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRESS_NONE: no address mode
      \arg        OSPI_ADDRESS_1_LINE: address mode on a single line
      \arg        OSPI_ADDRESS_2_LINES: address mode on two lines
      \arg        OSPI_ADDRESS_4_LINES: address mode on four lines
      \arg        OSPI_ADDRESS_8_LINES: address mode on eight lines
    \param[in]  addrdtr: OSPI address double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRDTR_MODE_DISABLE: address double transfer rate mode disable
      \arg        OSPI_ADDTR_MODE_ENABLE: address double transfer rate mode enable
    \param[in]  addrsz: OSPI address size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRESS_8_BITS: address size on 8-bit address
      \arg        OSPI_ADDRESS_16_BITS: address size on 16-bit address
      \arg        OSPI_ADDRESS_24_BITS: address size on 24-bit address
      \arg        OSPI_ADDRESS_32_BITS: address size on 32-bit address
    \param[in]  addr: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void ospi_wrap_address_config(uint32_t ospi_periph, uint32_t addrmod, uint32_t addrdtr, uint32_t addrsz, uint32_t addr)
{
    OSPI_WPTCFG(ospi_periph) &= (uint32_t)~(OSPI_WPTCFG_ADDRMOD | OSPI_WPTCFG_ADDRSZ | OSPI_WPTCFG_ADDRDTR);
    OSPI_WPTCFG(ospi_periph) |= (uint32_t)((addrmod & OSPI_WPTCFG_ADDRMOD) | (addrsz & OSPI_WPTCFG_ADDRSZ) | (addrdtr & OSPI_WPTCFG_ADDRDTR));
    OSPI_ADDR(ospi_periph) &= (uint32_t)(~OSPI_ADDR_ADDR);
    OSPI_ADDR(ospi_periph) |= (uint32_t)(addr & OSPI_ADDR_ADDR);
}

/*!
    \brief      configure wrap alternate bytes mode (API_ID: 0x0042U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  atlemod: OSPI alternate bytes mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ALTERNATE_BYTES_NONE: no alternate bytes mode
      \arg        OSPI_ALTERNATE_BYTES_1_LINE: alternate mode on a single line
      \arg        OSPI_ALTERNATE_BYTES_2_LINES: alternate bytes mode on two lines
      \arg        OSPI_ALTERNATE_BYTES_4_LINES: alternate bytes mode on four lines
      \arg        OSPI_ALTERNATE_BYTES_8_LINES: alternate bytes mode on eight lines
    \param[in]  abdtr: OSPI alternate bytes double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ABDTR_MODE_DISABLE: alternate bytes double transfer rate mode disable
      \arg        OSPI_ABDTR_MODE_ENABLE: alternate bytes double transfer rate mode enable
    \param[in]  altesz: OSPI alternate bytes size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ALTERNATE_BYTES_8_BITS: alternate bytes size on 8-bit address
      \arg        OSPI_ALTERNATE_BYTES_16_BITS: alternate bytes size on 16-bit address
      \arg        OSPI_ALTERNATE_BYTES_24_BITS: alternate bytes size on 24-bit address
      \arg        OSPI_ALTERNATE_BYTES_32_BITS: alternate bytes size on 32-bit address
    \param[in]  alte: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void ospi_wrap_alternate_bytes_config(uint32_t ospi_periph, uint32_t atlemod, uint32_t abdtr, uint32_t altesz, uint32_t alte)
{
    OSPI_WPTCFG(ospi_periph) &= (uint32_t)~(OSPI_WPTCFG_ALTEMOD | OSPI_WPTCFG_ALTESZ | OSPI_WPTCFG_ABDTR);
    OSPI_WPTCFG(ospi_periph) |= (uint32_t)((atlemod & OSPI_WPTCFG_ALTEMOD) | (altesz & OSPI_WPTCFG_ALTESZ) | (abdtr & OSPI_WPTCFG_ABDTR));
    OSPI_WPALTE(ospi_periph) &= (uint32_t)(~OSPI_ALTE_ALTE);
    OSPI_WPALTE(ospi_periph) |= (uint32_t)(alte & OSPI_ALTE_ALTE);
}

/*!
    \brief      configure wrap data mode (API_ID: 0x0043U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  datamod: OSPI data mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DATA_NONE: no data mode
      \arg        OSPI_DATA_1_LINE: data mode on a single line
      \arg        OSPI_DATA_2_LINES: data mode on two lines
      \arg        OSPI_DATA_4_LINES: data mode on four lines
      \arg        OSPI_DATA_8_LINES: data mode on eight lines
    \param[in]  dadtr: OSPI data double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DADTR_MODE_DISABLE: data double transfer rate mode disable
      \arg        OSPI_DADTR_MODE_ENABLE: data double transfer rate mode enable
    \param[out] none
    \retval     none
*/
void ospi_wrap_data_config(uint32_t ospi_periph, uint32_t datamod, uint32_t dadtr)
{
    OSPI_WPTCFG(ospi_periph) &= (uint32_t)~(OSPI_WPTCFG_DATAMOD | OSPI_WPTCFG_DADTR);
    OSPI_WPTCFG(ospi_periph) |= (uint32_t)((datamod & OSPI_WPTCFG_DATAMOD) | (dadtr & OSPI_WPTCFG_DADTR));
}

/*!
    \brief      enable OSPI wrap dqs mode (API_ID: 0x0044U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_wrap_dqs_enable(uint32_t ospi_periph)
{
    OSPI_WPTCFG(ospi_periph) |= (uint32_t)OSPI_WPTCFG_DQSEN;
}

/*!
    \brief      disable OSPI wrap dqs mode (API_ID: 0x0045U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_wrap_dqs_disable(uint32_t ospi_periph)
{
    OSPI_WPTCFG(ospi_periph) &= (uint32_t)(~OSPI_WPTCFG_DQSEN);
}

/*!
    \brief      configure wrap dummy cycles number (API_ID: 0x0046U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  dumyc: number of dummy cycles
      \arg        OSPI_DUMYC_CYCLES_x (x = 0, 1, 2, ..., 30, 31)
    \param[out] none
    \retval     none
*/
void ospi_wrap_dummy_cycles_config(uint32_t ospi_periph, uint32_t dumyc)
{
    OSPI_WPTIMCFG(ospi_periph) &= (uint32_t)(~OSPI_WPTIMCFG_DUMYC);
    OSPI_WPTIMCFG(ospi_periph) |= (uint32_t)(dumyc & OSPI_WPTIMCFG_DUMYC);
}

/*!
    \brief      delay hold 1/4 cycle in wrap (API_ID: 0x0047U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  dehqc: OSPI delay hold quarter cycle
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DELAY_HOLD_NONE: OSPI no delay hold cycle
      \arg        OSPI_DELAY_HOLD_QUARTER_CYCLE: OSPI delay hold 1/4 cycle
    \param[out] none
    \retval     none
*/
void ospi_wrap_delay_hold_cycle_config(uint32_t ospi_periph, uint32_t dehqc)
{
    OSPI_WPTIMCFG(ospi_periph) &= (uint32_t)(~OSPI_WPTIMCFG_DEHQC);
    OSPI_WPTIMCFG(ospi_periph) |= (uint32_t)(dehqc & OSPI_WPTIMCFG_DEHQC);
}

/*!
    \brief      configure sample shift in wrap (API_ID: 0x0048U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  ssample: OSPI sample shift
                only one parameter can be selected which is shown as below:
      \arg        OSPI_SAMPLE_SHIFTING_NONE: OSPI no sample shift
      \arg        OSPI_SAMPLE_SHIFTING_HALF_CYCLE: OSPI have 1/2 cycle sample shift
    \param[out] none
    \retval     none
*/
void ospi_wrap_sample_shift_config(uint32_t ospi_periph, uint32_t ssample)
{
    OSPI_WPTIMCFG(ospi_periph) &= (uint32_t)(~OSPI_WPTIMCFG_SSAMPLE);
    OSPI_WPTIMCFG(ospi_periph) |= (uint32_t)(ssample & OSPI_WPTIMCFG_SSAMPLE);
}

/*!
    \brief      configure write instruction mode (API_ID: 0x0049U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  imod: OSPI instruction mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INSTRUCTION_NONE: no instruction mode
      \arg        OSPI_INSTRUCTION_1_LINE: instruction mode on a single line
      \arg        OSPI_INSTRUCTION_2_LINES: instruction mode on two lines
      \arg        OSPI_INSTRUCTION_4_LINES: instruction mode on four lines
      \arg        OSPI_INSTRUCTION_8_LINES: instruction mode on eight lines
    \param[in]  insdtr: OSPI instruction double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INSDTR_MODE_DISABLE: instruction double transfer rate mode disable
      \arg        OSPI_INSDTR_MODE_ENABLE: instruction double transfer rate mode enable
    \param[in]  inssz: OSPI instruction size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INSTRUCTION_8_BITS: instruction size on 8-bit address
      \arg        OSPI_INSTRUCTION_16_BITS: instruction size on 16-bit address
      \arg        OSPI_INSTRUCTION_24_BITS: instruction size on 24-bit address
      \arg        OSPI_INSTRUCTION_32_BITS: instruction size on 32-bit address
    \param[in]  instruction: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void ospi_write_instruction_config(uint32_t ospi_periph, uint32_t imod, uint32_t insdtr, uint32_t inssz, uint32_t instruction)
{
    OSPI_WTCFG(ospi_periph) &= (uint32_t)~(OSPI_WTCFG_IMOD | OSPI_WTCFG_INSSZ | OSPI_WTCFG_INSDTR);
    OSPI_WTCFG(ospi_periph) |= (uint32_t)((imod & OSPI_WTCFG_IMOD) | (inssz & OSPI_WTCFG_INSSZ) | (insdtr & OSPI_WTCFG_INSDTR));
    OSPI_WINS(ospi_periph) &= (uint32_t)(~OSPI_INS_INSTRUCTION);
    OSPI_WINS(ospi_periph) |= (uint32_t)(instruction & OSPI_INS_INSTRUCTION);
}

/*!
    \brief      configure write address mode (API_ID: 0x004AU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  addrmod: OSPI address mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRESS_NONE: no address mode
      \arg        OSPI_ADDRESS_1_LINE: address mode on a single line
      \arg        OSPI_ADDRESS_2_LINES: address mode on two lines
      \arg        OSPI_ADDRESS_4_LINES: address mode on four lines
      \arg        OSPI_ADDRESS_8_LINES: address mode on eight lines
    \param[in]  addrdtr: OSPI address double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRDTR_MODE_DISABLE: address double transfer rate mode disable
      \arg        OSPI_ADDTR_MODE_ENABLE: address double transfer rate mode enable
    \param[in]  addrsz: OSPI address size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRESS_8_BITS: address size on 8-bit address
      \arg        OSPI_ADDRESS_16_BITS: address size on 16-bit address
      \arg        OSPI_ADDRESS_24_BITS: address size on 24-bit address
      \arg        OSPI_ADDRESS_32_BITS: address size on 32-bit address
    \param[in]  addr: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void ospi_write_address_config(uint32_t ospi_periph, uint32_t addrmod, uint32_t addrdtr, uint32_t addrsz, uint32_t addr)
{
    OSPI_WTCFG(ospi_periph) &= (uint32_t)~(OSPI_WTCFG_ADDRMOD | OSPI_WTCFG_ADDRSZ | OSPI_WTCFG_ADDRDTR);
    OSPI_WTCFG(ospi_periph) |= (uint32_t)((addrmod & OSPI_WTCFG_ADDRMOD) | (addrsz & OSPI_WTCFG_ADDRSZ) | (addrdtr & OSPI_WTCFG_ADDRDTR));
    OSPI_ADDR(ospi_periph) &= (uint32_t)(~OSPI_ADDR_ADDR);
    OSPI_ADDR(ospi_periph) |= (uint32_t)(addr & OSPI_ADDR_ADDR);
}

/*!
    \brief      configure write alternate bytes mode (API_ID: 0x004BU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  atlemod: OSPI alternate bytes mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ALTERNATE_BYTES_NONE: no alternate bytes mode
      \arg        OSPI_ALTERNATE_BYTES_1_LINE: alternate mode on a single line
      \arg        OSPI_ALTERNATE_BYTES_2_LINES: alternate bytes mode on two lines
      \arg        OSPI_ALTERNATE_BYTES_4_LINES: alternate bytes mode on four lines
      \arg        OSPI_ALTERNATE_BYTES_8_LINES: alternate bytes mode on eight lines
    \param[in]  abdtr: OSPI alternate bytes double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ABDTR_MODE_DISABLE: alternate bytes double transfer rate mode disable
      \arg        OSPI_ABDTR_MODE_ENABLE: alternate bytes double transfer rate mode enable
    \param[in]  altesz: OSPI alternate bytes size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ALTERNATE_BYTES_8_BITS: alternate bytes size on 8-bit address
      \arg        OSPI_ALTERNATE_BYTES_16_BITS: alternate bytes size on 16-bit address
      \arg        OSPI_ALTERNATE_BYTES_24_BITS: alternate bytes size on 24-bit address
      \arg        OSPI_ALTERNATE_BYTES_32_BITS: alternate bytes size on 32-bit address
    \param[in]  alte: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void ospi_write_alternate_bytes_config(uint32_t ospi_periph, uint32_t atlemod, uint32_t abdtr, uint32_t altesz, uint32_t alte)
{
    OSPI_WTCFG(ospi_periph) &= (uint32_t)~(OSPI_WTCFG_ALTEMOD | OSPI_WTCFG_ALTESZ | OSPI_WTCFG_ABDTR);
    OSPI_WTCFG(ospi_periph) |= (uint32_t)((atlemod & OSPI_WTCFG_ALTEMOD) | (altesz & OSPI_WTCFG_ALTESZ) | (abdtr & OSPI_WTCFG_ABDTR));
    OSPI_WALTE(ospi_periph) &= (uint32_t)(~OSPI_ALTE_ALTE);
    OSPI_WALTE(ospi_periph) |= (uint32_t)(alte & OSPI_ALTE_ALTE);
}

/*!
    \brief      configure write data mode (API_ID: 0x004CU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  datamod: OSPI data mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DATA_NONE: no data mode
      \arg        OSPI_DATA_1_LINE: data mode on a single line
      \arg        OSPI_DATA_2_LINES: data mode on two lines
      \arg        OSPI_DATA_4_LINES: data mode on four lines
      \arg        OSPI_DATA_8_LINES: data mode on eight lines
    \param[in]  dadtr: OSPI data double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DADTR_MODE_DISABLE: data double transfer rate mode disable
      \arg        OSPI_DADTR_MODE_ENABLE: data double transfer rate mode enable
    \param[out] none
    \retval     none
*/
void ospi_write_data_config(uint32_t ospi_periph, uint32_t datamod, uint32_t dadtr)
{
    OSPI_WTCFG(ospi_periph) &= (uint32_t)~(OSPI_WTCFG_DATAMOD | OSPI_WTCFG_DADTR);
    OSPI_WTCFG(ospi_periph) |= (uint32_t)((datamod & OSPI_WTCFG_DATAMOD) | (dadtr & OSPI_WTCFG_DADTR));
}

/*!
    \brief      enable OSPI write dqs mode (API_ID: 0x004DU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_write_dqs_enable(uint32_t ospi_periph)
{
    OSPI_WTCFG(ospi_periph) |= (uint32_t)OSPI_WTCFG_DQSEN;
}

/*!
    \brief      disable OSPI write dqs mode (API_ID: 0x004EU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_write_dqs_disable(uint32_t ospi_periph)
{
    OSPI_WTCFG(ospi_periph) &= (uint32_t)(~OSPI_WTCFG_DQSEN);
}

/*!
    \brief      configure write dummy cycles number (API_ID: 0x004FU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  dumyc: number of dummy cycles
      \arg        OSPI_DUMYC_CYCLES_x (x = 0, 1, 2, ..., 30, 31)
    \param[out] none
    \retval     none
*/
void ospi_write_dummy_cycles_config(uint32_t ospi_periph, uint32_t dumyc)
{
    OSPI_WTIMCFG(ospi_periph) &= (uint32_t)(~OSPI_WTIMCFG_DUMYC);
    OSPI_WTIMCFG(ospi_periph) |= (uint32_t)(dumyc & OSPI_WTIMCFG_DUMYC);
}

/*!
    \brief      hyperbus latency mode (API_ID: 0x0050U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  lmod: OSPI latency mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_VAR_INIT_LATENCY: variable initial latency mode
      \arg        OSPI_FIXED_LATENCY: fixed latency mode
    \param[out] none
    \retval     none
*/
void ospi_latency_mode_config(uint32_t ospi_periph, uint32_t lmod)
{
    OSPI_HBLCFG(ospi_periph) &= (uint32_t)(~OSPI_HBLCFG_LMOD);
    OSPI_HBLCFG(ospi_periph) |= (uint32_t)(lmod & OSPI_HBLCFG_LMOD);
}

/*!
    \brief      enable write zero latency (API_ID: 0x0051U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_write_zero_latency_enable(uint32_t ospi_periph)
{
    OSPI_HBLCFG(ospi_periph) |= (uint32_t)OSPI_HBLCFG_WZLAT;
}

/*!
    \brief      disable write zero latency (API_ID: 0x0052U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void ospi_write_zero_latency_disable(uint32_t ospi_periph)
{
    OSPI_HBLCFG(ospi_periph) &= (uint32_t)(~OSPI_HBLCFG_WZLAT);
}

/*!
    \brief      configure OSPI regular command parameter (API_ID: 0x0053U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  ospi_struct: OSPI parameter initialization stuct members of the structure
                             and the member values are shown as below:
      \arg        prescaler: between 0 and 255
      \arg        fifo_threshold: OSPI_FIFO_THRESHOLD_x (x = 1, 2, ..., 31, 32)
      \arg        sample_shift: OSPI_SAMPLE_SHIFTING_NONE, OSPI_SAMPLE_SHIFTING_HALF_CYCLE
      \arg        device_size: OSPI_MESZ_x_BYTES (x = 2, 4, 8, ..., 512, 1024)
                               OSPI_MESZ_x_KBS (x = 2, 4, 8, ..., 512, 1024)
                               OSPI_MESZ_x_MBS (x = 2, 4, 8, ..., 2048, 4096)
      \arg        cs_hightime: OSPI_CS_HIGH_TIME_1_CYCLE, OSPI_CS_HIGH_TIME_2_CYCLE
                               OSPI_CS_HIGH_TIME_3_CYCLE, OSPI_CS_HIGH_TIME_4_CYCLE
                               OSPI_CS_HIGH_TIME_5_CYCLE, OSPI_CS_HIGH_TIME_6_CYCLE
                               OSPI_CS_HIGH_TIME_7_CYCLE, OSPI_CS_HIGH_TIME_8_CYCLE
      \arg        clock_mode: OSPI_CLOCK_MODE_LOW, OSPI_CLOCK_MODE_HIGH
      \arg        dual_quad: OSPI_DUAL_QUAD_MODE_DISABLE, OSPI_DUAL_QUAD_MODE_ENABLE
      \arg        memory_type: OSPI_MICRON_MODE, OSPI_MACRONIX_MODE, OSPI_STANDARD_MODE
                                      OSPI_MACRONIX_RAM_MODE, OSPI_HYPERBUS_MEMORY_MODE, OSPI_HYPERBUS_REGISTER_MODE
      \arg        free_clock_run: OSPI_CLOCK_FREE_RUN_DISABLE, OSPI_CLOCK_FREE_RUN_ENABLE
      \arg        delay_hold_cycle: OSPI_DELAY_HOLD_NONE, OSPI_DELAY_HOLD_QUARTER_CYCLE
      \arg        clk_cs_hightime: between 0 and 7
      \arg        delay_block_bypass: OSPI_DELAY_CLOCK_BYPASS_DISABLE, OSPI_DELAY_CLOCK_BYPASS_ENABLE
      \arg        refrate: between 0 and 0xFFFFFFFF
    \param[in]  cmd_struct: structure that contains the command configuration information
                            and the member values are shown as below:
      \arg        operation_type: OSPI_OPTYPE_COMMON_CFG, OSPI_OPTYPE_READ_CFG
                                  OSPI_OPTYPE_WRITE_CFG, OSPI_OPTYPE_WRAP_CFG
      \arg        device_id: OSPI_DEVICE0_SELECT, OSPI_DEVICE1_SELECT
      \arg        instruction: between 0 and 0xFFFFFFFF
      \arg        ins_mode: OSPI_INSTRUCTION_NONE, OSPI_INSTRUCTION_1_LINE, OSPI_INSTRUCTION_2_LINES
                            OSPI_INSTRUCTION_4_LINES, OSPI_INSTRUCTION_8_LINES
      \arg        ins_size: OSPI_INSTRUCTION_8_BITS, OSPI_INSTRUCTION_16_BITS
                            OSPI_INSTRUCTION_24_BITS, OSPI_INSTRUCTION_32_BITS
      \arg        ins_dtr_mode: OSPI_INSDTR_MODE_DISABLE, OSPI_INSDTR_MODE_ENABLE
      \arg        address: between 0 and 0xFFFFFFFF
      \arg        addr_mode: OSPI_ADDRESS_NONE, OSPI_ADDRESS_1_LINE, OSPI_ADDRESS_2_LINES
                             OSPI_ADDRESS_4_LINES, OSPI_ADDRESS_8_LINES
      \arg        addr_size: OSPI_ADDRESS_8_BITS, OSPI_ADDRESS_16_BITS
                             OSPI_ADDRESS_24_BITS, OSPI_ADDRESS_32_BITS
      \arg        addr_dtr_mode: OSPI_ADDRDTR_MODE_DISABLE, OSPI_ADDTR_MODE_ENABLE
      \arg        alter_bytes: between 0 and 0xFFFFFFFF
      \arg        alter_bytes_mode: OSPI_ALTERNATE_BYTES_NONE, OSPI_ALTERNATE_BYTES_1_LINE
                                    OSPI_ALTERNATE_BYTES_2_LINES, OSPI_ALTERNATE_BYTES_4_LINES
                                    OSPI_ALTERNATE_BYTES_8_LINES
      \arg        alter_bytes_size: OSPI_ALTERNATE_BYTES_8_BITS, OSPI_ALTERNATE_BYTES_16_BITS
                                    OSPI_ALTERNATE_BYTES_24_BITS, OSPI_ALTERNATE_BYTES_32_BITS
      \arg        alter_bytes_dtr_mode: OSPI_ABDTR_MODE_DISABLE, OSPI_ABDTR_MODE_ENABLE
      \arg        data_mode: OSPI_DATA_NONE, OSPI_DATA_1_LINE, OSPI_DATA_2_LINES
                             OSPI_DATA_4_LINES, OSPI_DATA_8_LINES
      \arg        nbdata: between 1 and 0xFFFFFFFF
      \arg        data_dtr_mode: OSPI_DADTR_MODE_DISABLE, OSPI_DADTR_MODE_ENABLE
      \arg        dummy_cycles: OSPI_DUMYC_CYCLES_x (x = 0, 1, 2, ..., 30, 31)
      \arg        dqs_mode: OSPI_DQSEN_DISABLE, OSPI_DQSEN_ENABLE
      \arg        sioo_mode: OSPI_SIOO_INST_EVERY_CMD, OSPI_SIOO_INST_ONLY_FIRST_CMD
    \param[out] none
    \retval     none
*/
void ospi_command_config(uint32_t ospi_periph, ospi_parameter_struct *ospi_struct, ospi_regular_cmd_struct *cmd_struct)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_VALID_POINTER(ospi_struct)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x0053U), ERR_PARAM_POINTER);
    } else if(NOT_VALID_POINTER(cmd_struct)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x0053U), ERR_PARAM_POINTER);
    } else
#endif /* FW_DEBUG_ERR_REPORT */
    {
        if((ospi_struct->memory_type != OSPI_HYPERBUS_MEMORY_MODE) || ((cmd_struct->operation_type == OSPI_OPTYPE_WRITE_CFG) ||
                (cmd_struct->operation_type == OSPI_OPTYPE_WRAP_CFG)) || ((cmd_struct->operation_type == OSPI_OPTYPE_READ_CFG)  ||
                        (cmd_struct->operation_type == OSPI_OPTYPE_COMMON_CFG))) {
            /* wait till busy flag is reset */
            while(RESET != (OSPI_STAT(ospi_periph) & OSPI_FLAG_BUSY)) {
            }

            /* configure the registers */
            ospi_config(ospi_periph, ospi_struct, cmd_struct);

            if(cmd_struct->data_mode == OSPI_DATA_NONE) {
                /* when there is no data phase, the transfer start as soon as the configuration is done
                so wait until TC flag is set to go back in idle state */
                while(RESET == (OSPI_STAT(ospi_periph) & OSPI_FLAG_TC)) {
                }

                OSPI_STATC(ospi_periph) = OSPI_STATC_TCC;
            }
        }
    }
}

/*!
    \brief      configure OSPI regular command in interrupt mode parameter (API_ID: 0x0054U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  ospi_struct: OSPI parameter initialization stuct members of the structure
                             and the member values are shown as below:
      \arg        prescaler: between 0 and 255
      \arg        fifo_threshold: OSPI_FIFO_THRESHOLD_x (x = 1, 2, ..., 31, 32)
      \arg        sample_shift: OSPI_SAMPLE_SHIFTING_NONE, OSPI_SAMPLE_SHIFTING_HALF_CYCLE
      \arg        device_size: OSPI_MESZ_x_BYTES (x = 2, 4, 8, ..., 512, 1024)
                               OSPI_MESZ_x_KBS (x = 2, 4, 8, ..., 512, 1024)
                               OSPI_MESZ_x_MBS (x = 2, 4, 8, ..., 2048, 4096)
      \arg        cs_hightime: OSPI_CS_HIGH_TIME_1_CYCLE, OSPI_CS_HIGH_TIME_2_CYCLE
                               OSPI_CS_HIGH_TIME_3_CYCLE, OSPI_CS_HIGH_TIME_4_CYCLE
                               OSPI_CS_HIGH_TIME_5_CYCLE, OSPI_CS_HIGH_TIME_6_CYCLE
                               OSPI_CS_HIGH_TIME_7_CYCLE, OSPI_CS_HIGH_TIME_8_CYCLE
      \arg        clock_mode: OSPI_CLOCK_MODE_LOW, OSPI_CLOCK_MODE_HIGH
      \arg        dual_quad: OSPI_DUAL_QUAD_MODE_DISABLE, OSPI_DUAL_QUAD_MODE_ENABLE
      \arg        memory_type: OSPI_MICRON_MODE, OSPI_MACRONIX_MODE, OSPI_STANDARD_MODE
                                      OSPI_MACRONIX_RAM_MODE, OSPI_HYPERBUS_MEMORY_MODE, OSPI_HYPERBUS_REGISTER_MODE
      \arg        free_clock_run: OSPI_CLOCK_FREE_RUN_DISABLE, OSPI_CLOCK_FREE_RUN_ENABLE
      \arg        delay_hold_cycle: OSPI_DELAY_HOLD_NONE, OSPI_DELAY_HOLD_QUARTER_CYCLE
      \arg        clk_cs_hightime: between 0 and 7
      \arg        delay_block_bypass: OSPI_DELAY_CLOCK_BYPASS_DISABLE, OSPI_DELAY_CLOCK_BYPASS_ENABLE
      \arg        refrate: between 0 and 0xFFFFFFFF
    \param[in]  cmd_struct: structure that contains the command configuration information
                            and the member values are shown as below:
      \arg        operation_type: OSPI_OPTYPE_COMMON_CFG, OSPI_OPTYPE_READ_CFG
                                  OSPI_OPTYPE_WRITE_CFG, OSPI_OPTYPE_WRAP_CFG
      \arg        device_id: OSPI_DEVICE0_SELECT, OSPI_DEVICE1_SELECT
      \arg        instruction: between 0 and 0xFFFFFFFF
      \arg        ins_mode: OSPI_INSTRUCTION_NONE, OSPI_INSTRUCTION_1_LINE, OSPI_INSTRUCTION_2_LINES
                            OSPI_INSTRUCTION_4_LINES, OSPI_INSTRUCTION_8_LINES
      \arg        ins_size: OSPI_INSTRUCTION_8_BITS, OSPI_INSTRUCTION_16_BITS
                            OSPI_INSTRUCTION_24_BITS, OSPI_INSTRUCTION_32_BITS
      \arg        ins_dtr_mode: OSPI_INSDTR_MODE_DISABLE, OSPI_INSDTR_MODE_ENABLE
      \arg        address: between 0 and 0xFFFFFFFF
      \arg        addr_mode: OSPI_ADDRESS_NONE, OSPI_ADDRESS_1_LINE, OSPI_ADDRESS_2_LINES
                             OSPI_ADDRESS_4_LINES, OSPI_ADDRESS_8_LINES
      \arg        addr_size: OSPI_ADDRESS_8_BITS, OSPI_ADDRESS_16_BITS
                             OSPI_ADDRESS_24_BITS, OSPI_ADDRESS_32_BITS
      \arg        addr_dtr_mode: OSPI_ADDRDTR_MODE_DISABLE, OSPI_ADDTR_MODE_ENABLE
      \arg        alter_bytes: between 0 and 0xFFFFFFFF
      \arg        alter_bytes_mode: OSPI_ALTERNATE_BYTES_NONE, OSPI_ALTERNATE_BYTES_1_LINE
                                    OSPI_ALTERNATE_BYTES_2_LINES, OSPI_ALTERNATE_BYTES_4_LINES
                                    OSPI_ALTERNATE_BYTES_8_LINES
      \arg        alter_bytes_size: OSPI_ALTERNATE_BYTES_8_BITS, OSPI_ALTERNATE_BYTES_16_BITS
                                    OSPI_ALTERNATE_BYTES_24_BITS, OSPI_ALTERNATE_BYTES_32_BITS
      \arg        alter_bytes_dtr_mode: OSPI_ABDTR_MODE_DISABLE, OSPI_ABDTR_MODE_ENABLE
      \arg        data_mode: OSPI_DATA_NONE, OSPI_DATA_1_LINE, OSPI_DATA_2_LINES
                             OSPI_DATA_4_LINES, OSPI_DATA_8_LINES
      \arg        nbdata: between 1 and 0xFFFFFFFF
      \arg        data_dtr_mode: OSPI_DADTR_MODE_DISABLE, OSPI_DADTR_MODE_ENABLE
      \arg        dummy_cycles: OSPI_DUMYC_CYCLES_x (x = 0, 1, 2, ..., 30, 31)
      \arg        dqs_mode: OSPI_DQSEN_DISABLE, OSPI_DQSEN_ENABLE
      \arg        sioo_mode: OSPI_SIOO_INST_EVERY_CMD, OSPI_SIOO_INST_ONLY_FIRST_CMD
    \param[out] none
    \retval     none
*/
void ospi_command_interrupt_config(uint32_t ospi_periph, ospi_parameter_struct *ospi_struct, ospi_regular_cmd_struct *cmd_struct)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_VALID_POINTER(ospi_struct)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x0054U), ERR_PARAM_POINTER);
    } else if(NOT_VALID_POINTER(cmd_struct)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x0054U), ERR_PARAM_POINTER);
    } else
#endif /* FW_DEBUG_ERR_REPORT */
    {
        /* check the state of the driver */
        if((cmd_struct->operation_type == OSPI_OPTYPE_COMMON_CFG) && (cmd_struct->data_mode == OSPI_DATA_NONE) &&
                (ospi_struct->memory_type != OSPI_HYPERBUS_MEMORY_MODE)) {
            /* wait till busy flag is reset */
            while(RESET != (OSPI_STAT(ospi_periph) & OSPI_FLAG_BUSY));

            /* clear flags related to interrupt */
            OSPI_STATC(ospi_periph) = OSPI_STATC_TCC | OSPI_STATC_TERRC;

            /* configure the registers */
            ospi_config(ospi_periph, ospi_struct, cmd_struct);

            /* enable the transfer complete and transfer error interrupts */
            OSPI_CTL(ospi_periph) |= OSPI_CTL_TERRIE | OSPI_CTL_TCIE;
        }
    }
}

/*!
    \brief      configure the hyperbus parameters (API_ID: 0x0055U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  hyperbus_cfg_struct: structure containing the hyperbus configuration
                                     and the member values are shown as below:
      \arg        rw_recovery_time: between 0 and 255
      \arg        access_time: between 0 and 255
      \arg        write_zero_latency: OSPI_WRITE_ZERO_LATENCY_DISABLE, OSPI_WRITE_ZERO_LATENCY_ENABLE
      \arg        latency_mode: OSPI_VAR_INIT_LATENCY, OSPI_FIXED_LATENCY
    \param[out] none
    \retval     none
*/
void ospi_hyperbus_config(uint32_t ospi_periph, ospi_hyperbus_cfg_struct *hyperbus_cfg_struct)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_VALID_POINTER(hyperbus_cfg_struct)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x0055U), ERR_PARAM_POINTER);
    } else
#endif /* FW_DEBUG_ERR_REPORT */
    {
        /* wait till busy flag is reset */
        while(RESET != (OSPI_STAT(ospi_periph) & OSPI_FLAG_BUSY));

        /* configure hyperbus configuration latency register */
        OSPI_HBLCFG(ospi_periph) = OSPI_RWRTM(hyperbus_cfg_struct->rw_recovery_time) | OSPI_ACCTM(hyperbus_cfg_struct->access_time) |
                                   hyperbus_cfg_struct->write_zero_latency | hyperbus_cfg_struct->latency_mode;
    }
}

/*!
    \brief      set the hyperbus command configuration (API_ID: 0x0056U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  ospi_struct: OSPI parameter initialization stuct members of the structure
                             and the member values are shown as below:
      \arg        prescaler: between 0 and 255
      \arg        fifo_threshold: OSPI_FIFO_THRESHOLD_x (x = 1, 2, ..., 31, 32)
      \arg        sample_shift: OSPI_SAMPLE_SHIFTING_NONE, OSPI_SAMPLE_SHIFTING_HALF_CYCLE
      \arg        device_size: OSPI_MESZ_x_BYTES (x = 2, 4, 8, ..., 512, 1024)
                               OSPI_MESZ_x_KBS (x = 2, 4, 8, ..., 512, 1024)
                               OSPI_MESZ_x_MBS (x = 2, 4, 8, ..., 2048, 4096)
      \arg        cs_hightime: OSPI_CS_HIGH_TIME_1_CYCLE, OSPI_CS_HIGH_TIME_2_CYCLE
                               OSPI_CS_HIGH_TIME_3_CYCLE, OSPI_CS_HIGH_TIME_4_CYCLE
                               OSPI_CS_HIGH_TIME_5_CYCLE, OSPI_CS_HIGH_TIME_6_CYCLE
                               OSPI_CS_HIGH_TIME_7_CYCLE, OSPI_CS_HIGH_TIME_8_CYCLE
      \arg        clock_mode: OSPI_CLOCK_MODE_LOW, OSPI_CLOCK_MODE_HIGH
      \arg        dual_quad: OSPI_DUAL_QUAD_MODE_DISABLE, OSPI_DUAL_QUAD_MODE_ENABLE
      \arg        memory_type: OSPI_MICRON_MODE, OSPI_MACRONIX_MODE, OSPI_STANDARD_MODE
                                      OSPI_MACRONIX_RAM_MODE, OSPI_HYPERBUS_MEMORY_MODE, OSPI_HYPERBUS_REGISTER_MODE
      \arg        free_clock_run: OSPI_CLOCK_FREE_RUN_DISABLE, OSPI_CLOCK_FREE_RUN_ENABLE
      \arg        delay_hold_cycle: OSPI_DELAY_HOLD_NONE, OSPI_DELAY_HOLD_QUARTER_CYCLE
      \arg        clk_cs_hightime: between 0 and 7
      \arg        delay_block_bypass: OSPI_DELAY_CLOCK_BYPASS_DISABLE, OSPI_DELAY_CLOCK_BYPASS_ENABLE
      \arg        refrate: between 0 and 0xFFFFFFFF
    \param[in]  hyperbus_cmd_struct: structure containing the hyperbus command
                                     and the member values are shown as below:
      \arg        address_space: OSPI_MEMORY_ADDRESS_SPACE, OSPI_REGISTER_ADDRESS_SPACE
      \arg        address: between 0 and 0xFFFFFFFF
      \arg        address_size: OSPI_ADDRESS_8_BITS, OSPI_ADDRESS_16_BITS
      \arg                      OSPI_ADDRESS_24_BITS, OSPI_ADDRESS_32_BITS
      \arg        nbdata: between between 1 and 0xFFFFFFFF
      \arg        dqs_mode: OSPI_DQSEN_DISABLE, OSPI_DQSEN_ENABLE
    \param[out] none
    \retval     none
*/
void ospi_hyperbus_cmd_config(uint32_t ospi_periph, ospi_parameter_struct *ospi_struct, ospi_hyperbus_cmd_struct *hyperbus_cmd_struct)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_VALID_POINTER(ospi_struct)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x0056U), ERR_PARAM_POINTER);
    } else if(NOT_VALID_POINTER(hyperbus_cmd_struct)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x0056U), ERR_PARAM_POINTER);
    } else
#endif /* FW_DEBUG_ERR_REPORT */
    {
        /* Check the state of the driver */
        if(ospi_struct->memory_type == OSPI_HYPERBUS_MEMORY_MODE) {
            /* wait till busy flag is reset */
            while(RESET != (OSPI_STAT(ospi_periph) & OSPI_FLAG_BUSY));

            /* re-initialize the value of the functional mode */
            OSPI_CTL(ospi_periph) = (OSPI_CTL(ospi_periph) & ~OSPI_CTL_FMOD);

            /* Configure the address space in the DCFG0 register */
            OSPI_DCFG0(ospi_periph) = (OSPI_DCFG0(ospi_periph) & ~OSPI_RESERVE_MODE) | hyperbus_cmd_struct->addr_space;

            /* configure the TCFG and WTCFG registers with the address size and the following configuration :
             - DQS signal enabled (used as RWDS)
             - DTR mode enabled on address and data
             - address and data on 8 lines */
            OSPI_TCFG(ospi_periph) = hyperbus_cmd_struct->dqs_mode | OSPI_DADTR_MODE_ENABLE | OSPI_DATA_8_LINES |
                                     hyperbus_cmd_struct->addr_size | OSPI_ADDTR_MODE_ENABLE | OSPI_ADDRESS_8_LINES;

            OSPI_WTCFG(ospi_periph) = hyperbus_cmd_struct->dqs_mode | OSPI_DADTR_MODE_ENABLE | OSPI_DATA_8_LINES |
                                      hyperbus_cmd_struct->addr_size | OSPI_ADDTR_MODE_ENABLE | OSPI_ADDRESS_8_LINES;

            /* configure the DTLEN register with the number of data */
            OSPI_DTLEN(ospi_periph) = hyperbus_cmd_struct->nbdata - 1U;

            /* configure the ADDR register with the address value */
            OSPI_ADDR(ospi_periph) = hyperbus_cmd_struct->address;
        }
    }
}

/*!
    \brief      transmit an amount of data in blocking mode (API_ID: 0x0057U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  pdata: pointer to data buffer
    \param[out] none
    \note       this function is used only in indirect write mode
    \retval     none
*/
void ospi_transmit(uint32_t ospi_periph, uint8_t *pdata)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_VALID_POINTER(pdata)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x0057U), ERR_PARAM_POINTER);
    } else
#endif /* FW_DEBUG_ERR_REPORT */
    {
        uint32_t txcounter;
        uint8_t *txdata;
        __IO uint32_t *data_reg = &OSPI_DATA(ospi_periph);

        /* configure counters and size */
        txcounter = OSPI_DTLEN(ospi_periph) + 1U;
        txdata = pdata;

        /* configure CTL register with functional mode as indirect write */
        OSPI_CTL(ospi_periph) = (OSPI_CTL(ospi_periph) & ~OSPI_CTL_FMOD) | OSPI_INDIRECT_WRITE;

        do {
            /* wait till fifo threshold flag is set to send data */
            while(RESET == (OSPI_STAT(ospi_periph) & OSPI_FLAG_FT));

            *((__IO uint8_t *)data_reg) = *txdata;
            txdata++;
            txcounter--;
        } while(txcounter > 0U);

        /* wait till transfer complete flag is set to go back in idle state */
        while(RESET == (OSPI_STAT(ospi_periph) & OSPI_FLAG_TC)) {
        }

        /* clear transfer complete flag */
        OSPI_STATC(ospi_periph) = OSPI_STATC_TCC;
    }
}

/*!
    \brief      receive an amount of data in blocking mode (API_ID: 0x005AU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  ospi_struct: OSPI parameter initialization stuct members of the structure
                             and the member values are shown as below:
      \arg        prescaler: between 0 and 255
      \arg        fifo_threshold: OSPI_FIFO_THRESHOLD_x (x = 1, 2, ..., 31, 32)
      \arg        sample_shift: OSPI_SAMPLE_SHIFTING_NONE, OSPI_SAMPLE_SHIFTING_HALF_CYCLE
      \arg        device_size: OSPI_MESZ_x_BYTES (x = 2, 4, 8, ..., 512, 1024)
                               OSPI_MESZ_x_KBS (x = 2, 4, 8, ..., 512, 1024)
                               OSPI_MESZ_x_MBS (x = 2, 4, 8, ..., 2048, 4096)
      \arg        cs_hightime: OSPI_CS_HIGH_TIME_1_CYCLE, OSPI_CS_HIGH_TIME_2_CYCLE
                               OSPI_CS_HIGH_TIME_3_CYCLE, OSPI_CS_HIGH_TIME_4_CYCLE
                               OSPI_CS_HIGH_TIME_5_CYCLE, OSPI_CS_HIGH_TIME_6_CYCLE
                               OSPI_CS_HIGH_TIME_7_CYCLE, OSPI_CS_HIGH_TIME_8_CYCLE
      \arg        clock_mode: OSPI_CLOCK_MODE_LOW, OSPI_CLOCK_MODE_HIGH
      \arg        dual_quad: OSPI_DUAL_QUAD_MODE_DISABLE, OSPI_DUAL_QUAD_MODE_ENABLE
      \arg        memory_type: OSPI_MICRON_MODE, OSPI_MACRONIX_MODE, OSPI_STANDARD_MODE
                                      OSPI_MACRONIX_RAM_MODE, OSPI_HYPERBUS_MEMORY_MODE, OSPI_HYPERBUS_REGISTER_MODE
      \arg        free_clock_run: OSPI_CLOCK_FREE_RUN_DISABLE, OSPI_CLOCK_FREE_RUN_ENABLE
      \arg        delay_hold_cycle: OSPI_DELAY_HOLD_NONE, OSPI_DELAY_HOLD_QUARTER_CYCLE
      \arg        clk_cs_hightime: between 0 and 7
      \arg        delay_block_bypass: OSPI_DELAY_CLOCK_BYPASS_DISABLE, OSPI_DELAY_CLOCK_BYPASS_ENABLE
      \arg        refrate: between 0 and 0xFFFFFFFF
    \param[in]  pdata: pointer to data buffer
    \param[out] none
    \note       this function is used only in indirect read mode
    \retval     none
*/
void ospi_receive(uint32_t ospi_periph, ospi_parameter_struct *ospi_struct, uint8_t *pdata)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_VALID_POINTER(ospi_struct)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x005AU), ERR_PARAM_POINTER);
    } else if(NOT_VALID_POINTER(pdata)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x005AU), ERR_PARAM_POINTER);
    } else
#endif /* FW_DEBUG_ERR_REPORT */
    {
        uint32_t rxcounter;
        uint8_t *rxdata;
        __IO uint32_t *data_reg = &OSPI_DATA(ospi_periph);
        uint32_t addr_reg = OSPI_ADDR(ospi_periph);
        uint32_t ins_reg = OSPI_INS(ospi_periph);

        /* configure counters and size */
        rxcounter = OSPI_DTLEN(ospi_periph) + 1U;
        rxdata  = pdata;

        /* configure CTL register with functional mode as indirect read */
        OSPI_CTL(ospi_periph) = (OSPI_CTL(ospi_periph) & ~OSPI_CTL_FMOD) | OSPI_INDIRECT_READ;

        /* trig the transfer by re-writing address or instruction register */
        if(OSPI_HYPERBUS_MEMORY_MODE == ospi_struct->memory_type) {
            OSPI_ADDR(ospi_periph) = addr_reg;
        } else {
            if((OSPI_TCFG(ospi_periph) & OSPI_TCFG_ADDRMOD) != OSPI_ADDRESS_NONE) {
                OSPI_ADDR(ospi_periph) = addr_reg;
            } else {
                OSPI_INS(ospi_periph) = ins_reg;
            }
        }

        do {
            /* wait till fifo threshold or transfer complete flags are set to read received data */
            while(RESET == (OSPI_STAT(ospi_periph) & (OSPI_FLAG_FT | OSPI_FLAG_TC))) {
            }

            *rxdata = *((__IO uint8_t *)data_reg);
            rxdata++;
            rxcounter--;

        } while(rxcounter > 0U);

        /* wait till transfer complete flag is set to go back in idle state */
        while(RESET == (OSPI_STAT(ospi_periph) & OSPI_FLAG_TC)) {
        }

        /* clear transfer complete flag */
        OSPI_STATC(ospi_periph) = OSPI_STATC_TCC;
    }
}

/*!
    \brief      configure the OSPI automatic polling mode in blocking mode (API_ID: 0x005DU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  ospi_struct: OSPI parameter initialization stuct members of the structure
                             and the member values are shown as below:
      \arg        prescaler: between 0 and 255
      \arg        fifo_threshold: OSPI_FIFO_THRESHOLD_x (x = 1, 2, ..., 31, 32)
      \arg        sample_shift: OSPI_SAMPLE_SHIFTING_NONE, OSPI_SAMPLE_SHIFTING_HALF_CYCLE
      \arg        device_size: OSPI_MESZ_x_BYTES (x = 2, 4, 8, ..., 512, 1024)
                               OSPI_MESZ_x_KBS (x = 2, 4, 8, ..., 512, 1024)
                               OSPI_MESZ_x_MBS (x = 2, 4, 8, ..., 2048, 4096)
      \arg        cs_hightime: OSPI_CS_HIGH_TIME_1_CYCLE, OSPI_CS_HIGH_TIME_2_CYCLE
                               OSPI_CS_HIGH_TIME_3_CYCLE, OSPI_CS_HIGH_TIME_4_CYCLE
                               OSPI_CS_HIGH_TIME_5_CYCLE, OSPI_CS_HIGH_TIME_6_CYCLE
                               OSPI_CS_HIGH_TIME_7_CYCLE, OSPI_CS_HIGH_TIME_8_CYCLE
      \arg        clock_mode: OSPI_CLOCK_MODE_LOW, OSPI_CLOCK_MODE_HIGH
      \arg        dual_quad: OSPI_DUAL_QUAD_MODE_DISABLE, OSPI_DUAL_QUAD_MODE_ENABLE
      \arg        memory_type: OSPI_MICRON_MODE, OSPI_MACRONIX_MODE, OSPI_STANDARD_MODE
                                      OSPI_MACRONIX_RAM_MODE, OSPI_HYPERBUS_MEMORY_MODE, OSPI_HYPERBUS_REGISTER_MODE
      \arg        free_clock_run: OSPI_CLOCK_FREE_RUN_DISABLE, OSPI_CLOCK_FREE_RUN_ENABLE
      \arg        delay_hold_cycle: OSPI_DELAY_HOLD_NONE, OSPI_DELAY_HOLD_QUARTER_CYCLE
      \arg        clk_cs_hightime: between 0 and 7
      \arg        delay_block_bypass: OSPI_DELAY_CLOCK_BYPASS_DISABLE, OSPI_DELAY_CLOCK_BYPASS_ENABLE
      \arg        refrate: between 0 and 0xFFFFFFFF
    \param[in]  autopl_cfg_struct: structure that contains the polling configuration information
                                   and the member values are shown as below:
      \arg        match: between 0 and 0xFFFFFFFF
      \arg        mask: between 0 and 0xFFFFFFFF
      \arg        interval: between 0 and 0xFFFF
      \arg        match_mode: OSPI_MATCH_MODE_AND, OSPI_MATCH_MODE_OR
      \arg        automatic_stop: OSPI_AUTOMATIC_STOP_ABORT, OSPI_AUTOMATIC_STOP_MATCH
    \param[out] none
    \note       this function is used only in automatic polling mode
    \retval     none
*/
void ospi_autopolling_mode(uint32_t ospi_periph, ospi_parameter_struct *ospi_struct, ospi_autopolling_struct *autopl_cfg_struct)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_VALID_POINTER(ospi_struct)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x005DU), ERR_PARAM_POINTER);
    } else if(NOT_VALID_POINTER(autopl_cfg_struct)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x005DU), ERR_PARAM_POINTER);
    } else
#endif /* FW_DEBUG_ERR_REPORT */
    {
        uint32_t addr_reg = OSPI_ADDR(ospi_periph);
        uint32_t ins_reg = OSPI_INS(ospi_periph);

        if(autopl_cfg_struct->automatic_stop == OSPI_AUTOMATIC_STOP_MATCH) {
            /* wait till busy flag is reset */
            while(RESET != (OSPI_STAT(ospi_periph) & OSPI_FLAG_BUSY)) {
            }

            /* configure registers */
            OSPI_STATMATCH(ospi_periph) = autopl_cfg_struct->match;
            OSPI_STATMK(ospi_periph) = autopl_cfg_struct->mask;
            OSPI_INTERVAL(ospi_periph) = autopl_cfg_struct->interval;
            OSPI_CTL(ospi_periph) = (OSPI_CTL(ospi_periph) & (~OSPI_CTL_SPMOD | ~OSPI_CTL_SPS | ~OSPI_CTL_FMOD)) |
                                    (autopl_cfg_struct->match_mode | autopl_cfg_struct->automatic_stop | OSPI_STATUS_POLLING);

            /* trig the transfer by re-writing address or instruction register */
            if(ospi_struct->memory_type == OSPI_HYPERBUS_MEMORY_MODE) {
                OSPI_ADDR(ospi_periph) = addr_reg;
            } else {
                if((OSPI_TCFG(ospi_periph) & OSPI_TCFG_ADDRMOD) != OSPI_ADDRESS_NONE) {
                    OSPI_ADDR(ospi_periph) = addr_reg;
                } else {
                    OSPI_INS(ospi_periph) = ins_reg;
                }
            }

            /* wait till status match flag is set to go back in idle state */
            while(RESET == (OSPI_STAT(ospi_periph) & OSPI_FLAG_SM)) {
            }

            /* clear status match flag */
            OSPI_STATC(ospi_periph) = OSPI_STATC_SMC;
        }
    }
}

/*!
    \brief      configure the OSPI automatic polling mode in non-blocking mode (API_ID: 0x005EU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  ospi_struct: OSPI parameter initialization stuct members of the structure
                             and the member values are shown as below:
      \arg        prescaler: between 0 and 255
      \arg        fifo_threshold: OSPI_FIFO_THRESHOLD_x (x = 1, 2, ..., 31, 32)
      \arg        sample_shift: OSPI_SAMPLE_SHIFTING_NONE, OSPI_SAMPLE_SHIFTING_HALF_CYCLE
      \arg        device_size: OSPI_MESZ_x_BYTES (x = 2, 4, 8, ..., 512, 1024)
                               OSPI_MESZ_x_KBS (x = 2, 4, 8, ..., 512, 1024)
                               OSPI_MESZ_x_MBS (x = 2, 4, 8, ..., 2048, 4096)
      \arg        cs_hightime: OSPI_CS_HIGH_TIME_1_CYCLE, OSPI_CS_HIGH_TIME_2_CYCLE
                               OSPI_CS_HIGH_TIME_3_CYCLE, OSPI_CS_HIGH_TIME_4_CYCLE
                               OSPI_CS_HIGH_TIME_5_CYCLE, OSPI_CS_HIGH_TIME_6_CYCLE
                               OSPI_CS_HIGH_TIME_7_CYCLE, OSPI_CS_HIGH_TIME_8_CYCLE
      \arg        clock_mode: OSPI_CLOCK_MODE_LOW, OSPI_CLOCK_MODE_HIGH
      \arg        dual_quad: OSPI_DUAL_QUAD_MODE_DISABLE, OSPI_DUAL_QUAD_MODE_ENABLE
      \arg        memory_type: OSPI_MICRON_MODE, OSPI_MACRONIX_MODE, OSPI_STANDARD_MODE
                                      OSPI_MACRONIX_RAM_MODE, OSPI_HYPERBUS_MEMORY_MODE, OSPI_HYPERBUS_REGISTER_MODE
      \arg        free_clock_run: OSPI_CLOCK_FREE_RUN_DISABLE, OSPI_CLOCK_FREE_RUN_ENABLE
      \arg        delay_hold_cycle: OSPI_DELAY_HOLD_NONE, OSPI_DELAY_HOLD_QUARTER_CYCLE
      \arg        clk_cs_hightime: between 0 and 7
      \arg        delay_block_bypass: OSPI_DELAY_CLOCK_BYPASS_DISABLE, OSPI_DELAY_CLOCK_BYPASS_ENABLE
      \arg        refrate: between 0 and 0xFFFFFFFF
    \param[in]  autopl_cfg_struct: structure that contains the polling configuration information
                                   and the member values are shown as below:
      \arg        match: between 0 and 0xFFFFFFFF
      \arg        mask: between 0 and 0xFFFFFFFF
      \arg        interval: between 0 and 0xFFFF
      \arg        match_mode: OSPI_MATCH_MODE_AND, OSPI_MATCH_MODE_OR
      \arg        automatic_stop: OSPI_AUTOMATIC_STOP_ABORT, OSPI_AUTOMATIC_STOP_MATCH
    \param[out] none
    \note       this function is used only in automatic polling mode
    \retval     none
*/
void ospi_interrupt_autopolling_mode(uint32_t ospi_periph, ospi_parameter_struct *ospi_struct, ospi_autopolling_struct *autopl_cfg_struct)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_VALID_POINTER(ospi_struct)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x005EU), ERR_PARAM_POINTER);
    } else if(NOT_VALID_POINTER(autopl_cfg_struct)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x005EU), ERR_PARAM_POINTER);
    } else
#endif /* FW_DEBUG_ERR_REPORT */
    {
        uint32_t addr_reg = OSPI_ADDR(ospi_periph);
        uint32_t ins_reg = OSPI_INS(ospi_periph);

        /* wait till busy flag is reset */
        while(RESET != (OSPI_STAT(ospi_periph) & OSPI_FLAG_BUSY));

        /* configure registers */
        OSPI_STATMATCH(ospi_periph) = autopl_cfg_struct->match;
        OSPI_STATMK(ospi_periph) = autopl_cfg_struct->mask;
        OSPI_INTERVAL(ospi_periph) = autopl_cfg_struct->interval;
        OSPI_CTL(ospi_periph) = (OSPI_CTL(ospi_periph) & (~OSPI_CTL_SPMOD | ~OSPI_CTL_SPS | ~OSPI_CTL_FMOD)) |
                                (autopl_cfg_struct->match_mode | autopl_cfg_struct->automatic_stop | OSPI_STATUS_POLLING);

        /* clear flags related to interrupt */
        OSPI_STATC(ospi_periph) = OSPI_STATC_SMC | OSPI_STATC_TERRC;

        /* enable the status match and transfer error interrupts */
        OSPI_CTL(ospi_periph) |= OSPI_CTL_TERRIE | OSPI_CTL_SMIE;

        /* trig the transfer by re-writing address or instruction register */
        if(ospi_struct->memory_type == OSPI_HYPERBUS_MEMORY_MODE) {
            OSPI_ADDR(ospi_periph) = addr_reg;
        } else {
            if((OSPI_TCFG(ospi_periph) & OSPI_WTCFG_ADDRMOD) != OSPI_ADDRESS_NONE) {
                OSPI_ADDR(ospi_periph) = addr_reg;
            } else {
                OSPI_INS(ospi_periph) = ins_reg;
            }
        }
    }
}

/*!
    \brief      configure the OSPI memory mapped mode (API_ID: 0x005FU)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  memapped_cfg_struct: structure that contains the memory mapped configuration information
                                     and the member values are shown as below:
      \arg        timeout_activation: OSPI_TIMEOUT_COUNT_DISABLE, OSPI_TIMEOUT_COUNT_ENABLE
      \arg        timeout_period: between 0 and 0xFFFF
    \param[out] none
    \note       this function is used only in memory mapped mode
    \retval     none
*/
void ospi_memorymapped_mode(uint32_t ospi_periph, ospi_memorymapped_struct *memapped_cfg_struct)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_VALID_POINTER(memapped_cfg_struct)) {
        fw_debug_report_err(OSPI_MODULE_ID, API_ID(0x005FU), ERR_PARAM_POINTER);
    } else
#endif /* FW_DEBUG_ERR_REPORT */
    {
        /* wait till busy flag is reset */
        while(RESET != (OSPI_STAT(ospi_periph) & OSPI_FLAG_BUSY));

        if(memapped_cfg_struct->timeout_activation == OSPI_TIMEOUT_COUNT_ENABLE) {
            /* configure register */
            OSPI_LPOUT(ospi_periph) = memapped_cfg_struct->timeout_period;

            /* clear flags related to interrupt */
            OSPI_STATC(ospi_periph) = OSPI_STATC_TMOUTC;

            /* enable the timeout interrupt */
            OSPI_CTL(ospi_periph) = OSPI_CTL_TMOUTIE;
        }

        /* configure CTL register with functional mode as memory-mapped */
        OSPI_CTL(ospi_periph) = (OSPI_CTL(ospi_periph) & (~OSPI_CTL_TMOUTEN | ~OSPI_CTL_FMOD)) |
                                (memapped_cfg_struct->timeout_activation | OSPI_MEMORY_MAPPED);
    }
}

/*!
    \brief      configure the registers for the regular command mode (API_ID: 0x0060U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  ospi_struct: OSPI parameter initialization stuct members of the structure
                             and the member values are shown as below:
      \arg        prescaler: between 0 and 255
      \arg        fifo_threshold: OSPI_FIFO_THRESHOLD_x (x = 1, 2, ..., 31, 32)
      \arg        sample_shift: OSPI_SAMPLE_SHIFTING_NONE, OSPI_SAMPLE_SHIFTING_HALF_CYCLE
      \arg        device_size: OSPI_MESZ_x_BYTES (x = 2, 4, 8, ..., 512, 1024)
                               OSPI_MESZ_x_KBS (x = 2, 4, 8, ..., 512, 1024)
                               OSPI_MESZ_x_MBS (x = 2, 4, 8, ..., 2048, 4096)
      \arg        cs_hightime: OSPI_CS_HIGH_TIME_1_CYCLE, OSPI_CS_HIGH_TIME_2_CYCLE
                               OSPI_CS_HIGH_TIME_3_CYCLE, OSPI_CS_HIGH_TIME_4_CYCLE
                               OSPI_CS_HIGH_TIME_5_CYCLE, OSPI_CS_HIGH_TIME_6_CYCLE
                               OSPI_CS_HIGH_TIME_7_CYCLE, OSPI_CS_HIGH_TIME_8_CYCLE
      \arg        clock_mode: OSPI_CLOCK_MODE_LOW, OSPI_CLOCK_MODE_HIGH
      \arg        dual_quad: OSPI_DUAL_QUAD_MODE_DISABLE, OSPI_DUAL_QUAD_MODE_ENABLE
      \arg        memory_type: OSPI_MICRON_MODE, OSPI_MACRONIX_MODE, OSPI_STANDARD_MODE
                                      OSPI_MACRONIX_RAM_MODE, OSPI_HYPERBUS_MEMORY_MODE, OSPI_HYPERBUS_REGISTER_MODE
      \arg        free_clock_run: OSPI_CLOCK_FREE_RUN_DISABLE, OSPI_CLOCK_FREE_RUN_ENABLE
      \arg        delay_hold_cycle: OSPI_DELAY_HOLD_NONE, OSPI_DELAY_HOLD_QUARTER_CYCLE
      \arg        clk_cs_hightime: between 0 and 7
      \arg        delay_block_bypass: OSPI_DELAY_CLOCK_BYPASS_DISABLE, OSPI_DELAY_CLOCK_BYPASS_ENABLE
      \arg        refrate: between 0 and 0xFFFFFFFF
    \param[in]  cmd_struct: structure that contains the command configuration information
                            and the member values are shown as below:
      \arg        operation_type: OSPI_OPTYPE_COMMON_CFG, OSPI_OPTYPE_READ_CFG
                                  OSPI_OPTYPE_WRITE_CFG, OSPI_OPTYPE_WRAP_CFG
      \arg        device_id: OSPI_DEVICE0_SELECT, OSPI_DEVICE1_SELECT
      \arg        instruction: between 0 and 0xFFFFFFFF
      \arg        ins_mode: OSPI_INSTRUCTION_NONE, OSPI_INSTRUCTION_1_LINE, OSPI_INSTRUCTION_2_LINES
                            OSPI_INSTRUCTION_4_LINES, OSPI_INSTRUCTION_8_LINES
      \arg        ins_size: OSPI_INSTRUCTION_8_BITS, OSPI_INSTRUCTION_16_BITS
                            OSPI_INSTRUCTION_24_BITS, OSPI_INSTRUCTION_32_BITS
      \arg        ins_dtr_mode: OSPI_INSDTR_MODE_DISABLE, OSPI_INSDTR_MODE_ENABLE
      \arg        address: between 0 and 0xFFFFFFFF
      \arg        addr_mode: OSPI_ADDRESS_NONE, OSPI_ADDRESS_1_LINE, OSPI_ADDRESS_2_LINES
                             OSPI_ADDRESS_4_LINES, OSPI_ADDRESS_8_LINES
      \arg        addr_size: OSPI_ADDRESS_8_BITS, OSPI_ADDRESS_16_BITS
                             OSPI_ADDRESS_24_BITS, OSPI_ADDRESS_32_BITS
      \arg        addr_dtr_mode: OSPI_ADDRDTR_MODE_DISABLE, OSPI_ADDTR_MODE_ENABLE
      \arg        alter_bytes: between 0 and 0xFFFFFFFF
      \arg        alter_bytes_mode: OSPI_ALTERNATE_BYTES_NONE, OSPI_ALTERNATE_BYTES_1_LINE
                                    OSPI_ALTERNATE_BYTES_2_LINES, OSPI_ALTERNATE_BYTES_4_LINES
                                    OSPI_ALTERNATE_BYTES_8_LINES
      \arg        alter_bytes_size: OSPI_ALTERNATE_BYTES_8_BITS, OSPI_ALTERNATE_BYTES_16_BITS
                                    OSPI_ALTERNATE_BYTES_24_BITS, OSPI_ALTERNATE_BYTES_32_BITS
      \arg        alter_bytes_dtr_mode: OSPI_ABDTR_MODE_DISABLE, OSPI_ABDTR_MODE_ENABLE
      \arg        data_mode: OSPI_DATA_NONE, OSPI_DATA_1_LINE, OSPI_DATA_2_LINES
                             OSPI_DATA_4_LINES, OSPI_DATA_8_LINES
      \arg        nbdata: between 1 and 0xFFFFFFFF
      \arg        data_dtr_mode: OSPI_DADTR_MODE_DISABLE, OSPI_DADTR_MODE_ENABLE
      \arg        dummy_cycles: OSPI_DUMYC_CYCLES_x (x = 0, 1, 2, ..., 30, 31)
      \arg        dqs_mode: OSPI_DQSEN_DISABLE, OSPI_DQSEN_ENABLE
      \arg        sioo_mode: OSPI_SIOO_INST_EVERY_CMD, OSPI_SIOO_INST_ONLY_FIRST_CMD
    \param[out] none
    \retval     none
*/
static void ospi_config(uint32_t ospi_periph, ospi_parameter_struct *ospi_struct, ospi_regular_cmd_struct *cmd_struct)
{
    __IO uint32_t *tcfg_reg, *timcfg_reg, *ins_reg, *alte_reg;

    /* re-initialize the value of the functional mode */
    OSPI_CTL(ospi_periph) &= ~OSPI_CTL_FMOD;

    /* configure the flash ID */
    if(ospi_struct->dual_quad == OSPI_DUAL_QUAD_MODE_DISABLE) {
        OSPI_CTL(ospi_periph) = (OSPI_CTL(ospi_periph) & ~OSPI_DEVICE1_SELECT) | cmd_struct->device_id;
    }

    if(cmd_struct->operation_type == OSPI_OPTYPE_WRITE_CFG) {
        tcfg_reg = &(OSPI_WTCFG(ospi_periph));
        timcfg_reg = &(OSPI_WTIMCFG(ospi_periph));
        ins_reg  = &(OSPI_WINS(ospi_periph));
        alte_reg = &(OSPI_WALTE(ospi_periph));
    } else if(cmd_struct->operation_type == OSPI_OPTYPE_WRAP_CFG) {
        tcfg_reg = &(OSPI_WPTCFG(ospi_periph));
        timcfg_reg = &(OSPI_WPTIMCFG(ospi_periph));
        ins_reg  = &(OSPI_WPINS(ospi_periph));
        alte_reg = &(OSPI_WPALTE(ospi_periph));
    } else {
        tcfg_reg = &(OSPI_TCFG(ospi_periph));
        timcfg_reg = &(OSPI_TIMCFG(ospi_periph));
        ins_reg  = &(OSPI_INS(ospi_periph));
        alte_reg = &(OSPI_ALTE(ospi_periph));
    }

    if(cmd_struct->alter_bytes_mode != OSPI_ALTERNATE_BYTES_NONE) {
        /* configure the ALTE register with alternate bytes value */
        *alte_reg = cmd_struct->alter_bytes;

        /* configure the TCFG register with alternate bytes communication parameters */
        *tcfg_reg = (*tcfg_reg & ~(OSPI_TCFG_ALTEMOD | OSPI_TCFG_ABDTR | OSPI_TCFG_ALTESZ)) |
                    (cmd_struct->alter_bytes_mode | cmd_struct->alter_bytes_dtr_mode | cmd_struct->alter_bytes_size);
    } else {
        *tcfg_reg = (*tcfg_reg & ~(OSPI_TCFG_ALTEMOD | OSPI_TCFG_ABDTR | OSPI_TCFG_ALTESZ));
    }

    /* configure the TIMCFG register with the number of dummy cycles */
    *timcfg_reg = (*timcfg_reg & ~OSPI_TIMCFG_DUMYC) | cmd_struct->dummy_cycles;

    if(cmd_struct->data_mode != OSPI_DATA_NONE) {
        if(cmd_struct->operation_type == OSPI_OPTYPE_COMMON_CFG) {
            /* configure the DTLEN register with the number of data */
            OSPI_DTLEN(ospi_periph) = (cmd_struct->nbdata - 1U);
        }
    }

    if(cmd_struct->ins_mode != OSPI_INSTRUCTION_NONE) {
        if(cmd_struct->addr_mode != OSPI_ADDRESS_NONE) {
            if(cmd_struct->data_mode != OSPI_DATA_NONE) {
                /* command with instruction, address and data */
                /* configure the TCFG register with all communication parameters */
                *tcfg_reg &= ~(OSPI_TCFG_IMOD | OSPI_TCFG_INSDTR | OSPI_TCFG_INSSZ |
                               OSPI_TCFG_ADDRMOD | OSPI_TCFG_ADDRDTR | OSPI_TCFG_ADDRSZ |
                               OSPI_TCFG_DATAMOD | OSPI_TCFG_DADTR);

                *tcfg_reg |= cmd_struct->ins_mode | cmd_struct->ins_dtr_mode | cmd_struct->ins_size |
                             cmd_struct->addr_mode | cmd_struct->addr_dtr_mode | cmd_struct->addr_size |
                             cmd_struct->data_mode | cmd_struct->data_dtr_mode;
            } else {
                /* command with instruction and address */
                /* configure the TCFG register with all communication parameters */
                *tcfg_reg &= ~(OSPI_TCFG_IMOD | OSPI_TCFG_INSDTR | OSPI_TCFG_INSSZ |
                               OSPI_TCFG_ADDRMOD | OSPI_TCFG_ADDRDTR | OSPI_TCFG_ADDRSZ |
                               OSPI_TCFG_DATAMOD | OSPI_TCFG_DADTR);

                *tcfg_reg |= cmd_struct->ins_mode | cmd_struct->ins_dtr_mode | cmd_struct->ins_size |
                             cmd_struct->addr_mode | cmd_struct->addr_dtr_mode | cmd_struct->addr_size;

                /* the DHQC bit is linked with DDTR bit which should be activated */
                if((ospi_struct->delay_hold_cycle == OSPI_DELAY_HOLD_QUARTER_CYCLE) &&
                        (cmd_struct->ins_dtr_mode == OSPI_INSDTR_MODE_ENABLE)) {
                    *tcfg_reg = (*tcfg_reg & ~OSPI_DADTR_MODE_ENABLE) | OSPI_DADTR_MODE_ENABLE;
                }
            }
            /* configure the TCFG register with DQS and SIOO modes */
            *tcfg_reg &= ~(OSPI_TCFG_DQSEN | OSPI_TCFG_SIOO);
            *tcfg_reg |= (cmd_struct->dqs_mode | cmd_struct->sioo_mode);
            /* configure the INS register with the instruction value */
            *ins_reg = cmd_struct->instruction;

            /* configure the ADDR register with the address value */
            OSPI_ADDR(ospi_periph) = cmd_struct->address;
        } else {
            if(cmd_struct->data_mode != OSPI_DATA_NONE) {
                /* command with instruction and data */
                /* configure the TCFG register with all communication parameters */
                *tcfg_reg &= ~(OSPI_TCFG_IMOD | OSPI_TCFG_INSDTR | OSPI_TCFG_INSSZ |
                               OSPI_TCFG_DATAMOD | OSPI_TCFG_DADTR);

                *tcfg_reg = cmd_struct->ins_mode | cmd_struct->ins_dtr_mode | cmd_struct->ins_size |
                            cmd_struct->data_mode | cmd_struct->data_dtr_mode;
            } else {
                /* command with only instruction */
                /* configure the TCFG register with all communication parameters */
                *tcfg_reg &= ~(OSPI_TCFG_IMOD | OSPI_TCFG_INSDTR | OSPI_TCFG_INSSZ);

                *tcfg_reg = cmd_struct->ins_mode | cmd_struct->ins_dtr_mode | cmd_struct->ins_size;

                /* the DEHQC bit is linked with DDTR bit which should be activated */
                if((ospi_struct->delay_hold_cycle == OSPI_DELAY_HOLD_QUARTER_CYCLE) &&
                        (cmd_struct->ins_dtr_mode == OSPI_INSDTR_MODE_ENABLE)) {
                    *tcfg_reg = (*tcfg_reg & ~OSPI_DADTR_MODE_ENABLE) | OSPI_DADTR_MODE_ENABLE;
                }
            }
            /* configure the TCFG register with DQS and SIOO modes */
            *tcfg_reg |= (cmd_struct->dqs_mode | cmd_struct->sioo_mode);
            /* configure the INS register with the instruction value */
            *ins_reg = cmd_struct->instruction;
        }
    } else {
        if(cmd_struct->addr_mode != OSPI_ADDRESS_NONE) {
            if(cmd_struct->data_mode != OSPI_DATA_NONE) {
                /* command with address and data */
                /* configure the TCFG register with all communication parameters */
                *tcfg_reg &= ~(OSPI_TCFG_ADDRMOD | OSPI_TCFG_ADDRDTR | OSPI_TCFG_ADDRSZ |
                               OSPI_TCFG_DATAMOD | OSPI_TCFG_DADTR);

                *tcfg_reg = cmd_struct->addr_mode | cmd_struct->addr_dtr_mode | cmd_struct->addr_size |
                            cmd_struct->data_mode | cmd_struct->data_dtr_mode;
            } else {
                /* command with only address */

                /* configure the TCFG register with all communication parameters */
                *tcfg_reg &= ~(OSPI_TCFG_ADDRMOD | OSPI_TCFG_ADDRDTR | OSPI_TCFG_ADDRSZ);

                *tcfg_reg = cmd_struct->addr_mode | cmd_struct->addr_dtr_mode | cmd_struct->addr_size;
            }
            /* configure the TCFG register with DQS and SIOO modes */
            *tcfg_reg |= (cmd_struct->dqs_mode | cmd_struct->sioo_mode);
            /* configure the ADDR register with the instruction value */
            OSPI_ADDR(ospi_periph) = cmd_struct->address;
        }
    }
}

/*!
    \brief      enable OSPI interrupt (API_ID: 0x0061U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  interrupt: OSPI interrupt
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INT_TERR: transfer error interrupt
      \arg        OSPI_INT_TC: transfer complete interrupt
      \arg        OSPI_INT_FT: fifo threshold interrupt
      \arg        OSPI_INT_SM: status match interrupt
      \arg        OSPI_INT_TMOUT: timeout interrupt
    \param[out] none
    \retval     none
*/
void ospi_interrupt_enable(uint32_t ospi_periph, uint8_t interrupt)
{
    OSPI_CTL(ospi_periph) |= interrupt;
}

/*!
    \brief      disable OSPI interrupt (API_ID: 0x0062U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  interrupt: OSPI interrupt
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INT_TERR: transfer error interrupt
      \arg        OSPI_INT_TC: transfer complete interrupt
      \arg        OSPI_INT_FT: fifo threshold interrupt
      \arg        OSPI_INT_SM: status match interrupt
      \arg        OSPI_INT_TMOUT: timeout interrupt
    \param[out] none
    \retval     none
*/
void ospi_interrupt_disable(uint32_t ospi_periph, uint8_t interrupt)
{
    OSPI_CTL(ospi_periph) &= ~interrupt;
}

/*!
    \brief      get OSPI fifo level (API_ID: 0x0063U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     6-bit fifo level
*/
uint32_t ospi_fifo_level_get(uint32_t ospi_periph)
{
    uint32_t fl;
    fl = (OSPI_STAT(ospi_periph) & (uint32_t)OSPI_STAT_FL) >> 8U;
    return fl;
}

/*!
    \brief      get OSPI flag status (API_ID: 0x0064U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  flag: OSPI flag status
                only one parameter can be selected which is shown as below:
      \arg        OSPI_FLAG_TERR: transfer error flag
      \arg        OSPI_FLAG_TC: transfer complete flag
      \arg        OSPI_FLAG_FT: fifo threshold flag
      \arg        OSPI_FLAG_SM: status match flag
      \arg        OSPI_FLAG_TMOUT: timeout flag
      \arg        OSPI_FLAG_BUSY: busy flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus ospi_flag_get(uint32_t ospi_periph, uint32_t flag)
{
    if(RESET != (OSPI_STAT(ospi_periph) & flag)) {
        return SET;
    } else {
        return RESET;
    }
}

/*!
    \brief      clear OSPI flag status (API_ID: 0x0065U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  flag: OSPI flag status
                only one parameter can be selected which is shown as below:
      \arg        OSPI_FLAG_TERR: transfer error flag
      \arg        OSPI_FLAG_TC: transfer complete flag
      \arg        OSPI_FLAG_SM: status match flag
      \arg        OSPI_FLAG_TMOUT: timeout flag
    \param[out] none
    \retval     none
*/
void ospi_flag_clear(uint32_t ospi_periph, uint32_t flag)
{
    OSPI_STATC(ospi_periph) = (uint32_t)flag;
}

/*!
    \brief      get OSPI interrupt flag status (API_ID: 0x0066U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  interrupt: OSPI interrupt flag status
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INT_FLAG_TERR: transmit buffer empty interrupt flag
      \arg        OSPI_INT_FLAG_TC: receive buffer not empty interrupt flag
      \arg        OSPI_INT_FLAG_FT: overrun interrupt flag
      \arg        OSPI_INT_FLAG_SM: config error interrupt flag
      \arg        OSPI_INT_FLAG_TMOUT: CRC error interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus ospi_interrupt_flag_get(uint32_t ospi_periph, uint8_t interrupt)
{
    uint32_t reg1 = OSPI_STAT(ospi_periph);
    uint32_t reg2 = OSPI_CTL(ospi_periph);

    switch(interrupt) {
    /* OSPI transfer error interrupt flag */
    case OSPI_INT_FLAG_TERR:
        reg1 = reg1 & OSPI_STAT_TERR;
        reg2 = reg2 & OSPI_CTL_TERRIE;
        break;
    /* OSPI transfer complete interrupt flag */
    case OSPI_INT_FLAG_TC:
        reg1 = reg1 & OSPI_STAT_TC;
        reg2 = reg2 & OSPI_CTL_TCIE;
        break;
    /* OSPI fifo threshold interrupt flag */
    case OSPI_INT_FLAG_FT:
        reg1 = reg1 & OSPI_STAT_FT;
        reg2 = reg2 & OSPI_CTL_FTIE;
        break;
    /* OSPI status match interrupt flag */
    case OSPI_INT_FLAG_SM:
        reg1 = reg1 & OSPI_STAT_SM;
        reg2 = reg2 & OSPI_CTL_SMIE;
        break;
    /* OSPI timeout interrupt flag */
    case OSPI_INT_FLAG_TMOUT:
        reg1 = reg1 & OSPI_STAT_TMOUT;
        reg2 = reg2 & OSPI_CTL_TMOUTIE;
        break;
    default :
        break;
    }
    /*get OSPI interrupt flag status */
    if(RESET != (reg1 && reg2)) {
        return SET;
    } else {
        return RESET;
    }
}

/*!
    \brief      clear OSPI interrupt flag status (API_ID: 0x0067U)
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  flag: OSPI interrupt flag status
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INT_FLAG_TERR: transmit buffer empty interrupt flag
      \arg        OSPI_INT_FLAG_TC: receive buffer not empty interrupt flag
      \arg        OSPI_INT_FLAG_SM: config error interrupt flag
      \arg        OSPI_INT_FLAG_TMOUT: CRC error interrupt flag
    \param[out] none
    \retval     none
*/
void ospi_interrupt_flag_clear(uint32_t ospi_periph, uint32_t flag)
{
    uint32_t reg = OSPI_CTL(ospi_periph);

    switch(flag) {
    /* clear OSPI transfer error interrupt flag */
    case OSPI_INT_FLAG_TERR:
        if(RESET != (reg & OSPI_CTL_TERRIE)) {
            OSPI_STATC(ospi_periph) = (uint32_t)OSPI_STATC_TERRC;
        }
        break;
    /* clear OSPI transfer complete interrupt flag */
    case OSPI_INT_FLAG_TC:
        if(RESET != (reg & OSPI_CTL_TERRIE)) {
            OSPI_STATC(ospi_periph) = (uint32_t)OSPI_STATC_TCC;
        }
        break;
    /* clear OSPI status match interrupt flag */
    case OSPI_INT_FLAG_SM:
        if(RESET != (reg & OSPI_CTL_TERRIE)) {
            OSPI_STATC(ospi_periph) = (uint32_t)OSPI_STATC_SMC;
        }
        break;
    /* clear OSPI timeout interrupt flag */
    case OSPI_INT_FLAG_TMOUT:
        if(RESET != (reg & OSPI_CTL_TERRIE)) {
            OSPI_STATC(ospi_periph) = (uint32_t)OSPI_STATC_TMOUTC;
        }
        break;
    default :
        break;
    }
}
