/*!
    \file    gd32h77x_cpdm.c
    \brief   CPDM driver

    \version 2025-08-10, V0.0.0, firmware for GD32H77x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h77x_cpdm.h"

#define CPDM_DELAY_STEP_CNT_MASK   ((uint32_t)0x0000007FU) /*!< CPDM delay line length mask */
#define CPDM_DLLEN_OFFSET          ((uint32_t)16U)        /*!< CPDM delay line length offset */
#define CPDM_DLSTCNT_OFFSET        ((uint32_t)8U)         /*!< CPDM delay step count for a unit delay UINT offset */

/*!
    \brief      enable CPDM (API_ID: 0x0001U)
    \param[in]  cpdm_periph: the clock phase delay module
                only one parameter can be selected which is shown as below:
      \arg        CPDM_SDIO0: clock phase delay module of SDIO0
      \arg        CPDM_SDIO1: clock phase delay module of SDIO1
      \arg        CPDM_OSPI0: Clock Phase Delay Module of OSPI0
      \arg        CPDM_OSPI1: Clock Phase Delay Module of OSPI1
    \param[out] none
    \retval     none
*/
void cpdm_enable(uint32_t cpdm_periph)
{
    /* enable CPDM */
    CPDM_CTL(cpdm_periph) |= CPDM_CTL_CPDMEN;
}

/*!
    \brief      disable CPDM (API_ID: 0x0002U)
    \param[in]  cpdm_periph: the clock phase delay module
                only one parameter can be selected which is shown as below:
      \arg        CPDM_SDIO0: clock phase delay module of SDIO0
      \arg        CPDM_SDIO1: clock phase delay module of SDIO1
      \arg        CPDM_OSPI0: Clock Phase Delay Module of OSPI0
      \arg        CPDM_OSPI1: Clock Phase Delay Module of OSPI1
    \param[out] none
    \retval     none
*/
void cpdm_disable(uint32_t cpdm_periph)
{
    /* disable CPDM */
    CPDM_CTL(cpdm_periph) &= ~CPDM_CTL_CPDMEN;
}

/*!
    \brief      enable CPDM delay line sample module (API_ID: 0x0003U)
    \param[in]  cpdm_periph: the clock phase delay module
                only one parameter can be selected which is shown as below:
      \arg        CPDM_SDIO0: clock phase delay module of SDIO0
      \arg        CPDM_SDIO1: clock phase delay module of SDIO1
      \arg        CPDM_OSPI0: Clock Phase Delay Module of OSPI0
      \arg        CPDM_OSPI1: Clock Phase Delay Module of OSPI1
    \param[out] none
    \retval     none
*/
void cpdm_delayline_sample_enable(uint32_t cpdm_periph)
{
    /* enable CPDM delay line sample module */
    CPDM_CTL(cpdm_periph) |= CPDM_CTL_DLSEN;
}

/*!
    \brief      disable CPDM delay line sample module (API_ID: 0x0004U)
    \param[in]  cpdm_periph: the clock phase delay module
                only one parameter can be selected which is shown as below:
      \arg        CPDM_SDIO0: clock phase delay module of SDIO0
      \arg        CPDM_SDIO1: clock phase delay module of SDIO1
      \arg        CPDM_OSPI0: Clock Phase Delay Module of OSPI0
      \arg        CPDM_OSPI1: Clock Phase Delay Module of OSPI1
    \param[out] none
    \retval     none
*/
void cpdm_delayline_sample_disable(uint32_t cpdm_periph)
{
    /* disable CPDM delay line sample module */
    CPDM_CTL(cpdm_periph) &= ~CPDM_CTL_DLSEN;
}

/*!
    \brief      select CPDM output clock phase (API_ID: 0x0005U)
    \param[in]  cpdm_periph: the clock phase delay module
                only one parameter can be selected which is shown as below:
      \arg        CPDM_SDIO0: clock phase delay module of SDIO0
      \arg        CPDM_SDIO1: clock phase delay module of SDIO1
      \arg        CPDM_OSPI0: Clock Phase Delay Module of OSPI0
      \arg        CPDM_OSPI1: Clock Phase Delay Module of OSPI1
    \param[in]  output_clock_phase: the output clock phase, refer to cpdm_output_phase_enum
                only one parameter can be selected which is shown as below:
      \arg        CPDM_OUTPUT_PHASE_SELECTION_0: output clock phase = input clock
      \arg        CPDM_OUTPUT_PHASE_SELECTION_1: output clock phase = input clock + 1 * UNIT delay
      \arg        CPDM_OUTPUT_PHASE_SELECTION_2: output clock phase = input clock + 2 * UNIT delay
      \arg        CPDM_OUTPUT_PHASE_SELECTION_3: output clock phase = input clock + 3 * UNIT delay
      \arg        CPDM_OUTPUT_PHASE_SELECTION_4: output clock phase = input clock + 4 * UNIT delay
      \arg        CPDM_OUTPUT_PHASE_SELECTION_5: output clock phase = input clock + 5 * UNIT delay
      \arg        CPDM_OUTPUT_PHASE_SELECTION_6: output clock phase = input clock + 6 * UNIT delay
      \arg        CPDM_OUTPUT_PHASE_SELECTION_7: output clock phase = input clock + 7 * UNIT delay
      \arg        CPDM_OUTPUT_PHASE_SELECTION_8: output clock phase = input clock + 8 * UNIT delay
      \arg        CPDM_OUTPUT_PHASE_SELECTION_9: output clock phase = input clock + 9 * UNIT delay
      \arg        CPDM_OUTPUT_PHASE_SELECTION_10: output clock phase = input clock + 10 * UNIT delay
      \arg        CPDM_OUTPUT_PHASE_SELECTION_11: output clock phase = input clock + 11 * UNIT delay
      \arg        CPDM_OUTPUT_PHASE_SELECTION_12: output clock phase = input clock + 12 * UNIT delay
    \param[out] none
    \retval     none
*/
void cpdm_output_clock_phase_select(uint32_t cpdm_periph, cpdm_output_phase_enum output_clock_phase)
{
    uint32_t reg = 0U;

    reg = CPDM_CFG(cpdm_periph);
    reg &= ~CPDM_CFG_CPSEL;
    /* select CPDM output clock phase */
    reg |= ((uint32_t)output_clock_phase & CPDM_CFG_CPSEL);
    CPDM_CFG(cpdm_periph) = reg;
}

/*!
    \brief      configure CPDM delay step (API_ID: 0x0006U)
    \param[in]  cpdm_periph: the clock phase delay module
                only one parameter can be selected which is shown as below:
      \arg        CPDM_SDIO0: clock phase delay module of SDIO0
      \arg        CPDM_SDIO1: clock phase delay module of SDIO1
      \arg        CPDM_OSPI0: Clock Phase Delay Module of OSPI0
      \arg        CPDM_OSPI1: Clock Phase Delay Module of OSPI1
    \param[in]  delay_step: 0 ~ 127
    \param[out] none
    \retval     none
*/
void cpdm_delay_step_config(uint32_t cpdm_periph, uint8_t delay_step)
{
    uint32_t reg = 0U;

    reg = CPDM_CFG(cpdm_periph);
    reg &= ~CPDM_CFG_DLSTCNT;
    /* configure delay step */
    reg |= (((uint32_t)delay_step & CPDM_DELAY_STEP_CNT_MASK) << CPDM_DLSTCNT_OFFSET);
    CPDM_CFG(cpdm_periph) = reg;
}

/*!
    \brief      get delay line length valid flag (API_ID: 0x0007U)
    \param[in]  cpdm_periph: the clock phase delay module
                only one parameter can be selected which is shown as below:
      \arg        CPDM_SDIO0: clock phase delay module of SDIO0
      \arg        CPDM_SDIO1: clock phase delay module of SDIO1
      \arg        CPDM_OSPI0: Clock Phase Delay Module of OSPI0
      \arg        CPDM_OSPI1: Clock Phase Delay Module of OSPI1
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus cpdm_delayline_length_valid_flag_get(uint32_t cpdm_periph)
{
    FlagStatus reval = RESET;
    uint32_t reg = CPDM_CFG(cpdm_periph);

    if(reg & CPDM_CFG_DLLENF) {
        reval = SET;
    }
    return reval;
}

/*!
    \brief      get delay line length (API_ID: 0x0008U)
    \param[in]  cpdm_periph: the clock phase delay module
                only one parameter can be selected which is shown as below:
      \arg        CPDM_SDIO0: clock phase delay module of SDIO0
      \arg        CPDM_SDIO1: clock phase delay module of SDIO1
      \arg        CPDM_OSPI0: Clock Phase Delay Module of OSPI0
      \arg        CPDM_OSPI1: Clock Phase Delay Module of OSPI1
    \param[out] none
    \retval     the value of delay line length, 0x00~0xFF
*/
uint16_t cpdm_delayline_length_get(uint32_t cpdm_periph)
{
    return (uint16_t)((CPDM_CFG(cpdm_periph) & CPDM_CFG_DLLEN) >> CPDM_DLLEN_OFFSET);
}
