/*!
    \file    gd32h77x_pmu.h
    \brief   definitions for the PMU

    \version 2025-08-10, V0.0.0, firmware for GD32H77x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this 
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice, 
       this list of conditions and the following disclaimer in the documentation 
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors 
       may be used to endorse or promote products derived from this software without 
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
*/

#ifndef GD32F77X_PMU_H
#define GD32F77X_PMU_H

#include "gd32h77x.h"

/* PMU definitions */
#define PMU                             PMU_BASE                            /*!< PMU base address */

/* registers definitions */
#define PMU_CTL0                        REG32((PMU) + 0x00000000U)          /*!< PMU control register 0 */
#define PMU_CS                          REG32((PMU) + 0x00000004U)          /*!< PMU control and status register */
#define PMU_CTL1                        REG32((PMU) + 0x00000008U)          /*!< PMU control register 1 */
#define PMU_CTL2                        REG32((PMU) + 0x00000010U)          /*!< PMU control register 2 */
#define PMU_CTL3                        REG32((PMU) + 0x00000014U)          /*!< PMU control register 3 */
#define PMU_PAR                         REG32((PMU) + 0x00000018U)          /*!< PMU parameter register */

/* bits definitions */
/* PMU_CTL0 */
#define PMU_CTL0_COREPOROFFOUT          BIT(0)                              /*!< CORE POR OFF output */
#define PMU_CTL0_STBMOD                 BIT(1)                              /*!< standby mode */
#define PMU_CTL0_WURST                  BIT(2)                              /*!< wakeup flag reset */
#define PMU_CTL0_STBRST                 BIT(3)                              /*!< standby flag reset */
#define PMU_CTL0_LVDEN                  BIT(4)                              /*!< low voltage detector enable */
#define PMU_CTL0_LVDT                   BITS(5,7)                           /*!< low voltage detector threshold */
#define PMU_CTL0_BKPWEN                 BIT(8)                              /*!< backup domain write enable */
#define PMU_CTL0_COREPOROFFP            BIT(9)                              /*!< CORE POR OFF output polarity selection */
#define PMU_CTL0_COREPDREN              BIT(10)                              /*!< CORE PDR circuit enable signal */
#define PMU_CTL0_SLDOVS                 BITS(14,15)                         /*!< stop mode LDO output voltage select */
#define PMU_CTL0_VAVDEN                 BIT(16)                             /*!< VDDA analog voltage detector enable */
#define PMU_CTL0_VAVDVC                 BITS(17,18)                         /*!< VDDA analog voltage detector level configure bits */
#define PMU_CTL0_VOVDEN                 BIT(19)                             /*!< V0.9V voltage detector enable */
#define PMU_CTL0_VUVDEN                 BIT(24)                             /*!< V0.9V under voltage detector enable */
#define PMU_CTL0_VOVDVC                 BITS(25,26)                         /*!< V0.9V over voltage detector voltage level configure bits */
#define PMU_CTL0_VUVDVC                 BITS(27,28)                         /*!< V0.9V under voltage detector voltage level configure bits */
#define PMU_CTL0_SRAM_MODE              BITS(29,30)                         /*!< Sram_mode in Deep-sleep */
#define PMU_CTL0_SRAM_PRDYN             BITS(31)                            /*!< Indicates if the device has entered Deep-sleep mode */

/* PMU_CS */
#define PMU_CS_WUF                      BIT(0)                              /*!< wakeup flag */
#define PMU_CS_STBF                     BIT(1)                              /*!< standby flag */
#define PMU_CS_LVDF                     BIT(2)                              /*!< low voltage detector status flag */
#define PMU_CS_UVDF                     BIT(3)                              /*!< Record the undervoltage reset of the V0.9V */
#define PMU_CS_VUVDF0                   BIT(5)                              /*!< V0.9V under voltage detector flag bit */
#define PMU_CS_VOVDF1                   BIT(6)                              /*!< V0.9V over voltage detector flag bit after digital filter */
#define PMU_CS_VUVDF1                   BIT(7)                              /*!< V0.9V under voltage detector flag bit after digital filter */
#define PMU_CS_WUPEN0                   BIT(8)                              /*!< wakeup pin0(PA0) enable */
#define PMU_CS_WUPEN1                   BIT(9)                              /*!< wakeup pin1(PA2) enable */
#define PMU_CS_WUPEN3                   BIT(11)                             /*!< wakeup pin3(PC13) enable */
#define PMU_CS_WUPEN5                   BIT(13)                             /*!< wakeup pin5(PC1) enable */
#define PMU_CS_VAVDF                    BIT(16)                             /*!< VDDA analog voltage detector voltage output on VDDA flag */
#define PMU_CS_VOVDF                    BIT(20)                             /*!< V0.9V voltage detector flag */

/* PMU_CTL1 */
#define PMU_CTL1_BKPVSEN                BIT(0)                              /*!< Backup regulator enable */
#define PMU_CTL1_VBTMEN                 BIT(4)                              /*!< VBAT and temperature monitoring enable */
#define PMU_CTL1_BKPVSRF                BIT(16)                             /*!< Backup regulator ready */
#define PMU_CTL1_VBATLF                 BIT(20)                             /*!< VBAT level monitoring versus low threshold */
#define PMU_CTL1_VBATHF                 BIT(21)                             /*!< VBAT level monitoring versus high threshold */
#define PMU_CTL1_TEMPLF                 BIT(22)                             /*!< temperature level monitoring versus low threshold */
#define PMU_CTL1_TEMPHF                 BIT(23)                             /*!< temperature level monitoring versus high threshold */

/* PMU_CTL2 */
#define PMU_CTL2_VOVDO_DNF              BITS(0,7)                           /*!< VUVD analog output digital noise filter */
#define PMU_CTL2_VUVDO_DNF              BITS(8,15)                          /*!< VOVD analog output digital noise filter */

/* PMU_CTL3 */
#define PMU_CTL3_BYPASS                 BIT(0)                              /*!< power management unit bypass control bit */
#define PMU_CTL3_DVSEN                  BIT(2)                              /*!< SMPS step-down converter enable SDEN */
#define PMU_CTL3_VCEN                   BIT(8)                              /*!< VBAT battery charging enable */
#define PMU_CTL3_VCRSEL                 BIT(9)                              /*!< VBAT battery charging resistor selection */
#define PMU_CTL3_MIPISW                 BIT(17)                             /*!< Powers off the MIPI */
#define PMU_CTL3_PKCAUSW                BIT(18)                             /*!< Powers off the PKCAU */
#define PMU_CTL3_ETHCSW                 BIT(19)                             /*!< Powers off the ETHC */
#define PMU_CTL3_MIPIHTF                BIT(20)                             /*!< Mipi power switch ack flag */
#define PMU_CTL3_MIPIHTEN               BIT(21)                             /*!< MIPI output isolation enable */
#define PMU_CTL3_PKCAUHTEN              BIT(22)                             /*!< PKCAU output isolation enable */
#define PMU_CTL3_ETHCHTEN               BIT(23)                             /*!< ETHC output isolation enable */
#define PMU_CTL3_VUSB33DEN              BIT(24)                             /*!< VDD33USB voltage level detector enable bit */
#define PMU_CTL3_USBSEN                 BIT(25)                             /*!< USB regulator enable */
#define PMU_CTL3_USB33RF                BIT(26)                             /*!< USB supply ready flag */
#define PMU_CTL3_MIPI_LDO_EN            BIT(27)                             /*!< MIPI_LDO enable signal */
#define PMU_CTL3_MIPI_LDO_BYP           BIT(28)                             /*!< MIPI_LDO BYPASS mode enable signal */
#define PMU_CTL3_MIPI_LDO_RDY           BIT(29)                             /*!< The MIPI_LDO output voltage ready signal */
#define PMU_CTL3_PKCAUHTF               BIT(30)                             /*!< PKCAU power switch ack flag */
#define PMU_CTL3_ETHCHTF                BIT(31)                             /*!< ETHC power switch ack flag */

/* PMU_PAR */
#define PMU_PAR_PWR_CNT                 BITS(0,11)                          /*!< When exit stop, switch to hsi(csi) clock. Wait the HIS/CSI COUNTER */
#define PMU_PAR_TSW_HSICNT              BITS(16,20)                         /*!< When enter stop, switch to hsi(csi) clock. Wait the HIS/CSI COUNTER */

/* constants definitions */
/* PMU low voltage detector threshold definitions */
#define CTL0_LVDT(regval)               (BITS(5,7) & ((uint32_t)(regval) << 5U))
#define PMU_LVDT_0                      CTL0_LVDT(0)                        /*!< voltage threshold is 2.1V */
#define PMU_LVDT_1                      CTL0_LVDT(1)                        /*!< voltage threshold is 2.3V */
#define PMU_LVDT_2                      CTL0_LVDT(2)                        /*!< voltage threshold is 2.4V */
#define PMU_LVDT_3                      CTL0_LVDT(3)                        /*!< voltage threshold is 2.6V */
#define PMU_LVDT_4                      CTL0_LVDT(4)                        /*!< voltage threshold is 2.7V */
#define PMU_LVDT_5                      CTL0_LVDT(5)                        /*!< voltage threshold is 2.9V */
#define PMU_LVDT_6                      CTL0_LVDT(6)                        /*!< voltage threshold is 3.0V */
#define PMU_LVDT_7                      CTL0_LVDT(7)                        /*!< external input */

/* PMU stop mode LDO output voltage select definitions */
#define CTL0_SLDOVS(regval)             (BITS(14,15) & ((uint32_t)(regval) << 14U))
#define PMU_SLDOVS_0                    CTL0_SLDOVS(0)                      /*!< Deep-sleep mode LDO voltage 0.6V */
#define PMU_SLDOVS_1                    CTL0_SLDOVS(1)                      /*!< Deep-sleep mode LDO voltage 0.7V */
#define PMU_SLDOVS_2                    CTL0_SLDOVS(2)                      /*!< Deep-sleep mode LDO voltage 0.8V (default)*/
#define PMU_SLDOVS_3                    CTL0_SLDOVS(3)                      /*!< Deep-sleep mode LDO voltage 0.9V */

/* PMU analog voltage detector threshold definitions */
#define CTL0_VAVDVC(regval)             (BITS(17,18) & ((uint32_t)(regval) << 17U))
#define PMU_AVDT_0                      CTL0_VAVDVC(0)                      /*!< voltage threshold is 1.7V */
#define PMU_AVDT_1                      CTL0_VAVDVC(1)                      /*!< voltage threshold is 2.1V */
#define PMU_AVDT_2                      CTL0_VAVDVC(2)                      /*!< voltage threshold is 2.5V */
#define PMU_AVDT_3                      CTL0_VAVDVC(3)                      /*!< voltage threshold is 2.8V */

/* PMU analog voltage detector threshold definitions */
#define CTL0_VOVDVC(regval)             (BITS(25,26) & ((uint32_t)(regval) << 25U))
#define PMU_OVDT_0                      CTL0_VOVDVC(0)                      /*!< vcore over voltage threshold is 0.9V */
#define PMU_OVDT_1                      CTL0_VOVDVC(1)                      /*!< vcore over voltage threshold is 1.0V */
#define PMU_OVDT_2                      CTL0_VOVDVC(2)                      /*!< vcore over voltage threshold is 1.1V */
#define PMU_OVDT_3                      CTL0_VOVDVC(3)                      /*!< vcore over voltage threshold is 1.2V */

/* PMU analog voltage detector threshold definitions */
#define CTL0_VUVDVC(regval)             (BITS(27,28) & ((uint32_t)(regval) << 27U))
#define PMU_UVDT_0                      CTL0_VUVDVC(0)                      /*!< vcore over voltage threshold is 0.8V */
#define PMU_UVDT_1                      CTL0_VUVDVC(1)                      /*!< vcore over voltage threshold is 0.7V */
#define PMU_UVDT_2                      CTL0_VUVDVC(2)                      /*!< vcore over voltage threshold is 0.6V */


/* PMU VBAT battery charging resistor selection */
#define CTL3_VCRSEL(regval)             (BIT(9) & ((uint32_t)(regval) << 9U))
#define PMU_VCRSEL_5K                   CTL3_VCRSEL(0)                      /*!< 5 kOhms resistor is selected for charing VBAT battery */
#define PMU_VCRSEL_1P5K                 CTL3_VCRSEL(1)                      /*!< 1.5 kOhms resistor is selected for charing VBAT battery */

/* PMU flag definitions */
#define PMU_REGIDX_BIT(regidx, bitpos)  (((uint32_t)(regidx) << 6) | (uint32_t)(bitpos))
#define PMU_REG_VAL(periph)             (REG32(PMU + ((uint32_t)(periph) >> 6U)))
#define PMU_BIT_POS(val)                ((uint32_t)(val) & 0x1FU)

#define PMU_CTL0_OFFSET                 (0x00000000U)                       /*!< PMU control register 0 offset */
#define PMU_CS_OFFSET                   (0x00000004U)                       /*!< PMU control and status register offset */
#define PMU_CTL1_OFFSET                 (0x00000008U)                       /*!< PMU control register 1 offset */
#define PMU_CTL2_OFFSET                 (0x00000010U)                       /*!< PMU control register 2 offset */
#define PMU_PAR_OFFSET                  (0x00000018U)                       /*!< PMU parameter register offset */

#define PMU_FLAG_WAKEUP                 PMU_REGIDX_BIT(PMU_CS_OFFSET, 0)    /*!< wakeup flag */
#define PMU_FLAG_STANDBY                PMU_REGIDX_BIT(PMU_CS_OFFSET, 1)    /*!< standby flag */
#define PMU_FLAG_LVDF                   PMU_REGIDX_BIT(PMU_CS_OFFSET, 2)    /*!< low voltage detector status flag */
#define PMU_FLAG_UVDF                   PMU_REGIDX_BIT(PMU_CS_OFFSET, 3)    /*!< record the undervoltage reset of the V0.9V */
#define PMU_FLAG_VUVDF0                 PMU_REGIDX_BIT(PMU_CS_OFFSET, 5)    /*!< V0.9V under voltage detector flag bit */
#define PMU_FLAG_VOVDF1                 PMU_REGIDX_BIT(PMU_CS_OFFSET, 6)    /*!< V0.9V over voltage detector flag bit after digital filter */
#define PMU_FLAG_VUVDF1                 PMU_REGIDX_BIT(PMU_CS_OFFSET, 7)    /*!< V0.9V under voltage detector flag bit after digital filter */
#define PMU_FLAG_VAVDF                  PMU_REGIDX_BIT(PMU_CS_OFFSET, 16)   /*!< VDDA analog voltage detector voltage output on VDDA flag */
#define PMU_FLAG_VOVDF0                 PMU_REGIDX_BIT(PMU_CS_OFFSET, 20)   /*!< peripheral voltage on V0.9V detector flag */
#define PMU_FLAG_BKPVSRF                PMU_REGIDX_BIT(PMU_CTL1_OFFSET, 16) /*!< backup voltage stabilizer ready flag */
#define PMU_FLAG_VBATLF                 PMU_REGIDX_BIT(PMU_CTL1_OFFSET, 20) /*!< VBAT level monitoring versus low threshold */
#define PMU_FLAG_VBATHF                 PMU_REGIDX_BIT(PMU_CTL1_OFFSET, 21) /*!< VBAT level monitoring versus high threshold */
#define PMU_FLAG_TEMPLF                 PMU_REGIDX_BIT(PMU_CTL1_OFFSET, 22) /*!< temperature level monitoring versus low threshold */
#define PMU_FLAG_TEMPHF                 PMU_REGIDX_BIT(PMU_CTL1_OFFSET, 23) /*!< temperature level monitoring versus high threshold */
#define PMU_FLAG_MIPIHTF                PMU_REGIDX_BIT(PMU_CTL2_OFFSET, 20) /*!< MIPI power switch ack flag */
#define PMU_FLAG_USB33RF                PMU_REGIDX_BIT(PMU_CTL2_OFFSET, 26) /*!< USB supply ready flag bit */

/* PMU wakeup pin definitions */
#define PMU_WAKEUP_PIN0                 PMU_CS_WUPEN0                       /*!< wakeup pin 0 */
#define PMU_WAKEUP_PIN1                 PMU_CS_WUPEN1                       /*!< wakeup pin 1 */
#define PMU_WAKEUP_PIN3                 PMU_CS_WUPEN3                       /*!< wakeup pin 3 */
#define PMU_WAKEUP_PIN5                 PMU_CS_WUPEN5                       /*!< wakeup pin 5 */

/* PMU SMPS LDO supply mode definitions */
#define PMU_DIRECT_SMPS_SUPPLY              PMU_CTL3_DVSEN                                                              /*!< Core domains are suppplied from the SMPS only */
#define PMU_BYPASS                          PMU_CTL3_BYPASS                                                             /*!< The SMPS disabled and the LDO Bypass. The Core domains are supplied from an external source */

/* PMU command constants definitions */
#define WFI_CMD                         ((uint8_t)0x00U)                    /*!< use WFI command */
#define WFE_CMD                         ((uint8_t)0x01U)                    /*!< use WFE command */

#ifdef FW_DEBUG_ERR_REPORT
/* check enter Deep-sleep mode wait time */
#define NOT_PMU_ENTER_WAITTIME(wait_time)                (((wait_time) < 0x0) || ((wait_time) > 0x1F))
/* check exit Deep-sleep mode wait time */
#define NOT_PMU_EXIT_WAITTIME(wait_time)                 (((wait_time) < 0x0) || ((wait_time) > 0xFFF))
#endif /* FW_DEBUG_ERR_REPORT */

/* function declarations */
/* reset PMU registers */
void pmu_deinit(void);
/* select low voltage detector threshold */
void pmu_lvd_select(uint32_t lvdt_n);
/* enable PMU lvd */
void pmu_lvd_enable(void);
/* disable PMU lvd */
void pmu_lvd_disable(void);
/* select analog voltage detector threshold */
void pmu_avd_select(uint32_t avdt_n);
/* enable PMU analog voltage detector */
void pmu_avd_enable(void);
/* disable PMU analog voltage detector */
void pmu_avd_disable(void);
/* select PMU V0.9V over detector threshold */
void pmu_ovd_select(uint32_t ovdt_n);
/* enable PMU V0.9V over voltage detector */
void pmu_ovd_enable(void);
/* disable PMU V0.9V over voltage detector */
void pmu_ovd_disable(void);
/* select PMU V0.9V under voltage detector threshold */
void pmu_uvd_select(uint32_t uvdt_n);
/* enable PMU V0.9V under voltage detector */
void pmu_uvd_enable(void);
/* disable PMU V0.9V under voltage detector */
void pmu_uvd_disable(void);
/* deep-sleep mode V0.9V core voltage select */
void pmu_sldo_output_select(uint32_t sldo);

/* PMU VBAT battery charging resistor selection */
void pmu_vbat_charging_select(uint32_t resistor);
/* enable VBAT battery charging */
void pmu_vbat_charging_enable(void);
/* disable VBAT battery charging */
void pmu_vbat_charging_disable(void);
/* VBAT and temperature monitoring enable */
void pmu_vbat_temp_moniter_enable(void);
/* VBAT and temperature monitoring disable */
void pmu_vbat_temp_moniter_disable(void);

/* USB regulator*/
/* USB regulator enable */
void pmu_usb_regulator_enable(void);
/* USB regulator disable */
void pmu_usb_regulator_disable(void);
/* VDD33USB voltage level detector enable */
void pmu_usb_voltage_detector_enable(void);
/* VDD33USB voltage level detector disable */
void pmu_usb_voltage_detector_disable(void);

/* power supply configurations*/
void pmu_smps_ldo_supply_config(uint32_t smpsmode);

/* set PMU mode */
/* enter sleep mode */
void pmu_to_sleepmode(uint8_t sleepmodecmd);
/* enter deepsleep mode */
void pmu_to_deepsleepmode(uint8_t deepsleepmodecmd);
/* enter standby mode */
void pmu_to_standbymode(void);
/* enable PMU wakeup pin */
void pmu_wakeup_pin_enable(uint32_t wakeup_pin);
/* disable PMU wakeup pin */
void pmu_wakeup_pin_disable(uint32_t wakeup_pin);

/* backup related functions */
/* enable backup domain write */
void pmu_backup_write_enable(void);
/* disable backup domain write */
void pmu_backup_write_disable(void);
/* enable backup voltage stabilizer */
void pmu_backup_voltage_stabilizer_enable(void);
/* disable backup voltage stabilizer */
void pmu_backup_voltage_stabilizer_disable(void);

/* configure IRC counter before enter Deep-sleep mode */
void pmu_enter_deepsleep_wait_time_config(uint32_t wait_time);
/* configure IRC counter before exit Deep-sleep mode */
void pmu_exit_deepsleep_wait_time_config(uint32_t wait_time);

/* flag functions */
/* clear flag bit */
void pmu_flag_clear(uint32_t flag_reset);
/* get flag state */
FlagStatus pmu_flag_get(uint32_t flag);

#endif /* GD32F77X_PMU_H */