/*!
    \file    flash_if.c
    \brief   USB DFU device flash interface functions

    \version 2024-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "hs1_inter_flash_if.h"
#include "gd32h7xx_hal_fmc.h"

__IO uint8_t hs1_vfimg_set = 0;
__IO uint32_t hs1_certificate_size = 0;

/* local function prototypes ('static') */
static uint8_t  hs1_flash_if_init       (void);
static uint8_t  hs1_flash_if_deinit     (void);
static uint8_t  hs1_flash_if_erase      (uint32_t addr);
static uint8_t  hs1_flash_if_write      (uint8_t *buf, uint32_t addr, uint32_t len);
static uint8_t* hs1_flash_if_read       (uint8_t *buf, uint32_t addr, uint32_t len);
static uint8_t  hs1_flash_if_checkaddr  (uint32_t addr);

static hal_fmc_state_enum hs1_fmc_ready_wait(uint32_t timeout);

dfu_mal_prop HS1_DFU_Flash_cb =
{
    (const uint8_t *)HS1_FLASH_IF_STRING,

    hs1_flash_if_init,
    hs1_flash_if_deinit,
    hs1_flash_if_erase,
    hs1_flash_if_write,
    hs1_flash_if_read,
    hs1_flash_if_checkaddr,
    60U, /* flash erase timeout in ms */
    80U  /* flash programming timeout in ms (80us * RAM Buffer size (1024 Bytes) */
};

/*!
    \brief      write the option byte
    \param[in]  Mem_Add: memory address
    \param[in]  data: pointer to the data
    \param[out] none
    \retval     state of FMC, refer to fmc_state_enum
*/
hal_fmc_state_enum hs1_option_byte_write(uint32_t mem_add, uint8_t* data)
{
    hal_fmc_state_enum status ;

    /* clear pending flags */
    hals_fmc_flag_clear(FMC_FLAG_PGSERR | FMC_FLAG_WPERR | FMC_FLAG_END);

    status = hs1_fmc_ready_wait(FMC_TIMEOUT_COUNT);

    if (FMC_READY != status) {
        return status;
    }

    uint32_t write_data = (uint32_t)data[0] |
                          (uint32_t)(data[1] << 8) |
                          (uint32_t)(data[2] << 16) |
                          (uint32_t)(data[3] << 24);

    *(__IO uint32_t*)mem_add = write_data;

    return status;
}


void hs1_ob_write_all(uint8_t *data)
{
    uint32_t ob_addr = HS1_FMC_OB_START + 4;
    uint8_t *ob_data_addr = data + 4;

    hs1_option_byte_write(ob_addr, ob_data_addr);
    hs1_option_byte_write(ob_addr + 0x0C, ob_data_addr + 0x0C);
    hs1_option_byte_write(ob_addr + 0x14, ob_data_addr + 0x14);
    hs1_option_byte_write(ob_addr + 0x1C, ob_data_addr + 0x1C);
    hs1_option_byte_write(ob_addr + 0x24, ob_data_addr + 0x24);
    hs1_option_byte_write(ob_addr + 0x34, ob_data_addr + 0x34);
}

/*!
    \brief      flash memory interface initialization routine
    \param[in]  none
    \param[out] none
    \retval     MAL_OK if the operation is right, MAL_FAIL else
*/
static uint8_t hs1_flash_if_init (void)
{
    /* unlock the internal flash */
    hal_fmc_unlock();

    return MAL_OK;
}

/*!
    \brief      flash memory interface de-initialization routine
    \param[in]  none
    \param[out] none
    \retval     MAL_OK if the operation is right, MAL_FAIL else
*/
static uint8_t hs1_flash_if_deinit (void)
{
    /* lock the internal flash */
    hal_fmc_lock();

    return MAL_OK;
}

/*!
    \brief      erase flash sector
    \param[in]  addr: flash address to be erased
    \param[out] none
    \retval     MAL_OK if the operation is right, MAL_FAIL else
*/
static uint8_t hs1_flash_if_erase (uint32_t addr)
{
    hal_fmc_unlock();
    hals_fmc_sector_erase(addr);
    hal_fmc_lock();

    return MAL_OK;
}

/*!
    \brief      flash memory write routine
    \param[in]  buf: data buffer pointer
    \param[in]  addr: flash address to be written
    \param[in]  len: length of data to be written (in bytes)
    \param[out] none
    \retval     MAL_OK if the operation is right, MAL_FAIL else
*/
static uint8_t hs1_flash_if_write (uint8_t *buf, uint32_t addr, uint32_t len)
{
     uint32_t idx = 0U;

    /* unlock the flash program erase controller */
    hal_fmc_unlock();

    /* not an aligned data */
    if (len & 0x03U) {
        for(idx = len; idx < ((len & 0xFFFCU) + 4U); idx++) {
            buf[idx] = 0xFFU;
        }
    }

    /* data received are word multiple */
    for (idx = 0U; idx < len; idx += 4U) {
        hals_fmc_word_program(addr, *(uint32_t *)(buf + idx));

        addr += 4U;
    }

    hal_fmc_lock();

    return MAL_OK;
}

/*!
    \brief      flash memory read routine
    \param[in]  buf: data buffer pointer
    \param[in]  addr: flash address to be read from
    \param[in]  len: length of data to be read (in bytes)
    \param[out] none
    \retval     pointer to the physical address where data should be read
*/
static uint8_t *hs1_flash_if_read (uint8_t *buf, uint32_t addr, uint32_t len)
{
    if (((addr >= HS1_FLASH_START_ADDR) && (addr < HS1_FLASH_END_ADDR)) ||
           ((addr >= HS1_OCRAM_START) && (addr < HS1_OCRAM_END)) ||
              ((addr >= HS1_FMC_OB_START) && (addr < HS1_FMC_OB_END))) {
        return  (uint8_t *)(addr);
    } else if ((addr == HS1_EFUSE_START) && (len == HS1_EFUSE_LEN)) {
        return (uint8_t *)(addr);
    } else if ((addr == HS1_AES_IV_START) && (len == HS1_AES_IV_LEN)) {
        return (uint8_t *)(addr);
    } else if (addr == 0xA0000300) {
        NVIC_SystemReset();

        return 0;
    } else if (addr == 0xA0000200) {
        hs1_certificate_size = 136;

        return (uint8_t *)(&hs1_certificate_size);
    } else if (addr == 0x20009000) {
        return (uint8_t *)(addr);
    } else if (addr == 0x1FF0FC40) {
        return (uint8_t *)(addr);
    } else {
        return buf;
    }
}

/*!
    \brief      check if the address is an allowed address for this memory
    \param[in]  addr: flash address to be checked
    \param[out] none
    \retval     MAL_OK if the operation is right, MAL_FAIL else
*/
static uint8_t hs1_flash_if_checkaddr (uint32_t addr)
{
    if ((addr >= HS1_FLASH_START_ADDR) && (addr <= HS1_FLASH_END_ADDR)) {
        return MAL_OK;
    } else if ((addr >= HS1_OCRAM_START) && (addr <= HS1_OCRAM_END)) {
        return MAL_OK;
    } else if ((addr >= HS1_FMC_OB_START) && (addr <= HS1_FMC_OB_END)) {
        return MAL_OK;
    } else if ((addr >= HS1_EFUSE_START) && (addr <= HS1_EFUSE_END)) {
        return MAL_OK;
    } else if ((addr >= HS1_AES_IV_START) && (addr <= HS1_AES_IV_END)) {
        return MAL_OK;
    } else {
        return MAL_FAIL;
    }
}

/*!
    \brief      get FMC state
    \param[in]  none
    \param[out] none
    \retval     state of FMC, refer to fmc_state_enum
      \arg        FMC_READY: operation has been completed
      \arg        FMC_BUSY: operation is in progress
      \arg        FMC_WPERR: erase/program protection error
      \arg        FMC_PGSERR: program sequence error
      \arg        FMC_RPERR: read protection error
      \arg        FMC_RSERR: read secure error
      \arg        FMC_ECCCOR: one bit correct error
      \arg        FMC_ECCDET: two bits detect error
      \arg        FMC_OBMERR: option byte modify error
*/
static hal_fmc_state_enum hs1_fmc_state_get(void)
{
    hal_fmc_state_enum fmc_state = FMC_READY;

    if((uint32_t)0x00U != (FMC_STAT & FMC_STAT_BUSY)) {
        fmc_state = FMC_BUSY;
    } else {
        if((uint32_t)0x00U != (FMC_STAT & FMC_STAT_WPERR)) {
            fmc_state = FMC_WPERR;
        } else if((uint32_t)0x00U != (FMC_STAT & FMC_STAT_PGSERR)) {
            fmc_state = FMC_PGSERR;
        } else if((uint32_t)0x00U != (FMC_STAT & FMC_STAT_PGSERR)) {
            fmc_state = FMC_PGSERR;
        } else if((uint32_t)0x00U != (FMC_STAT & FMC_STAT_RPERR)) {
            fmc_state = FMC_RPERR;
        } else if((uint32_t)0x00U != (FMC_STAT & FMC_STAT_RSERR)) {
            fmc_state = FMC_RSERR;
        } else if((uint32_t)0x00U != (FMC_STAT & FMC_STAT_ECCCOR)) {
            fmc_state = FMC_ECCCOR;
        } else if((uint32_t)0x00U != (FMC_STAT & FMC_STAT_ECCDET)) {
            fmc_state = FMC_ECCDET;
        } else if((uint32_t)0x00U != (FMC_STAT & FMC_STAT_OBMERR)) {
            fmc_state = FMC_OBMERR;
        } else {
            /* illegal parameters */
        }
    }

    /* return the FMC state */
    return fmc_state;
}

/*!
    \brief      check whether FMC is ready or not
    \param[in]  timeout: timeout count
    \param[out] none
    \retval     state of FMC, refer to fmc_state_enum
      \arg        FMC_READY: operation has been completed
      \arg        FMC_BUSY: operation is in progress
      \arg        FMC_WPERR: erase/program protection error
      \arg        FMC_PGSERR: program sequence error
      \arg        FMC_RPERR: read protection error
      \arg        FMC_RSERR: read secure error
      \arg        FMC_ECCCOR: one bit correct error
      \arg        FMC_ECCDET: two bits detect error
      \arg        FMC_OBMERR: option byte modify error
      \arg        FMC_TOERR: timeout error
*/
static hal_fmc_state_enum hs1_fmc_ready_wait(uint32_t timeout)
{
    hal_fmc_state_enum fmc_state = FMC_BUSY;

    /* wait for FMC ready */
    do{
        /* get FMC state */
        fmc_state = hs1_fmc_state_get();
        timeout--;
    }while((FMC_BUSY == fmc_state) && (0U != timeout));

    if(FMC_BUSY == fmc_state){
        fmc_state = FMC_TOERR;
    }
    /* return the FMC state */
    return fmc_state;
}
