/*!
    \file    usbd_endpoint.c
    \brief   USB device mode endpoint functions

    \version 2024-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "usbd_endpoint.h"
#include "usbd_enum.h"
#include "drv_usb_hw.h"

/* endpoint type */
const uint32_t ep_type[] = {
    [USB_EP_ATTR_CTL]  = (uint32_t)USB_EPTYPE_CTRL,
    [USB_EP_ATTR_BULK] = (uint32_t)USB_EPTYPE_BULK,
    [USB_EP_ATTR_INT]  = (uint32_t)USB_EPTYPE_INTR,
    [USB_EP_ATTR_ISO]  = (uint32_t)USB_EPTYPE_ISOC
};

/*!
    \brief      endpoint initialization
    \param[in]  udev: pointer to USB core instance
    \param[in]  ep_desc: pointer to endpoint descriptor
    \param[out] none
    \retval     none
*/
uint32_t usbd_ep_setup(usb_core_driver *udev, const usb_desc_ep *ep_desc)
{
    usb_transc *transc;

    uint8_t ep_addr = ep_desc->bEndpointAddress;
    uint16_t max_len = ep_desc->wMaxPacketSize;

    /* set endpoint direction */
    if(EP_DIR(ep_addr)) {
        transc = &udev->dev.transc_in[EP_ID(ep_addr)];

        transc->ep_addr.dir = 1U;
    } else {
        transc = &udev->dev.transc_out[ep_addr];

        transc->ep_addr.dir = 0U;
    }

    transc->ep_addr.num = EP_ID(ep_addr);
    transc->max_len = max_len;
    transc->ep_type = (uint8_t)ep_type[ep_desc->bmAttributes & (uint8_t)USB_EPTYPE_MASK];

    /* active USB endpoint function */
    if(USBHS0_REG_BASE == udev->bp.base_reg) {
#ifdef USE_USBHS0_DEVICE
        (void)hs0_usb_transc_active(udev, transc);
#endif /* USE_USBHS0_DEVICE*/
    } else if (USBHS1_REG_BASE == udev->bp.base_reg) {
#ifdef USE_USBHS1_DEVICE
        (void)hs1_usb_transc_active(udev, transc);
#endif /* USE_USBHS1_DEVICE*/
    }

    return 0U;
}

/*!
    \brief      configure the endpoint when it is disabled
    \param[in]  udev: pointer to USB core instance
    \param[in]  ep_addr: endpoint address
                  in this parameter:
                    bit0..bit6: endpoint number (0..7)
                    bit7: endpoint direction which can be IN(1) or OUT(0)
    \param[out] none
    \retval     none
*/
uint32_t usbd_ep_clear(usb_core_driver *udev, uint8_t ep_addr)
{
    usb_transc *transc;

    if(EP_DIR(ep_addr)) {
        transc = &udev->dev.transc_in[EP_ID(ep_addr)];
    } else {
        transc = &udev->dev.transc_out[ep_addr];
    }

    /* active USB endpoint function */
    if(USBHS0_REG_BASE == udev->bp.base_reg) {
#ifdef USE_USBHS0_DEVICE
        (void)hs0_usb_transc_deactivate(udev, transc);
#endif /* USE_USBHS0_DEVICE*/
    } else if (USBHS1_REG_BASE == udev->bp.base_reg) {
#ifdef USE_USBHS1_DEVICE
        (void)hs1_usb_transc_deactivate(udev, transc);
#endif /* USE_USBHS1_DEVICE*/
    }

    return 0U;
}

/*!
    \brief      endpoint prepare to receive data
    \param[in]  udev: pointer to USB core instance
    \param[in]  ep_addr: endpoint address
                  in this parameter:
                    bit0..bit6: endpoint number (0..7)
                    bit7: endpoint direction which can be IN(1) or OUT(0)
    \param[in]  pbuf: user buffer address pointer
    \param[in]  len: buffer length
    \param[out] none
    \retval     none
*/
uint32_t usbd_ep_recev(usb_core_driver *udev, uint8_t ep_addr, uint8_t *pbuf, uint32_t len)
{
    usb_transc *transc = &udev->dev.transc_out[EP_ID(ep_addr)];

    /* setup the transfer */
    transc->xfer_buf = pbuf;
    transc->xfer_len = len;
    transc->xfer_count = 0U;

    if((uint8_t)USB_USE_DMA == udev->bp.transfer_mode) {
        transc->dma_addr = (uint32_t)pbuf;
    }

    /* start the transfer */
    if(USBHS0_REG_BASE == udev->bp.base_reg) {
#ifdef USE_USBHS0_DEVICE
        (void)hs0_usb_transc_outxfer(udev, transc);
#endif /* USE_USBHS0_DEVICE*/
    } else if (USBHS1_REG_BASE == udev->bp.base_reg) {
#ifdef USE_USBHS1_DEVICE
        (void)hs1_usb_transc_outxfer(udev, transc);
#endif /* USE_USBHS1_DEVICE*/
    }

    return 0U;
}

/*!
    \brief      endpoint prepare to transmit data
    \param[in]  udev: pointer to USB core instance
    \param[in]  ep_addr: endpoint address
                  in this parameter:
                    bit0..bit6: endpoint number (0..7)
                    bit7: endpoint direction which can be IN(1) or OUT(0)
    \param[in]  pbuf: transmit buffer address pointer
    \param[in]  len: buffer length
    \param[out] none
    \retval     none
*/
uint32_t usbd_ep_send(usb_core_driver *udev, uint8_t ep_addr, uint8_t *pbuf, uint32_t len)
{
    usb_transc *transc = &udev->dev.transc_in[EP_ID(ep_addr)];

    /* setup the transfer */
    transc->xfer_buf = pbuf;
    transc->xfer_len = len;
    transc->xfer_count = 0U;

    if((uint8_t)USB_USE_DMA == udev->bp.transfer_mode) {
        transc->dma_addr = (uint32_t)pbuf;
    }

    /* start the transfer */
    if(USBHS0_REG_BASE == udev->bp.base_reg) {
#ifdef USE_USBHS0_DEVICE
        (void)hs0_usb_transc_inxfer(udev, transc);
#endif /* USE_USBHS0_DEVICE*/
    } else if (USBHS1_REG_BASE == udev->bp.base_reg) {
#ifdef USE_USBHS1_DEVICE
        (void)hs1_usb_transc_inxfer(udev, transc);
#endif /* USE_USBHS1_DEVICE*/
    }

    return 0U;
}

/*!
    \brief      set an endpoint to STALL status
    \param[in]  udev: pointer to USB core instance
    \param[in]  ep_addr: endpoint address
                  in this parameter:
                    bit0..bit6: endpoint number (0..7)
                    bit7: endpoint direction which can be IN(1) or OUT(0)
    \param[out] none
    \retval     none
*/
uint32_t usbd_ep_stall(usb_core_driver *udev, uint8_t ep_addr)
{
    usb_transc *transc = NULL;

    if(EP_DIR(ep_addr)) {
        transc = &udev->dev.transc_in[EP_ID(ep_addr)];
    } else {
        transc = &udev->dev.transc_out[ep_addr];
    }

    transc->ep_stall = 1U;

    if(USBHS0_REG_BASE == udev->bp.base_reg) {
#ifdef USE_USBHS0_DEVICE
        (void)hs0_usb_transc_stall(udev, transc);
#endif /* USE_USBHS0_DEVICE*/
    } else if (USBHS1_REG_BASE == udev->bp.base_reg) {
#ifdef USE_USBHS1_DEVICE
        (void)hs1_usb_transc_stall(udev, transc);
#endif /* USE_USBHS1_DEVICE*/
    }

    return (0U);
}

/*!
    \brief      clear endpoint STALLed status
    \param[in]  udev: pointer to USB core instance
    \param[in]  ep_addr: endpoint address
                  in this parameter:
                    bit0..bit6: endpoint number (0..7)
                    bit7: endpoint direction which can be IN(1) or OUT(0)
    \param[out] none
    \retval     none
*/
uint32_t usbd_ep_stall_clear(usb_core_driver *udev, uint8_t ep_addr)
{
    usb_transc *transc = NULL;

    if(EP_DIR(ep_addr)) {
        transc = &udev->dev.transc_in[EP_ID(ep_addr)];
    } else {
        transc = &udev->dev.transc_out[ep_addr];
    }

    transc->ep_stall = 0U;

    if(USBHS0_REG_BASE == udev->bp.base_reg) {
#ifdef USE_USBHS0_DEVICE
        (void)hs0_usb_transc_clrstall(udev, transc);
#endif /* USE_USBHS0_DEVICE*/
    } else if (USBHS1_REG_BASE == udev->bp.base_reg) {
#ifdef USE_USBHS1_DEVICE
        (void)hs1_usb_transc_clrstall(udev, transc);
#endif /* USE_USBHS1_DEVICE*/
    }

    return (0U);
}


