/*!
    \file    usbd_core.c
    \brief   USB device mode core functions

    \version 2023-08-01, V1.0.0, HAL firmware for GD32F3x0
*/

/*
    Copyright (c) 2023, GigaDevice Semiconductor Inc. 

    Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this 
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice, 
       this list of conditions and the following disclaimer in the documentation 
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors 
       may be used to endorse or promote products derived from this software without 
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
*/

#include "usbd_core.h"
#include "usbd_enum.h"
#include "drv_usb_hw.h"
extern  char* test0_str ;
extern  char* test1_str ;
extern  char* test2_str ;
extern  char* test3_str ;
extern  char* test4_str ;
extern  char* test5_str ;
extern usb_desc_LANGID usbd_language_id_desc;
extern usb_desc_str serial_string;
extern usb_desc_str manufacturer_string;
extern usb_desc_str product_string;
extern usb_desc_str configure_string;
extern usb_desc_str interface_string;
extern usb_desc_dev hid_dev_desc;
void string_get(void);
 
/*!
    \brief      initializes the USB device-mode stack and load the class driver
    \param[in]  udev: pointer to USB core instance
    \param[in]  core: USB core type
    \param[in]  desc: pointer to USB descriptor
    \param[in]  class_core: class driver
    \param[out] none
    \retval     none
*/
void usbd_init (usb_core_driver *udev, usb_core_enum core, usb_desc *desc, usb_class_core *class_core)
{
    /* configure USB capabilities */
    (void)usb_basic_init (&udev->bp, &udev->regs, core);

    udev->dev.desc = desc;

    /* class callbacks */
    udev->dev.class_core = class_core;

    /* initializes string */
    string_get();

    usb_globalint_disable(&udev->regs);

    /* initializes the USB core*/
    (void)usb_core_init (udev->bp, &udev->regs);

    /* set device disconnect */
    usbd_disconnect (udev);

#ifndef USE_OTG_MODE
    usb_curmode_set(&udev->regs, DEVICE_MODE);
#endif

    /* initializes device mode */
    (void)usb_devcore_init (udev);

    usb_globalint_enable(&udev->regs);

    /* set device connect */
    usbd_connect (udev);

    udev->dev.cur_status = (uint8_t)USBD_DEFAULT;
}

/*!
    \brief      flush the endpoint FIFOs
    \param[in]  udev: pointer to USB core instance
    \param[in]  ep_addr: endpoint address
                  in this parameter:
                    bit0..bit6: endpoint number (0..7)
                    bit7: endpoint direction which can be IN(1) or OUT(0)
    \param[out] none
    \retval     none
*/
uint32_t usbd_fifo_flush (usb_core_driver *udev, uint8_t ep_addr)
{
    if (EP_DIR(ep_addr)) {
        (void)usb_txfifo_flush (&udev->regs, EP_ID(ep_addr));
    } else {
        (void)usb_rxfifo_flush (&udev->regs);
    }

    return (0U);
}

/*!
    \brief      device connect
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     none
*/
void usbd_connect (usb_core_driver *udev)
{
#ifndef USE_OTG_MODE
    /* connect device */
    usb_dev_connect (udev);

    usb_mdelay(3U);
#endif /* USE_OTG_MODE */
}

/*!
    \brief      device disconnect
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     none
*/
void usbd_disconnect (usb_core_driver *udev)
{
#ifndef USE_OTG_MODE
    /* disconnect device for 3ms */
    usb_dev_disconnect (udev);

    usb_mdelay(3U);
#endif /* USE_OTG_MODE */
}


