/*!
    \file    gd32h7xx_hal_uart.c
    \brief   UART driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* uart private function */
/* uart deinit function */
static void _uart_deinit(uint32_t uart_periph);
/* config uart register */
static void _uart_config(uint32_t periph, hal_uart_init_struct *p_uart);
/* get the mask of data bit */
static uint16_t _uart_data_bit_mask_get(hal_uart_dev_struct *uart_dev);
/* handle the transmit complete interrupt */
static void _uart_transmit_complete_interrupt(void *uart_dev);
/* handle the transmit interrupt */
static void _uart_transmit_fifo_empty_interrupt(void *uart_dev);
/* handle the receive interrupt */
static void _uart_receive_interrupt(void *uart_dev);
/* handle the receive interrupt */
static void _uart_received_fifo_full_interrupt(void *uart_dev);
/* handle the idle line detect interrupt */
static void _uart_idle_line_detected_interrupt(void *uart_dev);
/* end ongoing transfer on usart peripheral */
static void _uart_end_transfer(hal_uart_dev_struct *uart_dev);
/* handle the error interrupt */
static void _uart_error_interrupt(void *uart_dev);
/* handle the irda DMA transmit complete process */
static void _uart_dma_tx_complete(void *dma);
/* handle the irda DMA transmit half complete process */
static void _uart_dma_tx_half_complete(void *dma);
/* handle the irda DMA receive complete process */
static void _uart_dma_rx_complete(void *dma);
/* handle the irda DMA receive half complete process */
static void _uart_dma_rx_half_complete(void *dma);
/* handle the irda DMA error process */
static void _uart_dma_error(void *dma);
/* handle the uart DMA Abort complete process */
static void _uart_dma_abort_complete(void *dma);
/* handle the uart DMA Abort only tx complete process */
static void _uart_dma_onlytx_abort_complete(void *dma);
/* handle the uart DMA Abort only rx complete process */
static void _uart_dma_onlyrx_abort_complete(void *dma);

/*!
    \brief      initialize uart asynchronous mode
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  periph: specify which UART is initialized
    \param[in]  p_uart: the initialization data needed to initialize uart
                  baudrate: communication baudrate
                  parity: USART_PARITY_NONE, USART_PARITY_EVEN, USART_PARITY_ODD
                  wordlength: USART_WORDLENGTH_7B, USART_WORDLENGTH_8B, USART_WORDLENGTH_9B, USART_WORDLENGTH_10B
                  stopbits: USART_STOPBITS_1, USART_STOPBITS_0_5, USART_STOPBITS_2, USART_STOPBITS_1_5
                  mode: USART_MODE_RX, USART_MODE_TX, USART_MODE_TX_RX
                  hwflowctl: USART_HWCONTROL_NONE, USART_HWCONTROL_RTS, USART_HWCONTROL_CTS, USART_HWCONTROL_RTS_CTS
                  oversampling: USART_OVERSAMPLING_8, USART_OVERSAMPLING_16
                  samplingmethod: UART_THREE_SAMPLE_BIT, UART_ONE_SAMPLE_BIT
                  timeoutenable: USART_TIMEOUT_DISABLE, USART_TIMEOUT_ENABLE
                  timeoutvalue: 0 - (2^24-1)
                  fifomode:USART_FIFOMODE_DISABLE, USART_FIFOMODE_ENABLE
                  msbfirst: USART_ADVFEATURE_MSBFIRST_DISABLE, USART_ADVFEATURE_MSBFIRST_ENABLE
                  swap: USART_ADVFEATURE_SWAP_DISABLE, USART_ADVFEATURE_SWAP_ENABLE
                  rxpinlevelinvert: USART_ADVFEATURE_RXINV_DISABLE, USART_ADVFEATURE_RXINV_ENABLE
                  txpinlevelinvert: USART_ADVFEATURE_TXINV_DISABLE, USART_ADVFEATURE_TXINV_ENABLE
                  datainvert: USART_ADVFEATURE_DATAINV_DISABLE, USART_ADVFEATURE_DATAINV_ENABLE
                  overrundisable: USART_ADVFEATURE_OVERRUN_ENABLE, USART_ADVFEATURE_OVERRUN_DISABLE
                  dmadisableonrxerror: USART_ADVFEATURE_DMA_ENABLEONRXERROR, USART_ADVFEATURE_DMA_DISABLEONRXERROR
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_init(hal_uart_dev_struct *uart_dev, uint32_t periph, hal_uart_init_struct *p_uart)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check uart pointer and p_uart address */
    if((NULL == uart_dev) || (NULL == p_uart)) {
        HAL_DEBUGE("pointer [uart_dev] or [p_uart] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check periph parameter */
    if((USART0 != periph) && (USART1 != periph) && (USART2 != periph) && (UART3 != periph) && \
        (UART4 != periph) && (USART5 != periph) && (UART6 != periph) && (UART7 != periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    uart_dev->periph = periph;
    uart_dev->init   = *p_uart;

    /* disable uart */
    USART_CTL0(periph) &= ~(USART_CTL0_UEN);

    /* config uart */
    _uart_config(periph, p_uart);

    /* get the data bit mask */
    uart_dev->data_bit_mask = _uart_data_bit_mask_get(uart_dev);

    /* reset the Rx and Tx state */
    uart_dev->tx_state = HAL_UART_STATE_READY;
    uart_dev->rx_state = HAL_UART_STATE_READY;

    /* enable usart */
    USART_CTL0(periph) |= USART_CTL0_UEN;

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize uart rs485 mode
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  periph: specify which UART is initialized
    \param[in]  p_uart: the initialization data needed to initialize uart
                  baudrate: communication baudrate
                  parity: USART_PARITY_NONE, USART_PARITY_EVEN, USART_PARITY_ODD
                  wordlength: USART_WORDLENGTH_7B, USART_WORDLENGTH_8B, USART_WORDLENGTH_9B, USART_WORDLENGTH_10B
                  stopbits: USART_STOPBITS_1, USART_STOPBITS_0_5, USART_STOPBITS_2, USART_STOPBITS_1_5
                  mode: USART_MODE_RX, USART_MODE_TX, USART_MODE_TX_RX
                  hwflowctl: USART_HWCONTROL_NONE, USART_HWCONTROL_RTS, USART_HWCONTROL_CTS, USART_HWCONTROL_RTS_CTS
                  oversampling: USART_OVERSAMPLING_8, USART_OVERSAMPLING_16
                  samplingmethod: UART_THREE_SAMPLE_BIT, UART_ONE_SAMPLE_BIT
                  timeoutenable: USART_TIMEOUT_DISABLE, USART_TIMEOUT_ENABLE
                  timeoutvalue: 0 - (2^24-1)
                  fifomode:USART_FIFOMODE_DISABLE, USART_FIFOMODE_ENABLE
                  msbfirst: USART_ADVFEATURE_MSBFIRST_DISABLE, USART_ADVFEATURE_MSBFIRST_ENABLE
                  swap: USART_ADVFEATURE_SWAP_DISABLE, USART_ADVFEATURE_SWAP_ENABLE
                  rxpinlevelinvert: USART_ADVFEATURE_RXINV_DISABLE, USART_ADVFEATURE_RXINV_ENABLE
                  txpinlevelinvert: USART_ADVFEATURE_TXINV_DISABLE, USART_ADVFEATURE_TXINV_ENABLE
                  datainvert: USART_ADVFEATURE_DATAINV_DISABLE, USART_ADVFEATURE_DATAINV_ENABLE
                  overrundisable: USART_ADVFEATURE_OVERRUN_ENABLE, ENABLE
                  dmadisableonrxerror: DISABLE, USART_ADVFEATURE_OVERRUN_DISABLE
    \param[in] polarity: specify the HAU interrupt source to be enabled
                one or more parameters can be selected which are shown as below:
      \arg        USART_DEP_HIGH: DE signal is active high
      \arg        USART_DEP_LOW: DE signal is active low
    \param[in] assertion_time:driver enable assertion time, must be between 0 and 0x1F
    \param[in] deassertion_time:driver enable deassertion time, must be between 0 and 0x1F
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_rs485_init(hal_uart_dev_struct *uart_dev, uint32_t periph, hal_uart_init_struct *p_uart, \
                       uint32_t polarity, uint32_t assertion_time, uint32_t deassertion_time)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check uart pointer and p_uart address */
    if((NULL == uart_dev) || (NULL == p_uart)) {
        HAL_DEBUGE("pointer [uart_dev] or [p_uart] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check periph parameter */
    if((USART0 != periph) && (USART1 != periph) && (USART2 != periph) && (UART3 != periph) && \
       (UART4 != periph)  && (USART5 != periph) && (UART6 != periph) && (UART7 != periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }

    if((USART_DEP_HIGH != polarity) && (USART_DEP_LOW != polarity)) {
        HAL_DEBUGE("parameter [polarity] value is invalid");
        return HAL_ERR_VAL;
    }

    if(assertion_time > 0x1FU) {
        HAL_DEBUGE("parameter [assertion_time] value is invalid");
        return HAL_ERR_VAL;
    }

    if(deassertion_time > 0x1FU) {
        HAL_DEBUGE("parameter [deassertion_time] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    uart_dev->periph = periph;
    uart_dev->init   = *p_uart;

    /* disable uart */
    USART_CTL0(periph) &= ~(USART_CTL0_UEN);

    /* config uart */
    _uart_config(periph, p_uart);

    /* get the data bit mask */
    uart_dev->data_bit_mask = _uart_data_bit_mask_get(uart_dev);

    /* configure driver enable assertion&de-assertion time */
    USART_CTL0(periph) &= ~(USART_CTL0_DEA | USART_CTL0_DED);
    USART_CTL0(periph) |= (((assertion_time) << 21U) | ((deassertion_time) << 16U));

    /* configure driver enable polarity mode */
    USART_CTL2(periph) &= ~(USART_CTL2_DEP);
    USART_CTL2(periph) |= (polarity);

    /* enable RS485 driver */
    USART_CTL2(periph) |= USART_CTL2_DEM;

    /* reset the Rx and Tx state */
    uart_dev->tx_state = HAL_UART_STATE_READY;
    uart_dev->rx_state = HAL_UART_STATE_READY;

    /* enable usart */
    USART_CTL0(periph) |= USART_CTL0_UEN;

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize uart single wire(half-duplex) mode
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  periph: specify which UART is initialized
    \param[in]  p_uart: the initialization data needed to initialize uart
                  baudrate: communication baudrate
                  parity: USART_PARITY_NONE, USART_PARITY_EVEN, USART_PARITY_ODD
                  wordlength: USART_WORDLENGTH_7B, USART_WORDLENGTH_8B, USART_WORDLENGTH_9B, USART_WORDLENGTH_10B
                  stopbits: USART_STOPBITS_1, USART_STOPBITS_0_5, USART_STOPBITS_2, USART_STOPBITS_1_5
                  mode: USART_MODE_RX, USART_MODE_TX, USART_MODE_TX_RX
                  hwflowctl: USART_HWCONTROL_NONE, USART_HWCONTROL_RTS, USART_HWCONTROL_CTS, USART_HWCONTROL_RTS_CTS
                  oversampling: USART_OVERSAMPLING_8, USART_OVERSAMPLING_16
                  samplingmethod: UART_THREE_SAMPLE_BIT, UART_ONE_SAMPLE_BIT
                  timeoutenable: USART_TIMEOUT_DISABLE, USART_TIMEOUT_ENABLE
                  timeoutvalue: 0 - (2^24-1)
                  fifomode:USART_FIFOMODE_DISABLE, USART_FIFOMODE_ENABLE
                  msbfirst: USART_ADVFEATURE_MSBFIRST_DISABLE, USART_ADVFEATURE_MSBFIRST_ENABLE
                  swap: USART_ADVFEATURE_SWAP_DISABLE, USART_ADVFEATURE_SWAP_ENABLE
                  rxpinlevelinvert: USART_ADVFEATURE_RXINV_DISABLE, USART_ADVFEATURE_RXINV_ENABLE
                  txpinlevelinvert: USART_ADVFEATURE_TXINV_DISABLE, USART_ADVFEATURE_TXINV_ENABLE
                  datainvert: USART_ADVFEATURE_DATAINV_DISABLE, USART_ADVFEATURE_DATAINV_ENABLE
                  overrundisable: USART_ADVFEATURE_OVERRUN_ENABLE, ENABLE
                  dmadisableonrxerror: DISABLE, USART_ADVFEATURE_OVERRUN_DISABLE
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_halfduplex_init(hal_uart_dev_struct *uart_dev, uint32_t periph, hal_uart_init_struct *p_uart)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check uart pointer and p_uart address */
    if((NULL == uart_dev) || (NULL == p_uart)) {
        HAL_DEBUGE("pointer [uart_dev] or [p_uart] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check periph parameter */
    if((USART0 != periph) && (USART1 != periph) && (USART2 != periph) && (UART3 != periph) && \
       (UART4 != periph)  && (USART5 != periph) && (UART6 != periph) && (UART7 != periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    uart_dev->periph = periph;
    uart_dev->init   = *p_uart;

    /* disable uart */
    USART_CTL0(periph) &= ~(USART_CTL0_UEN);

    /* config uart */
    _uart_config(periph, p_uart);

    /* enable single wire(half-duplex) mode */
    USART_CTL2(periph) |= USART_CTL2_HDEN;

    /* get the data bit mask */
    uart_dev->data_bit_mask = _uart_data_bit_mask_get(uart_dev);

    /* reset the Rx and Tx state */
    uart_dev->tx_state = HAL_UART_STATE_READY;
    uart_dev->rx_state = HAL_UART_STATE_READY;

    /* enable usart */
    USART_CTL0(periph) |= USART_CTL0_UEN;

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize uart lin mode
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  periph: specify which UART is initialized
    \param[in]  p_uart: the initialization data needed to initialize uart
                  baudrate: communication baudrate
                  parity: USART_PARITY_NONE, USART_PARITY_EVEN, USART_PARITY_ODD
                  wordlength: USART_WORDLENGTH_7B, USART_WORDLENGTH_8B, USART_WORDLENGTH_9B, USART_WORDLENGTH_10B
                  stopbits: USART_STOPBITS_1, USART_STOPBITS_0_5, USART_STOPBITS_2, USART_STOPBITS_1_5
                  mode: USART_MODE_RX, USART_MODE_TX, USART_MODE_TX_RX
                  hwflowctl: USART_HWCONTROL_NONE, USART_HWCONTROL_RTS, USART_HWCONTROL_CTS, USART_HWCONTROL_RTS_CTS
                  oversampling: USART_OVERSAMPLING_8, USART_OVERSAMPLING_16
                  samplingmethod: UART_THREE_SAMPLE_BIT, UART_ONE_SAMPLE_BIT
                  timeoutenable: USART_TIMEOUT_DISABLE, USART_TIMEOUT_ENABLE
                  timeoutvalue: 0 - (2^24-1)
                  fifomode:USART_FIFOMODE_DISABLE, USART_FIFOMODE_ENABLE
                  msbfirst: USART_ADVFEATURE_MSBFIRST_DISABLE, USART_ADVFEATURE_MSBFIRST_ENABLE
                  swap: USART_ADVFEATURE_SWAP_DISABLE, USART_ADVFEATURE_SWAP_ENABLE
                  rxpinlevelinvert: USART_ADVFEATURE_RXINV_DISABLE, USART_ADVFEATURE_RXINV_ENABLE
                  txpinlevelinvert: USART_ADVFEATURE_TXINV_DISABLE, USART_ADVFEATURE_TXINV_ENABLE
                  datainvert: USART_ADVFEATURE_DATAINV_DISABLE, USART_ADVFEATURE_DATAINV_ENABLE
                  overrundisable: USART_ADVFEATURE_OVERRUN_ENABLE, ENABLE
                  dmadisableonrxerror: DISABLE, USART_ADVFEATURE_OVERRUN_DISABLE
    \param[in] break_length:specify lin break detect length
                one or more parameters can be selected which are shown as below:
      \arg        USART_LBLEN_10B: 10 bits break detection
      \arg        USART_LBLEN_11B: 11 bits break detection
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_lin_init(hal_uart_dev_struct *uart_dev, uint32_t periph, hal_uart_init_struct *p_uart, \
                     uint32_t break_length)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check uart pointer and p_uart address */
    if((NULL == uart_dev) || (NULL == p_uart)) {
        HAL_DEBUGE("pointer [uart_dev] or [p_uart] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check periph parameter */
    if((USART0 != periph) && (USART1 != periph) && (USART2 != periph) && (UART3 != periph) && (UART4 != periph) && \
       (USART5 != periph) && (UART6 != periph) && (UART7 != periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }

    /* check periph parameter */
    if((USART_LBLEN_10B != break_length) && (USART_LBLEN_11B != break_length)) {
        HAL_DEBUGE("parameter [break_length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    uart_dev->periph = periph;
    uart_dev->init   = *p_uart;

    /* disable uart */
    USART_CTL0(periph) &= ~(USART_CTL0_UEN);

    /* config uart */
    _uart_config(periph, p_uart);

    /* LIN break detection length */
    USART_CTL1(periph) &= ~(USART_CTL1_LBLEN);
    USART_CTL1(periph) |= (break_length);

    /* enable LIN mode */
    USART_CTL1(periph) |= USART_CTL1_LMEN;

    /* get the data bit mask */
    uart_dev->data_bit_mask = _uart_data_bit_mask_get(uart_dev);

    /* reset the Rx and Tx state */
    uart_dev->tx_state = HAL_UART_STATE_READY;
    uart_dev->rx_state = HAL_UART_STATE_READY;

    /* enable usart */
    USART_CTL0(periph) |= USART_CTL0_UEN;

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize uart multiprocessor communication mode
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  periph: specify which UART is initialized
    \param[in]  p_uart: the initialization data needed to initialize uart
                  baudrate: communication baudrate
                  parity: USART_PARITY_NONE, USART_PARITY_EVEN, USART_PARITY_ODD
                  wordlength: USART_WORDLENGTH_7B, USART_WORDLENGTH_8B, USART_WORDLENGTH_9B, USART_WORDLENGTH_10B
                  stopbits: USART_STOPBITS_1, USART_STOPBITS_0_5, USART_STOPBITS_2, USART_STOPBITS_1_5
                  mode: USART_MODE_RX, USART_MODE_TX, USART_MODE_TX_RX
                  hwflowctl: USART_HWCONTROL_NONE, USART_HWCONTROL_RTS, USART_HWCONTROL_CTS, USART_HWCONTROL_RTS_CTS
                  oversampling: USART_OVERSAMPLING_8, USART_OVERSAMPLING_16
                  samplingmethod: UART_THREE_SAMPLE_BIT, UART_ONE_SAMPLE_BIT
                  timeoutenable: USART_TIMEOUT_DISABLE, USART_TIMEOUT_ENABLE
                  timeoutvalue: 0 - (2^24-1)
                  fifomode:USART_FIFOMODE_DISABLE, USART_FIFOMODE_ENABLE
                  msbfirst: USART_ADVFEATURE_MSBFIRST_DISABLE, USART_ADVFEATURE_MSBFIRST_ENABLE
                  swap: USART_ADVFEATURE_SWAP_DISABLE, USART_ADVFEATURE_SWAP_ENABLE
                  rxpinlevelinvert: USART_ADVFEATURE_RXINV_DISABLE, USART_ADVFEATURE_RXINV_ENABLE
                  txpinlevelinvert: USART_ADVFEATURE_TXINV_DISABLE, USART_ADVFEATURE_TXINV_ENABLE
                  datainvert: USART_ADVFEATURE_DATAINV_DISABLE, USART_ADVFEATURE_DATAINV_ENABLE
                  overrundisable: USART_ADVFEATURE_OVERRUN_ENABLE, ENABLE
                  dmadisableonrxerror: DISABLE, USART_ADVFEATURE_OVERRUN_DISABLE
    \param[in] wakeup_method:specify multiprocessor WakeUp Method
                one or more parameters can be selected which are shown as below:
      \arg        USART_WM_IDLE: idle line
      \arg        USART_WM_ADDR: address match
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_multiprocessor_init(hal_uart_dev_struct *uart_dev, uint32_t periph, hal_uart_init_struct *p_uart,
                                uint32_t wakeup_method)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check uart pointer and p_uart address */
    if((NULL == uart_dev) || (NULL == p_uart)) {
        HAL_DEBUGE("pointer [uart_dev] or [p_uart] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check periph parameter */
    if((USART0 != periph) && (USART1 != periph) && (USART2 != periph) && (UART3 != periph) && (UART4 != periph) &&
       (USART5 != periph) && (UART6 != periph) && (UART7 != periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }

    /* check periph parameter */
    if((USART_WM_IDLE != wakeup_method) && (USART_WM_ADDR != wakeup_method)) {
        HAL_DEBUGE("parameter [wakeup_method] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    uart_dev->periph = periph;
    uart_dev->init   = *p_uart;

    /* disable uart */
    USART_CTL0(periph) &= ~(USART_CTL0_UEN);

    /* config uart */
    _uart_config(periph, p_uart);

    /* multiprocessor mode configure */
    USART_CTL0(periph) &= ~(USART_CTL0_WM);
    USART_CTL0(periph) |= wakeup_method;

    /* enable mute mode */
    USART_CTL0(periph) |= USART_CTL0_MEN;

    /* get the data bit mask */
    uart_dev->data_bit_mask = _uart_data_bit_mask_get(uart_dev);

    /* reset the Rx and Tx state */
    uart_dev->tx_state = HAL_UART_STATE_READY;
    uart_dev->rx_state = HAL_UART_STATE_READY;

    /* enable usart */
    USART_CTL0(periph) |= USART_CTL0_UEN;

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize uart modbus mode
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  periph: specify which UART is initialized
    \param[in]  p_uart: the initialization data needed to initialize uart
                  baudrate: communication baudrate
                  parity: USART_PARITY_NONE, USART_PARITY_EVEN, USART_PARITY_ODD
                  wordlength: USART_WORDLENGTH_7B, USART_WORDLENGTH_8B, USART_WORDLENGTH_9B, USART_WORDLENGTH_10B
                  stopbits: USART_STOPBITS_1, USART_STOPBITS_0_5, USART_STOPBITS_2, USART_STOPBITS_1_5
                  mode: USART_MODE_RX, USART_MODE_TX, USART_MODE_TX_RX
                  hwflowctl: USART_HWCONTROL_NONE, USART_HWCONTROL_RTS, USART_HWCONTROL_CTS, USART_HWCONTROL_RTS_CTS
                  oversampling: USART_OVERSAMPLING_8, USART_OVERSAMPLING_16
                  samplingmethod: UART_THREE_SAMPLE_BIT, UART_ONE_SAMPLE_BIT
                  timeoutenable: USART_TIMEOUT_DISABLE, USART_TIMEOUT_ENABLE
                  timeoutvalue: 0 - (2^24-1)
                  msbfirst: USART_ADVFEATURE_MSBFIRST_DISABLE, USART_ADVFEATURE_MSBFIRST_ENABLE
                  swap: USART_ADVFEATURE_SWAP_DISABLE, USART_ADVFEATURE_SWAP_ENABLE
                  rxpinlevelinvert: USART_ADVFEATURE_RXINV_DISABLE, USART_ADVFEATURE_RXINV_ENABLE
                  txpinlevelinvert: USART_ADVFEATURE_TXINV_DISABLE, USART_ADVFEATURE_TXINV_ENABLE
                  datainvert: USART_ADVFEATURE_DATAINV_DISABLE, USART_ADVFEATURE_DATAINV_ENABLE
                  overrundisable: USART_ADVFEATURE_OVERRUN_ENABLE, USART_ADVFEATURE_OVERRUN_DISABLE
                  dmadisableonrxerror: USART_ADVFEATURE_DMA_ENABLEONRXERROR, USART_ADVFEATURE_DMA_DISABLEONRXERROR
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_modbus_init(hal_uart_dev_struct *uart_dev, uint32_t periph, hal_uart_init_struct *p_uart)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check uart pointer and p_uart address */
    if((NULL == uart_dev) || (NULL == p_uart)) {
        HAL_DEBUGE("pointer [uart_dev] or [p_uart] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check periph parameter */
    if((USART0 != periph) && (USART1 != periph) && (USART2 != periph) && (UART3 != periph) && (UART4 != periph) && \
       (USART5 != periph) && (UART6 != periph) && (UART7 != periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    uart_dev->periph = periph;
    uart_dev->init   = *p_uart;

    /* disable uart */
    USART_CTL0(periph) &= ~(USART_CTL0_UEN);

    /* config uart */
    _uart_config(periph, p_uart);

    /* config receive block tail char */
    hals_usart_address_0_detection_mode_config(periph, USART_ADDM0_FULLBIT);
    hals_usart_address_0_config(periph, '\n');
    hals_usart_address_0_match_mode_enable(periph);

    /* get the data bit mask */
    uart_dev->data_bit_mask = _uart_data_bit_mask_get(uart_dev);

    /* reset the Rx and Tx state */
    uart_dev->tx_state = HAL_UART_STATE_READY;
    uart_dev->rx_state = HAL_UART_STATE_READY;

    /* enable usart */
    USART_CTL0(periph) |= USART_CTL0_UEN;

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure address and address length of the USART
    \param[in]  uart_dev: uart device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  address0: 0x00-0xFF, address 0 of USART terminal
    \param[in]  address0length: USART_ADDRESS0_DETECT_4B, USART_ADDRESS0_DETECT_7B
    \param[in]  address1: 0x00-0xFF, address 1 of USART terminal
    \param[in]  address1length: USART_ADDRESS1_DETECT_4B, USART_ADDRESS1_DETECT_7B
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_address_config(hal_uart_dev_struct *uart_dev, uint8_t address0, uint32_t address0length, \
                                uint8_t address1, uint32_t address1length)
{
    uint32_t reg_temp = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and p_irq address */
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* save CTL0 register */
    reg_temp = USART_CTL0(uart_dev->periph);
    /* disable USART */
    USART_CTL0(uart_dev->periph) &= ~(USART_CTL0_UEN | USART_CTL0_REN);

    /* configure CTL1 register */
    USART_CTL1(uart_dev->periph) &= ~(USART_CTL1_ADDR0 | USART_CTL1_ADDM0);
    USART_CTL1(uart_dev->periph) |= ((((uint32_t)address0) << 24U) | address0length);
    USART_CTL1(uart_dev->periph) |= USART_CTL1_AMEN0;

    /* configure CTL2 register */
    USART_CTL2(uart_dev->periph) &= ~(USART_CTL2_ADDR1 | USART_CTL2_ADDM1);
    USART_CTL2(uart_dev->periph) |= ((((uint32_t)address1) << 24U) | address1length);
    USART_CTL2(uart_dev->periph) |= USART_CTL2_AMEN1;

    /* resume CTL0 register */
    USART_CTL0(uart_dev->periph) = reg_temp;

    return HAL_ERR_NONE;
}

/*!
   \brief      set wakeup from sleep mode interrupt flag selection
   \param[in]  uart_dev: uart device information structure
                 the structure is not necessary to be reconfigured after structure initialization,
                 the structure parameters altering is automatically configured by core
   \param[in]  wake_mode: wakeup mode selection
                only one parameter can be selected which is shown as below:
      \arg      USART_WUM_ADDR: WUF active on address match
      \arg      USART_WUM_STARTB: WUF active on start bit detection
      \arg      USART_WUM_RBNE: WUF active on receive buffer not empty
   \param[out] none
   \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_deepsleep_wakeupsource_config(hal_uart_dev_struct *uart_dev, uint32_t wake_mode)
{
    uint32_t tickstart;
    int32_t  ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and p_irq address */
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* process locked */
    HAL_LOCK(uart_dev);

    if(HAL_UART_STATE_BUSY == uart_dev->tx_state) {
        HAL_DEBUGE("UART is busy");
        ret                   = HAL_ERR_BUSY;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_BUSY;
    } else {
        uart_dev->tx_state = HAL_UART_STATE_BUSY;

        /* disable usart */
        USART_CTL0(uart_dev->periph) &= ~(USART_CTL0_UEN);

        /* configure the USART wakeup mode from deep-sleep mode */
        USART_CTL2(uart_dev->periph) &= ~(USART_CTL2_WUM);
        USART_CTL2(uart_dev->periph) |= USART_CTL2_WUM & (wake_mode);

        /* enable usart */
        USART_CTL0(uart_dev->periph) |= USART_CTL0_UEN;

        /* init tickstart for timeout management */
        tickstart = hal_sys_basetick_count_get();

        /* wait until reack flag is set */
        while(RESET == hals_usart_flag_get(uart_dev->periph, USART_FLAG_REA)) {
            if(SET == hal_sys_basetick_timeout_check(tickstart, 0xFFFFU)) {
                /* reset the state */
                HAL_DEBUGE("wait until reack flag timeout");
                ret                   = HAL_ERR_TIMEOUT;
                uart_dev->tx_state    = HAL_UART_STATE_READY;
                uart_dev->error_state = (uint16_t)HAL_UART_ERROR_TTO;
                break;
            } else {
                /* do nothing */
            }
        }

        if(HAL_ERR_NONE == ret) {
            /* enable uart wakeup mode */
            USART_CTL0(uart_dev->periph) |= USART_CTL0_UESM;

            /* EXTI configuration */
            if(USART0 == uart_dev->periph) {
                hal_exti_internal_init(EXTI_LINE_27_USART0_WAKEUP, EXTI_INTERRUPT_TRIG_RISING);
            } else if(USART1 == uart_dev->periph) {
                hal_exti_internal_init(EXTI_LINE_28_USART1_WAKEUP, EXTI_INTERRUPT_TRIG_RISING);
            } else if(USART2 == uart_dev->periph) {
                hal_exti_internal_init(EXTI_LINE_29_USART2_WAKEUP, EXTI_INTERRUPT_TRIG_RISING);
            } else if(USART5 == uart_dev->periph) {
                hal_exti_internal_init(EXTI_LINE_30_USART5_WAKEUP, EXTI_INTERRUPT_TRIG_RISING);
            } else {
                /* do nothing */
            }

            /* initialize the uart state */
            uart_dev->tx_state = HAL_UART_STATE_READY;
        } else {
            /* do nothing */
        }
    }

    HAL_UNLOCK(uart_dev);

    return ret;
}

/*!
    \brief      initialize the uart structure with the default values
                note: this function must be called after the structure is created
    \param[in]  hal_struct_type: type of uart structure for initialization
                only one parameters can be selected which are shown as below:
      \arg        HAL_UART_INIT_STRUCT: initialization structure
      \arg        HAL_UART_DEV_STRUCT: device information structure
      \arg        HAL_UART_IRQ_INIT_STRUCT: interrupt callback initialization structure
    \param[out] p_struct: pointer to UART structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_struct_init(hal_uart_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_UART_INIT_STRUCT:
        /* initialize uart initialization structure with the default values */
        ((hal_uart_init_struct *)p_struct)->baudrate            = 115200U;
        ((hal_uart_init_struct *)p_struct)->wordlength          = USART_WORDLENGTH_8B;
        ((hal_uart_init_struct *)p_struct)->stopbits            = USART_STOPBITS_1;
        ((hal_uart_init_struct *)p_struct)->parity              = USART_PARITY_NONE;
        ((hal_uart_init_struct *)p_struct)->mode                = USART_MODE_TX_RX;
        ((hal_uart_init_struct *)p_struct)->hwflowctl           = USART_HWCONTROL_NONE;
        ((hal_uart_init_struct *)p_struct)->oversampling        = USART_OVERSAMPLING_16;
        ((hal_uart_init_struct *)p_struct)->samplingmethod      = USART_THREE_BITS_SAMPLING;
        ((hal_uart_init_struct *)p_struct)->timeoutenable       = USART_TIMEOUT_DISABLE;
        ((hal_uart_init_struct *)p_struct)->timeoutvalue        = 0U;
        ((hal_uart_init_struct *)p_struct)->fifomode            = USART_FIFOMODE_DISABLE;
        ((hal_uart_init_struct *)p_struct)->txthreshold         = USART_TXFIFO_THRESHOLD_1_8;
        ((hal_uart_init_struct *)p_struct)->rxthreshold         = USART_RXFIFO_THRESHOLD_1_8;
        ((hal_uart_init_struct *)p_struct)->txpinlevelinvert    = USART_ADVFEATURE_TXINV_DISABLE;
        ((hal_uart_init_struct *)p_struct)->rxpinlevelinvert    = USART_ADVFEATURE_RXINV_DISABLE;
        ((hal_uart_init_struct *)p_struct)->datainvert          = USART_ADVFEATURE_DATAINV_DISABLE;
        ((hal_uart_init_struct *)p_struct)->swap                = USART_ADVFEATURE_SWAP_DISABLE;
        ((hal_uart_init_struct *)p_struct)->overrundisable      = USART_ADVFEATURE_OVERRUN_ENABLE;
        ((hal_uart_init_struct *)p_struct)->dmadisableonrxerror = USART_ADVFEATURE_DMA_ENABLEONRXERROR;
        ((hal_uart_init_struct *)p_struct)->msbfirst            = USART_ADVFEATURE_MSBFIRST_DISABLE;
        break;
    case HAL_UART_DEV_STRUCT:
        /* initialize uart device information structure with the default values */
        ((hal_uart_dev_struct *)p_struct)->periph                              = 0U;
        ((hal_uart_dev_struct *)p_struct)->uart_irq.transmit_fifo_empty_handle = NULL;
        ((hal_uart_dev_struct *)p_struct)->uart_irq.transmit_complete_handle   = NULL;
        ((hal_uart_dev_struct *)p_struct)->uart_irq.receive_complete_handle    = NULL;
        ((hal_uart_dev_struct *)p_struct)->uart_irq.receive_fifo_full_handle   = NULL;
        ((hal_uart_dev_struct *)p_struct)->uart_irq.receive_timeout_handle     = NULL;
        ((hal_uart_dev_struct *)p_struct)->uart_irq.error_handle               = NULL;
        ((hal_uart_dev_struct *)p_struct)->uart_irq.wakeup_handle              = NULL;
        ((hal_uart_dev_struct *)p_struct)->uart_irq.idle_line_detected_handle  = NULL;
        ((hal_uart_dev_struct *)p_struct)->uart_irq.address_match_handle       = NULL;
        ((hal_uart_dev_struct *)p_struct)->uart_irq.lin_break_detected_handle  = NULL;
        ((hal_uart_dev_struct *)p_struct)->uart_irq.cts_change_handle          = NULL;
        ((hal_uart_dev_struct *)p_struct)->p_dma_rx                            = NULL;
        ((hal_uart_dev_struct *)p_struct)->p_dma_tx                            = NULL;
        ((hal_uart_dev_struct *)p_struct)->txbuffer.buffer                     = NULL;
        ((hal_uart_dev_struct *)p_struct)->txbuffer.length                     = 0U;
        ((hal_uart_dev_struct *)p_struct)->txbuffer.remain                     = 0U;
        ((hal_uart_dev_struct *)p_struct)->rxbuffer.buffer                     = NULL;
        ((hal_uart_dev_struct *)p_struct)->rxbuffer.length                     = 0U;
        ((hal_uart_dev_struct *)p_struct)->rxbuffer.remain                     = 0U;
        ((hal_uart_dev_struct *)p_struct)->data_bit_mask                       = 0U;
        ((hal_uart_dev_struct *)p_struct)->last_error                          = (uint16_t)HAL_UART_ERROR_NONE;
        ((hal_uart_dev_struct *)p_struct)->error_state                         = (uint16_t)HAL_UART_ERROR_NONE;
        ((hal_uart_dev_struct *)p_struct)->tx_state                            = HAL_UART_STATE_RESET;
        ((hal_uart_dev_struct *)p_struct)->rx_state                            = HAL_UART_STATE_RESET;
        ((hal_uart_dev_struct *)p_struct)->transmit_half_complete_callback     = NULL;
        ((hal_uart_dev_struct *)p_struct)->transmit_complete_callback          = NULL;
        ((hal_uart_dev_struct *)p_struct)->transmit_fifo_empty_callback        = NULL;
        ((hal_uart_dev_struct *)p_struct)->receive_half_complete_callback      = NULL;
        ((hal_uart_dev_struct *)p_struct)->receive_complete_callback           = NULL;
        ((hal_uart_dev_struct *)p_struct)->receive_fifo_full_callback          = NULL;
        ((hal_uart_dev_struct *)p_struct)->idle_receive_callback               = NULL;
        ((hal_uart_dev_struct *)p_struct)->wakeup_callback                     = NULL;
        ((hal_uart_dev_struct *)p_struct)->error_callback                      = NULL;
        ((hal_uart_dev_struct *)p_struct)->mutex                               = HAL_MUTEX_UNLOCKED;
        ((hal_uart_dev_struct *)p_struct)->priv                                = NULL;
        break;
    case HAL_UART_IRQ_INIT_STRUCT:
        /* initialize interrupt callback structure with the default values */
        ((hal_uart_irq_struct *)p_struct)->transmit_fifo_empty_handle = NULL;
        ((hal_uart_irq_struct *)p_struct)->transmit_complete_handle   = NULL;
        ((hal_uart_irq_struct *)p_struct)->receive_complete_handle    = NULL;
        ((hal_uart_irq_struct *)p_struct)->receive_fifo_full_handle   = NULL;
        ((hal_uart_irq_struct *)p_struct)->receive_complete_handle    = NULL;
        ((hal_uart_irq_struct *)p_struct)->error_handle               = NULL;
        ((hal_uart_irq_struct *)p_struct)->wakeup_handle              = NULL;
        ((hal_uart_irq_struct *)p_struct)->idle_line_detected_handle  = NULL;
        ((hal_uart_irq_struct *)p_struct)->address_match_handle       = NULL;
        ((hal_uart_irq_struct *)p_struct)->lin_break_detected_handle  = NULL;
        ((hal_uart_irq_struct *)p_struct)->cts_change_handle          = NULL;
        break;
    case HAL_UART_IRQ_USER_STRUCT:
        /* initialize interrupt callback structure with the default values */
        ((hal_uart_irq_user_callback_struct *)p_struct)->transmit_half_complete_func = NULL;
        ((hal_uart_irq_user_callback_struct *)p_struct)->transmit_complete_func = NULL;
        ((hal_uart_irq_user_callback_struct *)p_struct)->transmit_fifo_empty_func = NULL;
        ((hal_uart_irq_user_callback_struct *)p_struct)->abort_tx_complete_func = NULL;
        ((hal_uart_irq_user_callback_struct *)p_struct)->receive_half_complete_func = NULL;
        ((hal_uart_irq_user_callback_struct *)p_struct)->receive_complete_func = NULL;
        ((hal_uart_irq_user_callback_struct *)p_struct)->receive_fifo_full_func = NULL;
        ((hal_uart_irq_user_callback_struct *)p_struct)->abort_rx_complete_func = NULL;
        ((hal_uart_irq_user_callback_struct *)p_struct)->abort_complete_func = NULL;
        ((hal_uart_irq_user_callback_struct *)p_struct)->idle_receive_func = NULL;
        ((hal_uart_irq_user_callback_struct *)p_struct)->error_func = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      deinitialize uart
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_deinit(hal_uart_dev_struct *uart_dev)
{
    uint32_t periph;
    int32_t  ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(uart_dev);

    periph = uart_dev->periph;

    if((USART0 == periph) || (USART1 == periph) || (USART2 == periph) || (UART3 == periph) || \
       (UART4 == periph)  || (USART5 == periph) || (UART6 == periph)  || (UART7 == periph)) {
        /* deinitialize the periph and the device information structure */
        _uart_deinit(periph);
        uart_dev->last_error  = (uint16_t)HAL_UART_ERROR_NONE;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_NONE;
        uart_dev->tx_state    = HAL_UART_STATE_RESET;
        uart_dev->rx_state    = HAL_UART_STATE_RESET;
    } else {
        HAL_DEBUGE("parameter [uart_dev->periph] value is invalid");
        ret = HAL_ERR_VAL;
    }

    HAL_UNLOCK(uart_dev);

    return ret;
}

/*!
    \brief      uart interrupt handler content function,which is merely used in USART_IRQHandler
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_irq(hal_uart_dev_struct *uart_dev)
{
    __IO uint32_t errorflags = 0U;
    __IO hal_exti_line_enum linex = EXTI_0;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* EXTI line number*/
    if(USART0 == uart_dev->periph) {
        linex = EXTI_27;
    } else if(USART1 == uart_dev->periph) {
        linex = EXTI_28;
    } else if(USART2 == uart_dev->periph) {
        linex = EXTI_29;
    } else if(USART5 == uart_dev->periph) {
        linex = EXTI_30;
    } else {
        /* do nothing */
    }

    /* if no error occurs */
    errorflags = (USART_STAT(uart_dev->periph) & (uint32_t)(USART_STAT_PERR | USART_STAT_FERR | \
                                                            USART_STAT_ORERR | USART_STAT_NERR));
    if(0U == errorflags) {
        /* if uart is in receiver mode */
        if(RESET != hals_usart_interrupt_flag_get(uart_dev->periph, USART_INT_FLAG_RBNE)) {
            hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_RBNE);
            if(NULL != uart_dev->uart_irq.receive_complete_handle) {
                uart_dev->uart_irq.receive_complete_handle(uart_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* if some errors occur */
    if(0U != errorflags) {
        /* uart parity error interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(uart_dev->periph, USART_INT_FLAG_PERR)) {
            hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_PERR);
            uart_dev->error_state |= (uint16_t)HAL_UART_ERROR_PERR;
            uart_dev->last_error = (uint16_t)HAL_UART_ERROR_PERR;
        } else {
            /* do nothing */
        }

        /* uart frame error interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(uart_dev->periph, USART_INT_FLAG_ERR_FERR)) {
            hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_ERR_FERR);
            uart_dev->error_state |= (uint16_t)HAL_UART_ERROR_FERR;
            uart_dev->last_error = (uint16_t)HAL_UART_ERROR_FERR;
        } else {
            /* do nothing */
        }

        /* uart noise error interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(uart_dev->periph, USART_INT_FLAG_ERR_NERR)) {
            hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_ERR_NERR);
            uart_dev->error_state |= (uint16_t)HAL_UART_ERROR_NERR;
            uart_dev->last_error = (uint16_t)HAL_UART_ERROR_NERR;
        } else {
            /* do nothing */
        }

        /* uart over-run interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(uart_dev->periph, USART_INT_FLAG_ERR_ORERR)) {
            hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_ERR_ORERR);
            uart_dev->error_state |= (uint16_t)HAL_UART_ERROR_ORERR;
            uart_dev->last_error = (uint16_t)HAL_UART_ERROR_ORERR;
        } else {
            /* do nothing */
        }

        /* check whether error state is none or not */
        if(HAL_UART_ERROR_NONE != uart_dev->error_state) {
            /* if UART is in receiver mode */
            if(RESET != hals_usart_interrupt_flag_get(uart_dev->periph, USART_INT_FLAG_RBNE)) {
                hals_usart_interrupt_flag_get(uart_dev->periph, USART_INT_FLAG_RBNE);
                if(NULL != uart_dev->uart_irq.receive_complete_handle) {
                    uart_dev->uart_irq.receive_complete_handle(uart_dev);
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            if(NULL != uart_dev->uart_irq.error_handle) {
                uart_dev->uart_irq.error_handle(uart_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* uart receiver timeout interrupt occurred */
    if(RESET != hals_usart_interrupt_flag_get(uart_dev->periph, USART_INT_FLAG_RT)) {
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_RT);
        if(NULL != uart_dev->uart_irq.receive_timeout_handle) {
            uart_dev->uart_irq.receive_timeout_handle(uart_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* wakeup from deepsleep mode interrupt handle */
    if(RESET != hals_usart_interrupt_flag_get(uart_dev->periph, USART_INT_FLAG_WU)) {
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_WU);
        hals_exti_interrupt_flag_clear(linex);
        if(NULL != uart_dev->uart_irq.wakeup_handle) {
            uart_dev->uart_irq.wakeup_handle(uart_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* uart in mode transmitter */
    if(RESET != hals_usart_interrupt_flag_get(uart_dev->periph, USART_INT_FLAG_TBE)) {
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_TBE);
        if(NULL != uart_dev->uart_irq.transmit_fifo_empty_handle) {
            uart_dev->uart_irq.transmit_fifo_empty_handle(uart_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* transmission complete interrupt handle */
    if(RESET != hals_usart_interrupt_flag_get(uart_dev->periph, USART_INT_FLAG_TC)) {
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_TC);
        if(NULL != uart_dev->uart_irq.transmit_complete_handle) {
            uart_dev->uart_irq.transmit_complete_handle(uart_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* idle line detected interrupt handle */
    if(RESET != hals_usart_interrupt_flag_get(uart_dev->periph, USART_INT_FLAG_IDLE)) {
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_IDLE);
        if(NULL != uart_dev->uart_irq.idle_line_detected_handle) {
            uart_dev->uart_irq.idle_line_detected_handle(uart_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* address match interrupt handle */
    if(RESET != hals_usart_interrupt_flag_get(uart_dev->periph, USART_INT_FLAG_AM0)) {
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_AM0);
        if(NULL != uart_dev->uart_irq.address_match_handle) {
            uart_dev->uart_irq.address_match_handle(uart_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* LIN mode interrupt handle */
    if(RESET != hals_usart_interrupt_flag_get(uart_dev->periph, USART_INT_FLAG_LBD)) {
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_LBD);
        if(NULL != uart_dev->uart_irq.lin_break_detected_handle) {
            uart_dev->uart_irq.lin_break_detected_handle(uart_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* hardware flow mode interrupt handle */
    if(RESET != hals_usart_interrupt_flag_get(uart_dev->periph, USART_INT_FLAG_CTS)) {
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_CTS);
        if(NULL != uart_dev->uart_irq.cts_change_handle) {
            uart_dev->uart_irq.cts_change_handle(uart_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* uart rx Fifo full*/
    if(RESET != hals_usart_interrupt_flag_get(uart_dev->periph, USART_INT_FLAG_RFF)) {
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_RFF);
        if(NULL != uart_dev->uart_irq.receive_fifo_full_handle) {
            uart_dev->uart_irq.receive_fifo_full_handle(uart_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  uart_dev: uart device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to uart interrupt callback functions structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg        NULL: The corresponding callback mechanism is out of use, and
                    disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_irq_handle_set(hal_uart_dev_struct *uart_dev, hal_uart_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and p_irq address */
    if((NULL == uart_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [uart_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set user-defined transmit complete interrupt callback */
    if(NULL != p_irq->transmit_complete_handle) {
        uart_dev->uart_irq.transmit_complete_handle = p_irq->transmit_complete_handle;
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_TC);
    } else {
        uart_dev->uart_irq.transmit_complete_handle = NULL;
        hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TC);
    }

    /* set user-defined receive complete interrupt callback */
    if(NULL != p_irq->receive_complete_handle) {
        uart_dev->uart_irq.receive_complete_handle = p_irq->receive_complete_handle;
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_RBNE);
    } else {
        uart_dev->uart_irq.receive_complete_handle = NULL;
        hals_usart_interrupt_disable(uart_dev->periph, USART_INT_RBNE);
    }

    /* set user-defined error interrupt callback */
    if(NULL != p_irq->error_handle) {
        uart_dev->uart_irq.error_handle = p_irq->error_handle;
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_ERR);
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_PERR);
    } else {
        uart_dev->uart_irq.error_handle = NULL;
        hals_usart_interrupt_disable(uart_dev->periph, USART_INT_ERR);
        hals_usart_interrupt_disable(uart_dev->periph, USART_INT_PERR);
    }

    /* set user-defined receive timeout interrupt callback */
    if(NULL != p_irq->receive_timeout_handle) {
        uart_dev->uart_irq.receive_timeout_handle = p_irq->receive_timeout_handle;
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_RT);
    } else {
        uart_dev->uart_irq.receive_timeout_handle = NULL;
        hals_usart_interrupt_disable(uart_dev->periph, USART_INT_RT);
    }

    /* set user-defined idle line detected interrupt callback */
    if(NULL != p_irq->idle_line_detected_handle) {
        uart_dev->uart_irq.idle_line_detected_handle = p_irq->idle_line_detected_handle;
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_IDLE);
    } else {
        uart_dev->uart_irq.idle_line_detected_handle = NULL;
        hals_usart_interrupt_disable(uart_dev->periph, USART_INT_IDLE);
    }

    /* set user-defined address match interrupt callback */
    if(NULL != p_irq->address_match_handle) {
        uart_dev->uart_irq.address_match_handle = p_irq->address_match_handle;
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_AM0);
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_AM1);
    } else {
        uart_dev->uart_irq.address_match_handle = NULL;
        hals_usart_interrupt_disable(uart_dev->periph, USART_INT_AM0);
        hals_usart_interrupt_disable(uart_dev->periph, USART_INT_AM1);
    }

    /* set user-defined CTS change interrupt callback */
    if(NULL != p_irq->cts_change_handle) {
        uart_dev->uart_irq.cts_change_handle = p_irq->cts_change_handle;
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_CTS);
    } else {
        uart_dev->uart_irq.cts_change_handle = NULL;
        hals_usart_interrupt_disable(uart_dev->periph, USART_INT_CTS);
    }

    /* set user-defined LIN break detected interrupt callback */
    if(NULL != p_irq->lin_break_detected_handle) {
        uart_dev->uart_irq.lin_break_detected_handle = p_irq->lin_break_detected_handle;
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_LBD);
    } else {
        uart_dev->uart_irq.lin_break_detected_handle = NULL;
        hals_usart_interrupt_disable(uart_dev->periph, USART_INT_LBD);
    }

    /* set user-defined transmit ready interrupt callback */
    if(NULL != p_irq->transmit_fifo_empty_handle) {
        uart_dev->uart_irq.transmit_fifo_empty_handle = p_irq->transmit_fifo_empty_handle;
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_TBE);
    } else {
        uart_dev->uart_irq.transmit_fifo_empty_handle = NULL;
        hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TBE);
    }

    /* set user-defined wakeup interrupt callback */
    if(NULL != p_irq->wakeup_handle) {
        uart_dev->uart_irq.wakeup_handle = p_irq->wakeup_handle;

        /* configuration EXTI line  */
        if(USART0 == uart_dev->periph) {
            hal_exti_internal_init(EXTI_LINE_27_USART0_WAKEUP, EXTI_INTERRUPT_TRIG_BOTH);
        } else if(USART1 == uart_dev->periph) {
            hal_exti_internal_init(EXTI_LINE_28_USART1_WAKEUP, EXTI_INTERRUPT_TRIG_BOTH);
        } else if(USART2 == uart_dev->periph) {
            hal_exti_internal_init(EXTI_LINE_29_USART2_WAKEUP, EXTI_INTERRUPT_TRIG_BOTH);
        } else if(USART5 == uart_dev->periph) {
            hal_exti_internal_init(EXTI_LINE_30_USART5_WAKEUP, EXTI_INTERRUPT_TRIG_BOTH);
        } else {
            /* do nothing */
        }

        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_WU);
    } else {
        uart_dev->uart_irq.wakeup_handle = NULL;
        hals_usart_interrupt_disable(uart_dev->periph, USART_INT_WU);
    }

    /* initialize uart transmit ready callback */
    if(NULL != p_irq->receive_fifo_full_handle) {
        uart_dev->uart_irq.receive_fifo_full_handle = p_irq->receive_fifo_full_handle;
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_RFF);
    } else {
        uart_dev->uart_irq.receive_fifo_full_handle = NULL;
        hals_usart_interrupt_disable(uart_dev->periph, USART_INT_RFF);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_irq_handle_all_reset(hal_uart_dev_struct *uart_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and p_irq address */
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure interrupt callback function to NULL */
    uart_dev->uart_irq.transmit_fifo_empty_handle = NULL;
    uart_dev->uart_irq.transmit_complete_handle   = NULL;
    uart_dev->uart_irq.receive_complete_handle    = NULL;
    uart_dev->uart_irq.receive_timeout_handle     = NULL;
    uart_dev->uart_irq.error_handle               = NULL;
    uart_dev->uart_irq.wakeup_handle              = NULL;
    uart_dev->uart_irq.idle_line_detected_handle  = NULL;
    uart_dev->uart_irq.address_match_handle       = NULL;
    uart_dev->uart_irq.lin_break_detected_handle  = NULL;
    uart_dev->uart_irq.cts_change_handle          = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      transmit amounts of data, poll transmit process and completed status
                the function is blocking
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to data buffer
    \param[in]  length: number of data to be transmitted
    \param[in]  timeout: timeout duration
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY,
                            HAL_ERR_TIMEOUT, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_transmit_poll(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint32_t length, uint32_t timeout)
{
    volatile uint8_t *p_data8bits;
    volatile uint16_t *p_data16bits;

    __IO uint32_t tick_start = 0U;
    uint16_t temp            = 0U;
    int32_t ret              = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == uart_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [uart_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length) || (0U == timeout)) {
        HAL_DEBUGE("parameter [length] || [timeout] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* check the tx_state whether is busy or not */
    if(HAL_UART_STATE_READY != uart_dev->tx_state) {
        HAL_DEBUGE("uart tx has already been used, please wait until run_state change to free");
        ret                   = HAL_ERR_BUSY;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_BUSY;
    } else {
        /* initialize transmit parameters */
        uart_dev->error_state     = (uint16_t)HAL_UART_ERROR_NONE;
        uart_dev->tx_state        = HAL_UART_STATE_BUSY_TX;
        uart_dev->txbuffer.buffer = p_buffer;
        uart_dev->txbuffer.length = length;
        uart_dev->txbuffer.remain = length;

        /* calculate the data length */
        if((USART_WORDLENGTH_10B == uart_dev->init.wordlength) || \
           ((USART_WORDLENGTH_9B == uart_dev->init.wordlength) && (USART_PARITY_NONE == uart_dev->init.parity))) {
            p_data8bits  = NULL;
            p_data16bits = (uint16_t *)(uint32_t)uart_dev->txbuffer.buffer;
        } else {
            p_data8bits  = uart_dev->txbuffer.buffer;
            p_data16bits = NULL;
        }

        /* configure timeout */
        tick_start = hal_sys_basetick_count_get();
        while(uart_dev->txbuffer.remain > 0U) {
            /* wait for transmit buffer empty */
            while(RESET == hals_usart_flag_get(uart_dev->periph, USART_FLAG_TBE)) {
                if(HAL_TIMEOUT_FOREVER != timeout) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                        HAL_DEBUGE("uart transmit timeout");
                        ret                   = HAL_ERR_TIMEOUT;
                        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_TTO;
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }

            if(HAL_ERR_NONE == ret) {
                /* send data: in case of 9bit&no parity,10bit, uint16_t data will be transmitted */
                if(NULL == p_data8bits) {
                    temp = (uint16_t)(*p_data16bits & 0x03FFU);
                    hals_usart_data_transmit(uart_dev->periph, temp);
                    p_data16bits++;
                } else {
                    temp = (uint8_t)(*p_data8bits & 0xFFU);
                    hals_usart_data_transmit(uart_dev->periph, temp);
                    p_data8bits++;
                }

                /* change the transmit pointer */
                uart_dev->txbuffer.remain--;
            } else {
                /* do nothing */
            }
        }

        /* wait for transmit complete */
        while(RESET == hals_usart_flag_get(uart_dev->periph, USART_FLAG_TC)) {
            if(HAL_TIMEOUT_FOREVER != timeout) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                    HAL_DEBUGE("uart transmit timeout");
                    ret                   = HAL_ERR_TIMEOUT;
                    uart_dev->error_state = (uint16_t)HAL_UART_ERROR_TTO;
                    break;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }

        /* reset the state */
        uart_dev->tx_state = HAL_UART_STATE_READY;
    }

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return ret;
}

/*!
    \brief      receive amounts of data, poll receive process and completed status
                the function is blocking
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  length: number of data to be received
    \param[in]  timeout: timeout duration
    \param[out] p_buffer: pointer to data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY,
                            HAL_ERR_TIMEOUT, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_receive_poll(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint32_t length, uint32_t timeout)
{
    volatile uint8_t *p_data8bits;
    volatile uint16_t *p_data16bits;

    __IO uint32_t tick_start = 0U;
    uint16_t temp            = 0U;
    int32_t ret              = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == uart_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [uart_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length) || (0U == timeout)) {
        HAL_DEBUGE("parameter [length] || [timeout] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* check the rx_state whether is busy or not */
    if(HAL_UART_STATE_READY != uart_dev->rx_state) {
        HAL_DEBUGE("uart rx has already been used, please wait until run_state change to free ");
        ret                   = HAL_ERR_BUSY;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_BUSY;
    } else {
        /* initialize receive parameters */
        uart_dev->error_state     = (uint16_t)HAL_UART_ERROR_NONE;
        uart_dev->rx_state        = HAL_UART_STATE_BUSY_RX;
        uart_dev->rxbuffer.buffer = p_buffer;
        uart_dev->rxbuffer.length = length;
        uart_dev->rxbuffer.remain = length;
        uart_dev->data_bit_mask   = _uart_data_bit_mask_get(uart_dev);

        /* calculate the data length */
        if((USART_WORDLENGTH_10B == uart_dev->init.wordlength) || \
           ((USART_WORDLENGTH_9B == uart_dev->init.wordlength) && (USART_PARITY_NONE == uart_dev->init.parity))) {
            p_data8bits  = NULL;
            p_data16bits = (uint16_t *)(uint32_t)uart_dev->rxbuffer.buffer;
        } else {
            p_data8bits  = uart_dev->rxbuffer.buffer;
            p_data16bits = NULL;
        }

        /* configure timeout */
        tick_start = hal_sys_basetick_count_get();

        while(uart_dev->rxbuffer.remain > 0U) {
            /* wait for read data buffer not empty */
            while(RESET == hals_usart_flag_get(uart_dev->periph, USART_FLAG_RBNE)) {
                if(HAL_TIMEOUT_FOREVER != timeout) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                        HAL_DEBUGE("uart receive timeout");
                        /* reset the state */
                        ret                   = HAL_ERR_TIMEOUT;
                        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_RTO;
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }

            if(HAL_ERR_NONE == ret) {
                /* read data: in case of 9bit&no parity,10 bits,uint16_t data will be receive */
                if(NULL == p_data8bits) {
                    temp = hals_usart_data_receive(uart_dev->periph);
                    *p_data16bits = (uint16_t)(temp & uart_dev->data_bit_mask);
                    p_data16bits++;
                } else {
                    temp = hals_usart_data_receive(uart_dev->periph);
                    *p_data8bits = (uint8_t)(temp & (uint8_t)uart_dev->data_bit_mask);
                    p_data8bits++;
                }

                uart_dev->rxbuffer.remain--;
            } else {
                /* do nothing */
            }
        }

        /* change the Rx state to ready */
        uart_dev->rx_state = HAL_UART_STATE_READY;
    }

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return ret;
}

/*!
    \brief      receive amounts of data, poll receive process and completed status
                the function is blocking
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  length: number of data to be received
    \param[in]  timeout: timeout duration
    \param[out] p_buffer: pointer to data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_BUSY, HAL_ERR_TIMEOUT,HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_receive_idle_poll(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint32_t length, uint32_t timeout)
{
    volatile uint8_t *p_data8bits;
    volatile uint16_t *p_data16bits;
    __IO uint32_t tick_start = 0U;
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == uart_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [uart_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length) || (0U == timeout)) {
        HAL_DEBUGE("parameter [length] || [timeout] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* check the rx_state whether is busy or not */
    if(HAL_UART_STATE_READY != uart_dev->rx_state) {
        HAL_DEBUGE("uart rx has already been used, please wait until run_state change to free ");
        ret                   = HAL_ERR_BUSY;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_BUSY;
    } else {
        /* initialize receive parameters */
        uart_dev->error_state     = (uint16_t)HAL_UART_ERROR_NONE;
        uart_dev->rx_state        = HAL_UART_STATE_BUSY_RX;
        uart_dev->rxbuffer.buffer = p_buffer;
        uart_dev->rxbuffer.length = length;
        uart_dev->rxbuffer.remain = length;
        uart_dev->data_bit_mask   = _uart_data_bit_mask_get(uart_dev);

        /* calculate the data length */
        if((USART_WORDLENGTH_10B == uart_dev->init.wordlength) || \
           ((USART_WORDLENGTH_9B == uart_dev->init.wordlength) && (USART_PARITY_NONE == uart_dev->init.parity))) {
            p_data8bits  = NULL;
            p_data16bits = (uint16_t *)(uint32_t)uart_dev->rxbuffer.buffer;
        } else {
            p_data8bits  = uart_dev->rxbuffer.buffer;
            p_data16bits = NULL;
        }

        /* configure timeout */
        tick_start = hal_sys_basetick_count_get();

        while(uart_dev->rxbuffer.remain > 0U) {
            /* wait for read data buffer not empty */
            while(RESET == hals_usart_flag_get(uart_dev->periph, USART_FLAG_RBNE)) {
                if(HAL_TIMEOUT_FOREVER != timeout) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                        HAL_DEBUGE("uart receive timeout");
                        /* reset the state */
                        ret                   = HAL_ERR_TIMEOUT;
                        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_RTO;
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }

                if(RESET != hals_usart_flag_get(uart_dev->periph, USART_FLAG_IDLE)) {
                    hals_usart_flag_clear(uart_dev->periph, USART_FLAG_IDLE);
                    if(uart_dev->rxbuffer.remain < length) {
                        /* change the Rx state to ready */
                        ret                   = HAL_ERR_NONE;
                        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_NONE;
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }

            if(HAL_ERR_NONE == ret) {
                /* read data: in case of 9bit&no parity,10 bits,uint16_t data will be receive */
                if(NULL == p_data8bits) {
                    *p_data16bits = (uint16_t)(hals_usart_data_receive(uart_dev->periph) & uart_dev->data_bit_mask);
                    p_data16bits++;
                } else {
                *p_data8bits = (uint8_t)(hals_usart_data_receive(uart_dev->periph) & (uint8_t)uart_dev->data_bit_mask);
                    p_data8bits++;
                }

                uart_dev->rxbuffer.remain--;
            }
        }

        /* change the Rx state to ready */
        uart_dev->rx_state = HAL_UART_STATE_READY;
    }

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return ret;
}

/*!
    \brief      transmit amounts of data by interrupt method,the function is non-blocking
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to data buffer
    \param[in]  length: number of data to be transmit
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_transmit_interrupt(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint32_t length, \
                                    hal_uart_irq_user_callback_struct *p_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == uart_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [uart_dev] or [p_buffer] address value is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* check the tx_state whether is busy or not */
    if(HAL_UART_STATE_READY != uart_dev->tx_state) {
        HAL_DEBUGE("uart tx has already been used, please wait until run_state change to free");
        ret                   = HAL_ERR_BUSY;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_BUSY;
    } else {
        /* initialize transmit parameters */
        uart_dev->error_state     = (uint16_t)HAL_UART_ERROR_NONE;
        uart_dev->tx_state        = HAL_UART_STATE_BUSY_TX;
        uart_dev->txbuffer.buffer = p_buffer;
        uart_dev->txbuffer.length = length;
        uart_dev->txbuffer.remain = length;

        /* configure the transmit ready and complete callback as the function implemented */
        uart_dev->uart_irq.transmit_fifo_empty_handle = _uart_transmit_fifo_empty_interrupt;
        uart_dev->uart_irq.transmit_complete_handle   = _uart_transmit_complete_interrupt;
        uart_dev->uart_irq.error_handle               = _uart_error_interrupt;

        /* clear user callback */
        uart_dev->error_callback               = NULL;
        uart_dev->transmit_complete_callback   = NULL;
        uart_dev->transmit_fifo_empty_callback = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->transmit_complete_func) {
                uart_dev->transmit_complete_callback = (void *)p_func->transmit_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->transmit_fifo_empty_func) {
                uart_dev->transmit_fifo_empty_callback = (void *)p_func->transmit_fifo_empty_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                uart_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear UART TC interrupt flag */
        hals_usart_flag_clear(uart_dev->periph, USART_FLAG_TC);

        /* enable the TBE interrupt */
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_TBE);
    }

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return ret;
}

/*!
    \brief      receive amounts of data by interrupt method,the function is non-blocking
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[in]  length: number of data to be received
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] p_buffer: pointer to data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_receive_interrupt(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint32_t length, \
                                   hal_uart_irq_user_callback_struct *p_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == uart_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [uart_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* check the rx_state whether is busy or not */
    if(HAL_UART_STATE_BUSY == uart_dev->rx_state) {
        HAL_DEBUGE("uart rx has already been used, please wait until run_state change to free ");
        ret = HAL_ERR_BUSY;
    } else {
        /* configuration state and error state */
        uart_dev->rx_state    = HAL_UART_STATE_BUSY_RX;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_NONE;

        /* initialize receive parameters */
        uart_dev->rxbuffer.buffer = p_buffer;
        uart_dev->rxbuffer.length = length;
        uart_dev->rxbuffer.remain = length;

        /* configure the receive callback as the function implemented */
        uart_dev->uart_irq.receive_fifo_full_handle = _uart_received_fifo_full_interrupt;
        uart_dev->uart_irq.receive_complete_handle  = _uart_receive_interrupt;
        uart_dev->uart_irq.error_handle             = _uart_error_interrupt;

        /* clear user callback */
        uart_dev->error_callback             = NULL;
        uart_dev->receive_complete_callback  = NULL;
        uart_dev->receive_fifo_full_callback = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->receive_complete_func) {
                uart_dev->receive_complete_callback = (void *)p_func->receive_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->receive_fifo_full_func) {
                uart_dev->receive_fifo_full_callback = (void *)p_func->receive_fifo_full_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                uart_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear RBNE interrupt flag */
        hals_usart_data_receive(uart_dev->periph);
        hals_usart_flag_clear(uart_dev->periph, USART_FLAG_RBNE);

        /* enable the uart receiver timeout interrupt */
        if(USART_TIMEOUT_DISABLE != uart_dev->init.timeoutenable) {
            hals_usart_interrupt_enable(uart_dev->periph, USART_INT_RT);
        } else {
            /* do nothing */
        }

        /* enable ERR, PERR, RBNE interrupt */
        if(USART_PARITY_NONE != uart_dev->init.parity) {
            hals_usart_interrupt_enable(uart_dev->periph, USART_INT_PERR);
        } else {
            /* do nothing */
        }

        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_ERR);
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_RBNE);
    }

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return ret;
}

/*!
    \brief      receive amounts of data by interrupt method,the function is non-blocking
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  length: number of data to be received
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] p_buffer: pointer to data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_receive_idle_interrupt(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint32_t length, \
                                        hal_uart_irq_user_callback_struct *p_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == uart_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [uart_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* check the rx_state whether is busy or not */
    if(HAL_UART_STATE_BUSY == uart_dev->rx_state) {
        HAL_DEBUGE("uart rx has already been used, please wait until run_state change to free ");
        ret                   = HAL_ERR_BUSY;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_BUSY;
    } else {
        /* configuration state and error state */
        uart_dev->rx_state    = HAL_UART_STATE_BUSY_RX;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_NONE;

        /* initialize receive parameters */
        uart_dev->rxbuffer.buffer = p_buffer;
        uart_dev->rxbuffer.length = length;
        uart_dev->rxbuffer.remain = length;

        /* configure the receive callback as the function implemented */
        uart_dev->uart_irq.idle_line_detected_handle = _uart_idle_line_detected_interrupt;
        uart_dev->uart_irq.receive_fifo_full_handle  = _uart_received_fifo_full_interrupt;
        uart_dev->uart_irq.receive_complete_handle   = _uart_receive_interrupt;

        /* clear user callback */
        uart_dev->receive_complete_callback  = NULL;
        uart_dev->receive_fifo_full_callback = NULL;
        uart_dev->idle_receive_callback      = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->receive_complete_func) {
                uart_dev->receive_complete_callback = (void *)p_func->receive_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->receive_fifo_full_func) {
                uart_dev->receive_fifo_full_callback = (void *)p_func->receive_fifo_full_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->idle_receive_func) {
                uart_dev->idle_receive_callback = (void *)p_func->idle_receive_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear IDLE interrupt flag */
        hals_usart_data_receive(uart_dev->periph);
        hals_usart_flag_clear(uart_dev->periph, USART_FLAG_IDLE);
        hals_usart_flag_clear(uart_dev->periph, USART_FLAG_RBNE);

        /* enable the uart receiver timeout interrupt */
        if(USART_TIMEOUT_DISABLE != uart_dev->init.timeoutenable) {
            hals_usart_interrupt_enable(uart_dev->periph, USART_INT_RT);
        } else {
            /* do nothing */
        }

        /* enable ERR, PERR, RBNE interrupt */
        if(USART_PARITY_NONE != uart_dev->init.parity) {
            hals_usart_interrupt_enable(uart_dev->periph, USART_INT_PERR);
        } else {
            /* do nothing */
        }

        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_ERR);
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_RBNE);

        /* enable IDLE interrupt */
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_IDLE);
    }

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return ret;
}

/*!
    \brief      transmit amounts of data by dma method,the function is non-blocking
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to data buffer
    \param[in]  length: number of data to be transmit
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_transmit_dma(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint16_t length, \
                              hal_uart_irq_user_callback_struct *p_func)
{
    hal_dma_irq_struct dma_irq = {0};
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == uart_dev) || (NULL == p_buffer) || (NULL == uart_dev->p_dma_tx)) {
        HAL_DEBUGE("pointer [uart_dev] or [p_buffer] or [uart_dev->p_dma_tx] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* check the tx_state whether is busy or not */
    if(HAL_UART_STATE_BUSY == uart_dev->tx_state) {
        HAL_DEBUGE("uart tx has already been used, please wait until run_state change to free");
        ret                   = HAL_ERR_BUSY;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_BUSY;
    } else {
        /* configuration state and error state */
        uart_dev->tx_state    = HAL_UART_STATE_BUSY_TX;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_NONE;

        /* initialize transmit parameters */
        uart_dev->txbuffer.buffer = p_buffer;
        uart_dev->txbuffer.length = length;
        uart_dev->txbuffer.remain = length;

        /* configure DMA interrupt callback function */
        dma_irq.full_finish_handle = _uart_dma_tx_complete;
        dma_irq.half_finish_handle = _uart_dma_tx_half_complete;
        dma_irq.error_handle       = _uart_dma_error;

        /* clear user callback */
        uart_dev->transmit_half_complete_callback = NULL;
        uart_dev->transmit_complete_callback      = NULL;
        uart_dev->error_callback                  = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->transmit_half_complete_func) {
                uart_dev->transmit_complete_callback = (void *)p_func->transmit_half_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->transmit_complete_func) {
                uart_dev->transmit_complete_callback = (void *)p_func->transmit_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                uart_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* start DMA interrupt mode transfer */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(uart_dev->p_dma_tx, (uint32_t)uart_dev->txbuffer.buffer, \
                                                   (uint32_t)&USART_TDATA(uart_dev->periph), \
                                                   (uint16_t)uart_dev->txbuffer.length, &dma_irq)) {
            HAL_DEBUGE("dma start interrupt failed");
            ret                   = HAL_ERR_BUSY;
            uart_dev->error_state = (uint16_t)HAL_UART_ERROR_DMA;
        } else {
            /* clear UART TC interrupt flag */
            hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_TC);

            /* DMA enable for transmission */
            hals_usart_dma_transmit_config(uart_dev->periph, USART_TRANSMIT_DMA_ENABLE);
        }

        /* set tx_state to ready */
        uart_dev->tx_state = HAL_UART_STATE_READY;
    }

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return ret;
}

/*!
    \brief      receive amounts of data by dma method,the function is non-blocking
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  length: number of data to be receive
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] p_buffer: pointer to data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_receive_dma(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint16_t length, \
                             hal_uart_irq_user_callback_struct *p_func)
{
    hal_dma_irq_struct dma_irq = {0};
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == uart_dev) || (NULL == p_buffer) || (NULL == uart_dev->p_dma_rx)) {
        HAL_DEBUGE("pointer [uart_dev] or [p_buffer] or [uart_dev->p_dma_rx] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* check the rx_state whether is busy or not */
    if(HAL_UART_STATE_BUSY == uart_dev->rx_state) {
        HAL_DEBUGE("uart rx has already been used, please wait until run_state change to free ");
        ret                   = HAL_ERR_BUSY;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_BUSY;
    } else {
        /* configuration state and error state */
        uart_dev->rx_state    = HAL_UART_STATE_BUSY_RX;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_NONE;

        /* initialize receive parameters */
        uart_dev->rxbuffer.buffer = p_buffer;
        uart_dev->rxbuffer.length = length;
        uart_dev->rxbuffer.remain = length;

        /* configure DMA interrupt interrupt callback function */
        dma_irq.full_finish_handle = _uart_dma_rx_complete;
        dma_irq.half_finish_handle = _uart_dma_rx_half_complete;
        dma_irq.error_handle       = _uart_dma_error;

        /* clear user callback */
        uart_dev->receive_complete_callback  = NULL;
        uart_dev->receive_fifo_full_callback = NULL;
        uart_dev->error_callback             = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->receive_half_complete_func) {
                uart_dev->receive_half_complete_callback = (void *)p_func->receive_half_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->receive_complete_func) {
                uart_dev->receive_complete_callback = (void *)p_func->receive_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                uart_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* start DMA interrupt mode transfer */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(uart_dev->p_dma_rx, (uint32_t)&USART_RDATA(uart_dev->periph), \
                                                   (uint32_t)uart_dev->rxbuffer.buffer, \
                                                   length, &dma_irq)) {
            HAL_DEBUGE("dma start interrupt failed");
            ret                   = HAL_ERR_BUSY;
            uart_dev->error_state = (uint16_t)HAL_UART_ERROR_DMA;
        } else {
            /* do nothing */
        }

        /* clear RBNE interrupt flag */
        hals_usart_data_receive(uart_dev->periph);

        /* enable the uart receiver timeout interrupt */
        if(USART_TIMEOUT_DISABLE != uart_dev->init.timeoutenable) {
            hals_usart_interrupt_enable(uart_dev->periph, USART_INT_RT);
        } else {
            /* do nothing */
        }

        /* enable ERR, PERR interrupt */
        if(USART_PARITY_NONE != uart_dev->init.parity) {
            hals_usart_interrupt_enable(uart_dev->periph, USART_INT_PERR);
        } else {
            /* do nothing */
        }

        /* enable error interrupt */
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_ERR);

        /* DMA enable for reception */
        hals_usart_dma_receive_config(uart_dev->periph, USART_RECEIVE_DMA_ENABLE);

        /* set rx_state to ready */
        uart_dev->rx_state = HAL_UART_STATE_READY;
    }

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return ret;
}

/*!
    \brief      receive amounts of data by dma method,the function is non-blocking
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[in]  length: number of data to be receive
    \param[out] p_buffer: pointer to data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_receive_idle_dma(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint16_t length, \
                                  hal_uart_irq_user_callback_struct *p_func)
{
    hal_dma_irq_struct dma_irq = {0};
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == uart_dev) || (NULL == p_buffer) || (NULL == uart_dev->p_dma_rx)) {
        HAL_DEBUGE("parameter [uart_dev] or [p_buffer] or [uart_dev->p_dma_rx] value is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* check the rx_state whether is busy or not */
    if(HAL_UART_STATE_BUSY == uart_dev->rx_state) {
        HAL_DEBUGE("uart rx has already been used, please wait until run_state change to free ");
        ret                   = HAL_ERR_BUSY;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_BUSY;
    } else {
        /* configuration state and error state */
        uart_dev->rx_state    = HAL_UART_STATE_BUSY_RX;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_NONE;

        /* initialize receive parameters */
        uart_dev->rxbuffer.buffer = p_buffer;
        uart_dev->rxbuffer.length = length;
        uart_dev->rxbuffer.remain = length;

        /* configure DMA interrupt interrupt callback function */
        dma_irq.full_finish_handle = _uart_dma_rx_complete;
        dma_irq.half_finish_handle = _uart_dma_rx_half_complete;
        dma_irq.error_handle       = _uart_dma_error;

        /* clear user callback */
        uart_dev->receive_half_complete_callback = NULL;
        uart_dev->receive_complete_callback      = NULL;
        uart_dev->error_callback                 = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->receive_half_complete_func) {
                uart_dev->receive_half_complete_callback = (void *)p_func->receive_half_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->receive_complete_func) {
                uart_dev->receive_complete_callback = (void *)p_func->receive_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                uart_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* start DMA interrupt mode transfer */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(uart_dev->p_dma_rx, (uint32_t)&USART_RDATA(uart_dev->periph), \
                                                   (uint32_t)uart_dev->rxbuffer.buffer, length, &dma_irq)) {
            HAL_DEBUGE("dma start interrupt failed");
            ret                   = HAL_ERR_BUSY;
            uart_dev->error_state = (uint16_t)HAL_UART_ERROR_DMA;
        } else {
            /* clear IDLE interrupt flag */
            hals_usart_flag_clear(uart_dev->periph, USART_FLAG_IDLE);

            /* enable the uart receiver timeout interrupt */
            if(USART_TIMEOUT_DISABLE != uart_dev->init.timeoutenable) {
                hals_usart_interrupt_enable(uart_dev->periph, USART_INT_RT);
            } else {
                /* do nothing */
            }

            /* enable ERR, PERR interrupt */
            if(USART_PARITY_NONE != uart_dev->init.parity) {
                hals_usart_interrupt_enable(uart_dev->periph, USART_INT_PERR);
            } else {
                /* do nothing */
            }

            /* enable error interrupt */
            hals_usart_interrupt_enable(uart_dev->periph, USART_INT_ERR);

            /* enable IDLE interrupt */
            hals_usart_interrupt_enable(uart_dev->periph, USART_INT_IDLE);

            /* DMA enable for reception */
            hals_usart_dma_receive_config(uart_dev->periph, USART_RECEIVE_DMA_ENABLE);
        }

        /* set rx_state to ready */
        uart_dev->rx_state = HAL_UART_STATE_READY;
    }

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return ret;
}

/*!
    \brief      pause uart DMA transfer during transmission process
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_dma_pause(hal_uart_dev_struct *uart_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* check the rx_state whether is busy or not */
    if(((USART_CTL2(uart_dev->periph) & USART_CTL2_DENT) == USART_CTL2_DENT) && \
       (HAL_UART_STATE_BUSY_TX == uart_dev->tx_state)) {
        /* disable DMA transmit */
        hals_usart_dma_transmit_config(uart_dev->periph, USART_TRANSMIT_DMA_DISABLE);
    } else {
        /* do nothing */
    }

    if(((USART_CTL2(uart_dev->periph) & USART_CTL2_DENT) == USART_CTL2_DENT) && \
       (HAL_UART_STATE_BUSY_RX == uart_dev->tx_state)) {
        /* disable the PERR and ERR interrupt */
        hals_usart_interrupt_disable(uart_dev->periph, USART_INT_PERR);
        hals_usart_interrupt_disable(uart_dev->periph, USART_INT_ERR);

        /* disable DMA receive */
        hals_usart_dma_receive_config(uart_dev->periph, USART_RECEIVE_DMA_DISABLE);
    } else {
        /* do nothing */
    }

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      resume uart DMA transfer during transmission process
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_dma_resume(hal_uart_dev_struct *uart_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    if(HAL_UART_STATE_BUSY_TX == uart_dev->tx_state) {
        /* enable DMA transmit */
        hals_usart_dma_transmit_config(uart_dev->periph, USART_TRANSMIT_DMA_ENABLE);
    } else {
        /* do nothing */
    }

    if(HAL_UART_STATE_BUSY_RX == uart_dev->rx_state) {
        /* clear the overrun flag before resuming the rx transfer*/
        hals_usart_flag_clear(uart_dev->periph, USART_FLAG_ORERR);

        /* enable the PERR interrupt */
        if(USART_PARITY_NONE != uart_dev->init.parity) {
            hals_usart_interrupt_enable(uart_dev->periph, USART_INT_PERR);
        } else {
            /* do nothing */
        }

        /* enable error interrupt */
        hals_usart_interrupt_enable(uart_dev->periph, USART_INT_ERR);

        /* enable DMA transmit & receive */
        hals_usart_dma_receive_config(uart_dev->periph, USART_RECEIVE_DMA_ENABLE);
    } else {
        /* do nothing */
    }

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop uart DMA transfer during transmission process
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_dma_stop(hal_uart_dev_struct *uart_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    if(((USART_CTL2(uart_dev->periph) & USART_CTL2_DENT) == USART_CTL2_DENT) && \
       (HAL_UART_STATE_BUSY_TX == uart_dev->tx_state)) {
        hals_usart_dma_transmit_config(uart_dev->periph, USART_TRANSMIT_DMA_DISABLE);
        /* abort the usart dma tx channel */
        if(NULL != uart_dev->p_dma_tx) {
            hal_dma_stop(uart_dev->p_dma_tx);
        } else {
            /* do nothing */
        }
        _uart_end_transfer(uart_dev);
    } else {
        /* do nothing */
    }

    if((USART_CTL2_DENR == (USART_CTL2(uart_dev->periph) & USART_CTL2_DENR)) && \
       (HAL_UART_STATE_BUSY_RX == uart_dev->rx_state)) {
        hals_usart_dma_receive_config(uart_dev->periph, USART_RECEIVE_DMA_DISABLE);
        /* abort the usart dma rx channel */
        if(NULL != uart_dev->p_dma_rx) {
            hal_dma_stop(uart_dev->p_dma_rx);
        } else {
            /* do nothing */
        }
        _uart_end_transfer(uart_dev);
    } else {
        /* do nothing */
    }

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort ongoing transfers
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_abort(hal_uart_dev_struct *uart_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* disable the RFT and TFT interrupt */
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_RFT);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TFT);

    /* disable the TBE, TC, RBNE, PERR and ERR interrupt */
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_RBNE);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TBE);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TC);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_ERR);

    /* disable the IDLE interrupt */
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_IDLE);

    /* disable DMA transmit and stop DMA */
    if(USART_CTL2_DENT == (USART_CTL2(uart_dev->periph) & USART_CTL2_DENT)) {
        hals_usart_dma_transmit_config(uart_dev->periph, USART_TRANSMIT_DMA_DISABLE);
        if(NULL != uart_dev->p_dma_tx) {
            hal_dma_stop(uart_dev->p_dma_tx);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* disable DMA transmit and stop DMA */
    if(USART_CTL2_DENR == (USART_CTL2(uart_dev->periph) & USART_CTL2_DENR)) {
        hals_usart_dma_receive_config(uart_dev->periph, USART_RECEIVE_DMA_DISABLE);
        if(NULL != uart_dev->p_dma_rx) {
            hal_dma_stop(uart_dev->p_dma_rx);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* clear user callback */
    uart_dev->txbuffer.remain = 0U;
    uart_dev->rxbuffer.remain = 0U;
    uart_dev->tx_state        = HAL_UART_STATE_READY;
    uart_dev->rx_state        = HAL_UART_STATE_READY;
    uart_dev->error_state     = (uint16_t)HAL_UART_ERROR_NONE;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_PERR);
    hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_ERR_FERR);
    hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_ERR_NERR);
    hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_ERR_ORERR);

    /* discard the transmit & received data */
    hals_usart_command_enable(uart_dev->periph, USART_CMD_TXFCMD);
    hals_usart_command_enable(uart_dev->periph, USART_CMD_RXFCMD);

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort ongoing transmit transfer
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_transmit_abort(hal_uart_dev_struct *uart_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* disable the TBE, TC and TFT interrupt */
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TBE);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TC);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TFT);

    /* disable DMA transmit and stop DMA */
    if(USART_CTL2_DENT == (USART_CTL2(uart_dev->periph) & USART_CTL2_DENT)) {
        hals_usart_dma_transmit_config(uart_dev->periph, USART_TRANSMIT_DMA_DISABLE);
        if(NULL != uart_dev->p_dma_tx) {
            hal_dma_stop(uart_dev->p_dma_tx);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* reset the position and state */
    uart_dev->txbuffer.remain = 0U;

    /* discard the transmit data */
    hals_usart_command_enable(uart_dev->periph, USART_CMD_TXFCMD);

    uart_dev->tx_state = HAL_UART_STATE_READY;

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort ongoing receive transfer
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_receive_abort(hal_uart_dev_struct *uart_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* disable the RBNE, PERR and ERR interrupt */
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_RBNE);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_ERR);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_RFT);

    /* disable DMA receive and stop DMA */
    if(USART_CTL2_DENR == (USART_CTL2(uart_dev->periph) & USART_CTL2_DENR)) {
        hals_usart_dma_receive_config(uart_dev->periph, USART_RECEIVE_DMA_DISABLE);
        if(NULL != uart_dev->p_dma_rx) {
            hal_dma_stop(uart_dev->p_dma_rx);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* reset the position and state */
    uart_dev->rxbuffer.remain = 0U;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_PERR);
    hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_ERR_FERR);
    hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_ERR_NERR);
    hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_ERR_ORERR);

    /* discard the receive data */
    hals_usart_command_enable(uart_dev->periph, USART_CMD_RXFCMD);

    uart_dev->rx_state = HAL_UART_STATE_READY;

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort ongoing transfers in interrupt/DMA method
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_transmit_abort_interrupt(hal_uart_dev_struct *uart_dev, hal_uart_irq_user_callback_struct *p_func)
{
    uint8_t abort_complete = ERROR;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* disable the TFNF, TFT, TC interrupt */
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TFNF);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TFT);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TC);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_ERR);

    /* if DMA Tx handle are associated to UART handle,
       DMA abort complete callbacks should be initialized before any call
       to DMA abort functions */
    /* if DMA Tx request is enable  */
    if(NULL != uart_dev->p_dma_tx) {
        if(USART_CTL2_DENT == (USART_CTL2(uart_dev->periph) & USART_CTL2_DENT)) {
            uart_dev->p_dma_tx->dma_irq.abort_handle = _uart_dma_onlytx_abort_complete;
        } else {
            uart_dev->p_dma_tx->dma_irq.abort_handle = NULL;
        }
    } else {
        /* do nothing */
    }

    /* user callback */
    if(NULL != p_func->abort_tx_complete_func) {
        uart_dev->abort_tx_complete_callback = (void *)p_func->abort_tx_complete_func;
    } else {
        /* do nothing */
    }

    /* disable the UART DMA tx request if enabled */
    if(USART_CTL2_DENT == (USART_CTL2(uart_dev->periph) & USART_CTL2_DENT)) {
        hals_usart_dma_transmit_config(uart_dev->periph, USART_TRANSMIT_DMA_DISABLE);
        if(NULL != uart_dev->p_dma_tx) {
            /* abort DMA tx */
            if(HAL_ERR_NONE != hal_dma_stop_interrupt(uart_dev->p_dma_tx)) {
                uart_dev->p_dma_tx->dma_irq.abort_handle = NULL;
                abort_complete = ERROR;
            } else {
                abort_complete = SUCCESS;
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if(ERROR == abort_complete) {
        /* reset transmit buff */
        uart_dev->txbuffer.remain = 0U;

        /* reset transmit callback */
        uart_dev->uart_irq.transmit_fifo_empty_handle = NULL;
        uart_dev->uart_irq.transmit_complete_handle   = NULL;
        uart_dev->uart_irq.error_handle               = NULL;

        /* reset user callback */
        uart_dev->transmit_complete_callback   = NULL;
        uart_dev->transmit_fifo_empty_callback = NULL;

        /* reset state and error state */
        uart_dev->tx_state    = HAL_UART_STATE_READY;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_NONE;

        /* clear interrupt error flags */
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_ERR_FERR);

        if(NULL != uart_dev->abort_tx_complete_callback) {
            ((hal_uart_user_cb)uart_dev->abort_tx_complete_callback)(uart_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort ongoing received in interrupt/DMA method
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_receive_abort_interrupt(hal_uart_dev_struct *uart_dev, hal_uart_irq_user_callback_struct *p_func)
{
    uint8_t abort_complete = ERROR;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* disable the RFT RBNE, PERR and ERR interrupt */
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_EB);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_RT);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_RFNE);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_RFT);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_ERR);

    /* if DMA Rx handle are associated to UART handle,
       DMA abort complete callbacks should be initialized before any call
       to DMA abort functions */
    /* if DMA Rx request is enable */
    if(NULL != uart_dev->p_dma_rx) {
        if(USART_CTL2_DENT == ((USART_CTL2(uart_dev->periph) & USART_CTL2_DENR) >> 6U)) {
            uart_dev->p_dma_rx->dma_irq.abort_handle = _uart_dma_onlyrx_abort_complete;
        } else {
            uart_dev->p_dma_rx->dma_irq.abort_handle = NULL;
        }
    } else {
        /* do nothing */
    }

    /* user callback */
    if(NULL != p_func->abort_rx_complete_func) {
        uart_dev->abort_rx_complete_callback = (void *)p_func->abort_rx_complete_func;
    } else {
        /* do nothing */
    }

    /* if DMA rx request is enable  */
    if(USART_CTL2_DENR == (USART_CTL2(uart_dev->periph) & USART_CTL2_DENR)) {
        hals_usart_dma_receive_config(uart_dev->periph, USART_RECEIVE_DMA_DISABLE);
        if(NULL != uart_dev->p_dma_rx) {
            /* abort DMA rx */
            if(HAL_ERR_NONE != hal_dma_stop_interrupt(uart_dev->p_dma_rx)) {
                uart_dev->p_dma_rx->dma_irq.abort_handle = NULL;
                abort_complete = ERROR;
            } else {
                abort_complete = SUCCESS;
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if(ERROR == abort_complete) {
        /* reset received buff */
        uart_dev->rxbuffer.remain = 0U;

        /* reset state and error state */
        uart_dev->rx_state    = HAL_UART_STATE_READY;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_NONE;

        /* reset the Rx callback function pointer */
        uart_dev->uart_irq.receive_fifo_full_handle = NULL;
        uart_dev->uart_irq.receive_complete_handle  = NULL;
        uart_dev->uart_irq.error_handle             = NULL;

        /* clear user callback */
        uart_dev->receive_complete_callback  = NULL;
        uart_dev->receive_fifo_full_callback = NULL;

        /* clear interrupt error flags */
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_PERR);
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_ERR_FERR);
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_ERR_NERR);
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_ERR_ORERR);
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_RT);
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_EB);

        if(NULL != uart_dev->abort_rx_complete_callback) {
            ((hal_uart_user_cb)uart_dev->abort_rx_complete_callback)(uart_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort ongoing transfers\received in interrupt/DMA method
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_abort_interrupt(hal_uart_dev_struct *uart_dev, hal_uart_irq_user_callback_struct *p_func)
{
    uint8_t abort_complete = SUCCESS;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* disable the RFT and TFT interrupt */
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_RFT);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TFT);

    /* disable the TBE, TC, RBNE, PERR and ERR interrupt */
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_RBNE);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TBE);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TC);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_ERR);

    /* if DMA Tx and/or DMA Rx handle are associated to UART handle,
       DMA abort complete callbacks should be initialized before any call
       to DMA abort functions */
    /* if DMA Tx request is enable  */
    if(NULL != uart_dev->p_dma_tx) {
        if(USART_CTL2_DENT == (USART_CTL2(uart_dev->periph) & USART_CTL2_DENT)) {
            uart_dev->p_dma_tx->dma_irq.abort_handle = _uart_dma_abort_complete;
        } else {
            uart_dev->p_dma_tx->dma_irq.abort_handle = NULL;
        }
    } else {
        /* do nothing */
    }

    /* if DMA Rx request is enable */
    if(NULL != uart_dev->p_dma_rx) {
        if(USART_CTL2_DENT == ((USART_CTL2(uart_dev->periph) & USART_CTL2_DENR) >> 6U)) {
            uart_dev->p_dma_rx->dma_irq.abort_handle = _uart_dma_abort_complete;
        } else {
            uart_dev->p_dma_rx->dma_irq.abort_handle = NULL;
        }
    } else {
        /* do nothing */
    }

    /* user callback */
    if(NULL != p_func->abort_complete_func) {
        uart_dev->abort_complete_callback = (void *)p_func->abort_complete_func;
    } else {
        /* do nothing */
    }

    /* disable the UART DMA tx request if enabled */
    if(USART_CTL2_DENT == (USART_CTL2(uart_dev->periph) & USART_CTL2_DENT)) {
        hals_usart_dma_transmit_config(uart_dev->periph, USART_TRANSMIT_DMA_DISABLE);
        if(NULL != uart_dev->p_dma_tx) {
            /* abort DMA tx */
            if(HAL_ERR_NONE != hal_dma_stop_interrupt(uart_dev->p_dma_tx)) {
                uart_dev->p_dma_tx->dma_irq.abort_handle = NULL;
                abort_complete = ERROR;
            } else {
                abort_complete = SUCCESS;
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* if DMA rx request is enable  */
    if(USART_CTL2_DENR == (USART_CTL2(uart_dev->periph) & USART_CTL2_DENR)) {
        hals_usart_dma_receive_config(uart_dev->periph, USART_RECEIVE_DMA_DISABLE);
        if(NULL != uart_dev->p_dma_rx) {
            /* abort DMA rx */
            if(HAL_ERR_NONE != hal_dma_stop_interrupt(uart_dev->p_dma_rx)) {
                uart_dev->p_dma_rx->dma_irq.abort_handle = NULL;
                abort_complete = ERROR;
            } else {
                abort_complete = SUCCESS;
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if(ERROR == abort_complete) {
        /* reset transmit and received buff */
        uart_dev->txbuffer.remain = 0U;
        uart_dev->rxbuffer.remain = 0U;

        /* reset state and error state */
        uart_dev->tx_state    = HAL_UART_STATE_READY;
        uart_dev->rx_state    = HAL_UART_STATE_READY;
        uart_dev->error_state = (uint16_t)HAL_UART_ERROR_NONE;

        /* reset transmit/received callback */
        uart_dev->uart_irq.transmit_fifo_empty_handle = NULL;
        uart_dev->uart_irq.transmit_complete_handle   = NULL;
        uart_dev->uart_irq.error_handle               = NULL;
        uart_dev->uart_irq.receive_fifo_full_handle   = NULL;
        uart_dev->uart_irq.receive_complete_handle    = NULL;
        uart_dev->uart_irq.error_handle               = NULL;

        /* reset user callback */
        uart_dev->transmit_complete_callback   = NULL;
        uart_dev->transmit_fifo_empty_callback = NULL;
        uart_dev->receive_complete_callback    = NULL;
        uart_dev->receive_fifo_full_callback   = NULL;

        /* clear interrupt error flags */
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_PERR);
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_ERR_FERR);
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_ERR_NERR);
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_ERR_ORERR);
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_RT);
        hals_usart_interrupt_flag_clear(uart_dev->periph, USART_INT_FLAG_EB);

        if(NULL != uart_dev->abort_complete_callback) {
            ((hal_uart_user_cb)uart_dev->abort_complete_callback)(uart_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* unlock uart */
    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      enable FIFO
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_fifo_enable(hal_uart_dev_struct *uart_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((UART3 != uart_dev->periph) && (UART4 != uart_dev->periph) && \
       (UART6 != uart_dev->periph) && (UART7 != uart_dev->periph)) {
        HAL_DEBUGE("parameter [uart_dev->periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* disable USART */
    USART_CTL0(uart_dev->periph) &= ~(USART_CTL0_UEN);
    /* set FEN bit */
    USART_FCS(uart_dev->periph) |= USART_FCS_FEN;

    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      disable FIFO
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_fifo_disable(hal_uart_dev_struct *uart_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((UART3 != uart_dev->periph) && (UART4 != uart_dev->periph) && \
       (UART6 != uart_dev->periph) && (UART7 != uart_dev->periph)) {
        HAL_DEBUGE("parameter [uart_dev->periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    /* disable USART */
    USART_CTL0(uart_dev->periph) &= ~(USART_CTL0_UEN);
    /* reset FEN bit */
    USART_FCS(uart_dev->periph) &= ~(USART_FCS_FEN);

    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure transmit FIFO threshold
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  txthreshold: transmit FIFO threshold
                only one parameter can be selected which is shown as below:
      \arg        USART_TFTCFG_THRESHOLD_1_8: transmit FIFO reaches 1/8 of its depth
      \arg        USART_TFTCFG_THRESHOLD_1_4: transmit FIFO reaches 1/4 of its depth
      \arg        USART_TFTCFG_THRESHOLD_1_2: transmit FIFO reaches 1/2 of its depth
      \arg        USART_TFTCFG_THRESHOLD_3_4: transmit FIFO reaches 3/4 of its depth
      \arg        USART_TFTCFG_THRESHOLD_7_8: transmit FIFO reaches 7/8 of its depth
      \arg        USART_TFTCFG_THRESHOLD_EMPTY: transmit FIFO becomes empty
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_transmit_fifo_threshold_config(hal_uart_dev_struct *uart_dev, uint32_t txthreshold)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((UART3 != uart_dev->periph) && (UART4 != uart_dev->periph) && \
       (UART6 != uart_dev->periph) && (UART7 != uart_dev->periph)) {
        HAL_DEBUGE("parameter [uart_dev->periph] value is invalid");
        return HAL_ERR_VAL;
    }

    if((USART_TFTCFG_THRESHOLD_1_8 != txthreshold) && (USART_TFTCFG_THRESHOLD_1_4 != txthreshold) && \
       (USART_TFTCFG_THRESHOLD_1_2 != txthreshold) && (USART_TFTCFG_THRESHOLD_3_4 != txthreshold) && \
       (USART_TFTCFG_THRESHOLD_7_8 != txthreshold) && (USART_TFTCFG_THRESHOLD_EMPTY != txthreshold)) {
        HAL_DEBUGE("parameter [txthreshold] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    USART_FCS(uart_dev->periph) &= ~(USART_FCS_TFTCFG);
    USART_FCS(uart_dev->periph) |= txthreshold;

    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure receive FIFO threshold
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  rxthreshold: receive FIFO threshold
                only one parameter can be selected which is shown as below:
      \arg        USART_RFTCFG_THRESHOLD_1_8: receive FIFO reaches 1/8 of its depth
      \arg        USART_RFTCFG_THRESHOLD_1_4: receive FIFO reaches 1/4 of its depth
      \arg        USART_RFTCFG_THRESHOLD_1_2: receive FIFO reaches 1/2 of its depth
      \arg        USART_RFTCFG_THRESHOLD_3_4: receive FIFO reaches 3/4 of its depth
      \arg        USART_RFTCFG_THRESHOLD_7_8: receive FIFO reaches 7/8 of its depth
      \arg        USART_RFTCFG_THRESHOLD_FULL: receive FIFO becomes full
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_receive_fifo_threshold_config(hal_uart_dev_struct *uart_dev, uint32_t rxthreshold)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((UART3 != uart_dev->periph) && (UART4 != uart_dev->periph) && \
       (UART6 != uart_dev->periph) && (UART7 != uart_dev->periph)) {
        HAL_DEBUGE("parameter [uart_dev->periph] value is invalid");
        return HAL_ERR_VAL;
    }

    if((USART_RFTCFG_THRESHOLD_1_8 != rxthreshold) && (USART_RFTCFG_THRESHOLD_1_4 != rxthreshold) && \
       (USART_RFTCFG_THRESHOLD_1_2 != rxthreshold) && (USART_RFTCFG_THRESHOLD_3_4 != rxthreshold) && \
       (USART_RFTCFG_THRESHOLD_7_8 != rxthreshold) && (USART_RFTCFG_THRESHOLD_FULL != rxthreshold)) {
        HAL_DEBUGE("parameter [rxthreshold] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    USART_FCS(uart_dev->periph) &= ~(USART_FCS_RFTCFG);
    USART_FCS(uart_dev->periph) |= rxthreshold;

    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure receiver timeout threshold
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  rtimeout: 0x00000000-0x00FFFFFF, receiver timeout value in terms of number of baud clocks
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_receiver_timeout_config(hal_uart_dev_struct *uart_dev, uint32_t rtimeout)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((UART3 != uart_dev->periph) && (UART4 != uart_dev->periph) && \
       (UART6 != uart_dev->periph) && (UART7 != uart_dev->periph)) {
        HAL_DEBUGE("parameter [uart_dev->periph] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x00FFFFFFU < rtimeout) {
        HAL_DEBUGE("parameter [rtimeout] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    USART_RT(uart_dev->periph) &= ~(USART_RT_RT);
    USART_RT(uart_dev->periph) |= rtimeout;

    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      enable receiver timeout
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_receiver_timeout_enable(hal_uart_dev_struct *uart_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((UART3 != uart_dev->periph) && (UART4 != uart_dev->periph) && \
       (UART6 != uart_dev->periph) && (UART7 != uart_dev->periph)) {
        HAL_DEBUGE("parameter [uart_dev->periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    USART_CTL1(uart_dev->periph) |= USART_CTL1_RTEN;

    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      disable receiver timeout
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_receiver_timeout_disable(hal_uart_dev_struct *uart_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((UART3 != uart_dev->periph) && (UART4 != uart_dev->periph) && \
       (UART6 != uart_dev->periph) && (UART7 != uart_dev->periph)) {
        HAL_DEBUGE("parameter [uart_dev->periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    USART_CTL1(uart_dev->periph) &= ~(USART_CTL1_RTEN);

    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      enable mute mode
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_mute_mode_enable(hal_uart_dev_struct *uart_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((UART3 != uart_dev->periph) && (UART4 != uart_dev->periph) && \
       (UART6 != uart_dev->periph) && (UART7 != uart_dev->periph)) {
        HAL_DEBUGE("parameter [uart_dev->periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    USART_CTL0(uart_dev->periph) |= USART_CTL0_MEN;

    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      disable mute mode
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_mute_mode_disable(hal_uart_dev_struct *uart_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((UART3 != uart_dev->periph) && (UART4 != uart_dev->periph) && \
       (UART6 != uart_dev->periph) && (UART7 != uart_dev->periph)) {
        HAL_DEBUGE("parameter [uart_dev->periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    USART_CTL0(uart_dev->periph) &= ~(USART_CTL0_MEN);

    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      enable UART command
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  cmdtype: command type
                only one parameter can be selected which is shown as below:
      \arg        USART_CMD_SBKCMD: send break command
      \arg        USART_CMD_MMCMD: mute mode command
      \arg        USART_CMD_RXFCMD: receive data flush command
      \arg        USART_CMD_TXFCMD: transmit data flush request
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_command_enable(hal_uart_dev_struct *uart_dev, uint32_t cmdtype)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if((UART3 != uart_dev->periph) && (UART4 != uart_dev->periph) && \
       (UART6 != uart_dev->periph) && (UART7 != uart_dev->periph)) {
        HAL_DEBUGE("parameter [uart_dev->periph] value is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((USART_CMD_SBKCMD != cmdtype) && (USART_CMD_MMCMD != cmdtype) && \
       (USART_CMD_SBKCMD != cmdtype) && (USART_CMD_MMCMD != cmdtype)) {
        HAL_DEBUGE("parameter [cmdtype] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    USART_CMD(uart_dev->periph) |= (cmdtype);

    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure UART transmitter
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  txconfig: enable or disable USART transmitter
                only one parameter can be selected which is shown as below:
      \arg        USART_TRANSMIT_ENABLE: enable USART transmission
      \arg        USART_TRANSMIT_DISABLE: disable USART transmission
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_transmit_config(hal_uart_dev_struct *uart_dev, uint32_t txconfig)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((UART3 != uart_dev->periph) && (UART4 != uart_dev->periph) && \
       (UART6 != uart_dev->periph) && (UART7 != uart_dev->periph)) {
        HAL_DEBUGE("parameter [uart_dev->periph] value is invalid");
        return HAL_ERR_VAL;
    }

    if((USART_TRANSMIT_ENABLE != txconfig) && (USART_TRANSMIT_DISABLE != txconfig)) {
        HAL_DEBUGE("parameter [txconfig] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    USART_CTL0(uart_dev->periph) &= ~USART_CTL0_TEN;
    /* configure transfer mode */
    USART_CTL0(uart_dev->periph) |= txconfig;

    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure UART receiver
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  rxconfig: enable or disable USART receiver
                only one parameter can be selected which is shown as below:
      \arg        USART_RECEIVE_ENABLE: enable USART reception
      \arg        USART_RECEIVE_DISABLE: disable USART reception
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_receive_config(hal_uart_dev_struct *uart_dev, uint32_t rxconfig)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((UART3 != uart_dev->periph) && (UART4 != uart_dev->periph) && \
       (UART6 != uart_dev->periph) && (UART7 != uart_dev->periph)) {
        HAL_DEBUGE("parameter [uart_dev->periph] value is invalid");
        return HAL_ERR_VAL;
    }

    if((USART_RECEIVE_ENABLE != rxconfig) && (USART_RECEIVE_DISABLE != rxconfig)) {
        HAL_DEBUGE("parameter [rxconfig] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    USART_CTL0(uart_dev->periph) &= ~USART_CTL0_REN;
    USART_CTL0(uart_dev->periph) |= rxconfig;

    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      enable UART to wakeup the mcu from deep-sleep mode
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_wakeup_enable(hal_uart_dev_struct *uart_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((UART3 != uart_dev->periph) && (UART4 != uart_dev->periph) && \
       (UART6 != uart_dev->periph) && (UART7 != uart_dev->periph)) {
        HAL_DEBUGE("parameter [uart_dev->periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    USART_CTL0(uart_dev->periph) |= USART_CTL0_UESM;

    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      disable UART to wakeup the mcu from deep-sleep mode
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_uart_wakeup_disable(hal_uart_dev_struct *uart_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == uart_dev) {
        HAL_DEBUGE("pointer [uart_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((UART3 != uart_dev->periph) && (UART4 != uart_dev->periph) && \
       (UART6 != uart_dev->periph) && (UART7 != uart_dev->periph)) {
        HAL_DEBUGE("pointer [uart_dev->periph] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock uart */
    HAL_LOCK(uart_dev);

    USART_CTL0(uart_dev->periph) &= ~(USART_CTL0_UESM);

    HAL_UNLOCK(uart_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      return the uart txstate
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_uart_run_state_enum
*/
hal_uart_run_state_enum hal_uart_tx_state_get(hal_uart_dev_struct *uart_dev)
{
    return (hal_uart_run_state_enum)uart_dev->tx_state;
}

/*!
    \brief      return the uart rx state
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_uart_run_state_enum
*/
hal_uart_run_state_enum hal_uart_rx_state_get(hal_uart_dev_struct *uart_dev)
{
    return (hal_uart_run_state_enum)uart_dev->rx_state;
}

/*!
    \brief      return the uart error code
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error state
*/
uint16_t hal_uart_error_get(hal_uart_dev_struct *uart_dev)
{
    return (uart_dev->error_state);
}

/*!
    \brief      reset USART
    \param[in]  uart_periph: UARTx(x=3,4,6,7)
    \param[out] none
    \retval     none
*/
static void _uart_deinit(uint32_t uart_periph)
{
    switch(uart_periph) {
    case USART5:
        /* reset USART5 */
        hal_rcu_periph_reset_enable(RCU_USART5RST);
        hal_rcu_periph_reset_disable(RCU_USART5RST);
        break;
    case UART6:
        /* reset UART6 */
        hal_rcu_periph_reset_enable(RCU_UART6RST);
        hal_rcu_periph_reset_disable(RCU_UART6RST);
        break;
    case UART7:
        /* reset UART7 */
        hal_rcu_periph_reset_enable(RCU_UART7RST);
        hal_rcu_periph_reset_disable(RCU_UART7RST);
        break;
    default:
        break;
    }
}

/*!
    \brief      config uart register
    \param[in]  periph: specify which UART is initialized
    \param[in]  p_uart: the initialization data needed to initialize uart
                  baudrate: communication baudrate
                  parity: USART_PARITY_NONE, USART_PARITY_EVEN, USART_PARITY_ODD
                  wordlength: USART_WORDLENGTH_7B, USART_WORDLENGTH_8B, USART_WORDLENGTH_9B, USART_WORDLENGTH_10B
                  stopbits: USART_STOPBITS_1, USART_STOPBITS_0_5, USART_STOPBITS_2, USART_STOPBITS_1_5
                  mode: USART_MODE_RX, USART_MODE_TX, USART_MODE_TX_RX
                  hwflowctl: USART_HWCONTROL_NONE, USART_HWCONTROL_RTS, USART_HWCONTROL_CTS, USART_HWCONTROL_RTS_CTS
                  oversampling: USART_OVERSAMPLING_8, USART_OVERSAMPLING_16
                  samplingmethod: UART_THREE_SAMPLE_BIT, UART_ONE_SAMPLE_BIT
                  timeoutenable: USART_TIMEOUT_DISABLE, USART_TIMEOUT_ENABLE
                  timeoutvalue: 0 - (2^24-1)
                  fifomode:USART_FIFOMODE_DISABLE, USART_FIFOMODE_ENABLE
                  msbfirst: USART_ADVFEATURE_MSBFIRST_DISABLE, USART_ADVFEATURE_MSBFIRST_ENABLE
                  swap: USART_ADVFEATURE_SWAP_DISABLE, USART_ADVFEATURE_SWAP_ENABLE
                  rxpinlevelinvert: USART_ADVFEATURE_RXINV_DISABLE, USART_ADVFEATURE_RXINV_ENABLE
                  txpinlevelinvert: USART_ADVFEATURE_TXINV_DISABLE, USART_ADVFEATURE_TXINV_ENABLE
                  datainvert: USART_ADVFEATURE_DATAINV_DISABLE, USART_ADVFEATURE_DATAINV_ENABLE
                  overrundisable: USART_ADVFEATURE_OVERRUN_ENABLE, USART_ADVFEATURE_OVERRUN_DISABLE
                  dmadisableonrxerror: USART_ADVFEATURE_DMA_ENABLEONRXERROR, USART_ADVFEATURE_DMA_DISABLEONRXERROR
    \param[out] none
    \retval     none
*/
static void _uart_config(uint32_t periph, hal_uart_init_struct *p_uart)
{
    uint32_t reg_temp = 0U;

    /* configure CTL0 register */
    reg_temp = USART_CTL0(periph);
    reg_temp &= ~(USART_CTL0_PM | USART_CTL0_PCEN | USART_CTL0_WL0 | USART_CTL0_WL1 | \
                  USART_CTL0_OVSMOD | USART_CTL0_REN | USART_CTL0_TEN);
    reg_temp |= (p_uart->mode | p_uart->parity | p_uart->wordlength | p_uart->oversampling);
    USART_CTL0(periph) = reg_temp;

    /* configure CTL1 register */
    reg_temp = USART_CTL1(periph);
    reg_temp &= ~(USART_CTL1_STB | USART_CTL1_STRP | USART_CTL1_RINV | USART_CTL1_TINV | \
                  USART_CTL1_DINV | USART_CTL1_MSBF | USART_CTL1_RTEN);
    reg_temp |= (p_uart->stopbits | p_uart->timeoutenable | p_uart->datainvert | p_uart->rxpinlevelinvert | \
                 p_uart->txpinlevelinvert | p_uart->swap | p_uart->msbfirst);
    USART_CTL1(periph) = reg_temp;

    /* configure CTL2 register */
    reg_temp = USART_CTL2(periph);
    reg_temp &= ~(USART_CTL2_CTSEN | USART_CTL2_RTSEN | USART_CTL2_OSB | USART_CTL2_OVRD | USART_CTL2_DDRE);
    reg_temp |= (p_uart->hwflowctl | p_uart->samplingmethod | p_uart->overrundisable | p_uart->dmadisableonrxerror);
    USART_CTL2(periph) = reg_temp;

    /* configure RT register */
    reg_temp = USART_RT(periph);
    reg_temp &= ~USART_RT_RT;
    reg_temp |= p_uart->timeoutvalue;
    USART_RT(periph) = reg_temp;

    /* configure FCS register */
    reg_temp = USART_FCS(periph);
    reg_temp &= ~(USART_FCS_FEN | USART_FCS_RFTCFG | USART_FCS_TFTCFG);
    reg_temp |= (p_uart->fifomode | p_uart->rxthreshold | p_uart->txthreshold);
    USART_FCS(periph) = reg_temp;

    /* configure baudrate */
    hals_usart_baudrate_set(periph, p_uart->baudrate);

    /* disable the UART mode, Half-Duplex mode, IRDA mode, LIN mode and clock */
    USART_CTL2(periph) &= ~(USART_CTL2_SCEN | USART_CTL2_HDEN | USART_CTL2_IREN);
    USART_CTL1(periph) &= ~(USART_CTL1_LMEN | USART_CTL1_CKEN);
    USART_CTL0(periph) &= ~(USART_CTL0_MEN);
}

/*!
    \brief      get the mask of data bit
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint16_t:0x3F or 0x7F or 0xFF or 0x1FF or 0x3FF
*/
static uint16_t _uart_data_bit_mask_get(hal_uart_dev_struct *uart_dev)
{
    uint16_t retval;

    /* check whether the PCEN is enabled */
    if(RESET != (USART_CTL0(uart_dev->periph) & USART_CTL0_PCEN)) {
        if(USART_WORDLENGTH_8B == (USART_CTL0(uart_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1))) {
            retval = 0x007FU;
        } else if(USART_WORDLENGTH_9B == (USART_CTL0(uart_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1))) {
            retval = 0x00FFU;
        } else if(USART_WORDLENGTH_7B == (USART_CTL0(uart_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1))) {
            retval = 0x003FU;
        } else {
            retval = 0x01FFU;
        }
    } else {
        if(USART_WORDLENGTH_8B == (USART_CTL0(uart_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1))) {
            retval = 0x00FFU;
        } else if(USART_WORDLENGTH_9B == (USART_CTL0(uart_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1))) {
            retval = 0x01FFU;
        } else if(USART_WORDLENGTH_7B == (USART_CTL0(uart_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1))) {
            retval = 0x007FU;
        } else {
            retval = 0x03FFU;
        }
    }

    return retval;
}

/*!
    \brief      handle the transmit complete interrupt
    \param[in]  uart_dev: pointer to a uart device information structure
    \param[out] none
    \retval     none
*/
static void _uart_transmit_complete_interrupt(void *uart_dev)
{
    hal_uart_dev_struct *p_uart = uart_dev;
    hal_uart_user_cb p_func     = (hal_uart_user_cb)p_uart->transmit_complete_callback;

    /* disable the transmit complete interrupt */
    hals_usart_interrupt_disable(p_uart->periph, USART_INT_TC);

    /* reset transmit_complete_handle and tx_state */
    p_uart->tx_state = HAL_UART_STATE_READY;
    p_uart->uart_irq.transmit_complete_handle = NULL;

    /* if there is a user transmit complete callback */
    if(NULL != p_func) {
        p_func(p_uart);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the transmit fifo empty interrupt
    \param[in]  uart_dev: pointer to a uart device information structure
    \param[out] none
    \retval     none
*/
static void _uart_transmit_fifo_empty_interrupt(void *uart_dev)
{
    hal_uart_dev_struct *p_uart = uart_dev;
    hal_smartcard_user_cb p_func;

    uint16_t tmp = 0U;

    p_func = (hal_smartcard_user_cb)p_uart->transmit_fifo_empty_callback;

    if(0U != p_uart->txbuffer.remain) {
        /* send data: in case of 9bit&no parity, uint16_t data will be transmitted */
        if((USART_WORDLENGTH_10B == p_uart->init.wordlength) || \
           ((USART_WORDLENGTH_9B == p_uart->init.wordlength) && (USART_PARITY_NONE == p_uart->init.parity))) {
            /* 9-bit data, none parity and 10 bit */
            tmp = (uint16_t)(*p_uart->txbuffer.buffer & 0x03FFU);
            hals_usart_data_transmit(p_uart->periph, tmp);
            p_uart->txbuffer.buffer += 2;
        } else {
            /* 9-bit data, with parity or 8-bit data */
            tmp = (uint8_t)(*p_uart->txbuffer.buffer & 0xFFU);
            hals_usart_data_transmit(p_uart->periph, tmp);
            p_uart->txbuffer.buffer++;
        }
        p_uart->txbuffer.remain--;
    } else {
        /* disable the TBE interrupt, enable the TC interrupt and reset the transmit_fifo_empty_handle */
        hals_usart_interrupt_disable(p_uart->periph, USART_INT_TBE);
        hals_usart_interrupt_enable(p_uart->periph, USART_INT_TC);

        /* reset transmit fifo empty callback pointer */
        p_uart->uart_irq.transmit_fifo_empty_handle = NULL;

        /* call transmit fifo empty callback function */
        if(NULL != p_func) {
            p_func(uart_dev);
        } else {
            /* do nothing */
        }
    }
}

/*!
    \brief      handle the receive interrupt
    \param[in]  uart_dev: pointer to a uart device information structure
    \param[out] none
    \retval     none
*/
static void _uart_receive_interrupt(void *uart_dev)
{
    hal_uart_dev_struct *p_uart = uart_dev;
    hal_uart_user_cb p_func     = (hal_uart_user_cb)p_uart->receive_complete_callback;

    /* store the received data */
    if((USART_WORDLENGTH_10B == p_uart->init.wordlength) || \
       ((USART_WORDLENGTH_9B == p_uart->init.wordlength) && (USART_PARITY_NONE == p_uart->init.parity))) {
        *(uint16_t *)(uint32_t)p_uart->rxbuffer.buffer = (hals_usart_data_receive(p_uart->periph) & p_uart->data_bit_mask);
        p_uart->rxbuffer.buffer += 2U;
    } else {
        *p_uart->rxbuffer.buffer = (uint8_t)(hals_usart_data_receive(p_uart->periph) & p_uart->data_bit_mask);
        p_uart->rxbuffer.buffer++;
    }
    p_uart->rxbuffer.remain--;

    /* receive is finished */
    if(0U == p_uart->rxbuffer.remain) {
        /* disable PERR, ERR, RBNE interrupt */
        hals_usart_interrupt_disable(p_uart->periph, USART_INT_PERR);
        hals_usart_interrupt_disable(p_uart->periph, USART_INT_ERR);
        hals_usart_interrupt_disable(p_uart->periph, USART_INT_RBNE);

        /* reset receive_complete_handle and rx_state */
        p_uart->rx_state = HAL_UART_STATE_READY;
        p_uart->uart_irq.receive_complete_handle = NULL;

        /* if there is a user receive complete callback */
        if(NULL != p_func) {
            p_func(p_uart);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      uart receive interrupt handler
    \param[in]  uart_dev: pointer to a uart device information structure
    \param[out] none
    \retval     none
*/
static void _uart_received_fifo_full_interrupt(void *uart_dev)
{
    hal_uart_dev_struct *p_uart = uart_dev;
    hal_uart_user_cb p_func     = (hal_uart_user_cb)p_uart->receive_fifo_full_callback;

    /* if there is a user receive fifo full callback */
    if(NULL != p_func) {
        p_func(p_uart);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      uart idle line detect handler
    \param[in]  uart_dev: pointer to a uart device information structure
    \param[out] none
    \retval     none
*/
static void _uart_idle_line_detected_interrupt(void *uart_dev)
{
    hal_uart_dev_struct *p_uart = uart_dev;
    hal_uart_user_cb p_func     = (hal_uart_user_cb)p_uart->idle_receive_callback;

    /* if there is a user idle receive callback */
    if(NULL != p_func) {
        p_func(p_uart);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      end ongoing transfer on usart peripheral
    \param[in]  uart_dev: uart device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _uart_end_transfer(hal_uart_dev_struct *uart_dev)
{
    /* disable the RBNE, TBE, PERR, TC, ERR interrupt */
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_RBNE);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TBE);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TC);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_ERR);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_RT);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_IDLE);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_RFT);
    hals_usart_interrupt_disable(uart_dev->periph, USART_INT_TFT);

    /* reset rx_state and tx_state */
    uart_dev->rx_state = HAL_UART_STATE_READY;
    uart_dev->tx_state = HAL_UART_STATE_READY;
}

/*!
    \brief      handle the error interrupt
    \param[in]  uart_dev: pointer to a uart device information structure
    \param[out] none
    \retval     none
*/
static void _uart_error_interrupt(void *uart_dev)
{
    hal_uart_dev_struct *p_uart = uart_dev;
    hal_uart_user_cb p_func     = (hal_uart_user_cb)p_uart->error_callback;

    p_uart->error_state           = (uint16_t)HAL_UART_ERROR_NONE;
    p_uart->uart_irq.error_handle = NULL;

    _uart_end_transfer(p_uart);

    /* if there is a user receive complete callback */
    if(NULL != p_func) {
        p_func(p_uart);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the uart DMA transmit process complete
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _uart_dma_tx_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_uart_dev_struct *p_uart;
    hal_uart_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_uart = (hal_uart_dev_struct *)p_dma->p_periph;
    p_func = (hal_uart_user_cb)p_uart->transmit_complete_callback;

    /* DMA normal mode */
    if(0U == (DMA_CHCTL(p_dma->dma_periph, p_dma->channel) & DMA_CHXCTL_CMEN)) {
        p_uart->txbuffer.remain = 0U;
        hals_usart_dma_transmit_config(p_uart->periph, USART_TRANSMIT_DMA_DISABLE);
        /* enable TC interrupt */
        hals_usart_interrupt_enable(p_uart->periph, USART_INT_TC);
        /* reset rx_state */
        p_uart->tx_state = HAL_UART_STATE_READY;
    } else {
        /* do nothing */
    }

    /* if there is a user tx complete callback */
    if(NULL != p_func) {
        p_func(p_uart);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the uart DMA transmit process half complete
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _uart_dma_tx_half_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_uart_dev_struct *p_uart;
    hal_uart_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_uart = (hal_uart_dev_struct *)p_dma->p_periph;
    p_func = (hal_uart_user_cb)p_uart->transmit_half_complete_callback;

    /* if there is a user tx half complete callback */
    if(NULL != p_func) {
        p_func(p_uart);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the uart DMA receive process complete
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _uart_dma_rx_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_uart_dev_struct *p_uart;
    hal_uart_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_uart = (hal_uart_dev_struct *)p_dma->p_periph;
    p_func = (hal_uart_user_cb)p_uart->receive_complete_callback;

    /* DMA normal mode */
    if(0U == (DMA_CHCTL(p_dma->dma_periph, p_dma->channel) & DMA_CHXCTL_CMEN)) {
        p_uart->rxbuffer.remain = 0U;
        /* disable DMA receive, PERR and ERR interrupt */
        hals_usart_dma_receive_config(p_uart->periph, USART_RECEIVE_DMA_DISABLE);
        hals_usart_interrupt_disable(p_uart->periph, USART_INT_PERR);
        hals_usart_interrupt_disable(p_uart->periph, USART_INT_ERR);
        /* reset rx_state */
        p_uart->rx_state = HAL_UART_STATE_READY;
    } else {
        /* do nothing */
    }

    /* if there is a user receive complete callback */
    if(NULL != p_func) {
        p_func(p_uart);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the uart DMA receive process half complete
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _uart_dma_rx_half_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_uart_dev_struct *p_uart;
    hal_uart_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_uart = (hal_uart_dev_struct *)p_dma->p_periph;
    p_func = (hal_uart_user_cb)p_uart->receive_half_complete_callback;

    /* if there is a user receive half complete callback */
    if(NULL != p_func) {
        p_func(p_uart);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the uart DMA error process
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _uart_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_uart_dev_struct *p_uart;
    hal_uart_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_uart = (hal_uart_dev_struct *)p_dma->p_periph;
    p_func = (hal_uart_user_cb)p_uart->error_callback;

    if(HAL_UART_STATE_BUSY == p_uart->tx_state) {
        p_uart->txbuffer.remain = 0U;
        /* disable DMA transmit and reset tx_state */
        hals_usart_dma_transmit_config(p_uart->periph, USART_TRANSMIT_DMA_DISABLE);
        p_uart->tx_state = HAL_UART_STATE_READY;
    } else if(HAL_UART_STATE_BUSY == p_uart->rx_state) {
        p_uart->rxbuffer.remain = 0U;
        /* disable DMA receive, PERR, ERR interrupt */
        hals_usart_dma_receive_config(p_uart->periph, USART_RECEIVE_DMA_DISABLE);
        hals_usart_interrupt_disable(p_uart->periph, USART_INT_PERR);
        hals_usart_interrupt_disable(p_uart->periph, USART_INT_ERR);
        /* reset rx_state */
        p_uart->rx_state = HAL_UART_STATE_READY;
    } else {
        /* do nothing */
    }

    /* configuration error state */
    p_uart->error_state |= (uint16_t)HAL_UART_ERROR_DMA;
    p_uart->last_error = (uint16_t)HAL_UART_ERROR_DMA;

    /* if there is a user error callback */
    if(NULL != p_func) {
        p_func(p_uart);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      uart dma abort complete handle
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _uart_dma_abort_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_uart_dev_struct *p_uart;
    hal_uart_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_uart = (hal_uart_dev_struct *)p_dma->p_periph;
    p_func = (hal_uart_user_cb)p_uart->abort_complete_callback;

    /* reset transmit and received buff */
    p_uart->txbuffer.remain = 0U;
    p_uart->rxbuffer.remain = 0U;

    /* reset state and error state */
    p_uart->tx_state    = HAL_UART_STATE_READY;
    p_uart->rx_state    = HAL_UART_STATE_READY;
    p_uart->error_state = (uint16_t)HAL_UART_ERROR_NONE;

    /* reset transmit/received callback */
    p_uart->uart_irq.transmit_fifo_empty_handle = NULL;
    p_uart->uart_irq.transmit_complete_handle   = NULL;
    p_uart->uart_irq.error_handle               = NULL;
    p_uart->uart_irq.receive_fifo_full_handle   = NULL;
    p_uart->uart_irq.receive_complete_handle    = NULL;
    p_uart->uart_irq.error_handle               = NULL;

    /* reset user callback */
    p_uart->transmit_complete_callback   = NULL;
    p_uart->transmit_fifo_empty_callback = NULL;
    p_uart->receive_complete_callback    = NULL;
    p_uart->receive_fifo_full_callback   = NULL;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(p_uart->periph, USART_INT_FLAG_PERR);
    hals_usart_interrupt_flag_clear(p_uart->periph, USART_INT_FLAG_ERR_FERR);
    hals_usart_interrupt_flag_clear(p_uart->periph, USART_INT_FLAG_ERR_NERR);
    hals_usart_interrupt_flag_clear(p_uart->periph, USART_INT_FLAG_ERR_ORERR);
    hals_usart_interrupt_flag_clear(p_uart->periph, USART_INT_FLAG_RT);
    hals_usart_interrupt_flag_clear(p_uart->periph, USART_INT_FLAG_EB);

    /* if there is a user error callback */
    if(NULL != p_func) {
        p_func(p_uart);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      uart dma abort only tx complete handle
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _uart_dma_onlytx_abort_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_uart_dev_struct *p_uart;
    hal_uart_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_uart = (hal_uart_dev_struct *)p_dma->p_periph;
    p_func = (hal_uart_user_cb)p_uart->abort_tx_complete_callback;

    /* reset transmit buff, state and error state */
    p_uart->txbuffer.remain = 0U;
    p_uart->tx_state        = HAL_UART_STATE_READY;
    p_uart->error_state     = (uint16_t)HAL_UART_ERROR_NONE;

    /* reset transmit callback */
    p_uart->uart_irq.transmit_fifo_empty_handle = NULL;
    p_uart->uart_irq.transmit_complete_handle   = NULL;
    p_uart->uart_irq.error_handle               = NULL;

    /* reset user callback */
    p_uart->transmit_complete_callback   = NULL;
    p_uart->transmit_fifo_empty_callback = NULL;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(p_uart->periph, USART_INT_FLAG_ERR_FERR);

    /* if there is a user error callback */
    if(NULL != p_func) {
        p_func(p_uart);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      uart dma abort only rx complete handle
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _uart_dma_onlyrx_abort_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_uart_dev_struct *p_uart;
    hal_uart_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_uart = (hal_uart_dev_struct *)p_dma->p_periph;
    p_func = (hal_uart_user_cb)p_uart->abort_rx_complete_callback;

    /* reset received buff, state and error state */
    p_uart->rxbuffer.remain = 0U;
    p_uart->rx_state        = HAL_UART_STATE_READY;
    p_uart->error_state     = (uint16_t)HAL_UART_ERROR_NONE;

    /* reset the Rx callback function pointer */
    p_uart->uart_irq.receive_fifo_full_handle = NULL;
    p_uart->uart_irq.receive_complete_handle  = NULL;
    p_uart->uart_irq.error_handle             = NULL;

    /* clear user callback */
    p_uart->receive_complete_callback  = NULL;
    p_uart->receive_fifo_full_callback = NULL;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(p_uart->periph, USART_INT_FLAG_PERR);
    hals_usart_interrupt_flag_clear(p_uart->periph, USART_INT_FLAG_ERR_NERR);
    hals_usart_interrupt_flag_clear(p_uart->periph, USART_INT_FLAG_ERR_ORERR);
    hals_usart_interrupt_flag_clear(p_uart->periph, USART_INT_FLAG_RT);
    hals_usart_interrupt_flag_clear(p_uart->periph, USART_INT_FLAG_EB);

    /* if there is a user error callback */
    if(NULL != p_func) {
        p_func(p_uart);
    } else {
        /* do nothing */
    }
}
