/*!
    \file    gd32h7xx_hal_trng.c
    \brief   TRNG driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define TRNG_STAT_SECS_TIMEOUT   ((uint32_t)0x0004FFFFU)
#define TRNG_STAT_DRDY_TIMEOUT   ((uint32_t)0x0000000FU)
#define TRNG_CTL_CONDRST_TIMEOUT ((uint32_t)0x0000000FU)

/* trng random data ready interrupt handler */
static void _trng_random_data_ready_callback(void *trng);
/* trng error interrupt handler */
static void __trng_error_callback(void *trng);

/*!
    \brief      deinitialize TRNG
    \param[in]  trng_dev: TRNG device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_trng_deinit(hal_trng_dev_struct *trng_dev)
{
   int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == trng_dev) {
        HAL_DEBUGE("pointer [trng_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* reset trng */
    hal_rcu_periph_reset_enable(RCU_TRNGRST);
    hal_rcu_periph_reset_disable(RCU_TRNGRST);
    /* Initialize the device structure to the default value */
    hal_trng_struct_init(HAL_TRNG_DEV_STRUCT, trng_dev);

    return error_code;
}

/*!
    \brief      initialize the TRNG structure with default values
    \param[in]  hal_struct_type: type of TRNG structure for initialization
                  only one  parameters can be selected which are shown as below
      \arg        HAL_TRNG_INIT_STRUCT: initialization structure
      \arg        HAL_TRNG_DEV_STRUCT: device information structure
    \param[out] p_struct: pointer to TRNG structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_trng_struct_init(hal_trng_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_TRNG_INIT_STRUCT:
        /* Initialize the init structure to the default value */
        ((hal_trng_init_struct *)p_struct)->mode             = TRNG_MODSEL_NIST;
        ((hal_trng_init_struct *)p_struct)->clkerr_detection = TRNG_CLKERR_DISABLE;
        ((hal_trng_init_struct *)p_struct)->clkdiv           = TRNG_CLK_DIV1;
        ((hal_trng_init_struct *)p_struct)->power_mode       = TRNG_NR_HIGH;
        ((hal_trng_init_struct *)p_struct)->cond             = TRNG_COND_ENABLE;
        ((hal_trng_init_struct *)p_struct)->hash_init        = TRNG_HASH_NOINIT;
        ((hal_trng_init_struct *)p_struct)->algorithm        = TRNG_ALGO_SHA1;
        ((hal_trng_init_struct *)p_struct)->input            = TRNG_INMOD_256BIT;
        ((hal_trng_init_struct *)p_struct)->output           = TRNG_OUTMOD_128BIT;
        ((hal_trng_init_struct *)p_struct)->replace_test     = TRNG_REPTEST_DISABLE;
        ((hal_trng_init_struct *)p_struct)->postprocess      = TRNG_PPROCESS_DISABLE;
        ((hal_trng_init_struct *)p_struct)->adpo_threshold   = 691U;
        ((hal_trng_init_struct *)p_struct)->rep_threshold    = 40U;
        break;
    case HAL_TRNG_DEV_STRUCT:
        /* Initialize the device structure to the default value */
        ((hal_trng_dev_struct *)p_struct)->state                       = HAL_TRNG_STATE_RESET;
        ((hal_trng_dev_struct *)p_struct)->mutex                       = HAL_MUTEX_UNLOCKED;
        ((hal_trng_dev_struct *)p_struct)->trng_irq.clock_error_handle = NULL;
        ((hal_trng_dev_struct *)p_struct)->trng_irq.seed_error_handle  = NULL;
        ((hal_trng_dev_struct *)p_struct)->trng_irq.random_data_ready  = NULL;
        ((hal_trng_dev_struct *)p_struct)->ready_callback              = NULL;
        ((hal_trng_dev_struct *)p_struct)->error_callback              = NULL;
        ((hal_trng_dev_struct *)p_struct)->random_data                 = 0U;
        ((hal_trng_dev_struct *)p_struct)->error_code                  = TRNG_ERROR_NONE;
        break;
    case HAL_TRNG_CONFIG_STRUCT:
        /* Initialize the config structure to the default value */
        ((hal_trng_config_struct *)p_struct)->mode              = TRNG_MODSEL_LFSR;
        ((hal_trng_config_struct *)p_struct)->clkerr_detection  = 0U;
        ((hal_trng_config_struct *)p_struct)->clkdiv            = 0U;
        ((hal_trng_config_struct *)p_struct)->power_mode        = 0U;
        ((hal_trng_config_struct *)p_struct)->cond              = 0U;
        ((hal_trng_config_struct *)p_struct)->hash_init         = 0U;
        ((hal_trng_config_struct *)p_struct)->algorithm         = 0U;
        ((hal_trng_config_struct *)p_struct)->input             = TRNG_INMOD_256BIT;
        ((hal_trng_config_struct *)p_struct)->output            = TRNG_OUTMOD_128BIT;
        ((hal_trng_config_struct *)p_struct)->replace_test      = 0U;
        ((hal_trng_config_struct *)p_struct)->postprocess       = 0U;
        ((hal_trng_config_struct *)p_struct)->adpo_threshold    = 0U;
        ((hal_trng_config_struct *)p_struct)->rep_threshold     = 0U;
        break;
    case HAL_TRNG_IRQ_STRUCT:
        /* Initialize the IRQ structure to the default value */
        ((hal_trng_irq_struct *)p_struct)->clock_error_handle   = NULL;
        ((hal_trng_irq_struct *)p_struct)->seed_error_handle    = NULL;
        ((hal_trng_irq_struct *)p_struct)->random_data_ready    = NULL;
        ((hal_trng_irq_struct *)p_struct)->nist_mode_error      = NULL;
        break;
    case HAL_TRNG_USER_CALLBACK_STRUCT:
        /* Initialize the user callback structure to the default value */
        ((hal_trng_user_callback_struct *)p_struct)->ready_data_func   = NULL;
        ((hal_trng_user_callback_struct *)p_struct)->error_func        = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        error_code = HAL_ERR_VAL;
        break;
    }

    return error_code;
}

/*!
    \brief      initialize TRNG
    \param[in]  trng_dev: TRNG device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_init: TRNG init parameter struct
                  mode: operation mode
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_MODSEL_LFSR: TRNG working in LFSR mode
      \arg          TRNG_MODSEL_NIST: TRNG working in NIST mode
                  clkerr_detection: clock error detection
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_CLKERR_DISABLE: TRNG clock error detection disable
      \arg          TRNG_CLKERR_ENABLE: TRNG clock error detection enable
                  clkdiv: clock division factor
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_CLK_DIV1: TRNG clock TRNG_CLK divider 1
      \arg          TRNG_CLK_DIV2: TRNG clock TRNG_CLK divider 2
      \arg          TRNG_CLK_DIV4: TRNG clock TRNG_CLK divider 4
      \arg          TRNG_CLK_DIV8: TRNG clock TRNG_CLK divider 8
      \arg          TRNG_CLK_DIV16: TRNG clock TRNG_CLK divider 16
      \arg          TRNG_CLK_DIV32: TRNG clock TRNG_CLK divider 32
      \arg          TRNG_CLK_DIV64: TRNG clock TRNG_CLK divider 64
      \arg          TRNG_CLK_DIV128: TRNG clock TRNG_CLK divider 128
      \arg          TRNG_CLK_DIV256: TRNG clock TRNG_CLK divider 256
      \arg          TRNG_CLK_DIV512: TRNG clock TRNG_CLK divider 512
      \arg          TRNG_CLK_DIV1024: TRNG clock TRNG_CLK divider 1024
      \arg          TRNG_CLK_DIV2048: TRNG clock TRNG_CLK divider 2048
      \arg          TRNG_CLK_DIV4096: TRNG clock TRNG_CLK divider 4096
      \arg          TRNG_CLK_DIV8192: TRNG clock TRNG_CLK divider 8192
      \arg          TRNG_CLK_DIV16384: TRNG clock TRNG_CLK divider 16384
      \arg          TRNG_CLK_DIV32768: TRNG clock TRNG_CLK divider 32768
                  power_mode: power mode
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_NR_ULTRALOW: TRNG analog power mode ultralow
      \arg          TRNG_NR_LOW: TRNG analog power mode low
      \arg          TRNG_NR_MEDIUM: TRNG analog power mode medium
      \arg          TRNG_NR_HIGH: TRNG analog power mode high
                  cond: training unit module
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_COND_DISABLE: TRNG conditioning module disable
      \arg          TRNG_COND_ENABLE: TRNG conditioning module enable
                  hash_init: whether to initialize the hash or not
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_HASH_NOINIT: TRNG hash algorithm no init
      \arg          TRNG_HASH_INIT: TRNG hash algorithm init
                  algorithm: algorithm configuration
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_ALGO_SHA1: TRNG conditioning module hash SHA1
      \arg          TRNG_ALGO_MD5: TRNG conditioning module hash MD5
      \arg          TRNG_ALGO_SHA224: TRNG conditioning module hash SHA224
      \arg          TRNG_ALGO_SHA256: TRNG conditioning module hash SHA256
                  input: input mode
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_INMOD_256BIT: conditioning module input bitwidth 256bits
      \arg          TRNG_INMOD_440BIT: conditioning module input bitwidth 440bits
                  output: output mode
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_OUTMOD_128BIT: conditioning module output bitwidth 128bits
      \arg          TRNG_OUTMOD_256BIT: conditioning module output bitwidth 256bits
                  replace_test: replace test
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_REPTEST_DISABLE: TRNG replace test disable
      \arg          TRNG_REPTEST_ENABLE: TRNG replace test enable
                  postprocess: post-processing unit config
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_PPROCESS_DISABLE: TRNG post-processing disable
      \arg          TRNG_PPROCESS_ENABLE: TRNG post-processing enable
                  adpo_threshold: adaptive proportion test thresholds value
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_HTCFG_APTTH: adaptive proportion test threshold
                  rep_threshold: repetitive (00/11) test thresholds value
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_HTCFG_RCTTH: repetition (00/11) count test threshold
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_trng_init(hal_trng_dev_struct *trng_dev, hal_trng_init_struct *p_init)
{
    uint32_t trng_config = 0U;
    int32_t error_code   = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == trng_dev) || (NULL == p_init)) {
        HAL_DEBUGE("pointer [p_init] or [trng_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Set state and  mutext lock*/
    trng_dev->mutex = HAL_MUTEX_LOCKED;
    trng_dev->state = HAL_TRNG_STATE_BUSY;

    /* The conditions for the configuration to take effect are set,set TRNG_CTL_CONDRST to 1*/
    TRNG_CTL &= ~TRNG_CTL_LK;
    TRNG_CTL |= TRNG_CTL_CONDRST;

    /* Clear the associated register bit */
    TRNG_CTL &= (~(uint32_t)(TRNG_CTL_MODSEL | TRNG_CTL_CED | TRNG_CTL_CLKDIV | TRNG_CTL_NR | TRNG_CTL_CONDEN | \
                             TRNG_CTL_INIT | TRNG_CTL_ALGO | TRNG_CTL_INMOD | TRNG_CTL_OUTMOD | TRNG_CTL_RTEN | \
                             TRNG_CTL_PPEN));

    /* operation mode */
    TRNG_CTL |= p_init->mode;
    /* Random number seed input mode selection */
    TRNG_CTL |= ((uint32_t)p_init->input | (uint32_t)p_init->output);
    /* post-processing unit config */
    TRNG_CTL |= p_init->postprocess;
    /* algorithm configuration */
    TRNG_CTL |= p_init->algorithm;
    /* clock error detection, clock division factor */
    TRNG_CTL |= (p_init->clkerr_detection | p_init->clkdiv);
    /* power mode */
    TRNG_CTL |= p_init->power_mode;
    /* training unit module */
    TRNG_CTL |= p_init->cond;
    /* hash init */
    TRNG_CTL |= p_init->hash_init;
    /* Random number output mode selection */
    TRNG_CTL |= p_init->replace_test;

    /* reset aptth and rctth value */
    TRNG_HTCFG &= ~(TRNG_HTCFG_APTTH | TRNG_HTCFG_RCTTH);

    /* adaptive proportion test thresholds value, repetitive test thresholds value*/
    TRNG_HTCFG |= (HTCFG_APTTH(p_init->adpo_threshold) | HTCFG_RCTTH(p_init->rep_threshold));

    trng_config = TRNG_CTL;
    trng_config &= ~TRNG_CTL_CONDRST;
    trng_config |= TRNG_CTL_TRNGEN;
    TRNG_CTL = trng_config;

    trng_dev->state = HAL_TRNG_STATE_READY;
    trng_dev->mutex = HAL_MUTEX_UNLOCKED;

    return error_code;
}

/*!
    \brief      set user-defined interrupt callback function, which will be registered and
                called when corresponding interrupt be triggered
    \param[in]  trng_dev: TRNG device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to TRNG interrupt callback functions structure
                  The structure contains the following members:
      \arg        clock_error_handle: TRNG clock error interrupt callback function
                  if not NULL, the callback mechanism is in use and corresponding interrupt is enabled
                  if NULL, the callback mechanism is out of use and corresponding interrupt is disabled
      \arg        seed_error_handle: TRNG seed error interrupt callback function
                  if not NULL, the callback mechanism is in use and corresponding interrupt is enabled
                  if NULL, the callback mechanism is out of use and corresponding interrupt is disabled
      \arg        random_data_ready: TRNG random data ready interrupt callback function
                  if not NULL, the callback mechanism is in use and corresponding interrupt is enabled
                  if NULL, the callback mechanism is out of use and corresponding interrupt is disabled
      \arg        nist_mode_error: TRNG NIST mode error interrupt callback function
                  if not NULL, the callback mechanism is in use and corresponding interrupt is enabled
                  if NULL, the callback mechanism is out of use and corresponding interrupt is disabled
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_trng_irq_handle_set(hal_trng_dev_struct *trng_dev, hal_trng_irq_struct *p_irq)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == trng_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [trng_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    trng_dev->mutex = HAL_MUTEX_LOCKED;

    trng_dev->trng_irq.clock_error_handle = p_irq->clock_error_handle;
    trng_dev->trng_irq.seed_error_handle = p_irq->seed_error_handle;
    trng_dev->trng_irq.random_data_ready = p_irq->random_data_ready;
    trng_dev->trng_irq.nist_mode_error = p_irq->nist_mode_error;

    if((NULL != p_irq->clock_error_handle) || (NULL != p_irq->seed_error_handle) || \
       (NULL != p_irq->random_data_ready) || (NULL != p_irq->nist_mode_error)) {
        hals_trng_interrupt_enable();
    } else {
        hals_trng_interrupt_disable();
    }

    trng_dev->mutex = HAL_MUTEX_UNLOCKED;

    return error_code;
}

/*!
    \brief      reset all user-defined interrupt callback function, which will be registered and
                called when corresponding interrupt be triggered
    \param[in]  trng_dev: TRNG device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_trng_irq_handle_all_reset(hal_trng_dev_struct *trng_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == trng_dev) {
        HAL_DEBUGE("pointer [trng_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    trng_dev->mutex = HAL_MUTEX_LOCKED;

    trng_dev->trng_irq.clock_error_handle = NULL;
    trng_dev->trng_irq.seed_error_handle = NULL;
    trng_dev->trng_irq.random_data_ready = NULL;
    trng_dev->trng_irq.nist_mode_error = NULL;

    hals_trng_interrupt_disable();

    trng_dev->mutex = HAL_MUTEX_UNLOCKED;

    return error_code;
}

/*!
    \brief      TRNG interrupt handler content function, which is merely used in  HAU_TRNG_IRQHandler
    \param[in]  trng_dev: TRNG device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
void hal_trng_irq(hal_trng_dev_struct *trng_dev)
{
    uint32_t stat_secs_timeout_cnt = 0U;
    /* clock error interrupt handler */
    if(SET == hals_trng_interrupt_flag_get(TRNG_INT_FLAG_CEIF)) {
        hals_trng_interrupt_flag_clear(TRNG_INT_FLAG_CEIF);
        trng_dev->state = HAL_TRNG_STATE_ERROR;
        trng_dev->error_code = TRNG_ERROR_CLOCK;
        if(NULL != trng_dev->trng_irq.clock_error_handle) {
            trng_dev->trng_irq.clock_error_handle(trng_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* seed error interrupt handler */
    if(SET == hals_trng_interrupt_flag_get(TRNG_INT_FLAG_SEIF)) {
        /* reset TRNG conditioning logic */
        TRNG_CTL |= TRNG_CTL_CONDRST;
        TRNG_CTL &= ~TRNG_CTL_CONDRST;
        /* clear seed error interrupt flag */
        hals_trng_interrupt_flag_clear(TRNG_INT_FLAG_SEIF);
        /* wait seed error interrupt flag be clear */
        while((stat_secs_timeout_cnt < TRNG_STAT_SECS_TIMEOUT) && (TRNG_STAT & TRNG_STAT_SECS)) {
            stat_secs_timeout_cnt++;
            trng_dev->state = HAL_TRNG_STATE_TIMEOUT;
        }

        if(NULL != trng_dev->trng_irq.seed_error_handle) {
            trng_dev->error_code = TRNG_ERROR_SEED;
            trng_dev->trng_irq.seed_error_handle(trng_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /*!< nist mode error handle*/
    if(SET == hals_trng_interrupt_flag_get(TRNG_INT_FLAG_ERRSTA)) {
        if(NULL != trng_dev->trng_irq.nist_mode_error) {
            trng_dev->trng_irq.nist_mode_error(trng_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* random Data ready interrupt handle */
    if(SET == hals_trng_interrupt_flag_get(TRNG_INT_FLAG_DRDY)) {
        if(NULL != trng_dev->trng_irq.random_data_ready) {
            trng_dev->trng_irq.random_data_ready(trng_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    hals_trng_interrupt_disable();
}

/*!
    \brief      check whether the TRNG lfsr is ready for run
    \param[in]  none
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_trng_lfsr_ready_check(void)
{
    uint32_t timeout     = 0U;
    int32_t error_code   = HAL_ERR_NONE;
    FlagStatus trng_flag = RESET;

    /* check whether the random data is valid */
    do {
        timeout++;
        trng_flag = hals_trng_flag_get(TRNG_FLAG_DRDY);
    } while((RESET == trng_flag) && (0xFFFFU > timeout));

    if(0xFFFFU <= timeout) {
        error_code = HAL_ERR_TIMEOUT;
    } else{
        if(SET == trng_flag) {
            /* Check for a clock error */
            if(RESET == hals_trng_flag_get(TRNG_FLAG_CECS)) {
                error_code = HAL_ERR_NONE;
            } else {
                error_code = HAL_ERR_BUSY;
            }
        } else {
            /* do nothing */
        }
    }

    return error_code;
}

/*!
    \brief      check whether the TRNG NIST is ready for run
    \param[in]  none
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE, HAL_ERR_NO_SUPPORT details refer to gd32h7xx_hal.h
*/
int32_t hal_trng_nist_ready_check(void)
{
    uint32_t timeout     = 0U;
    int32_t error_code   = HAL_ERR_NONE;
    FlagStatus trng_flag = RESET;

    /* check whether the random data is valid */
    do {
        timeout++;
        trng_flag = hals_trng_flag_get(TRNG_FLAG_DRDY);
    } while((RESET == trng_flag) && (0x5FFFFU > timeout));

    if(0x5FFFFU <= timeout) {
        error_code = HAL_ERR_TIMEOUT;
    } else {
        if(SET == trng_flag) {
            if(RESET == hals_trng_interrupt_flag_get(TRNG_INT_FLAG_CEIF)) {
                if(RESET == hals_trng_interrupt_flag_get(TRNG_INT_FLAG_SEIF)) {
                    if(RESET == hals_trng_flag_get(TRNG_FLAG_CECS)) {
                        if(RESET == hals_trng_flag_get(TRNG_FLAG_SECS)) {
                            error_code = HAL_ERR_NONE;
                        } else {
                            error_code = HAL_ERR_NO_SUPPORT;
                        }
                    } else {
                        error_code = HAL_ERR_NO_SUPPORT;
                    }
                } else {
                    error_code = HAL_ERR_NO_SUPPORT;
                }
            } else {
                error_code = HAL_ERR_NO_SUPPORT;
            }
        } else {
            /* do nothing */
        }
    }

    return error_code;
}

/*!
    \brief      sequence to recover from a seed error
    \param[in]  trng_dev: TRNG device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_trng_recover_seed_error(hal_trng_dev_struct *trng_dev)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t timeout   = 0U;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == trng_dev) {
        HAL_DEBUGE("pointer [trng_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    trng_dev->error_code = TRNG_ERROR_NONE;

    /* Check if seed error current status */
    if(RESET == hals_trng_flag_get(TRNG_FLAG_SECS)) {
        hals_trng_interrupt_flag_clear(TRNG_INT_FLAG_SEIF);
    } else {
        /* Writing bit CONDRST=1*/
        TRNG_CTL |= TRNG_CTL_CONDRST;
        /* Writing bit CONDRST=0*/
        TRNG_CTL &= ~TRNG_CTL_CONDRST;
        /* Wait for conditioning reset process to be completed */
        do {
            timeout++;
            if(timeout >= TRNG_CTL_CONDRST_TIMEOUT) {
                if(NULL != trng_dev->trng_irq.seed_error_handle) {
                    trng_dev->trng_irq.seed_error_handle(trng_dev);
                }
                trng_dev->error_code = TRNG_ERROR_TIMEOUT;
                HAL_DEBUGE("trng recover seed error time out!");
                error_code = HAL_ERR_TIMEOUT;
                break;
            }
        } while(TRNG_CTL & (TRNG_CTL_CONDRST));

        if(HAL_ERR_NONE == error_code) {
            /* Clear bit SEIS */
            if(RESET != hals_trng_interrupt_flag_get(TRNG_INT_FLAG_SEIF)) {
                hals_trng_interrupt_flag_clear(TRNG_INT_FLAG_SEIF);
            } else {
                /* do nothing */
            }

            /* Wait for SECS to be cleared */
            timeout = 0U;
            do {
                timeout++;
                if(timeout >= TRNG_CTL_CONDRST_TIMEOUT) {
                    if(NULL != trng_dev->trng_irq.seed_error_handle) {
                        trng_dev->trng_irq.seed_error_handle(trng_dev);
                    } else {
                        /* do nothing */
                    }
                    trng_dev->error_code = TRNG_ERROR_TIMEOUT;
                    HAL_DEBUGE("trng recover seed error time out!");
                    error_code = HAL_ERR_TIMEOUT;
                    break;
                } else {
                    /* do nothing */
                }
            } while(RESET != hals_trng_flag_get(TRNG_FLAG_SECS));
        }
    }

    return error_code;
}

/*!
    \brief      stop random number generation in interrupt mode
    \param[in]  trng_dev: TRNG device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_trng_stop_interrupt(hal_trng_dev_struct *trng_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == trng_dev) {
        HAL_DEBUGE("pointer [trng_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(trng_dev);

    TRNG_CTL &= ~TRNG_CTL_IE;

    trng_dev->state = HAL_TRNG_STATE_READY;

    HAL_UNLOCK(trng_dev);

    return error_code;
}

/*!
    \brief      get the true random data
    \param[in]  trng_dev: TRNG device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] random_data: get the generated random data
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_trng_random_data_get(hal_trng_dev_struct *trng_dev, uint32_t *random_data)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t timeout   = 0U;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == trng_dev) || (NULL == random_data)) {
        HAL_DEBUGE("pointer [trng_dev] or [random_data] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(trng_dev);

    trng_dev->error_code = TRNG_ERROR_NONE;

    if(HAL_TRNG_STATE_READY != trng_dev->state) {
        trng_dev->error_code = TRNG_ERROR_BUSY;
        error_code = HAL_ERR_BUSY;
    } else {
        trng_dev->state = HAL_TRNG_STATE_BUSY;
        /* wait DRDY=1 */
        while((SET != hals_trng_flag_get(TRNG_FLAG_DRDY)) && (timeout <= TRNG_STAT_DRDY_TIMEOUT)) {
            timeout++;
        }

        if(timeout > TRNG_STAT_DRDY_TIMEOUT) {
            trng_dev->error_code = TRNG_ERROR_TIMEOUT;
            HAL_DEBUGE("trng get random data time out!");
            error_code = HAL_ERR_TIMEOUT;
        } else {
            *random_data = (uint32_t)TRNG_DATA;
            trng_dev->random_data = random_data;
        }
    }

    trng_dev->state = HAL_TRNG_STATE_READY;

    HAL_UNLOCK(trng_dev);

    return error_code;
}

/*!
    \brief      get the true random data in interrupt mode
    \param[in]  trng_dev: TRNG device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  random_data: get the generated random data
    \param[in]  p_user_func: user callback function structure
                  The structure contains the following members:
      \arg        ready_data_func: callback function called when random data is ready
                  if not NULL, the callback will be executed after random data is generated
                  if NULL, no callback will be executed
      \arg        error_func: callback function called when error occurs
                  if not NULL, the callback will be executed when an error occurs during data generation
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_trng_random_data_get_interrupt(hal_trng_dev_struct *trng_dev, uint32_t *random_data, \
                                           hal_trng_user_callback_struct *p_user_func)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == trng_dev) || (NULL == random_data)) {
        HAL_DEBUGE("pointer [trng_dev] or [random_data] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(trng_dev);
    trng_dev->error_code = TRNG_ERROR_NONE;

    if(HAL_TRNG_STATE_READY != trng_dev->state) {
        trng_dev->error_code = TRNG_ERROR_BUSY;
        error_code = HAL_ERR_BUSY;
    } else {
        if(NULL != p_user_func) {
            trng_dev->ready_callback = (void *)p_user_func->ready_data_func;
            trng_dev->error_callback = (void *)p_user_func->error_func;
        } else {
            trng_dev->ready_callback = NULL;
            trng_dev->error_callback = NULL;
        }

        trng_dev->state = HAL_TRNG_STATE_BUSY;
        trng_dev->trng_irq.random_data_ready = _trng_random_data_ready_callback;
        trng_dev->trng_irq.nist_mode_error = __trng_error_callback;
        trng_dev->trng_irq.clock_error_handle = __trng_error_callback;
        trng_dev->trng_irq.seed_error_handle = __trng_error_callback;
        trng_dev->random_data = random_data;

        hals_trng_interrupt_enable();
    }

    HAL_UNLOCK(trng_dev);

    return error_code;
}

/*!
    \brief      get the last true random data
    \param[in]  trng_dev: TRNG device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t: last random data
*/
uint32_t hal_trng_last_random_data_get(hal_trng_dev_struct *trng_dev)
{
    return *(uint32_t *)trng_dev->random_data;
}

/*!
    \brief      configure health tests default value
    \param[in]  adpo_threshold: adaptive proportion test threshold value
    \param[in]  rep_threshold: repetitive (00/11) test threshold value
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_trng_health_tests_config(uint16_t adpo_threshold, uint8_t rep_threshold)
{
    int32_t error_code;
    uint32_t tmp;
    uint32_t aptth_value;
    uint32_t rctth_value;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((!IS_TRNG_HTCFG_APT_TH(adpo_threshold)) || (!IS_TRNG_HTCFG_RCT_HT(rep_threshold))) {
        HAL_DEBUGE("parameter [adpo_threshold] or [rep_threshold] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    error_code = HAL_ERR_NONE;
    tmp = TRNG_HTCFG;

    tmp &= ~(TRNG_HTCFG_APTTH | TRNG_HTCFG_RCTTH);
    aptth_value = (uint32_t)adpo_threshold << 16U;
    rctth_value = (uint32_t)rep_threshold;
    tmp |= ((aptth_value & TRNG_HTCFG_APTTH) | (rctth_value & TRNG_HTCFG_RCTTH));
    TRNG_HTCFG = tmp;

    return error_code;
}

/*!
    \brief      enable the TRNG interface
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_trng_enable(void)
{
    uint32_t trng_config = TRNG_CTL;
    trng_config &= ~TRNG_CTL_CONDRST;
    trng_config |= TRNG_CTL_TRNGEN;
    TRNG_CTL = trng_config;
}

/*!
    \brief      disable the TRNG interface
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_trng_disable(void)
{
    TRNG_CTL &= ~TRNG_CTL_TRNGEN;
}

/*!
    \brief      configure TRNG parameter
    \param[in]  config: TRNG config struct
                  mode: operation mode
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_MODSEL_LFSR: TRNG working in LFSR mode
      \arg          TRNG_MODSEL_NIST: TRNG working in NIST mode
                  clkerr_detection: clock error detection
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_CLKERR_DISABLE: TRNG clock error detection disable
      \arg          TRNG_CLKERR_ENABLE: TRNG clock error detection enable
                  clkdiv: clock division factor
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_CLK_DIV1: TRNG clock TRNG_CLK divider 1
      \arg          TRNG_CLK_DIV2: TRNG clock TRNG_CLK divider 2
      \arg          TRNG_CLK_DIV4: TRNG clock TRNG_CLK divider 4
      \arg          TRNG_CLK_DIV8: TRNG clock TRNG_CLK divider 8
      \arg          TRNG_CLK_DIV16: TRNG clock TRNG_CLK divider 16
      \arg          TRNG_CLK_DIV32: TRNG clock TRNG_CLK divider 32
      \arg          TRNG_CLK_DIV64: TRNG clock TRNG_CLK divider 64
      \arg          TRNG_CLK_DIV128: TRNG clock TRNG_CLK divider 128
      \arg          TRNG_CLK_DIV256: TRNG clock TRNG_CLK divider 256
      \arg          TRNG_CLK_DIV512: TRNG clock TRNG_CLK divider 512
      \arg          TRNG_CLK_DIV1024: TRNG clock TRNG_CLK divider 1024
      \arg          TRNG_CLK_DIV2048: TRNG clock TRNG_CLK divider 2048
      \arg          TRNG_CLK_DIV4096: TRNG clock TRNG_CLK divider 4096
      \arg          TRNG_CLK_DIV8192: TRNG clock TRNG_CLK divider 8192
      \arg          TRNG_CLK_DIV16384: TRNG clock TRNG_CLK divider 16384
      \arg          TRNG_CLK_DIV32768: TRNG clock TRNG_CLK divider 32768
                  power_mode: power mode
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_NR_ULTRALOW: TRNG analog power mode ultralow
      \arg          TRNG_NR_LOW: TRNG analog power mode low
      \arg          TRNG_NR_MEDIUM: TRNG analog power mode medium
      \arg          TRNG_NR_HIGH: TRNG analog power mode high
                  cond: training unit module
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_COND_DISABLE: TRNG conditioning module disable
      \arg          TRNG_COND_ENABLE: TRNG conditioning module enable
                  hash_init: whether to initialize the hash or not
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_HASH_NOINIT: TRNG hash algorithm no init
      \arg          TRNG_HASH_INIT: TRNG hash algorithm init
                  algorithm: algorithm configuration
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_ALGO_SHA1: TRNG conditioning module hash SHA1
      \arg          TRNG_ALGO_MD5: TRNG conditioning module hash MD5
      \arg          TRNG_ALGO_SHA224: TRNG conditioning module hash SHA224
      \arg          TRNG_ALGO_SHA256: TRNG conditioning module hash SHA256
                  input: input mode
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_INMOD_256BIT: conditioning module input bitwidth 256bits
      \arg          TRNG_INMOD_440BIT: conditioning module input bitwidth 440bits
                  output: output mode
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_OUTMOD_128BIT: conditioning module output bitwidth 128bits
      \arg          TRNG_OUTMOD_256BIT: conditioning module output bitwidth 256bits
                  replace_test: replace test
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_REPTEST_DISABLE: TRNG replace test disable
      \arg          TRNG_REPTEST_ENABLE: TRNG replace test enable
                  postprocess: post-processing unit config
                  only one parameter can be selected which is shown as below:
      \arg          TRNG_PPROCESS_DISABLE: TRNG post-processing disable
      \arg          TRNG_PPROCESS_ENABLE: TRNG post-processing enable
                  adpo_threshold: adaptive proportion test thresholds value
                  rep_threshold: repetitive (00/11) test thresholds value
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_trng_parameter_config(hal_trng_config_struct *config)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == config) {
        HAL_DEBUGE("pointer [config] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    TRNG_CTL |= TRNG_CTL_CONDRST;

    /* Clear the associated register bit */
    TRNG_CTL &= (~(uint32_t)(TRNG_CTL_MODSEL | TRNG_CTL_CED | TRNG_CTL_CLKDIV | TRNG_CTL_NR | TRNG_CTL_CONDEN | \
                             TRNG_CTL_INIT | TRNG_CTL_ALGO | TRNG_CTL_INMOD | TRNG_CTL_OUTMOD | TRNG_CTL_RTEN | \
                             TRNG_CTL_PPEN));

    /* operation mode */
    TRNG_CTL |= config->mode;
    /* Random number seed input mode selection */
    TRNG_CTL |= ((uint32_t)config->input | (uint32_t)config->output);
    /* post-processing unit config */
    TRNG_CTL |= config->postprocess;
    /* algorithm configuration */
    TRNG_CTL |= config->algorithm;
    /* clock error detection, clock division factor */
    TRNG_CTL |= (config->clkerr_detection | config->clkdiv);
    /* power mode */
    TRNG_CTL |= config->power_mode;
    /* training unit module */
    TRNG_CTL |= config->cond;
    /* hash init */
    TRNG_CTL |= config->hash_init;
    /* Random number output mode selection */
    TRNG_CTL |= config->replace_test;

    /* reset aptth and rctth value */
    TRNG_HTCFG &= ~(TRNG_HTCFG_APTTH | TRNG_HTCFG_RCTTH);

    /* adaptive proportion test thresholds value, repetitive test thresholds value*/
    TRNG_HTCFG |= (HTCFG_APTTH(config->adpo_threshold) | HTCFG_RCTTH(config->rep_threshold));

    return error_code;
}

/*!
    \brief      lock the TRNG control bits
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_trng_lock(void)
{
    TRNG_CTL |= TRNG_CTL_LK;
}

/*!
    \brief      unlock the TRNG control bits
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_trng_unlock(void)
{
    TRNG_CTL &= ~TRNG_CTL_LK;
}

/*!
    \brief      get the trng error code
    \param[in]  trng_dev: TRNG device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t: trng error code
*/
uint32_t hal_trng_get_error(hal_trng_dev_struct *trng_dev)
{
    return trng_dev->error_code;
}

/*!
    \brief      return the trng state
    \param[in]  trng_dev: TRNG device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_trng_state_enum
*/
hal_trng_state_enum hal_trng_state_get(hal_trng_dev_struct *trng_dev)
{
    return (hal_trng_state_enum)trng_dev->state;
}

/*!
    \brief      configure the conditioning module hash algorithm
    \param[in]  module_algo: module hash algorithm
                only one parameter can be selected which is shown as below:
      \arg        TRNG_ALGO_SHA1:   TRNG conditioning module hash SHA1
      \arg        TRNG_ALGO_MD5:    TRNG conditioning module hash MD5
      \arg        TRNG_ALGO_SHA224: TRNG conditioning module hash SHA224
      \arg        TRNG_ALGO_SHA256: TRNG conditioning module hash SHA256
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hals_trng_conditioning_algo_config(uint32_t module_algo)
{
    int32_t error_code = HAL_ERR_NONE;
      uint32_t tmp              = 0U;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(!IS_TRNG_CTL_ALGO(module_algo)) {
        HAL_DEBUGE("parameter [module_algo] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    tmp = TRNG_CTL;
    tmp &= ~(TRNG_CTL_ALGO);
    tmp |= module_algo;
    TRNG_CTL = tmp;

    return error_code;
}

/*!
    \brief      get the TRNG status flags
    \param[in]  flag: TRNG status flag, refer to trng_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        TRNG_FLAG_DRDY: random data ready status
      \arg        TRNG_FLAG_CECS: clock error current status
      \arg        TRNG_FLAG_SECS: seed error current status
      \arg        TRNG_FALG_ERRSTA: NIST mode error status
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_trng_flag_get(hal_trng_flag_enum flag)
{
    FlagStatus state = RESET;

    if(RESET != (TRNG_STAT & flag)) {
        state = SET;
    } else {
        /* do nothing */
    }

    return state;
}

/*!
    \brief      enable the TRNG interrupt
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_trng_interrupt_enable(void)
{
    TRNG_CTL |= TRNG_CTL_IE;
}

/*!
    \brief      disable the TRNG interrupt
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_trng_interrupt_disable(void)
{
    TRNG_CTL &= ~TRNG_CTL_IE;
}

/*!
    \brief      get the TRNG interrupt flags
    \param[in]  int_flag: TRNG interrupt flag, refer to trng_int_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        TRNG_INT_FLAG_CEIF: clock error interrupt flag
      \arg        TRNG_INT_FLAG_SEIF: seed error interrupt flag
      \arg        TRNG_INT_FLAG_DRDY: random Data ready interrupt flag
      \arg        TRNG_INT_FLAG_ERRSTA: NIST mode error interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_trng_interrupt_flag_get(hal_trng_int_flag_enum int_flag)
{
    FlagStatus state = RESET;

    if(RESET != (TRNG_STAT & int_flag)) {
        state = SET;
    } else {
        /* do nothing */
    }

    return state;
}

/*!
    \brief      clear the TRNG interrupt flags
    \param[in]  int_flag: TRNG interrupt flag, refer to trng_int_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        TRNG_INT_FLAG_CEIF: clock error interrupt flag
      \arg        TRNG_INT_FLAG_SEIF: seed error interrupt flag
      \arg        TRNG_INT_FLAG_DRDY: random Data ready interrupt flag
      \arg        TRNG_INT_FLAG_ERRSTA: NIST mode error interrupt flag
    \param[out] none
    \retval     none
*/
void hals_trng_interrupt_flag_clear(hal_trng_int_flag_enum int_flag)
{
    TRNG_STAT &= ~(uint32_t)int_flag;
}

/*!
    \brief      trng random data ready interrupt handler
    \param[in]  trng: TRNG device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _trng_random_data_ready_callback(void *trng)
{
    hal_trng_dev_struct *trng_dev = (hal_trng_dev_struct *)trng;
    hal_trng_user_cb p_func = (hal_trng_user_cb)trng_dev->ready_callback;

    *(trng_dev->random_data) = (uint32_t)TRNG_DATA;
    hals_trng_interrupt_disable();
    if(NULL != p_func) {
        p_func(trng_dev);
    } else {
        trng_dev->error_code = TRNG_ERROR_INVALID_CALLBACK;
    }

    trng_dev->state = HAL_TRNG_STATE_READY;
}

/*!
    \brief      trng error interrupt handler
    \param[in]  trng: TRNG device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void __trng_error_callback(void *trng)
{
    hal_trng_dev_struct *trng_dev = (hal_trng_dev_struct *)trng;
    hal_trng_user_cb p_func = (hal_trng_user_cb)trng_dev->error_callback;

    hals_trng_interrupt_disable();
    if(NULL != p_func) {
        p_func(trng_dev);
    } else {
        trng_dev->error_code = TRNG_ERROR_INVALID_CALLBACK;
    }

    trng_dev->state = HAL_TRNG_STATE_READY;
}
