/*!
    \file    gd32h7xx_hal_trigsel.c
    \brief   TRIGSEL driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* TRIGSEL target register redefine */
#define TRIGSEL_TARGET_REG(PERIPH) \
        (REG32(TRIGSEL + ((uint32_t)(PERIPH) & BITS(2, 31))))            /*!< target peripheral register */

#define TRIGSEL_TARGET_PERIPH_SHIFT(PERIPH) \
        (((uint32_t)(PERIPH) & BITS(0, 1)) << 3U)                        /*!< bit offset in target peripheral register */

#define TRIGSEL_TARGET_PERIPH_MASK(PERIPH) \
        ((uint32_t)(BITS(0, 7) << TRIGSEL_TARGET_PERIPH_SHIFT(PERIPH)))  /*!< bit mask in target peripheral register */

/* register reset configuration struct */
typedef struct {
    volatile uint32_t *reg;  /*!< register address */
    uint32_t value;          /*!< register reset value */
} hal_trigsel_reset_config_struct;

static hal_trigsel_reset_config_struct trigsel_reset_config[] = {
    /* EXTOUT group (all 0x00000000U) */
    { (volatile uint32_t *)&TRIGSEL_EXTOUT0, 0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_EXTOUT1, 0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_EXTOUT2, 0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_EXTOUT3, 0x00000000U },
    /* ADC group (all 0x00001113U) */
    { (volatile uint32_t *)&TRIGSEL_ADC0,    0x00001113U },
    { (volatile uint32_t *)&TRIGSEL_ADC1,    0x00001113U },
    { (volatile uint32_t *)&TRIGSEL_ADC2,    0x00001113U },
    /* DAC group (all 0x00000025U) */
    { (volatile uint32_t *)&TRIGSEL_DAC0OUT0, 0x00000025U },
    { (volatile uint32_t *)&TRIGSEL_DAC0OUT1, 0x00000025U },
    /* TIMER BRKIN group (unique values) */
    { (volatile uint32_t *)&TRIGSEL_TIMER0BRKIN,  0x00232221U },
    { (volatile uint32_t *)&TRIGSEL_TIMER7BRKIN,  0x0051504FU },
    { (volatile uint32_t *)&TRIGSEL_TIMER14BRKIN, 0x00000059U },
    { (volatile uint32_t *)&TRIGSEL_TIMER15BRKIN, 0x0000005EU },
    { (volatile uint32_t *)&TRIGSEL_TIMER16BRKIN, 0x00000063U },
    { (volatile uint32_t *)&TRIGSEL_TIMER40BRKIN, 0x00000082U },
    { (volatile uint32_t *)&TRIGSEL_TIMER41BRKIN, 0x00000089U },
    { (volatile uint32_t *)&TRIGSEL_TIMER42BRKIN, 0x00000090U },
    { (volatile uint32_t *)&TRIGSEL_TIMER43BRKIN, 0x00000097U },
    { (volatile uint32_t *)&TRIGSEL_TIMER44BRKIN, 0x0000009EU },
    /* CAN group (all 0x0000003DU) */
    { (volatile uint32_t *)&TRIGSEL_CAN0, 0x0000003DU },
    { (volatile uint32_t *)&TRIGSEL_CAN1, 0x0000003DU },
    { (volatile uint32_t *)&TRIGSEL_CAN2, 0x0000003DU },
    /* LPDTS (0x00000000U) */
    { (volatile uint32_t *)&TRIGSEL_LPDTS, 0x00000000U },
    /* TIMER ETI group (unique values) */
    { (volatile uint32_t *)&TRIGSEL_TIMER0ETI,  0x00000024U },
    { (volatile uint32_t *)&TRIGSEL_TIMER1ETI,  0x0000002AU },
    { (volatile uint32_t *)&TRIGSEL_TIMER2ETI,  0x00000030U },
    { (volatile uint32_t *)&TRIGSEL_TIMER3ETI,  0x00000036U },
    { (volatile uint32_t *)&TRIGSEL_TIMER4ETI,  0x0000003CU },
    { (volatile uint32_t *)&TRIGSEL_TIMER7ETI,  0x00000052U },
    { (volatile uint32_t *)&TRIGSEL_TIMER22ETI, 0x00000069U },
    { (volatile uint32_t *)&TRIGSEL_TIMER23ETI, 0x0000006FU },
    { (volatile uint32_t *)&TRIGSEL_TIMER30ETI, 0x00000075U },
    { (volatile uint32_t *)&TRIGSEL_TIMER31ETI, 0x0000007BU },
    /* EDOUT and HPDF (both 0x00000000U) */
    { (volatile uint32_t *)&TRIGSEL_EDOUT, 0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_HPDF,  0x00000000U },
    /* TIMER ITI14 group (all 0x00000000U) */
    { (volatile uint32_t *)&TRIGSEL_TIMER0ITI14,  0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_TIMER1ITI14,  0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_TIMER2ITI14,  0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_TIMER3ITI14,  0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_TIMER4ITI14,  0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_TIMER7ITI14,  0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_TIMER14ITI14, 0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_TIMER22ITI14, 0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_TIMER23ITI14, 0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_TIMER30ITI14, 0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_TIMER31ITI14, 0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_TIMER40ITI14, 0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_TIMER41ITI14, 0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_TIMER42ITI14, 0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_TIMER43ITI14, 0x00000000U },
    { (volatile uint32_t *)&TRIGSEL_TIMER44ITI14, 0x00000000U }
};

/*!
    \brief      deinitialize TRIGSEL
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_trigsel_deinit(void)
{
    uint32_t i = 0U;

    /* Apply resets via loop for compactness and maintainability */
    for (i = 0U; i < (sizeof(trigsel_reset_config) / sizeof(trigsel_reset_config[0])); i++) {
        *trigsel_reset_config[i].reg = trigsel_reset_config[i].value;
    }
}

/*!
    \brief      initialize trigsel
    \param[in]  p_init: pointer of the initialization data needed to initialize trigsel
                  target_periph: target peripheral value, refer to hal_trigsel_target_periph_enum
                  input_source: input source value, refer to hal_trigsel_input_source_enum
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_LOCK details refer to gd32h7xx_hal.h
*/
int32_t hal_trigsel_init(hal_trigsel_init_struct *p_init)
{
    /* initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

/* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == p_init)) {
        HAL_DEBUGE("pointer [p_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* if register write is enabled, set trigger source to target peripheral */
    if(RESET == hals_trigsel_register_lock_get(p_init->target_periph)) {
        /* if register write is enabled, set trigger source to target peripheral */
        TRIGSEL_TARGET_REG(p_init->target_periph) &= ~TRIGSEL_TARGET_PERIPH_MASK(p_init->target_periph);
        TRIGSEL_TARGET_REG(p_init->target_periph) |= ((uint32_t)(p_init->input_source) << \
                                                      TRIGSEL_TARGET_PERIPH_SHIFT(p_init->target_periph)) & \
                                                      TRIGSEL_TARGET_PERIPH_MASK(p_init->target_periph);
    } else {
        HAL_DEBUGE("target peripheral register is locked");
        retval = HAL_ERR_LOCK;
    }

    return retval;
}

/*!
    \brief      initialize the trigsel structure with the default values
    \param[in]  struct_type: type of TRIGSEL structure for initialization
                only one parameters can be selected which are shown as below:
      \arg      HAL_TRIGSEL_INIT_STRUCT: initialization structure
    \param[out] p_struct: pointer to TRIGSEL structure that contains the configuration information
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_trigsel_struct_init(hal_trigsel_struct_type_enum struct_type, void *p_struct)
{
    /* initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

/* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(struct_type) {
        /* initialize trigsel initialization structure with the default values */
    case HAL_TRIGSEL_INIT_STRUCT:
        ((hal_trigsel_init_struct *)p_struct)->group         = 0U;
        ((hal_trigsel_init_struct *)p_struct)->input_source  = TRIGSEL_INPUT_0;
        ((hal_trigsel_init_struct *)p_struct)->target_periph = TRIGSEL_OUTPUT_TRIGSEL_OUT0;
        break;
    default:
        HAL_DEBUGE("parameter [struct_type] value is undefine");
        retval = HAL_ERR_VAL;
        break;
    }

    return retval;
}

/*!
    \brief      lock the trigger register
    \param[in]  target_periph: target peripheral value
                only one parameter can be selected which is shown as below:
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT0: output target peripheral TRIGSEL_OUT0 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT1: output target peripheral TRIGSEL_OUT1 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT2: output target peripheral TRIGSEL_OUT2 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT3: output target peripheral TRIGSEL_OUT3 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT4: output target peripheral TRIGSEL_OUT4 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT5: output target peripheral TRIGSEL_OUT5 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT6: output target peripheral TRIGSEL_OUT6 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT7: output target peripheral TRIGSEL_OUT7 pin
      \arg        TRIGSEL_OUTPUT_ADC0_ROUTRG: output target peripheral ADC0_ROUTRG
      \arg        TRIGSEL_OUTPUT_ADC0_INSTRG: output target peripheral ADC0_INSTRG
      \arg        TRIGSEL_OUTPUT_ADC1_ROUTRG: output target peripheral ADC1_ROUTRG
      \arg        TRIGSEL_OUTPUT_ADC1_INSTRG: output target peripheral ADC1_INSTRG
      \arg        TRIGSEL_OUTPUT_ADC2_ROUTRG: output target peripheral ADC2_ROUTRG
      \arg        TRIGSEL_OUTPUT_ADC2_INSTRG: output target peripheral ADC2_INSTRG
      \arg        TRIGSEL_OUTPUT_DAC0_OUT0_EXTRG: output target peripheral DAC0_OUT0_EXTRG
      \arg        TRIGSEL_OUTPUT_DAC0_OUT1_EXTRG: output target peripheral DAC0_OUT1_EXTRG
      \arg        TRIGSEL_OUTPUT_TIMER0_BRKIN0: output target peripheral TIMER0_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER0_BRKIN1: output target peripheral TIMER0_BRKIN1
      \arg        TRIGSEL_OUTPUT_TIMER0_BRKIN2: output target peripheral TIMER0_BRKIN2
      \arg        TRIGSEL_OUTPUT_TIMER7_BRKIN0: output target peripheral TIMER7_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER7_BRKIN1: output target peripheral TIMER7_BRKIN1
      \arg        TRIGSEL_OUTPUT_TIMER7_BRKIN2: output target peripheral TIMER7_BRKIN2
      \arg        TRIGSEL_OUTPUT_TIMER14_BRKIN0: output target peripheral TIMER14_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER15_BRKIN0: output target peripheral TIMER15_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER16_BRKIN0: output target peripheral TIMER16_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER40_BRKIN0: output target peripheral TIMER40_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER41_BRKIN0: output target peripheral TIMER41_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER42_BRKIN0: output target peripheral TIMER42_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER43_BRKIN0: output target peripheral TIMER43_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER44_BRKIN0: output target peripheral TIMER44_BRKIN0
      \arg        TRIGSEL_OUTPUT_CAN0_EX_TIME_TICK: output target peripheral CAN0_EX_TIME_TICK
      \arg        TRIGSEL_OUTPUT_CAN1_EX_TIME_TICK: output target peripheral CAN1_EX_TIME_TICK
      \arg        TRIGSEL_OUTPUT_CAN2_EX_TIME_TICK: output target peripheral CAN2_EX_TIME_TICK
      \arg        TRIGSEL_OUTPUT_LPDTS_TRG: output target peripheral LPDTS_TRG
      \arg        TRIGSEL_OUTPUT_TIMER0_ETI: output target peripheral TIMER0_ETI
      \arg        TRIGSEL_OUTPUT_TIMER1_ETI: output target peripheral TIMER1_ETI
      \arg        TRIGSEL_OUTPUT_TIMER2_ETI: output target peripheral TIMER2_ETI
      \arg        TRIGSEL_OUTPUT_TIMER3_ETI: output target peripheral TIMER3_ETI
      \arg        TRIGSEL_OUTPUT_TIMER4_ETI: output target peripheral TIMER4_ETI
      \arg        TRIGSEL_OUTPUT_TIMER7_ETI: output target peripheral TIMER7_ETI
      \arg        TRIGSEL_OUTPUT_TIMER22_ETI: output target peripheral TIMER22_ETI
      \arg        TRIGSEL_OUTPUT_TIMER23_ETI: output target peripheral TIMER23_ETI
      \arg        TRIGSEL_OUTPUT_TIMER30_ETI: output target peripheral TIMER30_ETI
      \arg        TRIGSEL_OUTPUT_TIMER31_ETI: output target peripheral TIMER31_ETI
      \arg        TRIGSEL_OUTPUT_EDOUT_TRG: output target peripheral EDOUT_TRG
      \arg        TRIGSEL_OUTPUT_HPDF_ITR: output target peripheral HPDF_ITR
      \arg        TRIGSEL_OUTPUT_TIMER0_ITI14: output target peripheral TIMER0_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER1_ITI14: output target peripheral TIMER1_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER2_ITI14: output target peripheral TIMER2_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER3_ITI14: output target peripheral TIMER3_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER4_ITI14: output target peripheral TIMER4_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER7_ITI14: output target peripheral TIMER7_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER14_ITI14: output target peripheral TIMER14_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER22_ITI14: output target peripheral TIMER22_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER23_ITI14: output target peripheral TIMER23_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER30_ITI14: output target peripheral TIMER30_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER31_ITI14: output target peripheral TIMER31_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER40_ITI14: output target peripheral TIMER40_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER41_ITI14: output target peripheral TIMER41_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER42_ITI14: output target peripheral TIMER42_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER43_ITI14: output target peripheral TIMER43_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER44_ITI14: output target peripheral TIMER44_ITI14
    \param[out] none
    \retval     none
*/
void hal_trigsel_register_lock_set(hal_trigsel_periph_enum target_periph)
{
    /*!< lock target peripheral register */
    TRIGSEL_TARGET_REG(target_periph) |= TRIGSEL_TARGET_LK;
}

/*!
    \brief      get the trigger input signal for target peripheral
    \param[in]  target_periph: target peripheral value
                only one parameter can be selected which is shown as below:
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT0: output target peripheral TRIGSEL_OUT0 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT1: output target peripheral TRIGSEL_OUT1 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT2: output target peripheral TRIGSEL_OUT2 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT3: output target peripheral TRIGSEL_OUT3 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT4: output target peripheral TRIGSEL_OUT4 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT5: output target peripheral TRIGSEL_OUT5 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT6: output target peripheral TRIGSEL_OUT6 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT7: output target peripheral TRIGSEL_OUT7 pin
      \arg        TRIGSEL_OUTPUT_ADC0_ROUTRG: output target peripheral ADC0_ROUTRG
      \arg        TRIGSEL_OUTPUT_ADC0_INSTRG: output target peripheral ADC0_INSTRG
      \arg        TRIGSEL_OUTPUT_ADC1_ROUTRG: output target peripheral ADC1_ROUTRG
      \arg        TRIGSEL_OUTPUT_ADC1_INSTRG: output target peripheral ADC1_INSTRG
      \arg        TRIGSEL_OUTPUT_ADC2_ROUTRG: output target peripheral ADC2_ROUTRG
      \arg        TRIGSEL_OUTPUT_ADC2_INSTRG: output target peripheral ADC2_INSTRG
      \arg        TRIGSEL_OUTPUT_DAC0_OUT0_EXTRG: output target peripheral DAC0_OUT0_EXTRG
      \arg        TRIGSEL_OUTPUT_DAC0_OUT1_EXTRG: output target peripheral DAC0_OUT1_EXTRG
      \arg        TRIGSEL_OUTPUT_TIMER0_BRKIN0: output target peripheral TIMER0_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER0_BRKIN1: output target peripheral TIMER0_BRKIN1
      \arg        TRIGSEL_OUTPUT_TIMER0_BRKIN2: output target peripheral TIMER0_BRKIN2
      \arg        TRIGSEL_OUTPUT_TIMER7_BRKIN0: output target peripheral TIMER7_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER7_BRKIN1: output target peripheral TIMER7_BRKIN1
      \arg        TRIGSEL_OUTPUT_TIMER7_BRKIN2: output target peripheral TIMER7_BRKIN2
      \arg        TRIGSEL_OUTPUT_TIMER14_BRKIN0: output target peripheral TIMER14_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER15_BRKIN0: output target peripheral TIMER15_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER16_BRKIN0: output target peripheral TIMER16_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER40_BRKIN0: output target peripheral TIMER40_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER41_BRKIN0: output target peripheral TIMER41_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER42_BRKIN0: output target peripheral TIMER42_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER43_BRKIN0: output target peripheral TIMER43_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER44_BRKIN0: output target peripheral TIMER44_BRKIN0
      \arg        TRIGSEL_OUTPUT_CAN0_EX_TIME_TICK: output target peripheral CAN0_EX_TIME_TICK
      \arg        TRIGSEL_OUTPUT_CAN1_EX_TIME_TICK: output target peripheral CAN1_EX_TIME_TICK
      \arg        TRIGSEL_OUTPUT_CAN2_EX_TIME_TICK: output target peripheral CAN2_EX_TIME_TICK
      \arg        TRIGSEL_OUTPUT_LPDTS_TRG: output target peripheral LPDTS_TRG
      \arg        TRIGSEL_OUTPUT_TIMER0_ETI: output target peripheral TIMER0_ETI
      \arg        TRIGSEL_OUTPUT_TIMER1_ETI: output target peripheral TIMER1_ETI
      \arg        TRIGSEL_OUTPUT_TIMER2_ETI: output target peripheral TIMER2_ETI
      \arg        TRIGSEL_OUTPUT_TIMER3_ETI: output target peripheral TIMER3_ETI
      \arg        TRIGSEL_OUTPUT_TIMER4_ETI: output target peripheral TIMER4_ETI
      \arg        TRIGSEL_OUTPUT_TIMER7_ETI: output target peripheral TIMER7_ETI
      \arg        TRIGSEL_OUTPUT_TIMER22_ETI: output target peripheral TIMER22_ETI
      \arg        TRIGSEL_OUTPUT_TIMER23_ETI: output target peripheral TIMER23_ETI
      \arg        TRIGSEL_OUTPUT_TIMER30_ETI: output target peripheral TIMER30_ETI
      \arg        TRIGSEL_OUTPUT_TIMER31_ETI: output target peripheral TIMER31_ETI
      \arg        TRIGSEL_OUTPUT_EDOUT_TRG: output target peripheral EDOUT_TRG
      \arg        TRIGSEL_OUTPUT_HPDF_ITR: output target peripheral HPDF_ITR
      \arg        TRIGSEL_OUTPUT_TIMER0_ITI14: output target peripheral TIMER0_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER1_ITI14: output target peripheral TIMER1_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER2_ITI14: output target peripheral TIMER2_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER3_ITI14: output target peripheral TIMER3_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER4_ITI14: output target peripheral TIMER4_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER7_ITI14: output target peripheral TIMER7_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER14_ITI14: output target peripheral TIMER14_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER22_ITI14: output target peripheral TIMER22_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER23_ITI14: output target peripheral TIMER23_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER30_ITI14: output target peripheral TIMER30_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER31_ITI14: output target peripheral TIMER31_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER40_ITI14: output target peripheral TIMER40_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER41_ITI14: output target peripheral TIMER41_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER42_ITI14: output target peripheral TIMER42_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER43_ITI14: output target peripheral TIMER43_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER44_ITI14: output target peripheral TIMER44_ITI14
    \param[out] none
    \retval  hal_trigger_source_enum: trigger source value(0~177) details refer to gd32h7xx_hal_trigsel.h
*/
hal_trigger_source_enum hals_trigsel_trigger_source_get(hal_trigsel_periph_enum target_periph)
{
    hal_trigger_source_enum trigsel_source;

    trigsel_source = (hal_trigger_source_enum)((TRIGSEL_TARGET_REG(target_periph) & \
                                                TRIGSEL_TARGET_PERIPH_MASK(target_periph)) >> \
                                                TRIGSEL_TARGET_PERIPH_SHIFT(target_periph));

    return trigsel_source;
}

/*!
    \brief      get the trigger register lock status
    \param[in]  target_periph: target peripheral value
                only one parameter can be selected which is shown as below:
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT0: output target peripheral TRIGSEL_OUT0 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT1: output target peripheral TRIGSEL_OUT1 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT2: output target peripheral TRIGSEL_OUT2 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT3: output target peripheral TRIGSEL_OUT3 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT4: output target peripheral TRIGSEL_OUT4 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT5: output target peripheral TRIGSEL_OUT5 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT6: output target peripheral TRIGSEL_OUT6 pin
      \arg        TRIGSEL_OUTPUT_TRIGSEL_OUT7: output target peripheral TRIGSEL_OUT7 pin
      \arg        TRIGSEL_OUTPUT_ADC0_ROUTRG: output target peripheral ADC0_ROUTRG
      \arg        TRIGSEL_OUTPUT_ADC0_INSTRG: output target peripheral ADC0_INSTRG
      \arg        TRIGSEL_OUTPUT_ADC1_ROUTRG: output target peripheral ADC1_ROUTRG
      \arg        TRIGSEL_OUTPUT_ADC1_INSTRG: output target peripheral ADC1_INSTRG
      \arg        TRIGSEL_OUTPUT_ADC2_ROUTRG: output target peripheral ADC2_ROUTRG
      \arg        TRIGSEL_OUTPUT_ADC2_INSTRG: output target peripheral ADC2_INSTRG
      \arg        TRIGSEL_OUTPUT_DAC0_OUT0_EXTRG: output target peripheral DAC0_OUT0_EXTRG
      \arg        TRIGSEL_OUTPUT_DAC0_OUT1_EXTRG: output target peripheral DAC0_OUT1_EXTRG
      \arg        TRIGSEL_OUTPUT_TIMER0_BRKIN0: output target peripheral TIMER0_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER0_BRKIN1: output target peripheral TIMER0_BRKIN1
      \arg        TRIGSEL_OUTPUT_TIMER0_BRKIN2: output target peripheral TIMER0_BRKIN2
      \arg        TRIGSEL_OUTPUT_TIMER7_BRKIN0: output target peripheral TIMER7_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER7_BRKIN1: output target peripheral TIMER7_BRKIN1
      \arg        TRIGSEL_OUTPUT_TIMER7_BRKIN2: output target peripheral TIMER7_BRKIN2
      \arg        TRIGSEL_OUTPUT_TIMER14_BRKIN0: output target peripheral TIMER14_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER15_BRKIN0: output target peripheral TIMER15_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER16_BRKIN0: output target peripheral TIMER16_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER40_BRKIN0: output target peripheral TIMER40_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER41_BRKIN0: output target peripheral TIMER41_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER42_BRKIN0: output target peripheral TIMER42_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER43_BRKIN0: output target peripheral TIMER43_BRKIN0
      \arg        TRIGSEL_OUTPUT_TIMER44_BRKIN0: output target peripheral TIMER44_BRKIN0
      \arg        TRIGSEL_OUTPUT_CAN0_EX_TIME_TICK: output target peripheral CAN0_EX_TIME_TICK
      \arg        TRIGSEL_OUTPUT_CAN1_EX_TIME_TICK: output target peripheral CAN1_EX_TIME_TICK
      \arg        TRIGSEL_OUTPUT_CAN2_EX_TIME_TICK: output target peripheral CAN2_EX_TIME_TICK
      \arg        TRIGSEL_OUTPUT_LPDTS_TRG: output target peripheral LPDTS_TRG
      \arg        TRIGSEL_OUTPUT_TIMER0_ETI: output target peripheral TIMER0_ETI
      \arg        TRIGSEL_OUTPUT_TIMER1_ETI: output target peripheral TIMER1_ETI
      \arg        TRIGSEL_OUTPUT_TIMER2_ETI: output target peripheral TIMER2_ETI
      \arg        TRIGSEL_OUTPUT_TIMER3_ETI: output target peripheral TIMER3_ETI
      \arg        TRIGSEL_OUTPUT_TIMER4_ETI: output target peripheral TIMER4_ETI
      \arg        TRIGSEL_OUTPUT_TIMER7_ETI: output target peripheral TIMER7_ETI
      \arg        TRIGSEL_OUTPUT_TIMER22_ETI: output target peripheral TIMER22_ETI
      \arg        TRIGSEL_OUTPUT_TIMER23_ETI: output target peripheral TIMER23_ETI
      \arg        TRIGSEL_OUTPUT_TIMER30_ETI: output target peripheral TIMER30_ETI
      \arg        TRIGSEL_OUTPUT_TIMER31_ETI: output target peripheral TIMER31_ETI
      \arg        TRIGSEL_OUTPUT_EDOUT_TRG: output target peripheral EDOUT_TRG
      \arg        TRIGSEL_OUTPUT_HPDF_ITR: output target peripheral HPDF_ITR
      \arg        TRIGSEL_OUTPUT_TIMER0_ITI14: output target peripheral TIMER0_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER1_ITI14: output target peripheral TIMER1_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER2_ITI14: output target peripheral TIMER2_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER3_ITI14: output target peripheral TIMER3_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER4_ITI14: output target peripheral TIMER4_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER7_ITI14: output target peripheral TIMER7_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER14_ITI14: output target peripheral TIMER14_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER22_ITI14: output target peripheral TIMER22_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER23_ITI14: output target peripheral TIMER23_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER30_ITI14: output target peripheral TIMER30_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER31_ITI14: output target peripheral TIMER31_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER40_ITI14: output target peripheral TIMER40_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER41_ITI14: output target peripheral TIMER41_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER42_ITI14: output target peripheral TIMER42_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER43_ITI14: output target peripheral TIMER43_ITI14
      \arg        TRIGSEL_OUTPUT_TIMER44_ITI14: output target peripheral TIMER44_ITI14
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_trigsel_register_lock_get(hal_trigsel_periph_enum target_periph)
{
    /* initialize the function return value */
    FlagStatus retval = RESET;

    /* assess the lock status of the target peripheral */
    if(RESET != (TRIGSEL_TARGET_REG(target_periph) & TRIGSEL_TARGET_LK)) {
        /* Register is read-only */
        retval = SET;
    } else {
        /* Register is read-write */
        retval = RESET;
    }

    return retval;
}
