/*!
    \file    gd32h7xx_hal_spi.c
    \brief   SPI driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define SPI_TIMEOUT_VALUE                    (200UL)                                    /*!< SPI transmit or receive timeout */
#define SPI_FIFO_SIZE                        (16UL)                                     /*!< SPI max fifo length */
#define SPI_I2S_INT_ALL                      ((uint32_t)0x000000FFU)                    /*!< SPI_I2S all interrupt  */
#define SPI_INIT_CTL0_MASK                   ((uint32_t)0x00000000U)                    /*!< SPI_CTL0 parameter initialization mask */
#define SPI_INIT_CFG0_MASK                   ((uint32_t)0x00000000U)                    /*!< SPI_CFG0 parameter initialization mask */
#define SPI_INIT_CFG1_MASK                   ((uint32_t)0x00000000U)                    /*!< SPI_CFG1 parameter initialization mask */
#define SPI_INIT_QCTL_MASK                   ((uint32_t)0x00000000U)                    /*!< SPI_QCTL parameter initialization mask */
#define SPI_INIT_TDATA_MASK                  ((uint32_t)0x00000000U)                    /*!< SPI_TDATA parameter initialization mask */
#define SPI_INIT_RDATA_MASK                  ((uint32_t)0x00000000U)                    /*!< SPI_RDATA parameter initialization mask */
#define SPI_INIT_CRC_MASK                    ((uint32_t)0x00000000U)                    /*!< SPI CRC parameter initialization mask */

#define SWAP16_BE_TO_LE(x)                   _conversion_high2low_16bits(x)             /*!< 16-bits big-endian to little-endian conversion */
#define SWAP32_BE_TO_LE(x)                   _conversion_high2low_32bits(x)             /*!< 32-bits big-endian to little-endian conversion */
#define SWAP_HIGH_LOW_16BITS(x)              _swap_high2low_32bits(x)                    /*!< swap the high and low 16-bits */

/* SPI private function */
/* receive complete interrupt */
static void _spi_receive_complete_interrupt(void *spi);
/* transmit complete interrupt */
static void _spi_transmit_complete_interrupt(void *spi);
/* 2 lines receive interrupt */
static void _spi_2lines_receive_interrupt(void *spi);
/* 2 lines transmit interrupt */
static void _spi_2lines_transmit_interrupt(void *spi);
/* suspend spi interrupt */
static void _spi_suspend_interrupt(void *spi);
/* abort transmit */
static void _spi_abort_transmit(void *spi);
/* abort dma transmit */
static void _spi_abort_dma_transmit(void *spi);
/* abort dma receive */
static void _spi_abort_dma_receive(void *spi);
/* stop transfer */
static int32_t _spi_stop_transfer(hal_spi_dev_struct *spi_dev);
/* DMA transmit complete */
static void _spi_transmit_complete_dma(void *dma);
/* DMA transmit half complete */
static void _spi_transmit_half_complete_dma(void *dma);
/* DMA receive complete */
static void _spi_receive_complete_dma(void *dma);
/* DMA receive half complete */
static void _spi_receive_half_complete_dma(void *dma);
/* DMA error */
static void _spi_dma_error(void *dma);
/* SPI deinitialize */
static void _spi_deinit(uint32_t spi_periph);
/* wait timeout */
static int32_t _spi_wait_flag_timeout(uint32_t spi_periph, hal_spi_flag_enum flag, FlagStatus status, \
                                      uint32_t timeout_ms);
/* 32-bits big-endian to little-endian conversion */
static uint32_t _conversion_high2low_32bits(uint32_t x);
/* 16-bits big-endian to little-endian conversion */
static uint16_t _conversion_high2low_16bits(uint16_t x);
/* swap high to low 16-bits */
static uint32_t _swap_high2low_32bits(uint32_t x);

/*!
    \brief      initialize the SPI structure with the default values
    \param[in]  hal_struct_type: refer to hal_spi_struct_type_enum
    \param[out] p_struct: point to SPI structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_struct_init(hal_spi_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t error_code = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_SPI_INIT_STRUCT:
        /* initialize SPI initialization structure with the default values */
        ((hal_spi_init_struct *)p_struct)->device_mode         = SPI_SLAVE;
        ((hal_spi_init_struct *)p_struct)->trans_mode          = SPI_TRANSMODE_FULLDUPLEX;
        ((hal_spi_init_struct *)p_struct)->ti_mode             = SPI_TIMODE_DISABLE;
        ((hal_spi_init_struct *)p_struct)->data_size           = SPI_DATASIZE_8BIT;
        ((hal_spi_init_struct *)p_struct)->endian              = SPI_ENDIAN_MSB;
        ((hal_spi_init_struct *)p_struct)->prescaler           = SPI_PSC_2;
        ((hal_spi_init_struct *)p_struct)->ckol                = SPI_CKOL_LOW;
        ((hal_spi_init_struct *)p_struct)->ckoh                = SPI_CKOH_FIRST;
        ((hal_spi_init_struct *)p_struct)->nss                 = SPI_NSS_HARD;
        ((hal_spi_init_struct *)p_struct)->nss_level           = SPI_NSS_LOW;
        ((hal_spi_init_struct *)p_struct)->nss_output          = SPI_NSS_HOLD_UNTIL_TRANS_END;
        ((hal_spi_init_struct *)p_struct)->nss_signal_type     = SPI_NSS_OUTPUT;
        ((hal_spi_init_struct *)p_struct)->master_nss_delay    = SPI_NSS_IDLENESS_00CYCLE;
        ((hal_spi_init_struct *)p_struct)->fifo_threshold      = SPI_FIFO_TH_01DATA;
        ((hal_spi_init_struct *)p_struct)->crc_calculation     = SPI_CRC_DISABLE;
        ((hal_spi_init_struct *)p_struct)->crc_length          = SPI_CRCSIZE_8BIT;
        ((hal_spi_init_struct *)p_struct)->crc_poly            = SPI_CRC_DEFAULT;
        ((hal_spi_init_struct *)p_struct)->crc_tx              = SPI_CRC_TX_ALLZONE;
        ((hal_spi_init_struct *)p_struct)->crc_rx              = SPI_CRC_RX_ALLZONE;
        ((hal_spi_init_struct *)p_struct)->master_data_delay   = SPI_DATA_IDLENESS_00CYCLE;
        ((hal_spi_init_struct *)p_struct)->master_keep_iostate = SPI_MASTER_KEEPIO_DISABLE;
        ((hal_spi_init_struct *)p_struct)->io_swap             = SPI_MOSI_MISO_PIN_SWAP_DISABLE;
        ((hal_spi_init_struct *)p_struct)->word_access         = SPI_BYTEN_ACCESS;
        ((hal_spi_init_struct *)p_struct)->byte_access         = SPI_HALF_WORD_ACCESS;
        ((hal_spi_init_struct *)p_struct)->auto_suspend        = SPI_CONTINUOUS;
        break;
    case HAL_SPI_DEV_STRUCT:
        /* initialize SPI device information structure with the default values */
        ((hal_spi_dev_struct *)p_struct)->periph                           = 0U;
        ((hal_spi_dev_struct *)p_struct)->spi_irq.error_handler            = NULL;
        ((hal_spi_dev_struct *)p_struct)->spi_irq.receive_handler          = NULL;
        ((hal_spi_dev_struct *)p_struct)->spi_irq.transmit_receive_handler = NULL;
        ((hal_spi_dev_struct *)p_struct)->spi_irq.transmit_handler         = NULL;
        ((hal_spi_dev_struct *)p_struct)->spi_irq.suspend_handler          = NULL;
        ((hal_spi_dev_struct *)p_struct)->p_dma_rx                         = NULL;
        ((hal_spi_dev_struct *)p_struct)->p_dma_tx                         = NULL;
        ((hal_spi_dev_struct *)p_struct)->txbuffer.buffer                  = NULL;
        ((hal_spi_dev_struct *)p_struct)->txbuffer.length                  = 0U;
        ((hal_spi_dev_struct *)p_struct)->txbuffer.pos                     = 0U;
        ((hal_spi_dev_struct *)p_struct)->rxbuffer.buffer                  = NULL;
        ((hal_spi_dev_struct *)p_struct)->rxbuffer.length                  = 0U;
        ((hal_spi_dev_struct *)p_struct)->rxbuffer.pos                     = 0U;
        ((hal_spi_dev_struct *)p_struct)->rx_callback                      = NULL;
        ((hal_spi_dev_struct *)p_struct)->tx_callback                      = NULL;
        ((hal_spi_dev_struct *)p_struct)->error_callback                   = NULL;
        ((hal_spi_dev_struct *)p_struct)->suspend_callback                 = NULL;
        ((hal_spi_dev_struct *)p_struct)->abort_callback                   = NULL;
        ((hal_spi_dev_struct *)p_struct)->state                            = HAL_SPI_STATE_READY;
        ((hal_spi_dev_struct *)p_struct)->error_code                       = HAL_SPI_ERROR_NONE;
        ((hal_spi_dev_struct *)p_struct)->mutex                            = HAL_MUTEX_UNLOCKED;
        break;
    case HAL_SPI_IRQ_STRUCT:
        /* initialize SPI device information structure with the default values */
        ((hal_spi_irq_struct *)p_struct)->receive_handler          = NULL;
        ((hal_spi_irq_struct *)p_struct)->transmit_handler         = NULL;
        ((hal_spi_irq_struct *)p_struct)->transmit_receive_handler = NULL;
        ((hal_spi_irq_struct *)p_struct)->suspend_handler          = NULL;
        ((hal_spi_irq_struct *)p_struct)->error_handler            = NULL;
        break;
    case HAL_SPI_USER_CALLBACK_STRUCT:
        /* initialize SPI device information structure with the default values */
        ((hal_spi_user_callback_struct *)p_struct)->complete_func      = NULL;
        ((hal_spi_user_callback_struct *)p_struct)->half_complete_func = NULL;
        ((hal_spi_user_callback_struct *)p_struct)->abort_func         = NULL;
        ((hal_spi_user_callback_struct *)p_struct)->suspend_func       = NULL;
        ((hal_spi_user_callback_struct *)p_struct)->error_func         = NULL;
        break;
    default:
        error_code = HAL_ERR_VAL;
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        break;
    }

    return error_code;
}

/*!
    \brief      initialize SPI
    \param[in]  spi_dev: SPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  periph: SPI peripheral address
    \param[in]  p_init: the initialization data needed to initialize SPI
                  device_mode: SPI_MASTER, SPI_SLAVE
                  trans_mode: the argument could be selected from enumeration <hal_hal_spi_trans_mode_enum>
                  ti_mode: SPI_TIMODE_DISABLE, SPI_TIMODE_ENABLE
                  data_size: the argument could be selected from enumeration <hal_spi_datasize_enum>
                  endian: SPI_ENDIAN_MSB, SPI_ENDIAN_LSB
                  prescaler: the argument could be selected from enumeration <hal_spi_prescaler_enum>
                  ckol: SPI_CKOL_HIGH, SPI_CKOL_LOW
                  ckoh: SPI_CKOH_FIRST, SPI_CKOH_SECOND
                  nss_level: SPI_NSS_HIGH, SPI_NSS_LOW
                  fifo_threshold: the argument could be selected from enumeration <hal_spi_fifo_threshold_enum>
                  crc_calculation: SPI_CRC_DISABLE, SPI_CRC_ENABLE
                  crc_length: the argument could be selected from enumeration <hal_spi_crc_length_enum>
                  crc_poly: 0x00 ~ 0xFF
                  crc_tx: SPI_CRC_TX_ALLZONE, SPI_CRC_TX_ALLONE
                  crc_rx: SPI_CRC_RX_ALLZONE, SPI_CRC_RX_ALLONE
                  master_data_delay: the argument could be selected from enumeration <hal_spi_master_data_delay>
                  master_nss_delay: the argument could be selected from enumeration <hal_spi_master_nss_delay>
                  master_keep_iostate: SPI_MASTER_KEEPIO_ENABLE, SPI_MASTER_KEEPIO_DISABLE
                  io_swap:SPI_MOSI_MISO_PIN_SWAP_ENABLE,SPI_MOSI_MISO_PIN_SWAP_DISABLE
                  word_access:SPI_WORD_ACCESS,SPI_HALF_WORD_ACCESS
                  byte_access:SPI_BYTE_ACCESS,SPI_HALF_BYTE_ACCESS
                  auto_suspend:SPI_AUTO_SUSPEND,SPI_CONTINUOUS
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_init(hal_spi_dev_struct *spi_dev, uint32_t periph, hal_spi_init_struct *p_init)
{
    uint32_t reg0       = 0U;
    uint32_t reg1       = 0U;
    uint32_t reg2       = 0U;
    uint32_t reg3       = 0U;
    int32_t error_code  = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi_dev) || (NULL == p_init)) {
        HAL_DEBUGE("pointer [spi_dev] or [p_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((SPI0 != periph) && (SPI1 != periph) && (SPI2 != periph) && (SPI3 != periph) && (SPI4 != periph)
       && (SPI5 != periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(((SPI3 != periph) && (SPI4 != periph))
       && ((SPI_TRANSMODE_QUADRECEIVE == p_init->trans_mode) || (SPI_TRANSMODE_QUADTRANSMIT == p_init->trans_mode))) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    spi_dev->periph = periph;
    spi_dev->state  = HAL_SPI_STATE_BUSY;

    /* SPI init */
    reg0 = SPI_CFG0(periph);
    reg1 = SPI_CFG1(periph);
    reg2 = SPI_CTL0(periph);
    reg3 = SPI_QCTL(periph);

    reg0 &= SPI_INIT_CFG0_MASK;
    reg1 &= SPI_INIT_CFG1_MASK;
    reg2 &= SPI_INIT_CTL0_MASK;

    /* SPI3\SPI4 quad mode */
    if(((SPI3 == spi_dev->periph) || (SPI4 == spi_dev->periph)) && \
       ((SPI_TRANSMODE_QUADRECEIVE == p_init->trans_mode) || (SPI_TRANSMODE_QUADTRANSMIT == p_init->trans_mode))) {
        /* Set SPI3\SPI4 Quad Mode*/
        reg3 &= SPI_INIT_QCTL_MASK;

        reg3 |= ((uint32_t)p_init->trans_mode);
        /* write to SPI_QCTL register */
        SPI_QCTL(periph) = reg3;

        p_init->endian          = SPI_ENDIAN_MSB;
        p_init->crc_calculation = SPI_CRC_DISABLE;
        p_init->data_size       = SPI_DATASIZE_8BIT;
    } else {
        reg1 |= ((uint32_t)p_init->trans_mode);
    }

    /* spi normal mode configuration */
    /* master mode */
    if((SPI_MASTER == p_init->device_mode)) {
        /* hardware mode */
        if(SPI_NSS_HARD == p_init->nss) {
            reg1 |= SPI_CFG1_MSTMOD;
            reg1 &= (~SPI_CFG1_NSSIM);
            reg1 |= SPI_CFG1_NSSDRV;
            if(SPI_NSS_INPUT == p_init->nss_signal_type) {
                reg1 &= (~SPI_CFG1_NSSDRV);
            } else {
                /* do nothing */
            }
        } else {
            /* software mode */
            reg1 |= SPI_CFG1_MSTMOD;
            reg1 |= SPI_CFG1_NSSIM;
            reg2 |= SPI_CTL0_NSSI;
            if(SPI_NSS_INPUT == p_init->nss_signal_type) {
                reg2 &= (~SPI_CTL0_NSSI);
            } else {
                /* do nothing */
            }
        }
    } else {
        /* do nothing */
    }

    /* slave mode */
    if((SPI_SLAVE == p_init->device_mode)) {
        /* hardware mode */
        if(SPI_NSS_HARD == p_init->nss) {
            reg1 &= (~SPI_CFG1_MSTMOD);
            reg1 &= (~SPI_CFG1_NSSIM);
        } else {
            /* software mode */
            reg1 &= (~SPI_CFG1_MSTMOD);
            reg1 |= SPI_CFG1_NSSIM;
        }
    } else {
        /* do nothing */
    }

    if(SPI_NSS_INVALID_PULSE == p_init->nss_output) {
        reg1 |= SPI_CFG1_NSSCTL;
    } else {
        /* do nothing */
    }

    /* get configure info */
    reg0 |= ((uint32_t)p_init->fifo_threshold | (uint32_t)p_init->data_size | (uint32_t)p_init->byte_access | \
             (uint32_t)p_init->word_access | (uint32_t)p_init->prescaler);
    reg1 |= (p_init->device_mode | p_init->ti_mode | p_init->ckoh | p_init->ckol | p_init->endian | p_init->nss | \
             p_init->nss_level | p_init->master_data_delay | p_init ->master_nss_delay | \
             p_init->master_keep_iostate | p_init->io_swap);
    reg2 |= p_init->auto_suspend;

    /* use crc calculation */
    if(SPI_CRC_ENABLE == p_init->crc_calculation) {
        reg0 |= (p_init ->crc_calculation);
        reg0 |= ((uint32_t)p_init ->crc_length);
        reg2 |= SPI_CTL0_CRCFS;
        reg2 |= (p_init->crc_tx | p_init->crc_rx);
        SPI_CRCPOLY(periph) = p_init->crc_poly;
    } else {
        /* do nothing */
    }

    /* write value to register */
    SPI_CFG0(periph) = reg0;
    SPI_CFG1(periph) = reg1;
    SPI_CTL0(periph) = reg2;

    /* select SPI mode */
    SPI_I2SCTL(periph) &= (~SPI_I2SCTL_I2SSEL);

    /* enable spi */
    SPI_CTL0(spi_dev->periph) |= SPI_CTL0_SPIEN;

    spi_dev->state = HAL_SPI_STATE_READY;

    return error_code;
}

/*!
    \brief      deinitialize SPI
    \param[in]  spi_dev: SPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_deinit(hal_spi_dev_struct *spi_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == spi_dev) {
        HAL_DEBUGE("pointer [spi_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    spi_dev->state = HAL_SPI_STATE_BUSY;

    /* reset SPI */
    _spi_deinit(spi_dev->periph);

    spi_dev->state = HAL_SPI_STATE_READY;

    return error_code;
}

/*!
    \brief      SPI abort ongoing transfer function blocking mode
    \param[in]  spi_dev: SPI device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \note       this procedure could be used for aborting any ongoing transfer (Tx and Rx),
                    started in Interrupt or DMA mode.
    \note       This procedure is executed in blocking mode : when exiting function, Abort is considered as completed.
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_ABORT details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_abort(hal_spi_dev_struct *spi_dev)
{
    int32_t error_code  = HAL_ERR_NONE;
    int32_t result      = 0;
    uint32_t tick_start = 0U;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == spi_dev) {
        HAL_DEBUGE("pointer [spi_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Set SPI state to aborting  */
    spi_dev->state = HAL_SPI_STATE_ABORT;
    tick_start     = hal_sys_basetick_count_get();
    /* If master communication on going, make sure current frame is done before closing the connection */
    if(RESET != (SPI_CTL0(spi_dev->periph) & SPI_CTL0_MSTART)) {
        /* request transfer suspend */
        SPI_CTL0(spi_dev->periph) |= SPI_CTL0_MSPDR;
        while(RESET != (SPI_CTL0(spi_dev->periph) & SPI_CTL0_MSTART)) {
            if(HAL_TIMEOUT_FOREVER != SPI_TIMEOUT_VALUE) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, SPI_TIMEOUT_VALUE)) {
                    spi_dev->error_code = HAL_SPI_ERROR_ABORT;
                    break;
                }
            }
        }

        /* wait SPD flag set */
        while(RESET == (SPI_STAT(spi_dev->periph) & SPI_STAT_SPD)) {
            if(HAL_TIMEOUT_FOREVER != SPI_TIMEOUT_VALUE) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, SPI_TIMEOUT_VALUE)) {
                    spi_dev->error_code = HAL_SPI_ERROR_ABORT;
                    break;
                }
            }
        }
    }

    /* disable spi dma tx interrupt request */
    if(RESET != (SPI_CFG0(spi_dev->periph) & SPI_CFG0_DMATEN)) {
        if(NULL != spi_dev->p_dma_tx) {
            result = hal_dma_stop(spi_dev->p_dma_tx);
            if(HAL_ERR_NONE != result) {
                spi_dev->error_code = HAL_SPI_ERROR_ABORT;
            }
        }
    }

    /* disable spi dma rx interrupt request*/
    if(RESET != (SPI_CFG0(spi_dev->periph) & SPI_CFG0_DMAREN)) {
        if(NULL != spi_dev->p_dma_rx) {
            result = hal_dma_stop(spi_dev->p_dma_rx);
            if(HAL_ERR_NONE != result) {
                spi_dev->error_code = HAL_SPI_ERROR_ABORT;
            }
        }
    }

    /* abort transfer and clear flags */
    _spi_abort_transmit(spi_dev);

    if(HAL_SPI_ERROR_ABORT == spi_dev->error_code) {
        error_code = HAL_ERR_ABORT;
    } else {
        spi_dev->error_code = HAL_SPI_ERROR_NONE;
    }

    spi_dev->state = HAL_SPI_STATE_READY;

    return error_code;
}

/*!
    \brief      SPI abort ongoing transfer function interrupt mode
    \param[in]  spi_dev: SPI device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \note       this procedure could be used for aborting any ongoing transfer (Tx and Rx),
                    started in Interrupt or DMA mode.
    \note       This procedure is executed in blocking mode : when exiting function, Abort is considered as completed.
    \param[in]  p_user_func: point to user callback function
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_ABORT details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_abort_it(hal_spi_dev_struct *spi_dev, hal_spi_user_callback_struct *p_user_func)
{
    int32_t error_code               = HAL_ERR_NONE;
    int32_t result                   = 0;
    uint32_t tick_start              = 0U;
    uint32_t dma_transmit_abort_done = 1U;
    uint32_t dma_receive_abort_done  = 1U;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == spi_dev) {
        HAL_DEBUGE("pointer [spi_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Set SPI state to aborting  */
    spi_dev->state = HAL_SPI_STATE_ABORT;
    tick_start     = hal_sys_basetick_count_get();
    /* If master communication on going, make sure current frame is done before closing the connection */
    if(RESET != (SPI_CTL0(spi_dev->periph) & SPI_CTL0_MSTART)) {
        /* request transfer suspend */
        SPI_CTL0(spi_dev->periph) |= SPI_CTL0_MSPDR;
        while(RESET != (SPI_CTL0(spi_dev->periph) & SPI_CTL0_MSTART)) {
            if(HAL_TIMEOUT_FOREVER != SPI_TIMEOUT_VALUE) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, SPI_TIMEOUT_VALUE)) {
                    spi_dev->error_code = HAL_SPI_ERROR_ABORT;
                    break;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }

        while(RESET == (SPI_STAT(spi_dev->periph) & SPI_STAT_SPD)) {
            if(HAL_TIMEOUT_FOREVER != SPI_TIMEOUT_VALUE) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, SPI_TIMEOUT_VALUE)) {
                    spi_dev->error_code = HAL_SPI_ERROR_ABORT;
                    break;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }
    }

    /* disable spi dma tx interrupt request */
    if(RESET != (SPI_CFG0(spi_dev->periph) & SPI_CFG0_DMATEN)) {
        if(NULL != spi_dev->p_dma_tx) {
            /* check user callback valid */
            if(NULL != p_user_func->abort_func) {
                spi_dev->abort_callback = (void *)p_user_func->abort_func;
            } else {
                /* do nothing */
            }

            /* set DMA abort Complete callback if SPI DMA tx request if enabled */
            spi_dev->p_dma_tx->dma_irq.abort_handle = (hal_irq_handle_cb)_spi_abort_dma_transmit;

            /* abort DMA transmit to SPI periph */
            result = hal_dma_stop_interrupt(spi_dev->p_dma_tx);
            if(HAL_ERR_NONE != result) {
                dma_transmit_abort_done = 0U;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* disable spi dma rx interrupt request*/
    if(RESET != (SPI_CFG0(spi_dev->periph) & SPI_CFG0_DMAREN)) {
        if(NULL != spi_dev->p_dma_rx) {
            /* check user callback valid */
            if(NULL != p_user_func->abort_func) {
                spi_dev->abort_callback = (void *)p_user_func->abort_func;
            } else {
                /* do nothing */
            }

            /* set DMA abort Complete callback if SPI DMA rx request if enabled */
            spi_dev->p_dma_rx->dma_irq.abort_handle = (hal_irq_handle_cb)_spi_abort_dma_receive;

            /* abort DMA receive to SPI periph */
            result = hal_dma_stop_interrupt(spi_dev->p_dma_rx);
            if(HAL_ERR_NONE != result) {
                dma_receive_abort_done = 0U;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if((0U == dma_transmit_abort_done) && (0U == dma_receive_abort_done)) {

        _spi_abort_transmit(spi_dev);

        if(HAL_SPI_ERROR_ABORT == spi_dev->error_code) {
            error_code = HAL_ERR_ABORT;
        } else {
            spi_dev->error_code = HAL_SPI_ERROR_NONE;
        }

        if(NULL != p_user_func->abort_func) {
            p_user_func->abort_func(spi_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    spi_dev->state = HAL_SPI_STATE_READY;

    return error_code;
}

/*!
    \brief      SPI DMA pause function
    \param[in]  spi_dev: SPI device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_dma_pause(hal_spi_dev_struct *spi_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == spi_dev) {
        HAL_DEBUGE("pointer [spi_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_SPI_DMA_DISABLE(spi_dev->periph, SPI_DMA_TRANSMIT);
    HAL_SPI_DMA_DISABLE(spi_dev->periph, SPI_DMA_RECEIVE);

    return error_code;
}

/*!
    \brief      SPI DMA resume function
    \param[in]  spi_dev: SPI device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_dma_resume(hal_spi_dev_struct *spi_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == spi_dev) {
        HAL_DEBUGE("pointer [spi_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_SPI_DMA_ENABLE(spi_dev->periph, SPI_DMA_TRANSMIT);
    HAL_SPI_DMA_ENABLE(spi_dev->periph, SPI_DMA_RECEIVE);

    return error_code;
}

/*!
    \brief      SPI DMA stop function
    \param[in]  spi_dev: SPI device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_dma_stop(hal_spi_dev_struct *spi_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == spi_dev) {
        HAL_DEBUGE("pointer [spi_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(NULL != spi_dev->p_dma_rx) {
        hal_dma_stop(spi_dev->p_dma_rx);
    } else {
        /* do nothing */
    }

    if(NULL != spi_dev->p_dma_tx) {
        hal_dma_stop(spi_dev->p_dma_tx);
    } else {
        /* do nothing */
    }

    HAL_SPI_DMA_DISABLE(spi_dev->periph, SPI_DMA_TRANSMIT);
    HAL_SPI_DMA_DISABLE(spi_dev->periph, SPI_DMA_RECEIVE);
    spi_dev->state = HAL_SPI_STATE_READY;

    return error_code;
}

/*!
   \brief      flush the Rx fifo
   \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
   \param[out] none
   \retval     error code: HAL_ERR_NONE, HAL_ERR_TIMEOUT, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_flush_rxfifo(uint32_t spi_periph)
{
    int32_t error_code = HAL_ERR_NONE;
    uint8_t count      = 0U;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(0U == spi_periph) {
        HAL_DEBUGE("parameter [spi_periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    while((SPI_RXFIFO_0PACKET != (SPI_STAT(spi_periph) & SPI_STAT_RPLVL)) || \
          ((RESET != (SPI_STAT(spi_periph) & SPI_STAT_RWNE)))) {
        count += (uint8_t)1UL;
        if(count > SPI_FIFO_SIZE) {
            error_code = HAL_ERR_TIMEOUT;
            break;
        } else {
            /* do nothing */
        }
    }

    return error_code;
}

/*!
    \brief      transmit amounts of data, poll transmit process and completed status, the function is blocking
    \param[in]  spi_dev: SPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_txbuffer: pointer to txbuffer
    \param[in]  length: length of data to be sent
    \param[in]  timeout_ms: timeout duration
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, details refer to
                gd32h7xx_hal.h
*/
int32_t hal_spi_transmit_poll(hal_spi_dev_struct *spi_dev, uint8_t *p_txbuffer, uint32_t length, uint32_t timeout_ms)
{
    int32_t error_code = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi_dev) || (NULL == p_txbuffer)) {
        HAL_DEBUGE("pointer [spi_dev] or [p_txbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length) || (0U == timeout_ms)) {
        HAL_DEBUGE("parameter [length] or [timeout_ms] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock SPI */
    HAL_LOCK(spi_dev);

    if(spi_dev->state != HAL_SPI_STATE_READY) {
        error_code = HAL_ERR_BUSY;
    } else {
        SPI_CTL0(spi_dev->periph) &= (~SPI_CTL0_SPIEN);

        hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_TXURERR);

        SPI_CTL1(spi_dev->periph) &= ~SPI_CTL1_TXSIZE;
        SPI_CTL1(spi_dev->periph) |= length;

        /* set the transaction information */
        spi_dev->state      = HAL_SPI_STATE_BUSY_TX;
        spi_dev->error_code = HAL_SPI_ERROR_NONE;

        spi_dev->txbuffer.buffer = (uint8_t *)p_txbuffer;
        spi_dev->txbuffer.length = length;
        spi_dev->txbuffer.pos    = length;

        /* init field not used to zero */
        spi_dev->rxbuffer.buffer = (uint8_t *)NULL;
        spi_dev->rxbuffer.length = 0U;
        spi_dev->rxbuffer.pos    = 0U;

        /* enable crc */
        if(SPI_CRC_ENABLE == (SPI_CFG0(spi_dev->periph) & SPI_CFG0_CRCEN)) {
            hals_spi_crc_off(spi_dev->periph);
            hals_spi_crc_on(spi_dev->periph);
        } else {
            /* do nothing */
        }

        /* enable spi */
        SPI_CTL0(spi_dev->periph) |= SPI_CTL0_SPIEN;

        /* Master transfer start */
        if(RESET != ((SPI_CFG1(spi_dev->periph) & SPI_CFG1_MSTMOD))) {
            SPI_CTL0(spi_dev->periph) |= SPI_CTL0_MSTART;
        } else {
            /* do nothing */
        }

        /* Transmit data in 32 Bit mode */
        if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) > SPI_DATASIZE_16BIT) {
            while(0U < spi_dev->txbuffer.pos) {
                /* transfer data if tp set*/
                if(RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_TP))) {
                    SPI_TDATA(spi_dev->periph) = *(uint32_t *)p_txbuffer;
                    p_txbuffer += sizeof(uint32_t);
                    spi_dev->txbuffer.pos--;
                } else {
                    /* check timeout */
                    if(HAL_ERR_NONE != _spi_wait_flag_timeout(spi_dev->periph, SPI_FLAG_TP, SET, timeout_ms)) {
                        HAL_DEBUGE("spi get TP timeout");
                        _spi_stop_transfer(spi_dev);
                        error_code = HAL_ERR_TIMEOUT;
                        break;
                    }
                }
            }
        } else if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) > SPI_DATASIZE_8BIT) {
            /* Transmit data in 16 Bit mode */
            while(0U < spi_dev->txbuffer.pos) {
                /* transfer data if tp set*/
                if(RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_TP))) {
                    if((2U <= spi_dev->txbuffer.pos) && \
                       ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                        SPI_TDATA(spi_dev->periph) = *(uint32_t *)p_txbuffer;
                        p_txbuffer += sizeof(uint32_t);
                        spi_dev->txbuffer.pos -= 2UL;
                    } else {
                        SPI_TDATA(spi_dev->periph) = *(uint16_t *)p_txbuffer;
                        p_txbuffer += sizeof(uint16_t);
                        spi_dev->txbuffer.pos--;
                    }
                } else {
                    /* check timeout */
                    if(HAL_ERR_NONE != _spi_wait_flag_timeout(spi_dev->periph, SPI_FLAG_TP, SET, timeout_ms)) {
                        HAL_DEBUGE("spi get TP timeout");
                        _spi_stop_transfer(spi_dev);
                        error_code = HAL_ERR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                }
            }
        } else {
            /* Transmit data in 8 Bit mode */
            while(0U < spi_dev->txbuffer.pos) {
                /* wait until TP is set */
                if(RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_TP))) {
                    /* check fifo level */
                    if((4U <= spi_dev->txbuffer.pos) && \
                       ((uint32_t)SPI_FIFO_TH_03DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                        SPI_TDATA(spi_dev->periph) = *(uint32_t *)p_txbuffer;
                        p_txbuffer += sizeof(uint32_t);
                        spi_dev->txbuffer.pos -= 4UL;
                    } else if((2U <= spi_dev->txbuffer.pos) && \
                              ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                        SPI_TDATA(spi_dev->periph) = *(uint16_t *)p_txbuffer;
                        p_txbuffer += sizeof(uint16_t);
                        spi_dev->txbuffer.pos -= 2UL;
                    } else {
                        SPI_TDATA(spi_dev->periph) = (uint8_t)(*p_txbuffer);
                        p_txbuffer += sizeof(uint8_t);
                        spi_dev->txbuffer.pos--;
                    }
                } else {
                    if(HAL_ERR_NONE != _spi_wait_flag_timeout(spi_dev->periph, SPI_FLAG_TP, SET, timeout_ms)) {
                        HAL_DEBUGE("spi get TP timeout");
                        _spi_stop_transfer(spi_dev);
                        error_code = HAL_ERR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                }
            }
        }

        /* check crc flag */
        if(HAL_SPI_GET_CRC_USED(spi_dev->periph) == SPI_CRC_ENABLE) {
            if(RESET != hals_spi_flag_get(spi_dev->periph, SPI_FLAG_CRCERR)) {
                error_code = HAL_ERR_VAL;
            } else {
                /* do nothing */
            }
        }

        /* wait transmit or receive complete */
        if(HAL_ERR_NONE != _spi_wait_flag_timeout(spi_dev->periph, SPI_FLAG_ET, SET, timeout_ms)) {
            HAL_DEBUGE("spi get ET timeout");
            _spi_stop_transfer(spi_dev);
            error_code = HAL_ERR_TIMEOUT;
        } else {
            /* do nothing */
        }
    }

    spi_dev->state = HAL_SPI_STATE_READY;

    /* unlock SPI */
    HAL_UNLOCK(spi_dev);

    return error_code;
}

/*!
    \brief      receive amounts of data, poll receive process and completed status, the function is blocking
    \param[in]  spi_dev: SPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  length: length of data to be sent
    \param[in]  timeout_ms: timeout duration
    \param[out] p_rxbuffer: pointer to rxbuffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_HARDWARE, HAL_ERR_TIMEOUT, HAL_ERR_VAL
                            details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_receive_poll(hal_spi_dev_struct *spi_dev, uint8_t *p_rxbuffer, uint32_t length, uint32_t timeout_ms)
{
    int32_t error_code = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi_dev) || (NULL == p_rxbuffer)) {
        HAL_DEBUGE("pointer [spi_dev] or [p_rxbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length) || (0U == timeout_ms)) {
        HAL_DEBUGE("parameter [length] or [timeout_ms] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((SPI_MASTER == HAL_SPI_GET_DEVICE_MODE(spi_dev->periph))) {
        /* call transmit-receive function to send dummy data generate clock */
        error_code = hal_spi_transmit_receive_poll(spi_dev, p_rxbuffer, p_rxbuffer, length, timeout_ms);
    } else {
        /* lock SPI */
        HAL_LOCK(spi_dev);

        if(HAL_SPI_STATE_READY != spi_dev->state) {
            error_code = HAL_ERR_BUSY;
        } else {
            SPI_CTL1(spi_dev->periph) &= ~SPI_CTL1_TXSIZE;
            SPI_CTL1(spi_dev->periph) |= length;

            spi_dev->state      = HAL_SPI_STATE_BUSY_RX;
            spi_dev->error_code = HAL_SPI_ERROR_NONE;

            /* init field not used to zero */
            spi_dev->txbuffer.buffer = (uint8_t *)NULL;
            spi_dev->txbuffer.length = 0U;
            spi_dev->txbuffer.pos    = 0U;

            /* set the receiver information */
            spi_dev->rxbuffer.buffer = (uint8_t *)p_rxbuffer;
            spi_dev->rxbuffer.length = length;
            spi_dev->rxbuffer.pos    = length;

            /* clear callback */
            spi_dev->tx_callback = NULL;
            spi_dev->rx_callback = NULL;

            /* enable crc */
            if(SPI_CRC_ENABLE == (SPI_CFG0(spi_dev->periph) & SPI_CFG0_CRCEN)) {
                hals_spi_crc_off(spi_dev->periph);
                hals_spi_crc_on(spi_dev->periph);
            } else {
                /* do nothing */
            }

            /* enable spi */
            if(SPI_CTL0_SPIEN != (SPI_CTL0(spi_dev->periph) & SPI_CTL0_SPIEN)) {
                SPI_CTL0(spi_dev->periph) |= SPI_CTL0_SPIEN;
            } else {
                /* do nothing */
            }

            /* Master transfer start */
            if(RESET != ((SPI_CFG1(spi_dev->periph) & SPI_CFG1_MSTMOD))) {
                SPI_CTL0(spi_dev->periph) |= SPI_CTL0_MSTART;
            } else {
                /* do nothing */
            }

            /* received data in 32 Bit mode */
            if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) > SPI_DATASIZE_16BIT) {
                while(0U < spi_dev->rxbuffer.pos) {
                    /* transfer data if tp set*/
                    if(RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_RP))) {
                        *(uint32_t *)p_rxbuffer = SPI_RDATA(spi_dev->periph);
                        p_rxbuffer += sizeof(uint32_t);
                        spi_dev->rxbuffer.pos--;
                    } else {
                        /* check timeout */
                        if(HAL_ERR_NONE != _spi_wait_flag_timeout(spi_dev->periph, SPI_FLAG_RP, SET, timeout_ms)) {
                            HAL_DEBUGE("spi get TP timeout");
                            _spi_stop_transfer(spi_dev);
                            error_code = HAL_ERR_TIMEOUT;
                            break;
                        } else {
                            /* do nothing */
                        }
                    }
                }
            } else if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) > SPI_DATASIZE_8BIT) {
                /* received data in 16 Bit mode */
                while(0U < spi_dev->rxbuffer.pos) {
                    /* transfer data if tp set*/
                    if(RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_RP))) {
                        if((2U <= spi_dev->rxbuffer.pos) && \
                           ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                            *(uint32_t *)p_rxbuffer = SPI_RDATA(spi_dev->periph);
                            SWAP_HIGH_LOW_16BITS(*(uint32_t *)p_rxbuffer);
                            p_rxbuffer += sizeof(uint32_t);
                            spi_dev->rxbuffer.pos -= 2U;
                        } else {
                            *(uint16_t *)p_rxbuffer = (uint8_t)SPI_RDATA(spi_dev->periph);
                            p_rxbuffer += sizeof(uint16_t);
                            spi_dev->rxbuffer.pos--;
                        }
                    } else {
                        /* check timeout */
                        if(HAL_ERR_NONE != _spi_wait_flag_timeout(spi_dev->periph, SPI_FLAG_RP, SET, timeout_ms)) {
                            HAL_DEBUGE("spi get TP timeout");
                            _spi_stop_transfer(spi_dev);
                            error_code = HAL_ERR_TIMEOUT;
                            break;
                        } else {
                            /* do nothing */
                        }
                    }
                }
            } else {
                /* received data in 8 Bit mode */
                while(0U < spi_dev->rxbuffer.pos) {
                    /* transfer data if tp set*/
                    if(RESET != hals_spi_flag_get(spi_dev->periph, SPI_FLAG_RP)) {
                        if((4U <= spi_dev->rxbuffer.pos) && \
                           ((uint32_t)SPI_FIFO_TH_03DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                            *(uint32_t *)p_rxbuffer = SPI_RDATA(spi_dev->periph);
                            SWAP32_BE_TO_LE(*(uint32_t *)p_rxbuffer);
                            p_rxbuffer += sizeof(uint32_t);
                            spi_dev->rxbuffer.pos -= 4UL;
                        } else if((2U <= spi_dev->rxbuffer.pos) && \
                                  ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                            *(uint16_t *)p_rxbuffer = (uint16_t)(SPI_RDATA(spi_dev->periph)& 0xFFFFU);
                            SWAP16_BE_TO_LE(*(uint16_t *)p_rxbuffer);
                            p_rxbuffer += sizeof(uint16_t);
                            spi_dev->rxbuffer.pos -= 2UL;
                        } else {
                            *(uint8_t *)p_rxbuffer = (uint8_t)(SPI_RDATA(spi_dev->periph) & 0xFFU);;
                            p_rxbuffer += sizeof(uint8_t);
                            spi_dev->rxbuffer.pos--;
                        }
                    } else {
                        /* check timeout */
                        if(HAL_ERR_NONE != _spi_wait_flag_timeout(spi_dev->periph, SPI_FLAG_RP, SET, timeout_ms)) {
                            HAL_DEBUGE("spi get RP timeout");
                            _spi_stop_transfer(spi_dev);
                            error_code = HAL_ERR_TIMEOUT;
                            break;
                        } else {
                            /* do nothing */
                        }
                    }
                }
            }

            if (HAL_ERR_NONE == error_code) {
                /* check crc flag */
                if(HAL_SPI_GET_CRC_USED(spi_dev->periph) == SPI_CRC_ENABLE) {
                    if(RESET != hals_spi_flag_get(spi_dev->periph, SPI_FLAG_CRCERR)) {
                        error_code = HAL_ERR_HARDWARE;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            if (HAL_ERR_NONE == error_code) {
                /* wait transmit or receive complete */
                if(HAL_ERR_NONE != _spi_wait_flag_timeout(spi_dev->periph, SPI_FLAG_ET, SET, timeout_ms)) {
                    HAL_DEBUGE("spi get ET timeout");
                    _spi_stop_transfer(spi_dev);
                    error_code = HAL_ERR_TIMEOUT;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }
    }

    spi_dev->state = HAL_SPI_STATE_READY;

    /* unlock SPI */
    HAL_UNLOCK(spi_dev);

    return error_code;
}

/*!
    \brief      transmit and receive amounts of data, poll receive process and completed status, the function is
   blocking
    \param[in]  spi_dev: SPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_txbuffer: pointer to txbuffer
    \param[in]  length: length of data to be sent
    \param[in]  timeout_ms: timeout duration
    \param[out] p_rxbuffer: pointer to rxbuffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_TIMEOUT, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY details refer to
   gd32h7xx_hal.h
*/
int32_t hal_spi_transmit_receive_poll(hal_spi_dev_struct *spi_dev, uint8_t *p_txbuffer, uint8_t *p_rxbuffer, \
                                      uint32_t length, uint32_t timeout_ms)
{
    int32_t error_code = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi_dev) || (NULL == p_txbuffer) || (NULL == p_rxbuffer)) {
        HAL_DEBUGE("pointer [spi_dev] or [p_txbuffer] or [p_rxbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length) || (0U == timeout_ms)) {
        HAL_DEBUGE("parameter [length] or [timeout_ms] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock SPI */
    HAL_LOCK(spi_dev);

    if(HAL_SPI_STATE_READY != spi_dev->state) {
        error_code = HAL_ERR_BUSY;
    } else {
        SPI_CTL0(spi_dev->periph) &= (~SPI_CTL0_SPIEN);
        SPI_CTL1(spi_dev->periph) &= ~SPI_CTL1_TXSIZE;
        SPI_CTL1(spi_dev->periph) |= length;

        /* set transmit information */
        spi_dev->state      = HAL_SPI_STATE_BUSY_TX_RX;
        spi_dev->error_code = HAL_SPI_ERROR_NONE;

        spi_dev->txbuffer.buffer = (uint8_t *)p_txbuffer;
        spi_dev->txbuffer.length = length;
        spi_dev->txbuffer.pos    = length;

        /* set receive information*/
        spi_dev->rxbuffer.buffer = (uint8_t *)p_rxbuffer;
        spi_dev->rxbuffer.length = length;
        spi_dev->rxbuffer.pos    = length;

        /* enable crc */
        if(SPI_CRC_ENABLE == (SPI_CFG0(spi_dev->periph) & SPI_CFG0_CRCEN)) {
            hals_spi_crc_off(spi_dev->periph);
            hals_spi_crc_on(spi_dev->periph);
        }

        /* enable spi */
        SPI_CTL0(spi_dev->periph) |= SPI_CTL0_SPIEN;

        /* Master transfer start */
        if(RESET != ((SPI_CFG1(spi_dev->periph) & SPI_CFG1_MSTMOD))) {
            SPI_CTL0(spi_dev->periph) |= SPI_CTL0_MSTART;
        } else {
            /* do nothing */
        }

        /* Transmit and received data in 32 Bit mode */
        if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) > SPI_DATASIZE_16BIT) {
            while((0U < spi_dev->txbuffer.pos) || (0U < spi_dev->rxbuffer.pos)) {
                /* transfer data if tp set*/
                if((RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_TP))) && (0U < spi_dev->txbuffer.pos)) {
                    SPI_TDATA(spi_dev->periph) = *(uint32_t *)p_txbuffer;
                    p_txbuffer += sizeof(uint32_t);
                    spi_dev->txbuffer.pos--;
                } else {
                    if(HAL_ERR_NONE != (_spi_wait_flag_timeout(spi_dev->periph, SPI_FLAG_TP, SET, timeout_ms))) {
                        HAL_DEBUGE("spi get RP timeout");
                        _spi_stop_transfer(spi_dev);
                        error_code = HAL_ERR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                }
                /* wait RP flag set and receive over buffer length valid */
                if((RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_RP))) && (0U < spi_dev->rxbuffer.pos)) {
                    *(uint32_t *)p_rxbuffer = SPI_RDATA(spi_dev->periph);
                    p_rxbuffer += sizeof(uint32_t);
                    spi_dev->rxbuffer.pos--;
                } else {
                    /* wait RP flag timeout */
                    if(HAL_ERR_NONE != (_spi_wait_flag_timeout(spi_dev->periph, SPI_FLAG_RP, SET, timeout_ms))) {
                        HAL_DEBUGE("spi get RP timeout");
                        _spi_stop_transfer(spi_dev);
                        error_code = HAL_ERR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                }
            }
        } else if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) > SPI_DATASIZE_8BIT) {
            /* Transmit and Receive data in 16 Bit mode */
            while((0U < spi_dev->txbuffer.pos) || (0U < spi_dev->rxbuffer.pos)) {
                /* transfer data if tp set*/
                if(RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_TP))) {
                    if((2U <= spi_dev->txbuffer.pos) && \
                       ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                        SPI_TDATA(spi_dev->periph) = *(uint32_t *)p_txbuffer;
                        p_txbuffer += sizeof(uint32_t);
                        spi_dev->txbuffer.pos -= 2UL;
                    } else {
                        SPI_TDATA(spi_dev->periph) = *(uint16_t *)p_txbuffer;
                        p_txbuffer += sizeof(uint16_t);
                        spi_dev->txbuffer.pos--;
                    }
                } else {
                    /* wait TP flag timeout */
                    if(HAL_ERR_NONE != (_spi_wait_flag_timeout(spi_dev->periph, SPI_FLAG_TP, SET, timeout_ms))) {
                        HAL_DEBUGE("spi get TP timeout");
                        _spi_stop_transfer(spi_dev);
                        error_code = HAL_ERR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                }
                /* check RP is set */
                if(RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_RP))) {
                    if((2U <= spi_dev->rxbuffer.pos) && \
                       ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                        *(uint32_t *)p_rxbuffer = SPI_RDATA(spi_dev->periph);
                        SWAP_HIGH_LOW_16BITS(*(uint32_t *)p_rxbuffer);
                        p_rxbuffer += sizeof(uint32_t);
                        spi_dev->rxbuffer.pos -= 2UL;
                    } else {
                        *(uint16_t *)p_rxbuffer = (uint16_t)(SPI_RDATA(spi_dev->periph) & 0xFFFFU);
                        p_rxbuffer += sizeof(uint16_t);
                        spi_dev->rxbuffer.pos--;
                    }
                } else {
                    /* wait RP flag timeout */
                    if(HAL_ERR_NONE != (_spi_wait_flag_timeout(spi_dev->periph, SPI_FLAG_RP, SET, timeout_ms))) {
                        HAL_DEBUGE("spi get RP timeout");
                        _spi_stop_transfer(spi_dev);
                        error_code = HAL_ERR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                }
            }
        } else {
            /* Transmit and Receive data in 8 Bit mode */
            while((0U < spi_dev->txbuffer.pos) || (0U < spi_dev->rxbuffer.pos)) {
                /* wait until SPI_I2S_FLAG_DP is set */
                if((RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_TP))) && (0U < spi_dev->txbuffer.pos)) {
                    /* check fifo threshold */
                    if((4U <= spi_dev->txbuffer.pos) && \
                       ((uint32_t)SPI_FIFO_TH_03DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                        SPI_TDATA(spi_dev->periph) = *(uint32_t *)p_txbuffer;
                        p_txbuffer += sizeof(uint32_t);
                        spi_dev->txbuffer.pos -= 4UL;
                    } else if((2U <= spi_dev->txbuffer.pos) && \
                              ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                        SPI_TDATA(spi_dev->periph) = *(uint16_t *)p_txbuffer;
                        p_txbuffer += sizeof(uint16_t);
                        spi_dev->txbuffer.pos -= 2UL;
                    } else {
                        SPI_TDATA(spi_dev->periph) = (uint8_t)(*p_txbuffer);
                        p_txbuffer += sizeof(uint8_t);
                        spi_dev->txbuffer.pos--;
                    }
                } else {
                    /* wait TP flag timeout */
                    if(HAL_ERR_NONE != (_spi_wait_flag_timeout(spi_dev->periph, SPI_FLAG_TP, SET, timeout_ms))) {
                        HAL_DEBUGE("spi get TP timeout");
                        _spi_stop_transfer(spi_dev);
                        error_code = HAL_ERR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                }
                /* check RP set and data length valid */
                if((RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_RP))) && (0U < spi_dev->rxbuffer.pos)) {
                    /* check fifo threshold */
                    if((4U <= spi_dev->rxbuffer.pos) && \
                       ((uint32_t)SPI_FIFO_TH_03DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                        *(uint32_t *)p_rxbuffer = SPI_RDATA(spi_dev->periph);
                        SWAP32_BE_TO_LE(*(uint32_t *)p_rxbuffer);
                        p_rxbuffer += sizeof(uint32_t);
                        spi_dev->rxbuffer.pos -= 4UL;
                    } else if((2U <= spi_dev->rxbuffer.pos) && \
                              ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                        *(uint16_t *)p_rxbuffer = (uint16_t)(SPI_RDATA(spi_dev->periph) & 0xFFFFU);
                        SWAP16_BE_TO_LE(*(uint16_t *)p_rxbuffer);
                        p_rxbuffer += sizeof(uint16_t);
                        spi_dev->rxbuffer.pos -= 2UL;
                    } else {
                        *(uint8_t *)p_rxbuffer = (uint8_t)(SPI_RDATA(spi_dev->periph) & 0xFFU);
                        p_rxbuffer += sizeof(uint8_t);
                        spi_dev->rxbuffer.pos--;
                    }
                } else {
                    /* wait RP flag timeout */
                    if(HAL_ERR_NONE != (_spi_wait_flag_timeout(spi_dev->periph, SPI_FLAG_RP, SET, timeout_ms))) {
                        HAL_DEBUGE("spi get RP timeout");
                        _spi_stop_transfer(spi_dev);
                        error_code = HAL_ERR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                }
            }
        }

        if (HAL_ERR_NONE == error_code) {
            /* check crc flag */
            if(HAL_SPI_GET_CRC_USED(spi_dev->periph) == SPI_CRC_ENABLE) {
                if(RESET != hals_spi_flag_get(spi_dev->periph, SPI_FLAG_CRCERR)) {
                    error_code = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        if (HAL_ERR_NONE == error_code) {
            /* wait transmit or receive complete */
            if(HAL_ERR_NONE != _spi_wait_flag_timeout(spi_dev->periph, SPI_FLAG_ET, SET, timeout_ms)) {
                HAL_DEBUGE("spi get ET timeout");
                _spi_stop_transfer(spi_dev);
                error_code = HAL_ERR_TIMEOUT;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    }

    spi_dev->state = HAL_SPI_STATE_READY;

    /* unlock SPI */
    HAL_UNLOCK(spi_dev);

    return error_code;
}

/*!
    \brief      transmit amounts of data by interrupt method, the function is non-blocking
    \param[in]  spi_dev: SPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_txbuffer: pointer to txbuffer
    \param[in]  length: length of data to be sent
    \param[in]  p_user_func: pointer to call back function for user
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_BUSY, details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_transmit_interrupt(hal_spi_dev_struct *spi_dev, uint8_t *p_txbuffer, uint32_t length, \
                                   hal_spi_user_callback_struct *p_user_func)
{
    int32_t error_code = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi_dev) || (NULL == p_txbuffer)) {
        HAL_DEBUGE("pointer [spi_dev] or [p_txbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((length == 0U)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
    if((NULL == p_user_func)) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock SPI */
    HAL_LOCK(spi_dev);

    if(HAL_SPI_STATE_READY != spi_dev->state) {
        error_code = HAL_ERR_BUSY;
    } else {

        SPI_CTL0(spi_dev->periph) &= (~SPI_CTL0_SPIEN);
        SPI_CTL1(spi_dev->periph) &= ~SPI_CTL1_TXSIZE;
        SPI_CTL1(spi_dev->periph) |= length;

        spi_dev->state      = HAL_SPI_STATE_BUSY_TX;
        spi_dev->error_code = HAL_SPI_ERROR_NONE;

        /* init field not used to zero */
        spi_dev->txbuffer.buffer = (uint8_t *)p_txbuffer;
        spi_dev->txbuffer.length = length;
        spi_dev->txbuffer.pos    = length;

        /* set the receiver information */
        spi_dev->rxbuffer.buffer = NULL;
        spi_dev->rxbuffer.length = 0U;
        spi_dev->rxbuffer.pos    = 0U;

        /* clear callback */
        spi_dev->tx_callback    = NULL;
        spi_dev->rx_callback    = NULL;
        spi_dev->error_callback = NULL;

        if(NULL != p_user_func) {
            spi_dev->tx_callback        = (void *)p_user_func->complete_func;
            spi_dev->suspend_callback   = (void *)p_user_func->suspend_func;
            spi_dev->error_callback     = (void *)p_user_func->error_func;
        } else {
            /* do nothing */
        }

        spi_dev->spi_irq.transmit_handler = _spi_transmit_complete_interrupt;
        spi_dev->spi_irq.suspend_handler  = _spi_suspend_interrupt;

        /* enable crc */
        if(SPI_CRC_ENABLE == (SPI_CFG0(spi_dev->periph) & SPI_CFG0_CRCEN)) {
            hals_spi_crc_off(spi_dev->periph);
            hals_spi_crc_on(spi_dev->periph);
        } else {
            /* do nothing */
        }

        /* enable spi */
        SPI_CTL0(spi_dev->periph) |= SPI_CTL0_SPIEN;

        /* Master transfer start */
        if(RESET != ((SPI_CFG1(spi_dev->periph) & SPI_CFG1_MSTMOD))) {
            SPI_CTL0(spi_dev->periph) |= SPI_CTL0_MSTART;
        } else {
            /* do nothing */
        }

        /* enable interrupt */
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_TXURE);
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_TXSERF);
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_CRCER);
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_TP);
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_ESTC);
    }

    /* unlock SPI */
    HAL_UNLOCK(spi_dev);

    return error_code;
}

/*!
    \brief      receive amounts of data by interrupt method, the function is non-blocking
    \param[in]  spi_dev: SPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  length: length of data to be sent
    \param[in]  p_user_func: pointer to call back function for user
    \param[out] p_rxbuffer: pointer to rxbuffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_receive_interrupt(hal_spi_dev_struct *spi_dev, uint8_t *p_rxbuffer, uint32_t length, \
                                  hal_spi_user_callback_struct *p_user_func)
{
    int32_t error_code = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi_dev) || (NULL == p_rxbuffer)) {
        HAL_DEBUGE("pointer [spi_dev] or [p_rxbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
    if((NULL == p_user_func)) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((SPI_MASTER == HAL_SPI_GET_DEVICE_MODE(spi_dev->periph))) {
        /* call transmit-receive function to send dummy data generate clock */
        error_code  = hal_spi_transmit_receive_interrupt(spi_dev, p_rxbuffer, p_rxbuffer, length, p_user_func);
    } else {
        /* lock SPI */
        HAL_LOCK(spi_dev);

        if(HAL_SPI_STATE_READY != spi_dev->state) {
            error_code = HAL_ERR_BUSY;
        } else {
            SPI_CTL1(spi_dev->periph) &= ~SPI_CTL1_TXSIZE;
            SPI_CTL1(spi_dev->periph) |= length;

            spi_dev->state      = HAL_SPI_STATE_BUSY_RX;
            spi_dev->error_code = HAL_SPI_ERROR_NONE;

            /* init field not used to zero */
            spi_dev->txbuffer.buffer = (uint8_t *)NULL;
            spi_dev->txbuffer.length = 0U;
            spi_dev->txbuffer.pos    = 0U;

            /* set the receiver information */
            spi_dev->rxbuffer.buffer = (uint8_t *)p_rxbuffer;
            spi_dev->rxbuffer.length = length;
            spi_dev->rxbuffer.pos    = length;

            /* clear callback */
            spi_dev->tx_callback    = NULL;
            spi_dev->rx_callback    = NULL;
            spi_dev->error_callback = NULL;

            /* set usercallback */
            if(NULL != p_user_func) {
                spi_dev->rx_callback        = (void *)p_user_func->complete_func;
                spi_dev->suspend_callback   = (void *)p_user_func->suspend_func;
                spi_dev->error_callback     = (void *)p_user_func->error_func;
            } else {
                /* do nothing */
            }

            spi_dev->spi_irq.receive_handler  = _spi_receive_complete_interrupt;
            spi_dev->spi_irq.suspend_handler  = _spi_suspend_interrupt;

            /* enable crc */
            if(SPI_CRC_ENABLE == (SPI_CFG0(spi_dev->periph) & SPI_CFG0_CRCEN)) {
                hals_spi_crc_off(spi_dev->periph);
                hals_spi_crc_on(spi_dev->periph);
            } else {
                /* do nothing */
            }

            /* enable spi */
            SPI_CTL0(spi_dev->periph) |= SPI_CTL0_SPIEN;

            /* enable interrupt */
            hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_TXURE);
            hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_RXORE);
            hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_CRCER);
            hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_RP);
        }
    }

    /* unlock SPI */
    HAL_UNLOCK(spi_dev);

    return error_code;
}

/*!
    \brief      transmit and receive amounts of data by interrupt method, the function is non-blocking
    \param[in]  spi_dev: SPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_txbuffer: pointer to txbuffer
    \param[in]  length: length of data to be sent
    \param[in]  p_user_func: pointer to call back function for user
    \param[out] p_rxbuffer: pointer to rxbuffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_BUSY, details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_transmit_receive_interrupt(hal_spi_dev_struct *spi_dev, uint8_t *p_txbuffer, uint8_t *p_rxbuffer, \
                                           uint32_t length, hal_spi_user_callback_struct *p_user_func)
{
    int32_t error_code = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi_dev) || (NULL == p_txbuffer) || (NULL == p_rxbuffer)) {
        HAL_DEBUGE("pointer [spi_dev] or [p_txbuffer] or [p_rxbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
    if((NULL == p_user_func)) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock SPI */
    HAL_LOCK(spi_dev);

    if(HAL_SPI_STATE_READY != spi_dev->state) {
        error_code = HAL_ERR_BUSY;
    } else {

        SPI_CTL0(spi_dev->periph) &= (~SPI_CTL0_SPIEN);

        SPI_CTL1(spi_dev->periph) &= ~SPI_CTL1_TXSIZE;
        SPI_CTL1(spi_dev->periph) |= length;

        spi_dev->state      = HAL_SPI_STATE_BUSY_TX_RX;
        spi_dev->error_code = HAL_SPI_ERROR_NONE;

        /* set the transmit information */
        spi_dev->txbuffer.buffer = (uint8_t *)p_txbuffer;
        spi_dev->txbuffer.length = length;
        spi_dev->txbuffer.pos    = length;

        /* set the receiver information */
        spi_dev->rxbuffer.buffer = (uint8_t *)p_rxbuffer;
        spi_dev->rxbuffer.length = length;
        spi_dev->rxbuffer.pos    = length;

        /* clear callback */
        spi_dev->tx_callback    = NULL;
        spi_dev->rx_callback    = NULL;
        spi_dev->error_callback = NULL;

        /* set user callback */
        if(NULL != p_user_func) {
            spi_dev->rx_callback = (void *)p_user_func->complete_func;
            spi_dev->error_callback = (void *)p_user_func->error_func;
        } else {
            /* do nothing */
        }

        spi_dev->spi_irq.receive_handler  = _spi_2lines_receive_interrupt;
        spi_dev->spi_irq.transmit_handler = _spi_2lines_transmit_interrupt;

        /* enable crc */
        if(SPI_CRC_ENABLE == (SPI_CFG0(spi_dev->periph) & SPI_CFG0_CRCEN)) {
            hals_spi_crc_off(spi_dev->periph);
            hals_spi_crc_on(spi_dev->periph);
        } else {
            /* do nothing */
        }

        /* enable spi */
        SPI_CTL0(spi_dev->periph) |= SPI_CTL0_SPIEN;

        /* Master transfer start */
        if(RESET != ((SPI_CFG1(spi_dev->periph) & SPI_CFG1_MSTMOD))) {
            SPI_CTL0(spi_dev->periph) |= SPI_CTL0_MSTART;
        }  else {
            /* do nothing */
        }

        /* enable interrupt */
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_RP);
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_CRCER);
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_TP);
    }

    /* unlock SPI */
    HAL_UNLOCK(spi_dev);

    return error_code;
}

/*!
    \brief      transmit amounts of data by DMA method, the function is non-blocking
    \param[in]  spi_dev: SPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_txbuffer: pointer to txbuffer
    \param[in]  length: length of data to be sent
    \param[in]  p_user_func: pointer to call back function for user
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_transmit_dma(hal_spi_dev_struct *spi_dev, uint8_t *p_txbuffer, uint16_t length, \
                             hal_spi_user_callback_struct *p_user_func)
{
    int32_t error_code = HAL_ERR_NONE;
    hal_dma_irq_struct dma_irq = {0};

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi_dev) || (NULL == p_txbuffer)) {
        HAL_DEBUGE("pointer [spi_dev] or [p_txbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
    if((NULL == p_user_func)) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock SPI */
    HAL_LOCK(spi_dev);

    if(spi_dev->state != HAL_SPI_STATE_READY) {
        error_code = HAL_ERR_BUSY;
    } else {
        /* set the transaction information */
        spi_dev->state           = HAL_SPI_STATE_BUSY_TX;
        spi_dev->txbuffer.buffer = (uint8_t *)p_txbuffer;
        spi_dev->txbuffer.length = length;
        spi_dev->txbuffer.pos    = length;

        /* Init field not used to zero */
        spi_dev->rxbuffer.buffer = (uint8_t *)NULL;
        spi_dev->rxbuffer.length = 0U;
        spi_dev->rxbuffer.pos    = 0U;

        /* clear callback */
        spi_dev->tx_callback            = NULL;
        spi_dev->tx_half_callback       = NULL;
        spi_dev->rx_callback            = NULL;
        spi_dev->rx_half_callback       = NULL;
        spi_dev->abort_callback         = NULL;
        spi_dev->error_callback         = NULL;

        /* set user call back */
        if(NULL != p_user_func) {
            spi_dev->tx_callback = (void *)p_user_func->complete_func;
            spi_dev->error_callback = (void *)p_user_func->error_func;
        } else {
            /* do nothing */
        }

        SPI_CTL0(spi_dev->periph) &= (~SPI_CTL0_SPIEN);

        /* transmit size */
        SPI_CTL1(spi_dev->periph) &= ~SPI_CTL1_TXSIZE;
        SPI_CTL1(spi_dev->periph) |= length;

        SPI_TDATA(spi_dev->periph) = 0U;

        /* according to DMA alignment Data size */
        if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) <= SPI_DATASIZE_8BIT) {
            if(DMA_PERIPH_WIDTH_32BIT == (DMA_CHCTL(spi_dev->p_dma_tx->dma_periph, spi_dev->p_dma_tx->channel) & \
                                        DMA_CHXCTL_PWIDTH)) {
                spi_dev->txbuffer.length = (spi_dev->txbuffer.length + 3U) >> 2U;
            } else {
                /* do nothing */
            }

            if(DMA_PERIPH_WIDTH_16BIT == (DMA_CHCTL(spi_dev->p_dma_tx->dma_periph, spi_dev->p_dma_tx->channel) & \
                                        DMA_CHXCTL_PWIDTH)) {
                spi_dev->txbuffer.length = (spi_dev->txbuffer.length + 1U) >> 1;
            } else {
                /* do nothing */
            }
        } else if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) <= SPI_DATASIZE_16BIT) {
            if(DMA_PERIPH_WIDTH_32BIT == (DMA_CHCTL(spi_dev->p_dma_tx->dma_periph, spi_dev->p_dma_tx->channel) & \
                                        DMA_CHXCTL_PWIDTH)) {
                spi_dev->txbuffer.length = (spi_dev->txbuffer.length + 1U) >> 1;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* enable crc */
        if(SPI_CRC_ENABLE == (SPI_CFG0(spi_dev->periph) & SPI_CFG0_CRCEN)) {
            hals_spi_crc_off(spi_dev->periph);
            hals_spi_crc_on(spi_dev->periph);
        } else {
            /* do nothing */
        }

        /* set dma transmit complete interrupt callback */
        dma_irq.half_finish_handle = (hal_irq_handle_cb)_spi_transmit_half_complete_dma;
        dma_irq.full_finish_handle = (hal_irq_handle_cb)_spi_transmit_complete_dma;
        dma_irq.error_handle       = (hal_irq_handle_cb)_spi_dma_error;

        hal_dma_start_interrupt(spi_dev->p_dma_tx, (uint32_t)spi_dev->txbuffer.buffer, \
                                (uint32_t)&SPI_TDATA(spi_dev->periph), (uint16_t)spi_dev->txbuffer.length, &dma_irq);

        /* enable spi */
        SPI_CTL0(spi_dev->periph) |= SPI_CTL0_SPIEN;

        /* enable SPI DMA transmit */
        hals_spi_dma_enable(spi_dev->periph, (uint16_t)SPI_DMA_TRANSMIT);

        /* transfer start */
        if(RESET != ((SPI_CFG1(spi_dev->periph) & SPI_CFG1_MSTMOD))) {
            hals_spi_master_transfer_start(spi_dev->periph, SPI_TRANS_START);
        } else {
            /* do nothing */
        }
    }

    /* unlock SPI */
    HAL_UNLOCK(spi_dev);

    return error_code;
}

/*!
    \brief      receive amounts of data by DMA method, the function is non-blocking
    \param[in]  spi_dev: SPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  length: length of data to be sent
    \param[in]  p_user_func: pointer to call back function for user
    \param[out] p_rxbuffer: pointer to rxbuffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_receive_dma(hal_spi_dev_struct *spi_dev, uint8_t *p_rxbuffer, uint16_t length, \
                            hal_spi_user_callback_struct *p_user_func)
{
    int32_t error_code = HAL_ERR_NONE;
    hal_dma_irq_struct dma_irq = {0};

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi_dev) || (NULL == p_rxbuffer)) {
        HAL_DEBUGE("pointer [spi_dev] or [p_txbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
    if((NULL == p_user_func)) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((SPI_MASTER == HAL_SPI_GET_DEVICE_MODE(spi_dev->periph))) {
        /* call transmit-receive function to send dummy data generate clock on CLK line */
        error_code = hal_spi_transmit_receive_dma(spi_dev, p_rxbuffer, p_rxbuffer, length, p_user_func);
    } else {
        /* lock SPI */
        HAL_LOCK(spi_dev);

        if(spi_dev->state != HAL_SPI_STATE_READY) {
            error_code = HAL_ERR_BUSY;
        } else {
            /* set the transaction information */
            spi_dev->state           = HAL_SPI_STATE_BUSY_RX;
            spi_dev->txbuffer.buffer = (uint8_t *)NULL;
            spi_dev->txbuffer.length = 0U;
            spi_dev->txbuffer.pos    = 0U;

            /* Init field not used to zero */
            spi_dev->rxbuffer.buffer = (uint8_t *)p_rxbuffer;
            spi_dev->rxbuffer.length = length;
            spi_dev->rxbuffer.pos    = length;

            /* clear callback */
            spi_dev->tx_callback            = NULL;
            spi_dev->tx_half_callback       = NULL;
            spi_dev->rx_callback            = NULL;
            spi_dev->rx_half_callback       = NULL;
            spi_dev->abort_callback         = NULL;
            spi_dev->error_callback         = NULL;

            if(NULL != p_user_func) {
                spi_dev->rx_callback = (void *)p_user_func->complete_func;
                spi_dev->error_callback = (void *)p_user_func->error_func;
            } else {
                /* do nothing */
            }

            /* transmit size */
            SPI_CTL1(spi_dev->periph) &= ~SPI_CTL1_TXSIZE;
            SPI_CTL1(spi_dev->periph) |= length;

            /* according to DMA alignment Data size */
            if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) <= SPI_DATASIZE_8BIT) {
                if(DMA_PERIPH_WIDTH_32BIT == (DMA_CHCTL(spi_dev->p_dma_tx->dma_periph, spi_dev->p_dma_tx->channel) & \
                                            DMA_CHXCTL_PWIDTH)) {
                    spi_dev->rxbuffer.length = (spi_dev->rxbuffer.length + 3U) >> 2U;
                } else {
                    /* do nothing */
                }

                if(DMA_PERIPH_WIDTH_16BIT == (DMA_CHCTL(spi_dev->p_dma_tx->dma_periph, spi_dev->p_dma_tx->channel) & \
                                            DMA_CHXCTL_PWIDTH)) {
                    spi_dev->rxbuffer.length = (spi_dev->rxbuffer.length + 1U) >> 1U;
                } else {
                    /* do nothing */
                }
            } else if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) <= SPI_DATASIZE_16BIT) {
                if(DMA_PERIPH_WIDTH_32BIT == (DMA_CHCTL(spi_dev->p_dma_tx->dma_periph, spi_dev->p_dma_tx->channel) & \
                                            DMA_CHXCTL_PWIDTH)) {
                    spi_dev->rxbuffer.length = (spi_dev->rxbuffer.length + 1U) >> 1U;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            /* enable crc */
            if(SPI_CRC_ENABLE == (SPI_CFG0(spi_dev->periph) & SPI_CFG0_CRCEN)) {
                hals_spi_crc_off(spi_dev->periph);
                hals_spi_crc_on(spi_dev->periph);
            } else {
                /* do nothing */
            }

            /* set dma receive complete interrupt callback */
            dma_irq.half_finish_handle = (hal_irq_handle_cb)_spi_receive_half_complete_dma;
            dma_irq.full_finish_handle = (hal_irq_handle_cb)_spi_receive_complete_dma;
            dma_irq.error_handle       = (hal_irq_handle_cb)_spi_dma_error;

            /* configure dma transmit */
            hal_dma_start_interrupt(spi_dev->p_dma_rx, (uint32_t)&SPI_RDATA(spi_dev->periph), \
                                    (uint32_t)spi_dev->rxbuffer.buffer, (uint16_t)spi_dev->rxbuffer.length, &dma_irq);

            /* enable spi */
            SPI_CTL0(spi_dev->periph) |= SPI_CTL0_SPIEN;

            /* enable SPI DMA transmit */
            hals_spi_dma_enable(spi_dev->periph, (uint16_t)SPI_DMA_RECEIVE);
        }
    }
    /* unlock SPI */
    HAL_UNLOCK(spi_dev);

    return error_code;
}

/*!
    \brief      transmit and receive amounts of data by DMA method, the function is non-blocking
    \param[in]  spi_dev: SPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_txbuffer: pointer to txbuffer
    \param[in]  length: length of data to be sent
    \param[in]  p_user_func: pointer to call back function for user
    \param[out] p_rxbuffer: pointer to rxbuffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_transmit_receive_dma(hal_spi_dev_struct *spi_dev, uint8_t *p_txbuffer, uint8_t *p_rxbuffer, \
                                     uint16_t length, hal_spi_user_callback_struct *p_user_func)
{
    int32_t error_code = HAL_ERR_NONE;
    hal_dma_irq_struct dma_irq = {0};

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi_dev) || (NULL == p_txbuffer) || (NULL == p_rxbuffer)) {
        HAL_DEBUGE("pointer [spi_dev] or [p_txbuffer] or [p_rxbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
    if((NULL == p_user_func)) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock SPI */
    HAL_LOCK(spi_dev);

    if(spi_dev->state != HAL_SPI_STATE_READY) {
        error_code = HAL_ERR_BUSY;
    } else {
        /* set the transaction information */
        spi_dev->state           = HAL_SPI_STATE_BUSY_TX_RX;
        spi_dev->txbuffer.buffer = (uint8_t *)p_txbuffer;
        spi_dev->txbuffer.length = length;
        spi_dev->txbuffer.pos    = length;

        /* Init field not used to zero */
        spi_dev->rxbuffer.buffer = (uint8_t *)p_rxbuffer;
        spi_dev->rxbuffer.length = length;
        spi_dev->rxbuffer.pos    = length;

        /* clear callback */
        spi_dev->tx_callback            = NULL;
        spi_dev->tx_half_callback       = NULL;
        spi_dev->rx_callback            = NULL;
        spi_dev->rx_half_callback       = NULL;
        spi_dev->abort_callback         = NULL;
        spi_dev->error_callback         = NULL;

        /* set user callback */
        if(NULL != p_user_func) {
            spi_dev->rx_callback      = (void *)p_user_func->complete_func;
            spi_dev->rx_half_callback = (void *)p_user_func->half_complete_func;
            spi_dev->error_callback   = (void *)p_user_func->error_func;
        } else {
            /* do nothing */
        }

        SPI_CTL0(spi_dev->periph) &= (~SPI_CTL0_SPIEN);

        /* transmit size */
        SPI_CTL1(spi_dev->periph) &= ~SPI_CTL1_TXSIZE;
        SPI_CTL1(spi_dev->periph) |= length;

        /* clear data */
        SPI_TDATA(spi_dev->periph) = 0U;

        /* according to DMA alignment Data size */
        if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) <= SPI_DATASIZE_8BIT) {
            if(DMA_PERIPH_WIDTH_32BIT == (DMA_CHCTL(spi_dev->p_dma_tx->dma_periph, spi_dev->p_dma_tx->channel) & \
                                        DMA_CHXCTL_PWIDTH)) {
                spi_dev->rxbuffer.length = (spi_dev->rxbuffer.length + 3U) >> 2U;
                spi_dev->txbuffer.length = (spi_dev->txbuffer.length + 3U) >> 2U;
            } else {
                /* do nothing */
            }

            if(DMA_PERIPH_WIDTH_16BIT == (DMA_CHCTL(spi_dev->p_dma_tx->dma_periph, spi_dev->p_dma_tx->channel) & \
                                        DMA_CHXCTL_PWIDTH)) {
                spi_dev->rxbuffer.length = (spi_dev->rxbuffer.length + 1U) >> 1;
                spi_dev->txbuffer.length = (spi_dev->txbuffer.length + 1U) >> 1;
            } else {
                /* do nothing */
            }
        } else if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) <= SPI_DATASIZE_16BIT) {
            if(DMA_PERIPH_WIDTH_32BIT == (DMA_CHCTL(spi_dev->p_dma_tx->dma_periph, spi_dev->p_dma_tx->channel) & \
                                        DMA_CHXCTL_PWIDTH)) {
                spi_dev->rxbuffer.length = (spi_dev->rxbuffer.length + 1U) >> 1;
                spi_dev->txbuffer.length = (spi_dev->txbuffer.length + 1U) >> 1;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* enable crc */
        if(SPI_CRC_ENABLE == (SPI_CFG0(spi_dev->periph) & SPI_CFG0_CRCEN)) {
            hals_spi_crc_off(spi_dev->periph);
            hals_spi_crc_on(spi_dev->periph);
        } else {
            /* do nothing */
        }

        /* enable spi */
        SPI_CTL0(spi_dev->periph) |= SPI_CTL0_SPIEN;

        /* set dma receive complete interrupt callback */
        dma_irq.half_finish_handle = (hal_irq_handle_cb)_spi_receive_half_complete_dma;
        dma_irq.full_finish_handle = (hal_irq_handle_cb)_spi_receive_complete_dma;
        dma_irq.error_handle       = (hal_irq_handle_cb)_spi_dma_error;

        hal_dma_start_interrupt(spi_dev->p_dma_rx, (uint32_t)&SPI_RDATA(spi_dev->periph), \
                                (uint32_t)spi_dev->rxbuffer.buffer, (uint16_t)spi_dev->rxbuffer.length, &dma_irq);

        /* enable SPI DMA receive */
        hals_spi_dma_enable(spi_dev->periph, (uint16_t)SPI_DMA_RECEIVE);

        /* set dma receive complete interrupt callback */
        dma_irq.half_finish_handle = (hal_irq_handle_cb)_spi_transmit_half_complete_dma;
        dma_irq.full_finish_handle = (hal_irq_handle_cb)_spi_transmit_complete_dma;
        dma_irq.error_handle       = (hal_irq_handle_cb)_spi_dma_error;

        hal_dma_start_interrupt(spi_dev->p_dma_tx, (uint32_t)spi_dev->txbuffer.buffer, \
                                (uint32_t)&SPI_TDATA(spi_dev->periph), (uint16_t)spi_dev->txbuffer.length, &dma_irq);

        /* enable SPI DMA transmit */
        hals_spi_dma_enable(spi_dev->periph, (uint16_t)SPI_DMA_TRANSMIT);

        /* Master transfer start */
        if(RESET != ((SPI_CFG1(spi_dev->periph) & SPI_CFG1_MSTMOD))) {
            SPI_CTL0(spi_dev->periph) |= SPI_CTL0_MSTART;
        } else {
            /* do nothing */
        }
    }
    /* unlock SPI */
    HAL_UNLOCK(spi_dev);

    return error_code;
}

/*!
   \brief      transmit an additional amount of data in blocking mode
   \param[in]  spi_dev : pointer to a SPI_HandleTypeDef structure that contains
                  the configuration information for SPI module.
   \param[in]  p_txbuffer: pointer to data buffer
   \param[in]  length : amount of data to be sent
   \param[in]  p_user_func: pointer to call back function for user
   \param[out] none
   \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NO_SUPPORT,details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_reload_transmit_interrupt(hal_spi_dev_struct *spi_dev, uint8_t *p_txbuffer, uint16_t length, \
                                          hal_spi_user_callback_struct *p_user_func)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t reg1      = 0U;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi_dev) || (NULL == p_txbuffer)) {
        HAL_DEBUGE("pointer [spi_dev] or [p_txbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
    if((NULL == p_user_func)) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock SPI */
    HAL_LOCK(spi_dev);

    if(HAL_SPI_STATE_BUSY_TX == spi_dev->state) {
        SPI_CTL1(spi_dev->periph) &= ~SPI_CTL1_TXSIZE;
        SPI_CTL1(spi_dev->periph) |= (uint32_t)((uint16_t)(length << 16U));

        /* init field not used to zero */
        spi_dev->txbuffer.buffer = (uint8_t *)p_txbuffer;
        spi_dev->txbuffer.length = length;
        spi_dev->txbuffer.pos    = length;

        /* set the receiver information */
        spi_dev->rxbuffer.buffer = (uint8_t *)NULL;
        spi_dev->rxbuffer.length = 0U;
        spi_dev->rxbuffer.pos    = 0U;
        spi_dev->rx_callback     = NULL;
        if(NULL != p_user_func) {
            spi_dev->tx_callback    = (void *)p_user_func->complete_func;
            spi_dev->error_callback = (void *)p_user_func->error_func;
        } else {
            /* do nothing */
        }

        /* Check if the current transmit is already completed */
        reg1 = SPI_CTL1(spi_dev->periph);
        if((0U != (reg1 >> 16)) && (HAL_SPI_STATE_READY == spi_dev->state)) {
            hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_TXSERF);
            SPI_CTL1(spi_dev->periph) &= ~SPI_CTL1_TXSER;
            error_code = HAL_ERR_NO_SUPPORT;
        } else {
            /* do nothing */
        }
    } else {
        error_code = HAL_ERR_NO_SUPPORT;
    }

    /* unlock SPI */
    HAL_UNLOCK(spi_dev);

    return error_code;
}

/*!
   \brief      receive an additional amount of data in blocking mode
   \param[in]  spi_dev : pointer to a SPI_HandleTypeDef structure that contains
                  the configuration information for SPI module.
   \param[in]  length : amount of data to be sent
   \param[in]  p_user_func : pointer to call back function for user
   \param[out] p_rxbuffer: pointer to rxbuffer
   \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_reload_received_interrupt(hal_spi_dev_struct *spi_dev, uint8_t *p_rxbuffer, uint16_t length, \
                                          hal_spi_user_callback_struct *p_user_func)
{
    int32_t error_code  = HAL_ERR_NONE;
    uint32_t tmp_txdata = 0U;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi_dev) || (NULL == p_rxbuffer)) {
        HAL_DEBUGE("pointer [spi_dev] or [p_rxbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
    if((NULL == p_user_func)) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock SPI */
    HAL_LOCK(spi_dev);

    if(HAL_SPI_STATE_BUSY_RX == spi_dev->state) {
        SPI_CTL1(spi_dev->periph) &= ~SPI_CTL1_TXSIZE;
        SPI_CTL1(spi_dev->periph) |= ((uint32_t)length << 16U);

        /* init field not used to zero */
        spi_dev->txbuffer.buffer = (uint8_t *)NULL;
        spi_dev->txbuffer.length = 0U;
        spi_dev->txbuffer.pos    = 0U;

        /* set the receiver information */
        spi_dev->rxbuffer.buffer = (uint8_t *)p_rxbuffer;
        spi_dev->rxbuffer.length = length;
        spi_dev->rxbuffer.pos    = length;

        spi_dev->tx_callback    = NULL;
        if(NULL != p_user_func) {
            spi_dev->rx_callback    = (void *)p_user_func->complete_func;
            spi_dev->error_callback = (void *)p_user_func->error_func;
        } else {
            /* do nothing */
        }

        /* Check if the current transmit is already completed */
        tmp_txdata = SPI_CTL1(spi_dev->periph);
        if((0U != (tmp_txdata >> 16)) && (HAL_SPI_STATE_READY == spi_dev->state)) {
            hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_TXSERF);
            SPI_CTL1(spi_dev->periph) &= ~SPI_CTL1_TXSER;
            spi_dev->state = HAL_SPI_STATE_READY;
            error_code = HAL_ERR_NO_SUPPORT;
        } else {
            /* do nothing */
        }
    } else {
        error_code = HAL_ERR_NO_SUPPORT;
    }

    /* unlock SPI */
    HAL_UNLOCK(spi_dev);

    return error_code;
}

/*!
   \brief      transmit and receive an additional amount of data in blocking mode
   \param[in]  spi_dev : pointer to a SPI_HandleTypeDef structure that contains
                  the configuration information for SPI module.
   \param[in]  p_txbuffer: pointer to transmission data buffer
   \param[in]  length    : amount of data to be sent and received
   \param[in]  p_user_func : pointer to call back function for user
   \param[out] p_rxbuffer: pointer to reception data buffer
   \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_reload_transmit_received_interrupt(hal_spi_dev_struct *spi_dev, uint8_t *p_txbuffer, \
                                                   uint8_t *p_rxbuffer, uint16_t length, \
                                                   hal_spi_user_callback_struct *p_user_func)
{
    int32_t error_code  = HAL_ERR_NONE;
    uint32_t tmp_txdata = 0U;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi_dev) || (NULL == p_rxbuffer) || (NULL == p_txbuffer)) {
        HAL_DEBUGE("pointer [spi_dev] or [p_rxbuffer] or [p_txbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
    if((NULL == p_user_func)) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock SPI */
    HAL_LOCK(spi_dev);

    if(HAL_SPI_STATE_BUSY_TX_RX == spi_dev->state) {
        SPI_CTL1(spi_dev->periph) &= ~SPI_CTL1_TXSIZE;
        SPI_CTL1(spi_dev->periph) |= (uint16_t)(length << 16U);

        /* init field not used to zero */
        spi_dev->txbuffer.buffer = (uint8_t *)p_txbuffer;
        spi_dev->txbuffer.length = length;
        spi_dev->txbuffer.pos    = length;

        /* set the receiver information */
        spi_dev->rxbuffer.buffer = (uint8_t *)p_rxbuffer;
        spi_dev->rxbuffer.length = length;
        spi_dev->rxbuffer.pos    = length;

        spi_dev->tx_callback = NULL;
        spi_dev->rx_callback = NULL;
        if(NULL != p_user_func) {
            spi_dev->rx_callback = (void *)p_user_func->complete_func;
            spi_dev->error_callback = (void *)p_user_func->error_func;
        } else {
            /* do nothing */
        }

        /* Check if the current transmit is already completed */
        tmp_txdata = SPI_CTL1(spi_dev->periph);

        if((0U != (tmp_txdata >> 16)) && (HAL_SPI_STATE_READY == spi_dev->state)) {
            hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_TXSERF);
            SPI_CTL1(spi_dev->periph) &= ~SPI_CTL1_TXSER;
            error_code = HAL_ERR_NO_SUPPORT;
        } else {
            /* do nothing */
        }
    } else {
        error_code = HAL_ERR_NO_SUPPORT;
    }

    /* unlock SPI */
    HAL_UNLOCK(spi_dev);

    return error_code;
}

/*!
    \brief      SPI interrupt handler content function, which is merely used in spi_handler
    \param[in]  spi_dev: SPI device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_irq(hal_spi_dev_struct *spi_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == spi_dev) {
        HAL_DEBUGE("pointer [spi_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((RESET == (SPI_STAT(spi_dev->periph) & SPI_STAT_TXURERR))
       && (RESET != hals_spi_interrupt_flag_get(spi_dev->periph, SPI_I2S_INT_FLAG_TP))) {
        if(NULL != spi_dev->spi_irq.transmit_handler) {
            spi_dev->spi_irq.transmit_handler(spi_dev);
        } else {
            /* do nothing */
        }
    }

    if((RESET == (SPI_STAT(spi_dev->periph) & SPI_STAT_RXORERR))
       && (RESET != hals_spi_interrupt_flag_get(spi_dev->periph, SPI_I2S_INT_FLAG_RP))) {
        if(NULL != spi_dev->spi_irq.receive_handler) {
            spi_dev->spi_irq.receive_handler(spi_dev);
        } else {
            /* do nothing */
        }
    }

    if((RESET != hals_spi_interrupt_flag_get(spi_dev->periph, SPI_I2S_INT_FLAG_ET))
       && (RESET != hals_spi_interrupt_flag_get(spi_dev->periph, SPI_I2S_INT_FLAG_SPD))) {
        if(NULL != spi_dev->spi_irq.suspend_handler) {
            spi_dev->spi_irq.suspend_handler(spi_dev);
        } else {
            /* do nothing */
        }
    }

    if(RESET != (SPI_STAT(spi_dev->periph) & (SPI_STAT_CONFERR | SPI_STAT_RXORERR | SPI_STAT_FERR | \
                 SPI_STAT_CRCERR | SPI_STAT_TXURERR))) {
        /* SPI configuration error interrupt occurred */
        if(hals_spi_interrupt_flag_get(spi_dev->periph, SPI_I2S_INT_FLAG_CONFERR) != RESET) {
            hals_spi_interrupt_flag_clear(spi_dev->periph, SPI_I2S_INT_FLAG_CONFERR);
        } else {
            /* do nothing */
        }

        /* SPI receiver overrun error interrupt occurred */
        if(hals_spi_interrupt_flag_get(spi_dev->periph, SPI_I2S_INT_FLAG_RXORERR) != RESET) {
            hals_spi_interrupt_flag_clear(spi_dev->periph, SPI_I2S_INT_FLAG_RXORERR);
        } else {
            /* do nothing */
        }

        /* SPI frame error interrupt occurred */
        if(hals_spi_interrupt_flag_get(spi_dev->periph, SPI_I2S_INT_FLAG_FERR) != RESET) {
            hals_spi_interrupt_flag_clear(spi_dev->periph, SPI_I2S_INT_FLAG_FERR);
        } else {
            /* do nothing */
        }

        /* SPI CRC error interrupt occurred */
        if(hals_spi_interrupt_flag_get(spi_dev->periph, SPI_I2S_INT_FLAG_CRCERR) != RESET) {
            hals_spi_interrupt_flag_clear(spi_dev->periph, SPI_I2S_INT_FLAG_CRCERR);
        } else {
            /* do nothing */
        }

        /* SPI transfer underrun interrupt occurred */
        if(hals_spi_interrupt_flag_get(spi_dev->periph, SPI_I2S_INT_FLAG_TXURERR) != RESET) {
            hals_spi_interrupt_flag_clear(spi_dev->periph, SPI_I2S_INT_FLAG_TXURERR);
        } else {
            /* do nothing */
        }

        /* SPI transfer underrun interrupt occurred */
        if(hals_spi_interrupt_flag_get(spi_dev->periph, SPI_I2S_INT_FLAG_TXF) != RESET) {
            hals_spi_interrupt_flag_clear(spi_dev->periph, SPI_I2S_INT_FLAG_TXF);
        } else {
            /* do nothing */
        }

        if(NULL != spi_dev->spi_irq.error_handler) {
            spi_dev->spi_irq.error_handler(spi_dev);
        } else {
            /* do nothing */
        }
    }

    return error_code;
}

/*!
    \brief      set user-defined interrupt callback function
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  spi_dev: SPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to SPI interrupt callback function pointer structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg        NULL: the corresponding callback mechanism is out of use, and
                     disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_irq_handle_set(hal_spi_dev_struct *spi_dev, hal_spi_irq_struct *p_irq)
{
    int32_t error_code = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [spi_dev] or [p_irq] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set or reset error flag */
    if(NULL != p_irq->error_handler) {
        spi_dev->spi_irq.error_handler = p_irq->error_handler;
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_TXURE);
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_TXURE);
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_RXORE);
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_FE);
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_CONFE);
    } else {
        spi_dev->spi_irq.error_handler = NULL;
        hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_TXURE);
        hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_TXURE);
        hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_RXORE);
        hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_FE);
        hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_CONFE);
    }

    /* set or reset receive complete flag */
    if(NULL != p_irq->receive_handler) {
        spi_dev->spi_irq.receive_handler = p_irq->receive_handler;
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_RP);
    } else {
        spi_dev->spi_irq.receive_handler = NULL;
        hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_RP);
    }

    /* set or reset transmit complete flag */
    if(NULL != p_irq->transmit_handler) {
        spi_dev->spi_irq.transmit_handler = p_irq->transmit_handler;
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_TP);
    } else {
        spi_dev->spi_irq.transmit_handler = NULL;
        hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_TP);
    }

    /* set or reset suspend flag */
    if(NULL != p_irq->suspend_handler) {
        spi_dev->spi_irq.suspend_handler = p_irq->suspend_handler;
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_ESTC);
    } else {
        spi_dev->spi_irq.suspend_handler = NULL;
        hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_ESTC);
    }

    /* set or reset complete flag */
    if(NULL != p_irq->transmit_receive_handler) {
        spi_dev->spi_irq.transmit_receive_handler = p_irq->transmit_receive_handler;
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_TP);
        hals_spi_interrupt_enable(spi_dev->periph, SPI_I2S_INT_RP);
    } else {
        spi_dev->spi_irq.transmit_receive_handler = NULL;
        hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_TP);
        hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_RP);
    }

    return error_code;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  spi_dev: SPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_spi_irq_handle_all_reset(hal_spi_dev_struct *spi_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi_dev)) {
        HAL_DEBUGE("pointer [spi_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    spi_dev->spi_irq.error_handler            = NULL;
    spi_dev->spi_irq.transmit_handler         = NULL;
    spi_dev->spi_irq.receive_handler          = NULL;
    spi_dev->spi_irq.transmit_receive_handler = NULL;

    /* disable interrupt */
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_TP);
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_RP);
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_TXF);
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_TXURE);
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_TXURE);
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_RXORE);
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_FE);
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_CONFE);

    return error_code;
}

/*!
    \brief      configure slave transmit underrun detected
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[in]  ur_cfg: underrun config
                only one parameter can be selected which is shown as below:
      \arg        SPI_DETECT_BEGIN_DATA_FRAME: underrun detected at start of data frame (no bit 1 protection)
      \arg        SPI_DETECT_END_DATA_FRAME: underrun detected at end of last data frame
      \arg        SPI_DETECT_BEGIN_ACTIVE_NSS: underrun detected at start of NSS signal
    \param[in]  ur_ope: underrun operation
                only one parameter can be selected which is shown as below:
      \arg        SPI_CONFIG_REGISTER_PATTERN: slave send a constant value defined by the SPI_URDATA register
      \arg        SPI_CONFIG_LAST_RECEIVED: slave send the lastly data frame received from master
      \arg        SPI_CONFIG_LAST_TRANSMITTED: slave send its lastly transmitted data frame
    \param[out] none
    \retval     none
*/
void hal_spi_underrun_config(uint32_t spi_periph, uint32_t ur_cfg, uint32_t ur_ope)
{
    SPI_CFG0(spi_periph) |= ur_cfg;
    SPI_CFG0(spi_periph) |= ur_ope;
}

/*!
    \brief      configure af gpio lock
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[in]  af_gpio: af gpio config
                only one parameter can be selected which is shown as below:
      \arg        SPI_MASTER_KEEPIO_ENABLE: spi af gpio lock enable
      \arg        SPI_MASTER_KEEPIO_DISABLE: spi af gpio lock disable
    \param[out] none
    \retval     none
*/
void hal_spi_af_lock_config(uint32_t spi_periph, uint32_t af_gpio)
{
    SPI_CFG1(spi_periph) |= af_gpio;
}

/*!
    \brief      return the spi state
    \param[in]  spi_dev: SPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_spi_run_state_enum
*/
hal_spi_run_state_enum hal_spi_state_get(hal_spi_dev_struct *spi_dev)
{
    return (hal_spi_run_state_enum)spi_dev->state;
}

/*!
    \brief      return the spi error code
    \param[in]  spi_dev: SPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t: 0 - 0xFFFFFFFFU
*/
uint32_t hal_spi_error_code_get(hal_spi_dev_struct *spi_dev)
{
    return spi_dev->error_code;
}

/*!
    \brief      enable SPI
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[out] none
    \retval     none
*/
void hals_spi_enable(uint32_t spi_periph)
{
    SPI_CTL0(spi_periph) |= SPI_CTL0_SPIEN;
}

/*!
    \brief      disable SPI
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[out] none
    \retval     none
*/
void hals_spi_disable(uint32_t spi_periph)
{
    SPI_CTL0(spi_periph) &= (~SPI_CTL0_SPIEN);
}

/*!
    \brief      enable SPI NSS output
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[out] none
    \retval     none
*/
void hals_spi_nss_output_enable(uint32_t spi_periph)
{
    SPI_CFG1(spi_periph) |= SPI_CFG1_NSSDRV;
}

/*!
    \brief      enable SPI DMA
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[in]  spi_dma: SPI DMA mode
                only one parameter can be selected which is shown as below:
      \arg        SPI_DMA_TRANSMIT: SPI transmit data use DMA
      \arg        SPI_DMA_RECEIVE: SPI receive data use DMA
    \param[out] none
    \retval     none
*/
void hals_spi_dma_enable(uint32_t spi_periph, uint16_t spi_dma)
{
    if(SPI_DMA_TRANSMIT == spi_dma) {
        SPI_CFG0(spi_periph) |= SPI_CFG0_DMATEN;
    } else {
        SPI_CFG0(spi_periph) |= SPI_CFG0_DMAREN;
    }
}

/*!
    \brief      disable SPI DMA
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[in]  spi_dma: SPI DMA mode
                only one parameter can be selected which is shown as below:
      \arg        SPI_DMA_TRANSMIT: SPI transmit data use DMA
      \arg        SPI_DMA_RECEIVE: SPI receive data use DMA
    \param[out] none
    \retval     none
*/
void hals_spi_dma_disable(uint32_t spi_periph, uint16_t spi_dma)
{
    if(SPI_DMA_TRANSMIT == spi_dma) {
        SPI_CFG0(spi_periph) &= (~SPI_CFG0_DMATEN);
    } else {
        SPI_CFG0(spi_periph) &= (~SPI_CFG0_DMAREN);
    }
}

/*!
    \brief      turn on SPI CRC function
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[out] none
    \retval     none
*/
void hals_spi_crc_on(uint32_t spi_periph)
{
    SPI_CFG0(spi_periph) |= SPI_CFG0_CRCEN;
}

/*!
    \brief      turn off SPI CRC function
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[out] none
    \retval     none
*/
void hals_spi_crc_off(uint32_t spi_periph)
{
    SPI_CFG0(spi_periph) &= (~SPI_CFG0_CRCEN);
}

/*!
    \brief      enable SPI interrupt
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[in]  interrupt: SPI interrupt
                only one parameter can be selected which is shown as below:
      \arg        SPI_I2S_INT_RP: RP interrupt
      \arg        SPI_I2S_INT_TP: TP interrupt
      \arg        SPI_I2S_INT_DP: DP interrupt
      \arg        SPI_I2S_INT_ESTC: end of transfer or suspend or TXFIFO clear interrupt
      \arg        SPI_I2S_INT_TXF: transmission filled interrupt
      \arg        SPI_I2S_INT_TXURE: underrun error interrupt
      \arg        SPI_I2S_INT_RXORE: overrun error interrupt
      \arg        SPI_I2S_INT_CRCER: CRC error interrupt
      \arg        SPI_I2S_INT_FE: TI frame error interrupt
      \arg        SPI_I2S_INT_CONFE: mode error interrupt
      \arg        SPI_I2S_INT_TXSERF: TXSER reload interrupt
    \param[out] none
    \retval     none
*/
void hals_spi_interrupt_enable(uint32_t spi_periph, uint8_t interrupt)
{
    switch(interrupt) {
    /* SPI/I2S RP interrupt */
    case SPI_I2S_INT_RP:
        SPI_INT(spi_periph) |= SPI_INT_RPIE;
        break;
    /* SPI/I2S TP interrupt */
    case SPI_I2S_INT_TP:
        SPI_INT(spi_periph) |= SPI_INT_TPIE;
        break;
    /* SPI/I2S DP interrupt */
    case SPI_I2S_INT_DP:
        SPI_INT(spi_periph) |= SPI_INT_DPIE;
        break;
    /* SPI/I2S end of transfer or suspend or TXFIFO clear interrupt */
    case SPI_I2S_INT_ESTC:
        SPI_INT(spi_periph) |= SPI_INT_ESTCIE;
        break;
    /* SPI/I2S transmission filled interrupt */
    case SPI_I2S_INT_TXF:
        SPI_INT(spi_periph) |= SPI_INT_TXFIE;
        break;
    /* SPI/I2S underrun error interrupt */
    case SPI_I2S_INT_TXURE:
        SPI_INT(spi_periph) |= SPI_INT_TXUREIE;
        break;
    /* SPI/I2S overrun error interrupt*/
    case SPI_I2S_INT_RXORE:
        SPI_INT(spi_periph) |= SPI_INT_RXOREIE;
        break;
    /* SPI/I2S CRC error interrupt */
    case SPI_I2S_INT_CRCER:
        SPI_INT(spi_periph) |= SPI_INT_CRCERIE;
        break;
    /* SPI TI frame error interrupt */
    case SPI_I2S_INT_FE:
        SPI_INT(spi_periph) |= SPI_INT_FEIE;
        break;
    /* SPI/I2S mode error interrupt */
    case SPI_I2S_INT_CONFE:
        SPI_INT(spi_periph) |= SPI_INT_CONFEIE;
        break;
    /* SPI/I2S TXSER reload interrupt */
    case SPI_I2S_INT_TXSERF:
        SPI_INT(spi_periph) |= SPI_INT_TXSERFIE;
        break;
    default:
        break;
    }
}

/*!
    \brief      disable SPI interrupt
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[in]  interrupt: SPI interrupt
                only one parameter can be selected which is shown as below:
      \arg        SPI_I2S_INT_RP: RP interrupt
      \arg        SPI_I2S_INT_TP: TP interrupt
      \arg        SPI_I2S_INT_DP: DP interrupt
      \arg        SPI_I2S_INT_ESTC: end of transfer or suspend or TXFIFO clear interrupt
      \arg        SPI_I2S_INT_TXF: transmission filled interrupt
      \arg        SPI_I2S_INT_TXURE: underrun error interrupt
      \arg        SPI_I2S_INT_RXORE: overrun error interrupt
      \arg        SPI_I2S_INT_CRCER: CRC error interrupt
      \arg        SPI_I2S_INT_FE: TI frame error interrupt
      \arg        SPI_I2S_INT_CONFE: mode error interrupt
      \arg        SPI_I2S_INT_TXSERF: TXSER reload interrupt
      \arg        SPI_I2S_INT_ALL: all interrupt
    \param[out] none
    \retval     none
*/
void hals_spi_interrupt_disable(uint32_t spi_periph, uint8_t interrupt)
{
    switch(interrupt) {
    /* SPI/I2S RP interrupt */
    case SPI_I2S_INT_RP:
        SPI_INT(spi_periph) &= (~SPI_INT_RPIE);
        break;
    /* SPI/I2S TP interrupt */
    case SPI_I2S_INT_TP:
        SPI_INT(spi_periph) &= (~SPI_INT_TPIE);
        break;
    /* SPI/I2S DP interrupt */
    case SPI_I2S_INT_DP:
        SPI_INT(spi_periph) &= (~SPI_INT_DPIE);
        break;
    /* SPI/I2S end of transfer or suspend or TXFIFO clear interrupt */
    case SPI_I2S_INT_ESTC:
        SPI_INT(spi_periph) &= (~SPI_INT_ESTCIE);
        break;
    /* SPI/I2S transmission filled interrupt */
    case SPI_I2S_INT_TXF:
        SPI_INT(spi_periph) &= (~SPI_INT_TXFIE);
        break;
    /* SPI/I2S underrun error interrupt */
    case SPI_I2S_INT_TXURE:
        SPI_INT(spi_periph) &= (~SPI_INT_TXUREIE);
        break;
    /* SPI/I2S overrun error interrupt*/
    case SPI_I2S_INT_RXORE:
        SPI_INT(spi_periph) &= (~SPI_INT_RXOREIE);
        break;
    /* SPI/I2S CRC error interrupt */
    case SPI_I2S_INT_CRCER:
        SPI_INT(spi_periph) &= (~SPI_INT_CRCERIE);
        break;
    /* SPI TI frame error interrupt */
    case SPI_I2S_INT_FE:
        SPI_INT(spi_periph) &= (~SPI_INT_FEIE);
        break;
    /* SPI/I2S mode error interrupt */
    case SPI_I2S_INT_CONFE:
        SPI_INT(spi_periph) &= (~SPI_INT_CONFEIE);
        break;
    /* SPI/I2S TXSER reload interrupt */
    case SPI_I2S_INT_TXSERF:
        SPI_INT(spi_periph) &= (~SPI_INT_TXSERFIE);
        break;
    case SPI_I2S_INT_ALL:
        SPI_INT(spi_periph) = 0U;
        break;
    default:
        break;
    }
}

/*!
    \brief      configure SPI current data number
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[in]  current_num: SPI transfer current data number
                             it can be 0-0xFFFF
    \param[out] none
    \retval     none
*/
void hals_spi_current_data_num_config(uint32_t spi_periph, uint32_t current_num)
{
    uint32_t reg = SPI_CTL1(spi_periph);
    /* configure SPI current data number */
    reg &= (uint32_t)(~BITS(0, 15));
    reg |= current_num;
    SPI_CTL1(spi_periph) = reg;
}

/*!
    \brief      SPI master start transfer
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[in]  transfer_start: transfer start bit
                only one parameter can be selected which is shown as below:
      \arg        SPI_TRANS_START: the master transmission is occurring,
                                   or has been temporarily suspended by automatic suspend
      \arg        SPI_TRANS_IDLE: the master transfer is idle status
    \param[out] none
    \retval     none
*/
void hals_spi_master_transfer_start(uint32_t spi_periph, uint32_t transfer_start)
{
    if(SPI_TRANS_START == transfer_start) {
        SPI_CTL0(spi_periph) |= SPI_CTL0_MSTART;
    } else {
        SPI_CTL0(spi_periph) &= (~SPI_CTL0_MSTART);
    }
}

/*!
    \brief      SPI NSS pin low level in software mode
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[out] none
    \retval     none
*/
void hals_spi_nss_internal_low(uint32_t spi_periph)
{
    SPI_CTL0(spi_periph) &= (~SPI_CTL0_NSSI);
}

/*!
    \brief      enable quad wire SPI
    \param[in]  spi_periph: SPIx(x=3,4)
    \param[out] none
    \retval     none
*/
void hals_spi_quad_enable(uint32_t spi_periph)
{
    SPI_QCTL(spi_periph) |= SPI_QCTL_QMOD;
}

/*!
    \brief      disable quad wire SPI
    \param[in]  spi_periph: SPIx(x=3,4)
    \param[out] none
    \retval     none
*/
void hals_spi_quad_disable(uint32_t spi_periph)
{
    SPI_QCTL(spi_periph) &= (~SPI_QCTL_QMOD);
}

/*!
    \brief      enable quad wire SPI write
    \param[in]  spi_periph: SPIx(x=3,4)
    \param[out] none
    \retval     none
*/
void hals_spi_quad_write_enable(uint32_t spi_periph)
{
    SPI_QCTL(spi_periph) &= (~SPI_QCTL_QRD);
}

/*!
    \brief      enable quad wire SPI read
    \param[in]  spi_periph: SPIx(x=3,4)
    \param[out] none
    \retval     none
*/
void hals_spi_quad_read_enable(uint32_t spi_periph)
{
    SPI_QCTL(spi_periph) |= SPI_QCTL_QRD;
}

/*!
    \brief      enable quad wire SPI_IO2 and SPI_IO3 pin output
    \param[in]  spi_periph: SPIx(x=3,4)
    \param[out] none
    \retval     none
*/
void hals_spi_quad_io23_output_enable(uint32_t spi_periph)
{
    SPI_QCTL(spi_periph) |= SPI_QCTL_IO23_DRV;
}

/*!
   \brief      disable quad wire SPI_IO2 and SPI_IO3 pin output
   \param[in]  spi_periph: SPIx(x=3,4)
   \param[out] none
   \retval     none
*/
void hals_spi_quad_io23_output_disable(uint32_t spi_periph)
{
    SPI_QCTL(spi_periph) &= (~SPI_QCTL_IO23_DRV);
}

/*!
    \brief      get SPI interrupt flag status
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[in]  interrupt_flag: SPI/I2S interrupt flag status
                only one parameter can be selected which is shown as below:
      \arg        SPI_I2S_INT_FLAG_RP: RP interrupt flag
      \arg        SPI_I2S_INT_FLAG_TP: TP interrupt flag
      \arg        SPI_I2S_INT_FLAG_DP: DP interrupt flag
      \arg        SPI_I2S_INT_FLAG_ET: end of transfer or receive interrupt flag
      \arg        SPI_I2S_INT_FLAG_TXF: transmission filled interrupt flag
      \arg        SPI_I2S_INT_FLAG_TXURERR: underrun error interrupt flag
      \arg        SPI_I2S_INT_FLAG_RXORERR: overrun error interrupt flag
      \arg        SPI_I2S_INT_FLAG_CRCERR: CRC error interrupt flag
      \arg        SPI_I2S_INT_FLAG_FERR: TI frame error interrupt flag
      \arg        SPI_I2S_INT_FLAG_CONFERR: mode error interrupt flag
      \arg        SPI_I2S_INT_FLAG_TXSERF: TXSER reload interrupt flag
      \arg        SPI_I2S_INT_FLAG_SPD: suspend interrupt flag
      \arg        SPI_I2S_INT_FLAG_TC: TxFIFO clear interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_spi_interrupt_flag_get(uint32_t spi_periph, hal_spi_interrupt_flag_enum interrupt_flag)
{
    FlagStatus state = RESET;
    uint32_t reg1    = SPI_STAT(spi_periph);
    uint32_t reg2    = SPI_INT(spi_periph);

    switch(interrupt_flag) {
    /* SPI/I2S RP interrupt */
    case SPI_I2S_INT_FLAG_RP:
        reg1 = reg1 & SPI_STAT_RP;
        reg2 = reg2 & SPI_INT_RPIE;
        break;
    /* SPI/I2S TP interrupt */
    case SPI_I2S_INT_FLAG_TP:
        reg1 = reg1 & SPI_STAT_TP;
        reg2 = reg2 & SPI_INT_TPIE;
        break;
    /* SPI/I2S DP interrupt */
    case SPI_I2S_INT_FLAG_DP:
        reg1 = reg1 & SPI_STAT_DP;
        reg2 = reg2 & SPI_INT_DPIE;
        break;
    /* SPI end of transfer or receive interrupt */
    case SPI_I2S_INT_FLAG_ET:
        reg1 = reg1 & SPI_STAT_ET;
        reg2 = reg2 & SPI_INT_ESTCIE;
        break;
    /* SPI transmission filled interrupt */
    case SPI_I2S_INT_FLAG_TXF:
        reg1 = reg1 & SPI_STAT_TXF;
        reg2 = reg2 & SPI_INT_TXFIE;
        break;
    /* SPI/I2S underrun error interrupt */
    case SPI_I2S_INT_FLAG_TXURERR:
        reg1 = reg1 & SPI_STAT_TXURERR;
        reg2 = reg2 & SPI_INT_TXUREIE;
        break;
    /* SPI/I2S overrun error interrupt */
    case SPI_I2S_INT_FLAG_RXORERR:
        reg1 = reg1 & SPI_STAT_RXORERR;
        reg2 = reg2 & SPI_INT_RXOREIE;
        break;
    /* SPI/I2S CRC error interrupt */
    case SPI_I2S_INT_FLAG_CRCERR:
        reg1 = reg1 & SPI_STAT_CRCERR;
        reg2 = reg2 & SPI_INT_CRCERIE;
        break;
    /* SPI TI frame error interrupt */
    case SPI_I2S_INT_FLAG_FERR:
        reg1 = reg1 & SPI_STAT_FERR;
        reg2 = reg2 & SPI_INT_FEIE;
        break;
    /* SPI/I2S mode error interrupt */
    case SPI_I2S_INT_FLAG_CONFERR:
        reg1 = reg1 & SPI_STAT_CONFERR;
        reg2 = reg2 & SPI_INT_CONFEIE;
        break;
    /* SPI/I2S TXSER reload interrupt */
    case SPI_I2S_INT_FLAG_TXSERF:
        reg1 = reg1 & SPI_STAT_TXSERF;
        reg2 = reg2 & SPI_INT_TXSERFIE;
        break;
    /* SPI suspend interrupt */
    case SPI_I2S_INT_FLAG_SPD:
        reg1 = reg1 & SPI_STAT_SPD;
        reg2 = reg2 & SPI_INT_ESTCIE;
        break;
    /* SPI TXFIFO clear interrupt */
    case SPI_I2S_INT_FLAG_TC:
        reg1 = reg1 & SPI_STAT_TC;
        reg2 = reg2 & SPI_INT_ESTCIE;
        break;
    default:
        break;
    }

    /*get SPI/I2S interrupt flag status */
    if(reg1 && reg2) {
        state = SET;
    } else {
        /* do nothing */
    }

    return state;
}

/*!
    \brief      clear SPI interrupt flag status
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[in]  interrupt_flag: SPI/I2S interrupt flag status
                only one parameter can be selected which is shown as below:
      \arg        SPI_I2S_INT_FLAG_ET: end of transfer or receive interrupt flag
      \arg        SPI_I2S_INT_FLAG_TXF: transmission filled interrupt flag
      \arg        SPI_I2S_INT_FLAG_TXURERR: underrun error interrupt flag
      \arg        SPI_I2S_INT_FLAG_RXORERR: overrun error interrupt flag
      \arg        SPI_I2S_INT_FLAG_CRCERR: CRC error interrupt flag
      \arg        SPI_I2S_INT_FLAG_FERR: TI frame error interrupt flag
      \arg        SPI_I2S_INT_FLAG_CONFERR: mode error interrupt flag
      \arg        SPI_I2S_INT_FLAG_TXSERF: TXSER reload interrupt flag
      \arg        SPI_I2S_INT_FLAG_SPD: suspend interrupt flag
    \param[out] none
    \retval     none
*/
void hals_spi_interrupt_flag_clear(uint32_t spi_periph, hal_spi_interrupt_flag_enum interrupt_flag)
{
    if(SPI_I2S_INT_FLAG_ET == interrupt_flag) {
        SPI_STATC(spi_periph) |= SPI_STATC_ETC;
    } else {
        /* do nothing */
    }

    if(SPI_I2S_INT_FLAG_TXF == interrupt_flag) {
        SPI_STATC(spi_periph) |= SPI_STATC_TXFC;
    } else {
        /* do nothing */
    }

    if(SPI_I2S_INT_FLAG_RXORERR == interrupt_flag) {
        SPI_STATC(spi_periph) |= SPI_STATC_RXORERRC;
    } else {
        /* do nothing */
    }

    if(SPI_I2S_INT_FLAG_TXURERR == interrupt_flag) {
        SPI_STATC(spi_periph) |= SPI_STATC_TXURERRC;
    } else {
        /* do nothing */
    }

    if(SPI_I2S_INT_FLAG_CRCERR == interrupt_flag) {
        SPI_STATC(spi_periph) |= SPI_STATC_CRCERRC;
    } else {
        /* do nothing */
    }

    if(SPI_I2S_INT_FLAG_FERR == interrupt_flag) {
        SPI_STATC(spi_periph) |= SPI_STATC_FERRC;
    } else {
        /* do nothing */
    }

    if(SPI_I2S_INT_FLAG_CONFERR == interrupt_flag) {
        SPI_STATC(spi_periph) |= SPI_STATC_CONFERRC;
    } else {
        /* do nothing */
    }

    if(SPI_I2S_INT_FLAG_TXSERF == interrupt_flag) {
        SPI_STATC(spi_periph) |= SPI_STATC_TXSERFC;
    } else {
        /* do nothing */
    }

    if(SPI_I2S_INT_FLAG_SPD == interrupt_flag) {
        SPI_STATC(spi_periph) |= SPI_STATC_SPDC;
    } else {
        /* do nothing */
    }
}

/*!
    \brief      get SPI flag status
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[in]  flag: SPI flag status
                only one parameter can be selected which is shown as below:
      \arg        SPI_FLAG_RP: RP flag
      \arg        SPI_FLAG_TP: TP flag
      \arg        SPI_FLAG_DP: DP flag
      \arg        SPI_FLAG_ET: end of transfer or receive flag
      \arg        SPI_FLAG_TXF: transmission filled flag
      \arg        SPI_FLAG_TXURERR: underrun error flag
      \arg        SPI_FLAG_RXORERR: overrun error flag
      \arg        SPI_FLAG_CRCERR: CRC error flag
      \arg        SPI_FLAG_FERR: TI frame error flag
      \arg        SPI_FLAG_CONFERR: mode error flag
      \arg        SPI_FLAG_TXSERF: TXSER reload flag
      \arg        SPI_FLAG_SPD: suspend flag
      \arg        SPI_FLAG_TC: TxFIFO clear flag
      \arg        SPI_FLAG_RWNE: the word of RXFIFO is not empty flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_spi_flag_get(uint32_t spi_periph, hal_spi_flag_enum flag)
{
    FlagStatus state = RESET;

    if(SPI_STAT(spi_periph) & (uint32_t)flag) {
        state = SET;
    } else {
        /* do nothing */
    }

    return state;
}

/*!
    \brief      clear SPI flag status
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[in]  flag: SPI/I2S flag status
                only one parameter can be selected which is shown as below:
      \arg        SPI_STATC_ETC: clear the end of transfer flag
      \arg        SPI_STATC_TXFC: clear the send transmission filled flag
      \arg        SPI_STATC_TXURERRC: clear the transmission underrun error flag
      \arg        SPI_STATC_RXORERRC: clear the reception overrun error flag
      \arg        SPI_STATC_CRCERRC: clear the CRC error flag
      \arg        SPI_STATC_FERRC: clear the SPI TI format error flag
      \arg        SPI_STATC_CONFERRC: clear the configuration error flag
      \arg        SPI_STATC_TXSERFC: clear the TXSERF flag
      \arg        SPI_STATC_SPDC: clear the suspend flag
    \param[out] none
    \retval     none
*/
void hals_spi_flag_clear(uint32_t spi_periph, hal_spi_flag_enum flag)
{
    SPI_STATC(spi_periph) |= (uint32_t)flag;
}

/*!
    \brief      SPI receive interrupt handler
    \param[in]  spi: SPI device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _spi_receive_complete_interrupt(void *spi)
{
    hal_spi_dev_struct *spi_dev = (hal_spi_dev_struct *)spi;
    hal_spi_user_cb p_func = NULL;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi)) {
        HAL_DEBUGE("pointer [spi] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) > SPI_DATASIZE_16BIT) {
        /* Transmit and received data in 32 Bit mode */
        if((RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_RP))) && (0U < spi_dev->rxbuffer.pos)) {
            *(__IO uint32_t *)spi_dev->rxbuffer.buffer = SPI_RDATA(spi_dev->periph);
            spi_dev->rxbuffer.buffer += sizeof(uint32_t);
            spi_dev->rxbuffer.pos--;
        } else {
            /* do nothing */
        }
    } else if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) > SPI_DATASIZE_8BIT) {
        /* Transmit and Receive data in 16 Bit mode */
        if((RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_RP))) && (0U < spi_dev->rxbuffer.pos)) {
            if((2U <= spi_dev->rxbuffer.pos) && \
               ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                *(__IO uint32_t *)spi_dev->rxbuffer.buffer = SPI_RDATA(spi_dev->periph);
                SWAP_HIGH_LOW_16BITS(*(__IO uint32_t *)spi_dev->rxbuffer.buffer);
                spi_dev->rxbuffer.buffer += sizeof(uint32_t);
                spi_dev->rxbuffer.pos -= 2UL;
            } else {
                *(__IO uint16_t *)spi_dev->rxbuffer.buffer = (uint16_t)(SPI_RDATA(spi_dev->periph) & 0xFFFFU);
                spi_dev->rxbuffer.buffer += sizeof(uint16_t);
                spi_dev->rxbuffer.pos--;
            }
        }
    } else {
        /* Transmit and Receive data in 8 Bit mode */
        if((RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_RP))) && (0U < spi_dev->rxbuffer.pos)) {
            if((4U <= spi_dev->rxbuffer.pos) && \
               ((uint32_t)SPI_FIFO_TH_03DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                *(__IO uint32_t *)spi_dev->rxbuffer.buffer = SPI_RDATA(spi_dev->periph);
                SWAP32_BE_TO_LE(*(__IO uint32_t *)spi_dev->rxbuffer.buffer);
                spi_dev->rxbuffer.buffer += sizeof(uint32_t);
                spi_dev->rxbuffer.pos -= 4UL;
            } else if((2U <= spi_dev->rxbuffer.pos) && \
                      ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                *(__IO uint16_t *)spi_dev->rxbuffer.buffer = (uint16_t)(SPI_RDATA(spi_dev->periph) & 0xFFFFU);
                SWAP16_BE_TO_LE(*(__IO uint16_t *)spi_dev->rxbuffer.buffer);
                spi_dev->rxbuffer.buffer += sizeof(uint16_t);
                spi_dev->rxbuffer.pos -= 2UL;
            } else {
                *(__IO uint8_t *)spi_dev->rxbuffer.buffer = (uint8_t)(SPI_RDATA(spi_dev->periph) & 0xFFU);
                spi_dev->rxbuffer.buffer += sizeof(uint8_t);
                spi_dev->rxbuffer.pos--;
            }
        }
    }

    if(0U == spi_dev->rxbuffer.pos) {
        if(SPI_CRC_ENABLE == HAL_SPI_GET_CRC_USED(spi_dev->periph)) {
            if(RESET != hals_spi_flag_get(spi_dev->periph, SPI_FLAG_CRCERR)) {
                hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_CRCERR);
                spi_dev->error_code = HAL_SPI_ERROR_CRC;
                p_func = (hal_spi_user_cb)spi_dev->error_callback;
                if(NULL != p_func) {
                    p_func(spi_dev);
                } else {
                    /* do nothing */
                }
            }
        }

        p_func = (hal_spi_user_cb)spi_dev->rx_callback;
        if(NULL != p_func) {
            p_func(spi_dev);
        } else {
            /* do nothing */
        }

        spi_dev->state = HAL_SPI_STATE_READY;
        hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_FLAG_RP);
    }
}

/*!
    \brief      SPI transmit interrupt handler
    \param[in]  spi: SPI device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _spi_transmit_complete_interrupt(void *spi)
{
    hal_spi_dev_struct *spi_dev = (hal_spi_dev_struct *)spi;
    hal_spi_user_cb p_func      = NULL;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi)) {
        HAL_DEBUGE("pointer [spi] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) > SPI_DATASIZE_16BIT) {
        /* Transmit and received data in 32 Bit mode */
        /* transfer data if tp set*/
        if((RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_TP))) && (0U < spi_dev->txbuffer.pos)) {
            if(1U == spi_dev->txbuffer.pos) {
                /* request transfer suspend */
                SPI_CTL0(spi_dev->periph) |= SPI_CTL0_MSPDR;
            }
            SPI_TDATA(spi_dev->periph) = *(__IO uint32_t *)spi_dev->txbuffer.buffer;
            spi_dev->txbuffer.buffer += sizeof(uint32_t);
            spi_dev->txbuffer.pos--;
        } else {
            /* do nothing */
        }
    } else if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) > SPI_DATASIZE_8BIT) {
        /* Transmit and Receive data in 16 Bit mode */
        /* transfer data if tp set*/
        if(RESET != (SPI_STAT(spi_dev->periph) & SPI_STAT_TP)) {
            if((2U <= spi_dev->txbuffer.pos)
               && ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                if(2U == spi_dev->txbuffer.pos) {
                    /* request transfer suspend */
                    SPI_CTL0(spi_dev->periph) |= SPI_CTL0_MSPDR;
                }
                SPI_TDATA(spi_dev->periph) = *(__IO uint32_t *)spi_dev->txbuffer.buffer;
                spi_dev->txbuffer.buffer += sizeof(uint32_t);
                spi_dev->txbuffer.pos -= 2UL;
            } else {
                if(1U == spi_dev->txbuffer.pos) {
                    /* request transfer suspend */
                    SPI_CTL0(spi_dev->periph) |= SPI_CTL0_MSPDR;
                }
                SPI_TDATA(spi_dev->periph) = *(__IO uint16_t *)spi_dev->txbuffer.buffer;
                spi_dev->txbuffer.buffer += sizeof(uint16_t);
                spi_dev->txbuffer.pos--;
            }
        }
    } else {
        /* Transmit and Receive data in 8 Bit mode */
        /* transfer data if tp set*/
        if((RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_TP))) && (0U < spi_dev->txbuffer.pos)) {
            if((4U <= spi_dev->txbuffer.pos) && \
               ((uint32_t)SPI_FIFO_TH_03DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                if(4U == spi_dev->txbuffer.pos) {
                    /* request transfer suspend */
                    SPI_CTL0(spi_dev->periph) |= SPI_CTL0_MSPDR;
                }
                SPI_TDATA(spi_dev->periph) = *(__IO uint32_t *)spi_dev->txbuffer.buffer;
                spi_dev->txbuffer.buffer += sizeof(uint32_t);
                spi_dev->txbuffer.pos -= 4UL;
            } else if((2U <= spi_dev->txbuffer.pos) && \
                      ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                if(2U == spi_dev->txbuffer.pos) {
                    /* request transfer suspend */
                    SPI_CTL0(spi_dev->periph) |= SPI_CTL0_MSPDR;
                }
                SPI_TDATA(spi_dev->periph) = *(__IO uint16_t *)spi_dev->txbuffer.buffer;
                spi_dev->txbuffer.buffer += sizeof(uint16_t);
                spi_dev->txbuffer.pos -= 2UL;
            } else {
                if(1U == spi_dev->txbuffer.pos) {
                    /* request transfer suspend */
                    SPI_CTL0(spi_dev->periph) |= SPI_CTL0_MSPDR;
                }
                SPI_TDATA(spi_dev->periph) = (uint8_t)(*spi_dev->txbuffer.buffer);
                spi_dev->txbuffer.buffer += sizeof(uint8_t);
                spi_dev->txbuffer.pos--;
            }
        }
    }

    if(0U == spi_dev->txbuffer.pos) {
        if(SPI_CRC_ENABLE == HAL_SPI_GET_CRC_USED(spi_dev->periph)) {
            if(RESET != hals_spi_flag_get(spi_dev->periph, SPI_FLAG_CRCERR)) {
                hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_CRCERR);
                hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_CRCER);
                spi_dev->error_code = HAL_SPI_ERROR_CRC;
                p_func              = (hal_spi_user_cb)spi_dev->error_callback;
                if(NULL != p_func) {
                    p_func(spi_dev);
                } else {
                    /* do nothing */
                }
            }
        }

        p_func = (hal_spi_user_cb)spi_dev->tx_callback;
        if(NULL != p_func) {
            p_func(spi_dev);
        } else {
            /* do nothing */
        }

        spi_dev->state = HAL_SPI_STATE_READY;
        hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_TP);
    }
}

/*!
    \brief      SPI suspend interrupt handler
    \param[in]  spi: SPI device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _spi_suspend_interrupt(void *spi)
{
    hal_spi_dev_struct *spi_dev = (hal_spi_dev_struct *)spi;
    hal_spi_user_cb p_func      = (hal_spi_user_cb)spi_dev->suspend_callback;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi)) {
        HAL_DEBUGE("pointer [spi] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Clear interrupt flags */
    hals_spi_interrupt_flag_clear(spi_dev->periph, SPI_I2S_INT_FLAG_ET);
    hals_spi_interrupt_flag_clear(spi_dev->periph, SPI_I2S_INT_FLAG_SPD);

    /* Disable the SPI interrupt */
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_ESTC);

    if (NULL != p_func) {
        p_func(spi_dev);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      SPI 2 lines receive interrupt handler
    \param[in]  spi: SPI device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _spi_2lines_receive_interrupt(void *spi)
{
    hal_spi_dev_struct *spi_dev = (hal_spi_dev_struct *)spi;
    hal_spi_user_cb p_func = NULL;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi)) {
        HAL_DEBUGE("pointer [spi] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) > SPI_DATASIZE_16BIT) {
        /* Transmit and received data in 32 Bit mode */
        if((RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_RP))) && (0U < spi_dev->rxbuffer.pos)) {
            *(__IO uint32_t *)spi_dev->rxbuffer.buffer = SPI_RDATA(spi_dev->periph);
            spi_dev->rxbuffer.buffer += sizeof(uint32_t);
            spi_dev->rxbuffer.pos--;
        } else {
            /* do nothing */
        }
    } else if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) > SPI_DATASIZE_8BIT) {
        /* Transmit and Receive data in 16 Bit mode */
        if((RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_RP))) && (0U < spi_dev->rxbuffer.pos)) {
            if((2U <= spi_dev->rxbuffer.pos) && \
               ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                *(__IO uint32_t *)spi_dev->rxbuffer.buffer = SPI_RDATA(spi_dev->periph);
                SWAP_HIGH_LOW_16BITS(*(__IO uint32_t *)spi_dev->rxbuffer.buffer);
                spi_dev->rxbuffer.buffer += sizeof(uint32_t);
                spi_dev->rxbuffer.pos -= 2UL;
            } else {
                *(__IO uint16_t *)spi_dev->rxbuffer.buffer = (uint16_t)(SPI_RDATA(spi_dev->periph) & 0xFFFFU);
                spi_dev->rxbuffer.buffer += sizeof(uint16_t);
                spi_dev->rxbuffer.pos--;
            }
        }
    } else {
        /* Transmit and Receive data in 8 Bit mode */
        if((RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_RP))) && (0U < spi_dev->rxbuffer.pos)) {
            if((4U <= spi_dev->rxbuffer.pos) && \
               ((uint32_t)SPI_FIFO_TH_03DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                *(__IO uint32_t *)spi_dev->rxbuffer.buffer = SPI_RDATA(spi_dev->periph);
                SWAP32_BE_TO_LE(*(__IO uint32_t *)spi_dev->rxbuffer.buffer);
                spi_dev->rxbuffer.buffer += sizeof(uint32_t);
                spi_dev->rxbuffer.pos -= 4UL;
            } else if((2U <= spi_dev->rxbuffer.pos) && \
                      ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                *(__IO uint16_t *)spi_dev->rxbuffer.buffer = (uint16_t)(SPI_RDATA(spi_dev->periph) & 0xFFFFU);
                SWAP16_BE_TO_LE(*(__IO uint16_t *)spi_dev->rxbuffer.buffer);
                spi_dev->rxbuffer.buffer += sizeof(uint16_t);
                spi_dev->rxbuffer.pos -= 2UL;
            } else {
                *(__IO uint8_t *)spi_dev->rxbuffer.buffer = (uint8_t)(SPI_RDATA(spi_dev->periph) & 0xFFU);
                spi_dev->rxbuffer.buffer += sizeof(uint8_t);
                spi_dev->rxbuffer.pos--;
            }
        }
    }

    if(0U == spi_dev->rxbuffer.pos) {
        if(SPI_CRC_ENABLE == HAL_SPI_GET_CRC_USED(spi_dev->periph)) {
            if(RESET != hals_spi_flag_get(spi_dev->periph, SPI_FLAG_CRCERR)) {
                hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_CRCERR);
                p_func = (hal_spi_user_cb)spi_dev->error_callback;
                if(NULL != p_func) {
                    p_func(spi_dev);
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }

        p_func = (hal_spi_user_cb)spi_dev->rx_callback;
        if(NULL != p_func) {
            p_func(spi_dev);
        } else {
            /* do nothing */
        }

        spi_dev->state = HAL_SPI_STATE_READY;
        hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_RP);
    }
}

/*!
    \brief      SPI 2 lines transmit interrupt handler
    \param[in]  spi: SPI device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _spi_2lines_transmit_interrupt(void *spi)
{
    hal_spi_dev_struct *spi_dev = (hal_spi_dev_struct *)spi;
    hal_spi_user_cb p_func      = NULL;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == spi)) {
        HAL_DEBUGE("pointer [spi] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) > SPI_DATASIZE_16BIT) {
        /* Transmit and received data in 32 Bit mode */
        /* transfer data if tp set*/
        if((RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_TP))) && (0U < spi_dev->txbuffer.pos)) {
            SPI_TDATA(spi_dev->periph) = *(__IO uint32_t *)spi_dev->txbuffer.buffer;
            spi_dev->txbuffer.buffer += sizeof(uint32_t);
            spi_dev->txbuffer.pos--;
        } else {
            /* do nothing */
        }
    } else if(HAL_SPI_GET_FRAME_SIZE(spi_dev->periph) > SPI_DATASIZE_8BIT) {
        /* Transmit and Receive data in 16 Bit mode */
        /* transfer data if tp set*/
        if((RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_TP))) && (0U < spi_dev->txbuffer.pos)) {
            if((2U <= spi_dev->txbuffer.pos) && \
               ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                SPI_TDATA(spi_dev->periph) = *(__IO uint32_t *)spi_dev->txbuffer.buffer;
                spi_dev->txbuffer.buffer += sizeof(uint32_t);
                spi_dev->txbuffer.pos -= 2UL;
            } else {
                SPI_TDATA(spi_dev->periph) = *(__IO uint16_t *)spi_dev->txbuffer.buffer;
                spi_dev->txbuffer.buffer += sizeof(uint16_t);
                spi_dev->txbuffer.pos--;
            }
        }
    } else {
        /* Transmit and Receive data in 8 Bit mode */
        /* transfer data if tp set*/
        if((RESET != ((SPI_STAT(spi_dev->periph) & SPI_STAT_TP))) && (0U < spi_dev->txbuffer.pos)) {
            if((4U <= spi_dev->txbuffer.pos) && \
               ((uint32_t)SPI_FIFO_TH_03DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                SPI_TDATA(spi_dev->periph) = *(__IO uint32_t *)spi_dev->txbuffer.buffer;
                spi_dev->txbuffer.buffer += sizeof(uint32_t);
                spi_dev->txbuffer.pos -= 4UL;
            } else if((2U <= spi_dev->txbuffer.pos) && \
                      ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(spi_dev->periph) & SPI_CFG0_FIFOLVL)))) {
                SPI_TDATA(spi_dev->periph) = *(__IO uint16_t *)spi_dev->txbuffer.buffer;
                spi_dev->txbuffer.buffer += sizeof(uint16_t);
                spi_dev->txbuffer.pos -= 2UL;
            } else {
                SPI_TDATA(spi_dev->periph) = (uint8_t)(*spi_dev->txbuffer.buffer);
                spi_dev->txbuffer.buffer += sizeof(uint8_t);
                spi_dev->txbuffer.pos--;
            }
        }
    }

    if(0U == spi_dev->txbuffer.pos) {
        if(SPI_CRC_ENABLE == HAL_SPI_GET_CRC_USED(spi_dev->periph)) {
            if(RESET != hals_spi_flag_get(spi_dev->periph, SPI_FLAG_CRCERR)) {
                hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_CRCERR);
                p_func = (hal_spi_user_cb)spi_dev->error_callback;
                if(NULL != p_func) {
                    p_func(spi_dev);
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_TP);
    } else {
        /* do nothing */
    }
}

/*!
  \brief      abort transfer and clear flags
  \param[in]  spi_dev: pointer to a SPI_HandleTypeDef structure that contains
                the configuration information for SPI module.
  \param[out] none
  \retval     none
*/
static void _spi_abort_transmit(void *spi)
{
    hal_spi_dev_struct *spi_dev = (hal_spi_dev_struct *)spi;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == spi) {
        HAL_DEBUGE("pointer [spi_dev] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable peripheral and interrupt */
    hals_spi_disable(spi_dev->periph);
    hals_spi_interrupt_disable(spi_dev->periph, (uint8_t)SPI_I2S_INT_ALL);

    /* clear transfer end and transfer fail flag */
    hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_ET);
    hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_TXF);

    /* clear tx dma and rx dma request */
    SPI_CFG0(spi_dev->periph) &= (~SPI_CFG0_DMATEN);
    SPI_CFG0(spi_dev->periph) &= (~SPI_CFG0_DMAREN);

    /* clear error flag */
    hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_RXORERR);
    hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_TXURERR);
    hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_FERR);
    hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_CRCERR);
    hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_CONFERR);

    /* disable crc*/
    if(RESET != (SPI_CFG0(spi_dev->periph) & SPI_CFG0_CRCEN)) {
        hals_spi_crc_off(spi_dev->periph);
    } else {
        /* do nothing */
    }

    spi_dev->txbuffer.pos = 0U;
    spi_dev->rxbuffer.pos = 0U;

}

/*!
  \brief      abort dma transmit
  \param[in]  spi_dev: pointer to a SPI_HandleTypeDef structure that contains
                the configuration information for SPI module
  \param[out] none
  \retval     none
*/
static void _spi_abort_dma_transmit(void *spi)
{
    hal_spi_dev_struct *spi_dev = (hal_spi_dev_struct *)spi;
    hal_spi_user_cb p_func      = (hal_spi_user_cb)spi_dev->abort_callback;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == spi) {
        HAL_DEBUGE("pointer [spi_dev] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    spi_dev->p_dma_tx->dma_irq.abort_handle = NULL;

    /* check if an abort process is still ongoing */
    if(spi_dev->p_dma_rx != NULL) {
        if(spi_dev->p_dma_rx->dma_irq.abort_handle != NULL) {
            spi_dev->error_code = HAL_SPI_ERROR_ABORT;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if (HAL_SPI_ERROR_NONE == spi_dev->error_code) {
        /* call the abort procedure */
        _spi_abort_transmit(spi_dev);
        if(NULL != p_func) {
            p_func(spi_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
  \brief      abort dma receive
  \param[in]  spi_dev: pointer to a SPI_HandleTypeDef structure that contains
                the configuration information for SPI module
  \param[out] none
  \retval     none
*/
static void _spi_abort_dma_receive(void *spi)
{
    int32_t error_code = HAL_ERR_NONE;
    hal_spi_dev_struct *spi_dev = (hal_spi_dev_struct *)spi;
    hal_spi_user_cb p_func      = NULL;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == spi) {
        HAL_DEBUGE("pointer [spi_dev] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    p_func = (hal_spi_user_cb)spi_dev->abort_callback;
    spi_dev->p_dma_rx->dma_irq.abort_handle = NULL;

    /* check if an abort process is still ongoing */
    if(spi_dev->p_dma_tx != NULL) {
        if(spi_dev->p_dma_tx->dma_irq.abort_handle != NULL) {
            spi_dev->error_code = HAL_SPI_ERROR_ABORT;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if (HAL_ERR_NONE == error_code) {
        /* call the abort procedure */
        _spi_abort_transmit(spi_dev);
        if(NULL != p_func) {
            p_func(spi_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
   \brief      stop transfer and clear flags.
   \param[in]  spi_dev: pointer to a SPI_HandleTypeDef structure that contains
                 the configuration information for SPI module.
   \param[out] none
   \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,  details refer to gd32h7xx_hal.h
*/
static int32_t _spi_stop_transfer(hal_spi_dev_struct *spi_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == spi_dev) {
        HAL_DEBUGE("pointer [spi_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear error flag */
    hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_ET);
    hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_TXF);

    /* disable interrupt */
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_ESTC);
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_TP);
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_RP);
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_TXURE);
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_RXORE);
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_FE);
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_CONFE);
    hals_spi_interrupt_disable(spi_dev->periph, SPI_I2S_INT_DP);

    /* clear tx dma request */
    SPI_CFG0(spi_dev->periph) &= (~SPI_CFG0_DMATEN);

    /* Report OverRun error for non TX Only communication */
    if(HAL_SPI_STATE_BUSY_TX == spi_dev->state) {
        if(RESET != (SPI_STAT(spi_dev->periph) & (uint32_t)SPI_FLAG_TXURERR)) {
            spi_dev->error_code = HAL_SPI_ERROR_OVR;
            hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_TXURERR);
            SPI_TDATA(spi_dev->periph) = SPI_INIT_TDATA_MASK;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* Report UnderRun error for non RX Only communication */
    if(HAL_SPI_STATE_BUSY_RX == spi_dev->periph) {
        if(RESET != (SPI_STAT(spi_dev->periph) & (uint32_t)SPI_FLAG_RXORERR)) {
            spi_dev->error_code = HAL_SPI_ERROR_UDR;
            hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_RXORERR);
            SPI_RDATA(spi_dev->periph) = SPI_INIT_RDATA_MASK;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* Check if CRC error occurred */
    if(RESET != (SPI_CFG0(spi_dev->periph) & SPI_CFG0_CRCEN)) {
        if(RESET != (SPI_STAT(spi_dev->periph) & (uint32_t)SPI_FLAG_CRCERR)) {
            spi_dev->error_code = HAL_SPI_ERROR_CRC;
            hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_CRCERR);
            SPI_TCRC(spi_dev->periph) = SPI_INIT_CRC_MASK;
            SPI_RCRC(spi_dev->periph) = SPI_INIT_CRC_MASK;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* SPI Frame error interrupt occurred */
    if(RESET != (SPI_STAT(spi_dev->periph) & SPI_STAT_FERR)) {
        spi_dev->error_code = HAL_SPI_ERROR_FRE;
        hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_FERR);
    } else {
        /* do nothing */
    }

    /* SPI mode error interrupt occurred */
    if(RESET != (SPI_STAT(spi_dev->periph) & SPI_STAT_CONFERR)) {
        spi_dev->error_code = HAL_SPI_ERROR_MODF;
        hals_spi_flag_clear(spi_dev->periph, SPI_FLAG_CONFERR);
    } else {
        /* do nothing */
    }

    spi_dev->txbuffer.pos = 0U;
    spi_dev->rxbuffer.pos = 0U;

    return error_code;
}

/*!
    \brief      SPI DMA transmit complete handler
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _spi_transmit_complete_dma(void *dma)
{
    hal_spi_user_cb p_func     = NULL;
    hal_spi_user_cb p_func_err = NULL;

    hal_dma_dev_struct *p_dma;
    hal_spi_dev_struct *p_spi;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == dma) {
        HAL_DEBUGE("pointer [dma] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* get device struct info */
    p_dma      = (hal_dma_dev_struct *)dma;
    p_spi      = (hal_spi_dev_struct *)p_dma->p_periph;
    p_func     = (hal_spi_user_cb)p_spi->tx_callback;
    p_func_err = (hal_spi_user_cb)p_spi->error_callback;

    /* DMA normal Mode */
    if(RESET != (DMA_CHCTL(p_dma->dma_periph, p_dma->channel) & DMA_CHXCTL_CMEN)) {
        /* disable Tx DMA Request */
        HAL_SPI_DMA_DISABLE(p_spi->periph, SPI_DMA_TRANSMIT);

        /* clear flag */
        hals_spi_flag_clear(p_spi->periph, SPI_FLAG_TXF);
        hals_spi_flag_clear(p_spi->periph, SPI_FLAG_TXURERR);
        hals_spi_flag_clear(p_spi->periph, SPI_FLAG_RXORERR);
        hals_spi_flag_clear(p_spi->periph, SPI_FLAG_ET);

        /* get crc status */
        if(SPI_CRC_ENABLE == HAL_SPI_GET_CRC_USED(p_spi->periph)) {
            if(RESET != hals_spi_flag_get(p_spi->periph, SPI_FLAG_CRCERR)) {
                hals_spi_flag_clear(p_spi->periph, SPI_FLAG_CRCERR);
                p_spi->error_code = HAL_SPI_ERROR_CRC;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        if(p_spi->error_code != (uint16_t)HAL_ERR_NONE) {
            if(NULL != p_func_err) {
                p_func_err(p_spi);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    p_spi->txbuffer.pos = 0U;
    if(HAL_SPI_STATE_BUSY_TX == p_spi->state) {
        /* check user callback valid */
        if(NULL != p_func) {
            p_func(p_spi);
        } else {
            /* do nothing */
        }
        p_spi->state = HAL_SPI_STATE_READY;
    } else {
        /* do nothing */
    }

}

/*!
    \brief      SPI DMA transmit half complete handler
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _spi_transmit_half_complete_dma(void *dma)
{
    hal_spi_user_cb p_func     = NULL;
    hal_dma_dev_struct *p_dma  = NULL;
    hal_spi_dev_struct *p_spi  = NULL;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == dma) {
        HAL_DEBUGE("pointer [dma] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* get device struct info */
    p_dma      = (hal_dma_dev_struct *)dma;
    p_spi      = (hal_spi_dev_struct *)p_dma->p_periph;
    p_func     = (hal_spi_user_cb)p_spi->tx_half_callback;

    if(NULL != p_func) {
        p_func(p_spi);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      SPI DMA receive complete handler
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _spi_receive_complete_dma(void *dma)
{
    hal_spi_user_cb p_func     = NULL;
    hal_spi_user_cb p_func_err = NULL;
    hal_dma_dev_struct *p_dma  = NULL;
    hal_spi_dev_struct *p_spi  = NULL;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == dma) {
        HAL_DEBUGE("pointer [dma] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    p_dma      = (hal_dma_dev_struct *)dma;
    p_spi      = (hal_spi_dev_struct *)p_dma->p_periph;
    p_func     = (hal_spi_user_cb)p_spi->rx_callback;
    p_func_err = (hal_spi_user_cb)p_spi->error_callback;

    /* DMA Normal Mode */
    if(RESET != (DMA_CHCTL(p_dma->dma_periph, p_dma->channel) & DMA_CHXCTL_CMEN)) {
        /* disable Rx/Tx DMA Request */
        HAL_SPI_DMA_DISABLE(p_spi->periph, SPI_DMA_RECEIVE);

        if(SPI_CRC_ENABLE == HAL_SPI_GET_CRC_USED(p_spi->periph)) {
            if(RESET != hals_spi_flag_get(p_spi->periph, SPI_FLAG_CRCERR)) {
                hals_spi_flag_clear(p_spi->periph, SPI_FLAG_CRCERR);
                p_spi->error_code = HAL_SPI_ERROR_CRC;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        if(p_spi->error_code != HAL_SPI_ERROR_NONE) {
            if(NULL != p_func_err) {
                p_func_err(p_spi);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check frame size */
    if(HAL_SPI_GET_FRAME_SIZE(p_spi->periph) > SPI_DATASIZE_16BIT) {
        ;
    } else if(HAL_SPI_GET_FRAME_SIZE(p_spi->periph) > SPI_DATASIZE_8BIT) {
        while(p_spi->rxbuffer.pos) {
            if((2U <= p_spi->rxbuffer.pos) && ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(p_spi->periph) & SPI_CFG0_FIFOLVL)))) {
                SWAP_HIGH_LOW_16BITS(*(__IO uint32_t *)p_spi->rxbuffer.buffer);
                p_spi->rxbuffer.buffer += sizeof(uint32_t);
                p_spi->rxbuffer.pos -= 2UL;
            } else {
                break;
            }
        }
    } else {
        while(p_spi->rxbuffer.pos) {
            /* check fifo threshold */
            if((4U <= p_spi->rxbuffer.pos) && ((uint32_t)SPI_FIFO_TH_03DATA < ((SPI_CFG0(p_spi->periph) & SPI_CFG0_FIFOLVL)))) {
                SWAP32_BE_TO_LE(*(__IO uint32_t *)p_spi->rxbuffer.buffer);
                p_spi->rxbuffer.buffer += sizeof(uint32_t);
                p_spi->rxbuffer.pos -= 4UL;
            } else if((2U <= p_spi->rxbuffer.pos) && ((uint32_t)SPI_FIFO_TH_01DATA < ((SPI_CFG0(p_spi->periph) & SPI_CFG0_FIFOLVL)))) {
                SWAP16_BE_TO_LE(*(__IO uint16_t *)p_spi->rxbuffer.buffer);
                p_spi->rxbuffer.buffer += sizeof(uint16_t);
                p_spi->rxbuffer.pos -= 2UL;
            } else {
                break;
            }
        }
    }

    /* user call back */
    if(NULL != p_func) {
        p_func(p_spi);
    } else {
        /* do nothing */
    }

    p_spi->rxbuffer.pos = 0U;
    p_spi->state = HAL_SPI_STATE_READY;

}

/*!
    \brief      SPI DMA receive half complete handler
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _spi_receive_half_complete_dma(void *dma)
{
    hal_spi_user_cb p_func     = NULL;
    hal_dma_dev_struct *p_dma  = NULL;
    hal_spi_dev_struct *p_spi  = NULL;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == dma) {
        HAL_DEBUGE("pointer [dma] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    p_dma      = (hal_dma_dev_struct *)dma;
    p_spi      = (hal_spi_dev_struct *)p_dma->p_periph;
    p_func     = (hal_spi_user_cb)p_spi->rx_half_callback;

    /* user call back */
    if(NULL != p_func) {
        p_func(p_spi);
    } else {
        /* do nothing */
    }

}

/*!
    \brief      SPI DMA error handler
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _spi_dma_error(void *dma)
{
    hal_spi_user_cb p_func_err = NULL;
    hal_dma_dev_struct *p_dma  = NULL;
    hal_spi_dev_struct *p_spi  = NULL;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == dma) {
        HAL_DEBUGE("pointer [dma] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    p_dma = (hal_dma_dev_struct *)dma;
    p_spi = (hal_spi_dev_struct *)p_dma->p_periph;
    p_func_err = (hal_spi_user_cb)p_spi->error_callback;

    /* disable dma flag */
    HAL_SPI_DMA_DISABLE(p_spi->periph, SPI_DMA_RECEIVE);
    HAL_SPI_DMA_DISABLE(p_spi->periph, SPI_DMA_TRANSMIT);

    p_spi->error_code = HAL_SPI_ERROR_DMA;
    p_spi->state = HAL_SPI_STATE_READY;

    /* user callback */
    if(NULL != p_func_err) {
        p_func_err(p_spi);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      reset SPI
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[out] none
    \retval     none
*/
static void _spi_deinit(uint32_t spi_periph)
{
    switch(spi_periph) {
    case SPI0:
        /* reset SPI0 */
        hal_rcu_periph_reset_enable(RCU_SPI0RST);
        hal_rcu_periph_reset_disable(RCU_SPI0RST);
        break;
    case SPI1:
        /* reset SPI1 */
        hal_rcu_periph_reset_enable(RCU_SPI1RST);
        hal_rcu_periph_reset_disable(RCU_SPI1RST);
        break;
    case SPI2:
        /* reset SPI2 */
        hal_rcu_periph_reset_enable(RCU_SPI2RST);
        hal_rcu_periph_reset_disable(RCU_SPI2RST);
        break;
    case SPI3:
        /* reset SPI3 */
        hal_rcu_periph_reset_enable(RCU_SPI3RST);
        hal_rcu_periph_reset_disable(RCU_SPI3RST);
        break;
    case SPI4:
        /* reset SPI4 */
        hal_rcu_periph_reset_enable(RCU_SPI4RST);
        hal_rcu_periph_reset_disable(RCU_SPI4RST);
        break;
    case SPI5:
        /* reset SPI5 */
        hal_rcu_periph_reset_enable(RCU_SPI5RST);
        hal_rcu_periph_reset_disable(RCU_SPI5RST);
        break;
    default:
        break;
    }
}

/*!
    \brief      wait the flag status until timeout
    \param[in]  spi_periph: SPIx(x=0,1,2,3,4,5)
    \param[in]  flag: spi flags, refer to hal_spi_flag_enum
    \param[in]  status: the status of spi flag to wait
    \param[in]  timeout_ms: timeout duration
    \param[out] none
    \retval     none
*/
static int32_t _spi_wait_flag_timeout(uint32_t spi_periph, hal_spi_flag_enum flag, FlagStatus status, \
                                      uint32_t timeout_ms)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t tick_start = 0U;

    /* set timeout */
    tick_start = hal_sys_basetick_count_get();

    /* wait flag status RESET */
    if(RESET == status) {
        while(SET == hals_spi_flag_get(spi_periph, flag)) {
            if(HAL_TIMEOUT_FOREVER != timeout_ms) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) {
                    HAL_DEBUGE("spi get flag timeout");
                    error_code = HAL_ERR_TIMEOUT;
                    break;
                }
            } else {
                /* do nothing */
            }
        }
    } else {
        /* wait flag status SET */
        while(RESET == hals_spi_flag_get(spi_periph, flag)) {
            if(HAL_TIMEOUT_FOREVER != timeout_ms) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) {
                    HAL_DEBUGE("spi get flag timeout");
                    error_code = HAL_ERR_TIMEOUT;
                    break;
                }
            } else {
                /* do nothing */
            }
        }
    }

    return error_code;
}

/*!
    \brief      32-bits big-endian to little-endian conversion
    \param[in]  x: 32-bits data
    \param[out] none
    \retval     swapped data
*/
static uint32_t _conversion_high2low_32bits(uint32_t x)
{
    return ((uint32_t)((((uint32_t)(x) & 0x000000FFU) << 24U) | \
            (((uint32_t)(x) & 0x0000FF00U) << 8U)  | \
            (((uint32_t)(x) & 0x00FF0000U) >> 8U)  | \
            (((uint32_t)(x) & 0xFF000000U) >> 24U)));
}

/*!
    \brief      16-bits big-endian to little-endian conversion
    \param[in]  x: 16-bits data
    \param[out] none
    \retval     swapped data
*/
static uint16_t _conversion_high2low_16bits(uint16_t x)
{
    return ((uint16_t)((((uint16_t)(x) & 0x00FFU) << 8U) | \
            (((uint16_t)(x) & 0xFF00U) >> 8U)));
}

/*!
    \brief      swap high to low 32-bits
    \param[in]  x: 32-bits data
    \param[out] none
    \retval     swapped data
*/
static uint32_t _swap_high2low_32bits(uint32_t x)
{
    return (((x) & 0xFFFF0000U) >> 16U) | (((x) & 0x0000FFFFU) << 16U);
}
