/*!
    \file    gd32h7xx_hal_rspdif.c
    \brief   RSPDIF driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define RSPDIF_INIT_MASK           (uint32_t)0xFFC88004U /*!< RSPDIF parameter initialization mask */

#define RSPDIF_CKCNT5_OFFSET       (uint32_t)16U         /*!< RSPDIF the number of consecutive time clock cycles offset */

#define RSPDIF_DATA_F0_PREF_OFFSET (uint32_t)28U         /*!< RSPDIF data format 0 preamble type offset */
#define RSPDIF_DATA_F0_C_OFFSET    (uint32_t)27U         /*!< RSPDIF data format 0 channel status offset */
#define RSPDIF_DATA_F0_U_OFFSET    (uint32_t)26U         /*!< RSPDIF data format 0 user bit offset */
#define RSPDIF_DATA_F0_V_OFFSET    (uint32_t)25U         /*!< RSPDIF data format 0 validity bit offset */
#define RSPDIF_DATA_F0_P_OFFSET    (uint32_t)24U         /*!< RSPDIF data format 0 parity error bit offset */

#define RSPDIF_DATA_F1_PREF_OFFSET (uint32_t)4U          /*!< RSPDIF data format 1 preamble type offset */
#define RSPDIF_DATA_F1_C_OFFSET    (uint32_t)3U          /*!< RSPDIF data format 1 channel status offset */
#define RSPDIF_DATA_F1_U_OFFSET    (uint32_t)2U          /*!< RSPDIF data format 1 user bit offset */
#define RSPDIF_DATA_F1_V_OFFSET    (uint32_t)1U          /*!< RSPDIF data format 1 validity bit offset */
#define RSPDIF_DATA_F1_DATA_OFFSET (uint32_t)8U          /*!< RSPDIF data format 1 data offset */

#define RSPDIF_CTL_RESET_VALUE     (uint32_t)0x00000000U /*!< RSPDIF CTL register default value */
#define RSPDIFRX_TIMEOUT_VALUE     10U                   /*!< RSPDIF timeout value */

/* RSPDIF DMA full transmission complete callback */
static void _rspdif_dma_full_transfer_complete(void *dma);
/* RSPDIF DMA full transmission complete callback */
static void _rspdif_dma_half_transfer_complete(void *dma);
/* RSPDIF DMA underflow error callback */
static void _rspdif_dma_error(void *dma);
/* receive an amount of data (data flow) with interrupt */
static void _rspdif_receive_data_handle(hal_rspdif_dev_struct *rspdif_dev);
/* receive an amount of data (control flow) with interrupt */
static void _rspdif_receive_ctrl_handle(hal_rspdif_dev_struct *rspdif_dev);
/* RSPDIF perr callback */
static void _rspdif_error_handle(void *rspdif_dev);
/* wait on flag until timeout */
static int32_t _rspdif_WaitOnFlagUntilTimeout(hal_rspdif_dev_struct *rspdif_dev, uint32_t Flag, \
                                              FlagStatus Status, uint32_t Timeout, uint32_t tickstart);

/*!
    \brief      reset the RSPDIF
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rspdif_deinit(hal_rspdif_dev_struct *rspdif_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == rspdif_dev) {
        HAL_DEBUGE("pointer [rspdif_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    rspdif_dev->state = HAL_RSPDIF_STATE_BUSY;

    /* reset RSPDIF */
    hal_rcu_periph_reset_enable(RCU_RSPDIFRST);
    hal_rcu_periph_reset_disable(RCU_RSPDIFRST);

    /* change RSPDIF error state and state */
    rspdif_dev->state       = HAL_RSPDIF_STATE_RESET;
    rspdif_dev->error_state = HAL_RSPDIF_ERROR_NONE;

    /* return function state */
    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the RSPDIF parameters
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rspdif_struct: RSPDIF parameter initialization structure and the member values are shown as below:
                  input_sel:
                  only one parameter can be selected which is shown as below:
      \arg          RSPDIF_INPUT_INx (x = 0 ~ 3)
                  max_retrie:
                  only one parameter can be selected which is shown as below:
      \arg          RSPDIF_MAXRETRIES_NONE: no retry
      \arg          RSPDIF_MAXRETRIES_3: allow up to 3 retries
      \arg          RSPDIF_MAXRETRIES_15: allow up to 15 retries
      \arg          RSPDIF_MAXRETRIES_63: allow up to 63 retries
                  wait_activity:
                  only one parameter can be selected which is shown as below:
      \arg          RSPDIF_WAIT_FOR_ACTIVITY_OFF: RSPDIF does not wait for the valid conversion signal from the selected
                    RSPDIF channel
      \arg          RSPDIF_WAIT_FOR_ACTIVITY_ON: RSPDIF wait for the valid conversion signal from the selected RSPDIF
                    channel
                  channel_sel:
                  only one parameter can be selected which is shown as below:
      \arg          RSPDIF_CHANNEL_A: gets channel status from channel A
      \arg          RSPDIF_CHANNEL_B: gets channel status from channel B
                  sample_format:
                  only one parameter can be selected which is shown as below:
      \arg          RSPDIF_DATAFORMAT_LSB: audio data is right aligned (LSB)
      \arg          RSPDIF_DATAFORMAT_MSB: audio data is left aligned (MSB)
      \arg          RSPDIF_DATAFORMAT_32BITS: which packs two 16-bit audio data into one 32-bit data
                  sound_mode:
                  only one parameter can be selected which is shown as below:
      \arg          RSPDIF_STEREOMODE_DISABLE: MONO mode
      \arg          RSPDIF_STEREOMODE_ENABLE: STEREO mode
                  pre_type:
                  only one parameter can be selected which is shown as below:
      \arg          RSPDIF_PREAMBLE_TYPE_MASK_OFF: copy the preamble type bit into the RSPDIF_DATA
      \arg          RSPDIF_PREAMBLE_TYPE_MASK_ON: do not copy the preamble type bit into the RSPDIF_DATA, but write 0
                    instead
                  channel_status_bit:
                  only one parameter can be selected which is shown as below:
      \arg          RSPDIF_CHANNEL_STATUS_MASK_OFF: copy the channel status and user bit into the RSPDIF_DATA
      \arg          RSPDIF_CHANNEL_STATUS_MASK_ON: do not copy the channel status and user bit into the RSPDIF_DATA, but
                    write 0 instead
                  validity_bit:
                  only one parameter can be selected which is shown as below:
      \arg          RSPDIF_VALIDITY_MASK_OFF: copy the validity bit into the RSPDIF_DATA
      \arg          RSPDIF_VALIDITY_MASK_ON: do not copy the validity bit into the RSPDIF_DATA, but write 0 instead
                  parity_error_bit:
                  only one parameter can be selected which is shown as below:
      \arg          RSPDIF_PERROR_MASK_OFF: copy the parity error bit into the RSPDIF_DATA
      \arg          RSPDIF_PERROR_MASK_ON: do not copy the parity error bit into the RSPDIF_DATA, but write 0 instead
                  symbol_clk:
                  only one parameter can be selected which is shown as below:
      \arg          RSPDIF_SYMBOL_CLK_OFF: the RSPDIF does not generate a symbol clock
      \arg          RSPDIF_SYMBOL_CLK_ON: the RSPDIF generates a symbol clock
                  bak_symbol_clk:
                  only one parameter can be selected which is shown as below:
      \arg          RSPDIF_BACKUP_SYMBOL_CLK_OFF: the RSPDIF does not generate a backup symbol clock
      \arg          RSPDIF_BACKUP_SYMBOL_CLK_ON: the RSPDIF generates a backup symbol clock if SCKEN = 1
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rspdif_init(hal_rspdif_dev_struct *rspdif_dev, hal_rspdif_init_struct *rspdif_struct)
{
      /* configure the RSPDIF */
    uint32_t reg = 0x00000000U;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == rspdif_dev) || (NULL == rspdif_struct)) {
        HAL_DEBUGE("pointer [rspdif_dev] or [rspdif_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    rspdif_dev->state = HAL_RSPDIF_STATE_BUSY;

    reg = RSPDIF_CTL;
    reg &= RSPDIF_INIT_MASK;

    /* select the RSPDIF input 16-18 */
    reg |= rspdif_struct->input_sel;
    /* configure the RSPDIF maximum allowed re-tries during synchronization phase */
    reg |= rspdif_struct->max_retrie;
    /* configure the RSPDIF wait for activity on the selected input */
    reg |= rspdif_struct->wait_activity;
    /* select the channel status from channel A or B */
    reg |= rspdif_struct->channel_sel;
    /* configure the RSPDIF data samples format */
    reg |= rspdif_struct->sample_format;
    /* select stereo or mono mode */
    reg |= rspdif_struct->sound_mode;
    /* select whether the preamble type value into the RSPDIF_DATA */
    reg |= rspdif_struct->pre_type;
    /* select whether the channel status and user bits are copied or not into the received frame */
    reg |= rspdif_struct->channel_status_bit;
    /* select whether the validity bit is copied or not into the received frame */
    reg |= rspdif_struct->validity_bit;
    /* select whether the parity error bit is copied or not into the received frame */
    reg |= rspdif_struct->parity_error_bit;
    /* configure the RSPDIF symbol clock generation */
    reg |= rspdif_struct->symbol_clk;
    /* configure the RSPDIF backup symbol clock generation */
    reg |= rspdif_struct->bak_symbol_clk;

    /* write to SPDIFRX_CTL register */
    RSPDIF_CTL = reg;

    /* change RSPDIF error state and state */
    rspdif_dev->error_state = HAL_RSPDIF_ERROR_NONE;
    rspdif_dev->state       = HAL_RSPDIF_STATE_READY;

    /* return function state */
    return HAL_ERR_NONE;
}

/*!
    \brief      specifies the RSPDIF peripheral state
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rspdif_cfg:
                only one parameter can be selected which is shown as below:
      \arg        RSPDIF_STATE_SYNC: enable RSPDIF synchronization only
      \arg        RSPDIF_STATE_RCV : enable RSPDIF receiver
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rspdif_enable(hal_rspdif_dev_struct *rspdif_dev, uint32_t rspdif_cfg)
{
    uint32_t reg = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == rspdif_dev) {
        HAL_DEBUGE("pointer [rspdif_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((RSPDIF_STATE_IDLE != rspdif_cfg) && (RSPDIF_STATE_SYNC != rspdif_cfg) && (RSPDIF_STATE_RCV != rspdif_cfg)) {
        HAL_DEBUGE("parameter [rspdif_cfg] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set state */
    rspdif_dev->state = HAL_RSPDIF_STATE_BUSY;

    reg = RSPDIF_CTL;

    /* clear RSPDIF state */
    reg &= ~(uint32_t)RSPDIF_CTL_RXCFG;
    reg |= (uint32_t)rspdif_cfg;

    /* enable RSPDIF */
    RSPDIF_CTL = (uint32_t)reg;

    /* set state */
    rspdif_dev->state = HAL_RSPDIF_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      disable RSPDIF
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_rspdif_disable(void)
{
    /* clear RSPDIF state */
    RSPDIF_CTL &= ~(uint32_t)RSPDIF_CTL_RXCFG;
}

/*!
    \brief      initialize the RSPDIF device structure with the default values
    \param[in]  hal_struct_type: the argument could be selected from enumeration <hal_rspdif_struct_type_enum>
                only one parameter can be selected which is shown as below:
      \arg        HAL_RSPDIF_INIT_STRUCT: RSPDIF initialization structure
      \arg        HAL_RSPDIF_DATAFORMAT_STRUCT: RSPDIF data format structure
      \arg        HAL_RSPDIF_DATA_STRUCT: RSPDIF data structure
      \arg        HAL_RSPDIF_IRQ_STRUCT: RSPDIF interrupt callback structure
      \arg        HAL_RSPDIF_IRQ_USER_CALLBACK_STRUCT: RSPDIF user interrupt callback structure
      \arg        HAL_RSPDIF_DEV_STRUCT: RSPDIF device structure
    \param[out] p_struct: pointer to RSPDIF structure that contains the configuration information
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_rspdif_struct_init(hal_rspdif_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_RSPDIF_INIT_STRUCT:
        /* initialize RSPDIF initialization structure with the default values */
        ((hal_rspdif_init_struct *)p_struct)->input_sel          = RSPDIF_INPUT_IN0;
        ((hal_rspdif_init_struct *)p_struct)->max_retrie         = RSPDIF_MAXRETRIES_NONE;
        ((hal_rspdif_init_struct *)p_struct)->wait_activity      = RSPDIF_WAIT_FOR_ACTIVITY_OFF;
        ((hal_rspdif_init_struct *)p_struct)->channel_sel        = RSPDIF_CHANNEL_A;
        ((hal_rspdif_init_struct *)p_struct)->sample_format      = RSPDIF_DATAFORMAT_LSB;
        ((hal_rspdif_init_struct *)p_struct)->sound_mode         = RSPDIF_STEREOMODE_DISABLE;
        ((hal_rspdif_init_struct *)p_struct)->pre_type           = RSPDIF_PREAMBLE_TYPE_MASK_OFF;
        ((hal_rspdif_init_struct *)p_struct)->channel_status_bit = RSPDIF_CHANNEL_STATUS_MASK_OFF;
        ((hal_rspdif_init_struct *)p_struct)->validity_bit       = RSPDIF_VALIDITY_MASK_OFF;
        ((hal_rspdif_init_struct *)p_struct)->parity_error_bit   = RSPDIF_PERROR_MASK_OFF;
        ((hal_rspdif_init_struct *)p_struct)->symbol_clk         = RSPDIF_SYMBOL_CLK_OFF;
        ((hal_rspdif_init_struct *)p_struct)->bak_symbol_clk     = RSPDIF_BACKUP_SYMBOL_CLK_OFF;
        break;
    case HAL_RSPDIF_DATAFORMAT_STRUCT:
        /* initialize RSPDIF data format structure with the default values */
        ((hal_rspdif_dataformat_struct *)p_struct)->DataFormat         = 0U;
        ((hal_rspdif_dataformat_struct *)p_struct)->StereoMode         = 0U;
        ((hal_rspdif_dataformat_struct *)p_struct)->PreambleTypeMask   = 0U;
        ((hal_rspdif_dataformat_struct *)p_struct)->ChannelStatusMask  = 0U;
        ((hal_rspdif_dataformat_struct *)p_struct)->ValidityBitMask    = 0U;
        ((hal_rspdif_dataformat_struct *)p_struct)->ParityErrorMask    = 0U;
        break;
    case HAL_RSPDIF_DATA_STRUCT:
        /* initialize RSPDIF data structure with the default values */
        ((hal_rspdif_data_struct *)p_struct)->format         = 0U;
        ((hal_rspdif_data_struct *)p_struct)->preamble       = 0U;
        ((hal_rspdif_data_struct *)p_struct)->channel_status = 0U;
        ((hal_rspdif_data_struct *)p_struct)->user_bit       = 0U;
        ((hal_rspdif_data_struct *)p_struct)->validity       = 0U;
        ((hal_rspdif_data_struct *)p_struct)->parity_err     = 0U;
        ((hal_rspdif_data_struct *)p_struct)->data0          = 0U;
        ((hal_rspdif_data_struct *)p_struct)->data1          = 0U;
        break;
    case HAL_RSPDIF_IRQ_STRUCT:
        /* initialize RSPDIF IRQ structure with the default values */
        ((hal_rspdif_irq_struct *)p_struct)->rspdif_rx_cpl_handle    = NULL;
        ((hal_rspdif_irq_struct *)p_struct)->rspdif_cx_cpl_handle    = NULL;
        ((hal_rspdif_irq_struct *)p_struct)->rspdif_rx_half_handle   = NULL;
        ((hal_rspdif_irq_struct *)p_struct)->rspdif_cx_half_handle   = NULL;
        ((hal_rspdif_irq_struct *)p_struct)->rspdif_error_handle     = NULL;
        break;
    case HAL_RSPDIF_IRQ_USER_CALLBACK_STRUCT:
        /* initialize RSPDIF user interrupt callback structure with the default values */
        ((hal_rspdif_irq_user_callback_struct *)p_struct)->rspdif_rx_cpl_callback             = NULL;
        ((hal_rspdif_irq_user_callback_struct *)p_struct)->rspdif_cx_cpl_callback             = NULL;
        ((hal_rspdif_irq_user_callback_struct *)p_struct)->rspdif_rx_half_callback            = NULL;
        ((hal_rspdif_irq_user_callback_struct *)p_struct)->rspdif_cx_half_callback            = NULL;
        ((hal_rspdif_irq_user_callback_struct *)p_struct)->rspdif_error_callback              = NULL;
        break;
    case HAL_RSPDIF_DEV_STRUCT:
        /* initialize RSPDIF device structure with the default values */
        ((hal_rspdif_dev_struct *)p_struct)->rspdif_rx_cpl_callback             = NULL;
        ((hal_rspdif_dev_struct *)p_struct)->rspdif_cx_cpl_callback             = NULL;
        ((hal_rspdif_dev_struct *)p_struct)->rspdif_rx_half_callback            = NULL;
        ((hal_rspdif_dev_struct *)p_struct)->rspdif_cx_half_callback            = NULL;
        ((hal_rspdif_dev_struct *)p_struct)->rspdif_error_callback              = NULL;
        ((hal_rspdif_dev_struct *)p_struct)->rspdif_irq.rspdif_rx_cpl_handle    = NULL;
        ((hal_rspdif_dev_struct *)p_struct)->rspdif_irq.rspdif_cx_cpl_handle    = NULL;
        ((hal_rspdif_dev_struct *)p_struct)->rspdif_irq.rspdif_rx_half_handle   = NULL;
        ((hal_rspdif_dev_struct *)p_struct)->rspdif_irq.rspdif_cx_half_handle   = NULL;
        ((hal_rspdif_dev_struct *)p_struct)->rspdif_irq.rspdif_error_handle     = NULL;
        ((hal_rspdif_dev_struct *)p_struct)->p_rx_buffer                        = NULL;
        ((hal_rspdif_dev_struct *)p_struct)->rx_xfer_size                       = 0U;
        ((hal_rspdif_dev_struct *)p_struct)->rx_xfer_count                      = 0U;
        ((hal_rspdif_dev_struct *)p_struct)->p_cs_buffer                        = NULL;
        ((hal_rspdif_dev_struct *)p_struct)->cs_xfer_size                       = 0U;
        ((hal_rspdif_dev_struct *)p_struct)->cs_xfer_count                      = 0U;
        ((hal_rspdif_dev_struct *)p_struct)->error_state                        = HAL_RSPDIF_ERROR_NONE;
        ((hal_rspdif_dev_struct *)p_struct)->state                              = HAL_RSPDIF_STATE_NONE;
        ((hal_rspdif_dev_struct *)p_struct)->mutex                              = HAL_MUTEX_UNLOCKED;
        ((hal_rspdif_dev_struct *)p_struct)->priv                               = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefined");
        ret = HAL_ERR_VAL;
        break;
    }

    /* return function state */
    return ret;
}

/*!
    \brief      initialize the parameters of RSPDIF structure with the default values
    \param[in]  none
    \param[out] rspdif_struct: the initialized structure hal_rspdif_init_struct pointer
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_rspdif_struct_para_init(hal_rspdif_init_struct *rspdif_struct)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == rspdif_struct) {
        HAL_DEBUGE("pointer [rspdif_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    /* configure the RSPDIF structure with the default values */
    rspdif_struct->input_sel          = RSPDIF_INPUT_IN0;
    rspdif_struct->max_retrie         = RSPDIF_MAXRETRIES_15;
    rspdif_struct->wait_activity      = RSPDIF_WAIT_FOR_ACTIVITY_ON;
    rspdif_struct->channel_sel        = RSPDIF_CHANNEL_A;
    rspdif_struct->sample_format      = RSPDIF_DATAFORMAT_MSB;
    rspdif_struct->sound_mode         = RSPDIF_STEREOMODE_ENABLE;
    rspdif_struct->pre_type           = RSPDIF_PREAMBLE_TYPE_MASK_OFF;
    rspdif_struct->channel_status_bit = RSPDIF_CHANNEL_STATUS_MASK_OFF;
    rspdif_struct->validity_bit       = RSPDIF_VALIDITY_MASK_OFF;
    rspdif_struct->parity_error_bit   = RSPDIF_PERROR_MASK_OFF;
    rspdif_struct->symbol_clk         = RSPDIF_SYMBOL_CLK_OFF;
    rspdif_struct->bak_symbol_clk     = RSPDIF_BACKUP_SYMBOL_CLK_OFF;

    return HAL_ERR_NONE;
}

/*!
    \brief      enable RSPDIF and start the conversion of routine sequence with DMA
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  src_addr: the source memory buffer address
    \param[in]  dst_addr: the destination memory buffer address
    \param[in]  length: the number of data to be transferred from source to destination
    \param[in]  p_user_func: RSPDIF DMA transfer complete interrupt callback function structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_HARDWARE,HAL_ERR_BUSY details refer to
   gd32h7xx_hal.h
*/
int32_t hal_rspdif_start_dma(hal_rspdif_dev_struct *rspdif_dev, uint32_t *src_addr, uint32_t *dst_addr, uint16_t length, \
                             hal_rspdif_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;
    hal_dma_irq_struct dma_irq;
      /* if rspdif enabled */
    uint32_t reg = 0x00000000U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == rspdif_dev) || (NULL == src_addr)) {
        HAL_DEBUGE("pointer [rspdif_dev] or [src_addr] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(65535U > length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock RSPDIF */
    HAL_LOCK(rspdif_dev);

    if(HAL_RSPDIF_STATE_BUSY == rspdif_dev->state) {
        HAL_UNLOCK(rspdif_dev);
        ret = HAL_ERR_BUSY;
    }

    rspdif_dev->state = HAL_RSPDIF_STATE_BUSY;
    hal_dma_struct_init(HAL_DMA_IRQ_STRUCT, &dma_irq);
    RSPDIF_CTL &= ~(uint32_t)RSPDIF_CTL_RXCFG;
    RSPDIF_CTL |= (uint32_t)RSPDIF_STATE_RCV;

    reg = RSPDIF_CTL;

    if(RSPDIF_STATE_IDLE != (reg & RSPDIF_STATE_RCV)) {
        rspdif_dev->error_state = HAL_RSPDIF_ERROR_NONE;

        /* clear all RSPDIF flag */
        hals_rspdif_flag_clear((uint32_t)RSPDIF_FLAG_PERR | (uint32_t)RSPDIF_FLAG_RXORERR | \
                               (uint32_t)RSPDIF_FLAG_RBNE | (uint32_t)RSPDIF_FLAG_CBNE);

        /* set the DMA transfer complete callback */
        dma_irq.full_finish_handle         = _rspdif_dma_full_transfer_complete;
        dma_irq.half_finish_handle         = _rspdif_dma_half_transfer_complete;
        dma_irq.error_handle               = _rspdif_dma_error;

        rspdif_dev->rspdif_rx_cpl_callback  = NULL;
        rspdif_dev->rspdif_rx_half_callback = NULL;
        rspdif_dev->rspdif_cx_cpl_callback  = NULL;
        rspdif_dev->rspdif_cx_half_callback = NULL;
        rspdif_dev->rspdif_error_callback   = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->rspdif_rx_cpl_callback) {
                rspdif_dev->rspdif_rx_cpl_callback = (void*)p_user_func->rspdif_rx_cpl_callback;
            }

            if(NULL != p_user_func->rspdif_rx_half_callback) {
                rspdif_dev->rspdif_rx_half_callback = (void*)p_user_func->rspdif_rx_half_callback;
            }

            if(NULL != p_user_func->rspdif_cx_cpl_callback) {
                rspdif_dev->rspdif_cx_cpl_callback = (void*)p_user_func->rspdif_cx_cpl_callback;
            }

            if(NULL != p_user_func->rspdif_cx_half_callback) {
                rspdif_dev->rspdif_cx_half_callback = (void*)p_user_func->rspdif_cx_half_callback;
            }

            if(NULL != p_user_func->rspdif_error_callback) {
                rspdif_dev->rspdif_error_callback = (void*)p_user_func->rspdif_error_callback;
            }
        }

        /* enable RSPDIF DMA mode */
        hals_rspdif_dma_enable();
        /* start the DMA channel */
        hal_dma_start_interrupt(rspdif_dev->p_dma_rspdif, (uint32_t)src_addr, (uint32_t)dst_addr, length, &dma_irq);
    } else {
        /* unlock RSPDIF */
        HAL_UNLOCK(rspdif_dev);
        ret = HAL_ERR_HARDWARE;
    }

    /* unlock RSPDIF */
    HAL_UNLOCK(rspdif_dev);

    return ret;
}

/*!
    \brief      stop RSPDIF
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rspdif_stop(hal_rspdif_dev_struct *rspdif_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == rspdif_dev) {
        HAL_DEBUGE("pointer [rspdif_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    /* disable RSPDIF */
    hal_rspdif_disable();
    rspdif_dev->state = HAL_RSPDIF_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      stop RSPDIF DMA request
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rspdif_stop_dma(hal_rspdif_dev_struct *rspdif_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == rspdif_dev) {
        HAL_DEBUGE("pointer [rspdif_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    HAL_LOCK(rspdif_dev);
    /* disable RSPDIF DMA mode */
    hals_rspdif_dma_disable();
    hals_rspdif_control_buffer_dma_disable();
    /* disable dma interrupt */
    hal_dma_stop_interrupt(rspdif_dev->p_dma_rspdif);
    /* disable RSPDIF */
    hal_rspdif_disable();
    rspdif_dev->state = HAL_RSPDIF_STATE_READY;
    HAL_UNLOCK(rspdif_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop RSPDIF DMA request interrupt
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rspdif_stop_dma_interrupt(hal_rspdif_dev_struct *rspdif_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == rspdif_dev) {
        HAL_DEBUGE("pointer [rspdif_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    HAL_LOCK(rspdif_dev);
    /* disable interrupt */
    hal_rspdif_stop_interrupt(rspdif_dev);
    /* disable RSPDIF DMA mode */
    hals_rspdif_dma_disable();
    /* disable dma interrupt */
    hal_dma_stop_interrupt(rspdif_dev->p_dma_rspdif);
    /* disable RSPDIF */
    hal_rspdif_disable();
    rspdif_dev->state = HAL_RSPDIF_STATE_READY;
    HAL_UNLOCK(rspdif_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      RSPDIF interrupt handler content function,which is merely used in rspdif_handler
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rspdif_irq(hal_rspdif_dev_struct *rspdif_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == rspdif_dev) {
        HAL_DEBUGE("pointer [rspdif_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check interrupt state bit */
    if(SET == hals_rspdif_interrupt_flag_get((uint16_t)RSPDIF_INT_RBNE)) {
        /* clear interrupt flag */
        hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_RBNE);
        /* RSPDIF in mode data flow reception */
        if(NULL != (rspdif_dev->rspdif_irq.rspdif_rx_cpl_handle)) {
            rspdif_dev->rspdif_irq.rspdif_rx_cpl_handle(rspdif_dev);
        }
    }

    /* check interrupt state bit */
    if(SET == hals_rspdif_interrupt_flag_get((uint16_t)RSPDIF_INT_CBNE)) {
        /* clear interrupt flag */
        hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_CBNE);
        /* RSPDIF in mode control flow reception */
        if(NULL != (rspdif_dev->rspdif_irq.rspdif_cx_cpl_handle)) {
            rspdif_dev->rspdif_irq.rspdif_cx_cpl_handle(rspdif_dev);
        }
    }

    /* check interrupt state bit */
    if(SET == hals_rspdif_interrupt_flag_get((uint16_t)RSPDIF_INT_PERR)) {
        /* clear interrupt flag */
        hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_PERR);
         /* RSPDIF parity error interrupt occurred */
        if(NULL != (rspdif_dev->rspdif_irq.rspdif_error_handle)) {
            rspdif_dev->rspdif_irq.rspdif_error_handle(rspdif_dev);
        }
    }

    /* check interrupt state bit */
    if(SET == hals_rspdif_interrupt_flag_get((uint16_t)RSPDIF_INT_RXORERR)) {
        /* clear interrupt flag */
        hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_RXORERR);
        /* RSPDIF overrun error interrupt occurred */
        if(NULL != (rspdif_dev->rspdif_irq.rspdif_error_handle)) {
            rspdif_dev->rspdif_irq.rspdif_error_handle(rspdif_dev);
        }
    }

    /* check interrupt state bit */
    if(SET == hals_rspdif_interrupt_flag_get((uint16_t)RSPDIF_INT_RXDCERR)) {
        /* clear interrupt flag */
        hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_RXDCERR);
        /* RSPDIF data check error interrupt occurred */
        if(NULL != (rspdif_dev->rspdif_irq.rspdif_error_handle)) {
            rspdif_dev->rspdif_irq.rspdif_error_handle(rspdif_dev);
        }
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function, which will be registered and called when corresponding
                interrupt be triggered
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to RSPDIF interrupt callback functions structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rspdif_irq_handle_set(hal_rspdif_dev_struct *rspdif_dev, hal_rspdif_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == rspdif_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [rspdif_dev] or pointer [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* RSPDIF in mode data flow reception function interrupt handler set */
    if(NULL != p_irq->rspdif_rx_cpl_handle) {
        rspdif_dev->rspdif_irq.rspdif_rx_cpl_handle = p_irq->rspdif_rx_cpl_handle;
        hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_RBNE);
        hals_rspdif_interrupt_enable(RSPDIF_INT_RBNE);
    } else {
        rspdif_dev->rspdif_irq.rspdif_rx_cpl_handle = NULL;
        hals_rspdif_interrupt_disable(RSPDIF_INT_RBNE);
        hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_RBNE);
    }

    /* RSPDIF in mode control flow reception interrupt handler set */
    if(NULL != p_irq->rspdif_cx_cpl_handle) {
        rspdif_dev->rspdif_irq.rspdif_cx_cpl_handle = p_irq->rspdif_cx_cpl_handle;
        hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_CBNE);
        hals_rspdif_interrupt_enable(RSPDIF_INT_CBNE);
    } else {
        rspdif_dev->rspdif_irq.rspdif_cx_cpl_handle = NULL;
        hals_rspdif_interrupt_disable(RSPDIF_INT_CBNE);
        hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_CBNE);
    }

    /* RSPDIF parity error interrupt occurred interrupt handler set */
    if(NULL != p_irq->rspdif_error_handle) {
        rspdif_dev->rspdif_irq.rspdif_error_handle = p_irq->rspdif_error_handle;
        hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_PERR);
        hals_rspdif_interrupt_enable(RSPDIF_INT_PERR);
    } else {
        rspdif_dev->rspdif_irq.rspdif_error_handle = NULL;
        hals_rspdif_interrupt_disable(RSPDIF_INT_PERR);
        hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_PERR);
    }

    /* RSPDIF overrun error interrupt occurred interrupt handler set */
    if(NULL != p_irq->rspdif_error_handle) {
        rspdif_dev->rspdif_irq.rspdif_error_handle = p_irq->rspdif_error_handle;
        hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_RXORERR);
        hals_rspdif_interrupt_enable(RSPDIF_INT_RXORERR);
    } else {
        rspdif_dev->rspdif_irq.rspdif_error_handle = NULL;
        hals_rspdif_interrupt_disable(RSPDIF_INT_RXORERR);
        hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_RXORERR);
    }

    /* RSPDIF data error interrupt occurred interrupt handler set */
    if(NULL != p_irq->rspdif_error_handle) {
        rspdif_dev->rspdif_irq.rspdif_error_handle = p_irq->rspdif_error_handle;
        hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_TMOUTERR | \
                                         (uint16_t)RSPDIF_INT_FLAG_SYNERR | \
                                         (uint16_t)RSPDIF_INT_FLAG_FRERR);
        hals_rspdif_interrupt_enable(RSPDIF_INT_RXDCERR);
    } else {
        rspdif_dev->rspdif_irq.rspdif_error_handle = NULL;
        hals_rspdif_interrupt_disable(RSPDIF_INT_RXDCERR);
        hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_TMOUTERR | \
                                         (uint16_t)RSPDIF_INT_FLAG_SYNERR | \
                                         (uint16_t)RSPDIF_INT_FLAG_FRERR);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt is triggered
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rspdif_irq_handle_all_reset(hal_rspdif_dev_struct *rspdif_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == rspdif_dev) {
        HAL_DEBUGE("pointer [rspdif_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    /* configure interrupt callback function to NULL */
    rspdif_dev->rspdif_irq.rspdif_rx_cpl_handle    = NULL;
    rspdif_dev->rspdif_irq.rspdif_cx_cpl_handle    = NULL;
    rspdif_dev->rspdif_irq.rspdif_rx_half_handle   = NULL;
    rspdif_dev->rspdif_irq.rspdif_cx_half_handle   = NULL;
    rspdif_dev->rspdif_irq.rspdif_error_handle     = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      start RSPDIF with interrupt
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: call back function for user
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rspdif_start_interrupt(hal_rspdif_dev_struct *rspdif_dev, hal_rspdif_irq_user_callback_struct *p_user_func)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == rspdif_dev) || (NULL == p_user_func)) {
        HAL_DEBUGE("pointer [rspdif_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    /* lock RSPDIF */
    HAL_LOCK(rspdif_dev);

    rspdif_dev->rspdif_rx_cpl_callback    = NULL;
    rspdif_dev->rspdif_cx_cpl_callback    = NULL;
    rspdif_dev->rspdif_error_callback     = NULL;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->rspdif_rx_cpl_callback) {
            rspdif_dev->rspdif_rx_cpl_callback          = (void *)p_user_func->rspdif_rx_cpl_callback;
            rspdif_dev->rspdif_irq.rspdif_rx_cpl_handle = (hal_irq_handle_cb)_rspdif_receive_data_handle;
            hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_RBNE);
            hals_rspdif_interrupt_enable(RSPDIF_INT_RBNE);
        } else {
            hals_rspdif_interrupt_disable(RSPDIF_INT_RBNE);
            hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_RBNE);
        }

        if(NULL != p_user_func->rspdif_cx_cpl_callback) {
            rspdif_dev->rspdif_cx_cpl_callback          = (void *)p_user_func->rspdif_cx_cpl_callback;
            rspdif_dev->rspdif_irq.rspdif_cx_cpl_handle = (hal_irq_handle_cb)_rspdif_receive_ctrl_handle;
            hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_CBNE);
            hals_rspdif_interrupt_enable(RSPDIF_INT_CBNE);
        } else {
            hals_rspdif_interrupt_disable(RSPDIF_INT_CBNE);
            hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_CBNE);
        }

        if(NULL != p_user_func->rspdif_error_callback) {
            rspdif_dev->rspdif_error_callback          = (void *)p_user_func->rspdif_error_callback;
            rspdif_dev->rspdif_irq.rspdif_error_handle = (hal_irq_handle_cb)_rspdif_error_handle;
            hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_PERR | (uint16_t)RSPDIF_INT_FLAG_RXORERR);
            hals_rspdif_interrupt_enable(RSPDIF_INT_PERR | RSPDIF_INT_RXORERR);
        } else {
            hals_rspdif_interrupt_disable(RSPDIF_INT_PERR | RSPDIF_INT_RXORERR);
            hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_PERR | (uint16_t)RSPDIF_INT_FLAG_RXORERR);
        }

    }

    /* unlock RSPDIF */
    HAL_UNLOCK(rspdif_dev);

    /* return function state */
    return HAL_ERR_NONE;
}

/*!
    \brief      stop RSPDIF with interrupt
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rspdif_stop_interrupt(hal_rspdif_dev_struct *rspdif_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == rspdif_dev) {
        HAL_DEBUGE("pointer [rspdif_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    HAL_LOCK(rspdif_dev);

    /* configure interrupt callback function to NULL */
    rspdif_dev->rspdif_irq.rspdif_rx_cpl_handle    = NULL;
    rspdif_dev->rspdif_irq.rspdif_cx_cpl_handle    = NULL;
    rspdif_dev->rspdif_irq.rspdif_rx_half_handle   = NULL;
    rspdif_dev->rspdif_irq.rspdif_cx_half_handle   = NULL;
    rspdif_dev->rspdif_irq.rspdif_error_handle     = NULL;

    /* disable the RSPDIF interrupt*/
    hals_rspdif_interrupt_disable(RSPDIF_INT_RBNE);
    hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_RBNE);

    hals_rspdif_interrupt_disable(RSPDIF_INT_CBNE);
    hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_CBNE);

    hals_rspdif_interrupt_disable(RSPDIF_INT_PERR);
    hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_PERR);

    hals_rspdif_interrupt_disable(RSPDIF_INT_RXORERR);
    hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_RXORERR);

    hals_rspdif_interrupt_disable(RSPDIF_INT_SYNDB);
    hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_SYNDB);

    hals_rspdif_interrupt_disable(RSPDIF_INT_SYNDO);
    hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_SYNDO);

    hals_rspdif_interrupt_disable(RSPDIF_INT_RXDCERR);
    hals_rspdif_interrupt_flag_clear((uint16_t)RSPDIF_INT_FLAG_SYNERR);

    HAL_UNLOCK(rspdif_dev);

    /* return function state */
    return HAL_ERR_NONE;
}

/*!
    \brief      RSPDIF receives an amount of data (data flow) in blocking mode
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_data: pointer to data buffer
    \param[in]  size: amount of data to be received
    \param[in]  timeout: timeout duration
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, details refer to
   gd32h7xx_hal.h
*/
int32_t hal_rspdif_data_flow_receive(hal_rspdif_dev_struct *rspdif_dev, uint32_t *p_data, uint16_t size, \
                                     uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;
    uint32_t tick_start;
    uint16_t counter = size;
    uint32_t *p_temp = p_data;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == rspdif_dev) || (NULL == p_data)) {
        HAL_DEBUGE("pointer [rspdif_dev] or [p_data] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(0U >= size) {
        HAL_DEBUGE("parameter [size] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DMA */
    HAL_LOCK(rspdif_dev);
    if(HAL_RSPDIF_STATE_READY == rspdif_dev->state) {
        rspdif_dev->state = HAL_RSPDIF_STATE_BUSY;
        /* clear RXCFG bit */
        RSPDIF_CTL &= ~(uint32_t)RSPDIF_CTL_RXCFG;
        /* start synchronization */
        RSPDIF_CTL |= (uint32_t)RSPDIF_STATE_SYNC;

        /* get tick */
        tick_start = hal_sys_basetick_count_get();

        /* wait until SYNDO bit is set*/
        if(HAL_ERR_NONE != _rspdif_WaitOnFlagUntilTimeout(rspdif_dev, RSPDIF_FLAG_SYNDO, RESET, timeout, tick_start)) {
            ret = HAL_ERR_TIMEOUT;
        }

        /* start reception */
        RSPDIF_CTL |= (uint32_t)RSPDIF_STATE_RCV;

        while(counter > 0U) {
            /* set timeout */
            tick_start = hal_sys_basetick_count_get();

            /* Wait until RBNE flag is set */
            if(HAL_ERR_NONE != _rspdif_WaitOnFlagUntilTimeout(rspdif_dev, RSPDIF_FLAG_RBNE, RESET, timeout, tick_start)) {
                ret = HAL_ERR_TIMEOUT;
            }

            (*p_temp)  = RSPDIF_DATA;
            p_temp++;
            counter--;
        }

        /* RSPDIF ready */
        rspdif_dev->state = HAL_RSPDIF_STATE_READY;
        HAL_UNLOCK(rspdif_dev);
        ret = HAL_ERR_NONE;
    } else {
        HAL_UNLOCK(rspdif_dev);
        ret = HAL_ERR_BUSY;
    }

    return ret;
}

/*!
    \brief      RSPDIF receives an amount of data (control flow) in blocking mode
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_data: pointer to data buffer
    \param[in]  size: amount of data to be received
    \param[in]  timeout: timeout duration
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, details refer to
                gd32h7xx_hal.h
*/
int32_t hal_rspdif_ctrl_flow_receive(hal_rspdif_dev_struct *rspdif_dev, uint32_t *p_data, uint16_t size, \
                                     uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;
    uint32_t tick_start;
    uint16_t counter = size;
    uint32_t *p_temp = p_data;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == rspdif_dev) || (NULL == p_data)) {
        HAL_DEBUGE("pointer [rspdif_dev] or [p_data] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(0U >= size) {
        HAL_DEBUGE("parameter [size] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DMA */
    HAL_LOCK(rspdif_dev);
    if(HAL_RSPDIF_STATE_READY == rspdif_dev->state) {
        rspdif_dev->state = HAL_RSPDIF_STATE_BUSY;
        /* clear RXCFG bit */
        RSPDIF_CTL &= ~(uint32_t)RSPDIF_CTL_RXCFG;

        /* start synchronization */
        RSPDIF_CTL |= (uint32_t)RSPDIF_STATE_SYNC;

        /* get tick */
        tick_start = hal_sys_basetick_count_get();

        /* wait until SYNDO bit is set*/
        if( HAL_ERR_NONE != _rspdif_WaitOnFlagUntilTimeout(rspdif_dev, RSPDIF_FLAG_SYNDO, RESET, timeout, tick_start)) {
            ret = HAL_ERR_TIMEOUT;
        }

        /* start reception */
        RSPDIF_CTL |= (uint32_t)RSPDIF_STATE_RCV;

        while(counter > 0U) {
            /* set timeout */
            tick_start = hal_sys_basetick_count_get();

            /* wait until RBNE flag is set */
            if(HAL_ERR_NONE != _rspdif_WaitOnFlagUntilTimeout(rspdif_dev, RSPDIF_FLAG_RBNE, RESET, timeout, tick_start)) {
                ret = HAL_ERR_TIMEOUT;
            }

            (*p_temp)  = RSPDIF_CHSTAT;
            p_temp++;
            counter--;
        }

        rspdif_dev->state = HAL_RSPDIF_STATE_READY;
        HAL_UNLOCK(rspdif_dev);
        ret = HAL_ERR_NONE;
    } else {
        HAL_UNLOCK(rspdif_dev);
        ret = HAL_ERR_BUSY;
    }

    return ret;
}

/*!
    \brief      RSPDIF receives an amount of data (data flow) in blocking mode with interrupt
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_data: pointer to data buffer
    \param[in]  size: amount of data to be received
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, details refer to
                gd32h7xx_hal.h
*/
int32_t hal_rspdif_data_flow_receive_interrupt(hal_rspdif_dev_struct *rspdif_dev, uint32_t *p_data, uint16_t size)
{
    int32_t ret = HAL_ERR_NONE;
    const hal_rspdif_state_enum temp_state = rspdif_dev->state;
    /* timeout */
    uint32_t count = RSPDIFRX_TIMEOUT_VALUE * (SystemCoreClock / 24U / 1000U);
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == rspdif_dev) || (NULL == p_data)) {
        HAL_DEBUGE("pointer [rspdif_dev] or [p_data] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(0U == size) {
        HAL_DEBUGE("parameter [size] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DMA */
    HAL_LOCK(rspdif_dev);

    if((HAL_RSPDIF_STATE_READY == temp_state) || (HAL_RSPDIF_STATE_BUSY_CX == temp_state)) {
        /* receive data */
        rspdif_dev->p_rx_buffer   = p_data;
        rspdif_dev->rx_xfer_size  = size;
        rspdif_dev->rx_xfer_count = size;
        /* change rspdif state */
        rspdif_dev->state       = HAL_RSPDIF_STATE_BUSY_RX;
        rspdif_dev->error_state = HAL_RSPDIF_ERROR_NONE;
        /* enable interrupt */
        hals_rspdif_interrupt_enable(RSPDIF_INT_RBNE | RSPDIF_INT_PERR | RSPDIF_INT_RXORERR);

        if(RSPDIF_STATE_RCV != (RSPDIF_CTL & RSPDIF_STATE_RCV)) {
            /* clear RXCFG bit */
            RSPDIF_CTL &= ~(uint32_t)RSPDIF_CTL_RXCFG;

            /* start synchronization */
            RSPDIF_CTL |= (uint32_t)RSPDIF_STATE_SYNC;

            /* wait until SYNDO bit is set */
            do {
                if(0U == count) {
                    rspdif_dev->state = HAL_RSPDIF_STATE_READY;

                    HAL_UNLOCK(rspdif_dev);
                    ret = HAL_ERR_TIMEOUT;
                }
                count--;
            } while(RESET == hals_rspdif_flag_get(RSPDIF_FLAG_SYNDO));
        }

        /* start reception */
        RSPDIF_CTL |= (uint32_t)RSPDIF_STATE_RCV;

        HAL_UNLOCK(rspdif_dev);
        ret = HAL_ERR_NONE;
    } else {
        HAL_UNLOCK(rspdif_dev);
        ret = HAL_ERR_BUSY;
    }

    return ret;
}

/*!
    \brief      RSPDIF receives an amount of data (control flow) in blocking mode
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_data: pointer to data buffer
    \param[in]  size: amount of data to be received
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, details refer to
   gd32h7xx_hal.h
*/
int32_t hal_rspdif_ctrl_flow_receive_interrupt(hal_rspdif_dev_struct *rspdif_dev, uint32_t *p_data, uint16_t size)
{
    int32_t ret = HAL_ERR_NONE;
    const hal_rspdif_state_enum temp_state = rspdif_dev->state;
    /* timeout */
    uint32_t count = RSPDIFRX_TIMEOUT_VALUE * (SystemCoreClock / 24U / 1000U);
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == rspdif_dev) || (NULL == p_data)) {
        HAL_DEBUGE("pointer [rspdif_dev] or [p_data] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(0U == size) {
        HAL_DEBUGE("parameter [size] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DMA */
    HAL_LOCK(rspdif_dev);

    if((HAL_RSPDIF_STATE_READY == temp_state) || (HAL_RSPDIF_STATE_BUSY_RX == temp_state)) {
        /* receive data */
        rspdif_dev->p_cs_buffer   = p_data;
        rspdif_dev->cs_xfer_size  = size;
        rspdif_dev->cs_xfer_count = size;
        /* change rspdif state */
        rspdif_dev->state       = HAL_RSPDIF_STATE_BUSY_CX;
        rspdif_dev->error_state = HAL_RSPDIF_ERROR_NONE;
        /* enable interrupt */
        hals_rspdif_interrupt_enable(RSPDIF_INT_CBNE | RSPDIF_INT_PERR | RSPDIF_INT_RXORERR);

        if(RSPDIF_STATE_RCV != (RSPDIF_CTL & RSPDIF_STATE_RCV)) {
            /* clear RXCFG bit */
            RSPDIF_CTL &= ~(uint32_t)RSPDIF_CTL_RXCFG;

            /* start synchronization */
            RSPDIF_CTL |= (uint32_t)RSPDIF_STATE_SYNC;

            /* wait until SYNDO bit is set */
            do {
                if(0U == count) {
                    rspdif_dev->state = HAL_RSPDIF_STATE_READY;

                    HAL_UNLOCK(rspdif_dev);
                    ret = HAL_ERR_TIMEOUT;
                }
                count--;
            } while(RESET == hals_rspdif_flag_get(RSPDIF_FLAG_SYNDO));
        }

        /* start reception */
        RSPDIF_CTL |= (uint32_t)RSPDIF_STATE_RCV;

        HAL_UNLOCK(rspdif_dev);
        ret = HAL_ERR_NONE;
    } else {
        HAL_UNLOCK(rspdif_dev);
        ret = HAL_ERR_BUSY;
    }

    return ret;
}

/*!
    \brief      RSPDIF receives an amount of data (data flow) in blocking mode with dma
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_data: pointer to data buffer
    \param[in]  size: amount of data to be received
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, details refer to
   gd32h7xx_hal.h
*/
int32_t hal_rspdif_data_flow_receive_dma(hal_rspdif_dev_struct *rspdif_dev, uint32_t *p_data, uint16_t size)
{
    int32_t ret = HAL_ERR_NONE;
    const hal_rspdif_state_enum temp_state = rspdif_dev->state;
    /* timeout */
    uint32_t count = RSPDIFRX_TIMEOUT_VALUE * (SystemCoreClock / 24U / 1000U);
    hal_dma_irq_struct dma_irq;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == rspdif_dev) || (NULL == p_data)) {
        HAL_DEBUGE("pointer [rspdif_dev] or [p_data] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(0U == size) {
        HAL_DEBUGE("parameter [size] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DMA */
    HAL_LOCK(rspdif_dev);

    if((HAL_RSPDIF_STATE_READY == temp_state) || (HAL_RSPDIF_STATE_BUSY_CX == temp_state)) {
        /* receive data */
        rspdif_dev->p_rx_buffer   = p_data;
        rspdif_dev->rx_xfer_size  = size;
        rspdif_dev->rx_xfer_count = size;
        /* change rspdif state */
        rspdif_dev->state       = HAL_RSPDIF_STATE_BUSY_RX;
        rspdif_dev->error_state = HAL_RSPDIF_ERROR_NONE;
        /* set handler */
        dma_irq.full_finish_handle = _rspdif_dma_full_transfer_complete;
        dma_irq.half_finish_handle = _rspdif_dma_half_transfer_complete;
        dma_irq.error_handle       = _rspdif_dma_error;
        /* enable dma */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(rspdif_dev->p_dma_rspdif, (uint32_t)&RSPDIF_DATA, \
                                                   (uint32_t)rspdif_dev->p_rx_buffer, size, &dma_irq)) {
            rspdif_dev->state = HAL_RSPDIF_STATE_ERROR;
            HAL_UNLOCK(rspdif_dev);
            ret = HAL_ERR_BUSY;
        }

        hals_rspdif_dma_enable();

        if(RSPDIF_STATE_RCV != (RSPDIF_CTL & RSPDIF_STATE_RCV)) {
            /* clear RXCFG bit */
            RSPDIF_CTL &= ~(uint32_t)RSPDIF_CTL_RXCFG;

            /* start synchronization */
            RSPDIF_CTL |= (uint32_t)RSPDIF_STATE_SYNC;

            /* Wait until SYNDO flag is set */
            do
            {
                if (0U == count)
                {
                    /* set RSPDIF state*/
                    rspdif_dev->state = HAL_RSPDIF_STATE_READY;

                    /* set RSPDIF error*/
                    rspdif_dev->error_state = HAL_RSPDIF_ERROR_DMA;

                    HAL_UNLOCK(rspdif_dev);
                    ret = HAL_ERR_TIMEOUT;
                }
                count--;
            } while (RESET == hals_rspdif_flag_get(RSPDIF_FLAG_SYNDO));

            /* start reception */
            RSPDIF_CTL |= (uint32_t)RSPDIF_STATE_RCV;
        }

        HAL_UNLOCK(rspdif_dev);
        ret = HAL_ERR_NONE;
    } else {
        HAL_UNLOCK(rspdif_dev);
        ret = HAL_ERR_BUSY;
    }

    return ret;
}

/*!
    \brief      RSPDIF receives an amount of data (control flow) in blocking mode
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_data: pointer to data buffer
    \param[in]  size: amount of data to be received
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, details refer to
   gd32h7xx_hal.h
*/
int32_t hal_rspdif_ctrl_flow_receive_dma(hal_rspdif_dev_struct *rspdif_dev, uint32_t *p_data, uint16_t size)
{
    int32_t ret = HAL_ERR_NONE;
    /* timeout */
    uint32_t count = RSPDIFRX_TIMEOUT_VALUE * (SystemCoreClock / 24U / 1000U);
        const hal_rspdif_state_enum temp_state = rspdif_dev->state;
    hal_dma_irq_struct dma_irq;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == rspdif_dev) || (NULL == p_data)) {
        HAL_DEBUGE("pointer [rspdif_dev] or [p_data] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(0U == size) {
        HAL_DEBUGE("parameter [size] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DMA */
    HAL_LOCK(rspdif_dev);
    if((HAL_RSPDIF_STATE_READY == temp_state) || (HAL_RSPDIF_STATE_BUSY_RX == temp_state)) {
        /* receive data */
        rspdif_dev->p_cs_buffer   = p_data;
        rspdif_dev->cs_xfer_size  = size;
        rspdif_dev->cs_xfer_count = size;
        /* change rspdif state */
        rspdif_dev->state       = HAL_RSPDIF_STATE_BUSY_CX;
        rspdif_dev->error_state = HAL_RSPDIF_ERROR_NONE;
        /* set handler */
        dma_irq.full_finish_handle = _rspdif_dma_full_transfer_complete;
        dma_irq.half_finish_handle = _rspdif_dma_half_transfer_complete;
        dma_irq.error_handle       = _rspdif_dma_error;
        /* enable dma */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(rspdif_dev->p_dma_rspdif, (uint32_t)&RSPDIF_CHSTAT, \
                                                   (uint32_t)rspdif_dev->p_cs_buffer, size, &dma_irq)) {
            rspdif_dev->state = HAL_RSPDIF_STATE_ERROR;
            HAL_UNLOCK(rspdif_dev);
            ret = HAL_ERR_BUSY;
        }

        hals_rspdif_control_buffer_dma_enable();

        if(RSPDIF_STATE_RCV != (RSPDIF_CTL & RSPDIF_STATE_RCV)) {
            /* clear RXCFG bit */
            RSPDIF_CTL &= ~(uint32_t)RSPDIF_CTL_RXCFG;

            /* start synchronization */
            RSPDIF_CTL |= (uint32_t)RSPDIF_STATE_SYNC;

            /* Wait until SYNDO flag is set */
            do
            {
                if (0U == count)
                {
                    /* set RSPDIF state*/
                    rspdif_dev->state = HAL_RSPDIF_STATE_READY;

                    /* set RSPDIF error*/
                    rspdif_dev->error_state = HAL_RSPDIF_ERROR_DMA;

                    HAL_UNLOCK(rspdif_dev);
                    ret = HAL_ERR_TIMEOUT;
                }
                count--;
            } while (RESET == hals_rspdif_flag_get(RSPDIF_FLAG_SYNDO));

            /* start reception */
            RSPDIF_CTL |= (uint32_t)RSPDIF_STATE_RCV;
        }

        HAL_UNLOCK(rspdif_dev);
        ret = HAL_ERR_NONE;
    } else {
        HAL_UNLOCK(rspdif_dev);
        ret = HAL_ERR_BUSY;
    }

    return ret;
}

/*!
    \brief      get the rspdif state
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     state: <hal_rspdif_state_enum> refer to gd32h7xx_hal_rspdif.h
*/
hal_rspdif_state_enum hal_rspdif_state_get(hal_rspdif_dev_struct *rspdif_dev)
{
    return (hal_rspdif_state_enum)rspdif_dev->state;
}

/*!
    \brief      get the rspdif error state
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     tate: <hal_rspdif_error_enum> refer to gd32h7xx_hal_rspdif.h
*/
hal_rspdif_error_enum hal_rspdif_error_get(hal_rspdif_dev_struct *rspdif_dev)
{
    return (hal_rspdif_error_enum)rspdif_dev->error_state;
}

/*!
    \brief      RSPDIF read data
    \param[in]  none
    \param[out] data_struct: RSPDIF data structure and the member values are shown as below:
                  format         : RSPDIF_DATAFORMAT_LSB, RSPDIF_DATAFORMAT_MSB,RSPDIF_DATAFORMAT_32BITS
                  preamble       : RSPDIF_PREAMBLE_NONE,RSPDIF_PREAMBLE_B,RSPDIF_PREAMBLE_M,RSPDIF_PREAMBLE_W
                  channel_status : 0 or 1
                  user_bit       : 0 or 1
                  validity       : 0 or 1
                  parity_err     : 0 or 1
                  data0          : 0 ~ 65535
                  data1          : 0 ~ 65535
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rspdif_data_read(hal_rspdif_data_struct *data_struct)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == data_struct) {
        HAL_DEBUGE("pointer [data_struct] address is invalid.");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    /* get data format */
    data_struct->format = RSPDIF_CTL & RSPDIF_CTL_RXDF;

    switch(data_struct->format) {
    /* data format 0 */
    case RSPDIF_DATAFORMAT_LSB:
        /* the preamble type */
        data_struct->preamble = (uint32_t)((RSPDIF_DATA & RSPDIF_DATA_F0_PREF) >> RSPDIF_DATA_F0_PREF_OFFSET);
        /* channel status bit */
        data_struct->channel_status = (uint32_t)((RSPDIF_DATA & RSPDIF_DATA_F0_C) >> RSPDIF_DATA_F0_C_OFFSET);
        /* user bit */
        data_struct->user_bit = (uint32_t)((RSPDIF_DATA & RSPDIF_DATA_F0_U) >> RSPDIF_DATA_F0_U_OFFSET);
        /* validity bit */
        data_struct->validity = (uint32_t)((RSPDIF_DATA & RSPDIF_DATA_F0_V) >> RSPDIF_DATA_F0_V_OFFSET);
        /* parity error bit */
        data_struct->parity_err = (uint32_t)((RSPDIF_DATA & RSPDIF_DATA_F0_P) >> RSPDIF_DATA_F0_P_OFFSET);
        /* data value 0 */
        data_struct->data0 = (uint32_t)(RSPDIF_DATA & RSPDIF_DATA_F0_DATA0);
        /* data value 1 */
        data_struct->data1 = 0U;
        break;
    /* data format 1 */
    case RSPDIF_DATAFORMAT_MSB:
        /* the preamble type */
        data_struct->preamble = (uint32_t)((RSPDIF_DATA & RSPDIF_DATA_F1_PREF) >> RSPDIF_DATA_F1_PREF_OFFSET);
        /* channel status bit */
        data_struct->channel_status = (uint32_t)((RSPDIF_DATA & RSPDIF_DATA_F1_C) >> RSPDIF_DATA_F1_C_OFFSET);
        /* user bit */
        data_struct->user_bit = (uint32_t)((RSPDIF_DATA & RSPDIF_DATA_F1_U) >> RSPDIF_DATA_F1_U_OFFSET);
        /* validity bit */
        data_struct->validity = (uint32_t)((RSPDIF_DATA & RSPDIF_DATA_F1_V) >> RSPDIF_DATA_F1_V_OFFSET);
        /* parity error bit */
        data_struct->parity_err = (uint32_t)(RSPDIF_DATA & RSPDIF_DATA_F1_P);
        /* data value 0 */
        data_struct->data0 = (uint32_t)((RSPDIF_DATA & RSPDIF_DATA_F1_DATA0) >> RSPDIF_DATA_F1_DATA_OFFSET);
        /* data value 1 */
        data_struct->data1 = 0U;
        break;
    /* data format 2 */
    case RSPDIF_DATAFORMAT_32BITS:
        /* the preamble type */
        data_struct->preamble = 0U;
        /* channel status bit */
        data_struct->channel_status = 0U;
        /* user bit */
        data_struct->user_bit = 0U;
        /* validity bit */
        data_struct->validity = 0U;
        /* parity error bit */
        data_struct->parity_err = 0U;
        /* data value 0 */
        data_struct->data0 = (uint32_t)(RSPDIF_DATA & RSPDIF_DATA_F2_DATA1);
        /* data value 1 */
        data_struct->data1 = (uint32_t)(RSPDIF_DATA & RSPDIF_DATA_F2_DATA2);
        break;
    default:
        break;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the RSPDIF receive data format
    \param[in]  rspdif_format_struct:the RSPDIF data format structure and the member values are shown as below:
                only one parameter can be selected which is shown as below:
      \arg        RSPDIF_DATAFORMAT_LSB: audio data is right aligned (LSB)
      \arg        RSPDIF_DATAFORMAT_MSB: audio data is left aligned (MSB)
      \arg        RSPDIF_DATAFORMAT_32BITS: which packs two 16-bit audio data into one 32-bit data
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_ABORT, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rspdif_dataformat_set(hal_rspdif_dataformat_struct *rspdif_format_struct)
{
    int32_t ret = HAL_ERR_NONE;
    uint32_t reg = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == rspdif_format_struct) {
        HAL_DEBUGE("pointer [rspdif_format_struct] address is invalid.");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* reset the old RSPDIF CTL configuration */
    reg = RSPDIF_CTL;

    if ((RSPDIF_STATE_RCV == (reg & RSPDIF_STATE_RCV)) && \
        (((reg & RSPDIF_CTL_RXDF) != rspdif_format_struct->DataFormat) || \
        ((reg & RSPDIF_CTL_RXSTEOMEN) != rspdif_format_struct->StereoMode))) {
        ret = HAL_ERR_ABORT;
    }

    reg &= ~(uint32_t)(RSPDIF_CTL_RXDF | RSPDIF_CTL_RXSTEOMEN | RSPDIF_CTL_CUNCPEN | \
                        RSPDIF_CTL_PTNCPEN | RSPDIF_CTL_PNCPEN | RSPDIF_CTL_VNCPEN);

    /* configure the new data format */
    reg |= (rspdif_format_struct->DataFormat | rspdif_format_struct->StereoMode | \
                rspdif_format_struct->PreambleTypeMask | rspdif_format_struct->ChannelStatusMask | \
                rspdif_format_struct->ValidityBitMask | rspdif_format_struct->ParityErrorMask);

    RSPDIF_CTL = (uint32_t)reg;

    return ret;
}

/*!
    \brief      enable RSPDIF symbol clock
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_rspdif_symbol_clock_enable(void)
{
    RSPDIF_CTL |= (uint32_t)RSPDIF_CTL_SCKEN;
}

/*!
    \brief      disable RSPDIF symbol clock
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_rspdif_symbol_clock_disable(void)
{
    RSPDIF_CTL &= ~(uint32_t)RSPDIF_CTL_SCKEN;
}

/*!
    \brief      enable RSPDIF backup symbol clock
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_rspdif_backup_symbol_clock_enable(void)
{
    RSPDIF_CTL |= (uint32_t)RSPDIF_CTL_BKSCKEN;
}

/*!
    \brief      disable RSPDIF backup symbol clock
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_rspdif_backup_symbol_clock_disable(void)
{
    RSPDIF_CTL &= ~(uint32_t)RSPDIF_CTL_BKSCKEN;
}

/*!
    \brief      enable the RSPDIF receiver DMA
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_rspdif_dma_enable(void)
{
    RSPDIF_CTL |= RSPDIF_CTL_DMAREN;
}

/*!
    \brief      disable the RSPDIF receiver DMA
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_rspdif_dma_disable(void)
{
    RSPDIF_CTL &= ~RSPDIF_CTL_DMAREN;
}

/*!
    \brief      enable the RSPDIF control buffer DMA
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_rspdif_control_buffer_dma_enable(void)
{
    RSPDIF_CTL |= RSPDIF_CTL_DMACBEN;
}

/*!
    \brief      disable the RSPDIF control buffer DMA
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_rspdif_control_buffer_dma_disable(void)
{
    RSPDIF_CTL &= ~RSPDIF_CTL_DMACBEN;
}

/*!
    \brief      get duration of 5 symbols counted using rspdif_ck
    \param[in]  none
    \param[out] none
    \retval     uint16_t: 0x0000-0xFFFF
*/
uint16_t hals_rspdif_duration_of_symbols_get(void)
{
    return ((uint16_t)((RSPDIF_STAT & RSPDIF_STAT_CKCNT5) >> RSPDIF_CKCNT5_OFFSET));
}

/*!
    \brief      get user data information
    \param[in]  none
    \param[out] none
    \retval     uint16_t: 0x0000-0xFFFF
*/
uint16_t hals_rspdif_user_data_get(void)
{
    return ((uint16_t)(RSPDIF_CHSTAT & RSPDIF_CHSTAT_USER));
}

/*!
    \brief      get channel status information
    \param[in]  none
    \param[out] none
    \retval     uint8_t: 0x00-0xFF
*/
uint8_t hals_rspdif_channel_status_get(void)
{
    return ((uint8_t)((RSPDIF_CHSTAT & RSPDIF_CHSTAT_CHS) >> 16U));
}

/*!
    \brief      get start of block
    \param[in]  none
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_rspdif_start_block_status_get(void)
{
    FlagStatus ret = RESET;

    if(RESET != (RSPDIF_CHSTAT & RSPDIF_CHSTAT_SOB)) {
        ret = SET;
    } else {
        ret = RESET;
    }

    return ret;
}

/*!
    \brief      get threshold low estimation
    \param[in]  none
    \param[out] none
    \retval     uint16_t: 0x0000-0x1FFF
*/
uint16_t hals_rspdif_low_threshold_get(void)
{
    return ((uint16_t)((RSPDIF_DTH & RSPDIF_DTH_THLO) >> 16U));
}

/*!
    \brief      get threshold high estimation
    \param[in]  none
    \param[out] none
    \retval     uint16_t: 0x0000-0x1FFF
*/
uint16_t hals_rspdif_high_threshold_get(void)
{
    return ((uint16_t)(RSPDIF_DTH & RSPDIF_DTH_THHI));
}

/*!
    \brief      get RSPDIF flag status
    \param[in]  flag: RSPDIF flag status
                only one parameter can be selected which is shown as below:
      \arg        RSPDIF_FLAG_RBNE     : RSPDIF RX buffer is not empty
      \arg        RSPDIF_FLAG_CBNE     : RSPDIF RX control buffer is not empty
      \arg        RSPDIF_FLAG_PERR     : RSPDIF parity error
      \arg        RSPDIF_FLAG_RXORERR  : RSPDIF RX overrun error
      \arg        RSPDIF_FLAG_SYNDB    : RSPDIF synchronization block detected
      \arg        RSPDIF_FLAG_SYNDO    : RSPDIF synchronization done
      \arg        RSPDIF_FLAG_FRERR    : RSPDIF frame error
      \arg        RSPDIF_FLAG_SYNERR   : RSPDIF synchronization error
      \arg        RSPDIF_FLAG_TMOUTERR : RSPDIF time out error
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_rspdif_flag_get(uint32_t flag)
{
    FlagStatus ret = RESET;

    if(RESET != (RSPDIF_STAT & flag)) {
        ret = SET;
    } else {
        ret = RESET;
    }

    return ret;
}

/*!
    \brief      clear RSPDIF flag
    \param[in]  flag: RSPDIF flag status
                one or more parameters can be selected which is shown as below:
      \arg        RSPDIF_FLAG_PERR    : RSPDIF parity error
      \arg        RSPDIF_FLAG_RXORERR : RSPDIF RX overrun error
      \arg        RSPDIF_FLAG_SYNDB   : RSPDIF synchronization block detected
      \arg        RSPDIF_FLAG_SYNDO   : RSPDIF synchronization done
    \param[out] none
    \retval     none
*/
void hals_rspdif_flag_clear(uint32_t flag)
{
    RSPDIF_STATC |= flag;
}

/*!
    \brief      enable RSPDIF interrupt
    \param[in]  interrupt: RSPDIF interrupt
                one or more parameters can be selected which is shown as below:
      \arg        RSPDIF_INT_RBNE   : RSPDIF RX buffer is not empty interrupt
      \arg        RSPDIF_INT_CBNE   : RSPDIF RX control buffer is not empty interrupt
      \arg        RSPDIF_INT_PERR   : RSPDIF parity error interrupt
      \arg        RSPDIF_INT_RXORERR: RSPDIF RX overrun error interrupt
      \arg        RSPDIF_INT_SYNDB  : RSPDIF synchronization block detected interrupt
      \arg        RSPDIF_INT_SYNDO  : RSPDIF synchronization done interrupt
      \arg        RSPDIF_INT_RXDCERR: RSPDIF data decoding error interrupt
    \param[out] none
    \retval     none
*/
void hals_rspdif_interrupt_enable(uint32_t interrupt)
{
    RSPDIF_INTEN |= interrupt;
}

/*!
    \brief      disable RSPDIF interrupt
    \param[in]  interrupt: RSPDIF interrupt
                one or more parameters can be selected which is shown as below:
      \arg        RSPDIF_INT_RBNE   : RSPDIF RX buffer is not empty interrupt
      \arg        RSPDIF_INT_CBNE   : RSPDIF RX control buffer is not empty interrupt
      \arg        RSPDIF_INT_PERR   : RSPDIF parity error interrupt
      \arg        RSPDIF_INT_RXORERR: RSPDIF RX overrun error interrupt
      \arg        RSPDIF_INT_SYNDB  : RSPDIF synchronization block detected interrupt
      \arg        RSPDIF_INT_SYNDO  : RSPDIF synchronization done interrupt
      \arg        RSPDIF_INT_RXDCERR: RSPDIF data decoding error interrupt
    \param[out] none
    \retval     none
*/
void hals_rspdif_interrupt_disable(uint32_t interrupt)
{
    RSPDIF_INTEN &= ~interrupt;
}

/*!
    \brief      get RSPDIF interrupt flag status
    \param[in]  int_flag: RSPDIF interrupt flag status
                only one parameter can be selected which is shown as below:
      \arg        RSPDIF_INT_FLAG_RBNE     : RSPDIF RX buffer is not empty interrupt flag
      \arg        RSPDIF_INT_FLAG_CBNE     : RSPDIF RX control buffer is not empty interrupt flag
      \arg        RSPDIF_INT_FLAG_PERR     : RSPDIF parity error interrupt flag
      \arg        RSPDIF_INT_FLAG_RXORERR  : RSPDIF RX overrun error interrupt flag
      \arg        RSPDIF_INT_FLAG_SYNDB    : RSPDIF synchronization block detected interrupt flag
      \arg        RSPDIF_INT_FLAG_SYNDO    : RSPDIF synchronization done interrupt flag
      \arg        RSPDIF_INT_FLAG_FRERR    : RSPDIF frame error interrupt flag
      \arg        RSPDIF_INT_FLAG_SYNERR   : RSPDIF synchronization error interrupt flag
      \arg        RSPDIF_INT_FLAG_TMOUTERR : RSPDIF time out error interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_rspdif_interrupt_flag_get(uint32_t int_flag)
{
    FlagStatus ret = RESET;
    uint32_t reg1 = RSPDIF_STAT;
    uint32_t reg2 = RSPDIF_INTEN;

    switch(int_flag) {
    /* RSPDIF RX buffer is not empty interrupt */
    case RSPDIF_INT_FLAG_RBNE:
        reg1 = reg1 & RSPDIF_STAT_RBNE;
        reg2 = reg2 & RSPDIF_INTEN_RBNEIE;
        break;
    /* RSPDIF RX control buffer is not empty interrupt */
    case RSPDIF_INT_FLAG_CBNE:
        reg1 = reg1 & RSPDIF_STAT_CBNE;
        reg2 = reg2 & RSPDIF_INTEN_CBNEIE;
        break;
    /* RSPDIF parity error interrupt */
    case RSPDIF_INT_FLAG_PERR:
        reg1 = reg1 & RSPDIF_STAT_PERR;
        reg2 = reg2 & RSPDIF_INTEN_PERRIE;
        break;
    /* RSPDIF RX overrun error interrupt */
    case RSPDIF_INT_FLAG_RXORERR:
        reg1 = reg1 & RSPDIF_STAT_RXORERR;
        reg2 = reg2 & RSPDIF_INTEN_RXORERRIE;
        break;
    /* RSPDIF synchronization block detected interrupt */
    case RSPDIF_INT_FLAG_SYNDB:
        reg1 = reg1 & RSPDIF_STAT_SYNDB;
        reg2 = reg2 & RSPDIF_INTEN_SYNDBIE;
        break;
    /* RSPDIF synchronization done interrupt */
    case RSPDIF_INT_FLAG_SYNDO:
        reg1 = reg1 & RSPDIF_STAT_SYNDO;
        reg2 = reg2 & RSPDIF_INTEN_SYNDOIE;
        break;
    /* RSPDIF frame error interrupt */
    case RSPDIF_INT_FLAG_FRERR:
        reg1 = reg1 & RSPDIF_STAT_FRERR;
        reg2 = reg2 & RSPDIF_INTEN_RXDCERRIE;
        break;
    /* RSPDIF synchronization error interrupt */
    case RSPDIF_INT_FLAG_SYNERR:
        reg1 = reg1 & RSPDIF_STAT_SYNERR;
        reg2 = reg2 & RSPDIF_INTEN_RXDCERRIE;
        break;
    /* RSPDIF time out error interrupt */
    case RSPDIF_INT_FLAG_TMOUTERR:
        reg1 = reg1 & RSPDIF_STAT_TMOUTERR;
        reg2 = reg2 & RSPDIF_INTEN_RXDCERRIE;
        break;
    default:
        break;
    }

    /*get RSPDIF interrupt flag status */
    if((0U != reg1) && (0U != reg2)) {
        ret = SET;
    } else {
        ret = RESET;
    }

    return ret;
}

/*!
    \brief      clear RSPDIF interrupt flag status
    \param[in]  int_flag: RSPDIF interrupt flag status
                one or more parameters can be selected which is shown as below:
      \arg        RSPDIF_INT_FLAG_PERR    : RSPDIF parity error interrupt flag
      \arg        RSPDIF_INT_FLAG_RXORERR : RSPDIF RX overrun error interrupt flag
      \arg        RSPDIF_INT_FLAG_SYNDB   : RSPDIF synchronization block detected interrupt flag
      \arg        RSPDIF_INT_FLAG_SYNDO   : RSPDIF synchronization done interrupt flag
      \arg        RSPDIF_INT_FLAG_SYNERR  : RSPDIF synchronization error interrupt flag
    \param[out] none
    \retval     none
*/
void hals_rspdif_interrupt_flag_clear(uint32_t int_flag)
{
    RSPDIF_STATC |= int_flag;
}

/*!
    \brief      RSPDIF DMA full transmission complete callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _rspdif_dma_full_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_rspdif_dev_struct *rspdif_dev;
    p_dma      = (hal_dma_dev_struct *)dma;
    rspdif_dev = (hal_rspdif_dev_struct *)(p_dma->p_periph);

    if(0U == (DMA_CHCTL(p_dma->dma_periph, p_dma->channel) & DMA_CHXCTL_CMEN)) {
        rspdif_dev->state = HAL_RSPDIF_STATE_READY;
        hal_rspdif_stop_dma(rspdif_dev);
    }

    if(NULL != (rspdif_dev->rspdif_rx_cpl_callback)) {
        ((hal_irq_handle_cb)rspdif_dev->rspdif_rx_cpl_callback)(rspdif_dev);
    }

    if(NULL != (rspdif_dev->rspdif_cx_cpl_callback)) {
        ((hal_irq_handle_cb)rspdif_dev->rspdif_cx_cpl_callback)(rspdif_dev);
    }

    rspdif_dev->state = HAL_RSPDIF_STATE_READY;
}

/*!
    \brief      RSPDIF DMA full transmission complete callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _rspdif_dma_half_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_rspdif_dev_struct *rspdif_dev;
    p_dma      = (hal_dma_dev_struct *)dma;
    rspdif_dev = (hal_rspdif_dev_struct *)(p_dma->p_periph);

    if(NULL != (rspdif_dev->rspdif_rx_half_callback)) {
        ((hal_irq_handle_cb)rspdif_dev->rspdif_rx_half_callback)(rspdif_dev);
    }

    if(NULL != (rspdif_dev->rspdif_cx_half_callback)) {
        ((hal_irq_handle_cb)rspdif_dev->rspdif_cx_half_callback)(rspdif_dev);
    }

    rspdif_dev->state = HAL_RSPDIF_STATE_READY;
}

/*!
    \brief      RSPDIF DMA underflow error callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _rspdif_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_rspdif_dev_struct *rspdif_dev;
    p_dma      = (hal_dma_dev_struct *)dma;
    rspdif_dev = (hal_rspdif_dev_struct *)(p_dma->p_periph);

    if(NULL != (rspdif_dev->rspdif_error_callback)) {
        ((hal_irq_handle_cb)rspdif_dev->rspdif_error_callback)(rspdif_dev);
    }

    rspdif_dev->state = HAL_RSPDIF_STATE_READY;
}

/*!
    \brief      receive an amount of data (data flow) with interrupt
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _rspdif_receive_data_handle(hal_rspdif_dev_struct *rspdif_dev)
{
    /* receive data */
    (*rspdif_dev->p_rx_buffer) = RSPDIF_DATA;
    rspdif_dev->p_rx_buffer++;
    rspdif_dev->rx_xfer_count--;

    if(0U == rspdif_dev->rx_xfer_count) {
        hals_rspdif_interrupt_disable(RSPDIF_INT_RBNE | RSPDIF_INT_PERR | RSPDIF_INT_RXORERR);
        rspdif_dev->state = HAL_RSPDIF_STATE_READY;
        HAL_UNLOCK(rspdif_dev);

        if(NULL != (rspdif_dev->rspdif_rx_cpl_callback)) {
            ((hal_irq_handle_cb)rspdif_dev->rspdif_rx_cpl_callback)(rspdif_dev);
        }
    }
}

/*!
    \brief      receive an amount of data (control flow) with interrupt
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _rspdif_receive_ctrl_handle(hal_rspdif_dev_struct *rspdif_dev)
{
    /* receive data */
    (*rspdif_dev->p_cs_buffer) = RSPDIF_CHSTAT;
    rspdif_dev->p_cs_buffer++;
    rspdif_dev->cs_xfer_count--;

    if(0U == rspdif_dev->cs_xfer_count) {
        hals_rspdif_interrupt_disable(RSPDIF_INT_CBNE | RSPDIF_INT_PERR | RSPDIF_INT_RXORERR);
        rspdif_dev->state = HAL_RSPDIF_STATE_READY;
        HAL_UNLOCK(rspdif_dev);

        if(NULL != (rspdif_dev->rspdif_cx_cpl_callback)) {
            ((hal_irq_handle_cb)rspdif_dev->rspdif_cx_cpl_callback)(rspdif_dev);
        }
    }
}

/*!
    \brief      RSPDIF error callback
    \param[in]  rspdif_dev: RSPDIF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _rspdif_error_handle(void *rspdif_dev)
{
    hal_rspdif_dev_struct *p_rspdif;
    p_rspdif = (hal_rspdif_dev_struct *)rspdif_dev;

    hals_rspdif_interrupt_disable(RSPDIF_INT_PERR | RSPDIF_INT_RXORERR | RSPDIF_INT_RXDCERR);

    if(NULL != (p_rspdif->rspdif_error_callback)) {
        ((hal_irq_handle_cb)p_rspdif->rspdif_error_callback)(p_rspdif);
    }

    p_rspdif->state = HAL_RSPDIF_STATE_READY;
}

/*!
    \brief This function handles RSPDIF Communication Timeout.
    \param[in] rspdif_dev SPDIFRX handle
    \param[in] Flag Flag checked
    \param[in] Status Value of the flag expected
    \param[in] Timeout Duration of the timeout
    \param[in] tickstart Tick start value
    \param[out] none
    \retval error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, details refer to
   gd32h7xx_hal.h
*/
static int32_t _rspdif_WaitOnFlagUntilTimeout(hal_rspdif_dev_struct *rspdif_dev, uint32_t Flag, \
                                              FlagStatus Status, uint32_t Timeout, uint32_t tickstart)
{
    int32_t ret = HAL_ERR_NONE;
    /* Wait until flag is set */
    while (Status == hals_rspdif_flag_get(Flag))
    {
        /* Check for the Timeout */
        if (Timeout != HAL_TIMEOUT_FOREVER)
        {
            if (((hal_sys_basetick_count_get() - tickstart) > Timeout) || (0U == Timeout))
            {
                rspdif_dev->state = HAL_RSPDIF_STATE_READY;

                /* Process Unlocked */
                HAL_UNLOCK(rspdif_dev);

                ret = HAL_ERR_TIMEOUT;
            }
        }
    }

    return ret;
}
