/*!
    \file    gd32h7xx_hal_rcu.c
    \brief   RCU driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* define clock source */
#define RCU_MODIFY(__delay)         do {                                     \
                                        __IO uint32_t i;                          \
                                        if(0U != __delay) {                  \
                                            RCU_CFG0 |= RCU_AHB_CKSYS_DIV2;  \
                                            for(i = 0U; i<__delay; i++) {     \
                                            }                                \
                                            RCU_CFG0 |= RCU_AHB_CKSYS_DIV4;  \
                                            for(i = 0U; i<__delay; i++) {     \
                                            }                                \
                                        }                                    \
                                    } while(0)

/* CKOUT0 GPIOA */
#define CKOUT0_GPIO_CLK_ENABLE()    hal_rcu_periph_clock_enable(RCU_GPIOA)
#define CKOUT0_GPIO_PORT            GPIOA
#define CKOUT0_GPIO_PIN             GPIO_PIN_8

/* CKOUT1 GPIOA */
#define CKOUT1_GPIO_CLK_ENABLE()    hal_rcu_periph_clock_enable(RCU_GPIOC)
#define CKOUT1_GPIO_PORT            GPIOC
#define CKOUT1_GPIO_PIN             GPIO_PIN_9

/* I2S_CKIN GPIOC */
#define I2S_CKIN_GPIO_CLK_ENABLE()  hal_rcu_periph_clock_enable(RCU_GPIOC)
#define I2S_CKIN_GPIO_PORT          GPIOC
#define I2S_CKIN_GPIO_PIN           GPIO_PIN_9

#define RCU_CKSYSSRC_INDEX(x)       (x)                     /*!< RCU CKSYSSRC index */

/* define clock source */
#define SEL_IRC64MDIV               ((uint16_t)0U)          /* IRC64M is selected as CK_SYS */
#define SEL_HXTAL                   ((uint16_t)1U)          /* HXTAL is selected as CK_SYS */
#define SEL_LPIRC4M                 ((uint16_t)2U)          /* LPIRC4M is selected as CK_SYS */
#define SEL_PLL0P                   ((uint16_t)3U)          /* PLL0P is selected as CK_SYS */

/* RCU_PLLxQ offset */
#define RCU_PLL1QOFFSET             ((uint32_t)8U)          /* PLL1Q offset reset */
#define RCU_PLL2QOFFSET             ((uint32_t)16U)         /* PLL2Q offset reset */

/* RCU IRC64M and LPIRC4M adjust value mask and offset */
#define RCU_IRC64M_ADJUST_MASK      ((uint8_t)0x7FU)         /*!< RCU IRC64M adjust value mask */
#define RCU_LPIRC4M_ADJUST_MASK     ((uint8_t)0x3FU)         /*!< RCU LPIRC4M adjust value mask */
#define RCU_LPIRC4M_ADJUST_OFFSET   ((uint32_t)2U)           /*!< RCU LPIRC4M adjust value offset */
#define RCU_REG_RESET_VALUE         ((uint32_t)0x00000000U)  /*!< RCU register reset value */
#define OSC_STARTUP_TIMEOUT         ((uint32_t)0x000FFFFFU)  /*!< OSC startup timeout value */
#define HAL_IRC64M_ENABLE_TIMEOUT   ((uint32_t)0x000000FFU)  /*!< IRC64M timer out value */

/* RCU interrupt definition */
static hal_rcu_irq_struct rcu_irq = {NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL};

/* RCU system clock source timeout */
static const uint32_t s_rcu_timeout[] = {HAL_IRC64MDIV_TIMEOUT, HAL_HXTAL_TIMEOUT, \
                                         HAL_LPIRC4M_TIMEOUT, HAL_PLL_TIMEOUT};

volatile uint32_t g_systemcoreclock = 64000000U;

/* function to calculate the PLL output frequency*/
static uint32_t _rcu_pll_clock_freq_cal(uint32_t pll_inputfreq, uint32_t pll_psc, uint32_t pll_n, \
                                        uint32_t fracn, uint32_t pll_pqr);
/* configuration the PLL input and output clock range */
static int32_t _rcu_pll_input_output_clock_range_config(hal_pll_idx_enum pll_idx, uint32_t ck_input, uint32_t ck_output);
/* configuration the PLL fractional divider */
static int32_t _rcu_pll_fractional_config(hal_pll_idx_enum pll_idx, uint32_t pll_frac, uint32_t pll_frac_state);
/* configuration the PLL0 */
static int32_t _rcu_pll0_config(hal_rcu_pll0_struct *pll0);
/* configuration the PLL1 */
static int32_t _rcu_pll1_config(hal_rcu_pll1_struct *pll1);
/* configuration the PLL2 */
static int32_t _rcu_pll2_config(hal_rcu_pll2_struct *pll2);
/* configuration the PLL source */
static void _rcu_pll_source_config(hal_rcu_pll_src_enum pll_src);
/* configuration the PLLUSB0 */
static void _rcu_pllusb0_config(hal_rcu_pllusb0_struct *pllusb0);
/* configuration the PLLUSB1 */
static void _rcu_pllusb1_config(hal_rcu_pllusb1_struct *pllusb1);
/* configuration PLL Clock Output */
static void _rcu_pll_clock_output_config(uint32_t pllxy, uint32_t pll_clock_status);
/* configuration PLL2R Clock Divided*/
static void _rcu_pll2r_clock_div_config(hal_rcu_pll2r_div_enum pll2r_div);
/* configuration PER Clock source */
static int32_t _rcu_per_clock_source_select(hal_rcu_periphclk_struct *periph_clk);
/* configuration IRC64M Clock source */
static int32_t _rcu_irc64m_clock_source_select(hal_rcu_irc64m_div_enum irc64m_div);
/* configuration LPIRC4M Clock source */
static int32_t _rcu_lpirc4m_clock_source_select(void);
/* configuration HXTAL Clock source */
static int32_t _rcu_hxtal_clock_source_select(void);
/* configuration LXTAL Clock source */
static int32_t _rcu_lxtal_clock_source_select(void);
/* configuration IRC32K Clock source */
static int32_t _rcu_irc32k_clock_source_select(void);

extern hal_sys_timebase_source_enum g_basetick_source;

/*!
    \brief      configure the RCU oscillators
    \param[in]  rcu_osci: the pointer of the RCU oscillators structure
                  hxtal: HXTAL status structure
                    need_configure: configure flag, can be ENABLE or DISABLE
                    status: the argument could be selected from enumeration <hal_rcu_osc_state_enum>
                  lxtal: LXTAL status structure
                    need_configure: configure flag, can be ENABLE or DISABLE
                    status: the argument could be selected from enumeration <hal_rcu_osc_state_enum>
                    drive: the argument could be selected from enumeration <hal_lxtal_drive_capability_enum>
                  irc64m: IRC64M status structure
                    need_configure: configure flag, can be ENABLE or DISABLE
                    status: the argument could be selected from enumeration <hal_rcu_osc_state_enum>
                  irc48m: IRC48M status structure
                    need_configure: configure flag, can be ENABLE or DISABLE
                    calibration_value: calibration value
                    status: the argument could be selected from enumeration <hal_rcu_osc_state_enum>
                  lpirc4m: LPIRC4M status structure
                    need_configure: configure flag, can be ENABLE or DISABLE
                    calibration_value: calibration value
                    status: the argument could be selected from enumeration <hal_rcu_osc_state_enum>
                  irc32k: IRC32K status structure
                    need_configure: configure flag, can be ENABLE or DISABLE
                    status: the argument could be selected from enumeration <hal_rcu_osc_state_enum>
                  pll: PLL0 status structure
                    need_configure: configure flag, can be ENABLE or DISABLE
                    status: the argument could be selected from enumeration <hal_rcu_osc_state_enum>
                  pll_source: RCU_PLL_SRC_CK_IRC64MDIV, RCU_PLL_SRC_CK_LPIRC4M
                              RCU_PLL_SRC_CK_CK_HXTAL,RCU_PLL_SRC_CK_CK_None
                  pll0_psc: the argument could be selected from enumeration <hal_rcu_pll_psc_enum>
                  pll0_pll0n: the argument could be selected from enumeration <hal_rcu_plln_enum>
                  pll0_pll0P: the argument could be selected from enumeration <hal_rcu_pllp_enum>
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_TIMEOUT, details refer to gd32h7xx_hal.h
*/
int32_t hal_rcu_osci_config(hal_rcu_osci_struct *rcu_osci)
{
    FlagStatus pwr_state = RESET;
    int32_t  ret = HAL_ERR_NONE;

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rcu_osci) {
        HAL_DEBUGE("pointer [rcu_osci] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear all RCU configuration */
    if(HAL_ERR_NONE != hal_rcu_deinit()) {
        HAL_DEBUGE(" deinitialize the RCU fail !");
        ret = HAL_ERR_TIMEOUT;
    } else {
        /*------------------------------- HXTAL Configuration -------------------------------*/
        if(ENABLE == rcu_osci->hxtal.need_configure) {
            /* it is not allowed to be disabled when the HXTAL is used as system clock or clock source of PLL */
            if((RCU_SCSS_HXTAL == hals_rcu_system_clock_source_get()) || \
               ((RCU_SCSS_PLL0P == hals_rcu_system_clock_source_get()) && \
                ((uint32_t)RCU_PLL_SRC_CK_HXTAL == (RCU_PLLALL & RCU_PLLALL_PLLSEL)))) {
                if((RESET != (RCU_CTL & RCU_CTL_HXTALSTB)) && (RCU_OSC_OFF == rcu_osci->hxtal.state)) {
                    HAL_DEBUGE("HXTAL is not stabilized or the state setting of hxtal_state is not ON");
                    ret = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* configure the new HXTAL state */
                switch(rcu_osci->hxtal.state) {
                case RCU_OSC_OFF:
                    hals_rcu_osci_off(RCU_HXTAL);
                    hals_rcu_osci_bypass_mode_disable(RCU_HXTAL);
                    break;
                case RCU_OSC_ON:
                    hals_rcu_osci_on(RCU_HXTAL);
                    break;
                case RCU_OSC_BYPASS:
                    hals_rcu_osci_off(RCU_HXTAL);
                    hals_rcu_osci_bypass_mode_enable(RCU_HXTAL);
                    hals_rcu_osci_on(RCU_HXTAL);
                    break;
                case RCU_OSC_NONE:
                    break;
                default:
                    HAL_DEBUGE("parameter member [rcu_osci->hxtal.state] is invalid.");
                    ret = HAL_ERR_VAL;
                    break;
                }

                if(HAL_ERR_NONE == ret) {
                    /* wait for crystal oscillator stabilize status */
                    if((RCU_OSC_OFF != rcu_osci->hxtal.state) && (RCU_OSC_NONE != rcu_osci->hxtal.state)) {
                        if(ERROR == hals_rcu_osci_stab_wait(RCU_HXTAL)) {
                            HAL_DEBUGE("HXTAL is not stabilized for the required amount of time");
                            ret = HAL_ERR_TIMEOUT;
                        } else {
                            /* do nothing */
                        }
                    } else if(RCU_OSC_OFF == rcu_osci->hxtal.state) {
                        /* wait till HXTAL is disable */
                        uint32_t tick_start = hal_sys_basetick_count_get();
                        while(RESET != (RCU_CTL & RCU_CTL_HXTALEN)) {
                            if(SET == hal_sys_basetick_timeout_check(tick_start, HAL_HXTAL_TIMEOUT)) {
                                HAL_DEBUGE("HXTAL did not shut down within the specified time period");
                                ret = HAL_ERR_TIMEOUT;
                                break;
                            } else {
                                /* do nothing */
                            }
                        }
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }
        } else {
            /* do nothing */
        }

        /* ------------------------------- LXTAL Configuration -------------------------------*/
        if((ENABLE == rcu_osci->lxtal.need_configure) && (HAL_ERR_NONE == ret)) {
            /* enable whether the pmu state */
            if(RESET == (RCU_APB4EN & RCU_APB4EN_PMUEN)) {
                hal_rcu_periph_clock_enable(RCU_PMU);
                pwr_state = SET;
            } else {
                /* do nothing */
            }

            /* update LXTAL configuration in backup domain control register */
            if(RESET == (PMU_CTL0 & PMU_CTL0_BKPWEN)) {
                uint32_t tick_start = 0U;

                /* enable write access to backup domain */
                hal_pmu_backup_write_enable();
                tick_start = hal_sys_basetick_count_get();
                while(RESET == (PMU_CTL0 & PMU_CTL0_BKPWEN)) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, HAL_BP_TIMEOUT)) {
                        HAL_DEBUGE("PMU backup domain write enable is not set for the required amount of time");
                        ret = HAL_ERR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                }
            } else {
                /* do nothing */
            }

            if(HAL_ERR_NONE == ret) {
                /* configure the new LXTAL state */
                switch(rcu_osci->lxtal.state) {
                case RCU_OSC_OFF:
                    hals_rcu_osci_off(RCU_LXTAL);
                    hals_rcu_osci_bypass_mode_disable(RCU_LXTAL);
                    break;
                case RCU_OSC_ON:
                    hals_rcu_osci_on(RCU_LXTAL);
                    break;
                case RCU_OSC_BYPASS:
                    hals_rcu_osci_off(RCU_LXTAL);
                    hals_rcu_osci_bypass_mode_enable(RCU_LXTAL);
                    hals_rcu_osci_on(RCU_LXTAL);
                    break;
                case RCU_OSC_NONE:
                    break;
                default:
                    HAL_DEBUGE("parameter [rcu_osci->lxtal.state] value is undefine");
                    ret = HAL_ERR_VAL;
                    break;
                }

                if(HAL_ERR_NONE == ret) {
                    /* configure the new LXTAL drive capability */
                    switch(rcu_osci->lxtal.drive) {
                    case RCU_LXTAL_DRIVE_LOWDRI:
                        hals_rcu_lxtal_drive_capability_config(RCU_LXTAL_DRIVE_LOWDRI);
                        break;
                    case RCU_LXTAL_DRIVE_MED_LOWDRI:
                        hals_rcu_lxtal_drive_capability_config(RCU_LXTAL_DRIVE_MED_LOWDRI);
                        break;
                    case RCU_LXTAL_DRIVE_MED_HIGHDRI:
                        hals_rcu_lxtal_drive_capability_config(RCU_LXTAL_DRIVE_MED_HIGHDRI);
                        break;
                    case RCU_LXTAL_DRIVE_HIGHDRI:
                        hals_rcu_lxtal_drive_capability_config(RCU_LXTAL_DRIVE_HIGHDRI);
                        break;
                    default:
                        HAL_DEBUGE("parameter [rcu_osci->lxtal.drive] value is undefine");
                        ret = HAL_ERR_VAL;
                        break;
                    }
                } else {
                    /* do nothing */
                }

                /* wait for crystal oscillator stabilize status */
                if((RCU_OSC_OFF != rcu_osci->lxtal.state) && \
                   (RCU_OSC_NONE != rcu_osci->lxtal.state) && \
                   (HAL_ERR_NONE == ret)) {
                    if(ERROR == hals_rcu_osci_stab_wait(RCU_LXTAL)) {
                        HAL_DEBUGE("LXTAL is not stabilized for the required amount of time");
                        ret = HAL_ERR_TIMEOUT;
                    } else {
                        /* do nothing */
                    }
                } else if((RCU_OSC_OFF == rcu_osci->lxtal.state) && (HAL_ERR_NONE == ret)) {
                    /* wait till Lxtal is disable */
                    uint32_t tick_start = hal_sys_basetick_count_get();
                    while(RESET != (RCU_BDCTL & RCU_BDCTL_LXTALEN)) {
                        if(SET == hal_sys_basetick_timeout_check(tick_start, HAL_LXTAL_TIMEOUT)) {
                            HAL_DEBUGE("LXTAL did not shut down within the specified time period");
                            ret = HAL_ERR_TIMEOUT;
                            break;
                        } else {
                            /* do nothing */
                        }
                    }
                } else {
                    /* do nothing */
                }

                if((SET == pwr_state) && (HAL_ERR_NONE == ret)) {
                    hal_rcu_periph_clock_disable(RCU_PMU);
                    pwr_state = RESET;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /*------------------------------- IRC64M Configuration -------------------------------*/
        if((ENABLE == rcu_osci->irc64m.need_configure) && (HAL_ERR_NONE == ret)) {
            /* it is not allowed to be disabled when the HXTAL is used as system clock or clock source of PLL */
            if((RCU_SCSS_IRC64MDIV == hals_rcu_system_clock_source_get()) || \
               ((RCU_SCSS_PLL0P == hals_rcu_system_clock_source_get()) && \
                ((uint32_t)RCU_PLL_SRC_CK_IRC64MDIV == (RCU_PLLALL & RCU_PLLALL_PLLSEL)))) {
                if((RESET != (RCU_CTL & RCU_CTL_IRC64MSTB)) && (RCU_OSC_OFF == rcu_osci->hxtal.state)) {
                    HAL_DEBUGE("IRC64M is not stabilized or the state setting of hxtal_state is not ON");
                    ret = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* configure the new IRC64M state */
                switch(rcu_osci->irc64m.state) {
                case RCU_OSC_OFF:
                    hals_rcu_osci_off(RCU_IRC64M);
                    break;
                case RCU_OSC_ON:
                    /* adjusts the IRC64M calibration value */
                    if(rcu_osci->irc64m.irc64m_div) {
                        hals_rcu_irc64mdiv_clock_config(rcu_osci->irc64m.irc64m_div);
                    } else {
                        /* do nothing */
                    }

                    hals_rcu_osci_on(RCU_IRC64M);
                    break;
                case RCU_OSC_BYPASS:
                    break;
                case RCU_OSC_NONE:
                    break;
                default:
                    HAL_DEBUGE("parameter [rcu_osci->irc64m.state] value is undefine.");
                    ret = HAL_ERR_VAL;
                    break;
                }

                if(HAL_ERR_NONE == ret) {
                    /* adjusts the IRC64M calibration value */
                    if(rcu_osci->irc64m.adjust_value) {
                        hals_rcu_irc64m_adjust_value_set((uint32_t)rcu_osci->irc64m.adjust_value);
                    } else {
                        /* do nothing */
                    }

                    /* wait for crystal oscillator stabilize status */
                    if((RCU_OSC_OFF != rcu_osci->irc64m.state) && (RCU_OSC_NONE != rcu_osci->irc64m.state)) {
                        if(ERROR == hals_rcu_osci_stab_wait(RCU_IRC64M)) {
                            HAL_DEBUGE("IRC64M is not stabilized for the required amount of time");
                            ret = HAL_ERR_TIMEOUT;
                        } else {
                            /* do nothing */
                        }
                    } else if(RCU_OSC_OFF != rcu_osci->irc64m.state) {
                        /* wait till IRC64M is disable */
                        uint32_t tick_start = hal_sys_basetick_count_get();
                        while(RESET != (RCU_CTL & RCU_CTL_IRC64MEN)) {
                            if(SET == hal_sys_basetick_timeout_check(tick_start, HAL_IRC64M_ENABLE_TIMEOUT)) {
                                HAL_DEBUGE("IRC64M did not shut down within the specified time period");
                                ret =  HAL_ERR_TIMEOUT;
                                break;
                            } else {
                                /* do nothing */
                            }
                        }
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }
        } else {
            /* do nothing */
        }

        /*------------------------------- IRC48M Configuration -------------------------------*/
        if((ENABLE == rcu_osci->irc48m.need_configure) && (HAL_ERR_NONE == ret)) {
            /* set the IRC48M new state */
            if(RCU_OSC_ON == rcu_osci->irc48m.state) {
                hals_rcu_osci_on(RCU_IRC48M);

                /* wait till IRC48M is stable */
                if(ERROR == hals_rcu_osci_stab_wait(RCU_IRC48M)) {
                    HAL_DEBUGE("IRC48M is not stabilized for the required amount of time");
                    ret = HAL_ERR_TIMEOUT;
                } else {
                    /* do nothing */
                }
            } else if(RCU_OSC_OFF == rcu_osci->irc48m.state) {
                uint32_t tick_start = 0U;

                /* wait till IRC48M is disabled */
                hals_rcu_osci_off(RCU_IRC48M);
                tick_start = hal_sys_basetick_count_get();
                while(RESET != (RCU_ADDCTL0 & RCU_ADDCTL0_IRC48MEN)) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, HAL_IRC48M_TIMEOUT)) {
                        HAL_DEBUGE("IRC48M did not shut down within the specified time period");
                        ret = HAL_ERR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                }
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /*------------------------------- IRC32K Configuration -------------------------------*/
        if((ENABLE == rcu_osci->irc32k.need_configure) && (HAL_ERR_NONE == ret)) {
            /* set the IRC32K new state */
            if(RCU_OSC_ON == rcu_osci->irc32k.state) {
                hals_rcu_osci_on(RCU_IRC32K);

                /* wait till IRC32K is stable */
                if(ERROR == hals_rcu_osci_stab_wait(RCU_IRC32K)) {
                    HAL_DEBUGE("IRC32K is not stabilized for the required amount of time");
                    ret = HAL_ERR_TIMEOUT;
                } else {
                    /* do nothing */
                }
            } else if(RCU_OSC_OFF == rcu_osci->irc32k.state) {
                uint32_t tick_start = 0U;

                /* wait till IRC32K is disabled */
                hals_rcu_osci_off(RCU_IRC32K);
                tick_start = hal_sys_basetick_count_get();
                while(RESET != (RCU_RSTSCK & RCU_RSTSCK_IRC32KEN)) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, HAL_IRC32K_TIMEOUT)) {
                        HAL_DEBUGE("IRC32K did not shut down within the specified time period");
                        ret = HAL_ERR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                }
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /*------------------------------- LPIRC4M Configuration -------------------------------*/
        if((ENABLE == rcu_osci->lpirc4m.need_configure) && (HAL_ERR_NONE == ret)) {
            /* set the LPIRC4M new state */
            if(RCU_OSC_ON == rcu_osci->lpirc4m.state) {
                hals_rcu_osci_on(RCU_LPIRC4M);

                /* adjusts the LPIRC4M calibration value */
                if(rcu_osci->lpirc4m.adjust_value) {
                    hals_rcu_lpirc4m_adjust_value_set((uint32_t)rcu_osci->lpirc4m.adjust_value);
                } else {
                    /* do nothing */
                }

                /* wait till LPIRC4M is stable */
                if(ERROR == hals_rcu_osci_stab_wait(RCU_LPIRC4M)) {
                    HAL_DEBUGE("LPIRC4M is not stabilized for the required amount of time");
                    ret = HAL_ERR_TIMEOUT;
                } else {
                    /* do nothing */
                }
            } else if(RCU_OSC_OFF == rcu_osci->lpirc4m.state) {
                uint32_t tick_start = 0U;

                /* wait till LPIRC4M is disabled */
                hals_rcu_osci_off(RCU_LPIRC4M);
                tick_start = hal_sys_basetick_count_get();
                while(RESET != (RCU_ADDCTL1 & RCU_ADDCTL1_LPIRC4MEN)) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, HAL_IRC32K_TIMEOUT)) {
                        HAL_DEBUGE("LPIRC4M did not shut down within the specified time period");
                        ret = HAL_ERR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                }
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /*------------------------------- pll Configuration -------------------------------*/
        if((ENABLE == rcu_osci->pll.need_configure) && (HAL_ERR_NONE == ret)) {
            /* check if the PLL is used as system clock or not */
            if((RCU_OSC_NONE != rcu_osci->pll.state) && (RCU_OSC_BYPASS != rcu_osci->pll.state)) {
                if(RCU_SCSS_PLL0P != hals_rcu_system_clock_source_get()) {
                    uint32_t tick_start = 0U;

                    if(RCU_OSC_ON == rcu_osci->pll.state) {
                        hals_rcu_osci_off(RCU_PLL0_CK);

                        /* wait till PLL0 is disabled */
                        tick_start = hal_sys_basetick_count_get();
                        while(RESET != (RCU_CTL & RCU_CTL_PLL0EN)) {
                            if(SET == hal_sys_basetick_timeout_check(tick_start, HAL_PLL_TIMEOUT)) {
                                HAL_DEBUGE("PLL0 did not shut down within the specified time period");
                                ret = HAL_ERR_TIMEOUT;
                                break;
                            } else {
                                /* do nothing */
                            }
                        }

                        if(HAL_ERR_NONE == ret) {
                            /* configure the PLL */
                            _rcu_pll_source_config(rcu_osci->pll.pll_source);
                            if(HAL_ERR_NONE != _rcu_pll0_config(&rcu_osci->pll.pll0)) {
                                HAL_DEBUGE("parameter [rcu_osci->pll.pll0] value is invalid");
                                ret = HAL_ERR_VAL;
                            } else {
                                /* do nothing */
                            }

                            /* configuration PLL fractional divider */
                            if((0U != rcu_osci->pll.pll0.pll0_fractional_latch) && (HAL_ERR_NONE == ret)) {
                                if(HAL_ERR_NONE != _rcu_pll_fractional_config(INDEX_PLL0, \
                                                                              rcu_osci->pll.pll0.pll0_fractional_latch, \
                                                                              ENABLE)) {
                                    HAL_DEBUGE("parameter [rcu_osci->pll.pll0.pll0_fractional_latch] value is invalid");
                                    ret = HAL_ERR_VAL;
                                } else {
                                    /* do nothing */
                                }
                            } else {
                                _rcu_pll_fractional_config(INDEX_PLL0, 0U, DISABLE);
                            }

                            if(HAL_ERR_NONE == ret) {
                                /* configuration PLL input/output clock */
                                _rcu_pll_input_output_clock_range_config(INDEX_PLL0, \
                                                                         rcu_osci->pll.pll0.pll0_input_fre_range, \
                                                                         rcu_osci->pll.pll0.pll0_output_fre_range);

                                _rcu_pll_clock_output_config(RCU_PLL0P, ENABLE);
                                _rcu_pll_clock_output_config(RCU_PLL0R, ENABLE);
                                _rcu_pll_clock_output_config(RCU_PLL0Q, ENABLE);

                                /* wait till PLL is stable */
                                hals_rcu_osci_on(RCU_PLL0_CK);
                                if(ERROR == hals_rcu_osci_stab_wait(RCU_PLL0_CK)) {
                                    HAL_DEBUGE("PLL is not stabilized for the required amount of time");
                                    ret = HAL_ERR_TIMEOUT;
                                } else {
                                    /* do nothing */
                                }
                            } else {
                                /* do nothing */
                            }
                        } else {
                            /* do nothing */
                        }
                    } else {
                        hals_rcu_osci_off(RCU_PLL0_CK);

                        /* wait till PLL is disabled */
                        tick_start = hal_sys_basetick_count_get();
                        while(RESET != (RCU_CTL & RCU_CTL_PLL0EN)) {
                            if(SET == hal_sys_basetick_timeout_check(tick_start, HAL_PLL_TIMEOUT)) {
                                HAL_DEBUGE("PLL did not shut down within the specified time period");
                                ret = HAL_ERR_TIMEOUT;
                            } else {
                                /* do nothing */
                            }
                        }
                    }
                } else {
                    HAL_DEBUGE("Currently PLL0R is already the system clock");
                    ret = HAL_ERR_VAL;
                }
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    }

    return ret;
}

/*!
    \brief      configure the clock0 out to output on CKOUT0 pin
    \param[in]  ckout0_src: clock out0 source,
                  the argument could be selected from enumeration <hal_rcu_clkout0_src_enum>
    \param[in]  ckout0_div: clock out0 divider,
                  the argument could be selected from enumeration <hal_rcu_ckout0_div_enum>
    \param[out] none
    \retval     none
*/
void hal_rcu_clock0_out_config(hal_rcu_clkout0_src_enum ckout0_src, hal_rcu_ckout0_div_enum ckout0_div)
{
    hal_gpio_init_struct gpio_init_struct;

    uint32_t reg = 0U;

    CKOUT0_GPIO_CLK_ENABLE();

    /* configure clock out GPIO pin */
    gpio_init_struct.mode   = GPIO_MODE_AF_PP;
    gpio_init_struct.ospeed = GPIO_OSPEED_100_220MHZ;
    gpio_init_struct.pull   = GPIO_PULL_NONE;
    gpio_init_struct.af     = GPIO_AF_0;
    hal_gpio_init(CKOUT0_GPIO_PORT, CKOUT0_GPIO_PIN, &gpio_init_struct);

    /* reset the CKOUT0SRC, CKOUT0DIV and set according to ckout0_src and ckout0_div */
    reg = RCU_CFG2;
    reg &= ~(RCU_CFG2_CKOUT0SEL | RCU_CFG2_CKOUT0DIV);
    RCU_CFG2 = (reg | ckout0_src | ckout0_div);
}

/*!
    \brief      configure the clock1 out to output on CKOUT1 pin
    \param[in]  ckout1_src: clock out1 source,
                  the argument could be selected from enumeration <hal_rcu_clkout1_src_enum>
    \param[in]  ckout1_div: clock out1 divider,
                  the argument could be selected from enumeration <hal_rcu_ckout1_div_enum>
    \param[out] none
    \retval     none
*/
void hal_rcu_clock1_out_config(hal_rcu_clkout1_src_enum ckout1_src, hal_rcu_ckout1_div_enum ckout1_div)
{
    hal_gpio_init_struct gpio_init_struct;

    uint32_t reg = 0U;

    CKOUT1_GPIO_CLK_ENABLE();

    /* configure clock out GPIO pin */
    gpio_init_struct.mode   = GPIO_MODE_AF_PP;
    gpio_init_struct.ospeed = GPIO_OSPEED_100_220MHZ;
    gpio_init_struct.pull   = GPIO_PULL_NONE;
    gpio_init_struct.af     = GPIO_AF_0;
    hal_gpio_init(CKOUT1_GPIO_PORT, CKOUT1_GPIO_PIN, &gpio_init_struct);

    /* reset the CKOUT1SRC, CKOUT1DIV and set according to ckout1_src and ckout1_div */
    reg = RCU_CFG2;
    reg &= ~(RCU_CFG2_CKOUT1SEL | RCU_CFG2_CKOUT1DIV);
    RCU_CFG2 = (reg | (uint32_t)ckout1_src | (uint32_t)ckout1_div);
}

/*!
    \brief      deinitialize the RCU
    \param[in]  none
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rcu_deinit(void)
{
    int32_t ret             = HAL_ERR_NONE;
    __IO uint32_t tick_start = 0U;

    /* enable IRC64M */
    RCU_CTL |= RCU_CTL_IRC64MEN;

    /* wait till IRC64M is on */
    tick_start = hal_sys_basetick_count_get();
    while((uint32_t)RESET == (RCU_CTL & RCU_CTL_IRC64MSTB)) {
        if(SET == hal_sys_basetick_timeout_check(tick_start, HAL_IRC64M_TIMEOUT)) {
            HAL_DEBUGE("PLL did not shut down within the specified time period");
            ret = HAL_ERR_TIMEOUT;
        } else {
            /* do nothing */
        }
    }

    if(HAL_ERR_NONE == ret) {
        RCU_MODIFY(0x50U);

        RCU_CFG0 &= ~RCU_CFG0_SCS;

        /* reset CFG0 register */
        RCU_CFG0 &= ~(RCU_CFG0_SCS | RCU_CFG0_AHBPSC | RCU_CFG0_APB1PSC | RCU_CFG0_APB2PSC | RCU_CFG0_RTCDIV | \
                      RCU_CFG0_APB3PSC | RCU_CFG0_APB4PSC | RCU_CFG0_I2C0SEL);

        /* reset CTL register */
        RCU_CTL &= ~(RCU_CTL_HXTALEN | RCU_CTL_CKMEN | RCU_CTL_PLL0EN | RCU_CTL_PLL1EN | RCU_CTL_PLL2EN);
        RCU_CTL &= ~(RCU_CTL_HXTALBPS);

        /* reset CFG register */
        RCU_CFG1 = RCU_REG_RESET_VALUE;
        RCU_CFG2 = RCU_REG_RESET_VALUE;
        RCU_CFG3 = RCU_REG_RESET_VALUE;
        RCU_CFG4 = RCU_REG_RESET_VALUE;
        RCU_CFG5 = RCU_REG_RESET_VALUE;

        /* reset PLL register */
        RCU_PLL0 = 0x01002020U;
        RCU_PLL1 = 0x01012020U;
        RCU_PLL2 = 0x01012020U;

        RCU_PLLALL  = RCU_REG_RESET_VALUE;
        RCU_PLL0FRA = RCU_REG_RESET_VALUE;
        RCU_PLL1FRA = RCU_REG_RESET_VALUE;
        RCU_PLL2FRA = RCU_REG_RESET_VALUE;

        /* reset INT register*/
        RCU_INT = 0x14ff0000U;

        /* reset ADDINT register */
        RCU_ADDINT = 0x00700000U;

        g_systemcoreclock = IRC64M_VALUE;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      initialize the RCU structure with the default values
    \param[in]  rcu_struct_type: the argument could be selected from enumeration <hal_rcu_struct_type_enum>
    \param[out] p_struct: pointer of the RCU structure,
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rcu_struct_init(hal_rcu_struct_type_enum rcu_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(rcu_struct_type) {
    case HAL_RCU_CLK_STRUCT:
        /* initialize RCU clock structure with the default values */
        ((hal_rcu_clk_struct *)p_struct)->clock_type    = RCU_CLKTYPE_NONE;
        ((hal_rcu_clk_struct *)p_struct)->sysclk_source = RCU_SYSCLK_SRC_CK_IRC64MDIV;
        ((hal_rcu_clk_struct *)p_struct)->apb1clk_div   = RCU_AHBCLK_APB1DIV_1;
        ((hal_rcu_clk_struct *)p_struct)->apb2clk_div   = RCU_AHBCLK_APB2DIV_1;
        ((hal_rcu_clk_struct *)p_struct)->apb3clk_div   = RCU_AHBCLK_APB3DIV_1;
        ((hal_rcu_clk_struct *)p_struct)->apb4clk_div   = RCU_AHBCLK_APB4DIV_1;
        ((hal_rcu_clk_struct *)p_struct)->ahbclk_div    = RCU_SYSCLK_AHBDIV_1;
        break;
    case HAL_RCU_OSCI_STRUCT:
        /* initialize RCU oscillator structure with the default values */
        ((hal_rcu_osci_struct *)p_struct)->hxtal.need_configure           = DISABLE;
        ((hal_rcu_osci_struct *)p_struct)->hxtal.state                    = RCU_OSC_NONE;
        ((hal_rcu_osci_struct *)p_struct)->lxtal.need_configure           = DISABLE;
        ((hal_rcu_osci_struct *)p_struct)->lxtal.state                    = RCU_OSC_NONE;
        ((hal_rcu_osci_struct *)p_struct)->irc48m.need_configure          = DISABLE;
        ((hal_rcu_osci_struct *)p_struct)->irc48m.state                   = RCU_OSC_NONE;
        ((hal_rcu_osci_struct *)p_struct)->irc64m.need_configure          = DISABLE;
        ((hal_rcu_osci_struct *)p_struct)->irc64m.adjust_value            = 0U;
        ((hal_rcu_osci_struct *)p_struct)->irc64m.state                   = RCU_OSC_NONE;
        ((hal_rcu_osci_struct *)p_struct)->irc64m.irc64m_div              = RCU_IRC64M_DIV_1;
        ((hal_rcu_osci_struct *)p_struct)->lpirc4m.need_configure         = DISABLE;
        ((hal_rcu_osci_struct *)p_struct)->lpirc4m.adjust_value           = 0U;
        ((hal_rcu_osci_struct *)p_struct)->lpirc4m.state                  = RCU_OSC_NONE;
        ((hal_rcu_osci_struct *)p_struct)->irc32k.need_configure          = DISABLE;
        ((hal_rcu_osci_struct *)p_struct)->irc32k.state                   = RCU_OSC_NONE;
        ((hal_rcu_osci_struct *)p_struct)->pll.need_configure             = DISABLE;
        ((hal_rcu_osci_struct *)p_struct)->pll.pll_source                 = RCU_PLL_SRC_CK_IRC64MDIV;
        ((hal_rcu_osci_struct *)p_struct)->pll.pll0.pll0_state            = RCU_OSC_NONE;
        ((hal_rcu_osci_struct *)p_struct)->pll.pll0.pll0_psc              = RCU_PLL_PSC_1;
        ((hal_rcu_osci_struct *)p_struct)->pll.pll0.pll0_n                = RCU_PLLN_MUL_9;
        ((hal_rcu_osci_struct *)p_struct)->pll.pll0.pll0_p                = RCU_PLLP_DIV_1;
        ((hal_rcu_osci_struct *)p_struct)->pll.pll0.pll0_input_fre_range  = RCU_PLL0RNG_4M_8M;
        ((hal_rcu_osci_struct *)p_struct)->pll.pll0.pll0_output_fre_range = RCU_PLL0VCO_192M_836M;
        ((hal_rcu_osci_struct *)p_struct)->pll.pll0.pll0_fractional_latch = 0U;
        break;
    case HAL_RCU_PLL_STRUCT:
        /* initialize RCU oscillator structure with the default values */
        ((hal_rcu_pll_struct *)p_struct)->need_configure                  = DISABLE;
        ((hal_rcu_pll_struct *)p_struct)->pll_source                      = RCU_PLL_SRC_CK_IRC64MDIV;
        ((hal_rcu_pll_struct *)p_struct)->pll0.pll0_state                 = RCU_OSC_NONE;
        ((hal_rcu_pll_struct *)p_struct)->pll0.pll0_psc                   = RCU_PLL_PSC_1;
        ((hal_rcu_pll_struct *)p_struct)->pll0.pll0_n                     = RCU_PLLN_MUL_9;
        ((hal_rcu_pll_struct *)p_struct)->pll0.pll0_p                     = RCU_PLLP_DIV_1;
        ((hal_rcu_pll_struct *)p_struct)->pll0.pll0_q                     = RCU_PLLQ_DIV_1;
        ((hal_rcu_pll_struct *)p_struct)->pll0.pll0_r                     = RCU_PLLR_DIV_1;
        ((hal_rcu_pll_struct *)p_struct)->pll0.pll0_input_fre_range       = RCU_PLL0RNG_4M_8M;
        ((hal_rcu_pll_struct *)p_struct)->pll0.pll0_output_fre_range      = RCU_PLL0VCO_192M_836M;
        ((hal_rcu_pll_struct *)p_struct)->pll0.pll0_fractional_latch      = 0U;
        ((hal_rcu_pll_struct *)p_struct)->pll1.pll1_state                 = RCU_OSC_NONE;
        ((hal_rcu_pll_struct *)p_struct)->pll1.pll1_psc                   = RCU_PLL_PSC_1;
        ((hal_rcu_pll_struct *)p_struct)->pll1.pll1_n                     = RCU_PLLN_MUL_9;
        ((hal_rcu_pll_struct *)p_struct)->pll1.pll1_p                     = RCU_PLLP_DIV_1;
        ((hal_rcu_pll_struct *)p_struct)->pll1.pll1_q                     = RCU_PLLQ_DIV_1;
        ((hal_rcu_pll_struct *)p_struct)->pll1.pll1_r                     = RCU_PLLR_DIV_1;
        ((hal_rcu_pll_struct *)p_struct)->pll1.pll1_input_fre_range       = RCU_PLL1RNG_4M_8M;
        ((hal_rcu_pll_struct *)p_struct)->pll1.pll1_output_fre_range      = RCU_PLL1VCO_192M_836M;
        ((hal_rcu_pll_struct *)p_struct)->pll1.pll1_fractional_latch      = 0U;
        ((hal_rcu_pll_struct *)p_struct)->pll2.pll2_state                 = RCU_OSC_NONE;
        ((hal_rcu_pll_struct *)p_struct)->pll2.pll2_psc                   = RCU_PLL_PSC_1;
        ((hal_rcu_pll_struct *)p_struct)->pll2.pll2_n                     = RCU_PLLN_MUL_9;
        ((hal_rcu_pll_struct *)p_struct)->pll2.pll2_p                     = RCU_PLLP_DIV_1;
        ((hal_rcu_pll_struct *)p_struct)->pll2.pll2_q                     = RCU_PLLQ_DIV_1;
        ((hal_rcu_pll_struct *)p_struct)->pll2.pll2_r                     = RCU_PLLR_DIV_1;
        ((hal_rcu_pll_struct *)p_struct)->pll2.pll2_input_fre_range       = RCU_PLL2RNG_4M_8M;
        ((hal_rcu_pll_struct *)p_struct)->pll2.pll2_output_fre_range      = RCU_PLL2VCO_192M_836M;
        ((hal_rcu_pll_struct *)p_struct)->pll2.pll2_fractional_latch      = 0U;
        ((hal_rcu_pll_struct *)p_struct)->pllusb0.pllusb0_state           = RCU_OSC_NONE;
        ((hal_rcu_pll_struct *)p_struct)->pllusb0.pllusb0_preclock_source = RCU_PLLUSBHSx_CKSRC_HXTAL;
        ((hal_rcu_pll_struct *)p_struct)->pllusb0.pllusb0_presel          = RCU_PLLUSBHSx_PREDV_1;
        ((hal_rcu_pll_struct *)p_struct)->pllusb0.pllusb0_mul             = RCU_PLLUSBHSx_MUL_16;
        ((hal_rcu_pll_struct *)p_struct)->pllusb1.pllusb1_state           = RCU_OSC_NONE;
        ((hal_rcu_pll_struct *)p_struct)->pllusb1.pllusb1_preclock_source = RCU_PLLUSBHSx_CKSRC_HXTAL;
        ((hal_rcu_pll_struct *)p_struct)->pllusb1.pllusb1_presel          = RCU_PLLUSBHSx_PREDV_1;
        ((hal_rcu_pll_struct *)p_struct)->pllusb1.pllusb1_mul             = RCU_PLLUSBHSx_MUL_16;
        break;
    case HAL_RCU_PERIPHCLK_STRUCT:
        /* initialize RCU peripheral clock structure with the default values */
        ((hal_rcu_periphclk_struct *)p_struct)->periph_clock_type             = RCU_PERIPH_CLKTYPE_NONE;
        ((hal_rcu_periphclk_struct *)p_struct)->rtc_clock_source              = RCU_RTC_CLKSRC_LXTAL;
        ((hal_rcu_periphclk_struct *)p_struct)->rtc_div                       = RCU_RTC_HXTAL_DIV_2;
        ((hal_rcu_periphclk_struct *)p_struct)->timer_mul                     = RCU_TIMER_MUL2;
        ((hal_rcu_periphclk_struct *)p_struct)->ck48m_clock_source            = RCU_CK48M_CKSRC_PLL48M;
        ((hal_rcu_periphclk_struct *)p_struct)->pll48m_clock_source           = RCU_PLL48M_CKSRC_PLL0Q;
        ((hal_rcu_periphclk_struct *)p_struct)->usb0_clock.usbhs_div          = RCU_USBHSx_DIV_2;
        ((hal_rcu_periphclk_struct *)p_struct)->usb0_clock.usbhs_psc          = RCU_USBHSx_PSC_1;
        ((hal_rcu_periphclk_struct *)p_struct)->usb0_clock.usbhs48m_source    = RCU_USBHSx48M_CKSRC_PLL0R;
        ((hal_rcu_periphclk_struct *)p_struct)->usb0_clock.usbhs_clock_source = RCU_USBHSx_CKSRC_CK48M;
        ((hal_rcu_periphclk_struct *)p_struct)->usb1_clock.usbhs_div          = RCU_USBHSx_DIV_2;
        ((hal_rcu_periphclk_struct *)p_struct)->usb1_clock.usbhs_psc          = RCU_USBHSx_PSC_1;
        ((hal_rcu_periphclk_struct *)p_struct)->usb1_clock.usbhs48m_source    = RCU_USBHSx48M_CKSRC_PLL0R;
        ((hal_rcu_periphclk_struct *)p_struct)->usb1_clock.usbhs_clock_source = RCU_USBHSx_CKSRC_CK48M;
        ((hal_rcu_periphclk_struct *)p_struct)->can0_clock_source             = RCU_CAN_CKSRC_CK_HXTAL;
        ((hal_rcu_periphclk_struct *)p_struct)->can1_clock_source             = RCU_CAN_CKSRC_CK_HXTAL;
        ((hal_rcu_periphclk_struct *)p_struct)->can2_clock_source             = RCU_CAN_CKSRC_CK_HXTAL;
        ((hal_rcu_periphclk_struct *)p_struct)->sai0_clock_source             = RCU_SAI_CKSRC_CK_PLL0Q;
        ((hal_rcu_periphclk_struct *)p_struct)->sai1_clock_source             = RCU_SAI_CKSRC_CK_PLL0Q;
        ((hal_rcu_periphclk_struct *)p_struct)->sai2b0_clock_source           = RCU_SAI2B_CKSRC_CK_PLL0Q;
        ((hal_rcu_periphclk_struct *)p_struct)->sai2b1_clock_source           = RCU_SAI2B_CKSRC_CK_PLL0Q;
        ((hal_rcu_periphclk_struct *)p_struct)->exmc_clock_source             = RCU_EXMC_CKSRC_CK_AHB;
        ((hal_rcu_periphclk_struct *)p_struct)->spi0_clock_source             = RCU_SPI_012_CKSRC_CK_PLL0Q;
        ((hal_rcu_periphclk_struct *)p_struct)->spi1_clock_source             = RCU_SPI_012_CKSRC_CK_PLL0Q;
        ((hal_rcu_periphclk_struct *)p_struct)->spi2_clock_source             = RCU_SPI_012_CKSRC_CK_PLL0Q;
        ((hal_rcu_periphclk_struct *)p_struct)->spi3_clock_source             = RCU_SPI_34_CKSRC_CK_APB2;
        ((hal_rcu_periphclk_struct *)p_struct)->spi4_clock_source             = RCU_SPI_34_CKSRC_CK_APB2;
        ((hal_rcu_periphclk_struct *)p_struct)->spi5_clock_source             = RCU_SPI_5_CKSRC_CK_APB2;
        ((hal_rcu_periphclk_struct *)p_struct)->usart0_clock_source           = RCU_USART_CKSRC_CK_APB;
        ((hal_rcu_periphclk_struct *)p_struct)->usart1_clock_source           = RCU_USART_CKSRC_CK_APB;
        ((hal_rcu_periphclk_struct *)p_struct)->usart2_clock_source           = RCU_USART_CKSRC_CK_APB;
        ((hal_rcu_periphclk_struct *)p_struct)->usart5_clock_source           = RCU_USART_CKSRC_CK_APB;
        ((hal_rcu_periphclk_struct *)p_struct)->hpdf_clock_source             = RCU_HPDF_CKSRC_CK_APB2;
        ((hal_rcu_periphclk_struct *)p_struct)->sdio0_clock_source            = RCU_SDIO0_CKSRC_CK_PLL0Q;
        ((hal_rcu_periphclk_struct *)p_struct)->sdio1_clock_source            = RCU_SDIO0_CKSRC_CK_PLL0Q;
        ((hal_rcu_periphclk_struct *)p_struct)->adc01_clock_source            = RCU_ADC_CKSRC_CK_PLL1P;
        ((hal_rcu_periphclk_struct *)p_struct)->adc2_clock_source             = RCU_ADC_CKSRC_CK_PLL1P;
        ((hal_rcu_periphclk_struct *)p_struct)->rspdif_clock_source           = RCU_RSPDIF_CKSRC_CK_PLL0Q;
        ((hal_rcu_periphclk_struct *)p_struct)->i2c0_clock_source             = RCU_I2C_CKSRC_CK_APB1;
        ((hal_rcu_periphclk_struct *)p_struct)->i2c1_clock_source             = RCU_I2C_CKSRC_CK_APB1;
        ((hal_rcu_periphclk_struct *)p_struct)->i2c2_clock_source             = RCU_I2C_CKSRC_CK_APB1;
        ((hal_rcu_periphclk_struct *)p_struct)->i2c3_clock_source             = RCU_I2C_CKSRC_CK_APB1;
        ((hal_rcu_periphclk_struct *)p_struct)->pef_clock_source              = RCU_PRECLK_CKSRC_CK_IRC64MDIV;
        break;
    default:
        HAL_DEBUGE("parameter [rcu_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      enable the peripherals clock
    \param[in]  periph: RCU peripherals,
                the argument could be selected from enumeration <hal_rcu_periph_enum>
    \param[out] none
    \retval     none
*/
void hal_rcu_periph_clock_enable(hal_rcu_periph_enum periph)
{
    RCU_REG_VAL(periph) |= BIT(RCU_BIT_POS(periph));
}

/*!
    \brief      disable the peripherals clock
    \param[in]  periph: RCU peripherals,
                the argument could be selected from enumeration <hal_rcu_periph_enum>
    \param[out] none
    \retval     none
*/
void hal_rcu_periph_clock_disable(hal_rcu_periph_enum periph)
{
    RCU_REG_VAL(periph) &= ~BIT(RCU_BIT_POS(periph));
}

/*!
    \brief      get the RCU oscillators configuration
    \param[in]  rcu_osci: the pointer of the RCU oscillators structure
                    hxtal: HXTAL status structure
                      need_configure: configure flag, can be ENABLE or DISABLE
                      status: the argument could be selected from enumeration <hal_rcu_osc_state_enum>
                    lxtal: LXTAL status structure
                      need_configure: configure flag, can be ENABLE or DISABLE
                      status: the argument could be selected from enumeration <hal_rcu_osc_state_enum>
                    irc64m: IRC64M status structure
                      need_configure: configure flag, can be ENABLE or DISABLE
                      status: the argument could be selected from enumeration <hal_rcu_osc_state_enum>
                    irc48m: IRC48M status structure
                      need_configure: configure flag, can be ENABLE or DISABLE
                      calibration_value: calibration value
                      status: the argument could be selected from enumeration <hal_rcu_osc_state_enum>
                    lpirc4m: LPIRC4M status structure
                      need_configure: configure flag, can be ENABLE or DISABLE
                      calibration_value: calibration value
                      status: the argument could be selected from enumeration <hal_rcu_osc_state_enum>
                    irc32k: IRC32K status structure
                      need_configure: configure flag, can be ENABLE or DISABLE
                      status: the argument could be selected from enumeration <hal_rcu_osc_state_enum>
                    pll: PLL status structure
                      need_configure: configure flag, can be ENABLE or DISABLE
                      status: the argument could be selected from enumeration <hal_rcu_osc_state_enum>
                    pll_source: RCU_PLL_SRC_CK_IRC64MDIV, RCU_PLL_SRC_CK_LPIRC4M
                                RCU_PLL_SRC_CK_CK_HXTAL,RCU_PLL_SRC_CK_CK_None
                    pllx_psc: the argument could be selected from enumeration <hal_rcu_pll_psc_enum>
                    pllx_pllxn: the argument could be selected from enumeration <hal_rcu_pll_plln_enum>
                    pllx_pllxP: the argument could be selected from enumeration <hal_rcu_pll_pllp_enum>
                    pllx_pllxq: the argument could be selected from enumeration <hal_rcu_pll_psc_enum>
                    pllx_pllxR: the argument could be selected from enumeration <hal_rcu_pll_plln_enum>
                    pll2R_div: the argument could be selected from enumeration <hal_rcu_pll2r_div_enum>
                    pllusb_presrc: the argument could be selected from enumeration <hal_rcu_pllusbhsx_precksrc_enum>
                    pllusb_presel: the argument could be selected from enumeration <hal_rcu_pllusbhsx_prediv_enum>
                    pllusb_mul: the argument could be selected from enumeration <hal_rcu_pllusbhsx_mul_enum>
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rcu_osci_config_get(hal_rcu_osci_struct *rcu_osci)
{
    uint32_t temp_irc64m_adj  = 0U;
    uint32_t temp_lpirc4m_adj = 0U;
    uint32_t temp_div         = 0U;
    uint32_t temp_pll0_psc    = 0U;
    uint32_t temp_pll0_n      = 0U;
    uint32_t temp_pll0_p      = 0U;
    uint32_t temp_pll0_r      = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rcu_osci) {
        HAL_DEBUGE("pointer [rcu_osci] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /*------------------------------- oscillator type Configuration -------------------------------*/
    rcu_osci->hxtal.need_configure   = ENABLE;
    rcu_osci->lxtal.need_configure   = ENABLE;
    rcu_osci->irc64m.need_configure  = ENABLE;
    rcu_osci->irc48m.need_configure  = ENABLE;
    rcu_osci->lpirc4m.need_configure = ENABLE;
    rcu_osci->irc32k.need_configure  = ENABLE;
    rcu_osci->pll.need_configure     = ENABLE;

    /*------------------------------- get the current HXTAL state -------------------------------*/
    if(RESET != (RCU_CTL & RCU_CTL_HXTALBPS)) {
        rcu_osci->hxtal.state = RCU_OSC_BYPASS;
    } else if(RESET != (RCU_CTL & RCU_CTL_HXTALEN)) {
        rcu_osci->hxtal.state = RCU_OSC_ON;
    } else {
        rcu_osci->hxtal.state = RCU_OSC_OFF;
    }

    /*------------------------------- get the current LXTAL state -------------------------------*/
    if(RESET != (RCU_BDCTL & RCU_BDCTL_LXTALBPS)) {
        rcu_osci->lxtal.state = RCU_OSC_BYPASS;
    } else if(RESET != (RCU_BDCTL & RCU_BDCTL_LXTALEN)) {
        rcu_osci->lxtal.state = RCU_OSC_ON;
    } else {
        rcu_osci->lxtal.state = RCU_OSC_OFF;
    }

    /*---------------------- get the current IRC64M state and adjust value-----------------------*/
    if(RESET != (RCU_CTL & RCU_CTL_IRC64MEN)) {
        rcu_osci->irc64m.state = RCU_OSC_ON;
    } else {
        rcu_osci->irc64m.state = RCU_OSC_OFF;
    }

    temp_irc64m_adj               = (RCU_CTL & RCU_CTL_IRC64MCALIB) >> 7U;
    rcu_osci->irc64m.adjust_value = (uint8_t)temp_irc64m_adj;

    temp_div                    = (RCU_ADDCTL1 & RCU_ADDCTL1_IRC64MDIV) >> 16U;
    rcu_osci->irc64m.irc64m_div = (hal_rcu_irc64m_div_enum)temp_div;

    /*------------------------------- get the current IRC48M state -------------------------------*/
    if(RESET != (RCU_ADDCTL0 & RCU_ADDCTL0_IRC48MEN)) {
        rcu_osci->irc48m.state = RCU_OSC_ON;
    } else {
        rcu_osci->irc48m.state = RCU_OSC_OFF;
    }

    /*------------------------------- get the current IRC32K state -------------------------------*/
    if(RESET != (RCU_RSTSCK & RCU_RSTSCK_IRC32KEN)) {
        rcu_osci->irc32k.state = RCU_OSC_ON;
    } else {
        rcu_osci->irc32k.state = RCU_OSC_OFF;
    }

    /*---------------------- get the current LPIRC4M state and adjust value-----------------------*/
    if(RESET != (RCU_ADDCTL1 & RCU_ADDCTL1_LPIRC4MEN)) {
        rcu_osci->lpirc4m.state = RCU_OSC_ON;
    } else {
        rcu_osci->lpirc4m.state = RCU_OSC_OFF;
    }

    temp_lpirc4m_adj               = (RCU_ADDCTL1 & RCU_ADDCTL1_LPIRC4MCALIB) >> 24U;
    rcu_osci->lpirc4m.adjust_value = (uint8_t)temp_lpirc4m_adj;

    /*------------------------------- get the current PLL0 state -------------------------------*/
    if(RESET != (RCU_CTL & RCU_CTL_PLL0EN)) {
        rcu_osci->pll.state = RCU_OSC_ON;
    } else {
        rcu_osci->pll.state = RCU_OSC_OFF;
    }

    temp_pll0_psc = (RCU_CFG0 & RCU_PLL0_PLL0PSC);
    temp_pll0_n   = (RCU_CFG0 & RCU_PLL0_PLL0N);
    temp_pll0_p   = (RCU_CFG0 & RCU_PLL0_PLL0P);
    temp_pll0_r   = (RCU_CFG0 & RCU_PLL0_PLL0R);

    rcu_osci->pll.pll0.pll0_psc = (hal_rcu_pll_psc_enum)temp_pll0_psc;
    rcu_osci->pll.pll0.pll0_n   = (hal_rcu_plln_enum)temp_pll0_n;
    rcu_osci->pll.pll0.pll0_p   = (hal_rcu_pllp_enum)temp_pll0_p;
    rcu_osci->pll.pll0.pll0_r   = (hal_rcu_pllr_enum)temp_pll0_r;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the pll
    \param[in]  p_pll: pointer of pll configuration struct
                  need_configure: configure flag
                  only one parameter can be selected from the following:
      \arg          ENABLE:enable the pll
      \arg          DISABLE:disable the pll
                  pll_source source: refer to hal_rcu_pll_src_enum
                  pll0: PLL0 configuration struct, refer to hal_rcu_pll0_struct
                  pll1: PLL1 configuration struct, refer to hal_rcu_pll1_struct
                  pll2: PLL2 configuration struct, refer to hal_rcu_pll2_struct
                  pllusb0: PLLUSB0 configuration struct, refer to hal_rcu_pllusb0_struct
                  pllusb1: PLLUSB1 configuration struct, refer to hal_rcu_pllusb1_struct
                  state: the argument could be selected from enumeration hal_rcu_osc_state_enum
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_rcu_pll_config(hal_rcu_pll_struct *p_pll)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_pll) {
        HAL_DEBUGE("pointer [rcu_osci] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((RCU_OSC_ON == p_pll->state) && (ENABLE == p_pll->need_configure)) {
        /* configure the PLL */
        if(RCU_SCSS_PLL0P != hals_rcu_system_clock_source_get()) {
            _rcu_pll_source_config(p_pll->pll_source);
        } else {
            /* do nothing */
        }

        /* configuration pll0 */
        if((RCU_SCSS_PLL0P != hals_rcu_system_clock_source_get()) && (RCU_OSC_ON == p_pll->pll0.pll0_state)) {
            uint32_t tick_start = 0U;

            /* wait till PLL0 is disabled */
            hals_rcu_osci_off(RCU_PLL0_CK);
            tick_start = hal_sys_basetick_count_get();
            while(RESET != (RCU_CTL & RCU_CTL_PLL0EN)) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, HAL_PLL_TIMEOUT)) {
                    HAL_DEBUGE("PLL0 did not shut down within the specified time period");
                    ret = HAL_ERR_TIMEOUT;
                    break;
                } else {
                    /* do nothing */
                }
            }

            /* configuration pll0 parameter */
            if((HAL_ERR_NONE != _rcu_pll0_config(&p_pll->pll0)) && (HAL_ERR_NONE == ret)) {
                HAL_DEBUGE("parameter [p_pll->pll0] value is invalid");
                ret = HAL_ERR_VAL;
            } else {
                /* do nothing */
            }

            /* configuration PLL0 fractional */
            if((0U != p_pll->pll0.pll0_fractional_latch) && (HAL_ERR_NONE == ret)) {
                if(HAL_ERR_NONE != _rcu_pll_fractional_config(INDEX_PLL0, \
                                                              p_pll->pll0.pll0_fractional_latch, \
                                                              ENABLE)) {
                    HAL_DEBUGE("parameter [p_pll->pll0.pll0_fractional_latch] value is invalid");
                    ret = HAL_ERR_VAL;
                }
            } else {
                _rcu_pll_fractional_config(INDEX_PLL0, 0U, DISABLE);
            }

            if(HAL_ERR_NONE == ret) {
                _rcu_pll_input_output_clock_range_config(INDEX_PLL0, \
                                                         p_pll->pll0.pll0_input_fre_range, \
                                                         p_pll->pll0.pll0_output_fre_range);

                /* pll0r, pll0p, pll0q output and pll0 enable */
                _rcu_pll_clock_output_config(RCU_PLL0P, ENABLE);
                _rcu_pll_clock_output_config(RCU_PLL0R, ENABLE);
                _rcu_pll_clock_output_config(RCU_PLL0Q, ENABLE);
                hals_rcu_osci_on(RCU_PLL0_CK);

                /* wait till PLL0 is stable */
                if(ERROR == hals_rcu_osci_stab_wait(RCU_PLL0_CK)) {
                    HAL_DEBUGE("PLL0 is not stabilized for the required amount of time");
                    ret = HAL_ERR_TIMEOUT;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* configuration pll1 */
        if((RCU_OSC_ON == p_pll->pll1.pll1_state) && (HAL_ERR_NONE == ret)) {
            uint32_t tick_start = 0U;

            /* wait till PLL1 is disabled */
            hals_rcu_osci_off(RCU_PLL1_CK);
            tick_start = hal_sys_basetick_count_get();
            while(RESET != (RCU_CTL & RCU_CTL_PLL1EN)) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, HAL_PLL_TIMEOUT)) {
                    HAL_DEBUGE("PLL1 did not shut down within the specified time period");
                    ret = HAL_ERR_TIMEOUT;
                    break;
                } else {
                    /* do nothing */
                }
            }

            /* configuration pll1 parameter */
            if((HAL_ERR_NONE != _rcu_pll1_config(&p_pll->pll1)) && (HAL_ERR_NONE == ret)) {
                HAL_DEBUGE("parameter [p_pll->pll1] value is invalid");
                ret = HAL_ERR_VAL;
            } else {
                /* do nothing */
            }

            if((0U != p_pll->pll1.pll1_fractional_latch) && (HAL_ERR_NONE == ret)) {
                if(HAL_ERR_NONE != _rcu_pll_fractional_config(INDEX_PLL1, \
                                                              p_pll->pll1.pll1_fractional_latch, \
                                                              ENABLE)) {
                    HAL_DEBUGE("parameter [p_pll->pll1.pll1_fractional_latch] value is invalid");
                    ret = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                _rcu_pll_fractional_config(INDEX_PLL1, 0U, DISABLE);
            }

            if(HAL_ERR_NONE == ret) {
                _rcu_pll_input_output_clock_range_config(INDEX_PLL1, \
                                                         p_pll->pll1.pll1_input_fre_range, \
                                                         p_pll->pll1.pll1_output_fre_range);

                /* pll1r, pll1p, pll1q output and pll1 enable */
                _rcu_pll_clock_output_config(RCU_PLL1P, ENABLE);
                _rcu_pll_clock_output_config(RCU_PLL1R, ENABLE);
                _rcu_pll_clock_output_config(RCU_PLL1Q, ENABLE);
                hals_rcu_osci_on(RCU_PLL1_CK);

                /* wait till PLL1 is stable */
                if(ERROR == hals_rcu_osci_stab_wait(RCU_PLL1_CK)) {
                    HAL_DEBUGE("PLL1 is not stabilized for the required amount of time");
                    ret = HAL_ERR_TIMEOUT;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* configuration pll2 */
        if((RCU_OSC_ON == p_pll->pll2.pll2_state) && (HAL_ERR_NONE == ret)) {
            uint32_t tick_start = 0U;

            /* wait till PLL is disabled */
            hals_rcu_osci_off(RCU_PLL2_CK);
            tick_start = hal_sys_basetick_count_get();
            while(RESET != (RCU_CTL & RCU_CTL_PLL2EN)) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, HAL_PLL_TIMEOUT)) {
                    HAL_DEBUGE("PLL2 did not shut down within the specified time period");
                    ret = HAL_ERR_TIMEOUT;
                    break;
                } else {
                    /* do nothing */
                }
            }

            /* configuration pll2 parameter */
            if((HAL_ERR_NONE != _rcu_pll2_config(&p_pll->pll2)) && (HAL_ERR_NONE == ret)) {
                HAL_DEBUGE("parameter [p_pll->pll2] value is invalid");
                ret = HAL_ERR_VAL;
            } else {
                /* do nothing */
            }

            /* configuration pll2 divided */
            if((0U != (uint32_t)p_pll->pll2.pll2r_div) && (HAL_ERR_NONE == ret)) {
                _rcu_pll2r_clock_div_config(p_pll->pll2.pll2r_div);
            } else {
                /* do nothing */
            }

            /* configuration PLL2 fractional divided */
            if((0U != p_pll->pll2.pll2_fractional_latch) && (HAL_ERR_NONE == ret)) {
                if(HAL_ERR_NONE != _rcu_pll_fractional_config(INDEX_PLL2, \
                                                              p_pll->pll2.pll2_fractional_latch, \
                                                              ENABLE)) {
                    HAL_DEBUGE("parameter [p_pll->pll2.pll2_fractional_latch] value is invalid");
                    ret = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                _rcu_pll_fractional_config(INDEX_PLL2, 0U, DISABLE);
            }

            if(HAL_ERR_NONE == ret) {
                _rcu_pll_input_output_clock_range_config(INDEX_PLL2, \
                                                         p_pll->pll2.pll2_input_fre_range, \
                                                         p_pll->pll2.pll2_output_fre_range);

                /* pll2r, pll2p, pll2q output and pll2 enable */
                _rcu_pll_clock_output_config(RCU_PLL2P, ENABLE);
                _rcu_pll_clock_output_config(RCU_PLL2R, ENABLE);
                _rcu_pll_clock_output_config(RCU_PLL2Q, ENABLE);
                hals_rcu_osci_on(RCU_PLL2_CK);

                /* wait till PLL2 is stable */
                if(ERROR == hals_rcu_osci_stab_wait(RCU_PLL2_CK)) {
                    HAL_DEBUGE("PLL2 is not stabilized for the required amount of time");
                    ret = HAL_ERR_TIMEOUT;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* configuration pllusb0 */
        if((RCU_OSC_ON == p_pll->pllusb0.pllusb0_state) && (HAL_ERR_NONE == ret)) {
            uint32_t tick_start = 0U;

            /* wait till PLLUSB0 is disabled */
            hals_rcu_osci_off(RCU_PLLUSBHS0_CK);
            tick_start = hal_sys_basetick_count_get();
            while(RESET != (RCU_ADDCTL1 & RCU_ADDCTL1_PLLUSBHS0EN)) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, HAL_PLL_TIMEOUT)) {
                    HAL_DEBUGE("PLLUSB0 did not shut down within the specified time period");
                    ret = HAL_ERR_TIMEOUT;
                    break;
                } else {
                    /* do nothing */
                }
            }

            if(HAL_ERR_NONE == ret) {
                /* configuration pllusb0 parameter */
                _rcu_pllusb0_config(&p_pll->pllusb0);
                hals_rcu_osci_on(RCU_PLLUSBHS0_CK);

                /* wait till PLLUSBHS0 is stable */
                if(ERROR == hals_rcu_osci_stab_wait(RCU_PLLUSBHS0_CK)) {
                    HAL_DEBUGE("PLLUSB0 is not stabilized for the required amount of time");
                    ret = HAL_ERR_TIMEOUT;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* configuration pllusb1 */
        if((RCU_OSC_ON == p_pll->pllusb1.pllusb1_state) && (HAL_ERR_NONE == ret)) {
            uint32_t tick_start = 0U;

            /* wait till PLLUSB1 is disabled */
            hals_rcu_osci_off(RCU_PLLUSBHS1_CK);
            tick_start = hal_sys_basetick_count_get();
            while(RESET != (RCU_ADDCTL1 & RCU_ADDCTL1_PLLUSBHS1EN)) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, HAL_PLL_TIMEOUT)) {
                    HAL_DEBUGE("PLLUSB1 did not shut down within the specified time period");
                    ret = HAL_ERR_TIMEOUT;
                    break;
                } else {
                    /* do nothing */
                }
            }

            if(HAL_ERR_NONE == ret) {
                /* configuration pllusb1 parameter */
                _rcu_pllusb1_config(&p_pll->pllusb1);
                hals_rcu_osci_on(RCU_PLLUSBHS1_CK);

                /* wait till PLLUSBHS1 is stable */
                if(ERROR == hals_rcu_osci_stab_wait(RCU_PLLUSBHS1_CK)) {
                    HAL_DEBUGE("PLLUSB1 is not stabilized for the required amount of time");
                    ret = HAL_ERR_TIMEOUT;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        ret = HAL_ERR_VAL;
    }

    return ret;
}

/*!
    \brief      get the pll configuration
    \param[in]  p_pll: pointer of pll configuration struct
                  need_configure: configure flag
                  only one parameter can be selected from the following:
      \arg          ENABLE:enable the pll
      \arg          DISABLE:disable the pll
                  pll_source source: refer to hal_rcu_pll_src_enum
                  pll0: PLL0 configuration struct, refer to hal_rcu_pll0_struct
                  pll1: PLL1 configuration struct, refer to hal_rcu_pll1_struct
                  pll2: PLL2 configuration struct, refer to hal_rcu_pll2_struct
                  pllusb0: PLLUSB0 configuration struct, refer to hal_rcu_pllusb0_struct
                  pllusb1: PLLUSB1 configuration struct, refer to hal_rcu_pllusb1_struct
                  state: the argument could be selected from enumeration hal_rcu_osc_state_enum
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rcu_pll_config_get(hal_rcu_pll_struct *p_pll)
{
    uint32_t temp_pll_src = 0U;
    uint32_t temp_pll0_psc = 0U, temp_pll0_n = 0U, temp_pll0_p = 0U, temp_pll0_q = 0U, temp_pll0_r = 0U;
    uint32_t temp_pll1_psc = 0U, temp_pll1_n = 0U, temp_pll1_p = 0U, temp_pll1_q = 0U, temp_pll1_r = 0U;
    uint32_t temp_pll1_input = 0U, temp_pll1_output = 0U, temp_pll1_fra = 0U;
    uint32_t temp_pll2_psc = 0U, temp_pll2_n = 0U, temp_pll2_p = 0U, temp_pll2_q = 0U, temp_pll2_r = 0U;
    uint32_t temp_pll2_input = 0U, temp_pll2_output = 0U, temp_pll2_fra = 0U;
    uint32_t temp_pllusb0_precksrc = 0U, temp_pllusb0_prediv = 0U, temp_pllusb0_mul = 0U;
    uint32_t temp_pllusb1_precksrc = 0U, temp_pllusb1_prediv = 0U, temp_pllusb1_mul = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_pll) {
        HAL_DEBUGE("pointer [rcu_osci] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* get PLL clock source */
    temp_pll_src = RCU_PLLALL & RCU_PLLALL_PLLSEL;
    p_pll->pll_source     = (hal_rcu_pll_src_enum)temp_pll_src;

    /* get PLL0 parameters */
    temp_pll0_psc = RCU_PLL0 & RCU_PLL0_PLL0PSC;
    temp_pll0_n   = RCU_PLL0 & RCU_PLL0_PLL0N;
    temp_pll0_p   = RCU_PLL0 & RCU_PLL0_PLL0P;
    temp_pll0_q   = RCU_PLLADDCTL & RCU_PLLADDCTL_PLL0Q;
    temp_pll0_r   = RCU_PLL0 & RCU_PLL0_PLL0R;

    p_pll->pll0.pll0_psc              = (hal_rcu_pll_psc_enum)temp_pll0_psc;
    p_pll->pll0.pll0_n                = (hal_rcu_plln_enum)temp_pll0_n;
    p_pll->pll0.pll0_p                = (hal_rcu_pllp_enum)temp_pll0_p;
    p_pll->pll0.pll0_q                = (hal_rcu_pllq_enum)temp_pll0_q;
    p_pll->pll0.pll0_r                = (hal_rcu_pllr_enum)temp_pll0_r;
    p_pll->pll0.pll0_input_fre_range  = RCU_PLLALL & RCU_PLLALL_PLL0RNG;
    p_pll->pll0.pll0_output_fre_range = RCU_PLLALL & RCU_PLLALL_PLL0VCOSEL;
    p_pll->pll0.pll0_fractional_latch = RCU_PLL0FRA & RCU_PLL0FRA_PLL0FRAN;

    /* get PLL1 parameters */
    temp_pll1_psc    = RCU_PLL1 & RCU_PLL1_PLL1PSC;
    temp_pll1_n      = RCU_PLL1 & RCU_PLL1_PLL1N;
    temp_pll1_p      = RCU_PLL1 & RCU_PLL1_PLL1P;
    temp_pll1_q      = RCU_PLLADDCTL & RCU_PLLADDCTL_PLL1Q;
    temp_pll1_r      = RCU_PLL1 & RCU_PLL1_PLL1R;
    temp_pll1_input  = RCU_PLLALL & RCU_PLLALL_PLL1RNG;
    temp_pll1_output = RCU_PLLALL & RCU_PLLALL_PLL1VCOSEL;
    temp_pll1_fra    = RCU_PLL1FRA & RCU_PLL1FRA_PLL1FRAN;

    p_pll->pll1.pll1_psc              = (hal_rcu_pll_psc_enum)temp_pll1_psc;
    p_pll->pll1.pll1_n                = (hal_rcu_plln_enum)temp_pll1_n;
    p_pll->pll1.pll1_p                = (hal_rcu_pllp_enum)temp_pll1_p;
    p_pll->pll1.pll1_q                = (hal_rcu_pllq_enum)temp_pll1_q;
    p_pll->pll1.pll1_r                = (hal_rcu_pllr_enum)temp_pll1_r;
    p_pll->pll1.pll1_input_fre_range  = temp_pll1_input;
    p_pll->pll1.pll1_output_fre_range = temp_pll1_output;
    p_pll->pll1.pll1_fractional_latch = temp_pll1_fra;

    /* get PLL2 parameters */
    temp_pll2_psc    = RCU_PLL2 & RCU_PLL2_PLL2PSC;
    temp_pll2_n      = RCU_PLL2 & RCU_PLL2_PLL2N;
    temp_pll2_p      = RCU_PLL2 & RCU_PLL2_PLL2P;
    temp_pll2_q      = RCU_PLLADDCTL & RCU_PLLADDCTL_PLL2Q;
    temp_pll2_r      = RCU_PLL2 & RCU_PLL2_PLL2R;
    temp_pll2_input  = RCU_PLLALL & RCU_PLLALL_PLL2RNG;
    temp_pll2_output = RCU_PLLALL & RCU_PLLALL_PLL2VCOSEL;
    temp_pll2_fra    = RCU_PLL2FRA & RCU_PLL2FRA_PLL2FRAN;

    p_pll->pll2.pll2_psc              = (hal_rcu_pll_psc_enum)temp_pll2_psc;
    p_pll->pll2.pll2_n                = (hal_rcu_plln_enum)temp_pll2_n;
    p_pll->pll2.pll2_p                = (hal_rcu_pllp_enum)temp_pll2_p;
    p_pll->pll2.pll2_q                = (hal_rcu_pllq_enum)temp_pll2_q;
    p_pll->pll2.pll2_r                = (hal_rcu_pllr_enum)temp_pll2_r;
    p_pll->pll2.pll2_input_fre_range  = temp_pll2_input;
    p_pll->pll2.pll2_output_fre_range = temp_pll2_output;
    p_pll->pll2.pll2_fractional_latch = temp_pll2_fra;

    /* get PLLUSB0 parameters */
    temp_pllusb0_precksrc = RCU_PLLUSBCFG & RCU_USBCLKCTL_PLLUSBHS0PRESEL;
    temp_pllusb0_prediv   = RCU_PLLUSBCFG & RCU_PLLUSBCFG_PLLUSBHS0PREDV;
    temp_pllusb0_mul      = RCU_PLLUSBCFG & RCU_PLLUSBCFG_USBHS0DV;

    p_pll->pllusb0.pllusb0_preclock_source = (hal_rcu_pllusbhsx_precksrc_enum)temp_pllusb0_precksrc;
    p_pll->pllusb0.pllusb0_presel          = (hal_rcu_pllusbhsx_prediv_enum)temp_pllusb0_prediv;
    p_pll->pllusb0.pllusb0_mul             = (hal_rcu_pllusbhsx_mul_enum)temp_pllusb0_mul;

    /* get PLLUSB1 parameters */
    temp_pllusb1_precksrc = RCU_PLLUSBCFG & RCU_USBCLKCTL_PLLUSBHS1PRESEL;
    temp_pllusb1_prediv   = RCU_PLLUSBCFG & RCU_PLLUSBCFG_PLLUSBHS1PREDV;
    temp_pllusb1_mul      = RCU_PLLUSBCFG & RCU_PLLUSBCFG_USBHS1DV;

    p_pll->pllusb1.pllusb1_preclock_source = (hal_rcu_pllusbhsx_precksrc_enum)temp_pllusb1_precksrc;
    p_pll->pllusb1.pllusb1_presel          = (hal_rcu_pllusbhsx_prediv_enum)temp_pllusb1_prediv;
    p_pll->pllusb1.pllusb1_mul             = (hal_rcu_pllusbhsx_mul_enum)temp_pllusb1_mul;

    return HAL_ERR_NONE;
}


/*!
    \brief      initialize the RCU extended peripherals clocks
    \param[in]  periph_clk: the pointer of peripheral clock structure
                  periph_clock_type(member):
                  only one parameter can be selected which is shown as below:
      \arg          RCU_PERIPH_CLKTYPE_NONE : no peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_RTC       : RTC peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_APB1TIMER : APB1 timer peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_APB2TIMER : APB2 timer peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_TRNG : TRNG peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_USBHS0 : USBHS0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_USBHS1 : USBHS0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_CAN0 : CAN0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_CAN1 : CAN1 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_CAN2 : CAN2 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SAI2B0 : SAI2B0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SAI2B1 : SAI2B0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SAI0 : SAI0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SAI1 : SAI1 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_EXMC : EXMC peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SPI0 : SPI0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SPI1 : SPI1 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SPI2 : SPI2 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SPI3 : SPI3 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SPI4 : SPI4 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SPI5 : SPI5 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SDIO0 : SDIO0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SDIO1 : SDIO1 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_HPDF : HPDF peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_USART0 : USART0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_USART1 : USART1 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_USART2 : USART2 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_USART5 : USART5 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_I2C0 : I2C0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_I2C1 : I2C1 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_I2C2 : I2C2 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_I2C3 : I2C3 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_RSPDIF : RSPDIF peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_ADC0 : ADC0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_ADC1 : ADC1 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_ADC2 : ADC2 peripheral clock to configure
                  hals_rcu_rtc_clock_config(member1, member2):
                  parameter member1 only one can be selected which is shown as below:
      \arg          RCU_RTC_CLKSRC_NONE: RTC has no clock source
      \arg          RCU_RTC_CLKSRC_LXTAL: RTC select LXTAL as clock source
      \arg          RCU_RTC_CLKSRC_IRC40K: RTC select IRC40K as clock source
      \arg          RCU_RTC_CLKSRC_HXTAL_DIV32: RTC select HXTAL/32 as clock source
                  parameter member2 only one can be selected which is shown as below:
      \arg          RCU_RTC_HXTAL_DIV_NONE, RCU_RTC_HXTAL_DIV_2, ... , RCU_RTC_HXTAL_DIV_63
                  hals_rcu_timer_clock_multiple_config(member):
                  only one parameter can be selected which is shown as below:
      \arg          RCU_TIMER_MUL2: Timer prescaler Multiple by 2
      \arg          RCU_TIMER_MUL4: Timer prescaler Multiple by 4
                  hals_rcu_trng_clock_config(member1, member2)
                  parameter member1 only one can be selected which is shown as below:
      \arg          RCU_CK48M_CKSRC_PLL48M: CK48M Clock Source Selection: PLL48M
      \arg          RCU_CK48M_CKSRC_IRC48M: CK48M Clock Source Selection: IRC48M
                  parameter member2 only one can be selected which is shown as below:
      \arg          RCU_PLL48M_CKSRC_PLL0Q: PLL48MSEL Clock Source Selection: CK_PLL0Q
      \arg          RCU_PLL48M_CKSRC_PLL2P: PLL48MSEL Clock Source Selection: IRC48M
                  hals_rcu_trng_clock_config(member1, member2)
                  parameter member1 only one can be selected which is shown as below:
      \arg          RCU_CK48M_CKSRC_PLL48M: CK48M Clock Source Selection: PLL48M
      \arg          RCU_CK48M_CKSRC_IRC48M: CK48M Clock Source Selection: IRC48M
                  parameter member2 only one can be selected which is shown as below:
      \arg          RCU_PLL48M_CKSRC_PLL0Q: PLL48MSEL Clock Source Selection: CK_PLL0Q
      \arg          RCU_PLL48M_CKSRC_PLL2P: PLL48MSEL Clock Source Selection: IRC48M
                  hals_rcu_usbhs_div_config
                  only one parameter can be selected which is shown as below:
      \arg          RCU_USBHSx_DIV_2 \ RCU_USBHSx_DIV_4 \ RCU_USBHSx_DIV_6
                    RCU_USBHSx_DIV_8 \ RCU_USBHSx_DIV_10 \ RCU_USBHSx_DIV_12
                    RCU_USBHSx_DIV_14 \ RCU_USBHSx_DIV_16
                  hals_rcu_usbhs_psc_config
                  only one parameter can be selected which is shown as below:
      \arg          RCU_USBHSx_PSC_1 \ RCU_USBHSx_PSC_2 \ RCU_USBHSx_PSC_3
                    RCU_USBHSx_PSC_4 \ RCU_USBHSx_PSC_5 \ RCU_USBHSx_PSC_6
                    RCU_USBHSx_PSC_7 \ RCU_USBHSx_PSC_8
                  hals_rcu_usb48m_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_USBHSx48M_CKSRC_PLL0R :PLLUSBx 48M Clock Source: CK_PLL0R
    \arg            RCU_USBHSx48M_CKSRC_USBHSDV :PLLUSBx 48M Clock Source: CK_USBHSxDV\CK_PLLUSBHSx
    \arg            RCU_USBHSx48M_CKSRC_PLL1Q :PLLUSBx 48M Clock Source: CK_USBHSxPSC\PLL1Q
    \arg            RCU_USBHSx48M_CKSRC_IRC48M :PLLUSBx 48M Clock Source: CK_IRC48M
                  hals_rcu_usbhs_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_USBHSx_CKSRC_CK48M : USBHSx clock Source: 48M
    \arg            RCU_USBHSx_CKSRC_CK60M : USBHSx clock Source: 60M
                  hals_rcu_can_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_CAN_CKSRC_CK_HXTAL : CAN clock Source: hxtal
    \arg            RCU_CAN_CKSRC_CK_APB : CAN clock Source: apb2
    \arg            RCU_CAN_CKSRC_CK_APB2DIV2 : CAN clock Source: apb2 / 2
    \arg            RCU_CAN_CKSRC_CK_IRC64MDIV : CAN clock Source: irc64mdiv
                  hals_rcu_sai2b_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_SAI2B_CKSRC_CK_PLL0Q : SAI2Bx clock Source : CK_PLL0Q
    \arg            RCU_SAI2B_CKSRC_CK_PLL1P : SAI2Bx clock Source : CK_PLL1P
    \arg            RCU_SAI2B_CKSRC_CK_PLL2P : SAI2Bx clock Source : CK_PLL2P
    \arg            RCU_SAI2B_CKSRC_I2S_CKIN : SAI2Bx clock Source : I2S_CKIN
    \arg            RCU_SAI2B_CKSRC_CK_PER : SAI2Bx clock Source : CK_PER
    \arg            RCU_SAI2B_CKSRC_CK_RSPDIF_SYMB : SAI2Bx clock Source : CK_RSPDIF_SYMB
                  hals_rcu_sai_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_SAI_CKSRC_CK_PLL0Q : SAIx clock Source : CK_PLL0Q
    \arg            RCU_SAI_CKSRC_CK_PLL1P : SAIx clock Source : CK_PLL1P
    \arg            RCU_SAI_CKSRC_CK_PLL2P : SAIx clock Source : CK_PLL2P
    \arg            RCU_SAI_CKSRC_I2S_CKIN : SAIx clock Source : I2S_CKIN
    \arg            RCU_SAI_CKSRC_CK_PER : SAIx clock Source : CK_PER
                  hals_rcu_exmc_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_EXMC_CKSRC_CK_AHB : EXMC clock src: CK_AHB
    \arg            RCU_EXMC_CKSRC_CK_PLL0Q : EXMC clock src: CK_PLL0Q
    \arg            RCU_EXMC_CKSRC_CK_PLL1R : EXMC clock src: CK_PLL1R
    \arg            RCU_EXMC_CKSRC_CK_PER : EXMC clock src: CK_PER
                  hals_rcu_spi_clock_config
                  only one parameter can be selected which is shown as below:
                    <hal_rcu_spi012_cksrc_enum>, <hal_rcu_spi34_cksrc_enum>, <hal_rcu_spi5_cksrc_enum>
                  hals_rcu_sdio_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_SDIO0_CKSRC_CK_PLL0Q : SDIO0 clock src: CK_PLL0Q
    \arg            RCU_SDIO0_CKSRC_CK_PLL1R : SDIO0 clock src: CK_PLL1R
    \arg            RCU_SDIO1_CKSRC_CK_PLL0Q : SDIO1 clock src: CK_PLL0Q
    \arg            RCU_SDIO1_CKSRC_CK_PLL1R : SDIO1 clock src: CK_PLL1R
                  hals_rcu_hpdf_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_HPDF_CKSRC_CK_APB2 : HPDF clock src: CK_APB2
    \arg            RCU_HPDF_CKSRC_CK_AHB : HPDF clock src: CK_AHB
                  hals_rcu_usart_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_USART_CKSRC_CK_APB : USART clock src: CK_APB
    \arg            RCU_USART_CKSRC_CK_AHB : USART clock src: CK_AHB
    \arg            RCU_USART_CKSRC_CK_LXTAL : USART clock src: CK_LXTAL
    \arg            RCU_USART_CKSRC_CK_IRC64MDIV : USART clock src: CK_IRC64MDIV
                  hals_rcu_i2c_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_I2C_CKSRC_CK_APB1 : I2C clock src: CK_APB1
    \arg            RCU_I2C_CKSRC_CK_PLL2R : I2C clock src: CK_PLL2R
    \arg            RCU_I2C_CKSRC_CK_IRC64MDIV : I2C clock src: CK_IRC64MDIV
    \arg            RCU_I2C_CKSRC_CK_LPIRC4M : I2C clock src: CK_LPIRC4M
                  hals_rcu_rspdif_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_RSPDIF_CKSRC_CK_PLL0Q : RSPDIF clock src: CK_PLL0Q
    \arg            RCU_RSPDIF_CKSRC_CK_PLL1R : RSPDIF clock src: CK_PLL1R
    \arg            RCU_RSPDIF_CKSRC_CK_PLL2R : RSPDIF clock src: CK_PLL2R
    \arg            RCU_RSPDIF_CKSRC_CK_IRC64MDIV : RSPDIF clock src: CK_IRC64MDIV
                  hals_rcu_adc_clock_config(member):
                  only one parameter can be selected which is shown as below:
      \arg          RCU_ADC_CKSRC_CK_PLL1P: ADC clock source select CK_PLL1P
      \arg          RCU_ADC_CKSRC_CK_PLL2R: ADC clock source select CK_PLL2R
      \arg          RCU_ADC_CKSRC_CK_PER: ADC clock source select CK_PER
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_rcu_periph_clock_config(hal_rcu_periphclk_struct *periph_clk)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == periph_clk) {
        HAL_DEBUGE("pointer [periph_clk] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(periph_clk->periph_clock_type > RCU_PERIPH_CLKTYPE_MAX) {
        HAL_DEBUGE("parameter [periph_clk->periph_clock_type] value is invalid.");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure the RTC clock source */
    if(RCU_PERIPH_CLKTYPE_RTC == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_RTC)) {
        /* need to enable pmu clock */
        if(RESET == (RCU_APB4EN & RCU_APB4EN_PMUEN)) {
            hal_rcu_periph_clock_enable(RCU_PMU);
        } else {
            /* do nothing */
        }

        /* enable the backup domain access */
        hal_pmu_backup_write_enable();

        /* configuration RTC clock source */
        switch(periph_clk->rtc_clock_source) {
        case RCU_RTC_CLKSRC_LXTAL:
            _rcu_lxtal_clock_source_select();
            hals_rcu_rtc_clock_config(RCU_RTC_CLKSRC_LXTAL, RCU_RTC_HXTAL_DIV_NONE);
            break;
        case RCU_RTC_CLKSRC_IRC32K:
            _rcu_irc32k_clock_source_select();
            hals_rcu_rtc_clock_config(RCU_RTC_CLKSRC_IRC32K, RCU_RTC_HXTAL_DIV_NONE);
            break;
        case RCU_RTC_CLKSRC_HXTAL_DIV_RTCDIV:
            _rcu_hxtal_clock_source_select();
            hals_rcu_rtc_clock_config(RCU_RTC_CLKSRC_HXTAL_DIV_RTCDIV, periph_clk->rtc_div);
            break;
        case RCU_RTC_CLKSRC_NONE:
            break;
        default:
            HAL_DEBUGE("parameter [periph_clk->rtc_clock_source] value is invalid.");
            ret = HAL_ERR_VAL;
            break;
        }

        /* disable PMU clock */
        hal_rcu_periph_clock_disable(RCU_PMU);
    } else {
        /* do nothing */
    }

    if(HAL_ERR_NONE == ret) {
        /* configure the timer multiple */
        if(RCU_PERIPH_CLKTYPE_TIMER == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_TIMER)) {
            hals_rcu_timer_clock_multiple_config(periph_clk->timer_mul);
        } else {
            /* do nothing */
        }

        /* configure the TRNG clock source */
        if(RCU_PERIPH_CLKTYPE_TRNG == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_TRNG)) {
            hals_rcu_trng_clock_config(periph_clk->pll48m_clock_source, periph_clk->ck48m_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the usbhs0 clock source */
        if(RCU_PERIPH_CLKTYPE_USBHS0 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_USBHS0)) {
            /* configuration usbhs48m source */
            if(RCU_USBHSx48M_CKSRC_PLL1Q == periph_clk->usb0_clock.usbhs48m_source) {
                hals_rcu_usbhs_psc_config(INDEX_USBHS0, periph_clk->usb0_clock.usbhs_psc);
            } else if(RCU_USBHSx48M_CKSRC_USBHSDV == periph_clk->usb0_clock.usbhs48m_source) {
                hals_rcu_usbhs_div_config(INDEX_USBHS0, periph_clk->usb0_clock.usbhs_div);
            } else {
                /* do nothing */
            }

            /* configuration USBHS0 usb48m clock source */
            hals_rcu_usb48m_clock_config(INDEX_USBHS0, periph_clk->usb0_clock.usbhs48m_source);

            /* configuration USBHS0 usbhs source */
            hals_rcu_usbhs_clock_config(INDEX_USBHS0, periph_clk->usb0_clock.usbhs_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the usbhs1 clock source */
        if(RCU_PERIPH_CLKTYPE_USBHS1 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_USBHS1)) {
            /* configuration usbhs48m source */
            if(RCU_USBHSx48M_CKSRC_PLL1Q == periph_clk->usb1_clock.usbhs48m_source) {
                hals_rcu_usbhs_psc_config(INDEX_USBHS1, periph_clk->usb1_clock.usbhs_psc);
            } else if(RCU_USBHSx48M_CKSRC_USBHSDV == periph_clk->usb1_clock.usbhs48m_source) {
                hals_rcu_usbhs_div_config(INDEX_USBHS1, periph_clk->usb1_clock.usbhs_div);
            } else {
                /* do nothing */
            }

            /* configuration USBHS1 usb48m clock source */
            hals_rcu_usb48m_clock_config(INDEX_USBHS1, periph_clk->usb1_clock.usbhs48m_source);

            /* configuration USBHS1 usbhs source */
            hals_rcu_usbhs_clock_config(INDEX_USBHS1, periph_clk->usb1_clock.usbhs_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the CAN0 clock */
        if(RCU_PERIPH_CLKTYPE_CAN0 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_CAN0)) {
            if(RCU_CAN_CKSRC_CK_IRC64MDIV == periph_clk->can0_clock_source) {
                _rcu_irc64m_clock_source_select(periph_clk->irc64m_div);
            } else if(RCU_CAN_CKSRC_CK_HXTAL == periph_clk->can0_clock_source) {
                _rcu_hxtal_clock_source_select();
            } else {
                /* do nothing */
            }

            /* configuration CAN0 clock source */
            hals_rcu_can_clock_config(INDEX_CAN0, periph_clk->can0_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the CAN1 clock */
        if(RCU_PERIPH_CLKTYPE_CAN1 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_CAN1)) {
            if(RCU_CAN_CKSRC_CK_IRC64MDIV == periph_clk->can1_clock_source) {
                _rcu_irc64m_clock_source_select(periph_clk->irc64m_div);
            } else if(RCU_CAN_CKSRC_CK_HXTAL == periph_clk->can1_clock_source) {
                _rcu_hxtal_clock_source_select();
            } else {
                /* do nothing */
            }

            /* configuration CAN1 clock source */
            hals_rcu_can_clock_config(INDEX_CAN1, periph_clk->can1_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the CAN2 clock */
        if(RCU_PERIPH_CLKTYPE_CAN2 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_CAN2)) {
            if(RCU_CAN_CKSRC_CK_IRC64MDIV == periph_clk->can2_clock_source) {
                _rcu_irc64m_clock_source_select(periph_clk->irc64m_div);
            } else if(RCU_CAN_CKSRC_CK_HXTAL == periph_clk->can2_clock_source) {
                _rcu_hxtal_clock_source_select();
            } else {
                /* do nothing */
            }

            /* configuration CAN2 clock source */
            hals_rcu_can_clock_config(INDEX_CAN2, periph_clk->can2_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the SAI2B0 clock */
        if(RCU_PERIPH_CLKTYPE_SAI2B0 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_SAI2B0)) {
            if(RCU_SAI2B_CKSRC_CK_PER == periph_clk->sai2b0_clock_source) {
                if(HAL_ERR_NONE != _rcu_per_clock_source_select(periph_clk)) {
                    HAL_DEBUGE("parameter [periph_clk->pef_clock_source] value is invalid.");
                    ret = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            /* configuration SAI2B0 clock source */
            if((HAL_ERR_NONE == ret)) {
                hals_rcu_sai2b_clock_config(INDEX_SAI2B0, periph_clk->sai2b0_clock_source);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* configure the SAI2B1 clock */
        if((RCU_PERIPH_CLKTYPE_SAI2B1 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_SAI2B1)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_SAI2B_CKSRC_CK_PER == periph_clk->sai2b1_clock_source) {
                if(HAL_ERR_NONE != _rcu_per_clock_source_select(periph_clk)) {
                    HAL_DEBUGE("parameter [periph_clk->pef_clock_source] value is invalid");
                    ret = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            /* configuration SAI2B1 clock source */
            if((HAL_ERR_NONE == ret)) {
                hals_rcu_sai2b_clock_config(INDEX_SAI2B1, periph_clk->sai2b1_clock_source);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* configure the SAI0 clock */
        if((RCU_PERIPH_CLKTYPE_SAI0 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_SAI0)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_SAI_CKSRC_CK_PER == periph_clk->sai0_clock_source) {
                if(HAL_ERR_NONE != _rcu_per_clock_source_select(periph_clk)) {
                    HAL_DEBUGE("parameter [periph_clk->pef_clock_source] value is invalid");
                    ret = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            /* configuration SAI0 clock source */
            if(HAL_ERR_NONE == ret) {
                hals_rcu_sai_clock_config(INDEX_SAI0, periph_clk->sai0_clock_source);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* configure the SAI1 clock */
        if((RCU_PERIPH_CLKTYPE_SAI1 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_SAI1)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_SAI_CKSRC_CK_PER == periph_clk->sai1_clock_source) {
                if(HAL_ERR_NONE != _rcu_per_clock_source_select(periph_clk)) {
                    HAL_DEBUGE("parameter [periph_clk->pef_clock_source] value is invalid");
                    ret = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            /* configuration SAI1 clock source */
            if(HAL_ERR_NONE == ret) {
                hals_rcu_sai_clock_config(INDEX_SAI1, periph_clk->sai1_clock_source);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* configure the EXMC clock */
        if((RCU_PERIPH_CLKTYPE_EXMC == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_EXMC)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_EXMC_CKSRC_CK_PER == periph_clk->exmc_clock_source) {
                if(HAL_ERR_NONE != _rcu_per_clock_source_select(periph_clk)) {
                    HAL_DEBUGE("parameter [periph_clk->pef_clock_source] value is invalid");
                    ret = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            if(HAL_ERR_NONE == ret) {
                /* configuration EXMC clock source */
                hals_rcu_exmc_clock_config(periph_clk->exmc_clock_source);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* configure the SPI0 clock */
        if((RCU_PERIPH_CLKTYPE_SPI0 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_SPI0)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_SPI_012_CKSRC_CK_PER == periph_clk->spi0_clock_source) {
                if(HAL_ERR_NONE != _rcu_per_clock_source_select(periph_clk)) {
                    HAL_DEBUGE("parameter [periph_clk->pef_clock_source] value is invalid");
                    ret = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            /* configuration SPI0 clock source */
            if(HAL_ERR_NONE == ret) {
                hals_rcu_spi_clock_config(INDEX_SPI0, (uint32_t)periph_clk->spi0_clock_source);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* configure the SPI1 clock */
        if((RCU_PERIPH_CLKTYPE_SPI1 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_SPI1)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_SPI_012_CKSRC_CK_PER == periph_clk->spi1_clock_source) {
                if(HAL_ERR_NONE != _rcu_per_clock_source_select(periph_clk)) {
                    HAL_DEBUGE("parameter [periph_clk->pef_clock_source] value is invalid");
                    ret = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            /* configuration SPI1 clock source */
            if(HAL_ERR_NONE == ret) {
                hals_rcu_spi_clock_config(INDEX_SPI1, (uint32_t)periph_clk->spi1_clock_source);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* configure the SPI2 clock */
        if((RCU_PERIPH_CLKTYPE_SPI2 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_SPI2)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_SPI_012_CKSRC_CK_PER == periph_clk->spi2_clock_source) {
                if(HAL_ERR_NONE != _rcu_per_clock_source_select(periph_clk)) {
                    HAL_DEBUGE("parameter [periph_clk->pef_clock_source] value is invalid");
                    ret = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            /* configuration SPI2 clock source */
            if(HAL_ERR_NONE == ret) {
                hals_rcu_spi_clock_config(INDEX_SPI2, (uint32_t)periph_clk->spi2_clock_source);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* configure the SPI3 clock */
        if((RCU_PERIPH_CLKTYPE_SPI3 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_SPI3)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_SPI_34_CKSRC_CK_IRC64MDIV == periph_clk->spi3_clock_source) {
                _rcu_irc64m_clock_source_select(periph_clk->irc64m_div);
            } else if(RCU_SPI_34_CKSRC_CK_LPIRC4M == periph_clk->spi3_clock_source) {
                _rcu_lpirc4m_clock_source_select();
            } else if(RCU_SPI_34_CKSRC_CK_HXTAL == periph_clk->spi3_clock_source) {
                _rcu_hxtal_clock_source_select();
            } else {
                /* do nothing */
            }

            /* configuration SPI3 clock source */
            hals_rcu_spi_clock_config(INDEX_SPI3, (uint32_t)periph_clk->spi3_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the SPI4 clock */
        if((RCU_PERIPH_CLKTYPE_SPI4 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_SPI4)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_SPI_34_CKSRC_CK_IRC64MDIV == periph_clk->spi4_clock_source) {
                _rcu_irc64m_clock_source_select(periph_clk->irc64m_div);
            } else if(RCU_SPI_34_CKSRC_CK_LPIRC4M == periph_clk->spi4_clock_source) {
                _rcu_lpirc4m_clock_source_select();
            } else if(RCU_SPI_34_CKSRC_CK_HXTAL == periph_clk->spi4_clock_source) {
                _rcu_hxtal_clock_source_select();
            } else {
                /* do nothing */
            }

            /* configuration SPI4 clock */
            hals_rcu_spi_clock_config(INDEX_SPI4, (uint32_t)periph_clk->spi4_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the SPI5 clock */
        if((RCU_PERIPH_CLKTYPE_SPI5 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_SPI5)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_SPI_5_CKSRC_CK_IRC64MDIV == periph_clk->spi5_clock_source) {
                _rcu_irc64m_clock_source_select(periph_clk->irc64m_div);
            } else if(RCU_SPI_5_CKSRC_CK_LPIRC4M == periph_clk->spi5_clock_source) {
                _rcu_lpirc4m_clock_source_select();
            } else if(RCU_SPI_5_CKSRC_CK_HXTAL == periph_clk->spi5_clock_source) {
                _rcu_hxtal_clock_source_select();
            } else {
                /* do nothing */
            }

            /* configuration SPI5 clock */
            hals_rcu_spi_clock_config(INDEX_SPI5, (uint32_t)periph_clk->spi5_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the SDIO0 clock source */
        if((RCU_PERIPH_CLKTYPE_SDIO0 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_SDIO0)) && \
           (HAL_ERR_NONE == ret)) {
            hals_rcu_sdio_clock_config(INDEX_SDIO0, periph_clk->sdio0_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the SDIO1 clock source */
        if((RCU_PERIPH_CLKTYPE_SDIO1 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_SDIO1)) && \
           (HAL_ERR_NONE == ret)) {
            hals_rcu_sdio_clock_config(INDEX_SDIO1, periph_clk->sdio1_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the HPDF clock source */
        if((RCU_PERIPH_CLKTYPE_HPDF == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_HPDF)) && \
           (HAL_ERR_NONE == ret)) {
            hals_rcu_hpdf_clock_config(periph_clk->hpdf_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the USART0 clock */
        if((RCU_PERIPH_CLKTYPE_USART0 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_USART0)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_USART_CKSRC_CK_IRC64MDIV == periph_clk->usart0_clock_source) {
                _rcu_irc64m_clock_source_select(periph_clk->irc64m_div);
            } else if(RCU_USART_CKSRC_CK_LXTAL == periph_clk->usart0_clock_source) {
                _rcu_lxtal_clock_source_select();
            } else {
                /* do nothing */
            }

            /* configuration USART0 clock source */
            hals_rcu_usart_clock_config(INDEX_USART0, periph_clk->usart0_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the USART1 clock */
        if((RCU_PERIPH_CLKTYPE_USART1 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_USART1)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_USART_CKSRC_CK_IRC64MDIV == periph_clk->usart1_clock_source) {
                _rcu_irc64m_clock_source_select(periph_clk->irc64m_div);
            } else if(RCU_USART_CKSRC_CK_LXTAL == periph_clk->usart1_clock_source) {
                _rcu_lxtal_clock_source_select();
            } else {
                /* do nothing */
            }

            /* configuration USART1 clock source */
            hals_rcu_usart_clock_config(INDEX_USART1, periph_clk->usart1_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the USART2 clock */
        if((RCU_PERIPH_CLKTYPE_USART2 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_USART2)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_USART_CKSRC_CK_IRC64MDIV == periph_clk->usart2_clock_source) {
                _rcu_irc64m_clock_source_select(periph_clk->irc64m_div);
            } else if(RCU_USART_CKSRC_CK_LXTAL == periph_clk->usart2_clock_source) {
                _rcu_lxtal_clock_source_select();
            } else {
                /* do nothing */
            }

            /* configuration USART2 clock source */
            hals_rcu_usart_clock_config(INDEX_USART2, periph_clk->usart2_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the USART5 clock */
        if((RCU_PERIPH_CLKTYPE_USART5 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_USART5)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_USART_CKSRC_CK_IRC64MDIV == periph_clk->usart5_clock_source) {
                _rcu_irc64m_clock_source_select(periph_clk->irc64m_div);
            } else if(RCU_USART_CKSRC_CK_LXTAL == periph_clk->usart5_clock_source) {
                _rcu_lxtal_clock_source_select();
            } else {
                /* do nothing */
            }

            /* configuration USART5 clock source */
            hals_rcu_usart_clock_config(INDEX_USART5, periph_clk->usart5_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the I2C0 clock */
        if((RCU_PERIPH_CLKTYPE_I2C0 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_I2C0)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_I2C_CKSRC_CK_IRC64MDIV == periph_clk->i2c0_clock_source) {
                _rcu_irc64m_clock_source_select(periph_clk->irc64m_div);
            } else if(RCU_I2C_CKSRC_CK_LPIRC4M == periph_clk->i2c0_clock_source) {
                _rcu_lpirc4m_clock_source_select();
            } else {
                /* do nothing */
            }

            /* configuration I2C0 clock source */
            hals_rcu_i2c_clock_config(INDEX_I2C0, periph_clk->i2c0_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the I2C1 clock */
        if((RCU_PERIPH_CLKTYPE_I2C1 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_I2C1)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_I2C_CKSRC_CK_IRC64MDIV == periph_clk->i2c1_clock_source) {
                _rcu_irc64m_clock_source_select(periph_clk->irc64m_div);
            } else if(RCU_I2C_CKSRC_CK_LPIRC4M == periph_clk->i2c1_clock_source) {
                _rcu_lpirc4m_clock_source_select();
            } else {
                /* do nothing */
            }

            /* configuration I2C1 clock source */
            hals_rcu_i2c_clock_config(INDEX_I2C1, periph_clk->i2c1_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the I2C2 clock */
        if((RCU_PERIPH_CLKTYPE_I2C2 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_I2C2)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_I2C_CKSRC_CK_IRC64MDIV == periph_clk->i2c2_clock_source) {
                _rcu_irc64m_clock_source_select(periph_clk->irc64m_div);
            } else if(RCU_I2C_CKSRC_CK_LPIRC4M == periph_clk->i2c2_clock_source) {
                _rcu_lpirc4m_clock_source_select();
            } else {
                /* do nothing */
            }

            /* configuration I2C2 clock source */
            hals_rcu_i2c_clock_config(INDEX_I2C2, periph_clk->i2c2_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the I2C3 clock */
        if((RCU_PERIPH_CLKTYPE_I2C3 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_I2C3)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_I2C_CKSRC_CK_IRC64MDIV == periph_clk->i2c3_clock_source) {
                _rcu_irc64m_clock_source_select(periph_clk->irc64m_div);
            } else if(RCU_I2C_CKSRC_CK_LPIRC4M == periph_clk->i2c3_clock_source) {
                _rcu_lpirc4m_clock_source_select();
            } else {
                /* do nothing */
            }

            /* configuration I2C3 clock source */
            hals_rcu_i2c_clock_config(INDEX_I2C3, periph_clk->i2c3_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the RSPDIF clock source */
        if((RCU_PERIPH_CLKTYPE_RSPDIF == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_RSPDIF)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_RSPDIF_CKSRC_CK_IRC64MDIV == periph_clk->rspdif_clock_source) {
                _rcu_irc64m_clock_source_select(periph_clk->irc64m_div);
            } else {
                /* do nothing */
            }
            hals_rcu_rspdif_clock_config(periph_clk->rspdif_clock_source);
        } else {
            /* do nothing */
        }

        /* configure the ADC0/ADC1 clock */
        if((RCU_PERIPH_CLKTYPE_ADC01 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_ADC01)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_ADC_CKSRC_CK_PER == periph_clk->adc01_clock_source) {
                if(HAL_ERR_NONE != _rcu_per_clock_source_select(periph_clk)) {
                    HAL_DEBUGE("parameter [periph_clk->pef_clock_source] value is invalid");
                    ret = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            /* configuration ADC0/ADC1 clock source */
            if(HAL_ERR_NONE == ret) {
                hals_rcu_adc_clock_config(INDEX_ADC01, periph_clk->adc01_clock_source);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* configure the ADC2 clock */
        if((RCU_PERIPH_CLKTYPE_ADC2 == (periph_clk->periph_clock_type & RCU_PERIPH_CLKTYPE_ADC2)) && \
           (HAL_ERR_NONE == ret)) {
            if(RCU_ADC_CKSRC_CK_PER == periph_clk->adc2_clock_source) {
                if(HAL_ERR_NONE != _rcu_per_clock_source_select(periph_clk)) {
                    HAL_DEBUGE("parameter [periph_clk->pef_clock_source] value is invalid");
                    ret = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            /* configuration ADC2 clock source */
            if(HAL_ERR_NONE == ret) {
                hals_rcu_adc_clock_config(INDEX_ADC2, periph_clk->adc2_clock_source);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      get the RCU extended peripherals clocks
    \param[in]  periph_clk: the pointer of peripheral clock structure
                  periph_clock_type(member):
                  only one parameter can be selected which is shown as below:
      \arg          RCU_PERIPH_CLKTYPE_NONE : no peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_RTC       : RTC peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_APB1TIMER : APB1 timer peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_APB2TIMER : APB2 timer peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_TRNG : TRNG peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_USBHS0 : USBHS0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_USBHS1 : USBHS0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_CAN0 : CAN0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_CAN1 : CAN1 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_CAN2 : CAN2 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SAI2B0 : SAI2B0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SAI2B1 : SAI2B0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SAI0 : SAI0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SAI1 : SAI1 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_EXMC : EXMC peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SPI0 : SPI0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SPI1 : SPI1 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SPI2 : SPI2 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SPI3 : SPI3 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SPI4 : SPI4 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SPI5 : SPI5 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SDIO0 : SDIO0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_SDIO1 : SDIO1 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_HPDF : HPDF peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_USART0 : USART0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_USART1 : USART1 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_USART2 : USART2 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_USART5 : USART5 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_I2C0 : I2C0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_I2C1 : I2C1 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_I2C2 : I2C2 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_I2C3 : I2C3 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_RSPDIF : RSPDIF peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_ADC0 : ADC0 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_ADC1 : ADC1 peripheral clock to configure
      \arg          RCU_PERIPH_CLKTYPE_ADC2 : ADC2 peripheral clock to configure
                  hals_rcu_rtc_clock_config(member1, member2):
                  parameter member1 only one can be selected which is shown as below:
      \arg          RCU_RTC_CLKSRC_NONE: RTC has no clock source
      \arg          RCU_RTC_CLKSRC_LXTAL: RTC select LXTAL as clock source
      \arg          RCU_RTC_CLKSRC_IRC40K: RTC select IRC40K as clock source
      \arg          RCU_RTC_CLKSRC_HXTAL_DIV32: RTC select HXTAL/32 as clock source
                  parameter member2 only one can be selected which is shown as below:
      \arg          RCU_RTC_HXTAL_DIV_NONE, RCU_RTC_HXTAL_DIV_2, ... , RCU_RTC_HXTAL_DIV_63
                  hals_rcu_timer_clock_multiple_config(member):
                  only one parameter can be selected which is shown as below:
      \arg          RCU_TIMER_MUL2: Timer prescaler Multiple by 2
      \arg          RCU_TIMER_MUL4: Timer prescaler Multiple by 4
                  hals_rcu_trng_clock_config(member1, member2)
                  parameter member1 only one can be selected which is shown as below:
      \arg          RCU_CK48M_CKSRC_PLL48M: CK48M Clock Source Selection: PLL48M
      \arg          RCU_CK48M_CKSRC_IRC48M: CK48M Clock Source Selection: IRC48M
                  parameter member2 only one can be selected which is shown as below:
      \arg          RCU_PLL48M_CKSRC_PLL0Q: PLL48MSEL Clock Source Selection: CK_PLL0Q
      \arg          RCU_PLL48M_CKSRC_PLL2P: PLL48MSEL Clock Source Selection: IRC48M
                  hals_rcu_trng_clock_config(member1, member2)
                  parameter member1 only one can be selected which is shown as below:
      \arg          RCU_CK48M_CKSRC_PLL48M: CK48M Clock Source Selection: PLL48M
      \arg          RCU_CK48M_CKSRC_IRC48M: CK48M Clock Source Selection: IRC48M
                  parameter member2 only one can be selected which is shown as below:
      \arg          RCU_PLL48M_CKSRC_PLL0Q: PLL48MSEL Clock Source Selection: CK_PLL0Q
      \arg          RCU_PLL48M_CKSRC_PLL2P: PLL48MSEL Clock Source Selection: IRC48M
                  hals_rcu_usbhs_div_config
                  only one parameter can be selected which is shown as below:
      \arg          RCU_USBHSx_DIV_2 \ RCU_USBHSx_DIV_4 \ RCU_USBHSx_DIV_6
                    RCU_USBHSx_DIV_8 \ RCU_USBHSx_DIV_10 \ RCU_USBHSx_DIV_12
                    RCU_USBHSx_DIV_14 \ RCU_USBHSx_DIV_16
                  hals_rcu_usbhs_psc_config
                  only one parameter can be selected which is shown as below:
      \arg          RCU_USBHSx_PSC_1 \ RCU_USBHSx_PSC_2 \ RCU_USBHSx_PSC_3
                    RCU_USBHSx_PSC_4 \ RCU_USBHSx_PSC_5 \ RCU_USBHSx_PSC_6
                    RCU_USBHSx_PSC_7 \ RCU_USBHSx_PSC_8
                  hals_rcu_usb48m_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_USBHSx48M_CKSRC_PLL0R :PLLUSBx 48M Clock Source: CK_PLL0R
    \arg            RCU_USBHSx48M_CKSRC_USBHSDV :PLLUSBx 48M Clock Source: CK_USBHSxDV\CK_PLLUSBHSx
    \arg            RCU_USBHSx48M_CKSRC_PLL1Q :PLLUSBx 48M Clock Source: CK_USBHSxPSC\PLL1Q
    \arg            RCU_USBHSx48M_CKSRC_IRC48M :PLLUSBx 48M Clock Source: CK_IRC48M
                  hals_rcu_usbhs_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_USBHSx_CKSRC_CK48M : USBHSx clock Source: 48M
    \arg            RCU_USBHSx_CKSRC_CK60M : USBHSx clock Source: 60M
                  hals_rcu_can_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_CAN_CKSRC_CK_HXTAL : CAN clock Source: hxtal
    \arg            RCU_CAN_CKSRC_CK_APB : CAN clock Source: apb2
    \arg            RCU_CAN_CKSRC_CK_APB2DIV2 : CAN clock Source: apb2 / 2
    \arg            RCU_CAN_CKSRC_CK_IRC64MDIV : CAN clock Source: irc64mdiv
                  hals_rcu_sai2b_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_SAI2B_CKSRC_CK_PLL0Q : SAI2Bx clock Source : CK_PLL0Q
    \arg            RCU_SAI2B_CKSRC_CK_PLL1P : SAI2Bx clock Source : CK_PLL1P
    \arg            RCU_SAI2B_CKSRC_CK_PLL2P : SAI2Bx clock Source : CK_PLL2P
    \arg            RCU_SAI2B_CKSRC_I2S_CKIN : SAI2Bx clock Source : I2S_CKIN
    \arg            RCU_SAI2B_CKSRC_CK_PER : SAI2Bx clock Source : CK_PER
    \arg            RCU_SAI2B_CKSRC_CK_RSPDIF_SYMB : SAI2Bx clock Source : CK_RSPDIF_SYMB
                  hals_rcu_sai_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_SAI_CKSRC_CK_PLL0Q : SAIx clock Source : CK_PLL0Q
    \arg            RCU_SAI_CKSRC_CK_PLL1P : SAIx clock Source : CK_PLL1P
    \arg            RCU_SAI_CKSRC_CK_PLL2P : SAIx clock Source : CK_PLL2P
    \arg            RCU_SAI_CKSRC_I2S_CKIN : SAIx clock Source : I2S_CKIN
    \arg            RCU_SAI_CKSRC_CK_PER : SAIx clock Source : CK_PER
                  hals_rcu_exmc_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_EXMC_CKSRC_CK_AHB : EXMC clock src: CK_AHB
    \arg            RCU_EXMC_CKSRC_CK_PLL0Q : EXMC clock src: CK_PLL0Q
    \arg            RCU_EXMC_CKSRC_CK_PLL1R : EXMC clock src: CK_PLL1R
    \arg            RCU_EXMC_CKSRC_CK_PER : EXMC clock src: CK_PER
                  hals_rcu_spi_clock_config
                  only one parameter can be selected which is shown as below:
                    <hal_rcu_spi012_cksrc_enum>, <hal_rcu_spi34_cksrc_enum>, <hal_rcu_spi5_cksrc_enum>
                  hals_rcu_sdio_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_SDIO0_CKSRC_CK_PLL0Q : SDIO0 clock src: CK_PLL0Q
    \arg            RCU_SDIO0_CKSRC_CK_PLL1R : SDIO0 clock src: CK_PLL1R
    \arg            RCU_SDIO1_CKSRC_CK_PLL0Q : SDIO1 clock src: CK_PLL0Q
    \arg            RCU_SDIO1_CKSRC_CK_PLL1R : SDIO1 clock src: CK_PLL1R
                  hals_rcu_hpdf_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_HPDF_CKSRC_CK_APB2 : HPDF clock src: CK_APB2
    \arg            RCU_HPDF_CKSRC_CK_AHB : HPDF clock src: CK_AHB
                  hals_rcu_usart_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_USART_CKSRC_CK_APB : USART clock src: CK_APB
    \arg            RCU_USART_CKSRC_CK_AHB : USART clock src: CK_AHB
    \arg            RCU_USART_CKSRC_CK_LXTAL : USART clock src: CK_LXTAL
    \arg            RCU_USART_CKSRC_CK_IRC64MDIV : USART clock src: CK_IRC64MDIV
                  hals_rcu_i2c_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_I2C_CKSRC_CK_APB1 : I2C clock src: CK_APB1
    \arg            RCU_I2C_CKSRC_CK_PLL2R : I2C clock src: CK_PLL2R
    \arg            RCU_I2C_CKSRC_CK_IRC64MDIV : I2C clock src: CK_IRC64MDIV
    \arg            RCU_I2C_CKSRC_CK_LPIRC4M : I2C clock src: CK_LPIRC4M
                  hals_rcu_rspdif_clock_config
                  only one parameter can be selected which is shown as below:
    \arg            RCU_RSPDIF_CKSRC_CK_PLL0Q : RSPDIF clock src: CK_PLL0Q
    \arg            RCU_RSPDIF_CKSRC_CK_PLL1R : RSPDIF clock src: CK_PLL1R
    \arg            RCU_RSPDIF_CKSRC_CK_PLL2R : RSPDIF clock src: CK_PLL2R
    \arg            RCU_RSPDIF_CKSRC_CK_IRC64MDIV : RSPDIF clock src: CK_IRC64MDIV
                  hals_rcu_adc_clock_config(member):
                  only one parameter can be selected which is shown as below:
      \arg          RCU_ADC_CKSRC_CK_PLL1P: ADC clock source select CK_PLL1P
      \arg          RCU_ADC_CKSRC_CK_PLL2R: ADC clock source select CK_PLL2R
      \arg          RCU_ADC_CKSRC_CK_PER: ADC clock source select CK_PER
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_rcu_periph_get_clock_config(hal_rcu_periphclk_struct *periph_clk)
{
    uint32_t temp_rtc_cksrc    = 0U;
    uint32_t temp_rtc_div      = 0U;
    uint32_t temp_timer_mul    = 0U;
    uint32_t temp_ck48m_cksrc  = 0U;
    uint32_t temp_usb0_div     = 0U, temp_usb0_psc = 0U, temp_usb0_48msrc = 0U, temp_usb0_cksrc = 0U;
    uint32_t temp_usb1_div     = 0U, temp_usb1_psc = 0U, temp_usb1_48msrc = 0U, temp_usb1_cksrc = 0U;
    uint32_t temp_can0_cksrc   = 0U, temp_can1_cksrc = 0U, temp_can2_cksrc = 0U;
    uint32_t temp_sai0_cksrc   = 0U, temp_sai1_cksrc = 0U, temp_sai2b0_cksrc = 0U, temp_sai2b1_cksrc = 0U;
    uint32_t temp_exmc_cksrc   = 0U;
    uint32_t temp_spi0_cksrc   = 0U, temp_spi1_cksrc = 0U, temp_spi2_cksrc = 0U, temp_spi3_cksrc = 0U;
    uint32_t temp_spi4_cksrc   = 0U, temp_spi5_cksrc = 0U;
    uint32_t temp_sdio0_cksrc  = 0U, temp_sdio1_cksrc = 0U;
    uint32_t temp_hpdf_cksrc   = 0U;
    uint32_t temp_usart0_cksrc = 0U, temp_usart1_cksrc = 0U, temp_usart2_cksrc = 0U, temp_usart5_cksrc = 0U;
    uint32_t temp_i2c0_cksrc   = 0U, temp_i2c1_cksrc = 0U, temp_i2c2_cksrc = 0U, temp_i2c3_cksrc = 0U;
    uint32_t temp_rspdif_cksrc = 0U;
    uint32_t temp_adc01_cksrc  = 0U, temp_adc2_cksrc = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == periph_clk) {
        HAL_DEBUGE("pointer [periph_clk] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* get the RTC clock source and RTC div*/
    temp_rtc_cksrc = RCU_BDCTL & RCU_BDCTL_RTCSRC;
    temp_rtc_div   = RCU_CFG0 & RCU_CFG0_RTCDIV;

    periph_clk->rtc_clock_source = (hal_rcu_rtc_cksrc_enum)(temp_rtc_cksrc);
    periph_clk->rtc_div          = (hal_rcu_rtc_div_enum)(temp_rtc_div);

    /* get the timer multiple */
    temp_timer_mul = RCU_CFG1 & RCU_CFG1_TIMERSEL;
    periph_clk->timer_mul   = (hal_rcu_timer_ckmul_enum)(temp_timer_mul);

    /* get the TRNG clock source */
    temp_ck48m_cksrc      = RCU_ADDCTL0 & RCU_ADDCTL0_CK48MSEL;
    periph_clk->ck48m_clock_source = (hal_rcu_ck48m_cksrc_enum)(temp_ck48m_cksrc);

    /* get USBHS0/USBHS1 clock source */
    temp_usb0_div    = RCU_PLLUSBCFG & RCU_PLLUSBCFG_USBHS0DV;
    temp_usb0_psc    = RCU_USBCLKCTL & RCU_USBCLKCTL_USBHS0PSC;
    temp_usb0_48msrc = RCU_USBCLKCTL & RCU_USBCLKCTL_USBHS048MSEL;
    temp_usb0_cksrc  = RCU_USBCLKCTL & RCU_USBCLKCTL_PLLUSBHS0PRESEL;
    temp_usb1_div    = RCU_PLLUSBCFG & RCU_PLLUSBCFG_USBHS1DV;
    temp_usb1_psc    = RCU_USBCLKCTL & RCU_USBCLKCTL_USBHS1PSC;
    temp_usb1_48msrc = RCU_USBCLKCTL & RCU_USBCLKCTL_USBHS148MSEL;
    temp_usb1_cksrc  = RCU_USBCLKCTL & RCU_USBCLKCTL_PLLUSBHS1PRESEL;

    periph_clk->usb0_clock.usbhs_div          = (hal_rcu_usbhsx_div_enum)(temp_usb0_div);
    periph_clk->usb0_clock.usbhs_psc          = (hal_rcu_usbhsx_psc_enum)(temp_usb0_psc);
    periph_clk->usb0_clock.usbhs48m_source    = (hal_rcu_usbhsx48m_cksrc_enum)(temp_usb0_48msrc);
    periph_clk->usb0_clock.usbhs_clock_source = (hal_rcu_usbhsx_cksrc_enum)(temp_usb0_cksrc);
    periph_clk->usb1_clock.usbhs_div          = (hal_rcu_usbhsx_div_enum)(temp_usb1_div);
    periph_clk->usb1_clock.usbhs_psc          = (hal_rcu_usbhsx_psc_enum)(temp_usb1_psc);
    periph_clk->usb1_clock.usbhs48m_source    = (hal_rcu_usbhsx48m_cksrc_enum)(temp_usb1_48msrc);
    periph_clk->usb1_clock.usbhs_clock_source = (hal_rcu_usbhsx_cksrc_enum)(temp_usb1_cksrc);

    /* get CAN clock source */
    temp_can0_cksrc = RCU_CFG1 & RCU_CFG1_CAN0SEL;
    temp_can1_cksrc = RCU_CFG1 & RCU_CFG1_CAN1SEL;
    temp_can2_cksrc = RCU_CFG1 & RCU_CFG1_CAN2SEL;

    periph_clk->can0_clock_source = (hal_rcu_can_cksrc_enum)(temp_can0_cksrc);
    periph_clk->can1_clock_source = (hal_rcu_can_cksrc_enum)(temp_can1_cksrc);
    periph_clk->can2_clock_source = (hal_rcu_can_cksrc_enum)(temp_can2_cksrc);

    /* get SAI clock source */
    temp_sai0_cksrc   = RCU_CFG2 & RCU_CFG2_SAI0SEL;
    temp_sai1_cksrc   = RCU_CFG2 & RCU_CFG2_SAI1SEL;
    temp_sai2b0_cksrc = RCU_CFG2 & RCU_CFG2_SAI2B0SEL;
    temp_sai2b1_cksrc = RCU_CFG2 & RCU_CFG2_SAI2B1SEL;

    periph_clk->sai0_clock_source   = (hal_rcu_sai_cksrc_enum)(temp_sai0_cksrc);
    periph_clk->sai1_clock_source   = (hal_rcu_sai_cksrc_enum)(temp_sai1_cksrc);
    periph_clk->sai2b0_clock_source = (hal_rcu_sai2b_cksrc_enum)(temp_sai2b0_cksrc);
    periph_clk->sai2b1_clock_source = (hal_rcu_sai2b_cksrc_enum)(temp_sai2b1_cksrc);

    /* get EXMC clock source */
    temp_exmc_cksrc               = RCU_CFG4 & RCU_CFG4_EXMCSEL;
    periph_clk->exmc_clock_source = (hal_rcu_exmc_cksrc_enum)(temp_exmc_cksrc);

    /* get SPI clock source */
    temp_spi0_cksrc =  RCU_CFG5 & RCU_CFG5_SPI0SEL;
    temp_spi1_cksrc =  RCU_CFG5 & RCU_CFG5_SPI1SEL;
    temp_spi2_cksrc =  RCU_CFG5 & RCU_CFG5_SPI2SEL;
    temp_spi3_cksrc =  RCU_CFG5 & RCU_CFG5_SPI3SEL;
    temp_spi4_cksrc =  RCU_CFG5 & RCU_CFG5_SPI4SEL;
    temp_spi5_cksrc =  RCU_CFG5 & RCU_CFG5_SPI5SEL;

    periph_clk->spi0_clock_source = (hal_rcu_spi012_cksrc_enum)(temp_spi0_cksrc);
    periph_clk->spi1_clock_source = (hal_rcu_spi012_cksrc_enum)(temp_spi1_cksrc);
    periph_clk->spi2_clock_source = (hal_rcu_spi012_cksrc_enum)(temp_spi2_cksrc);
    periph_clk->spi3_clock_source = (hal_rcu_spi34_cksrc_enum)(temp_spi3_cksrc);
    periph_clk->spi4_clock_source = (hal_rcu_spi34_cksrc_enum)(temp_spi4_cksrc);
    periph_clk->spi5_clock_source = (hal_rcu_spi5_cksrc_enum)(temp_spi5_cksrc);

    /* get SDIO clock source */
    temp_sdio0_cksrc = RCU_CFG4 & RCU_CFG4_SDIO0SEL;
    temp_sdio1_cksrc = RCU_CFG3 & RCU_CFG3_SDIO1SEL;

    periph_clk->sdio0_clock_source = (hal_rcu_sdio_cksrc_enum)(temp_sdio0_cksrc);
    periph_clk->sdio1_clock_source = (hal_rcu_sdio_cksrc_enum)(temp_sdio1_cksrc);

    /* get HPDF clock source */
    temp_hpdf_cksrc               = RCU_CFG1 & RCU_CFG1_HPDFSEL;
    periph_clk->hpdf_clock_source = (hal_rcu_hpdf_cksrc_enum)(temp_hpdf_cksrc);

    /* get USART clock source */
    temp_usart0_cksrc = RCU_CFG1 & RCU_CFG1_USART0SEL;
    temp_usart1_cksrc = RCU_CFG1 & RCU_CFG1_USART1SEL;
    temp_usart2_cksrc = RCU_CFG1 & RCU_CFG1_USART2SEL;
    temp_usart5_cksrc = RCU_CFG1 & RCU_CFG1_USART2SEL;

    periph_clk->usart0_clock_source = (hal_rcu_usart_cksrc_enum)(temp_usart0_cksrc);
    periph_clk->usart1_clock_source = (hal_rcu_usart_cksrc_enum)(temp_usart1_cksrc);
    periph_clk->usart2_clock_source = (hal_rcu_usart_cksrc_enum)(temp_usart2_cksrc);
    periph_clk->usart5_clock_source = (hal_rcu_usart_cksrc_enum)(temp_usart5_cksrc);

    /* get I2C clock source */
    temp_i2c0_cksrc = RCU_CFG3 & RCU_CFG3_I2C1SEL;
    temp_i2c1_cksrc = RCU_CFG3 & RCU_CFG3_I2C1SEL;
    temp_i2c2_cksrc = RCU_CFG3 & RCU_CFG3_I2C2SEL;
    temp_i2c3_cksrc = RCU_CFG3 & RCU_CFG3_I2C3SEL;

    periph_clk->i2c0_clock_source = (hal_rcu_i2c_cksrc_enum)(temp_i2c0_cksrc);
    periph_clk->i2c1_clock_source = (hal_rcu_i2c_cksrc_enum)(temp_i2c1_cksrc);
    periph_clk->i2c2_clock_source = (hal_rcu_i2c_cksrc_enum)(temp_i2c2_cksrc);
    periph_clk->i2c3_clock_source = (hal_rcu_i2c_cksrc_enum)(temp_i2c3_cksrc);

    /* get RSPDIF clock source */
    temp_rspdif_cksrc = RCU_CFG1 & RCU_CFG1_RSPDIFSEL;
    periph_clk->rspdif_clock_source = (hal_rcu_rspdif_cksrc_enum)(temp_rspdif_cksrc);

    /* get ADC clock source */
    temp_adc01_cksrc = RCU_CFG3 & RCU_CFG3_ADC01SEL;
    temp_adc2_cksrc  = RCU_CFG3 & RCU_CFG3_ADC2SEL;

    periph_clk->adc01_clock_source = (hal_rcu_adc_cksrc_enum)(temp_adc01_cksrc);
    periph_clk->adc2_clock_source  = (hal_rcu_adc_cksrc_enum)(temp_adc2_cksrc);

    return HAL_ERR_NONE;
}

/*!
    \brief      get the RCU clock configuration
    \param[in]  rcu_clk: the pointer of the RCU clock structure
                  clock_type(member):
                  only one parameter can be selected which is shown as below:
      \arg          RCU_CLKTYPE_NONE: no clock type
      \arg          RCU_CLKTYPE_SYSCLK: system clock type
      \arg          RCU_CLKTYPE_CK48MCLK: ck48m clock to configure
      \arg          RCU_CLKTYPE_AHBCLK: AHB bus clock type
      \arg          RCU_CLKTYPE_APB1CLK: APB1 bus clock type
      \arg          RCU_CLKTYPE_APB2CLK: APB2 bus clock type
                  sysclk_source(member):
                  only one parameter can be selected which is shown as below:
      \arg          RCU_SYSCLK_SRC_CK_IRC64MDIV: IRC8M as system clock source
      \arg          RCU_SYSCLK_SRC_CK_HXTAL: HXTAL as system clock source
      \arg          RCU_SYSCLK_SRC_CK_LPIRC4M: CK_LPIRC4M as system clock source
      \arg          RCU_SYSCLK_SRC_CK_PLL0P: PLL0P as system clock source
                  ck48m_source(member):
                  only one parameter can be selected which is shown as below:
      \arg          RCU_CK48M_CKSRC_PLL48M: PLL48M to be ck48m clock source
      \arg          RCU_CK48M_CKSRC_IRC48M: IRC48M to be ck48m clock source
                  ref_clock(member):
                  only one parameter can be selected which is shown as below:
      \arg          RCU_PRECLK_CKSRC_CK_IRC64MDIV: IRC64MDIV to be reference clock source
      \arg          RCU_PRECLK_CKSRC_CK_LPIRC4M: LPIRC4M to be reference clock source
      \arg          RCU_PRECLK_CKSRC_CK_HXTAL: HXTAL to be reference clock source
                  dspwu_source(member):
                  only one parameter can be selected which is shown as below:
      \arg          RCU_DSPWUSSEL_CKSRC_IRC64MDIV: IRC64MDIV to be deepsleep wakeup clock source
      \arg          RCU_DSPWUSSEL_CKSRC_LPIRC4M: LPIRC4M to be deepsleep wakeup clock source
                  irc64m_div(member):
                    the argument could be selected from enumeration <hal_rcu_irc64m_div_enum>
                  irc64m_div(member):
                    the argument could be selected from enumeration <hal_rcu_irc64m_div_enum>
                  ahbclk_div(member): AHB clock divider
                    the argument could be selected from enumeration <hal_rcu_sysclk_ahbdiv_enum>
                  apb1clk_div(member): APB1 clock divider
                    the argument could be selected from enumeration <hal_rcu_sysclk_apb1div_enum>
                  apb2clk_div(member): APB2 clock divider
                    the argument could be selected from enumeration <hal_rcu_sysclk_apb2div_enum>
                  apb3clk_div(member): APB3 clock divider
                    the argument could be selected from enumeration <hal_rcu_sysclk_apb3div_enum>
                  apb4clk_div(member): APB4 clock divider
                    the argument could be selected from enumeration <hal_rcu_sysclk_apb4div_enum>
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rcu_clock_config_get(hal_rcu_clk_struct *rcu_clk)
{
    uint32_t temp_sysclk_source = 0U;
    uint32_t temp_ahbclk_div    = 0U;
    uint32_t temp_apb1clk_div   = 0U;
    uint32_t temp_apb2clk_div   = 0U;
    uint32_t temp_apb3clk_div   = 0U;
    uint32_t temp_apb4clk_div   = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rcu_clk) {
        HAL_DEBUGE("pointer [rcu_clk] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    temp_sysclk_source = (RCU_CFG0 & RCU_CFG0_SCS);
    temp_ahbclk_div    = (RCU_CFG0 & RCU_CFG0_AHBPSC);
    temp_apb1clk_div   = (RCU_CFG0 & RCU_CFG0_APB1PSC);
    temp_apb2clk_div   = (RCU_CFG0 & RCU_CFG0_APB2PSC);
    temp_apb3clk_div   = (RCU_CFG0 & RCU_CFG0_APB3PSC);
    temp_apb4clk_div   = (RCU_CFG0 & RCU_CFG0_APB4PSC);

    /* get all clock type configuration */
    rcu_clk->clock_type    = RCU_CLKTYPE_SYSCLK  | RCU_CLKTYPE_AHBCLK | RCU_CLKTYPE_APB1CLK | RCU_CLKTYPE_APB2CLK | \
                             RCU_CLKTYPE_APB3CLK | RCU_CLKTYPE_APB4CLK;
    rcu_clk->sysclk_source = (hal_rcu_cksys_src_enum)(temp_sysclk_source);
    rcu_clk->ahbclk_div    = (hal_rcu_sysclk_ahbdiv_enum)(temp_ahbclk_div);
    rcu_clk->apb1clk_div   = (hal_rcu_ahbclk_apb1div_enum)(temp_apb1clk_div);
    rcu_clk->apb2clk_div   = (hal_rcu_ahbclk_apb2div_enum)(temp_apb2clk_div);
    rcu_clk->apb3clk_div   = (hal_rcu_ahbclk_apb3div_enum)(temp_apb3clk_div);
    rcu_clk->apb4clk_div   = (hal_rcu_ahbclk_apb4div_enum)(temp_apb4clk_div);

    return HAL_ERR_NONE;
}

/*!
    \brief      enable the HXTAL clock monitor
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_rcu_hxtal_clock_monitor_enable(void)
{
    RCU_CTL |= RCU_CTL_CKMEN;
}

/*!
    \brief      disable the HXTAL clock monitor
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_rcu_hxtal_clock_monitor_disable(void)
{
    RCU_CTL &= ~RCU_CTL_CKMEN;
}

/*!
    \brief      enable the LXTAL clock monitor
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_rcu_lxtal_clock_monitor_enable(void)
{
    RCU_BDCTL |= RCU_BDCTL_LCKMEN;
}

/*!
    \brief      disable the LXTAL clock monitor
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_rcu_lxtal_clock_monitor_disable(void)
{
    RCU_BDCTL &= ~RCU_BDCTL_LCKMEN;
}

/*!
    \brief      enable the peripherals clock when sleep mode
    \param[in]  periph: RCU peripherals, refer to rcu_periph_sleep_enum
                only one parameter can be selected which is shown as below:
      \arg        RCU_ENETx_SLP (x = 0,1): ENET clock
      \arg        RCU_ENETxTX_SLP (x = 0,1): ENETTX clock
      \arg        RCU_ENETxRX_SLP (x = 0,1): ENETRX clock
      \arg        RCU_ENETxPTP_SLP (x = 0,1): ENETPTP clock
      \arg        RCU_USBHSx_SLP (x = 0,1): USBHS clock
      \arg        RCU_USBHSxULPI_SLP (x = 0,1): USBHSULPI clock
      \arg        RCU_SRAM0_SLP: SRAM0 clock
      \arg        RCU_SRAM1_SLP: SRAM1 clock
      \arg        RCU_DMAx_SLP (x = 0,1): DMA clock
      \arg        RCU_DMAMUX_SLP: DMAMUX clock
      \arg        RCU_DCI_SLP: DCI clock
      \arg        RCU_FAC_SLP: TRNG clock
      \arg        RCU_SDIOx_SLP (x = 0,1): SDIO clock
      \arg        RCU_CAU_SLP: CAU clock
      \arg        RCU_HAU_SLP: HAU clock
      \arg        RCU_TRNG_SLP: TRNG clock
      \arg        RCU_TMU_SLP: TMU clock
      \arg        RCU_RAMECCMUx_SLP (x = 0,1): RAMECCMU clock
      \arg        RCU_EXMC_SLP: EXMC clock
      \arg        RCU_IPA_SLP: IPA clock
      \arg        RCU_SDIOx_SLP (x = 0,1): SDIO clock
      \arg        RCU_MDMA_SLP: MDMMA clock
      \arg        RCU_OSPIM_SLP: OSPIM clock
      \arg        RCU_OSPIx_SLP (x = 0,1): OSPI0 clock
      \arg        RCU_RTDECx_SLP (x = 0,1): RTDEC0 clock
      \arg        RCU_AXISRAM_SLP: AXISRAM clock
      \arg        RCU_FMC_SLP: SRAM1 clock
      \arg        RCU_GPIOx_SLP (x = A,B,C,D,E,F,G,H,J,K): GPIO ports clock
      \arg        RCU_BKPSRAM_SLP: BKPSRAM clock
      \arg        RCU_CRC_SLP: CRC clock
      \arg        RCU_TIMERx_SLP (x = 0,1,2,3,4,5,6,7,14,15,16,22,23,30,31,40,41,42,43,44,50,51): TIMER clock
      \arg        RCU_RSPDIF_SLP: RSPDIF clock
      \arg        RCU_SPIx_SLP (x = 0,1,2,3,4,5): SPI clock
      \arg        RCU_MDIO_SLP: MDIO clock
      \arg        RCU_USARTx_SLP (x = 0,1,2,5): USART clock
      \arg        RCU_UARTx_SLP (x = 3,4,6,7): UART clock
      \arg        RCU_I2Cx_SLP (x = 0,1,2,3): I2C clock
      \arg        RCU_CTC_SLP: CTC clock
      \arg        RCU_DACHOLD_SLP: DACHOLD clock
      \arg        RCU_DAC_SLP: DAC clock
      \arg        RCU_ADCx_SLP (x = 0,1,2): ADC clock
      \arg        RCU_HPDF_SLP: HPDF clock
      \arg        RCU_SAIx_SLP (x = 0,1,2): SAI clock
      \arg        RCU_EDOUT_SLP: EDOUT clock
      \arg        RCU_TRIGSEL_SLP: TRIGSEL clock
      \arg        RCU_TLI_SLP: TLI clock
      \arg        RCU_WWDGT_SLP: WWDGT clock
      \arg        RCU_SYSCFG_SLP: SYSCFG clock
      \arg        RCU_CMP_SLP: CMP clock
      \arg        RCU_VREF_SLP: VREF clock
      \arg        RCU_LPDTS_SLP: LPDTS clock
      \arg        RCU_PMU_SLP: PMU clock
      \arg        RCU_CANx_SLP (x = 0,1,2): can clock
    \param[out] none
    \retval     none
*/
void hal_rcu_periph_clock_sleep_enable(hal_rcu_periph_sleep_enum periph)
{
    RCU_REG_VAL(periph) |= BIT(RCU_BIT_POS(periph));
}

/*!
    \brief      disable the peripherals clock when sleep mode
    \param[in]  periph: RCU peripherals, refer to rcu_periph_sleep_enum
                only one parameter can be selected which is shown as below:
      \arg        RCU_ENETx_SLP (x = 0,1): ENET clock
      \arg        RCU_ENETxTX_SLP (x = 0,1): ENETTX clock
      \arg        RCU_ENETxRX_SLP (x = 0,1): ENETRX clock
      \arg        RCU_ENETxPTP_SLP (x = 0,1): ENETPTP clock
      \arg        RCU_USBHSx_SLP (x = 0,1): USBHS clock
      \arg        RCU_USBHSxULPI_SLP (x = 0,1): USBHSULPI clock
      \arg        RCU_SRAM0_SLP: SRAM0 clock
      \arg        RCU_SRAM1_SLP: SRAM1 clock
      \arg        RCU_DMAx_SLP (x = 0,1): DMA clock
      \arg        RCU_DMAMUX_SLP: DMAMUX clock
      \arg        RCU_DCI_SLP: DCI clock
      \arg        RCU_FAC_SLP: TRNG clock
      \arg        RCU_SDIOx_SLP (x = 0,1): SDIO clock
      \arg        RCU_CAU_SLP: CAU clock
      \arg        RCU_HAU_SLP: HAU clock
      \arg        RCU_TRNG_SLP: TRNG clock
      \arg        RCU_TMU_SLP: TMU clock
      \arg        RCU_RAMECCMUx_SLP (x = 0,1): RAMECCMU clock
      \arg        RCU_EXMC_SLP: EXMC clock
      \arg        RCU_IPA_SLP: IPA clock
      \arg        RCU_SDIOx_SLP (x = 0,1): SDIO clock
      \arg        RCU_MDMA_SLP: MDMMA clock
      \arg        RCU_OSPIM_SLP: OSPIM clock
      \arg        RCU_OSPIx_SLP (x = 0,1): OSPI0 clock
      \arg        RCU_RTDECx_SLP (x = 0,1): RTDEC0 clock
      \arg        RCU_AXISRAM_SLP: AXISRAM clock
      \arg        RCU_FMC_SLP: SRAM1 clock
      \arg        RCU_GPIOx_SLP (x = A,B,C,D,E,F,G,H,J,K): GPIO ports clock
      \arg        RCU_BKPSRAM_SLP: BKPSRAM clock
      \arg        RCU_CRC_SLP: CRC clock
      \arg        RCU_TIMERx_SLP (x = 0,1,2,3,4,5,6,7,14,15,16,22,23,30,31,40,41,42,43,44,50,51): TIMER clock
      \arg        RCU_RSPDIF_SLP: RSPDIF clock
      \arg        RCU_SPIx_SLP (x = 0,1,2,3,4,5): SPI clock
      \arg        RCU_MDIO_SLP: MDIO clock
      \arg        RCU_USARTx_SLP (x = 0,1,2,5): USART clock
      \arg        RCU_UARTx_SLP (x = 3,4,6,7): UART clock
      \arg        RCU_I2Cx_SLP (x = 0,1,2,3): I2C clock
      \arg        RCU_CTC_SLP: CTC clock
      \arg        RCU_DACHOLD_SLP: DACHOLD clock
      \arg        RCU_DAC_SLP: DAC clock
      \arg        RCU_ADCx_SLP (x = 0,1,2): ADC clock
      \arg        RCU_HPDF_SLP: HPDF clock
      \arg        RCU_SAIx_SLP (x = 0,1,2): SAI clock
      \arg        RCU_EDOUT_SLP: EDOUT clock
      \arg        RCU_TRIGSEL_SLP: TRIGSEL clock
      \arg        RCU_TLI_SLP: TLI clock
      \arg        RCU_WWDGT_SLP: WWDGT clock
      \arg        RCU_SYSCFG_SLP: SYSCFG clock
      \arg        RCU_CMP_SLP: CMP clock
      \arg        RCU_VREF_SLP: VREF clock
      \arg        RCU_LPDTS_SLP: LPDTS clock
      \arg        RCU_PMU_SLP: PMU clock
      \arg        RCU_CANx_SLP (x = 0,1,2): can clock
    \param[out] none
    \retval     none
*/
void hal_rcu_periph_clock_sleep_disable(hal_rcu_periph_sleep_enum periph)
{
    RCU_REG_VAL(periph) &= ~BIT(RCU_BIT_POS(periph));
}

/*!
    \brief      reset the peripherals
    \param[in]  periph_reset: RCU peripherals reset, refer to hal_rcu_periph_reset_enum
                only one parameter can be selected which is shown as below:
      \arg        RCU_GPIOxRST (x = A,B,C,D,E,F,G,H,J,K): reset GPIO ports
      \arg        RCU_ENETxRST  : ENETx(x=0,1) clock reset
      \arg        RCU_USBHSxRST : USBHSx(x=0,1) clock reset
      \arg        RCU_DMAxRST   : DMAx(x=0,1) clock reset
      \arg        RCU_DMAMUXRST : DMAMUX clock reset
      \arg        RCU_DCIRST    : DCI clock reset
      \arg        RCU_FACRST    : FAC clock reset
      \arg        RCU_SDIOxRST  : SDIOx(x=0,1) clock reset
      \arg        RCU_CAURST    : CAU clock reset
      \arg        RCU_HAURST    : HAU clock reset
      \arg        RCU_TRNGRST   : TRNG clock reset
      \arg        RCU_TMURST    : TMU clock reset
      \arg        RCU_EXMCRST   : EXMC clock reset
      \arg        RCU_IPARST    : IPA clock reset
      \arg        RCU_MDMARST   : MDMMA clock reset
      \arg        RCU_OSPIMRST  : OSPIM clock reset
      \arg        RCU_OSPIxRST  : OSPIx(x=0,1) clock reset
      \arg        RCU_RTDECxRST : RTDECx(x=0,1) clock reset
      \arg        RCU_CRCRST    : CRC clock reset
      \arg        RCU_HWSEMRST  : HWSEM clock reset
      \arg        RCU_RSPDIFRST : RSPDIF clock reset
      \arg        RCU_MDIORST   : MDIO clock reset
      \arg        RCU_USARTxRST : USARTx(x:0,1) clock reset
      \arg        RCU_I2CxRST   : I2Cx(x=0~4) clock reset
      \arg        RCU_CTCRST    : CTC clock reset
      \arg        RCU_DACHOLDRST: DACHOLD clock reset
      \arg        RCU_DACRST    : DAC clock reset
      \arg        RCU_USARTxRST : USARTx(x=0,3,4,5~7) clock reset
      \arg        RCU_ADCxRST   : ADCx(x=0,1,2) clock reset
      \arg        RCU_SPIxRST   : SPIx(x=0,1,2,3,4,5) clock reset
      \arg        RCU_HPDFRST   : HPDF clock reset
      \arg        RCU_SAIxRST   : SAIx(x=0,1,2) clock reset
      \arg        RCU_TIMERxRST : TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51) clock reset
      \arg        RCU_EDOUTRST  : EDOUT clock reset
      \arg        RCU_TRIGSELRST: TRIGSEL clock reset
      \arg        RCU_TLIRST    : TLI clock reset
      \arg        RCU_WWDGTRST  : WWDGT clock reset
      \arg        RCU_SYSCFGRST : SYSCFG clock reset
      \arg        RCU_CMPRST    : CMP clock reset
      \arg        RCU_VREFRST   : VREF clock reset
      \arg        RCU_LPDTSRST  : LPDTS clock reset
      \arg        RCU_PMURST    : PMU clock reset
      \arg        RCU_CANxRST   : RCU_CANxRST(x=0,1,2) clock reset
    \param[out] none
    \retval     none
*/
void hal_rcu_periph_reset_enable(hal_rcu_periph_reset_enum periph_reset)
{
    RCU_REG_VAL(periph_reset) |= BIT(RCU_BIT_POS(periph_reset));
}

/*!
    \brief      disable reset the peripheral
    \param[in]  periph_reset: RCU peripherals reset, refer to hal_rcu_periph_reset_enum
                only one parameter can be selected which is shown as below:
      \arg        RCU_GPIOxRST (x = A,B,C,D,E,F,G,H,J,K): reset GPIO ports
      \arg        RCU_ENETxRST  : ENETx(x=0,1) clock reset
      \arg        RCU_USBHSxRST : USBHSx(x=0,1) clock reset
      \arg        RCU_DMAxRST   : DMAx(x=0,1) clock reset
      \arg        RCU_DMAMUXRST : DMAMUX clock reset
      \arg        RCU_DCIRST    : DCI clock reset
      \arg        RCU_FACRST    : FAC clock reset
      \arg        RCU_SDIOxRST  : SDIOx(x=0,1) clock reset
      \arg        RCU_CAURST    : CAU clock reset
      \arg        RCU_HAURST    : HAU clock reset
      \arg        RCU_TRNGRST   : TRNG clock reset
      \arg        RCU_TMURST    : TMU clock reset
      \arg        RCU_EXMCRST   : EXMC clock reset
      \arg        RCU_IPARST    : IPA clock reset
      \arg        RCU_MDMARST   : MDMMA clock reset
      \arg        RCU_OSPIMRST  : OSPIM clock reset
      \arg        RCU_OSPIxRST  : OSPIx(x=0,1) clock reset
      \arg        RCU_RTDECxRST : RTDECx(x=0,1) clock reset
      \arg        RCU_CRCRST    : CRC clock reset
      \arg        RCU_HWSEMRST  : HWSEM clock reset
      \arg        RCU_RSPDIFRST : RSPDIF clock reset
      \arg        RCU_MDIORST   : MDIO clock reset
      \arg        RCU_USARTxRST : USARTx(x:0,1) clock reset
      \arg        RCU_I2CxRST   : I2Cx(x=0~4) clock reset
      \arg        RCU_CTCRST    : CTC clock reset
      \arg        RCU_DACHOLDRST: DACHOLD clock reset
      \arg        RCU_DACRST    : DAC clock reset
      \arg        RCU_USARTxRST : USARTx(x=0,3,4,5~7) clock reset
      \arg        RCU_ADCxRST   : ADCx(x=0,1,2) clock reset
      \arg        RCU_SPIxRST   : SPIx(x=0,1,2,3,4,5) clock reset
      \arg        RCU_HPDFRST   : HPDF clock reset
      \arg        RCU_SAIxRST   : SAIx(x=0,1,2) clock reset
      \arg        RCU_TIMERxRST : TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51) clock reset
      \arg        RCU_EDOUTRST  : EDOUT clock reset
      \arg        RCU_TRIGSELRST: TRIGSEL clock reset
      \arg        RCU_TLIRST    : TLI clock reset
      \arg        RCU_WWDGTRST  : WWDGT clock reset
      \arg        RCU_SYSCFGRST : SYSCFG clock reset
      \arg        RCU_CMPRST    : CMP clock reset
      \arg        RCU_VREFRST   : VREF clock reset
      \arg        RCU_LPDTSRST  : LPDTS clock reset
      \arg        RCU_PMURST    : PMU clock reset
      \arg        RCU_CANxRST   : RCU_CANxRST(x=0,1,2) clock reset
    \param[out] none
    \retval     none
*/
void hal_rcu_periph_reset_disable(hal_rcu_periph_reset_enum periph_reset)
{
    RCU_REG_VAL(periph_reset) &= ~BIT(RCU_BIT_POS(periph_reset));
}

/*!
    \brief      update the SystemCoreClock with current core clock retrieved from cpu registers
    \param[in]  none
    \param[out] none
    \retval     none
*/
uint32_t hal_rcu_system_core_clock_update(void)
{
    uint32_t ck_src = 0U, irc64mdiv_freq = 0U;
    uint32_t pll0p = 0U, pll0n = 0U, pll0psc = 0U, pllsel = 0U, fracn = 0U;

    /* get IRC64MDIV frequency */
    irc64mdiv_freq = hals_rcu_irc64mdiv_freq_get();

    /* get current system clock */
    ck_src = GET_BITS(RCU_CFG0, 2, 3);

    /* get current system clock value */
    switch(ck_src) {
    case CK_IRC64MDIV:
        g_systemcoreclock = irc64mdiv_freq;
        break;
    case CK_HXTAL:
        g_systemcoreclock = HXTAL_VALUE;
        break;
    case CK_LPIRC4M:
        g_systemcoreclock = LPIRC4M_VALUE;
        break;
    case CK_PLL0P:
        pll0psc = GET_BITS(RCU_PLL0, 0U, 5U);
        pll0n   = (GET_BITS(RCU_PLL0, 6U, 14U) + 1U);
        pll0p   = (GET_BITS(RCU_PLL0, 16U, 22U) + 1U);

        if(RESET != (RCU_PLL0FRA & RCU_PLL0FRA_PLL0FRAEN)) {
            fracn = GET_BITS(RCU_PLL0FRA, 0U, 12U);
        } else {
            /* do nothing */
        }

        /* PLL clock source selection, HXTAL or IRC64MDIV */
        pllsel = (RCU_PLLALL & RCU_PLLALL_PLLSEL);

        if(RCU_PLLSRC_HXTAL == pllsel) {
            ck_src = HXTAL_VALUE;
        } else if(RCU_PLLSRC_IRC64MDIV == pllsel) {
            ck_src = irc64mdiv_freq;
        } else {
            ck_src = LPIRC4M_VALUE;
        }

        g_systemcoreclock = _rcu_pll_clock_freq_cal(ck_src, pll0psc, pll0n, fracn, pll0p);
        break;
    default:
        g_systemcoreclock = irc64mdiv_freq;
        break;
    }

    return g_systemcoreclock;
}

/*!
    \brief       RCU clock configuration
    \param[in]  rcu_clk: the pointer of the RCU clock structure
                  clock_type(member):
                  only one parameter can be selected which is shown as below:
      \arg          RCU_CLKTYPE_NONE: no clock type
      \arg          RCU_CLKTYPE_SYSCLK: system clock type
      \arg          RCU_CLKTYPE_CK48MCLK: ck48m clock to configure
      \arg          RCU_CLKTYPE_REFCLK: reference clock to configure
      \arg          RCU_CLKTYPE_DSPWUCLK: deepsleep wake up clock to configure
      \arg          RCU_CLKTYPE_AHBCLK: AHB bus clock type
      \arg          RCU_CLKTYPE_APB1CLK: APB1 bus clock type
      \arg          RCU_CLKTYPE_APB2CLK: APB2 bus clock type
      \arg          RCU_CLKTYPE_APB3CLK: APB1 bus clock type
      \arg          RCU_CLKTYPE_APB4CLK: APB2 bus clock type
                  sysclk_source(member):
                  only one parameter can be selected which is shown as below:
      \arg          RCU_SYSCLK_SRC_CK_IRC64MDIV: IRC8M as system clock source
      \arg          RCU_SYSCLK_SRC_CK_HXTAL: HXTAL as system clock source
      \arg          RCU_SYSCLK_SRC_CK_LPIRC4M: CK_LPIRC4M as system clock source
      \arg          RCU_SYSCLK_SRC_CK_PLL0P: PLL0P as system clock source
                  ck48m_source(member):
                  only one parameter can be selected which is shown as below:
      \arg          RCU_CK48M_CKSRC_PLL48M: PLL48M to be ck48m clock source
      \arg          RCU_CK48M_CKSRC_IRC48M: IRC48M to be ck48m clock source
                  ref_clock(member):
                  only one parameter can be selected which is shown as below:
      \arg          RCU_PRECLK_CKSRC_CK_IRC64MDIV: IRC64MDIV to be reference clock source
      \arg          RCU_PRECLK_CKSRC_CK_LPIRC4M: LPIRC4M to be reference clock source
      \arg          RCU_PRECLK_CKSRC_CK_HXTAL: HXTAL to be reference clock source
                  dspwu_source(member):
                  only one parameter can be selected which is shown as below:
      \arg          RCU_DSPWUSSEL_CKSRC_IRC64MDIV: IRC64MDIV to be deepsleep wakeup clock source
      \arg          RCU_DSPWUSSEL_CKSRC_LPIRC4M: LPIRC4M to be deepsleep wakeup clock source
                  irc64m_div(member):
                    the argument could be selected from enumeration <hal_rcu_irc64m_div_enum>
                  irc64m_div(member):
                    the argument could be selected from enumeration <hal_rcu_irc64m_div_enum>
                  ahbclk_div(member): AHB clock divider
                    the argument could be selected from enumeration <hal_rcu_sysclk_ahbdiv_enum>
                  apb1clk_div(member): APB1 clock divider
                    the argument could be selected from enumeration <hal_rcu_ahbclk_apb1div_enum>
                  apb2clk_div(member): APB2 clock divider
                    the argument could be selected from enumeration <hal_rcu_sysclk_apb2div_enum>
                  apb3clk_div(member): APB3 clock divider
                    the argument could be selected from enumeration <hal_rcu_sysclk_apb3div_enum>
                  apb4clk_div(member): APB4 clock divider
                    the argument could be selected from enumeration <hal_rcu_sysclk_apb4div_enum>
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_VAL, HAL_ERR_NONE
                            details refer to gd32h7xx_hal.h
*/
int32_t hal_rcu_clock_config(hal_rcu_clk_struct *rcu_clk)
{
    uint32_t flag       = 0U;
    uint32_t tick_start = 0U;
    uint32_t tick_out   = s_rcu_timeout[RCU_CKSYSSRC_INDEX(rcu_clk->sysclk_source)];
    int32_t  ret        = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rcu_clk) {
        HAL_DEBUGE("pointer [rcu_clk] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(rcu_clk->clock_type > RCU_CLKTYPE_MAX) {
        HAL_DEBUGE("parameter [rcu_clk->clock_type] value is invalid.");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* ------------------------------- system clock Configuration -------------------------------*/
    if(RCU_CLKTYPE_SYSCLK == (rcu_clk->clock_type & RCU_CLKTYPE_SYSCLK)) {
        switch(rcu_clk->sysclk_source) {
        case RCU_SYSCLK_SRC_CK_IRC64MDIV:
            flag = hals_rcu_flag_get(RCU_FLAG_IRC64MSTB);
            break;
        case RCU_SYSCLK_SRC_CK_HXTAL:
            flag = hals_rcu_flag_get(RCU_FLAG_HXTALSTB);
            break;
        case RCU_SYSCLK_SRC_CK_LPIRC4M:
            flag = hals_rcu_flag_get(RCU_FLAG_LPIRC4MSTB);
            break;
        case RCU_SYSCLK_SRC_CK_PLL0P:
            flag = hals_rcu_flag_get(RCU_FLAG_PLL0STB);
            break;
        default:
            HAL_DEBUGE("parameter [rcu_clk->sysclk_source] value is invalid.");
            ret = HAL_ERR_VAL;
            break;
        }

        if(RESET == flag) {
            HAL_DEBUGE("parameter [rcu_clk->clock_type] value is invalid.");
            ret = HAL_ERR_VAL;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if(HAL_ERR_NONE == ret) {
        /* ------------------------------- AHB clock Configuration ---------------------------------*/
        if(RCU_CLKTYPE_AHBCLK == (rcu_clk->clock_type & RCU_CLKTYPE_AHBCLK)) {
            /* configure AHB clock */
            hals_rcu_ahb_clock_config(rcu_clk->ahbclk_div);
        } else {
            /* do nothing */
        }

        /* ------------------------------- APB1 clock Configuration -------------------------------*/
        if(RCU_CLKTYPE_APB1CLK == (rcu_clk->clock_type & RCU_CLKTYPE_APB1CLK)) {
            hals_rcu_apb1_clock_config(rcu_clk->apb1clk_div);
        } else {
            /* do nothing */
        }

        /* ------------------------------- APB2 clock Configuration -------------------------------*/
        if(RCU_CLKTYPE_APB2CLK == (rcu_clk->clock_type & RCU_CLKTYPE_APB2CLK)) {
            hals_rcu_apb2_clock_config(rcu_clk->apb2clk_div);
        } else {
            /* do nothing */
        }

        /* ------------------------------- APB3 clock Configuration -------------------------------*/
        if(RCU_CLKTYPE_APB3CLK == (rcu_clk->clock_type & RCU_CLKTYPE_APB3CLK)) {
            hals_rcu_apb3_clock_config(rcu_clk->apb3clk_div);
        } else {
            /* do nothing */
        }

        /* ------------------------------- APB4 clock Configuration -------------------------------*/
        if(RCU_CLKTYPE_APB4CLK == (rcu_clk->clock_type & RCU_CLKTYPE_APB4CLK)) {
            hals_rcu_apb4_clock_config(rcu_clk->apb4clk_div);
        } else {
            /* do nothing */
        }

        /* configure the system clock source */
        hals_rcu_system_clock_source_config(rcu_clk->sysclk_source);

        /* wait till system clock source is stable */
        while(rcu_clk->sysclk_source != (hals_rcu_system_clock_source_get() >> 2U)) {
            if(SET == hal_sys_basetick_timeout_check(tick_start, tick_out)) {
                HAL_DEBUGE("system clock is not stabilized for the required amount of time");
                ret = HAL_ERR_TIMEOUT;
                break;
            } else {
                /* do nothing */
            }
        }

        if(HAL_ERR_NONE == ret) {
            /* update the SystemCoreClock global variable */
            g_systemcoreclock = hal_rcu_clock_freq_get(CK_SYS);

            /* configure the source of time base considering new system clocks settings */
            hal_sys_timesource_init(g_basetick_source);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief    get the system clock, bus and peripheral clock frequency
    \param[in]  clock: the clock frequency which to get
                only one parameter can be selected which is shown as below:
      \arg        CK_SYS: system clock frequency
      \arg        CK_AHB: AHB clock frequency
      \arg        CK_APB1: APB1 clock frequency
      \arg        CK_APB2: APB2 clock frequency
      \arg        CK_APB3: APB3 clock frequency
      \arg        CK_APB4: APB4 clock frequency
      \arg        CK_PLL0P: PLL0P clock frequency
      \arg        CK_PLL0Q: PLL0Q clock frequency
      \arg        CK_PLL0R: PLL0R clock frequency
      \arg        CK_PLL1P: PLL1P clock frequency
      \arg        CK_PLL1Q: PLL1Q clock frequency
      \arg        CK_PLL1R: PLL1R clock frequency
      \arg        CK_PLL2P: PLL1P clock frequency
      \arg        CK_PLL2Q: PLL1Q clock frequency
      \arg        CK_PLL2R: PLL1R clock frequency
      \arg        CK_PER: PER clock frequency
      \arg        CK_USART0: USART0 clock frequency
      \arg        CK_USART1: USART1 clock frequency
      \arg        CK_USART2: USART2 clock frequency
      \arg        CK_USART5: USART5 clock frequency
      \arg        CK_I2C0:   I2C0 clock frequency
      \arg        CK_I2C1:   I2C1 clock frequency
      \arg        CK_I2C2:   I2C2 clock frequency
      \arg        CK_I2C3:   I2C2 clock frequency
      \arg        CK_CAN0:   CAN0 clock frequency
      \arg        CK_CAN1:   CAN1 clock frequency
      \arg        CK_CAN2:   CAN2 clock frequency
      \arg        CK_ADC01:  ADC01 clock frequency
      \arg        CK_ADC2:   ADC2 clock frequency
      \arg        CK_SDIO0:  SDIO0 clock frequency
      \arg        CK_SDIO1:  SDIO1 clock frequency
      \arg        CK_SPI3:   SPI3 clock frequency
      \arg        CK_SPI4:   SPI4 clock frequency
      \arg        CK_RTC:    RTC clock frequency
      \arg        CK_EXMC:   EXMC clock frequency
      \arg        CK_RSPDIF: RSPDIF clock frequency
      \arg        CK_IRC64MDIV: IRC64MDIV clock frequency
      \arg        CK_HXTAL: HXTAL clock frequency
      \arg        CK_LPIRC4M: LPIRC4M clock frequency
    \param[out] none
    \retval     uint32_t : 0 - 0xFFFFFFFF
*/
uint32_t hal_rcu_clock_freq_get(hal_rcu_clock_freq_enum clock)
{
    uint32_t system_clock_Source = 0U, ck_freq = 0U, irc64mdiv_freq = 0U, fracn = 0U;
    uint32_t cksys_freq = 0U, ahb_freq = 0U, apb1_freq = 0U, apb2_freq = 0U, apb3_freq = 0U, apb4_freq = 0U;
    uint32_t pll0p_freq = 0U, pll0q_freq = 0U, pll0r_freq = 0U, pll1p_freq = 0U, pll1q_freq = 0U, pll1r_freq = 0U, \
             pll2p_freq = 0U, pll2q_freq = 0U, pll2r_freq = 0U, per_freq = 0U, persel = 0U;
    uint32_t pll0psc = 0U, pll1psc = 0U, pll2psc = 0U, pll0n = 0U, pll1n = 0U, pll2n = 0U, pllsel = 0U, pll0p = 0U, \
             pll0q = 0U, pll0r = 0U, pll1p = 0U, pll1q = 0U, pll1r = 0U, pll2p = 0U, pll2q = 0U, pll2r = 0U, \
             ck_src = 0U, idx = 0U, clk_exp = 0U;
    uint32_t usart0_freq = 0U, usart1_freq = 0U, usart2_freq = 0U, usart5_freq = 0U;
    uint32_t i2c0_freq = 0U, i2c1_freq = 0U, i2c2_freq = 0U, i2c3_freq = 0U;
    uint32_t can0_freq = 0U, can1_freq = 0U, can2_freq = 0U;
    uint32_t adc01_freq = 0U, adc2_freq = 0U, adc_sck = 0U, adc_div = 0U;
    uint32_t sdio0_freq = 0U, sdio1_freq = 0U;
    uint32_t spi0_freq = 0U, spi1_freq = 0U, spi2_freq = 0U, spi3_freq = 0U, spi4_freq = 0U, spi5_freq = 0U;
    uint32_t rtc_freq = 0U, exmc_freq = 0U, rspdif_freq = 0U, rtc_div = 0U;

    /* exponent of AHB, APB1 and APB2 clock divider */
    const uint8_t ahb_exp[16] = {0U, 0U, 0U, 0U, 0U, 0U, 0U, 0U, 1U, 2U, 3U, 4U, 6U, 7U, 8U, 9U};
    const uint8_t apb1_exp[8] = {0U, 0U, 0U, 0U, 1U, 2U, 3U, 4U};
    const uint8_t apb2_exp[8] = {0U, 0U, 0U, 0U, 1U, 2U, 3U, 4U};
    const uint8_t apb3_exp[8] = {0U, 0U, 0U, 0U, 1U, 2U, 3U, 4U};
    const uint8_t apb4_exp[8] = {0U, 0U, 0U, 0U, 1U, 2U, 3U, 4U};

    /*----------------------------- get system clock frequency -----------------------------*/
    system_clock_Source = GET_BITS(RCU_CFG0, 2U, 3U);
    irc64mdiv_freq      = hals_rcu_irc64mdiv_freq_get();

    switch(system_clock_Source) {
    /* IRC64MDIV is selected as CK_SYS */
    case SEL_IRC64MDIV:
        cksys_freq = irc64mdiv_freq;
        break;
    /* LPIRC4M is selected as CK_SYS */
    case SEL_LPIRC4M:
        cksys_freq = LPIRC4M_VALUE;
        break;
    /* HXTAL is selected as CK_SYS */
    case SEL_HXTAL:
        cksys_freq = HXTAL_VALUE;
        break;
    /* PLL0P is selected as CK_SYS */
    case SEL_PLL0P:
        pll0psc = GET_BITS(RCU_PLL0, 0U, 5U);
        pll0n   = (GET_BITS(RCU_PLL0, 6U, 14U) + 1U);
        pll0p   = (GET_BITS(RCU_PLL0, 16U, 22U) + 1U);

        if(RESET != (RCU_PLL0FRA & RCU_PLL0FRA_PLL0FRAEN)) {
            fracn = GET_BITS(RCU_PLL0FRA, 0U, 12U);
        } else {
            /* do nothing */
        }

        /* PLL clock source selection, HXTAL or IRC64MDIV */
        pllsel = (RCU_PLLALL & RCU_PLLALL_PLLSEL);

        if(RCU_PLLSRC_HXTAL == pllsel) {
            ck_src = HXTAL_VALUE;
        } else if(RCU_PLLSRC_IRC64MDIV == pllsel) {
            ck_src = irc64mdiv_freq;
        } else {
            ck_src = LPIRC4M_VALUE;
        }

        cksys_freq = _rcu_pll_clock_freq_cal(ck_src, pll0psc, pll0n, fracn, pll0p);
        break;
    /* IRC64MDIV is selected as CK_SYS */
    default:
        cksys_freq = irc64mdiv_freq;
        break;
    }

    /*----------------------------- get AHB\APB clock frequency -----------------------------*/
    /* calculate AHB clock frequency */
    idx      = GET_BITS(RCU_CFG0, 4U, 7U);
    clk_exp  = ahb_exp[idx];
    ahb_freq = cksys_freq >> clk_exp;

    /* calculate APB1 clock frequency */
    idx       = GET_BITS(RCU_CFG0, 10U, 12U);
    clk_exp   = apb1_exp[idx];
    apb1_freq = ahb_freq >> clk_exp;

    /* calculate APB2 clock frequency */
    idx       = GET_BITS(RCU_CFG0, 13U, 15U);
    clk_exp   = apb2_exp[idx];
    apb2_freq = ahb_freq >> clk_exp;

    /* calculate APB3 clock frequency */
    idx       = GET_BITS(RCU_CFG0, 27U, 29U);
    clk_exp   = apb3_exp[idx];
    apb3_freq = ahb_freq >> clk_exp;

    /* calculate APB4 clock frequency */
    idx       = GET_BITS(RCU_CFG0, 24U, 26U);
    clk_exp   = apb4_exp[idx];
    apb4_freq = ahb_freq >> clk_exp;

    /*----------------------------- get PLL0 clock frequency -----------------------------*/
    /* calculate pll0 clock frequency */
    pll0psc = GET_BITS(RCU_PLL0,  0U, 5U);
    pll0n   = (GET_BITS(RCU_PLL0, 6U, 14U) + 1U);
    pll0p   = (GET_BITS(RCU_PLL0, 16U, 22U) + 1U);
    pll0r   = (GET_BITS(RCU_PLL0, 24U, 30U) + 1U);
    pll0q   = (GET_BITS(RCU_PLLADDCTL, 0U, 6U) + 1U);

    if(RESET != (RCU_PLL0FRA & RCU_PLL0FRA_PLL0FRAEN)) {
        fracn = GET_BITS(RCU_PLL0FRA, 0U, 12U);
    } else {
        /* do nothing */
    }

    /* PLL clock source selection (HXTAL, IRC64MDIV or LPIRC4M) */
    pllsel = (RCU_PLLALL & RCU_PLLALL_PLLSEL);

    if(RCU_PLLSRC_HXTAL == pllsel) {
        ck_src = HXTAL_VALUE;
    } else if(RCU_PLLSRC_IRC64MDIV == pllsel) {
        ck_src = irc64mdiv_freq;
    } else {
        ck_src = LPIRC4M_VALUE;
    }

    if((0U != pll0psc) && (0U != ck_src)) {
        if(RESET != (RCU_PLLADDCTL & RCU_PLLADDCTL_PLL0PEN)) {
            pll0p_freq = _rcu_pll_clock_freq_cal(ck_src, pll0psc, pll0n, fracn, pll0p);
            pll0r_freq = _rcu_pll_clock_freq_cal(ck_src, pll0psc, pll0n, fracn, pll0r);
            pll0q_freq = _rcu_pll_clock_freq_cal(ck_src, pll0psc, pll0n, fracn, pll0q);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /*----------------------------- get PLL1 clock frequency -----------------------------*/
    /* calculate pll1 clock frequency */
    pll1psc = GET_BITS(RCU_PLL1, 0U, 5U);
    pll1n   = (GET_BITS(RCU_PLL1, 6U, 14U) + 1U);
    pll1p   = (GET_BITS(RCU_PLL1, 16U, 22U) + 1U);
    pll1r   = (GET_BITS(RCU_PLL1, 24U, 30U) + 1U);
    pll1q   = (GET_BITS(RCU_PLLADDCTL, 8U, 14U) + 1U);

    if(0U != (RCU_PLL1FRA & RCU_PLL1FRA_PLL1FRAEN)) {
        fracn = GET_BITS(RCU_PLL1FRA, 0U, 12U);
    } else {
        /* do nothing */
    }

    /* PLL clock source selection (HXTAL, IRC64MDIV or LPIRC4M) */
    pllsel = (RCU_PLLALL & RCU_PLLALL_PLLSEL);

    if(RCU_PLLSRC_HXTAL == pllsel) {
        ck_src = HXTAL_VALUE;
    } else if(RCU_PLLSRC_IRC64MDIV == pllsel) {
        ck_src = irc64mdiv_freq;
    } else {
        ck_src = LPIRC4M_VALUE;
    }

    if((0U != pll1psc) && (0U != ck_src)) {
        if(RESET != (RCU_PLLADDCTL & RCU_PLLADDCTL_PLL1PEN)) {
            pll1p_freq = _rcu_pll_clock_freq_cal(ck_src, pll1psc, pll1n, fracn, pll1p);
            pll1r_freq = _rcu_pll_clock_freq_cal(ck_src, pll1psc, pll1n, fracn, pll1r);
            pll1q_freq = _rcu_pll_clock_freq_cal(ck_src, pll1psc, pll1n, fracn, pll1q);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /*----------------------------- get PLL2 clock frequency -----------------------------*/
    /* calculate pll1 clock frequency */
    pll2psc = GET_BITS(RCU_PLL2, 0U, 5U);
    pll2n   = (GET_BITS(RCU_PLL2, 6U, 14U) + 1U);
    pll2p   = (GET_BITS(RCU_PLL2, 16U, 22U) + 1U);
    pll2r   = (GET_BITS(RCU_PLL2, 24U, 30U) + 1U);
    pll2q   = (GET_BITS(RCU_PLLADDCTL, 16U, 22U) + 1U);

    if(RESET != (RCU_PLL2FRA & RCU_PLL2FRA_PLL2FRAEN)) {
        fracn = GET_BITS(RCU_PLL2FRA, 0U, 12U);
    } else {
        /* do nothing */
    }

    /* PLL clock source selection (HXTAL, IRC64MDIV or LPIRC4M) */
    pllsel = (RCU_PLLALL & RCU_PLLALL_PLLSEL);

    if(RCU_PLLSRC_HXTAL == pllsel) {
        ck_src = HXTAL_VALUE;
    } else if(RCU_PLLSRC_IRC64MDIV == pllsel) {
        ck_src = irc64mdiv_freq;
    } else {
        ck_src = LPIRC4M_VALUE;
    }

    if((0U != pll2psc) && (0U != ck_src)) {
        if(RESET != (RCU_PLLADDCTL & RCU_PLLADDCTL_PLL2PEN)) {
            pll2p_freq = _rcu_pll_clock_freq_cal(ck_src, pll2psc, pll2n, fracn, pll2p);
            pll2r_freq = _rcu_pll_clock_freq_cal(ck_src, pll2psc, pll2n, fracn, pll2r);
            pll2q_freq = _rcu_pll_clock_freq_cal(ck_src, pll2psc, pll2n, fracn, pll2q);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /*----------------------------- get peripheral clock frequency -----------------------------*/
    persel = (RCU_CFG1 & RCU_CFG1_PERSEL);
    if(RCU_PERSRC_HXTAL == persel) {
        per_freq = HXTAL_VALUE;
    } else if(RCU_PLLSRC_IRC64MDIV == persel) {
        per_freq = irc64mdiv_freq;
    } else {
        per_freq = LPIRC4M_VALUE;
    }

    /*----------------------------- get USART clock frequency -----------------------------*/
    /* calculate USART0 clock frequency */
    if(RCU_USARTSRC_APB2 == (RCU_CFG1 & RCU_CFG1_USART0SEL)) {
        usart0_freq = apb2_freq;
    } else if(RCU_USARTSRC_AHB == (RCU_CFG1 & RCU_CFG1_USART0SEL)) {
        usart0_freq = ahb_freq;
    } else if(RCU_USARTSRC_LXTAL == (RCU_CFG1 & RCU_CFG1_USART0SEL)) {
        usart0_freq = LXTAL_VALUE;
    } else if(RCU_USARTSRC_IRC64MDIV == (RCU_CFG1 & RCU_CFG1_USART0SEL)) {
        usart0_freq = irc64mdiv_freq;
    } else {
        /* do nothing */
    }

    /* calculate USART1 clock frequency */
    if((RCU_USARTSRC_APB2 << 18U) == (RCU_CFG1 & RCU_CFG1_USART1SEL)) {
        usart1_freq = apb1_freq;
    } else if((RCU_USARTSRC_AHB << 18U) == (RCU_CFG1 & RCU_CFG1_USART1SEL)) {
        usart1_freq = ahb_freq;
    } else if((RCU_USARTSRC_LXTAL << 18U) == (RCU_CFG1 & RCU_CFG1_USART1SEL)) {
        usart1_freq = LXTAL_VALUE;
    } else if((RCU_USARTSRC_IRC64MDIV << 18U) == (RCU_CFG1 & RCU_CFG1_USART1SEL)) {
        usart1_freq = irc64mdiv_freq;
    } else {
        /* do nothing */
    }

    /* calculate USART2 clock frequency */
    if((RCU_USARTSRC_APB2 << 20U) == (RCU_CFG1 & RCU_CFG1_USART2SEL)) {
        usart2_freq = apb1_freq;
    } else if((RCU_USARTSRC_AHB << 20U) == (RCU_CFG1 & RCU_CFG1_USART2SEL)) {
        usart2_freq = ahb_freq;
    } else if((RCU_USARTSRC_LXTAL << 20U) == (RCU_CFG1 & RCU_CFG1_USART2SEL)) {
        usart2_freq = LXTAL_VALUE;
    } else if((RCU_USARTSRC_IRC64MDIV << 20U) == (RCU_CFG1 & RCU_CFG1_USART2SEL)) {
        usart2_freq = irc64mdiv_freq;
    } else {
        /* do nothing */
    }

    /* calculate USART5 clock frequency */
    if((RCU_USARTSRC_APB2 << 22U) == (RCU_CFG1 & RCU_CFG1_USART5SEL)) {
        usart5_freq = apb2_freq;
    } else if((RCU_USARTSRC_AHB << 22U) == (RCU_CFG1 & RCU_CFG1_USART5SEL)) {
        usart5_freq = ahb_freq;
    } else if((RCU_USARTSRC_LXTAL << 22U) == (RCU_CFG1 & RCU_CFG1_USART5SEL)) {
        usart5_freq = LXTAL_VALUE;
    } else if((RCU_USARTSRC_IRC64MDIV << 22U) == (RCU_CFG1 & RCU_CFG1_USART5SEL)) {
        usart5_freq = irc64mdiv_freq;
    } else {
        /* do nothing */
    }

    /*----------------------------- get I2C clock frequency -----------------------------*/
    /* calculate I2C0 clock frequency */
    if((RCU_I2CSRC_APB1 << 30U) == (RCU_CFG0 & RCU_CFG0_I2C0SEL)) {
        i2c0_freq = apb1_freq;
    } else if((RCU_I2CSRC_PLL2R << 30U) == (RCU_CFG0 & RCU_CFG0_I2C0SEL)) {
        i2c0_freq = pll2r_freq;
    } else if((RCU_I2CSRC_IRC64MDIV << 30U) == (RCU_CFG0 & RCU_CFG0_I2C0SEL)) {
        i2c0_freq = irc64mdiv_freq;
    } else if((RCU_I2CSRC_LPIRC4M << 30U) == (RCU_CFG0 & RCU_CFG0_I2C0SEL)) {
        i2c0_freq = LPIRC4M_VALUE;
    } else {
        /* do nothing */
    }

    /* calculate I2C1 clock frequency */
    if(RCU_I2CSRC_APB1 == (RCU_CFG3 & RCU_CFG3_I2C1SEL)) {
        i2c1_freq = apb1_freq;
    } else if(RCU_I2CSRC_PLL2R == (RCU_CFG3 & RCU_CFG3_I2C1SEL)) {
        i2c1_freq = pll2r_freq;
    } else if(RCU_I2CSRC_IRC64MDIV == (RCU_CFG3 & RCU_CFG3_I2C1SEL)) {
        i2c1_freq = irc64mdiv_freq;
    } else if(RCU_I2CSRC_LPIRC4M == (RCU_CFG3 & RCU_CFG3_I2C1SEL)) {
        i2c1_freq = LPIRC4M_VALUE;
    } else {
        /* do nothing */
    }

    /* calculate I2C2 clock frequency */
    if((RCU_I2CSRC_APB1 << 2U) == (RCU_CFG3 & RCU_CFG3_I2C2SEL)) {
        i2c2_freq = apb1_freq;
    } else if((RCU_I2CSRC_PLL2R << 2U) == (RCU_CFG3 & RCU_CFG3_I2C2SEL)) {
        i2c2_freq = pll2r_freq;
    } else if((RCU_I2CSRC_IRC64MDIV << 2U) == (RCU_CFG3 & RCU_CFG3_I2C2SEL)) {
        i2c2_freq = irc64mdiv_freq;
    } else if((RCU_I2CSRC_LPIRC4M << 2U) == (RCU_CFG3 & RCU_CFG3_I2C2SEL)) {
        i2c2_freq = LPIRC4M_VALUE;
    } else {
        /* do nothing */
    }

    /* calculate I2C3 clock frequency */
    if((RCU_I2CSRC_APB1 << 4U) == (uint8_t)(RCU_CFG3 & RCU_CFG3_I2C3SEL)) {
        i2c3_freq = apb1_freq;
    } else if((RCU_I2CSRC_PLL2R << 4U) == (uint8_t)(RCU_CFG3 & RCU_CFG3_I2C3SEL)) {
        i2c3_freq = pll2r_freq;
    } else if((RCU_I2CSRC_IRC64MDIV << 4U) == (uint8_t)(RCU_CFG3 & RCU_CFG3_I2C3SEL)) {
        i2c3_freq = irc64mdiv_freq;
    } else if((RCU_I2CSRC_LPIRC4M << 4U) == (uint8_t)(RCU_CFG3 & RCU_CFG3_I2C3SEL)) {
        i2c3_freq = LPIRC4M_VALUE;
    } else {
        /* do nothing */
    }

    /*----------------------------- get CAN clock frequency -----------------------------*/
    /* calculate CAN0 clock frequency */
    if((RCU_CANSRC_HXTAL) == (RCU_CFG1 & RCU_CFG1_CAN0SEL)) {
        can0_freq = HXTAL_VALUE;
    } else if((RCU_CANSRC_APB2) == (RCU_CFG1 & RCU_CFG1_CAN0SEL)) {
        can0_freq = apb2_freq;
    } else if((RCU_CANSRC_APB2_DIV2) == (RCU_CFG1 & RCU_CFG1_CAN0SEL)) {
        can0_freq = apb2_freq / 2U;
    } else if((RCU_CANSRC_IRC64MDIV) == (RCU_CFG1 & RCU_CFG1_CAN0SEL)) {
        can0_freq = irc64mdiv_freq;
    } else {
        /* do nothing */
    }

    /* calculate CAN1 clock frequency */
    if((RCU_CANSRC_HXTAL << 2U) == (RCU_CFG1 & RCU_CFG1_CAN1SEL)) {
        can1_freq = HXTAL_VALUE;
    } else if((RCU_CANSRC_APB2 << 2U) == (RCU_CFG1 & RCU_CFG1_CAN1SEL)) {
        can1_freq = apb2_freq;
    } else if((RCU_CANSRC_APB2_DIV2 << 2U) == (RCU_CFG1 & RCU_CFG1_CAN1SEL)) {
        can1_freq = apb2_freq / 2U;
    } else if((RCU_CANSRC_IRC64MDIV << 2U) == (RCU_CFG1 & RCU_CFG1_CAN1SEL)) {
        can1_freq = irc64mdiv_freq;
    } else {
        /* do nothing */
    }

    /* calculate CAN2 clock frequency */
    if((RCU_CANSRC_HXTAL << 4U) == (RCU_CFG1 & RCU_CFG1_CAN2SEL)) {
        can2_freq = HXTAL_VALUE;
    } else if((RCU_CANSRC_APB2 << 4U) == (RCU_CFG1 & RCU_CFG1_CAN2SEL)) {
        can2_freq = apb2_freq;
    } else if((RCU_CANSRC_APB2_DIV2 << 4U) == (RCU_CFG1 & RCU_CFG1_CAN2SEL)) {
        can2_freq = apb2_freq / 2U;
    } else if((RCU_CANSRC_IRC64MDIV << 4U) == (RCU_CFG1 & RCU_CFG1_CAN2SEL)) {
        can2_freq = irc64mdiv_freq;
    } else {
        /* do nothing */
    }

    /*----------------------------- get ADC clock frequency -----------------------------*/
    adc_sck = GET_BITS(ADC_SYNCCTL_ADCSCK, 16U, 19U);

    switch(adc_sck) {
    case ADC_CLK_SYNC_HCLK_DIV2:
        adc_div = 2U;
        break;
    case ADC_CLK_SYNC_HCLK_DIV4:
        adc_div = 4U;
        break;
    case ADC_CLK_SYNC_HCLK_DIV8:
        adc_div = 8U;
        break;
    case ADC_CLK_SYNC_HCLK_DIV10:
        adc_div = 10U;
        break;
    case ADC_CLK_SYNC_HCLK_DIV12:
        adc_div = 12U;
        break;
    case ADC_CLK_SYNC_HCLK_DIV14:
        adc_div = 14U;
        break;
    case ADC_CLK_SYNC_HCLK_DIV16:
        adc_div = 16U;
        break;
    default:
        break;
    }

    /* calculate ADC01 clock frequency */
    if((RCU_ADCSRC_PLL1P) == (RCU_CFG3 & RCU_CFG3_ADC01SEL)) {
        adc01_freq = pll1p_freq / adc_div;
    } else if((RCU_ADCSRC_PLL2R) == (RCU_CFG3 & RCU_CFG3_ADC01SEL)) {
        adc01_freq = pll2r_freq / adc_div;
    } else if((RCU_ADCSRC_PER) == (RCU_CFG3 & RCU_CFG3_ADC01SEL)) {
        adc01_freq = per_freq / adc_div;
    } else {
        /* do nothing */
    }

    /* calculate ADC2 clock frequency */
    if((RCU_ADCSRC_PLL1P << 2U) == (RCU_CFG3 & RCU_CFG3_ADC2SEL)) {
        adc2_freq = pll1p_freq / adc_div;
    } else if((RCU_ADCSRC_PLL2R << 2U) == (RCU_CFG3 & RCU_CFG3_ADC2SEL)) {
        adc2_freq = pll2r_freq / adc_div;
    } else if((RCU_ADCSRC_PER << 2U) == (RCU_CFG3 & RCU_CFG3_ADC2SEL)) {
        adc2_freq = per_freq / adc_div;
    } else {
        /* do nothing */
    }

    /*----------------------------- get SDIO clock frequency -----------------------------*/
    /* calculate SDIO0 clock frequency */
    if((RCU_SDIO0SRC_PLL0Q) == (RCU_CFG4 & RCU_CFG4_SDIO0SEL)) {
        sdio0_freq = pll1p_freq;
    } else if((RCU_SDIO0SRC_PLL1R) == (RCU_CFG4 & RCU_CFG4_SDIO0SEL)) {
        sdio0_freq = pll2r_freq / adc_div;
    } else {
        /* do nothing */
    }

    /* calculate SDIO1 clock frequency */
    if((RCU_SDIO1SRC_PLL0Q) == (RCU_CFG3 & RCU_CFG3_SDIO1SEL)) {
        sdio1_freq = pll1p_freq;
    } else if((RCU_SDIO1SRC_PLL1R) == (RCU_CFG4 & RCU_CFG3_SDIO1SEL)) {
        sdio1_freq = pll2r_freq;
    } else {
        /* do nothing */
    }

    /*----------------------------- get SPI clock frequency -----------------------------*/
    /* calculate SPI0 clock frequency */
    if((RCU_SPISRC_PLL0Q) == (uint32_t)(RCU_CFG5 & RCU_CFG5_SPI0SEL)) {
        spi0_freq = pll0q_freq;
    } else if((RCU_SPISRC_PLL1P) == (uint32_t)(RCU_CFG5 & RCU_CFG5_SPI0SEL)) {
        spi0_freq = pll1p_freq;
    } else if((RCU_SPISRC_PLL2P) == (uint32_t)(RCU_CFG5 & RCU_CFG5_SPI0SEL)) {
        spi0_freq = pll2p_freq;
    } else if((RCU_SPISRC_PER) == (uint32_t)(RCU_CFG5 & RCU_CFG5_SPI0SEL)) {
        spi0_freq = per_freq;
    } else {
        /* do nothing */
    }

    /* calculate SPI1 clock frequency */
    if((RCU_SPISRC_PLL0Q << 4U) == (RCU_CFG5 & RCU_CFG5_SPI1SEL)) {
        spi1_freq = pll0q_freq;
    } else if((RCU_SPISRC_PLL1P << 4U) == (RCU_CFG5 & RCU_CFG5_SPI1SEL)) {
        spi1_freq = pll1p_freq;
    } else if((RCU_SPISRC_PLL2P << 4U) == (RCU_CFG5 & RCU_CFG5_SPI1SEL)) {
        spi1_freq = pll2p_freq;
    } else if((RCU_SPISRC_PER << 4U) == (RCU_CFG5 & RCU_CFG5_SPI1SEL)) {
        spi1_freq = per_freq;
    } else {
        /* do nothing */
    }

    /* calculate SPI2 clock frequency */
    if((RCU_SPISRC_PLL0Q << 8U) == (RCU_CFG5 & RCU_CFG5_SPI2SEL)) {
        spi2_freq = pll0q_freq;
    } else if((RCU_SPISRC_PLL1P << 8U) == (RCU_CFG5 & RCU_CFG5_SPI2SEL)) {
        spi2_freq = pll1p_freq;
    } else if((RCU_SPISRC_PLL2P << 8U) ==(RCU_CFG5 & RCU_CFG5_SPI2SEL)) {
        spi2_freq = pll2p_freq;
    } else if((RCU_SPISRC_PER << 8U) == (RCU_CFG5 & RCU_CFG5_SPI2SEL)) {
        spi2_freq = per_freq;
    } else {
        /* do nothing */
    }

    /* calculate SPI3 clock frequency */
    if((RCU_SPISRC_APB2) == (RCU_CFG5 & RCU_CFG5_SPI3SEL)) {
        spi3_freq = apb2_freq;
    } else if((RCU_SPISRC_PLL1Q) == (RCU_CFG5 & RCU_CFG5_SPI3SEL)) {
        spi3_freq = pll1q_freq;
    } else if((RCU_SPISRC_PLL2Q) == (RCU_CFG5 & RCU_CFG5_SPI3SEL)) {
        spi3_freq = pll2q_freq;
    } else if((RCU_SPISRC_IRC64MDIV) == (RCU_CFG5 & RCU_CFG5_SPI3SEL)) {
        spi3_freq = irc64mdiv_freq;
    } else if((RCU_SPISRC_LPIRC4M) == (RCU_CFG5 & RCU_CFG5_SPI3SEL)) {
        spi3_freq = LPIRC4M_VALUE;
    } else if((RCU_SPISRC_HXTAL) == (RCU_CFG5 & RCU_CFG5_SPI3SEL)) {
        spi3_freq = HXTAL_VALUE;
    } else {
        /* do nothing */
    }

    /* calculate SPI4 clock frequency */
    if((RCU_SPISRC_APB2 << 4U) == (RCU_CFG5 & RCU_CFG5_SPI4SEL)) {
        spi4_freq = apb2_freq;
    } else if((RCU_SPISRC_PLL1Q << 4U) == (RCU_CFG5 & RCU_CFG5_SPI4SEL)) {
        spi4_freq = pll1q_freq;
    } else if((RCU_SPISRC_PLL2Q << 4U) == (RCU_CFG5 & RCU_CFG5_SPI4SEL)) {
        spi4_freq = pll2q_freq;
    } else if((RCU_SPISRC_IRC64MDIV << 4U) == (RCU_CFG5 & RCU_CFG5_SPI4SEL)) {
        spi4_freq = irc64mdiv_freq;
    } else if((RCU_SPISRC_LPIRC4M << 4U) == (RCU_CFG5 & RCU_CFG5_SPI4SEL)) {
        spi4_freq = LPIRC4M_VALUE;
    } else if((RCU_SPISRC_HXTAL << 4U) == (RCU_CFG5 & RCU_CFG5_SPI4SEL)) {
        spi4_freq = HXTAL_VALUE;
    } else {
        /* do nothing */
    }

    /* calculate SPI5 clock frequency */
    if((RCU_SPI5SRC_APB2) == (RCU_CFG5 & RCU_CFG5_SPI5SEL)) {
        spi5_freq = apb2_freq;
    } else if((RCU_SPI5SRC_PLL1Q) == (RCU_CFG5 & RCU_CFG5_SPI5SEL)) {
        spi5_freq = pll1q_freq;
    } else if((RCU_SPI5SRC_PLL2Q) == (RCU_CFG5 & RCU_CFG5_SPI5SEL)) {
        spi5_freq = pll2q_freq;
    } else if((RCU_SPI5SRC_IRC64MDIV) == (RCU_CFG5 & RCU_CFG5_SPI5SEL)) {
        spi5_freq = irc64mdiv_freq;
    } else if((RCU_SPI5SRC_LPIRC4M) == (RCU_CFG5 & RCU_CFG5_SPI5SEL)) {
        spi5_freq = LPIRC4M_VALUE;
    } else if((RCU_SPI5SRC_HXTAL) == (RCU_CFG5 & RCU_CFG5_SPI5SEL)) {
        spi5_freq = HXTAL_VALUE;
    } else {
        /* do nothing */
    }

    /*----------------------------- get RTC clock frequency -----------------------------*/
    rtc_div = GET_BITS(RCU_CFG0, 16U, 21U);

    /* calculate RTC clock frequency */
    if((RCU_RTCSRC_LXTAL) == (uint32_t)(RCU_BDCTL & RCU_BDCTL_RTCSRC)) {
        rtc_freq = LXTAL_VALUE;
    } else if((RCU_RTCSRC_IRC32K) == (uint32_t)(RCU_BDCTL & RCU_BDCTL_RTCSRC)) {
        rtc_freq = IRC32K_VALUE;
    } else if((RCU_RTCSRC_HXTAL_DIV_RTCDIV) == (uint32_t)(RCU_BDCTL & RCU_BDCTL_RTCSRC)) {
        rtc_freq = HXTAL_VALUE / rtc_div;
    } else {
        /* do nothing */
    }

    /*----------------------------- get EXMC clock frequency -----------------------------*/
    /* calculate EXMC clock frequency */
    if((RCU_EXMCSRC_AHB) == (RCU_CFG4 & RCU_CFG4_EXMCSEL)) {
        exmc_freq = ahb_freq;
    } else if((RCU_EXMCSRC_PLL0Q) == (RCU_CFG4 & RCU_CFG4_EXMCSEL)) {
        exmc_freq = pll0q_freq;
    } else if((RCU_EXMCSRC_PLL1R) == (RCU_CFG4 & RCU_CFG4_EXMCSEL)) {
        exmc_freq = pll1r_freq;
    } else if((RCU_EXMCSRC_PER) == (RCU_CFG4 & RCU_CFG4_EXMCSEL)) {
        exmc_freq = per_freq;
    } else {
        /* do nothing */
    }

    /*----------------------------- get RSPDIF clock frequency -----------------------------*/
    /* calculate RSPDIF clock frequency */
    if((RCU_RSPDIFSRC_PLL0Q) == (RCU_CFG1 & RCU_CFG1_RSPDIFSEL)) {
        rspdif_freq = pll0q_freq;
    } else if((RCU_RSPDIFSRC_PLL1R) == (RCU_CFG1 & RCU_CFG1_RSPDIFSEL)) {
        rspdif_freq = pll1r_freq;
    } else if((RCU_RSPDIFSRC_PLL2R) == (RCU_CFG1 & RCU_CFG1_RSPDIFSEL)) {
        rspdif_freq = pll2r_freq;
    } else if((RCU_RSPDIFSRC_IRC64MDIV) == (RCU_CFG1 & RCU_CFG1_RSPDIFSEL)) {
        rspdif_freq = irc64mdiv_freq;
    } else {
        /* do nothing */
    }

    /*----------------------------- return the clocks frequency -----------------------------*/
    switch(clock) {
    case CK_SYS:
        ck_freq = cksys_freq;
        break;
    case CK_AHB:
        ck_freq = ahb_freq;
        break;
    case CK_APB1:
        ck_freq = apb1_freq;
        break;
    case CK_APB2:
        ck_freq = apb2_freq;
        break;
    case CK_APB3:
        ck_freq = apb3_freq;
        break;
    case CK_APB4:
        ck_freq = apb4_freq;
        break;
    case CK_PLL0P:
        ck_freq = pll0p_freq;
        break;
    case CK_PLL0Q:
        ck_freq = pll0q_freq;
        break;
    case CK_PLL0R:
        ck_freq = pll0r_freq;
        break;
    case CK_PLL1P:
        ck_freq = pll1p_freq;
        break;
    case CK_PLL1Q:
        ck_freq = pll1q_freq;
        break;
    case CK_PLL1R:
        ck_freq = pll1r_freq;
        break;
    case CK_PLL2P:
        ck_freq = pll2p_freq;
        break;
    case CK_PLL2Q:
        ck_freq = pll2q_freq;
        break;
    case CK_PLL2R:
        ck_freq = pll2r_freq;
        break;
    case CK_PER:
        ck_freq = per_freq;
        break;
    case CK_USART0:
        ck_freq = usart0_freq;
        break;
    case CK_USART1:
        ck_freq = usart1_freq;
        break;
    case CK_USART2:
        ck_freq = usart2_freq;
        break;
    case CK_USART5:
        ck_freq = usart5_freq;
        break;
    case CK_I2C0:
        ck_freq = i2c0_freq;
        break;
    case CK_I2C1:
        ck_freq = i2c1_freq;
        break;
    case CK_I2C2:
        ck_freq = i2c2_freq;
        break;
    case CK_I2C3:
        ck_freq = i2c3_freq;
        break;
    case CK_CAN0:
        ck_freq = can0_freq;
        break;
    case CK_CAN1:
        ck_freq = can1_freq;
        break;
    case CK_CAN2:
        ck_freq = can2_freq;
        break;
    case CK_ADC01:
        ck_freq = adc01_freq;
        break;
    case CK_ADC2:
        ck_freq = adc2_freq;
        break;
    case CK_SDIO0:
        ck_freq = sdio0_freq;
        break;
    case CK_SDIO1:
        ck_freq = sdio1_freq;
        break;
    case CK_SPI0:
        ck_freq = spi0_freq;
        break;
    case CK_SPI1:
        ck_freq = spi1_freq;
        break;
    case CK_SPI2:
        ck_freq = spi2_freq;
        break;
    case CK_SPI3:
        ck_freq = spi3_freq;
        break;
    case CK_SPI4:
        ck_freq = spi4_freq;
        break;
    case CK_SPI5:
        ck_freq = spi5_freq;
        break;
    case CK_RTC:
        ck_freq = rtc_freq;
        break;
    case CK_EXMC:
        ck_freq = exmc_freq;
        break;
    case CK_RSPDIF:
        ck_freq = rspdif_freq;
        break;
    case CK_IRC64MDIV:
        ck_freq = irc64mdiv_freq;
        break;
    case CK_HXTAL:
        ck_freq = HXTAL_VALUE;
        break;
    case CK_LXTAL:
        ck_freq = LXTAL_VALUE;
        break;
    case CK_LPIRC4M:
        ck_freq = LPIRC4M_VALUE;
        break;
    case CK_IRC32K:
        ck_freq = IRC32K_VALUE;
        break;
    case CK_IRC48M:
        ck_freq = IRC48M_VALUE;
        break;
    default:
        break;
    }

    return ck_freq;
}

/*!
    \brief      RCU interrupt handler content function,which is merely used in RCU_IRQHandler
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_rcu_irq(void)
{
    /* ----------------------------------- pll0 stable Configuration -------------------------------*/
    if(SET == hals_rcu_interrupt_flag_get(RCU_INT_FLAG_PLL0STB)) {
        hals_rcu_interrupt_flag_clear(RCU_INT_FLAG_PLL0STB_CLR);
        if(NULL != rcu_irq.pll0_stable_handle) {
            rcu_irq.pll0_stable_handle(NULL);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* ----------------------------------- pll1 stable Configuration -------------------------------*/
    if(SET == hals_rcu_interrupt_flag_get(RCU_INT_FLAG_PLL1STB)) {
        hals_rcu_interrupt_flag_clear(RCU_INT_FLAG_PLL1STB_CLR);
        if(NULL != rcu_irq.pll1_stable_handle) {
            rcu_irq.pll1_stable_handle(NULL);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* ----------------------------------- pll2 stable Configuration -------------------------------*/
    if(SET == hals_rcu_interrupt_flag_get(RCU_INT_FLAG_PLL2STB)) {
        hals_rcu_interrupt_flag_clear(RCU_INT_FLAG_PLL2STB_CLR);
        if(NULL != rcu_irq.pll2_stable_handle) {
            rcu_irq.pll2_stable_handle(NULL);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* ------------------------------ pllusbhs0 stable Configuration -------------------------------*/
    if(SET == hals_rcu_interrupt_flag_get(RCU_INT_FLAG_PLLUSBHS0STB)) {
        hals_rcu_interrupt_flag_clear(RCU_INT_FLAG_PLLUSBHS0STB_CLR);
        if(NULL != rcu_irq.pllusbhs0_stable_handle) {
            rcu_irq.pllusbhs0_stable_handle(NULL);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* ----------------------------- pllusbhs1 stable Configuration -------------------------------*/
    if(SET == hals_rcu_interrupt_flag_get(RCU_INT_FLAG_PLLUSBHS1STB)) {
        hals_rcu_interrupt_flag_clear(RCU_INT_FLAG_PLLUSBHS1STB_CLR);
        if(NULL != rcu_irq.pllusbhs1_stable_handle) {
            rcu_irq.pllusbhs1_stable_handle(NULL);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* ----------------------------------- LPIRC4M stable Configuration -------------------------------*/
    if(SET == hals_rcu_interrupt_flag_get(RCU_INT_FLAG_LPIRC4MSTB)) {
        hals_rcu_interrupt_flag_clear(RCU_INT_FLAG_LPIRC4MSTB_CLR);
        if(NULL != rcu_irq.lpirc4m_stable_handle) {
            rcu_irq.lpirc4m_stable_handle(NULL);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* ------------------------------------ IRC32K stable Configuration -------------------------------*/
    if(SET == hals_rcu_interrupt_flag_get(RCU_INT_FLAG_IRC32KSTB)) {
        hals_rcu_interrupt_flag_clear(RCU_INT_FLAG_IRC32KSTB_CLR);
        if(NULL != rcu_irq.irc32K_stable_handle) {
            rcu_irq.irc32K_stable_handle(NULL);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* ------------------------------------ IRC48M stable Configuration -------------------------------*/
    if(SET == hals_rcu_interrupt_flag_get(RCU_INT_FLAG_IRC48MSTB)) {
        hals_rcu_interrupt_flag_clear(RCU_INT_FLAG_IRC48MSTB_CLR);
        if(NULL != rcu_irq.irc48m_stable_handle) {
            rcu_irq.irc48m_stable_handle(NULL);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* ------------------------------ IRC64M stable Configuration -------------------------------*/
    if(SET == hals_rcu_interrupt_flag_get(RCU_INT_FLAG_IRC64MSTB)) {
        hals_rcu_interrupt_flag_clear(RCU_INT_FLAG_IRC64MSTB_CLR);
        if(NULL != rcu_irq.irc64m_stable_handle) {
            rcu_irq.irc64m_stable_handle(NULL);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* ------------------------------ LXTAL stable Configuration -------------------------------*/
    if(SET == hals_rcu_interrupt_flag_get(RCU_INT_FLAG_LXTALSTB)) {
        hals_rcu_interrupt_flag_clear(RCU_INT_FLAG_LXTALSTB_CLR);
        if(NULL != rcu_irq.lxtal_stable_handle) {
            rcu_irq.lxtal_stable_handle(NULL);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* ----------------------------------- HXTAL stable Configuration -------------------------------*/
    if(SET == hals_rcu_interrupt_flag_get(RCU_INT_FLAG_HXTALSTB)) {
        hals_rcu_interrupt_flag_clear(RCU_INT_FLAG_HXTALSTB_CLR);
        if(NULL != rcu_irq.hxtal_stable_handle) {
            rcu_irq.hxtal_stable_handle(NULL);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* ------------------------------------ HXTAL clock stuck Configuration --------------------*/
    if(SET == hals_rcu_interrupt_flag_get(RCU_INT_FLAG_CKM)) {
        hals_rcu_interrupt_flag_clear(RCU_INT_FLAG_CKM_CLR);
        if(NULL != rcu_irq.hxtal_stuck_handle) {
            rcu_irq.hxtal_stuck_handle(NULL);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* ------------------------------------ LXTAL clock stuck Configuration --------------------*/
    if(SET == hals_rcu_interrupt_flag_get(RCU_INT_FLAG_LCKM)) {
        hals_rcu_interrupt_flag_clear(RCU_INT_FLAG_LCKM_CLR);
        if(NULL != rcu_irq.lxtal_stuck_handle) {
            rcu_irq.lxtal_stuck_handle(NULL);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  prcu_irq: point to RCU interrupt callback functions structure
                The structure member can be assigned as following
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                  the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg        NULL : The corresponding callback mechanism is out of use, and disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rcu_irq_handle_set(hal_rcu_irq_struct *prcu_irq)
{
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == prcu_irq) {
        HAL_DEBUGE("pointer [prcu_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* --------------------------------- PLL0 stable interrupt handler set --------------------*/
    if(NULL != prcu_irq->pll0_stable_handle) {
        rcu_irq.pll0_stable_handle = prcu_irq->pll0_stable_handle;
        hals_rcu_interrupt_enable(RCU_INT_PLL0STB);
    } else {
        rcu_irq.pll0_stable_handle = NULL;
        hals_rcu_interrupt_disable(RCU_INT_PLL0STB);
    }

    /* --------------------------------- PLL1 stable interrupt handler set --------------------*/
    if(NULL != prcu_irq->pll1_stable_handle) {
        rcu_irq.pll1_stable_handle = prcu_irq->pll1_stable_handle;
        hals_rcu_interrupt_enable(RCU_INT_PLL1STB);
    } else {
        rcu_irq.pll1_stable_handle = NULL;
        hals_rcu_interrupt_disable(RCU_INT_PLL1STB);
    }

    /* --------------------------------- pll2 stable interrupt handler set --------------------*/
    if(NULL != prcu_irq->pll2_stable_handle) {
        rcu_irq.pll2_stable_handle = prcu_irq->pll2_stable_handle;
        hals_rcu_interrupt_enable(RCU_INT_PLL2STB);
    } else {
        rcu_irq.pll2_stable_handle = NULL;
        hals_rcu_interrupt_disable(RCU_INT_PLL2STB);
    }

    /* --------------------------------- pllusbhs0 stable interrupt handler set --------------------*/
    if(NULL != prcu_irq->pllusbhs0_stable_handle) {
        rcu_irq.pllusbhs0_stable_handle = prcu_irq->pllusbhs0_stable_handle;
        hals_rcu_interrupt_enable(RCU_INT_PLLUSBHS0STB);
    } else {
        rcu_irq.pllusbhs0_stable_handle = NULL;
        hals_rcu_interrupt_disable(RCU_INT_PLLUSBHS0STB);
    }

    /* --------------------------------- pllusbhs1 stable interrupt handler set --------------------*/
    if(NULL != prcu_irq->pllusbhs1_stable_handle) {
        rcu_irq.pllusbhs1_stable_handle = prcu_irq->pllusbhs1_stable_handle;
        hals_rcu_interrupt_enable(RCU_INT_PLLUSBHS1STB);
    } else {
        rcu_irq.pllusbhs1_stable_handle = NULL;
        hals_rcu_interrupt_disable(RCU_INT_PLLUSBHS1STB);
    }

    /* --------------------------------- LPIRC4M stable interrupt handler set --------------------*/
    if(NULL != prcu_irq->lpirc4m_stable_handle) {
        rcu_irq.lpirc4m_stable_handle = prcu_irq->lpirc4m_stable_handle;
        hals_rcu_interrupt_enable(RCU_INT_LPIRC4MSTB);
    } else {
        rcu_irq.lpirc4m_stable_handle = NULL;
        hals_rcu_interrupt_disable(RCU_INT_LPIRC4MSTB);
    }

    /* --------------------------------- IRC32K stable interrupt handler set --------------------*/
    if(NULL != prcu_irq->irc32K_stable_handle) {
        rcu_irq.irc32K_stable_handle = prcu_irq->irc32K_stable_handle;
        hals_rcu_interrupt_enable(RCU_INT_IRC32KSTB);
    } else {
        rcu_irq.irc32K_stable_handle = NULL;
        hals_rcu_interrupt_disable(RCU_INT_IRC32KSTB);
    }

    /* --------------------------------- IRC48M stable interrupt handler set --------------------*/
    if(NULL != prcu_irq->irc48m_stable_handle) {
        rcu_irq.irc48m_stable_handle = prcu_irq->irc48m_stable_handle;
        hals_rcu_interrupt_enable(RCU_INT_IRC48MSTB);
    } else {
        rcu_irq.irc48m_stable_handle = NULL;
        hals_rcu_interrupt_disable(RCU_INT_IRC48MSTB);
    }

    /* --------------------------------- IRC64M stable interrupt handler set --------------------*/
    if(NULL != prcu_irq->irc64m_stable_handle) {
        rcu_irq.irc64m_stable_handle = prcu_irq->irc64m_stable_handle;
        hals_rcu_interrupt_enable(RCU_INT_IRC64MSTB);
    } else {
        rcu_irq.irc64m_stable_handle = NULL;
        hals_rcu_interrupt_disable(RCU_INT_IRC64MSTB);
    }

    /* --------------------------------- HXTAL stable interrupt handler set --------------------*/
    if(NULL != prcu_irq->hxtal_stable_handle) {
        rcu_irq.hxtal_stable_handle = prcu_irq->hxtal_stable_handle;
        hals_rcu_interrupt_enable(RCU_INT_HXTALSTB);
    } else {
        rcu_irq.hxtal_stable_handle = NULL;
        hals_rcu_interrupt_disable(RCU_INT_HXTALSTB);
    }

    /* --------------------------------- LXTAL stable interrupt handler set --------------------*/
    if(NULL != prcu_irq->lxtal_stable_handle) {
        rcu_irq.lxtal_stable_handle = prcu_irq->lxtal_stable_handle;
        hals_rcu_interrupt_enable(RCU_INT_LXTALSTB);
    } else {
        rcu_irq.lxtal_stable_handle = NULL;
        hals_rcu_interrupt_disable(RCU_INT_LXTALSTB);
    }

    /* --------------------------------- HXTAL stuck interrupt handler set --------------------*/
    if(NULL != prcu_irq->hxtal_stuck_handle) {
        rcu_irq.hxtal_stuck_handle = prcu_irq->hxtal_stuck_handle;
    } else {
        /* do nothing */
    }

    /* --------------------------------- LXTAL stuck interrupt handler set --------------------*/
    if(NULL != prcu_irq->lxtal_stuck_handle) {
        rcu_irq.lxtal_stuck_handle = prcu_irq->lxtal_stuck_handle;
        hal_exti_internal_init(EXTI_LINE_18_RTC_TAMPER_TIMESTAMP, EXTI_INTERRUPT_TRIG_BOTH);
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_rcu_irq_handle_all_reset(void)
{
    rcu_irq.pll0_stable_handle      = NULL;
    rcu_irq.pll1_stable_handle      = NULL;
    rcu_irq.pll2_stable_handle      = NULL;
    rcu_irq.pllusbhs0_stable_handle = NULL;
    rcu_irq.pllusbhs1_stable_handle = NULL;
    rcu_irq.lpirc4m_stable_handle   = NULL;
    rcu_irq.irc32K_stable_handle    = NULL;
    rcu_irq.irc48m_stable_handle    = NULL;
    rcu_irq.irc64m_stable_handle    = NULL;
    rcu_irq.lxtal_stable_handle     = NULL;
    rcu_irq.hxtal_stable_handle     = NULL;
    rcu_irq.hxtal_stuck_handle      = NULL;
    rcu_irq.lxtal_stuck_handle      = NULL;
}

/*!
    \brief      reset source detect
    \param[in]  none
    \param[out] none
    \retval     none
*/
uint8_t hal_rcu_reset_source_detect(void)
{
    uint8_t reset_source = 0U;

    /* BOR reset */
    if(SET == hals_rcu_flag_get(RCU_FLAG_BORRST)) {
        reset_source |= (1U << 0U);
    } else {
        /* do nothing */
    }

    /* External PIN reset flags */
    if(SET == hals_rcu_flag_get(RCU_FLAG_EPRST)) {
        reset_source |= (1U << 1U);
    } else {
        /* do nothing */
    }

    /* power reset flags */
    if(SET == hals_rcu_flag_get(RCU_FLAG_PORRST)) {
        reset_source |= (1U << 2U);
    } else {
        /* do nothing */
    }

    /* Software reset flags */
    if(SET == hals_rcu_flag_get(RCU_FLAG_SWRST)) {
        reset_source |= (1U << 3U);
    } else {
        /* do nothing */
    }

    /* FWDGT reset flags */
    if(SET == hals_rcu_flag_get(RCU_FLAG_FWDGTRST)) {
        reset_source |= (1U << 4U);
    } else {
        /* do nothing */
    }

    /* WWDGT reset flags */
    if(SET == hals_rcu_flag_get(RCU_FLAG_WWDGTRST)) {
        reset_source |= (1U << 5U);
    } else {
        /* do nothing */
    }

    /* low-power reset flags */
    if(SET == hals_rcu_flag_get(RCU_FLAG_LPRST)) {
        reset_source |= (1U << 6U);
    } else {
        /* do nothing */
    }

    /* clear all reset flags */
    hals_rcu_all_reset_flag_clear();

    return reset_source;
}

/*!
    \brief      configure the Deep-sleep wakeup system clock source selection
    \param[in]  ck_dspwussel: Deep-sleep wakeup system clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_DSPWUSSEL_CKSRC_IRC64MDIV: ck_dspwussel select IRC64MDIV
      \arg        RCU_DSPWUSSEL_CKSRC_LPIRC4M: ck_dspwussel select LPIRC4M
    \param[out] none
    \retval     none
*/
void hal_rcu_deepsleep_wakeup_sys_clock_config(hal_rcu_dspwu_cksrc_enum ck_dspwussel)
{
    /* reset the DSPWUSSEL bits and set according to ck_dspwussel */
    RCU_CFG3 &= ~RCU_CFG3_DSPWUSSEL;
    RCU_CFG3 |= (uint32_t)ck_dspwussel;
}

/*!
    \brief      configure the IRC64M clock divider selection
    \param[in]  ck_irc64mdiv: IRC64M clock divider selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_IRC64M_DIV_1: CK_IRC64M / 1
      \arg        RCU_IRC64M_DIV_2: CK_IRC64M / 2
      \arg        RCU_IRC64M_DIV_4: CK_IRC64M / 4
      \arg        RCU_IRC64M_DIV_8: CK_IRC64M / 8
    \param[out] none
    \retval     none
*/
void hals_rcu_irc64mdiv_clock_config(hal_rcu_irc64m_div_enum ck_irc64mdiv)
{
    uint32_t reg = 0U;

    /* reset the IRC64MDIV and set according to ck_irc64mdiv */
    reg = RCU_ADDCTL1;
    reg &= ~RCU_ADDCTL1_IRC64MDIV;
    RCU_ADDCTL1 = (reg | (uint32_t)ck_irc64mdiv);
}

/*!
    \brief      wait for oscillator stabilization flags is SET or oscillator startup is timeout
    \param[in]  osci: oscillator types, refer to hal_rcu_osci_type_enum
                only one parameter can be selected which is shown as below:
      \arg        RCU_HXTAL: HXTAL
      \arg        RCU_LXTAL: LXTAL
      \arg        RCU_IRC64M: IRC64M
      \arg        RCU_IRC48M: IRC48M
      \arg        RCU_IRC32K: IRC32K
      \arg        RCU_LPIRC4M: LPIRC4M
      \arg        RCU_PLL0_CK: PLL0
      \arg        RCU_PLL1_CK: PLL1
      \arg        RCU_PLL2_CK: PLL2
      \arg        RCU_PLLUSBHS0_CK: PLLUSBHS0
      \arg        RCU_PLLUSBHS1_CK: PLLUSBHS1
    \param[out] none
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_rcu_osci_stab_wait(hal_rcu_osci_type_enum osci)
{
    uint32_t stb_cnt      = 0U;
    ErrStatus ret       = ERROR;
    FlagStatus osci_state = RESET;

    switch(osci) {
    case RCU_HXTAL:
        /*--------------------------------- wait HXTAL stable -------------------------*/
        while((RESET == osci_state) && (HXTAL_STARTUP_TIMEOUT != stb_cnt)) {
            osci_state = hals_rcu_flag_get(RCU_FLAG_HXTALSTB);
            stb_cnt++;
        }

        /* check whether flag is set */
        if(RESET != hals_rcu_flag_get(RCU_FLAG_HXTALSTB)) {
            ret = SUCCESS;
        } else {
            /* do nothing */
        }
        break;
    case RCU_LXTAL:
        /*--------------------------------- wait LXTAL stable -------------------------*/
        while((RESET == osci_state) && (LXTAL_STARTUP_TIMEOUT != stb_cnt)) {
            osci_state = hals_rcu_flag_get(RCU_FLAG_LXTALSTB);
            stb_cnt++;
        }

        /* check whether flag is set */
        if(RESET != hals_rcu_flag_get(RCU_FLAG_LXTALSTB)) {
            ret = SUCCESS;
        } else {
            /* do nothing */
        }
        break;
    case RCU_IRC64M:
        /*--------------------------------- wait IRC64M stable -------------------------*/
        while((RESET == osci_state) && (IRC64M_STARTUP_TIMEOUT != stb_cnt)) {
            osci_state = hals_rcu_flag_get(RCU_FLAG_IRC64MSTB);
            stb_cnt++;
        }

        /* check whether flag is set */
        if(RESET != hals_rcu_flag_get(RCU_FLAG_IRC64MSTB)) {
            ret = SUCCESS;
        } else {
            /* do nothing */
        }
        break;
    case RCU_IRC48M:
        /*--------------------------------- wait IRC48M stable -------------------------*/
        while((RESET == osci_state) && (OSC_STARTUP_TIMEOUT != stb_cnt)) {
            osci_state = hals_rcu_flag_get(RCU_FLAG_IRC48MSTB);
            stb_cnt++;
        }

        /* check whether flag is set */
        if(RESET != hals_rcu_flag_get(RCU_FLAG_IRC48MSTB)) {
            ret = SUCCESS;
        } else {
            /* do nothing */
        }
        break;
    case RCU_IRC32K:
        /*--------------------------------- wait IRC32K stable -------------------------*/
        while((RESET == osci_state) && (OSC_STARTUP_TIMEOUT != stb_cnt)) {
            osci_state = hals_rcu_flag_get(RCU_FLAG_IRC32KSTB);
            stb_cnt++;
        }

        /* check whether flag is set */
        if(RESET != hals_rcu_flag_get(RCU_FLAG_IRC32KSTB)) {
            ret = SUCCESS;
        } else {
            /* do nothing */
        }
        break;
    case RCU_LPIRC4M:
        /*--------------------------------- wait LPIRC4M stable -------------------------*/
        while((RESET == osci_state) && (OSC_STARTUP_TIMEOUT != stb_cnt)) {
            osci_state = hals_rcu_flag_get(RCU_FLAG_LPIRC4MSTB);
            stb_cnt++;
        }

        /* check whether flag is set */
        if(RESET != hals_rcu_flag_get(RCU_FLAG_LPIRC4MSTB)) {
            ret = SUCCESS;
        } else {
            /* do nothing */
        }
        break;
    case RCU_PLL0_CK:
        /*--------------------------------- wait PLL0 stable -------------------------*/
        while((RESET == osci_state) && (OSC_STARTUP_TIMEOUT != stb_cnt)) {
            osci_state = hals_rcu_flag_get(RCU_FLAG_PLL0STB);
            stb_cnt++;
        }

        /* check whether flag is set */
        if(RESET != hals_rcu_flag_get(RCU_FLAG_PLL0STB)) {
            ret = SUCCESS;
        } else {
            /* do nothing */
        }
        break;
    case RCU_PLL1_CK:
        /*--------------------------------- wait PLL1 stable -------------------------*/
        while((RESET == osci_state) && (OSC_STARTUP_TIMEOUT != stb_cnt)) {
            osci_state = hals_rcu_flag_get(RCU_FLAG_PLL1STB);
            stb_cnt++;
        }

        /* check whether flag is set */
        if(RESET != hals_rcu_flag_get(RCU_FLAG_PLL1STB)) {
            ret = SUCCESS;
        } else {
            /* do nothing */
        }
        break;
    case RCU_PLL2_CK:
        /*--------------------------------- wait PLL2 stable -------------------------*/
        while((RESET == osci_state) && (OSC_STARTUP_TIMEOUT != stb_cnt)) {
            osci_state = hals_rcu_flag_get(RCU_FLAG_PLL2STB);
            stb_cnt++;
        }

        /* check whether flag is set */
        if(RESET != hals_rcu_flag_get(RCU_FLAG_PLL2STB)) {
            ret = SUCCESS;
        } else {
            /* do nothing */
        }
        break;
    case RCU_PLLUSBHS0_CK:
        /*--------------------------------- wait PLLUSBHS0 stable -------------------------*/
        while((RESET == osci_state) && (OSC_STARTUP_TIMEOUT != stb_cnt)) {
            osci_state = hals_rcu_flag_get(RCU_FLAG_PLLUSBHS0STB);
            stb_cnt++;
        }

        /* check whether flag is set */
        if(RESET != hals_rcu_flag_get(RCU_FLAG_PLLUSBHS0STB)) {
            ret = SUCCESS;
        } else {
            /* do nothing */
        }
        break;
    case RCU_PLLUSBHS1_CK:
        /*--------------------------------- wait PLLUSBHS1 stable -------------------------*/
        while((RESET == osci_state) && (OSC_STARTUP_TIMEOUT != stb_cnt)) {
            osci_state = hals_rcu_flag_get(RCU_FLAG_PLLUSBHS1STB);
            stb_cnt++;
        }

        /* check whether flag is set */
        if(RESET != hals_rcu_flag_get(RCU_FLAG_PLLUSBHS1STB)) {
            ret = SUCCESS;
        } else {
            /* do nothing */
        }
        break;
    default:
        HAL_DEBUGE("parameter [osci] value is invalid.");
        break;
    }

    /* return value */
    return ret;
}

/*!
    \brief      turn on the oscillator
    \param[in]  osci: oscillator types, refer to hal_rcu_osci_type_enum
                only one parameter can be selected which is shown as below:
      \arg        RCU_HXTAL: HXTAL
      \arg        RCU_LXTAL: LXTAL
      \arg        RCU_IRC64M: IRC64M
      \arg        RCU_IRC48M: IRC48M
      \arg        RCU_IRC32K: IRC32K
      \arg        RCU_LPIRC4M: LPIRC4M
      \arg        RCU_PLL0_CK: PLL0
      \arg        RCU_PLL1_CK: PLL1
      \arg        RCU_PLL2_CK: PLL2
      \arg        RCU_PLLUSBHS0_CK: PLLUSBHS0
      \arg        RCU_PLLUSBHS1_CK: PLLUSBHS1
    \param[out] none
    \retval     none
*/
void hals_rcu_osci_on(hal_rcu_osci_type_enum osci)
{
    RCU_REG_VAL(osci) |= BIT(RCU_BIT_POS(osci));
}

/*!
    \brief      turn off the oscillator
    \param[in]  osci: oscillator types, refer to hal_rcu_osci_type_enum
                only one parameter can be selected which is shown as below:
      \arg        RCU_HXTAL: HXTAL
      \arg        RCU_LXTAL: LXTAL
      \arg        RCU_IRC64M: IRC64M
      \arg        RCU_IRC48M: IRC48M
      \arg        RCU_IRC32K: IRC32K
      \arg        RCU_LPIRC4M: LPIRC4M
      \arg        RCU_PLL0_CK: PLL
      \arg        RCU_PLL1_CK: PLL1
      \arg        RCU_PLL2_CK: PLL2
      \arg        RCU_PLLUSBHS0_CK: PLLUSBHS0
      \arg        RCU_PLLUSBHS1_CK: PLLUSBHS1
    \param[out] none
    \retval     none
*/
void hals_rcu_osci_off(hal_rcu_osci_type_enum osci)
{
    RCU_REG_VAL(osci) &= ~BIT(RCU_BIT_POS(osci));
}

/*!
    \brief      configure the system clock source
    \param[in]  ck_sys: system clock source select
                only one parameter can be selected which is shown as below:
      \arg        RCU_SYSCLK_SRC_CK_IRC64MDIV: select CK_IRC64MDIV as the CK_SYS source
      \arg        RCU_SYSCLK_SRC_CK_HXTAL:   select CK_HXTAL as the CK_SYS source
      \arg        RCU_SYSCLK_SRC_CK_LPIRC4M: select CK_LPIRC4M as the CK_SYS source
      \arg        RCU_SYSCLK_SRC_CK_PLL0P:   select CK_PLL0P as the CK_SYS source
    \param[out] none
    \retval     none
*/
void hals_rcu_system_clock_source_config(hal_rcu_cksys_src_enum ck_sys)
{
    uint32_t cksys_source = 0U;

    /* reset the SCS bits and set according to ck_sys */
    cksys_source = RCU_CFG0;
    cksys_source &= ~RCU_CFG0_SCS;
    RCU_CFG0 = (cksys_source | ck_sys);
}

/*!
    \brief      configure the AHB clock prescaler selection
    \param[in]  ck_ahb: AHB clock prescaler selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_SYSCLK_AHBDIV1: select CK_SYS as CK_AHB
      \arg        RCU_SYSCLK_AHBDIV2: select CK_SYS/2 as CK_AHB
      \arg        RCU_SYSCLK_AHBDIV4: select CK_SYS/4 as CK_AHB
      \arg        RCU_SYSCLK_AHBDIV8: select CK_SYS/8 as CK_AHB
      \arg        RCU_SYSCLK_AHBDIV16: select CK_SYS/16 as CK_AHB
      \arg        RCU_SYSCLK_AHBDIV64: select CK_SYS/64 as CK_AHB
      \arg        RCU_SYSCLK_AHBDIV128: select CK_SYS/128 as CK_AHB
      \arg        RCU_SYSCLK_AHBDIV256: select CK_SYS/256 as CK_AHB
      \arg        RCU_SYSCLK_AHBDIV512: select CK_SYS/512 as CK_AHB
    \param[out] none
    \retval     none
*/
void hals_rcu_ahb_clock_config(hal_rcu_sysclk_ahbdiv_enum ck_ahb)
{
    uint32_t ahbpsc = 0U;

    /* reset the AHBPSC bits and set according to ck_ahb */
    ahbpsc = RCU_CFG0;
    ahbpsc &= ~RCU_CFG0_AHBPSC;
    RCU_CFG0 = (ck_ahb | ahbpsc);
}

/*!
    \brief      configure the APB1 clock prescaler selection
    \param[in]  ck_apb1: APB1 clock prescaler selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_AHBCLK_APB1DIV_1: select CK_AHB as CK_APB1
      \arg        RCU_AHBCLK_APB1DIV_2: select CK_AHB/2 as CK_APB1
      \arg        RCU_AHBCLK_APB1DIV_4: select CK_AHB/4 as CK_APB1
      \arg        RCU_AHBCLK_APB1DIV_8: select CK_AHB/8 as CK_APB1
      \arg        RCU_AHBCLK_APB1DIV_16: select CK_AHB/16 as CK_APB1
    \param[out] none
    \retval     none
*/
void hals_rcu_apb1_clock_config(hal_rcu_ahbclk_apb1div_enum ck_apb1)
{
    uint32_t apb1psc = 0U;

    /* reset the APB1PSC and set according to ck_apb1 */
    apb1psc = RCU_CFG0;
    apb1psc &= ~RCU_CFG0_APB1PSC;
    RCU_CFG0 = ((uint32_t)ck_apb1 | apb1psc);
}

/*!
    \brief      configure the APB2 clock prescaler selection
    \param[in]  ck_apb2: APB2 clock prescaler selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_AHBCLK_APB2DIV_1: select CK_AHB     as CK_APB2
      \arg        RCU_AHBCLK_APB2DIV_2: select CK_AHB/2   as CK_APB2
      \arg        RCU_AHBCLK_APB2DIV_4: select CK_AHB/4   as CK_APB2
      \arg        RCU_AHBCLK_APB2DIV_8: select CK_AHB/8   as CK_APB2
      \arg        RCU_AHBCLK_APB2DIV_16: select CK_AHB/16 as CK_APB2
    \param[out] none
    \retval     none
*/
void hals_rcu_apb2_clock_config(hal_rcu_ahbclk_apb2div_enum ck_apb2)
{
    uint32_t apb2psc = 0U;

    /* reset the APB2PSC and set according to ck_apb2 */
    apb2psc = RCU_CFG0;
    apb2psc &= ~RCU_CFG0_APB2PSC;

    RCU_CFG0 = (ck_apb2 | apb2psc);
}

/*!
    \brief      configure the APB3 clock prescaler selection
    \param[in]  ck_apb3: APB3 clock prescaler selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_AHBCLK_APB3DIV_1: select CK_AHB     as CK_APB3
      \arg        RCU_AHBCLK_APB3DIV_2: select CK_AHB/2   as CK_APB3
      \arg        RCU_AHBCLK_APB3DIV_4: select CK_AHB/4   as CK_APB3
      \arg        RCU_AHBCLK_APB3DIV_8: select CK_AHB/8   as CK_APB3
      \arg        RCU_AHBCLK_APB3DIV_16: select CK_AHB/16 as CK_APB3
    \param[out] none
    \retval     none
*/
void hals_rcu_apb3_clock_config(hal_rcu_ahbclk_apb3div_enum ck_apb3)
{
    uint32_t apb3psc = 0U;

    /* reset the APB3PSC and set according to ck_apb3 */
    apb3psc = RCU_CFG0;
    apb3psc &= ~RCU_CFG0_APB3PSC;
    RCU_CFG0 = ((uint32_t)ck_apb3 | apb3psc);
}

/*!
    \brief      configure the APB4 clock prescaler selection
    \param[in]  ck_apb4: APB4 clock prescaler selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_AHBCLK_APB4DIV_1: select CK_AHB     as CK_APB4
      \arg        RCU_AHBCLK_APB4DIV_2: select CK_AHB/2   as CK_APB4
      \arg        RCU_AHBCLK_APB4DIV_4: select CK_AHB/4   as CK_APB4
      \arg        RCU_AHBCLK_APB4DIV_8: select CK_AHB/8   as CK_APB4
      \arg        RCU_AHBCLK_APB4DIV_16: select CK_AHB/16 as CK_APB4
    \param[out] none
    \retval     none
*/
void hals_rcu_apb4_clock_config(hal_rcu_ahbclk_apb4div_enum ck_apb4)
{
    uint32_t apb4psc = 0U;

    /* reset the APB4PSC and set according to ck_apb4 */
    apb4psc = RCU_CFG0;
    apb4psc &= ~RCU_CFG0_APB4PSC;
    RCU_CFG0 = ((uint32_t)ck_apb4 | apb4psc);
}

/*!
    \brief      get the system clock source
    \param[in]  none
    \param[out] none
    \retval     uint32_t: 0 - 0xFFFFFFFF
*/
uint32_t hals_rcu_system_clock_source_get(void)
{
    return (RCU_CFG0 & RCU_CFG0_SCSS);
}

/*!
    \brief      get the irc64mdiv clock
    \param[in]  none
    \param[out] none
    \retval     uint32_t: 64000000, 32000000, 16000000, 8000000
*/
uint32_t hals_rcu_irc64mdiv_freq_get(void)
{
    uint32_t irc64m_freq = 0U;

    /* CK_IRC64MDIV = CK_IRC64M/1/2/4/8 */
    if(RCU_IRC64M_DIV1 == (RCU_ADDCTL1 & RCU_ADDCTL1_IRC64MDIV)) {
        irc64m_freq = IRC64M_VALUE;
    } else if(RCU_IRC64M_DIV2 == (RCU_ADDCTL1 & RCU_ADDCTL1_IRC64MDIV)) {
        irc64m_freq = IRC64M_VALUE / 2U;
    } else if(RCU_IRC64M_DIV4 == (RCU_ADDCTL1 & RCU_ADDCTL1_IRC64MDIV)) {
        irc64m_freq = IRC64M_VALUE / 4U;
    } else if(RCU_IRC64M_DIV8 == (RCU_ADDCTL1 & RCU_ADDCTL1_IRC64MDIV)) {
        irc64m_freq = IRC64M_VALUE / 8U;
    } else {
        /* do nothing */
    }

    return irc64m_freq;
}

/*!
    \brief      configure the RTC clock source selection
    \param[in]  rtc_clock_source: RTC clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_RTC_CLKSRC_NONE: no clock selected
      \arg        RCU_RTC_CLKSRC_LXTAL: CK_LXTAL selected as RTC source clock
      \arg        RCU_RTC_CLKSRC_IRC32K: CK_IRC32K selected as RTC source clock
      \arg        RCU_RTC_CLKSRC_HXTAL_DIV_RTCDIV: CK_HXTAL / RTCDIV selected as RTC source clock
    \param[in]  rtc_div: RTC clock frequency division
                only one parameter can be selected which is shown as below:
      \arg        RCU_RTC_HXTAL_DIV_NONE: no clock for RTC
      \arg        RCU_RTC_HXTAL_DIVx: RTCDIV clock select CK_HXTAL / x (x = 2,3...63)
    \param[out] none
    \retval     none
*/
void hals_rcu_rtc_clock_config(hal_rcu_rtc_cksrc_enum rtc_clock_source, hal_rcu_rtc_div_enum rtc_div)
{
    uint32_t reg = 0U;

    /* reset the RTCSRC bits and set according to rtc_clock_source */
    reg = RCU_BDCTL;
    reg &= ~RCU_BDCTL_RTCSRC;
    RCU_BDCTL = (reg | (uint32_t)rtc_clock_source);

    /* reset the RTCDIV bits and set according to rtc_div value */
    reg = RCU_CFG0;
    reg &= ~RCU_CFG0_RTCDIV;
    RCU_CFG0 = (reg | (uint32_t)rtc_div);
}

/*!
    \brief      configure the TRNG clock selection
    \param[in]  pll48m_clock_source: PLL48M clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_PLL48M_CKSRC_PLL0Q: CK_PLL0Q selected as PLL48M source clock
      \arg        RCU_PLL48M_CKSRC_PLL2P: CK_PLL2P selected as PLL48M source clock
    \param[in]  ck48m_clock_source: CK48M clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_CK48M_CKSRC_PLL48M: CK_PLL48M selected as CK48M source clock
      \arg        RCU_CK48M_CKSRC_IRC48M: CK_IRC48M selected as CK48M source clock
    \param[out] none
    \retval     none
*/
void hals_rcu_trng_clock_config(hal_rcu_pll48m_cksrc_enum pll48m_clock_source, \
                                hal_rcu_ck48m_cksrc_enum ck48m_clock_source)
{
    uint32_t reg = 0U;

    /* reset the PLL48MSEL bit and set according to pll48m_clock_source */
    reg = RCU_ADDCTL0;
    reg &= ~RCU_ADDCTL0_PLL48MSEL;
    RCU_ADDCTL0 = (reg | pll48m_clock_source);

    /* reset the CK48MSEL bit and set according to ck48m_clock_source */
    reg = RCU_ADDCTL0;
    reg &= ~RCU_ADDCTL0_CK48MSEL;
    RCU_ADDCTL0 = (reg | ck48m_clock_source);
}

/*!
    \brief      configure the TIMER clock multiple selection
    \param[in]  timer_clock_prescaler: TIMER clock selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_TIMER_PSC_MUL2: if APB1PSC / APB2PSC in RCU_CFG0 register is 0b0xx (CK_APBx = CK_AHB)
                                      or 0b100 (CK_APBx = CK_AHB / 2), the TIMER clock is equal to CK_AHB (CK_TIMERx = CK_AHB).
                                      or else, the TIMER clock is twice the corresponding APB clock (TIMER in APB1 domain: CK_TIMERx = 2 x CK_APB1;
                                      TIMER in APB2 domain: CK_TIMERx = 2 x CK_APB2)
      \arg        RCU_TIMER_PSC_MUL4: if APB1PSC / APB2PSC in RCU_CFG0 register is 0b0xx(CK_APBx = CK_AHB),
                                      0b100 (CK_APBx = CK_AHB / 2), or 0b101 (CK_APBx = CK_AHB / 4), the TIMER clock is equal to CK_AHB (CK_TIMERx = CK_AHB).
                                      or else, the TIMER clock is four timers the corresponding APB clock (TIMER in APB1 domain: CK_TIMERx = 4 x CK_APB1;
                                      TIMER in APB2 domain: CK_TIMERx = 4 x CK_APB2)
    \param[out] none
    \retval     none
*/
void hals_rcu_timer_clock_multiple_config(hal_rcu_timer_ckmul_enum timer_clock_prescaler)
{
    /* configure the TIMERSEL bit and select the TIMER clock prescaler */
    if(RCU_TIMER_PSC_MUL2 == (uint32_t)timer_clock_prescaler) {
        RCU_CFG1 &= (uint32_t)timer_clock_prescaler;
    } else {
        RCU_CFG1 |= (uint32_t)timer_clock_prescaler;
    }
}

/*!
    \brief      configure the SPI / I2S clock source selection
    \param[in]  spi_idx: INDEX_SPIx (x = 0,1,2,3,4,5)
    \param[in]  ck_spi: SPI0 / SPI1 / SPI2 clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_SPI_012_CKSRC_CK_PLL0Q: CK_SPI select PLL0Q
      \arg        RCU_SPI_012_CKSRC_CK_PLL1P: CK_SPI select PLL1P
      \arg        RCU_SPI_012_CKSRC_CK_PLL2P: CK_SPI select PLL2P
      \arg        RCU_SPI_012_CKSRC_I2S_CKIN: CK_SPI select I2SCKIN
      \arg        RCU_SPI_012_CKSRC_CK_PER: CK_SPI select PER
    \param[in]  ck_spi: SPI3 / SPI4 / SPI5 clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_SPI_345_CKSRC_CK_APB2: CK_SPI select CKAPB2
      \arg        RCU_SPI_345_CKSRC_CK_PLL1Q: CK_SPI select PLL1Q
      \arg        RCU_SPI_345_CKSRC_CK_PLL2Q: CK_SPI select PLL2Q
      \arg        RCU_SPI_345_CKSRC_CK_IRC64MDIV: CK_SPI select IRC64MDIV
      \arg        RCU_SPI_345_CKSRC_CK_LPIRC4M: CK_SPI select LPIRC4M
      \arg        RCU_SPI_345_CKSRC_CK_HXTAL: CK_SPI select HXTAL
      \arg        RCU_SPI_5_CKSRC_I2S_CKIN: CK_SPI select I2SCKIN (only for SPI5)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_spi_clock_config(hal_spi_idx_enum spi_idx, uint32_t ck_spi)
{
    hal_gpio_init_struct gpio_init_parameter;
    int32_t ret = HAL_ERR_NONE;

    /* initial I2S_CKIN GPIO */
    if(((uint32_t)RCU_SPI_012_CKSRC_I2S_CKIN == ck_spi) || ((uint32_t)RCU_SPI_5_CKSRC_I2S_CKIN == ck_spi)) {
        I2S_CKIN_GPIO_CLK_ENABLE();

        gpio_init_parameter.mode   = GPIO_MODE_AF_PP;
        gpio_init_parameter.pull   = GPIO_PULL_NONE;
        gpio_init_parameter.af     = GPIO_AF_5;
        gpio_init_parameter.ospeed = GPIO_OSPEED_100_220MHZ;
        hal_gpio_init(I2S_CKIN_GPIO_PORT, I2S_CKIN_GPIO_PIN, &gpio_init_parameter);
    } else {
        /* do nothing */
    }

    switch(spi_idx) {
    case INDEX_SPI0:
        /* reset the SPI0SEL bits and set according to ck_spi */
        RCU_CFG5 &= ~RCU_CFG5_SPI0SEL;
        RCU_CFG5 |= (uint32_t)ck_spi;
        break;
    case INDEX_SPI1:
        /* reset the SPI1SEL bits and set according to ck_spi */
        RCU_CFG5 &= ~RCU_CFG5_SPI1SEL;
        RCU_CFG5 |= (uint32_t)ck_spi << 4U;
        break;
    case INDEX_SPI2:
        /* reset the SPI2SEL bits and set according to ck_spi */
        RCU_CFG5 &= ~RCU_CFG5_SPI2SEL;
        RCU_CFG5 |= (uint32_t)ck_spi << 8U;
        break;
    case INDEX_SPI3:
        /* reset the SPI3SEL bits and set according to ck_spi */
        RCU_CFG5 &= ~RCU_CFG5_SPI3SEL;
        RCU_CFG5 |= (uint32_t)ck_spi;
        break;
    case INDEX_SPI4:
        /* reset the SPI4SEL bits and set according to ck_spi */
        RCU_CFG5 &= ~RCU_CFG5_SPI4SEL;
        RCU_CFG5 |= (uint32_t)ck_spi << 4U;
        break;
    case INDEX_SPI5:
        /* reset the SPI5SEL bits and set according to ck_spi */
        RCU_CFG5 &= ~RCU_CFG5_SPI5SEL;
        RCU_CFG5 |= (uint32_t)ck_spi;
        break;
    default:
        HAL_DEBUGE("parameter [spi_idx] value is invalid.");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      configure the SDIO clock source selection
    \param[in]  sdio_idx: INDEX_SDIOx (x = 0,1)
    \param[in]  ck_sdio: SDIO clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_SDIO0_CKSRC_CK_PLL0Q: CK_SDIO0 select PLL0Q (only for SDIO0)
      \arg        RCU_SDIO0_CKSRC_CK_PLL1R: CK_SDIO0 select PLL1R (only for SDIO0)
      \arg        RCU_SDIO1_CKSRC_CK_PLL0Q: CK_SDIO1 select PLL0Q (only for SDIO1)
      \arg        RCU_SDIO1_CKSRC_CK_PLL1R: CK_SDIO1 select PLL1R (only for SDIO1)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_sdio_clock_config(hal_sdio_idx_enum sdio_idx, hal_rcu_sdio_cksrc_enum ck_sdio)
{
    int32_t ret = HAL_ERR_NONE;

    switch(sdio_idx) {
    case INDEX_SDIO0:
        /* reset the SDIO0SEL bits and set according to ck_sdio */
        RCU_CFG4 &= ~RCU_CFG4_SDIO0SEL;
        RCU_CFG4 |= (uint32_t)ck_sdio;
        break;
    case INDEX_SDIO1:
        /* reset the SDIO1SEL bits and set according to ck_sdio */
        RCU_CFG3 &= ~RCU_CFG3_SDIO1SEL;
        RCU_CFG3 |= (uint32_t)ck_sdio;
        break;
    default:
        HAL_DEBUGE("parameter [sdio_idx] value is invalid.");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      configure the USARTx (x = 0,1,2,5) clock source selection
    \param[in]  usart_idx: INDEX_USARTx (x = 0,1,2,5)
    \param[in]  ck_usart: USART clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_USART_CKSRC_CK_APB: CK_USART select CK_APB1(USART1 / USART2) or CK_APB2(USART0 / USART5)
      \arg        RCU_USART_CKSRC_CK_AHB: CK_USART select CK_AHB
      \arg        RCU_USART_CKSRC_CK_LXTAL: CK_USART select CK_LXTAL
      \arg        RCU_USART_CKSRC_CK_IRC64MDIV: CK_USART select CK_IRC64MDIV
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_usart_clock_config(hal_usart_idx_enum usart_idx, hal_rcu_usart_cksrc_enum ck_usart)
{
    int32_t ret = HAL_ERR_NONE;

    switch(usart_idx) {
    case INDEX_USART0:
        /* reset the USART0SEL bits and set according to ck_usart */
        RCU_CFG1 &= ~RCU_CFG1_USART0SEL;
        RCU_CFG1 |= ck_usart;
        break;
    case INDEX_USART1:
        /* reset the USART1SEL bits and set according to ck_usart */
        RCU_CFG1 &= ~RCU_CFG1_USART1SEL;
        RCU_CFG1 |= (uint32_t)ck_usart << 18U;
        break;
    case INDEX_USART2:
        /* reset the USART2SEL bits and set according to ck_usart */
        RCU_CFG1 &= ~RCU_CFG1_USART2SEL;
        RCU_CFG1 |= (uint32_t)ck_usart << 20U;
        break;
    case INDEX_USART5:
        /* reset the USART5SEL bits and set according to ck_usart */
        RCU_CFG1 &= ~RCU_CFG1_USART5SEL;
        RCU_CFG1 |= (uint32_t)ck_usart << 22U;
        break;
    default:
        HAL_DEBUGE("parameter [usart_idx] value is invalid.");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      configure the I2Cx (x = 0,1,2,3) clock source selection
    \param[in]  i2c_idx: INDEX_I2Cx (x = 0,1,2,3)
    \param[in]  ck_i2c: I2C clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_I2C_CKSRC_CK_APB1: CK_I2C select CK_APB1
      \arg        RCU_I2C_CKSRC_CK_PLL2R: CK_I2C select CK_PLL2R
      \arg        RCU_I2C_CKSRC_CK_IRC64MDIV: CK_I2C select CK_IRC64MDIV
      \arg        RCU_I2C_CKSRC_CK_LPIRC4M: CK_I2C select CK_LPIRC4M
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_i2c_clock_config(hal_i2c_idx_enum i2c_idx, hal_rcu_i2c_cksrc_enum ck_i2c)
{
    int32_t ret = HAL_ERR_NONE;

    switch(i2c_idx) {
    case INDEX_I2C0:
        /* reset the I2C0SEL bits and set according to ck_i2c */
        RCU_CFG0 &= ~RCU_CFG0_I2C0SEL;
        RCU_CFG0 |= (uint32_t)ck_i2c << 30U;
        break;
    case INDEX_I2C1:
        /* reset the I2C1SEL bits and set according to ck_i2c */
        RCU_CFG3 &= ~RCU_CFG3_I2C1SEL;
        RCU_CFG3 |= (uint32_t)ck_i2c;
        break;
    case INDEX_I2C2:
        /* reset the I2C2SEL bits and set according to ck_i2c */
        RCU_CFG3 &= ~RCU_CFG3_I2C2SEL;
        RCU_CFG3 |= (uint32_t)ck_i2c << 2U;
        break;
    case INDEX_I2C3:
        /* reset the I2C4SEL bits and set according to ck_i2c */
        RCU_CFG3 &= ~RCU_CFG3_I2C3SEL;
        RCU_CFG3 |= (uint32_t)ck_i2c << 4U;
        break;
    default:
        HAL_DEBUGE("parameter [i2c_idx] value is invalid.");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      configure the CANx (x = 0,1,2) clock source selection
    \param[in]  can_idx: INDEX_CANx (x = 0,1,2)
    \param[in]  ck_can: CAN clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_CAN_CKSRC_CK_HXTAL: CK_CAN select CK_HXTAL
      \arg        RCU_CAN_CKSRC_CK_APB: CK_CAN select CK_APB2
      \arg        RCU_CAN_CKSRC_CK_APB2DIV2: CK_CAN select CK_APB2 / 2
      \arg        RCU_CAN_CKSRC_CK_IRC64MDIV: CK_CAN select CK_IRC64MDIV
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_can_clock_config(hal_can_idx_enum can_idx, hal_rcu_can_cksrc_enum ck_can)
{
    int32_t ret = HAL_ERR_NONE;

    switch(can_idx) {
    case INDEX_CAN0:
        /* reset the CAN0SEL bits and set according to ck_can */
        RCU_CFG1 &= ~RCU_CFG1_CAN0SEL;
        RCU_CFG1 |= (uint32_t)ck_can;
        break;
    case INDEX_CAN1:
        /* reset the CAN1SEL bits and set according to ck_can */
        RCU_CFG1 &= ~RCU_CFG1_CAN1SEL;
        RCU_CFG1 |= (uint32_t)ck_can << 2U;
        break;
    case INDEX_CAN2:
        /* reset the CAN2SEL bits and set according to ck_can */
        RCU_CFG1 &= ~RCU_CFG1_CAN2SEL;
        RCU_CFG1 |= (uint32_t)ck_can << 4U;
        break;
    default:
        HAL_DEBUGE("parameter [can_idx] value is invalid.");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      configure the ADCx (x = 0,1,2) clock source selection
    \param[in]  adc_idx: INDEX_ADCx (x = 0,1,2)
    \param[in]  ck_adc: ADC clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_ADC_CKSRC_CK_PLL1P: CK_ADC select CK_PLL1P
      \arg        RCU_ADC_CKSRC_CK_PLL2R: CK_ADC select CK_PLL2R
      \arg        RCU_ADC_CKSRC_CK_PER: CK_ADC select CK_PER
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_adc_clock_config(hal_adc_idx_enum adc_idx, hal_rcu_adc_cksrc_enum ck_adc)
{
    int32_t ret = HAL_ERR_NONE;

    switch(adc_idx) {
    case INDEX_ADC01:
        /* reset the ADC0SEL/ADC1SEL bits and set according to ck_adc */
        RCU_CFG3 &= ~RCU_CFG3_ADC01SEL;
        RCU_CFG3 |= (uint32_t)ck_adc;
        break;
    case INDEX_ADC2:
        /* reset the ADC2SEL bits and set according to ck_adc */
        RCU_CFG3 &= ~RCU_CFG3_ADC2SEL;
        RCU_CFG3 |= (uint32_t)ck_adc << 2U;
        break;
    default:
        HAL_DEBUGE("parameter [adc_idx] value is invalid.");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      configure the SAIx (x = 0,1) clock source selection
    \param[in]  sai_idx: INDEX_SAIx (x = 0,1)
    \param[in]  ck_sai: SAI clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_SAI_CKSRC_CK_PLL0Q: CK_SAI select CK_PLL0Q
      \arg        RCU_SAI_CKSRC_CK_PLL1P: CK_SAI select CK_PLL2R
      \arg        RCU_SAI_CKSRC_CK_PLL2P: CK_SAI select CK_PLL2P
      \arg        RCU_SAI_CKSRC_I2S_CKIN: CK_SAI select CK_PLL2R
      \arg        RCU_SAI_CKSRC_CK_PER: CK_SAI select CK_PER
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_sai_clock_config(hal_sai_idx_enum sai_idx, hal_rcu_sai_cksrc_enum ck_sai)
{
    hal_gpio_init_struct gpio_init_parameter;
    int32_t ret = HAL_ERR_NONE;

    /* initial I2S_CKIN GPIO */
    if(RCU_SAI_CKSRC_I2S_CKIN == ck_sai) {
        I2S_CKIN_GPIO_CLK_ENABLE();

        gpio_init_parameter.mode   = GPIO_MODE_AF_PP;
        gpio_init_parameter.pull   = GPIO_PULL_NONE;
        gpio_init_parameter.af     = GPIO_AF_5;
        gpio_init_parameter.ospeed = GPIO_OSPEED_100_220MHZ;
        hal_gpio_init(I2S_CKIN_GPIO_PORT, I2S_CKIN_GPIO_PIN, &gpio_init_parameter);
    } else {
        /* do nothing */
    }

    switch(sai_idx) {
    case INDEX_SAI0:
        /* reset the SAI0SEL bits and set according to ck_sai */
        RCU_CFG2 &= ~RCU_CFG2_SAI0SEL;
        RCU_CFG2 |= (uint32_t)ck_sai;
        break;
    case INDEX_SAI1:
        /* reset the SAI1SEL bits and set according to ck_sai */
        RCU_CFG2 &= ~RCU_CFG2_SAI1SEL;
        RCU_CFG2 |= (uint32_t)ck_sai << 4U;
        break;
    default:
        HAL_DEBUGE("parameter [sai_idx] value is invalid.");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      configure the SAI2Bx (x = 0,1) clock source selection
    \param[in]  sai2b_idx: INDEX_SAI2Bx (x = 0,1)
    \param[in]  ck_sai2b: SAI clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_SAI2B_CKSRC_CK_PLL0Q: CK_SAI2B select CK_PLL0Q
      \arg        RCU_SAI2B_CKSRC_CK_PLL1P: CK_SAI2B select CK_PLL2R
      \arg        RCU_SAI2B_CKSRC_CK_PLL2P: CK_SAI2B select CK_PLL2P
      \arg        RCU_SAI2B_CKSRC_I2S_CKIN: CK_SAI2B select I2S_CKIN
      \arg        RCU_SAI2B_CKSRC_CK_PER: CK_SAI2B select CK_PER
      \arg        RCU_SAI2B_CKSRC_CK_RSPDIF_SYMB: CK_SAI2B select CK_RSPDIF_SYMB
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_sai2b_clock_config(hal_sai2b_idx_enum sai2b_idx, hal_rcu_sai2b_cksrc_enum ck_sai2b)
{
    hal_gpio_init_struct gpio_init_parameter;
    int32_t ret = HAL_ERR_NONE;

    /* initial I2S_CKIN GPIO */
    if(RCU_SAI2B_CKSRC_I2S_CKIN == ck_sai2b) {
        I2S_CKIN_GPIO_CLK_ENABLE();

        gpio_init_parameter.mode   = GPIO_MODE_AF_PP;
        gpio_init_parameter.pull   = GPIO_PULL_NONE;
        gpio_init_parameter.af     = GPIO_AF_5;
        gpio_init_parameter.ospeed = GPIO_OSPEED_100_220MHZ;
        hal_gpio_init(I2S_CKIN_GPIO_PORT, I2S_CKIN_GPIO_PIN, &gpio_init_parameter);
    } else {
        /* do nothing */
    }

    switch(sai2b_idx) {
    case INDEX_SAI2B0:
        /* reset the SAI2B0SEL bits and set according to ck_sai2b */
        RCU_CFG2 &= ~RCU_CFG2_SAI2B0SEL;
        RCU_CFG2 |= (uint32_t)ck_sai2b;
        break;
    case INDEX_SAI2B1:
        /* reset the SAI2B1SEL bits and set according to ck_sai2b */
        RCU_CFG2 &= ~RCU_CFG2_SAI2B1SEL;
        RCU_CFG2 |= (uint32_t)ck_sai2b << 4U;
        break;
    default:
        HAL_DEBUGE("parameter [sai2b_idx] value is invalid.");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      configure the RSPDIF clock source selection
    \param[in]  ck_rspdif: RSPDIF clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_RSPDIF_CKSRC_CK_PLL0Q: CK_RSPDIF select CK_PLL0Q
      \arg        RCU_RSPDIF_CKSRC_CK_PLL1R: CK_RSPDIF select CK_PLL1R
      \arg        RCU_RSPDIF_CKSRC_CK_PLL2R: CK_RSPDIF select CK_PLL2R
      \arg        RCU_RSPDIF_CKSRC_CK_IRC64MDIV: CK_RSPDIF select CK_IRC64MDIV
    \param[out] none
    \retval     none
*/
void hals_rcu_rspdif_clock_config(hal_rcu_rspdif_cksrc_enum ck_rspdif)
{
    /* reset the RSPDIFSEL bits and set according to ck_rspdif */
    RCU_CFG1 &= ~RCU_CFG1_RSPDIFSEL;
    RCU_CFG1 |= ck_rspdif;
}

/*!
    \brief      configure the EXMC clock source selection
    \param[in]  ck_exmc: EXMC clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_EXMC_CKSRC_CK_AHB: CK_EXMC select CK_AHB
      \arg        RCU_EXMC_CKSRC_CK_PLL0Q: CK_EXMC select CK_PLL0Q
      \arg        RCU_EXMC_CKSRC_CK_PLL1R: CK_EXMC select CK_PLL1R
      \arg        RCU_EXMC_CKSRC_CK_PER: CK_EXMC select CK_PER
    \param[out] none
    \retval     none
*/
void hals_rcu_exmc_clock_config(hal_rcu_exmc_cksrc_enum ck_exmc)
{
    /* reset the EXMCSEL bits and set according to ck_exmc */
    RCU_CFG4 &= ~(RCU_CFG4_EXMCSEL);
    RCU_CFG4 |= (uint32_t)ck_exmc;
}

/*!
    \brief      configure the HPDF clock source selection
    \param[in]  ck_hpdf: HPDF clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_HPDF_CKSRC_CK_APB2: CK_HPDF select CK_APB2
      \arg        RCU_HPDF_CKSRC_CK_AHB: CK_HPDF select CK_AHB
    \param[out] none
    \retval     none
*/
void hals_rcu_hpdf_clock_config(hal_rcu_hpdf_cksrc_enum ck_hpdf)
{
    /* reset the HPDFSEL bits and set according to ck_hpdf */
    RCU_CFG1 &= ~RCU_CFG1_HPDFSEL;
    RCU_CFG1 |= (uint32_t)ck_hpdf;
}

/*!
    \brief      configure the PER clock source selection
    \param[in]  ck_per: PER clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_PRECLK_CKSRC_CK_IRC64MDIV: CK_PER select CK_IRC64MDIV
      \arg        RCU_PRECLK_CKSRC_CK_LPIRC4M: CK_PER select CK_LPIRC4M
      \arg        RCU_PRECLK_CKSRC_CK_HXTAL: CK_PER select CK_HXTAL
    \param[out] none
    \retval     none
*/
void hals_rcu_per_clock_config(hal_rcu_pef_cksrc_enum ck_per)
{
    /* reset the PERSEL bits and set according to ck_per */
    RCU_CFG1 &= ~RCU_CFG1_PERSEL;
    RCU_CFG1 |= ck_per;
}

/*!
    \brief      configure the usbhs divided
    \param[in]  usbhs_idx: INDEX_USBHSx (x = 0,1)
    \param[in]  ck_usbhsdiv: USBHS clock divided
                only one parameter can be selected which is shown as below:
      \arg        RCU_USBHSx_DIV_2 : Div frequency by USBHS/2
      \arg        RCU_USBHSx_DIV_4 : Div frequency by USBHS/4
      \arg        RCU_USBHSx_DIV_6 : Div frequency by USBHS/6
      \arg        RCU_USBHSx_DIV_8 : Div frequency by USBHS/8
      \arg        RCU_USBHSx_DIV_10: Div frequency by USBHS/10
      \arg        RCU_USBHSx_DIV_12: Div frequency by USBHS/12
      \arg        RCU_USBHSx_DIV_14: Div frequency by USBHS/14
      \arg        RCU_USBHSx_DIV_16: Div frequency by USBHS/16
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_usbhs_div_config(hal_usbhs_idx_enum usbhs_idx, hal_rcu_usbhsx_div_enum ck_usbhsdiv)
{
    int32_t ret = HAL_ERR_NONE;

    switch(usbhs_idx) {
    case INDEX_USBHS0:
        /* reset the USBHS0DV bits and set according to ck_usbhsdiv */
        RCU_PLLUSBCFG &= ~RCU_PLLUSBCFG_USBHS0DV;
        RCU_PLLUSBCFG |= ck_usbhsdiv;
        break;
    case INDEX_USBHS1:
        /* reset the USBHS1DV bits and set according to ck_usbhsdiv */
        RCU_PLLUSBCFG &= ~RCU_PLLUSBCFG_USBHS1DV;
        RCU_PLLUSBCFG |= (uint32_t)((uint32_t)ck_usbhsdiv << 16U);
        break;
    default:
        HAL_DEBUGE("parameter [usbhs_idx] value is invalid.");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      configure the USBHS prescaler
    \param[in]  usbhs_idx: INDEX_USBHSx (x = 0,1)
    \param[in]  ck_usbhspsc: USBHS clock prescaler from CK_PLL1Q
                only one parameter can be selected which is shown as below:
      \arg        RCU_USBHSx_PSC_1: CK_PLL1Q / 1
      \arg        RCU_USBHSx_PSC_2: CK_PLL1Q / 2
      \arg        RCU_USBHSx_PSC_3: CK_PLL1Q / 3
      \arg        RCU_USBHSx_PSC_4: CK_PLL1Q / 4
      \arg        RCU_USBHSx_PSC_5: CK_PLL1Q / 5
      \arg        RCU_USBHSx_PSC_6: CK_PLL1Q / 6
      \arg        RCU_USBHSx_PSC_7: CK_PLL1Q / 7
      \arg        RCU_USBHSx_PSC_8: CK_PLL1Q / 8
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_usbhs_psc_config(hal_usbhs_idx_enum usbhs_idx, hal_rcu_usbhsx_psc_enum ck_usbhspsc)
{
    int32_t ret = HAL_ERR_NONE;

    switch(usbhs_idx) {
    case INDEX_USBHS0:
        /* reset the USBHS0PSC bits and set according to ck_usbhspsc */
        RCU_USBCLKCTL &= ~RCU_USBCLKCTL_USBHS0PSC;
        RCU_USBCLKCTL |= (uint32_t)ck_usbhspsc;
        break;
    case INDEX_USBHS1:
        /* reset the USBHS1PSC bits and set according to ck_usbhspsc */
        RCU_USBCLKCTL &= ~RCU_USBCLKCTL_USBHS1PSC;
        RCU_USBCLKCTL |= (uint32_t)((uint32_t)ck_usbhspsc << 3U);
        break;
    default:
        HAL_DEBUGE("parameter [usbhs_idx] value is invalid.");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      configure the USBHS48M clock source selection
    \param[in]  usbhs_idx: INDEX_USBHSx (x = 0,1)
    \param[in]  ck_usb48m: USBHS48MSEL clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_USBHSx48M_CKSRC_PLL0R: CK_USB48M select CK_PLL0R
      \arg        RCU_USBHSx48M_CKSRC_USBHSDV: CK_USB48M select CK_PLLUSBHSx / USBHxSDV
      \arg        RCU_USBHSx48M_CKSRC_PLL1Q: CK_USB48M select CK_PLL1Q / USBHSxPSC
      \arg        RCU_USBHSx48M_CKSRC_IRC48M: CK_USB48M select CK_IRC48M
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_usb48m_clock_config(hal_usbhs_idx_enum usbhs_idx, hal_rcu_usbhsx48m_cksrc_enum ck_usb48m)
{
    int32_t ret = HAL_ERR_NONE;

    switch(usbhs_idx) {
    case INDEX_USBHS0:
        /* reset the USB048MSEL bits and set according to ck_usb48m */
        RCU_USBCLKCTL &= ~RCU_USBCLKCTL_USBHS048MSEL;
        RCU_USBCLKCTL |= ck_usb48m;
        break;
    case INDEX_USBHS1:
        /* reset the USB148MSEL bits and set according to ck_usb48m */
        RCU_USBCLKCTL &= ~RCU_USBCLKCTL_USBHS148MSEL;
        RCU_USBCLKCTL |= (uint32_t)((uint32_t)ck_usb48m << 8U);
        break;
    default:
        HAL_DEBUGE("parameter [usbhs_idx] value is invalid.");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      configure the USBHSSEL clock source selection
    \param[in]  usbhs_idx: INDEX_USBHSx (x = 0,1)
    \param[in]  ck_usbhs: USBHSSEL clock source selection
                only one parameter can be selected which is shown as below:
      \arg        RCU_USBHSx_CKSRC_CK48M: CK_USBHS select 48M
      \arg        RCU_USBHSx_CKSRC_CK60M: CK_USBHS select 60M
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_usbhs_clock_config(hal_usbhs_idx_enum usbhs_idx, hal_rcu_usbhsx_cksrc_enum ck_usbhs)
{
    int32_t ret = HAL_ERR_NONE;

    switch(usbhs_idx) {
    case INDEX_USBHS0:
        /* reset the USBHS0SEL bits and set according to ck_usbhs */
        RCU_USBCLKCTL &= ~RCU_USBCLKCTL_USBHS0SEL;
        RCU_USBCLKCTL |= ck_usbhs;
        break;
    case INDEX_USBHS1:
        /* reset the USBHS1SEL bits and set according to ck_usbhs */
        RCU_USBCLKCTL &= ~RCU_USBCLKCTL_USBHS1SEL;
        RCU_USBCLKCTL |= (uint32_t)((uint32_t)ck_usbhs << 8U);
        break;
    default:
        HAL_DEBUGE("parameter [usbhs_idx] value is invalid.");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      config the USBHS clock source selection
    \param[in]  usbhs_idx: INDEX_USBHSx (x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_usbhs_clock_selection_config(hal_usbhs_idx_enum usbhs_idx)
{
    int32_t ret = HAL_ERR_NONE;

    switch(usbhs_idx) {
    case INDEX_USBHS0:
        /* set the USB0SWEN bit */
        RCU_USBCLKCTL |= RCU_USBCLKCTL_USBHS0SWEN;
        break;
    case INDEX_USBHS1:
        /* set the USB1SWEN bit */
        RCU_USBCLKCTL |= RCU_USBCLKCTL_USBHS1SWEN;
        break;
    default:
        HAL_DEBUGE("parameter [usbhs_idx] value is invalid.");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      configure the LXTAL drive capability
    \param[in]  lxtal_dricap: drive capability of LXTAL
                only one parameter can be selected which is shown as below:
      \arg        RCU_LXTAL_DRIVE_LOWDRI: lower driving capability
      \arg        RCU_LXTAL_DRIVE_MED_LOWDRI: middle driving capability
      \arg        RCU_LXTAL_DRIVE_MED_HIGHDRI: higher driving capability
      \arg        RCU_LXTAL_DRIVE_HIGHDRI: stronger driving capability
    \param[out] none
    \retval     none
*/
void hals_rcu_lxtal_drive_capability_config(hal_rcu_lxtal_drive_capability_enum lxtal_dricap)
{
    uint32_t reg = 0U;

    /* reset the LXTALDRI bits and set according to lxtal_dricap */
    reg = RCU_BDCTL;
    reg &= ~RCU_BDCTL_LXTALDRI;
    RCU_BDCTL = (reg | lxtal_dricap);
}

/*!
    \brief      reset the BKP
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_rcu_bkp_reset_enable(void)
{
    RCU_BDCTL |= RCU_BDCTL_BKPRST;
}

/*!
    \brief      disable the BKP reset
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_rcu_bkp_reset_disable(void)
{
    RCU_BDCTL &= ~RCU_BDCTL_BKPRST;
}

/*!
    \brief      enable the oscillator bypass mode, HXTALEN or LXTALEN must be reset before it
    \param[in]  osci: oscillator types, refer to hal_rcu_osci_type_enum
                only one parameter can be selected which is shown as below:
      \arg        RCU_HXTAL: high speed crystal oscillator (HXTAL)
      \arg        RCU_LXTAL: low speed crystal oscillator (LXTAL)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_osci_bypass_mode_enable(hal_rcu_osci_type_enum osci)
{
    uint32_t reg = 0U;
    int32_t ret = HAL_ERR_NONE;

    switch(osci) {
    /* enable HXTAL to bypass mode */
    case RCU_HXTAL:
        reg = RCU_CTL;
        RCU_CTL &= ~RCU_CTL_HXTALEN;
        RCU_CTL = (reg | RCU_CTL_HXTALBPS);
        break;
    /* enable LXTAL to bypass mode */
    case RCU_LXTAL:
        reg = RCU_BDCTL;
        RCU_BDCTL &= ~RCU_BDCTL_LXTALEN;
        RCU_BDCTL = (reg | RCU_BDCTL_LXTALBPS);
        break;
    default:
        HAL_DEBUGE("parameter [osci] value invalid.");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      disable the oscillator bypass mode, HXTALEN or LXTALEN must be reset before it
    \param[in]  osci: oscillator types, refer to hal_rcu_osci_type_enum
                only one parameter can be selected which is shown as below:
      \arg        RCU_HXTAL: high speed crystal oscillator (HXTAL)
      \arg        RCU_LXTAL: low speed crystal oscillator (LXTAL)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_osci_bypass_mode_disable(hal_rcu_osci_type_enum osci)
{
    uint32_t reg = 0U;
    int32_t ret = HAL_ERR_NONE;

    switch(osci) {
    /* disable HXTAL to bypass mode */
    case RCU_HXTAL:
        reg = RCU_CTL;
        RCU_CTL &= ~RCU_CTL_HXTALEN;
        RCU_CTL = (reg & ~RCU_CTL_HXTALBPS);
        break;
    /* disable LXTAL to bypass mode */
    case RCU_LXTAL:
        reg = RCU_BDCTL;
        RCU_BDCTL &= ~RCU_BDCTL_LXTALEN;
        RCU_BDCTL = (reg & ~RCU_BDCTL_LXTALBPS);
        break;
    default:
        HAL_DEBUGE("parameter [osci] value is invalid.");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      set the IRC64M adjust value
    \param[in]  irc64m_adjval: IRC64M adjust value, must be between 0 and 0x7F
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_irc64m_adjust_value_set(uint32_t irc64m_adjval)
{
    uint32_t reg = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(0x7FU < irc64m_adjval) {
        HAL_DEBUGE("parameter [irc64m_adjval] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* reset the IRC64MADJ bits and set according to irc64m_adjval */
    reg = RCU_CTL;
    reg &= ~RCU_CTL_IRC64MADJ;
    RCU_CTL = (reg | ((irc64m_adjval & RCU_IRC64M_ADJUST_MASK)));

    return HAL_ERR_NONE;
}

/*!
    \brief      set the LPIRC4M adjust value
    \param[in]  lpirc4m_adjval: LPIRC4M adjust value, must be between 0 and 0x3F
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_rcu_lpirc4m_adjust_value_set(uint32_t lpirc4m_adjval)
{
    uint32_t reg = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(0x3FU < lpirc4m_adjval) {
        HAL_DEBUGE("parameter [lpirc4m_adjval] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* reset the LPIRC4MTRIM bits and set according to lpirc4M_adjval */
    reg = RCU_ADDCTL1;
    reg &= ~RCU_ADDCTL1_LPIRC4MTRIM;
    RCU_ADDCTL1 = (reg | ((lpirc4m_adjval & RCU_LPIRC4M_ADJUST_MASK) << RCU_LPIRC4M_ADJUST_OFFSET));

    return HAL_ERR_NONE;
}

/*!
    \brief      get the clock stabilization and peripheralreset flags
    \param[in]  flag: the clock stabilization and peripheralreset flags, refer to hal_rcu_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        RCU_FLAG_IRC64MSTB: IRC64M stabilization flag
      \arg        RCU_FLAG_HXTALSTB: HXTAL stabilization flag
      \arg        RCU_FLAG_PLL0STB: PLL0 stabilization flag
      \arg        RCU_FLAG_PLL1STB: PLL1 stabilization flag
      \arg        RCU_FLAG_PLL2STB: PLL2 stabilization flag
      \arg        RCU_FLAG_LXTALSTB: LXTAL stabilization flag
      \arg        RCU_FLAG_IRC32KSTB: IRC32K stabilization flag
      \arg        RCU_FLAG_IRC48MSTB: IRC48M stabilization flag
      \arg        RCU_FLAG_LPIRC4MSTB: LPIRC4M stabilization flag
      \arg        RCU_FLAG_PLLUSBHS0STB: PLLUSBHS0 stabilization flag
      \arg        RCU_FLAG_PLLUSBHS1STB: PLLUSBHS1 stabilization flag
      \arg        RCU_FLAG_LCKMD: LXTAL clock failure detection flags
      \arg        RCU_FLAG_BORRST: BOR reset flags
      \arg        RCU_FLAG_EPRST: external PIN reset flag
      \arg        RCU_FLAG_PORRST: Power reset flag
      \arg        RCU_FLAG_SWRST: software reset flag
      \arg        RCU_FLAG_FWDGTRST: free watchdog timer reset flag
      \arg        RCU_FLAG_WWDGTRST: window watchdog timer reset flag
      \arg        RCU_FLAG_LPRST: low-power reset flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_rcu_flag_get(hal_rcu_flag_enum flag)
{
    FlagStatus ret = RESET;

    /* get the RCU flag */
    if(RESET != (RCU_REG_VAL(flag) & BIT(RCU_BIT_POS(flag)))) {
        ret = SET;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief    clear all the reset flag
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_rcu_all_reset_flag_clear(void)
{
    RCU_RSTSCK |= RCU_RSTSCK_RSTFC;
}

/*!
    \brief      enable the stabilization interrupt
    \param[in]  interrupt: clock stabilization interrupt, refer to hal_rcu_int_enum
                only one parameter can be selected which is shown as below:
      \arg        RCU_INT_IRC32KSTB: IRC32K stabilization interrupt enable
      \arg        RCU_INT_LXTALSTB: LXTAL stabilization interrupt enable
      \arg        RCU_INT_IRC64MSTB: IRC64M stabilization interrupt enable
      \arg        RCU_INT_HXTALSTB: HXTAL stabilization interrupt enable
      \arg        RCU_INT_PLL0STB: PLL0 stabilization interrupt enable
      \arg        RCU_INT_PLL1STB: PLL1 stabilization interrupt enable
      \arg        RCU_INT_PLL2STB: PLL2 stabilization interrupt enable
      \arg        RCU_INT_IRC48MSTB: IRC48M stabilization interrupt enable
      \arg        RCU_INT_LPIRC4MSTB: LPIRC4M stabilization interrupt enable
      \arg        RCU_INT_PLLUSBHS0STB: PLLUSBHS0 stabilization interrupt enable
      \arg        RCU_INT_PLLUSBHS1STB: PLLUSBHS1 stabilization interrupt enable
    \param[out] none
    \retval     none
*/
void hals_rcu_interrupt_enable(hal_rcu_int_enum interrupt)
{
    RCU_REG_VAL(interrupt) |= BIT(RCU_BIT_POS(interrupt));
}

/*!
    \brief      disable the stabilization interrupt
    \param[in]  interrupt: clock stabilization interrupt, refer to hal_rcu_int_enum
                only one parameter can be selected which is shown as below:
      \arg        RCU_INT_IRC32KSTB: IRC32K stabilization interrupt disable
      \arg        RCU_INT_LXTALSTB: LXTAL stabilization interrupt disable
      \arg        RCU_INT_IRC64MSTB: IRC64M stabilization interrupt disable
      \arg        RCU_INT_HXTALSTB: HXTAL stabilization interrupt disable
      \arg        RCU_INT_PLL0STB: PLL0 stabilization interrupt disable
      \arg        RCU_INT_PLL1STB: PLL1 stabilization interrupt disable
      \arg        RCU_INT_PLL2STB: PLL2 stabilization interrupt disable
      \arg        RCU_INT_IRC48MSTB: IRC48M stabilization interrupt disable
      \arg        RCU_INT_LPIRC4MSTB: LPIRC4M stabilization interrupt disable
      \arg        RCU_INT_PLLUSBHS0STB: PLLUSBHS0 stabilization interrupt disable
      \arg        RCU_INT_PLLUSBHS1STB: PLLUSBHS1 stabilization interrupt disable
    \param[out] none
    \retval     none
*/
void hals_rcu_interrupt_disable(hal_rcu_int_enum interrupt)
{
    RCU_REG_VAL(interrupt) &= ~BIT(RCU_BIT_POS(interrupt));
}

/*!
    \brief      get the clock stabilization interrupt and ckm flags
    \param[in]  int_flag: interrupt and ckm flags, refer to hal_rcu_int_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        RCU_INT_FLAG_IRC32KSTB: IRC32K stabilization interrupt flag
      \arg        RCU_INT_FLAG_LXTALSTB: LXTAL stabilization interrupt flag
      \arg        RCU_INT_FLAG_IRC64MSTB: IRC64M stabilization interrupt flag
      \arg        RCU_INT_FLAG_HXTALSTB: HXTAL stabilization interrupt flag
      \arg        RCU_INT_FLAG_PLL0STB: PLL0 stabilization interrupt flag
      \arg        RCU_INT_FLAG_PLL1STB: PLL1 stabilization interrupt flag
      \arg        RCU_INT_FLAG_PLL2STB: PLL2 stabilization interrupt flag
      \arg        RCU_INT_FLAG_CKM: HXTAL clock stuck interrupt flag
      \arg        RCU_INT_FLAG_LCKM: LXTAL clock stuck interrupt flag
      \arg        RCU_INT_FLAG_LPIRC4MSTB: LPIRC4M stabilization interrupt flag
      \arg        RCU_INT_FLAG_IRC48MSTB: IRC48M stabilization interrupt flag
      \arg        RCU_INT_FLAG_PLLUSBHS0STB: PLLUSBHS0 stabilization interrupt flag
      \arg        RCU_INT_FLAG_PLLUSBHS1STB: PLLUSBHS1 stabilization interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_rcu_interrupt_flag_get(hal_rcu_int_flag_enum int_flag)
{
    FlagStatus ret = RESET;

    if(RESET != (RCU_REG_VAL(int_flag) & BIT(RCU_BIT_POS(int_flag)))) {
        return SET;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      clear the interrupt flags
    \param[in]  int_flag: clock stabilization and stuck interrupt flags clear, refer to hal_rcu_int_flag_clear_enum
                only one parameter can be selected which is shown as below:
      \arg        RCU_INT_FLAG_IRC32KSTB_CLR: IRC32K stabilization interrupt flag clear
      \arg        RCU_INT_FLAG_LXTALSTB_CLR: LXTAL stabilization interrupt flag clear
      \arg        RCU_INT_FLAG_IRC64MSTB_CLR: IRC64M stabilization interrupt flag clear
      \arg        RCU_INT_FLAG_HXTALSTB_CLR: HXTAL stabilization interrupt flag clear
      \arg        RCU_INT_FLAG_PLL0STB_CLR: PLL0 stabilization interrupt flag clear
      \arg        RCU_INT_FLAG_PLL1STB_CLR: PLL1 stabilization interrupt flag clear
      \arg        RCU_INT_FLAG_PLL2STB_CLR: PLL2 stabilization interrupt flag clear
      \arg        RCU_INT_FLAG_CKM_CLR: clock stuck interrupt flag clear
      \arg        RCU_INT_FLAG_LCKM_CLR: LXTAL clock stuck interrupt flag clear
      \arg        RCU_INT_FLAG_LPIRC4MSTB_CLR: LPIRC4M stabilization interrupt flag clear
      \arg        RCU_INT_FLAG_IRC48MSTB_CLR: IRC48M stabilization interrupt flag clear
      \arg        RCU_INT_FLAG_PLLUSBHS0STB_CLR: PLLUSBHS0 stabilization interrupt flag clear
      \arg        RCU_INT_FLAG_PLLUSBHS1STB_CLR: PLLUSBHS1 stabilization interrupt flag clear
    \param[out] none
    \retval     none
*/
void hals_rcu_interrupt_flag_clear(hal_rcu_int_flag_clear_enum int_flag)
{
    RCU_REG_VAL(int_flag) |= BIT(RCU_BIT_POS(int_flag));
}

/*!
    \brief      function to calculate the PLL output frequency
    \param[in]  pll_inputfreq: PLL0 / PLL1 / PLL2 input frequency,
                this parameter should be selected between HXTAL_VALUE, irc64mdiv_freq or LPIRC4M_VALUE
    \param[in]  pll_psc: the PLL0 / PLL1 / PLL2 source clock prescaler,
                this parameter should be selected between 1 and 63.
    \param[in]  pll_n: the PLL0 / PLL1 / PLL2 clock multi factor,
                this parameter should be selected between 9 and 512
    \param[in]  fracn: Fractional part of the multiplication factor for PLL0 / PLL1 / PLL2 VCO,
                this parameter should be selected between 0 and 0x1FFF
    \param[in]  pll_pqr: the PLL P / Q / R output frequency division factor from PLL0 / PLL1 / PLL2 VCO clock,
                this parameter should be selected between 1 and 128
    \param[out] none
    \retval     uint32_t: PLL clock frequency
*/
static uint32_t _rcu_pll_clock_freq_cal(uint32_t pll_inputfreq, uint32_t pll_psc, uint32_t pll_n, \
                                        uint32_t fracn, uint32_t pll_pqr)
{
    float freq;

    freq = ((float)pll_inputfreq / (float)pll_psc) * ((float)pll_n + ((float)fracn / (float)0x2000));
    freq = freq / (float)pll_pqr;

    return (uint32_t)freq;
}

/*!
    \brief      configure the pll input and output clock range
    \param[in]  pll_idx: INDEX_PLL, INDEX_PLLx (x = 0,1,2)
    \param[in]  ck_input: input clock range
                only one parameter can be selected which is shown as below:
      \arg        RCU_PLLxRNG_1M_2M (x = 0,1,2): input clock frequency: 1-2MHz
      \arg        RCU_PLLxRNG_2M_4M (x = 0,1,2): input clock frequency: 2-4MHz
      \arg        RCU_PLLxRNG_4M_8M (x = 0,1,2): input clock frequency: 4-8MHz
      \arg        RCU_PLLxRNG_8M_16M: input clock frequency: 8-16MHz
     \param[in]  ck_output: output clock range
                 only one parameter can be selected which is shown as below:
      \arg        RCU_PLLxVCO_192M_836M (x = 0,1,2): select wide VCO, range: 192-836MHz
      \arg        RCU_PLLxVCO_150M_420M (x = 0,1,2): select narrow VCO, range: 150-420MHz
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
static int32_t _rcu_pll_input_output_clock_range_config(hal_pll_idx_enum pll_idx, uint32_t ck_input, uint32_t ck_output)
{
    int32_t ret = HAL_ERR_NONE;

    switch(pll_idx) {
    case INDEX_PLL0:
        /* reset the PLLRNG/PLLVCOSEL bits and set according to ck_input/ck_output */
        RCU_PLLALL &= ~(RCU_PLLALL_PLL0RNG | RCU_PLLALL_PLL0VCOSEL);
        RCU_PLLALL |= (ck_input | ck_output);
        break;
    case INDEX_PLL1:
        /* reset the PLLRNG/PLLVCOSEL bits and set according to ck_input/ck_output */
        RCU_PLLALL &= ~(RCU_PLLALL_PLL1RNG | RCU_PLLALL_PLL1VCOSEL);
        RCU_PLLALL |= (ck_input | ck_output);
        break;
    case INDEX_PLL2:
        /* reset the PLLRNG/PLLVCOSEL bits and set according to ck_input/ck_output */
        RCU_PLLALL &= ~(RCU_PLLALL_PLL2RNG | RCU_PLLALL_PLL2VCOSEL);
        RCU_PLLALL |= (ck_input | ck_output);
        break;
    default:
        HAL_DEBUGE("parameter [pll_idx] address is invalid.");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      configure fractional part of the multiplication factor for PLL VCO
    \param[in]  pll_idx: INDEX_PLLx (x = 0,1,2)
    \param[in]  pll_frac: fractional part of the multiplication factor,
                this parameter should be selected between 0 and 8191
    \param[in]  pll_frac_state: fractional part of the multiplication factor status
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable fractional part of the multiplication factor
      \arg        DISABLE: disable fractional part of the multiplication factor
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
static int32_t _rcu_pll_fractional_config(hal_pll_idx_enum pll_idx, uint32_t pll_frac, uint32_t pll_frac_state)
{
    int32_t ret = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((ENABLE != pll_frac_state) && (DISABLE != pll_frac_state)) {
        HAL_DEBUGE("parameter [pll_frac_state] value is invalid.");
        return HAL_ERR_VAL;
    }

    if(8191U < pll_frac) {
        HAL_DEBUGE("parameter [pll_frac] value is invalid.");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(ENABLE == pll_frac_state) {
        switch(pll_idx) {
        case INDEX_PLL0:
            /* reset the PLLFRAN and set according to pll_fracn */
            RCU_PLL0FRA &= ~(RCU_PLL0FRA_PLL0FRAN);
            RCU_PLL0FRA |= pll_frac;
            RCU_PLL0FRA |= RCU_PLL0FRA_PLL0FRAEN;
            break;
        case INDEX_PLL1:
            /* reset the PLLFRAN and set according to pll_fracn */
            RCU_PLL1FRA &= ~(RCU_PLL1FRA_PLL1FRAN);
            RCU_PLL1FRA |= pll_frac;
            RCU_PLL1FRA |= RCU_PLL1FRA_PLL1FRAEN;
            break;
        case INDEX_PLL2:
            /* reset the PLLFRAN and set according to pll_fracn */
            RCU_PLL2FRA &= ~(RCU_PLL2FRA_PLL2FRAN);
            RCU_PLL2FRA |= pll_frac;
            RCU_PLL2FRA |= RCU_PLL2FRA_PLL2FRAEN;
            break;
        default:
            HAL_DEBUGE("parameter [pll_idx] value is invalid.");
            ret = HAL_ERR_VAL;
            break;
        }
    } else {
        switch(pll_idx) {
        case INDEX_PLL0:
            /* reset the PLL0FRAEN */
            RCU_PLL0FRA &= ~RCU_PLL0FRA_PLL0FRAEN;
            break;
        case INDEX_PLL1:
            /* reset the PLL1FRAEN */
            RCU_PLL1FRA &= ~RCU_PLL1FRA_PLL1FRAEN;
            break;
        case INDEX_PLL2:
            /* reset the PLL2FRAEN */
            RCU_PLL2FRA &= ~RCU_PLL2FRA_PLL2FRAEN;
            break;
        default:
            HAL_DEBUGE("parameter [pll_idx] value is invalid.");
            ret = HAL_ERR_VAL;
            break;
        }
    }

    return ret;
}

/*!
    \brief      configure the main PLL0 clock
    \param[in]  pll0: PLL0
                  pll0_psc: the PLL0 VCO source clock prescaler, this parameter should be selected between 1 and 63.
                  when pll0_psc=0, the PLL0 VCO source clock prescaler close.
      \arg        pll0_n: the PLL0 VCO clock multi factor, this parameter should be selected between 9 and 512
      \arg        pll0_p: the PLL0P output frequency division factor from PLL0 VCO clock,
                          this parameter should be selected between 1 and 128
      \arg        pll0_q: the PLL0 Q output frequency division factor from PLL0 VCO clock
                          this parameter should be selected between 1 and 128
      \arg        pll0_r: the PLL0 R output frequency division factor from PLL0 VCO clock
                          this parameter should be selected between 1 and 128
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
static int32_t _rcu_pll0_config(hal_rcu_pll0_struct *pll0)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* Check the pll0_psc parameter */
    if(RCU_PLL_PSC_63 < pll0->pll0_psc) {
        HAL_DEBUGE("parameter [pll0.pll0_psc] value is invalid.");
        return HAL_ERR_VAL;
    }

    /* Check the pll0_n parameter */
    if(RCU_PLLN_MUL_512 < pll0->pll0_n) {
        HAL_DEBUGE("parameter [Pll0.pll0_n] value is invalid.");
        return HAL_ERR_VAL;
    }

    /* Check the pll0_p parameter */
    if(RCU_PLLP_DIV_128 < pll0->pll0_p) {
        HAL_DEBUGE("parameter [Pll0.pll0_p] value is invalid.");
        return HAL_ERR_VAL;
    }

    /* Check the pll0_r parameter */
    if(RCU_PLLR_DIV_128 < pll0->pll0_r) {
        HAL_DEBUGE("parameter [Pll0.pll0_r] value is invalid.");
        return HAL_ERR_VAL;
    }

    /* Check the pll0_q parameter */
    if(RCU_PLLQ_DIV_128 < pll0->pll0_q) {
        HAL_DEBUGE("parameter [Pll0.pll0_q] value is invalid.");
        return HAL_ERR_VAL;
    }

#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configuration the PLL0 */
    RCU_PLL0 &= ~(RCU_PLL0_PLL0PSC | RCU_PLL0_PLL0N | RCU_PLL0_PLL0P | RCU_PLL0_PLL0R);
    RCU_PLL0 |= ((uint32_t)pll0->pll0_psc | (uint32_t)pll0->pll0_n | \
                 (uint32_t)pll0->pll0_p   | (uint32_t)pll0->pll0_r);

    /* configuration the PLL0 Q output frequency division factor */
    RCU_PLLADDCTL &= ~RCU_PLLADDCTL_PLL0Q;
    RCU_PLLADDCTL |= pll0->pll0_q;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the main PLL1 clock
    \param[in]  pll1: PLL1
                  pll1_psc: the PLL1 VCO source clock prescaler, this parameter should be selected between 1 and 63.
                  when pll1_psc=0, the PLL1 VCO source clock prescaler close.
      \arg        pll1_n: the PLL1 VCO clock multi factor, this parameter should be selected between 9 and 512
      \arg        pll1_p: the PLL1P output frequency division factor from PLL1 VCO clock,
                          this parameter should be selected between 1 and 128
      \arg        pll1_q: the PLL1 Q output frequency division factor from PLL1 VCO clock
                          this parameter should be selected between 1 and 128
      \arg        pll1_r: the PLL1 R output frequency division factor from PLL1 VCO clock
                          this parameter should be selected between 1 and 128
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
static int32_t _rcu_pll1_config(hal_rcu_pll1_struct *pll1)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* Check the pll1_psc parameter */
    if(RCU_PLL_PSC_63 < pll1->pll1_psc) {
        HAL_DEBUGE("parameter [pll1->pll1_psc] value is invalid.");
        return HAL_ERR_VAL;
    }

    /* Check the pll1_n parameter */
    if(RCU_PLLN_MUL_512 < pll1->pll1_n) {
        HAL_DEBUGE("parameter [pll1->pll1_n] value is invalid.");
        return HAL_ERR_VAL;
    }

    /* Check the pll1_p parameter */
    if(RCU_PLLP_DIV_128 < pll1->pll1_p) {
        HAL_DEBUGE("parameter [pll1->pll1_p] value is invalid.");
        return HAL_ERR_VAL;
    }

    /* Check the pll1_r parameter */
    if(RCU_PLLR_DIV_128 < pll1->pll1_r) {
        HAL_DEBUGE("parameter [pll1->pll1_r] value is invalid.");
        return HAL_ERR_VAL;
    }

    /* Check the pll1_q parameter */
    if(RCU_PLLQ_DIV_128 < pll1->pll1_q) {
        HAL_DEBUGE("parameter [Pll1->pll1_q] value is invalid.");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configuration the PLL1 */
    RCU_PLL1 &= ~(RCU_PLL1_PLL1PSC | RCU_PLL1_PLL1N | RCU_PLL1_PLL1P | RCU_PLL1_PLL1R);
    RCU_PLL1 |= ((uint32_t)pll1->pll1_psc | (uint32_t)pll1->pll1_n | \
                 (uint32_t)pll1->pll1_p   | (uint32_t)pll1->pll1_r);

    /* configuration the PLL1 Q output frequency division factor */
    RCU_PLLADDCTL &= ~RCU_PLLADDCTL_PLL1Q;
    RCU_PLLADDCTL |= ((uint32_t)pll1->pll1_q << RCU_PLL1QOFFSET);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the main PLL2 clock
    \param[in]  pll2: PLL2
                  pll2_psc: the PLL2 VCO source clock prescaler, this parameter should be selected between 1 and 63.
                  when pll2_psc=0, the PLL2 VCO source clock prescaler close.
      \arg        pll2_n: the PLL2 VCO clock multi factor, this parameter should be selected between 9 and 512
      \arg        pll2_p: the PLL2P output frequency division factor from PLL2 VCO clock,
                          this parameter should be selected between 1 and 128
      \arg        pll2_q: the PLL2 Q output frequency division factor from PLL2 VCO clock
                          this parameter should be selected between 1 and 128
      \arg        pll2_r: the PLL2 R output frequency division factor from PLL2 VCO clock
                          this parameter should be selected between 1 and 128
    \param[out] none
    \retval     none
*/
static int32_t _rcu_pll2_config(hal_rcu_pll2_struct *pll2)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* Check the pll1_psc parameter */
    if(RCU_PLL_PSC_63 < pll2->pll2_psc) {
        HAL_DEBUGE("parameter [pll2->pll2_psc] value is invalid.");
        return HAL_ERR_VAL;
    }

    /* Check the pll2_n parameter */
    if(RCU_PLLN_MUL_512 < pll2->pll2_n) {
        HAL_DEBUGE("parameter [pll2->pll2_n] value is invalid.");
        return HAL_ERR_VAL;
    }

    /* Check the pll2_p parameter */
    if(RCU_PLLP_DIV_128 < pll2->pll2_p) {
        HAL_DEBUGE("parameter [pll2->pll2_p] value is invalid.");
        return HAL_ERR_VAL;
    }

    /* Check the pll2_r parameter */
    if(RCU_PLLR_DIV_128 < pll2->pll2_r) {
        HAL_DEBUGE("parameter [pll2->pll2_r] value is invalid.");
        return HAL_ERR_VAL;
    }

    /* Check the pll2_q parameter */
    if(RCU_PLLQ_DIV_128 < pll2->pll2_q) {
        HAL_DEBUGE("parameter [pll2->pll2_q] value is invalid.");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure PLL2 source selection */
    RCU_PLL2 &= ~(RCU_PLL2_PLL2PSC | RCU_PLL2_PLL2N | RCU_PLL2_PLL2P | RCU_PLL2_PLL2R);
    RCU_PLL2 |= ((uint32_t)pll2->pll2_psc | (uint32_t)pll2->pll2_n | \
                 (uint32_t)pll2->pll2_p | (uint32_t)pll2->pll2_r);

    /* configuration the PLL2 Q output frequency division factor */
    RCU_PLLADDCTL &= ~RCU_PLLADDCTL_PLL2Q;
    RCU_PLLADDCTL |= ((uint32_t)pll2->pll2_q << RCU_PLL2QOFFSET);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure PLL clock source
    \param[in]  pll_src: PLL clock source selection
                only one parameters can be selected which are shown as below
      \arg        RCU_PLL_SRC_CK_IRC64MDIV: select IRC64MDIV as PLL source clock
      \arg        RCU_PLL_SRC_CK_LPIRC4M: select LPIRC4M as PLL source clock
      \arg        RCU_PLL_SRC_CK_HXTAL: select HXTAL as PLL source clock
      \arg        RCU_PLL_SRC_CK_NONE: select no clock as PLL source clock
    \param[out] none
    \retval     none
*/
static void _rcu_pll_source_config(hal_rcu_pll_src_enum pll_src)
{
    RCU_PLLALL &= ~(RCU_PLLALL_PLLSEL);
    RCU_PLLALL |= (uint32_t)pll_src;
}

/*!
    \brief      configure the PLLUSBHS0 clock
    \param[in]  pllusb0: PLLUSBHS0
    \arg          pllusb0_preclock_source: PLLUSB0 Clock Source pre_Selection
                                           this parameter should be selected RCU_PLLUSBHSx_CKSRC_HXTAL,
                                                                             RCU_PLLUSBHSx_CKSRC_IRC48M
    \arg          pllusb0_presel: the divider factor from PLLUSBHS clock
                                  this parameter should be selected between RCU_PLLUSBHSPRE_DIVx (x = 1...15)
    \arg          pllusb0_mul: PLLUSBHS0 clock multiplication factor
                               this parameter should be selected RCU_PLLUSBHS_MULx (x = 16,17...127)
    \param[out] none
    \retval     none
*/
static void _rcu_pllusb0_config(hal_rcu_pllusb0_struct *pllusb0)
{
    /* configure PLLUSBHS source selection */
    RCU_USBCLKCTL &= ~(RCU_USBCLKCTL_PLLUSBHS0PRESEL);
    RCU_USBCLKCTL |= ((uint32_t)pllusb0->pllusb0_preclock_source);

    RCU_PLLUSBCFG &= ~(RCU_PLLUSBCFG_PLLUSBHS0PREDV | RCU_PLLUSBCFG_PLLUSBHS0MF);
    RCU_PLLUSBCFG |= (((uint32_t)pllusb0->pllusb0_presel | (uint32_t)pllusb0->pllusb0_mul));
}

/*!
    \brief      configure the PLLUSBHS1 clock
    \param[in]  pllusb1: PLLUSBHS1
    \arg          pllusb1_preclock_source: pllusb1 clock source selection
                                           this parameter should be selected RCU_PLLUSBHSx_CKSRC_HXTAL,
                                                                             RCU_PLLUSBHSx_CKSRC_IRC48M
    \arg          pllusb1_presel: the divider factor from PLLUSBHS clock
                                  this parameter should be selected between RCU_PLLUSBHSPRE_DIVx (x = 1...15)
    \arg          pllusb1_mul: PLLUSBHS1 clock multiplication factor
                               this parameter should be selected RCU_PLLUSBHS_MULx (x = 16,17...127)
    \param[out] none
    \retval     none
*/
static void _rcu_pllusb1_config(hal_rcu_pllusb1_struct *pllusb1)
{
    /* configure PLLUSBHS source selection */
    RCU_USBCLKCTL &= ~(RCU_USBCLKCTL_PLLUSBHS1PRESEL);
    RCU_USBCLKCTL |= ((uint32_t)pllusb1->pllusb1_preclock_source << 8U);

    RCU_PLLUSBCFG &= ~(RCU_PLLUSBCFG_PLLUSBHS1PREDV | RCU_PLLUSBCFG_PLLUSBHS1MF);
    RCU_PLLUSBCFG |= (((uint32_t)pllusb1->pllusb1_presel | (uint32_t)pllusb1->pllusb1_mul) << 16U);
}

/*!
    \brief      enable the pllp pllq pllr divider output
    \param[in]  pllxy: the output pll enable
                only one parameters can be selected which are shown as below
      \arg        RCU_PLL0P: PLL0P divider output enable
      \arg        RCU_PLL0Q: PLL0Q divider output enable
      \arg        RCU_PLL0R: PLL0R divider output enable
      \arg        RCU_PLL1P: PLL1P divider output enable
      \arg        RCU_PLL1Q: PLL1Q divider output enable
      \arg        RCU_PLL1R: PLL1R divider output enable
      \arg        RCU_PLL2P: PLL2P divider output enable
      \arg        RCU_PLL2Q: PLL2Q divider output enable
      \arg        RCU_PLL2R: PLL2R divider output enable
    \param[in]  pll_clock_status: the output pll status: ENABLE,DISBALE
    \param[out] none
    \retval     none
*/
static void _rcu_pll_clock_output_config(uint32_t pllxy, uint32_t pll_clock_status)
{
    if(ENABLE == pll_clock_status) {
        RCU_PLLADDCTL |= pllxy;
    } else {
        RCU_PLLADDCTL &= ~(pllxy);
    }

    /* PLL divider output enable */
    RCU_PLLADDCTL |= pllxy;
}

/*!
    \brief      configure the PLL2R divider used as input of TLI
    \param[in]  pll2r_div: PLL2R divider used as input of TLI
                only one parameter can be selected which is shown as below:
      \arg        RCU_PLL2R_DIVx (x = 2,4,8,16): PLL2R divided x used as input of TLI
    \param[out] none
    \retval     none
*/
static void _rcu_pll2r_clock_div_config(hal_rcu_pll2r_div_enum pll2r_div)
{
    uint32_t reg = 0U;

    /* reset the PLL2RDIV bit and set according to pll2r_div */
    reg = RCU_CFG1;
    reg &= ~RCU_CFG1_PLL2RDIV;
    RCU_CFG1 = (reg | (uint32_t)pll2r_div);
}

/*!
    \brief      configure the per clock source select
    \param[in]  periph_clk: the pointer of peripheral clock structure
                only one parameter can be selected which is shown as below:
      \arg        RCU_PRECLK_CKSRC_CK_IRC64MDIV: irc64m divided clock used as clock source
      \arg        RCU_PRECLK_CKSRC_CK_LPIRC4M: lpirc4m clock used as clock source
      \arg        RCU_PRECLK_CKSRC_CK_HXTAL: hxtal clock used as clock source
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
static int32_t _rcu_per_clock_source_select(hal_rcu_periphclk_struct *periph_clk)
{
    int32_t ret = HAL_ERR_NONE;

    switch(periph_clk->pef_clock_source) {
    /* disable HXTAL to bypass mode */
    case RCU_PRECLK_CKSRC_CK_IRC64MDIV:
        _rcu_irc64m_clock_source_select(periph_clk->irc64m_div);
        break;
    case RCU_PRECLK_CKSRC_CK_LPIRC4M:
        _rcu_lpirc4m_clock_source_select();
        break;
    case RCU_PRECLK_CKSRC_CK_HXTAL:
        _rcu_hxtal_clock_source_select();
        break;
    default:
        HAL_DEBUGE("parameter [periph_clk->pef_clock_source] value is invalid");
        ret = HAL_ERR_VAL;
        break;
    }

    hals_rcu_per_clock_config(periph_clk->pef_clock_source);

    return ret;
}

/*!
    \brief      configure the irc64m as clock source select
    \param[in]  irc64m_div: irc64m divided value,
                only one parameter can be selected which is shown as below:
      \arg        RCU_IRC64M_DIV1: irc64m divided 1 used as clock source
      \arg        RCU_IRC64M_DIV2: irc64m divided 2 used as clock source
      \arg        RCU_IRC64M_DIV4: irc64m divided 4 used as clock source
      \arg        RCU_IRC64M_DIV8: irc64m divided 8 used as clock source
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
static int32_t _rcu_irc64m_clock_source_select(hal_rcu_irc64m_div_enum irc64m_div)
{
    int32_t ret = HAL_ERR_NONE;

    if(RESET == (RCU_CTL & RCU_CTL_IRC64MEN)) {
        /* configuration irc64m divided value */
        if((0U != (uint32_t)irc64m_div) && (RCU_SCSS_IRC64MDIV != hals_rcu_system_clock_source_get())) {
            hals_rcu_irc64mdiv_clock_config(irc64m_div);
        } else {
            /* do nothing */
        }

        hals_rcu_osci_on(RCU_IRC64M);
        /* wait till IRC64M is stable */
        if(ERROR == hals_rcu_osci_stab_wait(RCU_IRC64M)) {
            ret = HAL_ERR_TIMEOUT;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      configure the lpirc4m as clock source select
    \param[in]  none
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
static int32_t _rcu_lpirc4m_clock_source_select(void)
{
    int32_t ret = HAL_ERR_NONE;

    if(RESET == (RCU_ADDCTL1 & RCU_ADDCTL1_LPIRC4MEN)) {
        hals_rcu_osci_on(RCU_LPIRC4M);

        /* wait till LPIRC4M is stable */
        if(ERROR == hals_rcu_osci_stab_wait(RCU_LPIRC4M)) {
            ret = HAL_ERR_TIMEOUT;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      configure the lxtal as clock source select
    \param[in]  none
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
static int32_t _rcu_lxtal_clock_source_select(void)
{
    int32_t ret = HAL_ERR_NONE;

    if(RESET == (RCU_BDCTL & RCU_BDCTL_LXTALEN)) {
        hals_rcu_osci_on(RCU_LXTAL);

        /* wait till LXTAL is stable */
        if(ERROR == hals_rcu_osci_stab_wait(RCU_LXTAL)) {
            ret =  HAL_ERR_TIMEOUT;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      configure the hxtal as clock source select
    \param[in]  none
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
static int32_t _rcu_hxtal_clock_source_select(void)
{
    int32_t ret = HAL_ERR_NONE;

    if(RESET == (RCU_CTL & RCU_CTL_HXTALEN)) {
        hals_rcu_osci_on(RCU_HXTAL);

        /* wait till HXTAL is stable */
        if(ERROR == hals_rcu_osci_stab_wait(RCU_HXTAL)) {
            ret =  HAL_ERR_TIMEOUT;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      configure the lxtal as clock source select
    \param[in]  none
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
static int32_t _rcu_irc32k_clock_source_select(void)
{
    int32_t ret = HAL_ERR_NONE;

    if(RESET == (RCU_RSTSCK & RCU_RSTSCK_IRC32KEN)) {
        hals_rcu_osci_on(RCU_IRC32K);

        /* wait till irc32k is stable */
        if(ERROR == hals_rcu_osci_stab_wait(RCU_IRC32K)) {
            ret =  HAL_ERR_TIMEOUT;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return ret;
}
