/*!
    \file    gd32h7xx_hal_irda.c
    \brief   IRDA driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* irda private function */
/* irda deinit function */
static void _irda_deinit(uint32_t irda_periph);
/* get the mask of data bit */
static uint16_t _irda_data_bit_mask_get(hal_irda_dev_struct *irda_dev);
/* handle the transmit complete interrupt */
static void _irda_transmit_complete_interrupt(void *irda_dev);
/* handle the transmit interrupt */
static void _irda_transmit_fifo_empty_interrupt(void *irda_dev);
/* handle the receive interrupt */
static void _irda_receive_interrupt(void *irda_dev);
/* handle the error interrupt */
static void _irda_error_interrupt(void *irda_dev);
/* handle the irda DMA transmit complete process */
static void _irda_dma_tx_complete(void *dma);
/* handle the irda DMA transmit half complete process */
static void _irda_dma_tx_halfcomplete(void *dma);
/* handle the irda DMA receive complete process */
static void _irda_dma_rx_complete(void *dma);
/* handle the irda DMA receive half complete process */
static void _irda_dm_rx_halfcomplete(void *dma);
/* handle the irda DMA error process */
static void _irda_dma_error(void *dma);

/*!
    \brief      initialize irda
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  periph: specify which IRDA is initialized, USART0/1/2/5 only
    \param[in]  p_irda: the initialization data needed to initialize irda
                  baudrate: communication baudrate
                  parity: USART_PARITY_NONE, USART_PARITY_EVEN, USART_PARITY_ODD
                  wordlength: USART_WORDLENGTH_7B, USART_WORDLENGTH_8B, USART_WORDLENGTH_9B, USART_WORDLENGTH_10B
                  mode: USART_MODE_RX, USART_MODE_TX, USART_MODE_TX_RX
                  prescaler: 1 - 255
                  powermode:USART_POWERMODE_NORMAL, USART_POWERMODE_LOWPOWER
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_init(hal_irda_dev_struct *irda_dev, uint32_t periph, hal_irda_init_struct *p_irda)
{
    uint32_t reg_temp = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check irda pointer and p_irda address */
    if((NULL == irda_dev) || (NULL == p_irda)) {
        HAL_DEBUGE("pointer [irda_dev] or [p_irda] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check periph parameter */
    if((USART0 != periph) && (USART1 != periph) && (USART2 != periph) && (USART5 != periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock irda */
    HAL_LOCK(irda_dev);

    irda_dev->periph = periph;
    irda_dev->init   = *p_irda;

    /* disable irda */
    USART_CTL0(periph) &= ~(USART_CTL0_UEN);

    /* configure CTL0 register */
    reg_temp = USART_CTL0(periph);
    reg_temp &= ~(USART_CTL0_OVSMOD | USART_CTL0_PM | USART_CTL0_PCEN | USART_CTL0_WL0 | \
                  USART_CTL0_WL1 | USART_CTL0_REN | USART_CTL0_TEN);
    reg_temp |= (p_irda->mode | p_irda->parity | p_irda->wordlength);
    USART_CTL0(periph) = reg_temp;

    /* configure CTL1 register */
    reg_temp = USART_CTL1(periph);
    reg_temp &= ~USART_CTL1_STB;
    USART_CTL1(periph) = reg_temp;

    /* configure GP register */
    reg_temp = USART_GP(periph);
    reg_temp &= ~USART_GP_PSC;
    reg_temp |= p_irda->prescaler;
    USART_GP(periph) = reg_temp;

    /* configure RFCS register */
    reg_temp = USART_FCS(periph);
    reg_temp &= ~(USART_FCS_FEN | USART_FCS_RFTCFG | USART_FCS_TFTCFG);
    reg_temp |= (p_irda->fifomode | p_irda->rxthreshold | p_irda->txthreshold);
    USART_FCS(periph) = reg_temp;

    /* clear LMEN, HDEN, CKEN, SCEN, MUTE */
    USART_CTL2(periph) &= ~(USART_CTL2_SCEN | USART_CTL2_HDEN);
    USART_CTL1(periph) &= ~(USART_CTL1_LMEN | USART_CTL1_CKEN);
    USART_CTL0(periph) &= ~(USART_CTL0_MEN);

    /* configure irda in low power mode */
    USART_CTL2(periph) &= ~(USART_CTL2_IRLP);
    USART_CTL2(periph) |= (USART_CTL2_IREN | p_irda->powermode);

    /* configure baudrate */
    hals_usart_baudrate_set(periph, p_irda->baudrate);
    irda_dev->data_bit_mask = _irda_data_bit_mask_get(irda_dev);

    /* reset the Rx and Tx state */
    irda_dev->tx_state = HAL_IRDA_STATE_READY;
    irda_dev->rx_state = HAL_IRDA_STATE_READY;

    /* enable usart */
    USART_CTL0(periph) |= USART_CTL0_UEN;

    /* unlock irda */
    HAL_UNLOCK(irda_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the irda structure with the default values
                note: this function must be called after the structure is created
    \param[in]  hal_struct_type: type of irda structure for initialization
      \arg        HAL_IRDA_INIT_STRUCT: initialization structure
      \arg        HAL_IRDA_DEV_STRUCT: device information structure
      \arg        HAL_IRDA_IRQ_INIT_STRUCT: interrupt callback initialization structure
    \param[out] p_struct: pointer to IRDA structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_struct_init(hal_irda_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_IRDA_INIT_STRUCT:
        /* initialize irda initialization structure with the default values */
        ((hal_irda_init_struct *)p_struct)->baudrate    = 115200U;
        ((hal_irda_init_struct *)p_struct)->wordlength  = USART_WORDLENGTH_8B;
        ((hal_irda_init_struct *)p_struct)->parity      = USART_PARITY_NONE;
        ((hal_irda_init_struct *)p_struct)->mode        = USART_MODE_TX_RX;
        ((hal_irda_init_struct *)p_struct)->prescaler   = 10U;
        ((hal_irda_init_struct *)p_struct)->powermode   = USART_POWERMODE_NORMAL;
        ((hal_irda_init_struct *)p_struct)->fifomode    = USART_FIFOMODE_DISABLE;
        ((hal_irda_init_struct *)p_struct)->txthreshold = USART_TXFIFO_THRESHOLD_1_8;
        ((hal_irda_init_struct *)p_struct)->rxthreshold = USART_RXFIFO_THRESHOLD_1_8;
        break;
    case HAL_IRDA_DEV_STRUCT:
        /* initialize irda device information structure with the default values */
        ((hal_irda_dev_struct *)p_struct)->periph                              = 0U;
        ((hal_irda_dev_struct *)p_struct)->irda_irq.transmit_complete_handle   = NULL;
        ((hal_irda_dev_struct *)p_struct)->irda_irq.transmit_fifo_empty_handle = NULL;
        ((hal_irda_dev_struct *)p_struct)->irda_irq.receive_complete_handle    = NULL;
        ((hal_irda_dev_struct *)p_struct)->irda_irq.error_handle               = NULL;
        ((hal_irda_dev_struct *)p_struct)->p_dma_rx                            = NULL;
        ((hal_irda_dev_struct *)p_struct)->p_dma_tx                            = NULL;
        ((hal_irda_dev_struct *)p_struct)->txbuffer.buffer                     = NULL;
        ((hal_irda_dev_struct *)p_struct)->txbuffer.length                     = 0U;
        ((hal_irda_dev_struct *)p_struct)->txbuffer.remain                     = 0U;
        ((hal_irda_dev_struct *)p_struct)->rxbuffer.buffer                     = NULL;
        ((hal_irda_dev_struct *)p_struct)->rxbuffer.length                     = 0U;
        ((hal_irda_dev_struct *)p_struct)->rxbuffer.remain                     = 0U;
        ((hal_irda_dev_struct *)p_struct)->data_bit_mask                       = 0U;
        ((hal_irda_dev_struct *)p_struct)->last_error                          = HAL_IRDA_ERROR_NONE;
        ((hal_irda_dev_struct *)p_struct)->error_state                         = HAL_IRDA_ERROR_NONE;
        ((hal_irda_dev_struct *)p_struct)->tx_state                            = HAL_IRDA_STATE_RESET;
        ((hal_irda_dev_struct *)p_struct)->rx_state                            = HAL_IRDA_STATE_RESET;
        ((hal_irda_dev_struct *)p_struct)->transmit_complete_callback          = NULL;
        ((hal_irda_dev_struct *)p_struct)->transmit_half_complete_callback     = NULL;
        ((hal_irda_dev_struct *)p_struct)->receive_complete_callback           = NULL;
        ((hal_irda_dev_struct *)p_struct)->receive_half_complete_callback      = NULL;
        ((hal_irda_dev_struct *)p_struct)->error_callback                      = NULL;
        ((hal_irda_dev_struct *)p_struct)->mutex                               = HAL_MUTEX_UNLOCKED;
        ((hal_irda_dev_struct *)p_struct)->priv                                = NULL;
        break;
    case HAL_IRDA_IRQ_INIT_STRUCT:
        /* initialize interrupt callback structure with the default values */
        ((hal_irda_irq_struct *)p_struct)->transmit_complete_handle   = NULL;
        ((hal_irda_irq_struct *)p_struct)->transmit_fifo_empty_handle = NULL;
        ((hal_irda_irq_struct *)p_struct)->receive_complete_handle    = NULL;
        ((hal_irda_irq_struct *)p_struct)->error_handle               = NULL;
        break;
    case HAL_IRDA_USER_CALLBACK_STRUCT:
        /* initialize user callback structure with the default values */
        ((hal_irda_irq_user_callback_struct *)p_struct)->transmit_complete_func      = NULL;
        ((hal_irda_irq_user_callback_struct *)p_struct)->transmit_half_complete_func = NULL;
        ((hal_irda_irq_user_callback_struct *)p_struct)->receive_complete_func       = NULL;
        ((hal_irda_irq_user_callback_struct *)p_struct)->receive_half_complete_func  = NULL;
        ((hal_irda_irq_user_callback_struct *)p_struct)->error_func                  = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      deinitialize irda
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_deinit(hal_irda_dev_struct *irda_dev)
{
    uint32_t periph = 0U;
    int32_t  ret    = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == irda_dev) {
        HAL_DEBUGE("pointer [irda_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(irda_dev);

    periph = irda_dev->periph;

    if((USART0 == periph) || (USART1 == periph) || (USART2 == periph) || (USART5 == periph)) {
        /* deinitialize the periph and the device information structure */
        _irda_deinit(periph);

        irda_dev->last_error  = HAL_IRDA_ERROR_NONE;
        irda_dev->error_state = HAL_IRDA_ERROR_NONE;
        irda_dev->tx_state    = HAL_IRDA_STATE_RESET;
        irda_dev->rx_state    = HAL_IRDA_STATE_RESET;
    } else {
        HAL_DEBUGE("parameter [irda_dev->periph] value is invalid");
        ret = HAL_ERR_VAL;
    }

    HAL_UNLOCK(irda_dev);

    return ret;
}

/*!
    \brief      irda interrupt handler content function,which is merely used in USART_IRQHandler
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_irq(hal_irda_dev_struct *irda_dev)
{
    __IO uint32_t errorflags = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == irda_dev) {
        HAL_DEBUGE("pointer [irda_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* if no error occurs */
    errorflags = (USART_STAT(irda_dev->periph) & (uint32_t)(USART_STAT_PERR  | USART_STAT_FERR | \
                                                            USART_STAT_ORERR | USART_STAT_NERR | USART_STAT_RTF));
    if(0U == errorflags) {
        /* if irda is in receiver mode */
        if(RESET != hals_usart_interrupt_flag_get(irda_dev->periph, USART_INT_FLAG_RBNE)) {
            hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_RBNE);
            if(NULL != irda_dev->irda_irq.receive_complete_handle) {
                irda_dev->irda_irq.receive_complete_handle(irda_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* if some errors occur */
    if(0U != errorflags) {
        /* irda parity error interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(irda_dev->periph, USART_INT_FLAG_PERR)) {
            hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_PERR);
            irda_dev->error_state |= (uint16_t)HAL_IRDA_ERROR_PERR;
            irda_dev->last_error = (uint16_t)HAL_IRDA_ERROR_PERR;
        } else {
            /* do nothing */
        }

        /* irda frame error interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(irda_dev->periph, USART_INT_FLAG_ERR_FERR)) {
            hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_ERR_FERR);
            irda_dev->error_state |= (uint16_t)HAL_IRDA_ERROR_FERR;
            irda_dev->last_error = (uint16_t)HAL_IRDA_ERROR_FERR;
        } else {
            /* do nothing */
        }

        /* irda noise error interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(irda_dev->periph, USART_INT_FLAG_ERR_NERR)) {
            hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_ERR_NERR);
            irda_dev->error_state |= (uint16_t)HAL_IRDA_ERROR_NERR;
            irda_dev->last_error = (uint16_t)HAL_IRDA_ERROR_NERR;
        } else {
            /* do nothing */
        }

        /* irda over-run interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(irda_dev->periph, USART_INT_FLAG_ERR_ORERR)) {
            hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_ERR_ORERR);
            irda_dev->error_state |= (uint16_t)HAL_IRDA_ERROR_ORERR;
            irda_dev->last_error = (uint16_t)HAL_IRDA_ERROR_ORERR;
        } else {
            /* do nothing */
        }

        /* check whether error state is none or not */
        if(HAL_IRDA_ERROR_NONE != irda_dev->error_state) {
            /* if irda is in receiver mode */
            if(RESET != hals_usart_interrupt_flag_get(irda_dev->periph, USART_INT_FLAG_RBNE)) {
                hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_RBNE);
                if(NULL != irda_dev->irda_irq.receive_complete_handle) {
                    irda_dev->irda_irq.receive_complete_handle(irda_dev);
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            if(NULL != irda_dev->irda_irq.error_handle) {
                irda_dev->irda_irq.error_handle(irda_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* transmitter buffer empty interrupt handle */
    if(RESET != hals_usart_interrupt_flag_get(irda_dev->periph, USART_INT_FLAG_TBE)) {
        hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_TBE);
        if(NULL != irda_dev->irda_irq.transmit_fifo_empty_handle) {
            irda_dev->irda_irq.transmit_fifo_empty_handle(irda_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* transmission complete interrupt handle */
    if(RESET != hals_usart_interrupt_flag_get(irda_dev->periph, USART_INT_FLAG_TC)) {
        hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_TC);
        if(NULL != irda_dev->irda_irq.transmit_complete_handle) {
            irda_dev->irda_irq.transmit_complete_handle(irda_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  irda_dev: irda device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to irda interrupt callback functions structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg        NULL: The corresponding callback mechanism is out of use, and
                    disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_irq_handle_set(hal_irda_dev_struct *irda_dev, hal_irda_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and p_irq address */
    if((NULL == irda_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [irda_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set user-defined error interrupt callback */
    if(NULL != p_irq->error_handle) {
        irda_dev->irda_irq.error_handle = p_irq->error_handle;
        hals_usart_interrupt_enable(irda_dev->periph, USART_INT_ERR);
        hals_usart_interrupt_enable(irda_dev->periph, USART_INT_PERR);
    } else {
        irda_dev->irda_irq.error_handle = NULL;
        hals_usart_interrupt_disable(irda_dev->periph, USART_INT_ERR);
        hals_usart_interrupt_disable(irda_dev->periph, USART_INT_PERR);
    }

    /* set user-defined receive complete interrupt callback */
    if(NULL != p_irq->receive_complete_handle) {
        irda_dev->irda_irq.receive_complete_handle = p_irq->receive_complete_handle;
        hals_usart_interrupt_enable(irda_dev->periph, USART_INT_RBNE);
    } else {
        irda_dev->irda_irq.receive_complete_handle = NULL;
        hals_usart_interrupt_disable(irda_dev->periph, USART_INT_RBNE);
    }

    /* set user-defined transmit complete interrupt callback */
    if(NULL != p_irq->transmit_complete_handle) {
        irda_dev->irda_irq.transmit_complete_handle = p_irq->transmit_complete_handle;
        hals_usart_interrupt_enable(irda_dev->periph, USART_INT_TC);
    } else {
        irda_dev->irda_irq.transmit_complete_handle = NULL;
        hals_usart_interrupt_disable(irda_dev->periph, USART_INT_TC);
    }

    /* set user-defined transmit ready interrupt callback */
    if(NULL != p_irq->transmit_fifo_empty_handle) {
        irda_dev->irda_irq.transmit_fifo_empty_handle = p_irq->transmit_fifo_empty_handle;
        hals_usart_interrupt_enable(irda_dev->periph, USART_INT_TBE);
    } else {
        irda_dev->irda_irq.transmit_fifo_empty_handle = NULL;
        hals_usart_interrupt_disable(irda_dev->periph, USART_INT_TBE);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_irq_handle_all_reset(hal_irda_dev_struct *irda_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and p_irq address */
    if(NULL == irda_dev) {
        HAL_DEBUGE("pointer [irda_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure interrupt callback function to NULL */
    irda_dev->irda_irq.error_handle               = NULL;
    irda_dev->irda_irq.receive_complete_handle    = NULL;
    irda_dev->irda_irq.transmit_complete_handle   = NULL;
    irda_dev->irda_irq.transmit_fifo_empty_handle = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      transmit amounts of data, poll transmit process and completed status
                the function is blocking
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to data buffer
    \param[in]  length: number of data to be transmitted
    \param[in]  timeout: timeout duration
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT,
                            HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_transmit_poll(hal_irda_dev_struct *irda_dev, uint8_t *p_buffer, uint32_t length, uint32_t timeout)
{
    volatile uint8_t *p_data8bits;
    volatile uint16_t *p_data16bits;
    uint32_t tick_start;
    int32_t  ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == irda_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [irda_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock irda */
    HAL_LOCK(irda_dev);

    /* check the tx_state whether is busy or not */
    if(HAL_IRDA_STATE_READY != irda_dev->tx_state) {
        HAL_DEBUGE("irda tx has already been used, please wait until run_state change to free");
        irda_dev->error_state = (uint16_t)HAL_IRDA_ERROR_BUSY;
        ret                   = HAL_ERR_BUSY;
    } else {
        /* configure the state */
        irda_dev->error_state = HAL_IRDA_ERROR_NONE;
        irda_dev->tx_state    = HAL_IRDA_STATE_BUSY_TX;

        /* initialize transmit parameters */
        irda_dev->txbuffer.buffer = p_buffer;
        irda_dev->txbuffer.length = length;
        irda_dev->txbuffer.remain = length;

        /* calculate the data length */
        if((USART_WORDLENGTH_10B == irda_dev->init.wordlength) || \
           ((USART_WORDLENGTH_9B == irda_dev->init.wordlength) && (USART_PARITY_NONE == irda_dev->init.parity))) {
            p_data8bits  = NULL;
            p_data16bits = (uint16_t *)(uint32_t)irda_dev->txbuffer.buffer;
        } else {
            p_data8bits  = irda_dev->txbuffer.buffer;
            p_data16bits = NULL;
        }

        /* configure timeout */
        tick_start = hal_sys_basetick_count_get();

        while(0U < irda_dev->txbuffer.remain) {
            /* wait for transmit buffer empty */
            while(RESET == hals_usart_flag_get(irda_dev->periph, USART_FLAG_TBE)) {
                if(HAL_TIMEOUT_FOREVER != timeout) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                        HAL_DEBUGE("irda transmit timeout");
                        ret                   = HAL_ERR_TIMEOUT;
                        irda_dev->error_state = (uint16_t)HAL_IRDA_ERROR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }

            if(HAL_ERR_NONE == ret) {
                /* send the data to be transmitted */
                if(NULL == p_data8bits) {
                    hals_usart_data_transmit(irda_dev->periph, (uint16_t)((uint16_t)*p_data16bits & 0x03FFU));
                    p_data16bits++;
                } else {
                    hals_usart_data_transmit(irda_dev->periph, (uint16_t)((uint16_t)*p_data8bits & 0xFFU));
                    p_data8bits++;
                }
                /* change the transmit pointer */
                irda_dev->txbuffer.remain--;
            } else {
                /* do nothing */
            }
        }

        if(HAL_ERR_NONE == ret) {
            /* wait for transmit complete */
            while(RESET == hals_usart_flag_get(irda_dev->periph, USART_FLAG_TC)) {
                if(HAL_TIMEOUT_FOREVER != timeout) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                        HAL_DEBUGE("irda transmit timeout");
                        /* reset the state */
                        ret                   = HAL_ERR_TIMEOUT;
                        irda_dev->error_state = (uint16_t)HAL_IRDA_ERROR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }
        } else {
            /* do nothing */
        }

        /* change the Tx state to ready */
        irda_dev->tx_state = HAL_IRDA_STATE_READY;
    }

    /* unlock irda */
    HAL_UNLOCK(irda_dev);

    return ret;
}

/*!
    \brief      receive amounts of data, poll receive process and completed status
                the function is blocking
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  length: number of data to be received
    \param[in]  timeout: timeout duration
    \param[out] p_buffer: pointer to data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL,
                            HAL_ERR_TIMEOUT, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_receive_poll(hal_irda_dev_struct *irda_dev, uint8_t *p_buffer, uint32_t length, uint32_t timeout)
{
    volatile uint8_t *p_data8bits;
    volatile uint16_t *p_data16bits;
    uint32_t tick_start;
    int32_t  ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == irda_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [irda_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock irda */
    HAL_LOCK(irda_dev);

    /* check the rx_state whether is busy or not */
    if(HAL_IRDA_STATE_READY != irda_dev->rx_state) {
        HAL_DEBUGE("irda rx has already been used, please wait until run_state change to free ");
        irda_dev->error_state = (uint16_t)HAL_IRDA_ERROR_BUSY;
        ret                   = HAL_ERR_BUSY;
    } else {
        /* configure the state */
        irda_dev->error_state = HAL_IRDA_ERROR_NONE;
        irda_dev->rx_state    = HAL_IRDA_STATE_BUSY_RX;

        /* initialize receive parameters */
        irda_dev->rxbuffer.buffer = p_buffer;
        irda_dev->rxbuffer.length = length;
        irda_dev->rxbuffer.remain = length;
        irda_dev->data_bit_mask   = _irda_data_bit_mask_get(irda_dev);

        /* calculate the data length */
        if((USART_WORDLENGTH_10B == irda_dev->init.wordlength) || \
           ((USART_WORDLENGTH_9B == irda_dev->init.wordlength) && (USART_PARITY_NONE == irda_dev->init.parity))) {
            p_data8bits  = NULL;
            p_data16bits = (uint16_t *)(uint32_t)irda_dev->txbuffer.buffer;
        } else {
            p_data8bits  = irda_dev->rxbuffer.buffer;
            p_data16bits = NULL;
        }

        /* configure timeout */
        tick_start = hal_sys_basetick_count_get();

        while(irda_dev->rxbuffer.remain > 0U) {
            /* wait for read data buffer not empty */
            while(RESET == hals_usart_flag_get(irda_dev->periph, USART_FLAG_RBNE)) {
                if(HAL_TIMEOUT_FOREVER != timeout) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                        HAL_DEBUGE("irda receive timeout");
                        ret                   = HAL_ERR_TIMEOUT;
                        irda_dev->error_state = (uint16_t)HAL_IRDA_ERROR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }

            if(HAL_ERR_NONE == ret) {
                /* read data from data register */
                if(NULL == p_data8bits) {
                    *p_data16bits = (uint16_t)(hals_usart_data_receive(irda_dev->periph) & irda_dev->data_bit_mask);
                    p_data16bits++;
                } else {
                    *p_data8bits = (uint8_t)(hals_usart_data_receive(irda_dev->periph) & (uint8_t)irda_dev->data_bit_mask);
                    p_data8bits++;
                }

                /* change the receive pointer */
                irda_dev->rxbuffer.remain--;
            } else {
                /* do nothing */
            }
        }

        /* change the Rx state to ready */
        irda_dev->rx_state = HAL_IRDA_STATE_READY;
    }

    /* unlock irda */
    HAL_UNLOCK(irda_dev);

    return ret;
}

/*!
    \brief      transmit amounts of data by interrupt method,the function is non-blocking
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to data buffer
    \param[in]  length: number of data to be transmitted
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_transmit_interrupt(hal_irda_dev_struct *irda_dev, uint8_t *p_buffer, uint32_t length, \
                                    hal_irda_irq_user_callback_struct *p_func)
{
    int32_t  ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == irda_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [irda_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock irda */
    HAL_LOCK(irda_dev);

    /* check the tx_state whether is busy or not */
    if(HAL_IRDA_STATE_READY != irda_dev->tx_state) {
        HAL_DEBUGE("irda tx has already been used, please wait until run_state change to free");
        ret                   = HAL_ERR_BUSY;
        irda_dev->error_state = (uint16_t)HAL_IRDA_ERROR_BUSY;
    } else {
        /* configure the state */
        irda_dev->tx_state    = HAL_IRDA_STATE_BUSY_TX;
        irda_dev->error_state = HAL_IRDA_ERROR_NONE;

        /* initialize transmit parameters */
        irda_dev->txbuffer.buffer = p_buffer;
        irda_dev->txbuffer.length = length;
        irda_dev->txbuffer.remain = length;

        /* configure the transmit callback as the function implemented */
        irda_dev->irda_irq.transmit_fifo_empty_handle = _irda_transmit_fifo_empty_interrupt;
        irda_dev->irda_irq.transmit_complete_handle   = _irda_transmit_complete_interrupt;
        irda_dev->irda_irq.error_handle               = _irda_error_interrupt;

        irda_dev->transmit_complete_callback = NULL;
        irda_dev->error_callback             = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->transmit_complete_func) {
                irda_dev->transmit_complete_callback = (void *)p_func->transmit_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                irda_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear IRDA TC interrupt flag */
        hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_TC);

        /* enable the TBE interrupt */
        hals_usart_interrupt_enable(irda_dev->periph, USART_INT_TBE);
    }

    /* unlock irda */
    HAL_UNLOCK(irda_dev);

    return ret;
}

/*!
    \brief      receive amounts of data by interrupt method,the function is non-blocking
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  length: number of data to be received
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] p_buffer: pointer to data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_receive_interrupt(hal_irda_dev_struct *irda_dev, uint8_t *p_buffer, uint32_t length, \
                                   hal_irda_irq_user_callback_struct *p_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == irda_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [irda_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock irda */
    HAL_LOCK(irda_dev);

    /* check the rx_state whether is busy or not */
    if(HAL_IRDA_STATE_READY != irda_dev->rx_state) {
        HAL_DEBUGE("irda rx has already been used, please wait until run_state change to free ");
        ret                   = HAL_ERR_BUSY;
        irda_dev->error_state = (uint16_t)HAL_IRDA_ERROR_BUSY;
    } else {
        /* configure the state */
        irda_dev->rx_state    = HAL_IRDA_STATE_BUSY_RX;
        irda_dev->error_state = HAL_IRDA_ERROR_NONE;

        /* initialize receive parameters */
        irda_dev->rxbuffer.buffer = p_buffer;
        irda_dev->rxbuffer.length = length;
        irda_dev->rxbuffer.remain = length;

        /* configure the receive complete callback as the function implemented */
        irda_dev->irda_irq.receive_complete_handle = _irda_receive_interrupt;
        irda_dev->irda_irq.error_handle            = _irda_error_interrupt;

        irda_dev->receive_complete_callback = NULL;
        irda_dev->error_callback            = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->receive_complete_func) {
                irda_dev->receive_complete_callback = (void *)p_func->receive_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                irda_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* enable PERR, ERR, RBNE interrupt */
        if(USART_PARITY_NONE != irda_dev->init.parity) {
            /* Enable the IRDA Parity Error and Data Register not empty Interrupts */
            hals_usart_interrupt_enable(irda_dev->periph, USART_INT_PERR);
        } else {
            /* do nothing */
        }

        hals_usart_interrupt_enable(irda_dev->periph, USART_INT_RBNE);
        hals_usart_interrupt_enable(irda_dev->periph, USART_INT_ERR);
    }

    /* unlock irda */
    HAL_UNLOCK(irda_dev);

    return ret;
}

/*!
    \brief      transmit amounts of data by dma method,the function is non-blocking
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to data buffer
    \param[in]  length: number of data to be transmitted
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_transmit_dma(hal_irda_dev_struct *irda_dev, uint8_t *p_buffer, uint16_t length, \
                              hal_irda_irq_user_callback_struct *p_func)
{
    hal_dma_irq_struct dma_irq = {0};
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == irda_dev) || (NULL == p_buffer) || (NULL == irda_dev->p_dma_tx)) {
        HAL_DEBUGE("pointer [irda_dev] or [p_buffer] or [irda_dev->p_dma_tx] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock irda */
    HAL_LOCK(irda_dev);

    /* check the tx_state whether is busy or not */
    if(HAL_IRDA_STATE_READY != irda_dev->tx_state) {
        HAL_DEBUGE("irda tx has already been used, please wait until run_state change to free");
        ret                   = HAL_ERR_BUSY;
        irda_dev->error_state = (uint16_t)HAL_IRDA_ERROR_BUSY;
    } else {
        /* configure the state */
        irda_dev->tx_state    = HAL_IRDA_STATE_BUSY_TX;
        irda_dev->error_state = HAL_IRDA_ERROR_NONE;

        /* initialize transmit parameters */
        irda_dev->txbuffer.buffer = p_buffer;
        irda_dev->txbuffer.length = length;
        irda_dev->txbuffer.remain = length;

        /* configure DMA interrupt callback function */
        dma_irq.half_finish_handle = _irda_dma_tx_halfcomplete;
        dma_irq.full_finish_handle = _irda_dma_tx_complete;
        dma_irq.error_handle       = _irda_dma_error;

        irda_dev->transmit_half_complete_callback = NULL;
        irda_dev->transmit_complete_callback      = NULL;
        irda_dev->error_callback                  = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->transmit_half_complete_func) {
                irda_dev->transmit_half_complete_callback = (void *)p_func->transmit_half_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->transmit_complete_func) {
                irda_dev->transmit_complete_callback = (void *)p_func->transmit_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                irda_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* start DMA interrupt mode transfer */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(irda_dev->p_dma_tx, (uint32_t)irda_dev->txbuffer.buffer, \
                                                   (uint32_t)&USART_TDATA(irda_dev->periph), \
                                                   (uint16_t)irda_dev->txbuffer.length, &dma_irq)) {
            HAL_DEBUGE("dma start interrupt failed");
            ret                   = HAL_ERR_BUSY;
            irda_dev->error_state = (uint16_t)HAL_IRDA_ERROR_DMA;
        } else {
            /* clear IRDA TC interrupt flag */
            hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_TC);

            /* DMA enable for transmission */
            hals_usart_dma_transmit_config(irda_dev->periph, USART_TRANSMIT_DMA_ENABLE);
        }

        /* change the tx state to ready */
        irda_dev->tx_state = HAL_IRDA_STATE_READY;
    }

    /* unlock irda */
    HAL_UNLOCK(irda_dev);

    return ret;
}

/*!
    \brief      receive amounts of data by dma method,the function is non-blocking
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  length: number of data to be received
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] p_buffer: pointer to data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_receive_dma(hal_irda_dev_struct *irda_dev, uint8_t *p_buffer, uint16_t length,
                             hal_irda_irq_user_callback_struct *p_func)
{
    hal_dma_irq_struct dma_irq = {0};
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == irda_dev) || (NULL == p_buffer) || (NULL == irda_dev->p_dma_rx)) {
        HAL_DEBUGE("pointer [irda_dev] or [p_buffer] or [irda_dev->p_dma_rx] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock irda */
    HAL_LOCK(irda_dev);

    /* check the rx_state whether is busy or not */
    if(HAL_IRDA_STATE_READY != irda_dev->rx_state) {
        HAL_DEBUGE("irda rx has already been used, please wait until run_state change to free ");
        ret                   = HAL_ERR_BUSY;
        irda_dev->error_state = (uint16_t)HAL_IRDA_ERROR_BUSY;
    } else {
        /* configure the state */
        irda_dev->rx_state    = HAL_IRDA_STATE_BUSY_RX;
        irda_dev->error_state = HAL_IRDA_ERROR_NONE;

        /* initialize receive parameters */
        irda_dev->rxbuffer.buffer = p_buffer;
        irda_dev->rxbuffer.length = length;
        irda_dev->rxbuffer.remain = length;

        /* configure DMA interrupt interrupt callback function */
        dma_irq.half_finish_handle = _irda_dm_rx_halfcomplete;
        dma_irq.full_finish_handle = _irda_dma_rx_complete;
        dma_irq.error_handle       = _irda_dma_error;

        irda_dev->receive_half_complete_callback = NULL;
        irda_dev->receive_complete_callback      = NULL;
        irda_dev->error_callback                 = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->receive_half_complete_func) {
                irda_dev->receive_half_complete_callback = (void *)p_func->receive_half_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->receive_complete_func) {
                irda_dev->receive_complete_callback = (void *)p_func->receive_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                irda_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* start DMA interrupt mode transfer */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(irda_dev->p_dma_rx, (uint32_t)&USART_RDATA(irda_dev->periph), \
                                                   (uint32_t)irda_dev->rxbuffer.buffer, \
                                                   (uint16_t)irda_dev->rxbuffer.length, &dma_irq)) {
            HAL_DEBUGE("dma start interrupt failed");
            ret                   = HAL_ERR_BUSY;
            irda_dev->error_state = (uint16_t)HAL_IRDA_ERROR_DMA;
        } else {
            /* enable the usart parity error and error interrupt: (frame error, noise error, overrun error) */
            if(USART_PARITY_NONE != irda_dev->init.parity) {
                hals_usart_interrupt_enable(irda_dev->periph, USART_INT_PERR);
            } else {
                /* do nothing */
            }
            hals_usart_interrupt_enable(irda_dev->periph, USART_INT_ERR);

            /* DMA enable for reception */
            hals_usart_dma_receive_config(irda_dev->periph, USART_RECEIVE_DMA_ENABLE);
        }

        /* change the rx state to ready */
        irda_dev->rx_state = HAL_IRDA_STATE_READY;
    }

    /* unlock irda */
    HAL_UNLOCK(irda_dev);

    return ret;
}

/*!
    \brief      pause irda DMA transfer during transmission process
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_dma_pause(hal_irda_dev_struct *irda_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == irda_dev) {
        HAL_DEBUGE("pointer [irda_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock irda */
    HAL_LOCK(irda_dev);

    /* check the tx state whether is busy or not */
    if(((USART_CTL2(irda_dev->periph) & USART_CTL2_DENT) == USART_CTL2_DENT) && \
       (HAL_IRDA_STATE_BUSY_TX == irda_dev->tx_state)) {
        /* disable DMA transmit */
        hals_usart_dma_transmit_config(irda_dev->periph, USART_TRANSMIT_DMA_DISABLE);
    } else {
        /* do nothing */
    }

    /* check the rx state whether is busy or not */
    if(((USART_CTL2(irda_dev->periph) & USART_CTL2_DENT) == USART_CTL2_DENT) && \
       (HAL_IRDA_STATE_BUSY_RX == irda_dev->rx_state)) {
        /* disable the PERR and ERR interrupt */
        hals_usart_interrupt_disable(irda_dev->periph, USART_INT_PERR);
        hals_usart_interrupt_disable(irda_dev->periph, USART_INT_ERR);

        /* disable DMA receive */
        hals_usart_dma_receive_config(irda_dev->periph, USART_RECEIVE_DMA_DISABLE);
    } else {
        /* do nothing */
    }

    /* unlock irda */
    HAL_UNLOCK(irda_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      resume irda DMA transfer during transmission process
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_dma_resume(hal_irda_dev_struct *irda_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == irda_dev) {
        HAL_DEBUGE("pointer [irda_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock irda */
    HAL_LOCK(irda_dev);

    if(HAL_IRDA_STATE_BUSY_TX == irda_dev->tx_state) {
        /* enable DMA transmit */
        hals_usart_dma_transmit_config(irda_dev->periph, USART_TRANSMIT_DMA_ENABLE);
    } else {
        /* do nothing */
    }

    if(HAL_IRDA_STATE_BUSY_RX == irda_dev->rx_state) {
        /* clear the overrun flag before resuming the rx transfer*/
        hals_usart_flag_clear(irda_dev->periph, USART_FLAG_ORERR);

        if(USART_PARITY_NONE != irda_dev->init.parity) {
            /* enable the PERR and ERR interrupt */
            hals_usart_interrupt_enable(irda_dev->periph, USART_INT_PERR);
        } else {
            /* do nothing */
        }
        hals_usart_interrupt_enable(irda_dev->periph, USART_INT_ERR);

        /* enable DMA transmit & receive */
        hals_usart_dma_receive_config(irda_dev->periph, USART_RECEIVE_DMA_ENABLE);
    } else {
        /* do nothing */
    }

    /* unlock irda */
    HAL_UNLOCK(irda_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop irda DMA transfer during transmission process
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_dma_stop(hal_irda_dev_struct *irda_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == irda_dev) {
        HAL_DEBUGE("pointer [irda_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock irda */
    HAL_LOCK(irda_dev);

    if(((USART_CTL2(irda_dev->periph) & USART_CTL2_DENT) == USART_CTL2_DENT) && \
       (HAL_IRDA_STATE_BUSY_TX == irda_dev->tx_state)) {
        hals_usart_dma_transmit_config(irda_dev->periph, USART_TRANSMIT_DMA_DISABLE);
        /* abort the usart dma tx channel */
        if(NULL != irda_dev->p_dma_tx) {
            hal_dma_stop(irda_dev->p_dma_tx);
        } else {
            /* do nothing */
        }

        hals_usart_interrupt_disable(irda_dev->periph, USART_INT_TBE);
        hals_usart_interrupt_disable(irda_dev->periph, USART_INT_TC);

        irda_dev->tx_state = HAL_IRDA_STATE_READY;
    } else {
        /* do nothing */
    }

    if(((USART_CTL2(irda_dev->periph) & USART_CTL2_DENT) == USART_CTL2_DENT) && \
       (HAL_IRDA_STATE_BUSY_RX == irda_dev->rx_state)) {
        hals_usart_dma_receive_config(irda_dev->periph, USART_RECEIVE_DMA_DISABLE);
        /* abort the usart dma rx channel */
        if(NULL != irda_dev->p_dma_rx) {
            hal_dma_stop(irda_dev->p_dma_rx);
        } else {
            /* do nothing */
        }

        hals_usart_interrupt_disable(irda_dev->periph, USART_INT_RBNE);
        hals_usart_interrupt_disable(irda_dev->periph, USART_INT_PERR);
        hals_usart_interrupt_disable(irda_dev->periph, USART_INT_ERR);

        irda_dev->rx_state = HAL_IRDA_STATE_READY;
    } else {
        /* do nothing */
    }

    /* unlock irda */
    HAL_UNLOCK(irda_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort irda ongoing transfer the function is blocking
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_abort(hal_irda_dev_struct *irda_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == irda_dev) {
        HAL_DEBUGE("pointer [irda_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock irda */
    HAL_LOCK(irda_dev);

    /* disable the TBE, TC, RBNE, PERR and ERR interrupt */
    hals_usart_interrupt_disable(irda_dev->periph, USART_INT_RBNE);
    hals_usart_interrupt_disable(irda_dev->periph, USART_INT_TBE);
    hals_usart_interrupt_disable(irda_dev->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(irda_dev->periph, USART_INT_TC);
    hals_usart_interrupt_disable(irda_dev->periph, USART_INT_ERR);

    /* disable DMA transmit and stop DMA */
    if((USART_CTL2(irda_dev->periph) & USART_CTL2_DENT) == USART_CTL2_DENT) {
        hals_usart_dma_transmit_config(irda_dev->periph, USART_TRANSMIT_DMA_DISABLE);
        if(NULL != irda_dev->p_dma_tx) {
            hal_dma_stop(irda_dev->p_dma_tx);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* disable DMA transmit and stop DMA */
    if((USART_CTL2(irda_dev->periph) & USART_CTL2_DENR) == USART_CTL2_DENR) {
        hals_usart_dma_receive_config(irda_dev->periph, USART_RECEIVE_DMA_DISABLE);
        if(NULL != irda_dev->p_dma_rx) {
            hal_dma_stop(irda_dev->p_dma_rx);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* reset the tx and rx buffer */
    irda_dev->txbuffer.remain = 0U;
    irda_dev->rxbuffer.remain = 0U;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_PERR);
    hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_ERR_FERR);
    hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_ERR_NERR);
    hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_ERR_ORERR);

    /* reset the tx and rx state */
    irda_dev->tx_state    = HAL_IRDA_STATE_READY;
    irda_dev->rx_state    = HAL_IRDA_STATE_READY;
    irda_dev->error_state = HAL_IRDA_ERROR_NONE;

    /* unlock irda */
    HAL_UNLOCK(irda_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort irda transmit transfer the function is blocking
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_transmit_abort(hal_irda_dev_struct *irda_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == irda_dev) {
        HAL_DEBUGE("pointer [irda_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock irda */
    HAL_LOCK(irda_dev);

    /* disable the TBE, TC interrupt */
    hals_usart_interrupt_disable(irda_dev->periph, USART_INT_TBE);
    hals_usart_interrupt_disable(irda_dev->periph, USART_INT_TC);

    /* disable DMA transmit and stop DMA */
    if((USART_CTL2(irda_dev->periph) & USART_CTL2_DENT) == USART_CTL2_DENT) {
        hals_usart_dma_transmit_config(irda_dev->periph, USART_TRANSMIT_DMA_DISABLE);
        if(NULL != irda_dev->p_dma_tx) {
            hal_dma_stop(irda_dev->p_dma_tx);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* reset the position and state */
    irda_dev->txbuffer.remain = 0U;
    irda_dev->tx_state        = HAL_IRDA_STATE_READY;

    /* unlock irda */
    HAL_UNLOCK(irda_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort irda receive transfer the function is blocking
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_irda_receive_abort(hal_irda_dev_struct *irda_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == irda_dev) {
        HAL_DEBUGE("pointer [irda_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock irda */
    HAL_LOCK(irda_dev);

    /* disable the RBNE, PERR and ERR interrupt */
    hals_usart_interrupt_disable(irda_dev->periph, USART_INT_RBNE);
    hals_usart_interrupt_disable(irda_dev->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(irda_dev->periph, USART_INT_ERR);

    /* disable DMA receive and stop DMA */
    if((USART_CTL2(irda_dev->periph) & USART_CTL2_DENR) == USART_CTL2_DENR) {
        hals_usart_dma_receive_config(irda_dev->periph, USART_RECEIVE_DMA_DISABLE);
        if(NULL != irda_dev->p_dma_rx) {
            hal_dma_stop(irda_dev->p_dma_rx);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* reset the position and state */
    irda_dev->rxbuffer.remain = 0U;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_PERR);
    hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_ERR_FERR);
    hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_ERR_NERR);
    hals_usart_interrupt_flag_clear(irda_dev->periph, USART_INT_FLAG_ERR_ORERR);

    irda_dev->rx_state = HAL_IRDA_STATE_READY;

    /* unlock irda */
    HAL_UNLOCK(irda_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      get irda transmit state
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     irda transmit state, refer to hal_irda_run_state_enum for details
*/
hal_irda_run_state_enum hal_irda_tx_state_get(hal_irda_dev_struct *irda_dev)
{
    return irda_dev->tx_state;
}

/*!
    \brief      get irda receive state
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     irda receive state, refer to hal_irda_run_state_enum for details
*/
hal_irda_run_state_enum hal_irda_rx_state_get(hal_irda_dev_struct *irda_dev)
{
    return irda_dev->rx_state;
}

/*!
    \brief      get irda error state
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     irda error state: 0x0 - 0xFFFF
*/
uint16_t hal_irda_error_state_get(hal_irda_dev_struct *irda_dev)
{
    return irda_dev->error_state;
}

/*!
    \brief      reset USARTx(x=0,1,2,5)
    \param[in]  irda_periph: USARTx(x=0,1,2,5)
    \param[out] none
    \retval     none
*/
static void _irda_deinit(uint32_t irda_periph)
{
    switch(irda_periph) {
    case USART0:
        /* reset USART0 */
        hal_rcu_periph_reset_enable(RCU_USART0RST);
        hal_rcu_periph_reset_disable(RCU_USART0RST);
        break;
    case USART1:
        /* reset USART1 */
        hal_rcu_periph_reset_enable(RCU_USART1RST);
        hal_rcu_periph_reset_disable(RCU_USART1RST);
        break;
    case USART2:
        /* reset USART2 */
        hal_rcu_periph_reset_enable(RCU_USART2RST);
        hal_rcu_periph_reset_disable(RCU_USART2RST);
        break;
    case USART5:
        /* reset USART5 */
        hal_rcu_periph_reset_enable(RCU_USART5RST);
        hal_rcu_periph_reset_disable(RCU_USART5RST);
        break;
    default:
        break;
    }
}

/*!
    \brief      get the mask of data bit
    \param[in]  irda_dev: irda device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     the mask of data bit(0x3F, 0x7F, 0xFF, 0x1FF, 0x3FF)
*/
static uint16_t _irda_data_bit_mask_get(hal_irda_dev_struct *irda_dev)
{
    uint16_t retval;

    /* check whether the PCEN is enabled */
    if(RESET != (USART_CTL0(irda_dev->periph) & USART_CTL0_PCEN)) {
        if((USART_CTL0(irda_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1)) == USART_WORDLENGTH_8B) {
            retval = 0x7FU;
        } else if((USART_CTL0(irda_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1)) == USART_WORDLENGTH_9B) {
            retval = 0xFFU;
        } else if((USART_CTL0(irda_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1)) == USART_WORDLENGTH_7B) {
            retval = 0x3FU;
        } else {
            retval = 0x1FFU;
        }
    } else {
        if((USART_CTL0(irda_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1)) == USART_WORDLENGTH_8B) {
            retval = 0xFFU;
        } else if((USART_CTL0(irda_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1)) == USART_WORDLENGTH_9B) {
            retval = 0x1FFU;
        } else if((USART_CTL0(irda_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1)) == USART_WORDLENGTH_7B) {
            retval = 0x7FU;
        } else {
            retval = 0x3FFU;
        }
    }

    return retval;
}

/*!
    \brief      handle the transmit complete interrupt
    \param[in]  irda_dev: pointer to a irda device information structure
    \param[out] none
    \retval     none
*/
static void _irda_transmit_complete_interrupt(void *irda_dev)
{
    hal_irda_dev_struct *p_irda = irda_dev;
    hal_irda_user_cb p_func     = (hal_irda_user_cb)p_irda->transmit_complete_callback;

    /* disable the transmit complete interrupt */
    hals_usart_interrupt_disable(p_irda->periph, USART_INT_TC);

    /* reset transmit_complete_handle and tx_state */
    p_irda->tx_state = HAL_IRDA_STATE_READY;
    p_irda->irda_irq.transmit_complete_handle = NULL;

    if(NULL != p_func) {
        /* if there is a user transmit complete callback */
        p_func(p_irda);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the transmit interrupt
    \param[in]  irda_dev: pointer to a irda device information structure
    \param[out] none
    \retval     none
*/
static void _irda_transmit_fifo_empty_interrupt(void *irda_dev)
{
    uint16_t *tmp;
    hal_irda_dev_struct *p_irda = irda_dev;

    if(p_irda->tx_state == HAL_IRDA_STATE_BUSY_TX) {
        if(0U != p_irda->txbuffer.remain) {
            if((USART_WORDLENGTH_10B == p_irda->init.wordlength) || \
               ((USART_WORDLENGTH_9B == p_irda->init.wordlength) && (USART_PARITY_NONE == p_irda->init.parity))) {
                /* 9-bit data, none parity and 10-bit data */
                tmp = (uint16_t *)(uint32_t)p_irda->txbuffer.buffer;
                hals_usart_data_transmit(p_irda->periph, (uint16_t)(*tmp & 0x03FFU));
                p_irda->txbuffer.buffer += 2;
            } else {
                /* 9-bit data, with parity or 8-bit data */
                hals_usart_data_transmit(p_irda->periph, (uint16_t)((uint16_t)*p_irda->txbuffer.buffer & 0x000000FFU));
                p_irda->txbuffer.buffer++;
            }
            p_irda->txbuffer.remain--;
        } else {
            /* disable the TBE interrupt, enable the TC interrupt and reset the transmit_fifo_empty_handle */
            hals_usart_interrupt_disable(p_irda->periph, USART_INT_TBE);
            hals_usart_interrupt_enable(p_irda->periph, USART_INT_TC);
            p_irda->irda_irq.transmit_fifo_empty_handle = NULL;
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the receive interrupt
    \param[in]  irda_dev: pointer to a irda device information structure
    \param[out] none
    \retval     none
*/
static void _irda_receive_interrupt(void *irda_dev)
{
    hal_irda_dev_struct *p_irda = irda_dev;
    hal_irda_user_cb p_func     = (hal_irda_user_cb)p_irda->receive_complete_callback;

    if(HAL_IRDA_STATE_BUSY_RX == p_irda->rx_state) {
        /* store the received data */
        if((USART_WORDLENGTH_10B == p_irda->init.wordlength) || \
           ((USART_WORDLENGTH_9B == p_irda->init.wordlength) && (USART_PARITY_NONE == p_irda->init.parity))) {
            *(uint16_t *)(uint32_t)p_irda->rxbuffer.buffer = (hals_usart_data_receive(p_irda->periph) & \
                                                                                      p_irda->data_bit_mask);
            p_irda->rxbuffer.buffer += 2U;
        } else {
            *p_irda->rxbuffer.buffer = (uint8_t)(hals_usart_data_receive(p_irda->periph) & p_irda->data_bit_mask);
            p_irda->rxbuffer.buffer++;
        }
        p_irda->rxbuffer.remain--;

        if(0U == p_irda->rxbuffer.remain) {
            /* disable PERR, ERR, RBNE interrupt */
            hals_usart_interrupt_disable(p_irda->periph, USART_INT_PERR);
            hals_usart_interrupt_disable(p_irda->periph, USART_INT_ERR);
            hals_usart_interrupt_disable(p_irda->periph, USART_INT_RBNE);

            /* reset rx_state */
            p_irda->rx_state = HAL_IRDA_STATE_READY;
            p_irda->irda_irq.receive_complete_handle = NULL;

            if(NULL != p_func) {
                /* if there is a user receive complete callback */
                p_func(p_irda);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        hals_usart_command_enable(p_irda->periph, USART_CMD_RXFCMD);
    }
}

/*!
    \brief      handle the error interrupt
    \param[in]  irda_dev: pointer to a irda device information structure
    \param[out] none
    \retval     none
*/
static void _irda_error_interrupt(void *irda_dev)
{
    hal_irda_dev_struct *p_irda = irda_dev;
    hal_irda_user_cb p_func     = (hal_irda_user_cb)p_irda->error_callback;

    p_irda->error_state = HAL_IRDA_ERROR_NONE;
    p_irda->irda_irq.error_handle = NULL;

    /* if there is a user receive complete callback */
    if(NULL != p_func) {
        p_func(p_irda);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the irda DMA transmit half complete process
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _irda_dma_tx_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_irda_dev_struct *p_irda;
    hal_irda_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_irda = (hal_irda_dev_struct *)p_dma->p_periph;
    p_func = (hal_irda_user_cb)p_irda->transmit_complete_callback;

    /* DMA normal mode */
    if(0U == (DMA_CHCTL(p_dma->dma_periph, p_dma->channel) & DMA_CHXCTL_CMEN)) {
        p_irda->txbuffer.remain = 0U;
        hals_usart_dma_transmit_config(p_irda->periph, USART_TRANSMIT_DMA_DISABLE);
        /* enable TC interrupt */
        hals_usart_interrupt_enable(p_irda->periph, USART_INT_TC);
    } else {
        if(NULL != p_func) {
            /* if there is a user transmit complete callback */
            p_func(p_irda);
        } else {
            /* do nothing */
        }
    }
}

/*!
    \brief      handle the irda DMA transmit half complete process
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _irda_dma_tx_halfcomplete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_irda_dev_struct *p_irda;
    hal_irda_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_irda = (hal_irda_dev_struct *)p_dma->p_periph;
    p_func = (hal_irda_user_cb)p_irda->transmit_half_complete_callback;

    /* if there is a user transmit halfcomplete callback */
    if(NULL != p_func) {
        p_func(p_irda);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the irda DMA receive complete process
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _irda_dma_rx_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_irda_dev_struct *p_irda;
    hal_irda_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_irda = (hal_irda_dev_struct *)p_dma->p_periph;
    p_func = (hal_irda_user_cb)p_irda->receive_complete_callback;

    /* DMA normal mode */
    if(0U == (DMA_CHCTL(p_dma->dma_periph, p_dma->channel) & DMA_CHXCTL_CMEN)) {
        p_irda->rxbuffer.remain = 0U;

        /* disable DMA receive, PERR and ERR interrupt */
        hals_usart_dma_receive_config(p_irda->periph, USART_RECEIVE_DMA_DISABLE);
        hals_usart_interrupt_disable(p_irda->periph, USART_INT_PERR);
        hals_usart_interrupt_disable(p_irda->periph, USART_INT_ERR);

        /* reset rx_state */
        p_irda->rx_state = HAL_IRDA_STATE_READY;
    } else {
        /* do nothing */
    }

    /* if there is a user receive complete callback */
    if(NULL != p_func) {
        p_func(p_irda);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the irda DMA receive half complete process
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _irda_dm_rx_halfcomplete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_irda_dev_struct *p_irda;
    hal_irda_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_irda = (hal_irda_dev_struct *)p_dma->p_periph;
    p_func = (hal_irda_user_cb)p_irda->receive_half_complete_callback;

    /* if there is a user receive complete callback */
    if(NULL != p_func) {
        p_func(p_irda);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the irda DMA error process
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _irda_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_irda_dev_struct *p_irda;
    hal_irda_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_irda = (hal_irda_dev_struct *)p_dma->p_periph;
    p_func = (hal_irda_user_cb)p_irda->error_callback;

    /* transmit state is busy */
    if(HAL_IRDA_STATE_BUSY_RX == p_irda->tx_state) {
        /* disable TC and TBE interrupt */
        hals_usart_interrupt_disable(p_irda->periph, USART_INT_TBE);
        hals_usart_interrupt_disable(p_irda->periph, USART_INT_TC);

        p_irda->tx_state        = HAL_IRDA_STATE_READY;
        p_irda->txbuffer.remain = 0U;
    } else {
        /* do nothing */
    }

    /* receive state is busy */
    if(HAL_IRDA_STATE_BUSY_TX == p_irda->rx_state) {
        /* disable PERR, ERR interrupt */
        hals_usart_interrupt_disable(p_irda->periph, USART_INT_RBNE);
        hals_usart_interrupt_disable(p_irda->periph, USART_INT_PERR);
        hals_usart_interrupt_disable(p_irda->periph, USART_INT_ERR);

        p_irda->rx_state        = HAL_IRDA_STATE_READY;
        p_irda->rxbuffer.remain = 0U;
    } else {
        /* do nothing */
    }

    /* reset IRDA state */
    p_irda->last_error = (uint16_t)HAL_IRDA_ERROR_DMA;
    p_irda->error_state |= (uint16_t)HAL_IRDA_ERROR_DMA;

    /* if there is a user error callback */
    if(NULL != p_func) {
        p_func(p_irda);
    } else {
        /* do nothing */
    }
}
