/*!
    \file    gd32h7xx_hal_fwdgt.c
    \brief   FWDGT driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/*!
    \brief      initialize the FWDGT structure with default values
    \param[in]  hal_struct_type: type of FWDGT structure for initialization
                only one parameter can be selected which is shown as below:
      \arg        HAL_FWDGT_INIT_STRUCT: FWDGT initialization structure
      \arg        HAL_FWDGT_DEV_STRUCT: FWDGT device information structure
    \param[out] p_struct: pointer to FWDGT structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_fwdgt_struct_init(hal_fwdgt_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_FWDGT_INIT_STRUCT:
        /* initialize WWDGT initialization structure with the default values */
        ((hal_fwdgt_init_struct *)p_struct)->fwdgt_pre_select       = FWDGT_PSC_DIV4;
        ((hal_fwdgt_init_struct *)p_struct)->fwdgt_cnt_window_value = 0xFFFU;
        ((hal_fwdgt_init_struct *)p_struct)->fwdgt_cnt_reload_value = 0xFFFU;
        break;
    case HAL_FWDGT_DEV_STRUCT:
        /* initialize WWDGT DEV structure with the default values */
        ((hal_fwdgt_dev_struct *)p_struct)->state = HAL_FWDGT_STATE_NONE;
        ((hal_fwdgt_dev_struct *)p_struct)->mutex = HAL_MUTEX_UNLOCKED;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      initialize FWDGT
    \param[in]  fwdgt_dev: FWDGT device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_fwdgt_init: pointer to a hal_fwdgt_init_struct structure which contains
                parameters for initialization of the FWDGT peripheral
                members of the structure and the member values are shown as below:
                fwdgt_pre_select: free Watchdog Timer prescaler selection
                only one parameter can be selected which is shown as below:
      \arg        FWDGT_PSC_DIV4  ： FWDGT prescaler set to 4
      \arg        FWDGT_PSC_DIV8  ： FWDGT prescaler set to 8
      \arg        FWDGT_PSC_DIV16 ： FWDGT prescaler set to 16
      \arg        FWDGT_PSC_DIV32 ： FWDGT prescaler set to 32
      \arg        FWDGT_PSC_DIV64 ： FWDGT prescaler set to 64
      \arg        FWDGT_PSC_DIV128： FWDGT prescaler set to 128
      \arg        FWDGT_PSC_DIV256： FWDGT prescaler set to 256
               fwdgt_cnt_reload_value: 0x0 - 0xFFF
               wdgt_cnt_value: 0x0 - 0xFFF, window function is disabled when the value is 0xFFF
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_TIMEOUT, HAL_ERR_NONE, details refer to gd32h7x0_hal.h
*/
int32_t hal_fwdgt_init(hal_fwdgt_dev_struct *fwdgt_dev, hal_fwdgt_init_struct *p_fwdgt_init)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_fwdgt_init) {
        HAL_DEBUGE("pointer [p_fwdgt_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(p_fwdgt_init->fwdgt_cnt_window_value < p_fwdgt_init->fwdgt_cnt_reload_value) {
        HAL_DEBUGE("fwdgt window value is smaller than reload value, it will lead to system reset!");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    fwdgt_dev->state = HAL_FWDGT_STATE_BUSY;

    hals_fwdgt_write_enable();

    /* configure FWDGT prescaler and reload value */
    if(HAL_ERR_TIMEOUT == hals_fwdgt_config(p_fwdgt_init->fwdgt_cnt_reload_value, \
                                         (uint32_t)p_fwdgt_init->fwdgt_pre_select)) {
        HAL_DEBUGE("hals_fwdgt_config timeout");
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* do nothing */
    }

    /* configure FWDGT window value */
    if(HAL_ERR_TIMEOUT == hals_fwdgt_window_value_config(p_fwdgt_init->fwdgt_cnt_window_value)) {
        HAL_DEBUGE("hals_fwdgt_window_value_config timeout");
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* do nothing */
    }

    hals_fwdgt_write_disable();

    return ret;
}

/*!
    \brief      deinitialize FWDGT
    \param[in]  fwdgt_dev: FWDGT device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_fwdgt_deinit(hal_fwdgt_dev_struct *fwdgt_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == fwdgt_dev) {
        HAL_DEBUGE("pointer [*fwdgt_dev] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* reset FWDGT register value */
    FWDGT_CTL = FWDGT_WRITEACCESS_ENABLE;
    FWDGT_PSC = 0x00000000U;
    FWDGT_CTL = FWDGT_WRITEACCESS_ENABLE;
    FWDGT_RLD = 0x00000000U;
    FWDGT_CTL = 0x00000000U;
    FWDGT_WND = 0x00000FFFU;

    fwdgt_dev->state = HAL_FWDGT_STATE_RESET;

    return HAL_ERR_NONE;
}

/*!
    \brief      reload the counter of FWDGT
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_fwdgt_reload(void)
{
    FWDGT_CTL = FWDGT_KEY_RELOAD;
}

/*!
    \brief      start FWDGT module function
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_fwdgt_start(void)
{
    FWDGT_CTL = FWDGT_KEY_ENABLE;
}

/*!
    \brief      enable write access to FWDGT_PSC and FWDGT_RLD and FWDGT_WND
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_fwdgt_write_enable(void)
{
    FWDGT_CTL = FWDGT_WRITEACCESS_ENABLE;
}

/*!
    \brief      disable write access to FWDGT_PSC,FWDGT_RLD and FWDGT_WND
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_fwdgt_write_disable(void)
{
    FWDGT_CTL = FWDGT_WRITEACCESS_DISABLE;
}

/*!
    \brief      configure the free watchdog timer counter window value
    \param[in]  wdgt_cnt_value: specify window value(0x0000 - 0x0FFF)
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE, details refer to gd32h7x0_hal.h
*/
int32_t hals_fwdgt_window_value_config(uint32_t wdgt_cnt_value)
{
    __IO uint32_t tick_start = 0U;
    int32_t ret = HAL_ERR_NONE;

    /* enable write access to FWDGT_WND */
    FWDGT_CTL = FWDGT_WRITEACCESS_ENABLE;

    /* wait until the WUD flag to be reset */
    tick_start = hal_sys_basetick_count_get();
    while(RESET != (FWDGT_STAT & FWDGT_STAT_WUD)) {
        if(SET == hal_sys_basetick_timeout_check(tick_start, FWDGT_WND_TIMEOUT)) {
            HAL_DEBUGE("WUD flag was not reset within the specified time period");
            ret = HAL_ERR_TIMEOUT;
            break;
        } else {
            /* do nothing */
        }
    }

    FWDGT_WND = WND_WND(wdgt_cnt_value);

    return ret;
}

/*!
    \brief      configure counter reload value, and prescaler divider value
    \param[in]  fwdgt_cnt_reload_value: specify reload value(0x0000 - 0x0FFF)
    \param[in]  fwdgt_pre_select: FWDGT prescaler value
                only one parameter can be selected which is shown as below:
      \arg        FWDGT_PSC_DIV4  : FWDGT prescaler set to 4
      \arg        FWDGT_PSC_DIV8  : FWDGT prescaler set to 8
      \arg        FWDGT_PSC_DIV16 : FWDGT prescaler set to 16
      \arg        FWDGT_PSC_DIV32 : FWDGT prescaler set to 32
      \arg        FWDGT_PSC_DIV64 : FWDGT prescaler set to 64
      \arg        FWDGT_PSC_DIV128: FWDGT prescaler set to 128
      \arg        FWDGT_PSC_DIV256: FWDGT prescaler set to 256
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE, details refer to gd32h7x0_hal.h
*/
int32_t hals_fwdgt_config(uint32_t fwdgt_cnt_reload_value, uint32_t fwdgt_pre_select)
{
    __IO uint32_t tick_start = 0U;
    int32_t ret = HAL_ERR_NONE;

    /* enable write access to FWDGT_PSC,and FWDGT_RLD */
    FWDGT_CTL = FWDGT_WRITEACCESS_ENABLE;

    /* wait until the PUD flag to be reset */
    tick_start = hal_sys_basetick_count_get();
    while(RESET != (FWDGT_STAT & FWDGT_STAT_PUD)) {
        if(SET == hal_sys_basetick_timeout_check(tick_start, FWDGT_PSC_TIMEOUT)) {
            HAL_DEBUGE("PUD flag was not reset within the specified time period");
            ret = HAL_ERR_TIMEOUT;
            break;
        } else {
            /* do nothing */
        }
    }

    /* configure FWDGT */
    FWDGT_PSC = (uint32_t)fwdgt_pre_select;

    /* wait until the RUD flag to be reset */
    tick_start = hal_sys_basetick_count_get();
    while(RESET != (FWDGT_STAT & FWDGT_STAT_RUD)) {
        if(SET == hal_sys_basetick_timeout_check(tick_start, FWDGT_RLD_TIMEOUT)) {
            HAL_DEBUGE("RUD flag was not reset within the specified time period");
            ret = HAL_ERR_TIMEOUT;
            break;
        } else {
            /* do nothing */
        }
    }

    FWDGT_RLD = RLD_RLD(fwdgt_cnt_reload_value);

    /* reload the counter */
    FWDGT_CTL = FWDGT_KEY_RELOAD;

    return ret;
}

/*!
    \brief      get flag state of FWDGT
    \param[in]  flag: flag to get
                only one parameter can be selected which is shown as below:
      \arg        FWDGT_FLAG_PUD: a write operation to FWDGT_PSC register is on going
      \arg        FWDGT_FLAG_RUD: a write operation to FWDGT_RLD register is on going
      \arg        FWDGT_FLAG_WUD: a write operation to FWDGT_WND register is on going
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_fwdgt_flag_get(uint16_t flag)
{
    FlagStatus ret = RESET;

    if(RESET != (FWDGT_STAT & flag)) {
        ret = SET;
    } else {
        /* do nothing */
    }

    return ret;
}
