/*!
    \file    gd32h7xx_hal_exmc_sram.c
    \brief   EXMC SRAM driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* EXMC SRAM private function */
static void _sram_dma_complete(void *dma);
static void _sram_dma_complete_protected(void *dma);
static void _sram_dma_error(void *dma);

/*!
    \brief      initialize EXMC NOR/SRAM region
    \param[in]  exmc_sram_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  sram_init: pointer to EXMC SRAM initialization structure
                This parameter contains following fields:
                norsram_region: select the region of EXMC SRAM region
                only one parameter can be selected which is shown as below:
      \arg        EXMC_BANK0_NORSRAM_REGION0: EXMC bank0 NOR/SRAM region0
      \arg        EXMC_BANK0_NORSRAM_REGION1: EXMC bank0 NOR/SRAM region1
      \arg        EXMC_BANK0_NORSRAM_REGION2: EXMC bank0 NOR/SRAM region2
      \arg        EXMC_BANK0_NORSRAM_REGION3: EXMC bank0 NOR/SRAM region3
                bank_remap: select EXMC NOR/PSRAM remap region
                only one parameter can be selected which is shown as below:
      \arg        EXMC_BANK_REMAP_DEFAULT: default bank remap configuration
      \arg        EXMC_BANK_NORPSRAM_SDRAM_SWAP: NOR/PSRAM and SDRAM bank swap
                consecutive_clock:configure consecutive clock mode
                only one parameter can be selected which is shown as below:
      \arg        EXMC_CLOCK_SYN_MODE: the clock is generated only during synchronous access
      \arg        EXMC_CLOCK_UNCONDITIONALLY: the clock is generated unconditionally
                asyn_access_mode: asynchronous access mode
                only one parameter can be selected which is shown as below:
      \arg        EXMC_ACCESS_MODE_A: access mode A
      \arg        EXMC_ACCESS_MODE_B: access mode B
      \arg        EXMC_ACCESS_MODE_C: access mode C
      \arg        EXMC_ACCESS_MODE_D: access mode D
                syn_data_latency: configure the data latency
                only one parameter can be selected which is shown as below:
      \arg        EXMC_DATALAT_x_CLK(x=2..17): data latency of first burst access is x EXMC_CLK
                syn_clk_division: configure the clock divide ratio
                only one parameter can be selected which is shown as below:
      \arg        EXMC_SYN_CLOCK_RATIO_DISABLE: clock divide ratio disable
      \arg        EXMC_SYN_CLOCK_RATIO_x_CLK(x=2..16): EXMC_CLK period = x*CK_EXMC period
                asyn_address_holdtime: 0 - 15, configure the address hold time
                bus_latency: 0 - 15, configure the bus latency
                asyn_address_setuptime: 1 - 15, configure the address setup time
                asyn_data_setuptime: 1 - 255, configure the data setup time
                write_asyn_access_mode: write asynchronous access mode
                only one parameter can be selected which is shown as below:
      \arg        EXMC_ACCESS_MODE_A: access mode A
      \arg        EXMC_ACCESS_MODE_B: access mode B
      \arg        EXMC_ACCESS_MODE_C: access mode C
      \arg        EXMC_ACCESS_MODE_D: access mode D
                write_asyn_address_setuptime: 1 - 15, write configure the address setup time
                write_bus_latency: 0 - 15, write configure the bus latency
                write_asyn_data_setuptime: 1 - 255, write configure the data setup time
                write_asyn_address_holdtime: 0 - 15, write configure the address hold time
                write_mode: the write mode
                only one parameter can be selected which is shown as below:
      \arg        EXMC_ASYN_WRITE: asynchronous write mode
      \arg        EXMC_SYN_WRITE: synchronous write mode
                extended_mode: enable or disable the extended mode
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable the extended mode
      \arg        DISABLE: disable the extended mode
                asyn_wait: enable or disable the asynchronous wait function
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable the asynchronous wait function
      \arg        DISABLE: disable the asynchronous wait function
                nwait_signal: enable or disable the NWAIT signal while in synchronous bust mode
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable the NWAIT signal
      \arg        DISABLE: disable the NWAIT signal
                memory_write: enable or disable the write operation
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable the write operation
      \arg        DISABLE: disable the write operation
                nwait_config: ENWAIT signal configuration
                only one parameter can be selected which is shown as below:
      \arg        EXMC_NWAIT_CONFIG_BEFORE: NWAIT signal is active one clock cycle before the wait state
      \arg        EXMC_NWAIT_CONFIG_DURING: NWAIT signal is active during wait state
                nwait_polarity: specify the polarity of NWAIT signal from memory
                only one parameter can be selected which is shown as below:
      \arg        EXMC_NWAIT_POLARITY_LOW: NWAIT active low
      \arg        XMC_NWAIT_POLARITY_HIGH: NWAIT active high
                burst_mode: enable or disable the burst mode
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable the burst mode
      \arg        DISABLE: disable the burst mode
                databus_width: specify the data bus width of external memory
                only one parameter can be selected which is shown as below:
      \arg        EXMC_NOR_DATABUS_WIDTH_8B: NOR data width is 8 bits
      \arg        EXMC_NOR_DATABUS_WIDTH_16B: NOR data width is 16 bits
      \arg        EXMC_NOR_DATABUS_WIDTH_32B: NOR data width is 32 bits
                memory_type: specify the type of external memory
                only one parameter can be selected which is shown as below:
      \arg        EXMC_MEMORY_TYPE_SRAM: SRAM memory type
                address_data_mux: specify whether the data bus and address bus are multiplexed
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: address/data multiplexed
      \arg        DISABLE: address/data not multiplexed
                cram_page_size: cram page size
                only one parameter can be selected which is shown as below:
      \arg        EXMC_CRAM_AUTO_SPLIT: the clock is generated only during synchronous access
      \arg        EXMC_CRAM_PAGE_SIZE_128_BYTES: page size is 128 bytes
      \arg        EXMC_CRAM_PAGE_SIZE_256_BYTES: page size is 256 bytes
      \arg        EXMC_CRAM_PAGE_SIZE_512_BYTES: page size is 512 bytes
      \arg        EXMC_CRAM_PAGE_SIZE_1024_BYTES: page size is 1024 bytes
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sram_init(hal_exmc_sram_dev_struct *exmc_sram_dev, hal_exmc_sram_init_struct *sram_init)
{
    uint32_t snctl = 0U, sntcfg = 0U, snwtcfg = 0U;
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_sram_dev) || (NULL == sram_init)) {
        HAL_DEBUGE("parameter [exmc_sram_dev] or [sram_init ] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(exmc_sram_dev);
    exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_RESET;

    /* disable exmc sram region */
    EXMC_SNCTL(sram_init->norsram_region) &= (uint32_t)(~EXMC_SNCTL_NRBKEN);

    exmc_sram_dev->sram_region = sram_init->norsram_region;
    snctl   = EXMC_SNCTL(sram_init->norsram_region);
    sntcfg  = EXMC_SNTCFG(sram_init->norsram_region);
    snwtcfg = EXMC_SNWTCFG(sram_init->norsram_region);

    if(EXMC_BANK0_NORSRAM_REGION0 == sram_init->norsram_region) {
        snctl &= (uint32_t)(~(EXMC_SNCTL_BKREMAP | EXMC_SNCTL_CCK));
    } else {
        /* do nothing */
    }
    snctl &= (uint32_t)(~(EXMC_SNCTL_NREN | EXMC_SNCTL_NRTP | EXMC_SNCTL_NRW | EXMC_SNCTL_SBRSTEN | EXMC_SNCTL_NRWTPOL | \
                          EXMC_SNCTL_NRWTCFG | EXMC_SNCTL_WEN | EXMC_SNCTL_NRWTEN | EXMC_SNCTL_EXMODEN | \
                          EXMC_SNCTL_ASYNCWTEN | EXMC_SNCTL_SYNCWR | EXMC_SNCTL_NRMUX | EXMC_SNCTL_CPS));

    /* configure control register */
    snctl = (uint32_t)((sram_init->bank_remap & EXMC_SNCTL_BKREMAP) | (sram_init->consecutive_clock & EXMC_SNCTL_CCK) | \
                      ((sram_init->address_data_mux << SNCTL_NRMUX_OFFSET) & EXMC_SNCTL_NRMUX) | \
                      (sram_init->memory_type & EXMC_SNCTL_NRTP) | (sram_init->databus_width & EXMC_SNCTL_NRW) | \
                      ((sram_init->burst_mode << SNCTL_SBRSTEN_OFFSET) & EXMC_SNCTL_SBRSTEN) | \
                      (sram_init->nwait_polarity & EXMC_SNCTL_NRWTPOL) | (sram_init->nwait_config & EXMC_SNCTL_NRWTCFG) | \
                      ((sram_init->memory_write << SNCTL_WREN_OFFSET) & EXMC_SNCTL_WEN) | \
                      ((sram_init->nwait_signal << SNCTL_NRWTEN_OFFSET) & EXMC_SNCTL_NRWTEN) | \
                      ((sram_init->extended_mode << SNCTL_EXMODEN_OFFSET) & EXMC_SNCTL_EXMODEN) | \
                      ((sram_init->asyn_wait << SNCTL_ASYNCWAITEN_OFFSET) & EXMC_SNCTL_ASYNCWTEN) | \
                      (sram_init->cram_page_size & EXMC_SNCTL_CPS) | (sram_init->write_mode & EXMC_SNCTL_SYNCWR));

    /* nor flash access enable */
    if(EXMC_MEMORY_TYPE_NOR == sram_init->memory_type) {
        snctl |= (uint32_t)EXMC_SNCTL_NREN;
    } else {
        /* do nothing */
    }

    /* configure timing register */
    sntcfg = (uint32_t)((sram_init->asyn_access_mode & EXMC_SNTCFG_ASYNCMOD) | \
                        SNTCFG_SYN_DATA_LATENCY(sram_init->syn_data_latency) | \
                        SNTCFG_SYN_CLK_DIVISION(sram_init->syn_clk_division) | \
                        ((sram_init->asyn_address_holdtime << SNTCFG_AHLD_OFFSET) & EXMC_SNTCFG_AHLD) | \
                        ((sram_init->bus_latency << SNTCFG_BUSLAT_OFFSET) & EXMC_SNTCFG_BUSLAT) | \
                        SNTCFG_ASYN_ADDR_SETUPTIME(sram_init->asyn_address_setuptime) | \
                        ((sram_init->asyn_data_setuptime << SNTCFG_DSET_OFFSET) & EXMC_SNTCFG_DSET));

    /* for extended mode, configure write timing */
    if(ENABLE == sram_init->extended_mode) {
        snwtcfg = (uint32_t)((sram_init->write_asyn_access_mode & EXMC_SNWTCFG_WASYNCMOD) | \
                             SNTCFG_WRITE_ASYN_ADDR_SETUPTIME(sram_init->write_asyn_address_setuptime) | \
                             ((sram_init->write_bus_latency << SNTCFG_BUSLAT_OFFSET) & EXMC_SNWTCFG_WBUSLAT) | \
                             ((sram_init->write_asyn_data_setuptime << SNTCFG_DSET_OFFSET) & EXMC_SNWTCFG_WDSET) | \
                             ((sram_init->write_asyn_address_holdtime << SNTCFG_AHLD_OFFSET) & EXMC_SNWTCFG_WAHLD));
    } else {
        snwtcfg = BANK0_SNWTCFG_RESET;
    }

    EXMC_SNCTL(sram_init->norsram_region)   = snctl;
    EXMC_SNTCFG(sram_init->norsram_region)  = sntcfg;
    EXMC_SNWTCFG(sram_init->norsram_region) = snwtcfg;

    /* enable exmc sram region */
    EXMC_SNCTL(sram_init->norsram_region) |= (uint32_t)EXMC_SNCTL_NRBKEN;

    exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_READY;
    HAL_UNLOCK(exmc_sram_dev);


    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the SRAM structure with default values
    \param[in]  hal_struct_type: The type of the struct to initialize
                only one parameters can be selected which are shown as below:
      \arg        HAL_EXMC_SRAM_INIT_STRUCT: SRAM initialization structure
      \arg        HAL_EXMC_SRAM_DEV_STRUCT: SRAM device structure
    \param[out] p_struct: pointer to SRAM structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sram_struct_init(hal_exmc_sram_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    switch(hal_struct_type) {
    case HAL_EXMC_SRAM_INIT_STRUCT:
        ((hal_exmc_sram_init_struct *)p_struct)->norsram_region    = EXMC_BANK0_NORSRAM_REGION0;
        ((hal_exmc_sram_init_struct *)p_struct)->bank_remap        = EXMC_BANK_REMAP_DEFAULT;
        ((hal_exmc_sram_init_struct *)p_struct)->consecutive_clock = EXMC_CLOCK_SYN_MODE;
        ((hal_exmc_sram_init_struct *)p_struct)->address_data_mux  = ENABLE;
        ((hal_exmc_sram_init_struct *)p_struct)->memory_type       = EXMC_MEMORY_TYPE_NOR;
        ((hal_exmc_sram_init_struct *)p_struct)->databus_width     = EXMC_NOR_DATABUS_WIDTH_16B;
        ((hal_exmc_sram_init_struct *)p_struct)->burst_mode        = DISABLE;
        ((hal_exmc_sram_init_struct *)p_struct)->nwait_polarity    = EXMC_NWAIT_POLARITY_LOW;
        ((hal_exmc_sram_init_struct *)p_struct)->nwait_config      = EXMC_NWAIT_CONFIG_BEFORE;
        ((hal_exmc_sram_init_struct *)p_struct)->memory_write      = ENABLE;
        ((hal_exmc_sram_init_struct *)p_struct)->nwait_signal      = ENABLE;
        ((hal_exmc_sram_init_struct *)p_struct)->extended_mode     = DISABLE;
        ((hal_exmc_sram_init_struct *)p_struct)->asyn_wait         = DISABLE;
        ((hal_exmc_sram_init_struct *)p_struct)->cram_page_size    = EXMC_CRAM_AUTO_SPLIT;
        ((hal_exmc_sram_init_struct *)p_struct)->write_mode        = EXMC_ASYN_WRITE;

        ((hal_exmc_sram_init_struct *)p_struct)->asyn_access_mode       = EXMC_ACCESS_MODE_A;
        ((hal_exmc_sram_init_struct *)p_struct)->syn_data_latency       = 15U;
        ((hal_exmc_sram_init_struct *)p_struct)->syn_clk_division       = 15U;
        ((hal_exmc_sram_init_struct *)p_struct)->bus_latency            = 15U;
        ((hal_exmc_sram_init_struct *)p_struct)->asyn_data_setuptime    = 255U;
        ((hal_exmc_sram_init_struct *)p_struct)->asyn_address_holdtime  = 15U;
        ((hal_exmc_sram_init_struct *)p_struct)->asyn_address_setuptime = 15U;

        ((hal_exmc_sram_init_struct *)p_struct)->write_asyn_access_mode       = EXMC_WACCESS_MODE_A;
        ((hal_exmc_sram_init_struct *)p_struct)->write_bus_latency            = 15U;
        ((hal_exmc_sram_init_struct *)p_struct)->write_asyn_data_setuptime    = 255U;
        ((hal_exmc_sram_init_struct *)p_struct)->write_asyn_address_holdtime  = 15U;
        ((hal_exmc_sram_init_struct *)p_struct)->write_asyn_address_setuptime = 15U;
        break;
    case HAL_EXMC_SRAM_DEV_STRUCT:
        ((hal_exmc_sram_dev_struct *)p_struct)->sram_region           = EXMC_BANK0_NORSRAM_REGION0;
        ((hal_exmc_sram_dev_struct *)p_struct)->p_dma_rx              = NULL;
        ((hal_exmc_sram_dev_struct *)p_struct)->p_dma_tx              = NULL;
        ((hal_exmc_sram_dev_struct *)p_struct)->dma_transfer_complete = NULL;
        ((hal_exmc_sram_dev_struct *)p_struct)->dma_transfer_error    = NULL;
        ((hal_exmc_sram_dev_struct *)p_struct)->mutex                 = HAL_MUTEX_UNLOCKED;
        ((hal_exmc_sram_dev_struct *)p_struct)->state                 = HAL_EXMC_SRAM_STATE_RESET;
        ((hal_exmc_sram_dev_struct *)p_struct)->priv                  = NULL;
        break;
    case HAL_EXMC_SRAM_USER_CALLBACK_STRUCT:
        ((hal_exmc_sram_user_callback_struct *)p_struct)->dma_transfer_complete = NULL;
        ((hal_exmc_sram_user_callback_struct *)p_struct)->dma_transfer_error    = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      deinitialize EXMC NOR/SRAM region
    \param[in]  exmc_sram_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sram_deinit(hal_exmc_sram_dev_struct *exmc_sram_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == exmc_sram_dev) {
        HAL_DEBUGE("pointer [exmc_sram_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* reset the registers */
    EXMC_SNCTL(exmc_sram_dev->sram_region)   = SRAM_BANK0_SNCTL_RESET;
    EXMC_SNTCFG(exmc_sram_dev->sram_region)  = SRAM_BANK0_SNTCFG_RESET;
    EXMC_SNWTCFG(exmc_sram_dev->sram_region) = SRAM_BANK0_SNWTCFG_RESET;
    exmc_sram_dev->mutex                     = HAL_MUTEX_UNLOCKED;
    exmc_sram_dev->state                     = HAL_EXMC_SRAM_STATE_RESET;

    return HAL_ERR_NONE;
}

/*!
    \brief      write a Byte buffer(data is 8 bits ) to the EXMC SRAM memory
    \param[in]  exmc_sram_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  sram_addr: pointer to write start address
    \param[in]  in_addr: pointer to source buffer to write
    \param[in]  size: size of the buffer to write to memory
    \param[out] none
    \retval     error code:HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sram_write_8bit(hal_exmc_sram_dev_struct *exmc_sram_dev, uint8_t *in_addr, uint32_t *sram_addr, uint32_t size)
{
    __IO uint8_t *tmp_sramaddr = (uint8_t *)sram_addr;
    __IO uint8_t *tmp_inaddr = in_addr;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_sram_dev) || (NULL == in_addr) || (NULL == sram_addr)) {
        HAL_DEBUGE("pointer [exmc_sram_dev] or [in_addr] or [sram_addr] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == size)) {
        HAL_DEBUGE("parameter [size] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_EXMC_SRAM_STATE_READY == exmc_sram_dev->state) {
        HAL_LOCK(exmc_sram_dev);
        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_BUSY;

        /* while there is data to write */
        for(; 0U != size ; size--) {
            /* transfer data to the memory */
            *(uint8_t volatile *)(tmp_sramaddr) = *tmp_inaddr;
            /* increment the address*/
            tmp_inaddr++;
            tmp_sramaddr++;
        }

        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_READY;
        HAL_UNLOCK(exmc_sram_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      read a block of 8-bit data from the EXMC SRAM memory
    \param[in]  exmc_sram_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  sram_addr: pointer to read start address
    \param[out] out_addr: pointer to source buffer to read
    \param[in]  size: size of the buffer to be read
    \retval      error code:HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sram_read_8bit(hal_exmc_sram_dev_struct *exmc_sram_dev, uint8_t *out_addr, uint32_t *sram_addr, uint32_t size)
{
    __IO uint8_t *tmp_outaddr = out_addr;
    __IO uint8_t *tmp_sramaddr = (uint8_t *)sram_addr;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_sram_dev) || (NULL == out_addr) || (NULL == sram_addr)) {
        HAL_DEBUGE("pointer [exmc_sram_dev] or [out_addr] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == size)) {
        HAL_DEBUGE("parameter [size] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((HAL_EXMC_SRAM_STATE_READY == exmc_sram_dev->state) ||(HAL_EXMC_SRAM_STATE_PROTECTED == exmc_sram_dev->state)) {
        HAL_LOCK(exmc_sram_dev);
        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_BUSY;

        /* while there is data to read */
        for(; 0U != size; size--) {
            /* read a byte from the memory */
            *tmp_outaddr = *((__IO uint8_t *)(tmp_sramaddr));
            /* increment the address */
            tmp_outaddr++;
            tmp_sramaddr++;
        }

        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_READY;
        HAL_UNLOCK(exmc_sram_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      write a Half-word buffer(data is 16 bits) to the EXMC SRAM memory
    \param[in]  exmc_sram_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  sram_addr: pointer to write start address
    \param[in]  in_addr: pointer to source buffer to write
    \param[in]  size: size of the buffer to write to memory
    \param[out] none
    \retval     error code:HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sram_write_16bit(hal_exmc_sram_dev_struct *exmc_sram_dev, uint16_t *in_addr, uint32_t *sram_addr, uint32_t size)
{
    __IO uint32_t *tmp_sramaddr = sram_addr;
    __IO uint16_t *tmp_inaddr = in_addr;
    uint32_t input_value, sram_value;
    uint8_t limit;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_sram_dev) || (NULL == in_addr) || (NULL == sram_addr)) {
        HAL_DEBUGE("pointer [exmc_sram_dev] or [in_addr] or [sram_addr] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == size)) {
        HAL_DEBUGE("parameter [size] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_EXMC_SRAM_STATE_READY == exmc_sram_dev->state) {
        HAL_LOCK(exmc_sram_dev);
        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_BUSY;

        /* check if the size is a 32-bits multiple */
        if((size % 2U) != 0U) {
            limit = 1U;
        } else {
            limit = 0U;
        }

        /* while there is data to write */
        for(; size != limit; size -= 2U) {
            /* transfer data to the memory */
            *(__IO uint32_t *)(tmp_sramaddr) = (uint32_t)(*tmp_inaddr);
            /* increment the address*/
            tmp_inaddr++;
            /* transfer data to the memory */
            sram_value = *(__IO uint32_t *)(tmp_sramaddr);
            input_value = (uint32_t)(*tmp_inaddr);
            sram_value |= input_value << 16U;
            *(__IO uint32_t *)(tmp_sramaddr) = sram_value;
            /* increment the address*/
            tmp_inaddr++;
            tmp_sramaddr++;
        }

        if(0U != limit) {
            sram_value = *(__IO uint32_t *)(tmp_sramaddr);
            input_value = (uint32_t)(*tmp_inaddr);
            sram_value = (input_value & 0x0000FFFFU) | (sram_value & 0xFFFF0000U);
            *(__IO uint32_t *)(tmp_sramaddr) = sram_value;
        } else {
            /* do nothing */
        }

        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_READY;
        HAL_UNLOCK(exmc_sram_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      read a block of 16-bit data from the EXMC SRAM memory
    \param[in]  exmc_sram_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  sram_addr: pointer to read start address
    \param[out] out_addr: pointer to source buffer to read
    \param[in]  size: size of the buffer to be read
    \retval      error code:HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sram_read_16bit(hal_exmc_sram_dev_struct *exmc_sram_dev, uint16_t *out_addr, uint32_t *sram_addr, uint32_t size)
{
    __IO uint32_t *tmp_sramaddr = sram_addr;
    __IO uint16_t *tmp_outaddr = out_addr;
    uint32_t sramaddr;
    uint8_t limit;
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_sram_dev) || (NULL == out_addr) || (NULL == sram_addr)) {
        HAL_DEBUGE("pointer [exmc_sram_dev] or [out_addr] or [sram_addr] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == size)) {
        HAL_DEBUGE("parameter [size] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((HAL_EXMC_SRAM_STATE_READY == exmc_sram_dev->state) || (HAL_EXMC_SRAM_STATE_PROTECTED == exmc_sram_dev->state)) {
        HAL_LOCK(exmc_sram_dev);
        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_BUSY;

        /* check if the size is a 32-bits multiple */
        if((size % 2U) != 0U) {
            limit = 1U;
        } else {
            limit = 0U;
        }

        /* while there is data to read */
        for(; limit != size ; size -= 2U) {
            sramaddr = (uint32_t)(*tmp_sramaddr);
            *tmp_outaddr = (uint16_t)(sramaddr & 0x0000FFFFU);
            /* increment the address */
            tmp_outaddr++;
            /* read a half-word from the memory */
            sramaddr = (uint32_t)(*tmp_sramaddr);
            *tmp_outaddr = (uint16_t)((sramaddr & 0xFFFF0000U) >> 16U);
            /* increment the address */
            tmp_outaddr++;
            tmp_sramaddr++;
        }

        if(0U != limit) {
            sramaddr = (uint32_t)(*tmp_sramaddr);
            *tmp_outaddr = (uint16_t)(sramaddr & 0x0000FFFFU);
        } else {
            /* do nothing */
        }

        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_READY;
        HAL_UNLOCK(exmc_sram_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      write a word buffer(data is 32 bits) to the EXMC SRAM memory
    \param[in]  exmc_sram_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  sram_addr: pointer to write start address
    \param[in]  in_addr: pointer to source buffer to write
    \param[in]  size: size of the buffer to write to memory
    \param[out] none
    \retval      error code:HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sram_write_32bit(hal_exmc_sram_dev_struct *exmc_sram_dev, uint32_t *in_addr, uint32_t *sram_addr, uint32_t size)
{
    __IO uint32_t *tmp_inaddr = in_addr;
    __IO uint32_t *tmp_sramaddr = sram_addr;
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_sram_dev) || (NULL == in_addr) || (NULL == sram_addr)) {
        HAL_DEBUGE("pointer [exmc_sram_dev] or [in_addr] or [sram_addr] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == size)) {
        HAL_DEBUGE("parameter [size] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_EXMC_SRAM_STATE_READY == exmc_sram_dev->state) {
        HAL_LOCK(exmc_sram_dev);
        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_BUSY;

        /* while there is data to write */
        for(; 0U != size; size--) {
            /* transfer data to the memory */
            *(__IO uint32_t *)(tmp_sramaddr) = *tmp_inaddr;
            /* increment the address*/
            tmp_inaddr++;
            tmp_sramaddr++;
        }

        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_READY;
        HAL_UNLOCK(exmc_sram_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      read a block of 32-bit data from the EXMC SRAM memory
    \param[in]  exmc_sram_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  sram_addr: pointer to read start address
    \param[out] out_addr: pointer to source buffer to read
    \param[in]  size: size of the buffer to be read
    \retval      error code:HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sram_read_32bit(hal_exmc_sram_dev_struct *exmc_sram_dev, uint32_t *out_addr, uint32_t *sram_addr, uint32_t size)
{
    __IO uint32_t *tmp_outaddr = out_addr;
    __IO uint32_t *tmp_sramaddr = sram_addr;
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_sram_dev) || (NULL == out_addr) || (NULL == sram_addr)) {
        HAL_DEBUGE("pointer [exmc_sram_dev] or [out_addr] or [sram_addr] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == size)) {
        HAL_DEBUGE("parameter [size] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((HAL_EXMC_SRAM_STATE_READY == exmc_sram_dev->state) || (HAL_EXMC_SRAM_STATE_PROTECTED == exmc_sram_dev->state)) {
        HAL_LOCK(exmc_sram_dev);
        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_BUSY;

        /* while there is data to read */
        for(; 0U != size; size--) {
            /* read a word from the memory */
            *tmp_outaddr = *(__IO uint32_t *)(tmp_sramaddr);
            /* increment the address */
            tmp_outaddr++;
            tmp_sramaddr++;
        }

        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_READY;
        HAL_UNLOCK(exmc_sram_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      write a block of 32-bit data from the EXMC SRAM memory using DMA transfer.
    \param[in]  exmc_sram_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  in_addr: Pointer to source buffer to write
    \param[in]  sram_addr: Pointer to write start address
    \param[in]  size: write length in byte
    \param[in]  p_user_func: pointer to user callback function struct
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_ABORT, HAL_ERR_BUSY,
                details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sram_write_dma(hal_exmc_sram_dev_struct *exmc_sram_dev, uint32_t *in_addr, uint32_t *sram_addr, \
                           uint32_t size, hal_exmc_sram_user_callback_struct *p_user_func)
{
    hal_dma_irq_struct dma_irq = {0};
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_sram_dev) || (NULL == in_addr) || (NULL == sram_addr)) {
        HAL_DEBUGE("pointer [exmc_sram_dev] or [in_addr] or [sram_addr] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == size)) {
        HAL_DEBUGE("parameter [size] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check the SRAM controller state */
    if(HAL_EXMC_SRAM_STATE_READY == exmc_sram_dev->state) {
        /* process Locked */
        HAL_LOCK(exmc_sram_dev);
        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_BUSY;

        /* configure DMA user callbacks */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->dma_transfer_complete) {
                exmc_sram_dev->dma_transfer_complete = (hal_irq_handle_cb)p_user_func->dma_transfer_complete;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->dma_transfer_error) {
                exmc_sram_dev->dma_transfer_error = (hal_irq_handle_cb)p_user_func->dma_transfer_error;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
        /* configure DMA interrupt callback function */
        dma_irq.full_finish_handle = &_sram_dma_complete;
        dma_irq.error_handle = &_sram_dma_error;

        /* start DMA interrupt mode transfer */
        if(NULL != exmc_sram_dev->p_dma_tx) {
            if(HAL_ERR_NONE != hal_dma_start_interrupt(exmc_sram_dev->p_dma_tx, (uint32_t)in_addr, \
                    (uint32_t)sram_addr, (uint16_t)(size * 4u), &dma_irq)) {
                retval = HAL_ERR_ABORT;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_READY;
        /* process unlocked */
        HAL_UNLOCK(exmc_sram_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      read a block of 32-bit data from the EXMC SRAM memory using DMA transfer.
    \param[in]  exmc_sram_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  sram_addr: pointer to read start address
    \param[in]  size: read length in byte
    \param[out] out_addr: pointer to source buffer to read
    \param[in]  p_user_func: pointer to user callback function struct
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_ABORT, HAL_ERR_BUSY,
                details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sram_read_dma(hal_exmc_sram_dev_struct *exmc_sram_dev, uint32_t *out_addr, uint32_t *sram_addr, \
                               uint32_t size, hal_exmc_sram_user_callback_struct *p_user_func)
{
    hal_dma_irq_struct dma_irq = {0};
    hal_exmc_sram_state_enum state = exmc_sram_dev->state;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_sram_dev) || (NULL == out_addr) || (NULL == sram_addr)) {
        HAL_DEBUGE("pointer [exmc_sram_dev] or [out_addr] or [sram_addr] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == size)) {
        HAL_DEBUGE("parameter [size] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check the SRAM controller state */
    if((HAL_EXMC_SRAM_STATE_READY == exmc_sram_dev->state) || (HAL_EXMC_SRAM_STATE_PROTECTED == exmc_sram_dev->state)) {
        /* process Locked */
        HAL_LOCK(exmc_sram_dev);
        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_BUSY;

        /* configure DMA user callbacks */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->dma_transfer_complete) {
                exmc_sram_dev->dma_transfer_complete = (hal_irq_handle_cb)p_user_func->dma_transfer_complete;
            } else {
            /* do nothing */
            }
            if(NULL != p_user_func->dma_transfer_error) {
                exmc_sram_dev->dma_transfer_error = (hal_irq_handle_cb)p_user_func->dma_transfer_error;
            } else {
            /* do nothing */
            }
        } else {
            /* do nothing */
        }
        /* configure DMA interrupt callback function */
        if(HAL_EXMC_SRAM_STATE_READY == state) {
            dma_irq.full_finish_handle = &_sram_dma_complete;
        } else {
            dma_irq.full_finish_handle = &_sram_dma_complete_protected;
        }
        dma_irq.error_handle = &_sram_dma_error;

        /* start DMA interrupt mode transfer */
        if(NULL != exmc_sram_dev->p_dma_rx) {
            if(HAL_ERR_NONE != hal_dma_start_interrupt(exmc_sram_dev->p_dma_rx, (uint32_t)sram_addr, \
                    (uint32_t)out_addr, ((uint16_t)size * 4u), &dma_irq)) {
                retval = HAL_ERR_ABORT;
            } else {
            /* do nothing */
            }
        } else {
            /* do nothing */
        }

        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_READY;
        /* process unlocked */
        HAL_UNLOCK(exmc_sram_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      EXMC sram write operation enable
    \param[in]  exmc_sram_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_ABORT details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sram_write_operation_enable(hal_exmc_sram_dev_struct *exmc_sram_dev)
{
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_sram_dev)) {
        HAL_DEBUGE("pointer exmc_sram_dev address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_EXMC_SRAM_STATE_READY == exmc_sram_dev->state)
    {
        /* process locked */
        HAL_LOCK(exmc_sram_dev);
        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_BUSY;

        EXMC_SNCTL(exmc_sram_dev->sram_region) |= EXMC_SNCTL_WEN;

        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_PROTECTED;
        HAL_UNLOCK(exmc_sram_dev);
    } else {
        retval = HAL_ERR_ABORT;
    }

    return retval;
}

/*!
    \brief      EXMC sram write operation disable
    \param[in]  exmc_sram_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_ABORT details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sram_write_operation_disable(hal_exmc_sram_dev_struct *exmc_sram_dev)
{
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_sram_dev)) {
        HAL_DEBUGE("pointer exmc_sram_dev address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_EXMC_SRAM_STATE_PROTECTED == exmc_sram_dev->state) {
        /* process locked */
        HAL_LOCK(exmc_sram_dev);
        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_BUSY;

        EXMC_SNCTL(exmc_sram_dev->sram_region) &= (~EXMC_SNCTL_WEN);

        exmc_sram_dev->state = HAL_EXMC_SRAM_STATE_READY;
        HAL_UNLOCK(exmc_sram_dev);
    } else {
        retval = HAL_ERR_ABORT;
    }

    return retval;
}

/*!
    \brief      get NOR/PSRAM and SDRAM remap configuration
    \param[in]  none
    \param[out] none
    \retval     bank remap value
      \arg        EXMC_BANK_REMAP_DEFAULT: default mapping
      \arg        EXMC_BANK_NORPSRAM_SDRAM_SWAP: NOR/PSRAM bank and SDRAM device 0 swapped
*/
uint32_t hal_exmc_sram_sdram_remap_get(void)
{
    uint32_t bank_remap;

    bank_remap = EXMC_SNCTL(EXMC_BANK0_NORSRAM_REGION0) & EXMC_SNCTL_BKREMAP;

    return bank_remap;
}

/*!
    \brief      return the EXMC SRAM state
    \param[in]  exmc_sram_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_exmc_sram_state_enum details refer to gd32h7xx_hal_exmc_sram.h
*/
hal_exmc_sram_state_enum hal_exmc_sram_state_get(hal_exmc_sram_dev_struct *exmc_sram_dev)
{
    return exmc_sram_dev->state;
}

/*!
    \brief      configure NOR/PSRAM and SDRAM remap
    \param[in]  bank_remap: NOR/PSRAM and SDRAM map address
                only one parameter can be selected which is shown as below:
      \arg        EXMC_BANK_REMAP_DEFAULT: default mapping
      \arg        EXMC_BANK_NORPSRAM_SDRAM_SWAP: NOR/PSRAM bank and SDRAM device 0 swapped
    \param[out] none
    \retval     none
*/
void hals_exmc_sram_sdram_remap_config(uint32_t bank_remap)
{
    /* reset BKREMAP bits */
    EXMC_SNCTL(EXMC_BANK0_NORSRAM_REGION0) &= (uint32_t)(~EXMC_SNCTL_BKREMAP);

    EXMC_SNCTL(EXMC_BANK0_NORSRAM_REGION0) |= bank_remap;
}

/*!
    \brief      configure consecutive clock mode (consecutive clock is only supported in EXMC BANK0 REGION0)
    \param[in]  clock_mode: specify when the clock is generated
                only one parameter can be selected which is shown as below:
      \arg        EXMC_CLOCK_SYN_MODE: the clock is generated only during synchronous access
      \arg        EXMC_CLOCK_UNCONDITIONALLY: the clock is generated unconditionally
    \param[out] none
    \retval     none
*/
void hals_exmc_sram_consecutive_clock_config(uint32_t clock_mode)
{
    if(EXMC_CLOCK_UNCONDITIONALLY == clock_mode) {
        EXMC_SNCTL(EXMC_BANK0_NORSRAM_REGION0) |= EXMC_CLOCK_UNCONDITIONALLY;
    } else {
        EXMC_SNCTL(EXMC_BANK0_NORSRAM_REGION0) &= ~EXMC_CLOCK_UNCONDITIONALLY;
    }
}

/*!
    \brief      configure CRAM page size
    \param[in]  exmc_sram_region: select the region of bank0
                only one parameter can be selected which is shown as below:
      \arg        EXMC_BANK0_NORSRAM_REGIONx(x=0..3): SRAM REGIONx
    \param[in]  page_size: CRAM page size
                only one parameter can be selected which is shown as below:
      \arg        EXMC_CRAM_AUTO_SPLIT: the clock is generated only during synchronous access
      \arg        EXMC_CRAM_PAGE_SIZE_128_BYTES: page size is 128 bytes
      \arg        EXMC_CRAM_PAGE_SIZE_256_BYTES: page size is 256 bytes
      \arg        EXMC_CRAM_PAGE_SIZE_512_BYTES: page size is 512 bytes
      \arg        EXMC_CRAM_PAGE_SIZE_1024_BYTES: page size is 1024 bytes
    \param[out] none
    \retval     none
*/
void hals_exmc_sram_page_size_config(uint32_t exmc_sram_region, uint32_t page_size)
{
    /* reset the bits */
    EXMC_SNCTL(exmc_sram_region) &= ~EXMC_SNCTL_CPS;

    EXMC_SNCTL(exmc_sram_region) |= page_size;
}

/*!
    \brief      EXMC SRAM  process complete callback.
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _sram_dma_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_exmc_sram_dev_struct *p_sram;

    p_dma = (hal_dma_dev_struct *)dma;
    p_sram = (hal_exmc_sram_dev_struct *)p_dma->p_periph;

    /* Disable the MDMA channel */
    hals_dma_channel_disable(p_dma->dma_periph, p_dma->channel);

    /* Update the SRAM controller state */
    p_sram->state = HAL_EXMC_SRAM_STATE_READY;

    if(NULL != p_sram->dma_transfer_complete) {
        p_sram->dma_transfer_complete(p_sram);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      EXMC SRAM  process complete callback.
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _sram_dma_complete_protected(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_exmc_sram_dev_struct *p_sram;

    p_dma = (hal_dma_dev_struct *)dma;
    p_sram = (hal_exmc_sram_dev_struct *)p_dma->p_periph;

    /* Disable the MDMA channel */
    hals_dma_channel_disable(p_dma->dma_periph, p_dma->channel);

    /* Update the SRAM controller state */
    p_sram->state = HAL_EXMC_SRAM_STATE_PROTECTED;

    if(NULL != p_sram->dma_transfer_complete) {
        p_sram->dma_transfer_complete(p_sram);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      EXMC SRAM error callback.
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _sram_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_exmc_sram_dev_struct *p_sram;

    p_dma = (hal_dma_dev_struct *)dma;
    p_sram = (hal_exmc_sram_dev_struct *)p_dma->p_periph;

    /* Disable the MDMA channel */
    hals_dma_channel_disable(p_dma->dma_periph, p_dma->channel);

    /* Update the SRAM controller state */
    p_sram->state = HAL_EXMC_SRAM_STATE_ERROR;

    if(NULL != p_sram->dma_transfer_error) {
        p_sram->dma_transfer_error(p_sram);
    } else {
        /* do nothing */
    }
}
