/*!
    \file    gd32h7xx_hal_dma.c
    \brief   DMA driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* DMA register bit offset */
#define CHXFCTL_FCNT_OFFSET   ((uint32_t)0x00000003U) /*!< bit offset of FCNT in DMA_CHxFCTL */
#define DEV_VALID_ADDRESS     ((uint32_t)0x68000000U) /*!< the valid address */

/* DMAMUX register bit offset */
#define RM_CHXCFG_NBR_OFFSET  ((uint32_t)0x00000013U) /*!< bit offset of NBR in DMAMUX_RM_CHXCFG */
#define RG_CHXCFG_NBRG_OFFSET ((uint32_t)0x00000013U) /*!< bit offset of NBRG in DMAMUX_RG_CHXCFG */

/* configure dmamux request gennerator channel */
static int32_t _dmamux_request_gen_config(hal_dma_dev_struct *dma_dev, hal_dmamux_generator_channel_enum channelx, \
                                          hal_dmamux_gen_parameter_struct *init_struct);
/* configure dmamux sync channel */
static int32_t _dmamux_sync_config(hal_dmamux_multiplexer_channel_enum channelx, \
                                   hal_dmamux_sync_parameter_struct *init_struct);

/*!
    \brief      initialize DMA channel
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx:specify which DMA channel is initialized.
                  the argument could be selected from enumeration <hal_dma_channel_enum>
      \arg        DMA_CHx(x=0..7)
    \param[in]  dma: the pointer of DMA init structure
      \arg        dma_single_struct: single data mode init struct
      \arg        dma_multi_struct: multi data mode init struct
      \arg        dmamux_sync_struct: dmamux sync init base setting
      \arg        dmamux_gen_struct: dmamux generator init base setting
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_dma_init(hal_dma_dev_struct *dma_dev, uint32_t dma_periph, hal_dma_channel_enum channelx, \
                     hal_dma_init_struct *dma)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check for the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == dma_dev) || (NULL == dma)) {
        HAL_DEBUGE("pointer [dma] or [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    /* check DMA mode and direction parameters */
    if(DMA_MULTIDATA_TRANSFER_ENABLE == dma_dev->multidata_mode) {
        if((DMA_CIRCULAR_MODE_ENABLE == dma->dma_multi_struct.circular_mode)
           && (DMA_MEMORY_TO_MEMORY == dma->dma_multi_struct.direction)) {
            HAL_DEBUGE("circular mode is invalid due to 'memory to memory' has been configured");
            return HAL_ERR_VAL;
        }
    } else {
        if((DMA_CIRCULAR_MODE_ENABLE == dma->dma_single_struct.circular_mode)
           && (DMA_MEMORY_TO_MEMORY == dma->dma_single_struct.direction)) {
            HAL_DEBUGE("circular mode is invalid due to 'memory to memory' has been configured");
            return HAL_ERR_VAL;
        }
    }
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* change DMA device state */
    dma_dev->state      = HAL_DMA_STATE_BUSY;
    dma_dev->dma_periph = dma_periph;
    dma_dev->channel    = channelx;
    /* disable dma channelx */
    hals_dma_channel_disable(dma_periph, dma_dev->channel);
    if(DMA_MULTIDATA_TRANSFER_ENABLE == dma_dev->multidata_mode) {
        hals_dma_multi_data_mode_init(dma_periph, channelx, &dma->dma_multi_struct);
        switch(dma->dma_multi_struct.request) {
        case DMA_REQUEST_GENERATOR0:
            _dmamux_request_gen_config(dma_dev, DMAMUX_GENCH0, &dma->dmamux_gen_struct);
            break;
        case DMA_REQUEST_GENERATOR1:
            _dmamux_request_gen_config(dma_dev, DMAMUX_GENCH1, &dma->dmamux_gen_struct);
            break;
        case DMA_REQUEST_GENERATOR2:
            _dmamux_request_gen_config(dma_dev, DMAMUX_GENCH2, &dma->dmamux_gen_struct);
            break;
        case DMA_REQUEST_GENERATOR3:
            _dmamux_request_gen_config(dma_dev, DMAMUX_GENCH3, &dma->dmamux_gen_struct);
            break;
        case DMA_REQUEST_GENERATOR4:
            _dmamux_request_gen_config(dma_dev, DMAMUX_GENCH4, &dma->dmamux_gen_struct);
            break;
        case DMA_REQUEST_GENERATOR5:
            _dmamux_request_gen_config(dma_dev, DMAMUX_GENCH5, &dma->dmamux_gen_struct);
            break;
        case DMA_REQUEST_GENERATOR6:
            _dmamux_request_gen_config(dma_dev, DMAMUX_GENCH6, &dma->dmamux_gen_struct);
            break;
        case DMA_REQUEST_GENERATOR7:
            _dmamux_request_gen_config(dma_dev, DMAMUX_GENCH7, &dma->dmamux_gen_struct);
            break;
        default:
            break;
        }
    } else {
        hals_dma_single_data_mode_init(dma_periph, channelx, &dma->dma_single_struct);
        switch(dma->dma_single_struct.request) {
        case DMA_REQUEST_GENERATOR0:
            _dmamux_request_gen_config(dma_dev, DMAMUX_GENCH0, &dma->dmamux_gen_struct);
            break;
        case DMA_REQUEST_GENERATOR1:
            _dmamux_request_gen_config(dma_dev, DMAMUX_GENCH1, &dma->dmamux_gen_struct);
            break;
        case DMA_REQUEST_GENERATOR2:
            _dmamux_request_gen_config(dma_dev, DMAMUX_GENCH2, &dma->dmamux_gen_struct);
            break;
        case DMA_REQUEST_GENERATOR3:
            _dmamux_request_gen_config(dma_dev, DMAMUX_GENCH3, &dma->dmamux_gen_struct);
            break;
        case DMA_REQUEST_GENERATOR4:
            _dmamux_request_gen_config(dma_dev, DMAMUX_GENCH4, &dma->dmamux_gen_struct);
            break;
        case DMA_REQUEST_GENERATOR5:
            _dmamux_request_gen_config(dma_dev, DMAMUX_GENCH5, &dma->dmamux_gen_struct);
            break;
        case DMA_REQUEST_GENERATOR6:
            _dmamux_request_gen_config(dma_dev, DMAMUX_GENCH6, &dma->dmamux_gen_struct);
            break;
        case DMA_REQUEST_GENERATOR7:
            _dmamux_request_gen_config(dma_dev, DMAMUX_GENCH7, &dma->dmamux_gen_struct);
            break;
        default:
            break;
        }
    }

    if(DMA0 == dma_periph) {
        dma_dev->multi_channel = (hal_dmamux_multiplexer_channel_enum)channelx;
    } else {
        dma_dev->multi_channel = (hal_dmamux_multiplexer_channel_enum)(channelx + 8U);
    }

    _dmamux_sync_config(dma_dev->multi_channel, &dma->dmamux_sync_struct);

    /* change DMA device state */
    dma_dev->state = HAL_DMA_STATE_READY;

    return error_code;
}

/*!
    \brief      deinitialize DMA registers of a channel
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_dma_deinit(hal_dma_dev_struct *dma_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the DMA pointer address and the number length parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == dma_dev) {
        HAL_DEBUGE("pointer [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    dma_dev->state = HAL_DMA_STATE_BUSY;
    /* disable DMA a channel */
    DMA_CHCTL(dma_dev->dma_periph, dma_dev->channel)   &= ~DMA_CHXCTL_CHEN;
    /* reset DMA channel registers */
    DMA_CHCTL(dma_dev->dma_periph, dma_dev->channel)    = DMA_CHCTL_RESET_VALUE;
    DMA_CHCNT(dma_dev->dma_periph, dma_dev->channel)    = DMA_CHCNT_RESET_VALUE;
    DMA_CHPADDR(dma_dev->dma_periph, dma_dev->channel)  = DMA_CHPADDR_RESET_VALUE;
    DMA_CHM0ADDR(dma_dev->dma_periph, dma_dev->channel) = DMA_CHMADDR_RESET_VALUE;
    DMA_CHM1ADDR(dma_dev->dma_periph, dma_dev->channel) = DMA_CHMADDR_RESET_VALUE;
    DMA_CHFCTL(dma_dev->dma_periph, dma_dev->channel)   = DMA_CHFCTL_RESET_VALUE;
    DMAMUX_RM_CHXCFG(dma_dev->multi_channel)            = DMAMUX_RM_CHCFG_RESET_VALUE;
    DMAMUX_RG_CHXCFG(dma_dev->multi_channel)            = DMAMUX_RG_CHCFG_RESET_VALUE;

    /* check DMA channel */
    if(dma_dev->channel < DMA_CH4) {
        DMA_INTC0(dma_dev->dma_periph) |= DMA_FLAG_ADD(DMA_CHINTF_RESET_VALUE, dma_dev->channel);
    } else {
        DMA_INTC1(dma_dev->dma_periph) |= DMA_FLAG_ADD(DMA_CHINTF_RESET_VALUE, dma_dev->channel - 4UL);
    }

    /* check DMA periph */
    if(DMA0 == dma_dev->dma_periph) {
        hal_rcu_periph_reset_enable(RCU_DMA0RST);
        hal_rcu_periph_reset_disable(RCU_DMA0RST);
        hal_rcu_periph_reset_enable(RCU_DMAMUXRST);
        hal_rcu_periph_reset_disable(RCU_DMAMUXRST);
    } else {
        hal_rcu_periph_reset_enable(RCU_DMA1RST);
        hal_rcu_periph_reset_disable(RCU_DMA1RST);
        hal_rcu_periph_reset_enable(RCU_DMAMUXRST);
        hal_rcu_periph_reset_disable(RCU_DMAMUXRST);
    }

    dma_dev->multi_channel = DMAMUX_MUXCH0;
    dma_dev->gen_channel   = DMAMUX_GENCH0;
    dma_dev->gen_enable    = 0U;
    dma_dev->error_state   = HAL_DMA_ERROR_NONE;
    dma_dev->state         = HAL_DMA_STATE_RESET;

    return error_code;
}

/*!
    \brief      initialize the DMA structure with the default values
    \param[in]  struct_type: the argument could be selected from enumeration <hal_dma_struct_type_enum>
    \param[out] p_struct: pointer to DMA structure that contains the configuration information
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE,HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_dma_struct_init(hal_dma_struct_type_enum struct_type, void *p_struct)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check for the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(struct_type) {
    case HAL_DMA_INIT_STRUCT:
        /* initialize DMA initialization structure with the default values */
        /* init single structure */
        hals_dma_single_data_para_struct_init(&((hal_dma_init_struct *)p_struct)->dma_single_struct);
        /* init multi structure */
        hals_dma_multi_data_para_struct_init(&((hal_dma_init_struct *)p_struct)->dma_multi_struct);
        /* init gen structure */
        hals_dmamux_gen_struct_para_init(&((hal_dma_init_struct *)p_struct)->dmamux_gen_struct);
        /* init sync structure */
        hals_dmamux_sync_struct_para_init(&((hal_dma_init_struct *)p_struct)->dmamux_sync_struct);
        break;
    case HAL_DMA_SINGLE_STRUCT:
        /* set the DMA struct with the default values */
        ((hal_dma_single_data_parameter_struct *)p_struct)->request             = DMA_REQUEST_M2M;
        ((hal_dma_single_data_parameter_struct *)p_struct)->periph_addr         = 0U;
        ((hal_dma_single_data_parameter_struct *)p_struct)->periph_inc          = DMA_PERIPH_INCREASE_DISABLE;
        ((hal_dma_single_data_parameter_struct *)p_struct)->memory0_addr        = 0U;
        ((hal_dma_single_data_parameter_struct *)p_struct)->memory_inc          = DMA_MEMORY_INCREASE_DISABLE;
        ((hal_dma_single_data_parameter_struct *)p_struct)->periph_memory_width = 0U;
        ((hal_dma_single_data_parameter_struct *)p_struct)->circular_mode       = DMA_CIRCULAR_MODE_DISABLE;
        ((hal_dma_single_data_parameter_struct *)p_struct)->direction           = DMA_PERIPH_TO_MEMORY;
        ((hal_dma_single_data_parameter_struct *)p_struct)->number              = 0U;
        ((hal_dma_single_data_parameter_struct *)p_struct)->priority            = DMA_PRIORITY_LOW;
        ((hal_dma_single_data_parameter_struct *)p_struct)->switch_buffer       = DMA_SWITCH_BUFFER_MODE_DISABLE;
        ((hal_dma_single_data_parameter_struct *)p_struct)->buffer_selection    = DMA_MEMORY_0;
        break;
    case HAL_DMA_MULTI_STRUCT:
        /* set the DMA structure with the default values */
        ((hal_dma_multi_data_parameter_struct *)p_struct)->request            = DMA_REQUEST_M2M;
        ((hal_dma_multi_data_parameter_struct *)p_struct)->periph_addr        = 0U;
        ((hal_dma_multi_data_parameter_struct *)p_struct)->periph_width       = 0U;
        ((hal_dma_multi_data_parameter_struct *)p_struct)->periph_inc         = DMA_PERIPH_INCREASE_DISABLE;
        ((hal_dma_multi_data_parameter_struct *)p_struct)->memory0_addr       = 0U;
        ((hal_dma_multi_data_parameter_struct *)p_struct)->memory_width       = 0U;
        ((hal_dma_multi_data_parameter_struct *)p_struct)->memory_inc         = DMA_MEMORY_INCREASE_DISABLE;
        ((hal_dma_multi_data_parameter_struct *)p_struct)->memory_burst_width = 0U;
        ((hal_dma_multi_data_parameter_struct *)p_struct)->periph_burst_width = 0U;
        ((hal_dma_multi_data_parameter_struct *)p_struct)->circular_mode      = DMA_CIRCULAR_MODE_DISABLE;
        ((hal_dma_multi_data_parameter_struct *)p_struct)->direction          = DMA_PERIPH_TO_MEMORY;
        ((hal_dma_multi_data_parameter_struct *)p_struct)->number             = 0U;
        ((hal_dma_multi_data_parameter_struct *)p_struct)->priority           = DMA_PRIORITY_LOW;
        ((hal_dma_multi_data_parameter_struct *)p_struct)->switch_buffer      = DMA_SWITCH_BUFFER_MODE_DISABLE;
        ((hal_dma_multi_data_parameter_struct *)p_struct)->buffer_selection   = DMA_MEMORY_0;
        break;
    case HAL_DMAMUX_GENREQ_STRUCT:
        /* set the DMAMUX request generator structure with the default values */
        ((hal_dmamux_gen_parameter_struct *)p_struct)->trigger_id       = DMAMUX_SYNC_EVT0_OUT;
        ((hal_dmamux_gen_parameter_struct *)p_struct)->trigger_polarity = DMAMUX_SYNC_RISING;
        ((hal_dmamux_gen_parameter_struct *)p_struct)->request_number   = 1U;
        break;
    case HAL_DMAMUX_SYNC_STRUCT:
        /* set the DMAMUX synchronization structure with the default values */
        ((hal_dmamux_sync_parameter_struct *)p_struct)->event_enable   = DMAMUX_EVENT_INPUT_DISABLE;
        ((hal_dmamux_sync_parameter_struct *)p_struct)->sync_enable    = DMAMUX_SYNC_DISABLE;
        ((hal_dmamux_sync_parameter_struct *)p_struct)->sync_id        = DMAMUX_SYNC_EVT0_OUT;
        ((hal_dmamux_sync_parameter_struct *)p_struct)->sync_polarity  = DMAMUX_SYNC_RISING;
        ((hal_dmamux_sync_parameter_struct *)p_struct)->request_number = 1U;
        break;
    case HAL_DMA_DEV_STRUCT:
        /* initialize DMA device information structure with the default values */
        ((hal_dma_dev_struct *)p_struct)->dma_periph                         = DMA0;
        ((hal_dma_dev_struct *)p_struct)->channel                            = DMA_CH0;
        ((hal_dma_dev_struct *)p_struct)->multi_channel                      = DMAMUX_MUXCH0;
        ((hal_dma_dev_struct *)p_struct)->gen_channel                        = DMAMUX_GENCH0;
        ((hal_dma_dev_struct *)p_struct)->gen_enable                         = 0U;
        ((hal_dma_dev_struct *)p_struct)->dma_irq.error_handle               = NULL;
        ((hal_dma_dev_struct *)p_struct)->dma_irq.half_finish_handle         = NULL;
        ((hal_dma_dev_struct *)p_struct)->dma_irq.full_finish_handle         = NULL;
        ((hal_dma_dev_struct *)p_struct)->dma_irq.m1_half_finish_handle      = NULL;
        ((hal_dma_dev_struct *)p_struct)->dma_irq.m1_full_finish_handle      = NULL;
        ((hal_dma_dev_struct *)p_struct)->dma_irq.fifo_error_handle          = NULL;
        ((hal_dma_dev_struct *)p_struct)->dma_irq.single_error_handle        = NULL;
        ((hal_dma_dev_struct *)p_struct)->dmamux_irq.sync_overrun_handle     = NULL;
        ((hal_dma_dev_struct *)p_struct)->dmamux_irq.trigger_overrun_handle  = NULL;
        ((hal_dma_dev_struct *)p_struct)->error_state                        = HAL_DMA_ERROR_NONE;
        ((hal_dma_dev_struct *)p_struct)->state                              = HAL_DMA_STATE_NONE;
        ((hal_dma_dev_struct *)p_struct)->p_periph                           = NULL;
        ((hal_dma_dev_struct *)p_struct)->mutex                              = HAL_MUTEX_UNLOCKED;
        ((hal_dma_dev_struct *)p_struct)->multidata_mode                     = DMA_MULTIDATA_TRANSFER_DISABLE;
        ((hal_dma_dev_struct *)p_struct)->transfer_state                     = HAL_DMA_TRANSFER_HALF;
        break;
    case HAL_DMA_IRQ_STRUCT:
        /* initialize DMA device interrupt callback function structure with the default values */
        ((hal_dma_irq_struct *)p_struct)->single_error_handle    = NULL;
        ((hal_dma_irq_struct *)p_struct)->error_handle           = NULL;
        ((hal_dma_irq_struct *)p_struct)->full_finish_handle     = NULL;
        ((hal_dma_irq_struct *)p_struct)->half_finish_handle     = NULL;
        ((hal_dma_irq_struct *)p_struct)->m1_half_finish_handle  = NULL;
        ((hal_dma_irq_struct *)p_struct)->m1_full_finish_handle  = NULL;
        ((hal_dma_irq_struct *)p_struct)->fifo_error_handle      = NULL;
        break;
    case HAL_DMAMUX_IRQ_STRUCT:
        /* initialize DMA device interrupt callback function structure with the default values */
        ((hal_dmamux_irq_struct *)p_struct)->sync_overrun_handle    = NULL;
        ((hal_dmamux_irq_struct *)p_struct)->trigger_overrun_handle = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [struct_type] value is undefine");
        error_code = HAL_ERR_VAL;
        break;
    }

    return error_code;
}

/*!
    \brief      DMA interrupt handler content function,which is merely used in dma_handler
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_dma_irq(hal_dma_dev_struct *dma_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check for the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the DMA pointer address and the number length parameter */
    if(NULL == dma_dev) {
        HAL_DEBUGE("pointer [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* fifo error interrupt handler */
    if(SET == hals_dma_interrupt_flag_get(dma_dev->dma_periph, dma_dev->channel, DMA_INTF_FEEIF)) {
        hals_dma_interrupt_flag_clear(dma_dev->dma_periph, dma_dev->channel, DMA_INTF_FEEIF);
        /* unlock DMA */
        dma_dev->state = HAL_DMA_STATE_READY;
        HAL_UNLOCK(dma_dev);

        if(dma_dev->dma_irq.fifo_error_handle != NULL) {
            dma_dev->dma_irq.fifo_error_handle(dma_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* half transfer finish interrupt handler */
    if(SET == hals_dma_interrupt_flag_get(dma_dev->dma_periph, dma_dev->channel, DMA_INTF_HTFIF)) {
        /* if DMA not in circular mode */
        if(0U == (DMA_CHCTL(dma_dev->dma_periph, dma_dev->channel) & DMA_CHXCTL_CMEN)) {
            hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_HTF);
        } else {
            /* do nothing */
        }

        hals_dma_interrupt_flag_clear(dma_dev->dma_periph, dma_dev->channel, DMA_INTF_HTFIF);

        /* current memory buffer used is memory 1 */
        if(DMA_CHXCTL_MBS == (DMA_CHCTL(dma_dev->dma_periph, dma_dev->channel) & DMA_CHXCTL_MBS)) {
            if(dma_dev->dma_irq.m1_half_finish_handle != NULL) {
                dma_dev->dma_irq.m1_half_finish_handle(dma_dev);
            } else {
                /* do nothing */
            }
        } else {
            if(dma_dev->dma_irq.half_finish_handle != NULL) {
                dma_dev->dma_irq.half_finish_handle(dma_dev);
            } else {
                /* do nothing */
            }
        }
    } else {
        /* do nothing */
    }

    /* full transfer finish interrupt handler */
    if(SET == hals_dma_interrupt_flag_get(dma_dev->dma_periph, dma_dev->channel, DMA_INTF_FTFIF)) {
        hals_dma_interrupt_flag_clear(dma_dev->dma_periph, dma_dev->channel, DMA_INTF_FTFIF);
        if(0U == (DMA_CHCTL(dma_dev->dma_periph, dma_dev->channel) & DMA_CHXCTL_CMEN)) {
            /* unlock DMA */
            dma_dev->state = HAL_DMA_STATE_READY;
            HAL_UNLOCK(dma_dev);
        } else {
            /* do nothing */
        }

        /* current memory buffer used is memory 1 */
        if(DMA_CHXCTL_MBS == (DMA_CHCTL(dma_dev->dma_periph, dma_dev->channel) & DMA_CHXCTL_MBS)) {
            if(dma_dev->dma_irq.m1_full_finish_handle != NULL) {
                dma_dev->dma_irq.m1_full_finish_handle(dma_dev);
            } else {
                /* do nothing */
            }
        } else {
            if(dma_dev->dma_irq.full_finish_handle != NULL) {
                dma_dev->dma_irq.full_finish_handle(dma_dev);
            } else {
                /* do nothing */
            }
        }
    } else {
        /* do nothing */
    }

    /* error interrupt handler */
    if(SET == hals_dma_interrupt_flag_get(dma_dev->dma_periph, dma_dev->channel, DMA_INTF_TAEIF)) {
        hals_dma_interrupt_flag_clear(dma_dev->dma_periph, dma_dev->channel, DMA_INTF_TAEIF);
        dma_dev->state       = HAL_DMA_STATE_READY;
        dma_dev->error_state = HAL_DMA_ERROR_TRANSFER;
        /* unlock DMA */
        HAL_UNLOCK(dma_dev);

        if(dma_dev->dma_irq.error_handle != NULL) {
            dma_dev->dma_irq.error_handle(dma_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* single error interrupt handler */
    if(SET == hals_dma_interrupt_flag_get(dma_dev->dma_periph, dma_dev->channel, DMA_INTF_SDEIF)) {
        hals_dma_interrupt_flag_clear(dma_dev->dma_periph, dma_dev->channel, DMA_INTF_SDEIF);
        dma_dev->state = HAL_DMA_STATE_READY;
        /* unlock DMA */
        HAL_UNLOCK(dma_dev);

        if(dma_dev->dma_irq.single_error_handle != NULL) {
            dma_dev->dma_irq.single_error_handle(dma_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return error_code;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to DMA interrupt callback function pointer structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_dma_irq_handle_set(hal_dma_dev_struct *dma_dev, hal_dma_irq_struct *p_irq)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check for the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the DMA pointer address and the number length parameter */
    if((NULL == dma_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [dma_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable DMA channel */
    hals_dma_channel_disable(dma_dev->dma_periph, dma_dev->channel);
    /* set the handler pointer  */
    /* enable the FIFO error interrupt */
    if(NULL != p_irq->fifo_error_handle) {
        dma_dev->dma_irq.fifo_error_handle = p_irq->fifo_error_handle;
        hals_dma_interrupt_enable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_FEE);
    } else {
        hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_FEE);
    }

    /* enable the single mode transfer channel error interrupt */
    if(NULL != p_irq->single_error_handle) {
        dma_dev->dma_irq.single_error_handle = p_irq->single_error_handle;
        hals_dma_interrupt_enable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_SDE);
    } else {
        hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_SDE);
    }

    /* enable the transfer channel error interrupt */
    if(NULL != p_irq->error_handle) {
        dma_dev->dma_irq.error_handle = p_irq->error_handle;
        hals_dma_interrupt_enable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_TAE);
    } else {
        hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_TAE);
    }

    /* set the full finish handler pointer and enable the channel full transfer finish  interrupt */
    if((NULL != p_irq->full_finish_handle) || (NULL != p_irq->m1_full_finish_handle)) {

        dma_dev->dma_irq.m1_full_finish_handle = p_irq->m1_full_finish_handle;
        dma_dev->dma_irq.full_finish_handle = p_irq->full_finish_handle;

        hals_dma_interrupt_enable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_FTF);
    } else {
        hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_FTF);
    }

    /* set the half finish handler pointer and enable the channel half transfer finish interrupt */
    if((NULL != p_irq->half_finish_handle) || (NULL != p_irq->m1_half_finish_handle)) {

        dma_dev->dma_irq.m1_half_finish_handle = p_irq->m1_half_finish_handle;
        dma_dev->dma_irq.half_finish_handle = p_irq->half_finish_handle;

        hals_dma_interrupt_enable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_HTF);
    } else {
        hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_HTF);
    }

    return error_code;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_dma_irq_handle_all_reset(hal_dma_dev_struct *dma_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check for the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the DMA pointer address and the number length parameter */
    if(NULL == dma_dev) {
        HAL_DEBUGE("pointer [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* DMA interrupt handler reset */
    dma_dev->dma_irq.fifo_error_handle   = NULL;
    dma_dev->dma_irq.single_error_handle = NULL;
    dma_dev->dma_irq.error_handle        = NULL;
    dma_dev->dma_irq.half_finish_handle  = NULL;
    dma_dev->dma_irq.full_finish_handle  = NULL;
    dma_dev->dma_irq.m1_half_finish_handle  = NULL;
    dma_dev->dma_irq.m1_full_finish_handle  = NULL;

    return error_code;
}

/*!
    \brief      start transferring amounts of data, poll transmit process and completed status
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  transfer_state: refer to hal_dma_transfer_state_enum
    \param[in]  timeout_ms: time out duration
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_HARDWARE, HAL_ERR_NONE, HAL_ERR_NO_SUPPORT,
                HAL_ERR_ALREADY_DONE, HAL_ERR_VAL, HAL_ERR_TIMEOUT  details refer to gd32h7xx_hal.h
*/
int32_t hal_dma_start_poll(hal_dma_dev_struct *dma_dev, hal_dma_transfer_state_enum transfer_state, uint32_t timeout_ms)
{
    int32_t error_code  = HAL_ERR_NONE;
    uint32_t flag       = 0U;
    uint32_t tick_start = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == dma_dev) {
        HAL_DEBUGE("pointer [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    /* check the circulation mode of the data transfer */
    if(RESET != (DMA_CHCTL(dma_dev->dma_periph, dma_dev->channel) & DMA_CHXCTL_CMEN)) {
        /* polling mode is not supported in circular mode */
        HAL_DEBUGE("DMA poll function is invalid in circulation mode");
        return HAL_ERR_NO_SUPPORT;
    }
    if(1U < transfer_state) {
        HAL_DEBUGE("parameter [transfer_state] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DMA */
    HAL_LOCK(dma_dev);

    if(HAL_DMA_STATE_BUSY != dma_dev->state) {
        /* no transfer */
        dma_dev->error_state = HAL_DMA_ERROR_NOTRANSFER;
        error_code = HAL_ERR_ALREADY_DONE;
    } else {
        /* do nothing */
    }

    if(HAL_ERR_NONE == error_code) {
        /* get the transfer flag */
        if(HAL_DMA_TRANSFER_HALF == transfer_state) {
            flag = DMA_FLAG_ADD(DMA_FLAG_HTF, dma_dev->channel);
        } else {
            flag = DMA_FLAG_ADD(DMA_FLAG_FTF, dma_dev->channel);
        }

        /* set timeout */
        tick_start = hal_sys_basetick_count_get();
        if(DMA_CH4 >= dma_dev->channel) {
            while(RESET == (DMA_INTF0(dma_dev->dma_periph) & flag)) {
                if(SET == hals_dma_flag_get(dma_dev->dma_periph, dma_dev->channel, DMA_INTF_TAEIF)) {
                    /* when error occurs, clear all flags and output error message */
                    dma_dev->error_state = HAL_DMA_ERROR_TRANSFER;
                    dma_dev->state       = HAL_DMA_STATE_READY;
                    hals_dma_flag_clear(dma_dev->dma_periph, dma_dev->channel, \
                                        (DMA_FLAG_FEE | DMA_FLAG_SDE | DMA_FLAG_TAE | DMA_FLAG_HTF | DMA_FLAG_FTF));
                    HAL_DEBUGE("dma transfer error, poll stop");

                    error_code = HAL_ERR_HARDWARE;
                } else {
                    /* do nothing */
                }

                if(HAL_ERR_NONE == error_code) {
                    /* check for the timeout */
                    if(HAL_TIMEOUT_FOREVER != timeout_ms) {
                        if(SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) {
                            dma_dev->error_state = HAL_DMA_ERROR_TIMEOUT;
                            dma_dev->state       = HAL_DMA_STATE_TIMEOUT;
                            /* when timeout occurs, output timeout warning message */
                            HAL_DEBUGE("dma transfer state poll timeout");
                            error_code = HAL_ERR_TIMEOUT;
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }
        } else {
            while(RESET == (DMA_INTF1(dma_dev->dma_periph) & flag)) {
                if(SET == hals_dma_flag_get(dma_dev->dma_periph, dma_dev->channel, DMA_INTF_TAEIF)) {
                    /* when error occurs, clear all flags and output error message */
                    dma_dev->error_state = HAL_DMA_ERROR_TRANSFER;
                    dma_dev->state       = HAL_DMA_STATE_READY;
                    hals_dma_flag_clear(dma_dev->dma_periph, dma_dev->channel, \
                                        (DMA_FLAG_FEE | DMA_FLAG_SDE | DMA_FLAG_TAE | DMA_FLAG_HTF | DMA_FLAG_FTF));
                    HAL_DEBUGE("dma transfer error, poll stop");
                    error_code = HAL_ERR_HARDWARE;
                } else {
                    /* do nothing */
                }

                if(HAL_ERR_NONE == error_code) {
                    /* check for the timeout */
                    if(HAL_TIMEOUT_FOREVER != timeout_ms) {
                        if(SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) {
                            dma_dev->error_state = HAL_DMA_ERROR_TIMEOUT;
                            dma_dev->state       = HAL_DMA_STATE_TIMEOUT;
                            /* when timeout occurs, output timeout warning message */
                            HAL_DEBUGE("dma transfer state poll timeout");
                            error_code = HAL_ERR_TIMEOUT;
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }
        }
    } else {
        /* do nothing */
    }

    /* clear all flags */
    hals_dma_flag_clear(dma_dev->dma_periph, dma_dev->channel, \
                        (DMA_FLAG_FEE | DMA_FLAG_SDE | DMA_FLAG_TAE | DMA_FLAG_HTF | DMA_FLAG_FTF));

    if(HAL_DMA_TRANSFER_FULL == transfer_state) {
        dma_dev->state = HAL_DMA_STATE_READY;
    }

    /* unlock DMA */
    HAL_UNLOCK(dma_dev);

    return error_code;
}

/*!
    \brief      start transferring amounts of data by interrupt method,the function is non-blocking
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  src_addr: the source memory buffer address
    \param[in]  dst_addr: the destination memory buffer address
    \param[in]  length: the number of data to be transferred from source to destination
    \param[in]  p_irq: device interrupt callback function structure
                  The structure member can be assigned as following parameters:
      \arg        fifo_error_handle: fifo error interrupt handler pointer
      \arg        single_error_handle: channel single mode transfer error interrupt handler pointer
      \arg        error_handle: channel transfer error interrupt handler pointer
      \arg        half_finish_handle: channel half transfer finish interrupt handler pointer
      \arg        m1_half_finish_handle: memory1 channel half transfer finish interrupt handler pointer
      \arg        full_finish_handle: channel full transfer finish interrupt handler pointer
      \arg        m1_full_finish_handle: memory1 channel full transfer finish interrupt handler pointer
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_dma_start_interrupt(hal_dma_dev_struct *dma_dev, uint32_t src_addr, uint32_t dst_addr, uint16_t length, \
                                hal_dma_irq_struct *p_irq)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check for the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the DMA pointer address and the number length parameter */
    if((NULL == dma_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [dma_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    /* check src_addr parameter */
    if(0U == (src_addr & DEV_VALID_ADDRESS)) {
        HAL_DEBUGE("parameter [src_addr] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check dst_addr parameter */
    if(0U == (dst_addr & DEV_VALID_ADDRESS)) {
        HAL_DEBUGE("parameter [dst_addr] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DMA */
    HAL_LOCK(dma_dev);

    if(HAL_DMA_STATE_READY == dma_dev->state) {
        dma_dev->state       = HAL_DMA_STATE_BUSY;
        dma_dev->error_state = HAL_DMA_ERROR_NONE;
        /* set the handler pointer  */
        hals_dma_channel_disable(dma_dev->dma_periph, dma_dev->channel);

        /* enable the FIFO error interrupt */
        if(NULL != p_irq->fifo_error_handle) {
            dma_dev->dma_irq.fifo_error_handle = p_irq->fifo_error_handle;
            hals_dma_interrupt_enable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_FEE);
        } else {
            hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_FEE);
        }

        /* enable the single mode transfer channel error interrupt */
        if(NULL != p_irq->single_error_handle) {
            dma_dev->dma_irq.single_error_handle = p_irq->single_error_handle;
            hals_dma_interrupt_enable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_SDE);
        } else {
            hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_SDE);
        }

        /* enable the transfer channel error interrupt */
        if(NULL != p_irq->error_handle) {
            dma_dev->dma_irq.error_handle = p_irq->error_handle;
            hals_dma_interrupt_enable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_TAE);
        } else {
            hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_TAE);
        }

        /* set the full finish handler pointer and enable the channel full transfer finish  interrupt */
        if((NULL != p_irq->full_finish_handle) || (NULL != p_irq->m1_full_finish_handle)) {

            dma_dev->dma_irq.m1_full_finish_handle = p_irq->m1_full_finish_handle;
            dma_dev->dma_irq.full_finish_handle = p_irq->full_finish_handle;

            hals_dma_interrupt_enable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_FTF);
        } else {
            hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_FTF);
        }

        /* set the half finish handler pointer and enable the channel half transfer finish interrupt */
        if((NULL != p_irq->half_finish_handle) || (NULL != p_irq->m1_half_finish_handle)) {

            dma_dev->dma_irq.m1_half_finish_handle = p_irq->m1_half_finish_handle;
            dma_dev->dma_irq.half_finish_handle = p_irq->half_finish_handle;

            hals_dma_interrupt_enable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_HTF);
        } else {
            hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_HTF);
        }

        /* check the direction of the data transfer */
        if(DMA_MEMORY_TO_PERIPH == (DMA_CHCTL(dma_dev->dma_periph, dma_dev->channel) & DMA_CHXCTL_TM)) {
            /* configure the transfer destination and source address */
            hals_dma_memory_address_config(dma_dev->dma_periph, dma_dev->channel, (uint8_t)DMA_MEMORY_0, src_addr);
            hals_dma_periph_address_config(dma_dev->dma_periph, dma_dev->channel, dst_addr);
        } else {
            /* configure the transfer destination and source address */
            hals_dma_memory_address_config(dma_dev->dma_periph, dma_dev->channel, (uint8_t)DMA_MEMORY_0, dst_addr);
            hals_dma_periph_address_config(dma_dev->dma_periph, dma_dev->channel, src_addr);
        }
        /* configure the transfer number */
        hals_dma_transfer_number_config(dma_dev->dma_periph, dma_dev->channel, (uint32_t)length);

        /* enable DMA channel */
        hals_dma_channel_enable(dma_dev->dma_periph, dma_dev->channel);
    } else {
        error_code = HAL_ERR_BUSY;
    }

    /* unlock DMA */
    HAL_UNLOCK(dma_dev);

    return error_code;
}

/*!
    \brief      stop transferring amounts of data by interrupt method,the function is non-blocking
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_ALREADY_DONE \
                details refer to gd32h7xx_hal.h
*/
int32_t hal_dma_stop_interrupt(hal_dma_dev_struct *dma_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check for the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(NULL == dma_dev) {
        HAL_DEBUGE("pointer [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_DMA_STATE_BUSY != dma_dev->state) {
        /* no transfer ongoing */
        dma_dev->error_state = HAL_DMA_ERROR_NOTRANSFER;
        error_code = HAL_ERR_ALREADY_DONE;
    } else {
        /* disable DMA channel */
        hals_dma_channel_disable(dma_dev->dma_periph, dma_dev->channel);

        /* disable DMA IT */
        hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, \
                                   DMA_INT_FEE | DMA_INT_SDE | DMA_INT_TAE | DMA_INT_HTF | DMA_INT_FTF);

        /* check abort callback valid */
        if(NULL != dma_dev->dma_irq.abort_handle) {
            dma_dev->dma_irq.abort_handle(dma_dev);
        }

        dma_dev->state = HAL_DMA_STATE_READY;

        /* reset the interrupt handle */
        dma_dev->dma_irq.single_error_handle    = NULL;
        dma_dev->dma_irq.error_handle           = NULL;
        dma_dev->dma_irq.full_finish_handle     = NULL;
        dma_dev->dma_irq.half_finish_handle     = NULL;
        dma_dev->dma_irq.m1_full_finish_handle  = NULL;
        dma_dev->dma_irq.m1_half_finish_handle  = NULL;
        dma_dev->dma_irq.fifo_error_handle      = NULL;
        dma_dev->dma_irq.abort_handle           = NULL;

        /* clear all flags */
        hals_dma_flag_clear(dma_dev->dma_periph, dma_dev->channel, \
                            (DMA_FLAG_FEE | DMA_FLAG_SDE | DMA_FLAG_TAE | DMA_FLAG_HTF | DMA_FLAG_FTF));
    }

    /* unlock DMA */
    HAL_UNLOCK(dma_dev);

    return error_code;
}

/*!
    \brief      start DMA transferring
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  src_addr: the source memory buffer address
    \param[in]  dst_addr: the destination memory buffer address
    \param[in]  length: the number of data to be transferred from source to destination
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE,HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_dma_start(hal_dma_dev_struct *dma_dev, uint32_t src_addr, uint32_t dst_addr, uint16_t length)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check for the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(NULL == dma_dev) {
        HAL_DEBUGE("pointer [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(0U == (src_addr & DEV_VALID_ADDRESS)) {
        HAL_DEBUGE("parameter [src_addr] value is invalid");
        return HAL_ERR_VAL;
    }
    if(0U == (dst_addr & DEV_VALID_ADDRESS)) {
        HAL_DEBUGE("parameter [dst_addr] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DMA */
    HAL_LOCK(dma_dev);
    if(HAL_DMA_STATE_READY == dma_dev->state) {
        dma_dev->state       = HAL_DMA_STATE_BUSY;
        dma_dev->error_state = HAL_DMA_ERROR_NONE;
        /* disable DMA channel */
        hals_dma_channel_disable(dma_dev->dma_periph, dma_dev->channel);

        if(DMA_MEMORY_TO_PERIPH == (DMA_CHCTL(dma_dev->dma_periph, dma_dev->channel) & DMA_CHXCTL_TM)) {
            /* configure the transfer destination and source address */
            hals_dma_memory_address_config(dma_dev->dma_periph, dma_dev->channel, (uint8_t)DMA_MEMORY_0, src_addr);
            hals_dma_periph_address_config(dma_dev->dma_periph, dma_dev->channel, dst_addr);
        } else {
            /* configure the transfer destination and source address */
            hals_dma_memory_address_config(dma_dev->dma_periph, dma_dev->channel, (uint8_t)DMA_MEMORY_0, dst_addr);
            hals_dma_periph_address_config(dma_dev->dma_periph, dma_dev->channel, src_addr);
        }

        /* configure the transfer number */
        hals_dma_transfer_number_config(dma_dev->dma_periph, dma_dev->channel, (uint32_t)length);
        /* enable DMA channel */
        hals_dma_channel_enable(dma_dev->dma_periph, dma_dev->channel);
    } else {
        error_code = HAL_ERR_BUSY;
    }

    /* unlock DMA */
    HAL_UNLOCK(dma_dev);

    return error_code;
}

/*!
    \brief      stop DMA transferring
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE,HAL_ERR_ALREADY_DONE details refer to gd32h7xx_hal.h
*/
int32_t hal_dma_stop(hal_dma_dev_struct *dma_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check for the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(NULL == dma_dev) {
        HAL_DEBUGE("pointer [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_DMA_STATE_BUSY != dma_dev->state) {
        /* dma not in transfer state */
        dma_dev->error_state = HAL_DMA_ERROR_NOTRANSFER;
        error_code = HAL_ERR_ALREADY_DONE;
    } else {
        /* disable DMA channel */
        hals_dma_channel_disable(dma_dev->dma_periph, dma_dev->channel);
        /* clear all flags */
        hals_dma_flag_clear(dma_dev->dma_periph, dma_dev->channel, \
                            (DMA_FLAG_FEE | DMA_FLAG_SDE | DMA_FLAG_TAE | DMA_FLAG_HTF | DMA_FLAG_FTF));
    }

    dma_dev->state = HAL_DMA_STATE_READY;

    /* unlock DMA */
    HAL_UNLOCK(dma_dev);

    return error_code;
}

/*!
    \brief      get the DMA state
    \param[in]  dma_dev: dma device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     state: refer to <hal_dma_state_enum>
*/
hal_dma_state_enum hal_dma_state_get(hal_dma_dev_struct *dma_dev)
{
    return dma_dev->state;
}

/*!
    \brief      get the DMA error state
    \param[in]  dma_dev: dma device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error state: refer to <hal_dma_error_enum>
*/
hal_dma_error_enum hal_dma_error_get(hal_dma_dev_struct *dma_dev)
{
    return dma_dev->error_state;
}

/*!
    \brief      enable DMA switch buffer mode
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dma_switch_buffer_mode_enable(uint32_t dma_periph, hal_dma_channel_enum channelx)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* enable switch buffer mode */
    DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_SBMEN;

    return error_code;
}

/*!
    \brief      disable DMA switch buffer mode
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dma_switch_buffer_mode_disable(uint32_t dma_periph, hal_dma_channel_enum channelx)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable switch buffer mode */
    DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_SBMEN;

    return error_code;
}

/*!
    \brief      start DMA multi buffer transfer
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  src_addr: the source memory buffer address
    \param[in]  dst_addr: the destination memory buffer address
    \param[in]  second_memory_address: the second memory buffer address in case of multi buffer transfer
    \param[in]  length: the number of data to be transferred from source to destination
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_dma_multi_buffer_start(hal_dma_dev_struct *dma_dev, uint32_t src_addr, uint32_t dst_addr, \
                                   uint32_t second_memory_address, uint16_t length)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(NULL == dma_dev) {
        HAL_DEBUGE("pointer [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(0U == (src_addr & DEV_VALID_ADDRESS)) {
        HAL_DEBUGE("parameter [src_addr] value is invalid");
        return HAL_ERR_VAL;
    }
    if(0U == (dst_addr & DEV_VALID_ADDRESS)) {
        HAL_DEBUGE("parameter [dst_addr] value is invalid");
        return HAL_ERR_VAL;
    }
    if(0U == (second_memory_address & DEV_VALID_ADDRESS)) {
        HAL_DEBUGE("parameter [second_memory_address] value is invalid");
        return HAL_ERR_VAL;
    }
    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DMA */
    HAL_LOCK(dma_dev);

    if(HAL_DMA_STATE_READY == dma_dev->state) {
        /* memory to memory transfer not supported in switch buffer mode */
        if(DMA_MEMORY_TO_MEMORY == (DMA_CHCTL(dma_dev->dma_periph, dma_dev->channel) & DMA_MEMORY_TO_MEMORY)) {
            dma_dev->error_state = HAL_DMA_ERROR_UNSUPPORT;
            /* unlock DMA */
            HAL_UNLOCK(dma_dev);
            error_code = HAL_ERR_NO_SUPPORT;
        } else {
            /* do nothing */
        }

        if(HAL_ERR_NONE == error_code) {
            dma_dev->state       = HAL_DMA_STATE_BUSY;
            dma_dev->error_state = HAL_DMA_ERROR_NONE;

            /* disable DMA channel */
            hals_dma_channel_disable(dma_dev->dma_periph, dma_dev->channel);

            /* configure switch buffer */
            hals_dma_switch_buffer_mode_config(dma_dev->dma_periph, dma_dev->channel, second_memory_address, DMA_MEMORY_1);

            /* clear all flags */
            hals_dma_flag_clear(dma_dev->dma_periph, dma_dev->channel, \
                                (DMA_FLAG_FEE | DMA_FLAG_SDE | DMA_FLAG_TAE | DMA_FLAG_HTF | DMA_FLAG_FTF));

            if(DMA_MEMORY_TO_PERIPH == (DMA_CHCTL(dma_dev->dma_periph, dma_dev->channel) & DMA_CHXCTL_TM)) {
                /* configure the transfer destination and source address */
                hals_dma_memory_address_config(dma_dev->dma_periph, dma_dev->channel, (uint8_t)DMA_MEMORY_0, src_addr);
                hals_dma_periph_address_config(dma_dev->dma_periph, dma_dev->channel, dst_addr);
            } else {
                /* configure the transfer destination and source address */
                hals_dma_memory_address_config(dma_dev->dma_periph, dma_dev->channel, (uint8_t)DMA_MEMORY_0, dst_addr);
                hals_dma_periph_address_config(dma_dev->dma_periph, dma_dev->channel, src_addr);
            }

            /* configure the transfer number */
            hals_dma_transfer_number_config(dma_dev->dma_periph, dma_dev->channel, (uint32_t)length);

            /* enable DMA channel */
            hals_dma_channel_enable(dma_dev->dma_periph, dma_dev->channel);
        } else {
            /* do nothing */
        }
    } else {
        dma_dev->error_state = HAL_DMA_ERROR_BUSY;
        error_code = HAL_ERR_BUSY;
    }

    /* unlock DMA */
    HAL_UNLOCK(dma_dev);

    return error_code;
}

/*!
    \brief      start DMA multi buffer transfer
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  src_addr: the source memory buffer address
    \param[in]  dst_addr: the destination memory buffer address
    \param[in]  second_memory_address: the second memory buffer address in case of multi buffer transfer
    \param[in]  length: the number of data to be transferred from source to destination
    \param[in]  p_irq: the pointer to the DMA interrupt structure
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE,HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NO_SUPPORT details refer to gd32h7xx_hal.h
*/
int32_t hal_dma_multi_buffer_start_interrupt(hal_dma_dev_struct *dma_dev, uint32_t src_addr, uint32_t dst_addr, \
                                             uint32_t second_memory_address, uint16_t length, hal_dma_irq_struct *p_irq)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(NULL == dma_dev) {
        HAL_DEBUGE("pointer [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(0U == (src_addr & DEV_VALID_ADDRESS)) {
        HAL_DEBUGE("parameter [src_addr] value is invalid");
        return HAL_ERR_VAL;
    }
    if(0U == (dst_addr & DEV_VALID_ADDRESS)) {
        HAL_DEBUGE("parameter [dst_addr] value is invalid");
        return HAL_ERR_VAL;
    }
    if(0U == (second_memory_address & DEV_VALID_ADDRESS)) {
        HAL_DEBUGE("parameter [second_memory_address] value is invalid");
        return HAL_ERR_VAL;
    }
    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DMA */
    HAL_LOCK(dma_dev);

    if(HAL_DMA_STATE_READY == dma_dev->state) {
        /* memory to memory transfer not supported in switch buffer mode */
        if(DMA_MEMORY_TO_MEMORY == (DMA_CHCTL(dma_dev->dma_periph, dma_dev->channel) & DMA_MEMORY_TO_MEMORY)) {
            dma_dev->error_state = HAL_DMA_ERROR_UNSUPPORT;
            /* unlock DMA */
            HAL_UNLOCK(dma_dev);
            error_code = HAL_ERR_NO_SUPPORT;
        } else {
            /* do nothing */
        }

        if(HAL_ERR_NONE == error_code) {
            dma_dev->state       = HAL_DMA_STATE_BUSY;
            dma_dev->error_state = HAL_DMA_ERROR_NONE;

            /* disable DMA channel */
            hals_dma_channel_disable(dma_dev->dma_periph, dma_dev->channel);

            /* configure switch buffer */
            hals_dma_switch_buffer_mode_config(dma_dev->dma_periph, dma_dev->channel, second_memory_address, DMA_MEMORY_1);

            /* enable the FIFO error interrupt */
            if(NULL != p_irq->fifo_error_handle) {
                dma_dev->dma_irq.fifo_error_handle = p_irq->fifo_error_handle;
                hals_dma_interrupt_enable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_FEE);
            } else {
                hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_FEE);
            }

            /* enable the single mode transfer channel error interrupt */
            if(NULL != p_irq->single_error_handle) {
                dma_dev->dma_irq.single_error_handle = p_irq->single_error_handle;
                hals_dma_interrupt_enable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_SDE);
            } else {
                hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_SDE);
            }

            /* enable the transfer channel error interrupt */
            if(NULL != p_irq->error_handle) {
                dma_dev->dma_irq.error_handle = p_irq->error_handle;
                hals_dma_interrupt_enable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_TAE);
            } else {
                hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_TAE);
            }

            /* set the full finish handler pointer and enable the channel full transfer finish  interrupt */
            if((NULL != p_irq->full_finish_handle) || (NULL != p_irq->m1_full_finish_handle)) {

                dma_dev->dma_irq.m1_full_finish_handle = p_irq->m1_full_finish_handle;
                dma_dev->dma_irq.full_finish_handle = p_irq->full_finish_handle;

                hals_dma_interrupt_enable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_FTF);
            } else {
                hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_FTF);
            }

            /* set the half finish handler pointer and enable the channel half transfer finish interrupt */
            if((NULL != p_irq->half_finish_handle) || (NULL != p_irq->m1_half_finish_handle)) {

                dma_dev->dma_irq.m1_half_finish_handle = p_irq->m1_half_finish_handle;
                dma_dev->dma_irq.half_finish_handle = p_irq->half_finish_handle;

                hals_dma_interrupt_enable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_HTF);
            } else {
                hals_dma_interrupt_disable(dma_dev->dma_periph, dma_dev->channel, DMA_INT_HTF);
            }

            /* clear all flags */
            hals_dma_flag_clear(dma_dev->dma_periph, dma_dev->channel, \
                                (DMA_FLAG_FEE | DMA_FLAG_SDE | DMA_FLAG_TAE | DMA_FLAG_HTF | DMA_FLAG_FTF));

            if (DMA_MEMORY_TO_PERIPH == (DMA_CHCTL(dma_dev->dma_periph, dma_dev->channel) & DMA_CHXCTL_TM)) {
                /* configure the transfer destination and source address */
                hals_dma_memory_address_config(dma_dev->dma_periph, dma_dev->channel, (uint8_t)DMA_MEMORY_0, src_addr);
                hals_dma_periph_address_config(dma_dev->dma_periph, dma_dev->channel, dst_addr);
            } else {
                /* configure the transfer destination and source address */
                hals_dma_memory_address_config(dma_dev->dma_periph, dma_dev->channel, (uint8_t)DMA_MEMORY_0, dst_addr);
                hals_dma_periph_address_config(dma_dev->dma_periph, dma_dev->channel, src_addr);
            }

            /* configure the transfer number */
            hals_dma_transfer_number_config(dma_dev->dma_periph, dma_dev->channel, (uint32_t)length);

            /* enable DMA channel */
            hals_dma_channel_enable(dma_dev->dma_periph, dma_dev->channel);
        } else {
            /* do nothing */
        }
    } else {
        dma_dev->error_state = HAL_DMA_ERROR_BUSY;
        error_code = HAL_ERR_BUSY;
    }

    /* unlock DMA */
    HAL_UNLOCK(dma_dev);

    return error_code;

}

/*!
    \brief      DMAMUX interrupt handler content function,which is merely used in dmamux_handler
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_dmamux_irq(hal_dma_dev_struct *dma_dev)
{
    int32_t error_code            = HAL_ERR_NONE;
    uint32_t sync_overrun_intf    = 0U;
    uint32_t trigger_overrun_intf = 0U;

    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the DMA pointer address and the number length parameter */
    if(NULL == dma_dev) {
        HAL_DEBUGE("pointer [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    sync_overrun_intf = DMAMUX_REGIDX_BIT2(DMAMUX_RM_INTF_REG_OFFSET, dma_dev->multi_channel, \
                                           DMAMUX_RM_CH0CFG_REG_OFFSET + ((uint32_t)dma_dev->multi_channel * 4U), 8U);
    /* sync overrun interrupt handler */
    if(SET == hals_dmamux_interrupt_flag_get((hal_dmamux_interrupt_flag_enum)sync_overrun_intf)) {
        hals_dmamux_interrupt_flag_clear((hal_dmamux_interrupt_flag_enum)sync_overrun_intf);
        if(dma_dev->dmamux_irq.sync_overrun_handle != NULL) {
            dma_dev->dmamux_irq.sync_overrun_handle(dma_dev);
            /* unlock DMA */
            dma_dev->state = HAL_DMA_STATE_READY;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if(0U != dma_dev->gen_enable) {
        trigger_overrun_intf = DMAMUX_REGIDX_BIT2(DMAMUX_RG_INTF_REG_OFFSET, dma_dev->multi_channel, \
                                                  DMAMUX_RG_CH0CFG_REG_OFFSET + ((uint32_t)dma_dev->multi_channel * 4U), 8U);
        /* trigger overrun interrupt handler */
        if(SET == hals_dmamux_interrupt_flag_get((hal_dmamux_interrupt_flag_enum)trigger_overrun_intf)) {
            hals_dmamux_interrupt_flag_clear((hal_dmamux_interrupt_flag_enum)trigger_overrun_intf);
            if(dma_dev->dmamux_irq.sync_overrun_handle != NULL) {
                dma_dev->dmamux_irq.sync_overrun_handle(dma_dev);
                /* unlock DMA */
                dma_dev->state = HAL_DMA_STATE_READY;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    HAL_UNLOCK(dma_dev);

    return error_code;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to DMAMUX interrupt callback function pointer structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_dmamux_irq_handle_set(hal_dma_dev_struct *dma_dev, hal_dmamux_irq_struct *p_irq)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t interrupt = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the DMA pointer address and the number length parameter */
    if((NULL == dma_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [dma_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DMA */
    HAL_LOCK(dma_dev);

    /* disable DMAMUX channel */
    hals_dmamux_synchronization_disable(dma_dev->multi_channel);
    /* enable the sync overrun interrupt */
    if(NULL != p_irq->sync_overrun_handle) {
        interrupt = DMAMUX_REGIDX_BIT(DMAMUX_RM_CH0CFG_REG_OFFSET + (dma_dev->multi_channel * 4U), 8U);
        dma_dev->dmamux_irq.sync_overrun_handle = p_irq->sync_overrun_handle;
        hals_dmamux_interrupt_enable((hal_dmamux_interrupt_enum)interrupt);
    } else {
        interrupt = DMAMUX_REGIDX_BIT(DMAMUX_RM_CH0CFG_REG_OFFSET + (dma_dev->multi_channel * 4U), 8U);
        dma_dev->dmamux_irq.sync_overrun_handle = NULL;
        hals_dmamux_interrupt_disable((hal_dmamux_interrupt_enum)interrupt);
    }
    /* enable DMAMUX channel */
    hals_dmamux_synchronization_enable(dma_dev->multi_channel);

    if(0U != dma_dev->gen_enable) {
        /* disable DMAMUX channel */
        DMAMUX_RG_CHXCFG(dma_dev->gen_channel) &= (~DMAMUX_RG_CHXCFG_RGEN);
        /* enable the trigger overrun interrupt */
        if(NULL != p_irq->trigger_overrun_handle) {
            dma_dev->dmamux_irq.trigger_overrun_handle = p_irq->trigger_overrun_handle;
            interrupt = DMAMUX_REGIDX_BIT(DMAMUX_RG_CH0CFG_REG_OFFSET + (dma_dev->multi_channel * 4U), 8U);
            hals_dmamux_interrupt_enable((hal_dmamux_interrupt_enum)interrupt);
        } else {
            interrupt = DMAMUX_REGIDX_BIT(DMAMUX_RG_CH0CFG_REG_OFFSET + (dma_dev->multi_channel * 4U), 8U);
            dma_dev->dmamux_irq.trigger_overrun_handle = NULL;
            hals_dmamux_interrupt_disable((hal_dmamux_interrupt_enum)interrupt);
        }
        /* enable DMAMUX channel */
        DMAMUX_RG_CHXCFG(dma_dev->gen_channel) |= DMAMUX_RG_CHXCFG_RGEN;
    } else {
        /* do nothing */
    }

    /* unlock DMA */
    HAL_UNLOCK(dma_dev);

    return error_code;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_dmamux_irq_handle_all_reset(hal_dma_dev_struct *dma_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the DMA pointer address and the number length parameter */
    if(NULL == dma_dev) {
        HAL_DEBUGE("pointer [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DMA */
    HAL_LOCK(dma_dev);
    /* DMAMUX interrupt handler reset */
    dma_dev->dmamux_irq.sync_overrun_handle    = NULL;
    dma_dev->dmamux_irq.trigger_overrun_handle = NULL;
    /* unlock DMA */
    HAL_UNLOCK(dma_dev);

    return error_code;
}

/*!
    \brief      start transferring amounts of data by interrupt method,the function is non-blocking
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: device interrupt callback function structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dmamux_start_interrupt(hal_dma_dev_struct *dma_dev, hal_dmamux_irq_struct *p_irq)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t interrupt = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the DMA pointer address and the number length parameter */
    if((NULL == dma_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [dma_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DMA */
    HAL_LOCK(dma_dev);
    /* set the handler pointer  */
    hals_dmamux_synchronization_disable(dma_dev->multi_channel);
    /* enable the sync overrun interrupt */
    if(NULL != p_irq->sync_overrun_handle) {
        interrupt = DMAMUX_REGIDX_BIT(DMAMUX_RM_CH0CFG_REG_OFFSET + (dma_dev->multi_channel * 4U), 8U);
        dma_dev->dmamux_irq.sync_overrun_handle = p_irq->sync_overrun_handle;
        hals_dmamux_interrupt_enable((hal_dmamux_interrupt_enum)interrupt);
    } else {
        interrupt = DMAMUX_REGIDX_BIT(DMAMUX_RM_CH0CFG_REG_OFFSET + (dma_dev->multi_channel * 4U), 8U);
        dma_dev->dmamux_irq.sync_overrun_handle = NULL;
        hals_dmamux_interrupt_disable((hal_dmamux_interrupt_enum)interrupt);
    }

    interrupt = DMAMUX_REGIDX_BIT(DMAMUX_RM_INTF_REG_OFFSET, dma_dev->multi_channel);
    /* clear all flags */
    hals_dmamux_flag_clear((hal_dmamux_flag_enum)interrupt);
    /* enable DMAMUX channel */
    hals_dmamux_synchronization_enable(dma_dev->multi_channel);

    if(0U != dma_dev->gen_enable) {
        DMAMUX_RG_CHXCFG(dma_dev->gen_channel) &= (~DMAMUX_RG_CHXCFG_RGEN);
        /* enable the trigger overrun interrupt */
        if(NULL != p_irq->trigger_overrun_handle) {
            interrupt = DMAMUX_REGIDX_BIT(DMAMUX_RG_CH0CFG_REG_OFFSET + (dma_dev->gen_channel * 4U), 8U);
            dma_dev->dmamux_irq.trigger_overrun_handle = p_irq->trigger_overrun_handle;
            hals_dmamux_interrupt_enable((hal_dmamux_interrupt_enum)interrupt);
        } else {
            interrupt = DMAMUX_REGIDX_BIT(DMAMUX_RG_CH0CFG_REG_OFFSET + (dma_dev->gen_channel * 4U), 8U);
            dma_dev->dmamux_irq.trigger_overrun_handle = NULL;
            hals_dmamux_interrupt_disable((hal_dmamux_interrupt_enum)interrupt);
        }

        interrupt = DMAMUX_REGIDX_BIT(DMAMUX_RG_INTF_REG_OFFSET, dma_dev->gen_channel);
        /* clear all flags */
        hals_dmamux_flag_clear((hal_dmamux_flag_enum)interrupt);
        /* enable DMAMUX generator channel */
        DMAMUX_RG_CHXCFG(dma_dev->gen_channel) |= DMAMUX_RG_CHXCFG_RGEN;
    } else {
        /* do nothing */
    }

    /* unlock DMA */
    HAL_UNLOCK(dma_dev);

    return error_code;
}

/*!
    \brief      stop transferring amounts of data by interrupt method,the function is non-blocking
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_ALREADY_DONE, HAL_ERR_ALREADY_DONE
                  details refer to gd32h7xx_hal.h
*/
int32_t hal_dmamux_stop_interrupt(hal_dma_dev_struct *dma_dev)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t interrupt  = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(NULL == dma_dev) {
        HAL_DEBUGE("pointer [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DMA */
    HAL_LOCK(dma_dev);

    if(HAL_DMA_STATE_BUSY != dma_dev->state) {
        /* no transfer ongoing */
        dma_dev->error_state = HAL_DMA_ERROR_NOTRANSFER;
        error_code = HAL_ERR_ALREADY_DONE;
    } else {

        interrupt = (DMAMUX_REGIDX_BIT(DMAMUX_RM_CH0CFG_REG_OFFSET + (dma_dev->multi_channel * 4U), 8U));
        /* disable DMAMUX channel */
        hals_dmamux_synchronization_disable(dma_dev->multi_channel);
        /* disable DMAMUX IT */
        hals_dmamux_interrupt_disable((hal_dmamux_interrupt_enum)interrupt);
        /* reset the interrupt handle */
        dma_dev->dmamux_irq.sync_overrun_handle = NULL;
        interrupt = DMAMUX_REGIDX_BIT(DMAMUX_RM_INTF_REG_OFFSET, dma_dev->multi_channel);
        /* clear all flags */
        hals_dmamux_flag_clear((hal_dmamux_flag_enum)interrupt);

        if(0U != dma_dev->gen_enable) {
            /* disable DMAMUX channel */
            DMAMUX_RG_CHXCFG(dma_dev->gen_channel) &= (~DMAMUX_RG_CHXCFG_RGEN);

            interrupt = (DMAMUX_REGIDX_BIT(DMAMUX_RG_CH0CFG_REG_OFFSET + (dma_dev->gen_channel * 4U), 8U));
            /* disable DMAMUX IT */
            hals_dmamux_interrupt_disable((hal_dmamux_interrupt_enum)interrupt);
            /* reset the interrupt handle */
            dma_dev->dmamux_irq.sync_overrun_handle = NULL;
            interrupt = DMAMUX_REGIDX_BIT(DMAMUX_RG_INTF_REG_OFFSET, dma_dev->gen_channel);
            /* clear all flags */
            hals_dmamux_flag_clear((hal_dmamux_flag_enum)interrupt);
        } else {
            /* do nothing */
        }
    }

    /* unlock DMA */
    HAL_UNLOCK(dma_dev);

    return error_code;
}

/*!
    \brief      start DMAMUX
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_dmamux_start(hal_dma_dev_struct *dma_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(NULL == dma_dev) {
        HAL_DEBUGE("pointer [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    hals_dmamux_synchronization_enable(dma_dev->multi_channel);

    if(0U != dma_dev->gen_enable) {
        DMAMUX_RG_CHXCFG(dma_dev->gen_channel) |= DMAMUX_RG_CHXCFG_RGEN;
    }

    return error_code;
}

/*!
    \brief      stop DMAMUX
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_dmamux_stop(hal_dma_dev_struct *dma_dev)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t interrupt  = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(NULL == dma_dev) {
        HAL_DEBUGE("pointer [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    hals_dmamux_synchronization_disable(dma_dev->multi_channel);
    interrupt = DMAMUX_REGIDX_BIT(DMAMUX_RM_INTF_REG_OFFSET, dma_dev->multi_channel);
    hals_dmamux_flag_clear((hal_dmamux_flag_enum)interrupt);
    if(0U != dma_dev->gen_enable) {
        interrupt = DMAMUX_REGIDX_BIT(DMAMUX_RG_INTF_REG_OFFSET, dma_dev->gen_channel);
        DMAMUX_RG_CHXCFG(dma_dev->gen_channel) &= (~DMAMUX_RG_CHXCFG_RGEN);
        hals_dmamux_flag_clear((hal_dmamux_flag_enum)interrupt);
    }

    return error_code;
}

/*!
    \brief      initialize DMAMUX request multiplexer channel synchronization mode
    \param[in]  channelx: specify which DMAMUX request multiplexer channel is initialized
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_MUXCHx(x=0..15)
    \param[in]  init_struct: the data needed to initialize DMAMUX request multiplexer channel
                  sync_id:
                  only one parameter can be selected which is shown as below:
      \arg          DMAMUX_SYNC_x x is the type of request
                  sync_polarity:
      \arg          DMAMUX_SYNC_NO_EVENT:synchronization input polarity is no event
      \arg          DMAMUX_SYNC_RISING: synchronization input polarity is rising
      \arg          DMAMUX_SYNC_FALLING: synchronization input polarity is falling
      \arg          DMAMUX_SYNC_RISING_FALLING: synchronization input polarity is rising and falling
                  request_number: the number of DMA request that will be authorized after a sync event, from 1 to 32
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_dmamux_synchronization_init(hal_dmamux_multiplexer_channel_enum channelx, \
                                        hal_dmamux_sync_parameter_struct *init_struct)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t cfg       = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(NULL == init_struct) {
        HAL_DEBUGE("pointer [init_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(DMAMUX_MUXCH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable synchronization mode and event generation for DMA request forward number configuration */
    DMAMUX_RM_CHXCFG(channelx) &= ~(DMAMUX_RM_CHXCFG_SYNCEN | DMAMUX_RM_CHXCFG_EVGEN);

    /* configure synchronization input identification, synchronization input polarity, number of DMA requests to forward */
    cfg = DMAMUX_RM_CHXCFG(channelx);
    cfg &= ~(DMAMUX_RM_CHXCFG_SYNCID | DMAMUX_RM_CHXCFG_NBR | DMAMUX_RM_CHXCFG_SYNCP);
    cfg |= (init_struct->sync_polarity | (init_struct->sync_id) | RM_CHXCFG_NBR(init_struct->request_number - 1U));
    DMAMUX_RM_CHXCFG(channelx) = cfg;
    if(DMAMUX_SYNC_ENABLE == init_struct->sync_enable) {
        DMAMUX_RM_CHXCFG(channelx) |= DMAMUX_RM_CHXCFG_SYNCEN;
    } else {
        /* do nothing */
    }

    if(DMAMUX_EVENT_INPUT_ENABLE == init_struct->event_enable) {
        DMAMUX_RM_CHXCFG(channelx) |= DMAMUX_RM_CHXCFG_EVGEN;
    } else {
        /* do nothing */
    }

    return error_code;
}

/*!
    \brief      initialize DMAMUX request generator channel
    \param[in]  channelx: specify which DMAMUX request generator channel is initialized
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_GENCHx(x=0..7)
    \param[in]  init_struct: the data needed to initialize DMAMUX request generator channel
                  trigger_id:
                  only one parameter can be selected which is shown as below:
      \arg          DMAMUX_TRIGGER_x x is the type of request
                  trigger_polarity:
      \arg          DMAMUX_GEN_NO_EVENT: request generator trigger polarity is no event
      \arg          DMAMUX_GEN_RISING: request generator trigger polarity is rising
      \arg          DMAMUX_GEN_FALLING: request generator trigger polarity is falling
      \arg          DMAMUX_GEN_RISING_FALLING: request generator trigger polarity is rising and falling
                  request_number: the number of DMA request that will be generated after a signal event, from 1 to 32
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dmamux_request_generator_init(hal_dmamux_generator_channel_enum channelx, \
                                          hal_dmamux_gen_parameter_struct *init_struct)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t cfg       = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(NULL == init_struct) {
        HAL_DEBUGE("pointer [init_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(DMAMUX_GENCH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable DMAMUX request generator channel for DMA request generation number configuration */
    DMAMUX_RG_CHXCFG(channelx) &= ~(DMAMUX_RG_CHXCFG_RGEN);

    /* configure trigger input identification, trigger polarity, number of DMA requests to be generated */
    cfg = DMAMUX_RG_CHXCFG(channelx);
    cfg &= ~(DMAMUX_RG_CHXCFG_TID | DMAMUX_RG_CHXCFG_NBRG | DMAMUX_RG_CHXCFG_RGTP);
    cfg |= (RG_CHXCFG_NBRG(init_struct->request_number - 1U) | init_struct->trigger_id | init_struct->trigger_polarity);
    DMAMUX_RG_CHXCFG(channelx) = cfg;

    return error_code;
}

/*!
    \brief      enable DMAMUX request generator channel
    \param[in]  channelx: specify which DMAMUX request generator channel is configured
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_GENCHx(x=0..7)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dmamux_request_generator_channel_enable(hal_dmamux_generator_channel_enum channelx)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(DMAMUX_GENCH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMAMUX_RG_CHXCFG(channelx) |= DMAMUX_RG_CHXCFG_RGEN;

    return error_code;
}

/*!
    \brief      disable DMAMUX request generator channel
    \param[in]  channelx: specify which DMAMUX request generator channel is configured
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_GENCHx(x=0..7)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dmamux_request_generator_channel_disable(hal_dmamux_generator_channel_enum channelx)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(DMAMUX_GENCH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMAMUX_RG_CHXCFG(channelx) &= (~DMAMUX_RG_CHXCFG_RGEN);

    return error_code;
}

/*!
    \brief      change the memory0 or memory1 address
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  address: the new address
    \param[in]  memory: the memory to be changed, only one parameter can be selected which is shown as below:
      \arg        DMA_MEMORY_0
      \arg        DMA_MEMORY_1
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_dma_change_memory(hal_dma_dev_struct *dma_dev, uint32_t address, uint32_t memory)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(NULL == dma_dev) {
        HAL_DEBUGE("pointer [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if (DMA_MEMORY_0 == memory) {
        DMA_CHM0ADDR(dma_dev->dma_periph, dma_dev->channel) = address;
    } else {
        DMA_CHM1ADDR(dma_dev->dma_periph, dma_dev->channel) = address;
    }

    return error_code;
}

/*!
    \brief      initialize the DMA single data mode parameters structure with the default values
    \param[in]  none
    \param[out] init_struct: the initialization data needed to initialize DMA channel
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_single_data_para_struct_init(hal_dma_single_data_parameter_struct *init_struct)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == init_struct) {
        HAL_DEBUGE("pointer [init_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set the DMA struct with the default values */
    init_struct->request             = DMA_REQUEST_M2M;
    init_struct->periph_addr         = 0U;
    init_struct->periph_inc          = DMA_PERIPH_INCREASE_DISABLE;
    init_struct->memory0_addr        = 0U;
    init_struct->memory_inc          = DMA_MEMORY_INCREASE_DISABLE;
    init_struct->periph_memory_width = 0U;
    init_struct->circular_mode       = DMA_CIRCULAR_MODE_DISABLE;
    init_struct->direction           = DMA_PERIPH_TO_MEMORY;
    init_struct->number              = 0U;
    init_struct->priority            = DMA_PRIORITY_LOW;
    init_struct->switch_buffer       = DMA_SWITCH_BUFFER_MODE_DISABLE;
    init_struct->buffer_selection    = DMA_MEMORY_0;

    return error_code;
}

/*!
    \brief      initialize the DMA multi data mode parameters structure with the default values
    \param[in]  none
    \param[out] init_struct: the initialization data needed to initialize DMA channel
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_multi_data_para_struct_init(hal_dma_multi_data_parameter_struct *init_struct)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == init_struct) {
        HAL_DEBUGE("pointer [init_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set the DMA structure with the default values */
    init_struct->request            = DMA_REQUEST_M2M;
    init_struct->periph_addr        = 0U;
    init_struct->periph_width       = 0U;
    init_struct->periph_inc         = DMA_PERIPH_INCREASE_DISABLE;
    init_struct->memory0_addr       = 0U;
    init_struct->memory_width       = 0U;
    init_struct->memory_inc         = DMA_MEMORY_INCREASE_DISABLE;
    init_struct->memory_burst_width = 0U;
    init_struct->periph_burst_width = 0U;
    init_struct->critical_value     = DMA_FIFO_1_WORD;
    init_struct->circular_mode      = DMA_CIRCULAR_MODE_DISABLE;
    init_struct->direction          = DMA_PERIPH_TO_MEMORY;
    init_struct->number             = 0U;
    init_struct->priority           = DMA_PRIORITY_LOW;
    init_struct->switch_buffer      = DMA_SWITCH_BUFFER_MODE_DISABLE;
    init_struct->buffer_selection   = DMA_MEMORY_0;

    return error_code;
}

/*!
    \brief      initialize DMA single data mode
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel is initialized
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  init_struct: the data needed to initialize DMA single data mode
                  request:
      \arg          DMA_REQUEST_x x is the type of request
                  periph_addr: peripheral base address
                  periph_inc:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_PERIPH_INCREASE_ENABLE: peripheral increasing
      \arg          DMA_PERIPH_INCREASE_DISABLE: peripheral decreasing
      \arg          DMA_PERIPH_INCREASE_FIX: peripheral increasing fixed
                  memory0_addr: memory base address
                  memory_inc:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_MEMORY_INCREASE_ENABLE: memory increasing
      \arg          DMA_MEMORY_INCREASE_DISABLE: memory decreasing
                  periph_memory_width:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_PERIPH_WIDTH_8BIT: peripheral and memory width is 8bit
      \arg          DMA_PERIPH_WIDTH_16BIT: peripheral and memory width is 16bit
      \arg          DMA_PERIPH_WIDTH_32BIT: peripheral and memory width is 32bit
                  circular_mode:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_CIRCULAR_MODE_ENABLE: circular mode enable
      \arg          DMA_CIRCULAR_MODE_DISABLE: circular mode disable
                  direction:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_PERIPH_TO_MEMORY: peripheral to memory
      \arg          DMA_MEMORY_TO_PERIPH: memory to peripheral
      \arg          DMA_MEMORY_TO_MEMORY: memory to memory
                  number: the number of remaining data to be transferred by the DMA
      \arg        priority:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_PRIORITY_LOW: priority is low
      \arg          DMA_PRIORITY_MEDIUM: priority is medium
      \arg          DMA_PRIORITY_HIGH: priority is high
      \arg          DMA_PRIORITY_ULTRA_HIGH: priority is ultra high
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE,HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_single_data_mode_init(uint32_t dma_periph, hal_dma_channel_enum channelx, \
                                       hal_dma_single_data_parameter_struct *init_struct)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t ctl       = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == init_struct) {
        HAL_DEBUGE("pointer [init_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* select single data mode */
    DMA_CHFCTL(dma_periph, channelx) &= ~DMA_CHXFCTL_MDMEN;

    /* configure peripheral base address */
    DMA_CHPADDR(dma_periph, channelx) = init_struct->periph_addr;

    /* configure memory base address */
    DMA_CHM0ADDR(dma_periph, channelx) = init_struct->memory0_addr;

    /* configure the number of remaining data to be transferred */
    DMA_CHCNT(dma_periph, channelx) = init_struct->number;

    /* configure peripheral and memory transfer width, channel priority, transfer mode */
    ctl = DMA_CHCTL(dma_periph, channelx);
    ctl &= ~(DMA_CHXCTL_PWIDTH | DMA_CHXCTL_MWIDTH | DMA_CHXCTL_PRIO | DMA_CHXCTL_TM);
    ctl |= (init_struct->periph_memory_width | (init_struct->periph_memory_width << 2U) | init_struct->priority | \
            init_struct->direction);
    DMA_CHCTL(dma_periph, channelx) = ctl;

    /* configure switching buffer mode */
    if(DMA_SWITCH_BUFFER_MODE_ENABLE == init_struct->switch_buffer) {
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_SBMEN;
    } else {
        DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_SBMEN;
    }

    /* configure buffer selection */
    if(DMA_MEMORY_1 == init_struct->buffer_selection) {
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_MBS;
    } else {
        DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_MBS;
    }

    /* configure peripheral increasing mode */
    if(DMA_PERIPH_INCREASE_ENABLE == init_struct->periph_inc) {
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_PNAGA;
    } else if(DMA_PERIPH_INCREASE_DISABLE == init_struct->periph_inc) {
        DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_PNAGA;
        DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_PAIF;
    } else {
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_PNAGA;
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_PAIF;
    }

    /* configure memory increasing mode */
    if(DMA_MEMORY_INCREASE_ENABLE == init_struct->memory_inc) {
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_MNAGA;
    } else {
        DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_MNAGA;
    }

    /* configure DMA circular mode */
    if(DMA_CIRCULAR_MODE_ENABLE == init_struct->circular_mode) {
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_CMEN;
    } else {
        DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_CMEN;
    }

    if(DMA0 == dma_periph) {
        DMAMUX_RM_CHXCFG(channelx) &= ~DMAMUX_RM_CHXCFG_MUXID;
        DMAMUX_RM_CHXCFG(channelx) |= init_struct->request;
    } else {
        DMAMUX_RM_CHXCFG((uint32_t)channelx + 8U) &= ~DMAMUX_RM_CHXCFG_MUXID;
        DMAMUX_RM_CHXCFG((uint32_t)channelx + 8U) |= init_struct->request;
    }

    return error_code;
}

/*!
    \brief      initialize DMA multi data mode
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel is initialized
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  init_struct: the data needed to initialize DMA multi data mode
                  request:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_REQUEST_x x is the type of request
                  periph_addr: peripheral base address
                  periph_width:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_PERIPH_WIDTH_8BIT: peripheral width is 8bit
      \arg          DMA_PERIPH_WIDTH_16BIT: peripheral width is 16bit
      \arg          DMA_PERIPH_WIDTH_32BIT: peripheral width is 32bit
                  periph_inc:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_PERIPH_INCREASE_ENABLE: peripheral increasing
      \arg          DMA_PERIPH_INCREASE_DISABLE: peripheral decreasing
      \arg          DMA_PERIPH_INCREASE_FIX: peripheral increasing fixed
                  memory0_addr: memory0 base address
                  memory_width:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_MEMORY_WIDTH_8BIT: memory width is 8bit
      \arg          DMA_MEMORY_WIDTH_16BIT: memory width is 16bit
      \arg          DMA_MEMORY_WIDTH_32BIT: memory width is 32bit
                  memory_inc:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_MEMORY_INCREASE_ENABLE: memory increasing
      \arg          DMA_MEMORY_INCREASE_DISABLE: memory decreasing
                  memory_burst_width:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_MEMORY_BURST_SINGLE: memory burst type is single burst
      \arg          DMA_MEMORY_BURST_4_BEAT: memory burst type is 4-beat burst
      \arg          DMA_MEMORY_BURST_8_BEAT: memory burst type is 8-beat burst
      \arg          DMA_MEMORY_BURST_16_BEAT: memory burst type is 16-beat burst
                  periph_burst_width:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_PERIPH_BURST_SINGLE: peripheral burst type is single burst
      \arg          DMA_PERIPH_BURST_4_BEAT: peripheral burst type is 4-beat burst
      \arg          DMA_PERIPH_BURST_8_BEAT: peripheral burst type is 8-beat burst
      \arg          DMA_PERIPH_BURST_16_BEAT: peripheral burst type is 16-beat burst
                  critical_value:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_FIFO_1_WORD: FIFO counter critical value is 1 word
      \arg          DMA_FIFO_2_WORD: FIFO counter critical value is 2 word
      \arg          DMA_FIFO_3_WORD: FIFO counter critical value is 3 word
      \arg          DMA_FIFO_4_WORD: FIFO counter critical value is 4 word
                  circular_mode:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_CIRCULAR_MODE_ENABLE: circular mode enable
      \arg          DMA_CIRCULAR_MODE_DISABLE: circular mode disable
                  direction:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_PERIPH_TO_MEMORY: peripheral to memory
      \arg          DMA_MEMORY_TO_PERIPH: memory to peripheral
      \arg          DMA_MEMORY_TO_MEMORY: memory to memory
                  number: the number of remaining data to be transferred by the DMA
      \arg        priority:
                  only one parameter can be selected which is shown as below:
      \arg          DMA_PRIORITY_LOW: priority is low
      \arg          DMA_PRIORITY_MEDIUM: priority is medium
      \arg          DMA_PRIORITY_HIGH: priority is high
      \arg          DMA_PRIORITY_ULTRA_HIGH: priority is ultra high
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE,HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_multi_data_mode_init(uint32_t dma_periph, hal_dma_channel_enum channelx, \
                                      hal_dma_multi_data_parameter_struct *init_struct)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t ctl       = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == init_struct) {
        HAL_DEBUGE("pointer [init_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* select multi data mode and configure FIFO critical value */
    DMA_CHFCTL(dma_periph, channelx) |= (DMA_CHXFCTL_MDMEN | init_struct->critical_value);

    /* configure peripheral base address */
    DMA_CHPADDR(dma_periph, channelx) = init_struct->periph_addr;

    /* configure memory base address */
    DMA_CHM0ADDR(dma_periph, channelx) = init_struct->memory0_addr;

    /* configure the number of remaining data to be transferred */
    DMA_CHCNT(dma_periph, channelx) = init_struct->number;

    /* configure peripheral and memory transfer width,channel priority, transfer mode,peripheral and memory burst
     * transfer width */
    ctl = DMA_CHCTL(dma_periph, channelx);
    ctl &= ~(DMA_CHXCTL_PWIDTH | DMA_CHXCTL_MWIDTH | DMA_CHXCTL_PRIO | DMA_CHXCTL_TM | DMA_CHXCTL_PBURST | \
             DMA_CHXCTL_MBURST);
    ctl |= (init_struct->periph_width | (init_struct->memory_width) | init_struct->priority | init_struct->direction | \
            init_struct->memory_burst_width | init_struct->periph_burst_width);
    DMA_CHCTL(dma_periph, channelx) = ctl;

    /* configure switching buffer mode */
    if(DMA_SWITCH_BUFFER_MODE_ENABLE == init_struct->switch_buffer) {
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_SBMEN;
    } else {
        DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_SBMEN;
    }

    /* configure buffer selection */
    if(DMA_MEMORY_1 == init_struct->buffer_selection) {
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_MBS;
    } else {
        DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_MBS;
    }

    /* configure peripheral increasing mode */
    if(DMA_PERIPH_INCREASE_ENABLE == init_struct->periph_inc) {
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_PNAGA;
    } else if(DMA_PERIPH_INCREASE_DISABLE == init_struct->periph_inc) {
        DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_PNAGA;
        DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_PAIF;
    } else {
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_PNAGA;
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_PAIF;
    }

    /* configure memory increasing mode */
    if(DMA_MEMORY_INCREASE_ENABLE == init_struct->memory_inc) {
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_MNAGA;
    } else {
        DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_MNAGA;
    }

    /* configure DMA circular mode */
    if(DMA_CIRCULAR_MODE_ENABLE == init_struct->circular_mode) {
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_CMEN;
    } else {
        DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_CMEN;
    }

    if(DMA0 == dma_periph) {
        DMAMUX_RM_CHXCFG(channelx) &= ~DMAMUX_RM_CHXCFG_MUXID;
        DMAMUX_RM_CHXCFG(channelx) |= init_struct->request;
    } else {
        DMAMUX_RM_CHXCFG((uint32_t)channelx + 8U) &= ~DMAMUX_RM_CHXCFG_MUXID;
        DMAMUX_RM_CHXCFG((uint32_t)channelx + 8U) |= init_struct->request;
    }

    return error_code;
}

/*!
    \brief      configure DMA peripheral base address
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel to set peripheral base address
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  address: peripheral base address, ranges from 0x00000000 to 0xFFFFFFFF
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_periph_address_config(uint32_t dma_periph, hal_dma_channel_enum channelx, uint32_t address)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMA_CHPADDR(dma_periph, channelx) = address;

    return error_code;
}

/*!
    \brief      configure DMA memory base address
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel to set memory base address
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  memory_flag: the selected memory
                only one parameter can be selected which is shown as below:
      \arg        DMA_MEMORY_0: DMA memory 0
      \arg        DMA_MEMORY_1: DMA memory 1
    \param[in]  address: memory base address
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_memory_address_config(uint32_t dma_periph, hal_dma_channel_enum channelx, uint8_t memory_flag, uint32_t address)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
    if((DMA_MEMORY_0 != memory_flag) && (DMA_MEMORY_1 != memory_flag)) {
        HAL_DEBUGE("parameter [memory_flag] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(memory_flag) {
        DMA_CHM1ADDR(dma_periph, channelx) = address;
    } else {
        DMA_CHM0ADDR(dma_periph, channelx) = address;
    }

    return error_code;
}

/*!
    \brief      configure the number of remaining data to be transferred by the DMA
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel to set number
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  number: the number of remaining data to be transferred by the DMA, ranges from 0x00000000 to 0x0000FFFF
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_transfer_number_config(uint32_t dma_periph, hal_dma_channel_enum channelx, uint32_t number)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
    if(65535U < number) {
        HAL_DEBUGE("parameter [number] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMA_CHCNT(dma_periph, channelx) = number;

    return error_code;
}

/*!
    \brief      get the number of remaining data to be transferred by the DMA
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel to get number
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[out] none
    \retval     uint32_t: 0x00000000-0x0000FFFF
*/
uint32_t hals_dma_transfer_number_get(uint32_t dma_periph, hal_dma_channel_enum channelx)
{
    return (uint32_t)DMA_CHCNT(dma_periph, channelx);
}

/*!
    \brief      configure priority level of DMA channel
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  priority: priority level of this channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_PRIORITY_LOW: low priority
      \arg        DMA_PRIORITY_MEDIUM: medium priority
      \arg        DMA_PRIORITY_HIGH: high priority
      \arg        DMA_PRIORITY_ULTRA_HIGH: ultra high priority
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_priority_config(uint32_t dma_periph, hal_dma_channel_enum channelx, uint32_t priority)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t ctl       = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_PRIORITY_ULTRA_HIGH < priority) {
        HAL_DEBUGE("parameter [priority] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* acquire DMA_CHxCTL register */
    ctl = DMA_CHCTL(dma_periph, channelx);
    /* assign register */
    ctl &= ~DMA_CHXCTL_PRIO;
    ctl |= priority;
    DMA_CHCTL(dma_periph, channelx) = ctl;

    return error_code;
}

/*!
    \brief      configure transfer burst beats of memory
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  mbeat: memory transfer burst beats
                only one parameter can be selected which is shown as below:
      \arg        DMA_MEMORY_BURST_SINGLE: memory transfer single burst
      \arg        DMA_MEMORY_BURST_4_BEAT: memory transfer 4-beat burst
      \arg        DMA_MEMORY_BURST_8_BEAT: memory transfer 8-beat burst
      \arg        DMA_MEMORY_BURST_16_BEAT: memory transfer 16-beat burst
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_memory_burst_beats_config(uint32_t dma_periph, hal_dma_channel_enum channelx, uint32_t mbeat)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t ctl       = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_MEMORY_BURST_16_BEAT < mbeat) {
        HAL_DEBUGE("parameter [mbeat] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* acquire DMA_CHxCTL register */
    ctl = DMA_CHCTL(dma_periph, channelx);
    /* assign register */
    ctl &= ~DMA_CHXCTL_MBURST;
    ctl |= mbeat;
    DMA_CHCTL(dma_periph, channelx) = ctl;

    return error_code;
}

/*!
    \brief      configure transfer burst beats of peripheral
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  pbeat: peripheral transfer burst beats
                only one parameter can be selected which is shown as below:
      \arg        DMA_PERIPH_BURST_SINGLE: peripheral transfer single burst
      \arg        DMA_PERIPH_BURST_4_BEAT: peripheral transfer 4-beat burst
      \arg        DMA_PERIPH_BURST_8_BEAT: peripheral transfer 8-beat burst
      \arg        DMA_PERIPH_BURST_16_BEAT: peripheral transfer 16-beat burst
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_periph_burst_beats_config(uint32_t dma_periph, hal_dma_channel_enum channelx, uint32_t pbeat)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t ctl       = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_PERIPH_BURST_16_BEAT < pbeat) {
        HAL_DEBUGE("parameter [pbeat] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* acquire DMA_CHxCTL register */
    ctl = DMA_CHCTL(dma_periph, channelx);
    /* assign register */
    ctl &= ~DMA_CHXCTL_PBURST;
    ctl |= pbeat;
    DMA_CHCTL(dma_periph, channelx) = ctl;

    return error_code;
}

/*!
    \brief      configure transfer data size of memory
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  msize: transfer data size of memory
                only one parameter can be selected which is shown as below:
      \arg        DMA_MEMORY_WIDTH_8BIT: transfer data size of memory is 8-bit
      \arg        DMA_MEMORY_WIDTH_16BIT: transfer data size of memory is 16-bit
      \arg        DMA_MEMORY_WIDTH_32BIT: transfer data size of memory is 32-bit
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_memory_width_config(uint32_t dma_periph, hal_dma_channel_enum channelx, uint32_t msize)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t ctl       = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_MEMORY_WIDTH_32BIT < msize) {
        HAL_DEBUGE("parameter [msize] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* acquire DMA_CHxCTL register */
    ctl = DMA_CHCTL(dma_periph, channelx);
    /* assign register */
    ctl &= ~DMA_CHXCTL_MWIDTH;
    ctl |= msize;
    DMA_CHCTL(dma_periph, channelx) = ctl;

    return error_code;
}

/*!
    \brief      configure transfer data size of peripheral
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  psize: transfer data size of peripheral
                only one parameter can be selected which is shown as below:
      \arg        DMA_PERIPH_WIDTH_8BIT: transfer data size of peripheral is 8-bit
      \arg        DMA_PERIPH_WIDTH_16BIT: transfer data size of peripheral is 16-bit
      \arg        DMA_PERIPH_WIDTH_32BIT: transfer data size of peripheral is 32-bit
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_periph_width_config(uint32_t dma_periph, hal_dma_channel_enum channelx, uint32_t psize)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t ctl       = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_PERIPH_WIDTH_32BIT < psize) {
        HAL_DEBUGE("parameter [psize] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* acquire DMA_CHxCTL register */
    ctl = DMA_CHCTL(dma_periph, channelx);
    /* assign register */
    ctl &= ~DMA_CHXCTL_PWIDTH;
    ctl |= psize;
    DMA_CHCTL(dma_periph, channelx) = ctl;

    return error_code;
}

/*!
    \brief      configure memory address generation algorithm
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  generation_algorithm: the address generation algorithm
                only one parameter can be selected which is shown as below:
      \arg        DMA_MEMORY_INCREASE_ENABLE: next address of memory is increasing address mode
      \arg        DMA_MEMORY_INCREASE_DISABLE: next address of memory is fixed address mode
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_memory_address_generation_config(uint32_t dma_periph, hal_dma_channel_enum channelx, \
                                                  uint8_t generation_algorithm)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
    if((DMA_MEMORY_INCREASE_ENABLE != generation_algorithm) && (DMA_MEMORY_INCREASE_DISABLE != generation_algorithm)) {
        HAL_DEBUGE("parameter [generation_algorithm] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DMA_MEMORY_INCREASE_ENABLE == generation_algorithm) {
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_MNAGA;
    } else {
        DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_MNAGA;
    }

    return error_code;
}

/*!
    \brief      configure peripheral address generation algorithm
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  generation_algorithm: the address generation algorithm
                only one parameter can be selected which is shown as below:
      \arg        DMA_PERIPH_INCREASE_ENABLE: next address of peripheral is increasing address mode
      \arg        DMA_PERIPH_INCREASE_DISABLE: next address of peripheral is fixed address mode
      \arg        DMA_PERIPH_INCREASE_FIX: increasing steps of peripheral address is fixed
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_peripheral_address_generation_config(uint32_t dma_periph, hal_dma_channel_enum channelx, \
                                                      uint8_t generation_algorithm)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_PERIPH_INCREASE_FIX < generation_algorithm) {
        HAL_DEBUGE("parameter [generation_algorithm] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DMA_PERIPH_INCREASE_ENABLE == generation_algorithm) {
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_PNAGA;
    } else if(DMA_PERIPH_INCREASE_DISABLE == generation_algorithm) {
        DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_PNAGA;
    } else {
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_PNAGA;
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_PAIF;
    }

    return error_code;
}

/*!
    \brief      enable DMA circulation mode
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_circulation_enable(uint32_t dma_periph, hal_dma_channel_enum channelx)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_CMEN;

    return error_code;
}

/*!
    \brief      disable DMA circulation mode
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_circulation_disable(uint32_t dma_periph, hal_dma_channel_enum channelx)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_CMEN;

    return error_code;
}

/*!
    \brief      enable DMA channel
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_channel_enable(uint32_t dma_periph, hal_dma_channel_enum channelx)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_CHEN;

    return error_code;
}

/*!
    \brief      disable DMA channel
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_channel_disable(uint32_t dma_periph, hal_dma_channel_enum channelx)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_CHEN;

    return error_code;
}

/*!
    \brief      configure the direction of data transfer on the channel
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  direction: specify the direction of  data transfer
                only one parameter can be selected which is shown as below:
      \arg        DMA_PERIPH_TO_MEMORY: read from peripheral and write to memory
      \arg        DMA_MEMORY_TO_PERIPH: read from memory and write to peripheral
      \arg        DMA_MEMORY_TO_MEMORY: read from memory and write to memory
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_transfer_direction_config(uint32_t dma_periph, hal_dma_channel_enum channelx, uint32_t direction)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t ctl       = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_MEMORY_TO_MEMORY < direction) {
        HAL_DEBUGE("parameter [direction] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* acquire DMA_CHxCTL register */
    ctl = DMA_CHCTL(dma_periph, channelx);
    /* assign register */
    ctl &= ~DMA_CHXCTL_TM;
    ctl |= direction;
    DMA_CHCTL(dma_periph, channelx) = ctl;

    return error_code;
}

/*!
    \brief      configure DMA switch buffer mode
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  memory1_addr: memory1 base address, ranges from 0x00000000 to 0xFFFFFFFF
    \param[in]  memory_select: the selected memory
                only one parameter can be selected which is shown as below:
      \arg        DMA_MEMORY_0: DMA memory 0
      \arg        DMA_MEMORY_1: DMA memory 1
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_switch_buffer_mode_config(uint32_t dma_periph, hal_dma_channel_enum channelx, uint32_t memory1_addr, \
                                           uint32_t memory_select)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
    if((DMA_MEMORY_0 != memory_select) && (DMA_MEMORY_1 != memory_select)) {
        HAL_DEBUGE("parameter [memory_select] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure memory1 base address */
    DMA_CHM1ADDR(dma_periph, channelx) = memory1_addr;

    if(DMA_MEMORY_0 == memory_select) {
        DMA_CHCTL(dma_periph, channelx) &= ~DMA_CHXCTL_MBS;
    } else {
        DMA_CHCTL(dma_periph, channelx) |= DMA_CHXCTL_MBS;
    }

    return error_code;
}

/*!
    \brief      get DMA using memory
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[out] none
    \retval     the using memory: DMA_MEMORY_0 or DMA_MEMORY_1
*/
uint32_t hals_dma_using_memory_get(uint32_t dma_periph, hal_dma_channel_enum channelx)
{
    uint32_t the_using_memory = DMA_MEMORY_0;

    if((DMA_CHCTL(dma_periph, channelx)) & DMA_CHXCTL_MBS) {
        the_using_memory = DMA_MEMORY_1;
    } else {
        the_using_memory = DMA_MEMORY_0;
    }

    return the_using_memory;
}

/*!
    \brief      get DMA FIFO status
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[out] none
    \retval     the number of words stored in FIFO
      \arg        DMA_FIFO_CNT_NO_DATA: no data
      \arg        DMA_FIFO_CNT_1_WORD: 1 word
      \arg        DMA_FIFO_CNT_2_WORD: 2 words
      \arg        DMA_FIFO_CNT_3_WORD: 3 words
      \arg        DMA_FIFO_CNT_EMPTY: empty
      \arg        DMA_FIFO_CNT_FULL: full
*/
uint32_t hals_dma_fifo_status_get(uint32_t dma_periph, hal_dma_channel_enum channelx)
{
    return ((DMA_CHFCTL(dma_periph, channelx) & DMA_CHXFCTL_FCNT) >> CHXFCTL_FCNT_OFFSET);
}

/*!
    \brief      get DMA flag
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel to get flag
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  flag: specify get which flag
                only one parameter can be selected which is shown as below:
      \arg        DMA_FLAG_FEE: FIFO error and exception flag
      \arg        DMA_FLAG_SDE: single data mode exception flag
      \arg        DMA_FLAG_TAE: transfer access error flag
      \arg        DMA_FLAG_HTF: half transfer finish flag
      \arg        DMA_FLAG_FTF: full transfer finish flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_dma_flag_get(uint32_t dma_periph, hal_dma_channel_enum channelx, uint32_t flag)
{
    FlagStatus flag_status = RESET;

    if(channelx < DMA_CH4) {
        if(DMA_INTF0(dma_periph) & DMA_FLAG_ADD(flag, channelx)) {
            flag_status = SET;
        } else {
            /* do nothing */
        }
    } else {
        channelx -= (hal_dma_channel_enum)4;
        if(DMA_INTF1(dma_periph) & DMA_FLAG_ADD(flag, channelx)) {
            flag_status = SET;
        } else {
            /* do nothing */
        }
    }

    return flag_status;
}

/*!
    \brief      clear DMA flag
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel to clear flag
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  flag: specify get which flag
                only one parameter can be selected which is shown as below:
      \arg        DMA_FLAG_FEE: FIFO error and exception flag
      \arg        DMA_FLAG_SDE: single data mode exception flag
      \arg        DMA_FLAG_TAE: transfer access error flag
      \arg        DMA_FLAG_HTF: half transfer finish flag
      \arg        DMA_FLAG_FTF: full transfer finish flag
    \param[out] none
    \retval     none
*/
void hals_dma_flag_clear(uint32_t dma_periph, hal_dma_channel_enum channelx, uint32_t flag)
{
    if(channelx < DMA_CH4) {
        DMA_INTC0(dma_periph) |= DMA_FLAG_ADD(flag, channelx);
    } else {
        channelx -= (hal_dma_channel_enum)4;
        DMA_INTC1(dma_periph) |= DMA_FLAG_ADD(flag, channelx);
    }
}

/*!
    \brief      enable DMA interrupt
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  interrupt: specify which interrupt to enable
                one or more parameters can be selected which are shown as below:
      \arg        DMA_INT_SDE: single data mode exception interrupt
      \arg        DMA_INT_TAE: transfer access error interrupt
      \arg        DMA_INT_HTF: half transfer finish interrupt
      \arg        DMA_INT_FTF: full transfer finish interrupt
      \arg        DMA_INT_FEE: FIFO exception interrupt
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_interrupt_enable(uint32_t dma_periph, hal_dma_channel_enum channelx, uint32_t interrupt)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DMA_CHXFCTL_FEEIE != (DMA_CHXFCTL_FEEIE & interrupt)) {
        DMA_CHCTL(dma_periph, channelx) |= interrupt;
    } else {
        DMA_CHFCTL(dma_periph, channelx) |= DMA_CHXFCTL_FEEIE;
        DMA_CHCTL(dma_periph, channelx) |= (interrupt & (~DMA_CHXFCTL_FEEIE));
    }

    return error_code;
}

/*!
    \brief      disable DMA interrupt
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  interrupt: specify which interrupt to disable
                one or more parameters can be selected which are shown as below:
      \arg        DMA_INT_SDE: single data mode exception interrupt
      \arg        DMA_INT_TAE: transfer access error interrupt
      \arg        DMA_INT_HTF: half transfer finish interrupt
      \arg        DMA_INT_FTF: full transfer finish interrupt
      \arg        DMA_INT_FEE: FIFO exception interrupt
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dma_interrupt_disable(uint32_t dma_periph, hal_dma_channel_enum channelx, uint32_t interrupt)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((DMA0 != dma_periph) && (DMA1 != dma_periph)) {
        HAL_DEBUGE("parameter [dma_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMA_CH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DMA_CHXFCTL_FEEIE != (DMA_CHXFCTL_FEEIE & interrupt)) {
        DMA_CHCTL(dma_periph, channelx) &= ~interrupt;
    } else {
        DMA_CHFCTL(dma_periph, channelx) &= ~DMA_CHXFCTL_FEEIE;
        DMA_CHCTL(dma_periph, channelx) &= ~(interrupt & (~DMA_CHXFCTL_FEEIE));
    }

    return error_code;
}

/*!
    \brief      get DMA interrupt flag
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel to get interrupt flag
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  int_flag: specify get which flag
                only one parameter can be selected which is shown as below:
      \arg        DMA_INT_FLAG_FEE: FIFO error and exception flag
      \arg        DMA_INT_FLAG_SDE: single data mode exception flag
      \arg        DMA_INT_FLAG_TAE: transfer access error flag
      \arg        DMA_INT_FLAG_HTF: half transfer finish flag
      \arg        DMA_INT_FLAG_FTF: full transfer finish flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_dma_interrupt_flag_get(uint32_t dma_periph, hal_dma_channel_enum channelx, uint32_t int_flag)
{
    FlagStatus flag_status                   = RESET;
    uint32_t interrupt_enable                = 0U;
    uint32_t interrupt_flag                  = 0U;
    hal_dma_channel_enum channel_flag_offset = channelx;

    /* flags for channel0-3 */
    if(channelx < DMA_CH4) {
        switch(int_flag) {
        case DMA_INTF_FEEIF:
            interrupt_flag   = DMA_INTF0(dma_periph) & DMA_FLAG_ADD(int_flag, channelx);
            interrupt_enable = DMA_CHFCTL(dma_periph, channelx) & DMA_CHXFCTL_FEEIE;
            break;
        case DMA_INTF_SDEIF:
            interrupt_flag   = DMA_INTF0(dma_periph) & DMA_FLAG_ADD(int_flag, channelx);
            interrupt_enable = DMA_CHCTL(dma_periph, channelx) & DMA_CHXCTL_SDEIE;
            break;
        case DMA_INTF_TAEIF:
            interrupt_flag   = DMA_INTF0(dma_periph) & DMA_FLAG_ADD(int_flag, channelx);
            interrupt_enable = DMA_CHCTL(dma_periph, channelx) & DMA_CHXCTL_TAEIE;
            break;
        case DMA_INTF_HTFIF:
            interrupt_flag   = DMA_INTF0(dma_periph) & DMA_FLAG_ADD(int_flag, channelx);
            interrupt_enable = DMA_CHCTL(dma_periph, channelx) & DMA_CHXCTL_HTFIE;
            break;
        case DMA_INTF_FTFIF:
            interrupt_flag   = (DMA_INTF0(dma_periph) & DMA_FLAG_ADD(int_flag, channelx));
            interrupt_enable = (DMA_CHCTL(dma_periph, channelx) & DMA_CHXCTL_FTFIE);
            break;
        default:
            break;
        }
        /* flags for channel4-7 */
    } else {
        channel_flag_offset -= (hal_dma_channel_enum)4U;

        switch(int_flag) {
        case DMA_INTF_FEEIF:
            interrupt_flag   = DMA_INTF1(dma_periph) & DMA_FLAG_ADD(int_flag, channel_flag_offset);
            interrupt_enable = DMA_CHFCTL(dma_periph, channelx) & DMA_CHXFCTL_FEEIE;
            break;
        case DMA_INTF_SDEIF:
            interrupt_flag   = DMA_INTF1(dma_periph) & DMA_FLAG_ADD(int_flag, channel_flag_offset);
            interrupt_enable = DMA_CHCTL(dma_periph, channelx) & DMA_CHXCTL_SDEIE;
            break;
        case DMA_INTF_TAEIF:
            interrupt_flag   = DMA_INTF1(dma_periph) & DMA_FLAG_ADD(int_flag, channel_flag_offset);
            interrupt_enable = DMA_CHCTL(dma_periph, channelx) & DMA_CHXCTL_TAEIE;
            break;
        case DMA_INTF_HTFIF:
            interrupt_flag   = DMA_INTF1(dma_periph) & DMA_FLAG_ADD(int_flag, channel_flag_offset);
            interrupt_enable = DMA_CHCTL(dma_periph, channelx) & DMA_CHXCTL_HTFIE;
            break;
        case DMA_INTF_FTFIF:
            interrupt_flag   = DMA_INTF1(dma_periph) & DMA_FLAG_ADD(int_flag, channel_flag_offset);
            interrupt_enable = DMA_CHCTL(dma_periph, channelx) & DMA_CHXCTL_FTFIE;
            break;
        default:
            break;
        }
    }

    if(interrupt_flag && interrupt_enable) {
        flag_status = SET;
    } else {
        /* do nothing */
    }

    return flag_status;
}

/*!
    \brief      clear DMA interrupt flag
    \param[in]  dma_periph: DMAx(x=0,1)
    \param[in]  channelx: specify which DMA channel to clear interrupt flag
                only one parameter can be selected which is shown as below:
      \arg        DMA_CHx(x=0..7)
    \param[in]  int_flag: specify get which flag
                only one parameter can be selected which is shown as below:
      \arg        DMA_INT_FLAG_FEE: FIFO error and exception flag
      \arg        DMA_INT_FLAG_SDE: single data mode exception flag
      \arg        DMA_INT_FLAG_TAE: transfer access error flag
      \arg        DMA_INT_FLAG_HTF: half transfer finish flag
      \arg        DMA_INT_FLAG_FTF: full transfer finish flag
    \param[out] none
    \retval     none
*/
void hals_dma_interrupt_flag_clear(uint32_t dma_periph, hal_dma_channel_enum channelx, uint32_t int_flag)
{
    if(channelx < DMA_CH4) {
        DMA_INTC0(dma_periph) |= DMA_FLAG_ADD(int_flag, channelx);
    } else {
        channelx -= (hal_dma_channel_enum)4U;
        DMA_INTC1(dma_periph) |= DMA_FLAG_ADD(int_flag, channelx);
    }
}

/*!
    \brief      initialize the parameters of DMAMUX synchronization mode structure with the default values
    \param[in]  none
    \param[out] init_struct: the initialization data needed to initialize DMAMUX request multiplexer channel
   synchronization mode
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dmamux_sync_struct_para_init(hal_dmamux_sync_parameter_struct *init_struct)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(NULL == init_struct) {
        HAL_DEBUGE("pointer [init_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set the DMAMUX synchronization structure with the default values */
    init_struct->event_enable   = DMAMUX_EVENT_INPUT_DISABLE;
    init_struct->sync_enable    = DMAMUX_SYNC_DISABLE;
    init_struct->sync_id        = DMAMUX_SYNC_EVT0_OUT;
    init_struct->sync_polarity  = DMAMUX_SYNC_RISING;
    init_struct->request_number = 1U;

    return error_code;
}

/*!
    \brief      enable synchronization mode
    \param[in]  channelx: specify which DMAMUX request multiplexer channel is configured
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_MUXCHx(x=0..15)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dmamux_synchronization_enable(hal_dmamux_multiplexer_channel_enum channelx)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(DMAMUX_MUXCH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMAMUX_RM_CHXCFG(channelx) |= DMAMUX_RM_CHXCFG_SYNCEN;

    return error_code;
}

/*!
    \brief      disable synchronization mode
    \param[in]  channelx: specify which DMAMUX request multiplexer channel is configured
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_MUXCHx(x=0..15)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dmamux_synchronization_disable(hal_dmamux_multiplexer_channel_enum channelx)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(DMAMUX_MUXCH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMAMUX_RM_CHXCFG(channelx) &= (~DMAMUX_RM_CHXCFG_SYNCEN);

    return error_code;
}

/*!
    \brief      enable event generation
    \param[in]  channelx: specify which DMAMUX request multiplexer channel is configured
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_MUXCHx(x=0..15)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dmamux_event_generation_enable(hal_dmamux_multiplexer_channel_enum channelx)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(DMAMUX_MUXCH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMAMUX_RM_CHXCFG(channelx) |= DMAMUX_RM_CHXCFG_EVGEN;

    return error_code;
}

/*!
    \brief      disable event generation
    \param[in]  channelx: specify which DMAMUX request multiplexer channel is configured
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_MUXCHx(x=0..15)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dmamux_event_generation_disable(hal_dmamux_multiplexer_channel_enum channelx)
{

    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(DMAMUX_MUXCH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMAMUX_RM_CHXCFG(channelx) &= (~DMAMUX_RM_CHXCFG_EVGEN);

    return error_code;
}

/*!
    \brief      initialize the parameters of DMAMUX request generator structure with the default values
    \param[in]  none
    \param[out] init_struct: the initialization data needed to initialize DMAMUX request generator channel
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dmamux_gen_struct_para_init(hal_dmamux_gen_parameter_struct *init_struct)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(NULL == init_struct) {
        HAL_DEBUGE("pointer [init_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set the DMAMUX request generator structure with the default values */
    init_struct->trigger_id       = DMAMUX_SYNC_EVT0_OUT;
    init_struct->trigger_polarity = DMAMUX_SYNC_RISING;
    init_struct->request_number   = 1U;

    return error_code;
}

/*!
    \brief      configure synchronization input polarity
    \param[in]  channelx: specify which DMAMUX request multiplexer channel is configured
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_MUXCHx(x=0..15)
    \param[in]  polarity: synchronization input polarity
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_SYNC_NO_EVENT: no event detection
      \arg        DMAMUX_SYNC_RISING: rising edge
      \arg        DMAMUX_SYNC_FALLING: falling edge
      \arg        DMAMUX_SYNC_RISING_FALLING: rising and falling edges
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dmamux_synchronization_polarity_config(hal_dmamux_multiplexer_channel_enum channelx, uint32_t polarity)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(DMAMUX_MUXCH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMAMUX_SYNC_RISING_FALLING < polarity) {
        HAL_DEBUGE("parameter [polarity] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMAMUX_RM_CHXCFG(channelx) &= ~DMAMUX_RM_CHXCFG_SYNCP;
    DMAMUX_RM_CHXCFG(channelx) |= polarity;

    return error_code;
}

/*!
    \brief      configure number of DMA requests to forward
    \param[in]  channelx: specify which DMAMUX request multiplexer channel is configured
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_MUXCHx(x=0..15)
    \param[in]  number: DMA requests number to forward
                only one parameter can be selected which is shown as below:
      \arg        1 - 32
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dmamux_request_forward_number_config(hal_dmamux_multiplexer_channel_enum channelx, uint32_t number)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(DMAMUX_MUXCH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
    if((1U > number) || (32U < number)) {
        HAL_DEBUGE("parameter [number] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMAMUX_RM_CHXCFG(channelx) &= ~DMAMUX_RM_CHXCFG_NBR;
    DMAMUX_RM_CHXCFG(channelx) |= ((number - 1U) << RM_CHXCFG_NBR_OFFSET);

    return error_code;
}

/*!
    \brief      configure synchronization input identification
    \param[in]  channelx: specify which DMAMUX request multiplexer channel is configured
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_MUXCHx(x=0..15)
    \param[in]  id: synchronization input identification
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_SYNC_EXTI0: synchronization input is EXTI0
      \arg        DMAMUX_SYNC_EXTI1: synchronization input is EXTI1
      \arg        DMAMUX_SYNC_EXTI2: synchronization input is EXTI2
      \arg        DMAMUX_SYNC_EXTI3: synchronization input is EXTI3
      \arg        DMAMUX_SYNC_EXTI4: synchronization input is EXTI4
      \arg        DMAMUX_SYNC_EXTI5: synchronization input is EXTI5
      \arg        DMAMUX_SYNC_EXTI6: synchronization input is EXTI6
      \arg        DMAMUX_SYNC_EXTI7: synchronization input is EXTI7
      \arg        DMAMUX_SYNC_EXTI8: synchronization input is EXTI8
      \arg        DMAMUX_SYNC_EXTI9: synchronization input is EXTI9
      \arg        DMAMUX_SYNC_EXTI10: synchronization input is EXTI10
      \arg        DMAMUX_SYNC_EXTI11: synchronization input is EXTI11
      \arg        DMAMUX_SYNC_EXTI12: synchronization input is EXTI12
      \arg        DMAMUX_SYNC_EXTI13: synchronization input is EXTI13
      \arg        DMAMUX_SYNC_EXTI14: synchronization input is EXTI14
      \arg        DMAMUX_SYNC_EXTI15: synchronization input is EXTI15
      \arg        DMAMUX_SYNC_EVT0_OUT: synchronization input is Evt0_out
      \arg        DMAMUX_SYNC_EVT1_OUT: synchronization input is Evt1_out
      \arg        DMAMUX_SYNC_EVT2_OUT: synchronization input is Evt2_out
      \arg        DMAMUX_SYNC_EVT3_OUT: synchronization input is Evt3_out
      \arg        DMAMUX_SYNC_EVT4_OUT: synchronization input is Evt4_out
      \arg        DMAMUX_SYNC_EVT5_OUT: synchronization input is Evt5_out
      \arg        DMAMUX_SYNC_EVT6_OUT: synchronization input is Evt6_out
      \arg        DMAMUX_SYNC_RTC_WAKEUP: synchronization input is RTC wakeup
      \arg        DMAMUX_SYNC_CMP0_OUTPUT: synchronization input is CMP0 output
      \arg        DMAMUX_SYNC_I2C0_WAKEUP: synchronization input is I2C0 wakeup
      \arg        DMAMUX_SYNC_I2C1_WAKEUP: synchronization input is I2C1 wakeup
      \arg        DMAMUX_SYNC_I2C2_WAKEUP: synchronization input is I2C2 wakeup
      \arg        DMAMUX_SYNC_I2C3_WAKEUP: synchronization input is I2C3 wakeup
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dmamux_sync_id_config(hal_dmamux_multiplexer_channel_enum channelx, uint32_t id)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(DMAMUX_MUXCH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMAMUX_RM_CHXCFG(channelx) &= ~DMAMUX_RM_CHXCFG_SYNCID;
    DMAMUX_RM_CHXCFG(channelx) |= id;

    return error_code;
}

/*!
    \brief      configure multiplexer input identification
    \param[in]  channelx: specify which DMAMUX request multiplexer channel is configured
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_MUXCHx(x=0..15)
    \param[in]  id: input DMA request identification
                only one parameter can be selected which is shown as below:
      \arg        DMA_REQUEST_M2M: memory to memory transfer
      \arg        DMA_REQUEST_GENERATOR0: DMAMUX request generator 0
      \arg        DMA_REQUEST_GENERATOR1: DMAMUX request generator 1
      \arg        DMA_REQUEST_GENERATOR2: DMAMUX request generator 2
      \arg        DMA_REQUEST_GENERATOR3: DMAMUX request generator 3
      \arg        DMA_REQUEST_GENERATOR4: DMAMUX request generator 4
      \arg        DMA_REQUEST_GENERATOR5: DMAMUX request generator 5
      \arg        DMA_REQUEST_GENERATOR6: DMAMUX request generator 6
      \arg        DMA_REQUEST_GENERATOR7: DMAMUX request generator 7
      \arg        DMA_REQUEST_ADC0: DMAMUX ADC0 request
      \arg        DMA_REQUEST_ADC1: DMAMUX ADC1 request
      \arg        DMA_REQUEST_TIMER0_CH0: DMAMUX TIMER0 CH0 request
      \arg        DMA_REQUEST_TIMER0_CH1: DMAMUX TIMER0 CH1 request
      \arg        DMA_REQUEST_TIMER0_CH2: DMAMUX TIMER0 CH2 request
      \arg        DMA_REQUEST_TIMER0_CH3: DMAMUX TIMER0 CH3 request
      \arg        DMA_REQUEST_TIMER0_MCH0: DMAMUX TIMER0 MCH0 request
      \arg        DMA_REQUEST_TIMER0_MCH1: DMAMUX TIMER0 MCH1 request
      \arg        DMA_REQUEST_TIMER0_MCH2: DMAMUX TIMER0 MCH2 request
      \arg        DMA_REQUEST_TIMER0_MCH3: DMAMUX TIMER0 MCH3 request
      \arg        DMA_REQUEST_TIMER0_UP: DMAMUX TIMER0 UP request
      \arg        DMA_REQUEST_TIMER0_TRG: DMAMUX TIMER0 TRG request
      \arg        DMA_REQUEST_TIMER0_CMT: DMAMUX TIMER0 CMT request
      \arg        DMA_REQUEST_TIMER1_CH0: DMAMUX TIMER1 CH0 request
      \arg        DMA_REQUEST_TIMER1_CH1: DMAMUX TIMER1 CH1 request
      \arg        DMA_REQUEST_TIMER1_CH2: DMAMUX TIMER1 CH2 request
      \arg        DMA_REQUEST_TIMER1_CH3: DMAMUX TIMER1 CH3 request
      \arg        DMA_REQUEST_TIMER1_UP: DMAMUX TIMER1 UP request
      \arg        DMA_REQUEST_TIMER1_TRG: DMAMUX TIMER1 TRG request
      \arg        DMA_REQUEST_TIMER2_CH0: DMAMUX TIMER2 CH0 request
      \arg        DMA_REQUEST_TIMER2_CH1: DMAMUX TIMER2 CH1 request
      \arg        DMA_REQUEST_TIMER2_CH2: DMAMUX TIMER2 CH2 request
      \arg        DMA_REQUEST_TIMER2_CH3: DMAMUX TIMER2 CH3 request
      \arg        DMA_REQUEST_TIMER2_UP: DMAMUX TIMER2 UP request
      \arg        DMA_REQUEST_TIMER2_TRG: DMAMUX TIMER2 TRG request
      \arg        DMA_REQUEST_TIMER3_CH0: DMAMUX TIMER3 CH0 request
      \arg        DMA_REQUEST_TIMER3_CH1: DMAMUX TIMER3 CH1 request
      \arg        DMA_REQUEST_TIMER3_CH2: DMAMUX TIMER3 CH2 request
      \arg        DMA_REQUEST_TIMER3_CH3: DMAMUX TIMER3 CH3 request
      \arg        DMA_REQUEST_TIMER3_TRG: DMAMUX TIMER3 TRG request
      \arg        DMA_REQUEST_TIMER3_UP: DMAMUX TIMER3 UP request
      \arg        DMA_REQUEST_I2C0_RX: DMAMUX I2C0 RX request
      \arg        DMA_REQUEST_I2C0_TX: DMAMUX I2C0 TX request
      \arg        DMA_REQUEST_I2C1_RX: DMAMUX I2C1 RX request
      \arg        DMA_REQUEST_I2C1_TX: DMAMUX I2C1 TX request
      \arg        DMA_REQUEST_SPI0_RX: DMAMUX SPI0 RX request
      \arg        DMA_REQUEST_SPI0_TX: DMAMUX SPI0 TX request
      \arg        DMA_REQUEST_SPI1_RX: DMAMUX SPI1 RX request
      \arg        DMA_REQUEST_SPI1_TX: DMAMUX SPI1 TX request
      \arg        DMA_REQUEST_USART0_RX: DMAMUX USART0 RX request
      \arg        DMA_REQUEST_USART0_TX: DMAMUX USART0 TX request
      \arg        DMA_REQUEST_USART1_RX: DMAMUX USART1 RX request
      \arg        DMA_REQUEST_USART1_TX: DMAMUX USART1 TX request
      \arg        DMA_REQUEST_USART2_RX: DMAMUX USART2 RX request
      \arg        DMA_REQUEST_USART2_TX: DMAMUX USART2 TX request
      \arg        DMA_REQUEST_TIMER7_CH0: DMAMUX TIMER7 CH0 request
      \arg        DMA_REQUEST_TIMER7_CH1: DMAMUX TIMER7 CH1 request
      \arg        DMA_REQUEST_TIMER7_CH2: DMAMUX TIMER7 CH2 request
      \arg        DMA_REQUEST_TIMER7_CH3: DMAMUX TIMER7 CH3 request
      \arg        DMA_REQUEST_TIMER7_MCH0: DMAMUX TIMER7 MCH0 request
      \arg        DMA_REQUEST_TIMER7_MCH1: DMAMUX TIMER7 MCH1 request
      \arg        DMA_REQUEST_TIMER7_MCH2: DMAMUX TIMER7 MCH2 request
      \arg        DMA_REQUEST_TIMER7_MCH3: DMAMUX TIMER7 MCH3 request
      \arg        DMA_REQUEST_TIMER7_UP: DMAMUX TIMER7 UP request
      \arg        DMA_REQUEST_TIMER7_TRG: DMAMUX TIMER7 TRG request
      \arg        DMA_REQUEST_TIMER7_CMT: DMAMUX TIMER7 CMT request
      \arg        DMA_REQUEST_TIMER4_CH0: DMAMUX TIMER4 CH0 request
      \arg        DMA_REQUEST_TIMER4_CH1: DMAMUX TIMER4 CH1 request
      \arg        DMA_REQUEST_TIMER4_CH2: DMAMUX TIMER4 CH2 request
      \arg        DMA_REQUEST_TIMER4_CH3: DMAMUX TIMER4 CH3 request
      \arg        DMA_REQUEST_TIMER4_UP: DMAMUX TIMER4 UP request
      \arg        DMA_REQUEST_TIMER4_CMT: DMAMUX TIMER4 CMT request
      \arg        DMA_REQUEST_TIMER4_TRG: DMAMUX TIMER4 TRG request
      \arg        DMA_REQUEST_SPI2_RX: DMAMUX SPI2 RX request
      \arg        DMA_REQUEST_SPI2_TX: DMAMUX SPI2 TX request
      \arg        DMA_REQUEST_UART3_RX: DMAMUX UART3 RX request
      \arg        DMA_REQUEST_UART3_TX: DMAMUX UART3 TX request
      \arg        DMA_REQUEST_UART4_RX: DMAMUX UART4 RX request
      \arg        DMA_REQUEST_UART4_TX: DMAMUX UART4 TX request
      \arg        DMA_REQUEST_DAC_CH0: DMAMUX DAC CH0 request
      \arg        DMA_REQUEST_DAC_CH1: DMAMUX DAC CH1 request
      \arg        DMA_REQUEST_TIMER5_UP: DMAMUX TIMER5 UP request
      \arg        DMA_REQUEST_TIMER6_UP: DMAMUX TIMER6 UP request
      \arg        DMA_REQUEST_USART5_RX: DMAMUX USART5 RX request
      \arg        DMA_REQUEST_USART5_TX: DMAMUX USART5 TX request
      \arg        DMA_REQUEST_I2C2_RX: DMAMUX I2C2 RX request
      \arg        DMA_REQUEST_I2C2_TX: DMAMUX I2C2 TX request
      \arg        DMA_REQUEST_DCI: DMAMUX DCI request
      \arg        DMA_REQUEST_CAU_IN: DMAMUX CAU IN request
      \arg        DMA_REQUEST_CAU_OUT: DMAMUX CAU OUT request
      \arg        DMA_REQUEST_HAU_IN: DMAMUX HAU IN request
      \arg        DMA_REQUEST_UART6_RX: DMAMUX UART6 RX request
      \arg        DMA_REQUEST_UART6_TX: DMAMUX UART6 TX request
      \arg        DMA_REQUEST_UART7_RX: DMAMUX UART7 RX request
      \arg        DMA_REQUEST_UART7_TX: DMAMUX UART7 TX request
      \arg        DMA_REQUEST_SPI3_RX: DMAMUX SPI3 RX request
      \arg        DMA_REQUEST_SPI3_TX: DMAMUX SPI3 TX request
      \arg        DMA_REQUEST_SPI4_RX: DMAMUX SPI4 RX request
      \arg        DMA_REQUEST_SPI4_TX: DMAMUX SPI4 TX request
      \arg        DMA_REQUEST_SAI0_B0: DMAMUX SAI0 B0 request
      \arg        DMA_REQUEST_SAI0_B1: DMAMUX SAI0 B1 request
      \arg        DMA_REQUEST_RSPDIF_DATA: DMAMUX RSPDIF DATA request
      \arg        DMA_REQUEST_RSPDIF_CS: DMAMUX RSPDIF CS request
      \arg        DMA_REQUEST_HPDF_FLT0: DMAMUX HPDF FLT0 request
      \arg        DMA_REQUEST_HPDF_FLT1: DMAMUX HPDF FLT1 request
      \arg        DMA_REQUEST_HPDF_FLT2: DMAMUX HPDF FLT2 request
      \arg        DMA_REQUEST_HPDF_FLT3: DMAMUX HPDF FLT3 request
      \arg        DMA_REQUEST_TIMER14_CH0: DMAMUX TIMER14 CH0 request
      \arg        DMA_REQUEST_TIMER14_CH1: DMAMUX TIMER14 CH1 request
      \arg        DMA_REQUEST_TIMER14_MCH0: DMAMUX TIMER14 MCH0 request
      \arg        DMA_REQUEST_TIMER14_UP: DMAMUX TIMER14 UP request
      \arg        DMA_REQUEST_TIMER14_TRG: DMAMUX TIMER14 TRG request
      \arg        DMA_REQUEST_TIMER14_CMT: DMAMUX TIMER14 CMT request
      \arg        DMA_REQUEST_TIMER15_CH0: DMAMUX TIMER15 CH0 request
      \arg        DMA_REQUEST_TIMER15_MCH0: DMAMUX TIMER15 MCH0 request
      \arg        DMA_REQUEST_TIMER15_UP: DMAMUX TIMER15 UP request
      \arg        DMA_REQUEST_TIMER16_CH0: DMAMUX TIMER16 CH0 request
      \arg        DMA_REQUEST_TIMER16_MCH0: DMAMUX TIMER16 MCH0 request
      \arg        DMA_REQUEST_TIMER16_UP: DMAMUX TIMER16 TRG request
      \arg        DMA_REQUEST_ADC2: DMAMUX ADC2 request
      \arg        DMA_REQUEST_FAC_READ: DMAMUX FAC READ request
      \arg        DMA_REQUEST_FAC_WRITE: DMAMUX FAC WRITE request
      \arg        DMA_REQUEST_TMU_INPUT: DMAMUX TMU INPUT request
      \arg        DMA_REQUEST_TMU_OUTPUT: DMAMUX TMU OUTPUT request
      \arg        DMA_REQUEST_TIMER22_CH0: DMAMUX TIMER22 CH0 request
      \arg        DMA_REQUEST_TIMER22_CH1: DMAMUX TIMER22 CH1 request
      \arg        DMA_REQUEST_TIMER22_CH2: DMAMUX TIMER22 CH2 request
      \arg        DMA_REQUEST_TIMER22_CH3: DMAMUX TIMER22 CH3 request
      \arg        DMA_REQUEST_TIMER22_UP: DMAMUX TIMER22 UP request
      \arg        DMA_REQUEST_TIMER22_TRG: DMAMUX TIMER22 TRG request
      \arg        DMA_REQUEST_TIMER23_CH0: DMAMUX TIMER23 CH0 request
      \arg        DMA_REQUEST_TIMER23_CH1: DMAMUX TIMER23 CH1 request
      \arg        DMA_REQUEST_TIMER23_CH2: DMAMUX TIMER23 CH2 request
      \arg        DMA_REQUEST_TIMER23_CH3: DMAMUX TIMER23 CH3 request
      \arg        DMA_REQUEST_TIMER23_UP: DMAMUX TIMER23 UP request
      \arg        DMA_REQUEST_TIMER23_TRG: DMAMUX TIMER23 TRG request
      \arg        DMA_REQUEST_TIMER30_CH0: DMAMUX TIMER30 CH0 request
      \arg        DMA_REQUEST_TIMER30_CH1: DMAMUX TIMER30 CH1 request
      \arg        DMA_REQUEST_TIMER30_CH2: DMAMUX TIMER30 CH2 request
      \arg        DMA_REQUEST_TIMER30_CH3: DMAMUX TIMER30 CH3 request
      \arg        DMA_REQUEST_TIMER30_UP: DMAMUX TIMER30 UP request
      \arg        DMA_REQUEST_TIMER30_TRG: DMAMUX TIMER30 TRG request
      \arg        DMA_REQUEST_TIMER31_CH0: DMAMUX TIMER31 CH0 request
      \arg        DMA_REQUEST_TIMER31_CH1: DMAMUX TIMER31 CH1 request
      \arg        DMA_REQUEST_TIMER31_CH2: DMAMUX TIMER31 CH2 request
      \arg        DMA_REQUEST_TIMER31_CH3: DMAMUX TIMER31 CH3 request
      \arg        DMA_REQUEST_TIMER31_UP: DMAMUX TIMER31 UP request
      \arg        DMA_REQUEST_TIMER31_TRG: DMAMUX TIMER31 TRG request
      \arg        DMA_REQUEST_TIMER40_CH0: DMAMUX TIMER40 CH0 request
      \arg        DMA_REQUEST_TIMER40_MCH0: DMAMUX TIMER40 MCH0 request
      \arg        DMA_REQUEST_TIMER40_CMT: DMAMUX TIMER40 CMT request
      \arg        DMA_REQUEST_TIMER40_UP: DMAMUX TIMER40 UP request
      \arg        DMA_REQUEST_TIMER41_CH0: DMAMUX TIMER41 CH0 request
      \arg        DMA_REQUEST_TIMER41_MCH0: DMAMUX TIMER41 MCH0 request
      \arg        DMA_REQUEST_TIMER41_CMT: DMAMUX TIMER41 CMT request
      \arg        DMA_REQUEST_TIMER41_UP: DMAMUX TIMER41 UP request
      \arg        DMA_REQUEST_TIMER42_CH0: DMAMUX TIMER42 CH0 request
      \arg        DMA_REQUEST_TIMER42_MCH0: DMAMUX TIMER42 MCH0 request
      \arg        DMA_REQUEST_TIMER42_CMT: DMAMUX TIMER42 CMT request
      \arg        DMA_REQUEST_TIMER42_UP: DMAMUX TIMER42 UP request
      \arg        DMA_REQUEST_TIMER43_CH0: DMAMUX TIMER43 CH0 request
      \arg        DMA_REQUEST_TIMER43_MCH0: DMAMUX TIMER43 MCH0 request
      \arg        DMA_REQUEST_TIMER43_CMT: DMAMUX TIMER43 CMT request
      \arg        DMA_REQUEST_TIMER43_UP: DMAMUX TIMER43 UP request
      \arg        DMA_REQUEST_TIMER44_CH0: DMAMUX TIMER44 CH0 request
      \arg        DMA_REQUEST_TIMER44_MCH0: DMAMUX TIMER44 MCH0 request
      \arg        DMA_REQUEST_TIMER44_CMT: DMAMUX TIMER44 CMT request
      \arg        DMA_REQUEST_TIMER44_UP: DMAMUX TIMER44 UP request
      \arg        DMA_REQUEST_TIMER50_UP: DMAMUX TIMER50 UP request
      \arg        DMA_REQUEST_TIMER51_UP: DMAMUX TIMER51 UP request
      \arg        DMA_REQUEST_SAI1_B0: DMAMUX SAI1 B0 request
      \arg        DMA_REQUEST_SAI1_B1: DMAMUX SAI1 B1 request
      \arg        DMA_REQUEST_SAI2_B0: DMAMUX SAI2 B0 request
      \arg        DMA_REQUEST_SAI2_B1: DMAMUX SAI2 B1 request
      \arg        DMA_REQUEST_SPI5_RX: DMAMUX SPI5 RX request
      \arg        DMA_REQUEST_SPI5_TX: DMAMUX SPI5 TX request
      \arg        DMA_REQUEST_I2C3_RX: DMAMUX I2C3 RX request
      \arg        DMA_REQUEST_I2C3_TX: DMAMUX I2C3 TX request
      \arg        DMA_REQUEST_CAN0: DMAMUX CAN0 request
      \arg        DMA_REQUEST_CAN1: DMAMUX CAN1 request
      \arg        DMA_REQUEST_CAN2: DMAMUX CAN2 request
      \arg        DMA_REQUEST_TIMER40_CH1: DMAMUX TIMER40 CH1 request
      \arg        DMA_REQUEST_TIMER40_TRG: DMAMUX TIMER40 TRG request
      \arg        DMA_REQUEST_TIMER41_CH1: DMAMUX TIMER41 CH1 request
      \arg        DMA_REQUEST_TIMER41_TRG: DMAMUX TIMER41 TRG request
      \arg        DMA_REQUEST_TIMER42_CH1: DMAMUX TIMER42 CH1 request
      \arg        DMA_REQUEST_TIMER42_TRG: DMAMUX TIMER42 TRG request
      \arg        DMA_REQUEST_TIMER43_CH1: DMAMUX TIMER43 CH1 request
      \arg        DMA_REQUEST_TIMER43_TRG: DMAMUX TIMER43 TRG request
      \arg        DMA_REQUEST_TIMER44_CH1: DMAMUX TIMER44 CH1 request
      \arg        DMA_REQUEST_TIMER44_TRG: DMAMUX TIMER44 TRG request
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dmamux_request_id_config(hal_dmamux_multiplexer_channel_enum channelx, uint32_t id)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(DMAMUX_MUXCH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMAMUX_RM_CHXCFG(channelx) &= ~DMAMUX_RM_CHXCFG_MUXID;
    DMAMUX_RM_CHXCFG(channelx) |= id;

    return error_code;
}

/*!
    \brief      configure trigger input polarity
    \param[in]  channelx: specify which DMAMUX request generator channel is configured
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_GENCHx(x=0..7)
    \param[in]  polarity: trigger input polarity
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_GEN_NO_EVENT: no event detection
      \arg        DMAMUX_GEN_RISING: rising edge
      \arg        DMAMUX_GEN_FALLING: falling edge
      \arg        DMAMUX_GEN_RISING_FALLING: rising and falling edges
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dmamux_trigger_polarity_config(hal_dmamux_generator_channel_enum channelx, uint32_t polarity)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(DMAMUX_GENCH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
    if(DMAMUX_GEN_RISING_FALLING < polarity) {
        HAL_DEBUGE("parameter [polarity] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMAMUX_RG_CHXCFG(channelx) &= ~DMAMUX_RG_CHXCFG_RGTP;
    DMAMUX_RG_CHXCFG(channelx) |= polarity;

    return error_code;
}

/*!
    \brief      configure number of DMA requests to be generated
    \param[in]  channelx: specify which DMAMUX request generator channel is configured
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_GENCHx(x=0..7)
    \param[in]  number: DMA requests number to be generated
                only one parameter can be selected which is shown as below:
      \arg        1 - 32
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dmamux_request_generate_number_config(hal_dmamux_generator_channel_enum channelx, uint32_t number)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(DMAMUX_GENCH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
    if((1U > number) || (32U < number)) {
        HAL_DEBUGE("parameter [number] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMAMUX_RG_CHXCFG(channelx) &= ~DMAMUX_RG_CHXCFG_NBRG;
    DMAMUX_RG_CHXCFG(channelx) |= ((number - 1U) << RG_CHXCFG_NBRG_OFFSET);

    return error_code;
}

/*!
    \brief      configure trigger input identification
    \param[in]  channelx: specify which DMAMUX request generator channel is configured
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_GENCHx(x=0..7)
    \param[in]  id: trigger input identification
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_TRIGGER_EXTI0: trigger input is EXTI0
      \arg        DMAMUX_TRIGGER_EXTI1: trigger input is EXTI1
      \arg        DMAMUX_TRIGGER_EXTI2: trigger input is EXTI2
      \arg        DMAMUX_TRIGGER_EXTI3: trigger input is EXTI3
      \arg        DMAMUX_TRIGGER_EXTI4: trigger input is EXTI4
      \arg        DMAMUX_TRIGGER_EXTI5: trigger input is EXTI5
      \arg        DMAMUX_TRIGGER_EXTI6: trigger input is EXTI6
      \arg        DMAMUX_TRIGGER_EXTI7: trigger input is EXTI7
      \arg        DMAMUX_TRIGGER_EXTI8: trigger input is EXTI8
      \arg        DMAMUX_TRIGGER_EXTI9: trigger input is EXTI9
      \arg        DMAMUX_TRIGGER_EXTI10: trigger input is EXTI10
      \arg        DMAMUX_TRIGGER_EXTI11: trigger input is EXTI11
      \arg        DMAMUX_TRIGGER_EXTI12: trigger input is EXTI12
      \arg        DMAMUX_TRIGGER_EXTI13: trigger input is EXTI13
      \arg        DMAMUX_TRIGGER_EXTI14: trigger input is EXTI14
      \arg        DMAMUX_TRIGGER_EXTI15: trigger input is EXTI15
      \arg        DMAMUX_TRIGGER_EVT0_OUT: trigger input is Evt0_out
      \arg        DMAMUX_TRIGGER_EVT1_OUT: trigger input is Evt1_out
      \arg        DMAMUX_TRIGGER_EVT2_OUT: trigger input is Evt2_out
      \arg        DMAMUX_TRIGGER_EVT3_OUT: trigger input is Evt3_out
      \arg        DMAMUX_TRIGGER_EVT4_OUT: trigger input is Evt4_out
      \arg        DMAMUX_TRIGGER_EVT5_OUT: trigger input is Evt5_out
      \arg        DMAMUX_TRIGGER_EVT6_OUT: trigger input is Evt6_out
      \arg        DMAMUX_TRIGGER_RTC_WAKEUP: trigger input is wakeup
      \arg        DMAMUX_TRIGGER_CMP0_OUTPUT: trigger input is CMP0 output
      \arg        DMAMUX_TRIGGER_CMP1_OUTPUT: trigger input is CMP1 output
      \arg        DMAMUX_TRIGGER_I2C0_WAKEUP: trigger input is I2C0 wakeup
      \arg        DMAMUX_TRIGGER_I2C1_WAKEUP: trigger input is I2C1 wakeup
      \arg        DMAMUX_TRIGGER_I2C2_WAKEUP: trigger input is I2C2 wakeup
      \arg        DMAMUX_TRIGGER_I2C3_WAKEUP: trigger input is I2C3 wakeup
      \arg        DMAMUX_TRIGGER_I2C0_INT_EVENT: trigger input is I2C0 interrupt event
      \arg        DMAMUX_TRIGGER_I2C1_INT_EVENT: trigger input is I2C1 interrupt event
      \arg        DMAMUX_TRIGGER_I2C2_INT_EVENT: trigger input is I2C2 interrupt event
      \arg        DMAMUX_TRIGGER_I2C3_INT_EVENT: trigger input is I2C3 interrupt event
      \arg        DMAMUX_TRIGGER_ADC2_INT: ADC2 interrupt
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dmamux_trigger_id_config(hal_dmamux_generator_channel_enum channelx, uint32_t id)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(DMAMUX_GENCH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    DMAMUX_RG_CHXCFG(channelx) &= ~DMAMUX_RG_CHXCFG_TID;
    DMAMUX_RG_CHXCFG(channelx) |= id;

    return error_code;
}

/*!
    \brief      get DMAMUX flag
    \param[in]  flag: flag type
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_FLAG_MUXCH0_SO: DMAMUX request multiplexer channel 0 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH1_SO: DMAMUX request multiplexer channel 1 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH2_SO: DMAMUX request multiplexer channel 2 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH3_SO: DMAMUX request multiplexer channel 3 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH4_SO: DMAMUX request multiplexer channel 4 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH5_SO: DMAMUX request multiplexer channel 5 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH6_SO: DMAMUX request multiplexer channel 6 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH7_SO: DMAMUX request multiplexer channel 7 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH8_SO: DMAMUX request multiplexer channel 8 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH9_SO: DMAMUX request multiplexer channel 9 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH10_SO: DMAMUX request multiplexer channel 10 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH11_SO: DMAMUX request multiplexer channel 11 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH12_SO: DMAMUX request multiplexer channel 12 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH13_SO: DMAMUX request multiplexer channel 13 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH14_SO: DMAMUX request multiplexer channel 14 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH15_SO: DMAMUX request multiplexer channel 15 synchronization overrun flag
      \arg        DMAMUX_FLAG_GENCH0_TO: DMAMUX request generator channel 0 trigger overrun flag
      \arg        DMAMUX_FLAG_GENCH1_TO: DMAMUX request generator channel 1 trigger overrun flag
      \arg        DMAMUX_FLAG_GENCH2_TO: DMAMUX request generator channel 2 trigger overrun flag
      \arg        DMAMUX_FLAG_GENCH3_TO: DMAMUX request generator channel 3 trigger overrun flag
      \arg        DMAMUX_FLAG_GENCH4_TO: DMAMUX request generator channel 4 trigger overrun flag
      \arg        DMAMUX_FLAG_GENCH5_TO: DMAMUX request generator channel 5 trigger overrun flag
      \arg        DMAMUX_FLAG_GENCH6_TO: DMAMUX request generator channel 6 trigger overrun flag
      \arg        DMAMUX_FLAG_GENCH7_TO: DMAMUX request generator channel 7 trigger overrun flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_dmamux_flag_get(hal_dmamux_flag_enum flag)
{
    FlagStatus reval = RESET;

    if(RESET != (DMAMUX_REG_VAL(flag) & BIT(DMAMUX_BIT_POS(flag)))) {
        reval = SET;
    } else {
        /* do nothing */
    }

    return reval;
}

/*!
    \brief      clear DMAMUX flag
    \param[in]  flag: flag type
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_FLAG_MUXCH0_SO: DMAMUX request multiplexer channel 0 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH1_SO: DMAMUX request multiplexer channel 1 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH2_SO: DMAMUX request multiplexer channel 2 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH3_SO: DMAMUX request multiplexer channel 3 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH4_SO: DMAMUX request multiplexer channel 4 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH5_SO: DMAMUX request multiplexer channel 5 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH6_SO: DMAMUX request multiplexer channel 6 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH7_SO: DMAMUX request multiplexer channel 7 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH8_SO: DMAMUX request multiplexer channel 8 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH9_SO: DMAMUX request multiplexer channel 9 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH10_SO: DMAMUX request multiplexer channel 10 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH11_SO: DMAMUX request multiplexer channel 11 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH12_SO: DMAMUX request multiplexer channel 12 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH13_SO: DMAMUX request multiplexer channel 13 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH14_SO: DMAMUX request multiplexer channel 14 synchronization overrun flag
      \arg        DMAMUX_FLAG_MUXCH15_SO: DMAMUX request multiplexer channel 15 synchronization overrun flag
      \arg        DMAMUX_FLAG_GENCH0_TO: DMAMUX request generator channel 0 trigger overrun flag
      \arg        DMAMUX_FLAG_GENCH1_TO: DMAMUX request generator channel 1 trigger overrun flag
      \arg        DMAMUX_FLAG_GENCH2_TO: DMAMUX request generator channel 2 trigger overrun flag
      \arg        DMAMUX_FLAG_GENCH3_TO: DMAMUX request generator channel 3 trigger overrun flag
      \arg        DMAMUX_FLAG_GENCH4_TO: DMAMUX request generator channel 4 trigger overrun flag
      \arg        DMAMUX_FLAG_GENCH5_TO: DMAMUX request generator channel 5 trigger overrun flag
      \arg        DMAMUX_FLAG_GENCH6_TO: DMAMUX request generator channel 6 trigger overrun flag
      \arg        DMAMUX_FLAG_GENCH7_TO: DMAMUX request generator channel 7 trigger overrun flag
    \param[out] none
    \retval     none
*/
void hals_dmamux_flag_clear(hal_dmamux_flag_enum flag)
{
    DMAMUX_REG_VAL3(flag) = BIT(DMAMUX_BIT_POS(flag));
}

/*!
    \brief      enable DMAMUX interrupt
    \param[in]  interrupt: specify which interrupt to enable
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_INT_MUXCH0_SO: DMAMUX request multiplexer channel 0 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH1_SO: DMAMUX request multiplexer channel 1 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH2_SO: DMAMUX request multiplexer channel 2 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH3_SO: DMAMUX request multiplexer channel 3 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH4_SO: DMAMUX request multiplexer channel 4 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH5_SO: DMAMUX request multiplexer channel 5 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH6_SO: DMAMUX request multiplexer channel 6 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH7_SO: DMAMUX request multiplexer channel 7 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH8_SO: DMAMUX request multiplexer channel 8 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH9_SO: DMAMUX request multiplexer channel 9 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH10_SO: DMAMUX request multiplexer channel 10 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH11_SO: DMAMUX request multiplexer channel 11 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH12_SO: DMAMUX request multiplexer channel 12 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH13_SO: DMAMUX request multiplexer channel 13 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH14_SO: DMAMUX request multiplexer channel 14 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH15_SO: DMAMUX request multiplexer channel 15 synchronization overrun interrupt
      \arg        DMAMUX_INT_GENCH0_TO: DMAMUX request generator channel 0 trigger overrun interrupt
      \arg        DMAMUX_INT_GENCH1_TO: DMAMUX request generator channel 1 trigger overrun interrupt
      \arg        DMAMUX_INT_GENCH2_TO: DMAMUX request generator channel 2 trigger overrun interrupt
      \arg        DMAMUX_INT_GENCH3_TO: DMAMUX request generator channel 3 trigger overrun interrupt
      \arg        DMAMUX_INT_GENCH4_TO: DMAMUX request generator channel 4 trigger overrun interrupt
      \arg        DMAMUX_INT_GENCH5_TO: DMAMUX request generator channel 5 trigger overrun interrupt
      \arg        DMAMUX_INT_GENCH6_TO: DMAMUX request generator channel 6 trigger overrun interrupt
      \arg        DMAMUX_INT_GENCH7_TO: DMAMUX request generator channel 7 trigger overrun interrupt
    \param[out] none
    \retval     none
*/
void hals_dmamux_interrupt_enable(hal_dmamux_interrupt_enum interrupt)
{
    DMAMUX_REG_VAL(interrupt) |= BIT(DMAMUX_BIT_POS(interrupt));
}

/*!
    \brief      disable DMAMUX interrupt
    \param[in]  interrupt: specify which interrupt to disable
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_INT_MUXCH0_SO: DMAMUX request multiplexer channel 0 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH1_SO: DMAMUX request multiplexer channel 1 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH2_SO: DMAMUX request multiplexer channel 2 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH3_SO: DMAMUX request multiplexer channel 3 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH4_SO: DMAMUX request multiplexer channel 4 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH5_SO: DMAMUX request multiplexer channel 5 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH6_SO: DMAMUX request multiplexer channel 6 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH7_SO: DMAMUX request multiplexer channel 7 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH8_SO: DMAMUX request multiplexer channel 8 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH9_SO: DMAMUX request multiplexer channel 9 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH10_SO: DMAMUX request multiplexer channel 10 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH11_SO: DMAMUX request multiplexer channel 11 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH12_SO: DMAMUX request multiplexer channel 12 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH13_SO: DMAMUX request multiplexer channel 13 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH14_SO: DMAMUX request multiplexer channel 14 synchronization overrun interrupt
      \arg        DMAMUX_INT_MUXCH15_SO: DMAMUX request multiplexer channel 15 synchronization overrun interrupt
      \arg        DMAMUX_INT_GENCH0_TO: DMAMUX request generator channel 0 trigger overrun interrupt
      \arg        DMAMUX_INT_GENCH1_TO: DMAMUX request generator channel 1 trigger overrun interrupt
      \arg        DMAMUX_INT_GENCH2_TO: DMAMUX request generator channel 2 trigger overrun interrupt
      \arg        DMAMUX_INT_GENCH3_TO: DMAMUX request generator channel 3 trigger overrun interrupt
      \arg        DMAMUX_INT_GENCH4_TO: DMAMUX request generator channel 4 trigger overrun interrupt
      \arg        DMAMUX_INT_GENCH5_TO: DMAMUX request generator channel 5 trigger overrun interrupt
      \arg        DMAMUX_INT_GENCH6_TO: DMAMUX request generator channel 6 trigger overrun interrupt
      \arg        DMAMUX_INT_GENCH7_TO: DMAMUX request generator channel 7 trigger overrun interrupt
    \param[out] none
    \retval     none
*/
void hals_dmamux_interrupt_disable(hal_dmamux_interrupt_enum interrupt)
{
    DMAMUX_REG_VAL(interrupt) &= ~BIT(DMAMUX_BIT_POS(interrupt));
}

/*!
    \brief      get DMAMUX interrupt flag
    \param[in]  int_flag: flag type
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_INT_FLAG_MUXCH0_SO: DMAMUX request multiplexer channel 0 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH1_SO: DMAMUX request multiplexer channel 1 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH2_SO: DMAMUX request multiplexer channel 2 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH3_SO: DMAMUX request multiplexer channel 3 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH4_SO: DMAMUX request multiplexer channel 4 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH5_SO: DMAMUX request multiplexer channel 5 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH6_SO: DMAMUX request multiplexer channel 6 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH7_SO: DMAMUX request multiplexer channel 7 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH8_SO: DMAMUX request multiplexer channel 8 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH9_SO: DMAMUX request multiplexer channel 9 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH10_SO: DMAMUX request multiplexer channel 10 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH11_SO: DMAMUX request multiplexer channel 11 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH12_SO: DMAMUX request multiplexer channel 12 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH13_SO: DMAMUX request multiplexer channel 13 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH14_SO: DMAMUX request multiplexer channel 14 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH15_SO: DMAMUX request multiplexer channel 15 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_GENCH0_TO: DMAMUX request generator channel 0 trigger overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_GENCH1_TO: DMAMUX request generator channel 1 trigger overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_GENCH2_TO: DMAMUX request generator channel 2 trigger overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_GENCH3_TO: DMAMUX request generator channel 3 trigger overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_GENCH4_TO: DMAMUX request generator channel 4 trigger overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_GENCH5_TO: DMAMUX request generator channel 5 trigger overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_GENCH6_TO: DMAMUX request generator channel 6 trigger overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_GENCH7_TO: DMAMUX request generator channel 7 trigger overrun interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_dmamux_interrupt_flag_get(hal_dmamux_interrupt_flag_enum int_flag)
{
    FlagStatus reval    = RESET;
    uint32_t intenable  = 0U;
    uint32_t flagstatus = 0U;

    /* get the interrupt enable bit status */
    intenable = (DMAMUX_REG_VAL2(int_flag) & BIT(DMAMUX_BIT_POS2(int_flag)));
    /* get the corresponding flag bit status */
    flagstatus = (DMAMUX_REG_VAL(int_flag) & BIT(DMAMUX_BIT_POS(int_flag)));

    if(flagstatus && intenable) {
        reval = SET;
    } else {
        /* do nothing */
    }

    return reval;
}

/*!
    \brief      clear DMAMUX interrupt flag
    \param[in]  int_flag: flag type
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_INT_FLAG_MUXCH0_SO: DMAMUX request multiplexer channel 0 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH1_SO: DMAMUX request multiplexer channel 1 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH2_SO: DMAMUX request multiplexer channel 2 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH3_SO: DMAMUX request multiplexer channel 3 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH4_SO: DMAMUX request multiplexer channel 4 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH5_SO: DMAMUX request multiplexer channel 5 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH6_SO: DMAMUX request multiplexer channel 6 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH7_SO: DMAMUX request multiplexer channel 7 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH8_SO: DMAMUX request multiplexer channel 8 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH9_SO: DMAMUX request multiplexer channel 9 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH10_SO: DMAMUX request multiplexer channel 10 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH11_SO: DMAMUX request multiplexer channel 11 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH12_SO: DMAMUX request multiplexer channel 12 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH13_SO: DMAMUX request multiplexer channel 13 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH14_SO: DMAMUX request multiplexer channel 14 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_MUXCH15_SO: DMAMUX request multiplexer channel 15 synchronization overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_GENCH0_TO: DMAMUX request generator channel 0 trigger overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_GENCH1_TO: DMAMUX request generator channel 1 trigger overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_GENCH2_TO: DMAMUX request generator channel 2 trigger overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_GENCH3_TO: DMAMUX request generator channel 3 trigger overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_GENCH4_TO: DMAMUX request generator channel 4 trigger overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_GENCH5_TO: DMAMUX request generator channel 5 trigger overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_GENCH6_TO: DMAMUX request generator channel 6 trigger overrun interrupt flag
      \arg        DMAMUX_INT_FLAG_GENCH7_TO: DMAMUX request generator channel 7 trigger overrun interrupt flag
    \param[out] none
    \retval     none
*/
void hals_dmamux_interrupt_flag_clear(hal_dmamux_interrupt_flag_enum int_flag)
{
    DMAMUX_REG_VAL3(int_flag) = BIT(DMAMUX_BIT_POS(int_flag));
}

/*!
    \brief      configure dmamux request gennerator channel
    \param[in]  dma_dev: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channelx: specify which DMAMUX request generator channel is configured
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_GENCHx(x=0..7)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
static int32_t _dmamux_request_gen_config(hal_dma_dev_struct *dma_dev, hal_dmamux_generator_channel_enum channelx, \
                                          hal_dmamux_gen_parameter_struct *init_struct)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t cfg       = 0U;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if((NULL == init_struct) || (NULL == dma_dev)) {
        HAL_DEBUGE("pointer [init_struct] or [dma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(DMAMUX_GENCH7 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable DMAMUX request generator channel for DMA request generation number configuration */
    DMAMUX_RG_CHXCFG(channelx) &= ~(DMAMUX_RG_CHXCFG_RGEN);

    /* configure trigger input identification, trigger polarity, number of DMA requests to be generated */
    cfg = DMAMUX_RG_CHXCFG(channelx);
    cfg &= ~(DMAMUX_RG_CHXCFG_TID | DMAMUX_RG_CHXCFG_NBRG | DMAMUX_RG_CHXCFG_RGTP);
    cfg |= (RG_CHXCFG_NBRG(init_struct->request_number - 1U) | init_struct->trigger_id | init_struct->trigger_polarity);
    DMAMUX_RG_CHXCFG(channelx) = cfg;
    dma_dev->gen_channel       = channelx;
    dma_dev->gen_enable        = 1U;

    return error_code;
}

/*!
    \brief      configure dmamux request gennerator channel
    \param[in]  channelx: specify which DMAMUX request multiplexer channel is configured
                only one parameter can be selected which is shown as below:
      \arg        DMAMUX_MUXCHx(x=0..15)
    \param[in]  init_struct: pointer to a hal_dmamux_sync_parameter_struct structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
static int32_t _dmamux_sync_config(hal_dmamux_multiplexer_channel_enum channelx, \
                                   hal_dmamux_sync_parameter_struct *init_struct)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t cfg       = 0U;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check DMA pointer address */
    if(NULL == init_struct) {
        HAL_DEBUGE("pointer [init_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(DMAMUX_MUXCH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable synchronization mode and event generation for DMA request forward number configuration */
    DMAMUX_RM_CHXCFG(channelx) &= ~(DMAMUX_RM_CHXCFG_SYNCEN | DMAMUX_RM_CHXCFG_EVGEN);

    /* configure synchronization input identification, synchronization input polarity, number of DMA requests to forward */
    cfg = DMAMUX_RM_CHXCFG(channelx);
    cfg &= ~(DMAMUX_RM_CHXCFG_SYNCID | DMAMUX_RM_CHXCFG_NBR | DMAMUX_RM_CHXCFG_SYNCP);
    cfg |= (init_struct->sync_polarity | (init_struct->sync_id) | RM_CHXCFG_NBR(init_struct->request_number - 1U));

    DMAMUX_RM_CHXCFG(channelx) = cfg;
    if(DMAMUX_SYNC_ENABLE == init_struct->sync_enable) {
        DMAMUX_RM_CHXCFG(channelx) |= DMAMUX_RM_CHXCFG_SYNCEN;
    } else {
        /* do nothing */
    }

    if(DMAMUX_EVENT_INPUT_ENABLE == init_struct->event_enable) {
        DMAMUX_RM_CHXCFG(channelx) |= DMAMUX_RM_CHXCFG_EVGEN;
    } else {
        /* do nothing */
    }

    return error_code;
}
