/*!
    \file    gd32h7xx_hal_cmp.c
    \brief   CMP driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* delay CMP startup time */
#define  CMP_STATRUP_WAIT_TIME          (80U)

/* CMP Compare output handler function */
static void _cmp_compare_output_handle(void *cmp_dev);

/*!
    \brief      initialize CMP
    \param[in]  cmp_dev: CMP device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  periph:CMP0/CMP1
    \param[in]  p_init: the pointer to CMP init structure
                  window_mode:
                  only one parameter can be selected which is shown as below:
      \arg          CMP_WINDOWMODE_DISABLE: CMP1_IP is connected to CMP1 non-inverting input
      \arg          CMP_WINDOWMODE_ENABLE: CMP1_IP is connected to CMP0_IP
                  noninverting_input:
                  only one parameter can be selected which is shown as below:
      \arg          CMP_INPUT_PSEL_INPUT1:noinverting input IO1
      \arg          CMP_INPUT_PSEL_INPUT2:noinverting input IO2
                  inverting_input:
                  only one parameter can be selected which is shown as below:
      \arg          CMP_INPUT_MSEL_1_4VREFINT: inverting input 1/4 Vrefint
      \arg          CMP_INPUT_MSEL_1_2VREFINT: inverting input 1/2 Vrefint
      \arg          CMP_INPUT_MSEL_3_4VREFINT: inverting input 3/4 Vrefint
      \arg          CMP_INPUT_MSEL_VREFINT: inverting input Vrefint
      \arg          CMP_INPUT_MSEL_DAC0_OUT0: inverting input DAC0_OUT0
      \arg          CMP_INPUT_MSEL_DAC0_OUT1: inverting input DAC0_OUT1
      \arg          CMP_INPUT_MSEL_INPUT1: inverting input IO1
      \arg          CMP_INPUT_MSEL_INPUT2: inverting input IO2
                  mode:
                  only one parameter can be selected which is shown as below:
      \arg          CMP_MODE_HIGHSPEED: high speed and power mode
      \arg          CMP_MODE_MIDDLESPEED: middle speed and power mode
      \arg          CMP_MODE_VERYLOWSPEED: very low speed and power mode
                  polarity:
                  only one parameter can be selected which is shown as below:
      \arg          CMP_OUTPUT_POLARITY_NOINVERTED: CMP output not inverted
      \arg          CMP_OUTPUT_POLARITY_INVERTED: CMP output inverted
                  hysteresis:
                  only one parameter can be selected which is shown as below:
      \arg          CMP_HYSTERESIS_NO: input no hysteresis
      \arg          CMP_HYSTERESIS_LOW: input hysteresis level low
      \arg          CMP_HYSTERESIS_MIDDLE: input hysteresis level middle
      \arg          CMP_HYSTERESIS_HIGH: input hysteresis level high
                  blanking_source:
                  only one parameter can be selected which is shown as below:
      \arg          CMP_BLANKING_NONE:  no blanking source
      \arg          CMP_BLANKING_TIMER0_OC0: TIMER 0 output channel0
      \arg          CMP_BLANKING_TIMER1_OC2: TIMER 1 output channel2
      \arg          CMP_BLANKING_TIMER2_OC2: TIMER 2 output channel2
      \arg          CMP_BLANKING_TIMER2_OC3: TIMER 2 output channel3
      \arg          CMP_BLANKING_TIMER7_OC0: TIMER 7 output channel0
      \arg          CMP_BLANKING_TIMER14_OC0: TIMER 14 output channel0
                  mux_outx(x=0-10):
                  only one parameter can be selected which is shown as below:
      \arg          CMP_MUX_OUTx_NO(x=0-10): no alternate function select
      \arg          CMP_MUX_OUT0_PA6: PA6 alternate function select
      \arg          CMP_MUX_OUT1_PA8: PA8 alternate function select
      \arg          CMP_MUX_OUT2_PB12: PB12 alternate function select
      \arg          CMP_MUX_OUT3_PE6: PE6 alternate function select
      \arg          CMP_MUX_OUT4_PE15: PE15 alternate function select
      \arg          CMP_MUX_OUT5_PG2: PG2 alternate function select
      \arg          CMP_MUX_OUT6_PG3: PG3 alternate function select
      \arg          CMP_MUX_OUT7_PG4: PG4 alternate function select
      \arg          CMP_MUX_OUT8_PK0: PK0 alternate function select
      \arg          CMP_MUX_OUT9_PK1: PK1 alternate function select
      \arg          CMP_MUX_OUT10_PK2: PK2 alternate function select
                  exti_type:
      \arg          CMP_EXTI_NONE: no exti interrupt or event trigger
      \arg          CMP_EXTI_INT_RISING: exti interrupt with rising edge
      \arg          CMP_EXTI_INT_FALLING: exti interrupt with falling edge
      \arg          CMP_EXTI_INT_BOTH: exti interrupt with both rising and falling edge
      \arg          CMP_EXTI_EVENT_RISING: exti event with rising edge
      \arg          CMP_EXTI_EVENT_FALLING: exti event with falling edge
      \arg          CMP_EXTI_EVENT_BOTH: exti event with both rising and falling edge
    \param[out] none
    \retval     error code:HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_LOCK details refer to gd32h7xx_hal.h
*/
int32_t hal_cmp_init(hal_cmp_dev_struct *cmp_dev, uint32_t periph, hal_cmp_init_struct *p_init)
{
    /* Function return status */
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* Validate input parameters */
    if((NULL == cmp_dev) || (NULL == p_init)) {
        HAL_DEBUGE("pointer [cmp_dev] or [p_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* Validate peripheral selection */
    if((CMP0 != periph) && (CMP1 != periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Acquire peripheral lock */
    HAL_LOCK(cmp_dev);

    /* CMP peripheral initialization sequence */
    /* Check if CMP configuration is locked */
    if(CMP_CS(periph) & CMP_CSLOCK_ENABLE) {
        HAL_DEBUGE("CMP is locked, can not be initialized");

        /* Update CMP peripheral state to lock */
        cmp_dev->state = HAL_CMP_STATE_LOCK;
        ret            = HAL_ERR_LOCK;
    } else {
        /* Store peripheral and configuration parameters */
        cmp_dev->periph = periph;
        cmp_dev->init   = *p_init;

        /* Reset CMP control register */
        CMP_CS(periph) = 0x00000000U;

        /* Configure CMP basic operating parameters */
        CMP_CS(periph) = (p_init->window_mode | p_init->noninverting_input | p_init->inverting_input | \
                          p_init->mode | p_init->hysteresis  | p_init->polarity | p_init->blanking_source);

        /* Configure CMP output mux and EXTI line based on comparator instance */
        if(CMP0 == cmp_dev->periph) {
            /* Configure CMP0 output mux selection bits */
            CMP_SR &= ~(p_init->mux_out0 | p_init->mux_out1 | p_init->mux_out2 | p_init->mux_out3 | \
                        p_init->mux_out4 | p_init->mux_out5 | p_init->mux_out6 | p_init->mux_out7 | \
                        p_init->mux_out8 | p_init->mux_out9 | p_init->mux_out10);

            if(CMP_EXTI_NONE != p_init->exti_type) {
                /* Initialize EXTI line 20 for CMP0 output with specified trigger type */
                if(HAL_ERR_NONE != hal_exti_internal_init((hal_exti_internal_line_enum)EXTI_LINE_20_CMP0_OUTPUT, \
                                                         (hal_exti_type_enum)p_init->exti_type)) {
                    HAL_DEBUGE("CMP0 EXTI line 20 initialization failed");

                    /* Update CMP peripheral state to error */
                    cmp_dev->state = HAL_CMP_STATE_ERROR;
                    ret = HAL_ERR_VAL;
                } else {
                    /* Clear EXTI interrupt flag and disable interrupt until explicitly enabled */
                    hals_exti_interrupt_flag_clear(EXTI_20);
                    EXTI_INTEN0 &= ~EXTI_INTEN0_INTEN20;
                }
            } else {
                /* do nothing */
            }
        } else {
            /* Configure CMP1 output mux selection bits */
            CMP_SR |= (p_init->mux_out0 | p_init->mux_out1 | p_init->mux_out2 | p_init->mux_out3 | \
                       p_init->mux_out4 | p_init->mux_out5 | p_init->mux_out6 | p_init->mux_out7 | \
                       p_init->mux_out8 | p_init->mux_out9 | p_init->mux_out10);
            if(CMP_EXTI_NONE != p_init->exti_type) {
                /* Initialize EXTI line 21 for CMP1 output with specified trigger type */
                if(HAL_ERR_NONE != hal_exti_internal_init((hal_exti_internal_line_enum)EXTI_LINE_21_CMP1_OUTPUT, \
                                       (hal_exti_type_enum)p_init->exti_type)) {
                    HAL_DEBUGE("CMP1 EXTI line 21 initialization failed");

                    /* Update CMP peripheral state to error */
                    cmp_dev->state = HAL_CMP_STATE_ERROR;
                    ret = HAL_ERR_VAL;
                } else {
                    /* Clear EXTI interrupt flag and disable interrupt until explicitly enabled */
                    hals_exti_interrupt_flag_clear(EXTI_21);
                    EXTI_INTEN0 &= ~EXTI_INTEN0_INTEN21;
                }
            } else {
                /* do nothing */
            }
        }
    }

    if(HAL_ERR_NONE == ret) {
        /* Update CMP peripheral state to ready */
        cmp_dev->state = HAL_CMP_STATE_READY;
    } else {
        /* do nothing */
    }

    /* Acquire peripheral unlock */
    HAL_UNLOCK(cmp_dev);

    return ret;
}

/*!
    \brief      deinitialize CMP
    \param[in]  cmp_dev: CMP device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code:HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_cmp_deinit(hal_cmp_dev_struct *cmp_dev)
{
    /* Variable declarations */
    uint32_t periph = 0U;
    int32_t ret     = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* Parameter validation check */
    if(NULL == cmp_dev) {
        HAL_DEBUGE("pointer [cmp_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Acquire peripheral lock */
    HAL_LOCK(cmp_dev);

    /* Get peripheral identifier from device structure */
    periph = cmp_dev->periph;

    /* Validate peripheral selection */
    if((CMP0 == periph) || (CMP1 == periph)) {
        /* deinitialize the periph and the device information structure */
        hal_rcu_periph_reset_enable(RCU_CMPRST);
        hal_rcu_periph_reset_disable(RCU_CMPRST);
        cmp_dev->state = HAL_CMP_STATE_RESET;
    } else {
        HAL_DEBUGE("parameter [cmp_dev->periph] value is invalid");
        /* Update CMP peripheral state to error */
        cmp_dev->state = HAL_CMP_STATE_ERROR;
        ret = HAL_ERR_VAL;
    }

    /* Acquire peripheral unlock */
    HAL_UNLOCK(cmp_dev);

    return ret;
}

/*!
    \brief      initialize the CMP structure with the default values
    \param[in]  hal_struct_type: the type of the structure
                only one parameters can be selected which are shown as below:
      \arg        HAL_CMP_INIT_STRUCT: initialization structure
      \arg        HAL_CMP_DEV_STRUCT: device information structure
      \arg        HAL_CMP_IRQ_INIT_STRUCT: interrupt callback initialization structure
      \arg        HAL_CMP_IRQ_USER_CALLBACK_STRUCT: user callback function structure
    \param[out] p_struct: pointer to CMP structure that contains the configuration information
    \retval     error code:HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_cmp_struct_init(hal_cmp_struct_type_enum hal_struct_type, void *p_struct)
{
    /* Variable declarations */
    int ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((HAL_CMP_INIT_STRUCT != hal_struct_type) && (HAL_CMP_DEV_STRUCT != hal_struct_type) && \
       (HAL_CMP_IRQ_INIT_STRUCT != hal_struct_type)) {
        HAL_DEBUGE("parameter [hal_struct_type] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_CMP_INIT_STRUCT:
        /* initialize CMP initialization structure with the default values */
        ((hal_cmp_init_struct *)p_struct)->window_mode        = CMP_WINDOWMODE_DISABLE;
        ((hal_cmp_init_struct *)p_struct)->noninverting_input = CMP_INPUT_PSEL_INPUT1;
        ((hal_cmp_init_struct *)p_struct)->inverting_input    = CMP_INPUT_MSEL_INPUT1;
        ((hal_cmp_init_struct *)p_struct)->mode               = CMP_MODE_HIGHSPEED;
        ((hal_cmp_init_struct *)p_struct)->polarity           = CMP_OUTPUT_POLARITY_NOINVERTED;
        ((hal_cmp_init_struct *)p_struct)->hysteresis         = CMP_HYSTERESIS_NO;
        ((hal_cmp_init_struct *)p_struct)->blanking_source    = CMP_BLANKING_NONE;
        ((hal_cmp_init_struct *)p_struct)->mux_out0           = CMP_MUX_OUT0_NO;
        ((hal_cmp_init_struct *)p_struct)->mux_out1           = CMP_MUX_OUT1_NO;
        ((hal_cmp_init_struct *)p_struct)->mux_out2           = CMP_MUX_OUT2_NO;
        ((hal_cmp_init_struct *)p_struct)->mux_out3           = CMP_MUX_OUT3_NO;
        ((hal_cmp_init_struct *)p_struct)->mux_out4           = CMP_MUX_OUT4_NO;
        ((hal_cmp_init_struct *)p_struct)->mux_out5           = CMP_MUX_OUT5_NO;
        ((hal_cmp_init_struct *)p_struct)->mux_out6           = CMP_MUX_OUT6_NO;
        ((hal_cmp_init_struct *)p_struct)->mux_out7           = CMP_MUX_OUT7_NO;
        ((hal_cmp_init_struct *)p_struct)->mux_out8           = CMP_MUX_OUT8_NO;
        ((hal_cmp_init_struct *)p_struct)->mux_out9           = CMP_MUX_OUT9_NO;
        ((hal_cmp_init_struct *)p_struct)->mux_out10          = CMP_MUX_OUT10_NO;
        ((hal_cmp_init_struct *)p_struct)->exti_type          = CMP_EXTI_NONE;
        break;
    case HAL_CMP_DEV_STRUCT:
        /* initialize CMP device information structure with the default values */
        ((hal_cmp_dev_struct *)p_struct)->periph                        = 0U;
        ((hal_cmp_dev_struct *)p_struct)->init.window_mode              = CMP_WINDOWMODE_DISABLE;
        ((hal_cmp_dev_struct *)p_struct)->init.noninverting_input       = CMP_INPUT_PSEL_INPUT1;
        ((hal_cmp_dev_struct *)p_struct)->init.inverting_input          = CMP_INPUT_MSEL_INPUT1;
        ((hal_cmp_dev_struct *)p_struct)->init.mode                     = CMP_MODE_HIGHSPEED;
        ((hal_cmp_dev_struct *)p_struct)->init.polarity                 = CMP_OUTPUT_POLARITY_NOINVERTED;
        ((hal_cmp_dev_struct *)p_struct)->init.hysteresis               = CMP_HYSTERESIS_NO;
        ((hal_cmp_dev_struct *)p_struct)->init.blanking_source          = CMP_BLANKING_NONE;
        ((hal_cmp_dev_struct *)p_struct)->init.mux_out0                 = CMP_MUX_OUT0_NO;
        ((hal_cmp_dev_struct *)p_struct)->init.mux_out1                 = CMP_MUX_OUT1_NO;
        ((hal_cmp_dev_struct *)p_struct)->init.mux_out2                 = CMP_MUX_OUT2_NO;
        ((hal_cmp_dev_struct *)p_struct)->init.mux_out3                 = CMP_MUX_OUT3_NO;
        ((hal_cmp_dev_struct *)p_struct)->init.mux_out4                 = CMP_MUX_OUT4_NO;
        ((hal_cmp_dev_struct *)p_struct)->init.mux_out5                 = CMP_MUX_OUT5_NO;
        ((hal_cmp_dev_struct *)p_struct)->init.mux_out6                 = CMP_MUX_OUT6_NO;
        ((hal_cmp_dev_struct *)p_struct)->init.mux_out7                 = CMP_MUX_OUT7_NO;
        ((hal_cmp_dev_struct *)p_struct)->init.mux_out8                 = CMP_MUX_OUT8_NO;
        ((hal_cmp_dev_struct *)p_struct)->init.mux_out9                 = CMP_MUX_OUT9_NO;
        ((hal_cmp_dev_struct *)p_struct)->init.mux_out10                = CMP_MUX_OUT10_NO;
        ((hal_cmp_dev_struct *)p_struct)->init.exti_type                = CMP_EXTI_NONE;
        ((hal_cmp_dev_struct *)p_struct)->cmp_irq.compare_output_handle = NULL;
        ((hal_cmp_dev_struct *)p_struct)->state                         = HAL_CMP_STATE_RESET;
        ((hal_cmp_dev_struct *)p_struct)->error_state                   = 0U;
        ((hal_cmp_dev_struct *)p_struct)->mutex                         = HAL_MUTEX_UNLOCKED;
        ((hal_cmp_dev_struct *)p_struct)->compare_output_callback       = NULL;
        break;
    case HAL_CMP_IRQ_INIT_STRUCT:
        /* initialize interrupt callback structure with the default values */
        ((hal_cmp_irq_struct *)p_struct)->compare_output_handle = NULL;
        break;
    case HAL_CMP_IRQ_USER_CALLBACK_STRUCT:
        /* initialize user callback function structure with the default values */
        ((hal_cmp_irq_user_callback_struct *)p_struct)->compare_output_func = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      start CMP module function
    \param[in]  cmp_dev: CMP device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code:HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_LOCK details refer to gd32h7xx_hal.h
*/
int32_t hal_cmp_start(hal_cmp_dev_struct *cmp_dev)
{
    /* Variable declarations */
    __IO uint32_t wait_startup_timer = 0U;
    __IO uint32_t system_core_clock = 0U;
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == cmp_dev) {
        HAL_DEBUGE("pointer [cmp_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Acquire peripheral lock */
    HAL_LOCK(cmp_dev);

    /* check if CMP locked */
    if(CMP_CS(cmp_dev->periph) & CMP_CSLOCK_ENABLE) {
        HAL_DEBUGE("CMP is locked");
        /* Update CMP peripheral state to lock */
        cmp_dev->state = HAL_CMP_STATE_LOCK;
        ret = HAL_ERR_LOCK;
    } else {
        if(HAL_CMP_STATE_READY == cmp_dev->state) {
            /* enable the selected CMP */
            CMP_CS(cmp_dev->periph) |= (uint32_t)CMP_CS_EN;

            /* change CMP state */
            cmp_dev->state = HAL_CMP_STATE_BUSY;

            /* update the SystemCoreClock global variable */
            system_core_clock = hal_rcu_clock_freq_get(CK_SYS);

            /* delay cmp startup time */
            wait_startup_timer = ((CMP_STATRUP_WAIT_TIME / 10UL) * ((system_core_clock / (100000UL * 2UL)) + 1UL));
            while(0U != wait_startup_timer) {
                wait_startup_timer--;
            }
        } else {
            /* do nothing */
        }
    }

    /* Acquire peripheral unlock */
    HAL_UNLOCK(cmp_dev);

    return ret;
}

/*!
    \brief      stop CMP module function
    \param[in]  cmp_dev: CMP device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code:HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_LOCK details refer to gd32h7xx_hal.h
*/
int32_t hal_cmp_stop(hal_cmp_dev_struct *cmp_dev)
{
    /* Variable declarations */
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == cmp_dev) {
        HAL_DEBUGE("pointer [cmp_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Acquire peripheral lock */
    HAL_LOCK(cmp_dev);

    /* check if CMP locked */
    if(CMP_CS(cmp_dev->periph) & CMP_CSLOCK_ENABLE) {
        HAL_DEBUGE("CMP is locked");
        /* Update CMP peripheral state to lock */
        cmp_dev->state = HAL_CMP_STATE_LOCK;
        ret = HAL_ERR_LOCK;
    } else {
        if(HAL_CMP_STATE_RESET != cmp_dev->state) {
            /* stop CMP module */
            CMP_CS(cmp_dev->periph) &= ~CMP_CS_EN;

            /* change CMP state */
            cmp_dev->state = HAL_CMP_STATE_READY;
        } else {
            /* do nothing */
        }
    }

    /* Acquire peripheral unlock */
    HAL_UNLOCK(cmp_dev);

    return ret;
}

/*!
    \brief      start CMP module function in interrupt mode
    \param[in]  cmp_dev: CMP device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code:HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_LOCK details refer to gd32h7xx_hal.h
*/
int32_t hal_cmp_start_interrupt(hal_cmp_dev_struct *cmp_dev, hal_cmp_irq_user_callback_struct *p_user_func)
{
    /* Variable declarations */
    __IO uint32_t wait_startup_timer = 0U;
    __IO uint32_t system_core_clock = 0U;
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == cmp_dev) {
        HAL_DEBUGE("pointer [cmp_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Acquire peripheral lock */
    HAL_LOCK(cmp_dev);

    /* Check if CMP peripheral configuration is locked */
    if(CMP_CS(cmp_dev->periph) & CMP_CSLOCK_ENABLE) {
        HAL_DEBUGE("CMP is locked");
        /* Update CMP peripheral state to lock */
        cmp_dev->state = HAL_CMP_STATE_LOCK;
        ret = HAL_ERR_LOCK;
    } else {
        if(HAL_CMP_STATE_READY == cmp_dev->state) {
            /* Enable CMP module */
            CMP_CS(cmp_dev->periph) |= (uint32_t)CMP_CS_EN;

            if(CMP_EXTI_NONE != cmp_dev->init.exti_type) {
                /* Enable EXTI interrupt for CMP output */
                if(CMP0 == cmp_dev->periph) {
                    /* Enable EXTI line 20 interrupt for CMP0 */
                    EXTI_INTEN0 |= EXTI_INTEN0_INTEN20;
                } else {
                    /* Enable EXTI line 21 interrupt for CMP1 */
                    EXTI_INTEN0 |= EXTI_INTEN0_INTEN21;
                }
            } else {
                /* Enable CMP internal interrupt */
                CMP_CS(cmp_dev->periph) |= CMP_INTERRUPT_ENABLE;
            }

            /* Set default interrupt handler and clear user callback */
            cmp_dev->cmp_irq.compare_output_handle = &_cmp_compare_output_handle;
            cmp_dev->compare_output_callback       = NULL;

            /* Register user-defined callback function if provided */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->compare_output_func) {
                    cmp_dev->compare_output_callback = (void *)p_user_func->compare_output_func;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            /* Update system core clock frequency */
            system_core_clock = hal_rcu_clock_freq_get(CK_SYS);

            /* Calculate and execute CMP startup delay */
            wait_startup_timer = ((CMP_STATRUP_WAIT_TIME / 10UL) * ((system_core_clock / (100000UL * 2UL)) + 1UL));
            while(0U != wait_startup_timer) {
                wait_startup_timer--;
            }

            /* Update CMP peripheral state */
            cmp_dev->state = HAL_CMP_STATE_BUSY;
        } else {
            /* do nothing */
        }
    }

    /* Acquire peripheral unlock */
    HAL_UNLOCK(cmp_dev);

    return ret;
}

/*!
    \brief      stop CMP module function in interrupt mode
    \param[in]  cmp_dev: CMP device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code:HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_LOCK details refer to gd32h7xx_hal.h
*/
int32_t hal_cmp_stop_interrupt(hal_cmp_dev_struct *cmp_dev)
{
    /* Variable declarations */
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == cmp_dev) {
        HAL_DEBUGE("pointer [cmp_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Acquire peripheral lock */
    HAL_LOCK(cmp_dev);

    /* Check if CMP peripheral configuration is locked */
    if(CMP_CS(cmp_dev->periph) & CMP_CSLOCK_ENABLE) {
        HAL_DEBUGE("CMP is locked");
        /* Update CMP peripheral state to lock */
        cmp_dev->state = HAL_CMP_STATE_LOCK;
        ret = HAL_ERR_LOCK;
    } else {
        if(HAL_CMP_STATE_RESET != cmp_dev->state) {
            /* Clear interrupt handler pointer */
            cmp_dev->cmp_irq.compare_output_handle = NULL;

            /* Disable CMP module */
            CMP_CS(cmp_dev->periph) &= ~CMP_CS_EN;

            /* Disable CMP internal interrupt */
            CMP_CS(cmp_dev->periph) &= ~CMP_INTERRUPT_ENABLE;

            /* Disable EXTI interrupt for CMP output */
            if(CMP0 == cmp_dev->periph) {
                /* Disable EXTI line 20 interrupt for CMP0 */
                EXTI_INTEN0 &= ~EXTI_INTEN0_INTEN20;
            } else {
                /* Disable EXTI line 21 interrupt for CMP1 */
                EXTI_INTEN0 &= ~EXTI_INTEN0_INTEN21;
            }

            /* Update CMP peripheral state to ready */
            cmp_dev->state = HAL_CMP_STATE_READY;
        } else {
            /* do nothing */
        }
    }

    /* Release peripheral lock */
    HAL_UNLOCK(cmp_dev);

    return ret;
}

/*!
    \brief      CMP interrupt handler content function, which is merely used in CMP_IRQHandler
    \param[in]  cmp_dev: CMP device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code:HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_cmp_irq(hal_cmp_dev_struct *cmp_dev)
{
    /* EXTI line variable for CMP interrupt handling */
    __IO hal_exti_line_enum linex;

#if (1U == HAL_PARAMETER_CHECK)
    /* Validate CMP device pointer */
    if(NULL == cmp_dev) {
        HAL_DEBUGE("pointer [cmp_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Determine EXTI line based on CMP peripheral instance */
    if(CMP0 == cmp_dev->periph) {
        linex = EXTI_20;
    } else {
        linex = EXTI_21;
    }

    /* Handle EXTI interrupt if triggered */
    if(RESET != hals_exti_interrupt_flag_get(linex)) {
        /* Check if CMP is operating in window mode */
        if(RESET!= (CMP_CS(CMP1) & CMP_CS_WNDEN)) {
            /* Clear both EXTI interrupt flags for window mode */
            hals_exti_interrupt_flag_clear(EXTI_20);
            hals_exti_interrupt_flag_clear(EXTI_21);
        } else {
            /* Clear specific EXTI interrupt flag for independent mode */
            hals_exti_interrupt_flag_clear(linex);
        }

        /* Execute interrupt handler callback if registered */
        if(NULL != cmp_dev->cmp_irq.compare_output_handle) {
            cmp_dev->cmp_irq.compare_output_handle(cmp_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* Handle CMP internal interrupt if enabled */
    if(RESET!= (CMP_CS(cmp_dev->periph) & CMP_CS_INTEN)) {
        /* Clear CMP0 compare interrupt flag if set */
        if(RESET != hals_cmp_interrupt_flag_get(cmp_dev->periph, CMP0_INT_FLAG_COMPARE)) {
            hals_cmp_interrupt_flag_clear(cmp_dev->periph, CMP0_INT_FLAG_COMPARE);
        } else {
            /* do nothing */
        }

        /* Clear CMP1 compare interrupt flag if set */
        if(RESET != hals_cmp_interrupt_flag_get(cmp_dev->periph, CMP1_INT_FLAG_COMPARE)) {
            hals_cmp_interrupt_flag_clear(cmp_dev->periph, CMP1_INT_FLAG_COMPARE);
        } else {
            /* do nothing */
        }

        /* Update CMP peripheral state to ready */
        cmp_dev->state = HAL_CMP_STATE_READY;

        /* Execute interrupt handler callback if registered */
        if(NULL != cmp_dev->cmp_irq.compare_output_handle) {
            cmp_dev->cmp_irq.compare_output_handle(cmp_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set the CMP interrupt callback function for compare output
                which will be registered and called when corresponding interrupt is triggered
    \param[in]  cmp_dev: CMP device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: pointer to CMP interrupt callback functions structure
      \arg        non-NULL: structure containing compare output interrupt handler
      \arg        NULL: invalid parameter (function returns error)
    \param[out] none
    \retval     error code:HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_cmp_irq_handle_set(hal_cmp_dev_struct *cmp_dev, hal_cmp_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == cmp_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [cmp_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* CMP compare interrupt handler set */
    if(NULL != p_irq->compare_output_handle) {
        cmp_dev->cmp_irq.compare_output_handle = p_irq->compare_output_handle;
        hals_cmp_interrupt_enable(cmp_dev->periph, CMP_INTERRUPT_ENABLE);
    } else {
        cmp_dev->cmp_irq.compare_output_handle = NULL;
        hals_cmp_interrupt_disable(cmp_dev->periph, CMP_INTERRUPT_DISABLE);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset the CMP external trigger interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  cmp_dev: CMP device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code:HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_cmp_irq_handle_all_reset(hal_cmp_dev_struct *cmp_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == cmp_dev) {
        HAL_DEBUGE("pointer [cmp_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    cmp_dev->cmp_irq.compare_output_handle = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      lock the CMP
    \param[in]  cmp_dev: CMP device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code:HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_cmp_lock_enable(hal_cmp_dev_struct *cmp_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == cmp_dev) {
        HAL_DEBUGE("pointer [cmp_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock CMP */
    CMP_CS(cmp_dev->periph) |= (uint32_t)CMP_CS_LK;

    return HAL_ERR_NONE;
}

/*!
    \brief      config comparator output port
    \param[in]  cmp_dev: CMP device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cmp_output_sel:
      \arg        CMP_AFSE_PA6:  select PA6 as comparator alternate function output port
      \arg        CMP_AFSE_PA8:  select PA8 as comparator alternate function output port
      \arg        CMP_AFSE_PB12:  select PB12 as comparator alternate function output port
      \arg        CMP_AFSE_PE6:  select PE6 as comparator alternate function output port
      \arg        CMP_AFSE_PE15:  select PE15 as comparator alternate function output port
      \arg        CMP_AFSE_PG2:  select PG2 as comparator alternate function output port
      \arg        CMP_AFSE_PG3:  select PG3 as comparator alternate function output port
      \arg        CMP_AFSE_PG4:  select PG4 as comparator alternate function output port
      \arg        CMP_AFSE_PK0:  select PK0 as comparator alternate function output port
      \arg        CMP_AFSE_PK1:  select PK1 as comparator alternate function output port
      \arg        CMP_AFSE_PK2:  select PK2 as comparator alternate function output port
    \param[out]   none
    \retval     error code:HAL_ERR_NONE, HAL_ERR_ADDRESS. HAL_ERR_LOCK details refer to gd32h7xx_hal.h
*/
int32_t hal_cmp_output_mux_config(hal_cmp_dev_struct *cmp_dev, hal_cmp_output_select_enum cmp_output_sel)
{
    /* Variable declarations */
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == cmp_dev) {
        HAL_DEBUGE("pointer [cmp_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

   /* Check if CMP peripheral is locked (configuration protection) */
    if(CMP_CS(cmp_dev->periph) & CMP_CSLOCK_ENABLE) {
        HAL_DEBUGE(" CMP is locked ");
        /* Update CMP peripheral state to lock */
        cmp_dev->state = HAL_CMP_STATE_LOCK;
        ret = HAL_ERR_LOCK;
    } else {
        /* Configure output mux based on CMP instance (CMP0/CMP1) */
        if(CMP0 == cmp_dev->periph) {
            /* Clear output selection bits for CMP0 */
            CMP_SR &= ~(uint32_t)cmp_output_sel;
        } else if(CMP1 == cmp_dev->periph) {
            /* Set output selection bits for CMP1 */
            CMP_SR |= (uint32_t)cmp_output_sel;
        } else {
            /* do nothing */
        }
    }

    return ret;
}

/*!
    \brief      get output level
    \param[in]  cmp_dev: CMP device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_cmp_output_state_enum details refer to gd32h7xx_hal_cmp.h
*/
hal_cmp_output_state_enum hal_cmp_output_level_get(hal_cmp_dev_struct *cmp_dev)
{
    hal_cmp_output_state_enum output_level = CMP_OUTPUTLEVEL_LOW;

    if(CMP0 == cmp_dev->periph) {
        /* get output level of CMP0 */
        if(CMP_STAT & CMP_STAT_CMP0OT) {
            output_level = CMP_OUTPUTLEVEL_HIGH;
        } else {
            output_level = CMP_OUTPUTLEVEL_LOW;
        }
    } else if(CMP1 == cmp_dev->periph) {
        /* get output level of CMP1 */
        if(CMP_STAT & CMP_STAT_CMP1OT) {
            output_level = CMP_OUTPUTLEVEL_HIGH;
        } else {
            output_level = CMP_OUTPUTLEVEL_LOW;
        }
    } else {
        /* do nothing */
    }

    return output_level;
}

/*!
    \brief      CMP state get function
    \param[in]  cmp_dev: CMP device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_cmp_state_enum details refer to gd32h7xx_hal_cmp.h
*/
hal_cmp_state_enum hal_cmp_state_get(hal_cmp_dev_struct *cmp_dev)
{
    return (cmp_dev->state);
}

/*!
    \brief      CMP error state get function
    \param[in]  cmp_dev: CMP device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t: CMP error state
*/
uint32_t hal_cmp_error_state_get(hal_cmp_dev_struct *cmp_dev)
{
    return (cmp_dev->error_state);
}

/*!
    \brief      enable comparator
    \param[in]  cmp_periph:
      \arg        CMP0: comparator 0
      \arg        CMP1: comparator 1
    \param[out] none
    \retval     none
*/
void hals_cmp_enable(uint32_t cmp_periph)
{
    CMP_CS(cmp_periph) |= (uint32_t)CMP_CS_EN;
}

/*!
    \brief      disable comparator
    \param[in]  cmp_periph:
      \arg        CMP0: comparator 0
      \arg        CMP1: comparator 1
    \param[out] none
    \retval     none
*/
void hals_cmp_disable(uint32_t cmp_periph)
{
    CMP_CS(cmp_periph) &= ~(uint32_t)CMP_CS_EN;
}

/*!
    \brief      get CMP flag
    \param[in]  cmp_periph:
      \arg        CMP0: comparator 0
      \arg        CMP1: comparator 1
    \param[in]  flag: CMP flag:
      \arg        CMPx_INT_FLAG_COMPARE: CMPx compare flag (x=0,1)
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_cmp_flag_get(uint32_t cmp_periph, uint32_t flag)
{
    /* Initialize return value to RESET state */
    FlagStatus retval = RESET;

    /* Check if interrupt flag corresponds to CMP0 compare event */
    if(CMP0_INT_FLAG_COMPARE == flag) {
        /* Verify the peripheral is CMP0 */
        if((CMP0 == cmp_periph) && (RESET!= (CMP_STAT & CMP_STAT_CMP0IF))) {
            retval = SET;
        } else {
            /* do nothing */
        }
    /* Check if interrupt flag corresponds to CMP1 compare event */
    } else if(CMP1_INT_FLAG_COMPARE == flag) {
        /* Verify the peripheral is CMP1 */
        if((CMP1 == cmp_periph) & (RESET != (CMP_STAT & CMP_STAT_CMP1IF))) {
            retval = SET;
        } else {
            /* do nothing */
        }
    } else {
        /* Do nothing */
    }

    return retval;
}

/*!
    \brief      clear CMP flag
    \param[in]  cmp_periph:
      \arg        CMP0: comparator 0
      \arg        CMP1: comparator 1
    \param[in]  flag: CMP flag
      \arg        CMPx_INT_FLAG_CLEAR: CMP compare flag
    \param[out] none
    \retval     none
*/
void hals_cmp_flag_clear(uint32_t cmp_periph, uint32_t flag)
{
    /* Check if the peripheral is CMP0 */
    if(CMP0 == cmp_periph) {
        /* Clear the specified interrupt flag for CMP0 by writing to CMP_IFC register */
        CMP_IFC |= (uint32_t)flag;
    /* Check if the peripheral is CMP1 */
    } else if(CMP1 == cmp_periph) {
        /* Clear the specified interrupt flag for CMP1 by writing to CMP_IFC register */
        CMP_IFC |= (uint32_t)flag;
    } else {
        /* do nothing */
    }
}

/*!
    \brief      enable CMP interrupt
    \param[in]  cmp_periph:
      \arg        CMP0: comparator 0
      \arg        CMP1: comparator 1
    \param[in]  interrupt: CMP interrupt
                only one parameter can be selected which is shown as below:
      \arg        CMP_INTERRUPT_ENABLE: CMP compare interrupt
    \param[out] none
    \retval     none
*/
void hals_cmp_interrupt_enable(uint32_t cmp_periph, uint32_t interrupt)
{
    CMP_CS(cmp_periph) |= (uint32_t)interrupt;
}

/*!
    \brief      disable CMP interrupt
    \param[in]  cmp_periph:
      \arg        CMP0: comparator 0
      \arg        CMP1: comparator 1
    \param[in]  interrupt: CMP interrupt
                only one parameter can be selected which is shown as below:
      \arg        CMP_INTERRUPT_ENABLE: CMP compare interrupt
    \param[out] none
    \retval     none
*/
void hals_cmp_interrupt_disable(uint32_t cmp_periph, uint32_t interrupt)
{
    CMP_CS(cmp_periph) &= ~(uint32_t)interrupt;
}

/*!
    \brief      get CMP interrupt flag
    \param[in]  cmp_periph:
      \arg        CMP0: comparator 0
      \arg        CMP1: comparator 1
    \param[in]  flag: CMP interrupt flag
      \arg        CMPx_INT_FLAG_COMPARE: CMPx compare interrupt flag (x=0,1)
    \param[out]   none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_cmp_interrupt_flag_get(uint32_t cmp_periph, uint32_t flag)
{
    /* Status variables for interrupt flag and enable state */
    uint32_t intstatus = 0U, flagstatus = 0U;
    /* Initialize return value to RESET state */
    FlagStatus retval = RESET;

    if(CMP0 == cmp_periph) {
        /* Get the specified interrupt flag status from CMP_STAT register */
        flagstatus = CMP_STAT & flag;
        /* Get the interrupt enable status from CMP_CS register */
        intstatus = CMP_CS(cmp_periph) & CMP_CS_INTEN;
    } else if(CMP1 == cmp_periph) {
        /* Get the specified interrupt flag status from CMP_STAT register */
        flagstatus = CMP_STAT & flag;
        /* Get the interrupt enable status from CMP_CS register */
        intstatus = CMP_CS(cmp_periph) & CMP_CS_INTEN;
    } else {
        /* do nothing */
    }

    /* Return SET only if both interrupt flag is active AND interrupt is enabled */
    if((RESET!= flagstatus) && (RESET!= intstatus)) {
        retval = SET;
    }

    return retval;
}

/*!
    \brief      clear CMP interrupt flag
    \param[in]  cmp_periph:
      \arg        CMP0: comparator 0
      \arg        CMP1: comparator 1
    \param[in]  flag: CMP interrupt flag
      \arg        CMPx_INT_FLAG_COMPARE: CMPx compare interrupt flag (x=0,1)
    \param[out] none
    \retval     none
*/
void hals_cmp_interrupt_flag_clear(uint32_t cmp_periph, uint32_t flag)
{
    /* Check if the peripheral is CMP0 */
    if(CMP0 == cmp_periph) {
        /* Clear the specified interrupt flag for CMP0 by writing to CMP_IFC register */
        CMP_IFC |= (uint32_t)flag;
    /* Check if the peripheral is CMP1 */
    } else if(CMP1 == cmp_periph) {
        /* Clear the specified interrupt flag for CMP1 by writing to CMP_IFC register */
        CMP_IFC |= (uint32_t)flag;
    } else {
        /* do nothing */
    }
}

/*!
    \brief      CMP Compare output handler function
    \param[in]  cmp_dev: pointer to a cmp device information structure
    \param[out] none
    \retval     none
*/
static void _cmp_compare_output_handle(void *cmp_dev)
{
    /* Get pointer to CMP device structure */
    hal_cmp_dev_struct *p_cmp = cmp_dev;

    /* Check if compare output callback function is registered */
    if(p_cmp->compare_output_callback != NULL) {
        /* Cast callback to proper function pointer type */
        hal_cmp_user_cb p_func = (hal_cmp_user_cb)p_cmp->compare_output_callback;
        /* Execute the user-defined callback function with CMP device context */
        p_func(p_cmp);
    } else {
        /* do nothing */
    }
}
