/*!
    \file    gd32h7xx_hal_cau_tdes.c
    \brief   CAU TDES driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* cau fifo input interrupt callback function */
static void _cau_fifo_input_callback(void *cau_dev);
/* cau fifo output interrupt callback function */
static void _cau_fifo_output_callback(void *cau_dev);
/* TDES calculate process */
static ErrStatus _tdes_calculate(uint8_t *input, uint32_t in_length, uint8_t *output);

/*!
    \brief      encrypt and decrypt using TDES in ECB mode
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 8 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 8 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 8 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[out] output: pointer to the output structure
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_cau_tdes_ecb(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output)
{
    ErrStatus ret = ERROR;
    hal_cau_key_parameter_struct key_initpara;
    uint32_t keyaddr    = (uint32_t)(cau_parameter->key);
    uint32_t inputaddr  = (uint32_t)(cau_parameter->input);
    uint32_t outputaddr = (uint32_t)output;

    if(HAL_CAU_STATE_READY == cau_dev->state) {
        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_TDES_ECB, cau_parameter->data_change_mode);

        /* key initialization */
        key_initpara.key_1_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_1_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_2_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_2_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_3_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_3_low = __REV(*(uint32_t *)(keyaddr));
        hals_cau_key_init(&key_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;

        /* enable the CAU peripheral */
        hals_cau_enable();
        /* TDES calculate process */
        ret = _tdes_calculate((uint8_t *)inputaddr, cau_parameter->in_length, (uint8_t *)outputaddr);
        /* disable the CAU peripheral */
        hals_cau_disable();

        cau_dev->state = HAL_CAU_STATE_READY;
    } else {
        /* nothing to do*/
    }

    return ret;
}

/*!
    \brief      encrypt and decrypt using TDES in CBC mode
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 8 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 8 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 8 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[out] output: pointer to the output structure
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_cau_tdes_cbc(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output)
{
    ErrStatus ret = ERROR;
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    uint32_t keyaddr    = (uint32_t)(cau_parameter->key);
    uint32_t inputaddr  = (uint32_t)(cau_parameter->input);
    uint32_t outputaddr = (uint32_t)output;
    uint32_t ivaddr     = (uint32_t)(cau_parameter->iv);
    if(HAL_CAU_STATE_READY == cau_dev->state) {
        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_TDES_CBC, cau_parameter->data_change_mode);

        /* key initialization */
        key_initpara.key_1_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_1_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_2_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_2_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_3_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_3_low = __REV(*(uint32_t *)(keyaddr));
        hals_cau_key_init(&key_initpara);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ivaddr));
        hals_cau_iv_init(&iv_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;
        /* enable the CAU peripheral */
        hals_cau_enable();
        /* TDES calculate process */
        ret = _tdes_calculate((uint8_t *)inputaddr, cau_parameter->in_length, (uint8_t *)outputaddr);
        /* disable the CAU peripheral */
        hals_cau_disable();

        cau_dev->state = HAL_CAU_STATE_READY;
    } else {
        /* nothing to do*/
    }

    return ret;
}

/*!
    \brief      encrypt and decrypt using TDES in ECB mode for interrupt
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 8 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 8 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 8 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[in]  p_user_func: pointer to user callback functions structure
                  The structure contains the following members:
      \arg        fifo_input_callback: input data transfer complete callback function
      \arg        fifo_output_callback: output data transfer complete callback function
    \param[out] output: pointer to the output structure
    \retval     none
*/
void hals_cau_tdes_ecb_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                 hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output)
{
    hal_cau_key_parameter_struct key_initpara;
    uint32_t keyaddr = (uint32_t)(cau_parameter->key);

    if(HAL_CAU_STATE_READY == cau_dev->state) {
        cau_dev->state = HAL_CAU_STATE_BUSY;

        cau_dev->in_buffer  = (uint32_t *)(cau_parameter->input);
        cau_dev->in_size    = (uint32_t)(cau_parameter->in_length);
        cau_dev->out_buffer = (uint32_t *)output;

        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_TDES_ECB, cau_parameter->data_change_mode);

        /* key initialization */
        key_initpara.key_1_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_1_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_2_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_2_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_3_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_3_low = __REV(*(uint32_t *)(keyaddr));
        hals_cau_key_init(&key_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;

        cau_dev->fifo_input_callback  = NULL;
        cau_dev->fifo_output_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->fifo_input_callback) {
                cau_dev->fifo_input_callback = (void *)p_user_func->fifo_input_callback;
            } else {
                /* nothing to do */
            }

            if(NULL != p_user_func->fifo_output_callback) {
                cau_dev->fifo_output_callback = (void *)p_user_func->fifo_output_callback;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* config callback func */
        cau_dev->cau_irq.fifo_input_handle  = _cau_fifo_input_callback;
        cau_dev->cau_irq.fifo_output_handle = _cau_fifo_output_callback;
        /* enable interrupt */
        hals_cau_interrupt_enable(CAU_INT_INFIFO | CAU_INT_OUTFIFO);
        /* enable the CAU peripheral */
        hals_cau_enable();
    } else {
        /* nothing to do*/
    }
}

/*!
    \brief      encrypt and decrypt using TDES in CBC mode for interrupt
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 8 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 8 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 8 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[in]  p_user_func: pointer to user callback functions structure
                  The structure contains the following members:
      \arg        fifo_input_callback: input data transfer complete callback function
      \arg        fifo_output_callback: output data transfer complete callback function
    \param[out] output: pointer to the output structure
    \retval     none
*/
void hals_cau_tdes_cbc_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                 hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output)
{
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    uint32_t keyaddr = (uint32_t)(cau_parameter->key);
    uint32_t ivaddr  = (uint32_t)(cau_parameter->iv);

    if(HAL_CAU_STATE_READY == cau_dev->state) {
        cau_dev->state = HAL_CAU_STATE_BUSY;

        cau_dev->in_buffer  = (uint32_t *)(cau_parameter->input);
        cau_dev->in_size    = (uint32_t)(cau_parameter->in_length);
        cau_dev->out_buffer = (uint32_t *)output;

        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_TDES_CBC, cau_parameter->data_change_mode);

        /* key initialization */
        key_initpara.key_1_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_1_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_2_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_2_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_3_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_3_low = __REV(*(uint32_t *)(keyaddr));
        hals_cau_key_init(&key_initpara);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ivaddr));
        hals_cau_iv_init(&iv_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;

        cau_dev->fifo_input_callback  = NULL;
        cau_dev->fifo_output_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->fifo_input_callback) {
                cau_dev->fifo_input_callback = (void *)p_user_func->fifo_input_callback;
            } else {
                /* nothing to do */
            }

            if(NULL != p_user_func->fifo_output_callback) {
                cau_dev->fifo_output_callback = (void *)p_user_func->fifo_output_callback;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* config callback func */
        cau_dev->cau_irq.fifo_input_handle  = _cau_fifo_input_callback;
        cau_dev->cau_irq.fifo_output_handle = _cau_fifo_output_callback;

        /* enable interrupt */
        hals_cau_interrupt_enable(CAU_INT_INFIFO | CAU_INT_OUTFIFO);
        /* enable the CAU peripheral */
        hals_cau_enable();
    } else {
        /* nothing to do*/
    }
}

/*!
    \brief      encrypt and decrypt using TDES in ECB mode for DMA
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 8 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 8 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 8 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[in]  dmacb: CAU DMA callback function pointer structure
                only one parameter can be selected which is shown as below:
      \arg        in_full_transcom_handle: CAU DMA input full transfer complete interrupt handler
      \arg        in_half_transcom_handle: CAU DMA input half transfer complete interrupt handler
      \arg        out_full_transcom_handle: CAU DMA output full transfer complete interrupt handler
      \arg        out_half_transcom_handle: CAU DMA output half transfer complete interrupt handler
      \arg        error_handle:CAU DMA error interrupt handler
    \param[out] output: pointer to the output structure
    \retval     none
*/
void hals_cau_tdes_ecb_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                           hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output)
{
    hal_cau_key_parameter_struct key_initpara;
    uint32_t keyaddr = (uint32_t)(cau_parameter->key);

    if(HAL_CAU_STATE_READY == cau_dev->state) {
        cau_dev->state = HAL_CAU_STATE_BUSY;

        cau_dev->in_buffer  = (uint32_t *)(cau_parameter->input);
        cau_dev->in_size    = (uint32_t)(cau_parameter->in_length);
        cau_dev->out_buffer = (uint32_t *)output;

        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_TDES_ECB, cau_parameter->data_change_mode);

        /* key initialization */
        key_initpara.key_1_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_1_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_2_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_2_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_3_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_3_low = __REV(*(uint32_t *)(keyaddr));
        hals_cau_key_init(&key_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;
        /* dma transfer config */
        hals_cau_dma_transfer_config(cau_dev, dmacb);
    } else {
        /* nothing to do*/
    }
}

/*!
    \brief      encrypt and decrypt using TDES in CBC mode for DMA
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 8 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 8 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 8 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[in]  dmacb: CAU DMA callback function pointer structure
                only one parameter can be selected which is shown as below:
      \arg        in_full_transcom_handle: CAU DMA input full transfer complete interrupt handler
      \arg        in_half_transcom_handle: CAU DMA input half transfer complete interrupt handler
      \arg        out_full_transcom_handle: CAU DMA output full transfer complete interrupt handler
      \arg        out_half_transcom_handle: CAU DMA output half transfer complete interrupt handler
      \arg        error_handle:CAU DMA error interrupt handler
    \param[out] output: pointer to the output structure
    \retval     none
*/
void hals_cau_tdes_cbc_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                           hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output)
{
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    uint32_t keyaddr = (uint32_t)(cau_parameter->key);
    uint32_t ivaddr  = (uint32_t)(cau_parameter->iv);

    if(HAL_CAU_STATE_READY == cau_dev->state) {
        cau_dev->state = HAL_CAU_STATE_BUSY;

        cau_dev->in_buffer  = (uint32_t *)(cau_parameter->input);
        cau_dev->in_size    = (uint32_t)(cau_parameter->in_length);
        cau_dev->out_buffer = (uint32_t *)output;

        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_TDES_CBC, cau_parameter->data_change_mode);

        /* key initialization */
        key_initpara.key_1_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_1_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_2_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_2_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_3_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        key_initpara.key_3_low = __REV(*(uint32_t *)(keyaddr));
        hals_cau_key_init(&key_initpara);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ivaddr));
        hals_cau_iv_init(&iv_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;
        /* dma transfer config */
        hals_cau_dma_transfer_config(cau_dev, dmacb);
    } else {
        /* nothing to do*/
    }
}

/*!
    \brief      CAU fifo input callback
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _cau_fifo_input_callback(void *cau_dev)
{
    hal_cau_dev_struct *p_cau_dev = (hal_cau_dev_struct *)cau_dev;

    if(HAL_CAU_STATE_BUSY == p_cau_dev->state) {
        if(SET == hals_cau_flag_get(CAU_FLAG_INFIFO)) {
            if(NULL != p_cau_dev->fifo_input_callback) {
                ((hal_irq_handle_cb)(p_cau_dev->fifo_input_callback))(p_cau_dev);
            } else {
                /* nothing to do*/
            }
        } else {
            /* nothing to do*/
        }
    } else {
        /* nothing to do*/
    }
}

/*!
    \brief      CAU fifo output callback
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _cau_fifo_output_callback(void *cau_dev)
{
    hal_cau_dev_struct *p_cau_dev = (hal_cau_dev_struct *)cau_dev;

    if(HAL_CAU_STATE_BUSY == p_cau_dev->state) {
        if(SET == hals_cau_flag_get(CAU_FLAG_OUTFIFO)) {
            if(NULL != p_cau_dev->fifo_output_callback) {
                ((hal_irq_handle_cb)p_cau_dev->fifo_output_callback)(p_cau_dev);
            } else {
                /* nothing to do*/
            }
        } else {
            /* nothing to do*/
        }
    } else {
        /* nothing to do*/
    }
}

/*!
    \brief      TDES calculate process
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer in bytes, must be a multiple of 8 bytes
    \param[out] output: pointer to the returned buffer
    \retval     ErrStatus: SUCCESS or ERROR
*/
static ErrStatus _tdes_calculate(uint8_t *input, uint32_t in_length, uint8_t *output)
{
    uint32_t inputaddr    = (uint32_t)input;
    uint32_t outputaddr   = (uint32_t)output;
    uint32_t i            = 0U;
    __IO uint32_t counter = 0U;
    uint32_t busystatus   = 0U;
    ErrStatus status = SUCCESS;

    /* the clock is not enabled or there is no embedded CAU peripheral */
    if(RESET == (CAU_CTL & CAU_CTL_CAUEN)) {
        status = ERROR;
    } else {
        /* nothing to do*/
    }

    for(i = 0U; i < in_length; i += 8U) {
        /* write data to the in fifo */
        hals_cau_data_write(*(uint32_t *)(inputaddr));
        inputaddr += 4U;
        hals_cau_data_write(*(uint32_t *)(inputaddr));
        inputaddr += 4U;

        /* wait until the complete message has been processed */
        counter = 0U;
        do {
            busystatus = hals_cau_flag_get(CAU_FLAG_BUSY);
            counter++;
        } while((TDESBUSY_TIMEOUT != counter) && (RESET != busystatus));

        if(RESET != busystatus) {
            status = ERROR;
        } else {
            /* read the output block from the output FIFO */
            *(uint32_t *)(outputaddr) = hals_cau_data_read();
            outputaddr += 4U;
            *(uint32_t *)(outputaddr) = hals_cau_data_read();
            outputaddr += 4U;
        }
    }

    return status;
}
