/*!
    \file    gd32h7xx_hal_cau_aes.c
    \brief   CAU AES driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* cau fifo input interrupt callback function*/
static void _cau_fifo_input_callback(void *cau_dev);
/* cau fifo output interrupt callback function*/
static void _cau_fifo_output_callback(void *cau_dev);
/* configure AES key structure parameter */
static void _cau_aes_key_config(uint8_t *key, uint32_t keysize, hal_cau_key_parameter_struct *cau_key_initpara);
/* fill data into data input register */
static ErrStatus _cau_aes_fill_data(uint8_t *input, uint32_t in_length);
/* AES calculate process */
static ErrStatus _cau_aes_calculate(uint8_t *input, uint32_t in_length, uint8_t *output);

/*!
    \brief      encrypt and decrypt using AES in ECB mode
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[out] output: pointer to the returned buffer
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_cau_aes_ecb(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output)
{
    ErrStatus ret = SUCCESS;
    hal_cau_key_parameter_struct key_initpara;
    __IO uint32_t counter = 0U;
    uint32_t busystatus   = 0U;

    if(cau_dev->state == HAL_CAU_STATE_READY) {
        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* AES decryption */
        if(CAU_DECRYPT == cau_parameter->alg_dir) {
            /* flush the IN and out fifo */
            CAU_CTL |= CAU_CTL_FFLUSH;
            /* initialize the CAU peripheral */
            hals_cau_init_param(CAU_DECRYPT, CAU_MODE_AES_KEY, cau_parameter->data_change_mode);

            /* enable the CAU peripheral */
            hals_cau_enable();

            /* wait until the busy flag is RESET */
            do {
                busystatus = hals_cau_flag_get(CAU_FLAG_BUSY);
                counter++;
            } while((AESBUSY_TIMEOUT != counter) && (RESET != busystatus));

            if(RESET != busystatus) {
                ret = ERROR;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_ECB, cau_parameter->data_change_mode);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;

        /* enable the CAU peripheral */
        hals_cau_enable();
        /* AES calculate process */
        ret = _cau_aes_calculate(cau_parameter->input, cau_parameter->in_length, output);
        /* disable the CAU peripheral */
        hals_cau_disable();

        cau_dev->state = HAL_CAU_STATE_READY;
    } else {
        /* nothing to do */
    }

    return ret;
}

/*!
    \brief      encrypt and decrypt using AES in CBC mode
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[out] output: pointer to the returned buffer
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_cau_aes_cbc(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output)
{
    ErrStatus ret = SUCCESS;
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    __IO uint32_t counter = 0U;
    uint32_t busystatus   = 0U;

    uint32_t ivaddr = (uint32_t)cau_parameter->iv;

    if(cau_dev->state == HAL_CAU_STATE_READY) {
        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* AES decryption */
        if(CAU_DECRYPT == cau_parameter->alg_dir) {
            /* flush the IN and out fifo */
            CAU_CTL |= CAU_CTL_FFLUSH;
            /* initialize the CAU peripheral */
            hals_cau_init_param(CAU_DECRYPT, CAU_MODE_AES_KEY, cau_parameter->data_change_mode);

            /* enable the CAU peripheral */
            hals_cau_enable();

            /* wait until the busy flag is RESET */
            do {
                busystatus = hals_cau_flag_get(CAU_FLAG_BUSY);
                counter++;
            } while((AESBUSY_TIMEOUT != counter) && (RESET != busystatus));

            if(RESET != busystatus) {
                ret = ERROR;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_CBC, cau_parameter->data_change_mode);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_low = __REV(*(uint32_t *)(ivaddr));
        hals_cau_iv_init(&iv_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;

        /* enable the CAU peripheral */
        hals_cau_enable();
        /* AES calculate process */
        ret = _cau_aes_calculate(cau_parameter->input, cau_parameter->in_length, output);
        /* disable the CAU peripheral */
        hals_cau_disable();

        cau_dev->state = HAL_CAU_STATE_READY;
    } else {
        /* nothing to do */
    }

    return ret;
}

/*!
    \brief      encrypt and decrypt using AES in CTR mode
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[out] output: pointer to the returned buffer
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_cau_aes_ctr(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output)
{
    ErrStatus ret = SUCCESS;
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    uint32_t ivaddr = (uint32_t)cau_parameter->iv;

    if(cau_dev->state == HAL_CAU_STATE_READY) {
        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_CTR, cau_parameter->data_change_mode);

        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_low = __REV(*(uint32_t *)(ivaddr));
        hals_cau_iv_init(&iv_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;

        /* enable the CAU peripheral */
        hals_cau_enable();
        /* AES calculate process */
        ret = _cau_aes_calculate(cau_parameter->input, cau_parameter->in_length, output);
        /* disable the CAU peripheral */
        hals_cau_disable();

        cau_dev->state = HAL_CAU_STATE_READY;
    } else {
        /* nothing to do */
    }

    return ret;
}

/*!
    \brief      encrypt and decrypt using AES in CFB mode
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[out] output: pointer to the returned buffer
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_cau_aes_cfb(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output)
{
    ErrStatus ret = ERROR;
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    uint32_t ivaddr = (uint32_t)cau_parameter->iv;

    if(cau_dev->state == HAL_CAU_STATE_READY) {
        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_CFB, cau_parameter->data_change_mode);

        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_low = __REV(*(uint32_t *)(ivaddr));
        hals_cau_iv_init(&iv_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;
        /* enable the CAU peripheral */
        hals_cau_enable();
        /* AES calculate process */
        ret = _cau_aes_calculate(cau_parameter->input, cau_parameter->in_length, output);
        /* disable the CAU peripheral */
        hals_cau_disable();

        cau_dev->state = HAL_CAU_STATE_READY;
    } else {
        /* nothing to do */
    }

    return ret;
}

/*!
    \brief      encrypt and decrypt using AES in OFB mode
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[out] output: pointer to the returned buffer
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_cau_aes_ofb(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output)
{
    ErrStatus ret = SUCCESS;
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    uint32_t ivaddr = (uint32_t)cau_parameter->iv;

    if(cau_dev->state == HAL_CAU_STATE_READY) {
        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_OFB, cau_parameter->data_change_mode);

        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_low = __REV(*(uint32_t *)(ivaddr));
        hals_cau_iv_init(&iv_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;
        /* enable the CAU peripheral */
        hals_cau_enable();
        /* AES calculate process */
        ret = _cau_aes_calculate(cau_parameter->input, cau_parameter->in_length, output);
        /* disable the CAU peripheral */
        hals_cau_disable();

        cau_dev->state = HAL_CAU_STATE_READY;
    } else {
        /* nothing to do */
    }

    return ret;
}

/*!
    \brief      encrypt and decrypt using AES in GCM mode
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[out] output: pointer to the returned output data buffer
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_cau_aes_gcm(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output)
{
    ErrStatus ret = SUCCESS;
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    uint64_t aadlength   = (uint64_t)cau_parameter->aad_size * 8U;
    uint64_t inputlength = (uint64_t)cau_parameter->in_length * 8U;
    uint32_t ivaddr      = (uint32_t)cau_parameter->iv;
    uint32_t tagaddr     = (uint32_t)cau_dev->tag;
    if(HAL_CAU_STATE_READY == cau_dev->state) {
        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_GCM, cau_parameter->data_change_mode);

        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_low = __REV(*(uint32_t *)(ivaddr));
        hals_cau_iv_init(&iv_initpara);

        /* prepare phase */
        /* select prepare phase */
        hals_cau_phase_config(CAU_PREPARE_PHASE);
        /* enable the CAU peripheral */
        hals_cau_enable();
        /* wait for CAUEN bit to be 0 */
        while(RESET != (CAU_CTL & CAU_CTL_CAUEN)) {}

        /* aad phase */
        if((uint32_t)0U != cau_parameter->aad_size) {
            /* select aad phase */
            hals_cau_phase_config(CAU_AAD_PHASE);
            /* flush the IN and out fifo */
            CAU_CTL |= CAU_CTL_FFLUSH;
            /* enable the CAU peripheral */
            hals_cau_enable();

            ret = _cau_aes_fill_data(cau_parameter->aad, cau_parameter->aad_size);

            if(ERROR == ret) {
                return ret;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* encrypt or decrypt phase */
        if((uint32_t)0U != cau_parameter->in_length) {
            /* select encrypt or decrypt phase */
            hals_cau_phase_config(CAU_ENCRYPT_DECRYPT_PHASE);
            /* flush the IN and out fifo */
            CAU_CTL |= CAU_CTL_FFLUSH;
            /* enable the CAU peripheral */
            hals_cau_enable();

            /* AES calculate process */
            ret = _cau_aes_calculate(cau_parameter->input, cau_parameter->in_length, output);

            if(ERROR == ret) {
                return ret;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* tag phase */
        /* select tag phase */
        hals_cau_phase_config(CAU_TAG_PHASE);
        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;
        /* enable the CAU peripheral */
        hals_cau_enable();

        if(RESET == (CAU_CTL & CAU_CTL_CAUEN)) {
            ret = ERROR;
        } else {
            /* nothing to do */
        }

        hals_cau_data_write(__REV((uint32_t)(aadlength >> 32U)));
        hals_cau_data_write(__REV((uint32_t)aadlength));
        hals_cau_data_write(__REV((uint32_t)(inputlength >> 32U)));
        hals_cau_data_write(__REV((uint32_t)inputlength));

        /* wait until the out fifo not empty flag is set */
        while(RESET == hals_cau_flag_get(CAU_FLAG_OUTFIFO_NO_EMPTY)) {}

        /* read the tag in the out fifo */
        *(uint32_t *)(tagaddr) = hals_cau_data_read();
        tagaddr += 4U;
        *(uint32_t *)(tagaddr) = hals_cau_data_read();
        tagaddr += 4U;
        *(uint32_t *)(tagaddr) = hals_cau_data_read();
        tagaddr += 4U;
        *(uint32_t *)(tagaddr) = hals_cau_data_read();
        tagaddr += 4U;

        /* disable the CAU peripheral */
        hals_cau_disable();

        cau_dev->state = HAL_CAU_STATE_READY;
    } else {
        /* nothing to do */
    }

    return ret;
}

/*!
    \brief      encrypt and decrypt using AES in CCM mode
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[out] output: pointer to the returned output data buffer
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_cau_aes_ccm(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output)
{
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    ErrStatus ret           = SUCCESS;
    uint32_t inputaddr      = (uint32_t)cau_parameter->input;
    uint32_t inputsize      = cau_parameter->in_length;
    uint32_t aadaddr        = (uint32_t)cau_parameter->aad;
    uint32_t aadsize        = cau_parameter->aad_size;
    uint32_t aad_block_size = 0U;
    uint32_t ivaddr         = (uint32_t)cau_parameter->iv;
    uint32_t ivsize         = cau_parameter->iv_size;
    uint32_t outputaddr     = (uint32_t)output;
    uint32_t i = 0U, plen   = 0U;
    uint32_t head_index     = 0U;
    uint8_t blockb0[16U]    = {0U};
    uint8_t counter[16U]    = {0U};
    uint32_t ctraddr        = (uint32_t)counter;
    uint32_t b0addr         = (uint32_t)blockb0;
    uint32_t temp_tag[4U];

    if(HAL_CAU_STATE_READY == cau_dev->state) {
        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* formatting the aad block */
        if((uint32_t)0U != aadsize) {
            /* check that the aad length is lower than 2^16 - 2^8 = 65536 - 256 = 65280 */
            if(aadsize < 65280U) {
                cau_dev->aad_buffer[head_index] = (uint8_t)((aadsize >> 8U) & 0xFFU);
                head_index++;
                cau_dev->aad_buffer[head_index] = (uint8_t)((aadsize) & 0xFFU);
                head_index++;
                aad_block_size                  = aadsize + 2U;
            } else {
                /* aad is encoded as 0xFF || 0xFE || [aadsize]32, i.e., six octets */
                cau_dev->aad_buffer[head_index] = 0xFFU;
                head_index++;
                cau_dev->aad_buffer[head_index] = 0xFEU;
                head_index++;
                cau_dev->aad_buffer[head_index] = (uint8_t)((aadsize & 0xFF000000U) >> 24U);
                head_index++;
                cau_dev->aad_buffer[head_index] = (uint8_t)((aadsize & 0x00FF0000U) >> 16U);
                head_index++;
                cau_dev->aad_buffer[head_index] = (uint8_t)((aadsize & 0x0000FF00U) >> 8U);
                head_index++;
                cau_dev->aad_buffer[head_index] = (uint8_t)(aadsize & 0x000000FFU);
                head_index++;
                aad_block_size                    = aadsize + 6U;
            }

            /* copy the aad buffer in internal buffer "HBuffer" */
            for(i = 0U; i < aadsize; i++) {
                cau_dev->aad_buffer[head_index] = *(uint8_t *)((uint32_t)(aadaddr + i));
                head_index++;
            }

            /* check if the aad block size is modulo 16 */
            if(0U != (aad_block_size % 16U)) {
                /* pad the aad buffer with 0s till the HBuffer length is modulo 16 */
                for(i = aad_block_size; i <= ((aad_block_size / 16U) + 1U) * 16U; i++) {
                    cau_dev->aad_buffer[i] = 0U;
                }
                /* set the aad size to modulo 16 */
                aad_block_size = ((aad_block_size / 16U) + 1U) * 16U;
            } else {
                /* nothing to do */
            }

            /* set the pointer aadaddr to HBuffer */
            aadaddr = (uint32_t)cau_dev->aad_buffer;
        }

        /* formatting the block B0 */
        if(0U != aadsize) {
            blockb0[0] = 0x40U;
        } else {
            /* nothing to do */
        }

        /* flags byte */
        blockb0[0] |= (0U | (((((uint8_t)cau_dev->tag_size - 2U) / 2U) & 0x07U) << 3U) | \
                       (((uint8_t)(15U - ivsize) - 1U) & 0x07U));

        if((MIN_CCM_IV_SIZE > ivsize) || (MAX_CCM_IV_SIZE < ivsize)) {
            ret = ERROR;
        } else {
            /* nothing to do */
        }

        for(i = 0U; i < ivsize; i++) {
            blockb0[i + 1U] = *(uint8_t *)((uint32_t)(ivaddr + i));
        }

        /* the byte length for payload length expressing, which plus the ivsize must equal to 15 bytes */
        plen = 15U - ivsize;
        /* if the byte length for payload length expressing is more than 4 bytes */
        if(plen > 4U) {
            /* pad the blockb0 after vectors, and before the last 4 bytes */
            for(i = i; i < 11U; i++) {
                blockb0[i + 1U] = 0U;
            }
            blockb0[12U] = (uint8_t)((inputsize >> 24U) & 0xFFU);
            blockb0[13U] = (uint8_t)((inputsize >> 16U) & 0xFFU);
            blockb0[14U] = (uint8_t)((inputsize >> 8U) & 0xFFU);
            blockb0[15U] = (uint8_t)(inputsize & 0xFFU);
        } else {
            /* the payload length is expressed in plen bytes */
            for(i = i; i < 15U; i++) {
                blockb0[i + 1U] = (uint8_t)((inputsize >> ((plen - 1U) * 8U)) & 0xFFU);
                plen--;
            }
        }

        /* formatting the initial counter */
        /* byte 0: bits 0-2 contain the same encoding of q as in B0 */
        counter[0] = blockb0[0] & BLOCK_B0_MASK;
        for(i = 1U; i < ivsize + 1U; i++) {
            counter[i] = blockb0[i];
        }

        /* set the LSB to 1 */
        counter[15] |= 0x01U;

        /* prepare phase */
        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;
        /* clear CAUEN bit to ensure CAU is disable */
        hals_cau_disable();

        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_CCM, cau_parameter->data_change_mode);
        /* select init phase */
        hals_cau_phase_config(CAU_PREPARE_PHASE);

        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ctraddr));
        ctraddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ctraddr));
        ctraddr += 4U;
        iv_initpara.iv_1_high = __REV(*(uint32_t *)(ctraddr));
        ctraddr += 4U;
        iv_initpara.iv_1_low = __REV(*(uint32_t *)(ctraddr));
        hals_cau_iv_init(&iv_initpara);

        /* enable the CAU peripheral */
        hals_cau_enable();

        /* write block B0 in the IN fifo */
        hals_cau_data_write(*(uint32_t *)(b0addr));
        b0addr += 4U;
        hals_cau_data_write(*(uint32_t *)(b0addr));
        b0addr += 4U;
        hals_cau_data_write(*(uint32_t *)(b0addr));
        b0addr += 4U;
        hals_cau_data_write(*(uint32_t *)(b0addr));

        /* wait for CAUEN bit to be 0 */
        while(RESET != (CAU_CTL & CAU_CTL_CAUEN)) {}

        /* aad phase */
        if((uint32_t)0U != aadsize) {
            /* select aad phase */
            hals_cau_phase_config(CAU_AAD_PHASE);
            /* enable the CAU peripheral */
            hals_cau_enable();

            ret = _cau_aes_fill_data((uint8_t *)aadaddr, aad_block_size);

            if(ERROR == ret) {
                return ret;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* encrypt or decrypt phase */
        inputsize = cau_parameter->in_length;

        if((uint32_t)0U != inputsize) {
            /* select encrypt or decrypt phase */
            hals_cau_phase_config(CAU_ENCRYPT_DECRYPT_PHASE);
            /* enable the CAU peripheral */
            hals_cau_enable();

            /* AES calculate process */
            ret = _cau_aes_calculate((uint8_t *)inputaddr, inputsize, (uint8_t *)outputaddr);

            if(ERROR == ret) {
                return ret;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* tag phase */
        /* select final phase */
        hals_cau_phase_config(CAU_TAG_PHASE);
        /* enable the CAU peripheral */
        hals_cau_enable();

        if(RESET == (CAU_CTL & CAU_CTL_CAUEN)) {
            ret = ERROR;
        } else {
            /* nothing to do */
        }

        ctraddr = (uint32_t)counter;

        hals_cau_data_write(*(uint32_t *)(ctraddr));
        ctraddr += 4U;
        hals_cau_data_write(*(uint32_t *)(ctraddr));
        ctraddr += 4U;
        hals_cau_data_write(*(uint32_t *)(ctraddr));
        ctraddr += 4U;
        /* reset bit 0 (after 8-bit swap) is equivalent to reset bit 24 (before 8-bit swap) */
        hals_cau_data_write(*(uint32_t *)(ctraddr) & 0xFEFFFFFFU);

        /* wait until the out fifo not empty flag is set */
        while(RESET == hals_cau_flag_get(CAU_FLAG_OUTFIFO_NO_EMPTY)) {}

        /* read the tag in the out fifo */
        temp_tag[0] = hals_cau_data_read();
        temp_tag[1] = hals_cau_data_read();
        temp_tag[2] = hals_cau_data_read();
        temp_tag[3] = hals_cau_data_read();

        /* disable the CAU peripheral */
        hals_cau_disable();

        /* copy temporary authentication tag in user tag buffer */
        for(i = 0U; i < cau_dev->tag_size; i++) {
            cau_dev->tag[i] = (uint8_t)(temp_tag[i / 4U] >> (8U * (i % 4U)));
        }

        cau_dev->state = HAL_CAU_STATE_READY;
    } else {
        /* nothing to do */
    }

    return ret;
}

/*!
    \brief      get authentication tag from CAU peripheral after CCM process
    \param[in]  cau_dev: CAU device information structure
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[out] none
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_cau_aes_ccm_generate_tag(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter)
{
    ErrStatus ret           = SUCCESS;
    uint32_t inputsize      = cau_parameter->in_length;
    uint32_t aadsize        = cau_parameter->aad_size;
    uint32_t ivsize         = cau_parameter->iv_size;
    uint32_t ivaddr         = (uint32_t)cau_parameter->iv;
    uint32_t i = 0U, plen   = 0U;
    uint8_t blockb0[16U]    = {0U};
    uint8_t counter[16U]    = {0U};
    uint32_t ctraddr        = (uint32_t)counter;
    uint32_t temp_tag[4U];

    if(HAL_CAU_STATE_READY == cau_dev->state) {
        cau_dev->state = HAL_CAU_STATE_BUSY;

        /* formatting the block B0 */
        if(0U != aadsize) {
            blockb0[0] = 0x40U;
        } else {
            /* nothing to do */
        }

        /* flags byte */
        blockb0[0] |= (0U | (((((uint8_t)cau_dev->tag_size - 2U) / 2U) & 0x07U) << 3U) | \
                       (((uint8_t)(15U - ivsize) - 1U) & 0x07U));

        if((MIN_CCM_IV_SIZE > ivsize) || (MAX_CCM_IV_SIZE < ivsize)) {
            ret = ERROR;
        } else {
            /* nothing to do */
        }

        for(i = 0U; i < ivsize; i++) {
            blockb0[i + 1U] = *(uint8_t *)((uint32_t)(ivaddr + i));
        }

        /* the byte length for payload length expressing, which plus the ivsize must equal to 15 bytes */
        plen = 15U - ivsize;
        /* if the byte length for payload length expressing is more than 4 bytes */
        if(plen > 4U) {
            /* pad the blockb0 after vectors, and before the last 4 bytes */
            for(i = i; i < 11U; i++) {
                blockb0[i + 1U] = 0U;
            }
            blockb0[12U] = (uint8_t)((inputsize >> 24U) & 0xFFU);
            blockb0[13U] = (uint8_t)((inputsize >> 16U) & 0xFFU);
            blockb0[14U] = (uint8_t)((inputsize >> 8U) & 0xFFU);
            blockb0[15U] = (uint8_t)(inputsize & 0xFFU);
        } else {
            /* the payload length is expressed in plen bytes */
            for(i = i; i < 15U; i++) {
                blockb0[i + 1U] = (uint8_t)((inputsize >> ((plen - 1U) * 8U)) & 0xFFU);
                plen--;
            }
        }

        /* formatting the initial counter */
        /* byte 0: bits 0-2 contain the same encoding of q as in B0 */
        counter[0] = blockb0[0] & BLOCK_B0_MASK;
        for(i = 1U; i < ivsize + 1U; i++) {
            counter[i] = blockb0[i];
        }

        /* set the LSB to 1 */
        counter[15] |= 0x01U;

        /* tag phase */
        /* select final phase */
        hals_cau_phase_config(CAU_TAG_PHASE);
        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;
        /* enable the CAU peripheral */
        hals_cau_enable();

        if(RESET == (CAU_CTL & CAU_CTL_CAUEN)) {
            ret = ERROR;
        } else {
            /* nothing to do */
        }

        ctraddr = (uint32_t)counter;

        hals_cau_data_write(*(uint32_t *)(ctraddr));
        ctraddr += 4U;
        hals_cau_data_write(*(uint32_t *)(ctraddr));
        ctraddr += 4U;
        hals_cau_data_write(*(uint32_t *)(ctraddr));
        ctraddr += 4U;
        /* reset bit 0 (after 8-bit swap) is equivalent to reset bit 24 (before 8-bit swap) */
        hals_cau_data_write(*(uint32_t *)(ctraddr) & 0xFEFFFFFFU);

        /* wait until the out fifo not empty flag is set */
        while(RESET == hals_cau_flag_get(CAU_FLAG_OUTFIFO_NO_EMPTY)) {}

        /* read the tag in the out fifo */
        temp_tag[0] = hals_cau_data_read();
        temp_tag[1] = hals_cau_data_read();
        temp_tag[2] = hals_cau_data_read();
        temp_tag[3] = hals_cau_data_read();

        /* disable the CAU peripheral */
        hals_cau_disable();

        /* copy temporary authentication tag in user tag buffer */
        for(i = 0U; i < cau_dev->tag_size; i++) {
            cau_dev->tag[i] = (uint8_t)(temp_tag[i / 4U] >> (8U * (i % 4U)));
        }

        cau_dev->state = HAL_CAU_STATE_READY;
    } else {
        /* nothing to do */
    }

    return ret;
}

/*!
    \brief      get authentication tag from CAU peripheral after GCM process
    \param[in]  cau_dev: CAU device information structure
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[out] none
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_cau_aes_gcm_generate_tag(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter)
{
    ErrStatus ret = SUCCESS;

    uint64_t aadlength   = (uint64_t)cau_parameter->aad_size * 8U;
    uint64_t inputlength = (uint64_t)cau_parameter->in_length * 8U;
    uint32_t tagaddr     = (uint32_t)cau_dev->tag;
    if(HAL_CAU_STATE_READY == cau_dev->state) {
        cau_dev->state = HAL_CAU_STATE_BUSY;

        /* tag phase */
        /* select tag phase */
        hals_cau_phase_config(CAU_TAG_PHASE);
        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;
        /* enable the CAU peripheral */
        hals_cau_enable();

        if(RESET == (CAU_CTL & CAU_CTL_CAUEN)) {
            ret = ERROR;
        } else {
            /* nothing to do */
        }

        hals_cau_data_write(__REV((uint32_t)(aadlength >> 32U)));
        hals_cau_data_write(__REV((uint32_t)aadlength));
        hals_cau_data_write(__REV((uint32_t)(inputlength >> 32U)));
        hals_cau_data_write(__REV((uint32_t)inputlength));

        /* wait until the out fifo not empty flag is set */
        while(RESET == hals_cau_flag_get(CAU_FLAG_OUTFIFO_NO_EMPTY)) {}

        /* read the tag in the out fifo */
        *(uint32_t *)(tagaddr) = hals_cau_data_read();
        tagaddr += 4U;
        *(uint32_t *)(tagaddr) = hals_cau_data_read();
        tagaddr += 4U;
        *(uint32_t *)(tagaddr) = hals_cau_data_read();
        tagaddr += 4U;
        *(uint32_t *)(tagaddr) = hals_cau_data_read();
        tagaddr += 4U;

        /* disable the CAU peripheral */
        hals_cau_disable();

        cau_dev->state = HAL_CAU_STATE_READY;

        ret = SUCCESS;
    } else {
        /* nothing to do */
    }

    return ret;
}

/*!
    \brief      encrypt and decrypt using AES in ECB mode for interrupt
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[in]  p_user_func: pointer to user callback functions structure
                  The structure contains the following members:
      \arg        fifo_input_callback: input data transfer complete callback function
      \arg        fifo_output_callback: output data transfer complete callback function
    \param[out] output: pointer to the returned buffer
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_cau_aes_ecb_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                     hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output)
{
    ErrStatus ret = SUCCESS;
    hal_cau_key_parameter_struct key_initpara;
    __IO uint32_t counter = 0U;
    uint32_t busystatus   = 0U;

    if(cau_dev->state == HAL_CAU_STATE_READY) {
        cau_dev->in_buffer  = (uint32_t *)(cau_parameter->input);
        cau_dev->in_size    = (uint32_t)(cau_parameter->in_length);
        cau_dev->out_buffer = (uint32_t *)output;

        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* AES decryption */
        if(CAU_DECRYPT == cau_parameter->alg_dir) {
            /* flush the IN and out fifo */
            CAU_CTL |= CAU_CTL_FFLUSH;
            /* initialize the CAU peripheral */
            hals_cau_init_param(CAU_DECRYPT, CAU_MODE_AES_KEY, cau_parameter->data_change_mode);

            /* enable the CAU peripheral */
            hals_cau_enable();

            /* wait until the busy flag is RESET */
            do {
                busystatus = hals_cau_flag_get(CAU_FLAG_BUSY);
                counter++;
            } while((AESBUSY_TIMEOUT != counter) && (RESET != busystatus));

            if(RESET != busystatus) {
                ret = ERROR;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_ECB, cau_parameter->data_change_mode);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;

        cau_dev->fifo_input_callback  = NULL;
        cau_dev->fifo_output_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->fifo_input_callback) {
                cau_dev->fifo_input_callback = (void *)p_user_func->fifo_input_callback;
            } else {
                /* nothing to do */
            }

            if(NULL != p_user_func->fifo_output_callback) {
                cau_dev->fifo_output_callback = (void *)p_user_func->fifo_output_callback;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* config callback func */
        cau_dev->cau_irq.fifo_input_handle  = _cau_fifo_input_callback;
        cau_dev->cau_irq.fifo_output_handle = _cau_fifo_output_callback;

        /* enable interrupt */
        hals_cau_interrupt_enable(CAU_INT_INFIFO | CAU_INT_OUTFIFO);
        /* enable the CAU peripheral */
        hals_cau_enable();
    } else {
        /* nothing to do */
    }

    return ret;
}

/*!
    \brief      encrypt and decrypt using AES in CBC mode for interrupt
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[in]  p_user_func: pointer to user callback functions structure
                  The structure contains the following members:
      \arg        fifo_input_callback: input data transfer complete callback function
      \arg        fifo_output_callback: output data transfer complete callback function
    \param[out] output: pointer to the returned buffer
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_cau_aes_cbc_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                     hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output)
{
    ErrStatus ret = SUCCESS;
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    __IO uint32_t counter = 0U;
    uint32_t busystatus   = 0U;

    uint32_t ivaddr = (uint32_t)cau_parameter->iv;

    if(cau_dev->state == HAL_CAU_STATE_READY) {
        cau_dev->in_buffer  = (uint32_t *)(cau_parameter->input);
        cau_dev->in_size    = (uint32_t)(cau_parameter->in_length);
        cau_dev->out_buffer = (uint32_t *)output;

        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* AES decryption */
        if(CAU_DECRYPT == cau_parameter->alg_dir) {
            /* flush the IN and out fifo */
            CAU_CTL |= CAU_CTL_FFLUSH;
            /* initialize the CAU peripheral */
            hals_cau_init_param(CAU_DECRYPT, CAU_MODE_AES_KEY, cau_parameter->data_change_mode);

            /* enable the CAU peripheral */
            hals_cau_enable();

            /* wait until the busy flag is RESET */
            do {
                busystatus = hals_cau_flag_get(CAU_FLAG_BUSY);
                counter++;
            } while((AESBUSY_TIMEOUT != counter) && (RESET != busystatus));

            if(RESET != busystatus) {
                ret = ERROR;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_CBC, cau_parameter->data_change_mode);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_low = __REV(*(uint32_t *)(ivaddr));
        hals_cau_iv_init(&iv_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;

        cau_dev->fifo_input_callback  = NULL;
        cau_dev->fifo_output_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->fifo_input_callback) {
                cau_dev->fifo_input_callback = (void *)p_user_func->fifo_input_callback;
            } else {
                /* nothing to do */
            }

            if(NULL != p_user_func->fifo_output_callback) {
                cau_dev->fifo_output_callback = (void *)p_user_func->fifo_output_callback;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* config callback func */
        cau_dev->cau_irq.fifo_input_handle  = _cau_fifo_input_callback;
        cau_dev->cau_irq.fifo_output_handle = _cau_fifo_output_callback;

        /* enable interrupt */
        hals_cau_interrupt_enable(CAU_INT_INFIFO | CAU_INT_OUTFIFO);
        /* enable the CAU peripheral */
        hals_cau_enable();
    } else {
        /* nothing to do */
    }

    return ret;
}

/*!
    \brief      encrypt and decrypt using AES in CTR mode for interrupt
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[in]  p_user_func: pointer to user callback functions structure
                  The structure contains the following members:
      \arg        fifo_input_callback: input data transfer complete callback function
      \arg        fifo_output_callback: output data transfer complete callback function
    \param[out] output: pointer to the returned buffer
    \retval     void
*/
void hals_cau_aes_ctr_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output)
{
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    uint32_t ivaddr = (uint32_t)cau_parameter->iv;

    if(cau_dev->state == HAL_CAU_STATE_READY) {
        cau_dev->in_buffer  = (uint32_t *)(cau_parameter->input);
        cau_dev->in_size    = (uint32_t)(cau_parameter->in_length);
        cau_dev->out_buffer = (uint32_t *)output;

        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_CTR, cau_parameter->data_change_mode);

        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_low = __REV(*(uint32_t *)(ivaddr));
        hals_cau_iv_init(&iv_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;

        cau_dev->fifo_input_callback  = NULL;
        cau_dev->fifo_output_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->fifo_input_callback) {
                cau_dev->fifo_input_callback = (void *)p_user_func->fifo_input_callback;
            } else {
                /* nothing to do */
            }

            if(NULL != p_user_func->fifo_output_callback) {
                cau_dev->fifo_output_callback = (void *)p_user_func->fifo_output_callback;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* config callback func */
        cau_dev->cau_irq.fifo_input_handle  = _cau_fifo_input_callback;
        cau_dev->cau_irq.fifo_output_handle = _cau_fifo_output_callback;

        /* enable interrupt */
        hals_cau_interrupt_enable(CAU_INT_INFIFO | CAU_INT_OUTFIFO);
        /* enable the CAU peripheral */
        hals_cau_enable();
    }
}

/*!
    \brief      encrypt and decrypt using AES in CFB mode for interrupt
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[in]  p_user_func: pointer to user callback functions structure
                  The structure contains the following members:
      \arg        fifo_input_callback: input data transfer complete callback function
      \arg        fifo_output_callback: output data transfer complete callback function
    \param[out] output: pointer to the returned buffer
    \retval     none
*/
void hals_cau_aes_cfb_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output)
{
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    uint32_t ivaddr = (uint32_t)cau_parameter->iv;

    if(cau_dev->state == HAL_CAU_STATE_READY) {
        cau_dev->in_buffer  = (uint32_t *)(cau_parameter->input);
        cau_dev->in_size    = (uint32_t)(cau_parameter->in_length);
        cau_dev->out_buffer = (uint32_t *)output;

        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_CFB, cau_parameter->data_change_mode);

        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_low = __REV(*(uint32_t *)(ivaddr));
        hals_cau_iv_init(&iv_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;

        cau_dev->fifo_input_callback  = NULL;
        cau_dev->fifo_output_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->fifo_input_callback) {
                cau_dev->fifo_input_callback = (void *)p_user_func->fifo_input_callback;
            } else {
                /* nothing to do */
            }

            if(NULL != p_user_func->fifo_output_callback) {
                cau_dev->fifo_output_callback = (void *)p_user_func->fifo_output_callback;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* config callback func */
        cau_dev->cau_irq.fifo_input_handle  = _cau_fifo_input_callback;
        cau_dev->cau_irq.fifo_output_handle = _cau_fifo_output_callback;

        /* enable interrupt */
        hals_cau_interrupt_enable(CAU_INT_INFIFO | CAU_INT_OUTFIFO);
        /* enable the CAU peripheral */
        hals_cau_enable();
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      encrypt and decrypt using AES in OFB mode for interrupt
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[in]  p_user_func: pointer to user callback functions structure
                  The structure contains the following members:
      \arg        fifo_input_callback: input data transfer complete callback function
      \arg        fifo_output_callback: output data transfer complete callback function
    \param[out] output: pointer to the returned buffer
    \retval     none
*/
void hals_cau_aes_ofb_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output)
{
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    uint32_t ivaddr = (uint32_t)cau_parameter->iv;

    if(cau_dev->state == HAL_CAU_STATE_READY) {
        cau_dev->in_buffer  = (uint32_t *)(cau_parameter->input);
        cau_dev->in_size    = (uint32_t)(cau_parameter->in_length);
        cau_dev->out_buffer = (uint32_t *)output;

        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_OFB, cau_parameter->data_change_mode);

        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_low = __REV(*(uint32_t *)(ivaddr));
        hals_cau_iv_init(&iv_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;

        cau_dev->fifo_input_callback  = NULL;
        cau_dev->fifo_output_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->fifo_input_callback) {
                cau_dev->fifo_input_callback = (void *)p_user_func->fifo_input_callback;
            } else {
                /* nothing to do */
            }

            if(NULL != p_user_func->fifo_output_callback) {
                cau_dev->fifo_output_callback = (void *)p_user_func->fifo_output_callback;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* config callback func */
        cau_dev->cau_irq.fifo_input_handle  = _cau_fifo_input_callback;
        cau_dev->cau_irq.fifo_output_handle = _cau_fifo_output_callback;

        /* enable interrupt */
        hals_cau_interrupt_enable(CAU_INT_INFIFO | CAU_INT_OUTFIFO);
        /* enable the CAU peripheral */
        hals_cau_enable();
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      encrypt and decrypt using AES in CCM mode for interrupt
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[in]  p_user_func: pointer to user callback functions structure
                  The structure contains the following members:
      \arg        fifo_input_callback: input data transfer complete callback function
      \arg        fifo_output_callback: output data transfer complete callback function
    \param[out] output: pointer to the returned buffer
    \retval     none
*/
void hals_cau_aes_ccm_gcm_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                    hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output)
{
    cau_dev->p_irq_para = cau_parameter;
    cau_dev->state      = HAL_CAU_STATE_BUSY;
    cau_dev->out_buffer = (uint32_t *)output;
    cau_dev->in_buffer  = (uint32_t *)(cau_parameter->input);
    cau_dev->in_size    = (uint32_t)(cau_parameter->in_length);
    hals_cau_enable();

    cau_dev->fifo_input_callback  = NULL;
    cau_dev->fifo_output_callback = NULL;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->fifo_input_callback) {
            cau_dev->fifo_input_callback = (void *)p_user_func->fifo_input_callback;
        } else {
            /* nothing to do */
        }

        if(NULL != p_user_func->fifo_output_callback) {
            cau_dev->fifo_output_callback = (void *)p_user_func->fifo_output_callback;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* config callback func */
    cau_dev->cau_irq.fifo_input_handle  = _cau_fifo_input_callback;
    cau_dev->cau_irq.fifo_output_handle = _cau_fifo_output_callback;

    /* enable interrupt */
    hals_cau_interrupt_enable(CAU_INT_INFIFO | CAU_INT_OUTFIFO);
}

/*!
    \brief      encrypt and decrypt using AES in ECB mode for DMA
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[in]  dmacb: CAU DMA callback function pointer structure
                only one parameter can be selected which is shown as below:
      \arg        in_full_transcom_handle: CAU DMA input full transfer complete interrupt handler
      \arg        in_half_transcom_handle: CAU DMA input half transfer complete interrupt handler
      \arg        out_full_transcom_handle: CAU DMA output full transfer complete interrupt handler
      \arg        out_half_transcom_handle: CAU DMA output half transfer complete interrupt handler
      \arg        error_handle:CAU DMA error interrupt handler
    \param[out] output: pointer to the returned buffer
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_cau_aes_ecb_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                               hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output)
{
    ErrStatus ret = SUCCESS;
    hal_cau_key_parameter_struct key_initpara;
    __IO uint32_t counter = 0U;
    uint32_t busystatus   = 0U;

    if(cau_dev->state == HAL_CAU_STATE_READY) {
        cau_dev->in_buffer  = (uint32_t *)(cau_parameter->input);
        cau_dev->in_size    = (uint32_t)(cau_parameter->in_length);
        cau_dev->out_buffer = (uint32_t *)output;

        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* AES decryption */
        if(CAU_DECRYPT == cau_parameter->alg_dir) {
            CAU_CTL |= CAU_CTL_FFLUSH;
            /* initialize the CAU peripheral */
            hals_cau_init_param(CAU_DECRYPT, CAU_MODE_AES_KEY, cau_parameter->data_change_mode);

            /* enable the CAU peripheral */
            hals_cau_enable();

            /* wait until the busy flag is RESET */
            do {
                busystatus = hals_cau_flag_get(CAU_FLAG_BUSY);
                counter++;
            } while((AESBUSY_TIMEOUT != counter) && (RESET != busystatus));

            if(RESET != busystatus) {
                ret = ERROR;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_ECB, cau_parameter->data_change_mode);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;
        /* dma transfer config */
        hals_cau_dma_transfer_config(cau_dev, dmacb);
    } else {
        /* nothing to do */
    }

    return ret;
}

/*!
    \brief      encrypt and decrypt using AES in CBC mode for DMA
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[in]  dmacb: CAU DMA callback function pointer structure
                only one parameter can be selected which is shown as below:
      \arg        in_full_transcom_handle: CAU DMA input full transfer complete interrupt handler
      \arg        in_half_transcom_handle: CAU DMA input half transfer complete interrupt handler
      \arg        out_full_transcom_handle: CAU DMA output full transfer complete interrupt handler
      \arg        out_half_transcom_handle: CAU DMA output half transfer complete interrupt handler
      \arg        error_handle:CAU DMA error interrupt handler
    \param[out] output: pointer to the returned buffer
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_cau_aes_cbc_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                               hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output)
{
    ErrStatus ret = SUCCESS;
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    __IO uint32_t counter = 0U;
    uint32_t busystatus   = 0U;

    uint32_t ivaddr = (uint32_t)cau_parameter->iv;

    if(cau_dev->state == HAL_CAU_STATE_READY) {
        cau_dev->in_buffer  = (uint32_t *)(cau_parameter->input);
        cau_dev->in_size    = (uint32_t)(cau_parameter->in_length);
        cau_dev->out_buffer = (uint32_t *)output;

        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* AES decryption */
        if(CAU_DECRYPT == cau_parameter->alg_dir) {
            /* flush the IN and out fifo */
            CAU_CTL |= CAU_CTL_FFLUSH;
            /* initialize the CAU peripheral */
            hals_cau_init_param(CAU_DECRYPT, CAU_MODE_AES_KEY, cau_parameter->data_change_mode);

            /* enable the CAU peripheral */
            hals_cau_enable();

            /* wait until the busy flag is RESET */
            do {
                busystatus = hals_cau_flag_get(CAU_FLAG_BUSY);
                counter++;
            } while((AESBUSY_TIMEOUT != counter) && (RESET != busystatus));

            if(RESET != busystatus) {
                ret = ERROR;
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_CBC, cau_parameter->data_change_mode);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_low = __REV(*(uint32_t *)(ivaddr));
        hals_cau_iv_init(&iv_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;
        /* dma transfer config */
        hals_cau_dma_transfer_config(cau_dev, dmacb);
    }

    return ret;
}

/*!
    \brief      encrypt and decrypt using AES in CTR mode for DMA
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[in]  dmacb: CAU DMA callback function pointer structure
                only one parameter can be selected which is shown as below:
      \arg        in_full_transcom_handle: CAU DMA input full transfer complete interrupt handler
      \arg        in_half_transcom_handle: CAU DMA input half transfer complete interrupt handler
      \arg        out_full_transcom_handle: CAU DMA output full transfer complete interrupt handler
      \arg        out_half_transcom_handle: CAU DMA output half transfer complete interrupt handler
      \arg        error_handle:CAU DMA error interrupt handler
    \param[out] output: pointer to the returned buffer
    \retval     void
*/
void hals_cau_aes_ctr_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                          hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output)
{
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    uint32_t ivaddr = (uint32_t)cau_parameter->iv;

    if(cau_dev->state == HAL_CAU_STATE_READY) {
        cau_dev->in_buffer  = (uint32_t *)(cau_parameter->input);
        cau_dev->in_size    = (uint32_t)(cau_parameter->in_length);
        cau_dev->out_buffer = (uint32_t *)output;

        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_CTR, cau_parameter->data_change_mode);

        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_low = __REV(*(uint32_t *)(ivaddr));
        hals_cau_iv_init(&iv_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;
        /* dma transfer config */
        hals_cau_dma_transfer_config(cau_dev, dmacb);
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      encrypt and decrypt using AES in CFB mode for DMA
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[in]  dmacb: CAU DMA callback function pointer structure
                only one parameter can be selected which is shown as below:
      \arg        in_full_transcom_handle: CAU DMA input full transfer complete interrupt handler
      \arg        in_half_transcom_handle: CAU DMA input half transfer complete interrupt handler
      \arg        out_full_transcom_handle: CAU DMA output full transfer complete interrupt handler
      \arg        out_half_transcom_handle: CAU DMA output half transfer complete interrupt handler
      \arg        error_handle:CAU DMA error interrupt handler
    \param[out] output: pointer to the returned buffer
    \retval     none
*/
void hals_cau_aes_cfb_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                          hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output)
{
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    uint32_t ivaddr = (uint32_t)cau_parameter->iv;

    if(cau_dev->state == HAL_CAU_STATE_READY) {
        cau_dev->in_buffer  = (uint32_t *)(cau_parameter->input);
        cau_dev->in_size    = (uint32_t)(cau_parameter->in_length);
        cau_dev->out_buffer = (uint32_t *)output;

        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_CFB, cau_parameter->data_change_mode);

        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_low = __REV(*(uint32_t *)(ivaddr));
        hals_cau_iv_init(&iv_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;
        /* dma transfer config */
        hals_cau_dma_transfer_config(cau_dev, dmacb);
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      encrypt and decrypt using AES in OFB mode for DMA
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector, 16 bytes
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg           CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg           CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg           CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg           CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[in]  dmacb: CAU DMA callback function pointer structure
                only one parameter can be selected which is shown as below:
      \arg        in_full_transcom_handle: CAU DMA input full transfer complete interrupt handler
      \arg        in_half_transcom_handle: CAU DMA input half transfer complete interrupt handler
      \arg        out_full_transcom_handle: CAU DMA output full transfer complete interrupt handler
      \arg        out_half_transcom_handle: CAU DMA output half transfer complete interrupt handler
      \arg        error_handle:CAU DMA error interrupt handler
    \param[out] output: pointer to the returned buffer
    \retval     none
*/
void hals_cau_aes_ofb_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                          hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output)
{
    hal_cau_key_parameter_struct key_initpara;
    hal_cau_iv_parameter_struct iv_initpara;
    uint32_t ivaddr = (uint32_t)cau_parameter->iv;

    if(cau_dev->state == HAL_CAU_STATE_READY) {
        cau_dev->in_buffer  = (uint32_t *)(cau_parameter->input);
        cau_dev->in_size    = (uint32_t)(cau_parameter->in_length);
        cau_dev->out_buffer = (uint32_t *)output;

        cau_dev->state = HAL_CAU_STATE_BUSY;
        /* key structure initialization */
        hals_cau_key_struct_para_init(&key_initpara);
        /* initialize the CAU peripheral */
        hals_cau_init_param(cau_parameter->alg_dir, CAU_MODE_AES_OFB, cau_parameter->data_change_mode);

        /* AES key structure parameter config */
        _cau_aes_key_config(cau_parameter->key, cau_parameter->key_size, &key_initpara);
        /* key initialization */
        hals_cau_key_init(&key_initpara);

        /* vectors initialization */
        iv_initpara.iv_0_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_0_low = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_high = __REV(*(uint32_t *)(ivaddr));
        ivaddr += 4U;
        iv_initpara.iv_1_low = __REV(*(uint32_t *)(ivaddr));
        hals_cau_iv_init(&iv_initpara);

        /* flush the IN and out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;
        /* dma transfer config */
        hals_cau_dma_transfer_config(cau_dev, dmacb);
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      CAU fifo input callback
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _cau_fifo_input_callback(void *cau_dev)
{
    hal_cau_dev_struct *p_cau_dev = (hal_cau_dev_struct *)cau_dev;

    if(HAL_CAU_STATE_BUSY == p_cau_dev->state) {
        if(SET == hals_cau_flag_get(CAU_FLAG_INFIFO)) {
            if(NULL != p_cau_dev->fifo_input_callback) {
                ((hal_irq_handle_cb)(p_cau_dev->fifo_input_callback))(p_cau_dev);
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      CAU fifo output callback
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _cau_fifo_output_callback(void *cau_dev)
{
    hal_cau_dev_struct *p_cau_dev = (hal_cau_dev_struct *)cau_dev;

    if(HAL_CAU_STATE_BUSY == p_cau_dev->state) {
        if(SET == hals_cau_flag_get(CAU_FLAG_OUTFIFO)) {
            if(NULL != p_cau_dev->fifo_output_callback) {
                ((hal_irq_handle_cb)p_cau_dev->fifo_output_callback)(p_cau_dev);
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      AES key structure parameter config
    \param[in]  key: key used for AES algorithm
    \param[in]  keysize: length of the key in bits, must be either 128, 192 or 256
    \param[out] cau_key_initpara: key init parameter struct
                only one parameters can be selected which are shown as below:
                  key_0_high: key 0 high
                  key_0_low:  key 0 low
                  key_1_high: key 1 high
                  key_1_low:  key 1 low
                  key_2_high: key 2 high
                  key_2_low:  key 2 low
                  key_3_high: key 3 high
                  key_3_low:  key 3 low
    \retval     none
*/
static void _cau_aes_key_config(uint8_t *key, uint32_t keysize, hal_cau_key_parameter_struct *cau_key_initpara)
{
    uint32_t keyaddr = (uint32_t)key;

    /* clear key size register */
    CAU_CTL &= ~CAU_CTL_KEYM;

    switch(keysize) {
    /* 128-bit key initialization */
    case 128:
        CAU_CTL |= CAU_KEYSIZE_128BIT;
        cau_key_initpara->key_2_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_2_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_3_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_3_low = __REV(*(uint32_t *)(keyaddr));
        break;
    /* 192-bit key initialization */
    case 192:
        CAU_CTL |= CAU_KEYSIZE_192BIT;
        cau_key_initpara->key_1_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_1_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_2_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_2_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_3_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_3_low = __REV(*(uint32_t *)(keyaddr));
        break;
    /* 256-bit key initialization */
    case 256:
        CAU_CTL |= CAU_KEYSIZE_256BIT;
        cau_key_initpara->key_0_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_0_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_1_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_1_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_2_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_2_low = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_3_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_3_low = __REV(*(uint32_t *)(keyaddr));
        break;
    default:
        break;
    }
}

/*!
    \brief      fill data into data input register
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer in bytes, must be a multiple of 16 bytes
    \retval     ErrStatus: SUCCESS or ERROR
*/
static ErrStatus _cau_aes_fill_data(uint8_t *input, uint32_t in_length)
{
    uint32_t inputaddr    = (uint32_t)input;
    uint32_t i            = 0U;
    __IO uint32_t counter = 0U;
    uint32_t busystatus   = 0U;
    ErrStatus status = SUCCESS;

    if(RESET == (CAU_CTL & CAU_CTL_CAUEN)) {
        status = ERROR;
    } else {
        /* nothing to do */
    }

    for(i = 0U; i < in_length; i += BLOCK_DATA_SIZE) {
        /* wait until the in fifo empty flag is set */
        while(RESET == hals_cau_flag_get(CAU_FLAG_INFIFO_EMPTY)) {}

        if(i + BLOCK_DATA_SIZE > in_length) {
            /* the last block data number is less than 128bit */
            uint32_t block_data_temp[4] = {0U};

            /* fill the remaining bits with zero */
            memcpy(block_data_temp, (uint32_t *)inputaddr, in_length - i);
            inputaddr = (uint32_t)block_data_temp;

            /* if GCM encryption or CCM decryption, then configurate NBPILB bits in CTL register */
            if((CAU_CTL & CAU_CTL_GCM_CCMPH) == CAU_ENCRYPT_DECRYPT_PHASE) {
                if((CAU_CTL & (CAU_CTL_ALGM | CAU_CTL_CAUDIR)) == (CAU_MODE_AES_GCM | CAU_ENCRYPT)) {
                    CAU_CTL |= CAU_PADDING_BYTES(i + BLOCK_DATA_SIZE - in_length);
                } else if((CAU_CTL & (CAU_CTL_ALGM | CAU_CTL_CAUDIR)) == (CAU_MODE_AES_CCM | CAU_DECRYPT)) {
                    CAU_CTL |= CAU_PADDING_BYTES(i + BLOCK_DATA_SIZE - in_length);
                } else {
                    /* nothing to do */
                }
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* write data to the in fifo */
        hals_cau_data_write(*(uint32_t *)(inputaddr));
        inputaddr += 4U;
        hals_cau_data_write(*(uint32_t *)(inputaddr));
        inputaddr += 4U;
        hals_cau_data_write(*(uint32_t *)(inputaddr));
        inputaddr += 4U;
        hals_cau_data_write(*(uint32_t *)(inputaddr));
        inputaddr += 4U;
    }
    /* wait until the complete message has been processed */
    counter = 0U;
    do {
        busystatus = hals_cau_flag_get(CAU_FLAG_BUSY);
        counter++;
    } while((AESBUSY_TIMEOUT != counter) && (RESET != busystatus));

    if(RESET != busystatus) {
        status = ERROR;
    } else {
        /* nothing to do */
    }

    return status;
}

/*!
    \brief      AES calculate process
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer in bytes, must be a multiple of 16 bytes
    \param[out]  output: pointer to the returned buffer
    \retval     ErrStatus: SUCCESS or ERROR
*/
static ErrStatus _cau_aes_calculate(uint8_t *input, uint32_t in_length, uint8_t *output)
{
    uint32_t inputaddr    = (uint32_t)input;
    uint32_t outputaddr   = (uint32_t)output;
    uint32_t i            = 0U;
    __IO uint32_t counter = 0U;
    uint32_t busystatus   = 0U;
    ErrStatus status = SUCCESS;

    /* the clock is not enabled or there is no embedded CAU peripheral */
    if(RESET == (CAU_CTL & CAU_CTL_CAUEN)) {
        status = ERROR;
    } else {
        /* nothing to do */
    }

    for(i = 0U; i < in_length; i += BLOCK_DATA_SIZE) {
        /* wait until the in fifo empty flag is set */
        while(RESET == hals_cau_flag_get(CAU_FLAG_INFIFO_EMPTY)) {}

        /* check if the last input data block */
        if(i + BLOCK_DATA_SIZE > in_length) {
            /* the last block data number is less than 128bit */
            uint32_t block_data_temp[4] = {0};

            /* fill the remaining bits with zero */
            memcpy(block_data_temp, (uint32_t *)inputaddr, in_length - i);
            inputaddr = (uint32_t)block_data_temp;

            /* if GCM encryption or CCM decryption, then configurate NBPILB bits in CTL register */
            if((CAU_CTL & (CAU_CTL_ALGM | CAU_CTL_CAUDIR)) == (CAU_MODE_AES_GCM | CAU_ENCRYPT)) {
                CAU_CTL |= CAU_PADDING_BYTES(i + BLOCK_DATA_SIZE - in_length);
            } else if((CAU_CTL & (CAU_CTL_ALGM | CAU_CTL_CAUDIR)) == (CAU_MODE_AES_CCM | CAU_DECRYPT)) {
                CAU_CTL |= CAU_PADDING_BYTES(i + BLOCK_DATA_SIZE - in_length);
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* write data to the in fifo */
        hals_cau_data_write(*(uint32_t *)(inputaddr));
        inputaddr += 4U;
        hals_cau_data_write(*(uint32_t *)(inputaddr));
        inputaddr += 4U;
        hals_cau_data_write(*(uint32_t *)(inputaddr));
        inputaddr += 4U;
        hals_cau_data_write(*(uint32_t *)(inputaddr));
        inputaddr += 4U;

        /* wait until the complete message has been processed */
        counter = 0U;
        do {
            busystatus = hals_cau_flag_get(CAU_FLAG_BUSY);
            counter++;
        } while((AESBUSY_TIMEOUT != counter) && (RESET != busystatus));

        if(RESET != busystatus) {
            status = ERROR;
        } else {
            /* read the output block from the output FIFO */
            *(uint32_t *)(outputaddr) = hals_cau_data_read();
            outputaddr += 4U;
            *(uint32_t *)(outputaddr) = hals_cau_data_read();
            outputaddr += 4U;
            *(uint32_t *)(outputaddr) = hals_cau_data_read();
            outputaddr += 4U;
            *(uint32_t *)(outputaddr) = hals_cau_data_read();
            outputaddr += 4U;
        }
    }

    return status;
}
