/*!
    \file    gd32h7xx_hal_can.c
    \brief   CAN driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define CAN_MAILBOX_MAX_NUM     (31U)
#define CAN_FIFO_NUM            (8U)

/* DLC to data size in bytes definitions */
static const uint8_t dlc_to_databytes[16] = {0U, 1U, 2U, 3U, 4U, 5U, 6U, 7U, 8U, 12U, 16U, 20U, 24U, 32U, 48U, 64U};
/* enter the corresponding mode */
static int32_t _can_operation_mode_enter(uint32_t can_periph, hal_can_operation_mode_enum mode);
/* deinit CAN device */
static void _can_deinit(uint32_t can_periph);
/* reset CAN internal state machines and CAN registers */
static ErrStatus _can_software_reset(uint32_t can_periph);
/* get mailbox RAM address */
static uint32_t *_can_ram_address_get(uint32_t can_periph, uint32_t index);
/* computes the maximum payload size (in bytes), given a dlc */
static uint32_t _can_payload_size_compute(uint32_t dlc_value);
/* swap data to little endian */
static void _can_data_to_little_endian_swap(uint32_t *dest, uint32_t *src, uint32_t len);
/* swap data to big endian */
static void _can_data_to_big_endian_swap(uint32_t *dest, uint32_t *src, uint32_t len);
/* computes the dlc field value, given a payload size (in bytes) */
static uint32_t _can_dlc_value_compute(uint32_t payload_size);
/* handle the rxfifo receive interrupt */
static void _can_rxfifo_receive_interrupt(void *can);
/* handle the mailbox receive interrupt */
static void _can_mailbox_receive_interrupt(void *can);
/* handle the rxfifo receive interrupt */
static void _can_pn_mode_data_receive_interrupt(void *can);
/* handle the CAN DMA receive process complete */
static void _can_dmarx_complete(void *dma);
/* handle the CAN DMA error process */
static void _can_dma_error(void *dma);

/*!
    \brief      initialize CAN
    \param[in]  can_dev:CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  periph: CANx(x=0,1,2)
    \param[in]  p_init: can parameter struct
                  frame_format:
                  only one parameter can be selected which is shown as below:
       \arg         CAN_CLASSIC_MODE
       \arg         CAN_FD_MODE
                  operation_mode: the argument could be selected from enumeration<hal_can_operation_mode_enum>
                  self_reception:
                  only one parameter can be selected which is shown as below:
       \arg         CAN_SELF_RECEPTION_DISABLE
       \arg         CAN_SELF_RECEPTION_ENABLE
                  internal_counter_source:
                  only one parameter can be selected which is shown as below:
       \arg         CAN_TIMER_SOURCE_BIT_CLOCK
       \arg         CAN_TIMER_SOURCE_EXTERNAL_TIME_TICK
                  mb_tx_order:
                  only one parameter can be selected which is shown as below:
       \arg         CAN_TX_HIGH_PRIORITY_MB_FIRST
       \arg         CAN_TX_LOW_NUM_MB_FIRST
                  mb_tx_abort_enable:
                  only one parameter can be selected which is shown as below:
       \arg         CAN_TRANSMIT_ABORT_ENABLE
       \arg         CAN_TRANSMIT_ABORT_DISABLE
                  local_priority_enable:
                  only one parameter can be selected which is shown as below:
       \arg         CAN_LOCAL_PRIORITY_ENABLE
       \arg         CAN_LOCAL_PRIORITY_DISABLE
                  mb_rx_ide_rtr_type:
                  only one parameter can be selected which is shown as below:
       \arg         CAN_IDE_RTR_COMPARED
       \arg         CAN_IDE_RTR_FILTERED
                  mb_remote_frame:
                  only one parameter can be selected which is shown as below:
       \arg         CAN_GEN_REMOTE_RESPONSE_FRAME
       \arg         CAN_STORE_REMOTE_REQUEST_FRAME
                  rx_private_filter_queue_enable:
                  only one parameter can be selected which is shown as below:
       \arg         CAN_PRIVATE_FILTER_QUEUE_ENABLE
       \arg         CAN_PRIVATE_FILTER_QUEUE_DISABLE
                  edge_filter_enable:
                  only one parameter can be selected which is shown as below:
       \arg         CAN_EDGE_FILTER_ENABLE
       \arg         CAN_EDGE_FILTER_DISABLE
                  rx_fifo_enable:
                  only one parameter can be selected which is shown as below:
       \arg         CAN_RX_FIFO_ENABLE
       \arg         CAN_RX_FIFO_DISABLE
                  rx_filter_order:
                  only one parameter can be selected which is shown as below:
       \arg         CAN_RX_FILTER_ORDER_FIFO_FIRST
       \arg         CAN_RX_FILTER_ORDER_MAILBOX_FIRST
                  memory_size:
                  only one parameter can be selected which is shown as below:
       \arg         CAN_MEMSIZE_1_UNIT, CAN_MEMSIZE_2_UNIT, CAN_MEMSIZE_3_UNIT, CAN_MEMSIZE_4_UNIT,
       \arg         CAN_MEMSIZE_5_UNIT, CAN_MEMSIZE_6_UNIT, CAN_MEMSIZE_7_UNIT, CAN_MEMSIZE_8_UNIT,
       \arg         CAN_MEMSIZE_9_UNIT, CAN_MEMSIZE_10_UNIT, CAN_MEMSIZE_11_UNIT, CAN_MEMSIZE_12_UNIT,
       \arg         CAN_MEMSIZE_13_UNIT, CAN_MEMSIZE_14_UNIT, CAN_MEMSIZE_15_UNIT, CAN_MEMSIZE_16_UNIT,
       \arg         CAN_MEMSIZE_17_UNIT, CAN_MEMSIZE_18_UNIT, CAN_MEMSIZE_19_UNIT, CAN_MEMSIZE_20_UNIT,
       \arg         CAN_MEMSIZE_21_UNIT, CAN_MEMSIZE_22_UNIT, CAN_MEMSIZE_23_UNIT, CAN_MEMSIZE_24_UNIT,
       \arg         CAN_MEMSIZE_25_UNIT, CAN_MEMSIZE_26_UNIT, CAN_MEMSIZE_27_UNIT, CAN_MEMSIZE_28_UNIT,
       \arg         CAN_MEMSIZE_29_UNIT, CAN_MEMSIZE_30_UNIT, CAN_MEMSIZE_31_UNIT, CAN_MEMSIZE_32_UNIT
                  mb_public_filter: 0x00000000 ~ 0xFFFFFFFF
                  prescaler: 1~1024
                  resync_jump_width: 1~32
                  prop_time_segment: 1~64
                  time_segment_1: 1~32
                  time_segment_2: 1~32
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_HARDWARE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_init(hal_can_dev_struct *can_dev, uint32_t periph, hal_can_init_struct *p_init)
{
      int32_t error_code = HAL_ERR_NONE;
    uint32_t i         = 0U;
    uint32_t reg_temp  = 0U;
    uint32_t *canram   = (uint32_t *)(CAN_RAM(periph));
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == can_dev) || (NULL == p_init)) {
        HAL_DEBUGE("pointer [can_dev] or [p_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    /* check periph parameter */
    if((CAN0 != periph) && (CAN1 != periph) && (CAN2 != periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock can_dev */
    HAL_LOCK(can_dev);

    /* deinit can */
    _can_deinit(periph);

    /* clear CAN RAM */
    for(i = 0U; i < CAN_MAX_RAM_SIZE; i++) {
        canram[i] = 0U;
    }

    /* reset CAN_RFIFOMPFx */
    for(i = 0U; i < CAN_MAX_MAILBOX_NUM; i++) {
        CAN_RFIFOMPF(periph, i) = 0x00000000U;
    }

    /* reset internal state machines and CAN registers */
    if(ERROR == _can_software_reset(periph)) {
        error_code = HAL_ERR_HARDWARE;
    } else {
        can_dev->periph = periph;
        can_dev->init   = *p_init;

        /* reset CAN_INTEN */
        CAN_INTEN(periph) = 0U;
        /* reset CAN_STAT */
        CAN_STAT(periph) = (uint32_t)0xFFFFFFFFU;
        CAN_TIMER(periph);
        while(CAN_STAT(periph) & CAN_STAT_MS5_RFNE) {
            CAN_STAT(periph) = CAN_STAT_MS5_RFNE;
        }

        /* clear register bits */
        CAN_BT(periph) &= ~(CAN_BT_BAUDPSC | CAN_BT_SJW | CAN_BT_PTS | CAN_BT_PBS1 | CAN_BT_PBS2);

        /* configure CTL0 register */
        reg_temp = CAN_CTL0(periph);
        reg_temp &= ~(CAN_CTL0_RFEN | CAN_CTL0_FDEN | CAN_CTL0_SRDIS | CAN_CTL0_LAPRIOEN | CAN_CTL0_MST | CAN_CTL0_RPFQEN | \
                    CAN_CTL0_FS | CAN_CTL0_MSZ);
        reg_temp |= (p_init->frame_format | p_init->self_reception | p_init->local_priority_enable | \
                    p_init->rx_private_filter_queue_enable | p_init->mb_tx_abort_enable | p_init->rx_fifo_enable | \
                    (p_init->memory_size - 1U) | (p_init->filter_format_and_number & CAN_CTL0_FS) | CAN_CTL0_WERREN);
        CAN_CTL0(periph) = reg_temp;

        /* set mailbox arbitration process order */
        reg_temp = CAN_CTL1(periph);
        reg_temp &= ~(CAN_CTL1_MTO | CAN_CTL1_BSPMOD | CAN_CTL1_ABORDIS | CAN_CTL1_TSYNC);
        reg_temp |= (p_init->mb_tx_order | p_init->bit_sampling_mode | p_init->auto_busoff_recovery | \
                    p_init->time_synchronization);
        CAN_CTL1(periph) = reg_temp;

        /* configure CTL2 register */
        reg_temp = CAN_CTL2(periph);
        reg_temp &= ~(CAN_CTL2_ITSRC | CAN_CTL2_IDERTR_RMF | CAN_CTL2_RRFRMS | CAN_CTL2_ASD | CAN_CTL2_RFO | CAN_CTL2_RFFN | \
                    CAN_CTL2_EFDIS | CAN_CTL2_PREEN);
        reg_temp |= (p_init->edge_filter_enable | p_init->protocol_exception_enable | p_init->internal_counter_source | \
                    p_init->mb_rx_ide_rtr_type | p_init->mb_remote_frame | p_init->rx_filter_order | \
                    (p_init->filter_format_and_number & CAN_CTL2_RFFN)) | \
                    (p_init->arbitration_start_delay << 19U);
        CAN_CTL2(periph) = reg_temp;

        /* set mailbox public filter */
        CAN_RMPUBF(periph) = p_init->mb_public_filter;

        /* configure fifo public filter */
        CAN_RFIFOPUBF(periph) = p_init->fifo_public_filter;
        /* configure fifo private filter */
        if(0U == (CAN_CTL0(periph) & CAN_CTL0_RPFQEN)) {
            for(i = 0U; i < CAN_MAX_MAILBOX_NUM; i++) {
                CAN_RFIFOMPF(periph, i) = p_init->fifo_public_filter;
            }
        } else {
            /* do nothing */
        }

        /* set time segment */
        CAN_BT(periph) |= (uint32_t)(BT_BAUDPSC(p_init->prescaler - 1U) | \
                                    BT_SJW((uint32_t)p_init->resync_jump_width - 1U) | \
                                    BT_PTS((uint32_t)p_init->prop_time_segment - 1U) | \
                                    BT_PBS1((uint32_t)p_init->time_segment_1 - 1U) | \
                                    BT_PBS2((uint32_t)p_init->time_segment_2 - 1U));

    }

    can_dev->state = HAL_CAN_STATE_READY;

    /* unlock can */
    HAL_UNLOCK(can_dev);

    return error_code;
}

/*!
    \brief      configure FD mode parameter
    \param[in]  can_dev:CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  fd_config: can fd parameter struct
                  iso_can_fd_enable: ENABLE, DISABLE
                  bitrate_switch_enable: ENABLE, DISABLE
                  mailbox_data_size: CAN_MAILBOX_DATA_SIZE_8_BYTES, CAN_MAILBOX_DATA_SIZE_16_BYTES,
                                     CAN_MAILBOX_DATA_SIZE_32_BYTES, CAN_MAILBOX_DATA_SIZE_64_BYTES
                  tdc_enable: ENABLE, DISABLE
                  tdc_offset: 0 ~ 31
                  prescaler: 1~1024
                  resync_jump_width: 1~8
                  prop_time_segment: 0~31
                  time_segment_1: 1~8
                  time_segment_2: 2~8
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_can_fd_config(hal_can_dev_struct *can_dev, hal_can_fd_config_struct *fd_config)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and fd_config address */
    if((NULL == can_dev) || (NULL == fd_config)) {
        HAL_DEBUGE("pointer [can_dev] or [fd_config] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear register bits, then enable FD mode */
    CAN_CTL0(can_dev->periph) &= ~CAN_CTL0_RFEN;
    CAN_CTL1(can_dev->periph) &= ~CAN_CTL1_BSPMOD;
    CAN_CTL2(can_dev->periph) &= ~CAN_CTL2_ISO;
    CAN_FDCTL(can_dev->periph) &= ~(CAN_FDCTL_BRSEN | CAN_FDCTL_MDSZ | CAN_FDCTL_TDCEN | CAN_FDCTL_TDCO);
    CAN_FDBT(can_dev->periph) &= ~(CAN_FDBT_DBAUDPSC | CAN_FDBT_DSJW | CAN_FDBT_DPTS | CAN_FDBT_DPBS1 | CAN_FDBT_DPBS2);
    CAN_CTL0(can_dev->periph) |= CAN_CTL0_FDEN;

    /* support ISO or non-ISO mode */
    CAN_CTL2(can_dev->periph) |= fd_config->iso_can_fd_enable;

    /* set TDC parameter */
    CAN_FDCTL(can_dev->periph) |= (fd_config->tdc_enable | fd_config->bitrate_switch_enable | \
                                   fd_config->mailbox_data_size | FDCTL_TDCO(fd_config->tdc_offset));

    /* configure FD bit timing */
    CAN_FDBT(can_dev->periph) |= (uint32_t)(FDBT_DBAUDPSC(fd_config->prescaler - 1U) | \
                                           FDBT_DSJW((uint32_t)fd_config->resync_jump_width - 1U) | \
                                           FDBT_DPTS(fd_config->prop_time_segment) | \
                                           FDBT_DPBS1((uint32_t)fd_config->time_segment_1 - 1U) | \
                                           FDBT_DPBS2((uint32_t)fd_config->time_segment_2 - 1U));

    return error_code;
}

/*!
    \brief      configure Pretended Networking mode parameter
    \param[in]  can_dev:CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  pnmode_config: Pretended Networking mode config struct
                  num_matches: 1 ~ 255
                  match_timeout: 0 ~ 0xFFFF
                  frame_filter: CAN_PN_FRAME_FILTERING_ID, CAN_PN_FRAME_FILTERING_ID_DATA
                                CAN_PN_FRAME_FILTERING_ID_NMM, CAN_PN_FRAME_FILTERING_ID_DATA_NMM
                  id_filter: CAN_PN_ID_FILTERING_EXACT, CAN_PN_ID_FILTERING_GREATER
                             CAN_PN_ID_FILTERING_SMALLER, CAN_PN_ID_FILTERING_RANGE
                  data_filter: CAN_PN_DATA_FILTERING_EXACT, CAN_PN_DATA_FILTERING_GREATER
                               CAN_PN_DATA_FILTERING_SMALLER, CAN_PN_DATA_FILTERING_RANGE
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_can_pn_mode_config(hal_can_dev_struct *can_dev, hal_can_pn_mode_config_struct *pnmode_config)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t reg_temp  = 0U;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and pnmode_config address */
    if((NULL == can_dev) || (NULL == pnmode_config)) {
        HAL_DEBUGE("pointer [can_dev] or [pnmode_config] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure specific Pretended Networking mode settings */
    reg_temp = CAN_PN_CTL0(can_dev->periph);
    reg_temp &= ~(CAN_PN_CTL0_FFT | CAN_PN_CTL0_IDFT | CAN_PN_CTL0_DATAFT | CAN_PN_CTL0_NMM);
    reg_temp |= (PN_CTL0_NMM(pnmode_config->num_matches) | pnmode_config->data_filter | pnmode_config->id_filter | \
                 pnmode_config->frame_filter);

    CAN_PN_CTL0(can_dev->periph) = reg_temp;

    /* set timeout value */
    if(0U != pnmode_config->match_timeout) {
        CAN_PN_TO(can_dev->periph) &= ~CAN_PN_TO_WTO;
        CAN_PN_TO(can_dev->periph) |= PN_TO_WTO(pnmode_config->match_timeout);
    } else {
        /* do nothing */
    }

    /* enable Pretended Networking mode */
    CAN_CTL0(can_dev->periph) |= CAN_CTL0_PNMOD;

    /* EXTI configuration */
    if(CAN0 == can_dev->periph) {
        hal_exti_internal_init(EXTI_LINE_24_CAN0_WAKEUP, EXTI_INTERRUPT_TRIG_RISING);
    } else if(CAN1 == can_dev->periph) {
        hal_exti_internal_init(EXTI_LINE_25_CAN1_WAKEUP, EXTI_INTERRUPT_TRIG_RISING);
    } else {
        hal_exti_internal_init(EXTI_LINE_26_CAN2_WAKEUP, EXTI_INTERRUPT_TRIG_RISING);
    }

    return error_code;
}

/*!
    \brief      deinitialize CAN
    \param[in]  can_dev: CAN device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_can_deinit(hal_can_dev_struct *can_dev)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t periph = 0U;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    periph = can_dev->periph;
    if((CAN0 == periph) || (CAN1 == periph) || (CAN2 == periph)) {
        /* deinitialize the periph and the device information structure */
        _can_deinit(periph);
        can_dev->state       = HAL_CAN_STATE_RESET;
        can_dev->error_state = HAL_CAN_ERROR_NONE;
    } else {
        HAL_DEBUGE("parameter [can_dev->periph] value is invalid");
        error_code = HAL_ERR_HARDWARE;
    }

    return error_code;
}

/*!
    \brief      initialize CAN parameter structure with a default value
    \param[in]  hal_struct_type: the type of CAN parameter struct
                only one parameter can be selected which is shown as below:
      \arg        HAL_CAN_INIT_STRUCT: the CAN initial struct
      \arg        HAL_CAN_FD_CONFIG_STRUCT: the CAN FD initial struct
      \arg        HAL_CAN_PN_MODE_CONFIG_STRUCT: the CAN Pretended Networking mode initial struct
      \arg        HAL_CAN_PN_MODE_FILTER_STRUCT: the CAN Pretended Networking mode filter struct
      \arg        HAL_CAN_MDSC_STRUCT: mailbox descriptor struct
      \arg        HAL_CAN_FDES_STRUCT: Rx fifo descriptor struct
      \arg        HAL_CAN_FIFO_ID_FILTER_STRUCT: Rx fifo id filter struct
      \arg        HAL_CAN_CRC_STRUCT: CRC struct
      \arg        HAL_CAN_ERRCNT_STRUCT: error counter struct
      \arg        HAL_CAN_BUFFER_STRUCT: CAN buffer structure
      \arg        HAL_CAN_USER_CALLBACK_STRUCT: CAN user interrupt callback function pointer structure
      \arg        HAL_CAN_DEV_STRUCT:the CAN device information structure
      \arg        HAL_CAN_IRQ_INIT_STRUCT:the CAN interrupt callback function pointer structure
    \param[out] p_struct: pointer to CAN structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_struct_init(hal_can_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_CAN_INIT_STRUCT:
        /* initialize can initialization structure with the default values */
        ((hal_can_init_struct *)p_struct)->frame_format                   = CAN_CLASSIC_MODE;
        ((hal_can_init_struct *)p_struct)->operation_mode                 = CAN_NORMAL_MODE;
        ((hal_can_init_struct *)p_struct)->self_reception                 = CAN_SELF_RECEPTION_DISABLE;
        ((hal_can_init_struct *)p_struct)->internal_counter_source        = CAN_TIMER_SOURCE_BIT_CLOCK;
        ((hal_can_init_struct *)p_struct)->mb_tx_order                    = CAN_TX_HIGH_PRIORITY_MB_FIRST;
        ((hal_can_init_struct *)p_struct)->mb_tx_abort_enable             = CAN_TRANSMIT_ABORT_ENABLE;
        ((hal_can_init_struct *)p_struct)->local_priority_enable          = CAN_LOCAL_PRIORITY_DISABLE;
        ((hal_can_init_struct *)p_struct)->arbitration_start_delay        = 20U;
        ((hal_can_init_struct *)p_struct)->mb_remote_frame                = CAN_STORE_REMOTE_REQUEST_FRAME;
        ((hal_can_init_struct *)p_struct)->protocol_exception_enable      = CAN_PROTOCOL_EXCEPTION_DISABLE;
        ((hal_can_init_struct *)p_struct)->bit_sampling_mode              = CAN_BSP_MODE_THREE_SAMPLES;
        ((hal_can_init_struct *)p_struct)->auto_busoff_recovery           = CAN_AUTO_BUSOFF_RECOVERY_DISABLE;
        ((hal_can_init_struct *)p_struct)->time_synchronization           = CAN_TIME_SYNC_DISABLE;
        ((hal_can_init_struct *)p_struct)->mb_rx_ide_rtr_type             = CAN_IDE_RTR_FILTERED;
        ((hal_can_init_struct *)p_struct)->rx_private_filter_queue_enable = CAN_PRIVATE_FILTER_QUEUE_DISABLE;
        ((hal_can_init_struct *)p_struct)->edge_filter_enable             = CAN_EDGE_FILTER_DISABLE;
        ((hal_can_init_struct *)p_struct)->rx_filter_order                = CAN_RX_FILTER_ORDER_FIFO_FIRST;
        ((hal_can_init_struct *)p_struct)->memory_size                    = CAN_MEMSIZE_32_UNIT;
        ((hal_can_init_struct *)p_struct)->mb_public_filter               = 0x00000000U;
        ((hal_can_init_struct *)p_struct)->rx_fifo_enable                 = CAN_RX_FIFO_DISABLE;
        ((hal_can_init_struct *)p_struct)->filter_format_and_number       = CAN_RXFIFO_FILTER_A_NUM_8;
        ((hal_can_init_struct *)p_struct)->fifo_public_filter             = 0x00000000U;
        ((hal_can_init_struct *)p_struct)->prescaler                      = 0x00000015U;
        ((hal_can_init_struct *)p_struct)->resync_jump_width              = 0x01U;
        ((hal_can_init_struct *)p_struct)->prop_time_segment              = 0x02U;
        ((hal_can_init_struct *)p_struct)->time_segment_1                 = 0x05U;
        ((hal_can_init_struct *)p_struct)->time_segment_2                 = 0x02U;
        break;
    case HAL_CAN_FD_CONFIG_STRUCT:
        /* used for initialize hal_can_fd_config_struct */
        ((hal_can_fd_config_struct *)p_struct)->iso_can_fd_enable     = CAN_FD_SPEC_BOSCH;
        ((hal_can_fd_config_struct *)p_struct)->bitrate_switch_enable = CAN_BITRATE_SWITCH_DISABLE;
        ((hal_can_fd_config_struct *)p_struct)->mailbox_data_size     = CAN_MAILBOX_DATA_SIZE_8_BYTES;
        ((hal_can_fd_config_struct *)p_struct)->tdc_enable            = CAN_TX_DELAY_COMPENSATION_DISABLE;
        ((hal_can_fd_config_struct *)p_struct)->tdc_offset            = 0x02U;
        ((hal_can_fd_config_struct *)p_struct)->prescaler             = 0x00000015U;
        ((hal_can_fd_config_struct *)p_struct)->resync_jump_width     = 0x01U;
        ((hal_can_fd_config_struct *)p_struct)->prop_time_segment     = 0x02U;
        ((hal_can_fd_config_struct *)p_struct)->time_segment_1        = 0x05U;
        ((hal_can_fd_config_struct *)p_struct)->time_segment_2        = 0x02U;
        break;
    case HAL_CAN_PN_MODE_CONFIG_STRUCT:
        /* used for initialize hal_can_pn_mode_config_struct */
        ((hal_can_pn_mode_config_struct *)p_struct)->num_matches   = 0x01U;
        ((hal_can_pn_mode_config_struct *)p_struct)->match_timeout = 0x0000U;
        ((hal_can_pn_mode_config_struct *)p_struct)->frame_filter  = CAN_PN_FRAME_FILTERING_ID;
        ((hal_can_pn_mode_config_struct *)p_struct)->id_filter     = CAN_PN_ID_FILTERING_EXACT;
        ((hal_can_pn_mode_config_struct *)p_struct)->data_filter   = CAN_PN_DATA_FILTERING_EXACT;
        break;
    case HAL_CAN_PN_MODE_FILTER_STRUCT:
        /* used for initialize hal_can_pn_mode_filter_struct */
        ((hal_can_pn_mode_filter_struct *)p_struct)->remote_frame       = (uint32_t)RESET;
        ((hal_can_pn_mode_filter_struct *)p_struct)->extended_frame     = (uint32_t)RESET;
        ((hal_can_pn_mode_filter_struct *)p_struct)->id                 = 0x00000000U;
        ((hal_can_pn_mode_filter_struct *)p_struct)->dlc_high_threshold = 0x00000000U;
        ((hal_can_pn_mode_filter_struct *)p_struct)->dlc_low_threshold  = 0x00000000U;
        ((hal_can_pn_mode_filter_struct *)p_struct)->payload[0]         = 0x00000000U;
        ((hal_can_pn_mode_filter_struct *)p_struct)->payload[1]         = 0x00000000U;
        break;
    case HAL_CAN_MDSC_STRUCT:
        /* used for initialize hal_can_mailbox_descriptor_struct */
        ((hal_can_mailbox_descriptor_struct *)p_struct)->timestamp  = 0x00000000U;
        ((hal_can_mailbox_descriptor_struct *)p_struct)->dlc        = 0x00000000U;
        ((hal_can_mailbox_descriptor_struct *)p_struct)->rtr        = 0x00000000U;
        ((hal_can_mailbox_descriptor_struct *)p_struct)->ide        = 0x00000000U;
        ((hal_can_mailbox_descriptor_struct *)p_struct)->srr        = 0x00000000U;
        ((hal_can_mailbox_descriptor_struct *)p_struct)->reserve1   = 0x00000000U;
        ((hal_can_mailbox_descriptor_struct *)p_struct)->code       = 0x00000000U;
        ((hal_can_mailbox_descriptor_struct *)p_struct)->reserve2   = 0x00000000U;
        ((hal_can_mailbox_descriptor_struct *)p_struct)->esi        = 0x00000000U;
        ((hal_can_mailbox_descriptor_struct *)p_struct)->brs        = 0x00000000U;
        ((hal_can_mailbox_descriptor_struct *)p_struct)->fdf        = 0x00000000U;
        ((hal_can_mailbox_descriptor_struct *)p_struct)->id         = 0x00000000U;
        ((hal_can_mailbox_descriptor_struct *)p_struct)->prio       = 0x00000000U;
        ((hal_can_mailbox_descriptor_struct *)p_struct)->data       = NULL;
        ((hal_can_mailbox_descriptor_struct *)p_struct)->data_bytes = 0x00000000U;
        ((hal_can_mailbox_descriptor_struct *)p_struct)->padding    = 0x0000U;
        break;
    case HAL_CAN_FDES_STRUCT:
        /* used for initialize hal_can_rx_fifo_struct */
        ((hal_can_rx_fifo_struct *)p_struct)->timestamp = 0x00000000U;
        ((hal_can_rx_fifo_struct *)p_struct)->dlc       = 0x00000000U;
        ((hal_can_rx_fifo_struct *)p_struct)->rtr       = 0x00000000U;
        ((hal_can_rx_fifo_struct *)p_struct)->ide       = 0x00000000U;
        ((hal_can_rx_fifo_struct *)p_struct)->srr       = 0x00000000U;
        ((hal_can_rx_fifo_struct *)p_struct)->idhit     = 0x00000000U;
        ((hal_can_rx_fifo_struct *)p_struct)->id        = 0x00000000U;
        ((hal_can_rx_fifo_struct *)p_struct)->data[0]   = 0x00000000U;
        ((hal_can_rx_fifo_struct *)p_struct)->data[1]   = 0x00000000U;
        break;
    case HAL_CAN_FIFO_ID_FILTER_STRUCT:
        /* used for initialize hal_can_rx_fifo_id_filter_struct */
        ((hal_can_rx_fifo_id_filter_struct *)p_struct)->remote_frame   = 0x00000000U;
        ((hal_can_rx_fifo_id_filter_struct *)p_struct)->extended_frame = 0x00000000U;
        ((hal_can_rx_fifo_id_filter_struct *)p_struct)->id             = 0x00000000U;
        break;
    case HAL_CAN_CRC_STRUCT:
        /* used for initialize hal_can_crc_struct */
        ((hal_can_crc_struct *)p_struct)->classical_frm_mb_number          = 0x00000000U;
        ((hal_can_crc_struct *)p_struct)->classical_frm_transmitted_crc    = 0x00000000U;
        ((hal_can_crc_struct *)p_struct)->classical_fd_frm_mb_number       = 0x00000000U;
        ((hal_can_crc_struct *)p_struct)->classical_fd_frm_transmitted_crc = 0x00000000U;
        break;
    case HAL_CAN_ERRCNT_STRUCT:
        /* used for initialize hal_can_error_counter_struct */
        ((hal_can_error_counter_struct *)p_struct)->fd_data_phase_rx_errcnt = 0x00U;
        ((hal_can_error_counter_struct *)p_struct)->fd_data_phase_tx_errcnt = 0x00U;
        ((hal_can_error_counter_struct *)p_struct)->rx_errcnt               = 0x00U;
        ((hal_can_error_counter_struct *)p_struct)->tx_errcnt               = 0x00U;
        break;
    case HAL_CAN_BUFFER_STRUCT:
        /* used for initialize hal_can_buffer_struct */
        ((hal_can_buffer_struct *)p_struct)->mailbox_index = 0x00000000U;
        ((hal_can_buffer_struct *)p_struct)->mailbox       = NULL;
        ((hal_can_buffer_struct *)p_struct)->rx_fifo       = NULL;
        break;
    case HAL_CAN_USER_CALLBACK_STRUCT:
        /* used for initialize hal_can_user_callback_struct */
        ((hal_can_user_callback_struct *)p_struct)->complete_func = NULL;
        ((hal_can_user_callback_struct *)p_struct)->error_func    = NULL;
        break;
    case HAL_CAN_DEV_STRUCT:
        /* initialize device information structure with the default values */
        ((hal_can_dev_struct *)p_struct)->periph                          = 0U;
        ((hal_can_dev_struct *)p_struct)->can_irq.mailbox_transfer_handle = NULL;
        ((hal_can_dev_struct *)p_struct)->can_irq.mailbox_receive_handle  = NULL;
        ((hal_can_dev_struct *)p_struct)->can_irq.pn_receive_handle       = NULL;
        ((hal_can_dev_struct *)p_struct)->can_irq.rx_warning_handle       = NULL;
        ((hal_can_dev_struct *)p_struct)->can_irq.tx_warning_handle       = NULL;
        ((hal_can_dev_struct *)p_struct)->can_irq.rxfifo_receive_handle   = NULL;
        ((hal_can_dev_struct *)p_struct)->can_irq.rxfifo_warning_handle   = NULL;
        ((hal_can_dev_struct *)p_struct)->can_irq.rxfifo_overflow_handle  = NULL;
        ((hal_can_dev_struct *)p_struct)->can_irq.busoff_error_handle     = NULL;
        ((hal_can_dev_struct *)p_struct)->can_irq.busoff_recovery_handle  = NULL;
        ((hal_can_dev_struct *)p_struct)->can_irq.wakeup_match_handle     = NULL;
        ((hal_can_dev_struct *)p_struct)->can_irq.wakeup_timeout_handle   = NULL;
        ((hal_can_dev_struct *)p_struct)->can_irq.can_error_handle        = NULL;
        ((hal_can_dev_struct *)p_struct)->can_irq.canfd_error_handle      = NULL;
        ((hal_can_dev_struct *)p_struct)->p_dma_can                       = NULL;
        ((hal_can_dev_struct *)p_struct)->rxbuffer.mailbox_index          = 0U;
        ((hal_can_dev_struct *)p_struct)->rxbuffer.mailbox                = NULL;
        ((hal_can_dev_struct *)p_struct)->rxbuffer.rx_fifo                = NULL;
        ((hal_can_dev_struct *)p_struct)->state                           = HAL_CAN_STATE_RESET;
        ((hal_can_dev_struct *)p_struct)->error_state                     = HAL_CAN_ERROR_NONE;
        ((hal_can_dev_struct *)p_struct)->rx_fifo_callback                = NULL;
        ((hal_can_dev_struct *)p_struct)->rx_mailbox_callback             = NULL;
        ((hal_can_dev_struct *)p_struct)->error_callback                  = NULL;
        ((hal_can_dev_struct *)p_struct)->pn_callback                     = NULL;
        ((hal_can_dev_struct *)p_struct)->mutex                           = HAL_MUTEX_UNLOCKED;
        break;
    case HAL_CAN_IRQ_INIT_STRUCT:
        /* initialize interrupt callback structure with the default values */
        ((hal_can_irq_struct *)p_struct)->mailbox_transfer_handle = NULL;
        ((hal_can_irq_struct *)p_struct)->mailbox_receive_handle  = NULL;
        ((hal_can_irq_struct *)p_struct)->pn_receive_handle       = NULL;
        ((hal_can_irq_struct *)p_struct)->rx_warning_handle       = NULL;
        ((hal_can_irq_struct *)p_struct)->tx_warning_handle       = NULL;
        ((hal_can_irq_struct *)p_struct)->rxfifo_receive_handle   = NULL;
        ((hal_can_irq_struct *)p_struct)->rxfifo_warning_handle   = NULL;
        ((hal_can_irq_struct *)p_struct)->rxfifo_overflow_handle  = NULL;
        ((hal_can_irq_struct *)p_struct)->busoff_error_handle     = NULL;
        ((hal_can_irq_struct *)p_struct)->busoff_recovery_handle  = NULL;
        ((hal_can_irq_struct *)p_struct)->wakeup_match_handle     = NULL;
        ((hal_can_irq_struct *)p_struct)->wakeup_timeout_handle   = NULL;
        ((hal_can_irq_struct *)p_struct)->can_error_handle        = NULL;
        ((hal_can_irq_struct *)p_struct)->canfd_error_handle      = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        error_code = HAL_ERR_VAL;
        break;
    }

    return error_code;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  can_dev:CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to can interrupt callback functions structure
                  The structure member can be assigned as following parameters:
    \param[in]  mailboxmask: this parameter only applies to mailbox, FIFO does not need to care about it,
                  the parameter value is CAN_INTEN_MIEx(x=0~31)
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg        NULL: The corresponding callback mechanism is out of use, and
                    disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_can_irq_handle_set(hal_can_dev_struct *can_dev, hal_can_irq_struct *p_irq, uint32_t mailboxmask)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and p_irq address */
    if((NULL == can_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [can_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set user-defined transfer complete interrupt callback */
    if(NULL != p_irq->mailbox_transfer_handle) {
        CAN_INTEN(can_dev->periph) |= mailboxmask;
        can_dev->can_irq.mailbox_transfer_handle = p_irq->mailbox_transfer_handle;
    } else {
        CAN_INTEN(can_dev->periph) &= ~mailboxmask;
        can_dev->can_irq.mailbox_transfer_handle = NULL;
    }

    /* set user-defined mailbox receive complete interrupt callback */
    if(NULL != p_irq->mailbox_receive_handle) {
        CAN_INTEN(can_dev->periph) |= mailboxmask;
        can_dev->can_irq.mailbox_receive_handle = p_irq->mailbox_receive_handle;
    } else {
        CAN_INTEN(can_dev->periph) &= ~mailboxmask;
        can_dev->can_irq.mailbox_receive_handle = NULL;
    }

    /* set user-defined pn mode mailbox receive complete interrupt callback */
    if(NULL != p_irq->pn_receive_handle) {
        hals_can_interrupt_enable(can_dev->periph, CAN_INT_WAKEUP_MATCH);
        can_dev->can_irq.pn_receive_handle = p_irq->pn_receive_handle;
    } else {
        hals_can_interrupt_disable(can_dev->periph, CAN_INT_WAKEUP_MATCH);
        can_dev->can_irq.pn_receive_handle = NULL;
    }

    /* set user-defined receive warning interrupt callback */
    if(NULL != p_irq->rx_warning_handle) {
        hals_can_interrupt_enable(can_dev->periph, CAN_INT_RX_WARNING);
        can_dev->can_irq.rx_warning_handle = p_irq->rx_warning_handle;
    } else {
        hals_can_interrupt_disable(can_dev->periph, CAN_INT_RX_WARNING);
        can_dev->can_irq.rx_warning_handle = NULL;
    }

    /* set user-defined transmit warning interrupt callback */
    if(NULL != p_irq->tx_warning_handle) {
        hals_can_interrupt_enable(can_dev->periph, CAN_INT_TX_WARNING);
        can_dev->can_irq.tx_warning_handle = p_irq->tx_warning_handle;
    } else {
        hals_can_interrupt_disable(can_dev->periph, CAN_INT_TX_WARNING);
        can_dev->can_irq.tx_warning_handle = NULL;
    }

    /* set user-defined rxfifo receive interrupt callback */
    if(NULL != p_irq->rxfifo_receive_handle) {
        hals_can_interrupt_enable(can_dev->periph, CAN_INT_FIFO_AVAILABLE);
        can_dev->can_irq.rxfifo_receive_handle = p_irq->rxfifo_receive_handle;
    } else {
        hals_can_interrupt_disable(can_dev->periph, CAN_INT_FIFO_AVAILABLE);
        can_dev->can_irq.rxfifo_receive_handle = NULL;
    }

    /* set user-defined rxfifo warning interrupt callback */
    if(NULL != p_irq->rxfifo_warning_handle) {
        hals_can_interrupt_enable(can_dev->periph, CAN_INT_FIFO_WARNING);
        can_dev->can_irq.rxfifo_warning_handle = p_irq->rxfifo_warning_handle;
    } else {
        hals_can_interrupt_disable(can_dev->periph, CAN_INT_FIFO_WARNING);
        can_dev->can_irq.rxfifo_warning_handle = NULL;
    }

    /* set user-defined rxfifo overflow interrupt callback */
    if(NULL != p_irq->rxfifo_overflow_handle) {
        hals_can_interrupt_enable(can_dev->periph, CAN_INT_FIFO_OVERFLOW);
        can_dev->can_irq.rxfifo_overflow_handle = p_irq->rxfifo_overflow_handle;
    } else {
        hals_can_interrupt_disable(can_dev->periph, CAN_INT_FIFO_OVERFLOW);
        can_dev->can_irq.rxfifo_overflow_handle = NULL;
    }

    /* set user-defined can wakeup match interrupt callback */
    if(NULL != p_irq->wakeup_match_handle) {
        hals_can_interrupt_enable(can_dev->periph, CAN_INT_WAKEUP_MATCH);
        can_dev->can_irq.wakeup_match_handle = p_irq->wakeup_match_handle;
    } else {
        hals_can_interrupt_disable(can_dev->periph, CAN_INT_WAKEUP_MATCH);
        can_dev->can_irq.wakeup_match_handle = NULL;
    }

    /* set user-defined can wakeup timeout interrupt callback */
    if(NULL != p_irq->wakeup_timeout_handle) {
        hals_can_interrupt_enable(can_dev->periph, CAN_INT_WAKEUP_TIMEOUT);
        can_dev->can_irq.wakeup_timeout_handle = p_irq->wakeup_timeout_handle;
    } else {
        hals_can_interrupt_disable(can_dev->periph, CAN_INT_WAKEUP_TIMEOUT);
        can_dev->can_irq.wakeup_timeout_handle = NULL;
    }

    /* set user-defined can error interrupt callback */
    if(NULL != p_irq->can_error_handle) {
        hals_can_interrupt_enable(can_dev->periph, CAN_INT_ERR_SUMMARY);
        can_dev->can_irq.can_error_handle = p_irq->can_error_handle;
    } else {
        hals_can_interrupt_disable(can_dev->periph, CAN_INT_ERR_SUMMARY);
        can_dev->can_irq.can_error_handle = NULL;
    }

    /* set user-defined can fd error interrupt callback */
    if(NULL != p_irq->canfd_error_handle) {
        hals_can_interrupt_enable(can_dev->periph, CAN_INT_ERR_SUMMARY_FD);
        can_dev->can_irq.canfd_error_handle = p_irq->canfd_error_handle;
    } else {
        hals_can_interrupt_disable(can_dev->periph, CAN_INT_ERR_SUMMARY_FD);
        can_dev->can_irq.canfd_error_handle = NULL;
    }

    /* set user-defined busoff interrupt callback */
    if(NULL != p_irq->busoff_error_handle) {
        hals_can_interrupt_enable(can_dev->periph, CAN_INT_BUSOFF);
        can_dev->can_irq.busoff_error_handle = p_irq->busoff_error_handle;
    } else {
        hals_can_interrupt_disable(can_dev->periph, CAN_INT_BUSOFF);
        can_dev->can_irq.busoff_error_handle = NULL;
    }

    /* set user-defined busoff recovery interrupt callback */
    if(NULL != p_irq->busoff_recovery_handle) {
        hals_can_interrupt_enable(can_dev->periph, CAN_INT_BUSOFF_RECOVERY);
        can_dev->can_irq.busoff_recovery_handle = p_irq->busoff_recovery_handle;
    } else {
        hals_can_interrupt_disable(can_dev->periph, CAN_INT_BUSOFF_RECOVERY);
        can_dev->can_irq.busoff_recovery_handle = NULL;
    }

    return error_code;
}

/*!
    \brief      reset user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_can_irq_handle_all_reset(hal_can_dev_struct *can_dev)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* reset user-defined interrupt callback */
    can_dev->can_irq.mailbox_transfer_handle = NULL;
    can_dev->can_irq.rx_warning_handle       = NULL;
    can_dev->can_irq.tx_warning_handle       = NULL;
    can_dev->can_irq.rxfifo_receive_handle   = NULL;
    can_dev->can_irq.rxfifo_warning_handle   = NULL;
    can_dev->can_irq.rxfifo_overflow_handle  = NULL;
    can_dev->can_irq.busoff_error_handle     = NULL;
    can_dev->can_irq.busoff_recovery_handle  = NULL;
    can_dev->can_irq.wakeup_match_handle     = NULL;
    can_dev->can_irq.wakeup_timeout_handle   = NULL;
    can_dev->can_irq.can_error_handle        = NULL;
    can_dev->can_irq.canfd_error_handle      = NULL;
    can_dev->can_irq.mailbox_receive_handle  = NULL;
    can_dev->can_irq.pn_receive_handle       = NULL;

    /* reset interrupt flag */
    /* reset transfer complete interrupt */
    hals_can_interrupt_disable(can_dev->periph, CAN_INT_FIFO_AVAILABLE);

    /* reset mailbox receive complete interrupt */
    hals_can_interrupt_disable(can_dev->periph, CAN_INT_FIFO_AVAILABLE);

    /* reset pn mode mailbox receive complete interrupt */
    hals_can_interrupt_disable(can_dev->periph, CAN_INT_WAKEUP_MATCH);

    /* reset receive warning interrupt */
    hals_can_interrupt_disable(can_dev->periph, CAN_INT_RX_WARNING);

    /* reset transmit warning interrupt */
    hals_can_interrupt_disable(can_dev->periph, CAN_INT_TX_WARNING);

    /* reset rxfifo receive interrupt */
    hals_can_interrupt_disable(can_dev->periph, CAN_INT_FIFO_AVAILABLE);

    /* reset rxfifo warning interrupt */
    hals_can_interrupt_disable(can_dev->periph, CAN_INT_FIFO_WARNING);

    /* reset rxfifo overflow interrupt */
    hals_can_interrupt_disable(can_dev->periph, CAN_INT_FIFO_OVERFLOW);

    /* reset can wakeup match interrupt */
    hals_can_interrupt_disable(can_dev->periph, CAN_INT_WAKEUP_MATCH);

    /* reset can wakeup timeout interrupt */
    hals_can_interrupt_disable(can_dev->periph, CAN_INT_WAKEUP_TIMEOUT);

    /* reset can error interrupt */
    hals_can_interrupt_disable(can_dev->periph, CAN_INT_ERR_SUMMARY);

    /* reset can fd error interrupt */
    hals_can_interrupt_disable(can_dev->periph, CAN_INT_ERR_SUMMARY_FD);

    /* reset busoff interrupt */
    hals_can_interrupt_disable(can_dev->periph, CAN_INT_BUSOFF);

    /* reset  busoff recovery interrupt */
    hals_can_interrupt_disable(can_dev->periph, CAN_INT_BUSOFF_RECOVERY);

    return error_code;
}

/*!
    \brief      CAN interrupt handler content function,which is merely used in CAN_IRQHandler
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
void hal_can_irq(hal_can_dev_struct *can_dev)
{
    __IO hal_exti_line_enum linex;
    uint32_t i = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(CAN0 == can_dev->periph) {
        linex = EXTI_24;
    } else if(CAN1 == can_dev->periph) {
        linex = EXTI_25;
    } else {
        linex = EXTI_26;
    }

    if(0U != (CAN_CTL0(can_dev->periph) & CAN_CTL0_RFEN)) {
        /* check whether rxfifo available interrupt is set or not */
        if((CAN_INTEN(can_dev->periph) & CAN_INTEN_MIE5) && (CAN_STAT(can_dev->periph) & CAN_STAT_MS5_RFNE)) {
            can_dev->can_irq.rxfifo_receive_handle(can_dev);
            CAN_STAT(can_dev->periph) = CAN_STAT_MS5_RFNE;
        } else {
            /* do nothing */
        }

        /* check whether rxfifo warning interrupt is set or not */
        if((CAN_INTEN(can_dev->periph) & CAN_INTEN_MIE6) && (CAN_STAT(can_dev->periph) & CAN_STAT_MS6_RFW)) {
            CAN_STAT(can_dev->periph) = CAN_STAT_MS6_RFW;
            if(NULL != can_dev->can_irq.rxfifo_warning_handle) {
                can_dev->can_irq.rxfifo_warning_handle(can_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* check whether rxfifo overflow interrupt is set or not */
        if((CAN_INTEN(can_dev->periph) & CAN_INTEN_MIE7) && (CAN_STAT(can_dev->periph) & CAN_STAT_MS7_RFO)) {
            CAN_STAT(can_dev->periph) = CAN_STAT_MS7_RFO;
            if(NULL != can_dev->can_irq.rxfifo_overflow_handle) {
                can_dev->can_irq.rxfifo_overflow_handle(can_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* check whether mailbox interrupt is set or not */
        for(i = 8U; i <= (uint32_t)CAN_MAILBOX_MAX_NUM; i++) {
            if((CAN_INTEN(can_dev->periph) & BIT(i)) && (CAN_STAT(can_dev->periph) & BIT(i))) {
                can_dev->rxbuffer.mailbox_index = i;
                if(NULL != can_dev->can_irq.mailbox_receive_handle) {
                    can_dev->can_irq.mailbox_receive_handle(can_dev);
                } else {
                    /* do nothing */
                }
                CAN_STAT(can_dev->periph) = BIT(i);
            } else {
                /* do nothing */
            }
        }
    } else {
        /* check whether mailbox interrupt is set or not */
        for(i = 0U; i <= (uint32_t)CAN_MAILBOX_MAX_NUM; i++) {
            if((CAN_INTEN(can_dev->periph) & BIT(i)) && (CAN_STAT(can_dev->periph) & BIT(i))) {
                can_dev->rxbuffer.mailbox_index = i;
                if(NULL != can_dev->can_irq.mailbox_receive_handle) {
                    can_dev->can_irq.mailbox_receive_handle(can_dev);
                } else {
                    /* do nothing */
                }
                CAN_STAT(can_dev->periph) = BIT(i);
            } else {
                /* do nothing */
            }
        }
    }

    /* check whether wakeup match interrupt is set or not */
    if(RESET != hals_can_interrupt_flag_get(can_dev->periph, CAN_INT_FLAG_WAKEUP_MATCH)) {
        if(NULL != can_dev->can_irq.pn_receive_handle) {
            can_dev->can_irq.pn_receive_handle(can_dev);
        } else {
            /* do nothing */
        }
        hals_can_interrupt_flag_clear(can_dev->periph, CAN_INT_FLAG_WAKEUP_MATCH);
        hals_exti_interrupt_flag_clear(linex);
    } else {
        /* do nothing */
    }

    /* check whether wakeup timeout interrupt is set or not */
    if(RESET != hals_can_interrupt_flag_get(can_dev->periph, CAN_INT_FLAG_WAKEUP_TIMEOUT)) {
        hals_can_interrupt_flag_clear(can_dev->periph, CAN_INT_FLAG_WAKEUP_TIMEOUT);
        if(NULL != can_dev->can_irq.wakeup_timeout_handle) {
            can_dev->can_irq.wakeup_timeout_handle(can_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether rx warning interrupt is set or not */
    if(RESET != hals_can_interrupt_flag_get(can_dev->periph, CAN_INT_FLAG_RX_WARNING)) {
        hals_can_interrupt_flag_clear(can_dev->periph, CAN_INT_FLAG_RX_WARNING);
        if(NULL != can_dev->can_irq.rx_warning_handle) {
            can_dev->can_irq.rx_warning_handle(can_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether tx warning interrupt is set or not */
    if(RESET != hals_can_interrupt_flag_get(can_dev->periph, CAN_INT_FLAG_TX_WARNING)) {
        hals_can_interrupt_flag_clear(can_dev->periph, CAN_INT_FLAG_TX_WARNING);
        if(NULL != can_dev->can_irq.tx_warning_handle) {
            can_dev->can_irq.tx_warning_handle(can_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether busoff interrupt is set or not */
    if(RESET != hals_can_interrupt_flag_get(can_dev->periph, CAN_INT_FLAG_BUSOFF)) {
        hals_can_interrupt_flag_clear(can_dev->periph, CAN_INT_FLAG_BUSOFF);
        if(NULL != can_dev->can_irq.busoff_error_handle) {
            can_dev->can_irq.busoff_error_handle(can_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether busoff recovery interrupt is set or not */
    if(RESET != hals_can_interrupt_flag_get(can_dev->periph, CAN_INT_FLAG_BUSOFF_RECOVERY)) {
        hals_can_interrupt_flag_clear(can_dev->periph, CAN_INT_FLAG_BUSOFF_RECOVERY);
        if(NULL != can_dev->can_irq.busoff_recovery_handle) {
            can_dev->can_irq.busoff_recovery_handle(can_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether can error interrupt is set or not */
    if(RESET != hals_can_interrupt_flag_get(can_dev->periph, CAN_INT_FLAG_ERR_SUMMARY)) {
        hals_can_interrupt_flag_clear(can_dev->periph, CAN_INT_FLAG_ERR_SUMMARY);
        if(NULL != can_dev->can_irq.can_error_handle) {
            can_dev->can_irq.can_error_handle(can_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether can error interrupt is set or not */
    if(RESET != hals_can_interrupt_flag_get(can_dev->periph, CAN_INT_FLAG_ERR_SUMMARY_FD)) {
        hals_can_interrupt_flag_clear(can_dev->periph, CAN_INT_FLAG_ERR_SUMMARY_FD);
        if(NULL != can_dev->can_irq.canfd_error_handle) {
            can_dev->can_irq.canfd_error_handle(can_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      start CAN module function
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_can_start(hal_can_dev_struct *can_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* change can peripheral state */
    can_dev->state = HAL_CAN_STATE_READY;
    /* Reset the FDCAN ErrorCode */
    can_dev->error_state = HAL_CAN_ERROR_NONE;

    return _can_operation_mode_enter(can_dev->periph, (hal_can_operation_mode_enum)can_dev->init.operation_mode);
}

/*!
    \brief      stop CAN module function
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_can_stop(hal_can_dev_struct *can_dev)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* change can peripheral state */
    can_dev->state = HAL_CAN_STATE_STOP;
    /* reset the CAN ErrorCode */
    can_dev->error_state = HAL_CAN_ERROR_NONE;

    _can_operation_mode_enter(can_dev->periph, CAN_INACTIVE_MODE);

    /* reset CAN_INTEN */
    CAN_INTEN(can_dev->periph) = 0U;
    /* reset CAN_STAT */
    CAN_STAT(can_dev->periph) = (uint32_t)0xFFFFFFFFU;
    CAN_TIMER(can_dev->periph);
    while(CAN_STAT(can_dev->periph) & CAN_STAT_MS5_RFNE) {
        CAN_STAT(can_dev->periph) = CAN_STAT_MS5_RFNE;
    }

    return error_code;
}

/*!
    \brief      enter the corresponding mode
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  mode:Pattern selection
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_operation_mode_enter(hal_can_dev_struct *can_dev, hal_can_operation_mode_enum mode)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    return _can_operation_mode_enter(can_dev->periph, mode);
}

/*!
    \brief      get operation mode
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     CAN_NORMAL_MODE, CAN_MONITOR_MODE refer to <hal_can_operation_mode_enum> refer to gd32h7xx_hal_can.h
*/
hal_can_operation_mode_enum hal_can_operation_mode_get(hal_can_dev_struct *can_dev)
{
    uint32_t reg = 0U;
    hal_can_operation_mode_enum state = CAN_NORMAL_MODE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return CAN_INVALID_ADDR;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    reg = CAN_CTL0(can_dev->periph);
    reg &= (CAN_CTL0_NRDY | CAN_CTL0_INAS | CAN_CTL0_PNS | CAN_CTL0_LPS);

    if((CAN_CTL0_NRDY | CAN_CTL0_LPS) == reg) {
        state = CAN_DISABLE_MODE;
    } else if((CAN_CTL0_NRDY | CAN_CTL0_INAS) == reg) {
        state = CAN_INACTIVE_MODE;
    } else if(0U == reg) {
        if(CAN_CTL1(can_dev->periph) & CAN_CTL1_MMOD) {
            state = CAN_MONITOR_MODE;
        } else if(CAN_CTL1(can_dev->periph) & CAN_CTL1_LSCMOD) {
            state = CAN_LOOPBACK_SILENT_MODE;
        } else {
            state = CAN_NORMAL_MODE;
        }
    } else if(CAN_CTL0_PNS == reg) {
        state = CAN_PN_MODE;
    } else {
        /* should not get here */
    }

    return state;
}

/*!
    \brief      exit inactive mode
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_inactive_mode_exit(hal_can_dev_struct *can_dev)
{
    uint32_t timeout = 0U;
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* exit inactive mode */
    CAN_CTL0(can_dev->periph) &= ~CAN_CTL0_HALT;
    timeout = CAN_DELAY;
    while((CAN_CTL0(can_dev->periph) & CAN_CTL0_INAS) && (timeout)) {
        timeout--;
    }

    if(CAN_CTL0(can_dev->periph) & CAN_CTL0_INAS) {
        error_code = HAL_ERR_VAL;
    } else {
       /* do nothing */
    }

    return error_code;
}

/*!
    \brief      exit Pretended Networking mode
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_pn_mode_exit(hal_can_dev_struct *can_dev)
{
    uint32_t timeout = 0U;
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    CAN_CTL0(can_dev->periph) &= ~(CAN_CTL0_PNEN | CAN_CTL0_PNMOD);

    /* check is timeout */
    timeout = CAN_DELAY;
    while((CAN_CTL0(can_dev->periph) & CAN_CTL0_PNS) && (timeout)) {
        timeout--;
    }

    if(CAN_CTL0(can_dev->periph) & CAN_CTL0_PNS) {
        error_code = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    return error_code;
}

/*!
    \brief      configure receive fifo public filter
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  public_filter: rxfifo public filter data to configure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_can_rxfifo_public_filter_config(hal_can_dev_struct *can_dev, uint32_t public_filter)
{
    uint32_t i = 0U;
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure fifo public filter */
    CAN_RFIFOPUBF(can_dev->periph) = public_filter;
    /* configure fifo private filter */
    if(0U == (CAN_CTL0(can_dev->periph) & CAN_CTL0_RPFQEN)) {
        for(i = 0U; i < CAN_MAX_MAILBOX_NUM; i++) {
            CAN_RFIFOMPF(can_dev->periph, i) = public_filter;
        }
    } else {
        /* do nothing */
    }

    return error_code;
}

/*!
    \brief      configure receive fifo private filter
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  index: mailbox index
    \param[in]  private_filter: private filter data to configure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_private_filter_config(hal_can_dev_struct *can_dev, uint32_t index, uint32_t private_filter)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(CAN_MAILBOX_MAX_NUM< index) {
        HAL_DEBUGE("parameter [index] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    CAN_RFIFOMPF(can_dev->periph, index) = private_filter;

    return error_code;
}

/*!
    \brief      configure rx FIFO filter table
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  id_filter_table: id filter table struct
                  remote_frame: CAN_DATA_FRAME_ACCEPTED, CAN_REMOTE_FRAME_ACCEPTED
                  extended_frame: CAN_STANDARD_FRAME_ACCEPTED, CAN_EXTENDED_FRAME_ACCEPTED
                  id: 11 bits for standard frame, 29 bits for extended frame
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_rx_fifo_filter_table_config(hal_can_dev_struct *can_dev, \
                                            hal_can_rx_fifo_id_filter_struct id_filter_table[])
{
    /* set rx FIFO ID filter table elements */
    uint32_t i              = 0U;
    uint32_t j              = 0U;
    uint32_t num_of_filters = 0U;
    uint32_t val            = 0U;
    uint32_t id_format      = 0U;
    int32_t error_code      = HAL_ERR_NONE;
    uint32_t *filter_table  = (uint32_t *)(uint32_t)(CAN_RAM(can_dev->periph) + 0x00000060U);
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and id_filter_table address */
    if((NULL == can_dev) || (NULL == id_filter_table)) {
        HAL_DEBUGE("pointer [can_dev] or [id_filter_table] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    num_of_filters          = (GET_CTL2_RFFN(CAN_CTL2(can_dev->periph)) + 1U) * CAN_FIFO_NUM;
    id_format               = CAN_CTL0(can_dev->periph) & CAN_CTL0_FS;

    switch(id_format) {
    case(CAN_FIFO_FILTER_FORMAT_A):
        /* one full id (standard and extended) per id filter table element */
        for(i = 0U; i < num_of_filters; i++) {
            val = 0U;

            if(CAN_REMOTE_FRAME_ACCEPTED == id_filter_table[i].remote_frame) {
                val |= CAN_FDESX_RTR_A;
            } else {
                /* do nothing */
            }
            /* check frame format */
            if(CAN_EXTENDED_FRAME_ACCEPTED == id_filter_table[i].extended_frame) {
                val |= CAN_FDESX_IDE_A;
                val |= (uint32_t)FIFO_FILTER_ID_EXD_A(id_filter_table[i].id);
            } else {
                val |= (uint32_t)FIFO_FILTER_ID_STD_A(id_filter_table[i].id);
            }

            filter_table[i] = val;
        }
        break;
    case(CAN_FIFO_FILTER_FORMAT_B):
        /* two full standard id or two partial 14-bit (standard and extended) id */
        j = 0U;
        for(i = 0U; i < num_of_filters; i++) {
            val = 0U;

            if(CAN_REMOTE_FRAME_ACCEPTED == id_filter_table[j].remote_frame) {
                val |= CAN_FDESX_RTR_B0;
            } else {
                /* do nothing */
            }

            if(CAN_EXTENDED_FRAME_ACCEPTED == id_filter_table[j].extended_frame) {
                val |= CAN_FDESX_IDE_B0;
                val |= (uint32_t)FIFO_FILTER_ID_EXD_B0(id_filter_table[j].id);
            } else {
                val |= (uint32_t)FIFO_FILTER_ID_STD_B0(id_filter_table[j].id);
            }
            j++;

            if(CAN_REMOTE_FRAME_ACCEPTED == id_filter_table[j].remote_frame) {
                val |= CAN_FDESX_RTR_B1;
            } else {
                /* do nothing */
            }

            if(CAN_EXTENDED_FRAME_ACCEPTED == id_filter_table[j].extended_frame) {
                val |= CAN_FDESX_IDE_B1;
                val |= (uint32_t)FIFO_FILTER_ID_EXD_B1(id_filter_table[j].id);
            } else {
                val |= (uint32_t)FIFO_FILTER_ID_STD_B1(id_filter_table[j].id);
            }

            j++;
            filter_table[i] = val;
        }
        break;
    case(CAN_FIFO_FILTER_FORMAT_C):
        /* four partial 8-bit standard id per id filter table element */
        j = 0U;
        for(i = 0U; i < num_of_filters; i++) {
            val = 0U;
            if(CAN_EXTENDED_FRAME_ACCEPTED == id_filter_table[j].extended_frame) {
                val |= (uint32_t)FIFO_FILTER_ID_EXD_C0(id_filter_table[j].id);
            } else {
                val |= (uint32_t)FIFO_FILTER_ID_STD_C0(id_filter_table[j].id);
            }

            j++;
            if(CAN_EXTENDED_FRAME_ACCEPTED == id_filter_table[j].extended_frame) {
                val |= (uint32_t)FIFO_FILTER_ID_EXD_C1(id_filter_table[j].id);
            } else {
                val |= (uint32_t)FIFO_FILTER_ID_STD_C1(id_filter_table[j].id);
            }

            j++;
            if(CAN_EXTENDED_FRAME_ACCEPTED == id_filter_table[j].extended_frame) {
                val |= (uint32_t)FIFO_FILTER_ID_EXD_C2(id_filter_table[j].id);
            } else {
                val |= (uint32_t)FIFO_FILTER_ID_STD_C2(id_filter_table[j].id);
            }

            j++;
            if(CAN_EXTENDED_FRAME_ACCEPTED == id_filter_table[j].extended_frame) {
                val |= (uint32_t)FIFO_FILTER_ID_EXD_C3(id_filter_table[j].id);
            } else {
                val |= (uint32_t)FIFO_FILTER_ID_STD_C3(id_filter_table[j].id);
            }

            j++;
            filter_table[i] = val;
        }
        break;
    case(CAN_FIFO_FILTER_FORMAT_D):
        /* all frames rejected */
        break;
    default:
        /* should not get here */
        error_code = HAL_ERR_VAL;
        break;
    }

    return error_code;
}

/*!
    \brief      read rx FIFO data, poll completed status
                the function is blocking
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timeout: time out duration in millisecond
    \param[out] rx_fifo: rx FIFO struct
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_can_rx_fifo_read_poll(hal_can_dev_struct *can_dev, hal_can_rx_fifo_struct *rx_fifo, uint32_t timeout)
{
    uint32_t tick_start    = 0U;
    uint32_t *rx_fifo_addr = (uint32_t *)rx_fifo;
    uint32_t *can_mb       = (uint32_t *)CAN_RAM(can_dev->periph);
    int32_t error_code     = HAL_ERR_NONE;
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and rx_fifo address */
    if((NULL == can_dev) || (NULL == rx_fifo)) {
        HAL_DEBUGE("pointer [can_dev] or [rx_fifo] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock can_dev */
    HAL_LOCK(can_dev);

    can_dev->error_state = HAL_CAN_ERROR_NONE;
    can_dev->state       = HAL_CAN_STATE_BUSY_SYSTEM;

    /* configure timeout */
    tick_start = hal_sys_basetick_count_get();

    /* wait for mailbox flag */
    while(RESET == hals_can_flag_get(can_dev->periph, CAN_FLAG_FIFO_AVAILABLE)) {
        if(HAL_TIMEOUT_FOREVER != timeout) {
            if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                /* reset the state */
                can_dev->state = HAL_CAN_STATE_READY;
                /* unlock can_dev */
                error_code = HAL_ERR_TIMEOUT;
                break;
            }
        }
    }

    if (HAL_ERR_NONE == error_code) {
        /* read FIFO descriptor 0 */
        rx_fifo_addr[0]  = can_mb[0];
        rx_fifo_addr[1]  = can_mb[1];
        rx_fifo->data[0] = can_mb[2];
        rx_fifo->data[1] = can_mb[3];

        /* clear FIFO status */
        CAN_STAT(can_dev->periph) = CAN_STAT_MS5_RFNE;

        /* read FIFO id field */
        if(rx_fifo->ide) {
            rx_fifo->id = GET_FDES1_ID_EXD(rx_fifo->id);
        } else {
            rx_fifo->id = GET_FDES1_ID_STD(rx_fifo->id);
        }

        /* read FIFO data */
        _can_data_to_little_endian_swap(rx_fifo->data, rx_fifo->data, rx_fifo->dlc);

        hals_can_flag_clear(can_dev->periph, CAN_FLAG_FIFO_AVAILABLE);
    }

    can_dev->state = HAL_CAN_STATE_READY;

    /* unlock can_dev */
    HAL_UNLOCK(can_dev);

    return error_code;
}

/*!
    \brief      read rx FIFO data with interrupt
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: user callback
    \param[out] rx_fifo: rx FIFO struct
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_can_rx_fifo_read_interrupt(hal_can_dev_struct *can_dev, hal_can_rx_fifo_struct *rx_fifo, \
                                       hal_can_user_callback_struct *p_user_func)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and rx_fifo address */
    if((NULL == can_dev) || (NULL == rx_fifo) || (NULL == p_user_func)) {
        HAL_DEBUGE("pointer [can_dev] or [rx_fifo] or [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock can_dev */
    HAL_LOCK(can_dev);

    /* initialize receive parameters */
    can_dev->error_state = HAL_CAN_ERROR_NONE;
    can_dev->state       = HAL_CAN_STATE_BUSY_SYSTEM;

    can_dev->rxbuffer.rx_fifo = rx_fifo;

    /* clear callback */
    can_dev->rx_fifo_callback = NULL;
    can_dev->error_callback   = NULL;

    if(NULL != p_user_func) {
        can_dev->rx_fifo_callback = (void *)p_user_func->complete_func;
        can_dev->error_callback   = (void *)p_user_func->error_func;
    }

    /* set rxfifo receive interrupt callback */
    can_dev->can_irq.rxfifo_receive_handle  = (hal_irq_handle_cb)(uint32_t)_can_rxfifo_receive_interrupt;
    /* set mailbox receive interrupt callback */
    can_dev->can_irq.mailbox_receive_handle = (hal_irq_handle_cb)(uint32_t)_can_mailbox_receive_interrupt;

    /* enable CAN fifo interrupt */
    hals_can_interrupt_enable(can_dev->periph, CAN_INT_FIFO_AVAILABLE);

    /* unlock can_dev */
    HAL_UNLOCK(can_dev);

    return error_code;
}

/*!
    \brief      read rx FIFO data with DMA
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: user callback
    \param[out] rx_fifo: rx FIFO struct
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_rx_fifo_read_dma(hal_can_dev_struct *can_dev, hal_can_rx_fifo_struct *rx_fifo, \
                                 hal_can_user_callback_struct *p_user_func)
{
    hal_dma_irq_struct dma_irq = {NULL};
    uint32_t *can_mb           = (uint32_t *)CAN_RAM(can_dev->periph);
    int32_t error_code         = HAL_ERR_NONE;
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and rx_fifo address */
    if((NULL == can_dev) || (NULL == rx_fifo) || (NULL == p_user_func)) {
        HAL_DEBUGE("pointer [can_dev] or [rx_fifo] or [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock can_dev */
    HAL_LOCK(can_dev);

    can_dev->error_state = HAL_CAN_ERROR_NONE;
    can_dev->state       = HAL_CAN_STATE_BUSY_SYSTEM;

    can_dev->rxbuffer.rx_fifo = rx_fifo;

    /* wait mailbox data ready */
    while(SET == hals_can_flag_get(can_dev->periph, CAN_FLAG_FIFO_AVAILABLE)) {
        hals_can_flag_clear(can_dev->periph, CAN_FLAG_FIFO_AVAILABLE);
    }

    /* enable DMA */
    _can_operation_mode_enter(can_dev->periph, CAN_INACTIVE_MODE);
    CAN_CTL0(can_dev->periph) |= (CAN_CTL0_DMAEN);
    hal_can_inactive_mode_exit(can_dev);

    /* clear callback */
    can_dev->rx_fifo_callback = NULL;
    can_dev->error_callback   = NULL;

    if(NULL != p_user_func) {
        can_dev->rx_fifo_callback = (void *)p_user_func->complete_func;
        can_dev->error_callback   = (void *)p_user_func->error_func;
    } else {
        /* do nothing */
    }

    /* configure DMA interrupt interrupt callback function */
    dma_irq.half_finish_handle = NULL;
    dma_irq.full_finish_handle = _can_dmarx_complete;
    dma_irq.error_handle       = _can_dma_error;

    /* start DMA interrupt mode transfer */
    if(HAL_ERR_NONE != hal_dma_start_interrupt(can_dev->p_dma_can, (uint32_t)can_mb, (uint32_t)rx_fifo, 4U, &dma_irq)) {
        can_dev->state       = HAL_CAN_STATE_READY;
        can_dev->error_state = HAL_CAN_ERROR_DMA;
        error_code           = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    /* unlock can_dev */
    HAL_UNLOCK(can_dev);

    return error_code;
}

/*!
    \brief      get rx FIFO filter matching number
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint16_t:0x0000-0x01FF
*/
uint16_t hal_can_rx_fifo_filter_matching_number_get(hal_can_dev_struct *can_dev)
{
    return (uint16_t)(GET_RFIFOIFMN_IDFMN(CAN_RFIFOIFMN(can_dev->periph)));
}

/*!
    \brief      clear rx FIFO
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_can_rx_fifo_clear(hal_can_dev_struct *can_dev)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    CAN_STAT(can_dev->periph) = CAN_STAT_MS0_RFC;

    return error_code;
}

/*!
    \brief      configure mailbox public filter
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  public_filter: mailbox public filter data to configure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_can_mailbox_public_filter_config(hal_can_dev_struct *can_dev, uint32_t public_filter)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    CAN_RMPUBF(can_dev->periph) = public_filter;

    return error_code;
}

/*!
    \brief      configure mailbox
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  index: mailbox index
                only one parameter can be selected which is shown as below:
      \arg        0 - 31
    \param[in]  mdpara: mailbox descriptor struct
                  rtr: 0, 1
                  ide: 0, 1
                  id: 0 - 0x1FFFFFFF
                  code: CAN_MB_TX_STATUS_INACTIVE, CAN_MB_TX_STATUS_ABORT, CAN_MB_TX_STATUS_DATA
                  data_bytes: 0 - 64
                  data: point to the data
                  esi: 0, 1
                  fdf: 0, 1
                  brs: 0, 1
                  prio: 0 - 7
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_mailbox_transmit_poll(hal_can_dev_struct *can_dev, uint32_t index, hal_can_mailbox_descriptor_struct *mdpara)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t dlc = 0U;
    uint32_t mdes0 = 0U;
    uint32_t length = 0U;
    uint32_t *mdes;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and mdpara address */
    if((NULL == can_dev) || (NULL == mdpara)) {
        HAL_DEBUGE("pointer [can_dev] or [mdpara] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(CAN_MAILBOX_MAX_NUM < index) {
        HAL_DEBUGE("parameter [index] val is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear mailbox status */
    CAN_STAT(can_dev->periph) = STAT_MS(index);

    /* get mailbox base address */
    mdes    = _can_ram_address_get(can_dev->periph, index);
    mdes[0] = 0U;
    mdes[1] = 0U;
    mdes[2] = 0U;
    mdes[3] = 0U;

    /* set RTR bit */
    if(mdpara->rtr) {
        mdes0 |= CAN_MDES0_RTR;
    } else {
        /* do nothing */
    }

    /* set IDE bit and ID field */
    if(mdpara->ide) {
        mdes0 |= CAN_MDES0_IDE;
        mdes0 |= CAN_MDES0_SRR;
        mdes[1] |= MDES1_ID_EXD(mdpara->id);
    } else {
        mdes[1] |= MDES1_ID_STD(mdpara->id);
    }

    /* set CODE field */
    mdes0 |= MDES0_CODE(mdpara->code);

    if(CAN_MB_RX_STATUS_EMPTY != mdpara->code) {
        /* copy user's buffer into the mailbox data area */
        if(mdpara->data_bytes) {
            dlc = _can_dlc_value_compute(mdpara->data_bytes);
            mdes0 |= MDES0_DLC(dlc);
            length = (uint32_t)1U << (GET_FDCTL_MDSZ(CAN_FDCTL(can_dev->periph)) + 3U);
            if(mdpara->data_bytes < length) {
                length = mdpara->data_bytes;
            } else {
                /* do nothing */
            }
            _can_data_to_big_endian_swap(&mdes[2], mdpara->data, length);
        } else {
            /* do nothing */
        }

        /* prepare mailbox for transmission */
        if(CAN_MB_TX_STATUS_DATA == mdpara->code) {
            /* set ESI bit */
            if(mdpara->esi) {
                mdes0 |= CAN_MDES0_ESI;
            } else {
                /* do nothing */
            }

            /* set FDF and BRS bit */
            if(mdpara->fdf) {
                mdes0 |= CAN_MDES0_FDF;
                if(mdpara->brs) {
                    mdes0 |= CAN_MDES0_BRS;
                } else {
                    /* do nothing */
                }
                mdes0 &= ~CAN_MDES0_RTR;
            } else {
                /* do nothing */
            }
        }

        /* set PRIO field */
        mdes[1] |= MDES1_PRIO(mdpara->prio);
    }

    /* set mailbox descriptor 0 */
    mdes[0] = mdes0;

    return error_code;
}

/*!
    \brief      configure receive mailbox
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  index: mailbox index
                only one parameter can be selected which is shown as below:
      \arg        0 - 31
    \param[in]  mdpara: mailbox descriptor struct
                  rtr: 0, 1
                  ide: 0, 1
                  id: 0 - 0x1FFFFFFF
                  code: CAN_MB_RX_STATUS_INACTIVE, CAN_MB_RX_STATUS_FULL, CAN_MB_RX_STATUS_EMPTY,
                        CAN_MB_RX_STATUS_OVERRUN, CAN_MB_RX_STATUS_RANSWER, CAN_MB_RX_STATUS_BUSY
                  data_bytes: 0 - 64
                  data: point to the data
                  esi: 0, 1
                  fdf: 0, 1
                  brs: 0, 1
                  prio: 0 - 7
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_mailbox_receive_config(hal_can_dev_struct *can_dev, uint32_t index, hal_can_mailbox_descriptor_struct *mdpara)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t mdes0 = 0U;
    uint32_t *mdes;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and mdpara address */
    if((NULL == can_dev) || (NULL == mdpara)) {
        HAL_DEBUGE("pointer [can_dev] or [mdpara] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(CAN_MAILBOX_MAX_NUM < index) {
        HAL_DEBUGE("parameter [index] val is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear mailbox status */
    CAN_STAT(can_dev->periph) = STAT_MS(index);

    /* get mailbox base address */
    mdes    = _can_ram_address_get(can_dev->periph, index);
    mdes[0] = 0U;
    mdes[1] = 0U;
    mdes[2] = 0U;
    mdes[3] = 0U;

    /* set RTR bit */
    if(mdpara->rtr) {
        mdes0 |= CAN_MDES0_RTR;
    } else {
        /* do nothing */
    }

    /* set IDE bit and ID field */
    if(mdpara->ide) {
        mdes0 |= CAN_MDES0_IDE;
        mdes0 |= CAN_MDES0_SRR;
        mdes[1] |= MDES1_ID_EXD(mdpara->id);
    } else {
        mdes[1] |= MDES1_ID_STD(mdpara->id);
    }

    /* set CODE field */
    mdes0 |= MDES0_CODE(mdpara->code);

    /* set mailbox descriptor 0 */
    mdes[0] = mdes0;

    return error_code;
}

/*!
    \brief      read receive mailbox data, poll completed status
                the function is blocking
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  index: mailbox index
                only one parameter can be selected which is shown as below:
      \arg        0 - 31
    \param[in]  timeout: time out duration in millisecond
    \param[out] mdpara: mailbox descriptor struct
                  rtr: 0, 1
                  ide: 0, 1
                  id: 0 - 0x1FFFFFFF
                  code: CAN_MB_RX_STATUS_INACTIVE, CAN_MB_RX_STATUS_FULL, CAN_MB_RX_STATUS_EMPTY,
                        CAN_MB_RX_STATUS_OVERRUN, CAN_MB_RX_STATUS_RANSWER, CAN_MB_RX_STATUS_BUSY,
                        CAN_MB_TX_STATUS_INACTIVE, CAN_MB_TX_STATUS_ABORT, CAN_MB_TX_STATUS_DATA
                  data_bytes: 0 - 64
                  data: point to the data
                  esi: 0, 1
                  fdf: 0, 1
                  brs: 0, 1
                  prio: 0 - 7
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_can_mailbox_receive_data_read_poll(hal_can_dev_struct *can_dev, uint32_t index, \
                                               hal_can_mailbox_descriptor_struct *mdpara, uint32_t timeout)
{
    int32_t error_code  = HAL_ERR_NONE;
    uint32_t i          = 0U;
    uint32_t cnt        = 0U;
    uint32_t tick_start = 0U;
    uint32_t *mdes      = _can_ram_address_get(can_dev->periph, index);
    uint32_t *mdaddr    = (uint32_t *)mdpara;

    /* init flag */
    uint32_t flag = 0U;
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if((NULL == can_dev) || (NULL == mdpara)) {
        HAL_DEBUGE("pointer [can_dev] or [mdpara] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(CAN_MAILBOX_MAX_NUM < index) {
        HAL_DEBUGE("parameter [index] val is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock can_dev */
    HAL_LOCK(can_dev);

    can_dev->error_state = HAL_CAN_ERROR_NONE;
    can_dev->state       = HAL_CAN_STATE_BUSY_SYSTEM;

    /* configure timeout */
    tick_start = hal_sys_basetick_count_get();

    flag = CAN_REGIDX_BIT(STAT_REG_OFFSET, index);

    /* wait for mailbox flag */
    while(RESET == hals_can_flag_get(can_dev->periph, (hal_can_flag_enum)flag)) {
        if(HAL_TIMEOUT_FOREVER != timeout) {
            if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                error_code = HAL_ERR_TIMEOUT;
                break;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    }

    if(HAL_ERR_NONE == error_code) {
        /* wait mailbox data ready */
        while(mdes[0] & MDES0_CODE(CAN_MB_RX_STATUS_BUSY)) {
            if(HAL_TIMEOUT_FOREVER != timeout) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                    error_code = HAL_ERR_TIMEOUT;
                    break;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }

        if(HAL_ERR_NONE == error_code) {
            flag  = CAN_REGIDX_BIT(STAT_REG_OFFSET, index);
            hals_can_flag_clear(can_dev->periph, (hal_can_flag_enum)flag);

            /* get mailbox descriptor 0 */
            mdaddr[0]          = mdes[0];
            mdaddr[1]          = mdes[1];
            mdpara->data_bytes = _can_payload_size_compute(mdpara->dlc);
            cnt                = (mdpara->data_bytes + 3U) / 4U;
            mdaddr             = mdpara->data;

            mdes += 2U;
            for(i = 0U; i < cnt; i++) {
                mdaddr[i] = mdes[i];
            }

            /* clear mailbox status */
            CAN_STAT(can_dev->periph) = STAT_MS(index);
            /* unlock mailbox */
            CAN_TIMER(can_dev->periph);

            /* get mailbox ID */
            if(mdpara->ide) {
                mdpara->id = GET_MDES1_ID_EXD(mdpara->id);
            } else {
                mdpara->id = GET_MDES1_ID_STD(mdpara->id);
            }

            /* get mailbox data */
            if(mdpara->data_bytes) {
                _can_data_to_little_endian_swap(mdpara->data, mdpara->data, mdpara->data_bytes);
            } else {
                /* do nothing */
            }
        }
    }

    can_dev->state = HAL_CAN_STATE_READY;

    /* unlock can_dev */
    HAL_UNLOCK(can_dev);

    return error_code;
}

/*!
    \brief      read receive mailbox data with interrupt, the function is non-blocking
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  index: mailbox index
                only one parameter can be selected which is shown as below:
      \arg        0 - 31
    \param[in]  p_user_func: user callback
    \param[out] mdpara: mailbox descriptor struct
                  rtr: 0, 1
                  ide: 0, 1
                  id: 0 - 0x1FFFFFFF
                  code: CAN_MB_RX_STATUS_INACTIVE, CAN_MB_RX_STATUS_FULL, CAN_MB_RX_STATUS_EMPTY,
                        CAN_MB_RX_STATUS_OVERRUN, CAN_MB_RX_STATUS_RANSWER, CAN_MB_RX_STATUS_BUSY,
                        CAN_MB_TX_STATUS_INACTIVE, CAN_MB_TX_STATUS_ABORT, CAN_MB_TX_STATUS_DATA
                  data_bytes: 0 - 64
                  data: point to the data
                  esi: 0, 1
                  fdf: 0, 1
                  brs: 0, 1
                  prio: 0 - 7
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_mailbox_receive_data_read_interrupt(hal_can_dev_struct *can_dev, uint32_t index, \
                                                    hal_can_mailbox_descriptor_struct *mdpara, \
                                                    hal_can_user_callback_struct *p_user_func)
{
    int32_t error_code = HAL_ERR_NONE;
      /* init flag */
    uint32_t flag = 0U;
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and mdpara address */
    if((NULL == can_dev) || (NULL == mdpara) || (NULL == p_user_func)) {
        HAL_DEBUGE("pointer [can_dev] or [mdpara] or [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(CAN_MAILBOX_MAX_NUM < index) {
        HAL_DEBUGE("parameter [index] val is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock can_dev */
    HAL_LOCK(can_dev);

    can_dev->error_state = HAL_CAN_ERROR_NONE;
    can_dev->state       = HAL_CAN_STATE_BUSY_SYSTEM;

    can_dev->rxbuffer.mailbox = mdpara;

    /* clear callback */
    can_dev->rx_mailbox_callback = NULL;
    can_dev->error_callback      = NULL;

    if(NULL != p_user_func) {
        can_dev->rx_mailbox_callback = (void *)p_user_func->complete_func;
        can_dev->error_callback      = (void *)p_user_func->error_func;
    } else {
        /* do nothing */
    }

    can_dev->can_irq.mailbox_receive_handle = (hal_irq_handle_cb)_can_mailbox_receive_interrupt;

    flag = CAN_REGIDX_BIT(INTEN_REG_OFFSET, index);

    /* enable CAN mailbox interrupt */
    hals_can_interrupt_enable(can_dev->periph, (hal_can_interrupt_enum)flag);

    /* unlock can_dev */
    HAL_UNLOCK(can_dev);

    return error_code;
}

/*!
    \brief      abort mailbox transmit
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  index: mailbox index
                only one parameter can be selected which is shown as below:
      \arg        0 - 31
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_mailbox_transmit_abort(hal_can_dev_struct *can_dev, uint32_t index)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t mdes0 = 0U;
    uint32_t *mdes;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(CAN_MAILBOX_MAX_NUM < index) {
        HAL_DEBUGE("parameter [index] val is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* abort transmit mailbox */
    mdes  = _can_ram_address_get(can_dev->periph, index);
    mdes0 = mdes[0];
    mdes0 &= ~CAN_MDES0_CODE;
    mdes0 |= MDES0_CODE(CAN_MB_TX_STATUS_ABORT);
    mdes[0] = mdes0;

    return error_code;
}

/*!
    \brief      inactive transmit mailbox
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  index: mailbox index
                only one parameter can be selected which is shown as below:
      \arg        0 - 31
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_mailbox_transmit_inactive(hal_can_dev_struct *can_dev, uint32_t index)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t mdes0 = 0U;
    uint32_t *mdes;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(CAN_MAILBOX_MAX_NUM < index) {
        HAL_DEBUGE("parameter [index] val is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* inactive transmit mailbox */
    mdes  = _can_ram_address_get(can_dev->periph, index);
    mdes0 = mdes[0];
    mdes0 &= ~CAN_MDES0_CODE;
    mdes0 |= MDES0_CODE(CAN_MB_TX_STATUS_INACTIVE);
    mdes[0] = mdes0;

    return error_code;
}

/*!
    \brief      lock the receive mailbox
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  index: mailbox index
                only one parameter can be selected which is shown as below:
      \arg        0 - 31
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_mailbox_receive_lock(hal_can_dev_struct *can_dev, uint32_t index)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t *mdes;
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(CAN_MAILBOX_MAX_NUM < index) {
        HAL_DEBUGE("parameter [index] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    mdes = _can_ram_address_get(can_dev->periph, index);
    REG32((uint32_t)mdes);

    return error_code;
}

/*!
    \brief      unlock the receive mailbox
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_can_mailbox_receive_unlock(hal_can_dev_struct *can_dev)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    CAN_TIMER(can_dev->periph);

    return error_code;
}

/*!
    \brief      inactive the receive mailbox
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  index: mailbox index
                only one parameter can be selected which is shown as below:
      \arg        0 - 31
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_mailbox_receive_inactive(hal_can_dev_struct *can_dev, uint32_t index)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t mdes0 = 0U;
    uint32_t *mdes;
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer address */
    if(NULL == can_dev) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(CAN_MAILBOX_MAX_NUM < index) {
        HAL_DEBUGE("parameter [index] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* inactive receive mailbox */
    mdes  = _can_ram_address_get(can_dev->periph, index);
    mdes0 = mdes[0];
    mdes0 &= ~CAN_MDES0_CODE;
    mdes0 |= MDES0_CODE(CAN_MB_RX_STATUS_INACTIVE);
    mdes[0] = mdes0;

    return error_code;
}

/*!
    \brief      get mailbox code value
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  index: mailbox index(0~31)
                only one parameter can be selected which is shown as below:
      \arg        0 - 31
    \param[out] none
    \retval     uint32_t:0x00000000-0xFFFFFFFF
*/
uint32_t hal_can_mailbox_code_get(hal_can_dev_struct *can_dev, uint32_t index)
{
    uint32_t error_code = 0U;
    uint32_t *mdes;

    mdes = _can_ram_address_get(can_dev->periph, index);
    error_code = GET_MDES0_CODE(mdes[0]);

    return error_code;
}

/*!
    \brief      configure error counter
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  errcnt_struct:
                  fd_data_phase_rx_errcnt: 0-255
                  fd_data_phase_tx_errcnt: 0-255
                  rx_errcnt: 0-255
                  tx_errcnt: 0-255
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_can_error_counter_config(hal_can_dev_struct *can_dev, hal_can_error_counter_struct *errcnt_struct)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and errcnt_struct address */
    if((NULL == can_dev) || (NULL == errcnt_struct)) {
        HAL_DEBUGE("pointer [can_dev] or [errcnt_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    CAN_ERR0(can_dev->periph) = ERR0_REFCNT(errcnt_struct->fd_data_phase_rx_errcnt) | \
                                ERR0_TEFCNT(errcnt_struct->fd_data_phase_tx_errcnt) | \
                                ERR0_RECNT(errcnt_struct->rx_errcnt) | ERR0_TECNT(errcnt_struct->tx_errcnt);

    return error_code;
}

/*!
    \brief      get error counter
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] errcnt_struct:
                  fd_data_phase_rx_errcnt: 0-255
                  fd_data_phase_tx_errcnt: 0-255
                  rx_errcnt: 0-255
                  tx_errcnt: 0-255
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_can_error_counter_get(hal_can_dev_struct *can_dev, hal_can_error_counter_struct *errcnt_struct)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t reg = 0U;
        /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and errcnt_struct address */
    if((NULL == can_dev) || (NULL == errcnt_struct)) {
        HAL_DEBUGE("pointer [can_dev] or [errcnt_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* get error counter */
    reg = CAN_ERR0(can_dev->periph);
    errcnt_struct->fd_data_phase_rx_errcnt = (uint8_t)GET_ERR0_REFCNT(reg);
    errcnt_struct->fd_data_phase_tx_errcnt = (uint8_t)GET_ERR0_TEFCNT(reg);
    errcnt_struct->rx_errcnt = (uint8_t)GET_ERR0_RECNT(reg);
    errcnt_struct->tx_errcnt = (uint8_t)GET_ERR0_TECNT(reg);

    return error_code;
}

/*!
    \brief      get error state indicator
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_can_error_state_enum:CAN_ERROR_STATE_ACTIVE, CAN_ERROR_STATE_PASSIVE or CAN_ERROR_STATE_BUS_OFF
*/
hal_can_error_state_enum hal_can_error_state_get(hal_can_dev_struct *can_dev)
{
    uint32_t reg = 0U;

    /* get can state */
    reg = GET_ERR1_ERRSI(CAN_ERR1(can_dev->periph));
    if((uint32_t)CAN_ERROR_STATE_BUS_OFF <= reg) {
        reg = (uint32_t)CAN_ERROR_STATE_BUS_OFF;
    } else {
        /* do nothing */
    }

    return (hal_can_error_state_enum)reg;
}

/*!
    \brief      get mailbox CRC value
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] crc_struct:
                  classical_frm_mb_number: 0 - 0x1F
                  classical_frm_transmitted_crc: 0 - 0x7FFF
                  classical_fd_frm_mb_number: 0 - 0x1F
                  classical_fd_frm_transmitted_crc: 0 - 0x1FFFFF
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_can_crc_get(hal_can_dev_struct *can_dev, hal_can_crc_struct *crc_struct)
{
    uint32_t reg1      = 0U;
    uint32_t reg2      = 0U;
    int32_t error_code = HAL_ERR_NONE;
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and crc_struct address */
    if((NULL == can_dev) || (NULL == crc_struct)) {
        HAL_DEBUGE("pointer [can_dev] or [crc_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* get crc value */
    reg1 = CAN_CRCC(can_dev->periph);
    reg2 = CAN_CRCCFD(can_dev->periph);
    crc_struct->classical_frm_mb_number          = GET_CRCC_ANTM(reg1);
    crc_struct->classical_frm_transmitted_crc    = GET_CRCC_CRCTC(reg1);
    crc_struct->classical_fd_frm_mb_number       = GET_CRCCFD_ANTM(reg2);
    crc_struct->classical_fd_frm_transmitted_crc = GET_CRCCFD_CRCTCI(reg2);

    return error_code;
}

/*!
    \brief      configure Pretended Networking mode filter
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  expect: Pretended Networking mode struct of expected wakeup frame
                  remote_frame: SET, RESET
                  extended_frame: SET, RESET
                  id: 0x00000000~0x1FFFFFFF
                  dlc_high_threshold: 0~8
                  dlc_low_threshold: 0~8
                  payload[0]: 0x00000000~0xFFFFFFFF
                  payload[1]: 0x00000000~0xFFFFFFFF
    \param[in]  filter: Pretended Networking mode struct of filter data
                  remote_frame: SET, RESET
                  extended_frame: SET, RESET
                  id: 0x00000000~0x1FFFFFFF
                  dlc_high_threshold: 0~8
                  dlc_low_threshold: 0~8
                  payload[0]: 0x00000000~0xFFFFFFFF
                  payload[1]: 0x00000000~0xFFFFFFFF
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_can_pn_mode_filter_config(hal_can_dev_struct *can_dev, hal_can_pn_mode_filter_struct *expect, \
                                      hal_can_pn_mode_filter_struct *filter)
{
        uint32_t reg       = 0U;
    uint32_t temp      = 0U;
    int32_t error_code = HAL_ERR_NONE;
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and expect and filter address */
    if((NULL == can_dev) || (NULL == expect) || (NULL == filter)) {
        HAL_DEBUGE("pointer [can_dev] or [expect] or [expect] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set filter identifier 0 */
    if((uint32_t)SET == expect->extended_frame) {
        reg |= CAN_PN_EID0_EIDE;
        reg |= (uint32_t)PN_EID0_EIDF_ELT_EXD(expect->id);
    } else {
        reg |= (uint32_t)PN_EID0_EIDF_ELT_STD(expect->id);
    }
    if((uint32_t)SET == expect->remote_frame) {
        reg |= CAN_PN_EID0_ERTR;
    }
    CAN_PN_EID0(can_dev->periph) = reg;

    temp = CAN_PN_CTL0(can_dev->periph);
    /* ID field 1 is used when ID filtering type is EXACT or RANGE */
    if((CAN_PN_ID_FILTERING_EXACT == (temp & CAN_PN_CTL0_IDFT)) || \
       (CAN_PN_ID_FILTERING_RANGE == (temp & CAN_PN_CTL0_IDFT))) {
        /* set filter identifier 1 */
        reg = 0U;
        if((uint32_t)SET == filter->extended_frame) {
            reg |= CAN_PN_IFEID1_IDEFD;
            reg |= (uint32_t)PN_IFEID1_IDEFD_EXD(filter->id);
        } else {
            reg |= (uint32_t)PN_IFEID1_IDEFD_STD(filter->id);
        }
        if((uint32_t)SET == filter->remote_frame) {
            reg |= CAN_PN_IFEID1_RTRFD;
        } else {
            /* do nothing */
        }
        CAN_PN_IFEID1(can_dev->periph) = reg;
    } else {
        /* do nothing */
    }

    /* data field is used when frame filtering type is not MATCH or MATCH NMM */
    if((CAN_PN_FRAME_FILTERING_ID_DATA == (temp & CAN_PN_CTL0_FFT)) || \
       (CAN_PN_FRAME_FILTERING_ID_DATA_NMM == (temp & CAN_PN_CTL0_FFT))) {
        /* set filter data payload 0 */
        CAN_PN_EDLC(can_dev->periph) = PN_EDLC_DLCEHT(expect->dlc_high_threshold) | \
                                       PN_EDLC_DLCELT(expect->dlc_low_threshold);
        CAN_PN_EDL0(can_dev->periph) = ((expect->payload[0] << 24U) & CAN_PN_EDL0_DB0ELT) | \
                                        ((expect->payload[0] << 8U) & CAN_PN_EDL0_DB1ELT) | \
                                        ((expect->payload[0] >> 8U) & CAN_PN_EDL0_DB2ELT) | \
                                        ((expect->payload[0] >> 24U) & CAN_PN_EDL0_DB3ELT);
        CAN_PN_EDL1(can_dev->periph) = ((expect->payload[1] << 24U) & CAN_PN_EDL1_DB4ELT) | \
                                       ((expect->payload[1] << 8U) & CAN_PN_EDL1_DB5ELT) | \
                                       ((expect->payload[1] >> 8U) & CAN_PN_EDL1_DB6ELT) | \
                                       ((expect->payload[1] >> 24U) & CAN_PN_EDL1_DB7ELT);

        /* data field 1 is used when data filtering type is EXACT or RANGE */
        if((CAN_PN_DATA_FILTERING_EXACT == (temp & CAN_PN_CTL0_DATAFT))
           || (CAN_PN_DATA_FILTERING_RANGE == (temp & CAN_PN_CTL0_DATAFT))) {
            /* set filter data payload 1 */
            CAN_PN_DF0EDH0(can_dev->periph) = ((filter->payload[0] << 24U) & CAN_PN_DF0EDH0_DB0FD_EHT) | \
                                              ((filter->payload[0] << 8U) & CAN_PN_DF0EDH0_DB1FD_EHT) | \
                                              ((filter->payload[0] >> 8U) & CAN_PN_DF0EDH0_DB2FD_EHT) | \
                                              ((filter->payload[0] >> 24U) & CAN_PN_DF0EDH0_DB3FD_EHT);
            CAN_PN_DF1EDH1(can_dev->periph) = ((filter->payload[1] << 24U) & CAN_PN_DF1EDH1_DB4FD_EHT) | \
                                              ((filter->payload[1] << 8U) & CAN_PN_DF1EDH1_DB5FD_EHT) | \
                                              ((filter->payload[1] >> 8U) & CAN_PN_DF1EDH1_DB6FD_EHT) | \
                                              ((filter->payload[1] >> 24U) & CAN_PN_DF1EDH1_DB7FD_EHT);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return error_code;
}

/*!
    \brief      get matching message counter of Pretended Networking mode
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_ALREADY_DONE details refer to gd32h7xx_hal.h
*/
int32_t hal_can_pn_mode_num_of_match_get(hal_can_dev_struct *can_dev)
{
    int32_t ret = HAL_ERR_NONE;
    uint32_t reg = 0U;
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and p_irq address */
    if((NULL == can_dev)) {
        HAL_DEBUGE("pointer [can_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    reg = CAN_PN_STAT(can_dev->periph);
    if(0U != (reg & CAN_PN_STAT_MMCNTS)) {
        ret = (int32_t)(uint32_t)GET_PN_STAT_MMCNT(reg);
    } else {
        ret = HAL_ERR_ALREADY_DONE;
    }

    return ret;
}

/*!
    \brief      get matching message
                the function is blocking
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  index: Pretended Networking mailbox index
                only one parameter can be selected which is shown as below:
      \arg        0 - 3
    \param[out] mdpara: mailbox descriptor struct
                  rtr: 0, 1
                  ide: 0, 1
                  id: 0 - 0x1FFFFFFF
                  code: CAN_MB_RX_STATUS_INACTIVE, CAN_MB_RX_STATUS_FULL, CAN_MB_RX_STATUS_EMPTY,
                        CAN_MB_RX_STATUS_OVERRUN, CAN_MB_RX_STATUS_RANSWER, CAN_MB_RX_STATUS_BUSY,
                        CAN_MB_TX_STATUS_INACTIVE, CAN_MB_TX_STATUS_ABORT, CAN_MB_TX_STATUS_DATA
                  data_bytes: 0 - 64
                  data: point to the data
                  esi: 0, 1
                  fdf: 0, 1
                  brs: 0, 1
                  prio: 0 - 7
    \param[in]  timeout: time out duration in millisecond
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_pn_mode_data_read_poll(hal_can_dev_struct *can_dev, uint32_t index, \
                                       hal_can_mailbox_descriptor_struct *mdpara, uint32_t timeout)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t tick_start = 0U;
    uint32_t *mdaddr = (uint32_t *)mdpara;
    uint32_t *pnram  = (uint32_t *)(uint32_t)(CAN_PN_RAM(can_dev->periph) + index * 16U);
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and mdpara address */
    if((NULL == can_dev) || (NULL == mdpara)) {
        HAL_DEBUGE("pointer [can_dev] or [mdpara] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(3U < index) {
        HAL_DEBUGE("parameter [index] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock can_dev */
    HAL_LOCK(can_dev);

    can_dev->error_state = HAL_CAN_ERROR_NONE;
    can_dev->state       = HAL_CAN_STATE_BUSY_SYSTEM;

    /* configure timeout */
    tick_start = hal_sys_basetick_count_get();

    /* wait for pn wakeup match flag */
    while(RESET == hals_can_flag_get(can_dev->periph, CAN_FLAG_WAKEUP_MATCH)) {
        if(HAL_TIMEOUT_FOREVER != timeout) {
            if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                /* reset the state */
                can_dev->state = HAL_CAN_STATE_READY;
                /* unlock can_dev */
                error_code = HAL_ERR_TIMEOUT;
                break;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    }

    if(HAL_ERR_NONE == error_code) {
        mdaddr[0] = pnram[0];
        mdaddr[1] = pnram[1];

        /* get mailbox ID */
        if(0U != mdpara->ide) {
            mdpara->id = GET_MDES1_ID_EXD(mdpara->id);
        } else {
            mdpara->id = GET_MDES1_ID_STD(mdpara->id);
        }

        mdpara->data_bytes = mdpara->dlc;

        if(mdpara->data_bytes) {
            _can_data_to_little_endian_swap(mdpara->data, &pnram[2], mdpara->data_bytes);
        } else {
            /* do nothing */
        }

        hals_can_flag_clear(can_dev->periph, CAN_FLAG_WAKEUP_MATCH);
    } else {
        /* do nothing */
    }

    can_dev->state = HAL_CAN_STATE_READY;

    /* unlock can_dev */
    HAL_UNLOCK(can_dev);

    return error_code;
}

/*!
    \brief      get matching message with interrupt, the function is non-blocking
    \param[in]  can_dev: CAN device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  index: Pretended Networking mailbox index
                only one parameter can be selected which is shown as below:
      \arg        0 - 3
    \param[in]  p_user_func: user callback
    \param[out] mdpara: mailbox descriptor struct
                  rtr: 0, 1
                  ide: 0, 1
                  id: 0 - 0x1FFFFFFF
                  code: CAN_MB_RX_STATUS_INACTIVE, CAN_MB_RX_STATUS_FULL, CAN_MB_RX_STATUS_EMPTY,
                        CAN_MB_RX_STATUS_OVERRUN, CAN_MB_RX_STATUS_RANSWER, CAN_MB_RX_STATUS_BUSY,
                        CAN_MB_TX_STATUS_INACTIVE, CAN_MB_TX_STATUS_ABORT, CAN_MB_TX_STATUS_DATA
                  data_bytes: 0 - 64
                  data: point to the data
                  esi: 0, 1
                  fdf: 0, 1
                  brs: 0, 1
                  prio: 0 - 7
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_can_pn_mode_data_read_interrupt(hal_can_dev_struct *can_dev, uint32_t index, \
                                            hal_can_mailbox_descriptor_struct *mdpara, \
                                            hal_can_user_callback_struct *p_user_func)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and mdpara address */
    if((NULL == can_dev) || (NULL == mdpara) || (NULL == p_user_func)) {
        HAL_DEBUGE("pointer [can_dev] or [mdpara] or [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(3U < index) {
        HAL_DEBUGE("parameter [index] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock can_dev */
    HAL_LOCK(can_dev);

    can_dev->error_state = HAL_CAN_ERROR_NONE;
    can_dev->state       = HAL_CAN_STATE_BUSY_SYSTEM;

    can_dev->rxbuffer.mailbox_index = index;
    can_dev->rxbuffer.mailbox       = mdpara;

    /* clear callback */
    can_dev->pn_callback    = NULL;
    can_dev->error_callback = NULL;

    /* set user callback */
    if(NULL != p_user_func) {
        can_dev->pn_callback    = (void *)p_user_func->complete_func;
        can_dev->error_callback = (void *)p_user_func->error_func;
    } else {
        /* do nothing */
    }

    /* set pn receive interrupt callback */
    can_dev->can_irq.pn_receive_handle = (hal_irq_handle_cb)_can_pn_mode_data_receive_interrupt;

    hals_can_interrupt_enable(can_dev->periph, CAN_INT_WAKEUP_MATCH);

    /* unlock can_dev */
    HAL_UNLOCK(can_dev);

    return error_code;
}

/*!
    \brief      return CAN state
    \param[in]  can_dev: CAN device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_can_state_enum details refer to gd32h7xx_hal_can.h
*/
hal_can_state_enum hal_can_state_get(hal_can_dev_struct *can_dev)
{
    return (hal_can_state_enum)can_dev->state;
}

/*!
    \brief      return CAN error code
    \param[in]  can_dev: CAN device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_CAN_ERROR_NONE,HAL_CAN_ERROR_TIMEOUT,HAL_CAN_ERROR_SYSTEM,HAL_CAN_ERROR_DMA,\
                            HAL_CAN_ERROR_DMA details refer to gd32h7xx_hal_can.h
*/
uint32_t hal_can_error_get(hal_can_dev_struct *can_dev)
{
    return can_dev->error_state;
}

/*!
    \brief      get CAN flag
    \param[in]  can_periph: CANx(x=0,1,2)
    \param[in]  flag: CAN flags, refer to can_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        CAN_FLAG_CAN_PN: Pretended Networking state flag
      \arg        CAN_FLAG_SOFT_RST: software reset flag
      \arg        CAN_FLAG_ERR_SUMMARY: error summary flag
      \arg        CAN_FLAG_BUSOFF: bus off flag
      \arg        CAN_FLAG_RECEIVING: receiving state flag
      \arg        CAN_FLAG_TRANSMITTING: transmitting state flag
      \arg        CAN_FLAG_IDLE: IDLE state flag
      \arg        CAN_FLAG_RX_WARNING: receive warning flag
      \arg        CAN_FLAG_TX_WARNING: transmit warning flag
      \arg        CAN_FLAG_STUFF_ERR: stuff error flag
      \arg        CAN_FLAG_FORM_ERR: form error flag
      \arg        CAN_FLAG_CRC_ERR: CRC error flag
      \arg        CAN_FLAG_ACK_ERR: ACK error flag
      \arg        CAN_FLAG_BIT_DOMINANT_ERR: bit dominant error flag
      \arg        CAN_FLAG_BIT_RECESSIVE_ERR: bit recessive error flag
      \arg        CAN_FLAG_SYNC_ERR: synchronization flag
      \arg        CAN_FLAG_BUSOFF_RECOVERY: bus off recovery flag
      \arg        CAN_FLAG_ERR_SUMMARY_FD: fd error summary flag
      \arg        CAN_FLAG_ERR_OVERRUN: error overrun flag
      \arg        CAN_FLAG_STUFF_ERR_FD: stuff error in FD data phase flag
      \arg        CAN_FLAG_FORM_ERR_FD: form error in FD data phase flag
      \arg        CAN_FLAG_CRC_ERR_FD: CRC error in FD data phase flag
      \arg        CAN_FLAG_BIT_DOMINANT_ERR_FD: bit dominant error in FD data phase flag
      \arg        CAN_FLAG_BIT_RECESSIVE_ERR_FD: bit recessive error in FD data phase flag
      \arg        CAN_FLAG_MBx(x=0~31): mailbox x flag
      \arg        CAN_FLAG_FIFO_AVAILABLE: fifo available flag
      \arg        CAN_FLAG_FIFO_WARNING: fifo warning flag
      \arg        CAN_FLAG_FIFO_OVERFLOW: fifo overflow flag
      \arg        CAN_FLAG_WAKEUP_MATCH: Pretended Networking match flag
      \arg        CAN_FLAG_WAKEUP_TIMEOUT: Pretended Networking timeout wakeup flag
      \arg        CAN_FLAG_TDC_OUT_OF_RANGE: transmitter delay is out of compensation range flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_can_flag_get(uint32_t can_periph, hal_can_flag_enum flag)
{
    FlagStatus state = RESET;
    if(CAN_REG_VAL(can_periph, flag) & BIT(CAN_BIT_POS(flag))) {
        state = SET;
    } else {
        /* do nothing */
    }
    return state;
}

/*!
    \brief      clear CAN flag
    \param[in]  can_periph: CANx(x=0,1,2)
    \param[in]  flag: CAN flags, refer to can_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        CAN_FLAG_ERR_SUMMARY: error summary flag
      \arg        CAN_FLAG_BUSOFF: bus off flag
      \arg        CAN_FLAG_BUSOFF_RECOVERY: bus off recovery flag
      \arg        CAN_FLAG_ERR_SUMMARY_FD: fd error summary flag
      \arg        CAN_FLAG_ERR_OVERRUN: error overrun flag
      \arg        CAN_FLAG_MBx(x=0~31): mailbox x flag
      \arg        CAN_FLAG_FIFO_AVAILABLE: fifo available flag
      \arg        CAN_FLAG_FIFO_WARNING: fifo warning flag
      \arg        CAN_FLAG_FIFO_OVERFLOW: fifo overflow flag
      \arg        CAN_FLAG_WAKEUP_MATCH: Pretended Networking match flag
      \arg        CAN_FLAG_WAKEUP_TIMEOUT: Pretended Networking timeout wakeup flag
      \arg        CAN_FLAG_TDC_OUT_OF_RANGE: transmitter delay is out of compensation range flag
    \param[out] none
    \retval     none
*/
void hals_can_flag_clear(uint32_t can_periph, hal_can_flag_enum flag)
{
    if(CAN_FLAG_TDC_OUT_OF_RANGE == flag) {
        CAN_FDCTL(can_periph) |= CAN_FDCTL_TDCS;
    } else {
        CAN_REG_VAL(can_periph, flag) = BIT(CAN_BIT_POS(flag));
    }
}

/*!
    \brief      enable CAN interrupt
    \param[in]  can_periph: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  interrupt: CAN interrupt, refer to can_interrupt_enum
                only one parameter can be selected which is shown as below:
      \arg        CAN_INT_ERR_SUMMARY: error interrupt
      \arg        CAN_INT_RX_WARNING: receive warning interrupt
      \arg        CAN_INT_TX_WARNING: transmit warning interrupt
      \arg        CAN_INT_BUSOFF: bus off interrupt
      \arg        CAN_INT_BUSOFF_RECOVERY: bus off recovery interrupt
      \arg        CAN_INT_ERR_SUMMARY_FD: fd error interrupt
      \arg        CAN_INT_MBx(x=0~31): mailbox x interrupt
      \arg        CAN_INT_FIFO_AVAILABLE: fifo available interrupt
      \arg        CAN_INT_FIFO_WARNING: fifo warning interrupt
      \arg        CAN_INT_FIFO_OVERFLOW: fifo overflow interrupt
      \arg        CAN_INT_WAKEUP_MATCH: Pretended Networking match interrupt
      \arg        CAN_INT_WAKEUP_TIMEOUT: Pretended Networking timeout wakeup interrupt
    \param[out] none
    \retval     none
*/
void hals_can_interrupt_enable(uint32_t can_periph, hal_can_interrupt_enum interrupt)
{
    CAN_REG_VAL(can_periph, interrupt) |= BIT(CAN_BIT_POS(interrupt));
}

/*!
    \brief      disable CAN interrupt
    \param[in]  can_periph: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  interrupt: CAN interrupt, refer to can_interrupt_enum
                only one parameter can be selected which is shown as below:
      \arg        CAN_INT_ERR_SUMMARY: error interrupt
      \arg        CAN_INT_RX_WARNING: receive warning interrupt
      \arg        CAN_INT_TX_WARNING: transmit warning interrupt
      \arg        CAN_INT_BUSOFF: bus off interrupt
      \arg        CAN_INT_BUSOFF_RECOVERY: bus off recovery interrupt
      \arg        CAN_INT_ERR_SUMMARY_FD: fd error interrupt
      \arg        CAN_INT_MBx(x=0~31): mailbox x interrupt
      \arg        CAN_INT_FIFO_AVAILABLE: fifo available interrupt
      \arg        CAN_INT_FIFO_WARNING: fifo warning interrupt
      \arg        CAN_INT_FIFO_OVERFLOW: fifo overflow interrupt
      \arg        CAN_INT_WAKEUP_MATCH: Pretended Networking match interrupt
      \arg        CAN_INT_WAKEUP_TIMEOUT: Pretended Networking timeout wakeup interrupt
    \param[out] none
    \retval     none
*/
void hals_can_interrupt_disable(uint32_t can_periph, hal_can_interrupt_enum interrupt)
{
    CAN_REG_VAL(can_periph, interrupt) &= ~BIT(CAN_BIT_POS(interrupt));
}

/*!
    \brief      get CAN interrupt flag
    \param[in]  can_periph: CANx(x=0,1,2)
    \param[in]  int_flag: CAN interrupt flags, refer to can_interrupt_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        CAN_INT_FLAG_ERR_SUMMARY: error summary interrupt flag
      \arg        CAN_INT_FLAG_BUSOFF: bus off interrupt flag
      \arg        CAN_INT_FLAG_RX_WARNING: receive warning interrupt flag
      \arg        CAN_INT_FLAG_TX_WARNING: transmit warning interrupt flag
      \arg        CAN_INT_FLAG_BUSOFF_RECOVERY: bus off recovery interrupt flag
      \arg        CAN_INT_FLAG_ERR_SUMMARY_FD: fd error summary interrupt flag
      \arg        CAN_INT_FLAG_MBx(x=0~31): mailbox x interrupt flag
      \arg        CAN_INT_FLAG_FIFO_AVAILABLE: fifo available interrupt flag
      \arg        CAN_INT_FLAG_FIFO_WARNING: fifo warning interrupt flag
      \arg        CAN_INT_FLAG_FIFO_OVERFLOW: fifo overflow interrupt flag
      \arg        CAN_INT_FLAG_WAKEUP_MATCH: Pretended Networking match interrupt flag
      \arg        CAN_INT_FLAG_WAKEUP_TIMEOUT: Pretended Networking timeout wakeup interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_can_interrupt_flag_get(uint32_t can_periph, hal_can_interrupt_flag_enum int_flag)
{
    FlagStatus state = RESET;
    if(CAN_REG_VAL(can_periph, int_flag) & BIT(CAN_BIT_POS(int_flag))) {
        state = SET;
    } else {
        /* do nothing */
    }
    return state;
}

/*!
    \brief      clear CAN interrupt flag
    \param[in]  can_periph: CANx(x=0,1,2)
    \param[in]  int_flag: CAN interrupt flags, refer to can_interrupt_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        CAN_INT_FLAG_ERR_SUMMARY: error summary interrupt flag
      \arg        CAN_INT_FLAG_BUSOFF: bus off interrupt flag
      \arg        CAN_INT_FLAG_RX_WARNING: receive warning interrupt flag
      \arg        CAN_INT_FLAG_TX_WARNING: transmit warning interrupt flag
      \arg        CAN_INT_FLAG_BUSOFF_RECOVERY: bus off recovery interrupt flag
      \arg        CAN_INT_FLAG_ERR_SUMMARY_FD: fd error summary interrupt flag
      \arg        CAN_INT_FLAG_MBx(x=0~31): mailbox x interrupt flag
      \arg        CAN_INT_FLAG_FIFO_AVAILABLE: fifo available interrupt flag
      \arg        CAN_INT_FLAG_FIFO_WARNING: fifo warning interrupt flag
      \arg        CAN_INT_FLAG_FIFO_OVERFLOW: fifo overflow interrupt flag
      \arg        CAN_INT_FLAG_WAKEUP_MATCH: Pretended Networking match interrupt flag
      \arg        CAN_INT_FLAG_WAKEUP_TIMEOUT: Pretended Networking timeout wakeup interrupt flag
    \param[out] none
    \retval     none
*/
void hals_can_interrupt_flag_clear(uint32_t can_periph, hal_can_interrupt_flag_enum int_flag)
{
    CAN_REG_VAL(can_periph, int_flag) = BIT(CAN_BIT_POS(int_flag));
}

/*!
    \brief      enter the corresponding mode
    \param[in]  can_periph: CANx(x=0,1,2)
    \param[in]  mode: the argument could be selected from enumeration<hal_can_operation_mode_enum>
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
static int32_t _can_operation_mode_enter(uint32_t can_periph, hal_can_operation_mode_enum mode)
{
    uint32_t timeout    = 0U;
    int32_t  error_code = HAL_ERR_NONE;

    /* enter INACTIVE mode */
    /* exit can_disable mode */
    CAN_CTL0(can_periph) &= ~CAN_CTL0_CANDIS;
    /* enter inactive mode */
    CAN_CTL0(can_periph) |= CAN_CTL0_HALT | CAN_CTL0_INAMOD;
    /* exit Pretended Networking mode */
    CAN_CTL0(can_periph) &= ~(CAN_CTL0_PNEN | CAN_CTL0_PNMOD);
    timeout = CAN_DELAY;
    /* wait for inactive mode state */
    while(((CAN_CTL0_NRDY | CAN_CTL0_INAS) != (CAN_CTL0(can_periph) & (CAN_CTL0_NRDY | CAN_CTL0_INAS))) && (timeout)) {
        timeout--;
    }

    if((CAN_CTL0_NRDY | CAN_CTL0_INAS) != (CAN_CTL0(can_periph) & (CAN_CTL0_NRDY | CAN_CTL0_INAS))) {
        error_code = HAL_ERR_VAL;
    } else {
        /* configure the modes */
        switch(mode) {
        case CAN_NORMAL_MODE:
            CAN_CTL1(can_periph) &= ~(CAN_CTL1_LSCMOD | CAN_CTL1_MMOD);
            break;
        case CAN_MONITOR_MODE:
            CAN_CTL1(can_periph) &= ~CAN_CTL1_LSCMOD;
            CAN_CTL1(can_periph) |= CAN_CTL1_MMOD;
            break;
        case CAN_LOOPBACK_SILENT_MODE:
            CAN_CTL1(can_periph) &= ~CAN_CTL1_MMOD;
            CAN_CTL0(can_periph) &= ~CAN_CTL0_SRDIS;
            CAN_FDCTL(can_periph) &= ~CAN_FDCTL_TDCEN;
            CAN_CTL1(can_periph) |= CAN_CTL1_LSCMOD;
            break;
        case CAN_INACTIVE_MODE:
            break;
        case CAN_DISABLE_MODE:
            CAN_CTL0(can_periph) |= CAN_CTL0_CANDIS;
            break;
        case CAN_PN_MODE:
            CAN_CTL0(can_periph) |= (CAN_CTL0_PNEN | CAN_CTL0_PNMOD);
            break;
        default:
            error_code = HAL_ERR_VAL;
            break;
        }

        if(HAL_ERR_NONE == error_code) {
            /* exit INACTIVE mode */
            if(CAN_INACTIVE_MODE != mode) {
                /* exit inactive mode */
                CAN_CTL0(can_periph) &= ~(CAN_CTL0_HALT | CAN_CTL0_INAMOD);
                timeout = CAN_DELAY;
                while((CAN_CTL0(can_periph) & CAN_CTL0_INAS) && (timeout)) {
                    timeout--;
                }

                if(CAN_CTL0(can_periph) & CAN_CTL0_INAS) {
                    error_code = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            if((HAL_ERR_NONE == error_code) && (CAN_PN_MODE == mode)) {
                timeout = CAN_DELAY;
                while((0U == (CAN_CTL0(can_periph) & CAN_CTL0_PNS)) && (timeout)) {
                    timeout--;
                }

                if(0U == (CAN_CTL0(can_periph) & CAN_CTL0_PNS)) {
                    error_code = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    }

    return error_code;
}

/*!
    \brief      deinitialize CAN
    \param[in]  can_periph: CANx(x=0,1,2)
    \param[out] none
    \retval     none
*/
static void _can_deinit(uint32_t can_periph)
{
    if(CAN0 == can_periph) {
        /* reset CAN0 */
        hal_rcu_periph_reset_enable(RCU_CAN0RST);
        hal_rcu_periph_reset_disable(RCU_CAN0RST);
    }

    if(CAN1 == can_periph) {
        /* reset CAN1 */
        hal_rcu_periph_reset_enable(RCU_CAN1RST);
        hal_rcu_periph_reset_disable(RCU_CAN1RST);
    }

    if(CAN2 == can_periph) {
        /* reset CAN2 */
        hal_rcu_periph_reset_enable(RCU_CAN2RST);
        hal_rcu_periph_reset_disable(RCU_CAN2RST);
    }
}

/*!
    \brief      reset CAN internal state machines and CAN registers
    \param[in]  can_periph: CANx(x=0,1,2)
    \param[out] none
    \retval     ErrStatus:ERROR or SUCCESS
*/
static ErrStatus _can_software_reset(uint32_t can_periph)
{
    uint32_t timeout = CAN_DELAY;
    ErrStatus state = SUCCESS;

    /* reset internal state machines and CAN registers */
    CAN_CTL0(can_periph) |= CAN_CTL0_SWRST;

    /* wait reset complete */
    while((CAN_CTL0(can_periph) & CAN_CTL0_SWRST) && (timeout)) {
        timeout--;
    }

    if(CAN_CTL0(can_periph) & CAN_CTL0_SWRST) {
        state = ERROR;
    } else {
        /* do nothing */
    }

    return state;
}

/*!
    \brief      get mailbox RAM address
    \param[in]  can_periph: CANx(x=0,1,2)
    \param[in]  index: mailbox index, 0-31
    \param[out] none
    \retval     uint32_t *:0-0xFFFF
*/
static uint32_t *_can_ram_address_get(uint32_t can_periph, uint32_t index)
{
    uint32_t payload_size;
    uint32_t *address;

    /* if CAN FD mode is enabled */
    if(CAN_CTL0(can_periph) & CAN_CTL0_FDEN) {
        payload_size = (uint32_t)1U << (GET_FDCTL_MDSZ(CAN_FDCTL(can_periph)) + 3U);
    } else {
        payload_size = CAN_FIFO_NUM;
    }

    address = (uint32_t *)(uint32_t)(CAN_RAM(can_periph) + (payload_size + CAN_FIFO_NUM) * index);

    return address;
}

/*!
    \brief      computes the maximum payload size (in bytes), given a dlc
    \param[in]  dlc_value: dlc value
    \param[out] none
    \retval     uint8_t:0=127
*/
static uint32_t _can_payload_size_compute(uint32_t dlc_value)
{
    uint8_t ret = 0U;

    if(15U >= dlc_value) {
        ret = dlc_to_databytes[dlc_value];
    }

    return (uint32_t)ret;
}

/*!
    \brief      swap data to little endian
    \param[in]  src: data source address
    \param[in]  len: data length be byte
    \param[out] dest: data destination address
    \retval     none
*/
static void _can_data_to_little_endian_swap(uint32_t *dest, uint32_t *src, uint32_t len)
{
    volatile uint32_t i = 0U;
    uint32_t cnt;
    uint32_t temp_src = 0U;

    /* get the word length of the data */
    cnt = (len + 3U) / 4U;
    /* change each word from big endian to little endian */
    for(i = 0U; i < cnt; i++) {
        temp_src = src[i];
        dest[i]  = ((uint32_t)(temp_src >> 24U) & 0x000000FFU) | ((uint32_t)(temp_src >> 8U) & 0x0000FF00U) | \
                   ((uint32_t)(temp_src << 8U) & 0x00FF0000U) | ((uint32_t)(temp_src << 24U) & 0xFF000000U);
    }

    cnt = len % 4U;
    if(cnt) {
        dest[i - 1U] &= ((uint32_t)1U << (cnt * 8U)) - 1U;
    } else {
        /* do nothing */
    }
}

/*!
    \brief      swap data to big endian
    \param[in]  src: data source address
    \param[in]  len: data length be byte
    \param[out] dest: data destination address
    \retval     none
*/
static void _can_data_to_big_endian_swap(uint32_t *dest, uint32_t *src, uint32_t len)
{
    volatile uint32_t i = 0U;
    uint32_t cnt;
    uint32_t temp_src = 0U;

    /* get the word length of the data */
    cnt = (len + 3U) / 4U;
    for(i = 0U; i < cnt; i++) {
        /* change each word from little endian to big endian */
        temp_src = src[i];
        dest[i]  = ((uint32_t)(temp_src >> 24U) & 0x000000FFU) | ((uint32_t)(temp_src >> 8U) & 0x0000FF00U) | \
                   ((uint32_t)(temp_src << 8U) & 0x00FF0000U) | ((uint32_t)(temp_src << 24U) & 0xFF000000U);
    }

    cnt = len % 4U;
    if(cnt) {
        dest[i - 1U] &= ~(((uint32_t)1U << ((4U - cnt) * 8U)) - 1U);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      computes the DLC field value, given a payload size (in bytes)
    \param[in]  payload_size: payload size
    \param[out] none
    \retval     uint32_t:0-0xFFFFFFFF
*/
static uint32_t _can_dlc_value_compute(uint32_t payload_size)
{
    uint32_t ret = 8U;

    if(8U >= payload_size) {
        ret = payload_size;
    } else if(24U >= payload_size) {
        ret = (payload_size - 9U) / 4U + 9U;
    } else if(64U >= payload_size) {
        ret = (payload_size - 17U) / 16U + 13U;
    } else {
        ret = 8U;
    }

    return ret;
}

/*!
    \brief      handle the mailbox receive interrupt
    \param[in]  can_dev: pointer to a can device information structure
    \param[in]  index: index: mailbox index, 0-31
    \param[out] none
    \retval     none
*/
static void _can_mailbox_receive_interrupt(void *can)
{
    uint32_t timeout = 0U;
    uint32_t cnt = 0U;
    uint32_t i = 0U;

    hal_can_dev_struct *can_dev = (hal_can_dev_struct *)can;
    hal_can_mailbox_descriptor_struct *mdpara = can_dev->rxbuffer.mailbox;
    hal_can_user_cb p_func = (hal_can_user_cb)can_dev->rx_mailbox_callback;

    uint32_t *mdes = _can_ram_address_get(can_dev->periph, can_dev->rxbuffer.mailbox_index);
    uint32_t *mdaddr = (uint32_t *)can_dev->rxbuffer.mailbox;

    /* wait mailbox data ready */
    timeout = CAN_DELAY;
    while((mdes[0] & MDES0_CODE(CAN_MB_RX_STATUS_BUSY)) && (timeout)) {
        timeout--;
    }

    if(FALSE == (mdes[0] & MDES0_CODE(CAN_MB_RX_STATUS_BUSY))) {
        /* get mailbox descriptor 0 */
        mdaddr[0] = mdes[0];
        mdaddr[1] = mdes[1];
        mdpara->data_bytes = _can_payload_size_compute(mdpara->dlc);
        cnt = (mdpara->data_bytes + 3U) / 4U;
        mdaddr = mdpara->data;

        mdes += 2U;
        for(i = 0U; i < cnt; i++) {
            mdaddr[i] = mdes[i];
        }

        /* clear mailbox status */
        CAN_STAT(can_dev->periph) = STAT_MS(can_dev->rxbuffer.mailbox_index);
        /* unlock mailbox */
        CAN_TIMER(can_dev->periph);

        /* get mailbox ID */
        if(mdpara->ide) {
            mdpara->id = GET_MDES1_ID_EXD(mdpara->id);
        } else {
            mdpara->id = GET_MDES1_ID_STD(mdpara->id);
        }

        /* get mailbox data */
        if(mdpara->data_bytes) {
            _can_data_to_little_endian_swap(mdpara->data, mdpara->data, mdpara->data_bytes);
        } else {
            /* do nothing */
        }

        can_dev->state = HAL_CAN_STATE_READY;
        if(NULL != p_func) {
            /* if there is a user receive complete callback */
            p_func(can_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the rxfifo receive interrupt
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _can_rxfifo_receive_interrupt(void *can)
{
    hal_can_dev_struct *can_dev = (hal_can_dev_struct *)can;
    hal_can_user_cb p_func = (hal_can_user_cb)can_dev->rx_fifo_callback;
    uint32_t *can_mb = (uint32_t *)CAN_RAM(can_dev->periph);
    hal_can_rx_fifo_struct *rx_fifo = can_dev->rxbuffer.rx_fifo;
    uint32_t *rx_fifo_addr = (uint32_t *)can_dev->rxbuffer.rx_fifo;

    /* read FIFO descriptor 0 */
    rx_fifo_addr[0]  = can_mb[0];
    rx_fifo_addr[1]  = can_mb[1];
    rx_fifo->data[0] = can_mb[2];
    rx_fifo->data[1] = can_mb[3];

    /* read FIFO id field */
    if(rx_fifo->ide) {
        rx_fifo->id = GET_FDES1_ID_EXD(rx_fifo->id);
    } else {
        rx_fifo->id = GET_FDES1_ID_STD(rx_fifo->id);
    }

    /* read FIFO data */
    _can_data_to_little_endian_swap(rx_fifo->data, rx_fifo->data, rx_fifo->dlc);

    can_dev->state = HAL_CAN_STATE_READY;
    if(NULL != p_func) {
        /* if there is a user receive complete callback */
        p_func(can_dev);
    }
}

/*!
    \brief      handle the rxfifo receive interrupt
    \param[in]  can_dev: CAN device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _can_pn_mode_data_receive_interrupt(void *can)
{
    hal_can_dev_struct *can_dev = (hal_can_dev_struct *)can;
    hal_can_user_cb p_func = (hal_can_user_cb)can_dev->pn_callback;

    hal_can_mailbox_descriptor_struct *mdpara = can_dev->rxbuffer.mailbox;
    uint32_t *mdaddr = (uint32_t *)can_dev->rxbuffer.mailbox;
    uint32_t *pnram = (uint32_t *)(uint32_t)(CAN_PN_RAM(can_dev->periph) + can_dev->rxbuffer.mailbox_index * 16U);

    mdaddr[0] = pnram[0];
    mdaddr[1] = pnram[1];
    /* get mailbox ID */
    if(0U != mdpara->ide) {
        mdpara->id = GET_MDES1_ID_EXD(mdpara->id);
    } else {
        mdpara->id = GET_MDES1_ID_STD(mdpara->id);
    }

    mdpara->data_bytes = mdpara->dlc;
    if(mdpara->data_bytes) {
        _can_data_to_little_endian_swap(mdpara->data, &pnram[2], mdpara->data_bytes);
    } else {
        /* do nothing */
    }

    can_dev->state = HAL_CAN_STATE_READY;
    if(NULL != p_func) {
        /* if there is a user receive complete callback */
        p_func(can_dev);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the CAN DMA receive process complete
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _can_dmarx_complete(void *dma)
{
    hal_dma_dev_struct *p_dma = (hal_dma_dev_struct *)dma;
    hal_can_dev_struct *p_can = (hal_can_dev_struct *)p_dma->p_periph;
    hal_can_rx_fifo_struct *rx_fifo;
    hal_can_user_cb p_func = (hal_can_user_cb)p_can->rx_fifo_callback;
    rx_fifo = p_can->rxbuffer.rx_fifo;

    /* D-Cache Invalidate by address */
    SCB_InvalidateDCache_by_Addr((uint32_t *)rx_fifo, 16);

    /* DMA normal mode */
    if(0U == (DMA_CHCTL(p_dma->dma_periph, p_dma->channel) & DMA_CHXCTL_CMEN)) {
        _can_operation_mode_enter(p_can->periph, CAN_INACTIVE_MODE);
        CAN_STAT(p_can->periph) = CAN_STAT_MS0_RFC;
        /* disable DMA */
        CAN_CTL0(p_can->periph) &= ~(CAN_CTL0_DMAEN);
        hal_can_inactive_mode_exit(p_can);
        /* reset state */
        p_can->state = HAL_CAN_STATE_READY;
    } else {
        /* do nothing */
    }

    /* read FIFO id field */
    if(rx_fifo->ide) {
        rx_fifo->id = GET_FDES1_ID_EXD(rx_fifo->id);
    } else {
        rx_fifo->id = GET_FDES1_ID_STD(rx_fifo->id);
    }

    /* read FIFO data */
    _can_data_to_little_endian_swap(rx_fifo->data, rx_fifo->data, rx_fifo->dlc);

    if(NULL != p_func) {
        /* if there is a user receive complete callback */
        p_func(p_can);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the CAN DMA error process
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _can_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma = (hal_dma_dev_struct *)dma;
    hal_can_dev_struct *p_can = (hal_can_dev_struct *)p_dma->p_periph;
    hal_can_user_cb p_func = (hal_can_user_cb)p_can->error_callback;

    if(HAL_CAN_STATE_BUSY_SYSTEM == p_can->state) {
        _can_operation_mode_enter(p_can->periph, CAN_INACTIVE_MODE);
        CAN_STAT(p_can->periph) = CAN_STAT_MS0_RFC;
        hal_can_inactive_mode_exit(p_can);

        /* disable DMA */
        CAN_CTL0(p_can->periph) &= ~(CAN_CTL0_DMAEN);
        p_can->state = HAL_CAN_STATE_READY;
    } else {
        /* do nothing */
    }

    p_can->error_state = HAL_CAN_ERROR_DMA;

    if(NULL != p_func) {
        /* if there is a user error callback */
        p_func(p_can);
    } else {
        /* do nothing */
    }
}
