/*!
    \file    gd32h7xx_hal_vref.h
    \brief   definitions for the VREF

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/
#ifndef GD32H7XX_HAL_VREF_H
#define GD32H7XX_HAL_VREF_H

#include "gd32h7xx_hal.h"

/* VREF definitions */
#define VREF                       VREF_BASE                                  /*!< VREF base address */

/* registers definitions */
#define VREF_CS                    REG32(VREF + 0x00000000U)                  /*!< VREF control and status register */
#define VREF_CALIB                 REG32(VREF + 0x00000004U)                  /*!< VREF calibration register */

/* bits definitions */
/* VREF_CS */
#define VREF_CS_VREFEN             BIT(0)                                     /*!< VREF enable */
#define VREF_CS_HIPM               BIT(1)                                     /*!< VREF high impedance mode */
#define VREF_CS_VREFRDY            BIT(3)                                     /*!< VREF ready to output */
#define VREF_CS_VREFS              BITS(4,5)                                  /*!< VREF reference voltage value */

/* VREF_CALIB */
#define VREF_CALIB_VREFCAL         BITS(0,5)                                  /*!< VREF calibration value */

/* write value to VREF_CALIB_VREFCAL bit field */
#define CALIB_CALIB(regval)        (BITS(0,5) & ((uint32_t)(regval) << 0U))

/* @STRUCT_MEMBER: vref_ref_voltage */
/* @DEFINE: VREF reference voltage selection */
#define VREFS_VREFS(regval)        (BITS(4,5) & ((uint32_t)(regval) << 4U))   /*!< VREF reference voltage value */
#define VREF_CS_2V5                ((uint8_t)VREFS_VREFS(0))                  /*!< VREF reference voltage to 2.5V */
#define VREF_CS_2V048              ((uint8_t)VREFS_VREFS(1))                  /*!< VREF reference voltage to 2.048V */
#define VREF_CS_1V8                ((uint8_t)VREFS_VREFS(2))                  /*!< VREF reference voltage to 1.8V */
#define VREF_CS_1V5                ((uint8_t)VREFS_VREFS(3))                  /*!< VREF reference voltage to 1.5V */

/* @STRUCT_MEMBER: vref_user_calibration */
/* @=NULL */

/* VREF bit definitions */
#define VREF_EN                    VREF_CS_VREFEN                             /*!< VREF enable */
#define VREF_HIGH_IMPEDANCE_MODE   VREF_CS_HIPM                               /*!< VREF high impedance mode */
#define VREF_RDY                   VREF_CS_VREFRDY                            /*!< VREF ready */

/* @STRUCT_MEMBER: vref_calibration_mode */
/* @ENUM: calibration mode */
typedef enum {
    HAL_VREF_CALIB_MODE_FACTORY = 0,                                          /*!< factory calibration */
    HAL_VREF_CALIB_MODE_USER                                                  /*!< user calibration */
} hal_vref_calibration_mode_enum;

/* VREF structure type enum */
typedef enum {
    HAL_VREF_INIT_STRUCT = 0,                                                 /*!< VREF initialization structure */
} hal_vref_struct_type_enum;

/* @STRUCT_MEMBER: vref_mode */
/* @ENUM: VREF mode */
typedef enum {
    HAL_VREF_MODE_DISABLED = 0,                                               /*!< VREF disabled, VREFP pin pulled-down to VSSA */
    HAL_VREF_MODE_EXTERNAL,                                                   /*!< External voltage reference mode */
    HAL_VREF_MODE_INTERNAL,                                                   /*!< Internal voltage reference mode */
    HAL_VREF_MODE_HOLD                                                        /*!< Hold mode */
} hal_vref_mode_enum;

/* @PARA: p_vref_init */
/* @STRUCT: VREF init structure */
typedef struct {
    hal_vref_mode_enum              vref_mode;                                /*!< VREF mode */
    hal_vref_calibration_mode_enum  vref_calibration_mode;                    /*!< VREF calibration mode */
    uint32_t                        vref_user_calibration;                    /*!< VREF calibration value */
    uint8_t                         vref_ref_voltage;                         /*!< VREF reference voltage value */
} hal_vref_init_struct;

/* function declarations */
/* @FUNCTION: initialize VREF base */
/* deinitialize VREF */
void hal_vref_deinit(void);
/* initialize VREF */
int32_t hal_vref_init(hal_vref_init_struct *p_vref_init);
/* initialize the VREF structure */
int32_t hal_vref_struct_init(hal_vref_struct_type_enum hal_struct_type, void *p_struct);
/* @END */

/* pin high impedance mode disable */
void hal_vref_hipm_disable(void);
/* pin high impedance mode enable */
void hal_vref_hipm_enable(void);
/* get the calibration value of VREF */
uint8_t hal_vref_cali_value_get(void);
/* voltage reference disable */
void hal_vref_disable(void);
/* voltage reference enable */
void hal_vref_enable(void);
/* set the calibration value of VREF */
void hal_vref_cali_value_set(uint8_t vref_user_calibration);
/* configure the VREF voltage reference */
int32_t hal_vref_config(uint8_t vref_ref_voltage);

#endif /* GD32H7XX_HAL_VREF_H */
