/*!
    \file    gd32h7xx_hal_usrt.h
    \brief   definitions for the USRT

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_USRT_H
#define GD32H7XX_HAL_USRT_H

#include "gd32h7xx_hal.h"

/* usrt error code */
#define HAL_USRT_ERROR_NONE         BIT(0)                                      /*!< no error */
#define HAL_USRT_ERROR_PERR         BIT(1)                                      /*!< parity error */
#define HAL_USRT_ERROR_NERR         BIT(2)                                      /*!< noise error */
#define HAL_USRT_ERROR_FERR         BIT(3)                                      /*!< frame error */
#define HAL_USRT_ERROR_ORERR        BIT(4)                                      /*!< overrun error */
#define HAL_USRT_ERROR_DMA          BIT(5)                                      /*!< DMA transfer error */
#define HAL_USRT_ERROR_RTO          BIT(6)                                      /*!< receiver timeout error */
#define HAL_USRT_ERROR_TTO          BIT(7)                                      /*!< transmit timeout error */
#define HAL_USRT_ERROR_BUSY         BIT(8)                                      /*!< busy error */

/* usrt struct initialization type enum */
typedef enum {
    HAL_USRT_INIT_STRUCT = 0U,        /*!< initialization structure */
    HAL_USRT_DEV_STRUCT,              /*!< device information structure */
    HAL_USRT_IRQ_INIT_STRUCT,         /*!< interrupt callback initialization structure */
    HAL_USRT_USER_CALLBACK_STRUCT     /*!< user callback initialization structure */
} hal_usrt_struct_type_enum;

/* usrt run state enum */
typedef enum {
    HAL_USRT_STATE_RESET = 0U, /*!< not initialized state */
    HAL_USRT_STATE_READY,      /*!< initialized and ready for use */
    HAL_USRT_STATE_BUSY,       /*!< internal process is ongoing */
    HAL_USRT_STATE_BUSY_TX,    /*!< data transmission process is ongoing */
    HAL_USRT_STATE_BUSY_RX,    /*!< data reception process is ongoing */
    HAL_USRT_STATE_BUSY_TX_RX  /*!< data transmission reception process is ongoing */
} hal_usrt_run_state_enum;

/* usrt device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb transmit_fifo_empty_handle; /*!< transmit fifo empty callback function */
    __IO hal_irq_handle_cb transmit_complete_handle;   /*!< transmit complete callback function */
    __IO hal_irq_handle_cb receive_complete_handle;    /*!< receive complete callback function */
    __IO hal_irq_handle_cb receive_fifo_full_handle;   /*!< receive fifo full callback function */
    __IO hal_irq_handle_cb error_handle;               /*!< error callback function */
} hal_usrt_irq_struct;

/* usrt transfer buffer structure */
typedef struct {
    __IO uint8_t *buffer; /*!< pointer to transfer buffer */
    __IO uint32_t length; /*!< transfer byte length */
    __IO uint32_t remain; /*!< transfer remain byte */
} hal_usrt_buffer_struct;

/* @PARA: p_usrt */
/* @STRUCT: UART Init structure definition */
typedef struct {
    uint32_t baudrate;    /*!< configures the USRT communication baud rate */
    uint32_t wordlength;  /*!< specifies the number of data bits transmitted or received in a frame */
    uint32_t stopbits;    /*!< specifies the number of stop bits transmitted */
    uint32_t parity;      /*!< specifies the parity mode */
    uint32_t mode;        /*!< specifies whether the receive or transmit mode is enabled or disabled */
    uint32_t clkpolarity; /*!< specifies the steady state of the serial clock */
    uint32_t clkphase;    /*!< specifies the clock transition on which the bit capture is made */
    uint32_t clklastbit;  /*!< specifies whether the clock pulse corresponding to the last transmitted
                              data bit (MSB) has to be output on the SCLK pin in synchronous mode */
    uint32_t fifomode;    /*!< specifies whether the receive or transmit fifo is enabled or disabled*/
    uint32_t txthreshold; /*!< specifies USRT transmit FIFO threshold */
    uint32_t rxthreshold; /*!< specifies USRT receive FIFO threshold */
} hal_usrt_init_struct;

/* @PARA: usrt_dev */
/* @STRUCT: USRT device information definition */
typedef struct {
    uint32_t periph;                           /*!< USRT periph */
    hal_usrt_init_struct init;                 /*!< USRT init parameter */
    hal_usrt_irq_struct usrt_irq;              /*!< device interrupt callback function pointer */
    hal_dma_dev_struct *p_dma_rx;              /*!< DMA receive pointer */
    hal_dma_dev_struct *p_dma_tx;              /*!< DMA transmit pointer */
    hal_usrt_buffer_struct txbuffer;           /*!< transmit buffer */
    hal_usrt_buffer_struct rxbuffer;           /*!< receive buffer */
    uint16_t data_bit_mask;                    /*!< mask bit of data */
    __IO uint16_t last_error;                  /*!< the last error code */
    __IO uint16_t error_state;                 /*!< error state */
    __IO hal_usrt_run_state_enum state;        /*!< transmit receive state */
    void *transmit_half_complete_callback;     /*!< transmit half complete interrupt callback */
    void *transmit_complete_callback;          /*!< transmit complete interrupt callback */
    void *transmit_fifo_empty_callback;        /*!< transmit fifo empty interrupt callback */
    void *receive_half_complete_callback;      /*!< receive half complete interrupt callback */
    void *receive_complete_callback;           /*!< receive complete interrupt callback */
    void *receive_fifo_full_callback;          /*!< receive fifo full interrupt callback */
    void *transmit_received_complete_callback; /*!< transmit receive complete interrupt callback */
    void *abort_complete_callback;             /*!< abort complete interrupt callback */
    void *abort_tx_complete_callback;          /*!< abort transmit complete interrupt callback */
    void *abort_rx_complete_callback;          /*!< abort received complete interrupt callback */
    void *error_callback;                      /*!< error interrupt callback */
    hal_mutex_enum mutex;                      /*!< lock */
    void *priv;                                /*!< private pointer */
} hal_usrt_dev_struct;

typedef void (*hal_usrt_user_cb)(hal_usrt_dev_struct *usrt_dev);

typedef struct {
    __IO hal_usrt_user_cb transmit_half_complete_func;     /*!< transmit half complete interrupt callback */
    __IO hal_usrt_user_cb transmit_complete_func;          /*!< transmit complete interrupt callback */
    __IO hal_usrt_user_cb transmit_fifo_empty_func;        /*!< transmit fifo empty interrupt callback */
    __IO hal_usrt_user_cb abort_tx_complete_func;          /*!< abort transmit complete interrupt callback */
    __IO hal_usrt_user_cb receive_half_complete_func;      /*!< receive half interrupt callback */
    __IO hal_usrt_user_cb receive_complete_func;           /*!< receive complete interrupt callback */
    __IO hal_usrt_user_cb receive_fifo_full_func;          /*!< receive fifo full interrupt callback */
    __IO hal_usrt_user_cb transmit_received_complete_func; /*!< transmit receive complete interrupt callback */
    __IO hal_usrt_user_cb abort_rx_complete_func;          /*!< abort receive complete interrupt callback */
    __IO hal_usrt_user_cb abort_complete_func;             /*!< abort complete interrupt callback */
    __IO hal_usrt_user_cb error_func;                      /*!< error interrupt callback */
} hal_usrt_irq_user_callback_struct;

/* @STRUCT_MEMBER: baudrate */
/* @=NULL */

/* hal function declarations */
/* Initialization and de-initialization functions */
/* @FUNCTION: initialize USRT */
/* initialize usrt synchronous master mode */
int32_t hal_usrt_init(hal_usrt_dev_struct *usrt_dev, uint32_t periph, hal_usrt_init_struct *p_usrt);

/* @END */
/* initialize the usrt structure with the default values */
int32_t hal_usrt_struct_init(hal_usrt_struct_type_enum hal_struct_type, void *p_struct);
/* deinitialize the usrt */
int32_t hal_usrt_deinit(hal_usrt_dev_struct *usrt_dev);

/* usrt interrupt handle functions */
/* usrt interrupt handler content function,which is merely used in USART_IRQHandler */
int32_t hal_usrt_irq(hal_usrt_dev_struct *usrt_dev);
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_usrt_irq_handle_set(hal_usrt_dev_struct *usrt_dev, hal_usrt_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_usrt_irq_handle_all_reset(hal_usrt_dev_struct *usrt_dev);

/* transmit or receive functions */
/* transmit amounts of data, poll transmit process and completed status */
/* the function is blocking */
int32_t hal_usrt_transmit_poll(hal_usrt_dev_struct *usrt_dev, uint8_t *p_buffer, uint32_t length, uint32_t timeout);
/* receive amounts of data, poll receive process and completed status */
/* the function is blocking */
int32_t hal_usrt_receive_poll(hal_usrt_dev_struct *usrt_dev, uint8_t *p_buffer, uint32_t length, uint32_t timeout);
/* transmit & receive amounts of data, poll transfer process and completed status */
/* the function is blocking */
int32_t hal_usrt_transmit_receive_poll(hal_usrt_dev_struct *usrt_dev, uint8_t *p_txbuffer, uint8_t *p_rxbuffer, \
                                       uint32_t length, uint32_t timeout);
/* transmit amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_usrt_transmit_interrupt(hal_usrt_dev_struct *usrt_dev, uint8_t *p_buffer, uint32_t length, \
                                    hal_usrt_irq_user_callback_struct *p_func);
/* receive amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_usrt_receive_interrupt(hal_usrt_dev_struct *usrt_dev, uint8_t *p_buffer, uint32_t length, \
                                   hal_usrt_irq_user_callback_struct *p_func);
/* transmit & receive amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_usrt_transmit_receive_interrupt(hal_usrt_dev_struct *usrt_dev, uint8_t *p_txbuffer, uint8_t *p_rxbuffer, \
                                            uint16_t length, hal_usrt_irq_user_callback_struct *p_func);
/* transmit amounts of data by dma method */
/* the function is non-blocking */
int32_t hal_usrt_transmit_dma(hal_usrt_dev_struct *usrt_dev, uint8_t *p_buffer, uint16_t length, \
                              hal_usrt_irq_user_callback_struct *p_func);
/* receive amounts of data by dma method */
/* the function is non-blocking */
int32_t hal_usrt_receive_dma(hal_usrt_dev_struct *usrt_dev, uint8_t *p_buffer, uint16_t length, \
                             hal_usrt_irq_user_callback_struct *p_func);
/* transmit & receive amounts of data by dma method */
/* the function is non-blocking */
int32_t hal_usrt_transmit_receive_dma(hal_usrt_dev_struct *usrt_dev, uint8_t *p_txbuffer, uint8_t *p_rxbuffer, \
                                      uint16_t length, hal_usrt_irq_user_callback_struct *p_func);

/* transfer control functions */
/* pause usrt DMA transfer during transmission process */
int32_t hal_usrt_dma_pause(hal_usrt_dev_struct *usrt_dev);
/* resume usrt DMA transfer during transmission process */
int32_t hal_usrt_dma_resume(hal_usrt_dev_struct *usrt_dev);
/* stop usrt DMA transfer during transmission process */
int32_t hal_usrt_dma_stop(hal_usrt_dev_struct *usrt_dev);
/* abort usrt transmit and receive transfer */
int32_t hal_usrt_abort(hal_usrt_dev_struct *usrt_dev);

/* abort ongoing transmit transfer in interrupt method */
int32_t hal_usrt_transmit_abort_interrupt(hal_usrt_dev_struct *usrt_dev, hal_usrt_irq_user_callback_struct *p_func);
/* abort ongoing receive transfer in interrupt method */
int32_t hal_usrt_receive_abort_interrupt(hal_usrt_dev_struct *usrt_dev, hal_usrt_irq_user_callback_struct *p_func);
/* abort ongoing transfers in interrupt method */
int32_t hal_usrt_abort_interrupt(hal_usrt_dev_struct *usrt_dev, hal_usrt_irq_user_callback_struct *p_func);

/* USRT FIFO */
/* enable FIFO */
int32_t hal_usrt_fifo_enable(hal_usrt_dev_struct *usrt_dev);
/* disable FIFO */
int32_t hal_usrt_fifo_disable(hal_usrt_dev_struct *usrt_dev);
/* configure transmit FIFO threshold */
int32_t hal_usrt_transmit_fifo_threshold_config(hal_usrt_dev_struct *usrt_dev, uint32_t txthreshold);
/* configure receive FIFO threshold */
int32_t hal_usrt_receive_fifo_threshold_config(hal_usrt_dev_struct *usrt_dev, uint32_t rxthreshold);

/* return the usrt state */
hal_usrt_run_state_enum hal_usrt_state_get(hal_usrt_dev_struct *usrt_dev);
/* return the usrt error code */
uint16_t hal_usrt_error_get(hal_usrt_dev_struct *usrt_dev);

#endif /* GD32H7XX_HAL_USRT_H */
