/*!
    \file    gd32h7xx_hal_tmu.h
    \brief   definitions for the TMU

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_TMU_H
#define GD32H7XX_HAL_TMU_H

#include "gd32h7xx_hal.h"

/* TMU definitions */
#define TMU                         TMU_BASE                                    /*!< TMU periph */

/* registers definitions */
#define TMU_CS                      REG32(TMU + 0x00000000U)                    /*!< TMU control and status register */
#define TMU_IDATA                   REG32(TMU + 0x00000004U)                    /*!< TMU input data register */
#define TMU_ODATA                   REG32(TMU + 0x00000008U)                    /*!< TMU output data register */

/* bits definitions */
/* TMU_CS */
#define TMU_CS_MODE                 BITS(0,3)                                  /*!< TMU operation mode selection */
#define TMU_CS_ITRTNUM              BITS(4,7)                                  /*!< number of iterations selection */
#define TMU_CS_FACTOR               BITS(8,10)                                 /*!< scaling factor */
#define TMU_CS_RIE                  BIT(16)                                     /*!< read TMU_ODATA interrupt enable */
#define TMU_CS_RDEN                 BIT(17)                                     /*!< read TMU_ODATA DMA request enable */
#define TMU_CS_WDEN                 BIT(18)                                     /*!< write TMU_IDATA DMA request enable */
#define TMU_CS_ONUM                 BIT(19)                                     /*!< times the TMU_ODATA needs to be read */
#define TMU_CS_INUM                 BIT(20)                                     /*!< times the TMU_IDATA needs to be write */
#define TMU_CS_OWIDTH               BIT(21)                                     /*!< width of output data */
#define TMU_CS_IWIDTH               BIT(22)                                     /*!< width of input data */
#define TMU_CS_ENDF                 BIT(31)                                     /*!< end of TMU operation flag */

/* TMU_IDATA */
#define TMU_IDATA_IDATA             BITS(0,31)                                  /*!< the input data of TMU operation */

/* TMU_ODATA */
#define TMU_ODATA_ODATA             BITS(0,31)                                  /*!< the output data of TMU operation */

/* constants definitions */
#define TMU_MODE(regval)            (BITS(0,3) & ((uint32_t)(regval) << 0))     /* TMU mode definitions */
#define ITERATIONS(regval)          (BITS(4,7) & ((uint32_t)(regval) << 4))     /* TMU number of iterations definitions */
#define SCALE(regval)               (BITS(8,10) & ((uint32_t)(regval) << 8))    /* TMU scaling factor definitions */

/* TMU DMA read enable definitions */
#define TMU_READ_DMA_DISABLE        ((uint32_t)0x00000000U)                     /*!< disable DMA request to read TMU_ODATA */
#define TMU_READ_DMA_ENABLE         TMU_CS_RDEN                                 /*!< enable DMA request to read TMU_ODATA */

/* TMU DMA write enable definitions */
#define TMU_WRITE_DMA_DISABLE       ((uint32_t)0x00000000U)                     /*!< disable DMA request to write TMU_IDATA */
#define TMU_WRITE_DMA_ENABLE        TMU_CS_WDEN                                 /*!< enable DMA request to write TMU_IDATA */

/* @STRUCT_MEMBER: mode */
/* @DEFINE: TMU mode definitions */
#define TMU_MODE_COS                TMU_MODE(0)                                 /*!< mode0: m*cos */
#define TMU_MODE_SIN                TMU_MODE(1)                                 /*!< mode1: m*sin */
#define TMU_MODE_ATAN2              TMU_MODE(2)                                 /*!< mode2: atan2(y,x) */
#define TMU_MODE_MODULUS            TMU_MODE(3)                                 /*!< mode3: sqrt(x^2+y^2) */
#define TMU_MODE_ATAN               TMU_MODE(4)                                 /*!< mode4: atan(x) */
#define TMU_MODE_COSH               TMU_MODE(5)                                 /*!< mode5: cosh(x) */
#define TMU_MODE_SINH               TMU_MODE(6)                                 /*!< mode6: sinh(x) */
#define TMU_MODE_ATANH              TMU_MODE(7)                                 /*!< mode7: atanh(x) */
#define TMU_MODE_LN                 TMU_MODE(8)                                 /*!< mode8: ln(x) */
#define TMU_MODE_SQRT               TMU_MODE(9)                                 /*!< mode9: sqrt(x) */

/* @STRUCT_MEMBER: iteration */
/* @DEFINE: TMU number of iterations definitions */
#define TMU_ITERATION_STEPS_4       ITERATIONS(1)                               /*!< 4 iteration steps */
#define TMU_ITERATION_STEPS_8       ITERATIONS(2)                               /*!< 8 iteration steps */
#define TMU_ITERATION_STEPS_12      ITERATIONS(3)                               /*!< 12 iteration steps */
#define TMU_ITERATION_STEPS_16      ITERATIONS(4)                               /*!< 16 iteration steps */
#define TMU_ITERATION_STEPS_20      ITERATIONS(5)                               /*!< 20 iteration steps */
#define TMU_ITERATION_STEPS_24      ITERATIONS(6)                               /*!< 24 iteration steps */

/* @STRUCT_MEMBER: factor */
/* @DEFINE: TMU scaling factor definitions */
#define TMU_SCALING_FACTOR_1        SCALE(0)                                    /*!< scaling factor = 1 */
#define TMU_SCALING_FACTOR_2        SCALE(1)                                    /*!< scaling factor = 2 */
#define TMU_SCALING_FACTOR_4        SCALE(2)                                    /*!< scaling factor = 4 */
#define TMU_SCALING_FACTOR_8        SCALE(3)                                    /*!< scaling factor = 8 */
#define TMU_SCALING_FACTOR_16       SCALE(4)                                    /*!< scaling factor = 16 */
#define TMU_SCALING_FACTOR_32       SCALE(5)                                    /*!< scaling factor = 32 */
#define TMU_SCALING_FACTOR_64       SCALE(6)                                    /*!< scaling factor = 64 */
#define TMU_SCALING_FACTOR_128      SCALE(7)                                    /*!< scaling factor = 128 */

/* @STRUCT_MEMBER: read_times */
/* @DEFINE: TMU_ODATA read times definitions */
#define TMU_READ_TIMES_1            ((uint32_t)0x00000000U)                     /*!< one 32-bit read operation */
#define TMU_READ_TIMES_2            TMU_CS_ONUM                                 /*!< two 32-bit read operation */

/* @STRUCT_MEMBER: write_times */
/* @DEFINE: TMU_IDATA write times definitions */
#define TMU_WRITE_TIMES_1           ((uint32_t)0x00000000U)                     /*!< one 32-bit write operation */
#define TMU_WRITE_TIMES_2           TMU_CS_INUM                                 /*!< two 32-bit write operation */

/* @STRUCT_MEMBER: output_width */
/* @DEFINE: TMU output data width definitions */
#define TMU_OUTPUT_WIDTH_32         ((uint32_t)0x00000000U)                     /*!< TMU_ODATA contains the output data in q1.31 format */
#define TMU_OUTPUT_WIDTH_16         TMU_CS_OWIDTH                               /*!< TMU_ODATA contains the output data in q1.15 format */

/* @STRUCT_MEMBER: input_width */
/* @DEFINE: TMU input data width definitions */
#define TMU_INPUT_WIDTH_32          ((uint32_t)0x00000000U)                     /*!< TMU_IDATA contains the input data in q1.31 format */
#define TMU_INPUT_WIDTH_16          TMU_CS_IWIDTH                               /*!< TMU_IDATA contains the input data in q1.15 format */

#define TMU_DMA_DIR_NONE            ((uint32_t)0x00000000U)                     /*!< DMA none direction */
#define TMU_DMA_DIR_IN              ((uint32_t)0x00000001U)                     /*!< DMA input direction */
#define TMU_DMA_DIR_OUT             ((uint32_t)0x00000002U)                     /*!< DMA output direction */
#define TMU_DMA_DIR_IN_OUT          ((uint32_t)0x00000003U)                     /*!< DMA input and output direction */

#define HAL_TMU_ERROR_NONE          ((uint32_t)0x00000000U)                     /*!< no error */
#define HAL_TMU_ERROR_PARAM         ((uint32_t)0x00000001U)                     /*!< wrong parameter error */
#define HAL_TMU_ERROR_NOT_READY     ((uint32_t)0x00000002U)                     /*!< peripheral not ready */
#define HAL_TMU_ERROR_TIMEOUT       ((uint32_t)0x00000004U)                     /*!< timeout error */
#define HAL_TMU_ERROR_DMA           ((uint32_t)0x00000008U)                     /*!< DMA error */

/* TMU structure type enum */
typedef enum {
    HAL_TMU_INIT_STRUCT = 0U,        /*!< TMU initialization structure */
    HAL_TMU_DEV_STRUCT,              /*!< TMU device structure */
    HAL_TMU_IRQ_INIT_STRUCT,         /*!< interrupt callback initialization structure */
    HAL_TMU_IRQ_USER_CALLBACK_STRUCT /*!< user callback initialization structure */
} hal_tmu_struct_type_enum;

/* TMU state enum definition */
typedef enum {
    HAL_TMU_STATE_RESET = 0x00U, /*!< TMU not yet initialized or disabled */
    HAL_TMU_STATE_READY,         /*!< TMU initialized and ready for used */
    HAL_TMU_STATE_BUSY,          /*!< TMU internal process is ongoing */
    HAL_TMU_STATE_ERROR          /*!< TMU error state */
} hal_tmu_state_enum;

/* TMU device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb error_handle;         /*!< TMU calc error handler function */
    __IO hal_irq_handle_cb calc_complete_handle; /*!< TMU calc complete handler function */
} hal_tmu_irq_struct;

/* @PARA: p_init */
/* @STRUCT: TMU init parameter struct definitions */
typedef struct {
    uint32_t mode;         /*!< mode of TMU operation */
    uint32_t output_width; /*!< width of output data */
    uint32_t input_width;  /*!< width of input data */
    uint32_t read_times;   /*!< times the TMU_ODATA needs to be read */
    uint32_t write_times;  /*!< times the TMU_IDATA needs to be write */
    uint32_t iteration;    /*!< number of iterations selection */
    uint32_t factor;       /*!< scaling factor */
} hal_tmu_init_struct;

/* @PARA: tmu_dev */
/* @STRUCT: TMU device structure definition */
typedef struct {
    hal_tmu_init_struct init;          /*!< TMU init parameters */
    hal_tmu_irq_struct tmu_irq;        /*!< TMU interrupt callback */
    uint32_t *input_buffer;            /*!< pointer to TMU input data buffer */
    uint32_t *output_buffer;           /*!< pointer to TMU output data buffer */
    uint32_t order_number;             /*!< remain number of calculation to order */
    uint32_t get_number;               /*!< remain number of calculation result to get */
    uint32_t dma_direction;            /*!< TMU DMA transfers direction */
    hal_dma_dev_struct *p_dma_in;      /*!< input data DMA pointer */
    hal_dma_dev_struct *p_dma_out;     /*!< output data DMA pointer */
    hal_tmu_state_enum state;          /*!< TMU device state */
    uint32_t error_state;              /*!< TMU error state */
    hal_mutex_enum mutex;              /*!< lock */
    void *priv;                        /*!< priv data */
    void *calculate_complete_callback; /*!< calculate complete interrupt callback */
    void *dma_error_callback;          /*!< DMA error callback */
} hal_tmu_dev_struct;

typedef void (*hal_tmu_user_cb)(hal_tmu_dev_struct *tmu_dev);

typedef struct {
    __IO hal_tmu_user_cb calculate_complete_callback; /*!< calculate complete interrupt callback */
    __IO hal_tmu_user_cb dma_error_callback;          /*!< DMA error callback */
} hal_tmu_irq_user_callback_struct;

/* hal function declarations */
/* @FUNCTION: initialize TMU */
int32_t hal_tmu_init(hal_tmu_dev_struct *tmu_dev, hal_tmu_init_struct *p_init);
/* @END */
/* initialize the TMU structure with the default values */
int32_t hal_tmu_struct_init(hal_tmu_struct_type_enum hal_struct_type, void *p_struct);
/* deinitialize the TMU */
int32_t hal_tmu_deinit(hal_tmu_dev_struct *tmu_dev);

/* TMU interrupt handle functions */
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_tmu_irq_handle_set(hal_tmu_dev_struct *tmu_dev, hal_tmu_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
   which will be registered and called when corresponding interrupt be triggered */
int32_t hal_tmu_irq_handle_all_reset(hal_tmu_dev_struct *tmu_dev);
/* TMU interrupt handler content function,which is merely used in TMU_IRQHandler */
void hal_tmu_irq(hal_tmu_dev_struct *tmu_dev);

/* TMU calc data, poll completed status */
/* the function is blocking */
int32_t hal_tmu_calculate_poll(hal_tmu_dev_struct *tmu_dev, \
                               uint32_t *indata, uint32_t *outdata, \
                               uint32_t calc_num, uint32_t timeout);
/* TMU calc data by zero cost method */
/* the function is non-blocking */
int32_t hal_tmu_calculate_zero(hal_tmu_dev_struct *tmu_dev, \
                               uint32_t *indata, uint32_t *outdata, \
                               uint32_t calc_num, uint32_t timeout);
/* TMU calc data by interrupt method */
/* the function is non-blocking */
int32_t hal_tmu_calculate_interrupt(hal_tmu_dev_struct *tmu_dev, \
                                    uint32_t *indata, uint32_t *outdata, \
                                    uint32_t calc_num);
/* TMU calc data by dma method */
/* the function is non-blocking */
int32_t hal_tmu_calculate_dma(hal_tmu_dev_struct *tmu_dev, \
                              uint32_t *indata, uint32_t *outdata, \
                              uint32_t calc_num, uint32_t dma_direction, \
                              hal_tmu_irq_user_callback_struct *p_user_func);

/* return the TMU state */
hal_tmu_state_enum hal_tmu_state_get(hal_tmu_dev_struct *tmu_dev);
/* return the TMU error code */
uint32_t hal_tmu_error_get(hal_tmu_dev_struct *tmu_dev);

#endif /* GD32H7XX_HAL_TMU_H */
