/*!
    \file    gd32h7xx_hal_sdio_sdcard.h
    \brief   definitions for the SDIO SDCARD

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_SDIO_SDCARD_H
#define GD32H7XX_HAL_SDIO_SDCARD_H

#include "gd32h7xx_hal.h"

/* SD memory card bus commands index */
#define SD_CMD_GO_IDLE_STATE                ((uint8_t)0U)                       /* CMD0, GO_IDLE_STATE */
#define SD_CMD_ALL_SEND_CID                 ((uint8_t)2U)                       /* CMD2, ALL_SEND_CID */
#define SD_CMD_SEND_RELATIVE_ADDR           ((uint8_t)3U)                       /* CMD3, SEND_RELATIVE_ADDR */
#define SD_CMD_SET_DSR                      ((uint8_t)4U)                       /* CMD4, SET_DSR */
#define SD_CMD_SWITCH_FUNC                  ((uint8_t)6U)                       /* CMD6, SWITCH_FUNC */
#define SD_CMD_SELECT_DESELECT_CARD         ((uint8_t)7U)                       /* CMD7, SELECT_DESELECT_CARD */
#define SD_CMD_SEND_IF_COND                 ((uint8_t)8U)                       /* CMD8, SEND_IF_COND */
#define SD_CMD_SEND_CSD                     ((uint8_t)9U)                       /* CMD9, SEND_CSD */
#define SD_CMD_SEND_CID                     ((uint8_t)10U)                      /* CMD10, SEND_CID */
#define SD_CMD_VOLATAGE_SWITCH              ((uint8_t)11U)                      /* CMD11, VOLATAGE_SWITCH */
#define SD_CMD_STOP_TRANSMISSION            ((uint8_t)12U)                      /* CMD12, STOP_TRANSMISSION */
#define SD_CMD_SEND_STATUS                  ((uint8_t)13U)                      /* CMD13, SEND_STATUS */
#define SD_CMD_GO_INACTIVE_STATE            ((uint8_t)15U)                      /* CMD15, GO_INACTIVE_STATE */
#define SD_CMD_SET_BLOCKLEN                 ((uint8_t)16U)                      /* CMD16, SET_BLOCKLEN */
#define SD_CMD_READ_SINGLE_BLOCK            ((uint8_t)17U)                      /* CMD17, READ_SINGLE_BLOCK */
#define SD_CMD_READ_MULTIPLE_BLOCK          ((uint8_t)18U)                      /* CMD18, READ_MULTIPLE_BLOCK */
#define SD_SEND_TUNING_PATTERN              ((uint8_t)19U)                      /* CMD19, SEND_TUNING_PATTERN */
#define SD_CMD_WRITE_BLOCK                  ((uint8_t)24U)                      /* CMD24, WRITE_BLOCK */
#define SD_CMD_WRITE_MULTIPLE_BLOCK         ((uint8_t)25U)                      /* CMD25, WRITE_MULTIPLE_BLOCK */
#define SD_CMD_PROG_CSD                     ((uint8_t)27U)                      /* CMD27, PROG_CSD */
#define SD_CMD_SET_WRITE_PROT               ((uint8_t)28U)                      /* CMD28, SET_WRITE_PROT */
#define SD_CMD_CLR_WRITE_PROT               ((uint8_t)29U)                      /* CMD29, CLR_WRITE_PROT */
#define SD_CMD_SEND_WRITE_PROT              ((uint8_t)30U)                      /* CMD30, SEND_WRITE_PROT */
#define SD_CMD_ERASE_WR_BLK_START           ((uint8_t)32U)                      /* CMD32, ERASE_WR_BLK_START */
#define SD_CMD_ERASE_WR_BLK_END             ((uint8_t)33U)                      /* CMD33, ERASE_WR_BLK_END */
#define SD_CMD_ERASE                        ((uint8_t)38U)                      /* CMD38, ERASE */
#define SD_CMD_LOCK_UNLOCK                  ((uint8_t)42U)                      /* CMD42, LOCK_UNLOCK */
#define SD_CMD_APP_CMD                      ((uint8_t)55U)                      /* CMD55, APP_CMD */
#define SD_CMD_GEN_CMD                      ((uint8_t)56U)                      /* CMD56, GEN_CMD */

/* SD memory card application specific commands index */
#define SD_APPCMD_SET_BUS_WIDTH             ((uint8_t)6U)                       /* ACMD6, SET_BUS_WIDTH */
#define SD_APPCMD_SD_STATUS                 ((uint8_t)13U)                      /* ACMD13, SD_STATUS */
#define SD_APPCMD_SEND_NUM_WR_BLOCKS        ((uint8_t)22U)                      /* ACMD22, SEND_NUM_WR_BLOCKS */
#define SD_APPCMD_SET_WR_BLK_ERASE_COUNT    ((uint8_t)23U)                      /* ACMD23, SET_WR_BLK_ERASE_COUNT */
#define SD_APPCMD_SD_SEND_OP_COND           ((uint8_t)41U)                      /* ACMD41, SD_SEND_OP_COND */
#define SD_APPCMD_SET_CLR_CARD_DETECT       ((uint8_t)42U)                      /* ACMD42, SET_CLR_CARD_DETECT */
#define SD_APPCMD_SEND_SCR                  ((uint8_t)51U)                      /* ACMD51, SEND_SCR */

/* card command class */
#define SD_CCC_SWITCH                       BIT(10)                             /* class 10 */
#define SD_CCC_IO_MODE                      BIT(9)                              /* class 9 */
#define SD_CCC_APPLICATION_SPECIFIC         BIT(8)                              /* class 8 */
#define SD_CCC_LOCK_CARD                    BIT(7)                              /* class 7 */
#define SD_CCC_WRITE_PROTECTION             BIT(6)                              /* class 6 */
#define SD_CCC_ERASE                        BIT(5)                              /* class 5 */
#define SD_CCC_BLOCK_WRITE                  BIT(4)                              /* class 4 */
#define SD_CCC_BLOCK_READ                   BIT(2)                              /* class 2 */
#define SD_CCC_BASIC                        BIT(0)                              /* class 0 */

/* lock unlock status */
#define SD_LOCK                             ((uint8_t)0x05U)                    /* lock the SD card */
#define SD_UNLOCK                           ((uint8_t)0x02U)                    /* unlock the SD card */

#define SD_SDSC                             ((uint32_t)0x00000000U)             /*!< SD Standard Capacity <2G*/
#define SD_SDHC_SDXC                        ((uint32_t)0x00000001U)             /*!< SD High Capacity <32G, SD Extended Capacity <2T */

/* card status of R1 definitions */
#define SD_R1_OUT_OF_RANGE                  BIT(31)                             /* command's argument was out of the allowed range */
#define SD_R1_ADDRESS_ERROR                 BIT(30)                             /* misaligned address which did not match the block length */
#define SD_R1_BLOCK_LEN_ERROR               BIT(29)                             /* transferred block length is not allowed */
#define SD_R1_ERASE_SEQ_ERROR               BIT(28)                             /* an error in the sequence of erase commands occurred */
#define SD_R1_ERASE_PARAM                   BIT(27)                             /* an invalid selection of write-blocks for erase occurred */
#define SD_R1_WP_VIOLATION                  BIT(26)                             /* the host attempts to write to a protected block or to the temporary or permanent write protected card */
#define SD_R1_CARD_IS_LOCKED                BIT(25)                             /* the card is locked by the host */
#define SD_R1_LOCK_UNLOCK_FAILED            BIT(24)                             /* a sequence or password error has been detected in lock/unlock card command */
#define SD_R1_COM_CRC_ERROR                 BIT(23)                             /* CRC check of the previous command failed */
#define SD_R1_ILLEGAL_COMMAND               BIT(22)                             /* command not legal for the card state */
#define SD_R1_CARD_ECC_FAILED               BIT(21)                             /* card internal ECC was applied but failed to correct the data */
#define SD_R1_CC_ERROR                      BIT(20)                             /* internal card controller error */
#define SD_R1_GENERAL_UNKNOWN_ERROR         BIT(19)                             /* a general or an unknown error occurred during the operation */
#define SD_R1_CSD_OVERWRITE                 BIT(16)                             /* read only section of the CSD does not match or attempt to reverse the copy or permanent WP bits */
#define SD_R1_WP_ERASE_SKIP                 BIT(15)                             /* partial address space was erased */
#define SD_R1_CARD_ECC_DISABLED             BIT(14)                             /* command has been executed without using the internal ECC */
#define SD_R1_ERASE_RESET                   BIT(13)                             /* an erase sequence was cleared before executing */
#define SD_R1_READY_FOR_DATA                BIT(8)                              /* correspond to buffer empty signaling on the bus */
#define SD_R1_APP_CMD                       BIT(5)                              /* card will expect ACMD */
#define SD_R1_AKE_SEQ_ERROR                 BIT(3)                              /* error in the sequence of the authentication process */
#define SD_R1_ERROR_BITS                    ((uint32_t)0xFDF9E008U)             /* all the R1 error bits */

/* card status of R6 definitions */
#define SD_R6_COM_CRC_ERROR                 BIT(15)                             /* CRC check of the previous command failed */
#define SD_R6_ILLEGAL_COMMAND               BIT(14)                             /* command not legal for the card state */
#define SD_R6_GENERAL_UNKNOWN_ERROR         BIT(13)                             /* a general or an unknown error occurred during the operation */

/* card state */
#define SD_CARDSTATE_IDLE                   ((uint8_t)0x00U)                    /* card is in idle state */
#define SD_CARDSTATE_READY                  ((uint8_t)0x01U)                    /* card is in ready state */
#define SD_CARDSTATE_IDENTIFICAT            ((uint8_t)0x02U)                    /* card is in identification state */
#define SD_CARDSTATE_STANDBY                ((uint8_t)0x03U)                    /* card is in standby state */
#define SD_CARDSTATE_TRANSFER               ((uint8_t)0x04U)                    /* card is in transfer state */
#define SD_CARDSTATE_DATA                   ((uint8_t)0x05U)                    /* card is in data sending state */
#define SD_CARDSTATE_RECEIVING              ((uint8_t)0x06U)                    /* card is in receiving state */
#define SD_CARDSTATE_PROGRAMMING            ((uint8_t)0x07U)                    /* card is in programming state */
#define SD_CARDSTATE_DISCONNECT             ((uint8_t)0x08U)                    /* card is in disconnect state */
#define SD_CARDSTATE_LOCKED                 ((uint32_t)0x02000000U)             /* card is in locked state */

#define SD_CHECK_PATTERN                    ((uint32_t)0x000001AAU)             /* check pattern for CMD8 */
#define SD_VOLTAGE_WINDOW                   ((uint32_t)0x80100000U)             /* host 3.3V request in ACMD41 */
#define SD_VOLTAGE_18V                      ((uint32_t)0x01000000U)             /* host 1.8V request in ACMD41 */

/* parameters for ACMD41(voltage validation) */
#define SD_HIGH_CAPACITY                    ((uint32_t)0x40000000U)             /* high capacity SD memory card */
#define SD_STD_CAPACITY                     ((uint32_t)0x00000000U)             /* standard capacity SD memory card */

/* SD bus width, check SCR register */
#define SD_BUS_WIDTH_4BIT                   ((uint32_t)0x00040000U)             /* 4-bit width bus mode */
#define SD_BUS_WIDTH_1BIT                   ((uint32_t)0x00010000U)             /* 1-bit width bus mode */

/* masks for SCR register */
#define SD_MASK_0_7BITS                     ((uint32_t)0x000000FFU)             /* mask [7:0] bits */
#define SD_MASK_8_15BITS                    ((uint32_t)0x0000FF00U)             /* mask [15:8] bits */
#define SD_MASK_16_23BITS                   ((uint32_t)0x00FF0000U)             /* mask [23:16] bits */
#define SD_MASK_24_31BITS                   ((uint32_t)0xFF000000U)             /* mask [31:24] bits */

#define SD_FIFOHALF_WORDS                   ((uint32_t)0x00000008U)             /* words of FIFO half full/empty */
#define SD_FIFOHALF_BYTES                   ((uint32_t)0x00000020U)             /* bytes of FIFO half full/empty */

#define SD_DATATIMEOUT                      ((uint32_t)0xFFFF0000U)             /* DSM data timeout */
#define SD_MAX_VOLT_VALIDATION              ((uint32_t)0x0000FFFFU)             /* the maximum times of voltage validation */
#define SD_MAX_DATA_LENGTH                  ((uint32_t)0x01FFFFFFU)             /* the maximum length of data */
#define SD_ALLZERO                          ((uint32_t)0x00000000U)             /* all zero */
#define SD_RCA_SHIFT                        ((uint8_t)0x10U)                    /* RCA shift bits */

/* user can according to need to change the macro values */
#define SD_CLK_DIV_INIT                     ((uint32_t)0x000001F4U)             /* SD clock division in initialization phase */
#define SD_CLK_DIV_TRANS_DSPEED             ((uint32_t)0x00000008U)             /* SD clock division in default speed transmission phase */
#define SD_CLK_DIV_TRANS_HSPEED             ((uint32_t)0x00000004U)             /* SD clock division in high speed transmission phase */
#define SD_CLK_DIV_TRANS_SDR25SPEED         ((uint32_t)0x00000004U)             /* SD clock division in SDR25 high speed transmission phase */
#define SD_CLK_DIV_TRANS_SDR50SPEED         ((uint32_t)0x00000002U)             /* SD clock division in SDR50 high speed transmission phase */
#define SD_CLK_DIV_TRANS_SDR104SPEED        ((uint32_t)0x00000001U)             /* SD clock division in SDR104 high speed transmission phase */
#define SD_CLK_DIV_TRANS_DDR50SPEED         ((uint32_t)0x00000004U)             /* SD clock division in DDR50 high speed transmission phase */

#define SD_TRANSFER_NONE                    ((uint32_t)0x00000000U)             /*!< none */
#define SD_TRANSFER_READ_SINGLE_BLOCK       ((uint32_t)0x00000001U)             /*!< read single block operation */
#define SD_TRANSFER_READ_MULTIPLE_BLOCK     ((uint32_t)0x00000002U)             /*!< read multiple blocks operation */
#define SD_TRANSFER_WRITE_SINGLE_BLOCK      ((uint32_t)0x00000010U)             /*!< write single block operation */
#define SD_TRANSFER_WRITE_MULTIPLE_BLOCK    ((uint32_t)0x00000020U)             /*!< write multiple blocks operation */
#define SD_TRANSFER_IT                      ((uint32_t)0x00000008U)             /*!< process in interrupt mode */
#define SD_TRANSFER_DMA                     ((uint32_t)0x00000080U)             /*!< process in DMA mode */

#define SDCAED_VOLTAGE_18V                  ((uint32_t)0x00000000U)             /*!< sdio use 1.8v voltage */
#define SDCAED_VOLTAGE_33V                  ((uint32_t)0x00000001U)             /*!< sdio use 3.3v voltage */

/* SD struct initialization type enum */
typedef enum {
    HAL_SD_INIT_STRUCT = 0U,     /*!< initialization structure */
    HAL_SD_DEV_STRUCT,           /*!< device information structure */
    HAL_SD_IRQ_INIT_STRUCT,      /*!< interrupt callback initialization structure */
    HAL_SD_CARD_CID_STRUCT,      /*!< card identification structure */
    HAL_SD_CARD_CSD_STRUCT,      /*!< card identification structure */
    HAL_SD_CARDINFO_STRUCT,      /*!< card information structure */
    HAL_SD_IRQ_USER_STRUCT      /*!< user interrupt callback structure */
} hal_sdio_sdcard_struct_type_enum;

/* SD state enum */
typedef enum {
    HAL_SD_STATE_RESET       = ((uint32_t)0x00000000U), /*!< SD not yet initialized or disabled  */
    HAL_SD_STATE_READY       = ((uint32_t)0x00000001U), /*!< SD initialized and ready for use    */
    HAL_SD_STATE_TIMEOUT     = ((uint32_t)0x00000002U), /*!< SD timeout state                    */
    HAL_SD_STATE_BUSY        = ((uint32_t)0x00000003U), /*!< SD process ongoing                  */
    HAL_SD_STATE_PROGRAMMING = ((uint32_t)0x00000004U), /*!< SD programming State                */
    HAL_SD_STATE_RECEIVING   = ((uint32_t)0x00000005U), /*!< SD receiving State                  */
    HAL_SD_STATE_TRANSFER    = ((uint32_t)0x00000006U), /*!< SD transfer State                   */
    HAL_SD_STATE_ERROR       = ((uint32_t)0x0000000FU)  /*!< SD is in error state                */
} hal_sdio_sdcard_state_enum;

/* SD error flags */
typedef enum {
    HAL_SD_OUT_OF_RANGE = 0U,     /* command's argument was out of range */
    HAL_SD_ADDRESS_ERROR,         /* misaligned address which did not match the block length */
    HAL_SD_BLOCK_LEN_ERROR,       /* transferred block length is not allowed for the card or the number of transferred bytes
                                     does not match the block length */
    HAL_SD_ERASE_SEQ_ERROR,       /* an error in the sequence of erase command occurs */
    HAL_SD_ERASE_PARAM,           /* an invalid selection of write-blocks for erase occurred */
    HAL_SD_WP_VIOLATION,          /* attempt to program a write protect block or permanent write protected card */
    HAL_SD_LOCK_UNLOCK_FAILED,    /* sequence or password error has been detected in lock/unlock card command */
    HAL_SD_COM_CRC_ERROR,         /* CRC check of the previous command failed */
    HAL_SD_ILLEGAL_COMMAND,       /* command not legal for the card state */
    HAL_SD_CARD_ECC_FAILED,       /* card internal ECC was applied but failed to correct the data */
    HAL_SD_CC_ERROR,              /* internal card controller error */
    HAL_SD_GENERAL_UNKNOWN_ERROR, /* general or unknown error occurred during the operation */
    HAL_SD_CSD_OVERWRITE,         /* read only section of the CSD does not match the card content or an attempt to reverse the
                                     copy or permanent WP bits was made */
    HAL_SD_WP_ERASE_SKIP,         /* only partial address space was erased or the temporary or permanent write protected card
                                     was erased */
    HAL_SD_CARD_ECC_DISABLED,     /* command has been executed without using internal ECC */
    HAL_SD_ERASE_RESET,           /* erase sequence was cleared before executing because an out of erase sequence command was
                                     received */
    HAL_SD_AKE_SEQ_ERROR,         /* error in the sequence of the authentication process */
    HAL_SD_CMD_CRC_ERROR,         /* command response received (CRC check failed) */
    HAL_SD_DATA_CRC_ERROR,        /* data block sent/received (CRC check failed) */
    HAL_SD_CMD_RESP_TIMEOUT,      /* command response timeout */
    HAL_SD_DATA_TIMEOUT,          /* data timeout */
    HAL_SD_TX_UNDERRUN_ERROR,     /* transmit FIFO underrun error occurs */
    HAL_SD_RX_OVERRUN_ERROR,      /* received FIFO overrun error occurs */
    HAL_SD_START_BIT_ERROR,       /* start bit error in the bus */
    HAL_SD_VOLTRANGE_INVALID,     /* the voltage range is invalid */
    HAL_SD_PARAMETER_INVALID,     /* the parameter is invalid */
    HAL_SD_OPERATION_IMPROPER,    /* the operation is improper */
    HAL_SD_FUNCTION_UNSUPPORTED,  /* the function is unsupported */
    HAL_SD_ERROR,                 /* an error occurred */
    HAL_SD_DMA_ERROR,             /* an error when idma transfer */
    HAL_SD_OK                     /* no error occurred */
} hal_sdio_sdcard_error_enum;

/* supported memory cards types */
typedef enum {
    SDIO_STD_CAPACITY_SD_CARD_V1_1 = 0U, /* standard capacity SD card version 1.1 */
    SDIO_STD_CAPACITY_SD_CARD_V2_0,      /* standard capacity SD card version 2.0 */
    SDIO_STD_CAPACITY_SD_CARD_V3_0,      /* standard capacity SD card version 3.0 */
    SDIO_HIGH_CAPACITY_SD_CARD,          /* high capacity SD card */
    SDIO_SECURE_DIGITAL_IO_CARD,         /* secure digital IO card */
    SDIO_SECURE_DIGITAL_IO_COMBO_CARD,   /* secure digital IO combo card */
    SDIO_MULTIMEDIA_CARD,                /* multimedia card */
    SDIO_HIGH_CAPACITY_MULTIMEDIA_CARD,  /* high capacity multimedia card */
    SDIO_HIGH_SPEED_MULTIMEDIA_CARD      /* high speed multimedia card */
} hal_sdio_sdcard_card_type_enum;

/* card identification (CID) register */
typedef struct {
    __IO uint8_t mid;     /* manufacturer ID */
    __IO uint16_t oid;    /* OEM/application ID */
    __IO uint32_t pnm0;   /* product name */
    __IO uint8_t pnm1;    /* product name */
    __IO uint8_t prv;     /* product revision */
    __IO uint32_t psn;    /* product serial number */
    __IO uint16_t mdt;    /* manufacturing date */
    __IO uint8_t cid_crc; /* CID CRC7 checksum */
} hal_sdio_sdcard_card_cid_struct;

/* CSD register (CSD version 1.0 and 2.0) */
typedef struct {
    __IO uint8_t csd_struct;         /* CSD struct */
    __IO uint8_t taac;               /* data read access-time */
    __IO uint8_t nsac;               /* data read access-time in CLK cycles */
    __IO uint8_t tran_speed;         /* max. data transfer rate */
    __IO uint16_t ccc;               /* card command classes */
    __IO uint8_t read_bl_len;        /* max. read data block length */
    __IO uint8_t read_bl_partial;    /* partial blocks for read allowed */
    __IO uint8_t write_blk_misalign; /* write block misalignment */
    __IO uint8_t read_blk_misalign;  /* read block misalignment */
    __IO uint8_t dsp_imp;            /* DSR implemented */
    __IO uint32_t c_size;            /* device size, 12 bits in CSD version 1.0, 22 bits in CSD version 2.0 */
    __IO uint8_t vdd_r_curr_min;     /* max.read current @VDD min, CSD version 1.0 */
    __IO uint8_t vdd_r_curr_max;     /* max.read current @VDD max, CSD version 1.0 */
    __IO uint8_t vdd_w_curr_min;     /* max.write current @VDD min, CSD version 1.0 */
    __IO uint8_t vdd_w_curr_max;     /* max.write current @VDD max, CSD version 1.0 */
    __IO uint8_t c_size_mult;        /* device size multiplier, CSD version 1.0 */
    __IO uint8_t erase_blk_en;       /* erase single block enable */
    __IO uint8_t sector_size;        /* erase sector size */
    __IO uint8_t wp_grp_size;        /* write protect group size */
    __IO uint8_t wp_grp_enable;      /* write protect group enable */
    __IO uint8_t r2w_factor;         /* write speed factor */
    __IO uint8_t write_bl_len;       /* max. write data block length */
    __IO uint8_t write_bl_partial;   /* partial blocks for write allowed */
    __IO uint8_t file_format_grp;    /* file format group */
    __IO uint8_t copy_flag;          /* copy flag (OTP) */
    __IO uint8_t perm_write_protect; /* permanent write protection */
    __IO uint8_t tmp_write_protect;  /* temporary write protection */
    __IO uint8_t file_format;        /* file format */
    __IO uint8_t csd_crc;            /* CSD CRC checksum */
} hal_sdio_sdcard_card_csd_struct;

/* information of card */
typedef struct {
    uint32_t card_type;              /* card type */
    uint32_t card_speed;             /* card speed */
    uint16_t card_rca;               /* card relative card address */
    uint32_t card_blocknum;          /* card capacity */
    uint32_t card_blocksize;         /* card block size */
    uint32_t logic_blocknum;         /* card capacity */
    uint32_t logic_blocksize;        /* card block size */
} hal_sdio_sdcard_cardinfo_struct;

/* SDIO device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb transmit_fifo_half_empty_handle; /*!< SDIO transmit fifo half empty callback function */
    __IO hal_irq_handle_cb receive_fifo_half_empty_handle;  /*!< SDIO receive fifo half empty callback function */
    __IO hal_irq_handle_cb transfer_complete_handle;        /*!< SDIO transfer complete callback function */
    __IO hal_irq_handle_cb idma_complete_handle;            /*!< SDIO idma complete callback function */
    __IO hal_irq_handle_cb transfer_error_handle;           /*!< SDIO transfer error callback function */
} hal_sdio_sdcard_irq_struct;

/* @PARA: p_sd_init */
/* @STRUCT: sd card init structure */
typedef struct {
    uint32_t bus_width;        /*!< SDIO bus width mode */
    uint32_t clock_division;   /*!< SDIO clock division value */
    uint32_t clock_edge;       /*!< SDIO clock edge value */
    uint32_t clock_powersave;  /*!< SDIO clock dynamic switch on/off value */
    uint32_t bus_speed_mode;   /*!< SDIO card bus speed control value */
    uint32_t data_rate;        /*!< SDIO data rate value */
    uint32_t hardware_control; /*!< SDIO hardware control */
    uint32_t transceiver;      /*!< SDIO transceiver 1.8V mode */
} hal_sdio_sdcard_init_struct;

/* @STRUCT_MEMBER: transceiver */
/* @SP: ENABLE/DISABLE */

typedef void (*transceiver_callback)(FlagStatus flag);

/* @PARA: sd_dev */
/* @STRUCT: sdio device information structure */
typedef struct {
    uint32_t periph;                                /*!< SDIO periph */
    uint32_t *p_txbuffer;                           /*!< SDIO transmit buffer */
    uint32_t tx_length;                             /*!< SDIO transmit length */
    uint32_t *p_rxbuffer;                           /*!< SDIO receive buffer */
    uint32_t rx_length;                             /*!< SDIO receive length */
    uint32_t transfer;                              /*!< SDIO transfer context */
    hal_sdio_sdcard_error_enum error_state;         /*!< SDIO error state */
    hal_sdio_sdcard_state_enum state;               /*!< SDIO state */
    hal_sdio_sdcard_cardinfo_struct card_info;      /*!< SDIO card information */
    hal_sdio_sdcard_irq_struct sd_irq;              /*!< SDIO device interrupt callback function pointer */
    uint32_t csd_data[4];                           /*!< data from the CSD register */
    uint32_t cid_data[4];                           /*!< data from the CID register */
    uint32_t scr_data[2];                           /*!< data from the SCR register */
    void *transmit_complete_callback;               /*!< SDIO transmit complete callback function */
    void *receive_complete_callback;                /*!< SDIO receive complete callback function */
    void *transfer_error_callback;                  /*!< SDIO transfer error callback function */
    void *write_dma_buffer0_complete_callback;      /*!< SDIO write DMA buffer complete callback function */
    void *write_dma_buffer1_complete_callback;      /*!< SDIO write DMA buffer complete callback function */
    void *read_dma_buffer0_complete_callback;       /*!< SDIO write DMA buffer complete callback function */
    void *read_dma_buffer1_complete_callback;       /*!< SDIO write DMA buffer complete callback function */
    void *abort_complete_callback;                  /*!< SDIO abort complete callback function */
    transceiver_callback transceiver_1_8v_callback; /*!< SDIO transceiver present callback function */
    hal_mutex_enum mutex;                           /*!< SDIO mutex */
} hal_sdio_sdcard_dev_struct;

typedef void (*hal_sdio_sdcard_user_cb)(hal_sdio_sdcard_dev_struct *sd_dev);

typedef struct {
    __IO hal_sdio_sdcard_user_cb transmit_complete_func;             /*!< SDIO transmit complete callback function */
    __IO hal_sdio_sdcard_user_cb receive_complete_func;              /*!< SDIO receive complete callback function */
    __IO hal_sdio_sdcard_user_cb read_dma_buffer0_complete_func;     /*!< SDIO read DMA buffer complete callback function */
    __IO hal_sdio_sdcard_user_cb read_dma_buffer1_complete_func;     /*!< SDIO read DMA buffer complete callback function */
    __IO hal_sdio_sdcard_user_cb write_dma_buffer0_complete_func;    /*!< SDIO write DMA buffer complete callback function */
    __IO hal_sdio_sdcard_user_cb write_dma_buffer1_complete_func;    /*!< SDIO write DMA buffer complete callback function */
    __IO hal_sdio_sdcard_user_cb transfer_error_func;                /*!< SDIO transfer error callback function */
    __IO hal_sdio_sdcard_user_cb abort_complete_func;                /*!< SDIO read DMA buffer complete callback function */
} hal_sdio_sdcard_irq_user_callback_struct;

/* hal function declarations */
/* Initialization and de-initialization functions */
/* @FUNCTION: initialize sdcard */
hal_sdio_sdcard_error_enum hal_sdio_sdcard_init(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t periph, hal_sdio_sdcard_init_struct *p_sd_init);
/* @END */
/* initialize the sdcard structure with the default values */
int32_t hal_sdio_sdcard_struct_init(hal_sdio_sdcard_struct_type_enum hal_struct_type, void *p_struct);
/* deinitialize the sdcard */
int32_t hal_sdio_sdcard_deinit(hal_sdio_sdcard_dev_struct *sd_dev);

/* set user-defined transceiver callback function */
int32_t hal_sdio_sdcard_transceiver_callback_register(hal_sdio_sdcard_dev_struct *sd_dev, transceiver_callback p_user_callback);
/* reset user-defined transceiver callback function */
int32_t hal_sdio_sdcard_transceiver_callback_unregister(hal_sdio_sdcard_dev_struct *sd_dev);
/* SDIO interrupt handle functions */
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_sdio_sdcard_irq_handle_set(hal_sdio_sdcard_dev_struct *sd_dev, hal_sdio_sdcard_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_sdio_sdcard_irq_handle_all_reset(hal_sdio_sdcard_dev_struct *sd_dev);
/* SD card interrupt handler content function,which is merely used in SDIO_IRQHandler */
void hal_sdio_sdcard_irq(hal_sdio_sdcard_dev_struct *sd_dev);

/* configure the bus mode */
int32_t hal_sdio_sdcard_bus_mode_config(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t busmode, uint32_t speed);
/* configure dma dual buffer mode. the data transfer is managed by an internal dma */
int32_t hal_sdio_sdcard_dma_buffer_config(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *data_buffer0, uint32_t *data_buffer1, \
                                     uint32_t buffer_size);
/* change the dma buffer0 or buffer1 address on the fly */
int32_t hal_sdio_sdcard_change_dma_buffer(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t buffer_select, uint32_t *data_buffer);
/* transmit or receive functions */
/* read data into a buffer from the specified address of a card */
int32_t hal_sdio_sdcard_readblocks_poll(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pdata, \
                                   uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber);
/* read data into a buffer from the specified address of a card by interrupt method */
int32_t hal_sdio_sdcard_readblocks_interrupt(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pdata, \
                                        uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                        hal_sdio_sdcard_irq_user_callback_struct *p_user_func);
/* read data into a buffer from the specified address of a card by dma method */
int32_t hal_sdio_sdcard_readblocks_dma(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pdata, \
                                  uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                  hal_sdio_sdcard_irq_user_callback_struct *p_user_func);
/* read data into a buffer from the specified address of a card by dma method */
int32_t hal_sdio_sdcard_readblocks_multibuffer_dma(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pbuf0, uint32_t *pbuf1, \
                                              uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                              hal_sdio_sdcard_irq_user_callback_struct *p_user_func);
/* write data to the specified address of a card */
int32_t hal_sdio_sdcard_writeblocks_poll(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pdata, \
                                    uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber);
/* write data to the specified address of a card by interrupt method */
int32_t hal_sdio_sdcard_writeblocks_interrupt(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pdata, \
                                         uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                         hal_sdio_sdcard_irq_user_callback_struct *p_user_func);
/* write data to the specified address of a card by dma method */
int32_t hal_sdio_sdcard_writeblocks_dma(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pdata, \
                                   uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                   hal_sdio_sdcard_irq_user_callback_struct *p_user_func);
/* write data to the specified address of a card by dma method */
int32_t hal_sdio_sdcard_writeblocks_multibuffer_dma(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pbuf0, uint32_t *pbuf1, \
                                               uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                               hal_sdio_sdcard_irq_user_callback_struct *p_user_func);
/* erase a continuous area of a card */
int32_t hal_sdio_sdcard_erase(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t startaddr, uint32_t endaddr);
/* abort an ongoing data transfer */
int32_t hal_sdio_sdcard_abort(hal_sdio_sdcard_dev_struct *sd_dev);
/* abort an ongoing data transfer */
int32_t hal_sdio_sdcard_abort_interrupt(hal_sdio_sdcard_dev_struct *sd_dev);
/* lock and unlock sdcard */
int32_t hal_sdio_sdcard_lock_unlock(hal_sdio_sdcard_dev_struct *sd_dev, uint8_t lockstate);

/* get the state which the card is in */
int32_t hal_sdio_sdcard_cardstate_get(hal_sdio_sdcard_dev_struct *sd_dev, uint8_t *pcardstate);
/* get the cid data information of the card which are stored on */
int32_t hal_sdio_sdcard_cid_get(hal_sdio_sdcard_dev_struct *sd_dev, hal_sdio_sdcard_card_cid_struct *p_cid);
/* get the csd data information of the card which are stored on */
int32_t hal_sdio_sdcard_csd_get(hal_sdio_sdcard_dev_struct *sd_dev, hal_sdio_sdcard_card_csd_struct *p_csd);
/* get the card status whose response format R1 contains a 32-bit field */
int32_t hal_sdio_sdcard_cardstatus_get(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pcardstatus);
/* get the detailed information of the SD card based on received CID and CSD */
int32_t hal_sdio_sdcard_information_get(hal_sdio_sdcard_dev_struct *sd_dev, hal_sdio_sdcard_cardinfo_struct *p_cardinfo);

/* return the sdcard state */
hal_sdio_sdcard_state_enum hal_sdio_sdcard_state_get(hal_sdio_sdcard_dev_struct *sd_dev);
/* return the sdcard error code */
hal_sdio_sdcard_error_enum hal_sdio_sdcard_error_get(hal_sdio_sdcard_dev_struct *sd_dev);

#endif /* GD32H7XX_HAL_SDIO_SDCARD_H */
