/*!
    \file    gd32h7xx_hal_sai.h
    \brief   definitions for the SAI

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_SAI_H
#define GD32H7XX_HAL_SAI_H

#include "gd32h7xx_hal.h"

/* SAI definitions */
#define SAI0                                  (SAI_BASE + 0x00000000U)                          /*!< SAI0 base address */
#define SAI1                                  (SAI_BASE + 0x00000400U)                          /*!< SAI1 base address */
#define SAI2                                  (SAI_BASE + 0x00000800U)                          /*!< SAI2 base address */

/* registers definitions */
#define SAI_SYNCFG(saix)                      REG32((saix) + 0x00000000U)                       /*!< SAI synchronize configuration register */
#define SAI_B0CFG0(saix)                      REG32((saix) + 0x00000004U)                       /*!< SAI block 0 configuration register0 */
#define SAI_B0CFG1(saix)                      REG32((saix) + 0x00000008U)                       /*!< SAI block 0 configuration register1 */
#define SAI_B0FCFG(saix)                      REG32((saix) + 0x0000000CU)                       /*!< SAI block 0 frame configuration register */
#define SAI_B0SCFG(saix)                      REG32((saix) + 0x00000010U)                       /*!< SAI block 0 slot configuration register */
#define SAI_B0INTEN(saix)                     REG32((saix) + 0x00000014U)                       /*!< SAI block 0 interrupt enable register  */
#define SAI_B0STAT(saix)                      REG32((saix) + 0x00000018U)                       /*!< SAI block 0 status register */
#define SAI_B0INTC(saix)                      REG32((saix) + 0x0000001CU)                       /*!< SAI block 0 interrupt flag clear register */
#define SAI_B0DATA(saix)                      REG32((saix) + 0x00000020U)                       /*!< SAI block 0 data register */
#define SAI_B1CFG0(saix)                      REG32((saix) + 0x00000024U)                       /*!< SAI block 1 configuration register0 */
#define SAI_B1CFG1(saix)                      REG32((saix) + 0x00000028U)                       /*!< SAI block 1 configuration register1 */
#define SAI_B1FCFG(saix)                      REG32((saix) + 0x0000002CU)                       /*!< SAI block 1 frame configuration register */
#define SAI_B1SCFG(saix)                      REG32((saix) + 0x00000030U)                       /*!< SAI block 1 slot configuration register */
#define SAI_B1INTEN(saix)                     REG32((saix) + 0x00000034U)                       /*!< SAI block 1 interrupt enable register  */
#define SAI_B1STAT(saix)                      REG32((saix) + 0x00000038U)                       /*!< SAI block 1 status register */
#define SAI_B1INTC(saix)                      REG32((saix) + 0x0000003CU)                       /*!< SAI block 1 interrupt flag clear register */
#define SAI_B1DATA(saix)                      REG32((saix) + 0x00000040U)                       /*!< SAI block 1 data register */
#define SAI_PDMCTL(saix)                      REG32((saix) + 0x00000044U)                       /*!< SAI PDM control register */
#define SAI_PDMCFG(saix)                      REG32((saix) + 0x00000048U)                       /*!< SAI PDM configuration register */

#define SAI_CFG0(saix, blocky)                REG32(((saix) + 0x00000004U) + 0x20U * (blocky))  /*!< SAI block configuration register0 */
#define SAI_CFG1(saix, blocky)                REG32(((saix) + 0x00000008U) + 0x20U * (blocky))  /*!< SAI block configuration register1 */
#define SAI_FCFG(saix, blocky)                REG32(((saix) + 0x0000000CU) + 0x20U * (blocky))  /*!< SAI block frame configuration register */
#define SAI_SCFG(saix, blocky)                REG32(((saix) + 0x00000010U) + 0x20U * (blocky))  /*!< SAI block slot configuration register */
#define SAI_INTEN(saix, blocky)               REG32(((saix) + 0x00000014U) + 0x20U * (blocky))  /*!< SAI block interrupt enable register  */
#define SAI_STAT(saix, blocky)                REG32(((saix) + 0x00000018U) + 0x20U * (blocky))  /*!< SAI block status register */
#define SAI_INTC(saix, blocky)                REG32(((saix) + 0x0000001CU) + 0x20U * (blocky))  /*!< SAI block interrupt flag clear register */
#define SAI_DATA(saix, blocky)                REG32(((saix) + 0x00000020U) + 0x20U * (blocky))  /*!< SAI block data register */

/* bits definitions */
/* SAI_SYNCFG */
#define SAI_SYNCFG_SYNI                       BITS(0,1)                                   /*!< synchronization inputs */
#define SAI_SYNCFG_SYNO                       BITS(4,5)                                   /*!< synchronization outputs */

/* SAI_CFG0 */
#define SAI_CFG0_OPTMOD                       BITS(0,1)                                   /*!< operating mode */
#define SAI_CFG0_PROT                         BITS(2,3)                                   /*!< protocol selection */
#define SAI_CFG0_DATAWD                       BITS(5,7)                                   /*!< data width */
#define SAI_CFG0_SHIFTDIR                     BIT(8)                                      /*!< shift direction */
#define SAI_CFG0_SAMPEDGE                     BIT(9)                                      /*!< sampling clock edge */
#define SAI_CFG0_SYNCMOD                      BITS(10,11)                                 /*!< synchronization mode */
#define SAI_CFG0_MONO                         BIT(12)                                     /*!< stereo and mono mode selection */
#define SAI_CFG0_ODRIV                        BIT(13)                                     /*!< output drive */
#define SAI_CFG0_SAIEN                        BIT(16)                                     /*!< sai sub-block enable */
#define SAI_CFG0_DMAEN                        BIT(17)                                     /*!< DMA enable */
#define SAI_CFG0_BYPASS                       BIT(19)                                     /*!< clock divider logic bypass */
#define SAI_CFG0_MDIV                         BITS(20,25)                                 /*!< master clock divider ratio */
#define SAI_CFG0_MOSPR                        BIT(26)                                     /*!< the master clock oversampling rate */
#define SAI_CFG0_MCLKEN                       BIT(27)                                     /*!< the master clock enable */

/* SAI_CFG1 */
#define SAI_CFG1_FFTH                         BITS(0,2)                                   /*!< FIFO threshold */
#define SAI_CFG1_FLUSH                        BIT(3)                                      /*!< FIFO flush */
#define SAI_CFG1_SDOM                         BIT(4)                                      /*!< serial data output management */
#define SAI_CFG1_MT                           BIT(5)                                      /*!< mute mode on */
#define SAI_CFG1_MTVAL                        BIT(6)                                      /*!< mute value */
#define SAI_CFG1_MTFCNT                       BITS(7,12)                                  /*!< mute frame count */
#define SAI_CFG1_CPLMOD                       BIT(13)                                     /*!< complement mode */
#define SAI_CFG1_CPAMOD                       BITS(14,15)                                 /*!< compander mode */

/* SAI_FCFG */
#define SAI_FCFG_FWD                          BITS(0,7)                                   /*!< frame width */
#define SAI_FCFG_FSAWD                        BITS(8,14)                                  /*!< frame synchronization active width */
#define SAI_FCFG_FSFUNC                       BIT(16)                                     /*!< frame synchronization function */
#define SAI_FCFG_FSPL                         BIT(17)                                     /*!< Frame synchronization active polarity */
#define SAI_FCFG_FSOST                        BIT(18)                                     /*!< frame synchronization offset */

/* SAI_SCFG */
#define SAI_SCFG_DATAOST                      BITS(0,4)                                   /*!< data offset */
#define SAI_SCFG_SLOTWD                       BITS(6,7)                                   /*!< slot width */
#define SAI_SCFG_SLOTNUM                      BITS(8,11)                                  /*!< slot number within frame */
#define SAI_SCFG_SLOTAV                       BITS(16,31)                                 /*!< slot activation vector */

/* SAI_INTEN */
#define SAI_INTEN_OUERRIE                     BIT(0)                                      /*!< FIFO overrun or underrun interrupt enable */
#define SAI_INTEN_MTDETIE                     BIT(1)                                      /*!< mute detection interrupt enable */
#define SAI_INTEN_ERRCKIE                     BIT(2)                                      /*!< error clock interrupt enable */
#define SAI_INTEN_FFREQIE                     BIT(3)                                      /*!< FIFO request interrupt enable */
#define SAI_INTEN_ACNRDYIE                    BIT(4)                                      /*!< audio codec not ready interrupt enable */
#define SAI_INTEN_FSADETIE                    BIT(5)                                      /*!< frame synchronization advanced detection interrupt enable */
#define SAI_INTEN_FSPDETIE                    BIT(6)                                      /*!< frame synchronization postpone detection interrupt enable */

/* SAI_STAT */
#define SAI_STAT_OUERR                        BIT(0)                                      /*!< FIFO overrun or underrun */
#define SAI_STAT_MTDET                        BIT(1)                                      /*!< mute detection */
#define SAI_STAT_ERRCK                        BIT(2)                                      /*!< error clock */
#define SAI_STAT_FFREQ                        BIT(3)                                      /*!< FIFO request */
#define SAI_STAT_ACNRDY                       BIT(4)                                      /*!< audio codec not ready */
#define SAI_STAT_FSADET                       BIT(5)                                      /*!< frame synchronization advanced detection */
#define SAI_STAT_FSPDET                       BIT(6)                                      /*!< frame synchronization postpone detection */
#define SAI_STAT_FFSTAT                       BITS(16,18)                                 /*!< FIFO status */

/* SAI_INTC */
#define SAI_INTC_OUERRC                       BIT(0)                                      /*!< FIFO overrun or underrun interrupt clear */
#define SAI_INTC_MTDETC                       BIT(1)                                      /*!< mute detection interrupt clear */
#define SAI_INTC_ERRCKC                       BIT(2)                                      /*!< error clock interrupt clear */
#define SAI_INTC_ACNRDYC                      BIT(4)                                      /*!< audio codec not ready interrupt clear */
#define SAI_INTC_FSADETC                      BIT(5)                                      /*!< frame synchronization advanced detection interrupt clear */
#define SAI_INTC_FSPDETC                      BIT(6)                                      /*!< frame synchronization postpone detection interrupt clear */

/* SAI_DATA */
#define SAI_DATA_DATA                         BITS(0,31)                                  /*!< transfer data or receive data */

/* SAI_PDMCTL */
#define SAI_PDMCTL_PDMEN                      BIT(0)                                      /*!< PDM enable */
#define SAI_PDMCTL_MICNUMSEL                  BITS(4,5)                                   /*!< microphones number selected */
#define SAI_PDMCTL_CLKL0EN                    BIT(8)                                      /*!< PDM clock line 0 enable */
#define SAI_PDMCTL_CLKL1EN                    BIT(9)                                      /*!< PDM clock line 1 enable */

/* SAI_PDMCFG */
#define SAI_PDMCFG_DPL0                       BITS(0,2)                                   /*!< the 0 group of left channel microphone data flow delay period */
#define SAI_PDMCTL_DPR0                       BITS(4,6)                                   /*!< the 0 group of right channel microphone data flow delay period */
#define SAI_PDMCFG_DPL1                       BITS(8,10)                                  /*!< the 1st group of left channel microphone data flow delay period */
#define SAI_PDMCTL_DPR1                       BITS(12,14)                                 /*!< the 1st group of right channel microphone data flow delay period */
#define SAI_PDMCFG_DPL2                       BITS(16,18)                                 /*!< the 2nd group of left channel microphone data flow delay period */
#define SAI_PDMCTL_DPR2                       BITS(20,22)                                 /*!< the 2nd group of right channel microphone data flow delay period */
#define SAI_PDMCFG_DPL3                       BITS(24,26)                                 /*!< the 3rd group of left channel microphone data flow delay period */
#define SAI_PDMCTL_DPR3                       BITS(28,30)                                 /*!< the 3rd group of right channel microphone data flow delay period */

/* constants definitions */
#define SYNCFG_SYNO(regval)                   (BITS(4,5)&((uint32_t)(regval) << 4U))      /* SAI synchronization output definitions */
#define SYNCFG_SYNI(regval)                   (BITS(0,1)&((uint32_t)(regval) << 0U))      /* SAI synchronization input definitions */
#define CFG0_OPERATING(regval)                (BITS(0,1)&((uint32_t)(regval) << 0U))      /* SAI operating mode */
#define CFG0_PROTOCOL(regval)                 (BITS(2,3)&((uint32_t)(regval) << 2U))      /* SAI protocol selection */
#define CFG0_DW(regval)                       (BITS(5,7)&((uint32_t)(regval) << 5U))      /* SAI data width definitions */
#define CFG0_SYNCMOD(regval)                  (BITS(10,11)&((uint32_t)(regval) << 10U))   /* SAI synchronization mode definitions */
#define CFG1_FFTH(regval)                     (BITS(0,2)&((uint32_t)(regval) << 0U))      /* SAI FIFO threshold */
#define CFG1_COMPANDER(regval)                (BITS(14,15)&((uint32_t)(regval) << 14U))   /* SAI compander mode */
#define SCFG_SW(regval)                       (BITS(6,7)&((uint32_t)(regval) << 6U))      /* SAI slot width definitions */
#define STAT_FFSTAT(regval)                   (BITS(16,18)&((uint32_t)(regval) << 16U))   /* SAI FIFO status */
#define CFG0_MDIV(regval)                     (BITS(20,25)&((uint32_t)(regval) << 20U))   /* SAI master clock divider ratio definitions */

/* SAI interrupt enable or disable */
#define SAI_INT_OUERR                         SAI_INTEN_OUERRIE                           /*!< FIFO overrun or underrun interrupt enable */
#define SAI_INT_MTDET                         SAI_INTEN_MTDETIE                           /*!< mute detection interrupt enable */
#define SAI_INT_ERRCK                         SAI_INTEN_ERRCKIE                           /*!< error clock interrupt enable */
#define SAI_INT_FFREQ                         SAI_INTEN_FFREQIE                           /*!< FIFO request interrupt enable */
#define SAI_INT_ACNRDY                        SAI_INTEN_ACNRDYIE                          /*!< audio codec not ready interrupt enable */
#define SAI_INT_FSADET                        SAI_INTEN_FSADETIE                          /*!< frame synchronization advanced detection interrupt enable */
#define SAI_INT_FSPDET                        SAI_INTEN_FSPDETIE                          /*!< frame synchronization postpone detection interrupt enable */

/* SAI flags */
#define SAI_FLAG_OUERR                        SAI_STAT_OUERR                              /*!< FIFO overrun or underrun flag */
#define SAI_FLAG_MTDET                        SAI_STAT_MTDET                              /*!< mute detection flag */
#define SAI_FLAG_ERRCK                        SAI_STAT_ERRCK                              /*!< error clock flag */
#define SAI_FLAG_FFREQ                        SAI_STAT_FFREQ                              /*!< FIFO request flag */
#define SAI_FLAG_ACNRDY                       SAI_STAT_ACNRDY                             /*!< audio codec not ready flag */
#define SAI_FLAG_FSADET                       SAI_STAT_FSADET                             /*!< frame synchronization advanced detection flag */
#define SAI_FLAG_FSPDET                       SAI_STAT_FSPDET                             /*!< frame synchronization postpone detection flag */

/* SAI FIFO status */
#define SAI_FIFO_STAT_EMPTY                   STAT_FFSTAT(0)                              /*!< FIFO status empty */
#define SAI_FIFO_STAT_QUARTER                 STAT_FFSTAT(1)                              /*!< receiver: empty < FIFO <= 1/4,  transmitter: empty <  FIFO < 1/4 */
#define SAI_FIFO_STAT_HALF                    STAT_FFSTAT(2)                              /*!< receiver: 1/4   < FIFO <= 1/2,  transmitter: 1/4   <= FIFO < 1/2 */
#define SAI_FIFO_STAT_THREE_QUARTER           STAT_FFSTAT(3)                              /*!< receiver: 1/2   < FIFO <= 3/4,  transmitter: 1/2   <= FIFO < 3/4 */
#define SAI_FIFO_STAT_NEARFULL                STAT_FFSTAT(4)                              /*!< receiver: 3/4   < FIFO <  full, transmitter: 3/4   <= FIFO < full */
#define SAI_FIFO_STAT_FULL                    STAT_FFSTAT(5)                              /*!< FIFO status full */

/* SAI PDM microphone */
#define SAI_PDM_MICROPHONE0_L                 ((uint32_t)0x00000000U)                     /*!< the group 0 left channel microphone */
#define SAI_PDM_MICROPHONE0_R                 ((uint32_t)0x00000001U)                     /*!< the group 0 right channel microphone */
#define SAI_PDM_MICROPHONE1_L                 ((uint32_t)0x00000002U)                     /*!< the group 1 left channel microphone */
#define SAI_PDM_MICROPHONE1_R                 ((uint32_t)0x00000003U)                     /*!< the group 1 right channel microphone */
#define SAI_PDM_MICROPHONE2_L                 ((uint32_t)0x00000004U)                     /*!< the group 2 left channel microphone */
#define SAI_PDM_MICROPHONE2_R                 ((uint32_t)0x00000005U)                     /*!< the group 2 right channel microphone */
#define SAI_PDM_MICROPHONE3_L                 ((uint32_t)0x00000006U)                     /*!< the group 3 left channel microphone */
#define SAI_PDM_MICROPHONE3_R                 ((uint32_t)0x00000007U)                     /*!< the group 3 right channel microphone */

/* @PARA: block */
/* @DEFINE: SAI Block */
#define SAI_BLOCK0                            ((uint32_t)0x00000000U)                     /*!< block 0 */
#define SAI_BLOCK1                            ((uint32_t)0x00000001U)                     /*!< block 1 */

/* @STRUCT_MEMBER: syncout */
/* @DEFINE: SAI Synchro Output */
#define SAI_SYNCOUTPUT_OFF                    SYNCFG_SYNO(0)                              /*!< no synchronization output signals */
#define SAI_SYNCOUTPUT_BLOCK0                 SYNCFG_SYNO(1)                              /*!< block 0 used for further synchronization for others SAI */
#define SAI_SYNCOUTPUT_BLOCK1                 SYNCFG_SYNO(2)                              /*!< block 1 used for further synchronization for others SAI */

/* @STRUCT_MEMBER: syncin */
/* @DEFINE: SAI Synchro Input */
#define SAI_ASYNCHRONOUS                      (0U)                                        /*!< asynchronous */
#define SAI_SYNCHRONOUS                       (1U)                                        /*!< synchronous with other block of same sai */
#define SAI_SYNCHRONOUS_EXT_SAI0              (2U)                                        /*!< synchronous with other sai, sai1 */
#define SAI_SYNCHRONOUS_EXT_SAI1              (3U)                                        /*!< synchronous with other sai, sai2 */
#define SAI_SYNCHRONOUS_EXT_SAI2              (4U)                                        /*!< synchronous with other sai, sai3 */

#define SAI_SYNCIN_SAI0                       SYNCFG_SYNI(0)                              /*!< SAI 1 / 2 selects the synchronization coming from SAI 0 */
#define SAI_SYNCIN_SAI1                       SYNCFG_SYNI(1)                              /*!< SAI 0 / 2 selects the synchronization coming from SAI 1 */
#define SAI_SYNCIN_SAI2                       SYNCFG_SYNI(2)                              /*!< SAI 0 / 1 selects the synchronization coming from SAI 2 */

#define SAI_SYNCMODE_ASYNC                    CFG0_SYNCMOD(0)                             /*!< asynchronous with the other sub-block */
#define SAI_SYNCMODE_OTHERBLOCK               CFG0_SYNCMOD(1)                             /*!< synchronous with the other sub-block */
#define SAI_SYNCMODE_EXTERNALSAI              CFG0_SYNCMOD(2)                             /*!< synchronous with an external SAI audio sub-block */

/* @STRUCT_MEMBER: operation_mode */
/* @DEFINE: SAI Operating Mode */
#define SAI_MODE_MASTER_TX                    CFG0_OPERATING(0)                           /*!< master transmitter */
#define SAI_MODE_MASTER_RX                    CFG0_OPERATING(1)                           /*!< master receiver */
#define SAI_MODE_SLAVE_TX                     CFG0_OPERATING(2)                           /*!< slave transmitter */
#define SAI_MODE_SLAVE_RX                     CFG0_OPERATING(3)                           /*!< slave receiver */

/* @STRUCT_MEMBER: protocol */
/* @DEFINE: SAI Protocol */
#define SAI_FREE_PROTOCOL                     CFG0_PROTOCOL(0)                            /*!< polymorphic */
#define SAI_SPDIF_PROTOCOL                    CFG0_PROTOCOL(1)                            /*!< SPDIF */
#define SAI_AC97_PROTOCOL                     CFG0_PROTOCOL(2)                            /*!< AC97 */

#define SAI_I2S_STANDARD                      (0U)                                        /*!< I2S Standard */
#define SAI_I2S_MSBJUSTIFIED                  (1U)                                        /*!< I2S Msb justified */
#define SAI_I2S_LSBJUSTIFIED                  (2U)                                        /*!< I2S Lsb justified */
#define SAI_PCM_LONG                          (3U)                                        /*!< I2S Pulse Code Modulation Long */
#define SAI_PCM_SHORT                         (4U)                                        /*!< I2S Pulse Code Modulation Short */

/* @STRUCT_MEMBER: data_width */
/* @DEFINE: SAI data width */
#define SAI_DATA_WIDTH_8BIT                   CFG0_DW(2)                                  /*!< SAI data width 8 bit */
#define SAI_DATA_WIDTH_10BIT                  CFG0_DW(3)                                  /*!< SAI data width 10 bit */
#define SAI_DATA_WIDTH_16BIT                  CFG0_DW(4)                                  /*!< SAI data width 16 bit */
#define SAI_DATA_WIDTH_20BIT                  CFG0_DW(5)                                  /*!< SAI data width 20 bit */
#define SAI_DATA_WIDTH_24BIT                  CFG0_DW(6)                                  /*!< SAI data width 24 bit */
#define SAI_DATA_WIDTH_32BIT                  CFG0_DW(7)                                  /*!< SAI data width 32 bit */

#define SAI_PROTOCOL_DATAWIDTH_16BIT          (0U)                                        /*!< SAI protocol data width 16 bit */
#define SAI_PROTOCOL_DATAWIDTH_16BITEXTENDED  (1U)                                        /*!< SAI protocol data width 16 bit extended */
#define SAI_PROTOCOL_DATAWIDTH_24BIT          (2U)                                        /*!< SAI protocol data width 24 bit */
#define SAI_PROTOCOL_DATAWIDTH_32BIT          (3U)                                        /*!< SAI protocol data width 32 bit */

/* @STRUCT_MEMBER: shiftdir */
/* @DEFINE: SAI Shift direction */
#define SAI_SHIFT_MSB                         ((uint32_t)0x00000000U)                     /*!< data is shifted with MSB first */
#define SAI_SHIFT_LSB                         SAI_CFG0_SHIFTDIR                           /*!< data is shifted with LSB first */

/* @STRUCT_MEMBER: sampling_edge */
/* @DEFINE: SAI sampling clock edge */
#define SAI_SAMPEDGE_FALLING                  ((uint32_t)0x00000000U)                     /*!< data sampled on SCK falling edge */
#define SAI_SAMPEDGE_RISING                   SAI_CFG0_SAMPEDGE                           /*!< data sampled on SCK rising edge */

/* @STRUCT_MEMBER: mclk_output */
/* @DEFINE: SAI master clock enable */
#define SAI_MCLK_DISABLE                      ((uint32_t)0x00000000U)                     /*!< the master clock is enable when SAI enable */
#define SAI_MCLK_ENABLE                       SAI_CFG0_MCLKEN                             /*!< the master clock is enable now */

/* @STRUCT_MEMBER: mclk_oversampling */
/* @DEFINE: SAI master clock oversampling rate */
#define SAI_MCLK_OVERSAMP_256                 ((uint32_t)0x00000000U)                     /*!< MCLK = 256 * Ffs */
#define SAI_MCLK_OVERSAMP_512                 SAI_CFG0_MOSPR                              /*!< MCLK = 512 * Ffs */

/* @STRUCT_MEMBER: mclk_bypass */
/* @DEFINE: SAI clock divider logic bypass */
#define SAI_CLKDIV_BYPASS_OFF                 ((uint32_t)0x00000000U)                     /*!< clock divider ratio is applied to both primary and secondary divider logic */
#define SAI_CLKDIV_BYPASS_ON                  SAI_CFG0_BYPASS                             /*!< clock divider logic is bypassed */

/* @STRUCT_MEMBER: mclk_div */
/* @=NULL */
#define SAI_MCLKDIV_1                         CFG0_MDIV(0)                                /*!< primary frequency divider logic bypass */
#define SAI_MCLKDIV_2                         CFG0_MDIV(2)                                /*!< SAI clock is divided by 2 */
#define SAI_MCLKDIV_3                         CFG0_MDIV(3)                                /*!< SAI clock is divided by 3 */
#define SAI_MCLKDIV_4                         CFG0_MDIV(4)                                /*!< SAI clock is divided by 4 */
#define SAI_MCLKDIV_5                         CFG0_MDIV(5)                                /*!< SAI clock is divided by 5 */
#define SAI_MCLKDIV_6                         CFG0_MDIV(6)                                /*!< SAI clock is divided by 6 */
#define SAI_MCLKDIV_7                         CFG0_MDIV(7)                                /*!< SAI clock is divided by 7 */
#define SAI_MCLKDIV_8                         CFG0_MDIV(8)                                /*!< SAI clock is divided by 8 */
#define SAI_MCLKDIV_9                         CFG0_MDIV(9)                                /*!< SAI clock is divided by 9 */
#define SAI_MCLKDIV_10                        CFG0_MDIV(10)                               /*!< SAI clock is divided by 10 */
#define SAI_MCLKDIV_11                        CFG0_MDIV(11)                               /*!< SAI clock is divided by 11 */
#define SAI_MCLKDIV_12                        CFG0_MDIV(12)                               /*!< SAI clock is divided by 12 */
#define SAI_MCLKDIV_13                        CFG0_MDIV(13)                               /*!< SAI clock is divided by 13 */
#define SAI_MCLKDIV_14                        CFG0_MDIV(14)                               /*!< SAI clock is divided by 14 */
#define SAI_MCLKDIV_15                        CFG0_MDIV(15)                               /*!< SAI clock is divided by 15 */
#define SAI_MCLKDIV_16                        CFG0_MDIV(16)                               /*!< SAI clock is divided by 16 */
#define SAI_MCLKDIV_17                        CFG0_MDIV(17)                               /*!< SAI clock is divided by 17 */
#define SAI_MCLKDIV_18                        CFG0_MDIV(18)                               /*!< SAI clock is divided by 18 */
#define SAI_MCLKDIV_19                        CFG0_MDIV(19)                               /*!< SAI clock is divided by 19 */
#define SAI_MCLKDIV_20                        CFG0_MDIV(20)                               /*!< SAI clock is divided by 20 */
#define SAI_MCLKDIV_21                        CFG0_MDIV(21)                               /*!< SAI clock is divided by 21 */
#define SAI_MCLKDIV_22                        CFG0_MDIV(22)                               /*!< SAI clock is divided by 22 */
#define SAI_MCLKDIV_23                        CFG0_MDIV(23)                               /*!< SAI clock is divided by 23 */
#define SAI_MCLKDIV_24                        CFG0_MDIV(24)                               /*!< SAI clock is divided by 24 */
#define SAI_MCLKDIV_25                        CFG0_MDIV(25)                               /*!< SAI clock is divided by 25 */
#define SAI_MCLKDIV_26                        CFG0_MDIV(26)                               /*!< SAI clock is divided by 26 */
#define SAI_MCLKDIV_27                        CFG0_MDIV(27)                               /*!< SAI clock is divided by 27 */
#define SAI_MCLKDIV_28                        CFG0_MDIV(28)                               /*!< SAI clock is divided by 28 */
#define SAI_MCLKDIV_29                        CFG0_MDIV(29)                               /*!< SAI clock is divided by 29 */
#define SAI_MCLKDIV_30                        CFG0_MDIV(30)                               /*!< SAI clock is divided by 30 */
#define SAI_MCLKDIV_31                        CFG0_MDIV(31)                               /*!< SAI clock is divided by 31 */
#define SAI_MCLKDIV_32                        CFG0_MDIV(32)                               /*!< SAI clock is divided by 32 */
#define SAI_MCLKDIV_33                        CFG0_MDIV(33)                               /*!< SAI clock is divided by 33 */
#define SAI_MCLKDIV_34                        CFG0_MDIV(34)                               /*!< SAI clock is divided by 34 */
#define SAI_MCLKDIV_35                        CFG0_MDIV(35)                               /*!< SAI clock is divided by 35 */
#define SAI_MCLKDIV_36                        CFG0_MDIV(36)                               /*!< SAI clock is divided by 36 */
#define SAI_MCLKDIV_37                        CFG0_MDIV(37)                               /*!< SAI clock is divided by 37 */
#define SAI_MCLKDIV_38                        CFG0_MDIV(38)                               /*!< SAI clock is divided by 38 */
#define SAI_MCLKDIV_39                        CFG0_MDIV(39)                               /*!< SAI clock is divided by 39 */
#define SAI_MCLKDIV_40                        CFG0_MDIV(40)                               /*!< SAI clock is divided by 40 */
#define SAI_MCLKDIV_41                        CFG0_MDIV(41)                               /*!< SAI clock is divided by 41 */
#define SAI_MCLKDIV_42                        CFG0_MDIV(42)                               /*!< SAI clock is divided by 42 */
#define SAI_MCLKDIV_43                        CFG0_MDIV(43)                               /*!< SAI clock is divided by 43 */
#define SAI_MCLKDIV_44                        CFG0_MDIV(44)                               /*!< SAI clock is divided by 44 */
#define SAI_MCLKDIV_45                        CFG0_MDIV(45)                               /*!< SAI clock is divided by 45 */
#define SAI_MCLKDIV_46                        CFG0_MDIV(46)                               /*!< SAI clock is divided by 46 */
#define SAI_MCLKDIV_47                        CFG0_MDIV(47)                               /*!< SAI clock is divided by 47 */
#define SAI_MCLKDIV_48                        CFG0_MDIV(48)                               /*!< SAI clock is divided by 48 */
#define SAI_MCLKDIV_49                        CFG0_MDIV(49)                               /*!< SAI clock is divided by 49 */
#define SAI_MCLKDIV_50                        CFG0_MDIV(50)                               /*!< SAI clock is divided by 50 */
#define SAI_MCLKDIV_51                        CFG0_MDIV(51)                               /*!< SAI clock is divided by 51 */
#define SAI_MCLKDIV_52                        CFG0_MDIV(52)                               /*!< SAI clock is divided by 52 */
#define SAI_MCLKDIV_53                        CFG0_MDIV(53)                               /*!< SAI clock is divided by 53 */
#define SAI_MCLKDIV_54                        CFG0_MDIV(54)                               /*!< SAI clock is divided by 54 */
#define SAI_MCLKDIV_55                        CFG0_MDIV(55)                               /*!< SAI clock is divided by 55 */
#define SAI_MCLKDIV_56                        CFG0_MDIV(56)                               /*!< SAI clock is divided by 56 */
#define SAI_MCLKDIV_57                        CFG0_MDIV(57)                               /*!< SAI clock is divided by 57 */
#define SAI_MCLKDIV_58                        CFG0_MDIV(58)                               /*!< SAI clock is divided by 58 */
#define SAI_MCLKDIV_59                        CFG0_MDIV(59)                               /*!< SAI clock is divided by 59 */
#define SAI_MCLKDIV_60                        CFG0_MDIV(60)                               /*!< SAI clock is divided by 60 */
#define SAI_MCLKDIV_61                        CFG0_MDIV(61)                               /*!< SAI clock is divided by 61 */
#define SAI_MCLKDIV_62                        CFG0_MDIV(62)                               /*!< SAI clock is divided by 62 */
#define SAI_MCLKDIV_63                        CFG0_MDIV(63)                               /*!< SAI clock is divided by 63 */

/* @STRUCT_MEMBER: audio_frequency */
/* @DEFINE: SAI audio frequency */
#define SAI_AUDIO_FREQUENCY_192K              (192000U)                                   /*!< SAI audio frequency 192 kHz */
#define SAI_AUDIO_FREQUENCY_96K               (96000U)                                    /*!< SAI audio frequency 96 kHz */
#define SAI_AUDIO_FREQUENCY_48K               (48000U)                                    /*!< SAI audio frequency 48 kHz */
#define SAI_AUDIO_FREQUENCY_44K               (44100U)                                    /*!< SAI audio frequency 44.1 kHz */
#define SAI_AUDIO_FREQUENCY_32K               (32000U)                                    /*!< SAI audio frequency 32 kHz */
#define SAI_AUDIO_FREQUENCY_22K               (22050U)                                    /*!< SAI audio frequency 22.05 kHz */
#define SAI_AUDIO_FREQUENCY_16K               (16000U)                                    /*!< SAI audio frequency 16 kHz */
#define SAI_AUDIO_FREQUENCY_11K               (11025U)                                    /*!< SAI audio frequency 11.025 kHz */
#define SAI_AUDIO_FREQUENCY_8K                (8000U)                                     /*!< SAI audio frequency 8 kHz */
#define SAI_AUDIO_FREQUENCY_MDIV              (0U)                                        /*!< SAI audio frequency set by MDIV */

/* @STRUCT_MEMBER: pdm_clock_enable */
/* @DEFINE: SAI PDM Clock Enable */
#define SAI_PDM_CLOCK_DISABLE                 ((uint32_t)0x00000000U)                     /*!< SAI pdm clock disable */
#define SAI_PDM_CLOCK0_ENABLE                 SAI_PDMCTL_CLKL0EN                          /*!< SAI pdm clock0 enable */
#define SAI_PDM_CLOCK1_ENABLE                 SAI_PDMCTL_CLKL1EN                          /*!< SAI pdm clock1 enable */
#define SAI_PDM_CLOCK0_AND_CLOCK1_ENABLE      (SAI_PDMCTL_CLKL0EN|SAI_PDMCTL_CLKL1EN)     /*!< SAI pdm clock0 and clock1 enable */

/* @STRUCT_MEMBER: output_drive */
/* @DEFINE: SAI output drive */
#define SAI_OUTPUTDRIVE_DISABLE               ((uint32_t)0x00000000U)                     /*!< SAI sub-block output driven only when SAIEN is set */
#define SAI_OUTPUTDRIVE_ENABLE                SAI_CFG0_ODRIV                              /*!< SAI sub-block output driven according to ODRIV setting */

/* @STRUCT_MEMBER: mono_stereo_mode */
/* @DEFINE: SAI stereo and mono mode selection */
#define SAI_STEREO_MODE                       ((uint32_t)0x00000000U  )                   /*!< stereo mode */
#define SAI_MONO_MODE                         SAI_CFG0_MONO                               /*!< mono mode */

/* @STRUCT_MEMBER: fsfunction */
/* @DEFINE: SAI frame synchronization function */
#define SAI_FS_FUNC_START                     ((uint32_t)0x00000000U)                     /*!< FS only defines frame start */
#define SAI_FS_FUNC_START_CHANNEL             SAI_FCFG_FSFUNC                             /*!< FS define both frame start and channel number */

/* @STRUCT_MEMBER: fspolarity */
/* @DEFINE: SAI frame synchronization active polarity */
#define SAI_FS_POLARITY_LOW                   ((uint32_t)0x00000000U)                     /*!< FS low active polarity */
#define SAI_FS_POLARITY_HIGH                  SAI_FCFG_FSPL                               /*!< FS high active polarity */

/* @STRUCT_MEMBER: fsoffset */
/* @DEFINE: SAI frame synchronization offset */
#define SAI_FS_OFFSET_BEGINNING               ((uint32_t)0x00000000U)                     /*!< FS active edge asserted at the beginning of the first bit of the first slot */
#define SAI_FS_OFFSET_ONEBITBEFORE            SAI_FCFG_FSOST                              /*!< FS active edge asserted one bit cycle before normal FS when FSOST is 0 */

/* @STRUCT_MEMBER: slot_number */
/* @=NULL */
#define SAI_SLOT_NUMBER                       SAI_SCFG_SLOTNUM                            /*!< slot number */

/* @STRUCT_MEMBER: slot_width */
/* @DEFINE: SAI slot width definitions */
#define SAI_SLOT_WIDTH_DATA                   SCFG_SW(0)                                  /*!< slot width equals data width */
#define SAI_SLOT_WIDTH_16BIT                  SCFG_SW(1)                                  /*!< slot width of 16-bits */
#define SAI_SLOT_WIDTH_32BIT                  SCFG_SW(2)                                  /*!< slot width of 32-bits */

/* @STRUCT_MEMBER: slot_active_value */
/* @=NULL */
#define SAI_SLOT_ACTIVE_NONE                  ((uint32_t)0x00000000U)                     /*!< all slot inactive */
#define SAI_SLOT_ACTIVE_0                     BIT(16)                                     /*!< slot 0 active */
#define SAI_SLOT_ACTIVE_1                     BIT(17)                                     /*!< slot 1 active */
#define SAI_SLOT_ACTIVE_2                     BIT(18)                                     /*!< slot 2 active */
#define SAI_SLOT_ACTIVE_3                     BIT(19)                                     /*!< slot 3 active */
#define SAI_SLOT_ACTIVE_4                     BIT(20)                                     /*!< slot 4 active */
#define SAI_SLOT_ACTIVE_5                     BIT(21)                                     /*!< slot 5 active */
#define SAI_SLOT_ACTIVE_6                     BIT(22)                                     /*!< slot 6 active */
#define SAI_SLOT_ACTIVE_7                     BIT(23)                                     /*!< slot 7 active */
#define SAI_SLOT_ACTIVE_8                     BIT(24)                                     /*!< slot 8 active */
#define SAI_SLOT_ACTIVE_9                     BIT(25)                                     /*!< slot 9 active */
#define SAI_SLOT_ACTIVE_10                    BIT(26)                                     /*!< slot 10 active */
#define SAI_SLOT_ACTIVE_11                    BIT(27)                                     /*!< slot 11 active */
#define SAI_SLOT_ACTIVE_12                    BIT(28)                                     /*!< slot 12 active */
#define SAI_SLOT_ACTIVE_13                    BIT(29)                                     /*!< slot 13 active */
#define SAI_SLOT_ACTIVE_14                    BIT(30)                                     /*!< slot 14 active */
#define SAI_SLOT_ACTIVE_15                    BIT(31)                                     /*!< slot 15 active */
#define SAI_SLOT_ACTIVE_ALL                   BITS(16,31)                                 /*!< slot all active */

/* SAI mute value */
#define SAI_MUTESENT_0                        ((uint32_t)0x00000000U)                     /*!< 0 is sent via the serial data line when mute is on */
#define SAI_MUTESENT_LASTFREAM                SAI_CFG1_MTVAL                              /*!< if SLOTNB is less or equals to two, last frame is sent via the serial data line */

/* SAI mute on */
#define SAI_MUTE_OFF                          ((uint32_t)0x00000000U)                     /*!< mute mode off */
#define SAI_MUTE_ON                           SAI_CFG1_MT                                 /*!< mute mode on */

/* @STRUCT_MEMBER: serial_data_mode */
/* @DEFINE: SAI serial data line output management */
#define SAI_SDLINE_DRIVEN                     ((uint32_t)0x00000000U)                     /*!< SD line output is driven entirely during the audio frame */
#define SAI_SDLINE_RELEASE                    SAI_CFG1_SDOM                               /*!< SD line output is released near inactive slots */

/* @STRUCT_MEMBER: fifo_threshold */
/* @DEFINE: SAI FIFO threshold */
#define SAI_FIFOTH_EMPTY                      CFG1_FFTH(0)                                /*!< FIFO threshold empty */
#define SAI_FIFOTH_QUARTER                    CFG1_FFTH(1)                                /*!< FIFO threshold quarter full */
#define SAI_FIFOTH_HALF                       CFG1_FFTH(2)                                /*!< FIFO threshold half full */
#define SAI_FIFOTH_THREE_QUARTER              CFG1_FFTH(3)                                /*!< FIFO threshold three quarter full */
#define SAI_FIFOTH_FULL                       CFG1_FFTH(4)                                /*!< FIFO threshold full */

/* @STRUCT_MEMBER: companding_mode */
/* @DEFINE: SAI Companion */
#define SAI_NO_COMPANDING                     ((uint32_t)0x00000000U)                     /*!< no compansion applies */
#define SAI_ULAW_1CPL_COMPANDING              CFG1_COMPANDER(2)                           /*!< 1's complement form u-law algorithm */
#define SAI_ALAW_1CPL_COMPANDING              CFG1_COMPANDER(3)                           /*!< 1's complement form A-law algorithm */
#define SAI_ULAW_2CPL_COMPANDING              (CFG1_COMPANDER(2) | SAI_CFG1_CPLMOD)       /*!< 2's complement form u-law algorithm */
#define SAI_ALAW_2CPL_COMPANDING              (CFG1_COMPANDER(3) | SAI_CFG1_CPLMOD)       /*!< 2's complement form A-law algorithm */

/* SAI interrupt */
#define SAI_INT_OUERR                         SAI_INTEN_OUERRIE                           /*!< FIFO overrun or underrun interrupt enable */
#define SAI_INT_MTDET                         SAI_INTEN_MTDETIE                           /*!< mute detection interrupt enable */
#define SAI_INT_ERRCK                         SAI_INTEN_ERRCKIE                           /*!< error clock interrupt enable */
#define SAI_INT_FFREQ                         SAI_INTEN_FFREQIE                           /*!< FIFO request interrupt enable */
#define SAI_INT_ACNRDY                        SAI_INTEN_ACNRDYIE                          /*!< audio codec not ready interrupt enable */
#define SAI_INT_FSADET                        SAI_INTEN_FSADETIE                          /*!< frame synchronization advanced detection interrupt enable */
#define SAI_INT_FSPDET                        SAI_INTEN_FSPDETIE                          /*!< frame synchronization postpone detection interrupt enable */

/* SAI error code */
#define HAL_SAI_ERROR_NONE                    (0U)                                        /*!< No error */
#define HAL_SAI_ERROR_OUERR                   BIT(0)                                      /*!< overrun Error */
#define HAL_SAI_ERROR_UDERR                   BIT(1)                                      /*!< underrun error */
#define HAL_SAI_ERROR_FSADET                  BIT(2)                                      /*!< frame synchronization advanced detection */
#define HAL_SAI_ERROR_FSPDET                  BIT(3)                                      /*!< frame synchronization postpone detection */
#define HAL_SAI_ERROR_ACNRDY                  BIT(4)                                      /*!< codec not ready */
#define HAL_SAI_ERROR_ERRCK                   BIT(5)                                      /*!< wrong clock configuration */
#define HAL_SAI_ERROR_TIMEOUT                 BIT(6)                                      /*!< timeout error */
#define HAL_SAI_ERROR_DMA                     BIT(7)                                      /*!< DMA error */

/* SAI state enum */
typedef enum {
    HAL_SAI_STATE_RESET   = 0x00U,                                                  /*!< SAI not yet initialized or disabled  */
    HAL_SAI_STATE_READY   = 0x01U,                                                  /*!< SAI initialized and ready for use    */
    HAL_SAI_STATE_BUSY    = 0x02U,                                                  /*!< SAI internal process is ongoing      */
    HAL_SAI_STATE_BUSY_TX = 0x12U,                                                  /*!< data transmission process is ongoing */
    HAL_SAI_STATE_BUSY_RX = 0x22U                                                   /*!< data reception process is ongoing    */
} hal_sai_state_enum;

/* SAI structure type enum */
typedef enum {
    HAL_SAI_INIT_STRUCT = 0U,                                                       /*!< SAI initialization structure */
    HAL_SAI_DEV_STRUCT,                                                             /*!< SAI device structure */
    HAL_SAI_IRQ_INIT_STRUCT,                                                        /*!< SAI interrupt callback structure */
    HAL_SAI_USERER_CALLBACK_STRUCT                                                  /*!< SAI user error callback structure */
} hal_sai_struct_type_enum;

/* SAI FIFO status */
typedef enum {
    FIFO_EMPTY  = 0U,                                                               /*!< empty */
    FIFO_EMPTY_TO_1_4_FULL,                                                         /*!< empty < fifo_level <= 1/4_full. */
    FIFO_1_4_FULL_TO_1_2_FULL,                                                      /*!< 1/4_full < fifo_level <= 1/2_full. */
    FIFO_1_2_FULL_TO_3_4_FULL,                                                      /*!< 1/2_full < fifo_level <= 3/4_full. */
    FIFO_3_4_FULL_TO_FULL,                                                          /*!< 3/4_full < fifo_level < full */
    FIFO_FULL                                                                       /*!< full */
} hal_sai_fifo_state_enum;

/* SAI device interrupt callback function pointer structure */
typedef struct {
    hal_irq_handle_cb              receive_handler;                                 /*!< SAI receive complete callback function */
    hal_irq_handle_cb              transmit_handler;                                /*!< SAI transmit complete callback function */
    hal_irq_handle_cb              mute_handler;                                    /*!< SAI transmit complete callback function */
    hal_irq_handle_cb              error_handler;                                   /*!< SAI error complete callback function */

} hal_sai_irq_struct;

/* SAI transfer buffer structure */
typedef struct {
    __IO uint8_t *buffer;                                                           /*!< pointer to transfer buffer */
    __IO uint32_t length;                                                           /*!< transfer byte length */
    __IO uint32_t remain;                                                           /*!< transfer remain byte */
} hal_sai_buffer_struct;

/* @PARA: p_init */
/* @STRUCT: SAI Init Structure definition */
typedef struct {
    uint32_t operation_mode;                                                        /*!< specifies the sai block audio mode */
    uint32_t syncin;                                                                /*!< specifies sai block synchronization */
    uint32_t syncout;                                                               /*!< specifies sai external output synchronization */
    uint32_t output_drive;                                                          /*!< specifies when sai block outputs are driven */
    uint32_t fifo_threshold;                                                        /*!< specifies sai block fifo threshold */
    uint32_t mclk_output;                                                           /*!< specifies whether master clock output will be generated or not */
    uint32_t mclk_bypass;                                                           /*!< specifies whether master clock will be divided or bypass */
    uint32_t mclk_div;                                                              /*!< specifies the master clock divider */
    uint32_t mclk_oversampling;                                                     /*!< specifies the master clock oversampling */
    uint32_t audio_frequency;                                                       /*!< specifies the audio frequency */
    uint32_t mono_stereo_mode;                                                      /*!< specifies if the mono or stereo mode is selected*/
    uint32_t companding_mode;                                                       /*!< specifies the companding mode type */
    uint32_t serial_data_mode;                                                      /*!< specifies the tristate management on data line  */
    uint32_t pdm_clock_enable;                                                      /*!< specifies which clock must be enabled */
    uint32_t mic_pairs_number;                                                      /*!< specifies the number of microphone pairs used */
    uint32_t protocol;                                                              /*!< specifies the sai block protocol */
    uint32_t data_width;                                                            /*!< specifies the sai block data width */
    uint32_t shiftdir;                                                              /*!< specifies whether data transfers start from msb or lsb bit */
    uint32_t sampling_edge;                                                         /*!< specifies the sai block clock sampling edge sensitivity */
    /* sai frame structure definition */
    uint32_t frame_width;                                                           /*!< specifies the frame length, the number of sck clocks for each audio frame */
    uint32_t active_frame_width;                                                    /*!< specifies the frame synchronization active level length */
    uint32_t fsfunction;                                                            /*!< specifies the frame synchronization definition */
    uint32_t fspolarity;                                                            /*!< specifies the frame synchronization polarity */
    uint32_t fsoffset;                                                              /*!< specifies the frame synchronization offset */
    /* sai slot structure definition */
    uint32_t data_offset;                                                           /*!< specifies the position of first data transfer bit in the slot */
    uint32_t slot_width;                                                            /*!< specifies the slot width */
    uint32_t slot_number;                                                           /*!< specifies the number of slot in the audio frame */
    uint32_t slot_active_value;                                                     /*!< specifies the slots in audio frame that will be activated */
} hal_sai_init_struct;

/* @STRUCT_MEMBER: mic_pairs_number */
/* @=NULL */

/* @STRUCT_MEMBER: frame_width */
/* @=NULL */

/* @STRUCT_MEMBER: activeframewidth */
/* @=NULL */

/* @STRUCT_MEMBER: data_offset */
/* @=NULL */

/* @PARA: sai_dev */
/* @STRUCT: SAI device structure definition */
typedef struct {
    uint32_t                  periph;                                               /*!< SAI periph */
    uint32_t                  block;                                                /*!< SAI block */
    hal_sai_init_struct       init;                                                 /*!< SAI init parameter */
    hal_sai_irq_struct        sai_irq;                                              /*!< device interrupt callback function pointer */
    hal_dma_dev_struct        *p_dma_rx;                                            /*!< DMA receive pointer */
    hal_dma_dev_struct        *p_dma_tx;                                            /*!< DMA transmit pointer */
    hal_sai_buffer_struct     pbuffer;                                              /*!< transmit/receive buffer */
    void                      *transfer_callback;                                   /*!< SAI irq callback */
    void                      *half_transfer_callback;                              /*!< SAI irq callback */
    void                      *receive_callback;                                    /*!< SAI irq callback */
    void                      *half_receive_callback;                               /*!< SAI irq callback */
    void                      *mute_callback;                                       /*!< SAI mute callback */
    void                      *error_callback;                                      /*!< SAI error callback */
    hal_sai_state_enum        state;                                                /*!< SAI state */
    uint32_t                  error_state;                                          /*!< SAI error state */
    hal_mutex_enum            mutex;                                                /*!< lock */
} hal_sai_dev_struct;

/* SAI device user callback function pointer */
typedef void (*hal_sai_user_cb)(hal_sai_dev_struct *sai_dev);

/* SAI callback structure */
typedef struct {
    hal_sai_user_cb         half_complete_func;                                     /*!< SAI user half complete callback function */
    hal_sai_user_cb         complete_func;                                          /*!< SAI user complete callback function */
    hal_sai_user_cb         mute_func;                                              /*!< SAI mute callback */
    hal_sai_user_cb         error_func;                                             /*!< SAI error callback */
}hal_sai_user_callback_struct;

/* hal function declarations */
/* @FUNCTION: SAI deinitialization and initialization functions */
/* initialize SAI Protocol */
int32_t hal_sai_protocol_struct_init(hal_sai_init_struct *p_init, uint32_t protocol, uint32_t data_width, \
                                     uint32_t slot_number);
/* initialize SAI */
int32_t hal_sai_init(hal_sai_dev_struct *sai_dev, uint32_t periph, uint32_t block, hal_sai_init_struct *p_init);
/* @END */
/* initialize the SAI structure with the default values */
int32_t hal_sai_struct_init(hal_sai_struct_type_enum hal_struct_type, void *p_struct);
/* deinitialize SAI */
int32_t hal_sai_deinit(hal_sai_dev_struct *sai_dev);

/* transmit amounts of data, poll transmit process and completed status */
int32_t hal_sai_transmit_poll(hal_sai_dev_struct *sai_dev, uint8_t *p_buffer, uint16_t length, uint32_t timeout);
/* receive amounts of data, poll receive process and completed status */
int32_t hal_sai_receive_poll(hal_sai_dev_struct *sai_dev, uint8_t *p_buffer, uint16_t length, uint32_t timeout);
/* transmit amounts of data by interrupt method */
int32_t hal_sai_transmit_interrupt(hal_sai_dev_struct *sai_dev, uint8_t *p_buffer, uint16_t length, \
                                   hal_sai_user_callback_struct *p_user_callback);
/* receive amounts of data by interrupt method */
int32_t hal_sai_receive_interrupt(hal_sai_dev_struct *sai_dev, uint8_t *p_buffer, uint16_t length, \
                                  hal_sai_user_callback_struct *p_user_callback);
/* transmit amounts of data by dma method */
int32_t hal_sai_transmit_dma(hal_sai_dev_struct *sai_dev, uint8_t *p_buffer, uint16_t length, \
                             hal_sai_user_callback_struct *p_user_callback);
/* receive amounts of data by dma method */
int32_t hal_sai_receive_dma(hal_sai_dev_struct *sai_dev, uint8_t *p_buffer, uint16_t length, \
                            hal_sai_user_callback_struct *p_user_callback);

/* SAI interrupt handle functions */
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_sai_irq_handle_set(hal_sai_dev_struct *sai_dev, hal_sai_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_sai_irq_handle_all_reset(hal_sai_dev_struct *sai_dev);
/* SAI interrupt handler content function,which is merely used in SAIx_IRQHandler */
void hal_sai_irq(hal_sai_dev_struct *sai_dev);

/* SAI DMA functions */
/* pause the audio stream playing from the media */
int32_t hal_sai_dma_pause(hal_sai_dev_struct *sai_dev);
/* resume the audio stream playing from the media */
int32_t hal_sai_dma_resume(hal_sai_dev_struct *sai_dev);
/* stop the audio stream playing from the media */
int32_t hal_sai_dma_stop(hal_sai_dev_struct *sai_dev);
/* abort the current transfer and disable the SAI */
int32_t hal_sai_abort(hal_sai_dev_struct *sai_dev);

/* enable the tx mute detection */
int32_t hal_sai_tx_mute_enable(hal_sai_dev_struct *sai_dev, uint16_t value);
/* disable the tx mute detection */
int32_t hal_sai_tx_mute_disable(hal_sai_dev_struct *sai_dev);
/* enable the rx mute detection */
int32_t hal_sai_rx_mute_enable(hal_sai_dev_struct *sai_dev, uint16_t counter);
/* disable the rx mute detection */
int32_t hal_sai_rx_mute_disable(hal_sai_dev_struct *sai_dev);

/* SAI pdm mode functions */
/* configure SAI pdm mode microphone number */
int32_t hal_sai_pdm_microphone_number_config(hal_sai_dev_struct *sai_dev, uint32_t microphonenum);
/* configure SAI pdm mode microphone delay */
int32_t hal_sai_pdm_delay_config(hal_sai_dev_struct *sai_dev, uint32_t microphone, uint32_t delay);

/* state and error functions */
/* return SAI periph state */
hal_sai_state_enum hal_sai_state_get(hal_sai_dev_struct *sai_dev);
/* return SAI error code */
uint32_t hal_sai_error_get(hal_sai_dev_struct *sai_dev);

/* function declarations */
/* flag and interrupt functions */
/* enable the SAI interrupt */
void hals_sai_interrupt_enable(uint32_t sai_periph, uint32_t block, uint32_t interrupt);
/* disable the SAI interrupt */
void hals_sai_interrupt_disable(uint32_t sai_periph, uint32_t block, uint32_t interrupt);
/* get the SAI interrupt flag */
FlagStatus hals_sai_interrupt_flag_get(uint32_t sai_periph, uint32_t block, uint32_t interrupt);
/* clear the SAI interrupt flag */
void hals_sai_interrupt_flag_clear(uint32_t sai_periph, uint32_t block, uint32_t interrupt);
/* get the SAI flag */
FlagStatus hals_sai_flag_get(uint32_t sai_periph, uint32_t block, uint32_t flag);
/* clear the SAI flag */
void hals_sai_flag_clear(uint32_t sai_periph, uint32_t block, uint32_t flag);

#endif /* GD32H7XX_HAL_SAI_H */
