/*!
    \file    gd32h7xx_hal_rcu.h
    \brief   definitions for the RCU

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef  GD32H7XX_HAL_RCU_H
#define  GD32H7XX_HAL_RCU_H

#include "gd32h7xx_hal.h"

/* RCU definitions */
#define RCU                             RCU_BASE                                        /*!< RCU base address */

/* registers definitions */
#define RCU_CTL                         REG32(RCU + 0x00000000U)                        /*!< control register */
#define RCU_PLL0                        REG32(RCU + 0x00000004U)                        /*!< PLL 0 register */
#define RCU_CFG0                        REG32(RCU + 0x00000008U)                        /*!< configuration register 0 */
#define RCU_INT                         REG32(RCU + 0x0000000CU)                        /*!< interrupt register */
#define RCU_AHB1RST                     REG32(RCU + 0x00000010U)                        /*!< AHB1 reset register */
#define RCU_AHB2RST                     REG32(RCU + 0x00000014U)                        /*!< AHB2 reset register */
#define RCU_AHB3RST                     REG32(RCU + 0x00000018U)                        /*!< AHB3 reset register */
#define RCU_AHB4RST                     REG32(RCU + 0x0000001CU)                        /*!< AHB4 reset register */
#define RCU_APB1RST                     REG32(RCU + 0x00000020U)                        /*!< APB1 reset register */
#define RCU_APB2RST                     REG32(RCU + 0x00000024U)                        /*!< APB2 reset register */
#define RCU_APB3RST                     REG32(RCU + 0x00000028U)                        /*!< APB3 reset register */
#define RCU_APB4RST                     REG32(RCU + 0x0000002CU)                        /*!< APB4 reset register */
#define RCU_AHB1EN                      REG32(RCU + 0x00000030U)                        /*!< AHB1 enable register */
#define RCU_AHB2EN                      REG32(RCU + 0x00000034U)                        /*!< AHB2 enable register */
#define RCU_AHB3EN                      REG32(RCU + 0x00000038U)                        /*!< AHB3 enable register */
#define RCU_AHB4EN                      REG32(RCU + 0x0000003CU)                        /*!< AHB4 enable register */
#define RCU_APB1EN                      REG32(RCU + 0x00000040U)                        /*!< APB1 enable register */
#define RCU_APB2EN                      REG32(RCU + 0x00000044U)                        /*!< APB2 enable register */
#define RCU_APB3EN                      REG32(RCU + 0x00000048U)                        /*!< APB3 enable register */
#define RCU_APB4EN                      REG32(RCU + 0x0000004CU)                        /*!< APB4 enable register */
#define RCU_AHB1SPEN                    REG32(RCU + 0x00000050U)                        /*!< AHB1SPEN enable register */
#define RCU_AHB2SPEN                    REG32(RCU + 0x00000054U)                        /*!< AHB2SPEN enable register */
#define RCU_AHB3SPEN                    REG32(RCU + 0x00000058U)                        /*!< AHB3SPEN enable register */
#define RCU_AHB4SPEN                    REG32(RCU + 0x0000005CU)                        /*!< AHB4SPEN enable register */
#define RCU_APB1SPEN                    REG32(RCU + 0x00000060U)                        /*!< APB1SPEN enable register */
#define RCU_APB2SPEN                    REG32(RCU + 0x00000064U)                        /*!< APB2SPEN enable register */
#define RCU_APB3SPEN                    REG32(RCU + 0x00000068U)                        /*!< APB3SPEN enable register */
#define RCU_APB4SPEN                    REG32(RCU + 0x0000006CU)                        /*!< APB4SPEN enable register */
#define RCU_BDCTL                       REG32(RCU + 0x00000070U)                        /*!< backup domain control register */
#define RCU_RSTSCK                      REG32(RCU + 0x00000074U)                        /*!< reset source /clock register */
#define RCU_PLLADDCTL                   REG32(RCU + 0x00000080U)                        /*!< PLL clock attached control register */
#define RCU_PLL1                        REG32(RCU + 0x00000084U)                        /*!< PLL1  register */
#define RCU_PLL2                        REG32(RCU + 0x00000088U)                        /*!< PLL2  register */
#define RCU_CFG1                        REG32(RCU + 0x0000008cU)                        /*!< configuration register 1 */
#define RCU_CFG2                        REG32(RCU + 0x00000090U)                        /*!< configuration register 2 */
#define RCU_CFG3                        REG32(RCU + 0x00000094U)                        /*!< configuration register 3 */
#define RCU_PLLALL                      REG32(RCU + 0x00000098U)                        /*!< PLL Control register */
#define RCU_PLL0FRA                     REG32(RCU + 0x0000009cU)                        /*!< PLL0 decimal configuration register */
#define RCU_PLL1FRA                     REG32(RCU + 0x000000A0U)                        /*!< PLL1 decimal configuration register */
#define RCU_PLL2FRA                     REG32(RCU + 0x000000A4U)                        /*!< PLL2 decimal configuration register */
#define RCU_ADDCTL0                     REG32(RCU + 0x000000C0U)                        /*!< additional clock control0 register */
#define RCU_ADDCTL1                     REG32(RCU + 0x000000C4U)                        /*!< additional clock control1 register */
#define RCU_ADDINT                      REG32(RCU + 0x000000CCU)                        /*!< additional clock interrupt register */
#define RCU_CFG4                        REG32(RCU + 0x000000D0U)                        /*!< configuration register 4 */
#define RCU_USBCLKCTL                   REG32(RCU + 0x000000D4U)                        /*!< USB Clock control register */
#define RCU_PLLUSBCFG                   REG32(RCU + 0x000000D8U)                        /*!< USB Clock configuration register */
#define RCU_ADDAPB2RST                  REG32(RCU + 0x000000E0U)                        /*!< APB2 attached reset register */
#define RCU_ADDAPB3RST                  REG32(RCU + 0x000000E4U)                        /*!< APB3 attached reset register */
#define RCU_ADDAPB2SPEN                 REG32(RCU + 0x000000E8U)                        /*!< APB2 append the sleep mode enable register */
#define RCU_CFG5                        REG32(RCU + 0x000000F0U)                        /*!< configuration register 5 */

/* bits definitions */
/* RCU_CTL */
#define RCU_CTL_IRC64MADJ               BITS(0,6)                                       /*!< Internal 64MHz RC oscillator clock adjustment value */
#define RCU_CTL_IRC64MCALIB             BITS(7,15)                                      /*!< Internal 64MHz RC oscillator calibration value */
#define RCU_CTL_HXTALEN                 BIT(16)                                         /*!< High Speed Crystal Oscillator (HXTAL) enabled */
#define RCU_CTL_HXTALSTB                BIT(17)                                         /*!< High speed crystal oscillator (HXTAL) clock stable flag bit */
#define RCU_CTL_HXTALBPS                BIT(18)                                         /*!< The clock bypass mode of High Speed Crystal oscillator (HXTAL) is enabled */
#define RCU_CTL_CKMEN                   BIT(19)                                         /*!< HXTAL The clock monitor is enabled */
#define RCU_CTL_PLL0EN                  BIT(24)                                         /*!< PLL0 enable */
#define RCU_CTL_PLL0STB                 BIT(25)                                         /*!< PLL0 clock stability flag bit */
#define RCU_CTL_PLL1EN                  BIT(26)                                         /*!< PLL1 enable */
#define RCU_CTL_PLL1STB                 BIT(27)                                         /*!< PLL1 clock stability flag bit */
#define RCU_CTL_PLL2EN                  BIT(28)                                         /*!< PLL2 enabled */
#define RCU_CTL_PLL2STB                 BIT(29)                                         /*!< PLL2 clock stability flag bit */
#define RCU_CTL_IRC64MEN                BIT(30)                                         /*!< Internal 64MHz RC oscillator is enabled */
#define RCU_CTL_IRC64MSTB               BIT(31)                                         /*!< Internal 64MHz RC oscillator stable marker bit */

/* RCU_PLL0 */
#define RCU_PLL0_PLL0PSC                BITS(0,5)                                       /*!< PLL0 VCO source clock divider */
#define RCU_PLL0_PLL0N                  BITS(6,14)                                      /*!< PLL0 VCO clock multiplier factor */
#define RCU_PLL0_PLL0P                  BITS(16,22)                                     /*!< PLL0P output frequency division coefficient (PLL0 VCO clock as input）*/
#define RCU_PLL0_PLL0R                  BITS(24,30)                                     /*!< PLL0R output frequency dividing coefficient （PLL0 VCO clock as input）*/
#define RCU_PLL0_PLLSTBSRC              BIT(31)                                         /*!< PLL stabilized signal source */

/* RCU_CFG0 */
#define RCU_CFG0_SCS                    BITS(0,1)                                       /*!< System clock selection */
#define RCU_CFG0_SCSS                   BITS(2,3)                                       /*!< System clock selection Status */
#define RCU_CFG0_AHBPSC                 BITS(4,7)                                       /*!< AHB/AXI pre-split selection */
#define RCU_CFG0_APB1PSC                BITS(10,12)                                     /*!< APB1 pre-split selection */
#define RCU_CFG0_APB2PSC                BITS(13,15)                                     /*!< APB2 pre-split selection */
#define RCU_CFG0_RTCDIV                 BITS(16,21)                                     /*!< RTC clock frequency division coefficient */
#define RCU_CFG0_APB4PSC                BITS(24,26)                                     /*!< APB4 Pre-division selection */
#define RCU_CFG0_APB3PSC                BITS(27,29)                                     /*!< APB3 Pre-division selection */
#define RCU_CFG0_I2C0SEL                BITS(30,31)                                     /*!< I2C0 Select the clock source */

/* RCU_INT */
#define RCU_INT_IRC32KSTBIF             BIT(0)                                          /*!< IRC32K stabilization interrupt flag */
#define RCU_INT_LXTALSTBIF              BIT(1)                                          /*!< LXTAL stabilization interrupt flag */
#define RCU_INT_IRC64MSTBIF             BIT(2)                                          /*!< IRC64M stabilization interrupt flag */
#define RCU_INT_HXTALSTBIF              BIT(3)                                          /*!< HXTAL stabilization interrupt flag */
#define RCU_INT_PLL0STBIF               BIT(4)                                          /*!< PLL0 stabilization interrupt flag */
#define RCU_INT_PLL1STBIF               BIT(5)                                          /*!< PLL1 stabilization interrupt flag */
#define RCU_INT_PLL2STBIF               BIT(6)                                          /*!< PLL2 stabilization interrupt flag */
#define RCU_INT_CKMIF                   BIT(7)                                          /*!< HXTAL clock stuck interrupt flag */
#define RCU_INT_IRC32KSTBIE             BIT(8)                                          /*!< IRC32K stabilization interrupt flag */
#define RCU_INT_LXTALSTBIE              BIT(9)                                          /*!< LXTAL stabilization interrupt enable */
#define RCU_INT_IRC64MSTBIE             BIT(10)                                         /*!< IRC64M stabilization interrupt enable */
#define RCU_INT_HXTALSTBIE              BIT(11)                                         /*!< HXTAL stabilization interrupt enable */
#define RCU_INT_PLL0STBIE               BIT(12)                                         /*!< The PLL0 clock stability interrupt function was enabled */
#define RCU_INT_PLL1STBIE               BIT(13)                                         /*!< The PLL1 clock stability interrupt function was enabled */
#define RCU_INT_PLL2STBIE               BIT(14)                                         /*!< The PLL2 clock stability interrupt function was enabled */
#define RCU_INT_IRC32KSTBIC             BIT(16)                                         /*!< IRC32K stabilization interrupt clear */
#define RCU_INT_LXTALSTBIC              BIT(17)                                         /*!< LXTAL stabilization interrupt clear */
#define RCU_INT_IRC64MSTBIC             BIT(18)                                         /*!< IRC64M stabilization interrupt clear */
#define RCU_INT_HXTALSTBIC              BIT(19)                                         /*!< HXTAL stabilization interrupt clear */
#define RCU_INT_PLL0STBIC               BIT(20)                                         /*!< PLL stabilization interrupt clear */
#define RCU_INT_PLL1STBIC               BIT(21)                                         /*!< PLL stabilization interrupt clear */
#define RCU_INT_PLL2STBIC               BIT(22)                                         /*!< PLL stabilization interrupt clear */
#define RCU_INT_CKMIC                   BIT(23)                                         /*!< HXTAL clock stuck interrupt clear */
#define RCU_INT_LPIRC4MSTBIF            BIT(24)                                         /*!< LPIRC4M Clock stability interrupt flag bit */
#define RCU_INT_LPIRC4MSTBIE            BIT(25)                                         /*!< LPIRC4M clock stability interrupt enabled */
#define RCU_INT_LPIRC4MSTBIC            BIT(26)                                         /*!< LPIRC4M clock stability interrupts clearing */
#define RCU_INT_LCKMIF                  BIT(27)                                         /*!< LXTAL clock block interrupt flag bit */
#define RCU_INT_LCKMIC                  BIT(28)                                         /*!< LXTAL clock block interrupt clear */

/* RCU_AHB1RST */
#define RCU_AHB1RST_ENET1RST            BIT(0)                                          /*!< Ethernet1 Reset bit */
#define RCU_AHB1RST_USBHS0RST           BIT(14)                                         /*!< USBHS0 Reset bit */
#define RCU_AHB1RST_DMA0RST             BIT(21)                                         /*!< DMA0 Reset bit */
#define RCU_AHB1RST_DMA1RST             BIT(22)                                         /*!< DMA1 Reset bit */
#define RCU_AHB1RST_DMAMUXRST           BIT(23)                                         /*!< DMAMUX Reset bit */
#define RCU_AHB1RST_ENET0RST            BIT(25)                                         /*!< Ethernet0 Reset bit */
#define RCU_AHB1RST_USBHS1RST           BIT(29)                                         /*!< USBHS1 Reset bit */

/* RCU_AHB2RST */
#define RCU_AHB2RST_DCIRST              BIT(0)                                          /*!< DCI Reset bit */
#define RCU_AHB2RST_FACRST              BIT(1)                                          /*!< FAC Reset bit */
#define RCU_AHB2RST_SDIO1RST            BIT(2)                                          /*!< SDIO1 Reset bit */
#define RCU_AHB2RST_CAURST              BIT(3)                                          /*!< CAU Reset bit */
#define RCU_AHB2RST_HAURST              BIT(4)                                          /*!< HAU Reset bit */
#define RCU_AHB2RST_TRNGRST             BIT(6)                                          /*!< TRNG Reset bit */
#define RCU_AHB2RST_TMURST              BIT(7)                                          /*!< TMU Reset bit */

/* RCU_AHB3RST */
#define RCU_AHB3RST_EXMCRST             BIT(0)                                          /*!< EXMC Reset bit */
#define RCU_AHB3RST_IPARST              BIT(1)                                          /*!< IPA Reset bit */
#define RCU_AHB3RST_SDIO0RST            BIT(2)                                          /*!< SDIO0 Reset bit */
#define RCU_AHB3RST_MDMARST             BIT(3)                                          /*!< MDMA Reset bit */
#define RCU_AHB3RST_OSPIMRST            BIT(4)                                          /*!< OSPIM Reset bit */
#define RCU_AHB3RST_OSPI0RST            BIT(5)                                          /*!< OSPI0 Reset bit */
#define RCU_AHB3RST_OSPI1RST            BIT(6)                                          /*!< OSPI1 Reset bit */
#define RCU_AHB3RST_RTDEC0RST           BIT(8)                                          /*!< RTDEC0 Reset bit */
#define RCU_AHB3RST_RTDEC1RST           BIT(9)                                          /*!< RTDEC1 Reset bit */

/* RCU_AHB4RST */
#define RCU_AHB4RST_PARST               BIT(0)                                          /*!< GPIO A Reset bit */
#define RCU_AHB4RST_PBRST               BIT(1)                                          /*!< GPIO B Reset bit */
#define RCU_AHB4RST_PCRST               BIT(2)                                          /*!< GPIO C Reset bit */
#define RCU_AHB4RST_PDRST               BIT(3)                                          /*!< GPIO D Reset bit */
#define RCU_AHB4RST_PERST               BIT(4)                                          /*!< GPIO E Reset bit */
#define RCU_AHB4RST_PFRST               BIT(5)                                          /*!< GPIO F Reset bit */
#define RCU_AHB4RST_PGRST               BIT(6)                                          /*!< GPIO G Reset bit */
#define RCU_AHB4RST_PHRST               BIT(7)                                          /*!< GPIO H Reset bit */
#define RCU_AHB4RST_PJRST               BIT(8)                                          /*!< GPIO J Reset bit */
#define RCU_AHB4RST_PKRST               BIT(9)                                          /*!< GPIO K Reset bit */
#define RCU_AHB4RST_CRCRST              BIT(14)                                         /*!< CRC Reset bit */
#define RCU_AHB4RST_HWSEMRST            BIT(15)                                         /*!< HWSEM Reset bit */

/* RCU_APB1RST */
#define RCU_APB1RST_TIMER1RST           BIT(0)                                          /*!< TIMER1RST Reset bit */
#define RCU_APB1RST_TIMER2RST           BIT(1)                                          /*!< TIMER2RST Reset bit */
#define RCU_APB1RST_TIMER3RST           BIT(2)                                          /*!< TIMER3RST Reset bit */
#define RCU_APB1RST_TIMER4RST           BIT(3)                                          /*!< TIMER4RST Reset bit */
#define RCU_APB1RST_TIMER5RST           BIT(4)                                          /*!< TIMER5RST Reset bit */
#define RCU_APB1RST_TIMER6RST           BIT(5)                                          /*!< TIMER6RST Reset bit */
#define RCU_APB1RST_TIMER22RST          BIT(6)                                          /*!< TIMER22RST Reset bit */
#define RCU_APB1RST_TIMER23RST          BIT(7)                                          /*!< TIMER23RST Reset bit */
#define RCU_APB1RST_TIMER30RST          BIT(8)                                          /*!< TIMER30RST Reset bit */
#define RCU_APB1RST_TIMER31RST          BIT(9)                                          /*!< TIMER31RST Reset bit */
#define RCU_APB1RST_TIMER50RST          BIT(10)                                         /*!< TIMER50RST Reset bit */
#define RCU_APB1RST_TIMER51RST          BIT(11)                                         /*!< TIMER51RST Reset bit */
#define RCU_APB1RST_RSPDIFRST           BIT(13)                                         /*!< RSPDIF Reset bit */
#define RCU_APB1RST_SPI1RST             BIT(14)                                         /*!< SPI1 Reset bit */
#define RCU_APB1RST_SPI2RST             BIT(15)                                         /*!< SPI2 Reset bit */
#define RCU_APB1RST_MDIORST             BIT(16)                                         /*!< MDIO Reset bit */
#define RCU_APB1RST_USART1RST           BIT(17)                                         /*!< USART1 Reset bit */
#define RCU_APB1RST_USART2RST           BIT(18)                                         /*!< USART2 Reset bit */
#define RCU_APB1RST_USART3RST           BIT(19)                                         /*!< USART3 Reset bit */
#define RCU_APB1RST_USART4RST           BIT(20)                                         /*!< USART4 Reset bit */
#define RCU_APB1RST_I2C0RST             BIT(21)                                         /*!< I2C0 Reset bit */
#define RCU_APB1RST_I2C1RST             BIT(22)                                         /*!< I2C1 Reset bit */
#define RCU_APB1RST_I2C2RST             BIT(23)                                         /*!< I2C2 Reset bit */
#define RCU_APB1RST_I2C3RST             BIT(24)                                         /*!< I2C3 Reset bit */
#define RCU_APB1RST_CTCRST              BIT(27)                                         /*!< CTC Reset bit */
#define RCU_APB1RST_DACHOLDRST          BIT(28)                                         /*!< DACHOLD Reset bit */
#define RCU_APB1RST_DACRST              BIT(29)                                         /*!< DAC Reset bit */
#define RCU_APB1RST_UART6RST            BIT(30)                                         /*!< UART6 Reset bit */
#define RCU_APB1RST_UART7RST            BIT(31)                                         /*!< UART7 Reset bit */

/* RCU_APB2RST */
#define RCU_APB2RST_TIMER0RST           BIT(0)                                          /*!< TIMER0 Reset bit */
#define RCU_APB2RST_TIMER7RST           BIT(1)                                          /*!< TIMER7 Reset bit */
#define RCU_APB2RST_USART0RST           BIT(4)                                          /*!< USART0 Reset bit */
#define RCU_APB2RST_USART5RST           BIT(5)                                          /*!< USART5 Reset bit */
#define RCU_APB2RST_ADC0RSTRST          BIT(8)                                          /*!< ADC0 Reset bit */
#define RCU_APB2RST_ADC1RSTRST          BIT(9)                                          /*!< ADC1 Reset bit */
#define RCU_APB2RST_ADC2RSTRST          BIT(10)                                         /*!< ADC2 Reset bit */
#define RCU_APB2RST_SPI0RST             BIT(12)                                         /*!< SPI0 Reset bit */
#define RCU_APB2RST_SPI3RST             BIT(13)                                         /*!< SPI3 Reset bit */
#define RCU_APB2RST_TIMER14RST          BIT(16)                                         /*!< TIMER14 Reset bit */
#define RCU_APB2RST_TIMER15RST          BIT(17)                                         /*!< TIMER15 Reset bit */
#define RCU_APB2RST_TIMER16RST          BIT(18)                                         /*!< TIMER16 Reset bit */
#define RCU_APB2RST_HPDFRST             BIT(19)                                         /*!< HPDF Reset bit */
#define RCU_APB2RST_SPI4RSTRST          BIT(20)                                         /*!< SPI4 Reset bit */
#define RCU_APB2RST_SPI5RSTRST          BIT(21)                                         /*!< SPI5 Reset bit */
#define RCU_APB2RST_SAI0RST             BIT(22)                                         /*!< SAI0 Reset bit */
#define RCU_APB2RST_SAI1RST             BIT(23)                                         /*!< SAI1 Reset bit */
#define RCU_APB2RST_SAI2RST             BIT(24)                                         /*!< SAI2 Reset bit */
#define RCU_APB2RST_TIMER40RST          BIT(25)                                         /*!< TIMER40 Reset bit */
#define RCU_APB2RST_TIMER41RST          BIT(26)                                         /*!< TIMER41 Reset bit */
#define RCU_APB2RST_TIMER42RST          BIT(27)                                         /*!< TIMER42 Reset bit */
#define RCU_APB2RST_TIMER43RST          BIT(28)                                         /*!< TIMER43 Reset bit */
#define RCU_APB2RST_TIMER44RST          BIT(29)                                         /*!< TIMER44 Reset bit */
#define RCU_APB2RST_EDOUTRST            BIT(30)                                         /*!< EDOUT Reset bit */
#define RCU_APB2RST_TRIGSELRST          BIT(31)                                         /*!< TRIGSEL Reset bit */

/* RCU_APB3RST */
#define RCU_APB3RST_TLIRST              BIT(0)                                          /*!< TLIRST Reset bit */
#define RCU_APB3RST_WWDGTRST            BIT(1)                                          /*!< WWDGTRST Reset bit */

/* RCU_APB4RST */
#define RCU_APB4RST_SYSCFGRST           BIT(0)                                          /*!< SYSCFG Reset bit */
#define RCU_APB4RST_CMPRST              BIT(1)                                          /*!< CMP Reset bit */
#define RCU_APB4RST_VREFRST             BIT(2)                                          /*!< VREF Reset bit */
#define RCU_APB4RST_LPDTSRST            BIT(3)                                          /*!< LPDTS Reset bit */
#define RCU_APB4RST_PMURST              BIT(4)                                          /*!< LPDTS Reset bit */

/* RCU_AHB1EN */
#define RCU_AHB1EN_ENE1TEN              BIT(0)                                          /*!< Ethernet1 Enable bit */
#define RCU_AHB1EN_ENET1TXEN            BIT(1)                                          /*!< Ethernet1 TX Enable bit */
#define RCU_AHB1EN_ENET1RXEN            BIT(2)                                          /*!< Ethernet1 RX Enable bit */
#define RCU_AHB1EN_ENET1PTPEN           BIT(3)                                          /*!< Ethernet1 PTP Enable bit */
#define RCU_AHB1EN_USBHS0EN             BIT(14)                                         /*!< USBHS0 Enable bit */
#define RCU_AHB1EN_USBHS0ULPIEN         BIT(15)                                         /*!< USBHS0 ULPI Enable bit */
#define RCU_AHB1EN_DMA0EN               BIT(21)                                         /*!< DMA0 Enable bit */
#define RCU_AHB1EN_DMA1EN               BIT(22)                                         /*!< DMA1 Enable bit */
#define RCU_AHB1EN_DMAMUXEN             BIT(23)                                         /*!< DMAMUX Enable bit */
#define RCU_AHB1EN_ENE0TEN              BIT(25)                                         /*!< Ethernet0 Enable bit */
#define RCU_AHB1EN_ENET0TXEN            BIT(26)                                         /*!< Ethernet0 TX Enable bit */
#define RCU_AHB1EN_ENET0RXEN            BIT(27)                                         /*!< Ethernet0 RX Enable bit */
#define RCU_AHB1EN_ENET0PTPEN           BIT(28)                                         /*!< Ethernet0 PTP Enable bit */
#define RCU_AHB1EN_USBHS1EN             BIT(29)                                         /*!< USBHS1 Enable bit Enable bit */
#define RCU_AHB1EN_USBHS1ULPIEN         BIT(30)                                         /*!< USBHS1 ULPI Enable bit */

/* RCU_AHB2EN */
#define RCU_AHB2EN_DCIEN                BIT(0)                                          /*!< DCI Enable bit */
#define RCU_AHB2EN_FACEN                BIT(1)                                          /*!< FAC Enable bit */
#define RCU_AHB2EN_SDIO1EN              BIT(2)                                          /*!< SDIO1 Enable bit */
#define RCU_AHB2EN_CAUEN                BIT(3)                                          /*!< CAU Enable bit */
#define RCU_AHB2EN_HAUEN                BIT(4)                                          /*!< HAU Enable bit */
#define RCU_AHB2EN_TRNGEN               BIT(6)                                          /*!< TRNG Enable bit */
#define RCU_AHB2EN_TMUEN                BIT(7)                                          /*!< TMU Enable bit */
#define RCU_AHB2EN_RAMECCMU1EN          BIT(8)                                          /*!< RAMECCMU1EN Enable bit */

/* RCU_AHB3EN */
#define RCU_AHB3EN_EXMCEN               BIT(0)                                          /*!< EXMC ENable bit */
#define RCU_AHB3EN_IPAEN                BIT(1)                                          /*!< IPA ENable bit */
#define RCU_AHB3EN_SDIO0EN              BIT(2)                                          /*!< SDIO0 ENable bit */
#define RCU_AHB3EN_MDMAEN               BIT(3)                                          /*!< MDMA ENable bit */
#define RCU_AHB3EN_OSPIMEN              BIT(4)                                          /*!< OSPIM ENable bit */
#define RCU_AHB3EN_OSPI0EN              BIT(5)                                          /*!< OSPI0 ENable bit */
#define RCU_AHB3EN_OSPI1EN              BIT(6)                                          /*!< OSPI1 ENable bit */
#define RCU_AHB3EN_RTDEC0EN             BIT(8)                                          /*!< RTDEC0 ENable bit */
#define RCU_AHB3EN_RTDEC1EN             BIT(9)                                          /*!< RTDEC1 ENable bit */
#define RCU_AHB3EN_RAMECCMU0EN          BIT(10)                                         /*!< RAMECCMU0 ENable bit */
#define RCU_AHB3EN_CPUEN                BIT(15)                                         /*!< CPU ENable bit */

/* RCU_AHB4EN */
#define RCU_AHB4EN_PAEN                 BIT(0)                                          /*!< GPIO A ENable bit */
#define RCU_AHB4EN_PBEN                 BIT(1)                                          /*!< GPIO B ENable bit */
#define RCU_AHB4EN_PCEN                 BIT(2)                                          /*!< GPIO C ENable bit */
#define RCU_AHB4EN_PDEN                 BIT(3)                                          /*!< GPIO D ENable bit */
#define RCU_AHB4EN_PEEN                 BIT(4)                                          /*!< GPIO E ENable bit */
#define RCU_AHB4EN_PFEN                 BIT(5)                                          /*!< GPIO F ENable bit */
#define RCU_AHB4EN_PGEN                 BIT(6)                                          /*!< GPIO G ENable bit */
#define RCU_AHB4EN_PHEN                 BIT(7)                                          /*!< GPIO H ENable bit */
#define RCU_AHB4EN_PJEN                 BIT(8)                                          /*!< GPIO J ENable bit */
#define RCU_AHB4EN_PKEN                 BIT(9)                                          /*!< GPIO K ENable bit */
#define RCU_AHB4EN_BKPSRAMEN            BIT(13)                                         /*!< BKPSRAM ENable bit */
#define RCU_AHB4EN_CRCEN                BIT(14)                                         /*!< CRC ENable bit */
#define RCU_AHB4EN_HWSEMEN              BIT(15)                                         /*!< HWSEM  Enable bit */

/* RCU_APB1EN */
#define RCU_APB1EN_TIMER1EN             BIT(0)                                          /*!< TIMER1EN Enable bit */
#define RCU_APB1EN_TIMER2EN             BIT(1)                                          /*!< TIMER2EN Enable bit */
#define RCU_APB1EN_TIMER3EN             BIT(2)                                          /*!< TIMER3EN Enable bit */
#define RCU_APB1EN_TIMER4EN             BIT(3)                                          /*!< TIMER4EN Enable bit */
#define RCU_APB1EN_TIMER5EN             BIT(4)                                          /*!< TIMER5EN Enable bit */
#define RCU_APB1EN_TIMER6EN             BIT(5)                                          /*!< TIMER6EN Enable bit */
#define RCU_APB1EN_TIMER22EN            BIT(6)                                          /*!< TIMER22EN Enable bit */
#define RCU_APB1EN_TIMER23EN            BIT(7)                                          /*!< TIMER23EN Enable bit */
#define RCU_APB1EN_TIMER30EN            BIT(8)                                          /*!< TIMER30EN Enable bit */
#define RCU_APB1EN_TIMER31EN            BIT(9)                                          /*!< TIMER31EN Enable bit */
#define RCU_APB1EN_TIMER50EN            BIT(10)                                         /*!< TIMER50EN Enable bit */
#define RCU_APB1EN_TIMER51EN            BIT(11)                                         /*!< TIMER51EN Enable bit */
#define RCU_APB1EN_RSPDIFEN             BIT(13)                                         /*!< RSPDIF Enable bit */
#define RCU_APB1EN_SPI1EN               BIT(14)                                         /*!< SPI1 Enable bit */
#define RCU_APB1EN_SPI2EN               BIT(15)                                         /*!< SPI2 Enable bit */
#define RCU_APB1EN_MDIOEN               BIT(16)                                         /*!< MDIO Enable bit */
#define RCU_APB1EN_USART1EN             BIT(17)                                         /*!< USART1 Enable bit */
#define RCU_APB1EN_USART2EN             BIT(18)                                         /*!< USART2 Enable bit */
#define RCU_APB1EN_USART3EN             BIT(19)                                         /*!< USART3 Enable bit */
#define RCU_APB1EN_USART4EN             BIT(20)                                         /*!< USART4 Enable bit */
#define RCU_APB1EN_I2C0EN               BIT(21)                                         /*!< I2C0 Enable bit */
#define RCU_APB1EN_I2C1EN               BIT(22)                                         /*!< I2C1 Enable bit */
#define RCU_APB1EN_I2C2EN               BIT(23)                                         /*!< I2C2 Enable bit */
#define RCU_APB1EN_I2C3EN               BIT(24)                                         /*!< I2C3 Enable bit */
#define RCU_APB1EN_CTCEN                BIT(27)                                         /*!< CTC Enable bit */
#define RCU_APB1EN_DACHOLDEN            BIT(28)                                         /*!< DACHOLD Enable bit */
#define RCU_APB1EN_DACEN                BIT(29)                                         /*!< DAC Enable bit */
#define RCU_APB1EN_UART6EN              BIT(30)                                         /*!< UART6 Enable bit */
#define RCU_APB1EN_UART7EN              BIT(31)                                         /*!< UART7 Enable bit */

/* RCU_APB2EN */
#define RCU_APB2EN_TIMER0EN             BIT(0)                                          /*!< TIMER0 Enable bit */
#define RCU_APB2EN_TIMER7EN             BIT(1)                                          /*!< TIMER7 Enable bit */
#define RCU_APB2EN_USART0EN             BIT(4)                                          /*!< USART0 Enable bit */
#define RCU_APB2EN_USART5EN             BIT(5)                                          /*!< USART5 Enable bit */
#define RCU_APB2EN_ADC0ENEN             BIT(8)                                          /*!< ADC0 Enable bit */
#define RCU_APB2EN_ADC1ENEN             BIT(9)                                          /*!< ADC1 Enable bit */
#define RCU_APB2EN_ADC2ENEN             BIT(10)                                         /*!< ADC2 Enable bit */
#define RCU_APB2EN_SPI0EN               BIT(12)                                         /*!< SPI0 Enable bit */
#define RCU_APB2EN_SPI3EN               BIT(13)                                         /*!< SPI3 Enable bit */
#define RCU_APB2EN_TIMER14EN            BIT(16)                                         /*!< TIMER14 Enable bit */
#define RCU_APB2EN_TIMER15EN            BIT(17)                                         /*!< TIMER15 Enable bit */
#define RCU_APB2EN_TIMER16EN            BIT(18)                                         /*!< TIMER16  Enable bit */
#define RCU_APB2EN_HPDFEN               BIT(19)                                         /*!< HPDF Enable bit */
#define RCU_APB2EN_SPI4ENEN             BIT(20)                                         /*!< SPI4 Enable bit */
#define RCU_APB2EN_SPI5ENEN             BIT(21)                                         /*!< SPI5 Enable bit */
#define RCU_APB2EN_SAI0EN               BIT(22)                                         /*!< SAI0 Enable bit */
#define RCU_APB2EN_SAI1EN               BIT(23)                                         /*!< SAI1 Enable bit */
#define RCU_APB2EN_SAI2EN               BIT(24)                                         /*!< SAI2 Enable bit */
#define RCU_APB2EN_TIMER40EN            BIT(25)                                         /*!< TIMER40 Enable bit */
#define RCU_APB2EN_TIMER41EN            BIT(26)                                         /*!< TIMER41 Enable bit */
#define RCU_APB2EN_TIMER42EN            BIT(27)                                         /*!< TIMER42 Enable bit */
#define RCU_APB2EN_TIMER43EN            BIT(28)                                         /*!< TIMER43  Enable bit */
#define RCU_APB2EN_TIMER44EN            BIT(29)                                         /*!< TIMER44 Enable bit */
#define RCU_APB2EN_EDOUTEN              BIT(30)                                         /*!< EDOUT Enable bit */
#define RCU_APB2EN_TRIGSELEN            BIT(31)                                         /*!< TRIGSEL Enable bit */

/* RCU_APB3EN */
#define RCU_APB3EN_TLIEN                BIT(0)                                          /*!< TLIEN Enable bit */
#define RCU_APB3EN_WWDGTEN              BIT(1)                                          /*!< WWDGTEN Enable bit */

/* RCU_APB4EN */
#define RCU_APB4EN_SYSCFGEN             BIT(0)                                          /*!< SYSCFG Enable bit */
#define RCU_APB4EN_CMPEN                BIT(1)                                          /*!< CMP Enable bit */
#define RCU_APB4EN_VREFEN               BIT(2)                                          /*!< VREF Enable bit */
#define RCU_APB4EN_LPDTSEN              BIT(3)                                          /*!< LPDTS Enable bit */
#define RCU_APB4EN_PMUEN                BIT(4)                                          /*!< LPDTS Enable bit */

/* RCU_AHB1SPEN */
#define RCU_AHB1SPEN_ENET1SPEN          BIT(0)                                          /*!< Ethernet 1 Enable bit In Sleep mode */
#define RCU_AHB1SPEN_ENET1TXSPEN        BIT(1)                                          /*!< Ethernet 1 TX Enable bit In Sleep mode */
#define RCU_AHB1SPEN_ENET1RXSPEN        BIT(2)                                          /*!< Ethernet 1 RX Enable bit In Sleep mode */
#define RCU_AHB1SPEN_ENET1PTPSPEN       BIT(3)                                          /*!< Ethernet 1 PTP Enable bit In Sleep mode */
#define RCU_AHB1SPEN_USBHS0SPEN         BIT(14)                                         /*!< USBHS 0 Enable bit In Sleep mode */
#define RCU_AHB1SPEN_USBHS0ULPISPEN     BIT(15)                                         /*!< USBHS 0 ULPI Enable bit In Sleep mode */
#define RCU_AHB1SPEN_SRAM0SPEN          BIT(16)                                         /*!< LPDTS Enable bit In Sleep mode */
#define RCU_AHB1SPEN_SRAM1SPEN          BIT(17)                                         /*!< LPDTS Enable bit In Sleep mode */
#define RCU_AHB1SPEN_DMA0SPEN           BIT(21)                                         /*!< DMA0 Enable bit In Sleep mode */
#define RCU_AHB1SPEN_DMA1SPEN           BIT(22)                                         /*!< DMA1 Enable bit In Sleep mode */
#define RCU_AHB1SPEN_DMAMUXSPEN         BIT(23)                                         /*!< DMAMUX Enable bit In Sleep mode */
#define RCU_AHB1SPEN_ENET0SPEN          BIT(25)                                         /*!< Ethernet 0 Enable Enable bit In Sleep mode */
#define RCU_AHB1SPEN_ENET0TXSPEN        BIT(26)                                         /*!< Ethernet 0 TX Enable bit In Sleep mode */
#define RCU_AHB1SPEN_ENET0RXSPEN        BIT(27)                                         /*!< Ethernet 0 RX Enable bit In Sleep mode */
#define RCU_AHB1SPEN_ENET0PTPSPEN       BIT(28)                                         /*!< Ethernet 0 PTP Enable bit In Sleep mode */
#define RCU_AHB1SPEN_USBHS1SPEN         BIT(29)                                         /*!< USBHS 1 Enable bit In Sleep mode */
#define RCU_AHB1SPEN_USBHS1ULPISPEN     BIT(30)                                         /*!< USBHS 1 ULPI Enable bit In Sleep mode */

/* RCU_AHB2SPEN */
#define RCU_AHB2SPEN_DCISPEN            BIT(0)                                          /*!< DCI Enable bit In Sleep mode */
#define RCU_AHB2SPEN_FACSPEN            BIT(1)                                          /*!< FAC Enable bit In Sleep mode */
#define RCU_AHB2SPEN_SDIO1SPEN          BIT(2)                                          /*!< SDIO1 Enable bit In Sleep mode */
#define RCU_AHB2SPEN_CAUSPEN            BIT(3)                                          /*!< CAU Enable bit In Sleep mode */
#define RCU_AHB2SPEN_HAUSPEN            BIT(4)                                          /*!< HAU  Enable bit In Sleep mode */
#define RCU_AHB2SPEN_TRNGSPEN           BIT(6)                                          /*!< TRNG  ULPI Enable bit In Sleep mode */
#define RCU_AHB2SPEN_TMUSPEN            BIT(7)                                          /*!< TMU Enable bit In Sleep mode */
#define RCU_AHB2SPEN_RAMECCMU1SPEN      BIT(8)                                          /*!< RAMECCMU1 Enable bit In Sleep mode */

/* RCU_AHB3SPEN */
#define RCU_AHB3SPEN_EXMCSPEN           BIT(0)                                          /*!< EXMC Enable bit In Sleep mode */
#define RCU_AHB3SPEN_IPASPEN            BIT(1)                                          /*!< IPA Enable bit In Sleep mode */
#define RCU_AHB3SPEN_SDIO0SPEN          BIT(2)                                          /*!< SDIO0 Enable bit In Sleep mode */
#define RCU_AHB3SPEN_MDMASPEN           BIT(3)                                          /*!< MDMU Enable bit In Sleep mode */
#define RCU_AHB3SPEN_OSPIMSPEN          BIT(4)                                          /*!< OSPIM  Enable bit In Sleep mode */
#define RCU_AHB3SPEN_OSPI0SPEN          BIT(5)                                          /*!< OSPI0  ULPI Enable bit In Sleep mode */
#define RCU_AHB3SPEN_OSPI1SPEN          BIT(6)                                          /*!< OSPI1 Enable bit In Sleep mode */
#define RCU_AHB3SPEN_RTDEC0SPEN         BIT(8)                                          /*!< RTDEC0 Enable bit In Sleep mode */
#define RCU_AHB3SPEN_RTDEC1SPEN         BIT(9)                                          /*!< RTDEC1  ULPI Enable bit In Sleep mode */
#define RCU_AHB3SPEN_RAMECCMU0SPEN      BIT(10)                                         /*!< RAMECCMU0 Enable bit In Sleep mode */
#define RCU_AHB3SPEN_AXISRAMSPEN        BIT(14)                                         /*!< AXISRAM Enable bit In Sleep mode */
#define RCU_AHB3SPEN_FMCSPEN            BIT(15)                                         /*!< FMC Enable bit In Sleep mode */

/* RCU_AHB4SPEN */
#define RCU_AHB4SPEN_PASPEN             BIT(0)                                          /*!< GPIO A Enable bit In Sleep mode */
#define RCU_AHB4SPEN_PBSPEN             BIT(1)                                          /*!< GPIO B Enable bit In Sleep mode */
#define RCU_AHB4SPEN_PCSPEN             BIT(2)                                          /*!< GPIO C Enable bit In Sleep mode */
#define RCU_AHB4SPEN_PDSPEN             BIT(3)                                          /*!< GPIO D Enable bit In Sleep mode */
#define RCU_AHB4SPEN_PESPEN             BIT(4)                                          /*!< GPIO E Enable bit In Sleep mode */
#define RCU_AHB4SPEN_PFSPEN             BIT(5)                                          /*!< GPIO F Enable bit In Sleep mode */
#define RCU_AHB4SPEN_PGSPEN             BIT(6)                                          /*!< GPIO G Enable bit In Sleep mode */
#define RCU_AHB4SPEN_PHSPEN             BIT(7)                                          /*!< GPIO H Enable bit In Sleep mode */
#define RCU_AHB4SPEN_PJSPEN             BIT(8)                                          /*!< GPIO J Enable bit In Sleep mode */
#define RCU_AHB4SPEN_PKSPEN             BIT(9)                                          /*!< GPIO K ULPI Enable bit In Sleep mode */
#define RCU_AHB4SPEN_BKPSRAMSPEN        BIT(13)                                         /*!< BKPSRAM Enable bit In Sleep mode */
#define RCU_AHB4SPEN_CRCSPEN            BIT(14)                                         /*!< CRC Enable bit In Sleep mode */

/* RCU_APB1SPEN */
#define RCU_APB1SPEN_TIMER1SPEN         BIT(0)                                          /*!< TIMER1 clock enable when sleep mode */
#define RCU_APB1SPEN_TIMER2SPEN         BIT(1)                                          /*!< TIMER2 clock enable when sleep mode */
#define RCU_APB1SPEN_TIMER3SPEN         BIT(2)                                          /*!< TIMER3 clock enable when sleep mode */
#define RCU_APB1SPEN_TIMER4SPEN         BIT(3)                                          /*!< TIMER4 clock enable when sleep mode */
#define RCU_APB1SPEN_TIMER5SPEN         BIT(4)                                          /*!< TIMER5 clock enable when sleep mode */
#define RCU_APB1SPEN_TIMER6SPEN         BIT(5)                                          /*!< TIMER6 clock enable when sleep mode */
#define RCU_APB1SPEN_TIMER22SPEN        BIT(6)                                          /*!< TIMER22 clock enable when sleep mode */
#define RCU_APB1SPEN_TIMER23SPEN        BIT(7)                                          /*!< TIMER23 clock enable when sleep mode */
#define RCU_APB1SPEN_TIMER30SPEN        BIT(8)                                          /*!< TIMER30 clock enable when sleep mode */
#define RCU_APB1SPEN_TIMER31SPEN        BIT(9)                                          /*!< TIMER31 clock enable when sleep mode */
#define RCU_APB1SPEN_TIMER50SPEN        BIT(10)                                         /*!< TIMER50 clock enable when sleep mode */
#define RCU_APB1SPEN_TIMER51SPEN        BIT(11)                                         /*!< TIMER51 clock enable when sleep mode */
#define RCU_APB1SPEN_RSPDIFSPEN         BIT(13)                                         /*!< RSPDIF clock enable when sleep mode */
#define RCU_APB1SPEN_SPI1SPEN           BIT(14)                                         /*!< SPI1 clock enable when sleep mode */
#define RCU_APB1SPEN_SPI2SPEN           BIT(15)                                         /*!< SPI2 clock enable when sleep mode */
#define RCU_APB1SPEN_MDIOSPEN           BIT(16)                                         /*!< MDIO clock enable when sleep mode */
#define RCU_APB1SPEN_USART1SPEN         BIT(17)                                         /*!< USART1 clock enable when sleep mode */
#define RCU_APB1SPEN_USART2SPEN         BIT(18)                                         /*!< USART2 clock enable when sleep mode */
#define RCU_APB1SPEN_UART3SPEN          BIT(19)                                         /*!< UART3 clock enable when sleep mode */
#define RCU_APB1SPEN_UART4SPEN          BIT(20)                                         /*!< UART4 clock enable when sleep mode */
#define RCU_APB1SPEN_I2C0SPEN           BIT(21)                                         /*!< I2C0 clock enable when sleep mode */
#define RCU_APB1SPEN_I2C1SPEN           BIT(22)                                         /*!< I2C1 clock enable when sleep mode */
#define RCU_APB1SPEN_I2C2SPEN           BIT(23)                                         /*!< I2C2 clock enable when sleep mode */
#define RCU_APB1SPEN_I2C3SPEN           BIT(24)                                         /*!< I2C3 clock enable when sleep mode */
#define RCU_APB1SPEN_CTCSPEN            BIT(27)                                         /*!< CTC clock enable when sleep mode */
#define RCU_APB1SPEN_DACHOLDSPEN        BIT(28)                                         /*!< DAC hold clock enable when sleep mode */
#define RCU_APB1SPEN_DACSPEN            BIT(29)                                         /*!< DAC clock enable when sleep mode */
#define RCU_APB1SPEN_UART6SPEN          BIT(30)                                         /*!< UART6 clock enable when sleep mode */
#define RCU_APB1SPEN_UART7SPEN          BIT(31)                                         /*!< UART7 clock enable when sleep mode */

/* RCU_APB2SPEN */
#define RCU_APB2SPEN_TIMER0SPEN         BIT(0)                                          /*!< TIMER0 clock enable when sleep mode */
#define RCU_APB2SPEN_TIMER7SPEN         BIT(1)                                          /*!< TIMER7 clock enable when sleep mode */
#define RCU_APB2SPEN_USART0SPEN         BIT(4)                                          /*!< USART0 clock enable when sleep mode */
#define RCU_APB2SPEN_USART5SPEN         BIT(5)                                          /*!< USART5 clock enable when sleep mode */
#define RCU_APB2SPEN_ADC0SPEN           BIT(8)                                          /*!< ADC0 clock enable when sleep mode */
#define RCU_APB2SPEN_ADC1SPEN           BIT(9)                                          /*!< ADC1 clock enable when sleep mode */
#define RCU_APB2SPEN_ADC2SPEN           BIT(10)                                         /*!< ADC2 clock enable when sleep mode */
#define RCU_APB2SPEN_SPI0SPEN           BIT(12)                                         /*!< SPI0 clock enable when sleep mode */
#define RCU_APB2SPEN_SPI3SPEN           BIT(13)                                         /*!< SPI3 clock enable when sleep mode */
#define RCU_APB2SPEN_TIMER14SPEN        BIT(16)                                         /*!< TIMER14 clock enable when sleep mode */
#define RCU_APB2SPEN_TIMER15SPEN        BIT(17)                                         /*!< TIMER15 clock enable when sleep mode */
#define RCU_APB2SPEN_TIMER16SPEN        BIT(18)                                         /*!< TIMER16 clock enable when sleep mode */
#define RCU_APB2SPEN_HPDFSPEN           BIT(19)                                         /*!< HPDF clock enable when sleep mode */
#define RCU_APB2SPEN_SPI4SPEN           BIT(20)                                         /*!< SPI4 clock enable when sleep mode */
#define RCU_APB2SPEN_SPI5SPEN           BIT(21)                                         /*!< SPI5 clock enable when sleep mode */
#define RCU_APB2SPEN_SAI0SPEN           BIT(22)                                         /*!< SAI0 clock enable when sleep mode */
#define RCU_APB2SPEN_SAI1SPEN           BIT(23)                                         /*!< SAI1 clock enable when sleep mode */
#define RCU_APB2SPEN_SAI2SPEN           BIT(24)                                         /*!< SAI2 clock enable when sleep mode */
#define RCU_APB2SPEN_TIMER40SPEN        BIT(25)                                         /*!< TIMER40 clock enable when sleep mode */
#define RCU_APB2SPEN_TIMER41SPEN        BIT(26)                                         /*!< TIMER41 clock enable when sleep mode */
#define RCU_APB2SPEN_TIMER42SPEN        BIT(27)                                         /*!< TIMER42 clock enable when sleep mode */
#define RCU_APB2SPEN_TIMER43SPEN        BIT(28)                                         /*!< TIMER43 clock enable when sleep mode */
#define RCU_APB2SPEN_TIMER44SPEN        BIT(29)                                         /*!< TIMER44 clock enable when sleep mode */
#define RCU_APB2SPEN_EDOUTSPEN          BIT(30)                                         /*!< EDOUT clock enable when sleep mode */
#define RCU_APB2SPEN_TRIGSELSPEN        BIT(31)                                         /*!< TRIGSEL clock enable when sleep mode */

/* RCU_APB3SPEN */
#define RCU_APB3SPEN_TLISPEN            BIT(0)                                          /*!< TLI clock enable when sleep mode */
#define RCU_APB3SPEN_WWDGTSPEN          BIT(1)                                          /*!< WWDGT clock enable when sleep mode */

/* RCU_APB4SPEN */
#define RCU_APB4SPEN_SYSCFGSPEN         BIT(0)                                          /*!< SYSCFG clock enable when sleep mode */
#define RCU_APB4SPEN_CMPSPEN            BIT(1)                                          /*!< CMP clock enable when sleep mode */
#define RCU_APB4SPEN_VREFSPEN           BIT(2)                                          /*!< VREF clock enable when sleep mode */
#define RCU_APB4SPEN_LPDTSSPEN          BIT(3)                                          /*!< LPDTS clock enable when sleep mode */
#define RCU_APB4SPEN_PMUSPEN            BIT(4)                                          /*!< PMU clock enable when sleep mode */

/* RCU_BDCTL */
#define RCU_BDCTL_LXTALEN               BIT(0)                                          /*!< LXTAL enable */
#define RCU_BDCTL_LXTALSTB              BIT(1)                                          /*!< low speed crystal oscillator stabilization flag */
#define RCU_BDCTL_LXTALBPS              BIT(2)                                          /*!< LXTAL bypass mode enable */
#define RCU_BDCTL_LXTALDRI              BITS(3,4)                                       /*!< LXTAL drive capability */
#define RCU_BDCTL_LCKMEN                BIT(5)                                          /*!< LXTAL clock monitor enable */
#define RCU_BDCTL_LCKMD                 BIT(6)                                          /*!< LXTAL clock failure detection flag */
#define RCU_BDCTL_RTCSRC                BITS(8,9)                                       /*!< RTC clock entry selection */
#define RCU_BDCTL_RTCEN                 BIT(15)                                         /*!< RTC clock enable */
#define RCU_BDCTL_BKPRST                BIT(16)                                         /*!< backup domain reset */

/* RCU_RSTSCK */
#define RCU_RSTSCK_IRC32KEN             BIT(0)                                          /*!< IRC32K enable */
#define RCU_RSTSCK_IRC32KSTB            BIT(1)                                          /*!< IRC32K stabilization flag */
#define RCU_RSTSCK_RSTFC                BIT(24)                                         /*!< reset flag clear */
#define RCU_RSTSCK_BORRSTF              BIT(25)                                         /*!< BOR reset flag */
#define RCU_RSTSCK_EPRSTF               BIT(26)                                         /*!< external pin reset flag */
#define RCU_RSTSCK_PORRSTF              BIT(27)                                         /*!< power reset flag */
#define RCU_RSTSCK_SWRSTF               BIT(28)                                         /*!< software reset flag */
#define RCU_RSTSCK_FWDGTRSTF            BIT(29)                                         /*!< free watchdog timer reset flag */
#define RCU_RSTSCK_WWDGTRSTF            BIT(30)                                         /*!< window watchdog timer reset flag */
#define RCU_RSTSCK_LPRSTF               BIT(31)                                         /*!< low-power reset flag */

/* RCU_PLLADDCTL */
#define RCU_PLLADDCTL_PLL0Q             BITS(0,6)                                       /*!< the PLL0Q output frequency division factor from PLL0 VCO clock */
#define RCU_PLLADDCTL_PLL1Q             BITS(8,14)                                      /*!< the PLL1Q output frequency division factor from PLL1 VCO clock */
#define RCU_PLLADDCTL_PLL2Q             BITS(16,22)                                     /*!< the PLL2Q output frequency division factor from PLL2 VCO clock */
#define RCU_PLLADDCTL_PLL0QEN           BIT(23)                                         /*!< PLL0Q divider output enable */
#define RCU_PLLADDCTL_PLL0REN           BIT(24)                                         /*!< PLL0R divider output enable */
#define RCU_PLLADDCTL_PLL0PEN           BIT(25)                                         /*!< PLL0P divider output enable */
#define RCU_PLLADDCTL_PLL1QEN           BIT(26)                                         /*!< PLL1Q divider output enable */
#define RCU_PLLADDCTL_PLL1REN           BIT(27)                                         /*!< PLL1R divider output enable */
#define RCU_PLLADDCTL_PLL1PEN           BIT(28)                                         /*!< PLL1P divider output enable */
#define RCU_PLLADDCTL_PLL2QEN           BIT(29)                                         /*!< PLL2Q divider output enable */
#define RCU_PLLADDCTL_PLL2REN           BIT(30)                                         /*!< PLL2R divider output enable */
#define RCU_PLLADDCTL_PLL2PEN           BIT(31)                                         /*!< PLL2P divider output enable */

/* RCU_PLL1 */
#define RCU_PLL1_PLL1PSC                BITS(0,5)                                       /*!< the PLL1 VCO source clock prescaler */
#define RCU_PLL1_PLL1N                  BITS(6,14)                                      /*!< the PLL1 VCO clock multi factor */
#define RCU_PLL1_PLL1P                  BITS(16,22)                                     /*!< the PLL1P output frequency division factor from PLL1 VCO clock */
#define RCU_PLL1_PLL1R                  BITS(24,30)                                     /*!< the PLL1R output frequency division factor from PLL1 VCO clock */

/* RCU_PLL2 */
#define RCU_PLL2_PLL2PSC                BITS(0,5)                                       /*!< the PLL2 VCO source clock prescaler */
#define RCU_PLL2_PLL2N                  BITS(6,14)                                      /*!< the PLL2 VCO clock multi factor */
#define RCU_PLL2_PLL2P                  BITS(16,22)                                     /*!< the PLL2P output frequency division factor from PLL2 VCO clock */
#define RCU_PLL2_PLL2R                  BITS(24,30)                                     /*!< the PLL2R output frequency division factor from PLL2 VCO clock */

/* RCU_CFG1 */
#define RCU_CFG1_USART0SEL              BITS(0,1)                                       /*!< USART0 clock source selection */
#define RCU_CFG1_RSPDIFSEL              BITS(4,5)                                       /*!< RSPDIF clock source selection */
#define RCU_CFG1_CAN0SEL                BITS(8,9)                                       /*!< CAN0 clock source selection */
#define RCU_CFG1_CAN1SEL                BITS(10,11)                                     /*!< CAN1 clock source selection */
#define RCU_CFG1_CAN2SEL                BITS(12,13)                                     /*!< CAN2 clock source selection */
#define RCU_CFG1_PERSEL                 BITS(14,15)                                     /*!< CK_PER clock source selection */
#define RCU_CFG1_PLL2RDIV               BITS(16,17)                                     /*!< the divider factor from PLL2R clock */
#define RCU_CFG1_USART1SEL              BITS(18,19)                                     /*!< USART1 clock source selection */
#define RCU_CFG1_USART2SEL              BITS(20,21)                                     /*!< USART2 clock source selection */
#define RCU_CFG1_USART5SEL              BITS(22,23)                                     /*!< USART5 clock source selection */
#define RCU_CFG1_TIMERSEL               BIT(24)                                         /*!< TIMER clock selection */
#define RCU_CFG1_HPDFSEL                BIT(31)                                         /*!< HPDF clock source selection */

/* RCU_CFG2 */
#define RCU_CFG2_CKOUT0DIV              BITS(0,3)                                       /*!< The CK_OUT0 divider which the CK_OUT0 frequency can be reduced */
#define RCU_CFG2_CKOUT0SEL              BITS(4,6)                                       /*!< CKOUT0 clock source selection */
#define RCU_CFG2_CKOUT1DIV              BITS(8,11)                                      /*!< The CK_OUT1 divider which the CK_OUT1 frequency can be reduced */
#define RCU_CFG2_CKOUT1SEL              BITS(12,14)                                     /*!< CKOUT1 clock source selection */
#define RCU_CFG2_SAI0SEL                BITS(16,18)                                     /*!< SAI0 clock source selection */
#define RCU_CFG2_SAI1SEL                BITS(20,22)                                     /*!< SAI1 clock source selection */
#define RCU_CFG2_SAI2B0SEL              BITS(24,26)                                     /*!< SAI2 Block 0 clock source selection */
#define RCU_CFG2_SAI2B1SEL              BITS(28,30)                                     /*!< SAI2 Block 1 clock source selection */

/* RCU_CFG3 */
#define RCU_CFG3_I2C1SEL                BITS(0,1)                                       /*!< I2C1 clock source selection */
#define RCU_CFG3_I2C2SEL                BITS(2,3)                                       /*!< I2C2 clock source selection */
#define RCU_CFG3_I2C3SEL                BITS(4,5)                                       /*!< I2C3 clock source selection */
#define RCU_CFG3_SDIO1SEL               BIT(12)                                         /*!< SDIO1 function clock source selection */
#define RCU_CFG3_DSPWUSSEL              BIT(24)                                         /*!< deep-sleep wakeup system clock source selection */
#define RCU_CFG3_ADC01SEL               BITS(26,27)                                     /*!< ADC0 and ADC1 clock source selection */
#define RCU_CFG3_ADC2SEL                BITS(28,29)                                     /*!< ADC2 clock source selection */

/* RCU_PLLALL */
#define RCU_PLLALL_PLL0RNG              BITS(0,1)                                       /*!< PLL0 input clock range */
#define RCU_PLLALL_PLL0VCOSEL           BIT(2)                                          /*!< PLL0 VCO selection */
#define RCU_PLLALL_PLL1RNG              BITS(4,5)                                       /*!< PLL1 input clock range */
#define RCU_PLLALL_PLL1VCOSEL           BIT(6)                                          /*!< PLL1 VCO selection */
#define RCU_PLLALL_PLL2RNG              BITS(8,9)                                       /*!< PLL2 input clock range */
#define RCU_PLLALL_PLL2VCOSEL           BIT(10)                                         /*!< PLL2 VCO selection */
#define RCU_PLLALL_PLLSEL               BITS(16,17)                                     /*!< PLL clock source selection */

/* RCU_PLL0FRA */
#define RCU_PLL0FRA_PLL0FRAN            BITS(0,12)                                      /*!< fractional part of the multiplication factor for PLL0 VCO */
#define RCU_PLL0FRA_PLL0FRAEN           BIT(15)                                         /*!< PLL0 fractional latch enable */

/* RCU_PLL1FRA */
#define RCU_PLL1FRA_PLL1FRAN            BITS(0,12)                                      /*!< fractional part of the multiplication factor for PLL1 VCO */
#define RCU_PLL1FRA_PLL1FRAEN           BIT(15)                                         /*!< PLL1 fractional latch enable */

/* RCU_PLL2FRA */
#define RCU_PLL2FRA_PLL2FRAN            BITS(0,12)                                      /*!< fractional part of the multiplication factor for PLL2 VCO */
#define RCU_PLL2FRA_PLL2FRAEN           BIT(15)                                         /*!< PLL2 fractional latch enable */

/* RCU_ADDCTL0 */
#define RCU_ADDCTL0_CK48MSEL            BIT(0)                                          /*!< 48MHz clock selection */
#define RCU_ADDCTL0_PLL48MSEL           BIT(1)                                          /*!< PLL48M clock selection */
#define RCU_ADDCTL0_IRC48MEN            BIT(16)                                         /*!< internal 48MHz RC oscillator enable */
#define RCU_ADDCTL0_IRC48MSTB           BIT(17)                                         /*!< internal 48MHz RC oscillator clock stabilization flag */
#define RCU_ADDCTL0_IRC48MCALIB         BITS(24,31)                                    /*!< internal 48MHz RC oscillator calibration value register */

/* RCU_ADDCTL1 */
#define RCU_ADDCTL1_LPIRC4MEN           BIT(0)                                          /*!< LPIRC4M clock enable */
#define RCU_ADDCTL1_LPIRC4MSTB          BIT(1)                                          /*!< LPIRC4M clock stabilization flag */
#define RCU_ADDCTL1_LPIRC4MTRIM         BITS(2,7)                                       /*!< LPIRC4M clock trim adjust value */
#define RCU_ADDCTL1_LPIRC4MCALIB        BITS(8,15)                                      /*!< LPIRC4M clock calibration value */
#define RCU_ADDCTL1_IRC64MDIV           BITS(16,17)                                     /*!< IRC64M clock divider */
#define RCU_ADDCTL1_LPIRC4MDSPEN        BIT(20)                                         /*!< LPIRC4M clock enable in deepsleep mode */
#define RCU_ADDCTL1_PLLUSBHS0EN         BIT(28)                                         /*!< PLLUSBHS0 clock enable */
#define RCU_ADDCTL1_PLLUSBHS0STB        BIT(29)                                         /*!< PLLUSBHS0 clock stabilization flag */
#define RCU_ADDCTL1_PLLUSBHS1EN         BIT(30)                                         /*!< PLLUSBHS1 clock enable */
#define RCU_ADDCTL1_PLLUSBHS1STB        BIT(31)                                         /*!< PLLUSBHS1 clock stabilization flag */

/* RCU_ADDINT */
#define RCU_ADDINT_PLLUSBHS0STBIF       BIT(4)                                          /*!< Internal PLL of USBHS0 stabilization interrupt flag */
#define RCU_ADDINT_PLLUSBHS1STBIF       BIT(5)                                          /*!< Internal PLL of USBHS1 stabilization interrupt flag */
#define RCU_ADDINT_IRC48MSTBIF          BIT(6)                                          /*!< IRC48M stabilization interrupt flag */
#define RCU_ADDINT_PLLUSBHS0STBIE       BIT(12)                                         /*!< Internal PLL of USBHS0 stabilization interrupt enable */
#define RCU_ADDINT_PLLUSBHS1STBIE       BIT(13)                                         /*!< Internal PLL of USBHS1 stabilization interrupt enable */
#define RCU_ADDINT_IRC48MSTBIE          BIT(14)                                         /*!< Internal 48 MHz RC oscillator stabilization interrupt enable */
#define RCU_ADDINT_PLLUSBHS0STBIC       BIT(20)                                         /*!< Internal PLL of USBHS0 stabilization interrupt clear */
#define RCU_ADDINT_PLLUSBHS1STBIC       BIT(21)                                         /*!< Internal PLL of USBHS1 stabilization interrupt clear */
#define RCU_ADDINT_IRC48MSTBIC          BIT(22)                                         /*!< Internal 48 MHz RC oscillator stabilization interrupt clear */

/* RCU_CFG4 */
#define RCU_CFG4_SDIO0SEL               BIT(0)                                          /*!< SDIO0 clock source selection */
#define RCU_CFG4_EXMCSEL                BITS(8,9)                                       /*!< EXMC clock source selection */

/* RCU_USBCLKCTL*/
#define RCU_USBCLKCTL_USBHS0SEL         BIT(1)                                          /*!< USBHS0 clock selection */
#define RCU_USBCLKCTL_PLLUSBHS0PRESEL   BIT(3)                                          /*!< PLLUSBHS0 clock source selection */
#define RCU_USBCLKCTL_USBHS0SWEN        BIT(4)                                          /*!< USBHS0 clock source selection enable */
#define RCU_USBCLKCTL_USBHS048MSEL      BITS(5,6)                                       /*!< USBHS0 48M clock source selection */
#define RCU_USBCLKCTL_USBHS1SEL         BIT(9)                                          /*!< USBHS1 clock selection */
#define RCU_USBCLKCTL_PLLUSBHS1PRESEL   BIT(11)                                         /*!< PLLUSBHS1 clock source selection */
#define RCU_USBCLKCTL_USBHS1SWEN        BIT(12)                                         /*!< USBHS1 clock source selection enable */
#define RCU_USBCLKCTL_USBHS148MSEL      BITS(13,14)                                     /*!< USBHS1 48M clock source selection */
#define RCU_USBCLKCTL_USBHS0PSC         BITS(16,18)                                     /*!< USBHS0 clock prescaler selection */
#define RCU_USBCLKCTL_USBHS1PSC         BITS(19,21)                                     /*!< USBHS1 clock prescaler selection */

/* RCU_PLLUSBCFG*/
#define RCU_PLLUSBCFG_PLLUSBHS0PREDV    BITS(0,3)                                       /*!< the PLLUSBHS0PREDV clock prescaler */
#define RCU_PLLUSBCFG_USBHS0DV          BITS(4,6)                                       /*!< USBHS0 clock divider */
#define RCU_PLLUSBCFG_PLLUSBHS0MF       BITS(8,14)                                      /*!< the PLLUSBHS0 clock multiplication factor */
#define RCU_PLLUSBCFG_PLLUSBHS1PREDV    BITS(16,19)                                     /*!< the PLLUSBHS1PREDV clock prescaler */
#define RCU_PLLUSBCFG_USBHS1DV          BITS(20,22)                                     /*!< USBHS1 clock divider */
#define RCU_PLLUSBCFG_PLLUSBHS1MF       BITS(24,30)                                     /*!< the PLLUSBHS1 clock multiplication factor */

/* RCU_ADDAPB2RST */
#define RCU_ADDAPB2RST_CAN0RST          BIT(0)                                          /*!< CAN0 reset */
#define RCU_ADDAPB2RST_CAN1RST          BIT(1)                                          /*!< CAN1 reset */
#define RCU_ADDAPB2RST_CAN2RST          BIT(2)                                          /*!< CAN2 reset */

/* RCU_ADDAPB2EN */
#define RCU_ADDAPB2EN_CAN0EN            BIT(0)                                          /*!< CAN0 clock enable */
#define RCU_ADDAPB2EN_CAN1EN            BIT(1)                                          /*!< CAN1 clock enable */
#define RCU_ADDAPB2EN_CAN2EN            BIT(2)                                          /*!< CAN2 clock enable */

/* RCU_ADDAPB2SPEN */
#define RCU_ADDAPB2SPEN_CAN0SPEN        BIT(0)                                          /*!< CAN0 clock enable when sleep mode */
#define RCU_ADDAPB2SPEN_CAN1SPEN        BIT(1)                                          /*!< CAN1 clock enable when sleep mode */
#define RCU_ADDAPB2SPEN_CAN2SPEN        BIT(2)                                          /*!< CAN2 clock enable when sleep mode */

/* RCU_CFG5 */
#define RCU_CFG5_SPI0SEL                BITS(0,2)                                       /*!< SPI0 and I2S0 clock source selection */
#define RCU_CFG5_SPI1SEL                BITS(4,6)                                       /*!< SPI1 and I2S1 clock source selection */
#define RCU_CFG5_SPI2SEL                BITS(8,10)                                      /*!< SPI2 and I2S2 clock source selection */
#define RCU_CFG5_SPI3SEL                BITS(12,14)                                     /*!< SPI3 clock source selection */
#define RCU_CFG5_SPI4SEL                BITS(16,18)                                     /*!< SPI4 clock source selection */
#define RCU_CFG5_SPI5SEL                BITS(20,22)                                     /*!< SPI5 and I2S5 clock source selection */

/* constants definitions */
/* define the peripheral clock enable bit position and its register index offset */
#define RCU_REGIDX_BIT(regidx,bitpos)   (((uint32_t)(regidx) << 6) | (uint32_t)(bitpos))
#define RCU_REG_VAL(periph)             (REG32(RCU + ((uint32_t)(periph) >> 6)))
#define RCU_BIT_POS(val)                ((uint32_t)(val) & 0x0000001FU)
/* define the voltage key unlock value */
#define RCU_VKEY_UNLOCK                 ((uint32_t)0x1A2B3C4DU)

/* RCU state change timeout */
#define HAL_BP_TIMEOUT                  ((uint8_t)100U)                                 /*!< disable backup domain write protection state timeout (in ms) */
#define HAL_IRC64M_TIMEOUT              ((uint8_t)4U)                                   /*!< IRC64M state change timeout (in ms) */
#define HAL_IRC64MDIV_TIMEOUT           ((uint8_t)4U)                                   /*!< IRC64MDIV state change timeout (in ms) */
#define HAL_IRC48M_TIMEOUT              ((uint8_t)4U)                                   /*!< IRC48M state change timeout (in ms) */
#define HAL_IRC32K_TIMEOUT              ((uint8_t)4U)                                   /*!< IRC32K state change timeout (in ms) */
#define HAL_LPIRC4M_TIMEOUT             ((uint8_t)4U)                                   /*!< LPIRC4M state change timeout (in ms) */
#define HAL_PLL_TIMEOUT                 ((uint8_t)4U)                                   /*!< PLL state change timeout (in ms) */

/* peripheral clock type */
#define RCU_PERIPH_CLKTYPE_NONE         (uint64_t)(0x00000000U)                         /*!< no peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_RTC          (uint64_t)(0x00000001U)                         /*!< RTC peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_TIMER        (uint64_t)(0x00000002U)                         /*!< timer peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_TRNG         (uint64_t)(0x00000004U)                         /*!< TRNG peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_USBHS0       (uint64_t)(0x00000008U)                         /*!< USBHS0 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_USBHS1       (uint64_t)(0x00000010U)                         /*!< USBHS0 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_CAN0         (uint64_t)(0x00000020U)                         /*!< CAN0 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_CAN1         (uint64_t)(0x00000040U)                         /*!< CAN1 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_CAN2         (uint64_t)(0x00000080U)                         /*!< CAN2 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_SAI2B0       (uint64_t)(0x00000100U)                         /*!< SAI2B0 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_SAI2B1       (uint64_t)(0x00000200U)                         /*!< SAI2B0 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_SAI0         (uint64_t)(0x00000400U)                         /*!< SAI0 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_SAI1         (uint64_t)(0x00000800U)                         /*!< SAI1 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_EXMC         (uint64_t)(0x00001000U)                         /*!< EXMC peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_SPI0         (uint64_t)(0x00002000U)                         /*!< SPI0 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_SPI1         (uint64_t)(0x00004000U)                         /*!< SPI1 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_SPI2         (uint64_t)(0x00008000U)                         /*!< SPI2 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_SPI3         (uint64_t)(0x00010000U)                         /*!< SPI3 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_SPI4         (uint64_t)(0x00020000U)                         /*!< SPI4 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_SPI5         (uint64_t)(0x00040000U)                         /*!< SPI5 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_SDIO0        (uint64_t)(0x00080000U)                         /*!< SDIO0 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_SDIO1        (uint64_t)(0x00100000U)                         /*!< SDIO1 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_HPDF         (uint64_t)(0x00200000U)                         /*!< HPDF peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_USART0       (uint64_t)(0x00400000U)                         /*!< USART0 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_USART1       (uint64_t)(0x00800000U)                         /*!< USART1 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_USART2       (uint64_t)(0x01000000U)                         /*!< USART2 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_USART5       (uint64_t)(0x02000000U)                         /*!< USART5 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_I2C0         (uint64_t)(0x04000000U)                         /*!< I2C0 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_I2C1         (uint64_t)(0x08000000U)                         /*!< I2C1 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_I2C2         (uint64_t)(0x10000000U)                         /*!< I2C2 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_I2C3         (uint64_t)(0x20000000U)                         /*!< I2C3 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_RSPDIF       (uint64_t)(0x40000000U)                         /*!< RSPDIF peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_ADC01        (uint64_t)(0x80000000U)                         /*!< ADC01 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_ADC2         (uint64_t)(0x100000000U)                        /*!< ADC2 peripheral clock to configure */
#define RCU_PERIPH_CLKTYPE_MAX          (uint64_t)(0xFFFFFFFFFU)                        /*!< all peripheral clock type to configure */

/* RCU clock type */
#define RCU_CLKTYPE_NONE                (uint32_t)(0x00000000U)                         /*!< no clock to configure */
#define RCU_CLKTYPE_SYSCLK              (uint32_t)(0x00000001U)                         /*!< system clock to configure */
#define RCU_CLKTYPE_AHBCLK              (uint32_t)(0x00000002U)                         /*!< AHB clock to configure */
#define RCU_CLKTYPE_APB1CLK             (uint32_t)(0x00000004U)                         /*!< APB1 clock to configure */
#define RCU_CLKTYPE_APB2CLK             (uint32_t)(0x00000008U)                         /*!< APB2 clock to configure */
#define RCU_CLKTYPE_APB3CLK             (uint32_t)(0x00000010U)                         /*!< APB3 clock to configure */
#define RCU_CLKTYPE_APB4CLK             (uint32_t)(0x00000020U)                         /*!< APB4 clock to configure */
#define RCU_CLKTYPE_MAX                 (uint32_t)(0x000000FFU)                         /*!< all clock type to configure */

/* RCU_PLLADDCTL register bit define */
/* PLL P/Q/R divider output enable */
#define RCU_PLL0P                       RCU_PLLADDCTL_PLL0PEN                           /*!< PLL0P divider output enable */
#define RCU_PLL0Q                       RCU_PLLADDCTL_PLL0QEN                           /*!< PLL0Q divider output enable */
#define RCU_PLL0R                       RCU_PLLADDCTL_PLL0REN                           /*!< PLL0R divider output enable */
#define RCU_PLL1P                       RCU_PLLADDCTL_PLL1PEN                           /*!< PLL1P divider output enable */
#define RCU_PLL1Q                       RCU_PLLADDCTL_PLL1QEN                           /*!< PLL1Q divider output enable */
#define RCU_PLL1R                       RCU_PLLADDCTL_PLL1REN                           /*!< PLL1R divider output enable */
#define RCU_PLL2P                       RCU_PLLADDCTL_PLL2PEN                           /*!< PLL2P divider output enable */
#define RCU_PLL2Q                       RCU_PLLADDCTL_PLL2QEN                           /*!< PLL2Q divider output enable */
#define RCU_PLL2R                       RCU_PLLADDCTL_PLL2REN                           /*!< PLL2R divider output enable */

/* RCU_CFG1 register bit define */
/* USARTx(x=0,1,2,5) clock source selection */
#define CFG1_USART0SEL(regval)          (BITS(0,1) & ((uint32_t)(regval) << 0U))
#define RCU_USARTSRC_APB2               CFG1_USART0SEL(0)                               /*!< CK_USART select CK_APB2 */
#define RCU_USARTSRC_AHB                CFG1_USART0SEL(1)                               /*!< CK_USART select CK_AHB */
#define RCU_USARTSRC_LXTAL              CFG1_USART0SEL(2)                               /*!< CK_USART select CK_LXTAL */
#define RCU_USARTSRC_IRC64MDIV          CFG1_USART0SEL(3)                               /*!< CK_USART select CK_IRC16MDIV */

/* I2Cx(x=0,1,2,3) clock source selection */
#define CFG3_I2C1SEL(regval)            (BITS(0,1) & ((uint32_t)(regval) << 0U))
#define RCU_I2CSRC_APB1                 CFG3_I2C1SEL(0)                                 /*!< CK_I2C select CK_APB1 */
#define RCU_I2CSRC_PLL2R                CFG3_I2C1SEL(1)                                 /*!< CK_I2C select CK_PLL2R */
#define RCU_I2CSRC_IRC64MDIV            CFG3_I2C1SEL(2)                                 /*!< CK_I2C select CK_IRC64MDIV */
#define RCU_I2CSRC_LPIRC4M              CFG3_I2C1SEL(3)                                 /*!< CK_I2C select CK_LPIRC4M */

/* CANx(x=0,1,2) clock source selection */
#define CFG1_CAN0SEL(regval)            (BITS(8,9) & ((uint32_t)(regval) << 8U))
#define RCU_CANSRC_HXTAL                CFG1_CAN0SEL(0)                                 /*!< CK_CAN select CK_HXTAL */
#define RCU_CANSRC_APB2                 CFG1_CAN0SEL(1)                                 /*!< CK_CAN select CK_APB2 */
#define RCU_CANSRC_APB2_DIV2            CFG1_CAN0SEL(2)                                 /*!< CK_CAN select CK_APB2/2 */
#define RCU_CANSRC_IRC64MDIV            CFG1_CAN0SEL(3)                                 /*!< CK_CAN select CK_IRC64MDIV */

/* RSPDIF clock selection */
#define CFG1_RSPDIFSEL(regval)          (BITS(4,5) & ((uint32_t)(regval) << 4U))
#define RCU_RSPDIFSRC_PLL0Q             CFG1_RSPDIFSEL(0)                               /*!< CK_RSPDIF select CK_PLL0Q */
#define RCU_RSPDIFSRC_PLL1R             CFG1_RSPDIFSEL(1)                               /*!< CK_RSPDIF select CK_PLL1R */
#define RCU_RSPDIFSRC_PLL2R             CFG1_RSPDIFSEL(2)                               /*!< CK_RSPDIF select CK_PLL2R */
#define RCU_RSPDIFSRC_IRC64MDIV         CFG1_RSPDIFSEL(3)                               /*!< CK_RSPDIF select CK_IRC64MDIV */

/* EXMC clock selection */
#define CFG4_EXMCSEL(regval)            (BITS(8,9) & ((uint32_t)(regval) << 8U))
#define RCU_EXMCSRC_AHB                 CFG4_EXMCSEL(0)                                 /*!< CK_EXMC select CK_AHB */
#define RCU_EXMCSRC_PLL0Q               CFG4_EXMCSEL(1)                                 /*!< CK_EXMC select CK_PLL0Q */
#define RCU_EXMCSRC_PLL1R               CFG4_EXMCSEL(2)                                 /*!< CK_EXMC select CK_PLL1R */
#define RCU_EXMCSRC_PER                 CFG4_EXMCSEL(3)                                 /*!< CK_EXMC select CK_PER */

/* SPIx(x=0,1,2) clock source selection */
#define CFG5_SPI0SEL(regval)            (BITS(0,2) & ((uint32_t)(regval) << 0U))
#define RCU_SPISRC_PLL0Q                CFG5_SPI0SEL(0)                                 /*!< CK_SPI select CK_PLL0Q */
#define RCU_SPISRC_PLL1P                CFG5_SPI0SEL(1)                                 /*!< CK_SPI select CK_PLL1P */
#define RCU_SPISRC_PLL2P                CFG5_SPI0SEL(2)                                 /*!< CK_SPI select CK_PLL2P */
#define RCU_SPISRC_I2S_CKIN             CFG5_SPI0SEL(3)                                 /*!< CK_SPI select I2S_CKIN */
#define RCU_SPISRC_PER                  CFG5_SPI0SEL(4)                                 /*!< CK_SPI select CK_PER */

/* SPIx(x=3,4) clock source selection */
#define CFG5_SPI3SEL(regval)            (BITS(12,14) & ((uint32_t)(regval) << 12U))
#define RCU_SPISRC_APB2                 CFG5_SPI3SEL(0)                                 /*!< CK_SPI select CK_APB2 */
#define RCU_SPISRC_PLL1Q                CFG5_SPI3SEL(1)                                 /*!< CK_SPI select CK_PLL1Q */
#define RCU_SPISRC_PLL2Q                CFG5_SPI3SEL(2)                                 /*!< CK_SPI select CK_PLL2Q */
#define RCU_SPISRC_IRC64MDIV            CFG5_SPI3SEL(3)                                 /*!< CK_SPI select CK_IRC64MDIV */
#define RCU_SPISRC_LPIRC4M              CFG5_SPI3SEL(4)                                 /*!< CK_SPI select CK_LPIRC4M */
#define RCU_SPISRC_HXTAL                CFG5_SPI3SEL(5)                                 /*!< CK_SPI select CK_HXTAL */

/* SPI5 clock source selection */
#define CFG5_SPI5SEL(regval)            (BITS(20,22) & ((uint32_t)(regval) << 20U))
#define RCU_SPI5SRC_APB2                CFG5_SPI5SEL(0)                                 /*!< CK_SPI5 select CK_APB2 */
#define RCU_SPI5SRC_PLL1Q               CFG5_SPI5SEL(1)                                 /*!< CK_SPI5 select CK_PLL1Q */
#define RCU_SPI5SRC_PLL2Q               CFG5_SPI5SEL(2)                                 /*!< CK_SPI5 select CK_PLL2Q */
#define RCU_SPI5SRC_IRC64MDIV           CFG5_SPI5SEL(3)                                 /*!< CK_SPI5 select CK_IRC64MDIV */
#define RCU_SPI5SRC_LPIRC4M             CFG5_SPI5SEL(4)                                 /*!< CK_SPI5 select CK_LPIRC4M */
#define RCU_SPI5SRC_HXTAL               CFG5_SPI5SEL(5)                                 /*!< CK_SPI5 select CK_HXTAL */
#define RCU_SPI5SRC_I2S_CKIN            CFG5_SPI5SEL(6)                                 /*!< CK_SPI5 select I2S_CKIN */

/* SDIOx(x=0,1) clock source selection */
#define RCU_SDIO0SRC_PLL0Q              (uint32_t)(0x00000000U)                         /*!< CK_SDIO0 select CK_PLL0Q */
#define RCU_SDIO0SRC_PLL1R              RCU_CFG4_SDIO0SEL                               /*!< CK_SDIO0 select CK_PLL1R */
#define RCU_SDIO1SRC_PLL0Q              (uint32_t)(0x00000000U)                         /*!< CK_SDIO1 select CK_PLL0Q */
#define RCU_SDIO1SRC_PLL1R              RCU_CFG3_SDIO1SEL                               /*!< CK_SDIO1 select CK_PLL1R */

/* Efuse clock source selection */
#define RCU_DSPWUSSEL_IRC64MDIV         (uint32_t)(0x00000000U)                         /*!< CK_IRC64MDIV is selected as wake up system clock from deep-sleep mode */
#define RCU_DSPWUSSEL_LPIRC4M           RCU_CFG3_DSPWUSSEL                              /*!< CK_LPIRC4M is selected as wake up system clock from deep-sleep mode */

/* ADCx(x=0,1,2) clock source selection */
#define CFG3_ADC0SEL(regval)            (BITS(26,27) & ((uint32_t)((regval) << 26U)))
#define RCU_ADCSRC_PLL1P                CFG3_ADC0SEL(0)                                 /*!< CK_ADC select CK_PLL1P */
#define RCU_ADCSRC_PLL2R                CFG3_ADC0SEL(1)                                 /*!< CK_ADC select CK_PLL2R */
#define RCU_ADCSRC_PER                  CFG3_ADC0SEL(2)                                 /*!< CK_ADC select CK_PER */

/* SAIx(x=0,1) clock source selection */
#define CFG2_SAI0SEL(regval)            (BITS(16,18) & ((uint32_t)(regval) << 16U))
#define RCU_SAISRC_PLL0Q                CFG2_SAI0SEL(0)                                 /*!< CK_SAI0/1 select CK_PLL0Q */
#define RCU_SAISRC_PLL1P                CFG2_SAI0SEL(1)                                 /*!< CK_SAI0/1 select CK_PLL1P */
#define RCU_SAISRC_PLL2P                CFG2_SAI0SEL(2)                                 /*!< CK_SAI0/1 select CK_PLL2P */
#define RCU_SAISRC_I2S_CKIN             CFG2_SAI0SEL(3)                                 /*!< CK_SAI0/1 select I2S_CKIN */
#define RCU_SAISRC_PER                  CFG2_SAI0SEL(4)                                 /*!< CK_SAI0/1 select CK_PER */

/* SAI2Bx(x=0,1) clock source selection */
#define CFG2_SAI2B0SEL(regval)          (BITS(24,26) & ((uint32_t)(regval) << 24U))
#define RCU_SAI2BSRC_PLL0Q              CFG2_SAI2B0SEL(0)                               /*!< SAI2 block0/1 select CK_PLL0Q */
#define RCU_SAI2BSRC_PLL1P              CFG2_SAI2B0SEL(1)                               /*!< SAI2 block0/1 select CK_PLL1P */
#define RCU_SAI2BSRC_PLL2P              CFG2_SAI2B0SEL(2)                               /*!< SAI2 block0/1 select CK_PLL2P */
#define RCU_SAI2BSRC_I2S_CKIN           CFG2_SAI2B0SEL(3)                               /*!< SAI2 block0/1 select I2S_CKIN */
#define RCU_SAI2BSRC_PER                CFG2_SAI2B0SEL(4)                               /*!< SAI2 block0/1 select CK_PER */
#define RCU_SAI2BSRC_RSPDIF_SYMB        CFG2_SAI2B0SEL(5)                               /*!< SAI2 block0/1 select CK_RSPDIF_SYMB */

/* HPDF clock source selection */
#define RCU_HPDFSRC_APB2                (uint32_t)(0x00000000U)                         /*!< CK_HPDF select CK_APB2 */
#define RCU_HPDFSRC_AHB                 RCU_CFG1_HPDFSEL                                /*!< CK_HPDF select CK_AHB */

/* PER clock selection */
#define CFG1_PERSEL(regval)             (BITS(14,15) & ((uint32_t)(regval) << 14U))
#define RCU_PERSRC_IRC64MDIV            CFG1_PERSEL(0)                                  /*!< CK_PER select CK_IRC64MDIV */
#define RCU_PERSRC_LPIRC4M              CFG1_PERSEL(1)                                  /*!< CK_PER select CK_LPIRC4M */
#define RCU_PERSRC_HXTAL                CFG1_PERSEL(2)                                  /*!< CK_PER select CK_HXTAL */

/* system clock source select */
#define CFG0_SCS(regval)                (BITS(0,1) & ((uint32_t)(regval) << 0U))
#define RCU_CKSYSSRC_IRC64MDIV          CFG0_SCS(0)                                     /*!< system clock source select IRC64MDIV */
#define RCU_CKSYSSRC_HXTAL              CFG0_SCS(1)                                     /*!< system clock source select HXTAL */
#define RCU_CKSYSSRC_LPIRC4M            CFG0_SCS(2)                                     /*!< system clock source select LPIRC4M */
#define RCU_CKSYSSRC_PLL0P              CFG0_SCS(3)                                     /*!< system clock source select PLL0P */

/* system clock source select status */
#define CFG0_SCSS(regval)               (BITS(2,3) & ((uint32_t)(regval) << 2U))
#define RCU_SCSS_IRC64MDIV              CFG0_SCSS(0)                                    /*!< system clock source select CK_IRC64MDIV */
#define RCU_SCSS_HXTAL                  CFG0_SCSS(1)                                    /*!< system clock source select CK_HXTAL */
#define RCU_SCSS_LPIRC4M                CFG0_SCSS(2)                                    /*!< system clock source select CK_LPIRC4M */
#define RCU_SCSS_PLL0P                  CFG0_SCSS(3)                                    /*!< system clock source select CK_PLL0P */

/* IRC64M clock divider */
#define ADDCTL1_IRC64MDIV(regval)       (BITS(16,17) & ((uint32_t)(regval) << 16U))
#define RCU_IRC64M_DIV1                 ADDCTL1_IRC64MDIV(0)                            /*!< CK_IRC64MDIV is CK_IRC64M / 1 */
#define RCU_IRC64M_DIV2                 ADDCTL1_IRC64MDIV(1)                            /*!< CK_IRC64MDIV is CK_IRC64M / 2 */
#define RCU_IRC64M_DIV4                 ADDCTL1_IRC64MDIV(2)                            /*!< CK_IRC64MDIV is CK_IRC64M / 4 */
#define RCU_IRC64M_DIV8                 ADDCTL1_IRC64MDIV(3)                            /*!< CK_IRC64MDIV is CK_IRC64M / 8 */

/* AHB prescaler selection */
#define CFG0_AHBPSC(regval)             (BITS(4,7) & ((uint32_t)(regval) << 4U))
#define RCU_AHB_CKSYS_DIV1              CFG0_AHBPSC(0)                                  /*!< AHB prescaler select CK_SYS */
#define RCU_AHB_CKSYS_DIV2              CFG0_AHBPSC(8)                                  /*!< AHB prescaler select CK_SYS / 2 */
#define RCU_AHB_CKSYS_DIV4              CFG0_AHBPSC(9)                                  /*!< AHB prescaler select CK_SYS / 4 */
#define RCU_AHB_CKSYS_DIV8              CFG0_AHBPSC(10)                                 /*!< AHB prescaler select CK_SYS / 8 */
#define RCU_AHB_CKSYS_DIV16             CFG0_AHBPSC(11)                                 /*!< AHB prescaler select CK_SYS / 16 */
#define RCU_AHB_CKSYS_DIV64             CFG0_AHBPSC(12)                                 /*!< AHB prescaler select CK_SYS / 64 */
#define RCU_AHB_CKSYS_DIV128            CFG0_AHBPSC(13)                                 /*!< AHB prescaler select CK_SYS / 128 */
#define RCU_AHB_CKSYS_DIV256            CFG0_AHBPSC(14)                                 /*!< AHB prescaler select CK_SYS / 256 */
#define RCU_AHB_CKSYS_DIV512            CFG0_AHBPSC(15)                                 /*!< AHB prescaler select CK_SYS / 512 */

/* APB1 prescaler selection */
#define CFG0_APB1PSC(regval)            (BITS(10,12) & ((uint32_t)(regval) << 10U))
#define RCU_APB1_CKAHB_DIV1             CFG0_APB1PSC(0)                                 /*!< APB1 prescaler select CK_AHB */
#define RCU_APB1_CKAHB_DIV2             CFG0_APB1PSC(4)                                 /*!< APB1 prescaler select CK_AHB / 2 */
#define RCU_APB1_CKAHB_DIV4             CFG0_APB1PSC(5)                                 /*!< APB1 prescaler select CK_AHB / 4 */
#define RCU_APB1_CKAHB_DIV8             CFG0_APB1PSC(6)                                 /*!< APB1 prescaler select CK_AHB / 8 */
#define RCU_APB1_CKAHB_DIV16            CFG0_APB1PSC(7)                                 /*!< APB1 prescaler select CK_AHB / 16 */

/* APB2 prescaler selection */
#define CFG0_APB2PSC(regval)            (BITS(13,15) & ((uint32_t)(regval) << 13U))
#define RCU_APB2_CKAHB_DIV1             CFG0_APB2PSC(0)                                 /*!< APB2 prescaler select CK_AHB */
#define RCU_APB2_CKAHB_DIV2             CFG0_APB2PSC(4)                                 /*!< APB2 prescaler select CK_AHB / 2 */
#define RCU_APB2_CKAHB_DIV4             CFG0_APB2PSC(5)                                 /*!< APB2 prescaler select CK_AHB / 4 */
#define RCU_APB2_CKAHB_DIV8             CFG0_APB2PSC(6)                                 /*!< APB2 prescaler select CK_AHB / 8 */
#define RCU_APB2_CKAHB_DIV16            CFG0_APB2PSC(7)                                 /*!< APB2 prescaler select CK_AHB / 16 */

/* APB3 prescaler selection */
#define CFG0_APB3PSC(regval)            (BITS(27,29) & ((uint32_t)(regval) << 27U))
#define RCU_APB3_CKAHB_DIV1             CFG0_APB3PSC(0)                                 /*!< APB3 prescaler select CK_AHB */
#define RCU_APB3_CKAHB_DIV2             CFG0_APB3PSC(4)                                 /*!< APB3 prescaler select CK_AHB / 2 */
#define RCU_APB3_CKAHB_DIV4             CFG0_APB3PSC(5)                                 /*!< APB3 prescaler select CK_AHB / 4 */
#define RCU_APB3_CKAHB_DIV8             CFG0_APB3PSC(6)                                 /*!< APB3 prescaler select CK_AHB / 8 */
#define RCU_APB3_CKAHB_DIV16            CFG0_APB3PSC(7)                                 /*!< APB3 prescaler select CK_AHB / 16 */

/* APB4 prescaler selection */
#define CFG0_APB4PSC(regval)            (BITS(24,26) & ((uint32_t)(regval) << 24U))
#define RCU_APB4_CKAHB_DIV1             CFG0_APB4PSC(0)                                 /*!< APB4 prescaler select CK_AHB */
#define RCU_APB4_CKAHB_DIV2             CFG0_APB4PSC(4)                                 /*!< APB4 prescaler select CK_AHB / 2 */
#define RCU_APB4_CKAHB_DIV4             CFG0_APB4PSC(5)                                 /*!< APB4 prescaler select CK_AHB / 4 */
#define RCU_APB4_CKAHB_DIV8             CFG0_APB4PSC(6)                                 /*!< APB4 prescaler select CK_AHB / 8 */
#define RCU_APB4_CKAHB_DIV16            CFG0_APB4PSC(7)                                 /*!< APB4 prescaler select CK_AHB / 16 */

/* RTC clock divider factor from HXTAL clock */
#define CFG0_RTCDIV(regval)             (BITS(16,21) & ((uint32_t)(regval) << 16U))
#define RCU_RTC_NONE                    CFG0_RTCDIV(0)                                  /*!< no clock for RTC */
#define RCU_RTC_DIV2                    CFG0_RTCDIV(2)                                  /*!< RTCDIV clock select CK_HXTAL / 2 */
#define RCU_RTC_DIV3                    CFG0_RTCDIV(3)                                  /*!< RTCDIV clock select CK_HXTAL / 3 */
#define RCU_RTC_DIV4                    CFG0_RTCDIV(4)                                  /*!< RTCDIV clock select CK_HXTAL / 4 */
#define RCU_RTC_DIV5                    CFG0_RTCDIV(5)                                  /*!< RTCDIV clock select CK_HXTAL / 5 */
#define RCU_RTC_DIV6                    CFG0_RTCDIV(6)                                  /*!< RTCDIV clock select CK_HXTAL / 6 */
#define RCU_RTC_DIV7                    CFG0_RTCDIV(7)                                  /*!< RTCDIV clock select CK_HXTAL / 7 */
#define RCU_RTC_DIV8                    CFG0_RTCDIV(8)                                  /*!< RTCDIV clock select CK_HXTAL / 8 */
#define RCU_RTC_DIV9                    CFG0_RTCDIV(9)                                  /*!< RTCDIV clock select CK_HXTAL / 9 */
#define RCU_RTC_DIV10                   CFG0_RTCDIV(10)                                 /*!< RTCDIV clock select CK_HXTAL / 10 */
#define RCU_RTC_DIV11                   CFG0_RTCDIV(11)                                 /*!< RTCDIV clock select CK_HXTAL / 11 */
#define RCU_RTC_DIV12                   CFG0_RTCDIV(12)                                 /*!< RTCDIV clock select CK_HXTAL / 12 */
#define RCU_RTC_DIV13                   CFG0_RTCDIV(13)                                 /*!< RTCDIV clock select CK_HXTAL / 13 */
#define RCU_RTC_DIV14                   CFG0_RTCDIV(14)                                 /*!< RTCDIV clock select CK_HXTAL / 14 */
#define RCU_RTC_DIV15                   CFG0_RTCDIV(15)                                 /*!< RTCDIV clock select CK_HXTAL / 15 */
#define RCU_RTC_DIV16                   CFG0_RTCDIV(16)                                 /*!< RTCDIV clock select CK_HXTAL / 16 */
#define RCU_RTC_DIV17                   CFG0_RTCDIV(17)                                 /*!< RTCDIV clock select CK_HXTAL / 17 */
#define RCU_RTC_DIV18                   CFG0_RTCDIV(18)                                 /*!< RTCDIV clock select CK_HXTAL / 18 */
#define RCU_RTC_DIV19                   CFG0_RTCDIV(19)                                 /*!< RTCDIV clock select CK_HXTAL / 19 */
#define RCU_RTC_DIV20                   CFG0_RTCDIV(20)                                 /*!< RTCDIV clock select CK_HXTAL / 20 */
#define RCU_RTC_DIV21                   CFG0_RTCDIV(21)                                 /*!< RTCDIV clock select CK_HXTAL / 21 */
#define RCU_RTC_DIV22                   CFG0_RTCDIV(22)                                 /*!< RTCDIV clock select CK_HXTAL / 22 */
#define RCU_RTC_DIV23                   CFG0_RTCDIV(23)                                 /*!< RTCDIV clock select CK_HXTAL / 23 */
#define RCU_RTC_DIV24                   CFG0_RTCDIV(24)                                 /*!< RTCDIV clock select CK_HXTAL / 24 */
#define RCU_RTC_DIV25                   CFG0_RTCDIV(25)                                 /*!< RTCDIV clock select CK_HXTAL / 25 */
#define RCU_RTC_DIV26                   CFG0_RTCDIV(26)                                 /*!< RTCDIV clock select CK_HXTAL / 26 */
#define RCU_RTC_DIV27                   CFG0_RTCDIV(27)                                 /*!< RTCDIV clock select CK_HXTAL / 27 */
#define RCU_RTC_DIV28                   CFG0_RTCDIV(28)                                 /*!< RTCDIV clock select CK_HXTAL / 28 */
#define RCU_RTC_DIV29                   CFG0_RTCDIV(29)                                 /*!< RTCDIV clock select CK_HXTAL / 29 */
#define RCU_RTC_DIV30                   CFG0_RTCDIV(30)                                 /*!< RTCDIV clock select CK_HXTAL / 30 */
#define RCU_RTC_DIV31                   CFG0_RTCDIV(31)                                 /*!< RTCDIV clock select CK_HXTAL / 31 */
#define RCU_RTC_DIV32                   CFG0_RTCDIV(32)                                 /*!< RTCDIV clock select CK_HXTAL / 32 */
#define RCU_RTC_DIV33                   CFG0_RTCDIV(33)                                 /*!< RTCDIV clock select CK_HXTAL / 33 */
#define RCU_RTC_DIV34                   CFG0_RTCDIV(34)                                 /*!< RTCDIV clock select CK_HXTAL / 34 */
#define RCU_RTC_DIV35                   CFG0_RTCDIV(35)                                 /*!< RTCDIV clock select CK_HXTAL / 35 */
#define RCU_RTC_DIV36                   CFG0_RTCDIV(36)                                 /*!< RTCDIV clock select CK_HXTAL / 36 */
#define RCU_RTC_DIV37                   CFG0_RTCDIV(37)                                 /*!< RTCDIV clock select CK_HXTAL / 37 */
#define RCU_RTC_DIV38                   CFG0_RTCDIV(38)                                 /*!< RTCDIV clock select CK_HXTAL / 38 */
#define RCU_RTC_DIV39                   CFG0_RTCDIV(39)                                 /*!< RTCDIV clock select CK_HXTAL / 39 */
#define RCU_RTC_DIV40                   CFG0_RTCDIV(40)                                 /*!< RTCDIV clock select CK_HXTAL / 40 */
#define RCU_RTC_DIV41                   CFG0_RTCDIV(41)                                 /*!< RTCDIV clock select CK_HXTAL / 41 */
#define RCU_RTC_DIV42                   CFG0_RTCDIV(42)                                 /*!< RTCDIV clock select CK_HXTAL / 42 */
#define RCU_RTC_DIV43                   CFG0_RTCDIV(43)                                 /*!< RTCDIV clock select CK_HXTAL / 43 */
#define RCU_RTC_DIV44                   CFG0_RTCDIV(44)                                 /*!< RTCDIV clock select CK_HXTAL / 44 */
#define RCU_RTC_DIV45                   CFG0_RTCDIV(45)                                 /*!< RTCDIV clock select CK_HXTAL / 45 */
#define RCU_RTC_DIV46                   CFG0_RTCDIV(46)                                 /*!< RTCDIV clock select CK_HXTAL / 46 */
#define RCU_RTC_DIV47                   CFG0_RTCDIV(47)                                 /*!< RTCDIV clock select CK_HXTAL / 47 */
#define RCU_RTC_DIV48                   CFG0_RTCDIV(48)                                 /*!< RTCDIV clock select CK_HXTAL / 48 */
#define RCU_RTC_DIV49                   CFG0_RTCDIV(49)                                 /*!< RTCDIV clock select CK_HXTAL / 49 */
#define RCU_RTC_DIV50                   CFG0_RTCDIV(50)                                 /*!< RTCDIV clock select CK_HXTAL / 50 */
#define RCU_RTC_DIV51                   CFG0_RTCDIV(51)                                 /*!< RTCDIV clock select CK_HXTAL / 51 */
#define RCU_RTC_DIV52                   CFG0_RTCDIV(52)                                 /*!< RTCDIV clock select CK_HXTAL / 52 */
#define RCU_RTC_DIV53                   CFG0_RTCDIV(53)                                 /*!< RTCDIV clock select CK_HXTAL / 53 */
#define RCU_RTC_DIV54                   CFG0_RTCDIV(54)                                 /*!< RTCDIV clock select CK_HXTAL / 54 */
#define RCU_RTC_DIV55                   CFG0_RTCDIV(55)                                 /*!< RTCDIV clock select CK_HXTAL / 55 */
#define RCU_RTC_DIV56                   CFG0_RTCDIV(56)                                 /*!< RTCDIV clock select CK_HXTAL / 56 */
#define RCU_RTC_DIV57                   CFG0_RTCDIV(57)                                 /*!< RTCDIV clock select CK_HXTAL / 57 */
#define RCU_RTC_DIV58                   CFG0_RTCDIV(58)                                 /*!< RTCDIV clock select CK_HXTAL / 58 */
#define RCU_RTC_DIV59                   CFG0_RTCDIV(59)                                 /*!< RTCDIV clock select CK_HXTAL / 59 */
#define RCU_RTC_DIV60                   CFG0_RTCDIV(60)                                 /*!< RTCDIV clock select CK_HXTAL / 60 */
#define RCU_RTC_DIV61                   CFG0_RTCDIV(61)                                 /*!< RTCDIV clock select CK_HXTAL / 61 */
#define RCU_RTC_DIV62                   CFG0_RTCDIV(62)                                 /*!< RTCDIV clock select CK_HXTAL / 62 */
#define RCU_RTC_DIV63                   CFG0_RTCDIV(63)                                 /*!< RTCDIV clock select CK_HXTAL / 63 */

/* The CK_OUT0 divider */
#define CFG2_CKOUT0DIV(regval)          (BITS(0,3) & ((uint32_t)(regval) << 0U))
#define RCU_CKOUT0_DIV1                 CFG2_CKOUT0DIV(1)                               /*!< CK_OUT0 is divided by 1 */
#define RCU_CKOUT0_DIV2                 CFG2_CKOUT0DIV(2)                               /*!< CK_OUT0 is divided by 2 */
#define RCU_CKOUT0_DIV3                 CFG2_CKOUT0DIV(3)                               /*!< CK_OUT0 is divided by 3 */
#define RCU_CKOUT0_DIV4                 CFG2_CKOUT0DIV(4)                               /*!< CK_OUT0 is divided by 4 */
#define RCU_CKOUT0_DIV5                 CFG2_CKOUT0DIV(5)                               /*!< CK_OUT0 is divided by 5 */
#define RCU_CKOUT0_DIV6                 CFG2_CKOUT0DIV(6)                               /*!< CK_OUT0 is divided by 6 */
#define RCU_CKOUT0_DIV7                 CFG2_CKOUT0DIV(7)                               /*!< CK_OUT0 is divided by 7 */
#define RCU_CKOUT0_DIV8                 CFG2_CKOUT0DIV(8)                               /*!< CK_OUT0 is divided by 8 */
#define RCU_CKOUT0_DIV9                 CFG2_CKOUT0DIV(9)                               /*!< CK_OUT0 is divided by 9 */
#define RCU_CKOUT0_DIV10                CFG2_CKOUT0DIV(10)                              /*!< CK_OUT0 is divided by 10 */
#define RCU_CKOUT0_DIV11                CFG2_CKOUT0DIV(11)                              /*!< CK_OUT0 is divided by 11 */
#define RCU_CKOUT0_DIV12                CFG2_CKOUT0DIV(12)                              /*!< CK_OUT0 is divided by 12 */
#define RCU_CKOUT0_DIV13                CFG2_CKOUT0DIV(13)                              /*!< CK_OUT0 is divided by 13 */
#define RCU_CKOUT0_DIV14                CFG2_CKOUT0DIV(14)                              /*!< CK_OUT0 is divided by 14 */
#define RCU_CKOUT0_DIV15                CFG2_CKOUT0DIV(15)                              /*!< CK_OUT0 is divided by 15 */

/* CKOUT0 Clock source selection */
#define CFG2_CKOUT0SEL(regval)          (BITS(4,6) & ((uint32_t)(regval) << 4U))
#define RCU_CKOUT0SRC_IRC64MDIV         CFG2_CKOUT0SEL(0)                               /*!< internal 64MDIV RC oscillator clock selected */
#define RCU_CKOUT0SRC_LXTAL             CFG2_CKOUT0SEL(1)                               /*!< low speed crystal oscillator clock (LXTAL) selected */
#define RCU_CKOUT0SRC_HXTAL             CFG2_CKOUT0SEL(2)                               /*!< high speed crystal oscillator clock (HXTAL) selected */
#define RCU_CKOUT0SRC_PLL0P             CFG2_CKOUT0SEL(3)                               /*!< CK_PLL0P clock selected */
#define RCU_CKOUT0SRC_IRC48M            CFG2_CKOUT0SEL(4)                               /*!< CK_IRC48M clock selected */
#define RCU_CKOUT0SRC_PER               CFG2_CKOUT0SEL(5)                               /*!< CK_PER clock selected */
#define RCU_CKOUT0SRC_USBHS060M         CFG2_CKOUT0SEL(6)                               /*!< USBHS0 60M clock selected */
#define RCU_CKOUT0SRC_USBHS160M         CFG2_CKOUT0SEL(7)                               /*!< USBHS1 60M clock selected */

/* The CK_OUT1 divider */
#define CFG2_CKOUT1DIV(regval)          (BITS(8,11) & ((uint32_t)(regval) << 8U))
#define RCU_CKOUT1_DIV1                 CFG2_CKOUT1DIV(1)                               /*!< CK_OUT1 is divided by 1 */
#define RCU_CKOUT1_DIV2                 CFG2_CKOUT1DIV(2)                               /*!< CK_OUT1 is divided by 2 */
#define RCU_CKOUT1_DIV3                 CFG2_CKOUT1DIV(3)                               /*!< CK_OUT1 is divided by 3 */
#define RCU_CKOUT1_DIV4                 CFG2_CKOUT1DIV(4)                               /*!< CK_OUT1 is divided by 4 */
#define RCU_CKOUT1_DIV5                 CFG2_CKOUT1DIV(5)                               /*!< CK_OUT1 is divided by 5 */
#define RCU_CKOUT1_DIV6                 CFG2_CKOUT1DIV(6)                               /*!< CK_OUT1 is divided by 6 */
#define RCU_CKOUT1_DIV7                 CFG2_CKOUT1DIV(7)                               /*!< CK_OUT1 is divided by 7 */
#define RCU_CKOUT1_DIV8                 CFG2_CKOUT1DIV(8)                               /*!< CK_OUT1 is divided by 8 */
#define RCU_CKOUT1_DIV9                 CFG2_CKOUT1DIV(9)                               /*!< CK_OUT1 is divided by 9 */
#define RCU_CKOUT1_DIV10                CFG2_CKOUT1DIV(10)                              /*!< CK_OUT1 is divided by 10 */
#define RCU_CKOUT1_DIV11                CFG2_CKOUT1DIV(11)                              /*!< CK_OUT1 is divided by 11 */
#define RCU_CKOUT1_DIV12                CFG2_CKOUT1DIV(12)                              /*!< CK_OUT1 is divided by 12 */
#define RCU_CKOUT1_DIV13                CFG2_CKOUT1DIV(13)                              /*!< CK_OUT1 is divided by 13 */
#define RCU_CKOUT1_DIV14                CFG2_CKOUT1DIV(14)                              /*!< CK_OUT1 is divided by 14 */
#define RCU_CKOUT1_DIV15                CFG2_CKOUT1DIV(15)                              /*!< CK_OUT1 is divided by 15 */

/* CKOUT1 Clock source selection */
#define CFG2_CKOUT1SEL(regval)          (BITS(12,14) & ((uint32_t)(regval) << 12U))
#define RCU_CKOUT1SRC_SYSTEMCLOCK       CFG2_CKOUT1SEL(0)                               /*!< system clock selected */
#define RCU_CKOUT1SRC_PLL1R             CFG2_CKOUT1SEL(1)                               /*!< CK_PLL1R clock selected */
#define RCU_CKOUT1SRC_HXTAL             CFG2_CKOUT1SEL(2)                               /*!< high speed crystal oscillator clock (HXTAL) selected */
#define RCU_CKOUT1SRC_PLL0P             CFG2_CKOUT1SEL(3)                               /*!< CK_PLL0P clock selected */
#define RCU_CKOUT1SRC_LPIRC4M           CFG2_CKOUT1SEL(4)                               /*!< CK_LPIRC4M clock selected */
#define RCU_CKOUT1SRC_IRC32K            CFG2_CKOUT1SEL(5)                               /*!< CK_IRC32K clock selected */
#define RCU_CKOUT1SRC_PLL2R             CFG2_CKOUT1SEL(6)                               /*!< CK_PLL2R clock selected */

/* the divider factor from PLL2R clock */
#define CFG1_PLL2RDIV(regval)           (BITS(16,17) & ((uint32_t)(regval) << 16U))
#define RCU_PLL2R_DIV2                  CFG1_PLL2RDIV(0)                                /*!< CK_PLL2RDIV clock select CK_PLL2R / 2 */
#define RCU_PLL2R_DIV4                  CFG1_PLL2RDIV(1)                                /*!< CK_PLL2RDIV clock select CK_PLL2R / 4 */
#define RCU_PLL2R_DIV8                  CFG1_PLL2RDIV(2)                                /*!< CK_PLL2RDIV clock select CK_PLL2R / 8 */
#define RCU_PLL2R_DIV16                 CFG1_PLL2RDIV(3)                                /*!< CK_PLL2RDIV clock select CK_PLL2R / 16 */

/* CK_CST Clock Source*/
#define RCU_CST_SRC

/* TIMER clock selection */
#define RCU_TIMER_PSC_MUL2              ((uint32_t)(0x00000000))                         /*!< if APB1PSC/APB2PSC in RCU_CFG0 register is 0b0xx(CK_APBx = CK_AHB)
                                                                                             or 0b100(CK_APBx = CK_AHB/2), the TIMER clock is equal to CK_AHB(CK_TIMERx = CK_AHB).
                                                                                             or else, the TIMER clock is twice the corresponding APB clock (TIMER in APB1 domain: CK_TIMERx = 2 x CK_APB1;
                                                                                             TIMER in APB2 domain: CK_TIMERx = 2 x CK_APB2) */
#define RCU_TIMER_PSC_MUL4              RCU_CFG1_TIMERSEL                               /*!< if APB1PSC/APB2PSC in RCU_CFG0 register is 0b0xx(CK_APBx = CK_AHB),
                                                                                             0b100(CK_APBx = CK_AHB/2), or 0b101(CK_APBx = CK_AHB/4), the TIMER clock is equal to CK_AHB(CK_TIMERx = CK_AHB).
                                                                                             or else, the TIMER clock is four timers the corresponding APB clock (TIMER in APB1 domain: CK_TIMERx = 4 x CK_APB1;
                                                                                             TIMER in APB2 domain: CK_TIMERx = 4 x CK_APB2) */

/* RCU_PLL register bit define */
/* The PLL(PLL0\PLL1\PLL2) VCO source clock prescaler */
#define PLL_PLLPSC(regval)              (BITS(0,5) & ((uint32_t)(regval) << 0U))
#define RCU_PLL_PSC1                    PLL_PLLPSC(1)                                   /*!< PLLPSC_DIV 1 value */
#define RCU_PLL_PSC2                    PLL_PLLPSC(2)                                   /*!< PLLPSC_DIV 2 value */
#define RCU_PLL_PSC3                    PLL_PLLPSC(3)                                   /*!< PLLPSC_DIV 3 value */
#define RCU_PLL_PSC4                    PLL_PLLPSC(4)                                   /*!< PLLPSC_DIV 4 value */
#define RCU_PLL_PSC5                    PLL_PLLPSC(5)                                   /*!< PLLPSC_DIV 5 value */
#define RCU_PLL_PSC6                    PLL_PLLPSC(6)                                   /*!< PLLPSC_DIV 6 value */
#define RCU_PLL_PSC7                    PLL_PLLPSC(7)                                   /*!< PLLPSC_DIV 7 value */
#define RCU_PLL_PSC8                    PLL_PLLPSC(8)                                   /*!< PLLPSC_DIV 8 value */
#define RCU_PLL_PSC9                    PLL_PLLPSC(9)                                   /*!< PLLPSC_DIV 9 value */
#define RCU_PLL_PSC10                   PLL_PLLPSC(10)                                  /*!< PLLPSC_DIV 10 value */
#define RCU_PLL_PSC11                   PLL_PLLPSC(11)                                  /*!< PLLPSC_DIV 11 value */
#define RCU_PLL_PSC12                   PLL_PLLPSC(12)                                  /*!< PLLPSC_DIV 12 value */
#define RCU_PLL_PSC13                   PLL_PLLPSC(13)                                  /*!< PLLPSC_DIV 13 value */
#define RCU_PLL_PSC14                   PLL_PLLPSC(14)                                  /*!< PLLPSC_DIV 14 value */
#define RCU_PLL_PSC15                   PLL_PLLPSC(15)                                  /*!< PLLPSC_DIV 15 value */
#define RCU_PLL_PSC16                   PLL_PLLPSC(16)                                  /*!< PLLPSC_DIV 16 value */
#define RCU_PLL_PSC17                   PLL_PLLPSC(17)                                  /*!< PLLPSC_DIV 17 value */
#define RCU_PLL_PSC18                   PLL_PLLPSC(18)                                  /*!< PLLPSC_DIV 18 value */
#define RCU_PLL_PSC19                   PLL_PLLPSC(19)                                  /*!< PLLPSC_DIV 19 value */
#define RCU_PLL_PSC20                   PLL_PLLPSC(20)                                  /*!< PLLPSC_DIV 20 value */
#define RCU_PLL_PSC21                   PLL_PLLPSC(21)                                  /*!< PLLPSC_DIV 21 value */
#define RCU_PLL_PSC22                   PLL_PLLPSC(22)                                  /*!< PLLPSC_DIV 22 value */
#define RCU_PLL_PSC23                   PLL_PLLPSC(23)                                  /*!< PLLPSC_DIV 23 value */
#define RCU_PLL_PSC24                   PLL_PLLPSC(24)                                  /*!< PLLPSC_DIV 24 value */
#define RCU_PLL_PSC25                   PLL_PLLPSC(25)                                  /*!< PLLPSC_DIV 25 value */
#define RCU_PLL_PSC26                   PLL_PLLPSC(26)                                  /*!< PLLPSC_DIV 26 value */
#define RCU_PLL_PSC27                   PLL_PLLPSC(27)                                  /*!< PLLPSC_DIV 27 value */
#define RCU_PLL_PSC28                   PLL_PLLPSC(28)                                  /*!< PLLPSC_DIV 28 value */
#define RCU_PLL_PSC29                   PLL_PLLPSC(29)                                  /*!< PLLPSC_DIV 29 value */
#define RCU_PLL_PSC30                   PLL_PLLPSC(30)                                  /*!< PLLPSC_DIV 30 value */
#define RCU_PLL_PSC31                   PLL_PLLPSC(31)                                  /*!< PLLPSC_DIV 31 value */
#define RCU_PLL_PSC32                   PLL_PLLPSC(32)                                  /*!< PLLPSC_DIV 32 value */
#define RCU_PLL_PSC33                   PLL_PLLPSC(33)                                  /*!< PLLPSC_DIV 33 value */
#define RCU_PLL_PSC34                   PLL_PLLPSC(34)                                  /*!< PLLPSC_DIV 34 value */
#define RCU_PLL_PSC35                   PLL_PLLPSC(35)                                  /*!< PLLPSC_DIV 35 value */
#define RCU_PLL_PSC36                   PLL_PLLPSC(36)                                  /*!< PLLPSC_DIV 36 value */
#define RCU_PLL_PSC37                   PLL_PLLPSC(37)                                  /*!< PLLPSC_DIV 37 value */
#define RCU_PLL_PSC38                   PLL_PLLPSC(38)                                  /*!< PLLPSC_DIV 38 value */
#define RCU_PLL_PSC39                   PLL_PLLPSC(39)                                  /*!< PLLPSC_DIV 39 value */
#define RCU_PLL_PSC40                   PLL_PLLPSC(40)                                  /*!< PLLPSC_DIV 40 value */
#define RCU_PLL_PSC41                   PLL_PLLPSC(41)                                  /*!< PLLPSC_DIV 41 value */
#define RCU_PLL_PSC42                   PLL_PLLPSC(42)                                  /*!< PLLPSC_DIV 42 value */
#define RCU_PLL_PSC43                   PLL_PLLPSC(43)                                  /*!< PLLPSC_DIV 43 value */
#define RCU_PLL_PSC44                   PLL_PLLPSC(44)                                  /*!< PLLPSC_DIV 44 value */
#define RCU_PLL_PSC45                   PLL_PLLPSC(45)                                  /*!< PLLPSC_DIV 45 value */
#define RCU_PLL_PSC46                   PLL_PLLPSC(46)                                  /*!< PLLPSC_DIV 46 value */
#define RCU_PLL_PSC47                   PLL_PLLPSC(47)                                  /*!< PLLPSC_DIV 47 value */
#define RCU_PLL_PSC48                   PLL_PLLPSC(48)                                  /*!< PLLPSC_DIV 48 value */
#define RCU_PLL_PSC49                   PLL_PLLPSC(49)                                  /*!< PLLPSC_DIV 49 value */
#define RCU_PLL_PSC50                   PLL_PLLPSC(50)                                  /*!< PLLPSC_DIV 50 value */
#define RCU_PLL_PSC51                   PLL_PLLPSC(51)                                  /*!< PLLPSC_DIV 51 value */
#define RCU_PLL_PSC52                   PLL_PLLPSC(52)                                  /*!< PLLPSC_DIV 52 value */
#define RCU_PLL_PSC53                   PLL_PLLPSC(53)                                  /*!< PLLPSC_DIV 53 value */
#define RCU_PLL_PSC54                   PLL_PLLPSC(54)                                  /*!< PLLPSC_DIV 54 value */
#define RCU_PLL_PSC55                   PLL_PLLPSC(55)                                  /*!< PLLPSC_DIV 55 value */
#define RCU_PLL_PSC56                   PLL_PLLPSC(56)                                  /*!< PLLPSC_DIV 56 value */
#define RCU_PLL_PSC57                   PLL_PLLPSC(57)                                  /*!< PLLPSC_DIV 57 value */
#define RCU_PLL_PSC58                   PLL_PLLPSC(58)                                  /*!< PLLPSC_DIV 58 value */
#define RCU_PLL_PSC59                   PLL_PLLPSC(59)                                  /*!< PLLPSC_DIV 59 value */
#define RCU_PLL_PSC60                   PLL_PLLPSC(60)                                  /*!< PLLPSC_DIV 60 value */
#define RCU_PLL_PSC61                   PLL_PLLPSC(61)                                  /*!< PLLPSC_DIV 61 value */
#define RCU_PLL_PSC62                   PLL_PLLPSC(62)                                  /*!< PLLPSC_DIV 62 value */
#define RCU_PLL_PSC63                   PLL_PLLPSC(63)                                  /*!< PLLPSC_DIV 63 value */

/* The PLL(PLL0\PLL1\PLL2) VCO clock multiplication factor */
#define PLL_PLLN(regval)                (BITS(6,14) & ((uint32_t)(regval) << 6U))
#define RCU_PLLN_MUL9                   PLL_PLLN(8)                                     /*!< PLLN_MUL 9 */
#define RCU_PLLN_MUL10                  PLL_PLLN(9)                                     /*!< PLLN_MUL 10 */
#define RCU_PLLN_MUL11                  PLL_PLLN(10)                                    /*!< PLLN_MUL 11 */
#define RCU_PLLN_MUL12                  PLL_PLLN(11)                                    /*!< PLLN_MUL 12 */
#define RCU_PLLN_MUL13                  PLL_PLLN(12)                                    /*!< PLLN_MUL 13 */
#define RCU_PLLN_MUL14                  PLL_PLLN(13)                                    /*!< PLLN_MUL 14 */
#define RCU_PLLN_MUL15                  PLL_PLLN(14)                                    /*!< PLLN_MUL 15 */
#define RCU_PLLN_MUL16                  PLL_PLLN(15)                                    /*!< PLLN_MUL 16 */
#define RCU_PLLN_MUL17                  PLL_PLLN(16)                                    /*!< PLLN_MUL 17 */
#define RCU_PLLN_MUL18                  PLL_PLLN(17)                                    /*!< PLLN_MUL 18 */
#define RCU_PLLN_MUL19                  PLL_PLLN(18)                                    /*!< PLLN_MUL 19 */
#define RCU_PLLN_MUL20                  PLL_PLLN(19)                                    /*!< PLLN_MUL 20 */
#define RCU_PLLN_MUL21                  PLL_PLLN(20)                                    /*!< PLLN_MUL 21 */
#define RCU_PLLN_MUL22                  PLL_PLLN(21)                                    /*!< PLLN_MUL 22 */
#define RCU_PLLN_MUL23                  PLL_PLLN(22)                                    /*!< PLLN_MUL 23 */
#define RCU_PLLN_MUL24                  PLL_PLLN(23)                                    /*!< PLLN_MUL 24 */
#define RCU_PLLN_MUL25                  PLL_PLLN(24)                                    /*!< PLLN_MUL 25 */
#define RCU_PLLN_MUL26                  PLL_PLLN(25)                                    /*!< PLLN_MUL 26 */
#define RCU_PLLN_MUL27                  PLL_PLLN(26)                                    /*!< PLLN_MUL 27 */
#define RCU_PLLN_MUL28                  PLL_PLLN(27)                                    /*!< PLLN_MUL 28 */
#define RCU_PLLN_MUL29                  PLL_PLLN(28)                                    /*!< PLLN_MUL 29 */
#define RCU_PLLN_MUL30                  PLL_PLLN(39)                                    /*!< PLLN_MUL 30 */
#define RCU_PLLN_MUL31                  PLL_PLLN(30)                                    /*!< PLLN_MUL 31 */
#define RCU_PLLN_MUL32                  PLL_PLLN(31)                                    /*!< PLLN_MUL 32 */
#define RCU_PLLN_MUL33                  PLL_PLLN(32)                                    /*!< PLLN_MUL 33 */
#define RCU_PLLN_MUL34                  PLL_PLLN(33)                                    /*!< PLLN_MUL 34 */
#define RCU_PLLN_MUL35                  PLL_PLLN(34)                                    /*!< PLLN_MUL 35 */
#define RCU_PLLN_MUL36                  PLL_PLLN(35)                                    /*!< PLLN_MUL 36 */
#define RCU_PLLN_MUL37                  PLL_PLLN(36)                                    /*!< PLLN_MUL 37 */
#define RCU_PLLN_MUL38                  PLL_PLLN(37)                                    /*!< PLLN_MUL 38 */
#define RCU_PLLN_MUL39                  PLL_PLLN(38)                                    /*!< PLLN_MUL 39 */
#define RCU_PLLN_MUL40                  PLL_PLLN(39)                                    /*!< PLLN_MUL 40 */
#define RCU_PLLN_MUL41                  PLL_PLLN(40)                                    /*!< PLLN_MUL 41 */
#define RCU_PLLN_MUL42                  PLL_PLLN(41)                                    /*!< PLLN_MUL 42 */
#define RCU_PLLN_MUL43                  PLL_PLLN(42)                                    /*!< PLLN_MUL 43 */
#define RCU_PLLN_MUL44                  PLL_PLLN(43)                                    /*!< PLLN_MUL 44 */
#define RCU_PLLN_MUL45                  PLL_PLLN(44)                                    /*!< PLLN_MUL 45 */
#define RCU_PLLN_MUL46                  PLL_PLLN(45)                                    /*!< PLLN_MUL 46 */
#define RCU_PLLN_MUL47                  PLL_PLLN(46)                                    /*!< PLLN_MUL 47 */
#define RCU_PLLN_MUL48                  PLL_PLLN(47)                                    /*!< PLLN_MUL 48 */
#define RCU_PLLN_MUL49                  PLL_PLLN(48)                                    /*!< PLLN_MUL 49 */
#define RCU_PLLN_MUL50                  PLL_PLLN(49)                                    /*!< PLLN_MUL 50 */
#define RCU_PLLN_MUL51                  PLL_PLLN(50)                                    /*!< PLLN_MUL 51 */
#define RCU_PLLN_MUL52                  PLL_PLLN(51)                                    /*!< PLLN_MUL 52 */
#define RCU_PLLN_MUL53                  PLL_PLLN(52)                                    /*!< PLLN_MUL 53 */
#define RCU_PLLN_MUL54                  PLL_PLLN(53)                                    /*!< PLLN_MUL 54 */
#define RCU_PLLN_MUL55                  PLL_PLLN(54)                                    /*!< PLLN_MUL 55 */
#define RCU_PLLN_MUL56                  PLL_PLLN(55)                                    /*!< PLLN_MUL 56 */
#define RCU_PLLN_MUL57                  PLL_PLLN(56)                                    /*!< PLLN_MUL 57 */
#define RCU_PLLN_MUL58                  PLL_PLLN(57)                                    /*!< PLLN_MUL 58 */
#define RCU_PLLN_MUL59                  PLL_PLLN(58)                                    /*!< PLLN_MUL 59 */
#define RCU_PLLN_MUL60                  PLL_PLLN(59)                                    /*!< PLLN_MUL 60 */
#define RCU_PLLN_MUL61                  PLL_PLLN(60)                                    /*!< PLLN_MUL 61 */
#define RCU_PLLN_MUL62                  PLL_PLLN(61)                                    /*!< PLLN_MUL 62 */
#define RCU_PLLN_MUL63                  PLL_PLLN(62)                                    /*!< PLLN_MUL 63 */
#define RCU_PLLN_MUL64                  PLL_PLLN(63)                                    /*!< PLLN_MUL 64 */
#define RCU_PLLN_MUL65                  PLL_PLLN(64)                                    /*!< PLLN_MUL 65 */
#define RCU_PLLN_MUL66                  PLL_PLLN(65)                                    /*!< PLLN_MUL 66 */
#define RCU_PLLN_MUL67                  PLL_PLLN(66)                                    /*!< PLLN_MUL 67 */
#define RCU_PLLN_MUL68                  PLL_PLLN(67)                                    /*!< PLLN_MUL 68 */
#define RCU_PLLN_MUL69                  PLL_PLLN(68)                                    /*!< PLLN_MUL 69 */
#define RCU_PLLN_MUL70                  PLL_PLLN(69)                                    /*!< PLLN_MUL 70 */
#define RCU_PLLN_MUL71                  PLL_PLLN(70)                                    /*!< PLLN_MUL 71 */
#define RCU_PLLN_MUL72                  PLL_PLLN(71)                                    /*!< PLLN_MUL 72 */
#define RCU_PLLN_MUL73                  PLL_PLLN(72)                                    /*!< PLLN_MUL 73 */
#define RCU_PLLN_MUL74                  PLL_PLLN(73)                                    /*!< PLLN_MUL 74 */
#define RCU_PLLN_MUL75                  PLL_PLLN(74)                                    /*!< PLLN_MUL 75 */
#define RCU_PLLN_MUL76                  PLL_PLLN(75)                                    /*!< PLLN_MUL 76 */
#define RCU_PLLN_MUL77                  PLL_PLLN(76)                                    /*!< PLLN_MUL 77 */
#define RCU_PLLN_MUL78                  PLL_PLLN(77)                                    /*!< PLLN_MUL 78 */
#define RCU_PLLN_MUL79                  PLL_PLLN(78)                                    /*!< PLLN_MUL 79 */
#define RCU_PLLN_MUL80                  PLL_PLLN(79)                                    /*!< PLLN_MUL 80 */
#define RCU_PLLN_MUL81                  PLL_PLLN(80)                                    /*!< PLLN_MUL 81 */
#define RCU_PLLN_MUL82                  PLL_PLLN(81)                                    /*!< PLLN_MUL 82 */
#define RCU_PLLN_MUL83                  PLL_PLLN(82)                                    /*!< PLLN_MUL 83 */
#define RCU_PLLN_MUL84                  PLL_PLLN(83)                                    /*!< PLLN_MUL 84 */
#define RCU_PLLN_MUL85                  PLL_PLLN(84)                                    /*!< PLLN_MUL 85 */
#define RCU_PLLN_MUL86                  PLL_PLLN(85)                                    /*!< PLLN_MUL 86 */
#define RCU_PLLN_MUL87                  PLL_PLLN(86)                                    /*!< PLLN_MUL 87 */
#define RCU_PLLN_MUL88                  PLL_PLLN(87)                                    /*!< PLLN_MUL 88 */
#define RCU_PLLN_MUL89                  PLL_PLLN(88)                                    /*!< PLLN_MUL 89 */
#define RCU_PLLN_MUL90                  PLL_PLLN(89)                                    /*!< PLLN_MUL 90 */
#define RCU_PLLN_MUL91                  PLL_PLLN(90)                                    /*!< PLLN_MUL 91 */
#define RCU_PLLN_MUL92                  PLL_PLLN(91)                                    /*!< PLLN_MUL 92 */
#define RCU_PLLN_MUL93                  PLL_PLLN(92)                                    /*!< PLLN_MUL 93 */
#define RCU_PLLN_MUL94                  PLL_PLLN(93)                                    /*!< PLLN_MUL 94 */
#define RCU_PLLN_MUL95                  PLL_PLLN(94)                                    /*!< PLLN_MUL 95 */
#define RCU_PLLN_MUL96                  PLL_PLLN(95)                                    /*!< PLLN_MUL 96 */
#define RCU_PLLN_MUL97                  PLL_PLLN(96)                                    /*!< PLLN_MUL 97 */
#define RCU_PLLN_MUL98                  PLL_PLLN(97)                                    /*!< PLLN_MUL 98 */
#define RCU_PLLN_MUL99                  PLL_PLLN(98)                                    /*!< PLLN_MUL 99 */
#define RCU_PLLN_MUL100                 PLL_PLLN(99)                                    /*!< PLLN_MUL 100 */
#define RCU_PLLN_MUL101                 PLL_PLLN(100)                                   /*!< PLLN_MUL 101 */
#define RCU_PLLN_MUL102                 PLL_PLLN(101)                                   /*!< PLLN_MUL 102 */
#define RCU_PLLN_MUL103                 PLL_PLLN(102)                                   /*!< PLLN_MUL 103 */
#define RCU_PLLN_MUL104                 PLL_PLLN(103)                                   /*!< PLLN_MUL 104 */
#define RCU_PLLN_MUL105                 PLL_PLLN(104)                                   /*!< PLLN_MUL 105 */
#define RCU_PLLN_MUL106                 PLL_PLLN(105)                                   /*!< PLLN_MUL 106 */
#define RCU_PLLN_MUL107                 PLL_PLLN(106)                                   /*!< PLLN_MUL 107 */
#define RCU_PLLN_MUL108                 PLL_PLLN(107)                                   /*!< PLLN_MUL 108 */
#define RCU_PLLN_MUL109                 PLL_PLLN(108)                                   /*!< PLLN_MUL 109 */
#define RCU_PLLN_MUL110                 PLL_PLLN(109)                                   /*!< PLLN_MUL 110 */
#define RCU_PLLN_MUL111                 PLL_PLLN(110)                                   /*!< PLLN_MUL 111 */
#define RCU_PLLN_MUL112                 PLL_PLLN(111)                                   /*!< PLLN_MUL 112 */
#define RCU_PLLN_MUL113                 PLL_PLLN(112)                                   /*!< PLLN_MUL 113 */
#define RCU_PLLN_MUL114                 PLL_PLLN(113)                                   /*!< PLLN_MUL 114 */
#define RCU_PLLN_MUL115                 PLL_PLLN(114)                                   /*!< PLLN_MUL 115 */
#define RCU_PLLN_MUL116                 PLL_PLLN(115)                                   /*!< PLLN_MUL 116 */
#define RCU_PLLN_MUL117                 PLL_PLLN(116)                                   /*!< PLLN_MUL 117 */
#define RCU_PLLN_MUL118                 PLL_PLLN(117)                                   /*!< PLLN_MUL 118 */
#define RCU_PLLN_MUL119                 PLL_PLLN(118)                                   /*!< PLLN_MUL 119 */
#define RCU_PLLN_MUL120                 PLL_PLLN(119)                                   /*!< PLLN_MUL 120 */
#define RCU_PLLN_MUL121                 PLL_PLLN(120)                                   /*!< PLLN_MUL 121 */
#define RCU_PLLN_MUL122                 PLL_PLLN(121)                                   /*!< PLLN_MUL 122 */
#define RCU_PLLN_MUL123                 PLL_PLLN(122)                                   /*!< PLLN_MUL 123 */
#define RCU_PLLN_MUL124                 PLL_PLLN(123)                                   /*!< PLLN_MUL 124 */
#define RCU_PLLN_MUL125                 PLL_PLLN(124)                                   /*!< PLLN_MUL 125 */
#define RCU_PLLN_MUL126                 PLL_PLLN(125)                                   /*!< PLLN_MUL 126 */
#define RCU_PLLN_MUL127                 PLL_PLLN(126)                                   /*!< PLLN_MUL 127 */
#define RCU_PLLN_MUL128                 PLL_PLLN(127)                                   /*!< PLLN_MUL 128 */
#define RCU_PLLN_MUL129                 PLL_PLLN(128)                                   /*!< PLLN_MUL 129 */
#define RCU_PLLN_MUL130                 PLL_PLLN(139)                                   /*!< PLLN_MUL 130 */
#define RCU_PLLN_MUL131                 PLL_PLLN(130)                                   /*!< PLLN_MUL 131 */
#define RCU_PLLN_MUL132                 PLL_PLLN(131)                                   /*!< PLLN_MUL 132 */
#define RCU_PLLN_MUL133                 PLL_PLLN(132)                                   /*!< PLLN_MUL 133 */
#define RCU_PLLN_MUL134                 PLL_PLLN(133)                                   /*!< PLLN_MUL 134 */
#define RCU_PLLN_MUL135                 PLL_PLLN(134)                                   /*!< PLLN_MUL 135 */
#define RCU_PLLN_MUL136                 PLL_PLLN(135)                                   /*!< PLLN_MUL 136 */
#define RCU_PLLN_MUL137                 PLL_PLLN(136)                                   /*!< PLLN_MUL 137 */
#define RCU_PLLN_MUL138                 PLL_PLLN(137)                                   /*!< PLLN_MUL 138 */
#define RCU_PLLN_MUL139                 PLL_PLLN(138)                                   /*!< PLLN_MUL 139 */
#define RCU_PLLN_MUL140                 PLL_PLLN(139)                                   /*!< PLLN_MUL 140 */
#define RCU_PLLN_MUL141                 PLL_PLLN(140)                                   /*!< PLLN_MUL 141 */
#define RCU_PLLN_MUL142                 PLL_PLLN(141)                                   /*!< PLLN_MUL 142 */
#define RCU_PLLN_MUL143                 PLL_PLLN(142)                                   /*!< PLLN_MUL 143 */
#define RCU_PLLN_MUL144                 PLL_PLLN(143)                                   /*!< PLLN_MUL 144 */
#define RCU_PLLN_MUL145                 PLL_PLLN(144)                                   /*!< PLLN_MUL 145 */
#define RCU_PLLN_MUL146                 PLL_PLLN(145)                                   /*!< PLLN_MUL 146 */
#define RCU_PLLN_MUL147                 PLL_PLLN(146)                                   /*!< PLLN_MUL 147 */
#define RCU_PLLN_MUL148                 PLL_PLLN(147)                                   /*!< PLLN_MUL 148 */
#define RCU_PLLN_MUL149                 PLL_PLLN(148)                                   /*!< PLLN_MUL 149 */
#define RCU_PLLN_MUL150                 PLL_PLLN(149)                                   /*!< PLLN_MUL 150 */
#define RCU_PLLN_MUL151                 PLL_PLLN(150)                                   /*!< PLLN_MUL 151 */
#define RCU_PLLN_MUL152                 PLL_PLLN(151)                                   /*!< PLLN_MUL 152 */
#define RCU_PLLN_MUL153                 PLL_PLLN(152)                                   /*!< PLLN_MUL 153 */
#define RCU_PLLN_MUL154                 PLL_PLLN(153)                                   /*!< PLLN_MUL 154 */
#define RCU_PLLN_MUL155                 PLL_PLLN(154)                                   /*!< PLLN_MUL 155 */
#define RCU_PLLN_MUL156                 PLL_PLLN(155)                                   /*!< PLLN_MUL 156 */
#define RCU_PLLN_MUL157                 PLL_PLLN(156)                                   /*!< PLLN_MUL 157 */
#define RCU_PLLN_MUL158                 PLL_PLLN(157)                                   /*!< PLLN_MUL 158 */
#define RCU_PLLN_MUL159                 PLL_PLLN(158)                                   /*!< PLLN_MUL 159 */
#define RCU_PLLN_MUL160                 PLL_PLLN(159)                                   /*!< PLLN_MUL 160 */
#define RCU_PLLN_MUL161                 PLL_PLLN(160)                                   /*!< PLLN_MUL 161 */
#define RCU_PLLN_MUL162                 PLL_PLLN(161)                                   /*!< PLLN_MUL 162 */
#define RCU_PLLN_MUL163                 PLL_PLLN(162)                                   /*!< PLLN_MUL 163 */
#define RCU_PLLN_MUL164                 PLL_PLLN(163)                                   /*!< PLLN_MUL 164 */
#define RCU_PLLN_MUL165                 PLL_PLLN(164)                                   /*!< PLLN_MUL 165 */
#define RCU_PLLN_MUL166                 PLL_PLLN(165)                                   /*!< PLLN_MUL 166 */
#define RCU_PLLN_MUL167                 PLL_PLLN(166)                                   /*!< PLLN_MUL 167 */
#define RCU_PLLN_MUL168                 PLL_PLLN(167)                                   /*!< PLLN_MUL 168 */
#define RCU_PLLN_MUL169                 PLL_PLLN(168)                                   /*!< PLLN_MUL 169 */
#define RCU_PLLN_MUL170                 PLL_PLLN(169)                                   /*!< PLLN_MUL 170 */
#define RCU_PLLN_MUL171                 PLL_PLLN(170)                                   /*!< PLLN_MUL 171 */
#define RCU_PLLN_MUL172                 PLL_PLLN(171)                                   /*!< PLLN_MUL 172 */
#define RCU_PLLN_MUL173                 PLL_PLLN(172)                                   /*!< PLLN_MUL 173 */
#define RCU_PLLN_MUL174                 PLL_PLLN(173)                                   /*!< PLLN_MUL 174 */
#define RCU_PLLN_MUL175                 PLL_PLLN(174)                                   /*!< PLLN_MUL 175 */
#define RCU_PLLN_MUL176                 PLL_PLLN(175)                                   /*!< PLLN_MUL 176 */
#define RCU_PLLN_MUL177                 PLL_PLLN(176)                                   /*!< PLLN_MUL 177 */
#define RCU_PLLN_MUL178                 PLL_PLLN(177)                                   /*!< PLLN_MUL 178 */
#define RCU_PLLN_MUL179                 PLL_PLLN(178)                                   /*!< PLLN_MUL 179 */
#define RCU_PLLN_MUL180                 PLL_PLLN(179)                                   /*!< PLLN_MUL 180 */
#define RCU_PLLN_MUL181                 PLL_PLLN(180)                                   /*!< PLLN_MUL 181 */
#define RCU_PLLN_MUL182                 PLL_PLLN(181)                                   /*!< PLLN_MUL 182 */
#define RCU_PLLN_MUL183                 PLL_PLLN(182)                                   /*!< PLLN_MUL 183 */
#define RCU_PLLN_MUL184                 PLL_PLLN(183)                                   /*!< PLLN_MUL 184 */
#define RCU_PLLN_MUL185                 PLL_PLLN(184)                                   /*!< PLLN_MUL 185 */
#define RCU_PLLN_MUL186                 PLL_PLLN(185)                                   /*!< PLLN_MUL 186 */
#define RCU_PLLN_MUL187                 PLL_PLLN(186)                                   /*!< PLLN_MUL 187 */
#define RCU_PLLN_MUL188                 PLL_PLLN(187)                                   /*!< PLLN_MUL 188 */
#define RCU_PLLN_MUL189                 PLL_PLLN(188)                                   /*!< PLLN_MUL 189 */
#define RCU_PLLN_MUL190                 PLL_PLLN(189)                                   /*!< PLLN_MUL 190 */
#define RCU_PLLN_MUL191                 PLL_PLLN(190)                                   /*!< PLLN_MUL 191 */
#define RCU_PLLN_MUL192                 PLL_PLLN(191)                                   /*!< PLLN_MUL 192 */
#define RCU_PLLN_MUL193                 PLL_PLLN(192)                                   /*!< PLLN_MUL 193 */
#define RCU_PLLN_MUL194                 PLL_PLLN(193)                                   /*!< PLLN_MUL 194 */
#define RCU_PLLN_MUL195                 PLL_PLLN(194)                                   /*!< PLLN_MUL 195 */
#define RCU_PLLN_MUL196                 PLL_PLLN(195)                                   /*!< PLLN_MUL 196 */
#define RCU_PLLN_MUL197                 PLL_PLLN(196)                                   /*!< PLLN_MUL 197 */
#define RCU_PLLN_MUL198                 PLL_PLLN(197)                                   /*!< PLLN_MUL 198 */
#define RCU_PLLN_MUL199                 PLL_PLLN(198)                                   /*!< PLLN_MUL 199 */
#define RCU_PLLN_MUL200                 PLL_PLLN(199)                                   /*!< PLLN_MUL 200 */
#define RCU_PLLN_MUL201                 PLL_PLLN(200)                                   /*!< PLLN_MUL 201 */
#define RCU_PLLN_MUL202                 PLL_PLLN(201)                                   /*!< PLLN_MUL 202 */
#define RCU_PLLN_MUL203                 PLL_PLLN(202)                                   /*!< PLLN_MUL 203 */
#define RCU_PLLN_MUL204                 PLL_PLLN(203)                                   /*!< PLLN_MUL 204 */
#define RCU_PLLN_MUL205                 PLL_PLLN(204)                                   /*!< PLLN_MUL 205 */
#define RCU_PLLN_MUL206                 PLL_PLLN(205)                                   /*!< PLLN_MUL 206 */
#define RCU_PLLN_MUL207                 PLL_PLLN(206)                                   /*!< PLLN_MUL 207 */
#define RCU_PLLN_MUL208                 PLL_PLLN(207)                                   /*!< PLLN_MUL 208 */
#define RCU_PLLN_MUL209                 PLL_PLLN(208)                                   /*!< PLLN_MUL 209 */
#define RCU_PLLN_MUL210                 PLL_PLLN(209)                                   /*!< PLLN_MUL 210 */
#define RCU_PLLN_MUL211                 PLL_PLLN(210)                                   /*!< PLLN_MUL 211 */
#define RCU_PLLN_MUL212                 PLL_PLLN(211)                                   /*!< PLLN_MUL 212 */
#define RCU_PLLN_MUL213                 PLL_PLLN(212)                                   /*!< PLLN_MUL 213 */
#define RCU_PLLN_MUL214                 PLL_PLLN(213)                                   /*!< PLLN_MUL 214 */
#define RCU_PLLN_MUL215                 PLL_PLLN(214)                                   /*!< PLLN_MUL 215 */
#define RCU_PLLN_MUL216                 PLL_PLLN(215)                                   /*!< PLLN_MUL 216 */
#define RCU_PLLN_MUL217                 PLL_PLLN(216)                                   /*!< PLLN_MUL 217 */
#define RCU_PLLN_MUL218                 PLL_PLLN(217)                                   /*!< PLLN_MUL 218 */
#define RCU_PLLN_MUL219                 PLL_PLLN(218)                                   /*!< PLLN_MUL 219 */
#define RCU_PLLN_MUL220                 PLL_PLLN(219)                                   /*!< PLLN_MUL 220 */
#define RCU_PLLN_MUL221                 PLL_PLLN(220)                                   /*!< PLLN_MUL 221 */
#define RCU_PLLN_MUL222                 PLL_PLLN(221)                                   /*!< PLLN_MUL 222 */
#define RCU_PLLN_MUL223                 PLL_PLLN(222)                                   /*!< PLLN_MUL 223 */
#define RCU_PLLN_MUL224                 PLL_PLLN(223)                                   /*!< PLLN_MUL 224 */
#define RCU_PLLN_MUL225                 PLL_PLLN(224)                                   /*!< PLLN_MUL 225 */
#define RCU_PLLN_MUL226                 PLL_PLLN(225)                                   /*!< PLLN_MUL 226 */
#define RCU_PLLN_MUL227                 PLL_PLLN(226)                                   /*!< PLLN_MUL 227 */
#define RCU_PLLN_MUL228                 PLL_PLLN(227)                                   /*!< PLLN_MUL 228 */
#define RCU_PLLN_MUL229                 PLL_PLLN(228)                                   /*!< PLLN_MUL 229 */
#define RCU_PLLN_MUL230                 PLL_PLLN(239)                                   /*!< PLLN_MUL 230 */
#define RCU_PLLN_MUL231                 PLL_PLLN(230)                                   /*!< PLLN_MUL 231 */
#define RCU_PLLN_MUL232                 PLL_PLLN(231)                                   /*!< PLLN_MUL 232 */
#define RCU_PLLN_MUL233                 PLL_PLLN(232)                                   /*!< PLLN_MUL 233 */
#define RCU_PLLN_MUL234                 PLL_PLLN(233)                                   /*!< PLLN_MUL 234 */
#define RCU_PLLN_MUL235                 PLL_PLLN(234)                                   /*!< PLLN_MUL 235 */
#define RCU_PLLN_MUL236                 PLL_PLLN(235)                                   /*!< PLLN_MUL 236 */
#define RCU_PLLN_MUL237                 PLL_PLLN(236)                                   /*!< PLLN_MUL 237 */
#define RCU_PLLN_MUL238                 PLL_PLLN(237)                                   /*!< PLLN_MUL 238 */
#define RCU_PLLN_MUL239                 PLL_PLLN(238)                                   /*!< PLLN_MUL 239 */
#define RCU_PLLN_MUL240                 PLL_PLLN(239)                                   /*!< PLLN_MUL 240 */
#define RCU_PLLN_MUL241                 PLL_PLLN(240)                                   /*!< PLLN_MUL 241 */
#define RCU_PLLN_MUL242                 PLL_PLLN(241)                                   /*!< PLLN_MUL 242 */
#define RCU_PLLN_MUL243                 PLL_PLLN(242)                                   /*!< PLLN_MUL 243 */
#define RCU_PLLN_MUL244                 PLL_PLLN(243)                                   /*!< PLLN_MUL 244 */
#define RCU_PLLN_MUL245                 PLL_PLLN(244)                                   /*!< PLLN_MUL 245 */
#define RCU_PLLN_MUL246                 PLL_PLLN(245)                                   /*!< PLLN_MUL 246 */
#define RCU_PLLN_MUL247                 PLL_PLLN(246)                                   /*!< PLLN_MUL 247 */
#define RCU_PLLN_MUL248                 PLL_PLLN(247)                                   /*!< PLLN_MUL 248 */
#define RCU_PLLN_MUL249                 PLL_PLLN(248)                                   /*!< PLLN_MUL 249 */
#define RCU_PLLN_MUL250                 PLL_PLLN(249)                                   /*!< PLLN_MUL 250 */
#define RCU_PLLN_MUL251                 PLL_PLLN(250)                                   /*!< PLLN_MUL 251 */
#define RCU_PLLN_MUL252                 PLL_PLLN(251)                                   /*!< PLLN_MUL 252 */
#define RCU_PLLN_MUL253                 PLL_PLLN(252)                                   /*!< PLLN_MUL 253 */
#define RCU_PLLN_MUL254                 PLL_PLLN(253)                                   /*!< PLLN_MUL 254 */
#define RCU_PLLN_MUL255                 PLL_PLLN(254)                                   /*!< PLLN_MUL 255 */
#define RCU_PLLN_MUL256                 PLL_PLLN(255)                                   /*!< PLLN_MUL 256 */
#define RCU_PLLN_MUL257                 PLL_PLLN(256)                                   /*!< PLLN_MUL 257 */
#define RCU_PLLN_MUL258                 PLL_PLLN(257)                                   /*!< PLLN_MUL 258 */
#define RCU_PLLN_MUL259                 PLL_PLLN(258)                                   /*!< PLLN_MUL 259 */
#define RCU_PLLN_MUL260                 PLL_PLLN(259)                                   /*!< PLLN_MUL 260 */
#define RCU_PLLN_MUL261                 PLL_PLLN(260)                                   /*!< PLLN_MUL 261 */
#define RCU_PLLN_MUL262                 PLL_PLLN(261)                                   /*!< PLLN_MUL 262 */
#define RCU_PLLN_MUL263                 PLL_PLLN(262)                                   /*!< PLLN_MUL 263 */
#define RCU_PLLN_MUL264                 PLL_PLLN(263)                                   /*!< PLLN_MUL 264 */
#define RCU_PLLN_MUL265                 PLL_PLLN(264)                                   /*!< PLLN_MUL 265 */
#define RCU_PLLN_MUL266                 PLL_PLLN(265)                                   /*!< PLLN_MUL 266 */
#define RCU_PLLN_MUL267                 PLL_PLLN(266)                                   /*!< PLLN_MUL 267 */
#define RCU_PLLN_MUL268                 PLL_PLLN(267)                                   /*!< PLLN_MUL 268 */
#define RCU_PLLN_MUL269                 PLL_PLLN(268)                                   /*!< PLLN_MUL 269 */
#define RCU_PLLN_MUL270                 PLL_PLLN(269)                                   /*!< PLLN_MUL 270 */
#define RCU_PLLN_MUL271                 PLL_PLLN(270)                                   /*!< PLLN_MUL 271 */
#define RCU_PLLN_MUL272                 PLL_PLLN(271)                                   /*!< PLLN_MUL 272 */
#define RCU_PLLN_MUL273                 PLL_PLLN(272)                                   /*!< PLLN_MUL 273 */
#define RCU_PLLN_MUL274                 PLL_PLLN(273)                                   /*!< PLLN_MUL 274 */
#define RCU_PLLN_MUL275                 PLL_PLLN(274)                                   /*!< PLLN_MUL 275 */
#define RCU_PLLN_MUL276                 PLL_PLLN(275)                                   /*!< PLLN_MUL 276 */
#define RCU_PLLN_MUL277                 PLL_PLLN(276)                                   /*!< PLLN_MUL 277 */
#define RCU_PLLN_MUL278                 PLL_PLLN(277)                                   /*!< PLLN_MUL 278 */
#define RCU_PLLN_MUL279                 PLL_PLLN(278)                                   /*!< PLLN_MUL 279 */
#define RCU_PLLN_MUL280                 PLL_PLLN(279)                                   /*!< PLLN_MUL 280 */
#define RCU_PLLN_MUL281                 PLL_PLLN(280)                                   /*!< PLLN_MUL 281 */
#define RCU_PLLN_MUL282                 PLL_PLLN(281)                                   /*!< PLLN_MUL 282 */
#define RCU_PLLN_MUL283                 PLL_PLLN(282)                                   /*!< PLLN_MUL 283 */
#define RCU_PLLN_MUL284                 PLL_PLLN(283)                                   /*!< PLLN_MUL 284 */
#define RCU_PLLN_MUL285                 PLL_PLLN(284)                                   /*!< PLLN_MUL 285 */
#define RCU_PLLN_MUL286                 PLL_PLLN(285)                                   /*!< PLLN_MUL 286 */
#define RCU_PLLN_MUL287                 PLL_PLLN(286)                                   /*!< PLLN_MUL 287 */
#define RCU_PLLN_MUL288                 PLL_PLLN(287)                                   /*!< PLLN_MUL 288 */
#define RCU_PLLN_MUL289                 PLL_PLLN(288)                                   /*!< PLLN_MUL 289 */
#define RCU_PLLN_MUL290                 PLL_PLLN(289)                                   /*!< PLLN_MUL 290 */
#define RCU_PLLN_MUL291                 PLL_PLLN(290)                                   /*!< PLLN_MUL 291 */
#define RCU_PLLN_MUL292                 PLL_PLLN(291)                                   /*!< PLLN_MUL 292 */
#define RCU_PLLN_MUL293                 PLL_PLLN(292)                                   /*!< PLLN_MUL 293 */
#define RCU_PLLN_MUL294                 PLL_PLLN(293)                                   /*!< PLLN_MUL 294 */
#define RCU_PLLN_MUL295                 PLL_PLLN(294)                                   /*!< PLLN_MUL 295 */
#define RCU_PLLN_MUL296                 PLL_PLLN(295)                                   /*!< PLLN_MUL 296 */
#define RCU_PLLN_MUL297                 PLL_PLLN(296)                                   /*!< PLLN_MUL 297 */
#define RCU_PLLN_MUL298                 PLL_PLLN(297)                                   /*!< PLLN_MUL 298 */
#define RCU_PLLN_MUL299                 PLL_PLLN(298)                                   /*!< PLLN_MUL 299 */
#define RCU_PLLN_MUL300                 PLL_PLLN(299)                                   /*!< PLLN_MUL 300 */
#define RCU_PLLN_MUL301                 PLL_PLLN(300)                                   /*!< PLLN_MUL 301 */
#define RCU_PLLN_MUL302                 PLL_PLLN(301)                                   /*!< PLLN_MUL 302 */
#define RCU_PLLN_MUL303                 PLL_PLLN(302)                                   /*!< PLLN_MUL 303 */
#define RCU_PLLN_MUL304                 PLL_PLLN(303)                                   /*!< PLLN_MUL 304 */
#define RCU_PLLN_MUL305                 PLL_PLLN(304)                                   /*!< PLLN_MUL 305 */
#define RCU_PLLN_MUL306                 PLL_PLLN(305)                                   /*!< PLLN_MUL 306 */
#define RCU_PLLN_MUL307                 PLL_PLLN(306)                                   /*!< PLLN_MUL 307 */
#define RCU_PLLN_MUL308                 PLL_PLLN(307)                                   /*!< PLLN_MUL 308 */
#define RCU_PLLN_MUL309                 PLL_PLLN(308)                                   /*!< PLLN_MUL 309 */
#define RCU_PLLN_MUL310                 PLL_PLLN(309)                                   /*!< PLLN_MUL 310 */
#define RCU_PLLN_MUL311                 PLL_PLLN(310)                                   /*!< PLLN_MUL 311 */
#define RCU_PLLN_MUL312                 PLL_PLLN(311)                                   /*!< PLLN_MUL 312 */
#define RCU_PLLN_MUL313                 PLL_PLLN(312)                                   /*!< PLLN_MUL 313 */
#define RCU_PLLN_MUL314                 PLL_PLLN(313)                                   /*!< PLLN_MUL 314 */
#define RCU_PLLN_MUL315                 PLL_PLLN(314)                                   /*!< PLLN_MUL 315 */
#define RCU_PLLN_MUL316                 PLL_PLLN(315)                                   /*!< PLLN_MUL 316 */
#define RCU_PLLN_MUL317                 PLL_PLLN(316)                                   /*!< PLLN_MUL 317 */
#define RCU_PLLN_MUL318                 PLL_PLLN(317)                                   /*!< PLLN_MUL 318 */
#define RCU_PLLN_MUL319                 PLL_PLLN(318)                                   /*!< PLLN_MUL 319 */
#define RCU_PLLN_MUL320                 PLL_PLLN(319)                                   /*!< PLLN_MUL 320 */
#define RCU_PLLN_MUL321                 PLL_PLLN(320)                                   /*!< PLLN_MUL 321 */
#define RCU_PLLN_MUL322                 PLL_PLLN(321)                                   /*!< PLLN_MUL 322 */
#define RCU_PLLN_MUL323                 PLL_PLLN(322)                                   /*!< PLLN_MUL 323 */
#define RCU_PLLN_MUL324                 PLL_PLLN(323)                                   /*!< PLLN_MUL 324 */
#define RCU_PLLN_MUL325                 PLL_PLLN(324)                                   /*!< PLLN_MUL 325 */
#define RCU_PLLN_MUL326                 PLL_PLLN(325)                                   /*!< PLLN_MUL 326 */
#define RCU_PLLN_MUL327                 PLL_PLLN(326)                                   /*!< PLLN_MUL 327 */
#define RCU_PLLN_MUL328                 PLL_PLLN(327)                                   /*!< PLLN_MUL 328 */
#define RCU_PLLN_MUL329                 PLL_PLLN(328)                                   /*!< PLLN_MUL 329 */
#define RCU_PLLN_MUL330                 PLL_PLLN(339)                                   /*!< PLLN_MUL 330 */
#define RCU_PLLN_MUL331                 PLL_PLLN(330)                                   /*!< PLLN_MUL 331 */
#define RCU_PLLN_MUL332                 PLL_PLLN(331)                                   /*!< PLLN_MUL 332 */
#define RCU_PLLN_MUL333                 PLL_PLLN(332)                                   /*!< PLLN_MUL 333 */
#define RCU_PLLN_MUL334                 PLL_PLLN(333)                                   /*!< PLLN_MUL 334 */
#define RCU_PLLN_MUL335                 PLL_PLLN(334)                                   /*!< PLLN_MUL 335 */
#define RCU_PLLN_MUL336                 PLL_PLLN(335)                                   /*!< PLLN_MUL 336 */
#define RCU_PLLN_MUL337                 PLL_PLLN(336)                                   /*!< PLLN_MUL 337 */
#define RCU_PLLN_MUL338                 PLL_PLLN(337)                                   /*!< PLLN_MUL 338 */
#define RCU_PLLN_MUL339                 PLL_PLLN(338)                                   /*!< PLLN_MUL 339 */
#define RCU_PLLN_MUL340                 PLL_PLLN(339)                                   /*!< PLLN_MUL 340 */
#define RCU_PLLN_MUL341                 PLL_PLLN(340)                                   /*!< PLLN_MUL 341 */
#define RCU_PLLN_MUL342                 PLL_PLLN(341)                                   /*!< PLLN_MUL 342 */
#define RCU_PLLN_MUL343                 PLL_PLLN(342)                                   /*!< PLLN_MUL 343 */
#define RCU_PLLN_MUL344                 PLL_PLLN(343)                                   /*!< PLLN_MUL 344 */
#define RCU_PLLN_MUL345                 PLL_PLLN(344)                                   /*!< PLLN_MUL 345 */
#define RCU_PLLN_MUL346                 PLL_PLLN(345)                                   /*!< PLLN_MUL 346 */
#define RCU_PLLN_MUL347                 PLL_PLLN(346)                                   /*!< PLLN_MUL 347 */
#define RCU_PLLN_MUL348                 PLL_PLLN(347)                                   /*!< PLLN_MUL 348 */
#define RCU_PLLN_MUL349                 PLL_PLLN(348)                                   /*!< PLLN_MUL 349 */
#define RCU_PLLN_MUL350                 PLL_PLLN(349)                                   /*!< PLLN_MUL 350 */
#define RCU_PLLN_MUL351                 PLL_PLLN(350)                                   /*!< PLLN_MUL 351 */
#define RCU_PLLN_MUL352                 PLL_PLLN(351)                                   /*!< PLLN_MUL 352 */
#define RCU_PLLN_MUL353                 PLL_PLLN(352)                                   /*!< PLLN_MUL 353 */
#define RCU_PLLN_MUL354                 PLL_PLLN(353)                                   /*!< PLLN_MUL 354 */
#define RCU_PLLN_MUL355                 PLL_PLLN(354)                                   /*!< PLLN_MUL 355 */
#define RCU_PLLN_MUL356                 PLL_PLLN(355)                                   /*!< PLLN_MUL 356 */
#define RCU_PLLN_MUL357                 PLL_PLLN(356)                                   /*!< PLLN_MUL 357 */
#define RCU_PLLN_MUL358                 PLL_PLLN(357)                                   /*!< PLLN_MUL 358 */
#define RCU_PLLN_MUL359                 PLL_PLLN(358)                                   /*!< PLLN_MUL 359 */
#define RCU_PLLN_MUL360                 PLL_PLLN(359)                                   /*!< PLLN_MUL 360 */
#define RCU_PLLN_MUL361                 PLL_PLLN(360)                                   /*!< PLLN_MUL 361 */
#define RCU_PLLN_MUL362                 PLL_PLLN(361)                                   /*!< PLLN_MUL 362 */
#define RCU_PLLN_MUL363                 PLL_PLLN(362)                                   /*!< PLLN_MUL 363 */
#define RCU_PLLN_MUL364                 PLL_PLLN(363)                                   /*!< PLLN_MUL 364 */
#define RCU_PLLN_MUL365                 PLL_PLLN(364)                                   /*!< PLLN_MUL 365 */
#define RCU_PLLN_MUL366                 PLL_PLLN(365)                                   /*!< PLLN_MUL 366 */
#define RCU_PLLN_MUL367                 PLL_PLLN(366)                                   /*!< PLLN_MUL 367 */
#define RCU_PLLN_MUL368                 PLL_PLLN(367)                                   /*!< PLLN_MUL 368 */
#define RCU_PLLN_MUL369                 PLL_PLLN(368)                                   /*!< PLLN_MUL 369 */
#define RCU_PLLN_MUL370                 PLL_PLLN(369)                                   /*!< PLLN_MUL 370 */
#define RCU_PLLN_MUL371                 PLL_PLLN(370)                                   /*!< PLLN_MUL 371 */
#define RCU_PLLN_MUL372                 PLL_PLLN(371)                                   /*!< PLLN_MUL 372 */
#define RCU_PLLN_MUL373                 PLL_PLLN(372)                                   /*!< PLLN_MUL 373 */
#define RCU_PLLN_MUL374                 PLL_PLLN(373)                                   /*!< PLLN_MUL 374 */
#define RCU_PLLN_MUL375                 PLL_PLLN(374)                                   /*!< PLLN_MUL 375 */
#define RCU_PLLN_MUL376                 PLL_PLLN(375)                                   /*!< PLLN_MUL 376 */
#define RCU_PLLN_MUL377                 PLL_PLLN(376)                                   /*!< PLLN_MUL 377 */
#define RCU_PLLN_MUL378                 PLL_PLLN(377)                                   /*!< PLLN_MUL 378 */
#define RCU_PLLN_MUL379                 PLL_PLLN(378)                                   /*!< PLLN_MUL 379 */
#define RCU_PLLN_MUL380                 PLL_PLLN(379)                                   /*!< PLLN_MUL 380 */
#define RCU_PLLN_MUL381                 PLL_PLLN(380)                                   /*!< PLLN_MUL 381 */
#define RCU_PLLN_MUL382                 PLL_PLLN(381)                                   /*!< PLLN_MUL 382 */
#define RCU_PLLN_MUL383                 PLL_PLLN(382)                                   /*!< PLLN_MUL 383 */
#define RCU_PLLN_MUL384                 PLL_PLLN(383)                                   /*!< PLLN_MUL 384 */
#define RCU_PLLN_MUL385                 PLL_PLLN(384)                                   /*!< PLLN_MUL 385 */
#define RCU_PLLN_MUL386                 PLL_PLLN(385)                                   /*!< PLLN_MUL 386 */
#define RCU_PLLN_MUL387                 PLL_PLLN(386)                                   /*!< PLLN_MUL 387 */
#define RCU_PLLN_MUL388                 PLL_PLLN(387)                                   /*!< PLLN_MUL 388 */
#define RCU_PLLN_MUL389                 PLL_PLLN(388)                                   /*!< PLLN_MUL 389 */
#define RCU_PLLN_MUL390                 PLL_PLLN(389)                                   /*!< PLLN_MUL 390 */
#define RCU_PLLN_MUL391                 PLL_PLLN(390)                                   /*!< PLLN_MUL 391 */
#define RCU_PLLN_MUL392                 PLL_PLLN(391)                                   /*!< PLLN_MUL 392 */
#define RCU_PLLN_MUL393                 PLL_PLLN(392)                                   /*!< PLLN_MUL 393 */
#define RCU_PLLN_MUL394                 PLL_PLLN(393)                                   /*!< PLLN_MUL 394 */
#define RCU_PLLN_MUL395                 PLL_PLLN(394)                                   /*!< PLLN_MUL 395 */
#define RCU_PLLN_MUL396                 PLL_PLLN(395)                                   /*!< PLLN_MUL 396 */
#define RCU_PLLN_MUL397                 PLL_PLLN(396)                                   /*!< PLLN_MUL 397 */
#define RCU_PLLN_MUL398                 PLL_PLLN(397)                                   /*!< PLLN_MUL 398 */
#define RCU_PLLN_MUL399                 PLL_PLLN(398)                                   /*!< PLLN_MUL 399 */
#define RCU_PLLN_MUL400                 PLL_PLLN(399)                                   /*!< PLLN_MUL 400 */
#define RCU_PLLN_MUL401                 PLL_PLLN(400)                                   /*!< PLLN_MUL 401 */
#define RCU_PLLN_MUL402                 PLL_PLLN(401)                                   /*!< PLLN_MUL 402 */
#define RCU_PLLN_MUL403                 PLL_PLLN(402)                                   /*!< PLLN_MUL 403 */
#define RCU_PLLN_MUL404                 PLL_PLLN(403)                                   /*!< PLLN_MUL 404 */
#define RCU_PLLN_MUL405                 PLL_PLLN(404)                                   /*!< PLLN_MUL 405 */
#define RCU_PLLN_MUL406                 PLL_PLLN(405)                                   /*!< PLLN_MUL 406 */
#define RCU_PLLN_MUL407                 PLL_PLLN(406)                                   /*!< PLLN_MUL 407 */
#define RCU_PLLN_MUL408                 PLL_PLLN(407)                                   /*!< PLLN_MUL 408 */
#define RCU_PLLN_MUL409                 PLL_PLLN(408)                                   /*!< PLLN_MUL 409 */
#define RCU_PLLN_MUL410                 PLL_PLLN(409)                                   /*!< PLLN_MUL 410 */
#define RCU_PLLN_MUL411                 PLL_PLLN(410)                                   /*!< PLLN_MUL 411 */
#define RCU_PLLN_MUL412                 PLL_PLLN(411)                                   /*!< PLLN_MUL 412 */
#define RCU_PLLN_MUL413                 PLL_PLLN(412)                                   /*!< PLLN_MUL 413 */
#define RCU_PLLN_MUL414                 PLL_PLLN(413)                                   /*!< PLLN_MUL 414 */
#define RCU_PLLN_MUL415                 PLL_PLLN(414)                                   /*!< PLLN_MUL 415 */
#define RCU_PLLN_MUL416                 PLL_PLLN(415)                                   /*!< PLLN_MUL 416 */
#define RCU_PLLN_MUL417                 PLL_PLLN(416)                                   /*!< PLLN_MUL 417 */
#define RCU_PLLN_MUL418                 PLL_PLLN(417)                                   /*!< PLLN_MUL 418 */
#define RCU_PLLN_MUL419                 PLL_PLLN(418)                                   /*!< PLLN_MUL 419 */
#define RCU_PLLN_MUL420                 PLL_PLLN(419)                                   /*!< PLLN_MUL 420 */
#define RCU_PLLN_MUL421                 PLL_PLLN(420)                                   /*!< PLLN_MUL 421 */
#define RCU_PLLN_MUL422                 PLL_PLLN(421)                                   /*!< PLLN_MUL 422 */
#define RCU_PLLN_MUL423                 PLL_PLLN(422)                                   /*!< PLLN_MUL 423 */
#define RCU_PLLN_MUL424                 PLL_PLLN(423)                                   /*!< PLLN_MUL 424 */
#define RCU_PLLN_MUL425                 PLL_PLLN(424)                                   /*!< PLLN_MUL 425 */
#define RCU_PLLN_MUL426                 PLL_PLLN(425)                                   /*!< PLLN_MUL 426 */
#define RCU_PLLN_MUL427                 PLL_PLLN(426)                                   /*!< PLLN_MUL 427 */
#define RCU_PLLN_MUL428                 PLL_PLLN(427)                                   /*!< PLLN_MUL 428 */
#define RCU_PLLN_MUL429                 PLL_PLLN(428)                                   /*!< PLLN_MUL 429 */
#define RCU_PLLN_MUL430                 PLL_PLLN(439)                                   /*!< PLLN_MUL 430 */
#define RCU_PLLN_MUL431                 PLL_PLLN(430)                                   /*!< PLLN_MUL 431 */
#define RCU_PLLN_MUL432                 PLL_PLLN(431)                                   /*!< PLLN_MUL 432 */
#define RCU_PLLN_MUL433                 PLL_PLLN(432)                                   /*!< PLLN_MUL 433 */
#define RCU_PLLN_MUL434                 PLL_PLLN(433)                                   /*!< PLLN_MUL 434 */
#define RCU_PLLN_MUL435                 PLL_PLLN(434)                                   /*!< PLLN_MUL 435 */
#define RCU_PLLN_MUL436                 PLL_PLLN(435)                                   /*!< PLLN_MUL 436 */
#define RCU_PLLN_MUL437                 PLL_PLLN(436)                                   /*!< PLLN_MUL 437 */
#define RCU_PLLN_MUL438                 PLL_PLLN(437)                                   /*!< PLLN_MUL 438 */
#define RCU_PLLN_MUL439                 PLL_PLLN(438)                                   /*!< PLLN_MUL 439 */
#define RCU_PLLN_MUL440                 PLL_PLLN(439)                                   /*!< PLLN_MUL 440 */
#define RCU_PLLN_MUL441                 PLL_PLLN(440)                                   /*!< PLLN_MUL 441 */
#define RCU_PLLN_MUL442                 PLL_PLLN(441)                                   /*!< PLLN_MUL 442 */
#define RCU_PLLN_MUL443                 PLL_PLLN(442)                                   /*!< PLLN_MUL 443 */
#define RCU_PLLN_MUL444                 PLL_PLLN(443)                                   /*!< PLLN_MUL 444 */
#define RCU_PLLN_MUL445                 PLL_PLLN(444)                                   /*!< PLLN_MUL 445 */
#define RCU_PLLN_MUL446                 PLL_PLLN(445)                                   /*!< PLLN_MUL 446 */
#define RCU_PLLN_MUL447                 PLL_PLLN(446)                                   /*!< PLLN_MUL 447 */
#define RCU_PLLN_MUL448                 PLL_PLLN(447)                                   /*!< PLLN_MUL 448 */
#define RCU_PLLN_MUL449                 PLL_PLLN(448)                                   /*!< PLLN_MUL 449 */
#define RCU_PLLN_MUL450                 PLL_PLLN(449)                                   /*!< PLLN_MUL 450 */
#define RCU_PLLN_MUL451                 PLL_PLLN(450)                                   /*!< PLLN_MUL 451 */
#define RCU_PLLN_MUL452                 PLL_PLLN(451)                                   /*!< PLLN_MUL 452 */
#define RCU_PLLN_MUL453                 PLL_PLLN(452)                                   /*!< PLLN_MUL 453 */
#define RCU_PLLN_MUL454                 PLL_PLLN(453)                                   /*!< PLLN_MUL 454 */
#define RCU_PLLN_MUL455                 PLL_PLLN(454)                                   /*!< PLLN_MUL 455 */
#define RCU_PLLN_MUL456                 PLL_PLLN(455)                                   /*!< PLLN_MUL 456 */
#define RCU_PLLN_MUL457                 PLL_PLLN(456)                                   /*!< PLLN_MUL 457 */
#define RCU_PLLN_MUL458                 PLL_PLLN(457)                                   /*!< PLLN_MUL 458 */
#define RCU_PLLN_MUL459                 PLL_PLLN(458)                                   /*!< PLLN_MUL 459 */
#define RCU_PLLN_MUL460                 PLL_PLLN(459)                                   /*!< PLLN_MUL 460 */
#define RCU_PLLN_MUL461                 PLL_PLLN(460)                                   /*!< PLLN_MUL 461 */
#define RCU_PLLN_MUL462                 PLL_PLLN(461)                                   /*!< PLLN_MUL 462 */
#define RCU_PLLN_MUL463                 PLL_PLLN(462)                                   /*!< PLLN_MUL 463 */
#define RCU_PLLN_MUL464                 PLL_PLLN(463)                                   /*!< PLLN_MUL 464 */
#define RCU_PLLN_MUL465                 PLL_PLLN(464)                                   /*!< PLLN_MUL 465 */
#define RCU_PLLN_MUL466                 PLL_PLLN(465)                                   /*!< PLLN_MUL 466 */
#define RCU_PLLN_MUL467                 PLL_PLLN(466)                                   /*!< PLLN_MUL 467 */
#define RCU_PLLN_MUL468                 PLL_PLLN(467)                                   /*!< PLLN_MUL 468 */
#define RCU_PLLN_MUL469                 PLL_PLLN(468)                                   /*!< PLLN_MUL 469 */
#define RCU_PLLN_MUL470                 PLL_PLLN(469)                                   /*!< PLLN_MUL 470 */
#define RCU_PLLN_MUL471                 PLL_PLLN(470)                                   /*!< PLLN_MUL 471 */
#define RCU_PLLN_MUL472                 PLL_PLLN(471)                                   /*!< PLLN_MUL 472 */
#define RCU_PLLN_MUL473                 PLL_PLLN(472)                                   /*!< PLLN_MUL 473 */
#define RCU_PLLN_MUL474                 PLL_PLLN(473)                                   /*!< PLLN_MUL 474 */
#define RCU_PLLN_MUL475                 PLL_PLLN(474)                                   /*!< PLLN_MUL 475 */
#define RCU_PLLN_MUL476                 PLL_PLLN(475)                                   /*!< PLLN_MUL 476 */
#define RCU_PLLN_MUL477                 PLL_PLLN(476)                                   /*!< PLLN_MUL 477 */
#define RCU_PLLN_MUL478                 PLL_PLLN(477)                                   /*!< PLLN_MUL 478 */
#define RCU_PLLN_MUL479                 PLL_PLLN(478)                                   /*!< PLLN_MUL 479 */
#define RCU_PLLN_MUL480                 PLL_PLLN(479)                                   /*!< PLLN_MUL 480 */
#define RCU_PLLN_MUL481                 PLL_PLLN(480)                                   /*!< PLLN_MUL 481 */
#define RCU_PLLN_MUL482                 PLL_PLLN(481)                                   /*!< PLLN_MUL 482 */
#define RCU_PLLN_MUL483                 PLL_PLLN(482)                                   /*!< PLLN_MUL 483 */
#define RCU_PLLN_MUL484                 PLL_PLLN(483)                                   /*!< PLLN_MUL 484 */
#define RCU_PLLN_MUL485                 PLL_PLLN(484)                                   /*!< PLLN_MUL 485 */
#define RCU_PLLN_MUL486                 PLL_PLLN(485)                                   /*!< PLLN_MUL 486 */
#define RCU_PLLN_MUL487                 PLL_PLLN(486)                                   /*!< PLLN_MUL 487 */
#define RCU_PLLN_MUL488                 PLL_PLLN(487)                                   /*!< PLLN_MUL 488 */
#define RCU_PLLN_MUL489                 PLL_PLLN(488)                                   /*!< PLLN_MUL 489 */
#define RCU_PLLN_MUL490                 PLL_PLLN(489)                                   /*!< PLLN_MUL 490 */
#define RCU_PLLN_MUL491                 PLL_PLLN(490)                                   /*!< PLLN_MUL 491 */
#define RCU_PLLN_MUL492                 PLL_PLLN(491)                                   /*!< PLLN_MUL 492 */
#define RCU_PLLN_MUL493                 PLL_PLLN(492)                                   /*!< PLLN_MUL 493 */
#define RCU_PLLN_MUL494                 PLL_PLLN(493)                                   /*!< PLLN_MUL 494 */
#define RCU_PLLN_MUL495                 PLL_PLLN(494)                                   /*!< PLLN_MUL 495 */
#define RCU_PLLN_MUL496                 PLL_PLLN(495)                                   /*!< PLLN_MUL 496 */
#define RCU_PLLN_MUL497                 PLL_PLLN(496)                                   /*!< PLLN_MUL 497 */
#define RCU_PLLN_MUL498                 PLL_PLLN(497)                                   /*!< PLLN_MUL 498 */
#define RCU_PLLN_MUL499                 PLL_PLLN(498)                                   /*!< PLLN_MUL 499 */
#define RCU_PLLN_MUL500                 PLL_PLLN(499)                                   /*!< PLLN_MUL 500 */
#define RCU_PLLN_MUL501                 PLL_PLLN(500)                                   /*!< PLLN_MUL 501 */
#define RCU_PLLN_MUL502                 PLL_PLLN(501)                                   /*!< PLLN_MUL 502 */
#define RCU_PLLN_MUL503                 PLL_PLLN(502)                                   /*!< PLLN_MUL 503 */
#define RCU_PLLN_MUL504                 PLL_PLLN(503)                                   /*!< PLLN_MUL 504 */
#define RCU_PLLN_MUL505                 PLL_PLLN(504)                                   /*!< PLLN_MUL 505 */
#define RCU_PLLN_MUL506                 PLL_PLLN(505)                                   /*!< PLLN_MUL 506 */
#define RCU_PLLN_MUL507                 PLL_PLLN(506)                                   /*!< PLLN_MUL 507 */
#define RCU_PLLN_MUL508                 PLL_PLLN(507)                                   /*!< PLLN_MUL 508 */
#define RCU_PLLN_MUL509                 PLL_PLLN(508)                                   /*!< PLLN_MUL 509 */
#define RCU_PLLN_MUL510                 PLL_PLLN(509)                                   /*!< PLLN_MUL 510 */
#define RCU_PLLN_MUL511                 PLL_PLLN(510)                                   /*!< PLLN_MUL 511 */
#define RCU_PLLN_MUL512                 PLL_PLLN(511)                                   /*!< PLLN_MUL 512 */

/* The PLLP(PLL0\PLL1\PLL2) output frequency division factor from PLL0 VCO clock */
#define PLL_PLLP(regval)                (BITS(16,22) & ((uint32_t)(regval) << 16U))
#define RCU_PLLP_DIV1                   PLL_PLLP(0)                                     /*!< PLLP_DIV 1 */
#define RCU_PLLP_DIV2                   PLL_PLLP(1)                                     /*!< PLLP_DIV 2 */
#define RCU_PLLP_DIV3                   PLL_PLLP(2)                                     /*!< PLLP_DIV 3 */
#define RCU_PLLP_DIV4                   PLL_PLLP(3)                                     /*!< PLLP_DIV 4 */
#define RCU_PLLP_DIV5                   PLL_PLLP(4)                                     /*!< PLLP_DIV 5 */
#define RCU_PLLP_DIV6                   PLL_PLLP(5)                                     /*!< PLLP_DIV 6 */
#define RCU_PLLP_DIV7                   PLL_PLLP(6)                                     /*!< PLLP_DIV 7 */
#define RCU_PLLP_DIV8                   PLL_PLLP(7)                                     /*!< PLLP_DIV 8 */
#define RCU_PLLP_DIV9                   PLL_PLLP(8)                                     /*!< PLLP_DIV 9 */
#define RCU_PLLP_DIV10                  PLL_PLLP(9)                                     /*!< PLLP_DIV 10 */
#define RCU_PLLP_DIV11                  PLL_PLLP(10)                                    /*!< PLLP_DIV 11 */
#define RCU_PLLP_DIV12                  PLL_PLLP(11)                                    /*!< PLLP_DIV 12 */
#define RCU_PLLP_DIV13                  PLL_PLLP(12)                                    /*!< PLLP_DIV 13 */
#define RCU_PLLP_DIV14                  PLL_PLLP(13)                                    /*!< PLLP_DIV 14 */
#define RCU_PLLP_DIV15                  PLL_PLLP(14)                                    /*!< PLLP_DIV 15 */
#define RCU_PLLP_DIV16                  PLL_PLLP(15)                                    /*!< PLLP_DIV 16 */
#define RCU_PLLP_DIV17                  PLL_PLLP(16)                                    /*!< PLLP_DIV 17 */
#define RCU_PLLP_DIV18                  PLL_PLLP(17)                                    /*!< PLLP_DIV 18 */
#define RCU_PLLP_DIV19                  PLL_PLLP(18)                                    /*!< PLLP_DIV 19 */
#define RCU_PLLP_DIV20                  PLL_PLLP(19)                                    /*!< PLLP_DIV 20 */
#define RCU_PLLP_DIV21                  PLL_PLLP(20)                                    /*!< PLLP_DIV 21 */
#define RCU_PLLP_DIV22                  PLL_PLLP(21)                                    /*!< PLLP_DIV 22 */
#define RCU_PLLP_DIV23                  PLL_PLLP(22)                                    /*!< PLLP_DIV 23 */
#define RCU_PLLP_DIV24                  PLL_PLLP(23)                                    /*!< PLLP_DIV 24 */
#define RCU_PLLP_DIV25                  PLL_PLLP(24)                                    /*!< PLLP_DIV 25 */
#define RCU_PLLP_DIV26                  PLL_PLLP(25)                                    /*!< PLLP_DIV 26 */
#define RCU_PLLP_DIV27                  PLL_PLLP(26)                                    /*!< PLLP_DIV 27 */
#define RCU_PLLP_DIV28                  PLL_PLLP(27)                                    /*!< PLLP_DIV 28 */
#define RCU_PLLP_DIV29                  PLL_PLLP(28)                                    /*!< PLLP_DIV 29 */
#define RCU_PLLP_DIV30                  PLL_PLLP(29)                                    /*!< PLLP_DIV 30 */
#define RCU_PLLP_DIV31                  PLL_PLLP(30)                                    /*!< PLLP_DIV 31 */
#define RCU_PLLP_DIV32                  PLL_PLLP(31)                                    /*!< PLLP_DIV 32 */
#define RCU_PLLP_DIV33                  PLL_PLLP(32)                                    /*!< PLLP_DIV 33 */
#define RCU_PLLP_DIV34                  PLL_PLLP(33)                                    /*!< PLLP_DIV 34 */
#define RCU_PLLP_DIV35                  PLL_PLLP(34)                                    /*!< PLLP_DIV 35 */
#define RCU_PLLP_DIV36                  PLL_PLLP(35)                                    /*!< PLLP_DIV 36 */
#define RCU_PLLP_DIV37                  PLL_PLLP(36)                                    /*!< PLLP_DIV 37 */
#define RCU_PLLP_DIV38                  PLL_PLLP(37)                                    /*!< PLLP_DIV 38 */
#define RCU_PLLP_DIV39                  PLL_PLLP(38)                                    /*!< PLLP_DIV 39 */
#define RCU_PLLP_DIV40                  PLL_PLLP(39)                                    /*!< PLLP_DIV 40 */
#define RCU_PLLP_DIV41                  PLL_PLLP(40)                                    /*!< PLLP_DIV 41 */
#define RCU_PLLP_DIV42                  PLL_PLLP(41)                                    /*!< PLLP_DIV 42 */
#define RCU_PLLP_DIV43                  PLL_PLLP(42)                                    /*!< PLLP_DIV 43 */
#define RCU_PLLP_DIV44                  PLL_PLLP(43)                                    /*!< PLLP_DIV 44 */
#define RCU_PLLP_DIV45                  PLL_PLLP(44)                                    /*!< PLLP_DIV 45 */
#define RCU_PLLP_DIV46                  PLL_PLLP(45)                                    /*!< PLLP_DIV 46 */
#define RCU_PLLP_DIV47                  PLL_PLLP(46)                                    /*!< PLLP_DIV 47 */
#define RCU_PLLP_DIV48                  PLL_PLLP(47)                                    /*!< PLLP_DIV 48 */
#define RCU_PLLP_DIV49                  PLL_PLLP(48)                                    /*!< PLLP_DIV 49 */
#define RCU_PLLP_DIV50                  PLL_PLLP(49)                                    /*!< PLLP_DIV 50 */
#define RCU_PLLP_DIV51                  PLL_PLLP(50)                                    /*!< PLLP_DIV 51 */
#define RCU_PLLP_DIV52                  PLL_PLLP(51)                                    /*!< PLLP_DIV 52 */
#define RCU_PLLP_DIV53                  PLL_PLLP(52)                                    /*!< PLLP_DIV 53 */
#define RCU_PLLP_DIV54                  PLL_PLLP(53)                                    /*!< PLLP_DIV 54 */
#define RCU_PLLP_DIV55                  PLL_PLLP(54)                                    /*!< PLLP_DIV 55 */
#define RCU_PLLP_DIV56                  PLL_PLLP(55)                                    /*!< PLLP_DIV 56 */
#define RCU_PLLP_DIV57                  PLL_PLLP(56)                                    /*!< PLLP_DIV 57 */
#define RCU_PLLP_DIV58                  PLL_PLLP(57)                                    /*!< PLLP_DIV 58 */
#define RCU_PLLP_DIV59                  PLL_PLLP(58)                                    /*!< PLLP_DIV 59 */
#define RCU_PLLP_DIV60                  PLL_PLLP(59)                                    /*!< PLLP_DIV 60 */
#define RCU_PLLP_DIV61                  PLL_PLLP(60)                                    /*!< PLLP_DIV 61 */
#define RCU_PLLP_DIV62                  PLL_PLLP(61)                                    /*!< PLLP_DIV 62 */
#define RCU_PLLP_DIV63                  PLL_PLLP(62)                                    /*!< PLLP_DIV 63 */
#define RCU_PLLP_DIV64                  PLL_PLLP(63)                                    /*!< PLLP_DIV 64 */
#define RCU_PLLP_DIV65                  PLL_PLLP(64)                                    /*!< PLLP_DIV 65 */
#define RCU_PLLP_DIV66                  PLL_PLLP(65)                                    /*!< PLLP_DIV 66 */
#define RCU_PLLP_DIV67                  PLL_PLLP(66)                                    /*!< PLLP_DIV 67 */
#define RCU_PLLP_DIV68                  PLL_PLLP(67)                                    /*!< PLLP_DIV 68 */
#define RCU_PLLP_DIV69                  PLL_PLLP(68)                                    /*!< PLLP_DIV 69 */
#define RCU_PLLP_DIV70                  PLL_PLLP(69)                                    /*!< PLLP_DIV 70 */
#define RCU_PLLP_DIV71                  PLL_PLLP(70)                                    /*!< PLLP_DIV 71 */
#define RCU_PLLP_DIV72                  PLL_PLLP(71)                                    /*!< PLLP_DIV 72 */
#define RCU_PLLP_DIV73                  PLL_PLLP(72)                                    /*!< PLLP_DIV 73 */
#define RCU_PLLP_DIV74                  PLL_PLLP(73)                                    /*!< PLLP_DIV 74 */
#define RCU_PLLP_DIV75                  PLL_PLLP(74)                                    /*!< PLLP_DIV 75 */
#define RCU_PLLP_DIV76                  PLL_PLLP(75)                                    /*!< PLLP_DIV 76 */
#define RCU_PLLP_DIV77                  PLL_PLLP(76)                                    /*!< PLLP_DIV 77 */
#define RCU_PLLP_DIV78                  PLL_PLLP(77)                                    /*!< PLLP_DIV 78 */
#define RCU_PLLP_DIV79                  PLL_PLLP(78)                                    /*!< PLLP_DIV 79 */
#define RCU_PLLP_DIV80                  PLL_PLLP(79)                                    /*!< PLLP_DIV 80 */
#define RCU_PLLP_DIV81                  PLL_PLLP(80)                                    /*!< PLLP_DIV 81 */
#define RCU_PLLP_DIV82                  PLL_PLLP(81)                                    /*!< PLLP_DIV 82 */
#define RCU_PLLP_DIV83                  PLL_PLLP(82)                                    /*!< PLLP_DIV 83 */
#define RCU_PLLP_DIV84                  PLL_PLLP(83)                                    /*!< PLLP_DIV 84 */
#define RCU_PLLP_DIV85                  PLL_PLLP(84)                                    /*!< PLLP_DIV 85 */
#define RCU_PLLP_DIV86                  PLL_PLLP(85)                                    /*!< PLLP_DIV 86 */
#define RCU_PLLP_DIV87                  PLL_PLLP(86)                                    /*!< PLLP_DIV 87 */
#define RCU_PLLP_DIV88                  PLL_PLLP(87)                                    /*!< PLLP_DIV 88 */
#define RCU_PLLP_DIV89                  PLL_PLLP(88)                                    /*!< PLLP_DIV 89 */
#define RCU_PLLP_DIV90                  PLL_PLLP(89)                                    /*!< PLLP_DIV 90 */
#define RCU_PLLP_DIV91                  PLL_PLLP(90)                                    /*!< PLLP_DIV 91 */
#define RCU_PLLP_DIV92                  PLL_PLLP(91)                                    /*!< PLLP_DIV 92 */
#define RCU_PLLP_DIV93                  PLL_PLLP(92)                                    /*!< PLLP_DIV 93 */
#define RCU_PLLP_DIV94                  PLL_PLLP(93)                                    /*!< PLLP_DIV 94 */
#define RCU_PLLP_DIV95                  PLL_PLLP(94)                                    /*!< PLLP_DIV 95 */
#define RCU_PLLP_DIV96                  PLL_PLLP(95)                                    /*!< PLLP_DIV 96 */
#define RCU_PLLP_DIV97                  PLL_PLLP(96)                                    /*!< PLLP_DIV 97 */
#define RCU_PLLP_DIV98                  PLL_PLLP(97)                                    /*!< PLLP_DIV 98 */
#define RCU_PLLP_DIV99                  PLL_PLLP(98)                                    /*!< PLLP_DIV 99 */
#define RCU_PLLP_DIV100                 PLL_PLLP(99)                                    /*!< PLLP_DIV 100 */
#define RCU_PLLP_DIV101                 PLL_PLLP(100)                                   /*!< PLLP_DIV 101 */
#define RCU_PLLP_DIV102                 PLL_PLLP(101)                                   /*!< PLLP_DIV 102 */
#define RCU_PLLP_DIV103                 PLL_PLLP(102)                                   /*!< PLLP_DIV 103 */
#define RCU_PLLP_DIV104                 PLL_PLLP(103)                                   /*!< PLLP_DIV 104 */
#define RCU_PLLP_DIV105                 PLL_PLLP(104)                                   /*!< PLLP_DIV 105 */
#define RCU_PLLP_DIV106                 PLL_PLLP(105)                                   /*!< PLLP_DIV 106 */
#define RCU_PLLP_DIV107                 PLL_PLLP(106)                                   /*!< PLLP_DIV 107 */
#define RCU_PLLP_DIV108                 PLL_PLLP(107)                                   /*!< PLLP_DIV 108 */
#define RCU_PLLP_DIV109                 PLL_PLLP(108)                                   /*!< PLLP_DIV 109 */
#define RCU_PLLP_DIV110                 PLL_PLLP(109)                                   /*!< PLLP_DIV 110 */
#define RCU_PLLP_DIV111                 PLL_PLLP(110)                                   /*!< PLLP_DIV 111 */
#define RCU_PLLP_DIV112                 PLL_PLLP(111)                                   /*!< PLLP_DIV 112 */
#define RCU_PLLP_DIV113                 PLL_PLLP(112)                                   /*!< PLLP_DIV 113 */
#define RCU_PLLP_DIV114                 PLL_PLLP(113)                                   /*!< PLLP_DIV 114 */
#define RCU_PLLP_DIV115                 PLL_PLLP(114)                                   /*!< PLLP_DIV 115 */
#define RCU_PLLP_DIV116                 PLL_PLLP(115)                                   /*!< PLLP_DIV 116 */
#define RCU_PLLP_DIV117                 PLL_PLLP(116)                                   /*!< PLLP_DIV 117 */
#define RCU_PLLP_DIV118                 PLL_PLLP(117)                                   /*!< PLLP_DIV 118 */
#define RCU_PLLP_DIV119                 PLL_PLLP(118)                                   /*!< PLLP_DIV 119 */
#define RCU_PLLP_DIV120                 PLL_PLLP(119)                                   /*!< PLLP_DIV 120 */
#define RCU_PLLP_DIV121                 PLL_PLLP(120)                                   /*!< PLLP_DIV 121 */
#define RCU_PLLP_DIV122                 PLL_PLLP(121)                                   /*!< PLLP_DIV 122 */
#define RCU_PLLP_DIV123                 PLL_PLLP(122)                                   /*!< PLLP_DIV 123 */
#define RCU_PLLP_DIV124                 PLL_PLLP(123)                                   /*!< PLLP_DIV 124 */
#define RCU_PLLP_DIV125                 PLL_PLLP(124)                                   /*!< PLLP_DIV 125 */
#define RCU_PLLP_DIV126                 PLL_PLLP(125)                                   /*!< PLLP_DIV 126 */
#define RCU_PLLP_DIV127                 PLL_PLLP(126)                                   /*!< PLLP_DIV 127 */
#define RCU_PLLP_DIV128                 PLL_PLLP(127)                                   /*!< PLLP_DIV 128 */

/* The PLLR(PLL0\PLL1\PLL2) output frequency division factor from PLL0 VCO clock */
#define PLL_PLLR(regval)                (BITS(24,30) & ((uint32_t)(regval) << 24U))
#define RCU_PLLR_DIV1                   PLL_PLLR(0)                                     /*!< PLLR_DIV 1 */
#define RCU_PLLR_DIV2                   PLL_PLLR(1)                                     /*!< PLLR_DIV 2 */
#define RCU_PLLR_DIV3                   PLL_PLLR(2)                                     /*!< PLLR_DIV 3 */
#define RCU_PLLR_DIV4                   PLL_PLLR(3)                                     /*!< PLLR_DIV 4 */
#define RCU_PLLR_DIV5                   PLL_PLLR(4)                                     /*!< PLLR_DIV 5 */
#define RCU_PLLR_DIV6                   PLL_PLLR(5)                                     /*!< PLLR_DIV 6 */
#define RCU_PLLR_DIV7                   PLL_PLLR(6)                                     /*!< PLLR_DIV 7 */
#define RCU_PLLR_DIV8                   PLL_PLLR(7)                                     /*!< PLLR_DIV 8 */
#define RCU_PLLR_DIV9                   PLL_PLLR(8)                                     /*!< PLLR_DIV 9 */
#define RCU_PLLR_DIV10                  PLL_PLLR(9)                                    /*!< PLLR_DIV 10 */
#define RCU_PLLR_DIV11                  PLL_PLLR(10)                                    /*!< PLLR_DIV 11 */
#define RCU_PLLR_DIV12                  PLL_PLLR(11)                                    /*!< PLLR_DIV 12 */
#define RCU_PLLR_DIV13                  PLL_PLLR(12)                                    /*!< PLLR_DIV 13 */
#define RCU_PLLR_DIV14                  PLL_PLLR(13)                                    /*!< PLLR_DIV 14 */
#define RCU_PLLR_DIV15                  PLL_PLLR(14)                                    /*!< PLLR_DIV 15 */
#define RCU_PLLR_DIV16                  PLL_PLLR(15)                                    /*!< PLLR_DIV 16 */
#define RCU_PLLR_DIV17                  PLL_PLLR(16)                                    /*!< PLLR_DIV 17 */
#define RCU_PLLR_DIV18                  PLL_PLLR(17)                                    /*!< PLLR_DIV 18 */
#define RCU_PLLR_DIV19                  PLL_PLLR(18)                                    /*!< PLLR_DIV 19 */
#define RCU_PLLR_DIV20                  PLL_PLLR(19)                                    /*!< PLLR_DIV 20 */
#define RCU_PLLR_DIV21                  PLL_PLLR(20)                                    /*!< PLLR_DIV 21 */
#define RCU_PLLR_DIV22                  PLL_PLLR(21)                                    /*!< PLLR_DIV 22 */
#define RCU_PLLR_DIV23                  PLL_PLLR(22)                                    /*!< PLLR_DIV 23 */
#define RCU_PLLR_DIV24                  PLL_PLLR(23)                                    /*!< PLLR_DIV 24 */
#define RCU_PLLR_DIV25                  PLL_PLLR(24)                                    /*!< PLLR_DIV 25 */
#define RCU_PLLR_DIV26                  PLL_PLLR(25)                                    /*!< PLLR_DIV 26 */
#define RCU_PLLR_DIV27                  PLL_PLLR(26)                                    /*!< PLLR_DIV 27 */
#define RCU_PLLR_DIV28                  PLL_PLLR(27)                                    /*!< PLLR_DIV 28 */
#define RCU_PLLR_DIV29                  PLL_PLLR(28)                                    /*!< PLLR_DIV 29 */
#define RCU_PLLR_DIV30                  PLL_PLLR(29)                                    /*!< PLLR_DIV 30 */
#define RCU_PLLR_DIV31                  PLL_PLLR(30)                                    /*!< PLLR_DIV 31 */
#define RCU_PLLR_DIV32                  PLL_PLLR(31)                                    /*!< PLLR_DIV 32 */
#define RCU_PLLR_DIV33                  PLL_PLLR(32)                                    /*!< PLLR_DIV 33 */
#define RCU_PLLR_DIV34                  PLL_PLLR(33)                                    /*!< PLLR_DIV 34 */
#define RCU_PLLR_DIV35                  PLL_PLLR(34)                                    /*!< PLLR_DIV 35 */
#define RCU_PLLR_DIV36                  PLL_PLLR(35)                                    /*!< PLLR_DIV 36 */
#define RCU_PLLR_DIV37                  PLL_PLLR(36)                                    /*!< PLLR_DIV 37 */
#define RCU_PLLR_DIV38                  PLL_PLLR(37)                                    /*!< PLLR_DIV 38 */
#define RCU_PLLR_DIV39                  PLL_PLLR(38)                                    /*!< PLLR_DIV 39 */
#define RCU_PLLR_DIV40                  PLL_PLLR(39)                                    /*!< PLLR_DIV 40 */
#define RCU_PLLR_DIV41                  PLL_PLLR(40)                                    /*!< PLLR_DIV 41 */
#define RCU_PLLR_DIV42                  PLL_PLLR(41)                                    /*!< PLLR_DIV 42 */
#define RCU_PLLR_DIV43                  PLL_PLLR(42)                                    /*!< PLLR_DIV 43 */
#define RCU_PLLR_DIV44                  PLL_PLLR(43)                                    /*!< PLLR_DIV 44 */
#define RCU_PLLR_DIV45                  PLL_PLLR(44)                                    /*!< PLLR_DIV 45 */
#define RCU_PLLR_DIV46                  PLL_PLLR(45)                                    /*!< PLLR_DIV 46 */
#define RCU_PLLR_DIV47                  PLL_PLLR(46)                                    /*!< PLLR_DIV 47 */
#define RCU_PLLR_DIV48                  PLL_PLLR(47)                                    /*!< PLLR_DIV 48 */
#define RCU_PLLR_DIV49                  PLL_PLLR(48)                                    /*!< PLLR_DIV 49 */
#define RCU_PLLR_DIV50                  PLL_PLLR(49)                                    /*!< PLLR_DIV 50 */
#define RCU_PLLR_DIV51                  PLL_PLLR(50)                                    /*!< PLLR_DIV 51 */
#define RCU_PLLR_DIV52                  PLL_PLLR(51)                                    /*!< PLLR_DIV 52 */
#define RCU_PLLR_DIV53                  PLL_PLLR(52)                                    /*!< PLLR_DIV 53 */
#define RCU_PLLR_DIV54                  PLL_PLLR(53)                                    /*!< PLLR_DIV 54 */
#define RCU_PLLR_DIV55                  PLL_PLLR(54)                                    /*!< PLLR_DIV 55 */
#define RCU_PLLR_DIV56                  PLL_PLLR(55)                                    /*!< PLLR_DIV 56 */
#define RCU_PLLR_DIV57                  PLL_PLLR(56)                                    /*!< PLLR_DIV 57 */
#define RCU_PLLR_DIV58                  PLL_PLLR(57)                                    /*!< PLLR_DIV 58 */
#define RCU_PLLR_DIV59                  PLL_PLLR(58)                                    /*!< PLLR_DIV 59 */
#define RCU_PLLR_DIV60                  PLL_PLLR(59)                                    /*!< PLLR_DIV 60 */
#define RCU_PLLR_DIV61                  PLL_PLLR(60)                                    /*!< PLLR_DIV 61 */
#define RCU_PLLR_DIV62                  PLL_PLLR(61)                                    /*!< PLLR_DIV 62 */
#define RCU_PLLR_DIV63                  PLL_PLLR(62)                                    /*!< PLLR_DIV 63 */
#define RCU_PLLR_DIV64                  PLL_PLLR(63)                                    /*!< PLLR_DIV 64 */
#define RCU_PLLR_DIV65                  PLL_PLLR(64)                                    /*!< PLLR_DIV 65 */
#define RCU_PLLR_DIV66                  PLL_PLLR(65)                                    /*!< PLLR_DIV 66 */
#define RCU_PLLR_DIV67                  PLL_PLLR(66)                                    /*!< PLLR_DIV 67 */
#define RCU_PLLR_DIV68                  PLL_PLLR(67)                                    /*!< PLLR_DIV 68 */
#define RCU_PLLR_DIV69                  PLL_PLLR(68)                                    /*!< PLLR_DIV 69 */
#define RCU_PLLR_DIV70                  PLL_PLLR(69)                                    /*!< PLLR_DIV 70 */
#define RCU_PLLR_DIV71                  PLL_PLLR(70)                                    /*!< PLLR_DIV 71 */
#define RCU_PLLR_DIV72                  PLL_PLLR(71)                                    /*!< PLLR_DIV 72 */
#define RCU_PLLR_DIV73                  PLL_PLLR(72)                                    /*!< PLLR_DIV 73 */
#define RCU_PLLR_DIV74                  PLL_PLLR(73)                                    /*!< PLLR_DIV 74 */
#define RCU_PLLR_DIV75                  PLL_PLLR(74)                                    /*!< PLLR_DIV 75 */
#define RCU_PLLR_DIV76                  PLL_PLLR(75)                                    /*!< PLLR_DIV 76 */
#define RCU_PLLR_DIV77                  PLL_PLLR(76)                                    /*!< PLLR_DIV 77 */
#define RCU_PLLR_DIV78                  PLL_PLLR(77)                                    /*!< PLLR_DIV 78 */
#define RCU_PLLR_DIV79                  PLL_PLLR(78)                                    /*!< PLLR_DIV 79 */
#define RCU_PLLR_DIV80                  PLL_PLLR(79)                                    /*!< PLLR_DIV 80 */
#define RCU_PLLR_DIV81                  PLL_PLLR(80)                                    /*!< PLLR_DIV 81 */
#define RCU_PLLR_DIV82                  PLL_PLLR(81)                                    /*!< PLLR_DIV 82 */
#define RCU_PLLR_DIV83                  PLL_PLLR(82)                                    /*!< PLLR_DIV 83 */
#define RCU_PLLR_DIV84                  PLL_PLLR(83)                                    /*!< PLLR_DIV 84 */
#define RCU_PLLR_DIV85                  PLL_PLLR(84)                                    /*!< PLLR_DIV 85 */
#define RCU_PLLR_DIV86                  PLL_PLLR(85)                                    /*!< PLLR_DIV 86 */
#define RCU_PLLR_DIV87                  PLL_PLLR(86)                                    /*!< PLLR_DIV 87 */
#define RCU_PLLR_DIV88                  PLL_PLLR(87)                                    /*!< PLLR_DIV 88 */
#define RCU_PLLR_DIV89                  PLL_PLLR(88)                                    /*!< PLLR_DIV 89 */
#define RCU_PLLR_DIV90                  PLL_PLLR(89)                                    /*!< PLLR_DIV 90 */
#define RCU_PLLR_DIV91                  PLL_PLLR(90)                                    /*!< PLLR_DIV 91 */
#define RCU_PLLR_DIV92                  PLL_PLLR(91)                                    /*!< PLLR_DIV 92 */
#define RCU_PLLR_DIV93                  PLL_PLLR(92)                                    /*!< PLLR_DIV 93 */
#define RCU_PLLR_DIV94                  PLL_PLLR(93)                                    /*!< PLLR_DIV 94 */
#define RCU_PLLR_DIV95                  PLL_PLLR(94)                                    /*!< PLLR_DIV 95 */
#define RCU_PLLR_DIV96                  PLL_PLLR(95)                                    /*!< PLLR_DIV 96 */
#define RCU_PLLR_DIV97                  PLL_PLLR(96)                                    /*!< PLLR_DIV 97 */
#define RCU_PLLR_DIV98                  PLL_PLLR(97)                                    /*!< PLLR_DIV 98 */
#define RCU_PLLR_DIV99                  PLL_PLLR(98)                                    /*!< PLLR_DIV 99 */
#define RCU_PLLR_DIV100                 PLL_PLLR(99)                                    /*!< PLLR_DIV 100 */
#define RCU_PLLR_DIV101                 PLL_PLLR(100)                                   /*!< PLLR_DIV 101 */
#define RCU_PLLR_DIV102                 PLL_PLLR(101)                                   /*!< PLLR_DIV 102 */
#define RCU_PLLR_DIV103                 PLL_PLLR(102)                                   /*!< PLLR_DIV 103 */
#define RCU_PLLR_DIV104                 PLL_PLLR(103)                                   /*!< PLLR_DIV 104 */
#define RCU_PLLR_DIV105                 PLL_PLLR(104)                                   /*!< PLLR_DIV 105 */
#define RCU_PLLR_DIV106                 PLL_PLLR(105)                                   /*!< PLLR_DIV 106 */
#define RCU_PLLR_DIV107                 PLL_PLLR(106)                                   /*!< PLLR_DIV 107 */
#define RCU_PLLR_DIV108                 PLL_PLLR(107)                                   /*!< PLLR_DIV 108 */
#define RCU_PLLR_DIV109                 PLL_PLLR(108)                                   /*!< PLLR_DIV 109 */
#define RCU_PLLR_DIV110                 PLL_PLLR(109)                                   /*!< PLLR_DIV 110 */
#define RCU_PLLR_DIV111                 PLL_PLLR(110)                                   /*!< PLLR_DIV 111 */
#define RCU_PLLR_DIV112                 PLL_PLLR(111)                                   /*!< PLLR_DIV 112 */
#define RCU_PLLR_DIV113                 PLL_PLLR(112)                                   /*!< PLLR_DIV 113 */
#define RCU_PLLR_DIV114                 PLL_PLLR(113)                                   /*!< PLLR_DIV 114 */
#define RCU_PLLR_DIV115                 PLL_PLLR(114)                                   /*!< PLLR_DIV 115 */
#define RCU_PLLR_DIV116                 PLL_PLLR(115)                                   /*!< PLLR_DIV 116 */
#define RCU_PLLR_DIV117                 PLL_PLLR(116)                                   /*!< PLLR_DIV 117 */
#define RCU_PLLR_DIV118                 PLL_PLLR(117)                                   /*!< PLLR_DIV 118 */
#define RCU_PLLR_DIV119                 PLL_PLLR(118)                                   /*!< PLLR_DIV 119 */
#define RCU_PLLR_DIV120                 PLL_PLLR(119)                                   /*!< PLLR_DIV 120 */
#define RCU_PLLR_DIV121                 PLL_PLLR(120)                                   /*!< PLLR_DIV 121 */
#define RCU_PLLR_DIV122                 PLL_PLLR(121)                                   /*!< PLLR_DIV 122 */
#define RCU_PLLR_DIV123                 PLL_PLLR(122)                                   /*!< PLLR_DIV 123 */
#define RCU_PLLR_DIV124                 PLL_PLLR(123)                                   /*!< PLLR_DIV 124 */
#define RCU_PLLR_DIV125                 PLL_PLLR(124)                                   /*!< PLLR_DIV 125 */
#define RCU_PLLR_DIV126                 PLL_PLLR(125)                                   /*!< PLLR_DIV 126 */
#define RCU_PLLR_DIV127                 PLL_PLLR(126)                                   /*!< PLLR_DIV 127 */
#define RCU_PLLR_DIV128                 PLL_PLLR(127)                                   /*!< PLLR_DIV 128 */

/* The PLL0Q(PLL0\PLL1\PLL2) output frequency division factor from PLL0 VCO clock */
#define PLL_PLLQ(regval)                (BITS(0,6) & ((uint32_t)(regval) << 0U))
#define RCU_PLLQ_DIV1                   PLL_PLLQ(0)                                     /*!< PLL_PLLQ 1 */
#define RCU_PLLQ_DIV2                   PLL_PLLQ(1)                                     /*!< PLL_PLLQ 2 */
#define RCU_PLLQ_DIV3                   PLL_PLLQ(2)                                     /*!< PLL_PLLQ 3 */
#define RCU_PLLQ_DIV4                   PLL_PLLQ(3)                                     /*!< PLL_PLLQ 4 */
#define RCU_PLLQ_DIV5                   PLL_PLLQ(4)                                     /*!< PLL_PLLQ 5 */
#define RCU_PLLQ_DIV6                   PLL_PLLQ(5)                                     /*!< PLL_PLLQ 6 */
#define RCU_PLLQ_DIV7                   PLL_PLLQ(6)                                     /*!< PLL_PLLQ 7 */
#define RCU_PLLQ_DIV8                   PLL_PLLQ(7)                                     /*!< PLL_PLLQ 8 */
#define RCU_PLLQ_DIV9                   PLL_PLLQ(8)                                     /*!< PLL_PLLQ 9 */
#define RCU_PLLQ_DIV10                  PLL_PLLQ(9)                                    /*!< PLL_PLLQ 10 */
#define RCU_PLLQ_DIV11                  PLL_PLLQ(10)                                    /*!< PLL_PLLQ 11 */
#define RCU_PLLQ_DIV12                  PLL_PLLQ(11)                                    /*!< PLL_PLLQ 12 */
#define RCU_PLLQ_DIV13                  PLL_PLLQ(12)                                    /*!< PLL_PLLQ 13 */
#define RCU_PLLQ_DIV14                  PLL_PLLQ(13)                                    /*!< PLL_PLLQ 14 */
#define RCU_PLLQ_DIV15                  PLL_PLLQ(14)                                    /*!< PLL_PLLQ 15 */
#define RCU_PLLQ_DIV16                  PLL_PLLQ(15)                                    /*!< PLL_PLLQ 16 */
#define RCU_PLLQ_DIV17                  PLL_PLLQ(16)                                    /*!< PLL_PLLQ 17 */
#define RCU_PLLQ_DIV18                  PLL_PLLQ(17)                                    /*!< PLL_PLLQ 18 */
#define RCU_PLLQ_DIV19                  PLL_PLLQ(18)                                    /*!< PLL_PLLQ 19 */
#define RCU_PLLQ_DIV20                  PLL_PLLQ(19)                                    /*!< PLL_PLLQ 20 */
#define RCU_PLLQ_DIV21                  PLL_PLLQ(20)                                    /*!< PLL_PLLQ 21 */
#define RCU_PLLQ_DIV22                  PLL_PLLQ(21)                                    /*!< PLL_PLLQ 22 */
#define RCU_PLLQ_DIV23                  PLL_PLLQ(22)                                    /*!< PLL_PLLQ 23 */
#define RCU_PLLQ_DIV24                  PLL_PLLQ(23)                                    /*!< PLL_PLLQ 24 */
#define RCU_PLLQ_DIV25                  PLL_PLLQ(24)                                    /*!< PLL_PLLQ 25 */
#define RCU_PLLQ_DIV26                  PLL_PLLQ(25)                                    /*!< PLL_PLLQ 26 */
#define RCU_PLLQ_DIV27                  PLL_PLLQ(26)                                    /*!< PLL_PLLQ 27 */
#define RCU_PLLQ_DIV28                  PLL_PLLQ(27)                                    /*!< PLL_PLLQ 28 */
#define RCU_PLLQ_DIV29                  PLL_PLLQ(28)                                    /*!< PLL_PLLQ 29 */
#define RCU_PLLQ_DIV30                  PLL_PLLQ(29)                                    /*!< PLL_PLLQ 30 */
#define RCU_PLLQ_DIV31                  PLL_PLLQ(30)                                    /*!< PLL_PLLQ 31 */
#define RCU_PLLQ_DIV32                  PLL_PLLQ(31)                                    /*!< PLL_PLLQ 32 */
#define RCU_PLLQ_DIV33                  PLL_PLLQ(32)                                    /*!< PLL_PLLQ 33 */
#define RCU_PLLQ_DIV34                  PLL_PLLQ(33)                                    /*!< PLL_PLLQ 34 */
#define RCU_PLLQ_DIV35                  PLL_PLLQ(34)                                    /*!< PLL_PLLQ 35 */
#define RCU_PLLQ_DIV36                  PLL_PLLQ(35)                                    /*!< PLL_PLLQ 36 */
#define RCU_PLLQ_DIV37                  PLL_PLLQ(36)                                    /*!< PLL_PLLQ 37 */
#define RCU_PLLQ_DIV38                  PLL_PLLQ(37)                                    /*!< PLL_PLLQ 38 */
#define RCU_PLLQ_DIV39                  PLL_PLLQ(38)                                    /*!< PLL_PLLQ 39 */
#define RCU_PLLQ_DIV40                  PLL_PLLQ(39)                                    /*!< PLL_PLLQ 40 */
#define RCU_PLLQ_DIV41                  PLL_PLLQ(40)                                    /*!< PLL_PLLQ 41 */
#define RCU_PLLQ_DIV42                  PLL_PLLQ(41)                                    /*!< PLL_PLLQ 42 */
#define RCU_PLLQ_DIV43                  PLL_PLLQ(42)                                    /*!< PLL_PLLQ 43 */
#define RCU_PLLQ_DIV44                  PLL_PLLQ(43)                                    /*!< PLL_PLLQ 44 */
#define RCU_PLLQ_DIV45                  PLL_PLLQ(44)                                    /*!< PLL_PLLQ 45 */
#define RCU_PLLQ_DIV46                  PLL_PLLQ(45)                                    /*!< PLL_PLLQ 46 */
#define RCU_PLLQ_DIV47                  PLL_PLLQ(46)                                    /*!< PLL_PLLQ 47 */
#define RCU_PLLQ_DIV48                  PLL_PLLQ(47)                                    /*!< PLL_PLLQ 48 */
#define RCU_PLLQ_DIV49                  PLL_PLLQ(48)                                    /*!< PLL_PLLQ 49 */
#define RCU_PLLQ_DIV50                  PLL_PLLQ(49)                                    /*!< PLL_PLLQ 50 */
#define RCU_PLLQ_DIV51                  PLL_PLLQ(50)                                    /*!< PLL_PLLQ 51 */
#define RCU_PLLQ_DIV52                  PLL_PLLQ(51)                                    /*!< PLL_PLLQ 52 */
#define RCU_PLLQ_DIV53                  PLL_PLLQ(52)                                    /*!< PLL_PLLQ 53 */
#define RCU_PLLQ_DIV54                  PLL_PLLQ(53)                                    /*!< PLL_PLLQ 54 */
#define RCU_PLLQ_DIV55                  PLL_PLLQ(54)                                    /*!< PLL_PLLQ 55 */
#define RCU_PLLQ_DIV56                  PLL_PLLQ(55)                                    /*!< PLL_PLLQ 56 */
#define RCU_PLLQ_DIV57                  PLL_PLLQ(56)                                    /*!< PLL_PLLQ 57 */
#define RCU_PLLQ_DIV58                  PLL_PLLQ(57)                                    /*!< PLL_PLLQ 58 */
#define RCU_PLLQ_DIV59                  PLL_PLLQ(58)                                    /*!< PLL_PLLQ 59 */
#define RCU_PLLQ_DIV60                  PLL_PLLQ(59)                                    /*!< PLL_PLLQ 60 */
#define RCU_PLLQ_DIV61                  PLL_PLLQ(60)                                    /*!< PLL_PLLQ 61 */
#define RCU_PLLQ_DIV62                  PLL_PLLQ(61)                                    /*!< PLL_PLLQ 62 */
#define RCU_PLLQ_DIV63                  PLL_PLLQ(62)                                    /*!< PLL_PLLQ 63 */
#define RCU_PLLQ_DIV64                  PLL_PLLQ(63)                                    /*!< PLL_PLLQ 64 */
#define RCU_PLLQ_DIV65                  PLL_PLLQ(64)                                    /*!< PLL_PLLQ 65 */
#define RCU_PLLQ_DIV66                  PLL_PLLQ(65)                                    /*!< PLL_PLLQ 66 */
#define RCU_PLLQ_DIV67                  PLL_PLLQ(66)                                    /*!< PLL_PLLQ 67 */
#define RCU_PLLQ_DIV68                  PLL_PLLQ(67)                                    /*!< PLL_PLLQ 68 */
#define RCU_PLLQ_DIV69                  PLL_PLLQ(68)                                    /*!< PLL_PLLQ 69 */
#define RCU_PLLQ_DIV70                  PLL_PLLQ(69)                                    /*!< PLL_PLLQ 70 */
#define RCU_PLLQ_DIV71                  PLL_PLLQ(70)                                    /*!< PLL_PLLQ 71 */
#define RCU_PLLQ_DIV72                  PLL_PLLQ(71)                                    /*!< PLL_PLLQ 72 */
#define RCU_PLLQ_DIV73                  PLL_PLLQ(72)                                    /*!< PLL_PLLQ 73 */
#define RCU_PLLQ_DIV74                  PLL_PLLQ(73)                                    /*!< PLL_PLLQ 74 */
#define RCU_PLLQ_DIV75                  PLL_PLLQ(74)                                    /*!< PLL_PLLQ 75 */
#define RCU_PLLQ_DIV76                  PLL_PLLQ(75)                                    /*!< PLL_PLLQ 76 */
#define RCU_PLLQ_DIV77                  PLL_PLLQ(76)                                    /*!< PLL_PLLQ 77 */
#define RCU_PLLQ_DIV78                  PLL_PLLQ(77)                                    /*!< PLL_PLLQ 78 */
#define RCU_PLLQ_DIV79                  PLL_PLLQ(78)                                    /*!< PLL_PLLQ 79 */
#define RCU_PLLQ_DIV80                  PLL_PLLQ(79)                                    /*!< PLL_PLLQ 80 */
#define RCU_PLLQ_DIV81                  PLL_PLLQ(80)                                    /*!< PLL_PLLQ 81 */
#define RCU_PLLQ_DIV82                  PLL_PLLQ(81)                                    /*!< PLL_PLLQ 82 */
#define RCU_PLLQ_DIV83                  PLL_PLLQ(82)                                    /*!< PLL_PLLQ 83 */
#define RCU_PLLQ_DIV84                  PLL_PLLQ(83)                                    /*!< PLL_PLLQ 84 */
#define RCU_PLLQ_DIV85                  PLL_PLLQ(84)                                    /*!< PLL_PLLQ 85 */
#define RCU_PLLQ_DIV86                  PLL_PLLQ(85)                                    /*!< PLL_PLLQ 86 */
#define RCU_PLLQ_DIV87                  PLL_PLLQ(86)                                    /*!< PLL_PLLQ 87 */
#define RCU_PLLQ_DIV88                  PLL_PLLQ(87)                                    /*!< PLL_PLLQ 88 */
#define RCU_PLLQ_DIV89                  PLL_PLLQ(88)                                    /*!< PLL_PLLQ 89 */
#define RCU_PLLQ_DIV90                  PLL_PLLQ(89)                                    /*!< PLL_PLLQ 90 */
#define RCU_PLLQ_DIV91                  PLL_PLLQ(90)                                    /*!< PLL_PLLQ 91 */
#define RCU_PLLQ_DIV92                  PLL_PLLQ(91)                                    /*!< PLL_PLLQ 92 */
#define RCU_PLLQ_DIV93                  PLL_PLLQ(92)                                    /*!< PLL_PLLQ 93 */
#define RCU_PLLQ_DIV94                  PLL_PLLQ(93)                                    /*!< PLL_PLLQ 94 */
#define RCU_PLLQ_DIV95                  PLL_PLLQ(94)                                    /*!< PLL_PLLQ 95 */
#define RCU_PLLQ_DIV96                  PLL_PLLQ(95)                                    /*!< PLL_PLLQ 96 */
#define RCU_PLLQ_DIV97                  PLL_PLLQ(96)                                    /*!< PLL_PLLQ 97 */
#define RCU_PLLQ_DIV98                  PLL_PLLQ(97)                                    /*!< PLL_PLLQ 98 */
#define RCU_PLLQ_DIV99                  PLL_PLLQ(98)                                    /*!< PLL_PLLQ 99 */
#define RCU_PLLQ_DIV100                 PLL_PLLQ(99)                                    /*!< PLL_PLLQ 100 */
#define RCU_PLLQ_DIV101                 PLL_PLLQ(100)                                   /*!< PLL_PLLQ 101 */
#define RCU_PLLQ_DIV102                 PLL_PLLQ(101)                                   /*!< PLL_PLLQ 102 */
#define RCU_PLLQ_DIV103                 PLL_PLLQ(102)                                   /*!< PLL_PLLQ 103 */
#define RCU_PLLQ_DIV104                 PLL_PLLQ(103)                                   /*!< PLL_PLLQ 104 */
#define RCU_PLLQ_DIV105                 PLL_PLLQ(104)                                   /*!< PLL_PLLQ 105 */
#define RCU_PLLQ_DIV106                 PLL_PLLQ(105)                                   /*!< PLL_PLLQ 106 */
#define RCU_PLLQ_DIV107                 PLL_PLLQ(106)                                   /*!< PLL_PLLQ 107 */
#define RCU_PLLQ_DIV108                 PLL_PLLQ(107)                                   /*!< PLL_PLLQ 108 */
#define RCU_PLLQ_DIV109                 PLL_PLLQ(108)                                   /*!< PLL_PLLQ 109 */
#define RCU_PLLQ_DIV110                 PLL_PLLQ(109)                                   /*!< PLL_PLLQ 110 */
#define RCU_PLLQ_DIV111                 PLL_PLLQ(110)                                   /*!< PLL_PLLQ 111 */
#define RCU_PLLQ_DIV112                 PLL_PLLQ(111)                                   /*!< PLL_PLLQ 112 */
#define RCU_PLLQ_DIV113                 PLL_PLLQ(112)                                   /*!< PLL_PLLQ 113 */
#define RCU_PLLQ_DIV114                 PLL_PLLQ(113)                                   /*!< PLL_PLLQ 114 */
#define RCU_PLLQ_DIV115                 PLL_PLLQ(114)                                   /*!< PLL_PLLQ 115 */
#define RCU_PLLQ_DIV116                 PLL_PLLQ(115)                                   /*!< PLL_PLLQ 116 */
#define RCU_PLLQ_DIV117                 PLL_PLLQ(116)                                   /*!< PLL_PLLQ 117 */
#define RCU_PLLQ_DIV118                 PLL_PLLQ(117)                                   /*!< PLL_PLLQ 118 */
#define RCU_PLLQ_DIV119                 PLL_PLLQ(118)                                   /*!< PLL_PLLQ 119 */
#define RCU_PLLQ_DIV120                 PLL_PLLQ(119)                                   /*!< PLL_PLLQ 120 */
#define RCU_PLLQ_DIV121                 PLL_PLLQ(120)                                   /*!< PLL_PLLQ 121 */
#define RCU_PLLQ_DIV122                 PLL_PLLQ(121)                                   /*!< PLL_PLLQ 122 */
#define RCU_PLLQ_DIV123                 PLL_PLLQ(122)                                   /*!< PLL_PLLQ 123 */
#define RCU_PLLQ_DIV124                 PLL_PLLQ(123)                                   /*!< PLL_PLLQ 124 */
#define RCU_PLLQ_DIV125                 PLL_PLLQ(124)                                   /*!< PLL_PLLQ 125 */
#define RCU_PLLQ_DIV126                 PLL_PLLQ(125)                                   /*!< PLL_PLLQ 126 */
#define RCU_PLLQ_DIV127                 PLL_PLLQ(126)                                   /*!< PLL_PLLQ 127 */
#define RCU_PLLQ_DIV128                 PLL_PLLQ(127)                                   /*!< PLL_PLLQ 128 */

/* PLL Clock Source Selection */
#define PLLALL_PLLSEL(regval)           (BITS(16,17) & ((uint32_t)(regval) << 16U))
#define RCU_PLLSRC_IRC64MDIV            PLLALL_PLLSEL(0)                                /*!< IRC64MDIV clock selected as source clock of PLL0, PLL1, PLL2 */
#define RCU_PLLSRC_LPIRC4M              PLLALL_PLLSEL(1)                                /*!< LPIRC4M clock selected as source clock of PLL0, PLL1, PLL2 */
#define RCU_PLLSRC_HXTAL                PLLALL_PLLSEL(2)                                /*!< HXTAL clock selected as source clock of PLL0, PLL1, PLL2 */
#define RCU_PLLSRC_NONE                 PLLALL_PLLSEL(3)                                /*!< No HXTAL clock selected as source clock of PLL0, PLL1, PLL2 */

/* RCU_BDCTL register bit define */
/* LXTAL drive capability */
#define BDCTL_LXTALDRI(regval)          (BITS(3,4) & ((uint32_t)(regval) << 3U))
#define RCU_LXTAL_LOWDRI                BDCTL_LXTALDRI(0)                               /*!< lower driving capability */
#define RCU_LXTAL_MED_LOWDRI            BDCTL_LXTALDRI(1)                               /*!< medium low driving capability */
#define RCU_LXTAL_MED_HIGHDRI           BDCTL_LXTALDRI(2)                               /*!< medium high driving capability */
#define RCU_LXTAL_HIGHDRI               BDCTL_LXTALDRI(3)                               /*!< higher driving capability */

/* RTC clock entry selection */
#define BDCTL_RTCSRC(regval)            (BITS(8,9) & ((uint32_t)(regval) << 8U))
#define RCU_RTCSRC_NONE                 BDCTL_RTCSRC(0)                                 /*!< no clock selected */
#define RCU_RTCSRC_LXTAL                BDCTL_RTCSRC(1)                                 /*!< RTC source clock select LXTAL */
#define RCU_RTCSRC_IRC32K               BDCTL_RTCSRC(2)                                 /*!< RTC source clock select IRC32K */
#define RCU_RTCSRC_HXTAL_DIV_RTCDIV     BDCTL_RTCSRC(3)                                 /*!< RTC source clock select HXTAL/RTCDIV */

/* PLL0 input clock range */
#define PLLALL_PLL0RNG(regval)          (BITS(0,1) & ((uint32_t)(regval) << 0U))
#define RCU_PLL0RNG_1M_2M               PLLALL_PLL0RNG(0)                               /*!< PLL0 input clock frequency: 1-2MHz */
#define RCU_PLL0RNG_2M_4M               PLLALL_PLL0RNG(1)                               /*!< PLL0 input clock frequency: 2-4MHz */
#define RCU_PLL0RNG_4M_8M               PLLALL_PLL0RNG(2)                               /*!< PLL0 input clock frequency: 4-8MHz */
#define RCU_PLL0RNG_8M_16M              PLLALL_PLL0RNG(3)                               /*!< PLL0 input clock frequency: 8-16MHz */

/* PLL1 input clock range */
#define PLLALL_PLL1RNG(regval)          (BITS(4,5) & ((uint32_t)(regval) << 4U))
#define RCU_PLL1RNG_1M_2M               PLLALL_PLL1RNG(0)                               /*!< PLL1 input clock frequency: 1-2MHz */
#define RCU_PLL1RNG_2M_4M               PLLALL_PLL1RNG(1)                               /*!< PLL1 input clock frequency: 2-4MHz */
#define RCU_PLL1RNG_4M_8M               PLLALL_PLL1RNG(2)                               /*!< PLL1 input clock frequency: 4-8MHz */
#define RCU_PLL1RNG_8M_16M              PLLALL_PLL1RNG(3)                               /*!< PLL1 input clock frequency: 8-16MHz */

/* PLL2 input clock range */
#define PLLALL_PLL2RNG(regval)          (BITS(8,9) & ((uint32_t)(regval) << 8U))
#define RCU_PLL2RNG_1M_2M               PLLALL_PLL2RNG(0)                               /*!< PLL2 input clock frequency: 1-2MHz */
#define RCU_PLL2RNG_2M_4M               PLLALL_PLL2RNG(1)                               /*!< PLL2 input clock frequency: 2-4MHz */
#define RCU_PLL2RNG_4M_8M               PLLALL_PLL2RNG(2)                               /*!< PLL2 input clock frequency: 4-8MHz */
#define RCU_PLL2RNG_8M_16M              PLLALL_PLL2RNG(3)                               /*!< PLL2 input clock frequency: 8-16MHz */

/* PLL0 VCO selection */
#define RCU_PLL0VCO_192M_836M           ((uint32_t)0x00000000U)                         /*!< PLL0 select wide VCO, range: 192-836MHz */
#define RCU_PLL0VCO_150M_420M           RCU_PLLALL_PLL0VCOSEL                           /*!< PLL0 select narrow VCO, range: 150-420MHz */

/* PLL1 VCO selection */
#define RCU_PLL1VCO_192M_836M           ((uint32_t)0x00000000U)                         /*!< PLL1 select wide VCO, range: 192-836MHz */
#define RCU_PLL1VCO_150M_420M           RCU_PLLALL_PLL1VCOSEL                           /*!< PLL1 select narrow VCO, range: 150-420MHz */

/* PLL2 VCO selection */
#define RCU_PLL2VCO_192M_836M           ((uint32_t)0x00000000U)                         /*!< PLL2 select wide VCO, range: 192-836MHz */
#define RCU_PLL2VCO_150M_420M           RCU_PLLALL_PLL2VCOSEL                           /*!< PLL2 select narrow VCO, range: 150-420MHz */

/* RCU_ADDCTL0 register bit define */
/* 48MHz clock selection */
#define RCU_CK48MSRC_PLL48M             ((uint32_t)0x00000000U)                         /*!< CK48M source clock select PLL48M */
#define RCU_CK48MSRC_IRC48M             RCU_ADDCTL0_CK48MSEL                            /*!< CK48M source clock select IRC48M */

/* PLL48M clock selection */
#define RCU_PLL48MSRC_PLL0Q             ((uint32_t)0x00000000U)                         /*!< PLL48M source clock select PLL0Q */
#define RCU_PLL48MSRC_PLL2P             RCU_ADDCTL0_PLL48MSEL                           /*!< PLL48M source clock select PLL2P */

/* USBHS clock selection */
#define RCU_USBHSSEL_48M                ((uint32_t)0x00000000U)                         /*!< 48M selected as USBHS source clock */
#define RCU_USBHSSEL_60M                RCU_USBCLKCTL_USBHS0SEL                         /*!< 60M selected as USBHS source clock */

/* PLLUSBHSPRE clock selection */
#define RCU_PLLUSBHSPRE_HXTAL           ((uint32_t)0x00000000U)                         /*!< CK_HATAL selected as PLLUSBHS source clock */
#define RCU_PLLUSBHSPRE_IRC48M          RCU_USBCLKCTL_PLLUSBHS0PRESEL                   /*!< CK_IRC48M selected as PLLUSBHS source clock */

/* PLLUSBHSxPREDV division factor*/
#define PLLUSBCFG_PLLUSBHSPREDV(regval) (BITS(0,3) & ((uint32_t)(regval) << 0U))
#define RCU_PLLUSBHSxPRE_DIV1           PLLUSBCFG_PLLUSBHSPREDV(1)                      /*!< PLLUSBHSPREDV input source clock divided by 1 */
#define RCU_PLLUSBHSxPRE_DIV2           PLLUSBCFG_PLLUSBHSPREDV(2)                      /*!< PLLUSBHSPREDV input source clock divided by 2 */
#define RCU_PLLUSBHSxPRE_DIV3           PLLUSBCFG_PLLUSBHSPREDV(3)                      /*!< PLLUSBHSPREDV input source clock divided by 3 */
#define RCU_PLLUSBHSxPRE_DIV4           PLLUSBCFG_PLLUSBHSPREDV(4)                      /*!< PLLUSBHSPREDV input source clock divided by 4 */
#define RCU_PLLUSBHSxPRE_DIV5           PLLUSBCFG_PLLUSBHSPREDV(5)                      /*!< PLLUSBHSPREDV input source clock divided by 5 */
#define RCU_PLLUSBHSxPRE_DIV6           PLLUSBCFG_PLLUSBHSPREDV(6)                      /*!< PLLUSBHSPREDV input source clock divided by 6 */
#define RCU_PLLUSBHSxPRE_DIV7           PLLUSBCFG_PLLUSBHSPREDV(7)                      /*!< PLLUSBHSPREDV input source clock divided by 7 */
#define RCU_PLLUSBHSxPRE_DIV8           PLLUSBCFG_PLLUSBHSPREDV(8)                      /*!< PLLUSBHSPREDV input source clock divided by 8 */
#define RCU_PLLUSBHSxPRE_DIV9           PLLUSBCFG_PLLUSBHSPREDV(9)                      /*!< PLLUSBHSPREDV input source clock divided by 9 */
#define RCU_PLLUSBHSxPRE_DIV10          PLLUSBCFG_PLLUSBHSPREDV(10)                     /*!< PLLUSBHSPREDV input source clock divided by 10 */
#define RCU_PLLUSBHSxPRE_DIV11          PLLUSBCFG_PLLUSBHSPREDV(11)                     /*!< PLLUSBHSPREDV input source clock divided by 11 */
#define RCU_PLLUSBHSxPRE_DIV12          PLLUSBCFG_PLLUSBHSPREDV(12)                     /*!< PLLUSBHSPREDV input source clock divided by 12 */
#define RCU_PLLUSBHSxPRE_DIV13          PLLUSBCFG_PLLUSBHSPREDV(13)                     /*!< PLLUSBHSPREDV input source clock divided by 13 */
#define RCU_PLLUSBHSxPRE_DIV14          PLLUSBCFG_PLLUSBHSPREDV(14)                     /*!< PLLUSBHSPREDV input source clock divided by 14 */
#define RCU_PLLUSBHSxPRE_DIV15          PLLUSBCFG_PLLUSBHSPREDV(15)                     /*!< PLLUSBHSPREDV input source clock divided by 15 */

/* USBHSxDV(x=0,1) division factor */
#define PLLUSBCFG_USBHSDV(regval)       (BITS(4,6) & ((uint32_t)(regval) << 4U))
#define RCU_USBHSx_DIV2                  PLLUSBCFG_USBHSDV(0)                           /*!< USBHSxDV(x=0,1) input source clock divided by 2 */
#define RCU_USBHSx_DIV4                  PLLUSBCFG_USBHSDV(1)                           /*!< USBHSxDV(x=0,1) input source clock divided by 4 */
#define RCU_USBHSx_DIV6                  PLLUSBCFG_USBHSDV(2)                           /*!< USBHSxDV(x=0,1) input source clock divided by 6 */
#define RCU_USBHSx_DIV8                  PLLUSBCFG_USBHSDV(3)                           /*!< USBHSxDV(x=0,1) input source clock divided by 8 */
#define RCU_USBHSx_DIV10                 PLLUSBCFG_USBHSDV(4)                           /*!< USBHSxDV(x=0,1) input source clock divided by 10 */
#define RCU_USBHSx_DIV12                 PLLUSBCFG_USBHSDV(5)                           /*!< USBHSxDV(x=0,1) input source clock divided by 12 */
#define RCU_USBHSx_DIV14                 PLLUSBCFG_USBHSDV(6)                           /*!< USBHSxDV(x=0,1) input source clock divided by 14 */
#define RCU_USBHSx_DIV16                 PLLUSBCFG_USBHSDV(7)                           /*!< USBHSxDV(x=0,1) input source clock divided by 16 */

/* PLLUSBx(x=0,1) clock multiplication factor */
#define PLLUSBCFG_PLLUSBHSxMF(regval)   (BITS(8,14) & ((uint32_t)(regval) << 8U))
#define RCU_PLLUSBHSx_MUL16             PLLUSBCFG_PLLUSBHSxMF(16)                       /*!< PLLUSBHSx source clock multiply by 16 */
#define RCU_PLLUSBHSx_MUL17             PLLUSBCFG_PLLUSBHSxMF(17)                       /*!< PLLUSBHSx source clock multiply by 17 */
#define RCU_PLLUSBHSx_MUL18             PLLUSBCFG_PLLUSBHSxMF(18)                       /*!< PLLUSBHSx source clock multiply by 18 */
#define RCU_PLLUSBHSx_MUL19             PLLUSBCFG_PLLUSBHSxMF(19)                       /*!< PLLUSBHSx source clock multiply by 19 */
#define RCU_PLLUSBHSx_MUL20             PLLUSBCFG_PLLUSBHSxMF(20)                       /*!< PLLUSBHSx source clock multiply by 20 */
#define RCU_PLLUSBHSx_MUL21             PLLUSBCFG_PLLUSBHSxMF(21)                       /*!< PLLUSBHSx source clock multiply by 21 */
#define RCU_PLLUSBHSx_MUL22             PLLUSBCFG_PLLUSBHSxMF(22)                       /*!< PLLUSBHSx source clock multiply by 22 */
#define RCU_PLLUSBHSx_MUL23             PLLUSBCFG_PLLUSBHSxMF(23)                       /*!< PLLUSBHSx source clock multiply by 23 */
#define RCU_PLLUSBHSx_MUL24             PLLUSBCFG_PLLUSBHSxMF(24)                       /*!< PLLUSBHSx source clock multiply by 24 */
#define RCU_PLLUSBHSx_MUL25             PLLUSBCFG_PLLUSBHSxMF(25)                       /*!< PLLUSBHSx source clock multiply by 25 */
#define RCU_PLLUSBHSx_MUL26             PLLUSBCFG_PLLUSBHSxMF(26)                       /*!< PLLUSBHSx source clock multiply by 26 */
#define RCU_PLLUSBHSx_MUL27             PLLUSBCFG_PLLUSBHSxMF(27)                       /*!< PLLUSBHSx source clock multiply by 27 */
#define RCU_PLLUSBHSx_MUL28             PLLUSBCFG_PLLUSBHSxMF(28)                       /*!< PLLUSBHSx source clock multiply by 28 */
#define RCU_PLLUSBHSx_MUL29             PLLUSBCFG_PLLUSBHSxMF(29)                       /*!< PLLUSBHSx source clock multiply by 29 */
#define RCU_PLLUSBHSx_MUL30             PLLUSBCFG_PLLUSBHSxMF(30)                       /*!< PLLUSBHSx source clock multiply by 30 */
#define RCU_PLLUSBHSx_MUL31             PLLUSBCFG_PLLUSBHSxMF(31)                       /*!< PLLUSBHSx source clock multiply by 31 */
#define RCU_PLLUSBHSx_MUL32             PLLUSBCFG_PLLUSBHSxMF(32)                       /*!< PLLUSBHSx source clock multiply by 32 */
#define RCU_PLLUSBHSx_MUL33             PLLUSBCFG_PLLUSBHSxMF(33)                       /*!< PLLUSBHSx source clock multiply by 33 */
#define RCU_PLLUSBHSx_MUL34             PLLUSBCFG_PLLUSBHSxMF(34)                       /*!< PLLUSBHSx source clock multiply by 34 */
#define RCU_PLLUSBHSx_MUL35             PLLUSBCFG_PLLUSBHSxMF(35)                       /*!< PLLUSBHSx source clock multiply by 35 */
#define RCU_PLLUSBHSx_MUL36             PLLUSBCFG_PLLUSBHSxMF(36)                       /*!< PLLUSBHSx source clock multiply by 36 */
#define RCU_PLLUSBHSx_MUL37             PLLUSBCFG_PLLUSBHSxMF(37)                       /*!< PLLUSBHSx source clock multiply by 37 */
#define RCU_PLLUSBHSx_MUL38             PLLUSBCFG_PLLUSBHSxMF(38)                       /*!< PLLUSBHSx source clock multiply by 38 */
#define RCU_PLLUSBHSx_MUL39             PLLUSBCFG_PLLUSBHSxMF(39)                       /*!< PLLUSBHSx source clock multiply by 39 */
#define RCU_PLLUSBHSx_MUL40             PLLUSBCFG_PLLUSBHSxMF(40)                       /*!< PLLUSBHSx source clock multiply by 40 */
#define RCU_PLLUSBHSx_MUL41             PLLUSBCFG_PLLUSBHSxMF(41)                       /*!< PLLUSBHSx source clock multiply by 41 */
#define RCU_PLLUSBHSx_MUL42             PLLUSBCFG_PLLUSBHSxMF(42)                       /*!< PLLUSBHSx source clock multiply by 42 */
#define RCU_PLLUSBHSx_MUL43             PLLUSBCFG_PLLUSBHSxMF(43)                       /*!< PLLUSBHSx source clock multiply by 43 */
#define RCU_PLLUSBHSx_MUL44             PLLUSBCFG_PLLUSBHSxMF(44)                       /*!< PLLUSBHSx source clock multiply by 44 */
#define RCU_PLLUSBHSx_MUL45             PLLUSBCFG_PLLUSBHSxMF(45)                       /*!< PLLUSBHSx source clock multiply by 45 */
#define RCU_PLLUSBHSx_MUL46             PLLUSBCFG_PLLUSBHSxMF(46)                       /*!< PLLUSBHSx source clock multiply by 46 */
#define RCU_PLLUSBHSx_MUL47             PLLUSBCFG_PLLUSBHSxMF(47)                       /*!< PLLUSBHSx source clock multiply by 47 */
#define RCU_PLLUSBHSx_MUL48             PLLUSBCFG_PLLUSBHSxMF(48)                       /*!< PLLUSBHSx source clock multiply by 48 */
#define RCU_PLLUSBHSx_MUL49             PLLUSBCFG_PLLUSBHSxMF(49)                       /*!< PLLUSBHSx source clock multiply by 49 */
#define RCU_PLLUSBHSx_MUL50             PLLUSBCFG_PLLUSBHSxMF(50)                       /*!< PLLUSBHSx source clock multiply by 50 */
#define RCU_PLLUSBHSx_MUL51             PLLUSBCFG_PLLUSBHSxMF(51)                       /*!< PLLUSBHSx source clock multiply by 51 */
#define RCU_PLLUSBHSx_MUL52             PLLUSBCFG_PLLUSBHSxMF(52)                       /*!< PLLUSBHSx source clock multiply by 52 */
#define RCU_PLLUSBHSx_MUL53             PLLUSBCFG_PLLUSBHSxMF(53)                       /*!< PLLUSBHSx source clock multiply by 53 */
#define RCU_PLLUSBHSx_MUL54             PLLUSBCFG_PLLUSBHSxMF(54)                       /*!< PLLUSBHSx source clock multiply by 54 */
#define RCU_PLLUSBHSx_MUL55             PLLUSBCFG_PLLUSBHSxMF(55)                       /*!< PLLUSBHSx source clock multiply by 55 */
#define RCU_PLLUSBHSx_MUL56             PLLUSBCFG_PLLUSBHSxMF(56)                       /*!< PLLUSBHSx source clock multiply by 56 */
#define RCU_PLLUSBHSx_MUL57             PLLUSBCFG_PLLUSBHSxMF(57)                       /*!< PLLUSBHSx source clock multiply by 57 */
#define RCU_PLLUSBHSx_MUL58             PLLUSBCFG_PLLUSBHSxMF(58)                       /*!< PLLUSBHSx source clock multiply by 58 */
#define RCU_PLLUSBHSx_MUL59             PLLUSBCFG_PLLUSBHSxMF(59)                       /*!< PLLUSBHSx source clock multiply by 59 */
#define RCU_PLLUSBHSx_MUL60             PLLUSBCFG_PLLUSBHSxMF(60)                       /*!< PLLUSBHSx source clock multiply by 60 */
#define RCU_PLLUSBHSx_MUL61             PLLUSBCFG_PLLUSBHSxMF(61)                       /*!< PLLUSBHSx source clock multiply by 61 */
#define RCU_PLLUSBHSx_MUL62             PLLUSBCFG_PLLUSBHSxMF(62)                       /*!< PLLUSBHSx source clock multiply by 62 */
#define RCU_PLLUSBHSx_MUL63             PLLUSBCFG_PLLUSBHSxMF(63)                       /*!< PLLUSBHSx source clock multiply by 63 */
#define RCU_PLLUSBHSx_MUL64             PLLUSBCFG_PLLUSBHSxMF(64)                       /*!< PLLUSBHSx source clock multiply by 64 */
#define RCU_PLLUSBHSx_MUL65             PLLUSBCFG_PLLUSBHSxMF(65)                       /*!< PLLUSBHSx source clock multiply by 65 */
#define RCU_PLLUSBHSx_MUL66             PLLUSBCFG_PLLUSBHSxMF(66)                       /*!< PLLUSBHSx source clock multiply by 66 */
#define RCU_PLLUSBHSx_MUL67             PLLUSBCFG_PLLUSBHSxMF(67)                       /*!< PLLUSBHSx source clock multiply by 67 */
#define RCU_PLLUSBHSx_MUL68             PLLUSBCFG_PLLUSBHSxMF(68)                       /*!< PLLUSBHSx source clock multiply by 68 */
#define RCU_PLLUSBHSx_MUL69             PLLUSBCFG_PLLUSBHSxMF(69)                       /*!< PLLUSBHSx source clock multiply by 69 */
#define RCU_PLLUSBHSx_MUL70             PLLUSBCFG_PLLUSBHSxMF(70)                       /*!< PLLUSBHSx source clock multiply by 70 */
#define RCU_PLLUSBHSx_MUL71             PLLUSBCFG_PLLUSBHSxMF(71)                       /*!< PLLUSBHSx source clock multiply by 71 */
#define RCU_PLLUSBHSx_MUL72             PLLUSBCFG_PLLUSBHSxMF(72)                       /*!< PLLUSBHSx source clock multiply by 72 */
#define RCU_PLLUSBHSx_MUL73             PLLUSBCFG_PLLUSBHSxMF(73)                       /*!< PLLUSBHSx source clock multiply by 73 */
#define RCU_PLLUSBHSx_MUL74             PLLUSBCFG_PLLUSBHSxMF(74)                       /*!< PLLUSBHSx source clock multiply by 74 */
#define RCU_PLLUSBHSx_MUL75             PLLUSBCFG_PLLUSBHSxMF(75)                       /*!< PLLUSBHSx source clock multiply by 75 */
#define RCU_PLLUSBHSx_MUL76             PLLUSBCFG_PLLUSBHSxMF(76)                       /*!< PLLUSBHSx source clock multiply by 76 */
#define RCU_PLLUSBHSx_MUL77             PLLUSBCFG_PLLUSBHSxMF(77)                       /*!< PLLUSBHSx source clock multiply by 77 */
#define RCU_PLLUSBHSx_MUL78             PLLUSBCFG_PLLUSBHSxMF(78)                       /*!< PLLUSBHSx source clock multiply by 78 */
#define RCU_PLLUSBHSx_MUL79             PLLUSBCFG_PLLUSBHSxMF(79)                       /*!< PLLUSBHSx source clock multiply by 79 */
#define RCU_PLLUSBHSx_MUL80             PLLUSBCFG_PLLUSBHSxMF(80)                       /*!< PLLUSBHSx source clock multiply by 80 */
#define RCU_PLLUSBHSx_MUL81             PLLUSBCFG_PLLUSBHSxMF(81)                       /*!< PLLUSBHSx source clock multiply by 81 */
#define RCU_PLLUSBHSx_MUL82             PLLUSBCFG_PLLUSBHSxMF(82)                       /*!< PLLUSBHSx source clock multiply by 82 */
#define RCU_PLLUSBHSx_MUL83             PLLUSBCFG_PLLUSBHSxMF(83)                       /*!< PLLUSBHSx source clock multiply by 83 */
#define RCU_PLLUSBHSx_MUL84             PLLUSBCFG_PLLUSBHSxMF(84)                       /*!< PLLUSBHSx source clock multiply by 84 */
#define RCU_PLLUSBHSx_MUL85             PLLUSBCFG_PLLUSBHSxMF(85)                       /*!< PLLUSBHSx source clock multiply by 85 */
#define RCU_PLLUSBHSx_MUL86             PLLUSBCFG_PLLUSBHSxMF(86)                       /*!< PLLUSBHSx source clock multiply by 86 */
#define RCU_PLLUSBHSx_MUL87             PLLUSBCFG_PLLUSBHSxMF(87)                       /*!< PLLUSBHSx source clock multiply by 87 */
#define RCU_PLLUSBHSx_MUL88             PLLUSBCFG_PLLUSBHSxMF(88)                       /*!< PLLUSBHSx source clock multiply by 88 */
#define RCU_PLLUSBHSx_MUL89             PLLUSBCFG_PLLUSBHSxMF(89)                       /*!< PLLUSBHSx source clock multiply by 89 */
#define RCU_PLLUSBHSx_MUL90             PLLUSBCFG_PLLUSBHSxMF(90)                       /*!< PLLUSBHSx source clock multiply by 90 */
#define RCU_PLLUSBHSx_MUL91             PLLUSBCFG_PLLUSBHSxMF(91)                       /*!< PLLUSBHSx source clock multiply by 91 */
#define RCU_PLLUSBHSx_MUL92             PLLUSBCFG_PLLUSBHSxMF(92)                       /*!< PLLUSBHSx source clock multiply by 92 */
#define RCU_PLLUSBHSx_MUL93             PLLUSBCFG_PLLUSBHSxMF(93)                       /*!< PLLUSBHSx source clock multiply by 93 */
#define RCU_PLLUSBHSx_MUL94             PLLUSBCFG_PLLUSBHSxMF(94)                       /*!< PLLUSBHSx source clock multiply by 94 */
#define RCU_PLLUSBHSx_MUL95             PLLUSBCFG_PLLUSBHSxMF(95)                       /*!< PLLUSBHSx source clock multiply by 95 */
#define RCU_PLLUSBHSx_MUL96             PLLUSBCFG_PLLUSBHSxMF(96)                       /*!< PLLUSBHSx source clock multiply by 96 */
#define RCU_PLLUSBHSx_MUL97             PLLUSBCFG_PLLUSBHSxMF(97)                       /*!< PLLUSBHSx source clock multiply by 97 */
#define RCU_PLLUSBHSx_MUL98             PLLUSBCFG_PLLUSBHSxMF(98)                       /*!< PLLUSBHSx source clock multiply by 98 */
#define RCU_PLLUSBHSx_MUL99             PLLUSBCFG_PLLUSBHSxMF(99)                       /*!< PLLUSBHSx source clock multiply by 99 */
#define RCU_PLLUSBHSx_MUL100            PLLUSBCFG_PLLUSBHSxMF(100)                      /*!< PLLUSBHSx source clock multiply by 100 */
#define RCU_PLLUSBHSx_MUL101            PLLUSBCFG_PLLUSBHSxMF(101)                      /*!< PLLUSBHSx source clock multiply by 101 */
#define RCU_PLLUSBHSx_MUL102            PLLUSBCFG_PLLUSBHSxMF(102)                      /*!< PLLUSBHSx source clock multiply by 102 */
#define RCU_PLLUSBHSx_MUL103            PLLUSBCFG_PLLUSBHSxMF(103)                      /*!< PLLUSBHSx source clock multiply by 103 */
#define RCU_PLLUSBHSx_MUL104            PLLUSBCFG_PLLUSBHSxMF(104)                      /*!< PLLUSBHSx source clock multiply by 104 */
#define RCU_PLLUSBHSx_MUL105            PLLUSBCFG_PLLUSBHSxMF(105)                      /*!< PLLUSBHSx source clock multiply by 105 */
#define RCU_PLLUSBHSx_MUL106            PLLUSBCFG_PLLUSBHSxMF(106)                      /*!< PLLUSBHSx source clock multiply by 106 */
#define RCU_PLLUSBHSx_MUL107            PLLUSBCFG_PLLUSBHSxMF(107)                      /*!< PLLUSBHSx source clock multiply by 107 */
#define RCU_PLLUSBHSx_MUL108            PLLUSBCFG_PLLUSBHSxMF(108)                      /*!< PLLUSBHSx source clock multiply by 108 */
#define RCU_PLLUSBHSx_MUL109            PLLUSBCFG_PLLUSBHSxMF(109)                      /*!< PLLUSBHSx source clock multiply by 109 */
#define RCU_PLLUSBHSx_MUL110            PLLUSBCFG_PLLUSBHSxMF(110)                      /*!< PLLUSBHSx source clock multiply by 110 */
#define RCU_PLLUSBHSx_MUL111            PLLUSBCFG_PLLUSBHSxMF(111)                      /*!< PLLUSBHSx source clock multiply by 111 */
#define RCU_PLLUSBHSx_MUL112            PLLUSBCFG_PLLUSBHSxMF(112)                      /*!< PLLUSBHSx source clock multiply by 112 */
#define RCU_PLLUSBHSx_MUL113            PLLUSBCFG_PLLUSBHSxMF(113)                      /*!< PLLUSBHSx source clock multiply by 113 */
#define RCU_PLLUSBHSx_MUL114            PLLUSBCFG_PLLUSBHSxMF(114)                      /*!< PLLUSBHSx source clock multiply by 114 */
#define RCU_PLLUSBHSx_MUL115            PLLUSBCFG_PLLUSBHSxMF(115)                      /*!< PLLUSBHSx source clock multiply by 115 */
#define RCU_PLLUSBHSx_MUL116            PLLUSBCFG_PLLUSBHSxMF(116)                      /*!< PLLUSBHSx source clock multiply by 116 */
#define RCU_PLLUSBHSx_MUL117            PLLUSBCFG_PLLUSBHSxMF(117)                      /*!< PLLUSBHSx source clock multiply by 117 */
#define RCU_PLLUSBHSx_MUL118            PLLUSBCFG_PLLUSBHSxMF(118)                      /*!< PLLUSBHSx source clock multiply by 118 */
#define RCU_PLLUSBHSx_MUL119            PLLUSBCFG_PLLUSBHSxMF(119)                      /*!< PLLUSBHSx source clock multiply by 119 */
#define RCU_PLLUSBHSx_MUL120            PLLUSBCFG_PLLUSBHSxMF(120)                      /*!< PLLUSBHSx source clock multiply by 120 */
#define RCU_PLLUSBHSx_MUL121            PLLUSBCFG_PLLUSBHSxMF(121)                      /*!< PLLUSBHSx source clock multiply by 121 */
#define RCU_PLLUSBHSx_MUL122            PLLUSBCFG_PLLUSBHSxMF(122)                      /*!< PLLUSBHSx source clock multiply by 122 */
#define RCU_PLLUSBHSx_MUL123            PLLUSBCFG_PLLUSBHSxMF(123)                      /*!< PLLUSBHSx source clock multiply by 123 */
#define RCU_PLLUSBHSx_MUL124            PLLUSBCFG_PLLUSBHSxMF(124)                      /*!< PLLUSBHSx source clock multiply by 124 */
#define RCU_PLLUSBHSx_MUL125            PLLUSBCFG_PLLUSBHSxMF(125)                      /*!< PLLUSBHSx source clock multiply by 125 */
#define RCU_PLLUSBHSx_MUL126            PLLUSBCFG_PLLUSBHSxMF(126)                      /*!< PLLUSBHSx source clock multiply by 126 */
#define RCU_PLLUSBHSx_MUL127            PLLUSBCFG_PLLUSBHSxMF(127)                      /*!< PLLUSBHSx source clock multiply by 127 */

/* USBHSx(x=0,1) prescaler factor */
#define USBCLKCTL_USBHSxPSC(regval)     (BITS(16,18) & ((uint32_t)(regval) << 16U))
#define RCU_USBHSxPSC_DIV1              USBCLKCTL_USBHSxPSC(0)
#define RCU_USBHSxPSC_DIV2              USBCLKCTL_USBHSxPSC(1)                          /*!< USBHSx prescaler select CK_PLL1Q / 2 */
#define RCU_USBHSxPSC_DIV3              USBCLKCTL_USBHSxPSC(2)                          /*!< USBHSx prescaler select CK_PLL1Q / 3 */
#define RCU_USBHSxPSC_DIV4              USBCLKCTL_USBHSxPSC(3)                          /*!< USBHSx prescaler select CK_PLL1Q / 4 */
#define RCU_USBHSxPSC_DIV5              USBCLKCTL_USBHSxPSC(4)                          /*!< USBHSx prescaler select CK_PLL1Q / 5 */
#define RCU_USBHSxPSC_DIV6              USBCLKCTL_USBHSxPSC(5)                          /*!< USBHSx prescaler select CK_PLL1Q / 6 */
#define RCU_USBHSxPSC_DIV7              USBCLKCTL_USBHSxPSC(6)                          /*!< USBHSx prescaler select CK_PLL1Q / 7 */
#define RCU_USBHSxPSC_DIV8              USBCLKCTL_USBHSxPSC(7)                          /*!< USBHSx prescaler select CK_PLL1Q / 8 */

/* USBHS 48MHz clock selection */
#define USBCLKCTL_USB48MSEL(regval)     (BITS(5,6) & ((uint32_t)(regval) << 5U))
#define RCU_USB48MSRC_PLL0R             USBCLKCTL_USB48MSEL(0)                          /*!< select PLL0R as USABHS 48MHz clock source */
#define RCU_USB48MSRC_PLLUSBHS          USBCLKCTL_USB48MSEL(1)                          /*!< select PLLUSBHS / USBHSDV as USABHS 48MHz clock source */
#define RCU_USB48MSRC_PLL1Q             USBCLKCTL_USB48MSEL(2)                          /*!< select PLL1Q / USBHSPSC as USABHS 48MHz clock source */
#define RCU_USB48MSRC_IRC48M            USBCLKCTL_USB48MSEL(3)                          /*!< select IRC48M as USABHS 48MHz clock source */

/* register index */
typedef enum {
    /* peripherals enable */
    IDX_AHB1EN        = ((uint32_t)0x00000030U), /*!< AHB1 enable register offset */
    IDX_AHB2EN        = ((uint32_t)0x00000034U), /*!< AHB2 enable register offset */
    IDX_AHB3EN        = ((uint32_t)0x00000038U), /*!< AHB3 enable register offset */
    IDX_AHB4EN        = ((uint32_t)0x0000003CU), /*!< AHB4 enable register offset */
    IDX_APB1EN        = ((uint32_t)0x00000040U), /*!< APB1 enable register offset */
    IDX_APB2EN        = ((uint32_t)0x00000044U), /*!< APB2 enable register offset */
    IDX_APB3EN        = ((uint32_t)0x00000048U), /*!< APB3 enable register offset */
    IDX_APB4EN        = ((uint32_t)0x0000004CU), /*!< APB4 enable register offset */
    IDX_AHB1SPEN      = ((uint32_t)0x00000050U), /*!< AHB1 sleep mode enable register offset */
    IDX_AHB2SPEN      = ((uint32_t)0x00000054U), /*!< AHB2 sleep mode enable register offset */
    IDX_AHB3SPEN      = ((uint32_t)0x00000058U), /*!< AHB3 sleep mode enable register offset */
    IDX_AHB4SPEN      = ((uint32_t)0x0000005CU), /*!< AHB4 sleep mode enable register offset */
    IDX_APB1SPEN      = ((uint32_t)0x00000060U), /*!< APB1 sleep mode enable register offset */
    IDX_APB2SPEN      = ((uint32_t)0x00000064U), /*!< APB2 sleep mode enable register offset */
    IDX_APB3SPEN      = ((uint32_t)0x00000068U), /*!< APB3 sleep mode enable register offset */
    IDX_APB4SPEN      = ((uint32_t)0x0000006CU), /*!< APB4 sleep mode enable register offset */
    IDX_ADD_APB2EN    = ((uint32_t)0x000000E4U), /*!< APB2 additional enable register offset */
    IDX_ADD_APB2SPEN  = ((uint32_t)0x000000E8U), /*!< APB2 additional sleep mode enable register offset */
    /* peripherals reset */
    IDX_AHB1RST       = ((uint32_t)0x00000010U), /*!< AHB1 reset register offset */
    IDX_AHB2RST       = ((uint32_t)0x00000014U), /*!< AHB2 reset register offset */
    IDX_AHB3RST       = ((uint32_t)0x00000018U), /*!< AHB3 reset register offset */
    IDX_AHB4RST       = ((uint32_t)0x0000001CU), /*!< AHB4 reset register offset */
    IDX_APB1RST       = ((uint32_t)0x00000020U), /*!< APB1 reset register offset */
    IDX_APB2RST       = ((uint32_t)0x00000024U), /*!< APB2 reset register offset */
    IDX_APB3RST       = ((uint32_t)0x00000028U), /*!< APB3 reset register offset */
    IDX_APB4RST       = ((uint32_t)0x0000002CU), /*!< APB4 reset register offset */
    IDX_ADD_APB2RST   = ((uint32_t)0x000000E0U), /*!< APB2 additional reset register offset */
    IDX_RSTSCK        = ((uint32_t)0x00000074U), /*!< reset source/clock register offset */
    /* clock control */
    IDX_CTL           = ((uint32_t)0x00000000U), /*!< control register offset */
    IDX_BDCTL         = ((uint32_t)0x00000070U), /*!< backup domain control register offset */
    IDX_PLLADDCTL     = ((uint32_t)0x00000080U), /*!< PLL clock attached control register */
    IDX_ADDCTL0       = ((uint32_t)0x000000C0U), /*!< additional clock control register 0 offset */
    IDX_ADDCTL1       = ((uint32_t)0x000000C4U), /*!< additional clock control register 1 offset */
    IDX_RCU_USBCLKCTL = ((uint32_t)0x000000D4U), /*!< USB clock control register */
    /* clock stabilization and stuck interrupt */
    IDX_INT           = ((uint32_t)0x0000000CU), /*!< clock interrupt register offset */
    IDX_ADDINT        = ((uint32_t)0x000000CCU), /*!< additional clock interrupt register offset */
    /* configuration register */
    IDX_PLL0          = ((uint32_t)0x00000004U), /*!< PLL0 register offset */
    IDX_CFG0          = ((uint32_t)0x00000008U), /*!< clock configuration register 0 offset */
    IDX_PLL1          = ((uint32_t)0x00000084U), /*!< PLL1 register offset */
    IDX_PLL2          = ((uint32_t)0x00000088U), /*!< PLL2 register offset */
    IDX_CFG1          = ((uint32_t)0x0000008CU), /*!< clock configuration register 1 offset */
    IDX_CFG2          = ((uint32_t)0x00000090U), /*!< clock configuration register 2 offset */
    IDX_CFG3          = ((uint32_t)0x00000094U), /*!< clock configuration register 3 offset */
    IDX_PLLALL        = ((uint32_t)0x00000098U), /*!< PLL configuration register */
    IDX_PLL0FRA       = ((uint32_t)0x0000009CU), /*!< PLL0 fraction configuration register */
    IDX_PLL1FRA       = ((uint32_t)0x000000A0U), /*!< PLL1 fraction configuration register */
    IDX_PLL2FRA       = ((uint32_t)0x000000A4U), /*!< PLL2 fraction configuration register */
    IDX_CFG5          = ((uint32_t)0x000000F0U), /*!< CFG5 fraction configuration register */
    IDX_PLLUSBCFG     = ((uint32_t)0x000000D8U)  /*!< PLLUSB Clock configuration register Clock configuration register */
} hal_reg_idx_enum;

/* peripheral clock enable */
typedef enum {
    /* AHB1 peripherals */
    RCU_ENET1      = RCU_REGIDX_BIT(IDX_AHB1EN, 0U),  /*!< ENET1 clock */
    RCU_ENET1TX    = RCU_REGIDX_BIT(IDX_AHB1EN, 1U),  /*!< ENET1 TX clock */
    RCU_ENET1RX    = RCU_REGIDX_BIT(IDX_AHB1EN, 2U),  /*!< ENET1 RX clock */
    RCU_ENET1PTP   = RCU_REGIDX_BIT(IDX_AHB1EN, 3U),  /*!< ENET1 PTP clock */
    RCU_USBHS0     = RCU_REGIDX_BIT(IDX_AHB1EN, 14U), /*!< USBHS0 clock */
    RCU_USBHS0ULPI = RCU_REGIDX_BIT(IDX_AHB1EN, 15U), /*!< USBHS0 ULPI clock */
    RCU_DMA0       = RCU_REGIDX_BIT(IDX_AHB1EN, 21U), /*!< DMA0 clock */
    RCU_DMA1       = RCU_REGIDX_BIT(IDX_AHB1EN, 22U), /*!< DMA1 clock */
    RCU_DMAMUX     = RCU_REGIDX_BIT(IDX_AHB1EN, 23U), /*!< IPA clock */
    RCU_ENET0      = RCU_REGIDX_BIT(IDX_AHB1EN, 25U), /*!< ENET0 clock */
    RCU_ENET0TX    = RCU_REGIDX_BIT(IDX_AHB1EN, 26U), /*!< ENET0 TX clock */
    RCU_ENET0RX    = RCU_REGIDX_BIT(IDX_AHB1EN, 27U), /*!< ENET0 RX clock */
    RCU_ENET0PTP   = RCU_REGIDX_BIT(IDX_AHB1EN, 28U), /*!< ENET0 PTP clock */
    RCU_USBHS1     = RCU_REGIDX_BIT(IDX_AHB1EN, 29U), /*!< USBHS1 clock */
    RCU_USBHS1ULPI = RCU_REGIDX_BIT(IDX_AHB1EN, 30U), /*!< USBHS1 ULPI clock */
    /* AHB2 peripherals */
    RCU_DCI        = RCU_REGIDX_BIT(IDX_AHB2EN, 0U), /*!< DCI clock */
    RCU_FAC        = RCU_REGIDX_BIT(IDX_AHB2EN, 1U), /*!< FAC clock */
    RCU_SDIO1      = RCU_REGIDX_BIT(IDX_AHB2EN, 2U), /*!< SDIO1 clock */
    RCU_CAU        = RCU_REGIDX_BIT(IDX_AHB2EN, 3U), /*!< CAU clock */
    RCU_HAU        = RCU_REGIDX_BIT(IDX_AHB2EN, 4U), /*!< HAU clock */
    RCU_TRNG       = RCU_REGIDX_BIT(IDX_AHB2EN, 6U), /*!< TRNG clock */
    RCU_TMU        = RCU_REGIDX_BIT(IDX_AHB2EN, 7U), /*!< TMU clock */
    RCU_RAMECCMU1  = RCU_REGIDX_BIT(IDX_AHB2EN, 8U), /*!< RAMECCMU1 clock */
    /* AHB3 peripherals */
    RCU_EXMC       = RCU_REGIDX_BIT(IDX_AHB3EN, 0U),  /*!< EXMC clock */
    RCU_IPA        = RCU_REGIDX_BIT(IDX_AHB3EN, 1U),  /*!< IPA clock */
    RCU_SDIO0      = RCU_REGIDX_BIT(IDX_AHB3EN, 2U),  /*!< SDIO0 clock */
    RCU_MDMA       = RCU_REGIDX_BIT(IDX_AHB3EN, 3U),  /*!< MDMMA clock */
    RCU_OSPIM      = RCU_REGIDX_BIT(IDX_AHB3EN, 4U),  /*!< OSPIM clock */
    RCU_OSPI0      = RCU_REGIDX_BIT(IDX_AHB3EN, 5U),  /*!< OSPI0 clock */
    RCU_OSPI1      = RCU_REGIDX_BIT(IDX_AHB3EN, 6U),  /*!< OSPI1 clock */
    RCU_RTDEC0     = RCU_REGIDX_BIT(IDX_AHB3EN, 8U),  /*!< RTDEC0 clock */
    RCU_RTDEC1     = RCU_REGIDX_BIT(IDX_AHB3EN, 9U),  /*!< RTDEC1 clock */
    RCU_RAMECCMU0  = RCU_REGIDX_BIT(IDX_AHB3EN, 10U), /*!< RAMECCMU0 clock */
    RCU_CPU        = RCU_REGIDX_BIT(IDX_AHB3EN, 15U), /*!< CPU clock */
    /* AHB4 peripherals */
    RCU_GPIOA      = RCU_REGIDX_BIT(IDX_AHB4EN, 0U),  /*!< GPIOA clock */
    RCU_GPIOB      = RCU_REGIDX_BIT(IDX_AHB4EN, 1U),  /*!< GPIOB clock */
    RCU_GPIOC      = RCU_REGIDX_BIT(IDX_AHB4EN, 2U),  /*!< GPIOC clock */
    RCU_GPIOD      = RCU_REGIDX_BIT(IDX_AHB4EN, 3U),  /*!< GPIOD clock */
    RCU_GPIOE      = RCU_REGIDX_BIT(IDX_AHB4EN, 4U),  /*!< GPIOE clock */
    RCU_GPIOF      = RCU_REGIDX_BIT(IDX_AHB4EN, 5U),  /*!< GPIOF clock */
    RCU_GPIOG      = RCU_REGIDX_BIT(IDX_AHB4EN, 6U),  /*!< GPIOG clock */
    RCU_GPIOH      = RCU_REGIDX_BIT(IDX_AHB4EN, 7U),  /*!< GPIOH clock */
    RCU_GPIOJ      = RCU_REGIDX_BIT(IDX_AHB4EN, 8U),  /*!< GPIOJ clock */
    RCU_GPIOK      = RCU_REGIDX_BIT(IDX_AHB4EN, 9U),  /*!< GPIOK clock */
    RCU_BKPSRAM    = RCU_REGIDX_BIT(IDX_AHB4EN, 13U), /*!< BKPSRAM clock */
    RCU_CRC        = RCU_REGIDX_BIT(IDX_AHB4EN, 14U), /*!< CRC clock */
    RCU_HWSEM      = RCU_REGIDX_BIT(IDX_AHB4EN, 15U), /*!< HWSEM  clock */
    /* APB1 peripherals */
    RCU_TIMER1     = RCU_REGIDX_BIT(IDX_APB1EN, 0U),  /*!< TIMER1 clock */
    RCU_TIMER2     = RCU_REGIDX_BIT(IDX_APB1EN, 1U),  /*!< TIMER2 clock */
    RCU_TIMER3     = RCU_REGIDX_BIT(IDX_APB1EN, 2U),  /*!< TIMER3 clock */
    RCU_TIMER4     = RCU_REGIDX_BIT(IDX_APB1EN, 3U),  /*!< TIMER4 clock */
    RCU_TIMER5     = RCU_REGIDX_BIT(IDX_APB1EN, 4U),  /*!< TIMER5 clock */
    RCU_TIMER6     = RCU_REGIDX_BIT(IDX_APB1EN, 5U),  /*!< TIMER6 clock */
    RCU_TIMER22    = RCU_REGIDX_BIT(IDX_APB1EN, 6U),  /*!< TIMER22 clock */
    RCU_TIMER23    = RCU_REGIDX_BIT(IDX_APB1EN, 7U),  /*!< TIMER23 clock */
    RCU_TIMER30    = RCU_REGIDX_BIT(IDX_APB1EN, 8U),  /*!< TIMER30 clock */
    RCU_TIMER31    = RCU_REGIDX_BIT(IDX_APB1EN, 9U),  /*!< TIMER31 clock */
    RCU_TIMER50    = RCU_REGIDX_BIT(IDX_APB1EN, 10U), /*!< TIMER50 clock */
    RCU_TIMER51    = RCU_REGIDX_BIT(IDX_APB1EN, 11U), /*!< TIMER51 clock */
    RCU_RSPDIF     = RCU_REGIDX_BIT(IDX_APB1EN, 13U), /*!< RSPDIF clock */
    RCU_SPI1       = RCU_REGIDX_BIT(IDX_APB1EN, 14U), /*!< SPI1 clock */
    RCU_SPI2       = RCU_REGIDX_BIT(IDX_APB1EN, 15U), /*!< SPI2 clock */
    RCU_MDIO       = RCU_REGIDX_BIT(IDX_APB1EN, 16U), /*!< MDIO clock */
    RCU_USART1     = RCU_REGIDX_BIT(IDX_APB1EN, 17U), /*!< USART1 clock */
    RCU_USART2     = RCU_REGIDX_BIT(IDX_APB1EN, 18U), /*!< USART2 clock */
    RCU_UART3      = RCU_REGIDX_BIT(IDX_APB1EN, 19U), /*!< UART3 clock */
    RCU_UART4      = RCU_REGIDX_BIT(IDX_APB1EN, 20U), /*!< UART4 clock */
    RCU_I2C0       = RCU_REGIDX_BIT(IDX_APB1EN, 21U), /*!< I2C0 clock */
    RCU_I2C1       = RCU_REGIDX_BIT(IDX_APB1EN, 22U), /*!< I2C1 clock */
    RCU_I2C2       = RCU_REGIDX_BIT(IDX_APB1EN, 23U), /*!< I2C2 clock */
    RCU_I2C3       = RCU_REGIDX_BIT(IDX_APB1EN, 24U), /*!< I2C3 clock */
    RCU_CTC        = RCU_REGIDX_BIT(IDX_APB1EN, 27U), /*!< CTC clock */
    RCU_DACHOLD    = RCU_REGIDX_BIT(IDX_APB1EN, 28U), /*!< DACHOLD clock */
    RCU_DAC        = RCU_REGIDX_BIT(IDX_APB1EN, 29U), /*!< DAC clock */
    RCU_UART6      = RCU_REGIDX_BIT(IDX_APB1EN, 30U), /*!< UART6 clock */
    RCU_UART7      = RCU_REGIDX_BIT(IDX_APB1EN, 31U), /*!< UART7 clock */
    /* APB2 peripherals */
    RCU_TIMER0     = RCU_REGIDX_BIT(IDX_APB2EN, 0U),  /*!< TIMER0 clock */
    RCU_TIMER7     = RCU_REGIDX_BIT(IDX_APB2EN, 1U),  /*!< TIMER7 clock */
    RCU_USART0     = RCU_REGIDX_BIT(IDX_APB2EN, 4U),  /*!< USART0 clock */
    RCU_USART5     = RCU_REGIDX_BIT(IDX_APB2EN, 5U),  /*!< USART5 clock */
    RCU_ADC0       = RCU_REGIDX_BIT(IDX_APB2EN, 8U),  /*!< ADC0 clock */
    RCU_ADC1       = RCU_REGIDX_BIT(IDX_APB2EN, 9U),  /*!< ADC1 clock */
    RCU_ADC2       = RCU_REGIDX_BIT(IDX_APB2EN, 10U), /*!< ADC2 clock */
    RCU_SPI0       = RCU_REGIDX_BIT(IDX_APB2EN, 12U), /*!< SPI0 clock */
    RCU_SPI3       = RCU_REGIDX_BIT(IDX_APB2EN, 13U), /*!< SPI3 clock */
    RCU_TIMER14    = RCU_REGIDX_BIT(IDX_APB2EN, 16U), /*!< TIMER14 clock */
    RCU_TIMER15    = RCU_REGIDX_BIT(IDX_APB2EN, 17U), /*!< TIMER15 clock */
    RCU_TIMER16    = RCU_REGIDX_BIT(IDX_APB2EN, 18U), /*!< TIMER16 clock */
    RCU_HPDF       = RCU_REGIDX_BIT(IDX_APB2EN, 19U), /*!< HPDF clock */
    RCU_SPI4       = RCU_REGIDX_BIT(IDX_APB2EN, 20U), /*!< SPI4 clock */
    RCU_SPI5       = RCU_REGIDX_BIT(IDX_APB2EN, 21U), /*!< SPI5 clock */
    RCU_SAI0       = RCU_REGIDX_BIT(IDX_APB2EN, 22U), /*!< SAI0 clock */
    RCU_SAI1       = RCU_REGIDX_BIT(IDX_APB2EN, 23U), /*!< SAI1 clock */
    RCU_SAI2       = RCU_REGIDX_BIT(IDX_APB2EN, 24U), /*!< SAI2 clock */
    RCU_TIMER40    = RCU_REGIDX_BIT(IDX_APB2EN, 25U), /*!< TIMER40 clock */
    RCU_TIMER41    = RCU_REGIDX_BIT(IDX_APB2EN, 26U), /*!< TIMER41 clock */
    RCU_TIMER42    = RCU_REGIDX_BIT(IDX_APB2EN, 27U), /*!< TIMER42 clock */
    RCU_TIMER43    = RCU_REGIDX_BIT(IDX_APB2EN, 28U), /*!< TIMER43 clock */
    RCU_TIMER44    = RCU_REGIDX_BIT(IDX_APB2EN, 29U), /*!< TIMER44 clock */
    RCU_EDOUT      = RCU_REGIDX_BIT(IDX_APB2EN, 30U), /*!< EDOUT clock */
    RCU_TRIGSEL    = RCU_REGIDX_BIT(IDX_APB2EN, 31U), /*!< TRIGSEL clock */
    /* APB3 peripherals */
    RCU_TLI        = RCU_REGIDX_BIT(IDX_APB3EN, 0U), /*!< TLI clock */
    RCU_WWDGT      = RCU_REGIDX_BIT(IDX_APB3EN, 1U), /*!< WWDGT clock */
    /* APB4 peripherals */
    RCU_SYSCFG     = RCU_REGIDX_BIT(IDX_APB4EN, 0U), /*!< SYSCFG clock */
    RCU_CMP        = RCU_REGIDX_BIT(IDX_APB4EN, 1U), /*!< CMP clock */
    RCU_VREF       = RCU_REGIDX_BIT(IDX_APB4EN, 2U), /*!< VREF clock */
    RCU_LPDTS      = RCU_REGIDX_BIT(IDX_APB4EN, 3U), /*!< LPDTS clock */
    RCU_PMU        = RCU_REGIDX_BIT(IDX_APB4EN, 4U), /*!< PMU clock */
    /* APB2 additional peripherals */
    RCU_CAN0       = RCU_REGIDX_BIT(IDX_ADD_APB2EN, 0U), /*!< CAN0 clock */
    RCU_CAN1       = RCU_REGIDX_BIT(IDX_ADD_APB2EN, 1U), /*!< CAN1 clock */
    RCU_CAN2       = RCU_REGIDX_BIT(IDX_ADD_APB2EN, 2U), /*!< CAN2 clock */
    /* HXTAL/LXTAL/IRC32K peripherals */
    RCU_RTC        = RCU_REGIDX_BIT(IDX_BDCTL, 15U) /*!< RTC clock */
} hal_rcu_periph_enum;

/* peripheral clock enable when sleep mode*/
typedef enum {
    /* AHB1 peripherals */
    RCU_ENET1_SLP      = RCU_REGIDX_BIT(IDX_AHB1SPEN, 0U),  /*!< ENET1 clock when sleep mode */
    RCU_ENET1TX_SLP    = RCU_REGIDX_BIT(IDX_AHB1SPEN, 1U),  /*!< ENET1 TX clock when sleep mode */
    RCU_ENET1RX_SLP    = RCU_REGIDX_BIT(IDX_AHB1SPEN, 2U),  /*!< ENET1 RX clock when sleep mode */
    RCU_ENET1PTP_SLP   = RCU_REGIDX_BIT(IDX_AHB1SPEN, 3U),  /*!< ENET1 PTP clock when sleep mode */
    RCU_USBHS0_SLP     = RCU_REGIDX_BIT(IDX_AHB1SPEN, 14U), /*!< USBHS0 clock when sleep mode */
    RCU_USBHS0ULPI_SLP = RCU_REGIDX_BIT(IDX_AHB1SPEN, 15U), /*!< USBHS0ULPI clock when sleep mode */
    RCU_SRAM0_SLP      = RCU_REGIDX_BIT(IDX_AHB1SPEN, 16U), /*!< SRAM0 clock when sleep mode */
    RCU_SRAM1_SLP      = RCU_REGIDX_BIT(IDX_AHB1SPEN, 17U), /*!< SRAM1 clock when sleep mode */
    RCU_DMA0_SLP       = RCU_REGIDX_BIT(IDX_AHB1SPEN, 21U), /*!< DMA0 clock when sleep mode */
    RCU_DMA1_SLP       = RCU_REGIDX_BIT(IDX_AHB1SPEN, 22U), /*!< DMA1 clock when sleep mode */
    RCU_DMAMUX_SLP     = RCU_REGIDX_BIT(IDX_AHB1SPEN, 23U), /*!< DMAMUX clock when sleep mode */
    RCU_ENET0_SLP      = RCU_REGIDX_BIT(IDX_AHB1SPEN, 25U), /*!< ENET0 clock when sleep mode */
    RCU_ENET0TX_SLP    = RCU_REGIDX_BIT(IDX_AHB1SPEN, 26U), /*!< ENET0 TX clock when sleep mode */
    RCU_ENET0RX_SLP    = RCU_REGIDX_BIT(IDX_AHB1SPEN, 27U), /*!< ENET0 RX clock when sleep mode */
    RCU_ENET0PTP_SLP   = RCU_REGIDX_BIT(IDX_AHB1SPEN, 28U), /*!< ENET0 PTP clock when sleep mode */
    RCU_USBHS1_SLP     = RCU_REGIDX_BIT(IDX_AHB1SPEN, 29U), /*!< USBHS1 clock when sleep mode */
    RCU_USBHS1ULPI_SLP = RCU_REGIDX_BIT(IDX_AHB1SPEN, 30U), /*!< USBHS1ULPI clock when sleep mode */
    /* AHB2 peripherals */
    RCU_DCI_SLP       = RCU_REGIDX_BIT(IDX_AHB2SPEN, 0U), /*!< DCI clock when sleep mode */
    RCU_FAC_SLP       = RCU_REGIDX_BIT(IDX_AHB2SPEN, 1U), /*!< FAC clock when sleep mode */
    RCU_SDIO1_SLP     = RCU_REGIDX_BIT(IDX_AHB2SPEN, 2U), /*!< SDIO1 clock when sleep mode */
    RCU_CAU_SLP       = RCU_REGIDX_BIT(IDX_AHB2SPEN, 3U), /*!< CAU clock when sleep mode */
    RCU_HAU_SLP       = RCU_REGIDX_BIT(IDX_AHB2SPEN, 4U), /*!< HAU clock when sleep mode */
    RCU_TRNG_SLP      = RCU_REGIDX_BIT(IDX_AHB2SPEN, 6U), /*!< TRNG clock when sleep mode */
    RCU_TMU_SLP       = RCU_REGIDX_BIT(IDX_AHB2SPEN, 7U), /*!< TMU clock when sleep mode */
    RCU_RAMECCMU1_SLP = RCU_REGIDX_BIT(IDX_AHB2SPEN, 8U), /*!< RAMECCMU1 clock when sleep mode */
    /* AHB3 peripherals */
    RCU_EXMC_SLP      = RCU_REGIDX_BIT(IDX_AHB3SPEN, 0U),  /*!< EXMC clock when sleep mode */
    RCU_IPA_SLP       = RCU_REGIDX_BIT(IDX_AHB3SPEN, 1U),  /*!< IPA clock when sleep mode */
    RCU_SDIO0_SLP     = RCU_REGIDX_BIT(IDX_AHB3SPEN, 2U),  /*!< SDIO0 clock when sleep mode */
    RCU_MDMA_SLP      = RCU_REGIDX_BIT(IDX_AHB3SPEN, 3U),  /*!< MDMMA clock when sleep mode */
    RCU_OSPIM_SLP     = RCU_REGIDX_BIT(IDX_AHB3SPEN, 4U),  /*!< OSPIM clock when sleep mode */
    RCU_OSPI0_SLP     = RCU_REGIDX_BIT(IDX_AHB3SPEN, 5U),  /*!< OSPI0 clock when sleep mode */
    RCU_OSPI1_SLP     = RCU_REGIDX_BIT(IDX_AHB3SPEN, 6U),  /*!< OSPI1 clock when sleep mode */
    RCU_RTDEC0_SLP    = RCU_REGIDX_BIT(IDX_AHB3SPEN, 8U),  /*!< RTDEC0 clock when sleep mode */
    RCU_RTDEC1_SLP    = RCU_REGIDX_BIT(IDX_AHB3SPEN, 9U),  /*!< RTDEC1 clock when sleep mode */
    RCU_RAMECCMU0_SLP = RCU_REGIDX_BIT(IDX_AHB3SPEN, 10U), /*!< RAMECCMU0 clock when sleep mode */
    RCU_AXISRAM_SLP   = RCU_REGIDX_BIT(IDX_AHB3SPEN, 14U), /*!< AXISRAM clock when sleep mode */
    RCU_FMC_SLP       = RCU_REGIDX_BIT(IDX_AHB3SPEN, 15U), /*!< FMC clock when sleep mode */
    /* AHB4 peripherals */
    RCU_GPIOA_SLP   = RCU_REGIDX_BIT(IDX_AHB4SPEN, 0U),  /*!< GPIOA clock when sleep mode */
    RCU_GPIOB_SLP   = RCU_REGIDX_BIT(IDX_AHB4SPEN, 1U),  /*!< GPIOB clock when sleep mode */
    RCU_GPIOC_SLP   = RCU_REGIDX_BIT(IDX_AHB4SPEN, 2U),  /*!< GPIOC clock when sleep mode */
    RCU_GPIOD_SLP   = RCU_REGIDX_BIT(IDX_AHB4SPEN, 3U),  /*!< GPIOD clock when sleep mode */
    RCU_GPIOE_SLP   = RCU_REGIDX_BIT(IDX_AHB4SPEN, 4U),  /*!< GPIOE clock when sleep mode */
    RCU_GPIOF_SLP   = RCU_REGIDX_BIT(IDX_AHB4SPEN, 5U),  /*!< GPIOF clock when sleep mode */
    RCU_GPIOG_SLP   = RCU_REGIDX_BIT(IDX_AHB4SPEN, 6U),  /*!< GPIOG clock when sleep mode */
    RCU_GPIOH_SLP   = RCU_REGIDX_BIT(IDX_AHB4SPEN, 7U),  /*!< GPIOH clock when sleep mode */
    RCU_GPIOJ_SLP   = RCU_REGIDX_BIT(IDX_AHB4SPEN, 8U),  /*!< GPIOJ clock when sleep mode */
    RCU_GPIOK_SLP   = RCU_REGIDX_BIT(IDX_AHB4SPEN, 9U),  /*!< GPIOK clock when sleep mode */
    RCU_BKPSRAM_SLP = RCU_REGIDX_BIT(IDX_AHB4SPEN, 13U), /*!< BKPSRAM clock when sleep mode */
    RCU_CRC_SLP     = RCU_REGIDX_BIT(IDX_AHB4SPEN, 14U), /*!< CRC clock when sleep mode */
    /* APB1 peripherals */
    RCU_TIMER1_SLP  = RCU_REGIDX_BIT(IDX_APB1SPEN, 0U),  /*!< TIMER1 clock when sleep mode */
    RCU_TIMER2_SLP  = RCU_REGIDX_BIT(IDX_APB1SPEN, 1U),  /*!< TIMER2 clock when sleep mode */
    RCU_TIMER3_SLP  = RCU_REGIDX_BIT(IDX_APB1SPEN, 2U),  /*!< TIMER3 clock when sleep mode */
    RCU_TIMER4_SLP  = RCU_REGIDX_BIT(IDX_APB1SPEN, 3U),  /*!< TIMER4 clock when sleep mode */
    RCU_TIMER5_SLP  = RCU_REGIDX_BIT(IDX_APB1SPEN, 4U),  /*!< TIMER5 clock when sleep mode */
    RCU_TIMER6_SLP  = RCU_REGIDX_BIT(IDX_APB1SPEN, 5U),  /*!< TIMER6 clock when sleep mode */
    RCU_TIMER22_SLP = RCU_REGIDX_BIT(IDX_APB1SPEN, 6U),  /*!< TIMER22 clock when sleep mode */
    RCU_TIMER23_SLP = RCU_REGIDX_BIT(IDX_APB1SPEN, 7U),  /*!< TIMER23 clock when sleep mode */
    RCU_TIMER30_SLP = RCU_REGIDX_BIT(IDX_APB1SPEN, 8U),  /*!< TIMER30 clock when sleep mode */
    RCU_TIMER31_SLP = RCU_REGIDX_BIT(IDX_APB1SPEN, 9U),  /*!< TIMER31 clock when sleep mode */
    RCU_TIMER50_SLP = RCU_REGIDX_BIT(IDX_APB1SPEN, 10U), /*!< TIMER50 clock when sleep mode */
    RCU_TIMER51_SLP = RCU_REGIDX_BIT(IDX_APB1SPEN, 11U), /*!< TIMER51 clock when sleep mode */
    RCU_RSPDIF_SLP  = RCU_REGIDX_BIT(IDX_APB1SPEN, 13U), /*!< RSPDIF clock when sleep mode */
    RCU_SPI1_SLP    = RCU_REGIDX_BIT(IDX_APB1SPEN, 14U), /*!< SPI1 clock when sleep mode */
    RCU_SPI2_SLP    = RCU_REGIDX_BIT(IDX_APB1SPEN, 15U), /*!< SPI2 clock when sleep mode */
    RCU_MDIO_SLP    = RCU_REGIDX_BIT(IDX_APB1SPEN, 16U), /*!< MDIO clock when sleep mode */
    RCU_USART1_SLP  = RCU_REGIDX_BIT(IDX_APB1SPEN, 17U), /*!< USART1 clock when sleep mode */
    RCU_USART2_SLP  = RCU_REGIDX_BIT(IDX_APB1SPEN, 18U), /*!< USART2 clock when sleep mode */
    RCU_UART3_SLP   = RCU_REGIDX_BIT(IDX_APB1SPEN, 19U), /*!< UART3 clock when sleep mode */
    RCU_UART4_SLP   = RCU_REGIDX_BIT(IDX_APB1SPEN, 20U), /*!< UART4 clock when sleep mode */
    RCU_I2C0_SLP    = RCU_REGIDX_BIT(IDX_APB1SPEN, 21U), /*!< I2C0 clock when sleep mode */
    RCU_I2C1_SLP    = RCU_REGIDX_BIT(IDX_APB1SPEN, 22U), /*!< I2C1 clock when sleep mode */
    RCU_I2C2_SLP    = RCU_REGIDX_BIT(IDX_APB1SPEN, 23U), /*!< I2C2 clock when sleep mode */
    RCU_I2C3_SLP    = RCU_REGIDX_BIT(IDX_APB1SPEN, 24U), /*!< I2C3 clock when sleep mode */
    RCU_CTC_SLP     = RCU_REGIDX_BIT(IDX_APB1SPEN, 27U), /*!< CTC clock when sleep mode */
    RCU_DACHOLD_SLP = RCU_REGIDX_BIT(IDX_APB1SPEN, 28U), /*!< DACHOLD clock when sleep mode */
    RCU_DAC_SLP     = RCU_REGIDX_BIT(IDX_APB1SPEN, 29U), /*!< DAC clock when sleep mode */
    RCU_UART6_SLP   = RCU_REGIDX_BIT(IDX_APB1SPEN, 30U), /*!< UART6 clock when sleep mode */
    RCU_UART7_SLP   = RCU_REGIDX_BIT(IDX_APB1SPEN, 31U), /*!< UART7 clock when sleep mode */
    /* APB2 peripherals */
    RCU_TIMER0_SLP  = RCU_REGIDX_BIT(IDX_APB2SPEN, 0U),  /*!< TIMER0 clock when sleep mode */
    RCU_TIMER7_SLP  = RCU_REGIDX_BIT(IDX_APB2SPEN, 1U),  /*!< TIMER7 clock when sleep mode */
    RCU_USART0_SLP  = RCU_REGIDX_BIT(IDX_APB2SPEN, 4U),  /*!< USART0 clock when sleep mode */
    RCU_USART5_SLP  = RCU_REGIDX_BIT(IDX_APB2SPEN, 5U),  /*!< USART5 clock when sleep mode */
    RCU_ADC0_SLP    = RCU_REGIDX_BIT(IDX_APB2SPEN, 8U),  /*!< ADC0 clock when sleep mode */
    RCU_ADC1_SLP    = RCU_REGIDX_BIT(IDX_APB2SPEN, 9U),  /*!< ADC1 clock when sleep mode */
    RCU_ADC2_SLP    = RCU_REGIDX_BIT(IDX_APB2SPEN, 10U), /*!< ADC2 clock when sleep mode */
    RCU_SPI0_SLP    = RCU_REGIDX_BIT(IDX_APB2SPEN, 12U), /*!< SPI0 clock when sleep mode */
    RCU_SPI3_SLP    = RCU_REGIDX_BIT(IDX_APB2SPEN, 13U), /*!< SPI3 clock when sleep mode */
    RCU_TIMER14_SLP = RCU_REGIDX_BIT(IDX_APB2SPEN, 16U), /*!< TIMER14 clock when sleep mode */
    RCU_TIMER15_SLP = RCU_REGIDX_BIT(IDX_APB2SPEN, 17U), /*!< TIMER15 clock when sleep mode */
    RCU_TIMER16_SLP = RCU_REGIDX_BIT(IDX_APB2SPEN, 18U), /*!< TIMER16 clock when sleep mode */
    RCU_HPDF_SLP    = RCU_REGIDX_BIT(IDX_APB2SPEN, 19U), /*!< HPDF clock when sleep mode */
    RCU_SPI4_SLP    = RCU_REGIDX_BIT(IDX_APB2SPEN, 20U), /*!< SPI4 clock when sleep mode */
    RCU_SPI5_SLP    = RCU_REGIDX_BIT(IDX_APB2SPEN, 21U), /*!< SPI5 clock when sleep mode */
    RCU_SAI0_SLP    = RCU_REGIDX_BIT(IDX_APB2SPEN, 22U), /*!< SAI0 clock when sleep mode */
    RCU_SAI1_SLP    = RCU_REGIDX_BIT(IDX_APB2SPEN, 23U), /*!< SAI1 clock when sleep mode */
    RCU_SAI2_SLP    = RCU_REGIDX_BIT(IDX_APB2SPEN, 24U), /*!< SAI2 clock when sleep mode */
    RCU_TIMER40_SLP = RCU_REGIDX_BIT(IDX_APB2SPEN, 25U), /*!< TIMER40 clock when sleep mode */
    RCU_TIMER41_SLP = RCU_REGIDX_BIT(IDX_APB2SPEN, 26U), /*!< TIMER41 clock when sleep mode */
    RCU_TIMER42_SLP = RCU_REGIDX_BIT(IDX_APB2SPEN, 27U), /*!< TIMER42 clock when sleep mode */
    RCU_TIMER43_SLP = RCU_REGIDX_BIT(IDX_APB2SPEN, 28U), /*!< TIMER43 clock when sleep mode */
    RCU_TIMER44_SLP = RCU_REGIDX_BIT(IDX_APB2SPEN, 29U), /*!< TIMER44 clock when sleep mode */
    RCU_EDOUT_SLP   = RCU_REGIDX_BIT(IDX_APB2SPEN, 30U), /*!< EDOUT clock when sleep mode */
    RCU_TRIGSEL_SLP = RCU_REGIDX_BIT(IDX_APB2SPEN, 31U), /*!< TRIGSEL clock when sleep mode */
    /* APB3 peripherals */
    RCU_TLI_SLP   = RCU_REGIDX_BIT(IDX_APB3SPEN, 0U), /*!< TLI clock when sleep mode */
    RCU_WWDGT_SLP = RCU_REGIDX_BIT(IDX_APB3SPEN, 1U), /*!< WWDGT clock when sleep mode */
    /* APB4 peripherals */
    RCU_SYSCFG_SLP = RCU_REGIDX_BIT(IDX_APB4SPEN, 0U), /*!< SYSCFG clock when sleep mode */
    RCU_CMP_SLP    = RCU_REGIDX_BIT(IDX_APB4SPEN, 1U), /*!< CMP clock when sleep mode */
    RCU_VREF_SLP   = RCU_REGIDX_BIT(IDX_APB4SPEN, 2U), /*!< VREF clock when sleep mode */
    RCU_LPDTS_SLP  = RCU_REGIDX_BIT(IDX_APB4SPEN, 3U), /*!< LPDTS clock when sleep mode */
    RCU_PMU_SLP    = RCU_REGIDX_BIT(IDX_APB4SPEN, 4U), /*!< PMU clock when sleep mode */
    /* APB2 additional peripherals */
    RCU_CAN0_SLP = RCU_REGIDX_BIT(IDX_ADD_APB2SPEN, 0U), /*!< CAN0 clock when sleep mode */
    RCU_CAN1_SLP = RCU_REGIDX_BIT(IDX_ADD_APB2SPEN, 1U), /*!< CAN1 clock when sleep mode */
    RCU_CAN2_SLP = RCU_REGIDX_BIT(IDX_ADD_APB2SPEN, 2U)  /*!< CAN2 clock when sleep mode */
} hal_rcu_periph_sleep_enum;

/* peripherals reset */
typedef enum {
    /* AHB1 peripherals */
    RCU_ENET1RST  = RCU_REGIDX_BIT(IDX_AHB1RST, 0U),  /*!< ENET1 clock reset */
    RCU_USBHS0RST = RCU_REGIDX_BIT(IDX_AHB1RST, 14U), /*!< USBHS0 clock reset */
    RCU_DMA0RST   = RCU_REGIDX_BIT(IDX_AHB1RST, 21U), /*!< DMA0 clock reset */
    RCU_DMA1RST   = RCU_REGIDX_BIT(IDX_AHB1RST, 22U), /*!< DMA1 clock reset */
    RCU_DMAMUXRST = RCU_REGIDX_BIT(IDX_AHB1RST, 23U), /*!< DMAMUX clock reset */
    RCU_ENET0RST  = RCU_REGIDX_BIT(IDX_AHB1RST, 25U), /*!< ENET clock reset */
    RCU_USBHS1RST = RCU_REGIDX_BIT(IDX_AHB1RST, 29U), /*!< USBHS1HS clock reset */
    /* AHB2 peripherals */
    RCU_DCIRST   = RCU_REGIDX_BIT(IDX_AHB2RST, 0U), /*!< DCI clock reset */
    RCU_FACRST   = RCU_REGIDX_BIT(IDX_AHB2RST, 1U), /*!< FAC clock reset */
    RCU_SDIO1RST = RCU_REGIDX_BIT(IDX_AHB2RST, 2U), /*!< SDIO1 clock reset */
    RCU_CAURST   = RCU_REGIDX_BIT(IDX_AHB2RST, 3U), /*!< CAU clock reset */
    RCU_HAURST   = RCU_REGIDX_BIT(IDX_AHB2RST, 4U), /*!< HAU clock reset */
    RCU_TRNGRST  = RCU_REGIDX_BIT(IDX_AHB2RST, 6U), /*!< TRNG clock reset */
    RCU_TMURST   = RCU_REGIDX_BIT(IDX_AHB2RST, 7U), /*!< TMU clock reset */
    /* AHB3 peripherals */
    RCU_EXMCRST   = RCU_REGIDX_BIT(IDX_AHB3RST, 0U), /*!< EXMC clock reset */
    RCU_IPARST    = RCU_REGIDX_BIT(IDX_AHB3RST, 1U), /*!< IPA clock reset */
    RCU_SDIO0RST  = RCU_REGIDX_BIT(IDX_AHB3RST, 2U), /*!< SDIO0 clock reset */
    RCU_MDMARST   = RCU_REGIDX_BIT(IDX_AHB3RST, 3U), /*!< MDMMA clock reset */
    RCU_OSPIMRST  = RCU_REGIDX_BIT(IDX_AHB3RST, 4U), /*!< OSPIM clock reset */
    RCU_OSPI0RST  = RCU_REGIDX_BIT(IDX_AHB3RST, 5U), /*!< OSPI0 clock reset */
    RCU_OSPI1RST  = RCU_REGIDX_BIT(IDX_AHB3RST, 6U), /*!< OSPI1 clock reset */
    RCU_RTDEC0RST = RCU_REGIDX_BIT(IDX_AHB3RST, 8U), /*!< RTDEC0 clock reset */
    RCU_RTDEC1RST = RCU_REGIDX_BIT(IDX_AHB3RST, 9U), /*!< RTDEC1 clock reset */
    /* AHB4 peripherals */
    RCU_GPIOARST = RCU_REGIDX_BIT(IDX_AHB4RST, 0U),  /*!< GPIOA clock reset */
    RCU_GPIOBRST = RCU_REGIDX_BIT(IDX_AHB4RST, 1U),  /*!< GPIOB clock reset */
    RCU_GPIOCRST = RCU_REGIDX_BIT(IDX_AHB4RST, 2U),  /*!< GPIOC clock reset */
    RCU_GPIODRST = RCU_REGIDX_BIT(IDX_AHB4RST, 3U),  /*!< GPIOD clock reset */
    RCU_GPIOERST = RCU_REGIDX_BIT(IDX_AHB4RST, 4U),  /*!< GPIOE clock reset */
    RCU_GPIOFRST = RCU_REGIDX_BIT(IDX_AHB4RST, 5U),  /*!< GPIOF clock reset */
    RCU_GPIOGRST = RCU_REGIDX_BIT(IDX_AHB4RST, 6U),  /*!< GPIOG clock reset */
    RCU_GPIOHRST = RCU_REGIDX_BIT(IDX_AHB4RST, 7U),  /*!< GPIOH clock reset */
    RCU_GPIOJRST = RCU_REGIDX_BIT(IDX_AHB4RST, 8U),  /*!< GPIOJ clock reset */
    RCU_GPIOKRST = RCU_REGIDX_BIT(IDX_AHB4RST, 9U),  /*!< GPIOK clock reset */
    RCU_CRCRST   = RCU_REGIDX_BIT(IDX_AHB4RST, 14U), /*!< CRC clock reset */
    RCU_HWSEMRST = RCU_REGIDX_BIT(IDX_AHB4RST, 15U), /*!< HWSEM clock reset */
    /* APB1 peripherals */
    RCU_TIMER1RST  = RCU_REGIDX_BIT(IDX_APB1RST, 0U),  /*!< TIMER1 clock reset */
    RCU_TIMER2RST  = RCU_REGIDX_BIT(IDX_APB1RST, 1U),  /*!< TIMER2 clock reset */
    RCU_TIMER3RST  = RCU_REGIDX_BIT(IDX_APB1RST, 2U),  /*!< TIMER3 clock reset */
    RCU_TIMER4RST  = RCU_REGIDX_BIT(IDX_APB1RST, 3U),  /*!< TIMER4 clock reset */
    RCU_TIMER5RST  = RCU_REGIDX_BIT(IDX_APB1RST, 4U),  /*!< TIMER5 clock reset */
    RCU_TIMER6RST  = RCU_REGIDX_BIT(IDX_APB1RST, 5U),  /*!< TIMER6 clock reset */
    RCU_TIMER22RST = RCU_REGIDX_BIT(IDX_APB1RST, 6U),  /*!< TIMER22 clock reset */
    RCU_TIMER23RST = RCU_REGIDX_BIT(IDX_APB1RST, 7U),  /*!< TIMER23 clock reset */
    RCU_TIMER30RST = RCU_REGIDX_BIT(IDX_APB1RST, 8U),  /*!< TIMER30 clock reset */
    RCU_TIMER31RST = RCU_REGIDX_BIT(IDX_APB1RST, 9U),  /*!< TIMER31 clock reset */
    RCU_TIMER50RST = RCU_REGIDX_BIT(IDX_APB1RST, 10U), /*!< TIMER50 clock reset */
    RCU_TIMER51RST = RCU_REGIDX_BIT(IDX_APB1RST, 11U), /*!< TIMER51 clock reset */
    RCU_RSPDIFRST  = RCU_REGIDX_BIT(IDX_APB1RST, 13U), /*!< RSPDIF clock reset */
    RCU_SPI1RST    = RCU_REGIDX_BIT(IDX_APB1RST, 14U), /*!< SPI1 clock reset */
    RCU_SPI2RST    = RCU_REGIDX_BIT(IDX_APB1RST, 15U), /*!< SPI2 clock reset */
    RCU_MDIORST    = RCU_REGIDX_BIT(IDX_APB1RST, 16U), /*!< MDIO clock reset */
    RCU_USART1RST  = RCU_REGIDX_BIT(IDX_APB1RST, 17U), /*!< USART1 clock reset */
    RCU_USART2RST  = RCU_REGIDX_BIT(IDX_APB1RST, 18U), /*!< USART2 clock reset */
    RCU_UART3RST   = RCU_REGIDX_BIT(IDX_APB1RST, 19U), /*!< UART3 clock reset */
    RCU_UART4RST   = RCU_REGIDX_BIT(IDX_APB1RST, 20U), /*!< UART4 clock reset */
    RCU_I2C0RST    = RCU_REGIDX_BIT(IDX_APB1RST, 21U), /*!< I2C0 clock reset */
    RCU_I2C1RST    = RCU_REGIDX_BIT(IDX_APB1RST, 22U), /*!< I2C1 clock reset */
    RCU_I2C2RST    = RCU_REGIDX_BIT(IDX_APB1RST, 23U), /*!< I2C2 clock reset */
    RCU_I2C3RST    = RCU_REGIDX_BIT(IDX_APB1RST, 24U), /*!< I2C3 clock reset */
    RCU_CTCRST     = RCU_REGIDX_BIT(IDX_APB1RST, 27U), /*!< CTC clock reset */
    RCU_DACHOLDRST = RCU_REGIDX_BIT(IDX_APB1RST, 28U), /*!< DACHOLD clock reset */
    RCU_DACRST     = RCU_REGIDX_BIT(IDX_APB1RST, 29U), /*!< DAC clock reset */
    RCU_UART6RST   = RCU_REGIDX_BIT(IDX_APB1RST, 30U), /*!< UART6 clock reset */
    RCU_UART7RST   = RCU_REGIDX_BIT(IDX_APB1RST, 31U), /*!< UART7 clock reset */
    /* APB2 peripherals */
    RCU_TIMER0RST  = RCU_REGIDX_BIT(IDX_APB2RST, 0U),  /*!< TIMER0 clock reset */
    RCU_TIMER7RST  = RCU_REGIDX_BIT(IDX_APB2RST, 1U),  /*!< TIMER7 clock reset */
    RCU_USART0RST  = RCU_REGIDX_BIT(IDX_APB2RST, 4U),  /*!< USART0 clock reset */
    RCU_USART5RST  = RCU_REGIDX_BIT(IDX_APB2RST, 5U),  /*!< USART5 clock reset */
    RCU_ADC0RST    = RCU_REGIDX_BIT(IDX_APB2RST, 8U),  /*!< ADC0 clock reset */
    RCU_ADC1RST    = RCU_REGIDX_BIT(IDX_APB2RST, 9U),  /*!< ADC1 clock reset */
    RCU_ADC2RST    = RCU_REGIDX_BIT(IDX_APB2RST, 10U), /*!< ADC2 clock reset */
    RCU_SPI0RST    = RCU_REGIDX_BIT(IDX_APB2RST, 12U), /*!< SPI0 clock reset */
    RCU_SPI3RST    = RCU_REGIDX_BIT(IDX_APB2RST, 13U), /*!< SPI3 clock reset */
    RCU_TIMER14RST = RCU_REGIDX_BIT(IDX_APB2RST, 16U), /*!< TIMER14 clock reset */
    RCU_TIMER15RST = RCU_REGIDX_BIT(IDX_APB2RST, 17U), /*!< TIMER15 clock reset */
    RCU_TIMER16RST = RCU_REGIDX_BIT(IDX_APB2RST, 18U), /*!< TIMER16 clock reset */
    RCU_HPDFRST    = RCU_REGIDX_BIT(IDX_APB2RST, 19U), /*!< HPDF clock reset */
    RCU_SPI4RST    = RCU_REGIDX_BIT(IDX_APB2RST, 20U), /*!< SPI4 clock reset */
    RCU_SPI5RST    = RCU_REGIDX_BIT(IDX_APB2RST, 21U), /*!< SPI5 clock reset */
    RCU_SAI0RST    = RCU_REGIDX_BIT(IDX_APB2RST, 22U), /*!< SAI0 clock reset */
    RCU_SAI1RST    = RCU_REGIDX_BIT(IDX_APB2RST, 23U), /*!< SAI1 clock reset */
    RCU_SAI2RST    = RCU_REGIDX_BIT(IDX_APB2RST, 24U), /*!< SAI2 clock reset */
    RCU_TIMER40RST = RCU_REGIDX_BIT(IDX_APB2RST, 25U), /*!< TIMER40 clock reset */
    RCU_TIMER41RST = RCU_REGIDX_BIT(IDX_APB2RST, 26U), /*!< TIMER41 clock reset */
    RCU_TIMER42RST = RCU_REGIDX_BIT(IDX_APB2RST, 27U), /*!< TIMER42 clock reset */
    RCU_TIMER43RST = RCU_REGIDX_BIT(IDX_APB2RST, 28U), /*!< TIMER43 clock reset */
    RCU_TIMER44RST = RCU_REGIDX_BIT(IDX_APB2RST, 29U), /*!< TIMER44 clock reset */
    RCU_EDOUTRST   = RCU_REGIDX_BIT(IDX_APB2RST, 30U), /*!< EDOUT clock reset */
    RCU_TRIGSELRST = RCU_REGIDX_BIT(IDX_APB2RST, 31U), /*!< TRIGSEL clock reset */
    /* APB3 peripherals */
    RCU_TLIRST   = RCU_REGIDX_BIT(IDX_APB3RST, 0U), /*!< TLI clock reset */
    RCU_WWDGTRST = RCU_REGIDX_BIT(IDX_APB3RST, 1U), /*!< WWDGT clock reset */
    /* APB4 peripherals */
    RCU_SYSCFGRST = RCU_REGIDX_BIT(IDX_APB4RST, 0U), /*!< SYSCFG clock reset */
    RCU_CMPRST    = RCU_REGIDX_BIT(IDX_APB4RST, 1U), /*!< CMP clock reset */
    RCU_VREFRST   = RCU_REGIDX_BIT(IDX_APB4RST, 2U), /*!< VREF clock reset */
    RCU_LPDTSRST  = RCU_REGIDX_BIT(IDX_APB4RST, 3U), /*!< LPDTS clock reset */
    RCU_PMURST    = RCU_REGIDX_BIT(IDX_APB4RST, 4U), /*!< PMU clock reset */
    /* APB2 additional peripherals */
    RCU_CAN0RST = RCU_REGIDX_BIT(IDX_ADD_APB2RST, 0U), /*!< CAN0 clock reset */
    RCU_CAN1RST = RCU_REGIDX_BIT(IDX_ADD_APB2RST, 1U), /*!< CAN1 clock reset */
    RCU_CAN2RST = RCU_REGIDX_BIT(IDX_ADD_APB2RST, 2U)  /*!< CAN2 clock reset */
} hal_rcu_periph_reset_enum;

/* clock stabilization and failure and peripheral reset flags */
typedef enum {
    /* clock stabilization flags */
    RCU_FLAG_IRC64MSTB    = RCU_REGIDX_BIT(IDX_CTL, 31U),     /*!< IRC64M stabilization flags */
    RCU_FLAG_HXTALSTB     = RCU_REGIDX_BIT(IDX_CTL, 17U),     /*!< HXTAL stabilization flags */
    RCU_FLAG_PLL0STB      = RCU_REGIDX_BIT(IDX_CTL, 25U),     /*!< PLL0 stabilization flags */
    RCU_FLAG_PLL1STB      = RCU_REGIDX_BIT(IDX_CTL, 27U),     /*!< PLL1 stabilization flags */
    RCU_FLAG_PLL2STB      = RCU_REGIDX_BIT(IDX_CTL, 29U),     /*!< PLL2 stabilization flags */
    RCU_FLAG_LXTALSTB     = RCU_REGIDX_BIT(IDX_BDCTL, 1U),    /*!< LXTAL stabilization flags */
    RCU_FLAG_IRC32KSTB    = RCU_REGIDX_BIT(IDX_RSTSCK, 1U),   /*!< IRC32K stabilization flags */
    RCU_FLAG_IRC48MSTB    = RCU_REGIDX_BIT(IDX_ADDCTL0, 17U), /*!< IRC48M stabilization flags */
    RCU_FLAG_LPIRC4MSTB   = RCU_REGIDX_BIT(IDX_ADDCTL1, 1U),  /*!< LPIRC4M stabilization flags */
    RCU_FLAG_PLLUSBHS0STB = RCU_REGIDX_BIT(IDX_ADDCTL1, 29U), /*!< PLLUSBHS0 stabilization flags */
    RCU_FLAG_PLLUSBHS1STB = RCU_REGIDX_BIT(IDX_ADDCTL1, 31U), /*!< PLLUSBHS1 stabilization flags */
    /* clock failure flag */
    RCU_FLAG_LCKMD = RCU_REGIDX_BIT(IDX_BDCTL, 6U), /*!< LXTAL clock failure detection flags */
    /* reset source flags */
    RCU_FLAG_BORRST   = RCU_REGIDX_BIT(IDX_RSTSCK, 25U), /*!< BOR reset flags */
    RCU_FLAG_EPRST    = RCU_REGIDX_BIT(IDX_RSTSCK, 26U), /*!< External PIN reset flags */
    RCU_FLAG_PORRST   = RCU_REGIDX_BIT(IDX_RSTSCK, 27U), /*!< power reset flags */
    RCU_FLAG_SWRST    = RCU_REGIDX_BIT(IDX_RSTSCK, 28U), /*!< Software reset flags */
    RCU_FLAG_FWDGTRST = RCU_REGIDX_BIT(IDX_RSTSCK, 29U), /*!< FWDGT reset flags */
    RCU_FLAG_WWDGTRST = RCU_REGIDX_BIT(IDX_RSTSCK, 30U), /*!< WWDGT reset flags */
    RCU_FLAG_LPRST    = RCU_REGIDX_BIT(IDX_RSTSCK, 31U)  /*!< low-power reset flags */
} hal_rcu_flag_enum;

/* clock stabilization and stuck interrupt flags */
typedef enum {
    RCU_INT_FLAG_IRC32KSTB    = RCU_REGIDX_BIT(IDX_INT, 0U),    /*!< IRC32K stabilization interrupt flag */
    RCU_INT_FLAG_LXTALSTB     = RCU_REGIDX_BIT(IDX_INT, 1U),    /*!< LXTAL stabilization interrupt flag */
    RCU_INT_FLAG_IRC64MSTB    = RCU_REGIDX_BIT(IDX_INT, 2U),    /*!< IRC64M stabilization interrupt flag */
    RCU_INT_FLAG_HXTALSTB     = RCU_REGIDX_BIT(IDX_INT, 3U),    /*!< HXTAL stabilization interrupt flag */
    RCU_INT_FLAG_PLL0STB      = RCU_REGIDX_BIT(IDX_INT, 4U),    /*!< PLL0 stabilization interrupt flag */
    RCU_INT_FLAG_PLL1STB      = RCU_REGIDX_BIT(IDX_INT, 5U),    /*!< PLL1 stabilization interrupt flag */
    RCU_INT_FLAG_PLL2STB      = RCU_REGIDX_BIT(IDX_INT, 6U),    /*!< PLL2 stabilization interrupt flag */
    RCU_INT_FLAG_CKM          = RCU_REGIDX_BIT(IDX_INT, 7U),    /*!< HXTAL clock stuck interrupt flag */
    RCU_INT_FLAG_LCKM         = RCU_REGIDX_BIT(IDX_INT, 27U),   /*!< LXTAL clock stuck interrupt flag */
    RCU_INT_FLAG_LPIRC4MSTB   = RCU_REGIDX_BIT(IDX_INT, 24U),   /*!< LPIRC4M stabilization interrupt flag */
    RCU_INT_FLAG_IRC48MSTB    = RCU_REGIDX_BIT(IDX_ADDINT, 6U), /*!< IRC48M stabilization interrupt flag */
    RCU_INT_FLAG_PLLUSBHS0STB = RCU_REGIDX_BIT(IDX_ADDINT, 4U), /*!< PLLUSBHS0 stabilization interrupt flag */
    RCU_INT_FLAG_PLLUSBHS1STB = RCU_REGIDX_BIT(IDX_ADDINT, 5U)  /*!< PLLUSBHS1 stabilization interrupt flag */
} hal_rcu_int_flag_enum;

/* clock stabilization and stuck interrupt flags clear */
typedef enum {
    RCU_INT_FLAG_IRC32KSTB_CLR    = RCU_REGIDX_BIT(IDX_INT, 16U),    /*!< IRC32K stabilization interrupt flags clear */
    RCU_INT_FLAG_LXTALSTB_CLR     = RCU_REGIDX_BIT(IDX_INT, 17U),    /*!< LXTAL stabilization interrupt flags clear */
    RCU_INT_FLAG_IRC64MSTB_CLR    = RCU_REGIDX_BIT(IDX_INT, 18U),    /*!< IRC64M stabilization interrupt flags clear */
    RCU_INT_FLAG_HXTALSTB_CLR     = RCU_REGIDX_BIT(IDX_INT, 19U),    /*!< HXTAL stabilization interrupt flags clear */
    RCU_INT_FLAG_PLL0STB_CLR      = RCU_REGIDX_BIT(IDX_INT, 20U),    /*!< PLL0 stabilization interrupt flags clear */
    RCU_INT_FLAG_PLL1STB_CLR      = RCU_REGIDX_BIT(IDX_INT, 21U),    /*!< PLL1 stabilization interrupt flags clear */
    RCU_INT_FLAG_PLL2STB_CLR      = RCU_REGIDX_BIT(IDX_INT, 22U),    /*!< PLL2 stabilization interrupt flags clear */
    RCU_INT_FLAG_CKM_CLR          = RCU_REGIDX_BIT(IDX_INT, 23U),    /*!< HXTAL clock stuck interrupt flags clear */
    RCU_INT_FLAG_LCKM_CLR         = RCU_REGIDX_BIT(IDX_INT, 28U),    /*!< LXTAL clock stuck interrupt flags clear */
    RCU_INT_FLAG_LPIRC4MSTB_CLR   = RCU_REGIDX_BIT(IDX_INT, 26U),    /*!< LPIRC4M  stabilization interrupt flag clear */
    RCU_INT_FLAG_IRC48MSTB_CLR    = RCU_REGIDX_BIT(IDX_ADDINT, 22U), /*!< IRC48M stabilization interrupt flag clear */
    RCU_INT_FLAG_PLLUSBHS0STB_CLR = RCU_REGIDX_BIT(IDX_ADDINT, 20U), /*!< PLLUSBHS0 stabilization interrupt flag clear */
    RCU_INT_FLAG_PLLUSBHS1STB_CLR = RCU_REGIDX_BIT(IDX_ADDINT, 21U)  /*!< PLLUSBHS1 stabilization interrupt flag clear */
} hal_rcu_int_flag_clear_enum;

/* clock stabilization interrupt enable or disable */
typedef enum {
    RCU_INT_IRC32KSTB  = RCU_REGIDX_BIT(IDX_INT, 8U),     /*!< IRC32K stabilization interrupt */
    RCU_INT_LXTALSTB   = RCU_REGIDX_BIT(IDX_INT, 9U),     /*!< LXTAL stabilization interrupt */
    RCU_INT_IRC64MSTB  = RCU_REGIDX_BIT(IDX_INT, 10U),    /*!< IRC64M stabilization interrupt */
    RCU_INT_HXTALSTB   = RCU_REGIDX_BIT(IDX_INT, 11U),    /*!< HXTAL stabilization interrupt */
    RCU_INT_PLL0STB    = RCU_REGIDX_BIT(IDX_INT, 12U),    /*!< PLL0 stabilization interrupt */
    RCU_INT_PLL1STB    = RCU_REGIDX_BIT(IDX_INT, 13U),    /*!< PLL1 stabilization interrupt */
    RCU_INT_PLL2STB    = RCU_REGIDX_BIT(IDX_INT, 14U),    /*!< PLL2 stabilization interrupt */
    RCU_INT_IRC48MSTB  = RCU_REGIDX_BIT(IDX_ADDINT, 14U), /*!< internal 48 MHz RC oscillator stabilization interrupt */
    RCU_INT_LPIRC4MSTB = RCU_REGIDX_BIT(IDX_INT, 25U),    /*!< LPIRC4M stabilization interrupt */
    RCU_INT_PLLUSBHS0STB = RCU_REGIDX_BIT(IDX_ADDINT, 12U), /*!< PLLUSBHS0 stabilization interrupt */
    RCU_INT_PLLUSBHS1STB = RCU_REGIDX_BIT(IDX_ADDINT, 13U)  /*!< PLLUSBHS1 stabilization interrupt */
} hal_rcu_int_enum;

/* oscillator types */
typedef enum {
    RCU_HXTAL        = RCU_REGIDX_BIT(IDX_CTL, 16U),     /*!< HXTAL */
    RCU_LXTAL        = RCU_REGIDX_BIT(IDX_BDCTL, 0U),    /*!< LXTAL */
    RCU_IRC64M       = RCU_REGIDX_BIT(IDX_CTL, 30U),     /*!< IRC64M */
    RCU_IRC48M       = RCU_REGIDX_BIT(IDX_ADDCTL0, 16U), /*!< IRC48M */
    RCU_IRC32K       = RCU_REGIDX_BIT(IDX_RSTSCK, 0U),   /*!< IRC32K */
    RCU_LPIRC4M      = RCU_REGIDX_BIT(IDX_ADDCTL1, 0U),  /*!< LPIRC4M */
    RCU_PLL0_CK      = RCU_REGIDX_BIT(IDX_CTL, 24U),     /*!< PLL0 */
    RCU_PLL1_CK      = RCU_REGIDX_BIT(IDX_CTL, 26U),     /*!< PLL1 */
    RCU_PLL2_CK      = RCU_REGIDX_BIT(IDX_CTL, 28U),     /*!< PLL2 */
    RCU_PLLUSBHS0_CK = RCU_REGIDX_BIT(IDX_ADDCTL1, 28U), /*!< PLLUSBHS0 */
    RCU_PLLUSBHS1_CK = RCU_REGIDX_BIT(IDX_ADDCTL1, 30U)  /*!< PLLUSBHS1 */
} hal_rcu_osci_type_enum;

/* rcu clock frequency */
typedef enum {
    CK_SYS = (uint32_t)0x00000000U, /*!< system clock */
    CK_AHB,                         /*!< AHB clock */
    CK_APB1,                        /*!< APB1 clock */
    CK_APB2,                        /*!< APB2 clock */
    CK_APB3,                        /*!< APB3 clock */
    CK_APB4,                        /*!< APB4 clock */
    CK_PLL0P,                       /*!< PLL0P clock */
    CK_PLL0Q,                       /*!< PLL0Q clock */
    CK_PLL0R,                       /*!< PLL0R clock */
    CK_PLL1P,                       /*!< PLL1P clock */
    CK_PLL1Q,                       /*!< PLL1Q clock */
    CK_PLL1R,                       /*!< PLL1R clock */
    CK_PLL2P,                       /*!< PLL2P clock */
    CK_PLL2Q,                       /*!< PLL2Q clock */
    CK_PLL2R,                       /*!< PLL2R clock */
    CK_PER,                         /*!< PER clock */
    CK_USART0,                      /*!< USART0 clock */
    CK_USART1,                      /*!< USART1 clock */
    CK_USART2,                      /*!< USART2 clock */
    CK_USART5,                      /*!< USART5 clock */
    CK_I2C0,                        /*!< I2C0 clock */
    CK_I2C1,                        /*!< I2C1 clock */
    CK_I2C2,                        /*!< I2C2 clock */
    CK_I2C3,                        /*!< I2C3 clock */
    CK_CAN0,                        /*!< CAN0 clock */
    CK_CAN1,                        /*!< CAN1 clock */
    CK_CAN2,                        /*!< CAN2 clock */
    CK_ADC01,                       /*!< ADC0 clock */
    CK_ADC2,                        /*!< ADC2 clock */
    CK_SDIO0,                       /*!< SDIO0 clock */
    CK_SDIO1,                       /*!< SDIO1 clock */
    CK_SPI0,                        /*!< SPI0 clock */
    CK_SPI1,                        /*!< SPI1 clock */
    CK_SPI2,                        /*!< SPI2 clock */
    CK_SPI3,                        /*!< SPI3 clock */
    CK_SPI4,                        /*!< SPI4 clock */
    CK_SPI5,                        /*!< SPI5 clock */
    CK_RTC,                         /*!< RTC clock */
    CK_EXMC,                        /*!< EXMC clock */
    CK_RSPDIF,                      /*!< RSPDIF clock */
    CK_IRC64MDIV,                   /*!< IRC64MDIV clock */
    CK_HXTAL,                       /*!< HXTAL clock */
    CK_LXTAL,                       /*!< LXTAL clock */
    CK_LPIRC4M,                     /*!< LPIRC4M clock */
    CK_IRC48M,                      /*!< IRC48M clock */
    CK_IRC32K                       /*!< IRC32K clock */
} hal_rcu_clock_freq_enum;

/* USART num */
typedef enum {
    INDEX_USART0 = (uint32_t)0x00000000U, /*!< index of USART0 */
    INDEX_USART1,                         /*!< index of USART1 */
    INDEX_USART2,                         /*!< index of USART2 */
    INDEX_USART5                          /*!< index of USART5 */
} hal_usart_idx_enum;

/* I2C num */
typedef enum {
    INDEX_I2C0 = (uint32_t)0x00000000U, /*!< index of I2C0 */
    INDEX_I2C1,                         /*!< index of I2C1 */
    INDEX_I2C2,                         /*!< index of I2C2 */
    INDEX_I2C3                          /*!< index of I2C2 */
} hal_i2c_idx_enum;

/* CAN num */
typedef enum {
    INDEX_CAN0 = (uint32_t)0x00000000U, /*!< index of CAN0 */
    INDEX_CAN1,                         /*!< index of CAN1 */
    INDEX_CAN2                          /*!< index of CAN2 */
} hal_can_idx_enum;

/* SAI num */
typedef enum {
    INDEX_SAI0 = (uint32_t)0x00000000U, /*!< index of SAI0 */
    INDEX_SAI1                          /*!< index of SAI1 */
} hal_sai_idx_enum;

/* SAI2B num */
typedef enum {
    INDEX_SAI2B0 = (uint32_t)0x00000000U, /*!< index of SAI2B0 */
    INDEX_SAI2B1                          /*!< index of SAI2B1 */
} hal_sai2b_idx_enum;

/* ADC num */
typedef enum {
    INDEX_ADC01 = (uint32_t)0x00000000U, /*!< index of ADC0 */
    INDEX_ADC2                           /*!< index of ADC2 */
} hal_adc_idx_enum;

/* USBHS num */
typedef enum {
    INDEX_USBHS0 = (uint32_t)0x00000000U, /*!< index of USBHS0 */
    INDEX_USBHS1                          /*!< index of USBHS1 */
} hal_usbhs_idx_enum;

/* PLL num */
typedef enum {
    INDEX_PLL0 = (uint32_t)0x00000000U, /*!< index of PLL0 */
    INDEX_PLL1,                         /*!< index of PLL1 */
    INDEX_PLL2                          /*!< index of PLL2 */
} hal_pll_idx_enum;

/* SDIO num */
typedef enum {
    INDEX_SDIO0 = (uint32_t)0x00000000U, /*!< index of SDIO0 */
    INDEX_SDIO1                          /*!< index of SDIO1 */
} hal_sdio_idx_enum;

/* SPI num */
typedef enum {
    INDEX_SPI0 = (uint32_t)0x00000000U, /*!< index of SPI0 */
    INDEX_SPI1,                         /*!< index of SPI1 */
    INDEX_SPI2,                         /*!< index of SPI2 */
    INDEX_SPI3,                         /*!< index of SPI3 */
    INDEX_SPI4,                         /*!< index of SPI4 */
    INDEX_SPI5                          /*!< index of SPI5 */
} hal_spi_idx_enum;

/* init struct type */
typedef enum {
    HAL_RCU_CLK_STRUCT = (uint32_t)0x00000000U, /*!< RCU clock structure type */
    HAL_RCU_OSCI_STRUCT,                        /*!< RCU oscillator structure type */
    HAL_RCU_PLL_STRUCT,                         /*!< RCU pll structure type */
    HAL_RCU_PERIPHCLK_STRUCT                    /*!< RCU peripheral clock structure type */
} hal_rcu_struct_type_enum;

/* RCU interrupt handler structure */
typedef struct {
    __IO hal_irq_handle_cb pll0_stable_handle;      /*!< PLL0 clock stable interrupt */
    __IO hal_irq_handle_cb pll1_stable_handle;      /*!< PLL1 clock stable interrupt */
    __IO hal_irq_handle_cb pll2_stable_handle;      /*!< PLL2 clock stable interrupt */
    __IO hal_irq_handle_cb pllusbhs0_stable_handle; /*!< pllusbhs0 clock stable interrupt */
    __IO hal_irq_handle_cb pllusbhs1_stable_handle; /*!< pllusbhs1 clock stable interrupt */
    __IO hal_irq_handle_cb lpirc4m_stable_handle;   /*!< LPIRC4M clock stable interrupt */
    __IO hal_irq_handle_cb irc32K_stable_handle;    /*!< IRC32K clock stable interrupt */
    __IO hal_irq_handle_cb irc48m_stable_handle;    /*!< IRC48M clock stable interrupt */
    __IO hal_irq_handle_cb irc64m_stable_handle;    /*!< IRC64M clock stable interrupt */
    __IO hal_irq_handle_cb lxtal_stable_handle;     /*!< LXTAL clock stable interrupt */
    __IO hal_irq_handle_cb hxtal_stable_handle;     /*!< HXTAL clock stable interrupt */
    __IO hal_irq_handle_cb hxtal_stuck_handle;      /*!< HXTAL clock stuck interrupt */
    __IO hal_irq_handle_cb lxtal_stuck_handle;      /*!< LXTAL clock stuck interrupt */
} hal_rcu_irq_struct;

/* @STRUCT_MEMBER: state */
/* @ENUM: RCU oscillator state */
typedef enum {
    RCU_OSC_NONE = (uint32_t)0x00000000U, /*!< the oscillator is not configured */
    RCU_OSC_OFF,                          /*!< the oscillator deactivation*/
    RCU_OSC_ON,                           /*!< the oscillator activation */
    RCU_OSC_BYPASS                        /*!< external clock source for the oscillator */
} hal_rcu_osc_state_enum;

/*!< PLL(PLL0\PLL1\PLL2) VCO Source clock divider*/
typedef enum {
    RCU_PLL_PSC_1  = RCU_PLL_PSC1,  /*!< PLL PSC div by 1 */
    RCU_PLL_PSC_2  = RCU_PLL_PSC2,  /*!< PLL PSC div by 2 */
    RCU_PLL_PSC_3  = RCU_PLL_PSC3,  /*!< PLL PSC div by 3 */
    RCU_PLL_PSC_4  = RCU_PLL_PSC4,  /*!< PLL PSC div by 4 */
    RCU_PLL_PSC_5  = RCU_PLL_PSC5,  /*!< PLL PSC div by 5 */
    RCU_PLL_PSC_6  = RCU_PLL_PSC6,  /*!< PLL PSC div by 6 */
    RCU_PLL_PSC_7  = RCU_PLL_PSC7,  /*!< PLL PSC div by 7 */
    RCU_PLL_PSC_8  = RCU_PLL_PSC8,  /*!< PLL PSC div by 8 */
    RCU_PLL_PSC_9  = RCU_PLL_PSC9,  /*!< PLL PSC div by 9 */
    RCU_PLL_PSC_10 = RCU_PLL_PSC10, /*!< PLL PSC div by 10 */
    RCU_PLL_PSC_11 = RCU_PLL_PSC11, /*!< PLL PSC div by 11 */
    RCU_PLL_PSC_12 = RCU_PLL_PSC12, /*!< PLL PSC div by 12 */
    RCU_PLL_PSC_13 = RCU_PLL_PSC13, /*!< PLL PSC div by 13 */
    RCU_PLL_PSC_14 = RCU_PLL_PSC14, /*!< PLL PSC div by 14 */
    RCU_PLL_PSC_15 = RCU_PLL_PSC15, /*!< PLL PSC div by 15 */
    RCU_PLL_PSC_16 = RCU_PLL_PSC16, /*!< PLL PSC div by 16 */
    RCU_PLL_PSC_17 = RCU_PLL_PSC17, /*!< PLL PSC div by 17 */
    RCU_PLL_PSC_18 = RCU_PLL_PSC18, /*!< PLL PSC div by 18 */
    RCU_PLL_PSC_19 = RCU_PLL_PSC19, /*!< PLL PSC div by 19 */
    RCU_PLL_PSC_20 = RCU_PLL_PSC20, /*!< PLL PSC div by 20 */
    RCU_PLL_PSC_21 = RCU_PLL_PSC21, /*!< PLL PSC div by 21 */
    RCU_PLL_PSC_22 = RCU_PLL_PSC22, /*!< PLL PSC div by 22 */
    RCU_PLL_PSC_23 = RCU_PLL_PSC23, /*!< PLL PSC div by 23 */
    RCU_PLL_PSC_24 = RCU_PLL_PSC24, /*!< PLL PSC div by 24 */
    RCU_PLL_PSC_25 = RCU_PLL_PSC25, /*!< PLL PSC div by 25 */
    RCU_PLL_PSC_26 = RCU_PLL_PSC26, /*!< PLL PSC div by 26 */
    RCU_PLL_PSC_27 = RCU_PLL_PSC27, /*!< PLL PSC div by 27 */
    RCU_PLL_PSC_28 = RCU_PLL_PSC28, /*!< PLL PSC div by 28 */
    RCU_PLL_PSC_29 = RCU_PLL_PSC29, /*!< PLL PSC div by 29 */
    RCU_PLL_PSC_30 = RCU_PLL_PSC30, /*!< PLL PSC div by 30 */
    RCU_PLL_PSC_31 = RCU_PLL_PSC31, /*!< PLL PSC div by 31 */
    RCU_PLL_PSC_32 = RCU_PLL_PSC32, /*!< PLL PSC div by 32 */
    RCU_PLL_PSC_33 = RCU_PLL_PSC33, /*!< PLL PSC div by 33 */
    RCU_PLL_PSC_34 = RCU_PLL_PSC34, /*!< PLL PSC div by 34 */
    RCU_PLL_PSC_35 = RCU_PLL_PSC35, /*!< PLL PSC div by 35 */
    RCU_PLL_PSC_36 = RCU_PLL_PSC36, /*!< PLL PSC div by 36 */
    RCU_PLL_PSC_37 = RCU_PLL_PSC37, /*!< PLL PSC div by 37 */
    RCU_PLL_PSC_38 = RCU_PLL_PSC38, /*!< PLL PSC div by 38 */
    RCU_PLL_PSC_39 = RCU_PLL_PSC39, /*!< PLL PSC div by 39 */
    RCU_PLL_PSC_40 = RCU_PLL_PSC40, /*!< PLL PSC div by 40 */
    RCU_PLL_PSC_41 = RCU_PLL_PSC41, /*!< PLL PSC div by 41 */
    RCU_PLL_PSC_42 = RCU_PLL_PSC42, /*!< PLL PSC div by 42 */
    RCU_PLL_PSC_43 = RCU_PLL_PSC43, /*!< PLL PSC div by 43 */
    RCU_PLL_PSC_44 = RCU_PLL_PSC44, /*!< PLL PSC div by 44 */
    RCU_PLL_PSC_45 = RCU_PLL_PSC45, /*!< PLL PSC div by 45 */
    RCU_PLL_PSC_46 = RCU_PLL_PSC46, /*!< PLL PSC div by 46 */
    RCU_PLL_PSC_47 = RCU_PLL_PSC47, /*!< PLL PSC div by 47 */
    RCU_PLL_PSC_48 = RCU_PLL_PSC48, /*!< PLL PSC div by 48 */
    RCU_PLL_PSC_49 = RCU_PLL_PSC49, /*!< PLL PSC div by 49 */
    RCU_PLL_PSC_50 = RCU_PLL_PSC50, /*!< PLL PSC div by 50 */
    RCU_PLL_PSC_51 = RCU_PLL_PSC51, /*!< PLL PSC div by 51 */
    RCU_PLL_PSC_52 = RCU_PLL_PSC52, /*!< PLL PSC div by 52 */
    RCU_PLL_PSC_53 = RCU_PLL_PSC53, /*!< PLL PSC div by 53 */
    RCU_PLL_PSC_54 = RCU_PLL_PSC54, /*!< PLL PSC div by 54 */
    RCU_PLL_PSC_55 = RCU_PLL_PSC55, /*!< PLL PSC div by 55 */
    RCU_PLL_PSC_56 = RCU_PLL_PSC56, /*!< PLL PSC div by 56 */
    RCU_PLL_PSC_57 = RCU_PLL_PSC57, /*!< PLL PSC div by 57 */
    RCU_PLL_PSC_58 = RCU_PLL_PSC58, /*!< PLL PSC div by 58 */
    RCU_PLL_PSC_59 = RCU_PLL_PSC59, /*!< PLL PSC div by 59 */
    RCU_PLL_PSC_60 = RCU_PLL_PSC60, /*!< PLL PSC div by 60 */
    RCU_PLL_PSC_61 = RCU_PLL_PSC61, /*!< PLL PSC div by 61 */
    RCU_PLL_PSC_62 = RCU_PLL_PSC62, /*!< PLL PSC div by 62 */
    RCU_PLL_PSC_63 = RCU_PLL_PSC63  /*!< PLL PSC div by 63 */
} hal_rcu_pll_psc_enum;

/*!< PLLN(PLL0\PLL1\PLL2) VCO clock multiplier factor*/
typedef enum {
    RCU_PLLN_MUL_9   = RCU_PLLN_MUL9,   /*!< PLLN Multiple by 9 */
    RCU_PLLN_MUL_10  = RCU_PLLN_MUL10,  /*!< PLLN Multiple by 10 */
    RCU_PLLN_MUL_11  = RCU_PLLN_MUL11,  /*!< PLLN Multiple by 11 */
    RCU_PLLN_MUL_12  = RCU_PLLN_MUL12,  /*!< PLLN Multiple by 12 */
    RCU_PLLN_MUL_13  = RCU_PLLN_MUL13,  /*!< PLLN Multiple by 13 */
    RCU_PLLN_MUL_14  = RCU_PLLN_MUL14,  /*!< PLLN Multiple by 14 */
    RCU_PLLN_MUL_15  = RCU_PLLN_MUL15,  /*!< PLLN Multiple by 15 */
    RCU_PLLN_MUL_16  = RCU_PLLN_MUL16,  /*!< PLLN Multiple by 16 */
    RCU_PLLN_MUL_17  = RCU_PLLN_MUL17,  /*!< PLLN Multiple by 17 */
    RCU_PLLN_MUL_18  = RCU_PLLN_MUL18,  /*!< PLLN Multiple by 18 */
    RCU_PLLN_MUL_19  = RCU_PLLN_MUL19,  /*!< PLLN Multiple by 19 */
    RCU_PLLN_MUL_20  = RCU_PLLN_MUL20,  /*!< PLLN Multiple by 20 */
    RCU_PLLN_MUL_21  = RCU_PLLN_MUL21,  /*!< PLLN Multiple by 21 */
    RCU_PLLN_MUL_22  = RCU_PLLN_MUL22,  /*!< PLLN Multiple by 22 */
    RCU_PLLN_MUL_23  = RCU_PLLN_MUL23,  /*!< PLLN Multiple by 23 */
    RCU_PLLN_MUL_24  = RCU_PLLN_MUL24,  /*!< PLLN Multiple by 24 */
    RCU_PLLN_MUL_25  = RCU_PLLN_MUL25,  /*!< PLLN Multiple by 25 */
    RCU_PLLN_MUL_26  = RCU_PLLN_MUL26,  /*!< PLLN Multiple by 26 */
    RCU_PLLN_MUL_27  = RCU_PLLN_MUL27,  /*!< PLLN Multiple by 27 */
    RCU_PLLN_MUL_28  = RCU_PLLN_MUL28,  /*!< PLLN Multiple by 28 */
    RCU_PLLN_MUL_29  = RCU_PLLN_MUL29,  /*!< PLLN Multiple by 29 */
    RCU_PLLN_MUL_30  = RCU_PLLN_MUL30,  /*!< PLLN Multiple by 30 */
    RCU_PLLN_MUL_31  = RCU_PLLN_MUL31,  /*!< PLLN Multiple by 31 */
    RCU_PLLN_MUL_32  = RCU_PLLN_MUL32,  /*!< PLLN Multiple by 32 */
    RCU_PLLN_MUL_33  = RCU_PLLN_MUL33,  /*!< PLLN Multiple by 33 */
    RCU_PLLN_MUL_34  = RCU_PLLN_MUL34,  /*!< PLLN Multiple by 34 */
    RCU_PLLN_MUL_35  = RCU_PLLN_MUL35,  /*!< PLLN Multiple by 35 */
    RCU_PLLN_MUL_36  = RCU_PLLN_MUL36,  /*!< PLLN Multiple by 36 */
    RCU_PLLN_MUL_37  = RCU_PLLN_MUL37,  /*!< PLLN Multiple by 37 */
    RCU_PLLN_MUL_38  = RCU_PLLN_MUL38,  /*!< PLLN Multiple by 38 */
    RCU_PLLN_MUL_39  = RCU_PLLN_MUL39,  /*!< PLLN Multiple by 39 */
    RCU_PLLN_MUL_40  = RCU_PLLN_MUL40,  /*!< PLLN Multiple by 40 */
    RCU_PLLN_MUL_41  = RCU_PLLN_MUL41,  /*!< PLLN Multiple by 41 */
    RCU_PLLN_MUL_42  = RCU_PLLN_MUL42,  /*!< PLLN Multiple by 42 */
    RCU_PLLN_MUL_43  = RCU_PLLN_MUL43,  /*!< PLLN Multiple by 43 */
    RCU_PLLN_MUL_44  = RCU_PLLN_MUL44,  /*!< PLLN Multiple by 44 */
    RCU_PLLN_MUL_45  = RCU_PLLN_MUL45,  /*!< PLLN Multiple by 45 */
    RCU_PLLN_MUL_46  = RCU_PLLN_MUL46,  /*!< PLLN Multiple by 46 */
    RCU_PLLN_MUL_47  = RCU_PLLN_MUL47,  /*!< PLLN Multiple by 47 */
    RCU_PLLN_MUL_48  = RCU_PLLN_MUL48,  /*!< PLLN Multiple by 48 */
    RCU_PLLN_MUL_49  = RCU_PLLN_MUL49,  /*!< PLLN Multiple by 49 */
    RCU_PLLN_MUL_50  = RCU_PLLN_MUL50,  /*!< PLLN Multiple by 50 */
    RCU_PLLN_MUL_51  = RCU_PLLN_MUL51,  /*!< PLLN Multiple by 51 */
    RCU_PLLN_MUL_52  = RCU_PLLN_MUL52,  /*!< PLLN Multiple by 52 */
    RCU_PLLN_MUL_53  = RCU_PLLN_MUL53,  /*!< PLLN Multiple by 53 */
    RCU_PLLN_MUL_54  = RCU_PLLN_MUL54,  /*!< PLLN Multiple by 54 */
    RCU_PLLN_MUL_55  = RCU_PLLN_MUL55,  /*!< PLLN Multiple by 55 */
    RCU_PLLN_MUL_56  = RCU_PLLN_MUL56,  /*!< PLLN Multiple by 56 */
    RCU_PLLN_MUL_57  = RCU_PLLN_MUL57,  /*!< PLLN Multiple by 57 */
    RCU_PLLN_MUL_58  = RCU_PLLN_MUL58,  /*!< PLLN Multiple by 58 */
    RCU_PLLN_MUL_59  = RCU_PLLN_MUL59,  /*!< PLLN Multiple by 59 */
    RCU_PLLN_MUL_60  = RCU_PLLN_MUL60,  /*!< PLLN Multiple by 60 */
    RCU_PLLN_MUL_61  = RCU_PLLN_MUL61,  /*!< PLLN Multiple by 61 */
    RCU_PLLN_MUL_62  = RCU_PLLN_MUL62,  /*!< PLLN Multiple by 62 */
    RCU_PLLN_MUL_63  = RCU_PLLN_MUL63,  /*!< PLLN Multiple by 63 */
    RCU_PLLN_MUL_64  = RCU_PLLN_MUL64,  /*!< PLLN Multiple by 64 */
    RCU_PLLN_MUL_65  = RCU_PLLN_MUL65,  /*!< PLLN Multiple by 65 */
    RCU_PLLN_MUL_66  = RCU_PLLN_MUL66,  /*!< PLLN Multiple by 66 */
    RCU_PLLN_MUL_67  = RCU_PLLN_MUL67,  /*!< PLLN Multiple by 67 */
    RCU_PLLN_MUL_68  = RCU_PLLN_MUL68,  /*!< PLLN Multiple by 68 */
    RCU_PLLN_MUL_69  = RCU_PLLN_MUL69,  /*!< PLLN Multiple by 69 */
    RCU_PLLN_MUL_70  = RCU_PLLN_MUL70,  /*!< PLLN Multiple by 70 */
    RCU_PLLN_MUL_71  = RCU_PLLN_MUL71,  /*!< PLLN Multiple by 71 */
    RCU_PLLN_MUL_72  = RCU_PLLN_MUL72,  /*!< PLLN Multiple by 72 */
    RCU_PLLN_MUL_73  = RCU_PLLN_MUL73,  /*!< PLLN Multiple by 73 */
    RCU_PLLN_MUL_74  = RCU_PLLN_MUL74,  /*!< PLLN Multiple by 74 */
    RCU_PLLN_MUL_75  = RCU_PLLN_MUL75,  /*!< PLLN Multiple by 75 */
    RCU_PLLN_MUL_76  = RCU_PLLN_MUL76,  /*!< PLLN Multiple by 76 */
    RCU_PLLN_MUL_77  = RCU_PLLN_MUL77,  /*!< PLLN Multiple by 77 */
    RCU_PLLN_MUL_78  = RCU_PLLN_MUL78,  /*!< PLLN Multiple by 78 */
    RCU_PLLN_MUL_79  = RCU_PLLN_MUL79,  /*!< PLLN Multiple by 79 */
    RCU_PLLN_MUL_80  = RCU_PLLN_MUL80,  /*!< PLLN Multiple by 80 */
    RCU_PLLN_MUL_81  = RCU_PLLN_MUL81,  /*!< PLLN Multiple by 81 */
    RCU_PLLN_MUL_82  = RCU_PLLN_MUL82,  /*!< PLLN Multiple by 82 */
    RCU_PLLN_MUL_83  = RCU_PLLN_MUL83,  /*!< PLLN Multiple by 83 */
    RCU_PLLN_MUL_84  = RCU_PLLN_MUL84,  /*!< PLLN Multiple by 84 */
    RCU_PLLN_MUL_85  = RCU_PLLN_MUL85,  /*!< PLLN Multiple by 85 */
    RCU_PLLN_MUL_86  = RCU_PLLN_MUL86,  /*!< PLLN Multiple by 86 */
    RCU_PLLN_MUL_87  = RCU_PLLN_MUL87,  /*!< PLLN Multiple by 87 */
    RCU_PLLN_MUL_88  = RCU_PLLN_MUL88,  /*!< PLLN Multiple by 88 */
    RCU_PLLN_MUL_89  = RCU_PLLN_MUL89,  /*!< PLLN Multiple by 89 */
    RCU_PLLN_MUL_90  = RCU_PLLN_MUL90,  /*!< PLLN Multiple by 90 */
    RCU_PLLN_MUL_91  = RCU_PLLN_MUL91,  /*!< PLLN Multiple by 91 */
    RCU_PLLN_MUL_92  = RCU_PLLN_MUL92,  /*!< PLLN Multiple by 92 */
    RCU_PLLN_MUL_93  = RCU_PLLN_MUL93,  /*!< PLLN Multiple by 93 */
    RCU_PLLN_MUL_94  = RCU_PLLN_MUL94,  /*!< PLLN Multiple by 94 */
    RCU_PLLN_MUL_95  = RCU_PLLN_MUL95,  /*!< PLLN Multiple by 95 */
    RCU_PLLN_MUL_96  = RCU_PLLN_MUL96,  /*!< PLLN Multiple by 96 */
    RCU_PLLN_MUL_97  = RCU_PLLN_MUL97,  /*!< PLLN Multiple by 97 */
    RCU_PLLN_MUL_98  = RCU_PLLN_MUL98,  /*!< PLLN Multiple by 98 */
    RCU_PLLN_MUL_99  = RCU_PLLN_MUL99,  /*!< PLLN Multiple by 99 */
    RCU_PLLN_MUL_100 = RCU_PLLN_MUL100, /*!< PLLN Multiple by 100 */
    RCU_PLLN_MUL_101 = RCU_PLLN_MUL101, /*!< PLLN Multiple by 101 */
    RCU_PLLN_MUL_102 = RCU_PLLN_MUL102, /*!< PLLN Multiple by 102 */
    RCU_PLLN_MUL_103 = RCU_PLLN_MUL103, /*!< PLLN Multiple by 103 */
    RCU_PLLN_MUL_104 = RCU_PLLN_MUL104, /*!< PLLN Multiple by 104 */
    RCU_PLLN_MUL_105 = RCU_PLLN_MUL105, /*!< PLLN Multiple by 105 */
    RCU_PLLN_MUL_106 = RCU_PLLN_MUL106, /*!< PLLN Multiple by 106 */
    RCU_PLLN_MUL_107 = RCU_PLLN_MUL107, /*!< PLLN Multiple by 107 */
    RCU_PLLN_MUL_108 = RCU_PLLN_MUL108, /*!< PLLN Multiple by 108 */
    RCU_PLLN_MUL_109 = RCU_PLLN_MUL109, /*!< PLLN Multiple by 109 */
    RCU_PLLN_MUL_110 = RCU_PLLN_MUL110, /*!< PLLN Multiple by 110 */
    RCU_PLLN_MUL_111 = RCU_PLLN_MUL111, /*!< PLLN Multiple by 111 */
    RCU_PLLN_MUL_112 = RCU_PLLN_MUL112, /*!< PLLN Multiple by 112 */
    RCU_PLLN_MUL_113 = RCU_PLLN_MUL113, /*!< PLLN Multiple by 113 */
    RCU_PLLN_MUL_114 = RCU_PLLN_MUL114, /*!< PLLN Multiple by 114 */
    RCU_PLLN_MUL_115 = RCU_PLLN_MUL115, /*!< PLLN Multiple by 115 */
    RCU_PLLN_MUL_116 = RCU_PLLN_MUL116, /*!< PLLN Multiple by 116 */
    RCU_PLLN_MUL_117 = RCU_PLLN_MUL117, /*!< PLLN Multiple by 117 */
    RCU_PLLN_MUL_118 = RCU_PLLN_MUL118, /*!< PLLN Multiple by 118 */
    RCU_PLLN_MUL_119 = RCU_PLLN_MUL119, /*!< PLLN Multiple by 119 */
    RCU_PLLN_MUL_120 = RCU_PLLN_MUL120, /*!< PLLN Multiple by 120 */
    RCU_PLLN_MUL_121 = RCU_PLLN_MUL121, /*!< PLLN Multiple by 121 */
    RCU_PLLN_MUL_122 = RCU_PLLN_MUL122, /*!< PLLN Multiple by 122 */
    RCU_PLLN_MUL_123 = RCU_PLLN_MUL123, /*!< PLLN Multiple by 123 */
    RCU_PLLN_MUL_124 = RCU_PLLN_MUL124, /*!< PLLN Multiple by 124 */
    RCU_PLLN_MUL_125 = RCU_PLLN_MUL125, /*!< PLLN Multiple by 125 */
    RCU_PLLN_MUL_126 = RCU_PLLN_MUL126, /*!< PLLN Multiple by 126 */
    RCU_PLLN_MUL_127 = RCU_PLLN_MUL127, /*!< PLLN Multiple by 127 */
    RCU_PLLN_MUL_128 = RCU_PLLN_MUL128, /*!< PLLN Multiple by 128 */
    RCU_PLLN_MUL_129 = RCU_PLLN_MUL129, /*!< PLLN Multiple by 129 */
    RCU_PLLN_MUL_130 = RCU_PLLN_MUL130, /*!< PLLN Multiple by 130 */
    RCU_PLLN_MUL_131 = RCU_PLLN_MUL131, /*!< PLLN Multiple by 131 */
    RCU_PLLN_MUL_132 = RCU_PLLN_MUL132, /*!< PLLN Multiple by 132 */
    RCU_PLLN_MUL_133 = RCU_PLLN_MUL133, /*!< PLLN Multiple by 133 */
    RCU_PLLN_MUL_134 = RCU_PLLN_MUL134, /*!< PLLN Multiple by 134 */
    RCU_PLLN_MUL_135 = RCU_PLLN_MUL135, /*!< PLLN Multiple by 135 */
    RCU_PLLN_MUL_136 = RCU_PLLN_MUL136, /*!< PLLN Multiple by 136 */
    RCU_PLLN_MUL_137 = RCU_PLLN_MUL137, /*!< PLLN Multiple by 137 */
    RCU_PLLN_MUL_138 = RCU_PLLN_MUL138, /*!< PLLN Multiple by 138 */
    RCU_PLLN_MUL_139 = RCU_PLLN_MUL139, /*!< PLLN Multiple by 139 */
    RCU_PLLN_MUL_140 = RCU_PLLN_MUL140, /*!< PLLN Multiple by 140 */
    RCU_PLLN_MUL_141 = RCU_PLLN_MUL141, /*!< PLLN Multiple by 141 */
    RCU_PLLN_MUL_142 = RCU_PLLN_MUL142, /*!< PLLN Multiple by 142 */
    RCU_PLLN_MUL_143 = RCU_PLLN_MUL143, /*!< PLLN Multiple by 143 */
    RCU_PLLN_MUL_144 = RCU_PLLN_MUL144, /*!< PLLN Multiple by 144 */
    RCU_PLLN_MUL_145 = RCU_PLLN_MUL145, /*!< PLLN Multiple by 145 */
    RCU_PLLN_MUL_146 = RCU_PLLN_MUL146, /*!< PLLN Multiple by 146 */
    RCU_PLLN_MUL_147 = RCU_PLLN_MUL147, /*!< PLLN Multiple by 147 */
    RCU_PLLN_MUL_148 = RCU_PLLN_MUL148, /*!< PLLN Multiple by 148 */
    RCU_PLLN_MUL_149 = RCU_PLLN_MUL149, /*!< PLLN Multiple by 149 */
    RCU_PLLN_MUL_150 = RCU_PLLN_MUL150, /*!< PLLN Multiple by 150 */
    RCU_PLLN_MUL_151 = RCU_PLLN_MUL151, /*!< PLLN Multiple by 151 */
    RCU_PLLN_MUL_152 = RCU_PLLN_MUL152, /*!< PLLN Multiple by 152 */
    RCU_PLLN_MUL_153 = RCU_PLLN_MUL153, /*!< PLLN Multiple by 153 */
    RCU_PLLN_MUL_154 = RCU_PLLN_MUL154, /*!< PLLN Multiple by 154 */
    RCU_PLLN_MUL_155 = RCU_PLLN_MUL155, /*!< PLLN Multiple by 155 */
    RCU_PLLN_MUL_156 = RCU_PLLN_MUL156, /*!< PLLN Multiple by 156 */
    RCU_PLLN_MUL_157 = RCU_PLLN_MUL157, /*!< PLLN Multiple by 157 */
    RCU_PLLN_MUL_158 = RCU_PLLN_MUL158, /*!< PLLN Multiple by 158 */
    RCU_PLLN_MUL_159 = RCU_PLLN_MUL159, /*!< PLLN Multiple by 159 */
    RCU_PLLN_MUL_160 = RCU_PLLN_MUL160, /*!< PLLN Multiple by 160 */
    RCU_PLLN_MUL_161 = RCU_PLLN_MUL161, /*!< PLLN Multiple by 161 */
    RCU_PLLN_MUL_162 = RCU_PLLN_MUL162, /*!< PLLN Multiple by 162 */
    RCU_PLLN_MUL_163 = RCU_PLLN_MUL163, /*!< PLLN Multiple by 163 */
    RCU_PLLN_MUL_164 = RCU_PLLN_MUL164, /*!< PLLN Multiple by 164 */
    RCU_PLLN_MUL_165 = RCU_PLLN_MUL165, /*!< PLLN Multiple by 165 */
    RCU_PLLN_MUL_166 = RCU_PLLN_MUL166, /*!< PLLN Multiple by 166 */
    RCU_PLLN_MUL_167 = RCU_PLLN_MUL167, /*!< PLLN Multiple by 167 */
    RCU_PLLN_MUL_168 = RCU_PLLN_MUL168, /*!< PLLN Multiple by 168 */
    RCU_PLLN_MUL_169 = RCU_PLLN_MUL169, /*!< PLLN Multiple by 169 */
    RCU_PLLN_MUL_170 = RCU_PLLN_MUL170, /*!< PLLN Multiple by 170 */
    RCU_PLLN_MUL_171 = RCU_PLLN_MUL171, /*!< PLLN Multiple by 171 */
    RCU_PLLN_MUL_172 = RCU_PLLN_MUL172, /*!< PLLN Multiple by 172 */
    RCU_PLLN_MUL_173 = RCU_PLLN_MUL173, /*!< PLLN Multiple by 173 */
    RCU_PLLN_MUL_174 = RCU_PLLN_MUL174, /*!< PLLN Multiple by 174 */
    RCU_PLLN_MUL_175 = RCU_PLLN_MUL175, /*!< PLLN Multiple by 175 */
    RCU_PLLN_MUL_176 = RCU_PLLN_MUL176, /*!< PLLN Multiple by 176 */
    RCU_PLLN_MUL_177 = RCU_PLLN_MUL177, /*!< PLLN Multiple by 177 */
    RCU_PLLN_MUL_178 = RCU_PLLN_MUL178, /*!< PLLN Multiple by 178 */
    RCU_PLLN_MUL_179 = RCU_PLLN_MUL179, /*!< PLLN Multiple by 179 */
    RCU_PLLN_MUL_180 = RCU_PLLN_MUL180, /*!< PLLN Multiple by 180 */
    RCU_PLLN_MUL_181 = RCU_PLLN_MUL181, /*!< PLLN Multiple by 181 */
    RCU_PLLN_MUL_182 = RCU_PLLN_MUL182, /*!< PLLN Multiple by 182 */
    RCU_PLLN_MUL_183 = RCU_PLLN_MUL183, /*!< PLLN Multiple by 183 */
    RCU_PLLN_MUL_184 = RCU_PLLN_MUL184, /*!< PLLN Multiple by 184 */
    RCU_PLLN_MUL_185 = RCU_PLLN_MUL185, /*!< PLLN Multiple by 185 */
    RCU_PLLN_MUL_186 = RCU_PLLN_MUL186, /*!< PLLN Multiple by 186 */
    RCU_PLLN_MUL_187 = RCU_PLLN_MUL187, /*!< PLLN Multiple by 187 */
    RCU_PLLN_MUL_188 = RCU_PLLN_MUL188, /*!< PLLN Multiple by 188 */
    RCU_PLLN_MUL_189 = RCU_PLLN_MUL189, /*!< PLLN Multiple by 189 */
    RCU_PLLN_MUL_190 = RCU_PLLN_MUL190, /*!< PLLN Multiple by 190 */
    RCU_PLLN_MUL_191 = RCU_PLLN_MUL191, /*!< PLLN Multiple by 191 */
    RCU_PLLN_MUL_192 = RCU_PLLN_MUL192, /*!< PLLN Multiple by 192 */
    RCU_PLLN_MUL_193 = RCU_PLLN_MUL193, /*!< PLLN Multiple by 193 */
    RCU_PLLN_MUL_194 = RCU_PLLN_MUL194, /*!< PLLN Multiple by 194 */
    RCU_PLLN_MUL_195 = RCU_PLLN_MUL195, /*!< PLLN Multiple by 195 */
    RCU_PLLN_MUL_196 = RCU_PLLN_MUL196, /*!< PLLN Multiple by 196 */
    RCU_PLLN_MUL_197 = RCU_PLLN_MUL197, /*!< PLLN Multiple by 197 */
    RCU_PLLN_MUL_198 = RCU_PLLN_MUL198, /*!< PLLN Multiple by 198 */
    RCU_PLLN_MUL_199 = RCU_PLLN_MUL199, /*!< PLLN Multiple by 199 */
    RCU_PLLN_MUL_200 = RCU_PLLN_MUL200, /*!< PLLN Multiple by 200 */
    RCU_PLLN_MUL_201 = RCU_PLLN_MUL201, /*!< PLLN Multiple by 201 */
    RCU_PLLN_MUL_202 = RCU_PLLN_MUL202, /*!< PLLN Multiple by 202 */
    RCU_PLLN_MUL_203 = RCU_PLLN_MUL203, /*!< PLLN Multiple by 203 */
    RCU_PLLN_MUL_204 = RCU_PLLN_MUL204, /*!< PLLN Multiple by 204 */
    RCU_PLLN_MUL_205 = RCU_PLLN_MUL205, /*!< PLLN Multiple by 205 */
    RCU_PLLN_MUL_206 = RCU_PLLN_MUL206, /*!< PLLN Multiple by 206 */
    RCU_PLLN_MUL_207 = RCU_PLLN_MUL207, /*!< PLLN Multiple by 207 */
    RCU_PLLN_MUL_208 = RCU_PLLN_MUL208, /*!< PLLN Multiple by 208 */
    RCU_PLLN_MUL_209 = RCU_PLLN_MUL209, /*!< PLLN Multiple by 209 */
    RCU_PLLN_MUL_210 = RCU_PLLN_MUL210, /*!< PLLN Multiple by 210 */
    RCU_PLLN_MUL_211 = RCU_PLLN_MUL211, /*!< PLLN Multiple by 211 */
    RCU_PLLN_MUL_212 = RCU_PLLN_MUL212, /*!< PLLN Multiple by 212 */
    RCU_PLLN_MUL_213 = RCU_PLLN_MUL213, /*!< PLLN Multiple by 213 */
    RCU_PLLN_MUL_214 = RCU_PLLN_MUL214, /*!< PLLN Multiple by 214 */
    RCU_PLLN_MUL_215 = RCU_PLLN_MUL215, /*!< PLLN Multiple by 215 */
    RCU_PLLN_MUL_216 = RCU_PLLN_MUL216, /*!< PLLN Multiple by 216 */
    RCU_PLLN_MUL_217 = RCU_PLLN_MUL217, /*!< PLLN Multiple by 217 */
    RCU_PLLN_MUL_218 = RCU_PLLN_MUL218, /*!< PLLN Multiple by 218 */
    RCU_PLLN_MUL_219 = RCU_PLLN_MUL219, /*!< PLLN Multiple by 219 */
    RCU_PLLN_MUL_220 = RCU_PLLN_MUL220, /*!< PLLN Multiple by 220 */
    RCU_PLLN_MUL_221 = RCU_PLLN_MUL221, /*!< PLLN Multiple by 221 */
    RCU_PLLN_MUL_222 = RCU_PLLN_MUL222, /*!< PLLN Multiple by 222 */
    RCU_PLLN_MUL_223 = RCU_PLLN_MUL223, /*!< PLLN Multiple by 223 */
    RCU_PLLN_MUL_224 = RCU_PLLN_MUL224, /*!< PLLN Multiple by 224 */
    RCU_PLLN_MUL_225 = RCU_PLLN_MUL225, /*!< PLLN Multiple by 225 */
    RCU_PLLN_MUL_226 = RCU_PLLN_MUL226, /*!< PLLN Multiple by 226 */
    RCU_PLLN_MUL_227 = RCU_PLLN_MUL227, /*!< PLLN Multiple by 227 */
    RCU_PLLN_MUL_228 = RCU_PLLN_MUL228, /*!< PLLN Multiple by 228 */
    RCU_PLLN_MUL_229 = RCU_PLLN_MUL229, /*!< PLLN Multiple by 229 */
    RCU_PLLN_MUL_230 = RCU_PLLN_MUL230, /*!< PLLN Multiple by 230 */
    RCU_PLLN_MUL_231 = RCU_PLLN_MUL231, /*!< PLLN Multiple by 231 */
    RCU_PLLN_MUL_232 = RCU_PLLN_MUL232, /*!< PLLN Multiple by 232 */
    RCU_PLLN_MUL_233 = RCU_PLLN_MUL233, /*!< PLLN Multiple by 233 */
    RCU_PLLN_MUL_234 = RCU_PLLN_MUL234, /*!< PLLN Multiple by 234 */
    RCU_PLLN_MUL_235 = RCU_PLLN_MUL235, /*!< PLLN Multiple by 235 */
    RCU_PLLN_MUL_236 = RCU_PLLN_MUL236, /*!< PLLN Multiple by 236 */
    RCU_PLLN_MUL_237 = RCU_PLLN_MUL237, /*!< PLLN Multiple by 237 */
    RCU_PLLN_MUL_238 = RCU_PLLN_MUL238, /*!< PLLN Multiple by 238 */
    RCU_PLLN_MUL_239 = RCU_PLLN_MUL239, /*!< PLLN Multiple by 239 */
    RCU_PLLN_MUL_240 = RCU_PLLN_MUL240, /*!< PLLN Multiple by 240 */
    RCU_PLLN_MUL_241 = RCU_PLLN_MUL241, /*!< PLLN Multiple by 241 */
    RCU_PLLN_MUL_242 = RCU_PLLN_MUL242, /*!< PLLN Multiple by 242 */
    RCU_PLLN_MUL_243 = RCU_PLLN_MUL243, /*!< PLLN Multiple by 243 */
    RCU_PLLN_MUL_244 = RCU_PLLN_MUL244, /*!< PLLN Multiple by 244 */
    RCU_PLLN_MUL_245 = RCU_PLLN_MUL245, /*!< PLLN Multiple by 245 */
    RCU_PLLN_MUL_246 = RCU_PLLN_MUL246, /*!< PLLN Multiple by 246 */
    RCU_PLLN_MUL_247 = RCU_PLLN_MUL247, /*!< PLLN Multiple by 247 */
    RCU_PLLN_MUL_248 = RCU_PLLN_MUL248, /*!< PLLN Multiple by 248 */
    RCU_PLLN_MUL_249 = RCU_PLLN_MUL249, /*!< PLLN Multiple by 249 */
    RCU_PLLN_MUL_250 = RCU_PLLN_MUL250, /*!< PLLN Multiple by 250 */
    RCU_PLLN_MUL_251 = RCU_PLLN_MUL251, /*!< PLLN Multiple by 251 */
    RCU_PLLN_MUL_252 = RCU_PLLN_MUL252, /*!< PLLN Multiple by 252 */
    RCU_PLLN_MUL_253 = RCU_PLLN_MUL253, /*!< PLLN Multiple by 253 */
    RCU_PLLN_MUL_254 = RCU_PLLN_MUL254, /*!< PLLN Multiple by 254 */
    RCU_PLLN_MUL_255 = RCU_PLLN_MUL255, /*!< PLLN Multiple by 255 */
    RCU_PLLN_MUL_256 = RCU_PLLN_MUL256, /*!< PLLN Multiple by 256 */
    RCU_PLLN_MUL_257 = RCU_PLLN_MUL257, /*!< PLLN Multiple by 257 */
    RCU_PLLN_MUL_258 = RCU_PLLN_MUL258, /*!< PLLN Multiple by 258 */
    RCU_PLLN_MUL_259 = RCU_PLLN_MUL259, /*!< PLLN Multiple by 259 */
    RCU_PLLN_MUL_260 = RCU_PLLN_MUL260, /*!< PLLN Multiple by 260 */
    RCU_PLLN_MUL_261 = RCU_PLLN_MUL261, /*!< PLLN Multiple by 261 */
    RCU_PLLN_MUL_262 = RCU_PLLN_MUL262, /*!< PLLN Multiple by 262 */
    RCU_PLLN_MUL_263 = RCU_PLLN_MUL263, /*!< PLLN Multiple by 263 */
    RCU_PLLN_MUL_264 = RCU_PLLN_MUL264, /*!< PLLN Multiple by 264 */
    RCU_PLLN_MUL_265 = RCU_PLLN_MUL265, /*!< PLLN Multiple by 265 */
    RCU_PLLN_MUL_266 = RCU_PLLN_MUL266, /*!< PLLN Multiple by 266 */
    RCU_PLLN_MUL_267 = RCU_PLLN_MUL267, /*!< PLLN Multiple by 267 */
    RCU_PLLN_MUL_268 = RCU_PLLN_MUL268, /*!< PLLN Multiple by 268 */
    RCU_PLLN_MUL_269 = RCU_PLLN_MUL269, /*!< PLLN Multiple by 269 */
    RCU_PLLN_MUL_270 = RCU_PLLN_MUL270, /*!< PLLN Multiple by 270 */
    RCU_PLLN_MUL_271 = RCU_PLLN_MUL271, /*!< PLLN Multiple by 271 */
    RCU_PLLN_MUL_272 = RCU_PLLN_MUL272, /*!< PLLN Multiple by 272 */
    RCU_PLLN_MUL_273 = RCU_PLLN_MUL273, /*!< PLLN Multiple by 273 */
    RCU_PLLN_MUL_274 = RCU_PLLN_MUL274, /*!< PLLN Multiple by 274 */
    RCU_PLLN_MUL_275 = RCU_PLLN_MUL275, /*!< PLLN Multiple by 275 */
    RCU_PLLN_MUL_276 = RCU_PLLN_MUL276, /*!< PLLN Multiple by 276 */
    RCU_PLLN_MUL_277 = RCU_PLLN_MUL277, /*!< PLLN Multiple by 277 */
    RCU_PLLN_MUL_278 = RCU_PLLN_MUL278, /*!< PLLN Multiple by 278 */
    RCU_PLLN_MUL_279 = RCU_PLLN_MUL279, /*!< PLLN Multiple by 279 */
    RCU_PLLN_MUL_280 = RCU_PLLN_MUL280, /*!< PLLN Multiple by 280 */
    RCU_PLLN_MUL_281 = RCU_PLLN_MUL281, /*!< PLLN Multiple by 281 */
    RCU_PLLN_MUL_282 = RCU_PLLN_MUL282, /*!< PLLN Multiple by 282 */
    RCU_PLLN_MUL_283 = RCU_PLLN_MUL283, /*!< PLLN Multiple by 283 */
    RCU_PLLN_MUL_284 = RCU_PLLN_MUL284, /*!< PLLN Multiple by 284 */
    RCU_PLLN_MUL_285 = RCU_PLLN_MUL285, /*!< PLLN Multiple by 285 */
    RCU_PLLN_MUL_286 = RCU_PLLN_MUL286, /*!< PLLN Multiple by 286 */
    RCU_PLLN_MUL_287 = RCU_PLLN_MUL287, /*!< PLLN Multiple by 287 */
    RCU_PLLN_MUL_288 = RCU_PLLN_MUL288, /*!< PLLN Multiple by 288 */
    RCU_PLLN_MUL_289 = RCU_PLLN_MUL289, /*!< PLLN Multiple by 289 */
    RCU_PLLN_MUL_290 = RCU_PLLN_MUL290, /*!< PLLN Multiple by 290 */
    RCU_PLLN_MUL_291 = RCU_PLLN_MUL291, /*!< PLLN Multiple by 291 */
    RCU_PLLN_MUL_292 = RCU_PLLN_MUL292, /*!< PLLN Multiple by 292 */
    RCU_PLLN_MUL_293 = RCU_PLLN_MUL293, /*!< PLLN Multiple by 293 */
    RCU_PLLN_MUL_294 = RCU_PLLN_MUL294, /*!< PLLN Multiple by 294 */
    RCU_PLLN_MUL_295 = RCU_PLLN_MUL295, /*!< PLLN Multiple by 295 */
    RCU_PLLN_MUL_296 = RCU_PLLN_MUL296, /*!< PLLN Multiple by 296 */
    RCU_PLLN_MUL_297 = RCU_PLLN_MUL297, /*!< PLLN Multiple by 297 */
    RCU_PLLN_MUL_298 = RCU_PLLN_MUL298, /*!< PLLN Multiple by 298 */
    RCU_PLLN_MUL_299 = RCU_PLLN_MUL299, /*!< PLLN Multiple by 299 */
    RCU_PLLN_MUL_300 = RCU_PLLN_MUL300, /*!< PLLN Multiple by 300 */
    RCU_PLLN_MUL_301 = RCU_PLLN_MUL301, /*!< PLLN Multiple by 301 */
    RCU_PLLN_MUL_302 = RCU_PLLN_MUL302, /*!< PLLN Multiple by 302 */
    RCU_PLLN_MUL_303 = RCU_PLLN_MUL303, /*!< PLLN Multiple by 303 */
    RCU_PLLN_MUL_304 = RCU_PLLN_MUL304, /*!< PLLN Multiple by 304 */
    RCU_PLLN_MUL_305 = RCU_PLLN_MUL305, /*!< PLLN Multiple by 305 */
    RCU_PLLN_MUL_306 = RCU_PLLN_MUL306, /*!< PLLN Multiple by 306 */
    RCU_PLLN_MUL_307 = RCU_PLLN_MUL307, /*!< PLLN Multiple by 307 */
    RCU_PLLN_MUL_308 = RCU_PLLN_MUL308, /*!< PLLN Multiple by 308 */
    RCU_PLLN_MUL_309 = RCU_PLLN_MUL309, /*!< PLLN Multiple by 309 */
    RCU_PLLN_MUL_310 = RCU_PLLN_MUL310, /*!< PLLN Multiple by 310 */
    RCU_PLLN_MUL_311 = RCU_PLLN_MUL311, /*!< PLLN Multiple by 311 */
    RCU_PLLN_MUL_312 = RCU_PLLN_MUL312, /*!< PLLN Multiple by 312 */
    RCU_PLLN_MUL_313 = RCU_PLLN_MUL313, /*!< PLLN Multiple by 313 */
    RCU_PLLN_MUL_314 = RCU_PLLN_MUL314, /*!< PLLN Multiple by 314 */
    RCU_PLLN_MUL_315 = RCU_PLLN_MUL315, /*!< PLLN Multiple by 315 */
    RCU_PLLN_MUL_316 = RCU_PLLN_MUL316, /*!< PLLN Multiple by 316 */
    RCU_PLLN_MUL_317 = RCU_PLLN_MUL317, /*!< PLLN Multiple by 317 */
    RCU_PLLN_MUL_318 = RCU_PLLN_MUL318, /*!< PLLN Multiple by 318 */
    RCU_PLLN_MUL_319 = RCU_PLLN_MUL319, /*!< PLLN Multiple by 319 */
    RCU_PLLN_MUL_320 = RCU_PLLN_MUL320, /*!< PLLN Multiple by 320 */
    RCU_PLLN_MUL_321 = RCU_PLLN_MUL321, /*!< PLLN Multiple by 321 */
    RCU_PLLN_MUL_322 = RCU_PLLN_MUL322, /*!< PLLN Multiple by 322 */
    RCU_PLLN_MUL_323 = RCU_PLLN_MUL323, /*!< PLLN Multiple by 323 */
    RCU_PLLN_MUL_324 = RCU_PLLN_MUL324, /*!< PLLN Multiple by 324 */
    RCU_PLLN_MUL_325 = RCU_PLLN_MUL325, /*!< PLLN Multiple by 325 */
    RCU_PLLN_MUL_326 = RCU_PLLN_MUL326, /*!< PLLN Multiple by 326 */
    RCU_PLLN_MUL_327 = RCU_PLLN_MUL327, /*!< PLLN Multiple by 327 */
    RCU_PLLN_MUL_328 = RCU_PLLN_MUL328, /*!< PLLN Multiple by 328 */
    RCU_PLLN_MUL_329 = RCU_PLLN_MUL329, /*!< PLLN Multiple by 329 */
    RCU_PLLN_MUL_330 = RCU_PLLN_MUL330, /*!< PLLN Multiple by 330 */
    RCU_PLLN_MUL_331 = RCU_PLLN_MUL331, /*!< PLLN Multiple by 331 */
    RCU_PLLN_MUL_332 = RCU_PLLN_MUL332, /*!< PLLN Multiple by 332 */
    RCU_PLLN_MUL_333 = RCU_PLLN_MUL333, /*!< PLLN Multiple by 333 */
    RCU_PLLN_MUL_334 = RCU_PLLN_MUL334, /*!< PLLN Multiple by 334 */
    RCU_PLLN_MUL_335 = RCU_PLLN_MUL335, /*!< PLLN Multiple by 335 */
    RCU_PLLN_MUL_336 = RCU_PLLN_MUL336, /*!< PLLN Multiple by 336 */
    RCU_PLLN_MUL_337 = RCU_PLLN_MUL337, /*!< PLLN Multiple by 337 */
    RCU_PLLN_MUL_338 = RCU_PLLN_MUL338, /*!< PLLN Multiple by 338 */
    RCU_PLLN_MUL_339 = RCU_PLLN_MUL339, /*!< PLLN Multiple by 339 */
    RCU_PLLN_MUL_340 = RCU_PLLN_MUL340, /*!< PLLN Multiple by 340 */
    RCU_PLLN_MUL_341 = RCU_PLLN_MUL341, /*!< PLLN Multiple by 341 */
    RCU_PLLN_MUL_342 = RCU_PLLN_MUL342, /*!< PLLN Multiple by 342 */
    RCU_PLLN_MUL_343 = RCU_PLLN_MUL343, /*!< PLLN Multiple by 343 */
    RCU_PLLN_MUL_344 = RCU_PLLN_MUL344, /*!< PLLN Multiple by 344 */
    RCU_PLLN_MUL_345 = RCU_PLLN_MUL345, /*!< PLLN Multiple by 345 */
    RCU_PLLN_MUL_346 = RCU_PLLN_MUL346, /*!< PLLN Multiple by 346 */
    RCU_PLLN_MUL_347 = RCU_PLLN_MUL347, /*!< PLLN Multiple by 347 */
    RCU_PLLN_MUL_348 = RCU_PLLN_MUL348, /*!< PLLN Multiple by 348 */
    RCU_PLLN_MUL_349 = RCU_PLLN_MUL349, /*!< PLLN Multiple by 349 */
    RCU_PLLN_MUL_350 = RCU_PLLN_MUL350, /*!< PLLN Multiple by 350 */
    RCU_PLLN_MUL_351 = RCU_PLLN_MUL351, /*!< PLLN Multiple by 351 */
    RCU_PLLN_MUL_352 = RCU_PLLN_MUL352, /*!< PLLN Multiple by 352 */
    RCU_PLLN_MUL_353 = RCU_PLLN_MUL353, /*!< PLLN Multiple by 353 */
    RCU_PLLN_MUL_354 = RCU_PLLN_MUL354, /*!< PLLN Multiple by 354 */
    RCU_PLLN_MUL_355 = RCU_PLLN_MUL355, /*!< PLLN Multiple by 355 */
    RCU_PLLN_MUL_356 = RCU_PLLN_MUL356, /*!< PLLN Multiple by 356 */
    RCU_PLLN_MUL_357 = RCU_PLLN_MUL357, /*!< PLLN Multiple by 357 */
    RCU_PLLN_MUL_358 = RCU_PLLN_MUL358, /*!< PLLN Multiple by 358 */
    RCU_PLLN_MUL_359 = RCU_PLLN_MUL359, /*!< PLLN Multiple by 359 */
    RCU_PLLN_MUL_360 = RCU_PLLN_MUL360, /*!< PLLN Multiple by 360 */
    RCU_PLLN_MUL_361 = RCU_PLLN_MUL361, /*!< PLLN Multiple by 361 */
    RCU_PLLN_MUL_362 = RCU_PLLN_MUL362, /*!< PLLN Multiple by 362 */
    RCU_PLLN_MUL_363 = RCU_PLLN_MUL363, /*!< PLLN Multiple by 363 */
    RCU_PLLN_MUL_364 = RCU_PLLN_MUL364, /*!< PLLN Multiple by 364 */
    RCU_PLLN_MUL_365 = RCU_PLLN_MUL365, /*!< PLLN Multiple by 365 */
    RCU_PLLN_MUL_366 = RCU_PLLN_MUL366, /*!< PLLN Multiple by 366 */
    RCU_PLLN_MUL_367 = RCU_PLLN_MUL367, /*!< PLLN Multiple by 367 */
    RCU_PLLN_MUL_368 = RCU_PLLN_MUL368, /*!< PLLN Multiple by 368 */
    RCU_PLLN_MUL_369 = RCU_PLLN_MUL369, /*!< PLLN Multiple by 369 */
    RCU_PLLN_MUL_370 = RCU_PLLN_MUL370, /*!< PLLN Multiple by 370 */
    RCU_PLLN_MUL_371 = RCU_PLLN_MUL371, /*!< PLLN Multiple by 371 */
    RCU_PLLN_MUL_372 = RCU_PLLN_MUL372, /*!< PLLN Multiple by 372 */
    RCU_PLLN_MUL_373 = RCU_PLLN_MUL373, /*!< PLLN Multiple by 373 */
    RCU_PLLN_MUL_374 = RCU_PLLN_MUL374, /*!< PLLN Multiple by 374 */
    RCU_PLLN_MUL_375 = RCU_PLLN_MUL375, /*!< PLLN Multiple by 375 */
    RCU_PLLN_MUL_376 = RCU_PLLN_MUL376, /*!< PLLN Multiple by 376 */
    RCU_PLLN_MUL_377 = RCU_PLLN_MUL377, /*!< PLLN Multiple by 377 */
    RCU_PLLN_MUL_378 = RCU_PLLN_MUL378, /*!< PLLN Multiple by 378 */
    RCU_PLLN_MUL_379 = RCU_PLLN_MUL379, /*!< PLLN Multiple by 379 */
    RCU_PLLN_MUL_380 = RCU_PLLN_MUL380, /*!< PLLN Multiple by 380 */
    RCU_PLLN_MUL_381 = RCU_PLLN_MUL381, /*!< PLLN Multiple by 381 */
    RCU_PLLN_MUL_382 = RCU_PLLN_MUL382, /*!< PLLN Multiple by 382 */
    RCU_PLLN_MUL_383 = RCU_PLLN_MUL383, /*!< PLLN Multiple by 383 */
    RCU_PLLN_MUL_384 = RCU_PLLN_MUL384, /*!< PLLN Multiple by 384 */
    RCU_PLLN_MUL_385 = RCU_PLLN_MUL385, /*!< PLLN Multiple by 385 */
    RCU_PLLN_MUL_386 = RCU_PLLN_MUL386, /*!< PLLN Multiple by 386 */
    RCU_PLLN_MUL_387 = RCU_PLLN_MUL387, /*!< PLLN Multiple by 387 */
    RCU_PLLN_MUL_388 = RCU_PLLN_MUL388, /*!< PLLN Multiple by 388 */
    RCU_PLLN_MUL_389 = RCU_PLLN_MUL389, /*!< PLLN Multiple by 389 */
    RCU_PLLN_MUL_390 = RCU_PLLN_MUL390, /*!< PLLN Multiple by 390 */
    RCU_PLLN_MUL_391 = RCU_PLLN_MUL391, /*!< PLLN Multiple by 391 */
    RCU_PLLN_MUL_392 = RCU_PLLN_MUL392, /*!< PLLN Multiple by 392 */
    RCU_PLLN_MUL_393 = RCU_PLLN_MUL393, /*!< PLLN Multiple by 393 */
    RCU_PLLN_MUL_394 = RCU_PLLN_MUL394, /*!< PLLN Multiple by 394 */
    RCU_PLLN_MUL_395 = RCU_PLLN_MUL395, /*!< PLLN Multiple by 395 */
    RCU_PLLN_MUL_396 = RCU_PLLN_MUL396, /*!< PLLN Multiple by 396 */
    RCU_PLLN_MUL_397 = RCU_PLLN_MUL397, /*!< PLLN Multiple by 397 */
    RCU_PLLN_MUL_398 = RCU_PLLN_MUL398, /*!< PLLN Multiple by 398 */
    RCU_PLLN_MUL_399 = RCU_PLLN_MUL399, /*!< PLLN Multiple by 399 */
    RCU_PLLN_MUL_400 = RCU_PLLN_MUL400, /*!< PLLN Multiple by 400 */
    RCU_PLLN_MUL_401 = RCU_PLLN_MUL401, /*!< PLLN Multiple by 401 */
    RCU_PLLN_MUL_402 = RCU_PLLN_MUL402, /*!< PLLN Multiple by 402 */
    RCU_PLLN_MUL_403 = RCU_PLLN_MUL403, /*!< PLLN Multiple by 403 */
    RCU_PLLN_MUL_404 = RCU_PLLN_MUL404, /*!< PLLN Multiple by 404 */
    RCU_PLLN_MUL_405 = RCU_PLLN_MUL405, /*!< PLLN Multiple by 405 */
    RCU_PLLN_MUL_406 = RCU_PLLN_MUL406, /*!< PLLN Multiple by 406 */
    RCU_PLLN_MUL_407 = RCU_PLLN_MUL407, /*!< PLLN Multiple by 407 */
    RCU_PLLN_MUL_408 = RCU_PLLN_MUL408, /*!< PLLN Multiple by 408 */
    RCU_PLLN_MUL_409 = RCU_PLLN_MUL409, /*!< PLLN Multiple by 409 */
    RCU_PLLN_MUL_410 = RCU_PLLN_MUL410, /*!< PLLN Multiple by 410 */
    RCU_PLLN_MUL_411 = RCU_PLLN_MUL411, /*!< PLLN Multiple by 411 */
    RCU_PLLN_MUL_412 = RCU_PLLN_MUL412, /*!< PLLN Multiple by 412 */
    RCU_PLLN_MUL_413 = RCU_PLLN_MUL413, /*!< PLLN Multiple by 413 */
    RCU_PLLN_MUL_414 = RCU_PLLN_MUL414, /*!< PLLN Multiple by 414 */
    RCU_PLLN_MUL_415 = RCU_PLLN_MUL415, /*!< PLLN Multiple by 415 */
    RCU_PLLN_MUL_416 = RCU_PLLN_MUL416, /*!< PLLN Multiple by 416 */
    RCU_PLLN_MUL_417 = RCU_PLLN_MUL417, /*!< PLLN Multiple by 417 */
    RCU_PLLN_MUL_418 = RCU_PLLN_MUL418, /*!< PLLN Multiple by 418 */
    RCU_PLLN_MUL_419 = RCU_PLLN_MUL419, /*!< PLLN Multiple by 419 */
    RCU_PLLN_MUL_420 = RCU_PLLN_MUL420, /*!< PLLN Multiple by 420 */
    RCU_PLLN_MUL_421 = RCU_PLLN_MUL421, /*!< PLLN Multiple by 421 */
    RCU_PLLN_MUL_422 = RCU_PLLN_MUL422, /*!< PLLN Multiple by 422 */
    RCU_PLLN_MUL_423 = RCU_PLLN_MUL423, /*!< PLLN Multiple by 423 */
    RCU_PLLN_MUL_424 = RCU_PLLN_MUL424, /*!< PLLN Multiple by 424 */
    RCU_PLLN_MUL_425 = RCU_PLLN_MUL425, /*!< PLLN Multiple by 425 */
    RCU_PLLN_MUL_426 = RCU_PLLN_MUL426, /*!< PLLN Multiple by 426 */
    RCU_PLLN_MUL_427 = RCU_PLLN_MUL427, /*!< PLLN Multiple by 427 */
    RCU_PLLN_MUL_428 = RCU_PLLN_MUL428, /*!< PLLN Multiple by 428 */
    RCU_PLLN_MUL_429 = RCU_PLLN_MUL429, /*!< PLLN Multiple by 429 */
    RCU_PLLN_MUL_430 = RCU_PLLN_MUL430, /*!< PLLN Multiple by 430 */
    RCU_PLLN_MUL_431 = RCU_PLLN_MUL431, /*!< PLLN Multiple by 431 */
    RCU_PLLN_MUL_432 = RCU_PLLN_MUL432, /*!< PLLN Multiple by 432 */
    RCU_PLLN_MUL_433 = RCU_PLLN_MUL433, /*!< PLLN Multiple by 433 */
    RCU_PLLN_MUL_434 = RCU_PLLN_MUL434, /*!< PLLN Multiple by 434 */
    RCU_PLLN_MUL_435 = RCU_PLLN_MUL435, /*!< PLLN Multiple by 435 */
    RCU_PLLN_MUL_436 = RCU_PLLN_MUL436, /*!< PLLN Multiple by 436 */
    RCU_PLLN_MUL_437 = RCU_PLLN_MUL437, /*!< PLLN Multiple by 437 */
    RCU_PLLN_MUL_438 = RCU_PLLN_MUL438, /*!< PLLN Multiple by 438 */
    RCU_PLLN_MUL_439 = RCU_PLLN_MUL439, /*!< PLLN Multiple by 439 */
    RCU_PLLN_MUL_440 = RCU_PLLN_MUL440, /*!< PLLN Multiple by 440 */
    RCU_PLLN_MUL_441 = RCU_PLLN_MUL441, /*!< PLLN Multiple by 441 */
    RCU_PLLN_MUL_442 = RCU_PLLN_MUL442, /*!< PLLN Multiple by 442 */
    RCU_PLLN_MUL_443 = RCU_PLLN_MUL443, /*!< PLLN Multiple by 443 */
    RCU_PLLN_MUL_444 = RCU_PLLN_MUL444, /*!< PLLN Multiple by 444 */
    RCU_PLLN_MUL_445 = RCU_PLLN_MUL445, /*!< PLLN Multiple by 445 */
    RCU_PLLN_MUL_446 = RCU_PLLN_MUL446, /*!< PLLN Multiple by 446 */
    RCU_PLLN_MUL_447 = RCU_PLLN_MUL447, /*!< PLLN Multiple by 447 */
    RCU_PLLN_MUL_448 = RCU_PLLN_MUL448, /*!< PLLN Multiple by 448 */
    RCU_PLLN_MUL_449 = RCU_PLLN_MUL449, /*!< PLLN Multiple by 449 */
    RCU_PLLN_MUL_450 = RCU_PLLN_MUL450, /*!< PLLN Multiple by 450 */
    RCU_PLLN_MUL_451 = RCU_PLLN_MUL451, /*!< PLLN Multiple by 451 */
    RCU_PLLN_MUL_452 = RCU_PLLN_MUL452, /*!< PLLN Multiple by 452 */
    RCU_PLLN_MUL_453 = RCU_PLLN_MUL453, /*!< PLLN Multiple by 453 */
    RCU_PLLN_MUL_454 = RCU_PLLN_MUL454, /*!< PLLN Multiple by 454 */
    RCU_PLLN_MUL_455 = RCU_PLLN_MUL455, /*!< PLLN Multiple by 455 */
    RCU_PLLN_MUL_456 = RCU_PLLN_MUL456, /*!< PLLN Multiple by 456 */
    RCU_PLLN_MUL_457 = RCU_PLLN_MUL457, /*!< PLLN Multiple by 457 */
    RCU_PLLN_MUL_458 = RCU_PLLN_MUL458, /*!< PLLN Multiple by 458 */
    RCU_PLLN_MUL_459 = RCU_PLLN_MUL459, /*!< PLLN Multiple by 459 */
    RCU_PLLN_MUL_460 = RCU_PLLN_MUL460, /*!< PLLN Multiple by 460 */
    RCU_PLLN_MUL_461 = RCU_PLLN_MUL461, /*!< PLLN Multiple by 461 */
    RCU_PLLN_MUL_462 = RCU_PLLN_MUL462, /*!< PLLN Multiple by 462 */
    RCU_PLLN_MUL_463 = RCU_PLLN_MUL463, /*!< PLLN Multiple by 463 */
    RCU_PLLN_MUL_464 = RCU_PLLN_MUL464, /*!< PLLN Multiple by 464 */
    RCU_PLLN_MUL_465 = RCU_PLLN_MUL465, /*!< PLLN Multiple by 465 */
    RCU_PLLN_MUL_466 = RCU_PLLN_MUL466, /*!< PLLN Multiple by 466 */
    RCU_PLLN_MUL_467 = RCU_PLLN_MUL467, /*!< PLLN Multiple by 467 */
    RCU_PLLN_MUL_468 = RCU_PLLN_MUL468, /*!< PLLN Multiple by 468 */
    RCU_PLLN_MUL_469 = RCU_PLLN_MUL469, /*!< PLLN Multiple by 469 */
    RCU_PLLN_MUL_470 = RCU_PLLN_MUL470, /*!< PLLN Multiple by 470 */
    RCU_PLLN_MUL_471 = RCU_PLLN_MUL471, /*!< PLLN Multiple by 471 */
    RCU_PLLN_MUL_472 = RCU_PLLN_MUL472, /*!< PLLN Multiple by 472 */
    RCU_PLLN_MUL_473 = RCU_PLLN_MUL473, /*!< PLLN Multiple by 473 */
    RCU_PLLN_MUL_474 = RCU_PLLN_MUL474, /*!< PLLN Multiple by 474 */
    RCU_PLLN_MUL_475 = RCU_PLLN_MUL475, /*!< PLLN Multiple by 475 */
    RCU_PLLN_MUL_476 = RCU_PLLN_MUL476, /*!< PLLN Multiple by 476 */
    RCU_PLLN_MUL_477 = RCU_PLLN_MUL477, /*!< PLLN Multiple by 477 */
    RCU_PLLN_MUL_478 = RCU_PLLN_MUL478, /*!< PLLN Multiple by 478 */
    RCU_PLLN_MUL_479 = RCU_PLLN_MUL479, /*!< PLLN Multiple by 479 */
    RCU_PLLN_MUL_480 = RCU_PLLN_MUL480, /*!< PLLN Multiple by 480 */
    RCU_PLLN_MUL_481 = RCU_PLLN_MUL481, /*!< PLLN Multiple by 481 */
    RCU_PLLN_MUL_482 = RCU_PLLN_MUL482, /*!< PLLN Multiple by 482 */
    RCU_PLLN_MUL_483 = RCU_PLLN_MUL483, /*!< PLLN Multiple by 483 */
    RCU_PLLN_MUL_484 = RCU_PLLN_MUL484, /*!< PLLN Multiple by 484 */
    RCU_PLLN_MUL_485 = RCU_PLLN_MUL485, /*!< PLLN Multiple by 485 */
    RCU_PLLN_MUL_486 = RCU_PLLN_MUL486, /*!< PLLN Multiple by 486 */
    RCU_PLLN_MUL_487 = RCU_PLLN_MUL487, /*!< PLLN Multiple by 487 */
    RCU_PLLN_MUL_488 = RCU_PLLN_MUL488, /*!< PLLN Multiple by 488 */
    RCU_PLLN_MUL_489 = RCU_PLLN_MUL489, /*!< PLLN Multiple by 489 */
    RCU_PLLN_MUL_490 = RCU_PLLN_MUL490, /*!< PLLN Multiple by 490 */
    RCU_PLLN_MUL_491 = RCU_PLLN_MUL491, /*!< PLLN Multiple by 491 */
    RCU_PLLN_MUL_492 = RCU_PLLN_MUL492, /*!< PLLN Multiple by 492 */
    RCU_PLLN_MUL_493 = RCU_PLLN_MUL493, /*!< PLLN Multiple by 493 */
    RCU_PLLN_MUL_494 = RCU_PLLN_MUL494, /*!< PLLN Multiple by 494 */
    RCU_PLLN_MUL_495 = RCU_PLLN_MUL495, /*!< PLLN Multiple by 495 */
    RCU_PLLN_MUL_496 = RCU_PLLN_MUL496, /*!< PLLN Multiple by 496 */
    RCU_PLLN_MUL_497 = RCU_PLLN_MUL497, /*!< PLLN Multiple by 497 */
    RCU_PLLN_MUL_498 = RCU_PLLN_MUL498, /*!< PLLN Multiple by 498 */
    RCU_PLLN_MUL_499 = RCU_PLLN_MUL499, /*!< PLLN Multiple by 499 */
    RCU_PLLN_MUL_500 = RCU_PLLN_MUL500, /*!< PLLN Multiple by 500 */
    RCU_PLLN_MUL_501 = RCU_PLLN_MUL501, /*!< PLLN Multiple by 501 */
    RCU_PLLN_MUL_502 = RCU_PLLN_MUL502, /*!< PLLN Multiple by 502 */
    RCU_PLLN_MUL_503 = RCU_PLLN_MUL503, /*!< PLLN Multiple by 503 */
    RCU_PLLN_MUL_504 = RCU_PLLN_MUL504, /*!< PLLN Multiple by 504 */
    RCU_PLLN_MUL_505 = RCU_PLLN_MUL505, /*!< PLLN Multiple by 505 */
    RCU_PLLN_MUL_506 = RCU_PLLN_MUL506, /*!< PLLN Multiple by 506 */
    RCU_PLLN_MUL_507 = RCU_PLLN_MUL507, /*!< PLLN Multiple by 507 */
    RCU_PLLN_MUL_508 = RCU_PLLN_MUL508, /*!< PLLN Multiple by 508 */
    RCU_PLLN_MUL_509 = RCU_PLLN_MUL509, /*!< PLLN Multiple by 509 */
    RCU_PLLN_MUL_510 = RCU_PLLN_MUL510, /*!< PLLN Multiple by 510 */
    RCU_PLLN_MUL_511 = RCU_PLLN_MUL511, /*!< PLLN Multiple by 511 */
    RCU_PLLN_MUL_512 = RCU_PLLN_MUL512  /*!< PLLN Multiple by 512 */
} hal_rcu_plln_enum;

/*!< PLL0P output frequency division coefficient */
typedef enum {
    RCU_PLLP_DIV_1   = RCU_PLLP_DIV1,   /*!< PLLP div by 1 */
    RCU_PLLP_DIV_2   = RCU_PLLP_DIV2,   /*!< PLLP div by 2 */
    RCU_PLLP_DIV_3   = RCU_PLLP_DIV3,   /*!< PLLP div by 3 */
    RCU_PLLP_DIV_4   = RCU_PLLP_DIV4,   /*!< PLLP div by 4 */
    RCU_PLLP_DIV_5   = RCU_PLLP_DIV5,   /*!< PLLP div by 5 */
    RCU_PLLP_DIV_6   = RCU_PLLP_DIV6,   /*!< PLLP div by 6 */
    RCU_PLLP_DIV_7   = RCU_PLLP_DIV7,   /*!< PLLP div by 7 */
    RCU_PLLP_DIV_8   = RCU_PLLP_DIV8,   /*!< PLLP div by 8 */
    RCU_PLLP_DIV_9   = RCU_PLLP_DIV9,   /*!< PLLP div by 9 */
    RCU_PLLP_DIV_10  = RCU_PLLP_DIV10,  /*!< PLLP div by 10 */
    RCU_PLLP_DIV_11  = RCU_PLLP_DIV11,  /*!< PLLP div by 11 */
    RCU_PLLP_DIV_12  = RCU_PLLP_DIV12,  /*!< PLLP div by 12 */
    RCU_PLLP_DIV_13  = RCU_PLLP_DIV13,  /*!< PLLP div by 13 */
    RCU_PLLP_DIV_14  = RCU_PLLP_DIV14,  /*!< PLLP div by 14 */
    RCU_PLLP_DIV_15  = RCU_PLLP_DIV15,  /*!< PLLP div by 15 */
    RCU_PLLP_DIV_16  = RCU_PLLP_DIV16,  /*!< PLLP div by 16 */
    RCU_PLLP_DIV_17  = RCU_PLLP_DIV17,  /*!< PLLP div by 17 */
    RCU_PLLP_DIV_18  = RCU_PLLP_DIV18,  /*!< PLLP div by 18 */
    RCU_PLLP_DIV_19  = RCU_PLLP_DIV19,  /*!< PLLP div by 19 */
    RCU_PLLP_DIV_20  = RCU_PLLP_DIV20,  /*!< PLLP div by 20 */
    RCU_PLLP_DIV_21  = RCU_PLLP_DIV21,  /*!< PLLP div by 21 */
    RCU_PLLP_DIV_22  = RCU_PLLP_DIV22,  /*!< PLLP div by 22 */
    RCU_PLLP_DIV_23  = RCU_PLLP_DIV23,  /*!< PLLP div by 23 */
    RCU_PLLP_DIV_24  = RCU_PLLP_DIV24,  /*!< PLLP div by 24 */
    RCU_PLLP_DIV_25  = RCU_PLLP_DIV25,  /*!< PLLP div by 25 */
    RCU_PLLP_DIV_26  = RCU_PLLP_DIV26,  /*!< PLLP div by 26 */
    RCU_PLLP_DIV_27  = RCU_PLLP_DIV27,  /*!< PLLP div by 27 */
    RCU_PLLP_DIV_28  = RCU_PLLP_DIV28,  /*!< PLLP div by 28 */
    RCU_PLLP_DIV_29  = RCU_PLLP_DIV29,  /*!< PLLP div by 29 */
    RCU_PLLP_DIV_30  = RCU_PLLP_DIV30,  /*!< PLLP div by 30 */
    RCU_PLLP_DIV_31  = RCU_PLLP_DIV31,  /*!< PLLP div by 31 */
    RCU_PLLP_DIV_32  = RCU_PLLP_DIV32,  /*!< PLLP div by 32 */
    RCU_PLLP_DIV_33  = RCU_PLLP_DIV33,  /*!< PLLP div by 33 */
    RCU_PLLP_DIV_34  = RCU_PLLP_DIV34,  /*!< PLLP div by 34 */
    RCU_PLLP_DIV_35  = RCU_PLLP_DIV35,  /*!< PLLP div by 35 */
    RCU_PLLP_DIV_36  = RCU_PLLP_DIV36,  /*!< PLLP div by 36 */
    RCU_PLLP_DIV_37  = RCU_PLLP_DIV37,  /*!< PLLP div by 37 */
    RCU_PLLP_DIV_38  = RCU_PLLP_DIV38,  /*!< PLLP div by 38 */
    RCU_PLLP_DIV_39  = RCU_PLLP_DIV39,  /*!< PLLP div by 39 */
    RCU_PLLP_DIV_40  = RCU_PLLP_DIV40,  /*!< PLLP div by 40 */
    RCU_PLLP_DIV_41  = RCU_PLLP_DIV41,  /*!< PLLP div by 41 */
    RCU_PLLP_DIV_42  = RCU_PLLP_DIV42,  /*!< PLLP div by 42 */
    RCU_PLLP_DIV_43  = RCU_PLLP_DIV43,  /*!< PLLP div by 43 */
    RCU_PLLP_DIV_44  = RCU_PLLP_DIV44,  /*!< PLLP div by 44 */
    RCU_PLLP_DIV_45  = RCU_PLLP_DIV45,  /*!< PLLP div by 45 */
    RCU_PLLP_DIV_46  = RCU_PLLP_DIV46,  /*!< PLLP div by 46 */
    RCU_PLLP_DIV_47  = RCU_PLLP_DIV47,  /*!< PLLP div by 47 */
    RCU_PLLP_DIV_48  = RCU_PLLP_DIV48,  /*!< PLLP div by 48 */
    RCU_PLLP_DIV_49  = RCU_PLLP_DIV49,  /*!< PLLP div by 49 */
    RCU_PLLP_DIV_50  = RCU_PLLP_DIV50,  /*!< PLLP div by 50 */
    RCU_PLLP_DIV_51  = RCU_PLLP_DIV51,  /*!< PLLP div by 51 */
    RCU_PLLP_DIV_52  = RCU_PLLP_DIV52,  /*!< PLLP div by 52 */
    RCU_PLLP_DIV_53  = RCU_PLLP_DIV53,  /*!< PLLP div by 53 */
    RCU_PLLP_DIV_54  = RCU_PLLP_DIV54,  /*!< PLLP div by 54 */
    RCU_PLLP_DIV_55  = RCU_PLLP_DIV55,  /*!< PLLP div by 55 */
    RCU_PLLP_DIV_56  = RCU_PLLP_DIV56,  /*!< PLLP div by 56 */
    RCU_PLLP_DIV_57  = RCU_PLLP_DIV57,  /*!< PLLP div by 57 */
    RCU_PLLP_DIV_58  = RCU_PLLP_DIV58,  /*!< PLLP div by 58 */
    RCU_PLLP_DIV_59  = RCU_PLLP_DIV59,  /*!< PLLP div by 59 */
    RCU_PLLP_DIV_60  = RCU_PLLP_DIV60,  /*!< PLLP div by 60 */
    RCU_PLLP_DIV_61  = RCU_PLLP_DIV61,  /*!< PLLP div by 61 */
    RCU_PLLP_DIV_62  = RCU_PLLP_DIV62,  /*!< PLLP div by 62 */
    RCU_PLLP_DIV_63  = RCU_PLLP_DIV63,  /*!< PLLP div by 63 */
    RCU_PLLP_DIV_64  = RCU_PLLP_DIV64,  /*!< PLLP div by 64 */
    RCU_PLLP_DIV_65  = RCU_PLLP_DIV65,  /*!< PLLP div by 65 */
    RCU_PLLP_DIV_66  = RCU_PLLP_DIV66,  /*!< PLLP div by 66 */
    RCU_PLLP_DIV_67  = RCU_PLLP_DIV67,  /*!< PLLP div by 67 */
    RCU_PLLP_DIV_68  = RCU_PLLP_DIV68,  /*!< PLLP div by 68 */
    RCU_PLLP_DIV_69  = RCU_PLLP_DIV69,  /*!< PLLP div by 69 */
    RCU_PLLP_DIV_70  = RCU_PLLP_DIV70,  /*!< PLLP div by 70 */
    RCU_PLLP_DIV_71  = RCU_PLLP_DIV71,  /*!< PLLP div by 71 */
    RCU_PLLP_DIV_72  = RCU_PLLP_DIV72,  /*!< PLLP div by 72 */
    RCU_PLLP_DIV_73  = RCU_PLLP_DIV73,  /*!< PLLP div by 73 */
    RCU_PLLP_DIV_74  = RCU_PLLP_DIV74,  /*!< PLLP div by 74 */
    RCU_PLLP_DIV_75  = RCU_PLLP_DIV75,  /*!< PLLP div by 75 */
    RCU_PLLP_DIV_76  = RCU_PLLP_DIV76,  /*!< PLLP div by 76 */
    RCU_PLLP_DIV_77  = RCU_PLLP_DIV77,  /*!< PLLP div by 77 */
    RCU_PLLP_DIV_78  = RCU_PLLP_DIV78,  /*!< PLLP div by 78 */
    RCU_PLLP_DIV_79  = RCU_PLLP_DIV79,  /*!< PLLP div by 79 */
    RCU_PLLP_DIV_80  = RCU_PLLP_DIV80,  /*!< PLLP div by 80 */
    RCU_PLLP_DIV_81  = RCU_PLLP_DIV81,  /*!< PLLP div by 81 */
    RCU_PLLP_DIV_82  = RCU_PLLP_DIV82,  /*!< PLLP div by 82 */
    RCU_PLLP_DIV_83  = RCU_PLLP_DIV83,  /*!< PLLP div by 83 */
    RCU_PLLP_DIV_84  = RCU_PLLP_DIV84,  /*!< PLLP div by 84 */
    RCU_PLLP_DIV_85  = RCU_PLLP_DIV85,  /*!< PLLP div by 85 */
    RCU_PLLP_DIV_86  = RCU_PLLP_DIV86,  /*!< PLLP div by 86 */
    RCU_PLLP_DIV_87  = RCU_PLLP_DIV87,  /*!< PLLP div by 87 */
    RCU_PLLP_DIV_88  = RCU_PLLP_DIV88,  /*!< PLLP div by 88 */
    RCU_PLLP_DIV_89  = RCU_PLLP_DIV89,  /*!< PLLP div by 89 */
    RCU_PLLP_DIV_90  = RCU_PLLP_DIV90,  /*!< PLLP div by 90 */
    RCU_PLLP_DIV_91  = RCU_PLLP_DIV91,  /*!< PLLP div by 91 */
    RCU_PLLP_DIV_92  = RCU_PLLP_DIV92,  /*!< PLLP div by 92 */
    RCU_PLLP_DIV_93  = RCU_PLLP_DIV93,  /*!< PLLP div by 93 */
    RCU_PLLP_DIV_94  = RCU_PLLP_DIV94,  /*!< PLLP div by 94 */
    RCU_PLLP_DIV_95  = RCU_PLLP_DIV95,  /*!< PLLP div by 95 */
    RCU_PLLP_DIV_96  = RCU_PLLP_DIV96,  /*!< PLLP div by 96 */
    RCU_PLLP_DIV_97  = RCU_PLLP_DIV97,  /*!< PLLP div by 97 */
    RCU_PLLP_DIV_98  = RCU_PLLP_DIV98,  /*!< PLLP div by 98 */
    RCU_PLLP_DIV_99  = RCU_PLLP_DIV99,  /*!< PLLP div by 99 */
    RCU_PLLP_DIV_100 = RCU_PLLP_DIV100, /*!< PLLP div by 100 */
    RCU_PLLP_DIV_101 = RCU_PLLP_DIV101, /*!< PLLP div by 101 */
    RCU_PLLP_DIV_102 = RCU_PLLP_DIV102, /*!< PLLP div by 102 */
    RCU_PLLP_DIV_103 = RCU_PLLP_DIV103, /*!< PLLP div by 103 */
    RCU_PLLP_DIV_104 = RCU_PLLP_DIV104, /*!< PLLP div by 104 */
    RCU_PLLP_DIV_105 = RCU_PLLP_DIV105, /*!< PLLP div by 105 */
    RCU_PLLP_DIV_106 = RCU_PLLP_DIV106, /*!< PLLP div by 106 */
    RCU_PLLP_DIV_107 = RCU_PLLP_DIV107, /*!< PLLP div by 107 */
    RCU_PLLP_DIV_108 = RCU_PLLP_DIV108, /*!< PLLP div by 108 */
    RCU_PLLP_DIV_109 = RCU_PLLP_DIV109, /*!< PLLP div by 109 */
    RCU_PLLP_DIV_110 = RCU_PLLP_DIV110, /*!< PLLP div by 110 */
    RCU_PLLP_DIV_111 = RCU_PLLP_DIV111, /*!< PLLP div by 111 */
    RCU_PLLP_DIV_112 = RCU_PLLP_DIV112, /*!< PLLP div by 112 */
    RCU_PLLP_DIV_113 = RCU_PLLP_DIV113, /*!< PLLP div by 113 */
    RCU_PLLP_DIV_114 = RCU_PLLP_DIV114, /*!< PLLP div by 114 */
    RCU_PLLP_DIV_115 = RCU_PLLP_DIV115, /*!< PLLP div by 115 */
    RCU_PLLP_DIV_116 = RCU_PLLP_DIV116, /*!< PLLP div by 116 */
    RCU_PLLP_DIV_117 = RCU_PLLP_DIV117, /*!< PLLP div by 117 */
    RCU_PLLP_DIV_118 = RCU_PLLP_DIV118, /*!< PLLP div by 118 */
    RCU_PLLP_DIV_119 = RCU_PLLP_DIV119, /*!< PLLP div by 119 */
    RCU_PLLP_DIV_120 = RCU_PLLP_DIV120, /*!< PLLP div by 120 */
    RCU_PLLP_DIV_121 = RCU_PLLP_DIV121, /*!< PLLP div by 121 */
    RCU_PLLP_DIV_122 = RCU_PLLP_DIV122, /*!< PLLP div by 122 */
    RCU_PLLP_DIV_123 = RCU_PLLP_DIV123, /*!< PLLP div by 123 */
    RCU_PLLP_DIV_124 = RCU_PLLP_DIV124, /*!< PLLP div by 124 */
    RCU_PLLP_DIV_125 = RCU_PLLP_DIV125, /*!< PLLP div by 125 */
    RCU_PLLP_DIV_126 = RCU_PLLP_DIV126, /*!< PLLP div by 126 */
    RCU_PLLP_DIV_127 = RCU_PLLP_DIV127, /*!< PLLP div by 127 */
    RCU_PLLP_DIV_128 = RCU_PLLP_DIV128  /*!< PLLP div by 128 */
} hal_rcu_pllp_enum;

/*!< PLL0Q output frequency division coefficient*/
typedef enum {
    RCU_PLLQ_DIV_1   = RCU_PLLQ_DIV1,   /*!< PLLQ div by 1 */
    RCU_PLLQ_DIV_2   = RCU_PLLQ_DIV2,   /*!< PLLQ div by 2 */
    RCU_PLLQ_DIV_3   = RCU_PLLQ_DIV3,   /*!< PLLQ div by 3 */
    RCU_PLLQ_DIV_4   = RCU_PLLQ_DIV4,   /*!< PLLQ div by 4 */
    RCU_PLLQ_DIV_5   = RCU_PLLQ_DIV5,   /*!< PLLQ div by 5 */
    RCU_PLLQ_DIV_6   = RCU_PLLQ_DIV6,   /*!< PLLQ div by 6 */
    RCU_PLLQ_DIV_7   = RCU_PLLQ_DIV7,   /*!< PLLQ div by 7 */
    RCU_PLLQ_DIV_8   = RCU_PLLQ_DIV8,   /*!< PLLQ div by 8 */
    RCU_PLLQ_DIV_9   = RCU_PLLQ_DIV9,   /*!< PLLQ div by 9 */
    RCU_PLLQ_DIV_10  = RCU_PLLQ_DIV10,  /*!< PLLQ div by 10 */
    RCU_PLLQ_DIV_11  = RCU_PLLQ_DIV11,  /*!< PLLQ div by 11 */
    RCU_PLLQ_DIV_12  = RCU_PLLQ_DIV12,  /*!< PLLQ div by 12 */
    RCU_PLLQ_DIV_13  = RCU_PLLQ_DIV13,  /*!< PLLQ div by 13 */
    RCU_PLLQ_DIV_14  = RCU_PLLQ_DIV14,  /*!< PLLQ div by 14 */
    RCU_PLLQ_DIV_15  = RCU_PLLQ_DIV15,  /*!< PLLQ div by 15 */
    RCU_PLLQ_DIV_16  = RCU_PLLQ_DIV16,  /*!< PLLQ div by 16 */
    RCU_PLLQ_DIV_17  = RCU_PLLQ_DIV17,  /*!< PLLQ div by 17 */
    RCU_PLLQ_DIV_18  = RCU_PLLQ_DIV18,  /*!< PLLQ div by 18 */
    RCU_PLLQ_DIV_19  = RCU_PLLQ_DIV19,  /*!< PLLQ div by 19 */
    RCU_PLLQ_DIV_20  = RCU_PLLQ_DIV20,  /*!< PLLQ div by 20 */
    RCU_PLLQ_DIV_21  = RCU_PLLQ_DIV21,  /*!< PLLQ div by 21 */
    RCU_PLLQ_DIV_22  = RCU_PLLQ_DIV22,  /*!< PLLQ div by 22 */
    RCU_PLLQ_DIV_23  = RCU_PLLQ_DIV23,  /*!< PLLQ div by 23 */
    RCU_PLLQ_DIV_24  = RCU_PLLQ_DIV24,  /*!< PLLQ div by 24 */
    RCU_PLLQ_DIV_25  = RCU_PLLQ_DIV25,  /*!< PLLQ div by 25 */
    RCU_PLLQ_DIV_26  = RCU_PLLQ_DIV26,  /*!< PLLQ div by 26 */
    RCU_PLLQ_DIV_27  = RCU_PLLQ_DIV27,  /*!< PLLQ div by 27 */
    RCU_PLLQ_DIV_28  = RCU_PLLQ_DIV28,  /*!< PLLQ div by 28 */
    RCU_PLLQ_DIV_29  = RCU_PLLQ_DIV29,  /*!< PLLQ div by 29 */
    RCU_PLLQ_DIV_30  = RCU_PLLQ_DIV30,  /*!< PLLQ div by 30 */
    RCU_PLLQ_DIV_31  = RCU_PLLQ_DIV31,  /*!< PLLQ div by 31 */
    RCU_PLLQ_DIV_32  = RCU_PLLQ_DIV32,  /*!< PLLQ div by 32 */
    RCU_PLLQ_DIV_33  = RCU_PLLQ_DIV33,  /*!< PLLQ div by 33 */
    RCU_PLLQ_DIV_34  = RCU_PLLQ_DIV34,  /*!< PLLQ div by 34 */
    RCU_PLLQ_DIV_35  = RCU_PLLQ_DIV35,  /*!< PLLQ div by 35 */
    RCU_PLLQ_DIV_36  = RCU_PLLQ_DIV36,  /*!< PLLQ div by 36 */
    RCU_PLLQ_DIV_37  = RCU_PLLQ_DIV37,  /*!< PLLQ div by 37 */
    RCU_PLLQ_DIV_38  = RCU_PLLQ_DIV38,  /*!< PLLQ div by 38 */
    RCU_PLLQ_DIV_39  = RCU_PLLQ_DIV39,  /*!< PLLQ div by 39 */
    RCU_PLLQ_DIV_40  = RCU_PLLQ_DIV40,  /*!< PLLQ div by 40 */
    RCU_PLLQ_DIV_41  = RCU_PLLQ_DIV41,  /*!< PLLQ div by 41 */
    RCU_PLLQ_DIV_42  = RCU_PLLQ_DIV42,  /*!< PLLQ div by 42 */
    RCU_PLLQ_DIV_43  = RCU_PLLQ_DIV43,  /*!< PLLQ div by 43 */
    RCU_PLLQ_DIV_44  = RCU_PLLQ_DIV44,  /*!< PLLQ div by 44 */
    RCU_PLLQ_DIV_45  = RCU_PLLQ_DIV45,  /*!< PLLQ div by 45 */
    RCU_PLLQ_DIV_46  = RCU_PLLQ_DIV46,  /*!< PLLQ div by 46 */
    RCU_PLLQ_DIV_47  = RCU_PLLQ_DIV47,  /*!< PLLQ div by 47 */
    RCU_PLLQ_DIV_48  = RCU_PLLQ_DIV48,  /*!< PLLQ div by 48 */
    RCU_PLLQ_DIV_49  = RCU_PLLQ_DIV49,  /*!< PLLQ div by 49 */
    RCU_PLLQ_DIV_50  = RCU_PLLQ_DIV50,  /*!< PLLQ div by 50 */
    RCU_PLLQ_DIV_51  = RCU_PLLQ_DIV51,  /*!< PLLQ div by 51 */
    RCU_PLLQ_DIV_52  = RCU_PLLQ_DIV52,  /*!< PLLQ div by 52 */
    RCU_PLLQ_DIV_53  = RCU_PLLQ_DIV53,  /*!< PLLQ div by 53 */
    RCU_PLLQ_DIV_54  = RCU_PLLQ_DIV54,  /*!< PLLQ div by 54 */
    RCU_PLLQ_DIV_55  = RCU_PLLQ_DIV55,  /*!< PLLQ div by 55 */
    RCU_PLLQ_DIV_56  = RCU_PLLQ_DIV56,  /*!< PLLQ div by 56 */
    RCU_PLLQ_DIV_57  = RCU_PLLQ_DIV57,  /*!< PLLQ div by 57 */
    RCU_PLLQ_DIV_58  = RCU_PLLQ_DIV58,  /*!< PLLQ div by 58 */
    RCU_PLLQ_DIV_59  = RCU_PLLQ_DIV59,  /*!< PLLQ div by 59 */
    RCU_PLLQ_DIV_60  = RCU_PLLQ_DIV60,  /*!< PLLQ div by 60 */
    RCU_PLLQ_DIV_61  = RCU_PLLQ_DIV61,  /*!< PLLQ div by 61 */
    RCU_PLLQ_DIV_62  = RCU_PLLQ_DIV62,  /*!< PLLQ div by 62 */
    RCU_PLLQ_DIV_63  = RCU_PLLQ_DIV63,  /*!< PLLQ div by 63 */
    RCU_PLLQ_DIV_64  = RCU_PLLQ_DIV64,  /*!< PLLQ div by 64 */
    RCU_PLLQ_DIV_65  = RCU_PLLQ_DIV65,  /*!< PLLQ div by 65 */
    RCU_PLLQ_DIV_66  = RCU_PLLQ_DIV66,  /*!< PLLQ div by 66 */
    RCU_PLLQ_DIV_67  = RCU_PLLQ_DIV67,  /*!< PLLQ div by 67 */
    RCU_PLLQ_DIV_68  = RCU_PLLQ_DIV68,  /*!< PLLQ div by 68 */
    RCU_PLLQ_DIV_69  = RCU_PLLQ_DIV69,  /*!< PLLQ div by 69 */
    RCU_PLLQ_DIV_70  = RCU_PLLQ_DIV70,  /*!< PLLQ div by 20 */
    RCU_PLLQ_DIV_71  = RCU_PLLQ_DIV71,  /*!< PLLQ div by 71 */
    RCU_PLLQ_DIV_72  = RCU_PLLQ_DIV72,  /*!< PLLQ div by 72 */
    RCU_PLLQ_DIV_73  = RCU_PLLQ_DIV73,  /*!< PLLQ div by 73 */
    RCU_PLLQ_DIV_74  = RCU_PLLQ_DIV74,  /*!< PLLQ div by 74 */
    RCU_PLLQ_DIV_75  = RCU_PLLQ_DIV75,  /*!< PLLQ div by 75 */
    RCU_PLLQ_DIV_76  = RCU_PLLQ_DIV76,  /*!< PLLQ div by 76 */
    RCU_PLLQ_DIV_77  = RCU_PLLQ_DIV77,  /*!< PLLQ div by 77 */
    RCU_PLLQ_DIV_78  = RCU_PLLQ_DIV78,  /*!< PLLQ div by 78 */
    RCU_PLLQ_DIV_79  = RCU_PLLQ_DIV79,  /*!< PLLQ div by 79 */
    RCU_PLLQ_DIV_80  = RCU_PLLQ_DIV80,  /*!< PLLQ div by 80 */
    RCU_PLLQ_DIV_81  = RCU_PLLQ_DIV81,  /*!< PLLQ div by 81 */
    RCU_PLLQ_DIV_82  = RCU_PLLQ_DIV82,  /*!< PLLQ div by 82 */
    RCU_PLLQ_DIV_83  = RCU_PLLQ_DIV83,  /*!< PLLQ div by 83 */
    RCU_PLLQ_DIV_84  = RCU_PLLQ_DIV84,  /*!< PLLQ div by 84 */
    RCU_PLLQ_DIV_85  = RCU_PLLQ_DIV85,  /*!< PLLQ div by 85 */
    RCU_PLLQ_DIV_86  = RCU_PLLQ_DIV86,  /*!< PLLQ div by 86 */
    RCU_PLLQ_DIV_87  = RCU_PLLQ_DIV87,  /*!< PLLQ div by 87 */
    RCU_PLLQ_DIV_88  = RCU_PLLQ_DIV88,  /*!< PLLQ div by 88 */
    RCU_PLLQ_DIV_89  = RCU_PLLQ_DIV89,  /*!< PLLQ div by 89 */
    RCU_PLLQ_DIV_90  = RCU_PLLQ_DIV90,  /*!< PLLQ div by 90 */
    RCU_PLLQ_DIV_91  = RCU_PLLQ_DIV91,  /*!< PLLQ div by 91 */
    RCU_PLLQ_DIV_92  = RCU_PLLQ_DIV92,  /*!< PLLQ div by 92 */
    RCU_PLLQ_DIV_93  = RCU_PLLQ_DIV93,  /*!< PLLQ div by 93 */
    RCU_PLLQ_DIV_94  = RCU_PLLQ_DIV94,  /*!< PLLQ div by 94 */
    RCU_PLLQ_DIV_95  = RCU_PLLQ_DIV95,  /*!< PLLQ div by 95 */
    RCU_PLLQ_DIV_96  = RCU_PLLQ_DIV96,  /*!< PLLQ div by 96 */
    RCU_PLLQ_DIV_97  = RCU_PLLQ_DIV97,  /*!< PLLQ div by 97 */
    RCU_PLLQ_DIV_98  = RCU_PLLQ_DIV98,  /*!< PLLQ div by 98 */
    RCU_PLLQ_DIV_99  = RCU_PLLQ_DIV99,  /*!< PLLQ div by 99 */
    RCU_PLLQ_DIV_100 = RCU_PLLQ_DIV100, /*!< PLLQ div by 100 */
    RCU_PLLQ_DIV_101 = RCU_PLLQ_DIV101, /*!< PLLQ div by 101 */
    RCU_PLLQ_DIV_102 = RCU_PLLQ_DIV102, /*!< PLLQ div by 102 */
    RCU_PLLQ_DIV_103 = RCU_PLLQ_DIV103, /*!< PLLQ div by 103 */
    RCU_PLLQ_DIV_104 = RCU_PLLQ_DIV104, /*!< PLLQ div by 104 */
    RCU_PLLQ_DIV_105 = RCU_PLLQ_DIV105, /*!< PLLQ div by 105 */
    RCU_PLLQ_DIV_106 = RCU_PLLQ_DIV106, /*!< PLLQ div by 106 */
    RCU_PLLQ_DIV_107 = RCU_PLLQ_DIV107, /*!< PLLQ div by 107 */
    RCU_PLLQ_DIV_108 = RCU_PLLQ_DIV108, /*!< PLLQ div by 108 */
    RCU_PLLQ_DIV_109 = RCU_PLLQ_DIV109, /*!< PLLQ div by 109 */
    RCU_PLLQ_DIV_110 = RCU_PLLQ_DIV110, /*!< PLLQ div by 110 */
    RCU_PLLQ_DIV_111 = RCU_PLLQ_DIV111, /*!< PLLQ div by 111 */
    RCU_PLLQ_DIV_112 = RCU_PLLQ_DIV112, /*!< PLLQ div by 112 */
    RCU_PLLQ_DIV_113 = RCU_PLLQ_DIV113, /*!< PLLQ div by 113 */
    RCU_PLLQ_DIV_114 = RCU_PLLQ_DIV114, /*!< PLLQ div by 114 */
    RCU_PLLQ_DIV_115 = RCU_PLLQ_DIV115, /*!< PLLQ div by 115 */
    RCU_PLLQ_DIV_116 = RCU_PLLQ_DIV116, /*!< PLLQ div by 116 */
    RCU_PLLQ_DIV_117 = RCU_PLLQ_DIV117, /*!< PLLQ div by 117 */
    RCU_PLLQ_DIV_118 = RCU_PLLQ_DIV118, /*!< PLLQ div by 118 */
    RCU_PLLQ_DIV_119 = RCU_PLLQ_DIV119, /*!< PLLQ div by 119 */
    RCU_PLLQ_DIV_120 = RCU_PLLQ_DIV120, /*!< PLLQ div by 120 */
    RCU_PLLQ_DIV_121 = RCU_PLLQ_DIV121, /*!< PLLQ div by 121 */
    RCU_PLLQ_DIV_122 = RCU_PLLQ_DIV122, /*!< PLLQ div by 122 */
    RCU_PLLQ_DIV_123 = RCU_PLLQ_DIV123, /*!< PLLQ div by 123 */
    RCU_PLLQ_DIV_124 = RCU_PLLQ_DIV124, /*!< PLLQ div by 124 */
    RCU_PLLQ_DIV_125 = RCU_PLLQ_DIV125, /*!< PLLQ div by 125 */
    RCU_PLLQ_DIV_126 = RCU_PLLQ_DIV126, /*!< PLLQ div by 126 */
    RCU_PLLQ_DIV_127 = RCU_PLLQ_DIV127, /*!< PLLQ div by 127 */
    RCU_PLLQ_DIV_128 = RCU_PLLQ_DIV128  /*!< PLLQ div by 128 */
} hal_rcu_pllq_enum;

/*!< PLL0R output frequency dividing coefficient */
typedef enum {
    RCU_PLLR_DIV_1   = RCU_PLLR_DIV1,   /*!< PLLR div by 1 */
    RCU_PLLR_DIV_2   = RCU_PLLR_DIV2,   /*!< PLLR div by 2 */
    RCU_PLLR_DIV_3   = RCU_PLLR_DIV3,   /*!< PLLR div by 3 */
    RCU_PLLR_DIV_4   = RCU_PLLR_DIV4,   /*!< PLLR div by 4 */
    RCU_PLLR_DIV_5   = RCU_PLLR_DIV5,   /*!< PLLR div by 5 */
    RCU_PLLR_DIV_6   = RCU_PLLR_DIV6,   /*!< PLLR div by 6 */
    RCU_PLLR_DIV_7   = RCU_PLLR_DIV7,   /*!< PLLR div by 7 */
    RCU_PLLR_DIV_8   = RCU_PLLR_DIV8,   /*!< PLLR div by 8 */
    RCU_PLLR_DIV_9   = RCU_PLLR_DIV9,   /*!< PLLR div by 9 */
    RCU_PLLR_DIV_10  = RCU_PLLR_DIV10,  /*!< PLLR div by 10 */
    RCU_PLLR_DIV_11  = RCU_PLLR_DIV11,  /*!< PLLR div by 11 */
    RCU_PLLR_DIV_12  = RCU_PLLR_DIV12,  /*!< PLLR div by 12 */
    RCU_PLLR_DIV_13  = RCU_PLLR_DIV13,  /*!< PLLR div by 13 */
    RCU_PLLR_DIV_14  = RCU_PLLR_DIV14,  /*!< PLLR div by 14 */
    RCU_PLLR_DIV_15  = RCU_PLLR_DIV15,  /*!< PLLR div by 15 */
    RCU_PLLR_DIV_16  = RCU_PLLR_DIV16,  /*!< PLLR div by 16 */
    RCU_PLLR_DIV_17  = RCU_PLLR_DIV17,  /*!< PLLR div by 17 */
    RCU_PLLR_DIV_18  = RCU_PLLR_DIV18,  /*!< PLLR div by 18 */
    RCU_PLLR_DIV_19  = RCU_PLLR_DIV19,  /*!< PLLR div by 19 */
    RCU_PLLR_DIV_20  = RCU_PLLR_DIV20,  /*!< PLLR div by 20 */
    RCU_PLLR_DIV_21  = RCU_PLLR_DIV21,  /*!< PLLR div by 21 */
    RCU_PLLR_DIV_22  = RCU_PLLR_DIV22,  /*!< PLLR div by 22 */
    RCU_PLLR_DIV_23  = RCU_PLLR_DIV23,  /*!< PLLR div by 23 */
    RCU_PLLR_DIV_24  = RCU_PLLR_DIV24,  /*!< PLLR div by 24 */
    RCU_PLLR_DIV_25  = RCU_PLLR_DIV25,  /*!< PLLR div by 25 */
    RCU_PLLR_DIV_26  = RCU_PLLR_DIV26,  /*!< PLLR div by 26 */
    RCU_PLLR_DIV_27  = RCU_PLLR_DIV27,  /*!< PLLR div by 27 */
    RCU_PLLR_DIV_28  = RCU_PLLR_DIV28,  /*!< PLLR div by 28 */
    RCU_PLLR_DIV_29  = RCU_PLLR_DIV29,  /*!< PLLR div by 29 */
    RCU_PLLR_DIV_30  = RCU_PLLR_DIV30,  /*!< PLLR div by 30 */
    RCU_PLLR_DIV_31  = RCU_PLLR_DIV31,  /*!< PLLR div by 31 */
    RCU_PLLR_DIV_32  = RCU_PLLR_DIV32,  /*!< PLLR div by 32 */
    RCU_PLLR_DIV_33  = RCU_PLLR_DIV33,  /*!< PLLR div by 33 */
    RCU_PLLR_DIV_34  = RCU_PLLR_DIV34,  /*!< PLLR div by 34 */
    RCU_PLLR_DIV_35  = RCU_PLLR_DIV35,  /*!< PLLR div by 35 */
    RCU_PLLR_DIV_36  = RCU_PLLR_DIV36,  /*!< PLLR div by 36 */
    RCU_PLLR_DIV_37  = RCU_PLLR_DIV37,  /*!< PLLR div by 37 */
    RCU_PLLR_DIV_38  = RCU_PLLR_DIV38,  /*!< PLLR div by 38 */
    RCU_PLLR_DIV_39  = RCU_PLLR_DIV39,  /*!< PLLR div by 39 */
    RCU_PLLR_DIV_40  = RCU_PLLR_DIV40,  /*!< PLLR div by 40 */
    RCU_PLLR_DIV_41  = RCU_PLLR_DIV41,  /*!< PLLR div by 41 */
    RCU_PLLR_DIV_42  = RCU_PLLR_DIV42,  /*!< PLLR div by 42 */
    RCU_PLLR_DIV_43  = RCU_PLLR_DIV43,  /*!< PLLR div by 43 */
    RCU_PLLR_DIV_44  = RCU_PLLR_DIV44,  /*!< PLLR div by 44 */
    RCU_PLLR_DIV_45  = RCU_PLLR_DIV45,  /*!< PLLR div by 45 */
    RCU_PLLR_DIV_46  = RCU_PLLR_DIV46,  /*!< PLLR div by 46 */
    RCU_PLLR_DIV_47  = RCU_PLLR_DIV47,  /*!< PLLR div by 47 */
    RCU_PLLR_DIV_48  = RCU_PLLR_DIV48,  /*!< PLLR div by 48 */
    RCU_PLLR_DIV_49  = RCU_PLLR_DIV49,  /*!< PLLR div by 49 */
    RCU_PLLR_DIV_50  = RCU_PLLR_DIV50,  /*!< PLLR div by 50 */
    RCU_PLLR_DIV_51  = RCU_PLLR_DIV51,  /*!< PLLR div by 51 */
    RCU_PLLR_DIV_52  = RCU_PLLR_DIV52,  /*!< PLLR div by 52 */
    RCU_PLLR_DIV_53  = RCU_PLLR_DIV53,  /*!< PLLR div by 53 */
    RCU_PLLR_DIV_54  = RCU_PLLR_DIV54,  /*!< PLLR div by 54 */
    RCU_PLLR_DIV_55  = RCU_PLLR_DIV55,  /*!< PLLR div by 55 */
    RCU_PLLR_DIV_56  = RCU_PLLR_DIV56,  /*!< PLLR div by 56 */
    RCU_PLLR_DIV_57  = RCU_PLLR_DIV57,  /*!< PLLR div by 57 */
    RCU_PLLR_DIV_58  = RCU_PLLR_DIV58,  /*!< PLLR div by 58 */
    RCU_PLLR_DIV_59  = RCU_PLLR_DIV59,  /*!< PLLR div by 59 */
    RCU_PLLR_DIV_60  = RCU_PLLR_DIV60,  /*!< PLLR div by 60 */
    RCU_PLLR_DIV_61  = RCU_PLLR_DIV61,  /*!< PLLR div by 61 */
    RCU_PLLR_DIV_62  = RCU_PLLR_DIV62,  /*!< PLLR div by 62 */
    RCU_PLLR_DIV_63  = RCU_PLLR_DIV63,  /*!< PLLR div by 63 */
    RCU_PLLR_DIV_64  = RCU_PLLR_DIV64,  /*!< PLLR div by 64 */
    RCU_PLLR_DIV_65  = RCU_PLLR_DIV65,  /*!< PLLR div by 65 */
    RCU_PLLR_DIV_66  = RCU_PLLR_DIV66,  /*!< PLLR div by 66 */
    RCU_PLLR_DIV_67  = RCU_PLLR_DIV67,  /*!< PLLR div by 67 */
    RCU_PLLR_DIV_68  = RCU_PLLR_DIV68,  /*!< PLLR div by 68 */
    RCU_PLLR_DIV_69  = RCU_PLLR_DIV69,  /*!< PLLR div by 69 */
    RCU_PLLR_DIV_70  = RCU_PLLR_DIV70,  /*!< PLLR div by 20 */
    RCU_PLLR_DIV_71  = RCU_PLLR_DIV71,  /*!< PLLR div by 71 */
    RCU_PLLR_DIV_72  = RCU_PLLR_DIV72,  /*!< PLLR div by 72 */
    RCU_PLLR_DIV_73  = RCU_PLLR_DIV73,  /*!< PLLR div by 73 */
    RCU_PLLR_DIV_74  = RCU_PLLR_DIV74,  /*!< PLLR div by 74 */
    RCU_PLLR_DIV_75  = RCU_PLLR_DIV75,  /*!< PLLR div by 75 */
    RCU_PLLR_DIV_76  = RCU_PLLR_DIV76,  /*!< PLLR div by 76 */
    RCU_PLLR_DIV_77  = RCU_PLLR_DIV77,  /*!< PLLR div by 77 */
    RCU_PLLR_DIV_78  = RCU_PLLR_DIV78,  /*!< PLLR div by 78 */
    RCU_PLLR_DIV_79  = RCU_PLLR_DIV79,  /*!< PLLR div by 79 */
    RCU_PLLR_DIV_80  = RCU_PLLR_DIV80,  /*!< PLLR div by 80 */
    RCU_PLLR_DIV_81  = RCU_PLLR_DIV81,  /*!< PLLR div by 81 */
    RCU_PLLR_DIV_82  = RCU_PLLR_DIV82,  /*!< PLLR div by 82 */
    RCU_PLLR_DIV_83  = RCU_PLLR_DIV83,  /*!< PLLR div by 83 */
    RCU_PLLR_DIV_84  = RCU_PLLR_DIV84,  /*!< PLLR div by 84 */
    RCU_PLLR_DIV_85  = RCU_PLLR_DIV85,  /*!< PLLR div by 85 */
    RCU_PLLR_DIV_86  = RCU_PLLR_DIV86,  /*!< PLLR div by 86 */
    RCU_PLLR_DIV_87  = RCU_PLLR_DIV87,  /*!< PLLR div by 87 */
    RCU_PLLR_DIV_88  = RCU_PLLR_DIV88,  /*!< PLLR div by 88 */
    RCU_PLLR_DIV_89  = RCU_PLLR_DIV89,  /*!< PLLR div by 89 */
    RCU_PLLR_DIV_90  = RCU_PLLR_DIV90,  /*!< PLLR div by 90 */
    RCU_PLLR_DIV_91  = RCU_PLLR_DIV91,  /*!< PLLR div by 91 */
    RCU_PLLR_DIV_92  = RCU_PLLR_DIV92,  /*!< PLLR div by 92 */
    RCU_PLLR_DIV_93  = RCU_PLLR_DIV93,  /*!< PLLR div by 93 */
    RCU_PLLR_DIV_94  = RCU_PLLR_DIV94,  /*!< PLLR div by 94 */
    RCU_PLLR_DIV_95  = RCU_PLLR_DIV95,  /*!< PLLR div by 95 */
    RCU_PLLR_DIV_96  = RCU_PLLR_DIV96,  /*!< PLLR div by 96 */
    RCU_PLLR_DIV_97  = RCU_PLLR_DIV97,  /*!< PLLR div by 97 */
    RCU_PLLR_DIV_98  = RCU_PLLR_DIV98,  /*!< PLLR div by 98 */
    RCU_PLLR_DIV_99  = RCU_PLLR_DIV99,  /*!< PLLR div by 99 */
    RCU_PLLR_DIV_100 = RCU_PLLR_DIV100, /*!< PLLR div by 100 */
    RCU_PLLR_DIV_101 = RCU_PLLR_DIV101, /*!< PLLR div by 101 */
    RCU_PLLR_DIV_102 = RCU_PLLR_DIV102, /*!< PLLR div by 102 */
    RCU_PLLR_DIV_103 = RCU_PLLR_DIV103, /*!< PLLR div by 103 */
    RCU_PLLR_DIV_104 = RCU_PLLR_DIV104, /*!< PLLR div by 104 */
    RCU_PLLR_DIV_105 = RCU_PLLR_DIV105, /*!< PLLR div by 105 */
    RCU_PLLR_DIV_106 = RCU_PLLR_DIV106, /*!< PLLR div by 106 */
    RCU_PLLR_DIV_107 = RCU_PLLR_DIV107, /*!< PLLR div by 107 */
    RCU_PLLR_DIV_108 = RCU_PLLR_DIV108, /*!< PLLR div by 108 */
    RCU_PLLR_DIV_109 = RCU_PLLR_DIV109, /*!< PLLR div by 109 */
    RCU_PLLR_DIV_110 = RCU_PLLR_DIV110, /*!< PLLR div by 110 */
    RCU_PLLR_DIV_111 = RCU_PLLR_DIV111, /*!< PLLR div by 111 */
    RCU_PLLR_DIV_112 = RCU_PLLR_DIV112, /*!< PLLR div by 112 */
    RCU_PLLR_DIV_113 = RCU_PLLR_DIV113, /*!< PLLR div by 113 */
    RCU_PLLR_DIV_114 = RCU_PLLR_DIV114, /*!< PLLR div by 114 */
    RCU_PLLR_DIV_115 = RCU_PLLR_DIV115, /*!< PLLR div by 115 */
    RCU_PLLR_DIV_116 = RCU_PLLR_DIV116, /*!< PLLR div by 116 */
    RCU_PLLR_DIV_117 = RCU_PLLR_DIV117, /*!< PLLR div by 117 */
    RCU_PLLR_DIV_118 = RCU_PLLR_DIV118, /*!< PLLR div by 118 */
    RCU_PLLR_DIV_119 = RCU_PLLR_DIV119, /*!< PLLR div by 119 */
    RCU_PLLR_DIV_120 = RCU_PLLR_DIV120, /*!< PLLR div by 120 */
    RCU_PLLR_DIV_121 = RCU_PLLR_DIV121, /*!< PLLR div by 121 */
    RCU_PLLR_DIV_122 = RCU_PLLR_DIV122, /*!< PLLR div by 122 */
    RCU_PLLR_DIV_123 = RCU_PLLR_DIV123, /*!< PLLR div by 123 */
    RCU_PLLR_DIV_124 = RCU_PLLR_DIV124, /*!< PLLR div by 124 */
    RCU_PLLR_DIV_125 = RCU_PLLR_DIV125, /*!< PLLR div by 125 */
    RCU_PLLR_DIV_126 = RCU_PLLR_DIV126, /*!< PLLR div by 126 */
    RCU_PLLR_DIV_127 = RCU_PLLR_DIV127, /*!< PLLR div by 127 */
    RCU_PLLR_DIV_128 = RCU_PLLR_DIV128  /*!< PLLR div by 128 */
} hal_rcu_pllr_enum;

/* RTC clock Source Selection*/
typedef enum {
    RCU_RTC_CLKSRC_NONE   = RCU_RTCSRC_NONE,   /*!< no clock selected */
    RCU_RTC_CLKSRC_LXTAL  = RCU_RTCSRC_LXTAL,  /*!< LXTAL selected as RTC source clock */
    RCU_RTC_CLKSRC_IRC32K = RCU_RTCSRC_IRC32K, /*!< IRC32K selected as RTC source clock */
    RCU_RTC_CLKSRC_HXTAL_DIV_RTCDIV = RCU_RTCSRC_HXTAL_DIV_RTCDIV /*!< RCU_RTCSRC_HXTAL_DIV_RTCDIV selected as RTC source clock */
} hal_rcu_rtc_cksrc_enum;

/* RTC clock divider factor from HXTAL clock Selection*/
typedef enum {
    RCU_RTC_HXTAL_DIV_NONE = RCU_RTCSRC_NONE, /*!< RTC Clock div by none */
    RCU_RTC_HXTAL_DIV_2    = RCU_RTC_DIV2,    /*!< RTC Clock div by 2 */
    RCU_RTC_HXTAL_DIV_3    = RCU_RTC_DIV3,    /*!< RTC Clock div by 3 */
    RCU_RTC_HXTAL_DIV_4    = RCU_RTC_DIV4,    /*!< RTC Clock div by 4 */
    RCU_RTC_HXTAL_DIV_5    = RCU_RTC_DIV5,    /*!< RTC Clock div by 5 */
    RCU_RTC_HXTAL_DIV_6    = RCU_RTC_DIV6,    /*!< RTC Clock div by 6 */
    RCU_RTC_HXTAL_DIV_7    = RCU_RTC_DIV7,    /*!< RTC Clock div by 7 */
    RCU_RTC_HXTAL_DIV_8    = RCU_RTC_DIV8,    /*!< RTC Clock div by 8 */
    RCU_RTC_HXTAL_DIV_9    = RCU_RTC_DIV9,    /*!< RTC Clock div by 9 */
    RCU_RTC_HXTAL_DIV_10   = RCU_RTC_DIV10,   /*!< RTC Clock div by 10 */
    RCU_RTC_HXTAL_DIV_11   = RCU_RTC_DIV11,   /*!< RTC Clock div by 11 */
    RCU_RTC_HXTAL_DIV_12   = RCU_RTC_DIV12,   /*!< RTC Clock div by 12 */
    RCU_RTC_HXTAL_DIV_13   = RCU_RTC_DIV13,   /*!< RTC Clock div by 13 */
    RCU_RTC_HXTAL_DIV_14   = RCU_RTC_DIV14,   /*!< RTC Clock div by 14 */
    RCU_RTC_HXTAL_DIV_15   = RCU_RTC_DIV15,   /*!< RTC Clock div by 15 */
    RCU_RTC_HXTAL_DIV_16   = RCU_RTC_DIV16,   /*!< RTC Clock div by 16 */
    RCU_RTC_HXTAL_DIV_17   = RCU_RTC_DIV17,   /*!< RTC Clock div by 17 */
    RCU_RTC_HXTAL_DIV_18   = RCU_RTC_DIV18,   /*!< RTC Clock div by 18 */
    RCU_RTC_HXTAL_DIV_19   = RCU_RTC_DIV19,   /*!< RTC Clock div by 19 */
    RCU_RTC_HXTAL_DIV_20   = RCU_RTC_DIV20,   /*!< RTC Clock div by 20 */
    RCU_RTC_HXTAL_DIV_21   = RCU_RTC_DIV21,   /*!< RTC Clock div by 21 */
    RCU_RTC_HXTAL_DIV_22   = RCU_RTC_DIV22,   /*!< RTC Clock div by 22 */
    RCU_RTC_HXTAL_DIV_23   = RCU_RTC_DIV23,   /*!< RTC Clock div by 23 */
    RCU_RTC_HXTAL_DIV_24   = RCU_RTC_DIV24,   /*!< RTC Clock div by 24 */
    RCU_RTC_HXTAL_DIV_25   = RCU_RTC_DIV25,   /*!< RTC Clock div by 25 */
    RCU_RTC_HXTAL_DIV_26   = RCU_RTC_DIV26,   /*!< RTC Clock div by 26 */
    RCU_RTC_HXTAL_DIV_27   = RCU_RTC_DIV27,   /*!< RTC Clock div by 27 */
    RCU_RTC_HXTAL_DIV_28   = RCU_RTC_DIV28,   /*!< RTC Clock div by 28 */
    RCU_RTC_HXTAL_DIV_29   = RCU_RTC_DIV29,   /*!< RTC Clock div by 29 */
    RCU_RTC_HXTAL_DIV_30   = RCU_RTC_DIV30,   /*!< RTC Clock div by 30 */
    RCU_RTC_HXTAL_DIV_31   = RCU_RTC_DIV31,   /*!< RTC Clock div by 31 */
    RCU_RTC_HXTAL_DIV_32   = RCU_RTC_DIV32,   /*!< RTC Clock div by 32 */
    RCU_RTC_HXTAL_DIV_33   = RCU_RTC_DIV33,   /*!< RTC Clock div by 33 */
    RCU_RTC_HXTAL_DIV_34   = RCU_RTC_DIV34,   /*!< RTC Clock div by 34 */
    RCU_RTC_HXTAL_DIV_35   = RCU_RTC_DIV35,   /*!< RTC Clock div by 35 */
    RCU_RTC_HXTAL_DIV_36   = RCU_RTC_DIV36,   /*!< RTC Clock div by 36 */
    RCU_RTC_HXTAL_DIV_37   = RCU_RTC_DIV37,   /*!< RTC Clock div by 37 */
    RCU_RTC_HXTAL_DIV_38   = RCU_RTC_DIV38,   /*!< RTC Clock div by 38 */
    RCU_RTC_HXTAL_DIV_39   = RCU_RTC_DIV39,   /*!< RTC Clock div by 39 */
    RCU_RTC_HXTAL_DIV_40   = RCU_RTC_DIV40,   /*!< RTC Clock div by 40 */
    RCU_RTC_HXTAL_DIV_41   = RCU_RTC_DIV41,   /*!< RTC Clock div by 41 */
    RCU_RTC_HXTAL_DIV_42   = RCU_RTC_DIV42,   /*!< RTC Clock div by 42 */
    RCU_RTC_HXTAL_DIV_43   = RCU_RTC_DIV43,   /*!< RTC Clock div by 43 */
    RCU_RTC_HXTAL_DIV_44   = RCU_RTC_DIV44,   /*!< RTC Clock div by 44 */
    RCU_RTC_HXTAL_DIV_45   = RCU_RTC_DIV45,   /*!< RTC Clock div by 45 */
    RCU_RTC_HXTAL_DIV_46   = RCU_RTC_DIV46,   /*!< RTC Clock div by 46 */
    RCU_RTC_HXTAL_DIV_47   = RCU_RTC_DIV47,   /*!< RTC Clock div by 47 */
    RCU_RTC_HXTAL_DIV_48   = RCU_RTC_DIV48,   /*!< RTC Clock div by 48 */
    RCU_RTC_HXTAL_DIV_49   = RCU_RTC_DIV49,   /*!< RTC Clock div by 49 */
    RCU_RTC_HXTAL_DIV_50   = RCU_RTC_DIV50,   /*!< RTC Clock div by 50 */
    RCU_RTC_HXTAL_DIV_51   = RCU_RTC_DIV51,   /*!< RTC Clock div by 51 */
    RCU_RTC_HXTAL_DIV_52   = RCU_RTC_DIV52,   /*!< RTC Clock div by 52 */
    RCU_RTC_HXTAL_DIV_53   = RCU_RTC_DIV53,   /*!< RTC Clock div by 53 */
    RCU_RTC_HXTAL_DIV_54   = RCU_RTC_DIV54,   /*!< RTC Clock div by 54 */
    RCU_RTC_HXTAL_DIV_55   = RCU_RTC_DIV55,   /*!< RTC Clock div by 55 */
    RCU_RTC_HXTAL_DIV_56   = RCU_RTC_DIV56,   /*!< RTC Clock div by 56 */
    RCU_RTC_HXTAL_DIV_57   = RCU_RTC_DIV57,   /*!< RTC Clock div by 57 */
    RCU_RTC_HXTAL_DIV_58   = RCU_RTC_DIV58,   /*!< RTC Clock div by 58 */
    RCU_RTC_HXTAL_DIV_59   = RCU_RTC_DIV59,   /*!< RTC Clock div by 59 */
    RCU_RTC_HXTAL_DIV_60   = RCU_RTC_DIV60,   /*!< RTC Clock div by 60 */
    RCU_RTC_HXTAL_DIV_61   = RCU_RTC_DIV61,   /*!< RTC Clock div by 61 */
    RCU_RTC_HXTAL_DIV_62   = RCU_RTC_DIV62,   /*!< RTC Clock div by 62 */
    RCU_RTC_HXTAL_DIV_63   = RCU_RTC_DIV63    /*!< RTC Clock div by 63 */
} hal_rcu_rtc_div_enum;

/* LXTAL LXTAL drive capability */
typedef enum {
    RCU_LXTAL_DRIVE_LOWDRI      = RCU_LXTAL_LOWDRI,      /* LXTAL drive capability : Low */
    RCU_LXTAL_DRIVE_MED_LOWDRI  = RCU_LXTAL_MED_LOWDRI,  /* LXTAL drive capability : MED_LOWDRI */
    RCU_LXTAL_DRIVE_MED_HIGHDRI = RCU_LXTAL_MED_HIGHDRI, /* LXTAL drive capability : MED_HIGHDRI */
    RCU_LXTAL_DRIVE_HIGHDRI     = RCU_LXTAL_HIGHDRI      /* LXTAL drive capability : HIGHDRI */
} hal_rcu_lxtal_drive_capability_enum;

/* configure the clock out 0 to output on CKOUT pin */
/* @PARA: ckout_src0 */
/* @ENUM: RCU clock out source */
typedef enum {
    RCU_CKOUT0_SRC_CK_IRC64MDIV = RCU_CKOUT0SRC_IRC64MDIV, /*!< CK_OUT clock source select IRC64MDIV */
    RCU_CKOUT0_SRC_CK_LXTAL     = RCU_CKOUT0SRC_LXTAL,     /*!< CK_OUT clock source select LXTAL */
    RCU_CKOUT0_SRC_CK_HXTAL     = RCU_CKOUT0SRC_HXTAL,     /*!< CK_OUT clock source select HXTAL */
    RCU_CKOUT0_SRC_CK_PLL0P     = RCU_CKOUT0SRC_PLL0P,     /*!< CK_OUT clock source select PLL0P */
    RCU_CKOUT0_SRC_CK_IRC48M    = RCU_CKOUT0SRC_IRC48M,    /*!< CK_OUT clock source select IRC48M */
    RCU_CKOUT0_SRC_CK_PER       = RCU_CKOUT0SRC_PER,       /*!< CK_OUT clock source select PER */
    RCU_CKOUT0_SRC_USBHS0_60M   = RCU_CKOUT0SRC_USBHS060M, /*!< CK_OUT clock source select USBHS0 60M */
    RCU_CKOUT0_SRC_USBHS1_60M   = RCU_CKOUT0SRC_USBHS160M  /*!< CK_OUT clock source select USBHS1 60M */
} hal_rcu_clkout0_src_enum;

/* configure the clock out 1 to output on CKOUT pin */
/* @PARA: ckout_src1 */
/* @ENUM: RCU clock out source */
typedef enum {
    RCU_CKOUT1_SRC_SYS     = RCU_CKOUT1SRC_SYSTEMCLOCK, /*!< CK_OUT clock source select CKSYS */
    RCU_CKOUT1_SRC_PLL1R   = RCU_CKOUT1SRC_PLL1R,       /*!< CK_OUT clock source select PLL1R */
    RCU_CKOUT1_SRC_HXTAL   = RCU_CKOUT1SRC_HXTAL,       /*!< CK_OUT clock source select HXTAL */
    RCU_CKOUT1_SRC_PLL0P   = RCU_CKOUT1SRC_PLL0P,       /*!< CK_OUT clock source select PLL0P */
    RCU_CKOUT1_SRC_LPIRC4M = RCU_CKOUT1SRC_LPIRC4M,     /*!< CK_OUT clock source select LPIRC4M */
    RCU_CKOUT1_SRC_IRC32K  = RCU_CKOUT1SRC_IRC32K,      /*!< CK_OUT clock source select IRC32K */
    RCU_CKOUT1_SRC_PLL2R   = RCU_CKOUT1SRC_PLL2R        /*!< CK_OUT clock source select PLL2R */
} hal_rcu_clkout1_src_enum;

/* @PARA: ckout0_div */
/* @ENUM: RCU clock 0 out divider */
typedef enum {
    RCU_CLKOUT0_DIV_1  = RCU_CKOUT0_DIV1,  /*!< CK_OUT 0 is divided by 1 */
    RCU_CLKOUT0_DIV_2  = RCU_CKOUT0_DIV2,  /*!< CK_OUT 0 is divided by 2 */
    RCU_CLKOUT0_DIV_3  = RCU_CKOUT0_DIV3,  /*!< CK_OUT 0 is divided by 3 */
    RCU_CLKOUT0_DIV_4  = RCU_CKOUT0_DIV4,  /*!< CK_OUT 0 is divided by 4 */
    RCU_CLKOUT0_DIV_5  = RCU_CKOUT0_DIV5,  /*!< CK_OUT 0 is divided by 5 */
    RCU_CLKOUT0_DIV_6  = RCU_CKOUT0_DIV6,  /*!< CK_OUT 0 is divided by 6 */
    RCU_CLKOUT0_DIV_7  = RCU_CKOUT0_DIV7,  /*!< CK_OUT 0 is divided by 7 */
    RCU_CLKOUT0_DIV_8  = RCU_CKOUT0_DIV8,  /*!< CK_OUT 0 is divided by 8 */
    RCU_CLKOUT0_DIV_9  = RCU_CKOUT0_DIV9,  /*!< CK_OUT 0 is divided by 9 */
    RCU_CLKOUT0_DIV_10 = RCU_CKOUT0_DIV10, /*!< CK_OUT 0 is divided by 10 */
    RCU_CLKOUT0_DIV_11 = RCU_CKOUT0_DIV11, /*!< CK_OUT 0 is divided by 11 */
    RCU_CLKOUT0_DIV_12 = RCU_CKOUT0_DIV12, /*!< CK_OUT 0 is divided by 12 */
    RCU_CLKOUT0_DIV_13 = RCU_CKOUT0_DIV13, /*!< CK_OUT 0 is divided by 13 */
    RCU_CLKOUT0_DIV_14 = RCU_CKOUT0_DIV14, /*!< CK_OUT 0 is divided by 14 */
    RCU_CLKOUT0_DIV_15 = RCU_CKOUT0_DIV15  /*!< CK_OUT 0 is divided by 15 */
} hal_rcu_ckout0_div_enum;

/* @PARA: ckout1_div */
/* @ENUM: RCU clock 1 out divider */
typedef enum {
    RCU_CLKOUT1_DIV_1  = RCU_CKOUT1_DIV1,  /*!< CK_OUT 1 is divided by 1 */
    RCU_CLKOUT1_DIV_2  = RCU_CKOUT1_DIV2,  /*!< CK_OUT 1 is divided by 2 */
    RCU_CLKOUT1_DIV_3  = RCU_CKOUT1_DIV3,  /*!< CK_OUT 1 is divided by 3 */
    RCU_CLKOUT1_DIV_4  = RCU_CKOUT1_DIV4,  /*!< CK_OUT 1 is divided by 4 */
    RCU_CLKOUT1_DIV_5  = RCU_CKOUT1_DIV5,  /*!< CK_OUT 1 is divided by 5 */
    RCU_CLKOUT1_DIV_6  = RCU_CKOUT1_DIV6,  /*!< CK_OUT 1 is divided by 6 */
    RCU_CLKOUT1_DIV_7  = RCU_CKOUT1_DIV7,  /*!< CK_OUT 1 is divided by 7 */
    RCU_CLKOUT1_DIV_8  = RCU_CKOUT1_DIV8,  /*!< CK_OUT 1 is divided by 8 */
    RCU_CLKOUT1_DIV_9  = RCU_CKOUT1_DIV9,  /*!< CK_OUT 1 is divided by 9 */
    RCU_CLKOUT1_DIV_10 = RCU_CKOUT1_DIV10, /*!< CK_OUT 1 is divided by 10 */
    RCU_CLKOUT1_DIV_11 = RCU_CKOUT1_DIV11, /*!< CK_OUT 1 is divided by 11 */
    RCU_CLKOUT1_DIV_12 = RCU_CKOUT1_DIV12, /*!< CK_OUT 1 is divided by 12 */
    RCU_CLKOUT1_DIV_13 = RCU_CKOUT1_DIV13, /*!< CK_OUT 1 is divided by 13 */
    RCU_CLKOUT1_DIV_14 = RCU_CKOUT1_DIV14, /*!< CK_OUT 1 is divided by 14 */
    RCU_CLKOUT1_DIV_15 = RCU_CKOUT1_DIV15  /*!< CK_OUT 1 is divided by 15 */
} hal_rcu_ckout1_div_enum;

/* @PARA: IRC64M MDiv */
/* @ENUM: IRC64M clock Multiple divider */
typedef enum {
    RCU_IRC64M_DIV_1 = RCU_IRC64M_DIV1, /*!< IRC64M clock divider by 1 */
    RCU_IRC64M_DIV_2 = RCU_IRC64M_DIV2, /*!< IRC64M clock divider by 2 */
    RCU_IRC64M_DIV_4 = RCU_IRC64M_DIV4, /*!< IRC64M clock divider by 4 */
    RCU_IRC64M_DIV_8 = RCU_IRC64M_DIV8  /*!< IRC64M clock divider by 8 */
} hal_rcu_irc64m_div_enum;

/* @PARA: SYSCLK */
/* @ENUM: System Clock Source */
typedef enum {
    RCU_SYSCLK_SRC_CK_IRC64MDIV = RCU_CKSYSSRC_IRC64MDIV, /*!< System Clock Source : CK_IRC64MDIV */
    RCU_SYSCLK_SRC_CK_HXTAL     = RCU_CKSYSSRC_HXTAL,     /*!< System Clock Source : CK_HXTAL */
    RCU_SYSCLK_SRC_CK_LPIRC4M   = RCU_CKSYSSRC_LPIRC4M,   /*!< System Clock Source : CK_LPIRC4M */
    RCU_SYSCLK_SRC_CK_PLL0P     = RCU_CKSYSSRC_PLL0P      /*!< System Clock Source : CK_PLL0P */
} hal_rcu_cksys_src_enum;

/* AHB prescaler selection */
typedef enum {
    RCU_SYSCLK_AHBDIV_1   = RCU_AHB_CKSYS_DIV1,   /*!< AHB prescaler select CK_SYS */
    RCU_SYSCLK_AHBDIV_2   = RCU_AHB_CKSYS_DIV2,   /*!< AHB prescaler select CK_SYS/2 */
    RCU_SYSCLK_AHBDIV_4   = RCU_AHB_CKSYS_DIV4,   /*!< AHB prescaler select CK_SYS/4 */
    RCU_SYSCLK_AHBDIV_8   = RCU_AHB_CKSYS_DIV8,   /*!< AHB prescaler select CK_SYS/8 */
    RCU_SYSCLK_AHBDIV_16  = RCU_AHB_CKSYS_DIV16,  /*!< AHB prescaler select CK_SYS/16 */
    RCU_SYSCLK_AHBDIV_64  = RCU_AHB_CKSYS_DIV64,  /*!< AHB prescaler select CK_SYS/64 */
    RCU_SYSCLK_AHBDIV_128 = RCU_AHB_CKSYS_DIV128, /*!< AHB prescaler select CK_SYS/128 */
    RCU_SYSCLK_AHBDIV_256 = RCU_AHB_CKSYS_DIV256, /*!< AHB prescaler select CK_SYS/256 */
    RCU_SYSCLK_AHBDIV_512 = RCU_AHB_CKSYS_DIV512  /*!< AHB prescaler select CK_SYS/512 */
} hal_rcu_sysclk_ahbdiv_enum;

/* APB1 prescaler selection */
typedef enum {
    RCU_AHBCLK_APB1DIV_1  = RCU_APB1_CKAHB_DIV1, /*!< APB1 prescaler select CK_AHB */
    RCU_AHBCLK_APB1DIV_2  = RCU_APB1_CKAHB_DIV2, /*!< APB1 prescaler select CK_AHB/2 */
    RCU_AHBCLK_APB1DIV_4  = RCU_APB1_CKAHB_DIV4, /*!< APB1 prescaler select CK_AHB/4 */
    RCU_AHBCLK_APB1DIV_8  = RCU_APB1_CKAHB_DIV8, /*!< APB1 prescaler select CK_AHB/8 */
    RCU_AHBCLK_APB1DIV_16 = RCU_APB1_CKAHB_DIV16 /*!< APB1 prescaler select CK_AHB/16 */
} hal_rcu_ahbclk_apb1div_enum;

/* APB2 prescaler selection */
typedef enum {
    RCU_AHBCLK_APB2DIV_1  = RCU_APB2_CKAHB_DIV1, /*!< APB2 prescaler select CK_AHB */
    RCU_AHBCLK_APB2DIV_2  = RCU_APB2_CKAHB_DIV2, /*!< APB2 prescaler select CK_AHB/2 */
    RCU_AHBCLK_APB2DIV_4  = RCU_APB2_CKAHB_DIV4, /*!< APB2 prescaler select CK_AHB/4 */
    RCU_AHBCLK_APB2DIV_8  = RCU_APB2_CKAHB_DIV8, /*!< APB2 prescaler select CK_AHB/8 */
    RCU_AHBCLK_APB2DIV_16 = RCU_APB2_CKAHB_DIV16 /*!< APB2 prescaler select CK_AHB/16 */
} hal_rcu_ahbclk_apb2div_enum;

/* APB3 prescaler selection */
typedef enum {
    RCU_AHBCLK_APB3DIV_1  = RCU_APB3_CKAHB_DIV1, /*!< APB3 prescaler select CK_AHB */
    RCU_AHBCLK_APB3DIV_2  = RCU_APB3_CKAHB_DIV2, /*!< APB3 prescaler select CK_AHB/2 */
    RCU_AHBCLK_APB3DIV_4  = RCU_APB3_CKAHB_DIV4, /*!< APB3 prescaler select CK_AHB/4 */
    RCU_AHBCLK_APB3DIV_8  = RCU_APB3_CKAHB_DIV8, /*!< APB3 prescaler select CK_AHB/8 */
    RCU_AHBCLK_APB3DIV_16 = RCU_APB3_CKAHB_DIV16 /*!< APB3 prescaler select CK_AHB/16 */
} hal_rcu_ahbclk_apb3div_enum;

/* APB4 prescaler selection */
typedef enum {
    RCU_AHBCLK_APB4DIV_1  = RCU_APB4_CKAHB_DIV1, /*!< APB4 prescaler select CK_AHB */
    RCU_AHBCLK_APB4DIV_2  = RCU_APB4_CKAHB_DIV2, /*!< APB4 prescaler select CK_AHB/2 */
    RCU_AHBCLK_APB4DIV_4  = RCU_APB4_CKAHB_DIV4, /*!< APB4 prescaler select CK_AHB/4 */
    RCU_AHBCLK_APB4DIV_8  = RCU_APB4_CKAHB_DIV8, /*!< APB4 prescaler select CK_AHB/8 */
    RCU_AHBCLK_APB4DIV_16 = RCU_APB4_CKAHB_DIV16 /*!< APB4 prescaler select CK_AHB/16 */
} hal_rcu_ahbclk_apb4div_enum;

typedef enum {
    RCU_TIMER_MUL2 = RCU_TIMER_PSC_MUL2, /*!< Timer prescaler Multiple by 2 */
    RCU_TIMER_MUL4 = RCU_TIMER_PSC_MUL4  /*!< Timer prescaler Multiple by 4 */
} hal_rcu_timer_ckmul_enum;

/*  PLL Clock Source Selection */
typedef enum {
    RCU_PLL_SRC_CK_IRC64MDIV = RCU_PLLSRC_IRC64MDIV, /*!< PLL clock source select CK_IRC64MDIV */
    RCU_PLL_SRC_CK_LPIRC4M   = RCU_PLLSRC_LPIRC4M,   /*!< PLL clock source select CK_LPIRC4M */
    RCU_PLL_SRC_CK_HXTAL     = RCU_PLLSRC_HXTAL,     /*!< PLL clock source select CK_HXTAL */
    RCU_PLL_SRC_CK_NONE      = RCU_PLLSRC_NONE       /*!< No PLL clock source select */
} hal_rcu_pll_src_enum;

/* PLL48M src SELECTION */
typedef enum {
    RCU_PLL48M_CKSRC_PLL0Q = RCU_PLL48MSRC_PLL0Q, /*!< PLL48MSEL Clock Source Selection: CK_PLL0Q */
    RCU_PLL48M_CKSRC_PLL2P = RCU_PLL48MSRC_PLL2P  /*!< PLL48MSEL Clock Source Selection: CK_PLL2P */
} hal_rcu_pll48m_cksrc_enum;

/* CK48M Clock Source Selection */
typedef enum {
    RCU_CK48M_CKSRC_PLL48M = RCU_CK48MSRC_PLL48M, /*!< CK48M Clock Source Selection: PLL48M */
    RCU_CK48M_CKSRC_IRC48M = RCU_CK48MSRC_IRC48M  /*!< CK48M Clock Source Selection: IRC48M */
} hal_rcu_ck48m_cksrc_enum;

/* PLL2R clock divided frequency */
typedef enum {
    RCU_PLL2R_DIV_2  = RCU_PLL2R_DIV2, /*!< PLL2R  clock divided frequency By 2 */
    RCU_PLL2R_DIV_4  = RCU_PLL2R_DIV4, /*!< PLL2R  clock divided frequency By 4 */
    RCU_PLL2R_DIV_8  = RCU_PLL2R_DIV8, /*!< PLL2R  clock divided frequency By 8 */
    RCU_PLL2R_DIV_16 = RCU_PLL2R_DIV16 /*!< PLL2R  clock divided frequency By 6 */
} hal_rcu_pll2r_div_enum;

/* PLLUSB  preference clock src */
typedef enum {
    RCU_PLLUSBHSx_CKSRC_HXTAL  = RCU_PLLUSBHSPRE_HXTAL, /*!< PLLUSB0  preference clock src : CK_HXTAL */
    RCU_PLLUSBHSx_CKSRC_IRC48M = RCU_PLLUSBHSPRE_IRC48M /*!< PLLUSB0  preference clock src : CK_IRC48M */
} hal_rcu_pllusbhsx_precksrc_enum;

/* PLLUSBx(x=0,1)  prescaler selection */
typedef enum {
    RCU_PLLUSBHSx_PREDV_1  = RCU_PLLUSBHSxPRE_DIV1,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS/1 */
    RCU_PLLUSBHSx_PREDV_2  = RCU_PLLUSBHSxPRE_DIV2,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS/2 */
    RCU_PLLUSBHSx_PREDV_3  = RCU_PLLUSBHSxPRE_DIV3,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS/3 */
    RCU_PLLUSBHSx_PREDV_4  = RCU_PLLUSBHSxPRE_DIV4,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS/4 */
    RCU_PLLUSBHSx_PREDV_5  = RCU_PLLUSBHSxPRE_DIV5,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS/5 */
    RCU_PLLUSBHSx_PREDV_6  = RCU_PLLUSBHSxPRE_DIV6,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS/6 */
    RCU_PLLUSBHSx_PREDV_7  = RCU_PLLUSBHSxPRE_DIV7,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS/7 */
    RCU_PLLUSBHSx_PREDV_8  = RCU_PLLUSBHSxPRE_DIV8,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS/8 */
    RCU_PLLUSBHSx_PREDV_9  = RCU_PLLUSBHSxPRE_DIV9,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS/9 */
    RCU_PLLUSBHSx_PREDV_10 = RCU_PLLUSBHSxPRE_DIV10, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS/10 */
    RCU_PLLUSBHSx_PREDV_11 = RCU_PLLUSBHSxPRE_DIV11, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS/11 */
    RCU_PLLUSBHSx_PREDV_12 = RCU_PLLUSBHSxPRE_DIV12, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS/12 */
    RCU_PLLUSBHSx_PREDV_13 = RCU_PLLUSBHSxPRE_DIV13, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS/13 */
    RCU_PLLUSBHSx_PREDV_14 = RCU_PLLUSBHSxPRE_DIV14, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS/14 */
    RCU_PLLUSBHSx_PREDV_15 = RCU_PLLUSBHSxPRE_DIV15  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS/15 */
} hal_rcu_pllusbhsx_prediv_enum;

/* PLLUSBx(x=0,1)  Multiple  selection */
typedef enum {
    RCU_PLLUSBHSx_MUL_16  = RCU_PLLUSBHSx_MUL16,  /*!< PLLUSBx(x=0,1)  Multiple selection  : PLLUSBHS*16 */
    RCU_PLLUSBHSx_MUL_17  = RCU_PLLUSBHSx_MUL17,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*17 */
    RCU_PLLUSBHSx_MUL_18  = RCU_PLLUSBHSx_MUL18,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*18 */
    RCU_PLLUSBHSx_MUL_19  = RCU_PLLUSBHSx_MUL19,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*19 */
    RCU_PLLUSBHSx_MUL_20  = RCU_PLLUSBHSx_MUL20,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*20 */
    RCU_PLLUSBHSx_MUL_21  = RCU_PLLUSBHSx_MUL21,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*21 */
    RCU_PLLUSBHSx_MUL_22  = RCU_PLLUSBHSx_MUL22,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*22 */
    RCU_PLLUSBHSx_MUL_23  = RCU_PLLUSBHSx_MUL23,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*23 */
    RCU_PLLUSBHSx_MUL_24  = RCU_PLLUSBHSx_MUL24,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*24 */
    RCU_PLLUSBHSx_MUL_25  = RCU_PLLUSBHSx_MUL25,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*25 */
    RCU_PLLUSBHSx_MUL_26  = RCU_PLLUSBHSx_MUL26,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*26 */
    RCU_PLLUSBHSx_MUL_27  = RCU_PLLUSBHSx_MUL27,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*27 */
    RCU_PLLUSBHSx_MUL_28  = RCU_PLLUSBHSx_MUL28,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*28 */
    RCU_PLLUSBHSx_MUL_29  = RCU_PLLUSBHSx_MUL29,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*29 */
    RCU_PLLUSBHSx_MUL_30  = RCU_PLLUSBHSx_MUL30,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*30 */
    RCU_PLLUSBHSx_MUL_31  = RCU_PLLUSBHSx_MUL31,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*31 */
    RCU_PLLUSBHSx_MUL_32  = RCU_PLLUSBHSx_MUL32,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*32 */
    RCU_PLLUSBHSx_MUL_33  = RCU_PLLUSBHSx_MUL33,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*33 */
    RCU_PLLUSBHSx_MUL_34  = RCU_PLLUSBHSx_MUL34,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*34 */
    RCU_PLLUSBHSx_MUL_35  = RCU_PLLUSBHSx_MUL35,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*35 */
    RCU_PLLUSBHSx_MUL_36  = RCU_PLLUSBHSx_MUL36,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*36 */
    RCU_PLLUSBHSx_MUL_37  = RCU_PLLUSBHSx_MUL37,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*37 */
    RCU_PLLUSBHSx_MUL_38  = RCU_PLLUSBHSx_MUL38,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*38 */
    RCU_PLLUSBHSx_MUL_39  = RCU_PLLUSBHSx_MUL39,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*39 */
    RCU_PLLUSBHSx_MUL_40  = RCU_PLLUSBHSx_MUL40,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*40 */
    RCU_PLLUSBHSx_MUL_41  = RCU_PLLUSBHSx_MUL41,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*41 */
    RCU_PLLUSBHSx_MUL_42  = RCU_PLLUSBHSx_MUL42,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*42 */
    RCU_PLLUSBHSx_MUL_43  = RCU_PLLUSBHSx_MUL43,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*43 */
    RCU_PLLUSBHSx_MUL_44  = RCU_PLLUSBHSx_MUL44,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*44 */
    RCU_PLLUSBHSx_MUL_45  = RCU_PLLUSBHSx_MUL45,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*45 */
    RCU_PLLUSBHSx_MUL_46  = RCU_PLLUSBHSx_MUL46,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*46 */
    RCU_PLLUSBHSx_MUL_47  = RCU_PLLUSBHSx_MUL47,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*47 */
    RCU_PLLUSBHSx_MUL_48  = RCU_PLLUSBHSx_MUL48,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*48 */
    RCU_PLLUSBHSx_MUL_49  = RCU_PLLUSBHSx_MUL49,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*49 */
    RCU_PLLUSBHSx_MUL_50  = RCU_PLLUSBHSx_MUL50,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*50 */
    RCU_PLLUSBHSx_MUL_51  = RCU_PLLUSBHSx_MUL51,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*51 */
    RCU_PLLUSBHSx_MUL_52  = RCU_PLLUSBHSx_MUL52,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*52 */
    RCU_PLLUSBHSx_MUL_53  = RCU_PLLUSBHSx_MUL53,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*53 */
    RCU_PLLUSBHSx_MUL_54  = RCU_PLLUSBHSx_MUL54,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*54 */
    RCU_PLLUSBHSx_MUL_55  = RCU_PLLUSBHSx_MUL55,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*55 */
    RCU_PLLUSBHSx_MUL_56  = RCU_PLLUSBHSx_MUL56,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*56 */
    RCU_PLLUSBHSx_MUL_57  = RCU_PLLUSBHSx_MUL57,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*57 */
    RCU_PLLUSBHSx_MUL_58  = RCU_PLLUSBHSx_MUL58,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*58 */
    RCU_PLLUSBHSx_MUL_59  = RCU_PLLUSBHSx_MUL59,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*59 */
    RCU_PLLUSBHSx_MUL_60  = RCU_PLLUSBHSx_MUL60,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*60 */
    RCU_PLLUSBHSx_MUL_61  = RCU_PLLUSBHSx_MUL61,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*61 */
    RCU_PLLUSBHSx_MUL_62  = RCU_PLLUSBHSx_MUL62,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*62 */
    RCU_PLLUSBHSx_MUL_63  = RCU_PLLUSBHSx_MUL63,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*63 */
    RCU_PLLUSBHSx_MUL_64  = RCU_PLLUSBHSx_MUL64,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*64 */
    RCU_PLLUSBHSx_MUL_65  = RCU_PLLUSBHSx_MUL65,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*65 */
    RCU_PLLUSBHSx_MUL_66  = RCU_PLLUSBHSx_MUL66,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*66 */
    RCU_PLLUSBHSx_MUL_67  = RCU_PLLUSBHSx_MUL67,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*67 */
    RCU_PLLUSBHSx_MUL_68  = RCU_PLLUSBHSx_MUL68,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*68 */
    RCU_PLLUSBHSx_MUL_69  = RCU_PLLUSBHSx_MUL69,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*69 */
    RCU_PLLUSBHSx_MUL_70  = RCU_PLLUSBHSx_MUL70,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*70 */
    RCU_PLLUSBHSx_MUL_71  = RCU_PLLUSBHSx_MUL71,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*71 */
    RCU_PLLUSBHSx_MUL_72  = RCU_PLLUSBHSx_MUL72,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*72 */
    RCU_PLLUSBHSx_MUL_73  = RCU_PLLUSBHSx_MUL73,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*73 */
    RCU_PLLUSBHSx_MUL_74  = RCU_PLLUSBHSx_MUL74,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*74 */
    RCU_PLLUSBHSx_MUL_75  = RCU_PLLUSBHSx_MUL75,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*75 */
    RCU_PLLUSBHSx_MUL_76  = RCU_PLLUSBHSx_MUL76,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*76 */
    RCU_PLLUSBHSx_MUL_77  = RCU_PLLUSBHSx_MUL77,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*77 */
    RCU_PLLUSBHSx_MUL_78  = RCU_PLLUSBHSx_MUL78,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*78 */
    RCU_PLLUSBHSx_MUL_79  = RCU_PLLUSBHSx_MUL79,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*79 */
    RCU_PLLUSBHSx_MUL_80  = RCU_PLLUSBHSx_MUL80,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*80 */
    RCU_PLLUSBHSx_MUL_81  = RCU_PLLUSBHSx_MUL81,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*81 */
    RCU_PLLUSBHSx_MUL_82  = RCU_PLLUSBHSx_MUL82,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*82 */
    RCU_PLLUSBHSx_MUL_83  = RCU_PLLUSBHSx_MUL83,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*83 */
    RCU_PLLUSBHSx_MUL_84  = RCU_PLLUSBHSx_MUL84,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*84 */
    RCU_PLLUSBHSx_MUL_85  = RCU_PLLUSBHSx_MUL85,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*85 */
    RCU_PLLUSBHSx_MUL_86  = RCU_PLLUSBHSx_MUL86,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*86 */
    RCU_PLLUSBHSx_MUL_87  = RCU_PLLUSBHSx_MUL87,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*87 */
    RCU_PLLUSBHSx_MUL_88  = RCU_PLLUSBHSx_MUL88,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*88 */
    RCU_PLLUSBHSx_MUL_89  = RCU_PLLUSBHSx_MUL89,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*89 */
    RCU_PLLUSBHSx_MUL_90  = RCU_PLLUSBHSx_MUL90,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*90 */
    RCU_PLLUSBHSx_MUL_91  = RCU_PLLUSBHSx_MUL91,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*91 */
    RCU_PLLUSBHSx_MUL_92  = RCU_PLLUSBHSx_MUL92,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*92 */
    RCU_PLLUSBHSx_MUL_93  = RCU_PLLUSBHSx_MUL93,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*93 */
    RCU_PLLUSBHSx_MUL_94  = RCU_PLLUSBHSx_MUL94,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*94 */
    RCU_PLLUSBHSx_MUL_95  = RCU_PLLUSBHSx_MUL95,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*95 */
    RCU_PLLUSBHSx_MUL_96  = RCU_PLLUSBHSx_MUL96,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*96 */
    RCU_PLLUSBHSx_MUL_97  = RCU_PLLUSBHSx_MUL97,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*97 */
    RCU_PLLUSBHSx_MUL_98  = RCU_PLLUSBHSx_MUL98,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*98 */
    RCU_PLLUSBHSx_MUL_99  = RCU_PLLUSBHSx_MUL99,  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*99 */
    RCU_PLLUSBHSx_MUL_100 = RCU_PLLUSBHSx_MUL100, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*100 */
    RCU_PLLUSBHSx_MUL_101 = RCU_PLLUSBHSx_MUL101, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*101 */
    RCU_PLLUSBHSx_MUL_102 = RCU_PLLUSBHSx_MUL102, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*102 */
    RCU_PLLUSBHSx_MUL_103 = RCU_PLLUSBHSx_MUL103, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*103 */
    RCU_PLLUSBHSx_MUL_104 = RCU_PLLUSBHSx_MUL104, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*104 */
    RCU_PLLUSBHSx_MUL_105 = RCU_PLLUSBHSx_MUL105, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*105 */
    RCU_PLLUSBHSx_MUL_106 = RCU_PLLUSBHSx_MUL106, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*106 */
    RCU_PLLUSBHSx_MUL_107 = RCU_PLLUSBHSx_MUL107, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*107 */
    RCU_PLLUSBHSx_MUL_108 = RCU_PLLUSBHSx_MUL108, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*108 */
    RCU_PLLUSBHSx_MUL_109 = RCU_PLLUSBHSx_MUL109, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*109 */
    RCU_PLLUSBHSx_MUL_110 = RCU_PLLUSBHSx_MUL110, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*110 */
    RCU_PLLUSBHSx_MUL_111 = RCU_PLLUSBHSx_MUL111, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*111 */
    RCU_PLLUSBHSx_MUL_112 = RCU_PLLUSBHSx_MUL112, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*112 */
    RCU_PLLUSBHSx_MUL_113 = RCU_PLLUSBHSx_MUL113, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*113 */
    RCU_PLLUSBHSx_MUL_114 = RCU_PLLUSBHSx_MUL114, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*114 */
    RCU_PLLUSBHSx_MUL_115 = RCU_PLLUSBHSx_MUL115, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*115 */
    RCU_PLLUSBHSx_MUL_116 = RCU_PLLUSBHSx_MUL116, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*116 */
    RCU_PLLUSBHSx_MUL_117 = RCU_PLLUSBHSx_MUL117, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*117 */
    RCU_PLLUSBHSx_MUL_118 = RCU_PLLUSBHSx_MUL118, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*118 */
    RCU_PLLUSBHSx_MUL_119 = RCU_PLLUSBHSx_MUL119, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*119 */
    RCU_PLLUSBHSx_MUL_120 = RCU_PLLUSBHSx_MUL120, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*120 */
    RCU_PLLUSBHSx_MUL_121 = RCU_PLLUSBHSx_MUL121, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*121 */
    RCU_PLLUSBHSx_MUL_122 = RCU_PLLUSBHSx_MUL122, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*122 */
    RCU_PLLUSBHSx_MUL_123 = RCU_PLLUSBHSx_MUL123, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*123 */
    RCU_PLLUSBHSx_MUL_124 = RCU_PLLUSBHSx_MUL124, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*124 */
    RCU_PLLUSBHSx_MUL_125 = RCU_PLLUSBHSx_MUL125, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*125 */
    RCU_PLLUSBHSx_MUL_126 = RCU_PLLUSBHSx_MUL126, /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*126 */
    RCU_PLLUSBHSx_MUL_127 = RCU_PLLUSBHSx_MUL127  /*!< PLLUSBx(x=0,1)  prescaler selection  : PLLUSBHS*127 */
} hal_rcu_pllusbhsx_mul_enum;

/* USBHSx(x=0,1) Div frequency selection */
typedef enum {
    RCU_USBHSx_DIV_2  = RCU_USBHSx_DIV2,  /*!< USBHSx(x=0,1) Div frequency by USBHS/2 */
    RCU_USBHSx_DIV_4  = RCU_USBHSx_DIV4,  /*!< USBHSx(x=0,1) Div frequency by USBHS/4 */
    RCU_USBHSx_DIV_6  = RCU_USBHSx_DIV6,  /*!< USBHSx(x=0,1) Div frequency by USBHS/6 */
    RCU_USBHSx_DIV_8  = RCU_USBHSx_DIV8,  /*!< USBHSx(x=0,1) Div frequency by USBHS/8 */
    RCU_USBHSx_DIV_10 = RCU_USBHSx_DIV10, /*!< USBHSx(x=0,1) Div frequency by USBHS/10 */
    RCU_USBHSx_DIV_12 = RCU_USBHSx_DIV12, /*!< USBHSx(x=0,1) Div frequency by USBHS/12 */
    RCU_USBHSx_DIV_14 = RCU_USBHSx_DIV14, /*!< USBHSx(x=0,1) Div frequency by USBHS/14 */
    RCU_USBHSx_DIV_16 = RCU_USBHSx_DIV16  /*!< USBHSx(x=0,1) Div frequency by USBHS/16 */
} hal_rcu_usbhsx_div_enum;

/* PLLUSBx(x=0,1)  Clock distribution coefficient */
typedef enum {
    RCU_USBHSx_PSC_1 = RCU_USBHSxPSC_DIV1, /*!< PLLUSBx(x=0,1)  Div frequency  by PLLUSBHS_PSC/1 */
    RCU_USBHSx_PSC_2 = RCU_USBHSxPSC_DIV2, /*!< PLLUSBx(x=0,1)  Div frequency  by PLLUSBHS_PSC/2 */
    RCU_USBHSx_PSC_3 = RCU_USBHSxPSC_DIV3, /*!< PLLUSBx(x=0,1)  Div frequency  by PLLUSBHS_PSC/3 */
    RCU_USBHSx_PSC_4 = RCU_USBHSxPSC_DIV4, /*!< PLLUSBx(x=0,1)  Div frequency  by PLLUSBHS_PSC/4 */
    RCU_USBHSx_PSC_5 = RCU_USBHSxPSC_DIV5, /*!< PLLUSBx(x=0,1)  Div frequency  by PLLUSBHS_PSC/5 */
    RCU_USBHSx_PSC_6 = RCU_USBHSxPSC_DIV6, /*!< PLLUSBx(x=0,1)  Div frequency  by PLLUSBHS_PSC/6 */
    RCU_USBHSx_PSC_7 = RCU_USBHSxPSC_DIV7, /*!< PLLUSBx(x=0,1)  Div frequency  by PLLUSBHS_PSC/7 */
    RCU_USBHSx_PSC_8 = RCU_USBHSxPSC_DIV8  /*!< PLLUSBx(x=0,1)  Div frequency  by PLLUSBHS_PSC/8 */
} hal_rcu_usbhsx_psc_enum;

/* USBHSx(x=0,1) 48M Clock Source */
typedef enum {
    RCU_USBHSx48M_CKSRC_PLL0R   = RCU_USB48MSRC_PLL0R,    /*!< PLLUSBx 48M Clock Source: CK_PLL0R */
    RCU_USBHSx48M_CKSRC_USBHSDV = RCU_USB48MSRC_PLLUSBHS, /*!< PLLUSBx 48M Clock Source: CK_USBHSxDV\CK_PLLUSBHSx */
    RCU_USBHSx48M_CKSRC_PLL1Q   = RCU_USB48MSRC_PLL1Q,    /*!< PLLUSBx 48M Clock Source: CK_USBHSxPSC\PLL1Q */
    RCU_USBHSx48M_CKSRC_IRC48M  = RCU_USB48MSRC_IRC48M    /*!< PLLUSBx 48M Clock Source: CK_IRC48M */
} hal_rcu_usbhsx48m_cksrc_enum;

/*  CK_USBHS src selection */
typedef enum {
    RCU_USBHSx_CKSRC_CK48M = RCU_USBHSSEL_48M, /*!< USBHSx clock Source: 48M */
    RCU_USBHSx_CKSRC_CK60M = RCU_USBHSSEL_60M  /*!< USBHSx clock Source: 60m */
} hal_rcu_usbhsx_cksrc_enum;

/*  CAN clock src selection */
typedef enum {
    RCU_CAN_CKSRC_CK_HXTAL     = RCU_CANSRC_HXTAL,     /*!< CAN clock Source: hxtal */
    RCU_CAN_CKSRC_CK_APB       = RCU_CANSRC_APB2,      /*!< CAN clock Source: apb2 */
    RCU_CAN_CKSRC_CK_APB2DIV2  = RCU_CANSRC_APB2_DIV2, /*!< CAN clock Source: apb2/2 */
    RCU_CAN_CKSRC_CK_IRC64MDIV = RCU_CANSRC_IRC64MDIV  /*!< CAN clock Source: irc64mdiv */
} hal_rcu_can_cksrc_enum;

/*  SAI2Bx(x=0,1) clock src selection */
typedef enum {
    RCU_SAI2B_CKSRC_CK_PLL0Q       = RCU_SAI2BSRC_PLL0Q,      /*!< SAI2Bx clock Source: CK_PLL0Q */
    RCU_SAI2B_CKSRC_CK_PLL1P       = RCU_SAI2BSRC_PLL1P,      /*!< SAI2Bx clock Source: CK_PLL1P */
    RCU_SAI2B_CKSRC_CK_PLL2P       = RCU_SAI2BSRC_PLL2P,      /*!< SAI2Bx clock Source: CK_PLL2P */
    RCU_SAI2B_CKSRC_I2S_CKIN       = RCU_SAI2BSRC_I2S_CKIN,   /*!< SAI2Bx clock Source: I2S_CKIN */
    RCU_SAI2B_CKSRC_CK_PER         = RCU_SAI2BSRC_PER,        /*!< SAI2Bx clock Source: CK_PER */
    RCU_SAI2B_CKSRC_CK_RSPDIF_SYMB = RCU_SAI2BSRC_RSPDIF_SYMB /*!< SAI2Bx clock Source: CK_RSPDIF_SYMB */
} hal_rcu_sai2b_cksrc_enum;

/*  EXMC clock src selection */
typedef enum {
    RCU_EXMC_CKSRC_CK_AHB   = RCU_EXMCSRC_AHB,   /*!< EXMC clock src: CK_AHB */
    RCU_EXMC_CKSRC_CK_PLL0Q = RCU_EXMCSRC_PLL0Q, /*!< EXMC clock src: CK_PLL0Q */
    RCU_EXMC_CKSRC_CK_PLL1R = RCU_EXMCSRC_PLL1R, /*!< EXMC clock src: CK_PLL1R */
    RCU_EXMC_CKSRC_CK_PER   = RCU_EXMCSRC_PER    /*!< EXMC clock src: CK_PER */
} hal_rcu_exmc_cksrc_enum;

/*  SPI(0,1,2) clock src selection */
typedef enum {
    RCU_SPI_012_CKSRC_CK_PLL0Q = RCU_SPISRC_PLL0Q,    /*!< SPI(0,1,2) clock src: CK_PLL0Q */
    RCU_SPI_012_CKSRC_CK_PLL1P = RCU_SPISRC_PLL1P,    /*!< SPI(0,1,2) clock src: CK_PLL1P */
    RCU_SPI_012_CKSRC_CK_PLL2P = RCU_SPISRC_PLL2P,    /*!< SPI(0,1,2) clock src: CK_PLL2P */
    RCU_SPI_012_CKSRC_I2S_CKIN = RCU_SPISRC_I2S_CKIN, /*!< SPI(0,1,2) clock src: I2S_CKIN */
    RCU_SPI_012_CKSRC_CK_PER   = RCU_SPISRC_PER       /*!< SPI(0,1,2) clock src: CK_PER */
} hal_rcu_spi012_cksrc_enum;

/*  SPI(3,4) clock src selection */
typedef enum {
    RCU_SPI_34_CKSRC_CK_APB2      = RCU_SPISRC_APB2,      /*!< SPI(3,4) clock src: CK_APB2 */
    RCU_SPI_34_CKSRC_CK_PLL1Q     = RCU_SPISRC_PLL1Q,     /*!< SPI(3,4) clock src: CK_PLL1Q */
    RCU_SPI_34_CKSRC_CK_PLL2Q     = RCU_SPISRC_PLL2Q,     /*!< SPI(3,4) clock src: CK_PLL2Q */
    RCU_SPI_34_CKSRC_CK_IRC64MDIV = RCU_SPISRC_IRC64MDIV, /*!< SPI(3,4) clock src: CK_IRC64MDIV */
    RCU_SPI_34_CKSRC_CK_LPIRC4M   = RCU_SPISRC_LPIRC4M,   /*!< SPI(3,4) clock src: CK_LPIRC4M */
    RCU_SPI_34_CKSRC_CK_HXTAL     = RCU_SPISRC_HXTAL      /*!< SPI(3,4) clock src: CK_HXTAL */
} hal_rcu_spi34_cksrc_enum;

/*  SPI5 clock src selection */
typedef enum {
    RCU_SPI_5_CKSRC_CK_APB2      = RCU_SPI5SRC_APB2,      /*!< SPI5 clock src: CK_APB2 */
    RCU_SPI_5_CKSRC_CK_PLL1Q     = RCU_SPI5SRC_PLL1Q,     /*!< SPI5 clock src: CK_PLL1Q */
    RCU_SPI_5_CKSRC_CK_PLL2Q     = RCU_SPI5SRC_PLL2Q,     /*!< SPI5 clock src: CK_PLL2Q */
    RCU_SPI_5_CKSRC_CK_IRC64MDIV = RCU_SPI5SRC_IRC64MDIV, /*!< SPI5 clock src: CK_IRC64MDIV */
    RCU_SPI_5_CKSRC_CK_LPIRC4M   = RCU_SPI5SRC_LPIRC4M,   /*!< SPI5 clock src: CK_LPIRC4M */
    RCU_SPI_5_CKSRC_CK_HXTAL     = RCU_SPI5SRC_HXTAL,     /*!< SPI5 clock src: CK_HXTAL */
    RCU_SPI_5_CKSRC_I2S_CKIN     = RCU_SPI5SRC_I2S_CKIN   /*!< SPI5 clock src: I2S_CKIN */
} hal_rcu_spi5_cksrc_enum;

/*  USART clock src selection */
typedef enum {
    RCU_USART_CKSRC_CK_APB       = RCU_USARTSRC_APB2,     /*!< USART clock src: CK_APB */
    RCU_USART_CKSRC_CK_AHB       = RCU_USARTSRC_AHB,      /*!< USART clock src: CK_AHB */
    RCU_USART_CKSRC_CK_LXTAL     = RCU_USARTSRC_LXTAL,    /*!< USART clock src: CK_LXTAL */
    RCU_USART_CKSRC_CK_IRC64MDIV = RCU_USARTSRC_IRC64MDIV /*!< USART clock src: CK_IRC64MDIV */
} hal_rcu_usart_cksrc_enum;

/*  SAIx(x=0,1) clock src selection */
typedef enum {
    RCU_SAI_CKSRC_CK_PLL0Q = RCU_SAISRC_PLL0Q,    /*!< SAIx clock src: CK_PLL0Q */
    RCU_SAI_CKSRC_CK_PLL1P = RCU_SAISRC_PLL1P,    /*!< SAIx clock src: CK_PLL1P */
    RCU_SAI_CKSRC_CK_PLL2P = RCU_SAISRC_PLL2P,    /*!< SAIx clock src: CK_PLL2P */
    RCU_SAI_CKSRC_I2S_CKIN = RCU_SAISRC_I2S_CKIN, /*!< SAIx clock src: I2S_CKIN */
    RCU_SAI_CKSRC_CK_PER   = RCU_SAISRC_PER       /*!< SAIx clock src: CK_PER */
} hal_rcu_sai_cksrc_enum;

/*  HPDF clock src selection */
typedef enum {
    RCU_HPDF_CKSRC_CK_APB2 = RCU_HPDFSRC_APB2,        /*!< HPDF clock src: CK_APB2 */
    RCU_HPDF_CKSRC_CK_AHB  = (int32_t)RCU_HPDFSRC_AHB /*!< HPDF clock src: CK_AHB */
} hal_rcu_hpdf_cksrc_enum;

/*  SDIOx(x=0,1) clock src selection */
typedef enum {
    RCU_SDIO0_CKSRC_CK_PLL0Q = RCU_SDIO0SRC_PLL0Q, /*!< SDIO0 clock src: CK_PLL0Q */
    RCU_SDIO0_CKSRC_CK_PLL1R = RCU_SDIO0SRC_PLL1R, /*!< SDIO0 clock src: CK_PLL1R */
    RCU_SDIO1_CKSRC_CK_PLL0Q = RCU_SDIO1SRC_PLL0Q, /*!< SDIO1 clock src: CK_PLL0Q */
    RCU_SDIO1_CKSRC_CK_PLL1R = RCU_SDIO1SRC_PLL1R  /*!< SDIO1 clock src: CK_PLL1R */
} hal_rcu_sdio_cksrc_enum;

/* Deepsleep clock source selection */
typedef enum {
    RCU_DSPWUSSEL_CKSRC_IRC64MDIV = RCU_DSPWUSSEL_IRC64MDIV, /*!< Deepsleep clock src: IRC64MDIV */
    RCU_DSPWUSSEL_CKSRC_LPIRC4M   = RCU_DSPWUSSEL_LPIRC4M    /*!< Deepsleep clock src: LPIRC4M */
} hal_rcu_dspwu_cksrc_enum;

/*  ADC clock src selection */
typedef enum {
    RCU_ADC_CKSRC_CK_PLL1P = RCU_ADCSRC_PLL1P, /*!< ADC clock src: CK_PLL1P */
    RCU_ADC_CKSRC_CK_PLL2R = RCU_ADCSRC_PLL2R, /*!< ADC clock src: CK_PLL2R */
    RCU_ADC_CKSRC_CK_PER   = RCU_ADCSRC_PER    /*!< ADC clock src: CK_PER */
} hal_rcu_adc_cksrc_enum;

/*  RSPDIF clock src selection */
typedef enum {
    RCU_RSPDIF_CKSRC_CK_PLL0Q     = RCU_RSPDIFSRC_PLL0Q,    /*!< RSPDIF clock src: CK_PLL0Q */
    RCU_RSPDIF_CKSRC_CK_PLL1R     = RCU_RSPDIFSRC_PLL1R,    /*!< RSPDIF clock src: CK_PLL1R */
    RCU_RSPDIF_CKSRC_CK_PLL2R     = RCU_RSPDIFSRC_PLL2R,    /*!< RSPDIF clock src: CK_PLL2R */
    RCU_RSPDIF_CKSRC_CK_IRC64MDIV = RCU_RSPDIFSRC_IRC64MDIV /*!< RSPDIF clock src: CK_IRC64MDIV */
} hal_rcu_rspdif_cksrc_enum;

/*  I2C clock src selection */
typedef enum {
    RCU_I2C_CKSRC_CK_APB1      = RCU_I2CSRC_APB1,      /*!< I2C clock src: CK_APB1 */
    RCU_I2C_CKSRC_CK_PLL2R     = RCU_I2CSRC_PLL2R,     /*!< I2C clock src: CK_PLL2R */
    RCU_I2C_CKSRC_CK_IRC64MDIV = RCU_I2CSRC_IRC64MDIV, /*!< I2C clock src: CK_IRC64MDIV */
    RCU_I2C_CKSRC_CK_LPIRC4M   = RCU_I2CSRC_LPIRC4M    /*!< I2C clock src: CK_LPIRC4M */
} hal_rcu_i2c_cksrc_enum;

/*  preference clock src selection */
typedef enum {
    RCU_PRECLK_CKSRC_CK_IRC64MDIV = RCU_PERSRC_IRC64MDIV, /*!<  Peripheral clock src: IRC64MDIV */
    RCU_PRECLK_CKSRC_CK_LPIRC4M   = RCU_PERSRC_LPIRC4M,   /*!<  Peripheral clock src: CK_LPIRC4M */
    RCU_PRECLK_CKSRC_CK_HXTAL     = RCU_PERSRC_HXTAL      /*!<  Peripheral clock src: CK_HXTAL */
} hal_rcu_pef_cksrc_enum;

/* @STRUCT_MEMBER: hxtal need_configure
   @brief        : RCU HXTAL status
   @SP           : need_configure = ENABLE/DISABLE */
typedef struct {
    ControlStatus need_configure; /*!< the oscillators configure flag */
    hal_rcu_osc_state_enum state; /*!< the oscillators state */
} hal_rcu_hxtal_struct;

/* @STRUCT_MEMBER: lxtal need_configure
   @brief        : RCU LXTAL status
   @SP           : need_configure = ENABLE/DISABLE */
typedef struct {
    ControlStatus need_configure;              /*!< the oscillators configure flag */
    hal_rcu_osc_state_enum state;              /*!< the oscillators state */
    hal_rcu_lxtal_drive_capability_enum drive; /*!< the oscillators drive capability */
} hal_rcu_lxtal_struct;

/* @STRUCT_MEMBER: irc64m need_configure
   @brief        : RCU IRC64M status
   @SP           : need_configure = ENABLE/DISABLE */
typedef struct {
    ControlStatus need_configure;       /*!< the oscillators configure flag */
    uint8_t adjust_value;               /*!< the oscillators adjust value */
    hal_rcu_osc_state_enum state;       /*!< the oscillators state */
    hal_rcu_irc64m_div_enum irc64m_div; /*!< IRC64M clock Multiple divider */
} hal_rcu_irc64m_struct;

/* @STRUCT_MEMBER: irc48m need_configure
   @brief        : RCU IRC48M status
   @SP           : need_configure = ENABLE/DISABLE */
typedef struct {
    ControlStatus need_configure; /*!< the oscillators configure flag */
    hal_rcu_osc_state_enum state; /*!< the oscillators state */
} hal_rcu_irc48m_struct;

/* @STRUCT_MEMBER: irc32k need_configure
   @brief        : RCU IRC32k status
   @SP           : need_configure = ENABLE/DISABLE */
typedef struct {
    ControlStatus need_configure; /*!< the oscillators configure flag */
    hal_rcu_osc_state_enum state; /*!< the oscillators state */
} hal_rcu_irc32k_struct;

/* @STRUCT_MEMBER: lpirc4m need_configure
   @brief        : RCU LPIRC4M status
   @SP           : need_configure = ENABLE/DISABLE */
typedef struct {
    ControlStatus need_configure; /*!< the oscillators configure flag */
    uint8_t adjust_value;         /*!< the oscillators adjust value */
    hal_rcu_osc_state_enum state; /*!< the oscillators state */
} hal_rcu_lpirc4m_struct;

/* RCU clock */
typedef struct {
    uint32_t clock_type;                     /*!< rcu clock type to be configured */
    hal_rcu_cksys_src_enum sysclk_source;    /*!< system clock source selection */
    hal_rcu_sysclk_ahbdiv_enum ahbclk_div;   /*!< AHB prescaler selection */
    hal_rcu_ahbclk_apb1div_enum apb1clk_div; /*!< apb1 prescaler selection */
    hal_rcu_ahbclk_apb2div_enum apb2clk_div; /*!< apb2 prescaler selection */
    hal_rcu_ahbclk_apb3div_enum apb3clk_div; /*!< apb3 prescaler selection */
    hal_rcu_ahbclk_apb4div_enum apb4clk_div; /*!< apb4 prescaler selection */
} hal_rcu_clk_struct;

typedef struct {
    hal_rcu_usbhsx_div_enum usbhs_div;            /*!< usbhs divided selection */
    hal_rcu_usbhsx_psc_enum usbhs_psc;            /*!< usbhs distribution coefficient selection */
    hal_rcu_usbhsx48m_cksrc_enum usbhs48m_source; /*!< usbhs48m clock source selection */
    hal_rcu_usbhsx_cksrc_enum usbhs_clock_source; /*!< usbhs clock source selection */
} hal_rcu_usb_struct;

/* RCU peripheral clock struct */
typedef struct {
    uint64_t periph_clock_type;                    /*!< peripherals clock type selection */
    hal_rcu_rtc_cksrc_enum rtc_clock_source;       /*!< RTC clock source selection */
    hal_rcu_rtc_div_enum rtc_div;                  /*!< RTC clock source selection */
    hal_rcu_timer_ckmul_enum timer_mul;            /*!< timer multiple selection */
    hal_rcu_ck48m_cksrc_enum ck48m_clock_source;   /*!< ck48m selection */
    hal_rcu_pll48m_cksrc_enum pll48m_clock_source; /*!< pll48m selection */
    hal_rcu_usb_struct usb0_clock;                 /*!< usb0 clock configuration*/
    hal_rcu_usb_struct usb1_clock;                 /*!< usb1 clock configuration*/
    hal_rcu_can_cksrc_enum can0_clock_source;      /*!< can clock source selection */
    hal_rcu_can_cksrc_enum can1_clock_source;      /*!< can clock source selection */
    hal_rcu_can_cksrc_enum can2_clock_source;      /*!< can clock source selection */
    hal_rcu_sai_cksrc_enum sai0_clock_source;      /*!< sai0 clock source selection */
    hal_rcu_sai_cksrc_enum sai1_clock_source;      /*!< sai1 clock source selection */
    hal_rcu_sai2b_cksrc_enum sai2b0_clock_source;  /*!< SAI2B0 clock src selection */
    hal_rcu_sai2b_cksrc_enum sai2b1_clock_source;  /*!< SAI2B1 clock src selection */
    hal_rcu_exmc_cksrc_enum exmc_clock_source;     /*!< exmc clock src selection */
    hal_rcu_spi012_cksrc_enum spi0_clock_source;   /*!< spi0 clock src selection */
    hal_rcu_spi012_cksrc_enum spi1_clock_source;   /*!< spi1 clock src selection */
    hal_rcu_spi012_cksrc_enum spi2_clock_source;   /*!< spi2 clock src selection */
    hal_rcu_spi34_cksrc_enum spi3_clock_source;    /*!< spi3 clock src selection */
    hal_rcu_spi34_cksrc_enum spi4_clock_source;    /*!< spi4 clock src selection */
    hal_rcu_spi5_cksrc_enum spi5_clock_source;     /*!< spi5 clock src selection */
    hal_rcu_usart_cksrc_enum usart0_clock_source;  /*!< usart0 clock source selection */
    hal_rcu_usart_cksrc_enum usart1_clock_source;  /*!< usart1 clock source selection */
    hal_rcu_usart_cksrc_enum usart2_clock_source;  /*!< usart2 clock source selection */
    hal_rcu_usart_cksrc_enum usart5_clock_source;  /*!< usart5 clock source selection */
    hal_rcu_hpdf_cksrc_enum hpdf_clock_source;     /*!< hpdf clock source selection */
    hal_rcu_sdio_cksrc_enum sdio0_clock_source;    /*!< sdio0 clock source selection */
    hal_rcu_sdio_cksrc_enum sdio1_clock_source;    /*!< sdio1 clock source selection */
    hal_rcu_adc_cksrc_enum adc01_clock_source;     /*!< adc0 clock source selection */
    hal_rcu_adc_cksrc_enum adc2_clock_source;      /*!< adc2 clock source selection */
    hal_rcu_rspdif_cksrc_enum rspdif_clock_source; /*!< rspdif clock source selection */
    hal_rcu_i2c_cksrc_enum i2c0_clock_source;      /*!< i2c0 clock source selection */
    hal_rcu_i2c_cksrc_enum i2c1_clock_source;      /*!< i2c1 clock source selection */
    hal_rcu_i2c_cksrc_enum i2c2_clock_source;      /*!< i2c2 clock source selection */
    hal_rcu_i2c_cksrc_enum i2c3_clock_source;      /*!< i2c3 clock source selection */
    hal_rcu_pef_cksrc_enum pef_clock_source;       /*!< pef clock source selection */
    hal_rcu_irc64m_div_enum irc64m_div;            /*!< IRC64M clock Multiple divider */
} hal_rcu_periphclk_struct;

typedef struct {
    hal_rcu_osc_state_enum pll0_state; /*!< the oscillators state */
    hal_rcu_pll_psc_enum pll0_psc;     /*!< PLL0 VCO Source clock divider */
    hal_rcu_plln_enum pll0_n;          /*!< PLL0 VCO clock multiplier factor */
    hal_rcu_pllp_enum pll0_p;          /*!< PLL0P output frequency division coefficient */
    hal_rcu_pllq_enum pll0_q;          /*!< PLL0Q output frequency division coefficient */
    hal_rcu_pllr_enum pll0_r;          /*!< PLL0R output frequency dividing coefficient */
    uint32_t pll0_input_fre_range;     /*!< pll0 input frequency range */
    uint32_t pll0_output_fre_range;    /*!< pll0 output frequency range */
    uint32_t pll0_fractional_latch;    /*!< pll0 fractional latch*/
} hal_rcu_pll0_struct;

typedef struct {
    hal_rcu_osc_state_enum pll1_state; /*!< the oscillators state */
    hal_rcu_pll_psc_enum pll1_psc;     /*!< PLL1 VCO Source clock divider */
    hal_rcu_plln_enum pll1_n;          /*!< PLL1 VCO clock multiplier factor */
    hal_rcu_pllp_enum pll1_p;          /*!< PLL1P output frequency division coefficient */
    hal_rcu_pllq_enum pll1_q;          /*!< PLL1Q output frequency division coefficient */
    hal_rcu_pllr_enum pll1_r;          /*!< PLL1R output frequency dividing coefficient */
    uint32_t pll1_input_fre_range;     /*!< pll1 input frequency range */
    uint32_t pll1_output_fre_range;    /*!< pll1 output frequency range */
    uint32_t pll1_fractional_latch;    /*!< pll1 fractional latch*/
} hal_rcu_pll1_struct;

typedef struct {
    hal_rcu_osc_state_enum pll2_state; /*!< the oscillators state */
    hal_rcu_pll_psc_enum pll2_psc;     /*!< PLL2 VCO Source clock divider */
    hal_rcu_plln_enum pll2_n;          /*!< PLL2 VCO clock multiplier factor */
    hal_rcu_pllp_enum pll2_p;          /*!< PLL2P output frequency division coefficient */
    hal_rcu_pllq_enum pll2_q;          /*!< PLL2Q output frequency division coefficient */
    hal_rcu_pllr_enum pll2_r;          /*!< PLL2R output frequency dividing coefficient */
    hal_rcu_pll2r_div_enum pll2r_div;  /*!< Frequency division factor of PLL2R clock */
    uint32_t pll2_input_fre_range;     /*!< pll2 input frequency range */
    uint32_t pll2_output_fre_range;    /*!< pll2 output frequency range */
    uint32_t pll2_fractional_latch;    /*!< pll2 fractional latch*/
} hal_rcu_pll2_struct;

typedef struct {
    hal_rcu_osc_state_enum        pllusb0_state;                /*!< the oscillators state */
    hal_rcu_pllusbhsx_precksrc_enum  pllusb0_preclock_source;  /*!< PLLUSBHS0 Pre-select the clock source */
    hal_rcu_pllusbhsx_prediv_enum pllusb0_presel;               /*!< PLLUSBHS0PREDV clock frequency division factor */
    hal_rcu_pllusbhsx_mul_enum    pllusb0_mul;                  /*!< PLLUSBHS0 clock multiplier */
} hal_rcu_pllusb0_struct;

typedef struct {
    hal_rcu_osc_state_enum pllusb1_state;                    /*!< the oscillators state */
    hal_rcu_pllusbhsx_precksrc_enum pllusb1_preclock_source; /*!< PLLUSBHS1 Pre-select the clock source */
    hal_rcu_pllusbhsx_prediv_enum pllusb1_presel;            /*!< PLLUSBHS1PREDV clock frequency division factor */
    hal_rcu_pllusbhsx_mul_enum pllusb1_mul;                  /*!< PLLUSBHS1 clock multiplier */
} hal_rcu_pllusb1_struct;

/* @STRUCT_MEMBER: pll
                   need_configure
   @brief        : RCU PLL
   @SP           : need_configure = ENABLE/DISABLE */
typedef struct {
    ControlStatus need_configure;    /*!< the oscillators configure flag */
    hal_rcu_pll_src_enum pll_source; /*!< PLL entry clock source */
    hal_rcu_pll0_struct pll0;        /*!< PLL0 status struct */
    hal_rcu_pll1_struct pll1;        /*!< PLL1 status struct */
    hal_rcu_pll2_struct pll2;        /*!< PLL1 status struct */
    hal_rcu_pllusb0_struct pllusb0;  /*!< PLLusb0 status struct */
    hal_rcu_pllusb1_struct pllusb1;  /*!< PLLusb1 status struct */
    hal_rcu_osc_state_enum state;    /*!< the oscillators state */
} hal_rcu_pll_struct;

/* configure the RCU oscillators */
/* @PARA: rcu_osci */
/* @STRUCT: RCU oscillators struct*/
typedef struct {
    hal_rcu_hxtal_struct hxtal;     /*!< HXTAL status structure */
    hal_rcu_lxtal_struct lxtal;     /*!< LXTAL status structure */
    hal_rcu_irc64m_struct irc64m;   /*!< IRC64M status structure */
    hal_rcu_irc48m_struct irc48m;   /*!< IRC48M status structure */
    hal_rcu_irc32k_struct irc32k;   /*!< IRC32K status structure */
    hal_rcu_lpirc4m_struct lpirc4m; /*!< LPIRC4M status structure */
    hal_rcu_pll_struct pll;         /*!< PLL structure parameters */
} hal_rcu_osci_struct;

/* function declarations */
/* @FUNCTION: configure the RCU oscillators */
int32_t hal_rcu_osci_config(hal_rcu_osci_struct *rcu_osci);
/* @FUNCTION: configure the clock0 out to output on CKOUT0 pin */
void hal_rcu_clock0_out_config(hal_rcu_clkout0_src_enum ckout0_src, hal_rcu_ckout0_div_enum ckout0_div);
/* @FUNCTION: configure the clock1 out to output on CKOUT1 pin */
void hal_rcu_clock1_out_config(hal_rcu_clkout1_src_enum ckout1_src, hal_rcu_ckout1_div_enum ckout1_div);
/* deinitialize the RCU */
int32_t hal_rcu_deinit(void);
/* @END */

/* initialize the RCU structure with the default values */
/* deinitiaize RCU */
int32_t hal_rcu_struct_init(hal_rcu_struct_type_enum rcu_struct_type, void *p_struct);
/* enable the peripherals clock */
void hal_rcu_periph_clock_enable(hal_rcu_periph_enum periph);
/* disable the peripherals clock */
void hal_rcu_periph_clock_disable(hal_rcu_periph_enum periph);

/* get the RCU oscillators configuration */
int32_t hal_rcu_osci_config_get(hal_rcu_osci_struct *rcu_osci);

/* configure the pll */
int32_t hal_rcu_pll_config(hal_rcu_pll_struct *p_pll);
/* get PLL configuration */
int32_t hal_rcu_pll_config_get(hal_rcu_pll_struct *p_pll);
/* initialize the RCU extended peripherals clocks */
int32_t hal_rcu_periph_clock_config(hal_rcu_periphclk_struct *periph_clk);
/* get the RCU extended peripherals clock configuration */
int32_t hal_rcu_periph_get_clock_config(hal_rcu_periphclk_struct *periph_clk);
/* get the RCU clock configuration */
int32_t hal_rcu_clock_config_get(hal_rcu_clk_struct *rcu_clk);

/* enable the HXTAL clock monitor */
void hal_rcu_hxtal_clock_monitor_enable(void);
/* disable the HXTAL clock monitor */
void hal_rcu_hxtal_clock_monitor_disable(void);
/* enable the LXTAL clock monitor */
void hal_rcu_lxtal_clock_monitor_enable(void);
/* disable the LXTAL clock monitor */
void hal_rcu_lxtal_clock_monitor_disable(void);

/* enable the peripherals clock when sleep mode */
void hal_rcu_periph_clock_sleep_enable(hal_rcu_periph_sleep_enum periph);
/* disable the peripherals clock when sleep mode */
void hal_rcu_periph_clock_sleep_disable(hal_rcu_periph_sleep_enum periph);

/* reset the peripherals */
void hal_rcu_periph_reset_enable(hal_rcu_periph_reset_enum periph_reset);
/* disable reset the peripheral */
void hal_rcu_periph_reset_disable(hal_rcu_periph_reset_enum periph_reset);

/* update the SystemCoreClock with current core clock retrieved from cpu registers */
uint32_t hal_rcu_system_core_clock_update(void);

/*  RCU clock configuration */
int32_t hal_rcu_clock_config(hal_rcu_clk_struct *rcu_clk);
/* get the system clock, bus and peripheral clock frequency */
uint32_t hal_rcu_clock_freq_get(hal_rcu_clock_freq_enum clock);

/* RCU interrupt handler content function,which is merely used in RCU_IRQHandler */
void hal_rcu_irq(void);
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_rcu_irq_handle_set(hal_rcu_irq_struct *prcu_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
void hal_rcu_irq_handle_all_reset(void);

/* reset source detect */
uint8_t hal_rcu_reset_source_detect(void);
/* configure the Deep-sleep wakeup system clock source selection */
void hal_rcu_deepsleep_wakeup_sys_clock_config(hal_rcu_dspwu_cksrc_enum ck_dspwussel);

/* configure the IRC64M clock divider selection */
void hals_rcu_irc64mdiv_clock_config(hal_rcu_irc64m_div_enum ck_irc64mdiv);

/* wait for oscillator stabilization flags is SET or oscillator startup is timeout */
ErrStatus hals_rcu_osci_stab_wait(hal_rcu_osci_type_enum osci);
/* turn on the oscillator */
void hals_rcu_osci_on(hal_rcu_osci_type_enum osci);
/* turn off the oscillator */
void hals_rcu_osci_off(hal_rcu_osci_type_enum osci);

/* system and peripherals clock source, system reset configure functions */
/* configure the system clock source */
void hals_rcu_system_clock_source_config(hal_rcu_cksys_src_enum ck_sys);
/* configure the AHB clock prescaler selection */
void hals_rcu_ahb_clock_config(hal_rcu_sysclk_ahbdiv_enum ck_ahb);
/* configure the APB1 clock prescaler selection */
void hals_rcu_apb1_clock_config(hal_rcu_ahbclk_apb1div_enum ck_apb1);
/* configure the APB2 clock prescaler selection */
void hals_rcu_apb2_clock_config(hal_rcu_ahbclk_apb2div_enum ck_apb2);
/* configure the APB3 clock prescaler selection */
void hals_rcu_apb3_clock_config(hal_rcu_ahbclk_apb3div_enum ck_apb3);
/* configure the APB4 clock prescaler selection */
void hals_rcu_apb4_clock_config(hal_rcu_ahbclk_apb4div_enum ck_apb4);

/* get the system clock source */
uint32_t hals_rcu_system_clock_source_get(void);

/* get the irc64mdiv clock */
uint32_t hals_rcu_irc64mdiv_freq_get(void);

/* configure the RTC clock source selection */
void hals_rcu_rtc_clock_config(hal_rcu_rtc_cksrc_enum rtc_clock_source, hal_rcu_rtc_div_enum rtc_div);
/* configure the trng clock selection */
void hals_rcu_trng_clock_config(hal_rcu_pll48m_cksrc_enum pll48m_clock_source, hal_rcu_ck48m_cksrc_enum ck48m_clock_source);
/* configure the TIMER clock multiple selection */
void hals_rcu_timer_clock_multiple_config(hal_rcu_timer_ckmul_enum timer_clock_prescaler);
/* configure the SPI / I2S clock source selection */
int32_t hals_rcu_spi_clock_config(hal_spi_idx_enum spi_idx, uint32_t ck_spi);
/* configure the SDIO clock source selection */
int32_t hals_rcu_sdio_clock_config(hal_sdio_idx_enum sdio_idx, hal_rcu_sdio_cksrc_enum ck_sdio);
/* configure the USARTx (x = 0,1,2,5) clock source selection */
int32_t hals_rcu_usart_clock_config(hal_usart_idx_enum usart_idx, hal_rcu_usart_cksrc_enum ck_usart);
/* configure the I2Cx (x = 0,1,2,3) clock source selection */
int32_t hals_rcu_i2c_clock_config(hal_i2c_idx_enum i2c_idx, hal_rcu_i2c_cksrc_enum ck_i2c);
/* configure the CANx (x = 0,1,2) clock source selection */
int32_t hals_rcu_can_clock_config(hal_can_idx_enum can_idx, hal_rcu_can_cksrc_enum ck_can);
/* configure the ADCx (x = 0,1,2) clock source selection */
int32_t hals_rcu_adc_clock_config(hal_adc_idx_enum adc_idx, hal_rcu_adc_cksrc_enum ck_adc);
/* configure the SAIx (x = 0,1) clock source selection */
int32_t hals_rcu_sai_clock_config(hal_sai_idx_enum sai_idx, hal_rcu_sai_cksrc_enum ck_sai);
/* configure the SAI2Bx (x = 0,1) clock source selection */
int32_t hals_rcu_sai2b_clock_config(hal_sai2b_idx_enum sai2b_idx, hal_rcu_sai2b_cksrc_enum ck_sai2b);
/* configure the RSPDIF clock source selection */
void hals_rcu_rspdif_clock_config(hal_rcu_rspdif_cksrc_enum ck_rspdif);
/* configure the EXMC clock source selection */
void hals_rcu_exmc_clock_config(hal_rcu_exmc_cksrc_enum ck_exmc);
/* configure the HPDF clock source selection */
void hals_rcu_hpdf_clock_config(hal_rcu_hpdf_cksrc_enum ck_hpdf);
/* configure the PER clock source selection */
void hals_rcu_per_clock_config(hal_rcu_pef_cksrc_enum ck_per);

/* configure the usbhs divided */
int32_t hals_rcu_usbhs_div_config(hal_usbhs_idx_enum usbhs_idx, hal_rcu_usbhsx_div_enum ck_usbhsdiv);
/* configure the USBHS prescaler */
int32_t hals_rcu_usbhs_psc_config(hal_usbhs_idx_enum usbhs_idx, hal_rcu_usbhsx_psc_enum ck_usbhspsc);
/* configure the USBHS48M clock source selection */
int32_t hals_rcu_usb48m_clock_config(hal_usbhs_idx_enum usbhs_idx, hal_rcu_usbhsx48m_cksrc_enum ck_usb48m);
/* configure the USBHSSEL clock source selection */
int32_t hals_rcu_usbhs_clock_config(hal_usbhs_idx_enum usbhs_idx, hal_rcu_usbhsx_cksrc_enum ck_usbhs);
/* config the USBHS clock source selection */
int32_t hals_rcu_usbhs_clock_selection_config(hal_usbhs_idx_enum usbhs_idx);

/* LXTAL, IRC64M, LPIRC4M, PLL and other oscillator configure functions */
/* configure the LXTAL drive capability */
void hals_rcu_lxtal_drive_capability_config(hal_rcu_lxtal_drive_capability_enum lxtal_dricap);

/* reset the BKP */
void hals_rcu_bkp_reset_enable(void);
/* disable the BKP reset */
void hals_rcu_bkp_reset_disable(void);

/* enable the oscillator bypass mode, HXTALEN or LXTALEN must be reset before it */
int32_t hals_rcu_osci_bypass_mode_enable(hal_rcu_osci_type_enum osci);
/* disable the oscillator bypass mode, HXTALEN or LXTALEN must be reset before it */
int32_t hals_rcu_osci_bypass_mode_disable(hal_rcu_osci_type_enum osci);

/* set the IRC64M adjust value */
int32_t hals_rcu_irc64m_adjust_value_set(uint32_t irc64m_adjval);
/* set the LPIRC4M adjust value */
int32_t hals_rcu_lpirc4m_adjust_value_set(uint32_t lpirc4m_adjval);

/* flag & interrupt functions */
/* get the clock stabilization and peripheralreset flags */
FlagStatus hals_rcu_flag_get(hal_rcu_flag_enum flag);
/* clear all the reset flag */
void hals_rcu_all_reset_flag_clear(void);

/* enable the stabilization interrupt */
void hals_rcu_interrupt_enable(hal_rcu_int_enum interrupt);
/* disable the stabilization interrupt */
void hals_rcu_interrupt_disable(hal_rcu_int_enum interrupt);
/* get the clock stabilization interrupt and ckm flags */
FlagStatus hals_rcu_interrupt_flag_get(hal_rcu_int_flag_enum int_flag);
/* clear the interrupt flags */
void hals_rcu_interrupt_flag_clear(hal_rcu_int_flag_clear_enum int_flag);

#endif /* GD32H7XX_HAL_RCU_H */
