/*!
    \file    gd32h7xx_hal_rameccmu.h
    \brief   definitions for the RAMECCMU

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_RAMECCMU_H
#define GD32H7XX_HAL_RAMECCMU_H

#include "gd32h7xx_hal.h"

/* RAMECCMU definitions */
/* @PARA: periph */
/* @DEFINE: rameccmu periph */
#define RAMECCMU0                                   (RAMECCMU_BASE + 0x09FE6000U)          /*!< RAMECCMU for region 0 */
#define RAMECCMU1                                   RAMECCMU_BASE                          /*!< RAMECCMU for region 1 */

/* registers definitions */
#define RAMECCMU_INT(rameccmux)                     REG32((rameccmux) + 0x00000000U)       /*!< RAMECCMU global interrupt register */

#define RAMECCMU_M0CTL(rameccmux)                   REG32((rameccmux) + 0x00000020U)       /*!< RAMECCMU monitor 0 control register */
#define RAMECCMU_M0STAT(rameccmux)                  REG32((rameccmux) + 0x00000024U)       /*!< RAMECCMU monitor 0 status register */
#define RAMECCMU_M0FADDR(rameccmux)                 REG32((rameccmux) + 0x00000028U)       /*!< RAMECCMU monitor 0 ECC failing address register */
#define RAMECCMU_M0FDL(rameccmux)                   REG32((rameccmux) + 0x0000002CU)       /*!< RAMECCMU monitor 0 ECC failing data low register */
#define RAMECCMU_M0FDH(rameccmux)                   REG32((rameccmux) + 0x00000030U)       /*!< RAMECCMU monitor 0 ECC failing data high register */
#define RAMECCMU_M0FECODE(rameccmux)                REG32((rameccmux) + 0x00000034U)       /*!< RAMECCMU monitor 0 failing ECC error code register */

#define RAMECCMU_M1CTL(rameccmux)                   REG32((rameccmux) + 0x00000040U)       /*!< RAMECCMU monitor 1 control register */
#define RAMECCMU_M1STAT(rameccmux)                  REG32((rameccmux) + 0x00000044U)       /*!< RAMECCMU monitor 1 status register */
#define RAMECCMU_M1FADDR(rameccmux)                 REG32((rameccmux) + 0x00000048U)       /*!< RAMECCMU monitor 1 ECC failing address register */
#define RAMECCMU_M1FDL(rameccmux)                   REG32((rameccmux) + 0x0000004CU)       /*!< RAMECCMU monitor 1 ECC failing data low register */
#define RAMECCMU_M1FDH(rameccmux)                   REG32((rameccmux) + 0x00000050U)       /*!< RAMECCMU monitor 1 ECC failing data high register */
#define RAMECCMU_M1FECODE(rameccmux)                REG32((rameccmux) + 0x00000054U)       /*!< RAMECCMU monitor 1 failing ECC error code register */

#define RAMECCMU_M2CTL(rameccmux)                   REG32((rameccmux) + 0x00000060U)       /*!< RAMECCMU monitor 2 control register */
#define RAMECCMU_M2STAT(rameccmux)                  REG32((rameccmux) + 0x00000064U)       /*!< RAMECCMU monitor 2 status register */
#define RAMECCMU_M2FADDR(rameccmux)                 REG32((rameccmux) + 0x00000068U)       /*!< RAMECCMU monitor 2 ECC failing address register */
#define RAMECCMU_M2FDL(rameccmux)                   REG32((rameccmux) + 0x0000006CU)       /*!< RAMECCMU monitor 2 ECC failing data low register */
#define RAMECCMU_M2FDH(rameccmux)                   REG32((rameccmux) + 0x00000070U)       /*!< RAMECCMU monitor 2 ECC failing data high register */
#define RAMECCMU_M2FECODE(rameccmux)                REG32((rameccmux) + 0x00000074U)       /*!< RAMECCMU monitor 2 failing ECC error code register */

#define RAMECCMU_M3CTL(rameccmux)                   REG32((rameccmux) + 0x00000080U)       /*!< RAMECCMU monitor 3 control register */
#define RAMECCMU_M3STAT(rameccmux)                  REG32((rameccmux) + 0x00000084U)       /*!< RAMECCMU monitor 3 status register */
#define RAMECCMU_M3FADDR(rameccmux)                 REG32((rameccmux) + 0x00000088U)       /*!< RAMECCMU monitor 3 ECC failing address register */
#define RAMECCMU_M3FDL(rameccmux)                   REG32((rameccmux) + 0x0000008CU)       /*!< RAMECCMU monitor 3 ECC failing data low register */
#define RAMECCMU_M3FDH(rameccmux)                   REG32((rameccmux) + 0x00000090U)       /*!< RAMECCMU monitor 3 ECC failing data high register */
#define RAMECCMU_M3FECODE(rameccmux)                REG32((rameccmux) + 0x00000094U)       /*!< RAMECCMU monitor 3 failing ECC error code register */

#define RAMECCMU_M4CTL(rameccmux)                   REG32((rameccmux) + 0x000000A0U)       /*!< RAMECCMU monitor 4 control register */
#define RAMECCMU_M4STAT(rameccmux)                  REG32((rameccmux) + 0x000000A4U)       /*!< RAMECCMU monitor 4 status register */
#define RAMECCMU_M4FADDR(rameccmux)                 REG32((rameccmux) + 0x000000A8U)       /*!< RAMECCMU monitor 4 ECC failing address register */
#define RAMECCMU_M4FDL(rameccmux)                   REG32((rameccmux) + 0x000000ACU)       /*!< RAMECCMU monitor 4 ECC failing data low register */
#define RAMECCMU_M4FDH(rameccmux)                   REG32((rameccmux) + 0x000000B0U)       /*!< RAMECCMU monitor 4 ECC failing data high register */
#define RAMECCMU_M4FECODE(rameccmux)                REG32((rameccmux) + 0x000000B4U)       /*!< RAMECCMU monitor 4 failing ECC error code register */

/* bits definitions */
/* RAMECCMU_INT */
#define RAMECCMU_INT_GEIE                           BIT(0)                                 /*!< global ECC interrupt enable */
#define RAMECCMU_INT_GESERRIE                       BIT(1)                                 /*!< global ECC single error interrupt enable */
#define RAMECCMU_INT_GEDERRIE                       BIT(2)                                 /*!< global ECC double error interrupt enable */
#define RAMECCMU_INT_GEDERRBWIE                     BIT(3)                                 /*!< global ECC double error on byte write interrupt enable */

/* RAMECCMU_MxCTL */
#define RAMECCMU_MXCTL_ECCSERRIE                    BIT(2)                                 /*!< ECC single error interrupt enable */
#define RAMECCMU_MXCTL_ECCDERRIE                    BIT(3)                                 /*!< ECC double error interrupt enable */
#define RAMECCMU_MXCTL_ECCDERRBWIE                  BIT(4)                                 /*!< ECC double error on byte write interrupt enable */
#define RAMECCMU_MXCTL_ECCERRLATEN                  BIT(5)                                 /*!< ECC error latching enable */

/* RAMECCMU_MxSTAT */
#define RAMECCMU_MXSTAT_ECCSERRDCF                  BIT(0)                                 /*!< ECC single error detected and corrected flag */
#define RAMECCMU_MXSTAT_ECCDERRDF                   BIT(1)                                 /*!< ECC double error detected flag */
#define RAMECCMU_MXSTAT_ECCDERRBWDF                 BIT(2)                                 /*!< ECC double error on byte write detected flag */

/* RAMECCMU_MxFADDR */
#define RAMECCMU_MXFADDR_ECCFADDR                   BITS(0,31)                             /*!< ECC error failing address */

/* RAMECCMU_MxFDL */
#define RAMECCMU_MXFDL_ECCFDL                       BITS(0,31)                             /*!< ECC failing data low bits */

/* RAMECCMU_MxFDH */
#define RAMECCMU_MXFDH_ECCFDH                       BITS(0,31)                             /*!< ECC failing data high bits */

/* RAMECCMU_MxFECODE */
#define RAMECCMU_MXFECODE_ECCFECODE                 BITS(0,31)                             /*!< ECC failing error code */

/* constants definitions */
/* RAMECCMUx monitor register address */
#define RAMECCMU_MXCTL(rameccmu_monitor)            REG32(RAMECCMU0 - ((uint32_t)(rameccmu_monitor) >> 4U) * 0x09FE6000U + \
                                                          0x20U + ((rameccmu_monitor) & 0x0FU) * 0x20U)                       /*!< RAMECCMU monitor control register */
#define RAMECCMU_MXSTAT(rameccmu_monitor)           REG32(RAMECCMU0 - ((uint32_t)(rameccmu_monitor) >> 4U) * 0x09FE6000U + \
                                                          0x24U + ((rameccmu_monitor) & 0x0FU) * 0x20U)                       /*!< RAMECCMU monitor status register */
#define RAMECCMU_MXFADDR(rameccmu_monitor)          REG32(RAMECCMU0 - ((uint32_t)(rameccmu_monitor) >> 4U) * 0x09FE6000U + \
                                                          0x28U + ((rameccmu_monitor) & 0x0FU) * 0x20U)                       /*!< RAMECCMU monitor ECC failing address register */
#define RAMECCMU_MXFDL(rameccmu_monitor)            REG32(RAMECCMU0 - ((uint32_t)(rameccmu_monitor) >> 4U) * 0x09FE6000U + \
                                                          0x2CU + ((rameccmu_monitor) & 0x0FU) * 0x20U)                       /*!< RAMECCMU monitor ECC failing data low register */
#define RAMECCMU_MXFDH(rameccmu_monitor)            REG32(RAMECCMU0 - ((uint32_t)(rameccmu_monitor) >> 4U) * 0x09FE6000U + \
                                                          0x30U + ((rameccmu_monitor) & 0x0FU) * 0x20U)                       /*!< RAMECCMU monitor ECC failing data high register */
#define RAMECCMU_MXFECODE(rameccmu_monitor)         REG32(RAMECCMU0 - ((uint32_t)(rameccmu_monitor) >> 4U) * 0x09FE6000U + \
                                                          0x34U + ((rameccmu_monitor) & 0x0FU) * 0x20U)                       /*!< RAMECCMU monitor failing ECC error code register */

#define RAMECCMU_INT_ECC_GLOBAL_ERROR               BIT(0)                                 /*!< ECC global error interrupt */
#define RAMECCMU_INT_ECC_SINGLE_ERROR               BIT(1)                                 /*!< ECC single error interrupt */
#define RAMECCMU_INT_ECC_DOUBLE_ERROR               BIT(2)                                 /*!< ECC double error interrupt */
#define RAMECCMU_INT_ECC_DOUBLE_ERROR_BYTE_WRITE    BIT(3)                                 /*!< ECC double error on byte write interrupt */
#define RAMECCMU_INT_ECC_ERROR_LATCHING             BIT(4)                                 /*!< ECC error latching */

#define RAMECCMU_FLAG_ECC_SINGLE_ERROR              RAMECCMU_MXSTAT_ECCSERRDCF             /*!< ECC single error detected and corrected flag */
#define RAMECCMU_FLAG_ECC_DOUBLE_ERROR              RAMECCMU_MXSTAT_ECCDERRDF              /*!< ECC double error detected flag */
#define RAMECCMU_FLAG_ECC_DOUBLE_ERROR_BYTE_WRITE   RAMECCMU_MXSTAT_ECCDERRBWDF            /*!< ECC double error on byte write detected flag */

/* @PARA: global_ecc */
/* @DEFINE: global_ecc */
#define RAMECCMU_GECC_ERROR_DISENABLE               ((uint32_t)0x00000000U)                /*!< global ECC interrupt disenable */
#define RAMECCMU_GECC_ERROR_ENABLE                  RAMECCMU_INT_GEIE                      /*!< global ECC interrupt enable */

/* @PARA: global_ecc_single */
/* @DEFINE: global_ecc_single */
#define RAMECCMU_GECC_SINGLE_ERROR_DISENABLE        ((uint32_t)0x00000000U)                /*!< global ECC single error interrupt disenable */
#define RAMECCMU_GECC_SINGLE_ERROR_ENABLE           RAMECCMU_INT_GESERRIE                  /*!< global ECC single error interrupt enable */

/* @PARA: global_ecc_double */
/* @DEFINE: global_ecc_double */
#define RAMECCMU_GECC_DOUBLE_ERROR_DISENABLE        ((uint32_t)0x00000000U)                /*!< global ECC double error interrupt disenable */
#define RAMECCMU_GECC_DOUBLE_ERROR_ENABLE           RAMECCMU_INT_GEDERRIE                  /*!< global ECC double error interrupt enable */

/* @PARA: global_ecc_double_byte */
/* @DEFINE: global_ecc_double_byte */
#define RAMECCMU_GECC_DOUBLE_BYTE_ERROR_DISENABLE   ((uint32_t)0x00000000U)                /*!< global ECC double  byte error interrupt disenable */
#define RAMECCMU_GECC_DOUBLE_BYTE_ERROR_ENABLE      RAMECCMU_INT_GEDERRBWIE                /*!< global ECC double byte error interrupt enable */

#define RAMECCMU_ECC_SINGLE_DISENABLE               ((uint32_t)0x00000000U)                /*!< ECC single error interrupt disenable */
#define RAMECCMU_ECC_SINGLE_ENABLE                  RAMECCMU_MXCTL_ECCSERRIE               /*!< gECC single error interrupt enable */

#define RAMECCMU_ECC_DOUBLE_ERROR_DISENABLE         ((uint32_t)0x00000000U)                /*!< ECC double error interrupt disenable */
#define RAMECCMU_ECC_DOUBLE_ERROR_ENABLE            RAMECCMU_MXCTL_ECCDERRIE               /*!<  ECC double error interrupt enable */

#define RAMECCMU_ECC_DOUBLE_BYTE_ERROR_DISENABLE    ((uint32_t)0x00000000U)                /*!< global ECC double error byte interrupt disenable */
#define RAMECCMU_ECC_DOUBLE_BYTE_ERROR_ENABLE       RAMECCMU_MXCTL_ECCDERRBWIE             /*!< global ECC double byte error interrupt enable */

#define RAMECCMU_ECC_LATCH_ERROR_DISENABLE          ((uint32_t)0x00000000U)                /*!< ECC single latch error interrupt disenable */
#define RAMECCMU_ECC_LATCH_ERROR_ENABLE             RAMECCMU_MXCTL_ECCERRLATEN             /*!< ECC latch error interrupt enable */

/* RAMECCU structure type enum */
typedef enum {
    HAL_RAMECCMU_INIT_STRUCT = 0U, /*!< RAMECCMU initialization structure */
    HAL_RAMECCMU_DEV_STRUCT,       /*!< RAMECCMU device information structure */
    HAL_RAMECCMU_IRQ_STRUCT        /*!< RAMECCMU IRQ structure */
} hal_rameccmu_struct_type_enum;

/* enumerator definition */
/* RAMECCMU state enum */
typedef enum {
    HAL_RAMECCMU_STATE_NONE = 0U, /*!< NONE(default value) */
    HAL_RAMECCMU_STATE_RESET,     /*!< RESET */
    HAL_RAMECCMU_STATE_READY,     /*!< READY */
    HAL_RAMECCMU_STATE_BUSY       /*!< BUSY */
} hal_rameccmu_state_enum;

/* RAMECCMU error state */
typedef enum {
    HAL_RAMECCMU_ERROR_STATE_NONE = 0U, /*!< NONE(default value) */
    HAL_RAMECCMU_ERROR_STATE_TIMEROUT,  /*!< TIMEOUT */
    HAL_RAMECCMU_ERROR_STATE            /*!< BUSY */
} hal_rameccmu_error_state_enum;

/* RAMECCMU error detect state */
typedef enum {
    HAL_RAMECCMU_DETECT_ERROR_STATE_NONE = 0U, /*!< NONE(default value) */
    HAL_RAMECCMU_DETECT_ERROR_STATE_SINGLE,    /*!< SINGLE */
    HAL_RAMECCMU_DETECT_ERROR_STATE_DOUBLE     /*!< DOUBLE */
} hal_rameccmu_detect_error_state_enum;

/* @STRUCT_MEMBER: rameccmu_monitor_select */
/* @ENUM: rameccmu_monitor_select */
typedef enum {
    RAMECCMU0_MONITOR0 = 0x00U, /*!< RAMECCMU0 monitor 0 */
    RAMECCMU0_MONITOR1 = 0x01U, /*!< RAMECCMU0 monitor 1 */
    RAMECCMU0_MONITOR2 = 0x02U, /*!< RAMECCMU0 monitor 2 */
    RAMECCMU0_MONITOR3 = 0x03U, /*!< RAMECCMU0 monitor 3 */
    RAMECCMU0_MONITOR4 = 0x04U, /*!< RAMECCMU0 monitor 4 */
    RAMECCMU1_MONITOR0 = 0x10U, /*!< RAMECCMU1 monitor 0 */
    RAMECCMU1_MONITOR1 = 0x11U, /*!< RAMECCMU1 monitor 1 */
    RAMECCMU1_MONITOR2 = 0x12U  /*!< RAMECCMU1 monitor 2 */
} hal_rameccmu_monitor_enum;

/* structure definition */
/* @PARA: p_init */
/* @STRUCT: init structure */
typedef struct {
    uint32_t global_ecc;                               /*!< global ECC interrupt */
    uint32_t global_ecc_single;                        /*!< global ECC single error interrupt */
    uint32_t global_ecc_double;                        /*!< global ECC double error interrupt */
    uint32_t global_ecc_double_byte;                   /*!< global ECC double error on byte write interrupt */
    hal_rameccmu_monitor_enum rameccmu_monitor_select; /*!< rameccmu monitor select */
} hal_rameccmu_init_struct;

/* RAMECCMU device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb global_ecc_single_error_handler;       /*!< RAMECCMU global ECC single error callback function */
    __IO hal_irq_handle_cb global_ecc_double_error_handler;       /*!< RAMECCMU global ECC double error callback function */
    __IO hal_irq_handle_cb global_ecc_double_bytes_error_handler; /*!< RAMECCMU global ECC double error
                                                                       on byte write callback function */
} hal_rameccmu_irq_struct;

/* @PARA: rameccmu_dev */
/* @STRUCT: rameccmu device information structure */
typedef struct {
    uint32_t periph;                                              /*!< RAMECCMU peripheral */
    __IO hal_rameccmu_state_enum state;                           /*!< RAMECCMU state */
    __IO hal_rameccmu_error_state_enum error_state;               /*!< RAMECCMU error state */
    __IO hal_rameccmu_detect_error_state_enum detect_error_state; /*!< RAMECCMU detect error state */
    hal_mutex_enum mutex;                                         /*!< mutex */
    hal_rameccmu_irq_struct rameccmu_irq;                         /*!< RAMECCMU device interrupt callback function pointer */
    hal_rameccmu_monitor_enum rameccmu_monitor;                   /*!< rameccmu monitor select */
} hal_rameccmu_dev_struct;

/* function declarations */
/* @FUNCTION: initialize RAMECCMU */
/* initialize RAMECCMU */
int32_t hal_rameccmu_init(hal_rameccmu_dev_struct *rameccmu_dev, uint32_t periph, \
                          hal_rameccmu_init_struct *p_init);
/* initialize the RAMECCMU structure */
int32_t hal_rameccmu_struct_init(hal_rameccmu_struct_type_enum hal_struct_type, void *p_struct);
/* deinitialize RAMECCMU */
int32_t hal_rameccmu_deinit(hal_rameccmu_dev_struct *rameccmu_dev);
/* @END */

/* RAMECCMU monitor ECC functions */
/* get RAMECCMU monitor ECC failing address */
uint32_t hal_rameccmu_monitor_failing_address_get(hal_rameccmu_dev_struct *rameccmu_dev);
/* get RAMECCMU monitor ECC failing data low 32 bits */
uint32_t hal_rameccmu_monitor_failing_data_low_bits_get(hal_rameccmu_dev_struct *rameccmu_dev);
/* get RAMECCMU monitor ECC failing data high 32 bits */
uint32_t hal_rameccmu_monitor_failing_data_high_bits_get(hal_rameccmu_dev_struct *rameccmu_dev);
/* get RAMECCMU monitor failing ECC error code */
uint32_t hal_rameccmu_monitor_failing_ecc_error_code_get(hal_rameccmu_dev_struct *rameccmu_dev);

/* interrupt & flag functions */
/* handles RAMECCMU interrupt request */
int32_t hal_rameccmu_irq(hal_rameccmu_dev_struct *rameccmu_dev);
/* set user-defined interrupt callback function,
   which will be registered and called when corresponding interrupt be triggered */
int32_t hal_rameccmu_irq_handle_set(hal_rameccmu_dev_struct *rameccmu_dev, hal_rameccmu_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
   which will be registered and called when corresponding interrupt be triggered */
int32_t hal_rameccmu_irq_handle_all_reset(hal_rameccmu_dev_struct *rameccmu_dev);

/* enable RAMECCMU latching error information */
int32_t hal_rameccmu_monitor_latch_enable(hal_rameccmu_dev_struct *rameccmu_dev);
/* disable RAMECCMU latching error information */
int32_t hal_rameccmu_monitor_latch_disable(hal_rameccmu_dev_struct *rameccmu_dev);

/* RAMECCMU state get */
hal_rameccmu_state_enum hal_rameccmu_state_get(hal_rameccmu_dev_struct *rameccmu_dev);
/* RAMECCMU error state get */
hal_rameccmu_error_state_enum hal_rameccmu_error_state_get(hal_rameccmu_dev_struct *rameccmu_dev);
/* RAMECCMU detect error state get */
hal_rameccmu_detect_error_state_enum hal_rameccmu_detect_error_state_get(hal_rameccmu_dev_struct *rameccmu_dev);

/* enable RAMECCMU global ECC interrupt */
void hals_rameccmu_global_interrupt_enable(uint32_t periph, uint32_t interrupt);
/* disable RAMECCMU global ECC interrupt */
void hals_rameccmu_global_interrupt_disable(uint32_t periph, uint32_t interrupt);
/* enable RAMECCMU monitor ECC error interrupt */
void hals_rameccmu_monitor_interrupt_enable(hal_rameccmu_monitor_enum rameccmu_monitor, uint32_t monitor_interrupt);
/* disable RAMECCMU monitor ECC error interrupt */
void hals_rameccmu_monitor_interrupt_disable(hal_rameccmu_monitor_enum rameccmu_monitor, uint32_t monitor_interrupt);

/* get RAMECCMU monitor ECC error flag */
FlagStatus hal_rameccmu_monitor_flag_get(hal_rameccmu_monitor_enum rameccmu_monitor, uint32_t flag);
/* clear RAMECCMU monitor ECC error flag */
void hal_rameccmu_monitor_flag_clear(hal_rameccmu_monitor_enum rameccmu_monitor, uint32_t flag);
/* get RAMECCMU monitor ECC interrupt error flag */
FlagStatus hals_rameccmu_monitor_interrupt_flag_get(hal_rameccmu_monitor_enum rameccmu_monitor, uint32_t int_flag);
/* clear RAMECCMU monitor interrupt ECC error flag */
void hals_rameccmu_monitor_interrupt_flag_clear(hal_rameccmu_monitor_enum rameccmu_monitor, uint32_t int_flag);

#endif /* GD32H7XX_HAL_RAMECCMU_H */
