/*!
    \file    gd32h7xx_hal_pmu.h
    \brief   definitions for the PMU

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_PMU_H
#define GD32H7XX_HAL_PMU_H

#include "gd32h7xx_hal.h"

/* PMU definitions */
#define PMU                                 PMU_BASE                                                                    /*!< PMU base address */

/* registers definitions */
#define PMU_CTL0                            REG32((PMU) + 0x00000000U)                                                  /*!< PMU control register 0 */
#define PMU_CS                              REG32((PMU) + 0x00000004U)                                                  /*!< PMU control and status register */
#define PMU_CTL1                            REG32((PMU) + 0x00000008U)                                                  /*!< PMU control register 1 */
#define PMU_CTL2                            REG32((PMU) + 0x00000010U)                                                  /*!< PMU control register 2 */
#define PMU_CTL3                            REG32((PMU) + 0x00000014U)                                                  /*!< PMU control register 3 */
#define PMU_PAR                             REG32((PMU) + 0x00000018U)                                                  /*!< PMU parameter register */

/* bits definitions */
/* PMU_CTL0 */
#define PMU_CTL0_STBMOD                     BIT(1)                                                                      /*!< standby mode */
#define PMU_CTL0_WURST                      BIT(2)                                                                      /*!< wakeup flag reset */
#define PMU_CTL0_STBRST                     BIT(3)                                                                      /*!< standby flag reset */
#define PMU_CTL0_LVDEN                      BIT(4)                                                                      /*!< low voltage detector enable */
#define PMU_CTL0_LVDT                       BITS(5,7)                                                                   /*!< low voltage detector threshold */
#define PMU_CTL0_BKPWEN                     BIT(8)                                                                      /*!< backup domain write enable */
#define PMU_CTL0_SLDOVS                     BITS(14,15)                                                                 /*!< deep sleep mode mode LDO output voltage select */
#define PMU_CTL0_VAVDEN                     BIT(16)                                                                     /*!< VDDA analog voltage detector enable */
#define PMU_CTL0_VAVDVC                     BITS(17,18)                                                                 /*!< VDDA analog voltage detector level configure bits */
#define PMU_CTL0_VOVDEN                     BIT(19)                                                                     /*!< peripheral voltage on V0.9V detector enable bit */

/* PMU_CS */
#define PMU_CS_WUF                          BIT(0)                                                                      /*!< wakeup flag */
#define PMU_CS_STBF                         BIT(1)                                                                      /*!< standby flag */
#define PMU_CS_LVDF                         BIT(2)                                                                      /*!< low voltage detector status flag */
#define PMU_CS_WUPEN0                       BIT(8)                                                                      /*!< wakeup pin0(PA0) enable */
#define PMU_CS_WUPEN1                       BIT(9)                                                                      /*!< wakeup pin1(PA2) enable */
#define PMU_CS_WUPEN3                       BIT(11)                                                                     /*!< wakeup pin3(PC13) enable */
#define PMU_CS_WUPEN5                       BIT(13)                                                                     /*!< wakeup pin5(PC1) enable */
#define PMU_CS_VAVDF                        BIT(16)                                                                     /*!< VDDA analog voltage detector voltage output on VDDA flag */
#define PMU_CS_VOVDF                        BIT(20)                                                                     /*!< peripheral voltage on V0.9V detector flag bit */

/* PMU_CTL1 */
#define PMU_CTL1_BKPVSEN                    BIT(0)                                                                      /*!< backup voltage stabilizer enable */
#define PMU_CTL1_VBTMEN                     BIT(4)                                                                      /*!< VBAT and temperature monitoring enable */
#define PMU_CTL1_BKPVSRF                    BIT(16)                                                                     /*!< backup regulator ready */
#define PMU_CTL1_VBATLF                     BIT(20)                                                                     /*!< VBAT level monitoring versus low threshold */
#define PMU_CTL1_VBATHF                     BIT(21)                                                                     /*!< VBAT level monitoring versus high threshold */
#define PMU_CTL1_TEMPLF                     BIT(22)                                                                     /*!< temperature level monitoring versus low threshold */
#define PMU_CTL1_TEMPHF                     BIT(23)                                                                     /*!< temperature level monitoring versus high threshold */

/* PMU_CTL2 */
#define PMU_CTL2_BYPASS                     BIT(0)                                                                      /*!< power management unit bypass control bit */
#define PMU_CTL2_LDOEN                      BIT(1)                                                                      /*!< Low drop-out voltage stabilizer enable bit */
#define PMU_CTL2_DVSEN                      BIT(2)                                                                      /*!< step-down voltage stabilizer enable bit */
#define PMU_CTL2_VCRSEL                     BIT(8)                                                                      /*!< VBAT battery charging resistor selection */
#define PMU_CTL2_VCEN                       BIT(9)                                                                      /*!< VBAT battery charging enable */
#define PMU_CTL2_DVSRF                      BIT(16)                                                                     /*!< step-down voltage stabilizer ready flag bit */
#define PMU_CTL2_VUSB33DEN                  BIT(24)                                                                     /*!< VDD33USB voltage level detector enable bit */
#define PMU_CTL2_USBSEN                     BIT(25)                                                                     /*!< USB voltage stabilizer enable */
#define PMU_CTL2_USB33RF                    BIT(26)                                                                     /*!< USB supply ready flag */

/* PMU_CTL3 */
#define PMU_CTL3_LDOVS                      BITS(12,14)                                                                 /*!< voltage scaling selection according to performance */
#define PMU_CTL3_VOVRF                      BIT(16)                                                                     /*!< V0.9V voltage ready bit */

/* PMU_PAR */
#define PMU_PAR_CNT                         BITS(0,11)                                                                  /*!< exit deepsleep mode wait time count configure bits */
#define PMU_PAR_TSW_IRCCNT                  BITS(16,20)                                                                 /*!< when enter deepsleep, switch to IRC wait clock */

/* constants definitions */
/* PMU low voltage detector threshold definitions */
/* @STRUCT_MEMBER: lvd_threshold */
/* @DEFINE: lvdt threshold */
#define CTL0_LVDT(regval)                   (BITS(5,7) & ((uint32_t)(regval) << 5U))
#define PMU_LVDT_0                          CTL0_LVDT(0)                                                                /*!< voltage threshold is 2.1V */
#define PMU_LVDT_1                          CTL0_LVDT(1)                                                                /*!< voltage threshold is 2.3V */
#define PMU_LVDT_2                          CTL0_LVDT(2)                                                                /*!< voltage threshold is 2.4V */
#define PMU_LVDT_3                          CTL0_LVDT(3)                                                                /*!< voltage threshold is 2.6V */
#define PMU_LVDT_4                          CTL0_LVDT(4)                                                                /*!< voltage threshold is 2.7V */
#define PMU_LVDT_5                          CTL0_LVDT(5)                                                                /*!< voltage threshold is 2.9V */
#define PMU_LVDT_6                          CTL0_LVDT(6)                                                                /*!< voltage threshold is 3.0V */
#define PMU_LVDT_7                          CTL0_LVDT(7)                                                                /*!< input analog voltage on PB7 (compared with 0.8V) */

/* PMU deepsleep mode voltage scaling selection */
/* @STRUCT_MEMBER: deep_sleep_voltage */
/* @DEFINE: sldo scale */
#define CTL0_SLDOVS(regval)                 (BITS(14,15) & ((uint32_t)(regval) << 14U))
#define PMU_SLDOVS_0                        CTL0_SLDOVS(0)                                                              /*!< SLDOVS scale 0.6V */
#define PMU_SLDOVS_1                        CTL0_SLDOVS(1)                                                              /*!< SLDOVS scale 0.7V */
#define PMU_SLDOVS_2                        CTL0_SLDOVS(2)                                                              /*!< SLDOVS scale 0.8V (default) */
#define PMU_SLDOVS_3                        CTL0_SLDOVS(3)                                                              /*!< SLDOVS scale 0.9V */

/* @STRUCT_MEMBER: back_up_write */
/* @DEFINE: ENABLE/DISABLE */
#define PMU_BACKUP_WRITE_DISABLE            ((uint32_t)0x00000000U)                                                     /*!< PMU back up write disable */
#define PMU_BACKUP_WRITE_ENABLE             PMU_CTL0_BKPWEN                                                             /*!< PMU back up write enable */

/* @STRUCT_MEMBER: lvd_enable */
/* @DEFINE: ENABLE/DISABLE */
#define PMU_LVD_DISABLE                     ((uint32_t)0x00000000U)                                                     /*!< low voltage detector disable */
#define PMU_LVD_ENABLE                      PMU_CTL0_LVDEN                                                              /*!< low voltage detector enable */

/* @STRUCT_MEMBER: vdda_vavd_enable */
/* @DEFINE: ENABLE/DISABLE */
#define PMU_VAVD_DISABLE                    ((uint32_t)0x00000000U)                                                     /*!< VDDA analog voltage detector disable */
#define PMU_VAVD_ENABLE                     PMU_CTL0_VAVDEN                                                             /*!< VDDA analog voltage detector enable */

/* @STRUCT_MEMBER: vovd_enable */
/* @DEFINE: ENABLE/DISABLE */
#define PMU_VOVD_DISABLE                    ((uint32_t)0x00000000U)                                                     /*!< peripheral voltage on V0.9V detector disable */
#define PMU_VOVD_ENABLE                     PMU_CTL0_VOVDEN                                                             /*!< peripheral voltage on V0.9V detector enable */

/* @STRUCT_MEMBER: vbat_charge_enable */
/* @DEFINE: ENABLE/DISABLE */
#define PMU_CHARGE_DISABLE                  ((uint32_t)0x00000000U)                                                     /*!< VBAT battery charging disable */
#define PMU_CHARGE_ENABLE                   PMU_CTL2_VCEN                                                               /*!< VBAT battery charging enable */

/* @STRUCT_MEMBER: vbat_temp_monitor_enable */
/* @DEFINE: ENABLE/DISABLE */
#define PMU_TEMP_MONITOR_DISABLE            ((uint32_t)0x00000000U)                                                     /*!< VBAT and temperature monitoring disable */
#define PMU_TEMP_MONITOR_ENABLE             PMU_CTL1_VBTMEN                                                             /*!< VBAT and temperature monitoring enable */

/* @STRUCT_MEMBER: backup_pvs */
/* @DEFINE: ENABLE/DISABLE */
#define PMU_BACKUP_PVS_DISABLE               ((uint32_t)0x00000000U)                                                    /*!< backup voltage stabilizer disable */
#define PMU_BACKUP_PVS_ENABLE                PMU_CTL1_BKPVSEN                                                           /*!< backup voltage stabilizer enable */

/* PMU analog voltage detector threshold definitions */
/* @STRUCT_MEMBER: vdda_vavd_level */
/* @DEFINE: vavd threshold */
#define CTL0_VAVDVC(regval)                 (BITS(17,18) & ((uint32_t)(regval) << 17U))
#define PMU_VAVDVC_0                        CTL0_VAVDVC(0)                                                              /*!< voltage threshold is 1.7V */
#define PMU_VAVDVC_1                        CTL0_VAVDVC(1)                                                              /*!< voltage threshold is 2.1V */
#define PMU_VAVDVC_2                        CTL0_VAVDVC(2)                                                              /*!< voltage threshold is 2.5V */
#define PMU_VAVDVC_3                        CTL0_VAVDVC(3)                                                              /*!< voltage threshold is 2.8V */

/* PMU VBAT battery charging resistor selection */
/* @STRUCT_MEMBER: vbat_charge_resistor */
/* @DEFINE: charging resistor */
#define PMU_VCRSEL_5K                       ((uint32_t)0x00000000U)                                                     /*!< 5 kOhms resistor is selected for charging VBAT battery */
#define PMU_VCRSEL_1P5K                     PMU_CTL2_VCRSEL                                                             /*!< 1.5 kOhms resistor is selected for charging VBAT battery */

/* PMU LDO output voltage select definitions */
/* @STRUCT_MEMBER: ldo_output_voltage */
/* @DEFINE: ldo output' */
#define CTL3_LDOVS(regval)                  (BITS(12,14) & ((uint32_t)(regval) << 12U))
#define PMU_LDOVS_0                         CTL3_LDOVS(0)                                                               /*!< LDO output voltage 0.8V mode */
#define PMU_LDOVS_1                         CTL3_LDOVS(1)                                                               /*!< LDO output voltage 0.85V mode */
#define PMU_LDOVS_2                         CTL3_LDOVS(2)                                                               /*!< LDO output voltage 0.9V mode */
#define PMU_LDOVS_3                         CTL3_LDOVS(3)                                                               /*!< LDO output voltage 0.95V mode */
#define PMU_LDOVS_4                         CTL3_LDOVS(4)                                                               /*!< LDO output voltage 0.975V mode */
#define PMU_LDOVS_5                         CTL3_LDOVS(5)                                                               /*!< LDO output voltage 1V mode */

/* PMU wakeup pin definitions */
/* @PARA: wakeup_pin */
/* @DEFINE: wakeup pin */
#define PMU_WAKEUP_PIN0                     PMU_CS_WUPEN0                                                               /*!< wakeup pin 0 */
#define PMU_WAKEUP_PIN1                     PMU_CS_WUPEN1                                                               /*!< wakeup pin 1 */
#define PMU_WAKEUP_PIN3                     PMU_CS_WUPEN3                                                               /*!< wakeup pin 3 */
#define PMU_WAKEUP_PIN5                     PMU_CS_WUPEN5                                                               /*!< wakeup pin 5 */

/* @STRUCT_MEMBER: ldo_enable */
/* @DEFINE: ldo enable */
#define PMU_LDO_DISABLE                     ((uint32_t)0x00000000U)                                                     /*!< pmu ldo disable */
#define PMU_LDO_ENABLE                      PMU_CTL2_LDOEN                                                              /*!< pmu ldo enable */

/* @STRUCT_MEMBER: vdd33usb_enable */
/* @DEFINE: vdd33usb enable */
#define PMU_VDD33USB_DISABLE                ((uint32_t)0x00000000U)                                                     /*!< VDD33USB disable */
#define PMU_VDD33USB_ENABLE                 PMU_CTL2_VUSB33DEN                                                          /*!< VDD33USB enable */

/* @STRUCT_MEMBER: usb_enable */
/* @DEFINE: usb enable */
#define PMU_USB_DISABLE                     ((uint32_t)0x00000000U)                                                     /*!< USB disable */
#define PMU_USB_ENABLE                      PMU_CTL2_USBSEN                                                             /*!< USB enable */

/* @STRUCT_MEMBER: dvs_enable */
/* @DEFINE: usb enable */
#define PMU_DVS_DISABLE                     ((uint32_t)0x00000000U)                                                     /*!< Down Voltage And Stabilize Voltage disable */
#define PMU_DVS_ENABLE                      PMU_CTL2_DVSEN                                                              /*!< Down Voltage And Stabilize Voltage enable */

/* @STRUCT_MEMBER: bypass */
/* @DEFINE: power management unit bypass */
#define PMU_BYPASS_NORMAL                   ((uint32_t)0x00000000U)                                                     /*!< Down Voltage And Stabilize Voltage disable */
#define PMU_BYPASS_ENABLE                   PMU_CTL2_BYPASS                                                             /*!< Down Voltage And Stabilize Voltage enable */

/* @STRUCT_MEMBER: pmu_cnt */
/* @=NULL */

/* @STRUCT_MEMBER: tsw_irccnt */
/* @=NULL */

/* PMU flag definitions */
#define PMU_REGIDX_BIT(regidx, bitpos)      (((uint32_t)(regidx) << 6) | (uint32_t)(bitpos))
#define PMU_REG_VAL(periph)                 (REG32(PMU + ((uint32_t)(periph) >> 6U)))
#define PMU_BIT_POS(val)                    ((uint32_t)(val) & 0x1FU)

#define PMU_CTL0_OFFSET                     (0x00000000U)                                                               /*!< PMU control register 0 offset */
#define PMU_CS_OFFSET                       (0x00000004U)                                                               /*!< PMU control and status register offset */
#define PMU_CTL1_OFFSET                     (0x00000008U)                                                               /*!< PMU control register 1 offset */
#define PMU_CTL2_OFFSET                     (0x00000010U)                                                               /*!< PMU control register 2 offset */
#define PMU_CTL3_OFFSET                     (0x00000014U)                                                               /*!< PMU control register 3 offset */
#define PMU_PAR_OFFSET                      (0x00000018U)                                                               /*!< PMU parameter register offset */

#define PMU_FLAG_WAKEUP                     PMU_REGIDX_BIT(PMU_CS_OFFSET,0)                                             /*!< wakeup flag */
#define PMU_FLAG_STANDBY                    PMU_REGIDX_BIT(PMU_CS_OFFSET,1)                                             /*!< standby flag */
#define PMU_FLAG_LVDF                       PMU_REGIDX_BIT(PMU_CS_OFFSET,2)                                             /*!< low voltage detector status flag */
#define PMU_FLAG_VAVDF                      PMU_REGIDX_BIT(PMU_CS_OFFSET,16)                                            /*!< VDDA analog voltage detector voltage output on VDDA flag */
#define PMU_FLAG_VOVDF                      PMU_REGIDX_BIT(PMU_CS_OFFSET,20)                                            /*!< peripheral voltage on VDDA detector flag */
#define PMU_FLAG_BKPVSRF                    PMU_REGIDX_BIT(PMU_CTL1_OFFSET,16)                                          /*!< backup voltage stabilizer ready flag */
#define PMU_FLAG_VBATLF                     PMU_REGIDX_BIT(PMU_CTL1_OFFSET,20)                                          /*!< VBAT level monitoring versus low threshold */
#define PMU_FLAG_VBATHF                     PMU_REGIDX_BIT(PMU_CTL1_OFFSET,21)                                          /*!< VBAT level monitoring versus high threshold */
#define PMU_FLAG_TEMPLF                     PMU_REGIDX_BIT(PMU_CTL1_OFFSET,22)                                          /*!< temperature level monitoring versus low threshold */
#define PMU_FLAG_TEMPHF                     PMU_REGIDX_BIT(PMU_CTL1_OFFSET,23)                                          /*!< temperature level monitoring versus high threshold */
#define PMU_FLAG_DVSRF                      PMU_REGIDX_BIT(PMU_CTL2_OFFSET,16)                                          /*!< step-down voltage stabilizer ready flag bit */
#define PMU_FLAG_USB33RF                    PMU_REGIDX_BIT(PMU_CTL2_OFFSET,26)                                          /*!< USB supply ready flag bit */
#define PMU_FLAG_PWRRF                      PMU_REGIDX_BIT(PMU_CTL3_OFFSET,16)                                          /*!< Power Ready flag bit */

/* PMU SMPS LDO supply mode definitions */
#define PMU_LDO_SUPPLY                      PMU_CTL2_LDOEN                                                              /*!< V0.9V domains are supplied from the LDO */
#define PMU_DIRECT_SMPS_SUPPLY              PMU_CTL2_DVSEN                                                              /*!< V0.9V domains are supplied from the SMPS only */
#define PMU_BYPASS                          PMU_CTL2_BYPASS                                                             /*!< The SMPS disabled and the LDO Bypass. The V0.9V domains are supplied from an external source */

/* PMU command constants definitions */
#define WFI_CMD                             ((uint8_t)0x00U)                                                            /*!< use WFI command */
#define WFE_CMD                             ((uint8_t)0x01U)                                                            /*!< use WFE command */

/* PMU structure type enum */
typedef enum {
    HAL_PMU_INIT_STRUCT = 0U,        /*!< PMU initialization structure */
    HAL_PMU_IRQ_STRUCT,              /*!< PMU irq handler structure */
    HAL_PMU_DEV_STRUCT,              /*!< PMU device structure */
    HAL_PMU_IRQ_USER_CALLBACK_STRUCT /*!< user callback initialization structure */
} hal_pmu_struct_type_enum;

/* PMU error type enum */
typedef enum {
    HAL_PMU_ERROR_NONE   = (uint32_t)0x00000000U,    /*!< no error */
    HAL_PMU_ERROR_SYSTEM = (uint32_t)0x00000001U,    /*!< PMU internal error: if problem of clocking, enable/disable, wrong state */
    HAL_PMU_ERROR_CONFIG = (uint32_t)0x00000002U     /*!< configuration error occurs */
} hal_pmu_error_enum;

/* PMU state enum */
typedef enum {
    HAL_PMU_STATE_NONE = 0U,    /*!< NONE(default value) */
    HAL_PMU_STATE_RESET,        /*!< RESET */
    HAL_PMU_STATE_BUSY,         /*!< BUSY */
    HAL_PMU_STATE_TIMEOUT,      /*!< TIMEOUT */
    HAL_PMU_STATE_ERROR,        /*!< ERROR */
    HAL_PMU_STATE_READY         /*!< READY */
} hal_pmu_state_enum;

/* PMU device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb pmu_lvd_handle;        /*!< LVD handler function */
    __IO hal_irq_handle_cb pmu_vavd_handle;       /*!< VAVD handler function */
    __IO hal_irq_handle_cb pmu_vovd_handle;       /*!< VOVD handler function */
    __IO hal_irq_handle_cb pmu_wakeup0_handle;    /*!< pmu wakeup 0 handler callback */
    __IO hal_irq_handle_cb pmu_wakeup1_handle;    /*!< pmu wakeup 1 handler callback */
    __IO hal_irq_handle_cb pmu_wakeup3_handle;    /*!< pmu wakeup 3 handler callback */
    __IO hal_irq_handle_cb pmu_wakeup5_handle;    /*!< pmu wakeup 5 handler callback */
} hal_pmu_irq_struct;

/* PMU device information structure */
typedef struct {
    hal_pmu_irq_struct pmu_irq;        /*!< PMU device interrupt callback function pointer structure */
    hal_pmu_error_enum error_state;    /*!< PMU error state */
    hal_pmu_state_enum state;          /*!< PMU state */
    hal_mutex_enum mutex;              /*!< lock */
    void *priv;                        /*!< priv data */
    void *pmu_lvd_callback;            /*!< lvd callback function */
    void *pmu_vovd_callback;           /*!< vovd callback function */
    void *pmu_vavd_callback;           /*!< vavd callback function */
    void *pmu_wakeup0_callback;        /*!< wakeup0 callback function */
    void *pmu_wakeup1_callback;        /*!< wakeup1 callback function */
    void *pmu_wakeup3_callback;        /*!< wakeup3 callback function */
    void *pmu_wakeup5_callback;        /*!< wakeup5 callback function */
} hal_pmu_dev_struct;

/* @PARA: pmu_struct_init */
/* @STRUCT: PMU basic config struct */
typedef struct {
    uint32_t deep_sleep_voltage;          /*!< deep sleep mode voltage selection */
    uint32_t back_up_write;               /*!< Backup domain write enable */
    uint32_t lvd_enable;                  /*!< low voltage detector enable */
    uint32_t lvd_threshold;               /*!< select low voltage detector threshold */
    uint32_t vdda_vavd_enable;            /*!< VDDA analog voltage detector enable */
    uint32_t vdda_vavd_level;             /*!< select VDDA analog voltage detector level */
    uint32_t vovd_enable;                 /*!< V0.9v peripheral voltage detector enable*/
    uint32_t vbat_charge_enable;          /*!< VBAT charge enable */
    uint32_t vbat_charge_resistor;        /*!< VBAT charge resistor selection */
    uint32_t vbat_temp_monitor_enable;    /*!< VBAT temperature monitor enable */
    uint32_t ldo_enable;                  /*!< LDO enable */
    uint32_t ldo_output_voltage;          /*!< LDO input voltage selection */
    uint32_t dvs_enable;                  /*!< Down  Voltage And Stabilize Voltage enable */
    uint32_t vdd33usb_enable;             /*!< VDD33USB voltage level detector enable */
    uint32_t usb_enable;                  /*!< USB voltage stabilizer enable */
    uint32_t pmu_cnt;                     /*!< exit deepsleep mode wait time count configure */
    uint32_t tsw_irccnt;                  /*!< When enter deepsleep, switch to LPIRC4M
                                               IRC64M (confirmed by DSPWUSSEL) clock. Wait the LPIRC4M
                                               IRC64M (confirmed by DSPWUSSEL) counter and then set stop_state signal.*/
    uint32_t bypass;                      /*!< power management unit bypass */
    uint32_t backup_pvs;                  /*!< backup voltage stabilizer enable */
} hal_pmu_init_struct;

typedef void (*hal_pmu_user_cb)(hal_pmu_dev_struct *pmu_dev);

/* PMU device interrupt callback function pointer structure */
typedef struct {
    __IO hal_pmu_user_cb pmu_lvd_callback;        /*!< LVD callback function */
    __IO hal_pmu_user_cb pmu_vavd_callback;       /*!< VAVD callback function */
    __IO hal_pmu_user_cb pmu_vovd_callback;       /*!< VOVD callback function */
    __IO hal_pmu_user_cb pmu_wakeup0_callback;    /*!< pmu wakeup 0 handler callback */
    __IO hal_pmu_user_cb pmu_wakeup1_callback;    /*!< pmu wakeup 1 handler callback */
    __IO hal_pmu_user_cb pmu_wakeup3_callback;    /*!< pmu wakeup 3 handler callback */
    __IO hal_pmu_user_cb pmu_wakeup5_callback;    /*!< pmu wakeup 5 handler callback */
} hal_pmu_irq_user_callback_struct;

/* function declarations */
/* @FUNCTION: init PMU peripheral */
int32_t hal_pmu_init(hal_pmu_dev_struct *pmu_dev, hal_pmu_init_struct *pmu_struct_init);
/* initialize the PMU structure with the default values */
int32_t hal_pmu_struct_init(hal_pmu_struct_type_enum hal_struct_type, void *p_struct);
/* reset PMU peripheral */
int32_t hal_pmu_deinit(hal_pmu_dev_struct *pmu_dev);
/* enable PMU wakeup pin */
int32_t hal_pmu_wakeup_pin_enable(hal_pmu_dev_struct *pmu_dev, uint32_t wakeup_pin);
/* @END */

/* LVD/VAVD config */
/* start LVD detector */
int32_t hal_pmu_lvd_start(hal_pmu_dev_struct *pmu_dev);
/* stop LVD detector */
int32_t hal_pmu_lvd_stop(hal_pmu_dev_struct *pmu_dev);
/* start VAVD detector */
int32_t hal_pmu_vavd_start(hal_pmu_dev_struct *pmu_dev);
/* stop VAVD detector */
int32_t hal_pmu_vavd_stop(hal_pmu_dev_struct *pmu_dev);

/* get the pmu supply configuration value */
uint32_t hal_pmu_supply_config_get(void);
/* get the pmu deepsleep mode voltage scaling selection value */
uint32_t hal_pmu_sldo_output_get(void);

/* backup domain and wakeup control functions */
/* enable backup domain write */
void hal_pmu_backup_write_enable(void);
/* disable backup domain write */
void hal_pmu_backup_write_disable(void);
/* disable PMU wakeup pin */
int32_t hal_pmu_wakeup_pin_disable(uint32_t wakeup_pin);

/* set PMU mode */
/* enter sleep mode */
int32_t hal_pmu_to_sleepmode(uint8_t sleepmodecmd);
/* enter deepsleep mode */
int32_t hal_pmu_to_deepsleepmode(uint8_t deepsleepmodecmd);
/* set sleep on exit mode */
void hal_pmu_set_sleep_on_exit(void);
/* reset sleep on exit mode */
void hal_pmu_reset_sleep_on_exit(void);

/* enable send event on pending */
void hal_pmu_enable_sevonpend(void);
/* disable send event on pending */
void hal_pmu_disable_sevonpend(void);

/* PMU power supply and voltage control functions */
/* get pmu power supply config */
uint32_t hal_pmu_get_power_supply_config(void);
/* enter standby mode */
void hal_pmu_to_standbymode(void);
/* power supply configurations */
int32_t hal_pmu_smps_ldo_supply_config(uint32_t smpsmode);
/* PMU get sldo voltage value */
uint32_t hal_pmu_sldo_voltage_get(void);

/* backup voltage function control */
/* enable backup voltage stabilizer */
int32_t hal_pmu_backup_voltage_stabilizer_enable(void);
/* disable backup voltage stabilizer */
void hal_pmu_backup_voltage_stabilizer_disable(void);

/* USB regulator */
/* enable USB regulator */
void hal_pmu_usb_regulator_enable(void);
/* disable USB regulator */
void hal_pmu_usb_regulator_disable(void);
/* enable VDD33USB voltage level detector */
void hal_pmu_usb_voltage_detector_enable(void);
/* disable VDD33USB voltage level detector */
void hal_pmu_usb_voltage_detector_disable(void);

/* VBAT battery charging and temperature monitoring functions */
/* PMU VBAT battery charging resistor selection */
int32_t hal_pmu_vbat_charging_select(uint32_t resistor);
/* enable VBAT battery charging */
void hal_pmu_vbat_charging_enable(void);
/* disable VBAT battery charging */
void hal_pmu_vbat_charging_disable(void);
/* enable VBAT and temperature monitoring */
void hal_pmu_vbat_temp_monitor_enable(void);
/* disable VBAT and temperature monitoring */
void hal_pmu_vbat_temp_monitor_disable(void);

/* PMU interrupt handling functions */
/* PMU interrupt handler content function, which will be called in PMU interrupt handler */
int32_t hal_pmu_irq(hal_pmu_dev_struct *pmu_dev);
/* set user-defined interrupt callback function,
   which will be registered and called when corresponding interrupt be triggered */
int32_t hal_pmu_irq_handle_set(hal_pmu_dev_struct *pmu_dev, hal_pmu_irq_struct *irq_handle);
/* reset all user-defined interrupt callback function,
   which will be registered and called when corresponding interrupt be triggered */
int32_t hal_pmu_irq_handle_all_reset(hal_pmu_dev_struct *pmu_dev);
/* start PMU with interrupt mode,
   which will enable the corresponding PMU interrupt */
int32_t hal_pmu_start_interrupt(hal_pmu_dev_struct *pmu_dev, hal_pmu_irq_struct *p_irq, \
                                hal_pmu_irq_user_callback_struct *p_user_func);
/* stop PMU with interrupt mode,
   which will disable the corresponding PMU interrupt */
int32_t hal_pmu_stop_interrupt(hal_pmu_dev_struct *pmu_dev);

/* LVD/VAVD/CVD function control */
/* select low voltage detector threshold */
int32_t hal_pmu_lvd_select(uint32_t lvdt_n);
/* enable PMU lvd */
void hal_pmu_lvd_enable(void);
/* disable PMU lvd */
void hal_pmu_lvd_disable(void);
/* select analog voltage detector threshold */
int32_t hal_pmu_vavd_select(uint32_t vavd_n);
/* enable PMU analog voltage detector */
void hal_pmu_vavd_enable(void);
/* disable PMU analog voltage detector */
void hal_pmu_vavd_disable(void);
/* enable PMU core voltage detector */
void hal_pmu_cvd_enable(void);
/* disable PMU V0.9V core voltage detector */
void hal_pmu_cvd_disable(void);

/* control the V0.9V core voltage level */
int32_t hal_pmu_ldo_output_select(uint32_t ldo_n);
/* deepsleep mode V0.9V core voltage select */
int32_t hal_pmu_sldo_output_select(uint32_t sldo_n);

/* PMU flag management functions */
/* get the pmu temperature flag */
uint32_t hal_pmu_temperature_flag_get(void);
/* get the pmu vbat flag */
uint32_t hal_pmu_vbat_flag_get(void);
/* get the pmu WUF flag */
FlagStatus hal_pmu_wuf_get(void);
/* clear pending event */
void hal_pmu_pending_event_clear(void);
/* clear the pmu WUF flag */
void hal_pmu_wuf_clear(void);

/* get the pmu state */
hal_pmu_state_enum hal_pmu_state_get(hal_pmu_dev_struct *pmu_dev);
/* get the pmu error state */
hal_pmu_error_enum hal_pmu_error_state_get(hal_pmu_dev_struct *pmu_dev);

/* configure IRC counter before enter deepsleep mode */
int32_t hals_pmu_enter_deepsleep_wait_time_config(uint32_t wait_time);
/* configure IRC counter before exit deepsleep mode */
int32_t hals_pmu_exit_deepsleep_wait_time_config(uint32_t wait_time);

/* flag functions */
/* get flag state */
FlagStatus hals_pmu_flag_get(uint32_t flag);
/* clear flag bit */
void hals_pmu_flag_clear(uint32_t flag_reset);

#endif /* GD32H7XX_HAL_PMU_H */
