/*!
    \file    gd32h7xx_hal_gpio.h
    \brief   definitions for the GPIO

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_GPIO_H
#define GD32H7XX_HAL_GPIO_H

#include "gd32h7xx_hal.h"

/* GPIO definition */
/* @PARA: gpio_periph */
/* @DEFINE: GPIOx(x=A,B,C,D,F,G,H,J,K) definitions */
#define GPIOA                      (GPIO_BASE + 0x00000000U)        /*!< GPIOA base address */
#define GPIOB                      (GPIO_BASE + 0x00000400U)        /*!< GPIOB base address */
#define GPIOC                      (GPIO_BASE + 0x00000800U)        /*!< GPIOC base address */
#define GPIOD                      (GPIO_BASE + 0x00000C00U)        /*!< GPIOD base address */
#define GPIOE                      (GPIO_BASE + 0x00001000U)        /*!< GPIOE base address */
#define GPIOF                      (GPIO_BASE + 0x00001400U)        /*!< GPIOF base address */
#define GPIOG                      (GPIO_BASE + 0x00001800U)        /*!< GPIOG base address */
#define GPIOH                      (GPIO_BASE + 0x00001C00U)        /*!< GPIOH base address */
#define GPIOJ                      (GPIO_BASE + 0x00002400U)        /*!< GPIOJ base address */
#define GPIOK                      (GPIO_BASE + 0x00002800U)        /*!< GPIOK base address */

/* registers definitions */
#define GPIO_CTL(gpiox)            REG32((gpiox) + 0x00000000U)     /*!< GPIO port control register */
#define GPIO_OMODE(gpiox)          REG32((gpiox) + 0x00000004U)     /*!< GPIO port output mode register */
#define GPIO_OSPD(gpiox)           REG32((gpiox) + 0x00000008U)     /*!< GPIO port output speed register */
#define GPIO_PUD(gpiox)            REG32((gpiox) + 0x0000000CU)     /*!< GPIO port pull-up/pull-down register */
#define GPIO_ISTAT(gpiox)          REG32((gpiox) + 0x00000010U)     /*!< GPIO port input status register */
#define GPIO_OCTL(gpiox)           REG32((gpiox) + 0x00000014U)     /*!< GPIO port output control register */
#define GPIO_BOP(gpiox)            REG32((gpiox) + 0x00000018U)     /*!< GPIO port bit operation register */
#define GPIO_LOCK(gpiox)           REG32((gpiox) + 0x0000001CU)     /*!< GPIO port configuration lock register */
#define GPIO_AFSEL0(gpiox)         REG32((gpiox) + 0x00000020U)     /*!< GPIO alternate function selected register 0 */
#define GPIO_AFSEL1(gpiox)         REG32((gpiox) + 0x00000024U)     /*!< GPIO alternate function selected register 1 */
#define GPIO_BC(gpiox)             REG32((gpiox) + 0x00000028U)     /*!< GPIO bit clear register */
#define GPIO_TG(gpiox)             REG32((gpiox) + 0x0000002CU)     /*!< GPIO port bit toggle register */
#define GPIO_IFL(gpiox)            REG32((gpiox) + 0x00000030U)     /*!< GPIO port Input Filter register */
#define GPIO_IFTP(gpiox)           REG32((gpiox) + 0x00000034U)     /*!< GPIO port Input Filter Type register */

/* bits definitions */
/* GPIO_CTL */
#define GPIO_CTL_CTL0              BITS(0,1)                        /*!< pin 0 configuration bits */
#define GPIO_CTL_CTL1              BITS(2,3)                        /*!< pin 1 configuration bits */
#define GPIO_CTL_CTL2              BITS(4,5)                        /*!< pin 2 configuration bits */
#define GPIO_CTL_CTL3              BITS(6,7)                        /*!< pin 3 configuration bits */
#define GPIO_CTL_CTL4              BITS(8,9)                        /*!< pin 4 configuration bits */
#define GPIO_CTL_CTL5              BITS(10,11)                      /*!< pin 5 configuration bits */
#define GPIO_CTL_CTL6              BITS(12,13)                      /*!< pin 6 configuration bits */
#define GPIO_CTL_CTL7              BITS(14,15)                      /*!< pin 7 configuration bits */
#define GPIO_CTL_CTL8              BITS(16,17)                      /*!< pin 8 configuration bits */
#define GPIO_CTL_CTL9              BITS(18,19)                      /*!< pin 9 configuration bits */
#define GPIO_CTL_CTL10             BITS(20,21)                      /*!< pin 10 configuration bits */
#define GPIO_CTL_CTL11             BITS(22,23)                      /*!< pin 11 configuration bits */
#define GPIO_CTL_CTL12             BITS(24,25)                      /*!< pin 12 configuration bits */
#define GPIO_CTL_CTL13             BITS(26,27)                      /*!< pin 13 configuration bits */
#define GPIO_CTL_CTL14             BITS(28,29)                      /*!< pin 14 configuration bits */
#define GPIO_CTL_CTL15             BITS(30,31)                      /*!< pin 15 configuration bits */

/* GPIO_OMODE */
#define GPIO_OMODE_OM0             BIT(0)                           /*!< pin 0 output mode bit */
#define GPIO_OMODE_OM1             BIT(1)                           /*!< pin 1 output mode bit */
#define GPIO_OMODE_OM2             BIT(2)                           /*!< pin 2 output mode bit */
#define GPIO_OMODE_OM3             BIT(3)                           /*!< pin 3 output mode bit */
#define GPIO_OMODE_OM4             BIT(4)                           /*!< pin 4 output mode bit */
#define GPIO_OMODE_OM5             BIT(5)                           /*!< pin 5 output mode bit */
#define GPIO_OMODE_OM6             BIT(6)                           /*!< pin 6 output mode bit */
#define GPIO_OMODE_OM7             BIT(7)                           /*!< pin 7 output mode bit */
#define GPIO_OMODE_OM8             BIT(8)                           /*!< pin 8 output mode bit */
#define GPIO_OMODE_OM9             BIT(9)                           /*!< pin 9 output mode bit */
#define GPIO_OMODE_OM10            BIT(10)                          /*!< pin 10 output mode bit */
#define GPIO_OMODE_OM11            BIT(11)                          /*!< pin 11 output mode bit */
#define GPIO_OMODE_OM12            BIT(12)                          /*!< pin 12 output mode bit */
#define GPIO_OMODE_OM13            BIT(13)                          /*!< pin 13 output mode bit */
#define GPIO_OMODE_OM14            BIT(14)                          /*!< pin 14 output mode bit */
#define GPIO_OMODE_OM15            BIT(15)                          /*!< pin 15 output mode bit */

/* GPIO_OSPD */
#define GPIO_OSPD_OSPD0            BITS(0,1)                        /*!< pin 0 output max speed bits */
#define GPIO_OSPD_OSPD1            BITS(2,3)                        /*!< pin 1 output max speed bits */
#define GPIO_OSPD_OSPD2            BITS(4,5)                        /*!< pin 2 output max speed bits */
#define GPIO_OSPD_OSPD3            BITS(6,7)                        /*!< pin 3 output max speed bits */
#define GPIO_OSPD_OSPD4            BITS(8,9)                        /*!< pin 4 output max speed bits */
#define GPIO_OSPD_OSPD5            BITS(10,11)                      /*!< pin 5 output max speed bits */
#define GPIO_OSPD_OSPD6            BITS(12,13)                      /*!< pin 6 output max speed bits */
#define GPIO_OSPD_OSPD7            BITS(14,15)                      /*!< pin 7 output max speed bits */
#define GPIO_OSPD_OSPD8            BITS(16,17)                      /*!< pin 8 output max speed bits */
#define GPIO_OSPD_OSPD9            BITS(18,19)                      /*!< pin 9 output max speed bits */
#define GPIO_OSPD_OSPD10           BITS(20,21)                      /*!< pin 10 output max speed bits */
#define GPIO_OSPD_OSPD11           BITS(22,23)                      /*!< pin 11 output max speed bits */
#define GPIO_OSPD_OSPD12           BITS(24,25)                      /*!< pin 12 output max speed bits */
#define GPIO_OSPD_OSPD13           BITS(26,27)                      /*!< pin 13 output max speed bits */
#define GPIO_OSPD_OSPD14           BITS(28,29)                      /*!< pin 14 output max speed bits */
#define GPIO_OSPD_OSPD15           BITS(30,31)                      /*!< pin 15 output max speed bits */

/* GPIO_PUD */
#define GPIO_PUD_PUD0              BITS(0,1)                        /*!< pin 0 pull-up or pull-down bits */
#define GPIO_PUD_PUD1              BITS(2,3)                        /*!< pin 1 pull-up or pull-down bits */
#define GPIO_PUD_PUD2              BITS(4,5)                        /*!< pin 2 pull-up or pull-down bits */
#define GPIO_PUD_PUD3              BITS(6,7)                        /*!< pin 3 pull-up or pull-down bits */
#define GPIO_PUD_PUD4              BITS(8,9)                        /*!< pin 4 pull-up or pull-down bits */
#define GPIO_PUD_PUD5              BITS(10,11)                      /*!< pin 5 pull-up or pull-down bits */
#define GPIO_PUD_PUD6              BITS(12,13)                      /*!< pin 6 pull-up or pull-down bits */
#define GPIO_PUD_PUD7              BITS(14,15)                      /*!< pin 7 pull-up or pull-down bits */
#define GPIO_PUD_PUD8              BITS(16,17)                      /*!< pin 8 pull-up or pull-down bits */
#define GPIO_PUD_PUD9              BITS(18,19)                      /*!< pin 9 pull-up or pull-down bits */
#define GPIO_PUD_PUD10             BITS(20,21)                      /*!< pin 10 pull-up or pull-down bits */
#define GPIO_PUD_PUD11             BITS(22,23)                      /*!< pin 11 pull-up or pull-down bits */
#define GPIO_PUD_PUD12             BITS(24,25)                      /*!< pin 12 pull-up or pull-down bits */
#define GPIO_PUD_PUD13             BITS(26,27)                      /*!< pin 13 pull-up or pull-down bits */
#define GPIO_PUD_PUD14             BITS(28,29)                      /*!< pin 14 pull-up or pull-down bits */
#define GPIO_PUD_PUD15             BITS(30,31)                      /*!< pin 15 pull-up or pull-down bits */

/* GPIO_ISTAT */
#define GPIO_ISTAT_ISTAT0          BIT(0)                           /*!< pin 0 input status */
#define GPIO_ISTAT_ISTAT1          BIT(1)                           /*!< pin 1 input status */
#define GPIO_ISTAT_ISTAT2          BIT(2)                           /*!< pin 2 input status */
#define GPIO_ISTAT_ISTAT3          BIT(3)                           /*!< pin 3 input status */
#define GPIO_ISTAT_ISTAT4          BIT(4)                           /*!< pin 4 input status */
#define GPIO_ISTAT_ISTAT5          BIT(5)                           /*!< pin 5 input status */
#define GPIO_ISTAT_ISTAT6          BIT(6)                           /*!< pin 6 input status */
#define GPIO_ISTAT_ISTAT7          BIT(7)                           /*!< pin 7 input status */
#define GPIO_ISTAT_ISTAT8          BIT(8)                           /*!< pin 8 input status */
#define GPIO_ISTAT_ISTAT9          BIT(9)                           /*!< pin 9 input status */
#define GPIO_ISTAT_ISTAT10         BIT(10)                          /*!< pin 10 input status */
#define GPIO_ISTAT_ISTAT11         BIT(11)                          /*!< pin 11 input status */
#define GPIO_ISTAT_ISTAT12         BIT(12)                          /*!< pin 12 input status */
#define GPIO_ISTAT_ISTAT13         BIT(13)                          /*!< pin 13 input status */
#define GPIO_ISTAT_ISTAT14         BIT(14)                          /*!< pin 14 input status */
#define GPIO_ISTAT_ISTAT15         BIT(15)                          /*!< pin 15 input status */

/* GPIO_OCTL */
#define GPIO_OCTL_OCTL0            BIT(0)                           /*!< pin 0 output bit */
#define GPIO_OCTL_OCTL1            BIT(1)                           /*!< pin 1 output bit */
#define GPIO_OCTL_OCTL2            BIT(2)                           /*!< pin 2 output bit */
#define GPIO_OCTL_OCTL3            BIT(3)                           /*!< pin 3 output bit */
#define GPIO_OCTL_OCTL4            BIT(4)                           /*!< pin 4 output bit */
#define GPIO_OCTL_OCTL5            BIT(5)                           /*!< pin 5 output bit */
#define GPIO_OCTL_OCTL6            BIT(6)                           /*!< pin 6 output bit */
#define GPIO_OCTL_OCTL7            BIT(7)                           /*!< pin 7 output bit */
#define GPIO_OCTL_OCTL8            BIT(8)                           /*!< pin 8 output bit */
#define GPIO_OCTL_OCTL9            BIT(9)                           /*!< pin 9 output bit */
#define GPIO_OCTL_OCTL10           BIT(10)                          /*!< pin 10 output bit */
#define GPIO_OCTL_OCTL11           BIT(11)                          /*!< pin 11 output bit */
#define GPIO_OCTL_OCTL12           BIT(12)                          /*!< pin 12 output bit */
#define GPIO_OCTL_OCTL13           BIT(13)                          /*!< pin 13 output bit */
#define GPIO_OCTL_OCTL14           BIT(14)                          /*!< pin 14 output bit */
#define GPIO_OCTL_OCTL15           BIT(15)                          /*!< pin 15 output bit */

/* GPIO_BOP */
#define GPIO_BOP_BOP0              BIT(0)                           /*!< pin 0 set bit */
#define GPIO_BOP_BOP1              BIT(1)                           /*!< pin 1 set bit */
#define GPIO_BOP_BOP2              BIT(2)                           /*!< pin 2 set bit */
#define GPIO_BOP_BOP3              BIT(3)                           /*!< pin 3 set bit */
#define GPIO_BOP_BOP4              BIT(4)                           /*!< pin 4 set bit */
#define GPIO_BOP_BOP5              BIT(5)                           /*!< pin 5 set bit */
#define GPIO_BOP_BOP6              BIT(6)                           /*!< pin 6 set bit */
#define GPIO_BOP_BOP7              BIT(7)                           /*!< pin 7 set bit */
#define GPIO_BOP_BOP8              BIT(8)                           /*!< pin 8 set bit */
#define GPIO_BOP_BOP9              BIT(9)                           /*!< pin 9 set bit */
#define GPIO_BOP_BOP10             BIT(10)                          /*!< pin 10 set bit */
#define GPIO_BOP_BOP11             BIT(11)                          /*!< pin 11 set bit */
#define GPIO_BOP_BOP12             BIT(12)                          /*!< pin 12 set bit */
#define GPIO_BOP_BOP13             BIT(13)                          /*!< pin 13 set bit */
#define GPIO_BOP_BOP14             BIT(14)                          /*!< pin 14 set bit */
#define GPIO_BOP_BOP15             BIT(15)                          /*!< pin 15 set bit */
#define GPIO_BOP_CR0               BIT(16)                          /*!< pin 0 clear bit */
#define GPIO_BOP_CR1               BIT(17)                          /*!< pin 1 clear bit */
#define GPIO_BOP_CR2               BIT(18)                          /*!< pin 2 clear bit */
#define GPIO_BOP_CR3               BIT(19)                          /*!< pin 3 clear bit */
#define GPIO_BOP_CR4               BIT(20)                          /*!< pin 4 clear bit */
#define GPIO_BOP_CR5               BIT(21)                          /*!< pin 5 clear bit */
#define GPIO_BOP_CR6               BIT(22)                          /*!< pin 6 clear bit */
#define GPIO_BOP_CR7               BIT(23)                          /*!< pin 7 clear bit */
#define GPIO_BOP_CR8               BIT(24)                          /*!< pin 8 clear bit */
#define GPIO_BOP_CR9               BIT(25)                          /*!< pin 9 clear bit */
#define GPIO_BOP_CR10              BIT(26)                          /*!< pin 10 clear bit */
#define GPIO_BOP_CR11              BIT(27)                          /*!< pin 11 clear bit */
#define GPIO_BOP_CR12              BIT(28)                          /*!< pin 12 clear bit */
#define GPIO_BOP_CR13              BIT(29)                          /*!< pin 13 clear bit */
#define GPIO_BOP_CR14              BIT(30)                          /*!< pin 14 clear bit */
#define GPIO_BOP_CR15              BIT(31)                          /*!< pin 15 clear bit */

/* GPIO_LOCK */
#define GPIO_LOCK_LK0              BIT(0)                           /*!< pin 0 lock bit */
#define GPIO_LOCK_LK1              BIT(1)                           /*!< pin 1 lock bit */
#define GPIO_LOCK_LK2              BIT(2)                           /*!< pin 2 lock bit */
#define GPIO_LOCK_LK3              BIT(3)                           /*!< pin 3 lock bit */
#define GPIO_LOCK_LK4              BIT(4)                           /*!< pin 4 lock bit */
#define GPIO_LOCK_LK5              BIT(5)                           /*!< pin 5 lock bit */
#define GPIO_LOCK_LK6              BIT(6)                           /*!< pin 6 lock bit */
#define GPIO_LOCK_LK7              BIT(7)                           /*!< pin 7 lock bit */
#define GPIO_LOCK_LK8              BIT(8)                           /*!< pin 8 lock bit */
#define GPIO_LOCK_LK9              BIT(9)                           /*!< pin 9 lock bit */
#define GPIO_LOCK_LK10             BIT(10)                          /*!< pin 10 lock bit */
#define GPIO_LOCK_LK11             BIT(11)                          /*!< pin 11 lock bit */
#define GPIO_LOCK_LK12             BIT(12)                          /*!< pin 12 lock bit */
#define GPIO_LOCK_LK13             BIT(13)                          /*!< pin 13 lock bit */
#define GPIO_LOCK_LK14             BIT(14)                          /*!< pin 14 lock bit */
#define GPIO_LOCK_LK15             BIT(15)                          /*!< pin 15 lock bit */
#define GPIO_LOCK_LKK              BIT(16)                          /*!< pin sequence lock key */

/* GPIO_AFSEL0 */
#define GPIO_AFSEL0_SEL0           BITS(0,3)                        /*!< pin 0 alternate function selected */
#define GPIO_AFSEL0_SEL1           BITS(4,7)                        /*!< pin 1 alternate function selected */
#define GPIO_AFSEL0_SEL2           BITS(8,11)                       /*!< pin 2 alternate function selected */
#define GPIO_AFSEL0_SEL3           BITS(12,15)                      /*!< pin 3 alternate function selected */
#define GPIO_AFSEL0_SEL4           BITS(16,19)                      /*!< pin 4 alternate function selected */
#define GPIO_AFSEL0_SEL5           BITS(20,23)                      /*!< pin 5 alternate function selected */
#define GPIO_AFSEL0_SEL6           BITS(24,27)                      /*!< pin 6 alternate function selected */
#define GPIO_AFSEL0_SEL7           BITS(28,31)                      /*!< pin 7 alternate function selected */

/* GPIO_AFSEL1 */
#define GPIO_AFSEL1_SEL8           BITS(0,3)                        /*!< pin 8 alternate function selected */
#define GPIO_AFSEL1_SEL9           BITS(4,7)                        /*!< pin 9 alternate function selected */
#define GPIO_AFSEL1_SEL10          BITS(8,11)                       /*!< pin 10 alternate function selected */
#define GPIO_AFSEL1_SEL11          BITS(12,15)                      /*!< pin 11 alternate function selected */
#define GPIO_AFSEL1_SEL12          BITS(16,19)                      /*!< pin 12 alternate function selected */
#define GPIO_AFSEL1_SEL13          BITS(20,23)                      /*!< pin 13 alternate function selected */
#define GPIO_AFSEL1_SEL14          BITS(24,27)                      /*!< pin 14 alternate function selected */
#define GPIO_AFSEL1_SEL15          BITS(28,31)                      /*!< pin 15 alternate function selected */

/* GPIO_BC */
#define GPIO_BC_CR0                BIT(0)                           /*!< pin 0 clear bit */
#define GPIO_BC_CR1                BIT(1)                           /*!< pin 1 clear bit */
#define GPIO_BC_CR2                BIT(2)                           /*!< pin 2 clear bit */
#define GPIO_BC_CR3                BIT(3)                           /*!< pin 3 clear bit */
#define GPIO_BC_CR4                BIT(4)                           /*!< pin 4 clear bit */
#define GPIO_BC_CR5                BIT(5)                           /*!< pin 5 clear bit */
#define GPIO_BC_CR6                BIT(6)                           /*!< pin 6 clear bit */
#define GPIO_BC_CR7                BIT(7)                           /*!< pin 7 clear bit */
#define GPIO_BC_CR8                BIT(8)                           /*!< pin 8 clear bit */
#define GPIO_BC_CR9                BIT(9)                           /*!< pin 9 clear bit */
#define GPIO_BC_CR10               BIT(10)                          /*!< pin 10 clear bit */
#define GPIO_BC_CR11               BIT(11)                          /*!< pin 11 clear bit */
#define GPIO_BC_CR12               BIT(12)                          /*!< pin 12 clear bit */
#define GPIO_BC_CR13               BIT(13)                          /*!< pin 13 clear bit */
#define GPIO_BC_CR14               BIT(14)                          /*!< pin 14 clear bit */
#define GPIO_BC_CR15               BIT(15)                          /*!< pin 15 clear bit */

/* GPIO_TG */
#define GPIO_TG_TG0                BIT(0)                           /*!< pin 0 toggle bit */
#define GPIO_TG_TG1                BIT(1)                           /*!< pin 1 toggle bit */
#define GPIO_TG_TG2                BIT(2)                           /*!< pin 2 toggle bit */
#define GPIO_TG_TG3                BIT(3)                           /*!< pin 3 toggle bit */
#define GPIO_TG_TG4                BIT(4)                           /*!< pin 4 toggle bit */
#define GPIO_TG_TG5                BIT(5)                           /*!< pin 5 toggle bit */
#define GPIO_TG_TG6                BIT(6)                           /*!< pin 6 toggle bit */
#define GPIO_TG_TG7                BIT(7)                           /*!< pin 7 toggle bit */
#define GPIO_TG_TG8                BIT(8)                           /*!< pin 8 toggle bit */
#define GPIO_TG_TG9                BIT(9)                           /*!< pin 9 toggle bit */
#define GPIO_TG_TG10               BIT(10)                          /*!< pin 10 toggle bit */
#define GPIO_TG_TG11               BIT(11)                          /*!< pin 11 toggle bit */
#define GPIO_TG_TG12               BIT(12)                          /*!< pin 12 toggle bit */
#define GPIO_TG_TG13               BIT(13)                          /*!< pin 13 toggle bit */
#define GPIO_TG_TG14               BIT(14)                          /*!< pin 14 toggle bit */
#define GPIO_TG_TG15               BIT(15)                          /*!< pin 15 toggle bit */

/* GPIO_IFL */
#define GPIO_IFL_FLPRD0            BITS(0,7)                        /*!< GPIO0 - GPIO7 Filter sampling frequency */
#define GPIO_IFL_FLPRD1            BITS(8,15)                       /*!< GPIO8 - GPIO15 Filter sampling frequency */

/* GPIO_IFTP */
#define GPIO_IFTP_IFTP0            BITS(0,1)                        /*!< pin 0 Filter Input Type */
#define GPIO_IFTP_IFTP1            BITS(2,3)                        /*!< pin 1 Filter Input Type */
#define GPIO_IFTP_IFTP2            BITS(4,5)                        /*!< pin 2 Filter Input Type */
#define GPIO_IFTP_IFTP3            BITS(6,7)                        /*!< pin 3 Filter Input Type */
#define GPIO_IFTP_IFTP4            BITS(8,9)                        /*!< pin 4 Filter Input Type */
#define GPIO_IFTP_IFTP5            BITS(10,11)                      /*!< pin 5 Filter Input Type */
#define GPIO_IFTP_IFTP6            BITS(12,13)                      /*!< pin 6 Filter Input Type */
#define GPIO_IFTP_IFTP7            BITS(14,15)                      /*!< pin 7 Filter Input Type */
#define GPIO_IFTP_IFTP8            BITS(16,17)                      /*!< pin 8 Filter Input Type */
#define GPIO_IFTP_IFTP9            BITS(18,19)                      /*!< pin 9 Filter Input Type */
#define GPIO_IFTP_IFTP10           BITS(20,21)                      /*!< pin 10 Filter Input Type */
#define GPIO_IFTP_IFTP11           BITS(22,23)                      /*!< pin 11 Filter Input Type */
#define GPIO_IFTP_IFTP12           BITS(24,25)                      /*!< pin 12 Filter Input Type */
#define GPIO_IFTP_IFTP13           BITS(26,27)                      /*!< pin 13 Filter Input Type */
#define GPIO_IFTP_IFTP14           BITS(28,29)                      /*!< pin 14 Filter Input Type */
#define GPIO_IFTP_IFTP15           BITS(30,31)                      /*!< pin 15 Filter Input Type */

/* constants definitions */
typedef FlagStatus bit_status;

/* @PARA: pin */
/* @DEFINE: pinx(x=0,1,2...15) definitions */
#define GPIO_PIN_0                 BIT(0)                           /*!< GPIO pin 0 */
#define GPIO_PIN_1                 BIT(1)                           /*!< GPIO pin 1 */
#define GPIO_PIN_2                 BIT(2)                           /*!< GPIO pin 2 */
#define GPIO_PIN_3                 BIT(3)                           /*!< GPIO pin 3 */
#define GPIO_PIN_4                 BIT(4)                           /*!< GPIO pin 4 */
#define GPIO_PIN_5                 BIT(5)                           /*!< GPIO pin 5 */
#define GPIO_PIN_6                 BIT(6)                           /*!< GPIO pin 6 */
#define GPIO_PIN_7                 BIT(7)                           /*!< GPIO pin 7 */
#define GPIO_PIN_8                 BIT(8)                           /*!< GPIO pin 8 */
#define GPIO_PIN_9                 BIT(9)                           /*!< GPIO pin 9 */
#define GPIO_PIN_10                BIT(10)                          /*!< GPIO pin 10 */
#define GPIO_PIN_11                BIT(11)                          /*!< GPIO pin 11 */
#define GPIO_PIN_12                BIT(12)                          /*!< GPIO pin 12 */
#define GPIO_PIN_13                BIT(13)                          /*!< GPIO pin 13 */
#define GPIO_PIN_14                BIT(14)                          /*!< GPIO pin 14 */
#define GPIO_PIN_15                BIT(15)                          /*!< GPIO pin 15 */
#define GPIO_PIN_ALL               BITS(0,15)                       /*!< GPIO pin all */

/* output mode definitions */
#define CTL_CLTR(regval)           (BITS(0,1) & ((uint32_t)(regval) << 0))
#define PUD_PUPD(regval)           (BITS(0,1) & ((uint32_t)(regval) << 0))
#define OSPD_OSPD0(regval)         (BITS(0,1) & ((uint32_t)(regval) << 0))
#define IFL_FLPRD0(regval)         (BITS(0,7) & ((uint32_t)(regval) << 0))
#define IFTYPE_IFTP0(regval)       (BITS(0,1) & ((uint32_t)(regval) << 0))
#define AF(regval)                 (BITS(0,3) & ((uint32_t)(regval) << 0))

/* output mode definitions */
#define CTL_CLTR(regval)           (BITS(0,1) & ((uint32_t)(regval) << 0))
#define GPIO_MODE_OUTPUT           CTL_CLTR(1)                      /*!< output mode */
#define GPIO_MODE_AF               CTL_CLTR(2)                      /*!< alternate function mode */

/* GPIO mode configuration values */
#define GPIO_MODE_SET(n, mode)     ((uint32_t)((uint32_t)(mode) << (2U * (n))))
#define GPIO_MODE_MASK(n)          ((uint32_t)((uint32_t)0x00000003U << (2U * (n))))

/* GPIO pull-up/pull-down values */
#define GPIO_PUPD_SET(n, pupd)     ((uint32_t)((uint32_t)(pupd) << (2U * (n))))
#define GPIO_PUPD_MASK(n)          ((uint32_t)((uint32_t)0x00000003U << (2U * (n))))

/* GPIO output speed values */
#define GPIO_OSPEED_SET(n, speed)  ((uint32_t)((uint32_t)(speed) << (2U * (n))))
#define GPIO_OSPEED_MASK(n)        ((uint32_t)((uint32_t)0x00000003U << (2U * (n))))

/* GPIO output type */
#define GPIO_OTYPE_PP              ((uint8_t)(0x00U))               /*!< push pull mode */
#define GPIO_OTYPE_OD              ((uint8_t)(0x01U))               /*!< open drain mode */

/* GPIO output max speed value */
#define OSPD_OSPD(regval)          (BITS(0,1) & ((uint32_t)(regval) << 0))

/* GPIO alternate function values */
#define GPIO_AFR_SET(n, af)        ((uint32_t)((uint32_t)(af) << (4U * (n))))
#define GPIO_AFR_MASK(n)           ((uint32_t)((uint32_t)0x0000000FU << (4U * (n))))

/* GPIO Input Filter Value */
#define GPIO_IFL_SET(n,IFL_Value)  ((uint32_t)((uint32_t)(IFL_Value) << (8U * (n))))
#define GPIO_IFL_MASK(n)           ((uint32_t)((uint32_t)0x000000FFU << (8U * (n))))

/* GPIO Input Filter Type */
#define GPIO_IFTP_SET(n,IFL_Type)  ((uint32_t)((uint32_t)(IFL_Type)  << (2U * (n))))
#define GPIO_IFTP_MASK(n)          ((uint32_t)((uint32_t)0x00000003U << (2U * (n))))

/* GPIO input filter type values */
#define GPIO_IFTYPE_SET(n, type)   ((uint32_t)((uint32_t)(type) << (2U * (n))))
#define GPIO_IFTYPE_MASK(n)        (0x3U << (2U * (n)))

/* @STRUCT_MEMBER: Pin */
/* @ENUM: GPIO Pin State definitions */
typedef enum {
    GPIO_PIN_RESET      = (uint32_t)0x00000000U,          /*!< GPIO Pin Level low */
    GPIO_PIN_SET                                          /*!< GPIO Pin Level High */
} hal_gpio_pin_enum;

/* @STRUCT_MEMBER: mode */
/* @ENUM: GPIO mode definitions */
typedef enum {
    GPIO_MODE_ANALOG      = (uint32_t)0x00000003U,                                   /*!< analog mode */
    GPIO_MODE_INPUT       = (uint32_t)0x00000000U,                                   /*!< input floating mode */
    GPIO_MODE_OUTPUT_PP   = (uint32_t)0x00000001U,                                   /*!< output push pull mode */
    GPIO_MODE_OUTPUT_OD   = (uint32_t)0x00000011U,                                   /*!< output open drain mode */
    GPIO_MODE_AF_PP       = (uint32_t)0x00000002U,                                   /*!< alternate function push pull mode */
    GPIO_MODE_AF_OD       = (uint32_t)0x00000012U,                                   /*!< alternate function open drain mode */
    GPIO_MODE_IT_RISING   = ((EXTI_INTERRUPT_TRIG_RISING << 8) | GPIO_MODE_INPUT),   /*!< EXTI interrupt rising edge trigger */
    GPIO_MODE_IT_FALLING  = ((EXTI_INTERRUPT_TRIG_FALLING << 8) | GPIO_MODE_INPUT),  /*!< EXTI interrupt falling edge trigger */
    GPIO_MODE_IT_BOTH     = ((EXTI_INTERRUPT_TRIG_BOTH << 8) | GPIO_MODE_INPUT),     /*!< EXTI interrupt both edge trigger */
    GPIO_MODE_EVT_RISING  = ((EXTI_EVENT_TRIG_RISING << 8) | GPIO_MODE_INPUT),       /*!< EXTI event rising edge trigger */
    GPIO_MODE_EVT_FALLING = ((EXTI_EVENT_TRIG_FALLING << 8) | GPIO_MODE_INPUT),      /*!< EXTI event falling edge trigger */
    GPIO_MODE_EVT_BOTH    = ((EXTI_EVENT_TRIG_BOTH << 8) | GPIO_MODE_INPUT)          /*!< EXTI event both edge trigger */
} hal_gpio_mode_enum;

/* @STRUCT_MEMBER: pull */
/* @ENUM: pull-up/pull-down definitions */
typedef enum {
    GPIO_PULL_NONE = (uint32_t)0x00000000U,               /*!< floating mode, no pull-up and pull-down resistors */
    GPIO_PULL_UP,                                         /*!< with pull-up resistor */
    GPIO_PULL_DOWN                                        /*!< with pull-down resistor */
} hal_gpio_pull_enum;

/* @STRUCT_MEMBER: ospeed */
/* @ENUM: GPIO output max speed value */
typedef enum {
    GPIO_OSPEED_12MHZ = (uint32_t)0x00000000U,            /*!< output max speed 12MHz */
    GPIO_OSPEED_60MHZ,                                    /*!< output max speed 60MHz */
    GPIO_OSPEED_85MHZ,                                    /*!< output max speed 85MHz */
    GPIO_OSPEED_100_220MHZ                                /*!< output max speed 100/220MHz */
} hal_gpio_ospeed_enum;

/* @STRUCT_MEMBER: af */
/* @ENUM: GPIO alternate function */
typedef enum {
    GPIO_AF_0 = (uint32_t)0x00000000U,                    /*!< alternate function 0 selected */
    GPIO_AF_1,                                            /*!< alternate function 1 selected */
    GPIO_AF_2,                                            /*!< alternate function 2 selected */
    GPIO_AF_3,                                            /*!< alternate function 3 selected */
    GPIO_AF_4,                                            /*!< alternate function 4 selected */
    GPIO_AF_5,                                            /*!< alternate function 5 selected */
    GPIO_AF_6,                                            /*!< alternate function 6 selected */
    GPIO_AF_7,                                            /*!< alternate function 7 selected */
    GPIO_AF_8,                                            /*!< alternate function 8 selected */
    GPIO_AF_9,                                            /*!< alternate function 9 selected */
    GPIO_AF_10,                                           /*!< alternate function 10 selected */
    GPIO_AF_11,                                           /*!< alternate function 11 selected */
    GPIO_AF_12,                                           /*!< alternate function 12 selected */
    GPIO_AF_13,                                           /*!< alternate function 13 selected */
    GPIO_AF_14,                                           /*!< alternate function 14 selected */
    GPIO_AF_15                                            /*!< alternate function 15 selected */
} hal_gpio_af_enum;

/* @STRUCT_MEMBER: filter */
/* @ENUM: GPIO Input Filter Value */
typedef enum {
    GPIO_IFL_CK_AHB = (uint32_t)0x00000000U,              /*!< GPIO Input Filter Value :  CK_AHB */
    GPIO_IFL_CK_AHB_2,                                    /*!< GPIO Input Filter Value :  CK_AHB\2 */
    GPIO_IFL_CK_AHB_4,                                    /*!< GPIO Input Filter Value :  CK_AHB\4 */
    GPIO_IFL_CK_AHB_6,                                    /*!< GPIO Input Filter Value :  CK_AHB\6 */
    GPIO_IFL_CK_AHB_8,                                    /*!< GPIO Input Filter Value :  CK_AHB\8 */
    GPIO_IFL_CK_AHB_10,                                   /*!< GPIO Input Filter Value :  CK_AHB\10 */
    GPIO_IFL_CK_AHB_12,                                   /*!< GPIO Input Filter Value :  CK_AHB\12 */
    GPIO_IFL_CK_AHB_14,                                   /*!< GPIO Input Filter Value :  CK_AHB\14 */
    GPIO_IFL_CK_AHB_16,                                   /*!< GPIO Input Filter Value :  CK_AHB\16 */
    GPIO_IFL_CK_AHB_18,                                   /*!< GPIO Input Filter Value :  CK_AHB\18 */
    GPIO_IFL_CK_AHB_20,                                   /*!< GPIO Input Filter Value :  CK_AHB\20 */
    GPIO_IFL_CK_AHB_22,                                   /*!< GPIO Input Filter Value :  CK_AHB\22 */
    GPIO_IFL_CK_AHB_24,                                   /*!< GPIO Input Filter Value :  CK_AHB\24 */
    GPIO_IFL_CK_AHB_26,                                   /*!< GPIO Input Filter Value :  CK_AHB\26 */
    GPIO_IFL_CK_AHB_28,                                   /*!< GPIO Input Filter Value :  CK_AHB\28 */
    GPIO_IFL_CK_AHB_30,                                   /*!< GPIO Input Filter Value :  CK_AHB\30 */
    GPIO_IFL_CK_AHB_32,                                   /*!< GPIO Input Filter Value :  CK_AHB\32 */
    GPIO_IFL_CK_AHB_34,                                   /*!< GPIO Input Filter Value :  CK_AHB\34 */
    GPIO_IFL_CK_AHB_36,                                   /*!< GPIO Input Filter Value :  CK_AHB\36 */
    GPIO_IFL_CK_AHB_38,                                   /*!< GPIO Input Filter Value :  CK_AHB\38 */
    GPIO_IFL_CK_AHB_40,                                   /*!< GPIO Input Filter Value :  CK_AHB\40 */
    GPIO_IFL_CK_AHB_42,                                   /*!< GPIO Input Filter Value :  CK_AHB\42 */
    GPIO_IFL_CK_AHB_44,                                   /*!< GPIO Input Filter Value :  CK_AHB\44 */
    GPIO_IFL_CK_AHB_46,                                   /*!< GPIO Input Filter Value :  CK_AHB\46 */
    GPIO_IFL_CK_AHB_48,                                   /*!< GPIO Input Filter Value :  CK_AHB\48 */
    GPIO_IFL_CK_AHB_50,                                   /*!< GPIO Input Filter Value :  CK_AHB\50 */
    GPIO_IFL_CK_AHB_52,                                   /*!< GPIO Input Filter Value :  CK_AHB\52 */
    GPIO_IFL_CK_AHB_54,                                   /*!< GPIO Input Filter Value :  CK_AHB\54 */
    GPIO_IFL_CK_AHB_56,                                   /*!< GPIO Input Filter Value :  CK_AHB\56 */
    GPIO_IFL_CK_AHB_58,                                   /*!< GPIO Input Filter Value :  CK_AHB\58 */
    GPIO_IFL_CK_AHB_60,                                   /*!< GPIO Input Filter Value :  CK_AHB\60 */
    GPIO_IFL_CK_AHB_62,                                   /*!< GPIO Input Filter Value :  CK_AHB\62 */
    GPIO_IFL_CK_AHB_64,                                   /*!< GPIO Input Filter Value :  CK_AHB\64 */
    GPIO_IFL_CK_AHB_66,                                   /*!< GPIO Input Filter Value :  CK_AHB\66 */
    GPIO_IFL_CK_AHB_68,                                   /*!< GPIO Input Filter Value :  CK_AHB\68 */
    GPIO_IFL_CK_AHB_70,                                   /*!< GPIO Input Filter Value :  CK_AHB\70 */
    GPIO_IFL_CK_AHB_72,                                   /*!< GPIO Input Filter Value :  CK_AHB\72 */
    GPIO_IFL_CK_AHB_74,                                   /*!< GPIO Input Filter Value :  CK_AHB\74 */
    GPIO_IFL_CK_AHB_76,                                   /*!< GPIO Input Filter Value :  CK_AHB\76 */
    GPIO_IFL_CK_AHB_78,                                   /*!< GPIO Input Filter Value :  CK_AHB\78 */
    GPIO_IFL_CK_AHB_80,                                   /*!< GPIO Input Filter Value :  CK_AHB\80 */
    GPIO_IFL_CK_AHB_82,                                   /*!< GPIO Input Filter Value :  CK_AHB\82 */
    GPIO_IFL_CK_AHB_84,                                   /*!< GPIO Input Filter Value :  CK_AHB\84 */
    GPIO_IFL_CK_AHB_86,                                   /*!< GPIO Input Filter Value :  CK_AHB\86 */
    GPIO_IFL_CK_AHB_88,                                   /*!< GPIO Input Filter Value :  CK_AHB\88 */
    GPIO_IFL_CK_AHB_90,                                   /*!< GPIO Input Filter Value :  CK_AHB\90 */
    GPIO_IFL_CK_AHB_92,                                   /*!< GPIO Input Filter Value :  CK_AHB\92 */
    GPIO_IFL_CK_AHB_94,                                   /*!< GPIO Input Filter Value :  CK_AHB\94 */
    GPIO_IFL_CK_AHB_96,                                   /*!< GPIO Input Filter Value :  CK_AHB\96 */
    GPIO_IFL_CK_AHB_98,                                   /*!< GPIO Input Filter Value :  CK_AHB\98 */
    GPIO_IFL_CK_AHB_100,                                  /*!< GPIO Input Filter Value :  CK_AHB\100 */
    GPIO_IFL_CK_AHB_102,                                  /*!< GPIO Input Filter Value :  CK_AHB\102 */
    GPIO_IFL_CK_AHB_104,                                  /*!< GPIO Input Filter Value :  CK_AHB\104 */
    GPIO_IFL_CK_AHB_106,                                  /*!< GPIO Input Filter Value :  CK_AHB\106 */
    GPIO_IFL_CK_AHB_108,                                  /*!< GPIO Input Filter Value :  CK_AHB\108 */
    GPIO_IFL_CK_AHB_110,                                  /*!< GPIO Input Filter Value :  CK_AHB\110 */
    GPIO_IFL_CK_AHB_112,                                  /*!< GPIO Input Filter Value :  CK_AHB\112 */
    GPIO_IFL_CK_AHB_114,                                  /*!< GPIO Input Filter Value :  CK_AHB\114 */
    GPIO_IFL_CK_AHB_116,                                  /*!< GPIO Input Filter Value :  CK_AHB\116 */
    GPIO_IFL_CK_AHB_118,                                  /*!< GPIO Input Filter Value :  CK_AHB\118 */
    GPIO_IFL_CK_AHB_120,                                  /*!< GPIO Input Filter Value :  CK_AHB\120 */
    GPIO_IFL_CK_AHB_122,                                  /*!< GPIO Input Filter Value :  CK_AHB\122 */
    GPIO_IFL_CK_AHB_124,                                  /*!< GPIO Input Filter Value :  CK_AHB\124 */
    GPIO_IFL_CK_AHB_126,                                  /*!< GPIO Input Filter Value :  CK_AHB\126 */
    GPIO_IFL_CK_AHB_128,                                  /*!< GPIO Input Filter Value :  CK_AHB\128 */
    GPIO_IFL_CK_AHB_130,                                  /*!< GPIO Input Filter Value :  CK_AHB\130 */
    GPIO_IFL_CK_AHB_132,                                  /*!< GPIO Input Filter Value :  CK_AHB\132 */
    GPIO_IFL_CK_AHB_134,                                  /*!< GPIO Input Filter Value :  CK_AHB\134 */
    GPIO_IFL_CK_AHB_136,                                  /*!< GPIO Input Filter Value :  CK_AHB\136 */
    GPIO_IFL_CK_AHB_138,                                  /*!< GPIO Input Filter Value :  CK_AHB\138 */
    GPIO_IFL_CK_AHB_140,                                  /*!< GPIO Input Filter Value :  CK_AHB\140 */
    GPIO_IFL_CK_AHB_142,                                  /*!< GPIO Input Filter Value :  CK_AHB\142 */
    GPIO_IFL_CK_AHB_144,                                  /*!< GPIO Input Filter Value :  CK_AHB\144 */
    GPIO_IFL_CK_AHB_146,                                  /*!< GPIO Input Filter Value :  CK_AHB\146 */
    GPIO_IFL_CK_AHB_148,                                  /*!< GPIO Input Filter Value :  CK_AHB\148 */
    GPIO_IFL_CK_AHB_150,                                  /*!< GPIO Input Filter Value :  CK_AHB\150 */
    GPIO_IFL_CK_AHB_152,                                  /*!< GPIO Input Filter Value :  CK_AHB\152 */
    GPIO_IFL_CK_AHB_154,                                  /*!< GPIO Input Filter Value :  CK_AHB\154 */
    GPIO_IFL_CK_AHB_156,                                  /*!< GPIO Input Filter Value :  CK_AHB\156 */
    GPIO_IFL_CK_AHB_158,                                  /*!< GPIO Input Filter Value :  CK_AHB\158 */
    GPIO_IFL_CK_AHB_160,                                  /*!< GPIO Input Filter Value :  CK_AHB\160 */
    GPIO_IFL_CK_AHB_162,                                  /*!< GPIO Input Filter Value :  CK_AHB\162 */
    GPIO_IFL_CK_AHB_164,                                  /*!< GPIO Input Filter Value :  CK_AHB\164 */
    GPIO_IFL_CK_AHB_166,                                  /*!< GPIO Input Filter Value :  CK_AHB\166 */
    GPIO_IFL_CK_AHB_168,                                  /*!< GPIO Input Filter Value :  CK_AHB\168 */
    GPIO_IFL_CK_AHB_170,                                  /*!< GPIO Input Filter Value :  CK_AHB\170 */
    GPIO_IFL_CK_AHB_172,                                  /*!< GPIO Input Filter Value :  CK_AHB\172 */
    GPIO_IFL_CK_AHB_174,                                  /*!< GPIO Input Filter Value :  CK_AHB\174 */
    GPIO_IFL_CK_AHB_176,                                  /*!< GPIO Input Filter Value :  CK_AHB\176 */
    GPIO_IFL_CK_AHB_178,                                  /*!< GPIO Input Filter Value :  CK_AHB\178 */
    GPIO_IFL_CK_AHB_180,                                  /*!< GPIO Input Filter Value :  CK_AHB\180 */
    GPIO_IFL_CK_AHB_182,                                  /*!< GPIO Input Filter Value :  CK_AHB\182 */
    GPIO_IFL_CK_AHB_184,                                  /*!< GPIO Input Filter Value :  CK_AHB\184 */
    GPIO_IFL_CK_AHB_186,                                  /*!< GPIO Input Filter Value :  CK_AHB\186 */
    GPIO_IFL_CK_AHB_188,                                  /*!< GPIO Input Filter Value :  CK_AHB\188 */
    GPIO_IFL_CK_AHB_190,                                  /*!< GPIO Input Filter Value :  CK_AHB\190 */
    GPIO_IFL_CK_AHB_192,                                  /*!< GPIO Input Filter Value :  CK_AHB\192 */
    GPIO_IFL_CK_AHB_194,                                  /*!< GPIO Input Filter Value :  CK_AHB\194 */
    GPIO_IFL_CK_AHB_196,                                  /*!< GPIO Input Filter Value :  CK_AHB\196 */
    GPIO_IFL_CK_AHB_198,                                  /*!< GPIO Input Filter Value :  CK_AHB\198 */
    GPIO_IFL_CK_AHB_200,                                  /*!< GPIO Input Filter Value :  CK_AHB\200 */
    GPIO_IFL_CK_AHB_202,                                  /*!< GPIO Input Filter Value :  CK_AHB\202 */
    GPIO_IFL_CK_AHB_204,                                  /*!< GPIO Input Filter Value :  CK_AHB\204 */
    GPIO_IFL_CK_AHB_206,                                  /*!< GPIO Input Filter Value :  CK_AHB\206 */
    GPIO_IFL_CK_AHB_208,                                  /*!< GPIO Input Filter Value :  CK_AHB\208 */
    GPIO_IFL_CK_AHB_210,                                  /*!< GPIO Input Filter Value :  CK_AHB\210 */
    GPIO_IFL_CK_AHB_212,                                  /*!< GPIO Input Filter Value :  CK_AHB\212 */
    GPIO_IFL_CK_AHB_214,                                  /*!< GPIO Input Filter Value :  CK_AHB\214 */
    GPIO_IFL_CK_AHB_216,                                  /*!< GPIO Input Filter Value :  CK_AHB\216 */
    GPIO_IFL_CK_AHB_218,                                  /*!< GPIO Input Filter Value :  CK_AHB\218 */
    GPIO_IFL_CK_AHB_220,                                  /*!< GPIO Input Filter Value :  CK_AHB\220 */
    GPIO_IFL_CK_AHB_222,                                  /*!< GPIO Input Filter Value :  CK_AHB\222 */
    GPIO_IFL_CK_AHB_224,                                  /*!< GPIO Input Filter Value :  CK_AHB\224 */
    GPIO_IFL_CK_AHB_226,                                  /*!< GPIO Input Filter Value :  CK_AHB\226 */
    GPIO_IFL_CK_AHB_228,                                  /*!< GPIO Input Filter Value :  CK_AHB\228 */
    GPIO_IFL_CK_AHB_230,                                  /*!< GPIO Input Filter Value :  CK_AHB\230 */
    GPIO_IFL_CK_AHB_232,                                  /*!< GPIO Input Filter Value :  CK_AHB\232 */
    GPIO_IFL_CK_AHB_234,                                  /*!< GPIO Input Filter Value :  CK_AHB\234 */
    GPIO_IFL_CK_AHB_236,                                  /*!< GPIO Input Filter Value :  CK_AHB\236 */
    GPIO_IFL_CK_AHB_238,                                  /*!< GPIO Input Filter Value :  CK_AHB\238 */
    GPIO_IFL_CK_AHB_240,                                  /*!< GPIO Input Filter Value :  CK_AHB\240 */
    GPIO_IFL_CK_AHB_242,                                  /*!< GPIO Input Filter Value :  CK_AHB\242 */
    GPIO_IFL_CK_AHB_244,                                  /*!< GPIO Input Filter Value :  CK_AHB\244 */
    GPIO_IFL_CK_AHB_246,                                  /*!< GPIO Input Filter Value :  CK_AHB\246 */
    GPIO_IFL_CK_AHB_248,                                  /*!< GPIO Input Filter Value :  CK_AHB\248 */
    GPIO_IFL_CK_AHB_250,                                  /*!< GPIO Input Filter Value :  CK_AHB\250 */
    GPIO_IFL_CK_AHB_252,                                  /*!< GPIO Input Filter Value :  CK_AHB\252 */
    GPIO_IFL_CK_AHB_254,                                  /*!< GPIO Input Filter Value :  CK_AHB\254 */
    GPIO_IFL_CK_AHB_256,                                  /*!< GPIO Input Filter Value :  CK_AHB\256 */
    GPIO_IFL_CK_AHB_258,                                  /*!< GPIO Input Filter Value :  CK_AHB\258 */
    GPIO_IFL_CK_AHB_260,                                  /*!< GPIO Input Filter Value :  CK_AHB\260 */
    GPIO_IFL_CK_AHB_262,                                  /*!< GPIO Input Filter Value :  CK_AHB\262 */
    GPIO_IFL_CK_AHB_264,                                  /*!< GPIO Input Filter Value :  CK_AHB\264 */
    GPIO_IFL_CK_AHB_266,                                  /*!< GPIO Input Filter Value :  CK_AHB\266 */
    GPIO_IFL_CK_AHB_268,                                  /*!< GPIO Input Filter Value :  CK_AHB\268 */
    GPIO_IFL_CK_AHB_270,                                  /*!< GPIO Input Filter Value :  CK_AHB\270 */
    GPIO_IFL_CK_AHB_272,                                  /*!< GPIO Input Filter Value :  CK_AHB\272 */
    GPIO_IFL_CK_AHB_274,                                  /*!< GPIO Input Filter Value :  CK_AHB\274 */
    GPIO_IFL_CK_AHB_276,                                  /*!< GPIO Input Filter Value :  CK_AHB\276 */
    GPIO_IFL_CK_AHB_278,                                  /*!< GPIO Input Filter Value :  CK_AHB\278 */
    GPIO_IFL_CK_AHB_280,                                  /*!< GPIO Input Filter Value :  CK_AHB\280 */
    GPIO_IFL_CK_AHB_282,                                  /*!< GPIO Input Filter Value :  CK_AHB\282 */
    GPIO_IFL_CK_AHB_284,                                  /*!< GPIO Input Filter Value :  CK_AHB\284 */
    GPIO_IFL_CK_AHB_286,                                  /*!< GPIO Input Filter Value :  CK_AHB\286 */
    GPIO_IFL_CK_AHB_288,                                  /*!< GPIO Input Filter Value :  CK_AHB\288 */
    GPIO_IFL_CK_AHB_290,                                  /*!< GPIO Input Filter Value :  CK_AHB\290 */
    GPIO_IFL_CK_AHB_292,                                  /*!< GPIO Input Filter Value :  CK_AHB\292 */
    GPIO_IFL_CK_AHB_294,                                  /*!< GPIO Input Filter Value :  CK_AHB\294 */
    GPIO_IFL_CK_AHB_296,                                  /*!< GPIO Input Filter Value :  CK_AHB\296 */
    GPIO_IFL_CK_AHB_298,                                  /*!< GPIO Input Filter Value :  CK_AHB\298 */
    GPIO_IFL_CK_AHB_300,                                  /*!< GPIO Input Filter Value :  CK_AHB\300 */
    GPIO_IFL_CK_AHB_302,                                  /*!< GPIO Input Filter Value :  CK_AHB\302 */
    GPIO_IFL_CK_AHB_304,                                  /*!< GPIO Input Filter Value :  CK_AHB\304 */
    GPIO_IFL_CK_AHB_306,                                  /*!< GPIO Input Filter Value :  CK_AHB\306 */
    GPIO_IFL_CK_AHB_308,                                  /*!< GPIO Input Filter Value :  CK_AHB\308 */
    GPIO_IFL_CK_AHB_310,                                  /*!< GPIO Input Filter Value :  CK_AHB\310 */
    GPIO_IFL_CK_AHB_312,                                  /*!< GPIO Input Filter Value :  CK_AHB\312 */
    GPIO_IFL_CK_AHB_314,                                  /*!< GPIO Input Filter Value :  CK_AHB\314 */
    GPIO_IFL_CK_AHB_316,                                  /*!< GPIO Input Filter Value :  CK_AHB\316 */
    GPIO_IFL_CK_AHB_318,                                  /*!< GPIO Input Filter Value :  CK_AHB\318 */
    GPIO_IFL_CK_AHB_320,                                  /*!< GPIO Input Filter Value :  CK_AHB\320 */
    GPIO_IFL_CK_AHB_322,                                  /*!< GPIO Input Filter Value :  CK_AHB\322 */
    GPIO_IFL_CK_AHB_324,                                  /*!< GPIO Input Filter Value :  CK_AHB\324 */
    GPIO_IFL_CK_AHB_326,                                  /*!< GPIO Input Filter Value :  CK_AHB\326 */
    GPIO_IFL_CK_AHB_328,                                  /*!< GPIO Input Filter Value :  CK_AHB\328 */
    GPIO_IFL_CK_AHB_330,                                  /*!< GPIO Input Filter Value :  CK_AHB\330 */
    GPIO_IFL_CK_AHB_332,                                  /*!< GPIO Input Filter Value :  CK_AHB\332 */
    GPIO_IFL_CK_AHB_334,                                  /*!< GPIO Input Filter Value :  CK_AHB\334 */
    GPIO_IFL_CK_AHB_336,                                  /*!< GPIO Input Filter Value :  CK_AHB\336 */
    GPIO_IFL_CK_AHB_338,                                  /*!< GPIO Input Filter Value :  CK_AHB\338 */
    GPIO_IFL_CK_AHB_340,                                  /*!< GPIO Input Filter Value :  CK_AHB\340 */
    GPIO_IFL_CK_AHB_342,                                  /*!< GPIO Input Filter Value :  CK_AHB\342 */
    GPIO_IFL_CK_AHB_344,                                  /*!< GPIO Input Filter Value :  CK_AHB\344 */
    GPIO_IFL_CK_AHB_346,                                  /*!< GPIO Input Filter Value :  CK_AHB\346 */
    GPIO_IFL_CK_AHB_348,                                  /*!< GPIO Input Filter Value :  CK_AHB\348 */
    GPIO_IFL_CK_AHB_350,                                  /*!< GPIO Input Filter Value :  CK_AHB\350 */
    GPIO_IFL_CK_AHB_352,                                  /*!< GPIO Input Filter Value :  CK_AHB\352 */
    GPIO_IFL_CK_AHB_354,                                  /*!< GPIO Input Filter Value :  CK_AHB\354 */
    GPIO_IFL_CK_AHB_356,                                  /*!< GPIO Input Filter Value :  CK_AHB\356 */
    GPIO_IFL_CK_AHB_358,                                  /*!< GPIO Input Filter Value :  CK_AHB\358 */
    GPIO_IFL_CK_AHB_360,                                  /*!< GPIO Input Filter Value :  CK_AHB\360 */
    GPIO_IFL_CK_AHB_362,                                  /*!< GPIO Input Filter Value :  CK_AHB\362 */
    GPIO_IFL_CK_AHB_364,                                  /*!< GPIO Input Filter Value :  CK_AHB\364 */
    GPIO_IFL_CK_AHB_366,                                  /*!< GPIO Input Filter Value :  CK_AHB\366 */
    GPIO_IFL_CK_AHB_368,                                  /*!< GPIO Input Filter Value :  CK_AHB\368 */
    GPIO_IFL_CK_AHB_370,                                  /*!< GPIO Input Filter Value :  CK_AHB\370 */
    GPIO_IFL_CK_AHB_372,                                  /*!< GPIO Input Filter Value :  CK_AHB\372 */
    GPIO_IFL_CK_AHB_374,                                  /*!< GPIO Input Filter Value :  CK_AHB\374 */
    GPIO_IFL_CK_AHB_376,                                  /*!< GPIO Input Filter Value :  CK_AHB\376 */
    GPIO_IFL_CK_AHB_378,                                  /*!< GPIO Input Filter Value :  CK_AHB\378 */
    GPIO_IFL_CK_AHB_380,                                  /*!< GPIO Input Filter Value :  CK_AHB\380 */
    GPIO_IFL_CK_AHB_382,                                  /*!< GPIO Input Filter Value :  CK_AHB\382 */
    GPIO_IFL_CK_AHB_384,                                  /*!< GPIO Input Filter Value :  CK_AHB\384 */
    GPIO_IFL_CK_AHB_386,                                  /*!< GPIO Input Filter Value :  CK_AHB\386 */
    GPIO_IFL_CK_AHB_388,                                  /*!< GPIO Input Filter Value :  CK_AHB\388 */
    GPIO_IFL_CK_AHB_390,                                  /*!< GPIO Input Filter Value :  CK_AHB\390 */
    GPIO_IFL_CK_AHB_392,                                  /*!< GPIO Input Filter Value :  CK_AHB\392 */
    GPIO_IFL_CK_AHB_394,                                  /*!< GPIO Input Filter Value :  CK_AHB\394 */
    GPIO_IFL_CK_AHB_396,                                  /*!< GPIO Input Filter Value :  CK_AHB\396 */
    GPIO_IFL_CK_AHB_398,                                  /*!< GPIO Input Filter Value :  CK_AHB\398 */
    GPIO_IFL_CK_AHB_400,                                  /*!< GPIO Input Filter Value :  CK_AHB\400 */
    GPIO_IFL_CK_AHB_402,                                  /*!< GPIO Input Filter Value :  CK_AHB\402 */
    GPIO_IFL_CK_AHB_404,                                  /*!< GPIO Input Filter Value :  CK_AHB\404 */
    GPIO_IFL_CK_AHB_406,                                  /*!< GPIO Input Filter Value :  CK_AHB\406 */
    GPIO_IFL_CK_AHB_408,                                  /*!< GPIO Input Filter Value :  CK_AHB\408 */
    GPIO_IFL_CK_AHB_410,                                  /*!< GPIO Input Filter Value :  CK_AHB\410 */
    GPIO_IFL_CK_AHB_412,                                  /*!< GPIO Input Filter Value :  CK_AHB\412 */
    GPIO_IFL_CK_AHB_414,                                  /*!< GPIO Input Filter Value :  CK_AHB\414 */
    GPIO_IFL_CK_AHB_416,                                  /*!< GPIO Input Filter Value :  CK_AHB\416 */
    GPIO_IFL_CK_AHB_418,                                  /*!< GPIO Input Filter Value :  CK_AHB\418 */
    GPIO_IFL_CK_AHB_420,                                  /*!< GPIO Input Filter Value :  CK_AHB\420 */
    GPIO_IFL_CK_AHB_422,                                  /*!< GPIO Input Filter Value :  CK_AHB\422 */
    GPIO_IFL_CK_AHB_424,                                  /*!< GPIO Input Filter Value :  CK_AHB\424 */
    GPIO_IFL_CK_AHB_426,                                  /*!< GPIO Input Filter Value :  CK_AHB\426 */
    GPIO_IFL_CK_AHB_428,                                  /*!< GPIO Input Filter Value :  CK_AHB\428 */
    GPIO_IFL_CK_AHB_430,                                  /*!< GPIO Input Filter Value :  CK_AHB\430 */
    GPIO_IFL_CK_AHB_432,                                  /*!< GPIO Input Filter Value :  CK_AHB\432 */
    GPIO_IFL_CK_AHB_434,                                  /*!< GPIO Input Filter Value :  CK_AHB\434 */
    GPIO_IFL_CK_AHB_436,                                  /*!< GPIO Input Filter Value :  CK_AHB\436 */
    GPIO_IFL_CK_AHB_438,                                  /*!< GPIO Input Filter Value :  CK_AHB\438 */
    GPIO_IFL_CK_AHB_440,                                  /*!< GPIO Input Filter Value :  CK_AHB\440 */
    GPIO_IFL_CK_AHB_442,                                  /*!< GPIO Input Filter Value :  CK_AHB\442 */
    GPIO_IFL_CK_AHB_444,                                  /*!< GPIO Input Filter Value :  CK_AHB\444 */
    GPIO_IFL_CK_AHB_446,                                  /*!< GPIO Input Filter Value :  CK_AHB\446 */
    GPIO_IFL_CK_AHB_448,                                  /*!< GPIO Input Filter Value :  CK_AHB\448 */
    GPIO_IFL_CK_AHB_450,                                  /*!< GPIO Input Filter Value :  CK_AHB\450 */
    GPIO_IFL_CK_AHB_452,                                  /*!< GPIO Input Filter Value :  CK_AHB\452 */
    GPIO_IFL_CK_AHB_454,                                  /*!< GPIO Input Filter Value :  CK_AHB\454 */
    GPIO_IFL_CK_AHB_456,                                  /*!< GPIO Input Filter Value :  CK_AHB\456 */
    GPIO_IFL_CK_AHB_458,                                  /*!< GPIO Input Filter Value :  CK_AHB\458 */
    GPIO_IFL_CK_AHB_460,                                  /*!< GPIO Input Filter Value :  CK_AHB\460 */
    GPIO_IFL_CK_AHB_462,                                  /*!< GPIO Input Filter Value :  CK_AHB\462 */
    GPIO_IFL_CK_AHB_464,                                  /*!< GPIO Input Filter Value :  CK_AHB\464 */
    GPIO_IFL_CK_AHB_466,                                  /*!< GPIO Input Filter Value :  CK_AHB\466 */
    GPIO_IFL_CK_AHB_468,                                  /*!< GPIO Input Filter Value :  CK_AHB\468 */
    GPIO_IFL_CK_AHB_470,                                  /*!< GPIO Input Filter Value :  CK_AHB\470 */
    GPIO_IFL_CK_AHB_472,                                  /*!< GPIO Input Filter Value :  CK_AHB\472 */
    GPIO_IFL_CK_AHB_474,                                  /*!< GPIO Input Filter Value :  CK_AHB\474 */
    GPIO_IFL_CK_AHB_476,                                  /*!< GPIO Input Filter Value :  CK_AHB\476 */
    GPIO_IFL_CK_AHB_478,                                  /*!< GPIO Input Filter Value :  CK_AHB\478 */
    GPIO_IFL_CK_AHB_480,                                  /*!< GPIO Input Filter Value :  CK_AHB\480 */
    GPIO_IFL_CK_AHB_482,                                  /*!< GPIO Input Filter Value :  CK_AHB\482 */
    GPIO_IFL_CK_AHB_484,                                  /*!< GPIO Input Filter Value :  CK_AHB\484 */
    GPIO_IFL_CK_AHB_486,                                  /*!< GPIO Input Filter Value :  CK_AHB\486 */
    GPIO_IFL_CK_AHB_488,                                  /*!< GPIO Input Filter Value :  CK_AHB\488 */
    GPIO_IFL_CK_AHB_490,                                  /*!< GPIO Input Filter Value :  CK_AHB\490 */
    GPIO_IFL_CK_AHB_492,                                  /*!< GPIO Input Filter Value :  CK_AHB\492 */
    GPIO_IFL_CK_AHB_494,                                  /*!< GPIO Input Filter Value :  CK_AHB\494 */
    GPIO_IFL_CK_AHB_496,                                  /*!< GPIO Input Filter Value :  CK_AHB\496 */
    GPIO_IFL_CK_AHB_498,                                  /*!< GPIO Input Filter Value :  CK_AHB\498 */
    GPIO_IFL_CK_AHB_500,                                  /*!< GPIO Input Filter Value :  CK_AHB\500 */
    GPIO_IFL_CK_AHB_502,                                  /*!< GPIO Input Filter Value :  CK_AHB\502 */
    GPIO_IFL_CK_AHB_504,                                  /*!< GPIO Input Filter Value :  CK_AHB\504 */
    GPIO_IFL_CK_AHB_506,                                  /*!< GPIO Input Filter Value :  CK_AHB\506 */
    GPIO_IFL_CK_AHB_508,                                  /*!< GPIO Input Filter Value :  CK_AHB\508 */
    GPIO_IFL_CK_AHB_510                                   /*!< GPIO Input Filter Value :  CK_AHB\510 */
} hal_gpio_filter_value_enum;

/* @STRUCT_MEMBER: filter_type */
/* @ENUM: GPIO Input Filter Type */
typedef enum {
    GPIO_IFTP_Synchronization = (uint32_t)0x00000000U,    /*!< GPIO Input Filter Type : synchronization */
    GPIO_IFTP_Filter_3,                                   /*!< GPIO Input Filter Type : Filter (3 sampling points) */
    GPIO_IFTP_Filter_6,                                   /*!< GPIO Input Filter Type : Filter (6 sampling points) */
    GPIO_IFTP_Asynchronization                            /*!< GPIO Input Filter Type : Asynchronization */
} hal_gpio_filter_type_enum;

/* @PARA: p_init */
/* @STRUCT: GPIO initialization structure */
typedef struct {
    hal_gpio_mode_enum mode;                 /* refer to hal_gpio_mode_enum */
    hal_gpio_pull_enum pull;                 /* refer to hal_gpio_pull_enum */
    hal_gpio_ospeed_enum ospeed;             /* refer to hal_gpio_ospeed_enum */
    hal_gpio_af_enum af;                     /* refer to hal_gpio_af_enum */
    hal_gpio_filter_value_enum filter;       /* refer to hal_gpio_filter_value_enum */
    hal_gpio_filter_type_enum filter_type;   /* refer to hal_gpio_filter_type_enum */
} hal_gpio_init_struct;

/* function declarations */
/* @FUNCTION: initialize GPIO */
int32_t hal_gpio_init(uint32_t gpio_periph, uint32_t pin, hal_gpio_init_struct *p_init);
/* @FUNCTION: deinitialize GPIO */
int32_t hal_gpio_deinit(uint32_t gpio_periph, uint32_t pin);
/* @FUNCTION: initialize the GPIO initialization structure with the default values */
/* deinitialize GPIO initialization structure */
int32_t hal_gpio_struct_init(hal_gpio_init_struct *p_init);
/* @END */

/* set GPIO alternate function */
int32_t hal_gpio_af_set(uint32_t gpio_periph, uint32_t pin, hal_gpio_af_enum alt_func_num);
/* set GPIO mode */
int32_t hal_gpio_mode_set(uint32_t gpio_periph, uint32_t pin, hal_gpio_mode_enum mode, hal_gpio_pull_enum pull_up_down);
/* set GPIO output type and speed */
int32_t hal_gpio_output_options_set(uint32_t gpio_periph, uint32_t pin, uint8_t otype, hal_gpio_ospeed_enum speed);
/* set GPIO input filter */
int32_t hal_gpio_input_filter_set(uint32_t gpio_periph, uint32_t pin, uint8_t filter, hal_gpio_filter_type_enum filter_type);

/* set GPIO pin bit */
void hal_gpio_bit_set(uint32_t gpio_periph, uint32_t pin);
/* reset GPIO pin bit */
void hal_gpio_bit_reset(uint32_t gpio_periph, uint32_t pin);
/* write data to the specified GPIO pin */
void hal_gpio_bit_write(uint32_t gpio_periph, uint32_t pin, bit_status bit_value);
/* write data to the specified GPIO port */
void hal_gpio_port_write(uint32_t gpio_periph, uint16_t data);

/* get GPIO pin input status */
FlagStatus hal_gpio_input_bit_get(uint32_t gpio_periph, uint32_t pin);
/* get GPIO port input status */
uint16_t hal_gpio_input_port_get(uint32_t gpio_periph);
/* get GPIO pin output status */
FlagStatus hal_gpio_output_bit_get(uint32_t gpio_periph, uint32_t pin);
/* get GPIO port output status */
uint16_t hal_gpio_output_port_get(uint32_t gpio_periph);

/* lock GPIO pin bit */
void hal_gpio_pin_lock(uint32_t gpio_periph, uint32_t pin);
/* toggle GPIO pin status */
void hal_gpio_bit_toggle(uint32_t gpio_periph, uint32_t pin);
/* toggle GPIO port status */
void hal_gpio_port_toggle(uint32_t gpio_periph);

#endif /* GD32H7XX_HAL_GPIO_H */
