/*!
    \file    gd32h7xx_hal_cau.h
    \brief   definitions for the CAU

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_CAU_H
#define GD32H7XX_HAL_CAU_H

#include "gd32h7xx_hal.h"

/* CAU definitions */
#define CAU                         CAU_BASE                                         /*!< CAU base address */

/* registers definitions */
#define CAU_CTL                     REG32(CAU + 0x00000000U)                         /*!< control register  */
#define CAU_STAT0                   REG32(CAU + 0x00000004U)                         /*!< status register 0 */
#define CAU_DI                      REG32(CAU + 0x00000008U)                         /*!< data input register */
#define CAU_DO                      REG32(CAU + 0x0000000CU)                         /*!< data output register */
#define CAU_DMAEN                   REG32(CAU + 0x00000010U)                         /*!< DMA enable register */
#define CAU_INTEN                   REG32(CAU + 0x00000014U)                         /*!< interrupt enable register */
#define CAU_STAT1                   REG32(CAU + 0x00000018U)                         /*!< status register 1 */
#define CAU_INTF                    REG32(CAU + 0x0000001CU)                         /*!< interrupt flag register */
#define CAU_KEY0H                   REG32(CAU + 0x00000020U)                         /*!< key 0 high register */
#define CAU_KEY0L                   REG32(CAU + 0x00000024U)                         /*!< key 0 low register */
#define CAU_KEY1H                   REG32(CAU + 0x00000028U)                         /*!< key 1 high register */
#define CAU_KEY1L                   REG32(CAU + 0x0000002CU)                         /*!< key 1 low register */
#define CAU_KEY2H                   REG32(CAU + 0x00000030U)                         /*!< key 2 high register */
#define CAU_KEY2L                   REG32(CAU + 0x00000034U)                         /*!< key 2 low register */
#define CAU_KEY3H                   REG32(CAU + 0x00000038U)                         /*!< key 3 high register */
#define CAU_KEY3L                   REG32(CAU + 0x0000003CU)                         /*!< key 3 low register */
#define CAU_IV0H                    REG32(CAU + 0x00000040U)                         /*!< initial vector 0 high register */
#define CAU_IV0L                    REG32(CAU + 0x00000044U)                         /*!< initial vector 0 low register */
#define CAU_IV1H                    REG32(CAU + 0x00000048U)                         /*!< initial vector 1 high register */
#define CAU_IV1L                    REG32(CAU + 0x0000004CU)                         /*!< initial vector 1 low register */
#define CAU_GCMCCMCTXSx(x)          REG32(CAU + 0x00000050U + (uint32_t)(4U * (x)))  /*!< GCM or CCM mode context switch register, x = 0...7 */
#define CAU_GCMCTXSx(x)             REG32(CAU + 0x00000070U + (uint32_t)(4U * (x)))  /*!< GCM mode context switch register, x = 0...7 */

/* bits definitions */
/* CAU_CTL */
#define CAU_CTL_KEY_SEL             BIT(0)                                           /*!< key select */
#define CAU_CTL_CAUDIR              BIT(2)                                           /*!< algorithm direction */
#define CAU_CTL_ALGM                (BITS(3,5) | BIT(19))                            /*!< cryptographic algorithm mode */
#define CAU_CTL_DATAM               BITS(6,7)                                        /*!< data swapping selection */
#define CAU_CTL_KEYM                BITS(8,9)                                        /*!< key length selection when aes mode */
#define CAU_CTL_FFLUSH              BIT(14)                                          /*!< FIFO flush */
#define CAU_CTL_CAUEN               BIT(15)                                          /*!< cryptographic module enable */
#define CAU_CTL_GCM_CCMPH           BITS(16,17)                                      /*!< GCM CCM phase */
#define CAU_CTL_NBPILB              BITS(20,23)                                      /*!< number of bytes padding in last block */

/* CAU_STAT0 */
#define CAU_STAT0_IEM               BIT(0)                                           /*!< in fifo empty flag */
#define CAU_STAT0_INF               BIT(1)                                           /*!< in fifo not full flag */
#define CAU_STAT0_ONE               BIT(2)                                           /*!< out fifo not empty flag */
#define CAU_STAT0_OFU               BIT(3)                                           /*!< out fifo full flag */
#define CAU_STAT0_BUSY              BIT(4)                                           /*!< busy flag */

/* CAU_DI */
#define CAU_DI_DI                   BITS(0,31)                                       /*!< data input */

/* CAU_DO */
#define CAU_DO_DO                   BITS(0,31)                                       /*!< data output */

/* CAU_DMAEN */
#define CAU_DMAEN_DMAIEN            BIT(0)                                           /*!< in fifo DMA enable */
#define CAU_DMAEN_DMAOEN            BIT(1)                                           /*!< out fifo DMA enable */

/* CAU_INTEN */
#define CAU_INTEN_IINTEN            BIT(0)                                           /*!< in fifo interrupt enable */
#define CAU_INTEN_OINTEN            BIT(1)                                           /*!< out fifo interrupt enable */

/* CAU_STAT1 */
#define CAU_STAT1_ISTA              BIT(0)                                           /*!< flag set when there is less than 4 words in in fifo */
#define CAU_STAT1_OSTA              BIT(1)                                           /*!< flag set when there is one or more word in out fifo */

/* CAU_INTF */
#define CAU_INTF_IINTF              BIT(0)                                           /*!< in fifo interrupt flag */
#define CAU_INTF_OINTF              BIT(1)                                           /*!< out fifo interrupt flag */

/* CAU_KEYxH x=0..3 */
#define CAU_KEYXH_KEYXH             BITS(0,31)                                       /*!< the key for des, tdes, aes */

/* CAU_KEYxL x=0..3 */
#define CAU_KEYXL_KEYXL             BITS(0,31)                                       /*!< the key for des, tdes, aes */

/* CAU_IVxH x=0..1 */
#define CAU_IVXH_IVXH               BITS(0,31)                                       /*!< the initialization vector for des, tdes, aes */

/* CAU_IVxL x=0..1 */
#define CAU_IVXL_IVXL               BITS(0,31)                                       /*!< the initialization vector for des, tdes, aes */

#define AESBUSY_TIMEOUT             ((uint32_t)0x00010000U)                          /*!< aes busy timeout */
#define DESBUSY_TIMEOUT             ((uint32_t)0x00010000U)                          /*!< des busy timeout */
#define TDESBUSY_TIMEOUT            ((uint32_t)0x00010000U)                          /*!< tdes busy timeout */
#define BLOCK_B0_MASK               ((uint8_t)0x07U)                                 /*!< block size mask */
#define BLOCK_DATA_SIZE             ((uint32_t)0x00000010U)                          /*!< block size */
#define MIN_CCM_IV_SIZE             ((uint32_t)0x00000007U)                          /*!< min ccm iv size */
#define MAX_CCM_IV_SIZE             ((uint32_t)0x0000000DU)                          /*!< max ccm iv size */

/* cau_ctl register value */
#define CAU_ENCRYPT                 ((uint32_t)0x00000000U)                          /*!< encrypt */
#define CAU_DECRYPT                 CAU_CTL_CAUDIR                                   /*!< decrypt */

#define CTL_ALGM(regval)            ((BITS(3,5) & ((uint32_t)(regval) << 3U)) |\
                                    (BIT(19) & ((uint32_t)(regval) << 16U)))         /*!< write value to CAU_CTL_ALGM bit field */
#define CAU_MODE_TDES_ECB           CTL_ALGM(0)                                      /*!< TDES-ECB (3DES Electronic codebook) */
#define CAU_MODE_TDES_CBC           CTL_ALGM(1)                                      /*!< TDES-CBC (3DES Cipher block chaining) */
#define CAU_MODE_DES_ECB            CTL_ALGM(2)                                      /*!< DES-ECB (simple DES Electronic codebook) */
#define CAU_MODE_DES_CBC            CTL_ALGM(3)                                      /*!< DES-CBC (simple DES Cipher block chaining) */
#define CAU_MODE_AES_ECB            CTL_ALGM(4)                                      /*!< AES-ECB (AES Electronic codebook) */
#define CAU_MODE_AES_CBC            CTL_ALGM(5)                                      /*!< AES-CBC (AES Cipher block chaining) */
#define CAU_MODE_AES_CTR            CTL_ALGM(6)                                      /*!< AES-CTR (AES counter mode) */
#define CAU_MODE_AES_KEY            CTL_ALGM(7)                                      /*!< AES decryption key preparation mode */
#define CAU_MODE_AES_GCM            CTL_ALGM(8)                                      /*!< AES-GCM (AES Galois/counter mode) */
#define CAU_MODE_AES_CCM            CTL_ALGM(9)                                      /*!< AES-CCM (AES combined cipher machine mode) */
#define CAU_MODE_AES_CFB            CTL_ALGM(10)                                     /*!< AES-CFB (cipher feedback mode) */
#define CAU_MODE_AES_OFB            CTL_ALGM(11)                                     /*!< AES-OFB (output feedback mode) */

#define CTL_DATAM(regval)           (BITS(6,7) & ((uint32_t)(regval) << 6U))         /*!< write value to CAU_CTL_DATAM bit field */
#define CAU_SWAPPING_32BIT          CTL_DATAM(0)                                     /*!< no swapping */
#define CAU_SWAPPING_16BIT          CTL_DATAM(1)                                     /*!< half-word swapping */
#define CAU_SWAPPING_8BIT           CTL_DATAM(2)                                     /*!< bytes swapping */
#define CAU_SWAPPING_1BIT           CTL_DATAM(3)                                     /*!< bit swapping */

#define CAU_KEY                     ((uint32_t)0x00000000U)                          /*!< use the key from CAU register */
#define CAU_EFUSE_KEY               CAU_CTL_KEY_SEL                                  /*!< use the key from EFUSE */

#define CTL_KEYM(regval)            (BITS(8,9) & ((uint32_t)(regval) << 8U))         /*!< write value to CAU_CTL_KEYM bit field */
#define CAU_KEYSIZE_128BIT          CTL_KEYM(0)                                      /*!< 128 bit key length */
#define CAU_KEYSIZE_192BIT          CTL_KEYM(1)                                      /*!< 192 bit key length */
#define CAU_KEYSIZE_256BIT          CTL_KEYM(2)                                      /*!< 256 bit key length */

#define CTL_GCM_CCMPH(regval)       (BITS(16,17) & ((uint32_t)(regval) << 16U))      /*!< write value to CAU_CTL_GCM_CCMPH bit field */
#define CAU_PREPARE_PHASE           CTL_GCM_CCMPH(0)                                 /*!< prepare phase */
#define CAU_AAD_PHASE               CTL_GCM_CCMPH(1)                                 /*!< AAD phase */
#define CAU_ENCRYPT_DECRYPT_PHASE   CTL_GCM_CCMPH(2)                                 /*!< encryption/decryption phase */
#define CAU_TAG_PHASE               CTL_GCM_CCMPH(3)                                 /*!< tag phase */

#define CAU_PADDING_BYTES(regval)   (BITS(20,23) & ((uint32_t)(regval) << 20U))     /* Fill bytes */

/* cau_stat0 register value */
#define CAU_FLAG_INFIFO_EMPTY       CAU_STAT0_IEM                                    /*!< in fifo empty */
#define CAU_FLAG_INFIFO_NO_FULL     CAU_STAT0_INF                                    /*!< in fifo is not full */
#define CAU_FLAG_OUTFIFO_NO_EMPTY   CAU_STAT0_ONE                                    /*!< out fifo not empty */
#define CAU_FLAG_OUTFIFO_FULL       CAU_STAT0_OFU                                    /*!< out fifo is full */
#define CAU_FLAG_BUSY               CAU_STAT0_BUSY                                   /*!< the CAU core is busy */

/* cau_dmaen register value */
#define CAU_DMA_INFIFO              CAU_DMAEN_DMAIEN                                 /*!< DMA input enable */
#define CAU_DMA_OUTFIFO             CAU_DMAEN_DMAOEN                                 /*!< DMA output enable */

/* cau_inten register value */
#define CAU_INT_INFIFO              CAU_INTEN_IINTEN                                 /*!< in fifo Interrupt */
#define CAU_INT_OUTFIFO             CAU_INTEN_OINTEN                                 /*!< out fifo Interrupt */

/* cau_stat1 register value */
#define CAU_FLAG_INFIFO             CAU_STAT1_ISTA                                   /*!< in fifo flag status */
#define CAU_FLAG_OUTFIFO            CAU_STAT1_OSTA                                   /*!< out fifo flag status */

/* cau_intf register value */
#define CAU_INT_FLAG_INFIFO         CAU_INTF_IINTF                                   /*!< in fifo interrupt status */
#define CAU_INT_FLAG_OUTFIFO        CAU_INTF_OINTF                                   /*!< out fifo interrupt status */

/* @STRUCT_MEMBER: cau_dir */
/* @DEFINE: cau algorithm direction */
#define CAU_ENCRYPTION_DIRECTION    CAU_ENCRYPT                                      /*!< encrypt */
#define CAU_DECRYPTION_DIRECTION    CAU_DECRYPT                                      /*!< decrypt */

/* @STRUCT_MEMBER: key_source */
/* @DEFINE: key selection */
#define CAU_KEY_FROM_REGISTER       CAU_KEY                                          /*!< use the key from CAU register */
#define CAU_KEY_FROM_EFUSE          CAU_EFUSE_KEY                                    /*!< use the key from EFUSE */

/* @PARA: hal_struct_type */
/* @ENUM: CAU structure type enumeration */
typedef enum {
    HAL_CAU_INIT_STRUCT = 0U,                                                        /*!< CAU initialization structure */
    HAL_CAU_IRQ_STRUCT,                                                              /*!< CAU interrupt structure*/
    HAL_CAU_DMA_HANDLE_CB_STRUCT,                                                    /*!< CAU dma handle structure*/
    HAL_CAU_DEV_STRUCT,                                                              /*!< CAU device information structure */
    HAL_CAU_CONTEXT_PARAMETER_STRUCT,                                                /*!< CAU context parameter structure */
    HAL_CAU_IRQ_USER_CALLBACK_STRUCT,                                                /*!< CAU interrupt user callback structure */
    HAL_CAU_IV_PARAMETER_STRUCT,                                                     /*!< CAU IV parameter structure */
    HAL_CAU_KEY_PARAMETER_STRUCT,                                                    /*!< CAU key parameter structure */
    HAL_CAU_PARAMETER_STRUCT                                                         /*!< CAU parameter structure */
} hal_cau_struct_type_enum;

/* the callback of CAU interrupt declaration */
typedef void (*hal_cau_dma_handle_cb)(void *ptr);

/* CAU state type enum */
typedef enum {
    HAL_CAU_STATE_RESET = (uint32_t)0x00000000U,                                     /*!< CAU is not initialized or disabled */
    HAL_CAU_STATE_READY = (uint32_t)0x00000001U,                                     /*!< CAU is ready */
    HAL_CAU_STATE_BUSY  = (uint32_t)0x00000002U                                      /*!< CAU is busy */
} hal_cau_state_enum;

/* CAU phase enum */
typedef enum {
    HAL_CAU_PHASE_PREPARE         = 0U,                                              /*!< CAU prepare phase */
    HAL_CAU_PHASE_AAD             = 1U,                                              /*!< CAU aad phase */
    HAL_CAU_PHASE_ENCRYPT_DECRYPT = 2U,                                              /*!< CAU encrypt and decrypt phase */
    HAL_CAU_PHASE_TAG             = 3U                                               /*!< CAU tag phase */
} hal_cau_phase_enum;

/* CAU error type enum */
typedef enum {
    HAL_CAU_ERROR_NONE   = (uint32_t)0x00000000U,                                    /*!< no error */
    HAL_CAU_ERROR_SYSTEM = (uint32_t)0x00000001U,                                    /*!< CAU internal error: if problem of clocking, enable/disable, wrong state */
    HAL_CAU_ERROR_DMA    = (uint32_t)0x00000002U,                                    /*!< DMA transfer error */
    HAL_CAU_ERROR_CONFIG = (uint32_t)0x00000004U                                     /*!< configuration error occurs */
} hal_cau_error_enum;

/* @STRUCT_MEMBER: mode */
/* @ENUM:  mode */
typedef enum {
    CAU_TDES_ECB = CAU_MODE_TDES_ECB,                                                /*!< TDES-ECB (3DES electronic codebook) */
    CAU_TDES_CBC = CAU_MODE_TDES_CBC,                                                /*!< TDES-CBC (3DES Cipher block chaining) */
    CAU_DES_ECB  = CAU_MODE_DES_ECB,                                                 /*!< DES-ECB (simple DES Electronic codebook) */
    CAU_DES_CBC  = CAU_MODE_DES_CBC,                                                 /*!< DES-CBC (simple DES Cipher block chaining) */
    CAU_AES_ECB  = CAU_MODE_AES_ECB,                                                 /*!< AES-ECB (AES electronic codebook) */
    CAU_AES_CBC  = CAU_MODE_AES_CBC,                                                 /*!< AES-CBC (AES Cipher block chaining) */
    CAU_AES_CTR  = CAU_MODE_AES_CTR,                                                 /*!< AES-CTR (AES counter mode) */
    CAU_AES_KEY  = CAU_MODE_AES_KEY,                                                 /*!< AES decryption key preparation mode */
    CAU_AES_CCM  = CAU_MODE_AES_CCM,                                                 /*!< AES-GCM (AES galois/counter mode) */
    CAU_AES_GCM  = CAU_MODE_AES_GCM,                                                 /*!< AES-CCM (AES combined cipher machine mode) */
    CAU_AES_CFB  = CAU_MODE_AES_CFB,                                                 /*!< AES-CFB (cipher feedback mode) */
    CAU_AES_OFB  = CAU_MODE_AES_OFB                                                  /*!< AES-OFB (output feedback mode) */
} hal_cau_mode_enum;

/* @STRUCT_MEMBER: aes_key_length */
/* @ENUM: aes key length */
typedef enum {
    CAU_AES_KEY_LENGTH_128BIT = CAU_KEYSIZE_128BIT,                                  /*!< 128 bit key length */
    CAU_AES_KEY_LENGTH_192BIT = CAU_KEYSIZE_192BIT,                                  /*!< 192 bit key length */
    CAU_AES_KEY_LENGTH_256BIT = CAU_KEYSIZE_256BIT                                   /*!< 256 bit key length */
} hal_cau_aes_key_length_enum;

/* @STRUCT_MEMBER: data_exchange_mode */
/* @ENUM: data exchange mode */
typedef enum {
    CAU_DATA_EXCHANGE_32BIT = CAU_SWAPPING_32BIT,                                    /*!< no swapping */
    CAU_DATA_EXCHANGE_16BIT = CAU_SWAPPING_16BIT,                                    /*!< half-word swapping */
    CAU_DATA_EXCHANGE_8BIT  = CAU_SWAPPING_8BIT,                                     /*!< bytes swapping */
    CAU_DATA_EXCHANGE_1BIT  = CAU_SWAPPING_1BIT                                      /*!< bit swapping */
} hal_cau_data_exchange_mode_enum;

/* constants definitions */
/* structure for keys initialization of the cau */
typedef struct {
    uint32_t key_0_high;                                                             /*!< key 0 high */
    uint32_t key_0_low;                                                              /*!< key 0 low  */
    uint32_t key_1_high;                                                             /*!< key 1 high */
    uint32_t key_1_low;                                                              /*!< key 1 low  */
    uint32_t key_2_high;                                                             /*!< key 2 high */
    uint32_t key_2_low;                                                              /*!< key 2 low  */
    uint32_t key_3_high;                                                             /*!< key 3 high */
    uint32_t key_3_low;                                                              /*!< key 3 low  */
} hal_cau_key_parameter_struct;

/* structure for vectors initialization of the cau */
typedef struct {
    uint32_t iv_0_high;                                                              /*!< init vector 0 high */
    uint32_t iv_0_low;                                                               /*!< init vector 0 low  */
    uint32_t iv_1_high;                                                              /*!< init vector 1 high */
    uint32_t iv_1_low;                                                               /*!< init vector 1 low  */
} hal_cau_iv_parameter_struct;

/* structure for cau context swapping */
typedef struct {
    uint32_t ctl_config;                                                             /*!< current configuration */
    uint32_t iv_0_high;                                                              /*!< init vector 0 high */
    uint32_t iv_0_low;                                                               /*!< init vector 0 low  */
    uint32_t iv_1_high;                                                              /*!< init vector 1 high */
    uint32_t iv_1_low;                                                               /*!< init vector 1 low  */
    uint32_t key_0_high;                                                             /*!< key 0 high */
    uint32_t key_0_low;                                                              /*!< key 0 low  */
    uint32_t key_1_high;                                                             /*!< key 1 high */
    uint32_t key_1_low;                                                              /*!< key 1 low  */
    uint32_t key_2_high;                                                             /*!< key 2 high */
    uint32_t key_2_low;                                                              /*!< key 2 low  */
    uint32_t key_3_high;                                                             /*!< key 3 high */
    uint32_t key_3_low;                                                              /*!< key 3 low  */
    uint32_t gcmccmctxs[8];                                                          /*!< GCM or CCM mode context switch */
    uint32_t gcmctxs[8];                                                             /*!< GCM mode context switch */
} hal_cau_context_parameter_struct;

/* CAU device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb fifo_input_handle;                                        /*!< fifo input interrupt callback */
    __IO hal_irq_handle_cb fifo_output_handle;                                       /*!< fifo output interrupt callback */
} hal_cau_irq_struct;

/* CAU DMA interrupt callback function pointer structure */
typedef struct {
    __IO hal_cau_dma_handle_cb in_full_transcom_handle;                              /*!< CAU DMA transfer complete interrupt handler function */
    __IO hal_cau_dma_handle_cb in_half_transcom_handle;                              /*!< CAU DMA transfer complete interrupt handler function */
    __IO hal_cau_dma_handle_cb out_full_transcom_handle;                             /*!< CAU DMA transfer complete interrupt handler function */
    __IO hal_cau_dma_handle_cb out_half_transcom_handle;                             /*!< CAU DMA transfer complete interrupt handler function */
    __IO hal_cau_dma_handle_cb error_handle;                                         /*!< CAU error handler function */
} hal_cau_dma_handle_cb_struct;

/* CAU device interrupt user callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb fifo_input_callback;                                      /*!< fifo input interrupt callback */
    __IO hal_irq_handle_cb fifo_output_callback;                                     /*!< fifo output interrupt callback */
} hal_cau_irq_user_callback_struct;

/* structure for encrypt and decrypt parameters */
typedef struct {
    uint32_t alg_dir;                                                                /*!< algorithm directory */
    uint8_t *key;                                                                    /*!< key */
    uint32_t key_size;                                                               /*!< key size in bytes */
    uint8_t *iv;                                                                     /*!< initialization vector */
    uint32_t iv_size;                                                                /*!< iv size in bytes */
    uint8_t *input;                                                                  /*!< input data */
    uint32_t in_length;                                                              /*!< input data length in bytes */
    uint8_t *aad;                                                                    /*!< additional authentication data */
    uint32_t aad_size;                                                               /*!< aad size */
    hal_cau_data_exchange_mode_enum data_change_mode;                                /*!< data change mode */
} hal_cau_parameter_struct;

/* @PARA: p_init */
/* @STRUCT: CAU common settings struct */
typedef struct {
    hal_cau_mode_enum mode;                                                          /*!< cau mode */
    uint32_t cau_dir;                                                                /*!< cau algorithm direction */
    uint32_t key_source;                                                             /*!< key selection */
    hal_cau_aes_key_length_enum aes_key_length;                                      /*!< aes key length */
    uint8_t *key;                                                                    /*!< encryption/decryption Key */
    uint8_t *init_vector;                                                            /*!< initialization vector */
    uint8_t *aes_verify_key;                                                         /*!< aes verify key */
    hal_cau_data_exchange_mode_enum data_exchange_mode;                              /*!< data exchange mode */
} hal_cau_init_struct;

/* @PARA: cau_dev */
/* @STRUCT: CAU device information structure */
typedef struct {
    hal_cau_irq_struct cau_irq;                                                      /*!< CAU device interrupt callback function pointer structure */
    hal_cau_dma_handle_cb_struct cau_dma;                                            /*!< CAU DMA interrupt callback function pointer structure */
    hal_dma_dev_struct *p_dma_cau_in;                                                /*!< DMA device information structure for cau INFIFO*/
    hal_dma_dev_struct *p_dma_cau_out;                                               /*!< DMA device information structure for cau out OUTFIFO */
    hal_cau_error_enum error_code;                                                   /*!< CAU error state */
    hal_mutex_enum mutex;                                                            /*!< CAU mutex set */
    hal_cau_state_enum state;                                                        /*!< CAU state */
    hal_cau_phase_enum phase;                                                        /*!< CAU phase mode config for func(poll,interrupt,dma)*/
    hal_cau_parameter_struct *p_irq_para;                                            /*!< CAU parameter for AES-GCM,AES-CCM func*/
    uint32_t *in_buffer;                                                             /*!< CAU input buffer for encryption/decryption */
    uint32_t in_size;                                                                /*!< CAU input buffer size*/
    uint32_t *out_buffer;                                                            /*!< CAU output buffer for encryption/decryption */
    uint32_t out_size;                                                               /*!< CAU output buffer size*/
    uint8_t *aad_buffer;                                                             /*!< CAU additional authentication buffer */
    uint32_t aad_size;                                                               /*!< CAU additional authentication buffer size */
    uint8_t *tag;                                                                    /*!< CAU tag buffer */
    uint32_t tag_size;                                                               /*!< CAU tag buffer size*/
    void *in_full_transcom_callback;                                                 /*!< CAU DMA transfer complete interrupt handler function */
    void *in_half_transcom_callback;                                                 /*!< CAU DMA transfer complete interrupt handler function */
    void *out_full_transcom_callback;                                                /*!< CAU DMA transfer complete interrupt handler function */
    void *out_half_transcom_callback;                                                /*!< CAU DMA transfer complete interrupt handler function */
    void *error_callback;                                                            /*!< CAU error handler function */
    void *fifo_input_callback;                                                       /*!< fifo input interrupt callback */
    void *fifo_output_callback;                                                      /*!< fifo output interrupt callback */
} hal_cau_dev_struct;

/* @STRUCT_MEMBER: key */
/* @=NULL */

/* @STRUCT_MEMBER: init_vector */
/* @=NULL */

/* @STRUCT_MEMBER: aes_verify_key */
/* @=NULL */

/* function declarations */
/* initialization functions */
/* @FUNCTION: deinitialize CAU */
int32_t hal_cau_deinit(hal_cau_dev_struct *cau_dev);
/* @FUNCTION: initialize CAU */
int32_t hal_cau_init(hal_cau_dev_struct *cau_dev, hal_cau_init_struct *p_init);
/* @FUNCTION: initialize the CAU structure with default values */
int32_t hal_cau_struct_init(hal_cau_struct_type_enum hal_struct_type, void *p_struct);
/* @END */
/* CAU interrupt handler content function, which is merely used in CAU_IRQHandler */
void hal_cau_irq(hal_cau_dev_struct *cau_dev);
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_cau_irq_handle_set(hal_cau_dev_struct *cau_dev, hal_cau_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_cau_irq_handle_all_reset(hal_cau_dev_struct *cau_dev);

/* initialize the CAU encrypt and decrypt parameter struct with the default values */
void hal_cau_encrypt_decrypt_para_init(hal_cau_parameter_struct *cau_parameter);

/* context switch functions */
/* save context before context switching */
void hal_cau_context_save(hal_cau_context_parameter_struct *cau_context, hal_cau_key_parameter_struct *key_initpara);
/* restore context after context switching */
void hal_cau_context_restore(hal_cau_context_parameter_struct *cau_context);

/* cau stop encrypt and decrypt in interrupt mode */
int32_t hal_cau_interrupt_stop(hal_cau_dev_struct *cau_dev);
/* cau stop encrypt and decrypt in dma mode */
int32_t hal_cau_dma_stop(hal_cau_dev_struct *cau_dev);
/* cau encrypt and decrypt for poll */
ErrStatus hal_cau_encrypt_decrypt_poll(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                       uint8_t *output, uint32_t cau_mode);
/* cau encrypt and decrypt for interrupt */
ErrStatus hal_cau_encrypt_decrypt_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                            hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output, \
                                            uint32_t cau_mode);
/* cau encrypt and decrypt for dma */
int32_t hal_cau_encrypt_decrypt_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                    hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output, uint32_t cau_mode);

/* return whether CAU peripheral is enabled or disabled */
ControlStatus hal_cau_enable_state_get(void);
/* get CAU state */
hal_cau_state_enum hal_cau_state_get(hal_cau_dev_struct *cau_dev);
/* get CAU error code */
hal_cau_error_enum hal_cau_error_code_get(hal_cau_dev_struct *cau_dev);

/* configure key selection */
void hals_cau_aes_key_select(uint32_t key_selection);
/* encrypt and decrypt using AES in ECB mode */
ErrStatus hals_cau_aes_ecb(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output);
/* encrypt and decrypt using AES in CBC mode */
ErrStatus hals_cau_aes_cbc(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output);
/* encrypt and decrypt using AES in CTR mode */
ErrStatus hals_cau_aes_ctr(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output);
/* encrypt and decrypt using AES in CFB mode */
ErrStatus hals_cau_aes_cfb(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output);
/* encrypt and decrypt using AES in OFB mode */
ErrStatus hals_cau_aes_ofb(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output);
/* encrypt and decrypt using AES in GCM mode */
ErrStatus hals_cau_aes_gcm(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output);
/* encrypt and decrypt using AES in CCM mode */
ErrStatus hals_cau_aes_ccm(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output);
/* encrypt and decrypt using TDES in ECB mode */
ErrStatus hals_cau_tdes_ecb(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output);
/* encrypt and decrypt using TDES in CBC mode */
ErrStatus hals_cau_tdes_cbc(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output);
/* encrypt and decrypt using DES in ECB mode */
ErrStatus hals_cau_des_ecb(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output);
/* encrypt and decrypt using DES in CBC mode */
ErrStatus hals_cau_des_cbc(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, uint8_t *output);
/* generate tag using AES in CCM mode */
ErrStatus hals_cau_aes_ccm_generate_tag(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter);
/* generate tag using AES in GCM mode */
ErrStatus hals_cau_aes_gcm_generate_tag(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter);

/* encrypt and decrypt using AES in ECB mode for interrupt */
ErrStatus hals_cau_aes_ecb_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                     hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output);
/* encrypt and decrypt using AES in CBC mode for interrupt */
ErrStatus hals_cau_aes_cbc_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                     hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output);
/* encrypt and decrypt using AES in CTR mode for interrupt */
void hals_cau_aes_ctr_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output);
/* encrypt and decrypt using AES in CFB mode for interrupt*/
void hals_cau_aes_cfb_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output);
/* encrypt and decrypt using AES in OFB mode for interrupt */
void hals_cau_aes_ofb_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output);
/* encrypt and decrypt using TDES in ECB mode for interrupt */
void hals_cau_tdes_ecb_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                 hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output);
/* encrypt and decrypt using TDES in CBC mode for interrupt */
void hals_cau_tdes_cbc_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                 hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output);
/* encrypt and decrypt using DES in ECB mode for interrupt */
void hals_cau_des_ecb_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output);
/* encrypt and decrypt using DES in CBC mode for interrupt */
void hals_cau_des_cbc_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output);
/* encrypt and decrypt using AES in CCM mode for interrupt */
void hals_cau_aes_ccm_gcm_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                    hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output);

/* encrypt and decrypt using AES in ECB mode for DMA */
ErrStatus hals_cau_aes_ecb_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                               hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output);
/* encrypt and decrypt using AES in CBC mode for DMA */
ErrStatus hals_cau_aes_cbc_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                               hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output);
/* encrypt and decrypt using AES in CTR mode for DMA */
void hals_cau_aes_ctr_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                          hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output);
/* encrypt and decrypt using AES in CFB mode for DMA */
void hals_cau_aes_cfb_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                          hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output);
/* encrypt and decrypt using AES in OFB mode for DMA */
void hals_cau_aes_ofb_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                          hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output);
/* encrypt and decrypt using TDES in ECB mode for DMA */
void hals_cau_tdes_ecb_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                           hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output);
/* encrypt and decrypt using TDES in CBC mode for DMA */
void hals_cau_tdes_cbc_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                           hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output);
/* encrypt and decrypt using DES in ECB mode for DMA */
void hals_cau_des_ecb_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                          hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output);
/* encrypt and decrypt using DES in CBC mode for DMA */
void hals_cau_des_cbc_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                          hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output);

/* initialize the vectors parameter struct with the default values */
void hals_cau_iv_struct_para_init(hal_cau_iv_parameter_struct *iv_initpara);
/* initialize the context parameter struct with the default values */
void hals_cau_context_struct_para_init(hal_cau_context_parameter_struct *cau_context);
/* initialize the CAU peripheral */
void hals_cau_init_param(uint32_t alg_dir, uint32_t algo_mode, hal_cau_data_exchange_mode_enum swapping);
/* get the current parameters of the CAU peripheral */
void hals_cau_get_current_param(uint32_t *alg_dir, uint32_t *algo_mode, hal_cau_data_exchange_mode_enum *swapping);
/* initialize the key parameters */
void hals_cau_key_init(hal_cau_key_parameter_struct *key_initpara);
/* initialize the vectors parameters */
void hals_cau_iv_init(hal_cau_iv_parameter_struct *iv_initpara);
/* configure phase */
void hals_cau_phase_config(uint32_t phase);
/* CAU dma transfer param config */
void hals_cau_dma_transfer_config(hal_cau_dev_struct *cau_dev, hal_cau_dma_handle_cb_struct *dmacb);
/* read and write functions */
/* write data to the in fifo */
void hals_cau_data_write(uint32_t data);
/* return the last data entered into the output FIFO */
uint32_t hals_cau_data_read(void);
/* enable the CAU peripheral */
void hals_cau_enable(void);
/* disable the CAU peripheral */
void hals_cau_disable(void);
/* initialize the key parameter struct with the default values */
void hals_cau_key_struct_para_init(hal_cau_key_parameter_struct *key_initpara);
/* get the CAU flag status */
FlagStatus hals_cau_flag_get(uint32_t flag);
/* enable the CAU interrupts */
void hals_cau_interrupt_enable(uint32_t interrupt);
/* disable the CAU interrupts */
void hals_cau_interrupt_disable(uint32_t interrupt);
/* get the interrupt flag */
FlagStatus hals_cau_interrupt_flag_get(uint32_t int_flag);

#endif /* GD32H7XX_HAL_CAU_H */
