/*!
    \file    gd32h7xx_hal_adc.h
    \brief   definitions for the ADC

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_ADC_H
#define GD32H7XX_HAL_ADC_H

#include "gd32h7xx_hal.h"

/* ADC definitions */
#define ADC0                                         ADC_BASE                                                      /*!< ADC0 base address */
#define ADC1                                         (ADC_BASE + 0x00000400U)                                      /*!< ADC1 base address */
#define ADC2                                         (ADC_BASE + 0x00000800U)                                      /*!< ADC2 base address */

/* registers definitions */
#define ADC_STAT(adcx)                               REG32((adcx) + 0x00000000U)                                   /*!< ADC status register */
#define ADC_CTL0(adcx)                               REG32((adcx) + 0x00000004U)                                   /*!< ADC control register 0 */
#define ADC_CTL1(adcx)                               REG32((adcx) + 0x00000008U)                                   /*!< ADC control register 1 */
#define ADC_WDHT0(adcx)                              REG32((adcx) + 0x0000001CU)                                   /*!< ADC watchdog high threshold register 0 */
#define ADC_WDLT0(adcx)                              REG32((adcx) + 0x00000020U)                                   /*!< ADC watchdog low threshold register 0 */
#define ADC_RSQ0(adcx)                               REG32((adcx) + 0x00000024U)                                   /*!< ADC routine sequence register 0 */
#define ADC_RSQ1(adcx)                               REG32((adcx) + 0x00000028U)                                   /*!< ADC routine sequence register 1 */
#define ADC_RSQ2(adcx)                               REG32((adcx) + 0x0000002CU)                                   /*!< ADC routine sequence register 2 */
#define ADC_RSQ3(adcx)                               REG32((adcx) + 0x00000030U)                                   /*!< ADC routine sequence register 3 */
#define ADC_RSQ4(adcx)                               REG32((adcx) + 0x00000034U)                                   /*!< ADC routine sequence register 4 */
#define ADC_RSQ5(adcx)                               REG32((adcx) + 0x00000038U)                                   /*!< ADC routine sequence register 5 */
#define ADC_RSQ6(adcx)                               REG32((adcx) + 0x0000003CU)                                   /*!< ADC routine sequence register 6 */
#define ADC_RSQ7(adcx)                               REG32((adcx) + 0x00000040U)                                   /*!< ADC routine sequence register 7 */
#define ADC_RSQ8(adcx)                               REG32((adcx) + 0x00000044U)                                   /*!< ADC routine sequence register 8 */
#define ADC_IDATA0(adcx)                             REG32((adcx) + 0x00000054U)                                   /*!< ADC inserted data register 0 */
#define ADC_IDATA1(adcx)                             REG32((adcx) + 0x00000058U)                                   /*!< ADC inserted data register 1 */
#define ADC_IDATA2(adcx)                             REG32((adcx) + 0x0000005CU)                                   /*!< ADC inserted data register 2 */
#define ADC_IDATA3(adcx)                             REG32((adcx) + 0x00000060U)                                   /*!< ADC inserted data register 3 */
#define ADC_RDATA(adcx)                              REG32((adcx) + 0x00000064U)                                   /*!< ADC routine data register */
#define ADC_OVSAMPCTL(adcx)                          REG32((adcx) + 0x00000080U)                                   /*!< ADC oversampling control register */
#define ADC_WD1SR(adcx)                              REG32((adcx) + 0x000000A0U)                                   /*!< ADC watchdog 1 channel selection register */
#define ADC_WD2SR(adcx)                              REG32((adcx) + 0x000000A4U)                                   /*!< ADC watchdog 2 channel selection register */
#define ADC_WDHT1(adcx)                              REG32((adcx) + 0x000000A8U)                                   /*!< ADC watchdog high threshold register 1 */
#define ADC_WDLT1(adcx)                              REG32((adcx) + 0x000000ACU)                                   /*!< ADC watchdog low threshold register 1 */
#define ADC_WDHT2(adcx)                              REG32((adcx) + 0x000000B0U)                                   /*!< ADC watchdog high threshold register 2 */
#define ADC_WDLT2(adcx)                              REG32((adcx) + 0x000000B4U)                                   /*!< ADC watchdog low threshold register 2 */
#define ADC_DIFCTL(adcx)                             REG32((adcx) + 0x000000B8U)                                   /*!< ADC differential mode control register */
#define ADC_SSTAT                                    REG32((ADC0) + 0x00000300U)                                   /*!< ADC summary status register */
#define ADC_SYNCCTL(adcx)                            REG32((adcx) + 0x00000304U)                                   /*!< ADC sync control register */
#define ADC_SYNCDATA0                                REG32((ADC0) + 0x00000308U)                                   /*!< ADC sync routine data register 0 */
#define ADC_SYNCDATA1                                REG32((ADC0) + 0x0000030CU)                                   /*!< ADC sync routine data register 1 */

/* bits definitions */
/* ADC_STAT */
#define ADC_STAT_WDE0                                BIT(0)                                                        /*!< analog watchdog 0 event flag */
#define ADC_STAT_EOC                                 BIT(1)                                                        /*!< end of sequence conversion flag */
#define ADC_STAT_EOIC                                BIT(2)                                                        /*!< end of inserted sequence conversion flag */
#define ADC_STAT_STIC                                BIT(3)                                                        /*!< start flag of inserted channel sequence */
#define ADC_STAT_STRC                                BIT(4)                                                        /*!< start flag of routine channel sequence */
#define ADC_STAT_ROVF                                BIT(5)                                                        /*!< routine data register overflow */
#define ADC_STAT_WDE1                                BIT(30)                                                       /*!< analog watchdog 1 event flag */
#define ADC_STAT_WDE2                                BIT(31)                                                       /*!< analog watchdog 2 event flag */

/* ADC_CTL0 */
#define ADC_CTL0_WD0CHSEL                            BITS(0,4)                                                     /*!< analog watchdog channel select */
#define ADC_CTL0_EOCIE                               BIT(5)                                                        /*!< interrupt enable for EOC */
#define ADC_CTL0_WDE0IE                              BIT(6)                                                        /*!< interrupt enable for WDE0 */
#define ADC_CTL0_EOICIE                              BIT(7)                                                        /*!< interrupt enable for EOIC */
#define ADC_CTL0_SM                                  BIT(8)                                                        /*!< scan mode */
#define ADC_CTL0_WD0SC                               BIT(9)                                                        /*!< when in scan mode, analog watchdog 0 is effective on a single channel */
#define ADC_CTL0_ICA                                 BIT(10)                                                       /*!< inserted channel sequence convert automatically */
#define ADC_CTL0_DISRC                               BIT(11)                                                       /*!< discontinuous mode on routine channels */
#define ADC_CTL0_DISIC                               BIT(12)                                                       /*!< discontinuous mode on inserted channels */
#define ADC_CTL0_DISNUM                              BITS(13,15)                                                   /*!< number of conversions in discontinuous mode */
#define ADC_CTL0_IWD0EN                              BIT(22)                                                       /*!< inserted channel analog watchdog 0 enable */
#define ADC_CTL0_RWD0EN                              BIT(23)                                                       /*!< routine channel analog watchdog 0 enable */
#define ADC_CTL0_DRES                                BITS(24,25)                                                   /*!< ADC data resolution */
#define ADC_CTL0_ROVFIE                              BIT(26)                                                       /*!< interrupt enable for ROVF */
#define ADC_CTL0_WDE1IE                              BIT(30)                                                       /*!< interrupt enable for WDE1 */
#define ADC_CTL0_WDE2IE                              BIT(31)                                                       /*!< interrupt enable for WDE2 */

/* ADC_CTL1 */
#define ADC_CTL1_ADCON                               BIT(0)                                                        /*!< ADC on */
#define ADC_CTL1_CTN                                 BIT(1)                                                        /*!< continuous mode */
#define ADC_CTL1_CLB                                 BIT(2)                                                        /*!< ADC calibration */
#define ADC_CTL1_RSTCLB                              BIT(3)                                                        /*!< reset calibration */
#define ADC_CTL1_CALNUM                              BITS(4,6)                                                     /*!< ADC calibration times */
#define ADC_CTL1_DMA                                 BIT(8)                                                        /*!< DMA request enable */
#define ADC_CTL1_DDM                                 BIT(9)                                                        /*!< DMA disable mode */
#define ADC_CTL1_EOCM                                BIT(10)                                                       /*!< end of conversion mode */
#define ADC_CTL1_DAL                                 BIT(11)                                                       /*!< data alignment */
#define ADC_CTL1_HPDFCFG                             BIT(12)                                                       /*!< HPDF mode configuration */
#define ADC_CTL1_ETMIC                               BITS(20,21)                                                   /*!< external trigger mode for inserted channels */
#define ADC_CTL1_SWICST                              BIT(22)                                                       /*!< software start on inserted channel */
#define ADC_CTL1_TSVEN1                              BIT(23)                                                       /*!< temperature sensor channel enable */
#define ADC_CTL1_INREFEN                             BIT(24)                                                       /*!< vrefint channel enable */
#define ADC_CTL1_VBATEN                              BIT(25)                                                       /*!< vbat channel enable */
#define ADC_CTL1_CALMOD                              BIT(27)                                                       /*!< ADC calibration mode */
#define ADC_CTL1_ETMRC                               BITS(28,29)                                                   /*!< external trigger mode for routine channels */
#define ADC_CTL1_SWRCST                              BIT(30)                                                       /*!< software start on routine channel. */
#define ADC_CTL1_TSVEN2                              BIT(31)                                                       /*!< high-precision temperature sensor channel enable */

/* ADC_WDHT0 */
#define ADC_WDHT0_WDHT0                              BITS(0,23)                                                    /*!< high threshold for analog watchdog 0 */

/* ADC_WDLT0 */
#define ADC_WDLT0_WDLT0                              BITS(0,23)                                                    /*!< low threshold for analog watchdog 0 */

/* ADC_RSQx x=0..8 */
#define ADC_RSQX_RSQN                                BITS(0,4)                                                     /*!< nth conversion channel number in the routine channel sequence */
#define ADC_RSQX_RSMPN                               BITS(5,14)                                                    /*!< nth conversion sample time in the routine channel sequence */
#define ADC_RSQ0_RL                                  BITS(20,23)                                                   /*!< routine channel sequence length */

/* ADC_RDATA */
#define ADC_RDATA_RDATA                              BITS(0,31)                                                    /*!< routine channel data */

/* ADC_OVSAMPCTL */
#define ADC_OVSAMPCTL_OVSEN                          BIT(0)                                                        /*!< oversampling enable */
#define ADC_OVSAMPCTL_OVSS                           BITS(5,8)                                                     /*!< oversampling shift */
#define ADC_OVSAMPCTL_TOVS                           BIT(9)                                                        /*!< triggered oversampling */
#define ADC_OVSAMPCTL_OVSR                           BITS(16,25)                                                   /*!< oversampling ratio */

/* ADC_WD1SR */
#define ADC_WD1SR_AWD1CS                             BITS(0,21)                                                    /*!< analog watchdog 1 channel selection */

/* ADC_WD2SR */
#define ADC_WD2SR_AWD2CS                             BITS(0,21)                                                    /*!< analog watchdog 2 channel selection */

/* ADC_WDHT1 */
#define ADC_WDHT1_WDHT1                              BITS(0,23)                                                    /*!< high threshold for analog watchdog 1 */

/* ADC_WDHT1 */
#define ADC_WDLT1_WDLT1                              BITS(0,23)                                                    /*!< low threshold for analog watchdog 1 */

/* ADC_WDHT2 */
#define ADC_WDHT2_WDHT2                              BITS(0,23)                                                    /*!< high threshold for analog watchdog 2 */

/* ADC_WDHT2 */
#define ADC_WDLT2_WDLT2                              BITS(0,23)                                                    /*!< low threshold for analog watchdog 2 */

/* ADC_DIFCTL */
#define ADC_DIFCTL_DIFCTL                            BITS(0,21)                                                    /*!< Differential mode for channel 0..21 */

/* ADC_SSTAT */
#define ADC_SSTAT_ADC0_WDE0                          BIT(0)                                                        /*!< the bit is mirror image of the WDE0 bit of ADC0 */
#define ADC_SSTAT_ADC0_WDE1                          BIT(1)                                                        /*!< the bit is mirror image of the WDE1 bit of ADC0 */
#define ADC_SSTAT_ADC0_WDE2                          BIT(2)                                                        /*!< the bit is mirror image of the WDE2 bit of ADC0 */
#define ADC_SSTAT_ADC0_EOC                           BIT(3)                                                        /*!< the bit is mirror image of the EOC bit of ADC0 */
#define ADC_SSTAT_ADC0_EOIC                          BIT(4)                                                        /*!< the bit is mirror image of the EOIC bit of ADC0 */
#define ADC_SSTAT_ADC0_STIC                          BIT(5)                                                        /*!< the bit is mirror image of the STIC bit of ADC0 */
#define ADC_SSTAT_ADC0_STRC                          BIT(6)                                                        /*!< the bit is mirror image of the STRC bit of ADC0 */
#define ADC_SSTAT_ADC0_ROVF                          BIT(7)                                                        /*!< the bit is mirror image of the ROVF bit of ADC0 */
#define ADC_SSTAT_ADC1_WDE0                          BIT(8)                                                        /*!< the bit is mirror image of the WDE0 bit of ADC1 */
#define ADC_SSTAT_ADC1_WDE1                          BIT(9)                                                        /*!< the bit is mirror image of the WDE1 bit of ADC1 */
#define ADC_SSTAT_ADC1_WDE2                          BIT(10)                                                       /*!< the bit is mirror image of the WDE2 bit of ADC1 */
#define ADC_SSTAT_ADC1_EOC                           BIT(11)                                                       /*!< the bit is mirror image of the EOC bit of ADC1 */
#define ADC_SSTAT_ADC1_EOIC                          BIT(12)                                                       /*!< the bit is mirror image of the EOIC bit of ADC1 */
#define ADC_SSTAT_ADC1_STIC                          BIT(13)                                                       /*!< the bit is mirror image of the STIC bit of ADC1 */
#define ADC_SSTAT_ADC1_STRC                          BIT(14)                                                       /*!< the bit is mirror image of the STRC bit of ADC1 */
#define ADC_SSTAT_ADC1_ROVF                          BIT(15)                                                       /*!< the bit is mirror image of the ROVF bit of ADC1 */
#define ADC_SSTAT_ADC2_WDE0                          BIT(16)                                                       /*!< the bit is mirror image of the WDE0 bit of ADC2 */
#define ADC_SSTAT_ADC2_WDE1                          BIT(17)                                                       /*!< the bit is mirror image of the WDE1 bit of ADC2 */
#define ADC_SSTAT_ADC2_WDE2                          BIT(18)                                                       /*!< the bit is mirror image of the WDE2 bit of ADC2 */
#define ADC_SSTAT_ADC2_EOC                           BIT(19)                                                       /*!< the bit is mirror image of the EOC bit of ADC2 */
#define ADC_SSTAT_ADC2_EOIC                          BIT(20)                                                       /*!< the bit is mirror image of the EOIC bit of ADC2 */
#define ADC_SSTAT_ADC2_STIC                          BIT(21)                                                       /*!< the bit is mirror image of the STIC bit of ADC2 */
#define ADC_SSTAT_ADC2_STRC                          BIT(22)                                                       /*!< the bit is mirror image of the STRC bit of ADC2 */
#define ADC_SSTAT_ADC2_ROVF                          BIT(23)                                                       /*!< the bit is mirror image of the ROVF bit of ADC2 */

/* ADC_SYNCCTL */
#define ADC_SYNCCTL_SYNCM                            BITS(0,3)                                                     /*!< ADC sync mode */
#define ADC_SYNCCTL_SYNCDLY                          BITS(8,11)                                                    /*!< ADC sync delay */
#define ADC_SYNCCTL_SYNCDDM                          BIT(13)                                                       /*!< ADC sync DMA disable mode */
#define ADC_SYNCCTL_SYNCDMA                          BITS(14,15)                                                   /*!< ADC sync DMA mode selection */
#define ADC_SYNCCTL_ADCSCK                           BITS(16,19)                                                   /*!< ADC sync clock mode */
#define ADC_SYNCCTL_ADCCK                            BITS(20,23)                                                   /*!< ADC clock prescaler */

/* ADC_SYNCDATA0 */
#define ADC_SYNCDATA0_SYNCDATA0                      BITS(0,15)                                                    /*!< ADC0 routine data in ADC synchronization mode */
#define ADC_SYNCDATA0_SYNCDATA1                      BITS(16,31)                                                   /*!< ADC1 routine data in ADC synchronization mode */

/* ADC_SYNCDATA1 */
#define ADC_SYNCDATA1_SYNCDATA                       BITS(0,31)                                                    /*!< ADC0 routine data in ADC synchronization mode */

/* number of conversions in discontinuous mode */
#define CTL0_DISNUM(regval)                          (BITS(13,15) & ((uint32_t)(regval) << 13U))                   /*!< write value to ADC_CTL0_DISNUM bit field */

/* ADC calibration times */
#define CTL1_CALNUM(regval)                          (BITS(4,6) & ((uint32_t)(regval) << 4U))                      /*!< write value to ADC_CTL1_CLBNUM bit field */
#define ADC_CALIBRATION_NUM1                         CTL1_CALNUM(0)                                                /*!< ADC calibration 1 time */
#define ADC_CALIBRATION_NUM2                         CTL1_CALNUM(1)                                                /*!< ADC calibration 2 times */
#define ADC_CALIBRATION_NUM4                         CTL1_CALNUM(2)                                                /*!< ADC calibration 4 times */
#define ADC_CALIBRATION_NUM8                         CTL1_CALNUM(3)                                                /*!< ADC calibration 8 times */
#define ADC_CALIBRATION_NUM16                        CTL1_CALNUM(4)                                                /*!< ADC calibration 16 times */
#define ADC_CALIBRATION_NUM32                        CTL1_CALNUM(5)                                                /*!< ADC calibration 32 times */

/* ADC high threshold for analog watchdog 0 */
#define WDHT0_WDHT0(regval)                          (BITS(0,23) & ((uint32_t)(regval) << 0U))                     /*!< write value to ADC_WDHT0_WDHT0 bit field */

/* ADC low threshold for analog watchdog 0 */
#define WDLT0_WDLT0(regval)                          (BITS(0,23) & ((uint32_t)(regval) << 0U))                     /*!< write value to ADC_WDLT0_WDLT0 bit field */

/* ADC high threshold for analog watchdog 1 */
#define WDHT1_WDHT1(regval)                          (BITS(0,23) & ((uint32_t)(regval) << 0U))                     /*!< write value to ADC_WDHT1_WDHT1 bit field */

/* ADC low threshold for analog watchdog 1 */
#define WDLT1_WDLT1(regval)                          (BITS(0,23) & ((uint32_t)(regval) << 0U))                     /*!< write value to ADC_WDLT1_WDLT1 bit field */

/* ADC high threshold for analog watchdog 2 */
#define WDHT2_WDHT2(regval)                          (BITS(0,23) & ((uint32_t)(regval) << 0U))                     /*!< write value to ADC_WDHT2_WDHT2 bit field */

/* ADC low threshold for analog watchdog 2 */
#define WDLT2_WDLT2(regval)                          (BITS(0,23) & ((uint32_t)(regval) << 0U))                     /*!< write value to ADC_WDLT2_WDLT2 bit field */

/* ADC sequence sample time */
#define SQX_SMP(regval)                              (BITS(5,14) & ((uint32_t)(regval) << 5U))                     /*!< write value to RSQX_SMPn or ISQX_SMPn bit field */

/* ADC routine channel sequence length */
#define RSQ0_RL(regval)                              (BITS(20,23) & ((uint32_t)(regval) << 20U))                   /*!< write value to ADC_RSQ0_RL bit field */

/* ADC resolution */
#define CTL0_DRES(regval)                            (BITS(24,25) & ((uint32_t)(regval) << 24U))                   /*!< write value to ADC_CTL0_DRES bit field */
#define ADC_RESOLUTION_14B                           ((uint8_t)0x00U)                                              /*!< 14-bit ADC resolution */
#define ADC_RESOLUTION_12B                           ((uint8_t)0x01U)                                              /*!< 12-bit ADC resolution */
#define ADC_RESOLUTION_10B                           ((uint8_t)0x02U)                                              /*!< 10-bit ADC resolution */
#define ADC_RESOLUTION_8B                            ((uint8_t)0x03U)                                              /*!< 8-bit ADC resolution */
#define ADC_RESOLUTION_6B                            ((uint8_t)0x04U)                                              /*!< 6-bit ADC resolution */

/*ADC oversampling shift */
#define OVSCR_OVSS(regval)                           (BITS(5,8) & ((uint32_t)(regval) << 5U))                      /*!< write value to ADC_OVSAMPCTL_OVSS bit field */
#define ADC_OVERSAMPLING_SHIFT_NONE                  OVSCR_OVSS(0)                                                 /*!< no oversampling shift */
#define ADC_OVERSAMPLING_SHIFT_1B                    OVSCR_OVSS(1)                                                 /*!< 1-bit oversampling shift */
#define ADC_OVERSAMPLING_SHIFT_2B                    OVSCR_OVSS(2)                                                 /*!< 2-bit oversampling shift */
#define ADC_OVERSAMPLING_SHIFT_3B                    OVSCR_OVSS(3)                                                 /*!< 3-bit oversampling shift */
#define ADC_OVERSAMPLING_SHIFT_4B                    OVSCR_OVSS(4)                                                 /*!< 4-bit oversampling shift */
#define ADC_OVERSAMPLING_SHIFT_5B                    OVSCR_OVSS(5)                                                 /*!< 5-bit oversampling shift */
#define ADC_OVERSAMPLING_SHIFT_6B                    OVSCR_OVSS(6)                                                 /*!< 6-bit oversampling shift */
#define ADC_OVERSAMPLING_SHIFT_7B                    OVSCR_OVSS(7)                                                 /*!< 7-bit oversampling shift */
#define ADC_OVERSAMPLING_SHIFT_8B                    OVSCR_OVSS(8)                                                 /*!< 8-bit oversampling shift */
#define ADC_OVERSAMPLING_SHIFT_9B                    OVSCR_OVSS(9)                                                 /*!< 9-bit oversampling shift */
#define ADC_OVERSAMPLING_SHIFT_10B                   OVSCR_OVSS(10)                                                /*!< 10-bit oversampling shift */
#define ADC_OVERSAMPLING_SHIFT_11B                   OVSCR_OVSS(11)                                                /*!< 11-bit oversampling shift */

/* ADC oversampling ratio */
#define OVSCR_OVSR(regval)                           (BITS(16,25) & ((uint32_t)(regval) << 16))                    /*!< ADC oversampling ratio */

/* configure the ADC clock */
#define SYNCCTL_ADCSCK(regval)                       (BITS(16,19) & ((uint32_t)(regval) << 16U))                   /*!< write value to ADC_SYNCCTL_ADCSCK bit field */
#define SYNCCTL_ADCCK(regval)                        (BITS(20,23) & ((uint32_t)(regval) << 20U))                   /*!< write value to ADC_SYNCCTL_ADCCK bit field */
#define ADC_CLK_SYNC_HCLK_DIV2                       (SYNCCTL_ADCCK(0) | SYNCCTL_ADCSCK(8))                        /*!< ADC sync clock mode HCLK div2 */
#define ADC_CLK_SYNC_HCLK_DIV4                       (SYNCCTL_ADCCK(0) | SYNCCTL_ADCSCK(9))                        /*!< ADC sync clock mode HCLK div4 */
#define ADC_CLK_SYNC_HCLK_DIV6                       (SYNCCTL_ADCCK(0) | SYNCCTL_ADCSCK(10))                       /*!< ADC sync clock mode HCLK div6 */
#define ADC_CLK_SYNC_HCLK_DIV8                       (SYNCCTL_ADCCK(0) | SYNCCTL_ADCSCK(11))                       /*!< ADC sync clock mode HCLK div8 */
#define ADC_CLK_SYNC_HCLK_DIV10                      (SYNCCTL_ADCCK(0) | SYNCCTL_ADCSCK(12))                       /*!< ADC sync clock mode HCLK div10 */
#define ADC_CLK_SYNC_HCLK_DIV12                      (SYNCCTL_ADCCK(0) | SYNCCTL_ADCSCK(13))                       /*!< ADC sync clock mode HCLK div12 */
#define ADC_CLK_SYNC_HCLK_DIV14                      (SYNCCTL_ADCCK(0) | SYNCCTL_ADCSCK(14))                       /*!< ADC sync clock mode HCLK div14 */
#define ADC_CLK_SYNC_HCLK_DIV16                      (SYNCCTL_ADCCK(0) | SYNCCTL_ADCSCK(15))                       /*!< ADC sync clock mode HCLK div16 */
#define ADC_CLK_ASYNC_DIV1                           (SYNCCTL_ADCCK(0) | SYNCCTL_ADCSCK(0))                        /*!< ADC async clock mode div1 */
#define ADC_CLK_ASYNC_DIV2                           (SYNCCTL_ADCCK(1) | SYNCCTL_ADCSCK(0))                        /*!< ADC async clock mode div2 */
#define ADC_CLK_ASYNC_DIV4                           (SYNCCTL_ADCCK(2) | SYNCCTL_ADCSCK(0))                        /*!< ADC async clock mode div4 */
#define ADC_CLK_ASYNC_DIV6                           (SYNCCTL_ADCCK(3) | SYNCCTL_ADCSCK(0))                        /*!< ADC async clock mode div6 */
#define ADC_CLK_ASYNC_DIV8                           (SYNCCTL_ADCCK(4) | SYNCCTL_ADCSCK(0))                        /*!< ADC async clock mode div8 */
#define ADC_CLK_ASYNC_DIV10                          (SYNCCTL_ADCCK(5) | SYNCCTL_ADCSCK(0))                        /*!< ADC async clock mode div10 */
#define ADC_CLK_ASYNC_DIV12                          (SYNCCTL_ADCCK(6) | SYNCCTL_ADCSCK(0))                        /*!< ADC async clock mode div12 */
#define ADC_CLK_ASYNC_DIV16                          (SYNCCTL_ADCCK(7) | SYNCCTL_ADCSCK(0))                        /*!< ADC async clock mode div16 */
#define ADC_CLK_ASYNC_DIV32                          (SYNCCTL_ADCCK(8) | SYNCCTL_ADCSCK(0))                        /*!< ADC async clock mode div32 */
#define ADC_CLK_ASYNC_DIV64                          (SYNCCTL_ADCCK(9) | SYNCCTL_ADCSCK(0))                        /*!< ADC async clock mode div64 */
#define ADC_CLK_ASYNC_DIV128                         (SYNCCTL_ADCCK(10) | SYNCCTL_ADCSCK(0))                       /*!< ADC async clock mode div128 */
#define ADC_CLK_ASYNC_DIV256                         (SYNCCTL_ADCCK(11) | SYNCCTL_ADCSCK(0))                       /*!< ADC async clock mode div256 */

/* ADC sync DMA mode selection */
#define SYNCCTL_SYNCDMA(regval)                      (BITS(14,15) & ((uint32_t)(regval) << 14U))                   /*!< write value to ADC_SYNCCTL_SYNCDMA bit field */
#define ADC_SYNC_DMA_DISABLE                         SYNCCTL_SYNCDMA(0)                                            /*!< ADC sync DMA disabled */
#define ADC_SYNC_DMA_MODE0                           SYNCCTL_SYNCDMA(1)                                            /*!< ADC sync DMA mode 0 */
#define ADC_SYNC_DMA_MODE1                           SYNCCTL_SYNCDMA(2)                                            /*!< ADC sync DMA mode 1 */

/* ADC sync delay */
#define SYNCCTL_SYNCDLY(regval)                      (BITS(8,11) & ((uint32_t)(regval) << 8U))                     /*!< write value to ADC_SYNCCTL_SYNCDLY bit field */
#define ADC_SYNC_DELAY_5CYCLE                        SYNCCTL_SYNCDLY(0)                                            /*!< the delay between 2 sampling phases in ADC synchronization modes to 5 ADC clock cycles. */
#define ADC_SYNC_DELAY_6CYCLE                        SYNCCTL_SYNCDLY(1)                                            /*!< the delay between 2 sampling phases in ADC synchronization modes to 6 ADC clock cycles. */
#define ADC_SYNC_DELAY_7CYCLE                        SYNCCTL_SYNCDLY(2)                                            /*!< the delay between 2 sampling phases in ADC synchronization modes to 7 ADC clock cycles. */
#define ADC_SYNC_DELAY_8CYCLE                        SYNCCTL_SYNCDLY(3)                                            /*!< the delay between 2 sampling phases in ADC synchronization modes to 8 ADC clock cycles. */
#define ADC_SYNC_DELAY_9CYCLE                        SYNCCTL_SYNCDLY(4)                                            /*!< the delay between 2 sampling phases in ADC synchronization modes to 9 ADC clock cycles. */
#define ADC_SYNC_DELAY_10CYCLE                       SYNCCTL_SYNCDLY(5)                                            /*!< the delay between 2 sampling phases in ADC synchronization modes to 10 ADC clock cycles. */
#define ADC_SYNC_DELAY_11CYCLE                       SYNCCTL_SYNCDLY(6)                                            /*!< the delay between 2 sampling phases in ADC synchronization modes to 11 ADC clock cycles. */
#define ADC_SYNC_DELAY_12CYCLE                       SYNCCTL_SYNCDLY(7)                                            /*!< the delay between 2 sampling phases in ADC synchronization modes to 12 ADC clock cycles. */
#define ADC_SYNC_DELAY_13CYCLE                       SYNCCTL_SYNCDLY(8)                                            /*!< the delay between 2 sampling phases in ADC synchronization modes to 13 ADC clock cycles. */
#define ADC_SYNC_DELAY_14CYCLE                       SYNCCTL_SYNCDLY(9)                                            /*!< the delay between 2 sampling phases in ADC synchronization modes to 14 ADC clock cycles. */
#define ADC_SYNC_DELAY_15CYCLE                       SYNCCTL_SYNCDLY(10)                                           /*!< the delay between 2 sampling phases in ADC synchronization modes to 15 ADC clock cycles. */
#define ADC_SYNC_DELAY_16CYCLE                       SYNCCTL_SYNCDLY(11)                                           /*!< the delay between 2 sampling phases in ADC synchronization modes to 16 ADC clock cycles. */
#define ADC_SYNC_DELAY_17CYCLE                       SYNCCTL_SYNCDLY(12)                                           /*!< the delay between 2 sampling phases in ADC synchronization modes to 17 ADC clock cycles. */
#define ADC_SYNC_DELAY_18CYCLE                       SYNCCTL_SYNCDLY(13)                                           /*!< the delay between 2 sampling phases in ADC synchronization modes to 18 ADC clock cycles. */
#define ADC_SYNC_DELAY_19CYCLE                       SYNCCTL_SYNCDLY(14)                                           /*!< the delay between 2 sampling phases in ADC synchronization modes to 19 ADC clock cycles. */
#define ADC_SYNC_DELAY_20CYCLE                       SYNCCTL_SYNCDLY(15)                                           /*!< the delay between 2 sampling phases in ADC synchronization modes to 20 ADC clock cycles. */

/* ADC sync mode */
#define SYNCCTL_SYNCM(regval)                        (BITS(0,3) & ((uint32_t)(regval) << 0U))                      /*!< write value to ADC_SYNCCTL_SYNCM bit field */
#define ADC_SYNC_MODE_INDEPENDENT                    SYNCCTL_SYNCM(0)                                              /*!< ADC synchronization mode disabled.All the ADCs work independently */
#define ADC_DAUL_REGULAL_PARALLEL_INSERTED_PARALLEL  SYNCCTL_SYNCM(1)                                              /*!< combined routine parallel & inserted parallel mode */
#define ADC_DAUL_REGULAL_PARALLEL_INSERTED_ROTATION  SYNCCTL_SYNCM(2)                                              /*!< combined routine parallel & trigger rotation mode */
#define ADC_DAUL_INSERTED_PARALLEL                   SYNCCTL_SYNCM(5)                                              /*!< inserted parallel mode */
#define ADC_DAUL_REGULAL_PARALLEL                    SYNCCTL_SYNCM(6)                                              /*!< routine parallel mode */
#define ADC_DAUL_REGULAL_FOLLOW_UP                   SYNCCTL_SYNCM(7)                                              /*!< follow-up mode */
#define ADC_DAUL_INSERTED_TRIGGER_ROTATION           SYNCCTL_SYNCM(9)                                              /*!< trigger rotation mode */

/* get ADC init value */
#define HAL_ADC_GET_SCAN_MODE(adcx)                  (uint32_t)((ADC_CTL0(adcx)) & (ADC_CTL0_SM))                  /*!< get ADC scan mode init value */
#define HAL_ADC_GET_CONTINUOUS_MODE(adcx)            (uint32_t)((ADC_CTL1(adcx)) & (ADC_CTL1_CTN))                 /*!< get ADC continuous mode init value */
#define HAL_ADC_GET_ROUTINECH_DIS_MODE(adcx)         (uint32_t)((ADC_CTL0(adcx)) & (ADC_CTL0_DISRC))               /*!< get ADC discontinuous mode init value */
#define HAL_ADC_GET_HPTEMPSENSOR_ENABLE(adcx)        (uint32_t)((ADC_CTL1(adcx)) & (ADC_CTL1_TSVEN2))              /*!< get ADC internal channel enable init value */
#define HAL_ADC_GET_TEMPSENSOR_ENABLE(adcx)          (uint32_t)((ADC_CTL1(adcx)) & (ADC_CTL1_TSVEN1))              /*!< get ADC internal channel enable init value */
#define HAL_ADC_GET_INVREFINT_ENABLE(adcx)           (uint32_t)((ADC_CTL1(adcx)) & (ADC_CTL1_INREFEN))             /*!< get ADC internal channel enable init value */
#define HAL_ADC_GET_BATTERY_ENABLE(adcx)             (uint32_t)((ADC_CTL1(adcx)) & (ADC_CTL1_VBATEN))              /*!< get ADC internal channel enable init value */
#define HAL_ADC_GET_OVERSAMPLE_ENABLE(adcx)          (uint32_t)((ADC_OVSAMPCTL(adcx)) & (ADC_OVSAMPCTL_OVSEN))     /*!< get ADC oversampling enable init value */
#define HAL_ADC_GET_DMA_MODE(adcx)                   (uint32_t)((ADC_CTL1(adcx)) & (ADC_CTL1_DMA))                 /*!< get ADC DMA mode init value */

#define HAL_ADC_GET_ROUTINECH_LENGTH(adcx)           (uint32_t)((ADC_RSQ0(adcx)) & (ADC_RSQ0_RL))                  /*!< get ADC routine sequence length value */
#define HAL_ADC_GET_ROUTINECH_EXTTRIGGER(adcx)       (uint32_t)((ADC_CTL1(adcx)) & (ADC_CTL1_ETMRC))               /*!< get ADC routine external trigger select */

/* constants definitions */
/* ADC event type */
#define ADC_EVENT_WDE0                               ADC_STAT_WDE0                                                 /*!< analog watchdog 0 event flag */
#define ADC_EVENT_WDE1                               ADC_STAT_WDE1                                                 /*!< analog watchdog 1 event flag */
#define ADC_EVENT_WDE2                               ADC_STAT_WDE2                                                 /*!< analog watchdog 2 event flag */

/* ADC status flag */
#define ADC_FLAG_WDE0                                ADC_STAT_WDE0                                                 /*!< analog watchdog 0 event flag */
#define ADC_FLAG_EOC                                 ADC_STAT_EOC                                                  /*!< end of sequence conversion flag */
#define ADC_FLAG_EOIC                                ADC_STAT_EOIC                                                 /*!< end of inserted sequence conversion flag */
#define ADC_FLAG_STIC                                ADC_STAT_STIC                                                 /*!< start flag of inserted channel sequence */
#define ADC_FLAG_STRC                                ADC_STAT_STRC                                                 /*!< start flag of routine channel sequence */
#define ADC_FLAG_ROVF                                ADC_STAT_ROVF                                                 /*!< routine data register overflow */
#define ADC_FLAG_WDE1                                ADC_STAT_WDE1                                                 /*!< analog watchdog 1 event flag */
#define ADC_FLAG_WDE2                                ADC_STAT_WDE2                                                 /*!< analog watchdog 2 event flag */

/* ADC interrupt */
#define ADC_INT_WDE0                                 ADC_CTL0_WDE0IE                                               /*!< interrupt enable for WDE0 */
#define ADC_INT_EOC                                  ADC_CTL0_EOCIE                                                /*!< interrupt enable for EOC */
#define ADC_INT_EOIC                                 ADC_CTL0_EOICIE                                               /*!< interrupt enable for EOIC */
#define ADC_INT_ROVF                                 ADC_CTL0_ROVFIE                                               /*!< interrupt enable for ROVF */
#define ADC_INT_WDE1                                 ADC_CTL0_WDE1IE                                               /*!< interrupt enable for WDE1 */
#define ADC_INT_WDE2                                 ADC_CTL0_WDE2IE                                               /*!< interrupt enable for WDE2 */

/* ADC interrupt flag */
#define ADC_INT_FLAG_WDE0                            ADC_STAT_WDE0                                                 /*!< analog watchdog 0 event interrupt flag */
#define ADC_INT_FLAG_EOC                             ADC_STAT_EOC                                                  /*!< end of sequence conversion interrupt flag */
#define ADC_INT_FLAG_EOIC                            ADC_STAT_EOIC                                                 /*!< end of inserted sequence conversion interrupt flag */
#define ADC_INT_FLAG_ROVF                            ADC_STAT_ROVF                                                 /*!< routine data register overflow interrupt flag */
#define ADC_INT_FLAG_WDE1                            ADC_STAT_WDE1                                                 /*!< analog watchdog 1 event interrupt flag */
#define ADC_INT_FLAG_WDE2                            ADC_STAT_WDE2                                                 /*!< analog watchdog 2 event interrupt flag */

/* ADC special function definitions */
#define ADC_SCAN_MODE                                ADC_CTL0_SM                                                   /*!< scan mode */
#define ADC_CONTINUOUS_MODE                          ADC_CTL1_CTN                                                  /*!< continuous mode */

/* ADC calibration mode */
#define ADC_CALIBRATION_OFFSET_MISMATCH              ((uint32_t)0x00000000U)                                       /*!< ADC calibration offset and mismatch mode */
#define ADC_CALIBRATION_OFFSET                       ADC_CTL1_CALMOD                                               /*!< ADC calibration mode */

/* ADC data alignment */
#define ADC_DATAALIGN_RIGHT                          ((uint32_t)0x00000000U)                                       /*!< LSB alignment */
#define ADC_DATAALIGN_LEFT                           ADC_CTL1_DAL                                                  /*!< MSB alignment */

/* end of conversion mode */
#define ADC_EOC_SET_SEQUENCE                         ((uint32_t)0x00000000U)                                       /*!< only at the end of a sequence of routine conversions, the EOC bit is set */
#define ADC_EOC_SET_CONVERSION                       ADC_CTL1_EOCM                                                 /*!< at the end of each routine conversion, the EOC bit is set */

/* ADC internal channel definitions */
#define ADC_CHANNEL_INTERNAL_TEMPSENSOR              ADC_CTL1_TSVEN1                                               /*!< temperature sensor channel */
#define ADC_CHANNEL_INTERNAL_VREFINT                 ADC_CTL1_INREFEN                                              /*!< vrefint channel */
#define ADC_CHANNEL_INTERNAL_VBAT                    ADC_CTL1_VBATEN                                               /*!< vbat channel */
#define ADC_CHANNEL_INTERNAL_HP_TEMPSENSOR           ADC_CTL1_TSVEN2                                               /*!< high-precision temperature sensor channel */

/* external trigger mode for routine and inserted  channel */
#define EXTERNAL_TRIGGER_DISABLE                     ((uint32_t)0x00000000U)                                       /*!< external trigger disable */
#define EXTERNAL_TRIGGER_RISING                      ((uint32_t)0x00000001U)                                       /*!< rising edge of external trigger */
#define EXTERNAL_TRIGGER_FALLING                     ((uint32_t)0x00000002U)                                       /*!< falling edge of external trigger */
#define EXTERNAL_TRIGGER_RISING_FALLING              ((uint32_t)0x00000003U)                                       /*!< rising and falling edge of external trigger */

/* ADC channel sequence definitions */
#define ADC_ROUTINE_CHANNEL                          ((uint8_t)0x01U)                                              /*!< ADC routine channel sequence */
#define ADC_INSERTED_CHANNEL                         ((uint8_t)0x02U)                                              /*!< ADC inserted channel sequence */
#define ADC_ROUTINE_INSERTED_CHANNEL                 ((uint8_t)0x03U)                                              /*!< both routine and inserted channel sequence */
#define ADC_CHANNEL_DISCON_DISABLE                   ((uint8_t)0x04U)                                              /*!< disable discontinuous mode of routine & inserted channel */

/* ADC channel definitions */
#define ADC_CHANNEL_0                                ((uint8_t)0x00U)                                              /*!< ADC channel 0 */
#define ADC_CHANNEL_1                                ((uint8_t)0x01U)                                              /*!< ADC channel 1 */
#define ADC_CHANNEL_2                                ((uint8_t)0x02U)                                              /*!< ADC channel 2 */
#define ADC_CHANNEL_3                                ((uint8_t)0x03U)                                              /*!< ADC channel 3 */
#define ADC_CHANNEL_4                                ((uint8_t)0x04U)                                              /*!< ADC channel 4 */
#define ADC_CHANNEL_5                                ((uint8_t)0x05U)                                              /*!< ADC channel 5 */
#define ADC_CHANNEL_6                                ((uint8_t)0x06U)                                              /*!< ADC channel 6 */
#define ADC_CHANNEL_7                                ((uint8_t)0x07U)                                              /*!< ADC channel 7 */
#define ADC_CHANNEL_8                                ((uint8_t)0x08U)                                              /*!< ADC channel 8 */
#define ADC_CHANNEL_9                                ((uint8_t)0x09U)                                              /*!< ADC channel 9 */
#define ADC_CHANNEL_10                               ((uint8_t)0x0AU)                                              /*!< ADC channel 10 */
#define ADC_CHANNEL_11                               ((uint8_t)0x0BU)                                              /*!< ADC channel 11 */
#define ADC_CHANNEL_12                               ((uint8_t)0x0CU)                                              /*!< ADC channel 12 */
#define ADC_CHANNEL_13                               ((uint8_t)0x0DU)                                              /*!< ADC channel 13 */
#define ADC_CHANNEL_14                               ((uint8_t)0x0EU)                                              /*!< ADC channel 14 */
#define ADC_CHANNEL_15                               ((uint8_t)0x0FU)                                              /*!< ADC channel 15 */
#define ADC_CHANNEL_16                               ((uint8_t)0x10U)                                              /*!< ADC channel 16 */
#define ADC_CHANNEL_17                               ((uint8_t)0x11U)                                              /*!< ADC channel 17 */
#define ADC_CHANNEL_18                               ((uint8_t)0x12U)                                              /*!< ADC channel 18 */
#define ADC_CHANNEL_19                               ((uint8_t)0x13U)                                              /*!< ADC channel 19 */
#define ADC_CHANNEL_20                               ((uint8_t)0x14U)                                              /*!< ADC channel 20 */

/* ADC internal channel */
#define ADC0_CHANNEL_DAC0_OUT0                       ADC_CHANNEL_20                                                /*!< ADC DAC0_OUT0 channel */
#define ADC1_CHANNEL_BATTERY                         ADC_CHANNEL_16                                                /*!< ADC backup battery voltage channel */
#define ADC1_CHANNEL_VREFINT                         ADC_CHANNEL_17                                                /*!< ADC VREFINT channel */
#define ADC1_CHANNEL_DAC0_OUT1                       ADC_CHANNEL_20                                                /*!< ADC dac0 out0 voltage channel */
#define ADC2_CHANNEL_BATTERY                         ADC_CHANNEL_17                                                /*!< ADC backup battery voltage channel */
#define ADC2_CHANNEL_TEMPSENSOR                      ADC_CHANNEL_18                                                /*!< ADC temperature sensor channel */
#define ADC2_CHANNEL_VREFINT                         ADC_CHANNEL_19                                                /*!< ADC VREFINT channel */
#define ADC2_CHANNEL_HIGH_PRECISION_TEMPSENSOR       ADC_CHANNEL_20                                                /*!< ADC high precision temperature sensor channel */

/* analog watchdog 1/2 channel selection for channel n(n=0..20) */
#define ADC_AWD1_2_SELECTION_CHANNEL_0               ((uint32_t)0x00000001U)                                       /*!< ADC channel 0 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_1               ((uint32_t)0x00000002U)                                       /*!< ADC channel 1 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_2               ((uint32_t)0x00000004U)                                       /*!< ADC channel 2 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_3               ((uint32_t)0x00000008U)                                       /*!< ADC channel 3 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_4               ((uint32_t)0x00000010U)                                       /*!< ADC channel 4 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_5               ((uint32_t)0x00000020U)                                       /*!< ADC channel 5 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_6               ((uint32_t)0x00000040U)                                       /*!< ADC channel 6 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_7               ((uint32_t)0x00000080U)                                       /*!< ADC channel 7 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_8               ((uint32_t)0x00000100U)                                       /*!< ADC channel 8 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_9               ((uint32_t)0x00000200U)                                       /*!< ADC channel 9 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_10              ((uint32_t)0x00000400U)                                       /*!< ADC channel 10 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_11              ((uint32_t)0x00000800U)                                       /*!< ADC channel 11 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_12              ((uint32_t)0x00001000U)                                       /*!< ADC channel 12 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_13              ((uint32_t)0x00002000U)                                       /*!< ADC channel 13 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_14              ((uint32_t)0x00004000U)                                       /*!< ADC channel 14 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_15              ((uint32_t)0x00008000U)                                       /*!< ADC channel 15 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_16              ((uint32_t)0x00010000U)                                       /*!< ADC channel 16 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_17              ((uint32_t)0x00020000U)                                       /*!< ADC channel 17 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_18              ((uint32_t)0x00040000U)                                       /*!< ADC channel 18 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_19              ((uint32_t)0x00080000U)                                       /*!< ADC channel 19 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_20              ((uint32_t)0x00100000U)                                       /*!< ADC channel 20 analog watchdog 1/2 selection */
#define ADC_AWD1_2_SELECTION_CHANNEL_ALL             ((uint32_t)0x003FFFFFU)                                       /*!< all ADC channels analog watchdog 1/2 selection */

/* Differential mode for channel n(n=0..21) */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_0              ((uint32_t)0x00000001U)                                       /*!< ADC channel 0 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_1              ((uint32_t)0x00000002U)                                       /*!< ADC channel 1 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_2              ((uint32_t)0x00000004U)                                       /*!< ADC channel 2 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_3              ((uint32_t)0x00000008U)                                       /*!< ADC channel 3 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_4              ((uint32_t)0x00000010U)                                       /*!< ADC channel 4 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_5              ((uint32_t)0x00000020U)                                       /*!< ADC channel 5 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_6              ((uint32_t)0x00000040U)                                       /*!< ADC channel 6 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_7              ((uint32_t)0x00000080U)                                       /*!< ADC channel 7 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_8              ((uint32_t)0x00000100U)                                       /*!< ADC channel 8 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_9              ((uint32_t)0x00000200U)                                       /*!< ADC channel 9 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_10             ((uint32_t)0x00000400U)                                       /*!< ADC channel 10 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_11             ((uint32_t)0x00000800U)                                       /*!< ADC channel 11 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_12             ((uint32_t)0x00001000U)                                       /*!< ADC channel 12 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_13             ((uint32_t)0x00002000U)                                       /*!< ADC channel 13 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_14             ((uint32_t)0x00004000U)                                       /*!< ADC channel 14 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_15             ((uint32_t)0x00008000U)                                       /*!< ADC channel 15 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_16             ((uint32_t)0x00010000U)                                       /*!< ADC channel 16 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_17             ((uint32_t)0x00020000U)                                       /*!< ADC channel 17 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_18             ((uint32_t)0x00040000U)                                       /*!< ADC channel 18 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_19             ((uint32_t)0x00080000U)                                       /*!< ADC channel 19 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_20             ((uint32_t)0x00100000U)                                       /*!< ADC channel 20 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_21             ((uint32_t)0x00200000U)                                       /*!< ADC channel 21 differential mode */
#define ADC_DIFFERENTIAL_MODE_CHANNEL_ALL            ((uint32_t)0x003FFFFFU)                                       /*!< all ADC channels differential mode */

/* delaytime for ADC enable and disable, in milliseconds */
#define ADC_ENABLE_DELAYTIME                         ((uint32_t)2U)                                                /*!< enable delay time */
#define ADC_DISABLE_DELAYTIME                        ((uint32_t)2U)                                                /*!< disable delay time */
/* delaytime for temperature sensor stabilization time, in microseconds */
#define ADC_TEMPSENSOR_DELAYTIME                     ((uint32_t)10U)                                               /*!< temp sensor delay time */

/* ADC routine channel macro */
#define ADC_ROUTINE_CHANNEL_RANK_ONE                 ((uint8_t)1U)                                                 /*!< inserted channel rank one */
#define ADC_ROUTINE_CHANNEL_RANK_THREE               ((uint8_t)3U)                                                 /*!< inserted channel rank three */
#define ADC_ROUTINE_CHANNEL_RANK_FIVE                ((uint8_t)5U)                                                 /*!< inserted channel rank five */
#define ADC_ROUTINE_CHANNEL_RANK_SEVEN               ((uint8_t)7U)                                                 /*!< inserted channel rank seven */
#define ADC_ROUTINE_CHANNEL_RANK_NINE                ((uint8_t)9U)                                                 /*!< inserted channel rank nine */
#define ADC_ROUTINE_CHANNEL_RANK_ELEVEN              ((uint8_t)11U)                                                /*!< inserted channel rank eleven */
#define ADC_ROUTINE_CHANNEL_RANK_THIRTEEN            ((uint8_t)13U)                                                /*!< inserted channel rank thirteen */
#define ADC_ROUTINE_CHANNEL_RANK_FIFTEEN             ((uint8_t)15U)                                                /*!< inserted channel rank fifteen */
#define ADC_ROUTINE_CHANNEL_RANK_SIXTEEN             ((uint8_t)16U)                                                /*!< inserted channel rank sixteen */
#define ADC_ROUTINE_CHANNEL_SHIFT_LENGTH             ((uint8_t)16U)                                                /*!< routine channel shift length */

/* external trigger mode macro */
#define ROUTINE_TRIGGER_MODE                         ((uint8_t)28U)                                                /*!< routine trigger mode */

/* @STRUCT_MEMBER: dma_sync_mode */
/* @DEFINE: dma sync mode */
#define ADC_DMA_SYNC_MODE_DISABLE                    ADC_SYNC_DMA_DISABLE                                          /*!< ADC sync DMA disabled */
#define ADC_DMA_SYNC_MODE0                           ADC_SYNC_DMA_MODE0                                            /*!< ADC sync DMA mode 0 */
#define ADC_DMA_SYNC_MODE1                           ADC_SYNC_DMA_MODE1                                            /*!< ADC sync DMA mode 1 */

/* @STRUCT_MEMBER: dma_sync_cont_req */
/* @DEFINE: dma sync cont mode */
#define ADC_DMA_SYNC_CONT_REQ_DISABLE                ((uint32_t)0x00000000U)                                       /*!< ADC sync DMA continuous mode disabled */
#define ADC_DMA_SYNC_CONT_REQ_ENABLE                 ADC_SYNCCTL_SYNCDDM                                           /*!< ADC sync DMA continuous mode enable */

/* @STRUCT_MEMBER: difctl_in0 */
/* @DEFINE: difctl in0 */
#define ADC_DIFCTL_IN0_DISABLE                       ((uint32_t)0x00000000U)                                       /*!< ADC channel 0 disable mode */
#define ADC_DIFCTL_IN0_SINGLE_MODE                   ADC_DIFCTL_IN0_DISABLE                                        /*!< ADC channel 0 single mode */
#define ADC_DIFCTL_IN0_DIF_MODE                      ADC_DIFFERENTIAL_MODE_CHANNEL_0                               /*!< ADC channel 0 differential mode */

/* @STRUCT_MEMBER: difctl_in1 */
/* @DEFINE: difctl in1 */
#define ADC_DIFCTL_IN1_DISABLE                       ((uint32_t)0x00000000U)                                       /*!< ADC channel 1 disable mode */
#define ADC_DIFCTL_IN1_SINGLE_MODE                   ADC_DIFCTL_IN1_DISABLE                                        /*!< ADC channel 1 single mode */
#define ADC_DIFCTL_IN1_DIF_MODE                      ADC_DIFFERENTIAL_MODE_CHANNEL_1                               /*!< ADC channel 1 differential mode */

/* @STRUCT_MEMBER: difctl_in2 */
/* @DEFINE: difctl in2 */
#define ADC_DIFCTL_IN2_DISABLE                       ((uint32_t)0x00000000U)                                       /*!< ADC channel 2 disable mode */
#define ADC_DIFCTL_IN2_SINGLE_MODE                   ADC_DIFCTL_IN2_DISABLE                                        /*!< ADC channel 2 single mode */
#define ADC_DIFCTL_IN2_DIF_MODE                      ADC_DIFFERENTIAL_MODE_CHANNEL_2                               /*!< ADC channel 2 differential mode */

/* @STRUCT_MEMBER: difctl_in3 */
/* @DEFINE: difctl in3 */
#define ADC_DIFCTL_IN3_DISABLE                       ((uint32_t)0x00000000U)                                       /*!< ADC channel 3 disable mode */
#define ADC_DIFCTL_IN3_SINGLE_MODE                   ADC_DIFCTL_IN3_DISABLE                                        /*!< ADC channel 3 single mode */
#define ADC_DIFCTL_IN3_DIF_MODE                      ADC_DIFFERENTIAL_MODE_CHANNEL_3                               /*!< ADC channel 3 differential mode */

/* @STRUCT_MEMBER: difctl_in4 */
/* @DEFINE: difctl in4 */
#define ADC_DIFCTL_IN4_DISABLE                       ((uint32_t)0x00000000U)                                       /*!< ADC channel 4 disable mode */
#define ADC_DIFCTL_IN4_SINGLE_MODE                   ADC_DIFCTL_IN4_DISABLE                                        /*!< ADC channel 4 single mode */
#define ADC_DIFCTL_IN4_DIF_MODE                      ADC_DIFFERENTIAL_MODE_CHANNEL_4                               /*!< ADC channel 4 differential mode */

/* @STRUCT_MEMBER: difctl_in5 */
/* @DEFINE: difctl in5 */
#define ADC_DIFCTL_IN5_DISABLE                       ((uint32_t)0x00000000U)                                       /*!< ADC channel 5 disable mode */
#define ADC_DIFCTL_IN5_SINGLE_MODE                   ADC_DIFCTL_IN5_DISABLE                                        /*!< ADC channel 5 single mode */
#define ADC_DIFCTL_IN5_DIF_MODE                      ADC_DIFFERENTIAL_MODE_CHANNEL_5                               /*!< ADC channel 5 differential mode */

/* @STRUCT_MEMBER: difctl_in6 */
/* @DEFINE: difctl in6 */
#define ADC_DIFCTL_IN6_DISABLE                       ((uint32_t)0x00000000U)                                       /*!< ADC channel 6 disable mode */
#define ADC_DIFCTL_IN6_SINGLE_MODE                   ADC_DIFCTL_IN6_DISABLE                                        /*!< ADC channel 6 single mode */
#define ADC_DIFCTL_IN6_DIF_MODE                      ADC_DIFFERENTIAL_MODE_CHANNEL_6                               /*!< ADC channel 6 differential mode */

/* @STRUCT_MEMBER: difctl_in7 */
/* @DEFINE: difctl in7 */
#define ADC_DIFCTL_IN7_DISABLE                       ((uint32_t)0x00000000U)                                       /*!< ADC channel 7 disable mode */
#define ADC_DIFCTL_IN7_SINGLE_MODE                   ADC_DIFCTL_IN7_DISABLE                                        /*!< ADC channel 7 single mode */
#define ADC_DIFCTL_IN7_DIF_MODE                      ADC_DIFFERENTIAL_MODE_CHANNEL_7                               /*!< ADC channel 7 differential mode */

/* @STRUCT_MEMBER: difctl_in8 */
/* @DEFINE: difctl in8 */
#define ADC_DIFCTL_IN8_DISABLE                       ((uint32_t)0x00000000U)                                       /*!< ADC channel 8 disable mode */
#define ADC_DIFCTL_IN8_SINGLE_MODE                   ADC_DIFCTL_IN8_DISABLE                                        /*!< ADC channel 8 single mode */
#define ADC_DIFCTL_IN8_DIF_MODE                      ADC_DIFFERENTIAL_MODE_CHANNEL_8                               /*!< ADC channel 8 differential mode */

/* @STRUCT_MEMBER: difctl_in9 */
/* @DEFINE: difctl in9 */
#define ADC_DIFCTL_IN9_DISABLE                       ((uint32_t)0x00000000U)                                       /*!< ADC channel 9 disable mode */
#define ADC_DIFCTL_IN9_SINGLE_MODE                   ADC_DIFCTL_IN9_DISABLE                                        /*!< ADC channel 9 single mode */
#define ADC_DIFCTL_IN9_DIF_MODE                      ADC_DIFFERENTIAL_MODE_CHANNEL_9                               /*!< ADC channel 9 differential mode */

/* @STRUCT_MEMBER: difctl_in10 */
/* @DEFINE: difctl in10 */
#define ADC_DIFCTL_IN10_DISABLE                      ((uint32_t)0x00000000U)                                       /*!< ADC channel 10 disable mode */
#define ADC_DIFCTL_IN10_SINGLE_MODE                  ADC_DIFCTL_IN10_DISABLE                                       /*!< ADC channel 10 single mode */
#define ADC_DIFCTL_IN10_DIF_MODE                     ADC_DIFFERENTIAL_MODE_CHANNEL_10                              /*!< ADC channel 10 differential mode */

/* @STRUCT_MEMBER: difctl_in11 */
/* @DEFINE: difctl in11 */
#define ADC_DIFCTL_IN11_DISABLE                      ((uint32_t)0x00000000U)                                       /*!< ADC channel 11 disable mode */
#define ADC_DIFCTL_IN11_SINGLE_MODE                  ADC_DIFCTL_IN11_DISABLE                                       /*!< ADC channel 11 single mode */
#define ADC_DIFCTL_IN11_DIF_MODE                     ADC_DIFFERENTIAL_MODE_CHANNEL_11                              /*!< ADC channel 11 differential mode */

/* @STRUCT_MEMBER: difctl_in12 */
/* @DEFINE: difctl in12 */
#define ADC_DIFCTL_IN12_DISABLE                      ((uint32_t)0x00000000U)                                       /*!< ADC channel 12 disable mode */
#define ADC_DIFCTL_IN12_SINGLE_MODE                  ADC_DIFCTL_IN12_DISABLE                                       /*!< ADC channel 12 single mode */
#define ADC_DIFCTL_IN12_DIF_MODE                     ADC_DIFFERENTIAL_MODE_CHANNEL_12                              /*!< ADC channel 12 differential mode */

/* @STRUCT_MEMBER: difctl_in13 */
/* @DEFINE: difctl in13 */
#define ADC_DIFCTL_IN13_DISABLE                      ((uint32_t)0x00000000U)                                       /*!< ADC channel 13 disable mode */
#define ADC_DIFCTL_IN13_SINGLE_MODE                  ADC_DIFCTL_IN13_DISABLE                                       /*!< ADC channel 13 single mode */
#define ADC_DIFCTL_IN13_DIF_MODE                     ADC_DIFFERENTIAL_MODE_CHANNEL_13                              /*!< ADC channel 13 differential mode */

/* @STRUCT_MEMBER: difctl_in14 */
/* @DEFINE: difctl in14 */
#define ADC_DIFCTL_IN14_DISABLE                      ((uint32_t)0x00000000U)                                       /*!< ADC channel 14 disable mode */
#define ADC_DIFCTL_IN14_SINGLE_MODE                  ADC_DIFCTL_IN14_DISABLE                                       /*!< ADC channel 14 single mode */
#define ADC_DIFCTL_IN14_DIF_MODE                     ADC_DIFFERENTIAL_MODE_CHANNEL_14                              /*!< ADC channel 14 differential mode */

/* @STRUCT_MEMBER: difctl_in15 */
/* @DEFINE: difctl in15 */
#define ADC_DIFCTL_IN15_DISABLE                      ((uint32_t)0x00000000U)                                       /*!< ADC channel 15 disable mode */
#define ADC_DIFCTL_IN15_SINGLE_MODE                  ADC_DIFCTL_IN15_DISABLE                                       /*!< ADC channel 15 single mode */
#define ADC_DIFCTL_IN15_DIF_MODE                     ADC_DIFFERENTIAL_MODE_CHANNEL_15                              /*!< ADC channel 15 differential mode */

/* @STRUCT_MEMBER: difctl_in16 */
/* @DEFINE: difctl in16 */
#define ADC_DIFCTL_IN16_DISABLE                      ((uint32_t)0x00000000U)                                       /*!< ADC channel 16 disable mode */
#define ADC_DIFCTL_IN16_SINGLE_MODE                  ADC_DIFCTL_IN16_DISABLE                                       /*!< ADC channel 16 single mode */
#define ADC_DIFCTL_IN16_DIF_MODE                     ADC_DIFFERENTIAL_MODE_CHANNEL_16                              /*!< ADC channel 16 differential mode */
#define ADC_DIFCTL_IN16_VBAT_MODE                    ADC_DIFCTL_IN16_DISABLE                                       /*!< ADC channel 16 ADC_CHANNEL_INTERNAL_VBAT */

/* @STRUCT_MEMBER: difctl_in17 */
/* @DEFINE: difctl in17 */
#define ADC_DIFCTL_IN17_DISABLE                      ((uint32_t)0x00000000U)                                       /*!< ADC channel 17 disable mode */
#define ADC_DIFCTL_IN17_SINGLE_MODE                  ADC_DIFCTL_IN17_DISABLE                                       /*!< ADC channel 17 single mode */
#define ADC_DIFCTL_IN17_DIF_MODE                     ADC_DIFFERENTIAL_MODE_CHANNEL_17                              /*!< ADC channel 17 differential mode */
#define ADC_DIFCTL_IN17_VBAT_MODE                    ADC_DIFCTL_IN17_DISABLE                                       /*!< ADC channel 17 ADC_CHANNEL_INTERNAL_VBAT */
#define ADC_DIFCTL_IN17_VREFINT_MODE                 ADC_DIFCTL_IN17_DISABLE                                       /*!< ADC channel 17 ADC_CHANNEL_INTERNAL_VREFINT */

/* @STRUCT_MEMBER: difctl_in18 */
/* @DEFINE: difctl in18 */
#define ADC_DIFCTL_IN18_DISABLE                      ((uint32_t)0x00000000U)                                       /*!< ADC channel 18 disable mode */
#define ADC_DIFCTL_IN18_SINGLE_MODE                  ADC_DIFCTL_IN18_DISABLE                                       /*!< ADC channel 18 single mode */
#define ADC_DIFCTL_IN18_DIF_MODE                     ADC_DIFFERENTIAL_MODE_CHANNEL_18                              /*!< ADC channel 18 differential mode */
#define ADC_DIFCTL_IN18_TS_MODE                      ADC_DIFCTL_IN18_DISABLE                                       /*!< ADC channel 18 ADC_CHANNEL_INTERNAL_TEMPSENSOR */

/* @STRUCT_MEMBER: difctl_in19 */
/* @DEFINE: difctl in19 */
#define ADC_DIFCTL_IN19_DISABLE                      ((uint32_t)0x00000000U)                                       /*!< ADC channel 19 disable mode */
#define ADC_DIFCTL_IN19_SINGLE_MODE                  ADC_DIFCTL_IN19_DISABLE                                       /*!< ADC channel 19 single mode */
#define ADC_DIFCTL_IN19_DIF_MODE                     ADC_DIFFERENTIAL_MODE_CHANNEL_19                              /*!< ADC channel 19 differential mode */
#define ADC_DIFCTL_IN19_VREFINT_MODE                 ADC_DIFCTL_IN19_DISABLE                                       /*!< ADC channel 19 ADC_CHANNEL_INTERNAL_VREFINT */

/* @STRUCT_MEMBER: difctl_in20 */
/* @DEFINE: difctl in20 */
#define ADC_DIFCTL_IN20_DISABLE                      ((uint32_t)0x00000000U)                                       /*!< ADC channel 20 disable mode */
#define ADC_DIFCTL_IN20_SINGLE_MODE                  ADC_DIFCTL_IN20_DISABLE                                       /*!< ADC channel 20 single mode */
#define ADC_DIFCTL_IN20_DIF_MODE                     ADC_DIFFERENTIAL_MODE_CHANNEL_20                              /*!< ADC channel 20 differential mode */
#define ADC_DIFCTL_IN20_DAC0_OUT0                    ADC_DIFCTL_IN20_DISABLE                                       /*!< ADC channel 20 DAC0_OUT0 */
#define ADC_DIFCTL_IN20_DAC0_OUT1                    ADC_DIFCTL_IN20_DISABLE                                       /*!< ADC channel 20 DAC0_OUT1 */
#define ADC_DIFCTL_IN20_HPTS_MODE                    ADC_DIFCTL_IN20_DISABLE                                       /*!< ADC channel 20 ADC_CHANNEL_INTERNAL_HP_TEMPSENSOR */

/* @STRUCT_MEMBER: difctl_in21 */
/* @DEFINE: difctl in21 */
#define ADC_DIFCTL_IN21_DISABLE                      ((uint32_t)0x00000000U)                                       /*!< ADC channel 21 disable mode */
#define ADC_DIFCTL_IN21_SINGLE_MODE                  ADC_DIFCTL_IN21_DISABLE                                       /*!< ADC channel 21 single mode */
#define ADC_DIFCTL_IN21_DIF_MODE                     ADC_DIFFERENTIAL_MODE_CHANNEL_21                              /*!< ADC channel 21 differential mode */

/* @STRUCT_MEMBER: data_alignment */
/* @DEFINE: data alignment */
#define ADC_LSB_ALIGNMENT                            ((uint32_t)0x00000000U)                                       /*!< LSB alignment */
#define ADC_MSB_ALIGNMENT                            ADC_CTL1_DAL                                                  /*!< MSB alignment */

/* @STRUCT_MEMBER: calibration */
/* @DEFINE: calibration disable or enable */
#define ADC_CALIBRATION_DISABLE                      DISABLE                                                       /*!< ADC calibration disable */
#define ADC_CALIBRATION_ENABLE                       ENABLE                                                        /*!< ADC calibration enable */

/* @STRUCT_MEMBER: calibration_mode */
/* @DEFINE: calibration mode */
#define ADC_CAL_MODE_OFFSET_MISMATCH                 ADC_CALIBRATION_OFFSET_MISMATCH                               /*!< ADC calibration offset and mismatch mode */
#define ADC_CAL_MODE_OFFSET                          ADC_CALIBRATION_OFFSET                                        /*!< ADC calibration offset mode */

/* @STRUCT_MEMBER: resolution */
/* @DEFINE: adc resolution */
#define ADC01_RESOLUTION_14B                         ADC_RESOLUTION_14B                                            /*!< 14-bit ADC resolution */
#define ADC01_RESOLUTION_12B                         ADC_RESOLUTION_12B                                            /*!< 12-bit ADC resolution */
#define ADC01_RESOLUTION_10B                         ADC_RESOLUTION_10B                                            /*!< 10-bit ADC resolution */
#define ADC01_RESOLUTION_8B                          ADC_RESOLUTION_8B                                             /*!< 8-bit ADC resolution */
#define ADC2_RESOLUTION_12B                          ADC_RESOLUTION_12B                                            /*!< 12-bit ADC resolution */
#define ADC2_RESOLUTION_10B                          ADC_RESOLUTION_10B                                            /*!< 10-bit ADC resolution */
#define ADC2_RESOLUTION_8B                           ADC_RESOLUTION_8B                                             /*!< 8-bit ADC resolution */
#define ADC2_RESOLUTION_6B                           ADC_RESOLUTION_6B                                             /*!< 6-bit ADC resolution */

/* @STRUCT_MEMBER: scan_mode */
/* @DEFINE: scan mode */
#define ADC_SCAN_MODE_DISABLE                        ((uint32_t)0x00000000U)                                       /*!< ADC scan mode disable */
#define ADC_SCAN_MODE_ENABLE                         ADC_SCAN_MODE                                                 /*!< ADC scan mode enable */

/* @STRUCT_MEMBER: hardware_oversampling */
/* @DEFINE: hardware oversampling */
#define ADC_OVERSAMPLING_DISABLE                     ((uint32_t)0x00000000U)                                       /*!< ADC hardware oversampling disable */
#define ADC_OVERSAMPLING_ENABLE                      ADC_OVSAMPCTL_OVSEN                                           /*!< ADC hardware oversampling enable */

/* @STRUCT_MEMBER: oversampling_ratio */
/* @=NULL */

/* @STRUCT_MEMBER: triggered_oversampling */
/* @DEFINE: triggered oversampling */
#define ADC_OVERSAMPLING_ALL_CONVERT                 ((uint32_t)0x00000000U)                                       /*!< all oversampled conversions for a channel are done consecutively after a trigger */
#define ADC_OVERSAMPLING_ONE_CONVERT                 ADC_OVSAMPCTL_TOVS                                            /*!< each oversampled conversion for a channel needs a trigger */

/* @STRUCT_MEMBER: hpdf_mode */
/* @DEFINE: hpdf mode */
#define ADC_HPDF_MODE_DISABLE                        ((uint32_t)0x00000000U)                                       /*!< HPDF mode disable */
#define ADC_HPDF_MODE_ENABLE                         ADC_CTL1_HPDFCFG                                              /*!< HPDF mode enable */

/* @STRUCT_MEMBER: routine_external_trigger_source */
/* @DEFINE: routine external trigger source */
#define ADC_ROUTINE_SOFTWARE_TRIGGER                 ((uint32_t)1U)                                                /*!< ADC routine external trigger source is software */
#define ADC_ROUTINE_TRIGSEL_TRIGGER                  ((uint32_t)2U)                                                /*!< ADC routine external trigger source is TRIGSEL */

/* @STRUCT_MEMBER: routine_conversions */
/* @SP: ENABLE/DISABLE */

/* @STRUCT_MEMBER: continuous_mode */
/* @DEFINE: continuous mode */
#define ADC_CONTINUOUS_MODE_DISABLE                  ((uint32_t)0x00000000U)                                       /*!< ADC continuous mode disable */
#define ADC_CONTINUOUS_MODE_ENABLE                   ADC_CONTINUOUS_MODE                                           /*!< ADC continuous mode enable */

/* @STRUCT_MEMBER: discontinuous_mode */
/* @DEFINE: discontinuous mode */
#define ADC_DISCONTINUOUS_MODE_DISABLE               ((uint32_t)0x00000000U)                                       /*!< ADC discontinuous mode enable */
#define ADC_DISCONTINUOUS_MODE_ENABLE                ADC_CTL0_DISRC                                                /*!< ADC discontinuous mode enable */

/* @STRUCT_MEMBER: number_of_conversions_in_discontinuous_mode */
/* @=NULL */

/* @STRUCT_MEMBER: dma_cont_req */
/* @DEFINE: dma cont req */
#define ADC_DMA_CONT_REQ_DISABLE                     ((uint32_t)0x00000000U)                                       /*!< ADC routine channel DMA continuous requests disabled */
#define ADC_DMA_CONT_REQ_ENABLE                      ADC_CTL1_DDM                                                  /*!< ADC routine channel DMA continuous requests enable */

/* @STRUCT_MEMBER: channel */
/* @DEFINE: ADC channel select */
#define ADC_CHANNEL_0                                ((uint8_t)0x00U)                                              /*!< ADC channel 0 */
#define ADC_CHANNEL_1                                ((uint8_t)0x01U)                                              /*!< ADC channel 1 */
#define ADC_CHANNEL_2                                ((uint8_t)0x02U)                                              /*!< ADC channel 2 */
#define ADC_CHANNEL_3                                ((uint8_t)0x03U)                                              /*!< ADC channel 3 */
#define ADC_CHANNEL_4                                ((uint8_t)0x04U)                                              /*!< ADC channel 4 */
#define ADC_CHANNEL_5                                ((uint8_t)0x05U)                                              /*!< ADC channel 5 */
#define ADC_CHANNEL_6                                ((uint8_t)0x06U)                                              /*!< ADC channel 6 */
#define ADC_CHANNEL_7                                ((uint8_t)0x07U)                                              /*!< ADC channel 7 */
#define ADC_CHANNEL_8                                ((uint8_t)0x08U)                                              /*!< ADC channel 8 */
#define ADC_CHANNEL_9                                ((uint8_t)0x09U)                                              /*!< ADC channel 9 */
#define ADC_CHANNEL_10                               ((uint8_t)0x0AU)                                              /*!< ADC channel 10 */
#define ADC_CHANNEL_11                               ((uint8_t)0x0BU)                                              /*!< ADC channel 11 */
#define ADC_CHANNEL_12                               ((uint8_t)0x0CU)                                              /*!< ADC channel 12 */
#define ADC_CHANNEL_13                               ((uint8_t)0x0DU)                                              /*!< ADC channel 13 */
#define ADC_CHANNEL_14                               ((uint8_t)0x0EU)                                              /*!< ADC channel 14 */
#define ADC_CHANNEL_15                               ((uint8_t)0x0FU)                                              /*!< ADC channel 15 */
#define ADC_CHANNEL_16                               ((uint8_t)0x10U)                                              /*!< ADC channel 16 */
#define ADC_CHANNEL_17                               ((uint8_t)0x11U)                                              /*!< ADC channel 17 */
#define ADC_CHANNEL_18                               ((uint8_t)0x12U)                                              /*!< ADC channel 18 */
#define ADC_CHANNEL_19                               ((uint8_t)0x13U)                                              /*!< ADC channel 19 */
#define ADC_CHANNEL_20                               ((uint8_t)0x14U)                                              /*!< ADC channel 20 */

/* @STRUCT_MEMBER: sampling_time */
/* @=NULL */
#define ADC01_SAMPLETIME_3POINT5                     SQX_SMP(0)                                                    /*!< 3.5 sampling cycles */
#define ADC01_SAMPLETIME_4POINT5                     SQX_SMP(1)                                                    /*!< 4.4 sampling cycles */
#define ADC01_SAMPLETIME_5POINT5                     SQX_SMP(2)                                                    /*!< 5.5 sampling cycles */
#define ADC01_SAMPLETIME_6POINT5                     SQX_SMP(3)                                                    /*!< 6.5 sampling cycles */
#define ADC01_SAMPLETIME_7POINT5                     SQX_SMP(4)                                                    /*!< 7.5 sampling cycles */
#define ADC01_SAMPLETIME_641POINT5                   SQX_SMP(638)                                                  /*!< 641.5 sampling cycles */
#define ADC01_SAMPLETIME_642POINT5                   SQX_SMP(639)                                                  /*!< 642.5 sampling cycles */
#define ADC01_SAMPLETIME_810POINT5                   SQX_SMP(809)                                                  /*!< 810.5 sampling cycles */
#define ADC2_SAMPLETIME_2POINT5                      SQX_SMP(0)                                                    /*!< 2.5 sampling cycles */
#define ADC2_SAMPLETIME_3POINT5                      SQX_SMP(1)                                                    /*!< 3.4 sampling cycles */
#define ADC2_SAMPLETIME_4POINT5                      SQX_SMP(2)                                                    /*!< 4.5 sampling cycles */
#define ADC2_SAMPLETIME_5POINT5                      SQX_SMP(3)                                                    /*!< 5.5 sampling cycles */
#define ADC2_SAMPLETIME_6POINT5                      SQX_SMP(4)                                                    /*!< 6.5 sampling cycles */
#define ADC2_SAMPLETIME_640POINT5                    SQX_SMP(638)                                                  /*!< 640.5 sampling cycles */

/* @STRUCT_MEMBER: routine_seq_watchdog0 */
/* @DEFINE: routine seq watchdog0 */
#define ADC_ROUTINE_SEQ_WATCHDOG0_DISABLE            ((uint32_t)0x00000000U)                                       /*!< ADC analog watchdog 0 monitor routine sequence disable */
#define ADC_ROUTINE_SEQ_WATCHDOG0_ENABLE             ADC_CTL0_RWD0EN                                               /*!< ADC analog watchdog 0 monitor routine sequence enable */

/* @STRUCT_MEMBER: watchdog0_mode */
/* @DEFINE: watchdog0 mode */
#define ADC_WATCHDOG0_MONITOR_ALL_CHANNELS           ((uint32_t)0x00000000U)                                       /*!< ADC analog watchdog 0 monitor all channels */
#define ADC_WATCHDOG0_MONITOR_SINGLE_CHANNEL         ADC_CTL0_WD0SC                                                /*!< ADC analog watchdog 0 monitor single channel */

/* @STRUCT_MEMBER: watchdog0_channel */
/* @DEFINE: watchdog 0 channel */
#define ADC_WDT0_CHANNEL_0                           ((uint8_t)0x00U)                                              /*!< ADC watchdog 0 channel 0 */
#define ADC_WDT0_CHANNEL_1                           ((uint8_t)0x01U)                                              /*!< ADC watchdog 0 channel 1 */
#define ADC_WDT0_CHANNEL_2                           ((uint8_t)0x02U)                                              /*!< ADC watchdog 0 channel 2 */
#define ADC_WDT0_CHANNEL_3                           ((uint8_t)0x03U)                                              /*!< ADC watchdog 0 channel 3 */
#define ADC_WDT0_CHANNEL_4                           ((uint8_t)0x04U)                                              /*!< ADC watchdog 0 channel 4 */
#define ADC_WDT0_CHANNEL_5                           ((uint8_t)0x05U)                                              /*!< ADC watchdog 0 channel 5 */
#define ADC_WDT0_CHANNEL_6                           ((uint8_t)0x06U)                                              /*!< ADC watchdog 0 channel 6 */
#define ADC_WDT0_CHANNEL_7                           ((uint8_t)0x07U)                                              /*!< ADC watchdog 0 channel 7 */
#define ADC_WDT0_CHANNEL_8                           ((uint8_t)0x08U)                                              /*!< ADC watchdog 0 channel 8 */
#define ADC_WDT0_CHANNEL_9                           ((uint8_t)0x09U)                                              /*!< ADC watchdog 0 channel 9 */
#define ADC_WDT0_CHANNEL_10                          ((uint8_t)0x0AU)                                              /*!< ADC watchdog 0 channel 10 */
#define ADC_WDT0_CHANNEL_11                          ((uint8_t)0x0BU)                                              /*!< ADC watchdog 0 channel 11 */
#define ADC_WDT0_CHANNEL_12                          ((uint8_t)0x0CU)                                              /*!< ADC watchdog 0 channel 12 */
#define ADC_WDT0_CHANNEL_13                          ((uint8_t)0x0DU)                                              /*!< ADC watchdog 0 channel 13 */
#define ADC_WDT0_CHANNEL_14                          ((uint8_t)0x0EU)                                              /*!< ADC watchdog 0 channel 14 */
#define ADC_WDT0_CHANNEL_15                          ((uint8_t)0x0FU)                                              /*!< ADC watchdog 0 channel 15 */
#define ADC_WDT0_CHANNEL_16                          ((uint8_t)0x10U)                                              /*!< ADC watchdog 0 channel 16 */
#define ADC_WDT0_CHANNEL_17                          ((uint8_t)0x11U)                                              /*!< ADC watchdog 0 channel 17 */
#define ADC_WDT0_CHANNEL_18                          ((uint8_t)0x12U)                                              /*!< ADC watchdog 0 channel 18 */
#define ADC_WDT0_CHANNEL_19                          ((uint8_t)0x13U)                                              /*!< ADC watchdog 0 channel 19 */
#define ADC_WDT0_CHANNEL_20                          ((uint8_t)0x14U)                                              /*!< ADC watchdog 0 channel 20 */

/* @STRUCT_MEMBER: watchdog1_channel0 */
/* @DEFINE: watchdog1 monitor channel0 */
#define ADC_WDT1_CHANNEL0_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 0 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL0_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_0                                /*!< ADC channel 0 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel1 */
/* @DEFINE: watchdog1 monitor channel1 */
#define ADC_WDT1_CHANNEL1_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 1 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL1_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_1                                /*!< ADC channel 1 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel2 */
/* @DEFINE: watchdog1 monitor channel2 */
#define ADC_WDT1_CHANNEL2_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 2 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL2_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_2                                /*!< ADC channel 2 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel3 */
/* @DEFINE: watchdog1 monitor channel3 */
#define ADC_WDT1_CHANNEL3_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 3 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL3_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_3                                /*!< ADC channel 3 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel4 */
/* @DEFINE: watchdog1 monitor channel4 */
#define ADC_WDT1_CHANNEL4_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 4 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL4_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_4                                /*!< ADC channel 4 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel5 */
/* @DEFINE: watchdog1 monitor channel5 */
#define ADC_WDT1_CHANNEL5_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 5 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL5_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_5                                /*!< ADC channel 5 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel6 */
/* @DEFINE: watchdog1 monitor channel6 */
#define ADC_WDT1_CHANNEL6_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 6 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL6_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_6                                /*!< ADC channel 6 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel7 */
/* @DEFINE: watchdog1 monitor channel7 */
#define ADC_WDT1_CHANNEL7_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 7 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL7_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_7                                /*!< ADC channel 7 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel8 */
/* @DEFINE: watchdog1 monitor channel8 */
#define ADC_WDT1_CHANNEL8_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 8 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL8_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_8                                /*!< ADC channel 8 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel9 */
/* @DEFINE: watchdog1 monitor channel9 */
#define ADC_WDT1_CHANNEL9_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 9 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL9_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_9                                /*!< ADC channel 9 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel10 */
/* @DEFINE: watchdog1 monitor channel10 */
#define ADC_WDT1_CHANNEL10_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 10 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL10_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_10                               /*!< ADC channel 10 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel11 */
/* @DEFINE: watchdog1 monitor channel11 */
#define ADC_WDT1_CHANNEL11_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 11 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL11_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_11                               /*!< ADC channel 11 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel12 */
/* @DEFINE: watchdog1 monitor channel12 */
#define ADC_WDT1_CHANNEL12_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 12 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL12_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_12                               /*!< ADC channel 12 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel13 */
/* @DEFINE: watchdog1 monitor channel13 */
#define ADC_WDT1_CHANNEL13_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 13 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL13_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_13                               /*!< ADC channel 13 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel14 */
/* @DEFINE: watchdog1 monitor channel14 */
#define ADC_WDT1_CHANNEL14_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 14 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL14_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_14                               /*!< ADC channel 14 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel15 */
/* @DEFINE: watchdog1 monitor channel15 */
#define ADC_WDT1_CHANNEL15_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 15 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL15_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_15                               /*!< ADC channel 15 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel16 */
/* @DEFINE: watchdog1 monitor channel16 */
#define ADC_WDT1_CHANNEL16_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 16 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL16_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_16                               /*!< ADC channel 16 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel17 */
/* @DEFINE: watchdog1 monitor channel17 */
#define ADC_WDT1_CHANNEL17_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 17 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL17_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_17                               /*!< ADC channel 17 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel18 */
/* @DEFINE: watchdog1 monitor channel18 */
#define ADC_WDT1_CHANNEL18_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 18 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL18_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_18                               /*!< ADC channel 18 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel19 */
/* @DEFINE: watchdog1 monitor channel19 */
#define ADC_WDT1_CHANNEL19_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 19 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL19_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_19                               /*!< ADC channel 19 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog1_channel20 */
/* @DEFINE: watchdog1 monitor channel20 */
#define ADC_WDT1_CHANNEL20_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 20 analog watchdog 1/2 disable */
#define ADC_WDT1_CHANNEL20_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_20                               /*!< ADC channel 20 analog watchdog 1/2 enable */

/* @STRUCT_MEMBER: watchdog2_channel0 */
/* @DEFINE: watchdog2 monitor channel0 */
#define ADC_WDT2_CHANNEL0_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 0 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL0_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_0                                /*!< ADC channel 0 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel1 */
/* @DEFINE: watchdog2 monitor channel1 */
#define ADC_WDT2_CHANNEL1_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 1 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL1_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_1                                /*!< ADC channel 1 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel2 */
/* @DEFINE: watchdog2 monitor channel2 */
#define ADC_WDT2_CHANNEL2_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 2 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL2_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_2                                /*!< ADC channel 2 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel3 */
/* @DEFINE: watchdog2 monitor channel3 */
#define ADC_WDT2_CHANNEL3_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 3 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL3_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_3                                /*!< ADC channel 3 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel4 */
/* @DEFINE: watchdog2 monitor channel4 */
#define ADC_WDT2_CHANNEL4_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 4 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL4_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_4                                /*!< ADC channel 4 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel5 */
/* @DEFINE: watchdog2 monitor channel5 */
#define ADC_WDT2_CHANNEL5_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 5 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL5_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_5                                /*!< ADC channel 5 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel6 */
/* @DEFINE: watchdog2 monitor channel6 */
#define ADC_WDT2_CHANNEL6_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 6 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL6_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_6                                /*!< ADC channel 6 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel7 */
/* @DEFINE: watchdog2 monitor channel7 */
#define ADC_WDT2_CHANNEL7_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 7 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL7_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_7                                /*!< ADC channel 7 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel8 */
/* @DEFINE: watchdog2 monitor channel8 */
#define ADC_WDT2_CHANNEL8_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 8 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL8_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_8                                /*!< ADC channel 8 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel9 */
/* @DEFINE: watchdog2 monitor channel9 */
#define ADC_WDT2_CHANNEL9_DISABLE                    ((uint32_t)0x00000000U)                                       /*!< ADC channel 9 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL9_ENABLE                     ADC_AWD1_2_SELECTION_CHANNEL_9                                /*!< ADC channel 9 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel10 */
/* @DEFINE: watchdog2 monitor channel10 */
#define ADC_WDT2_CHANNEL10_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 10 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL10_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_10                               /*!< ADC channel 10 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel11 */
/* @DEFINE: watchdog2 monitor channel11 */
#define ADC_WDT2_CHANNEL11_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 11 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL11_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_11                               /*!< ADC channel 11 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel12 */
/* @DEFINE: watchdog2 monitor channel12 */
#define ADC_WDT2_CHANNEL12_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 12 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL12_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_12                               /*!< ADC channel 12 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel13 */
/* @DEFINE: watchdog2 monitor channel13 */
#define ADC_WDT2_CHANNEL13_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 13 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL13_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_13                               /*!< ADC channel 13 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel14 */
/* @DEFINE: watchdog2 monitor channel14 */
#define ADC_WDT2_CHANNEL14_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 14 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL14_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_14                               /*!< ADC channel 14 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel15 */
/* @DEFINE: watchdog2 monitor channel15 */
#define ADC_WDT2_CHANNEL15_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 15 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL15_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_15                               /*!< ADC channel 15 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel16 */
/* @DEFINE: watchdog2 monitor channel16 */
#define ADC_WDT2_CHANNEL16_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 16 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL16_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_16                               /*!< ADC channel 16 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel17 */
/* @DEFINE: watchdog2 monitor channel17 */
#define ADC_WDT2_CHANNEL17_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 17 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL17_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_17                               /*!< ADC channel 17 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel18 */
/* @DEFINE: watchdog2 monitor channel18 */
#define ADC_WDT2_CHANNEL18_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 18 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL18_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_18                               /*!< ADC channel 18 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel19 */
/* @DEFINE: watchdog2 monitor channel19 */
#define ADC_WDT2_CHANNEL19_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 19 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL19_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_19                               /*!< ADC channel 19 analog watchdog 1/2 enable */
/* @STRUCT_MEMBER: watchdog2_channel20 */
/* @DEFINE: watchdog2 monitor channel20 */
#define ADC_WDT2_CHANNEL20_DISABLE                   ((uint32_t)0x00000000U)                                       /*!< ADC channel 20 analog watchdog 1/2 disable */
#define ADC_WDT2_CHANNEL20_ENABLE                    ADC_AWD1_2_SELECTION_CHANNEL_20                               /*!< ADC channel 20 analog watchdog 1/2 enable */

/* the callback of ADC interrupt declaration */
typedef void (*hal_adc_dma_handle_cb)(void *ptr);

/* ADC structure type enum */
typedef enum {
    HAL_ADC_INIT_STRUCT = 0U,                                                                                      /*!< ADC initialization structure */
    HAL_ADC_ROUTINE_CONFIG_STRUCT,                                                                                 /*!< ADC routine channel configuration structure */
    HAL_ADC_ROUTINE_RANK_CONFIG_STRUCT,                                                                            /*!< ADC routine rank configuration structure */
    HAL_ADC_WATCHDOG0_CONFIG_STRUCT,                                                                               /*!< ADC watchdog0 configuration structure */
    HAL_ADC_WATCHDOG1_CONFIG_STRUCT,                                                                               /*!< ADC watchdog1 configuration structure */
    HAL_ADC_WATCHDOG2_CONFIG_STRUCT,                                                                               /*!< ADC watchdog2 configuration structure */
    HAL_ADC_DEV_STRUCT,                                                                                            /*!< ADC device information structure */
    HAL_ADC_IRQ_STRUCT,                                                                                            /*!< ADC device interrupt callback function pointer structure */
    HAL_ADC_DMA_HANDLE_CB_STRUCT                                                                                   /*!< ADC DMA callback function pointer structure */
} hal_adc_struct_type_enum;

/* ADC state type enum */
typedef enum {
    HAL_ADC_STATE_RESET        = (uint32_t)0x00000000U,                                                            /*!< ADC is not initialized or disabled */
    HAL_ADC_STATE_READY        = (uint32_t)0x00000001U,                                                            /*!< ADC is ready */
    HAL_ADC_STATE_BUSY_SYSTEM  = (uint32_t)0x00000002U,                                                            /*!< ADC is busy to internal system (initialization, calibration) */
    HAL_ADC_STATE_TIMEOUT      = (uint32_t)0x00000004U,                                                            /*!< ADC timeout occurs */
    HAL_ADC_STATE_ROUTINE_BUSY = (uint32_t)0x00000010U,                                                            /*!< a conversion is ongoing on routine sequence */
    HAL_ADC_STATE_ROUTINE_EOC  = (uint32_t)0x00000020U,                                                            /*!< conversion data available on routine sequence */
    HAL_ADC_STATE_ROVF         = (uint32_t)0x00000040U,                                                            /*!< routine data register overflow event */
    HAL_ADC_STATE_WATCHDOG0    = (uint32_t)0x00001000U,                                                            /*!< analog watchdog0 */
    HAL_ADC_STATE_WATCHDOG1    = (uint32_t)0x00002000U,                                                            /*!< analog watchdog1 */
    HAL_ADC_STATE_WATCHDOG2    = (uint32_t)0x00004000U                                                             /*!< analog watchdog2 */
} hal_adc_state_enum;

/* ADC error type enum */
typedef enum {
    HAL_ADC_ERROR_NONE   = (uint32_t)0x00000000U,                                                                  /*!< no error */
    HAL_ADC_ERROR_SYSTEM = (uint32_t)0x00000001U,                                                                  /*!< ADC internal error: if problem of clocking, enable/disable, wrong state */
    HAL_ADC_ERROR_DMA    = (uint32_t)0x00000002U,                                                                  /*!< DMA transfer error */
    HAL_ADC_ERROR_CONFIG = (uint32_t)0x00000004U                                                                   /*!< configuration error occurs */
} hal_adc_error_enum;

/* @STRUCT_MEMBER: mode */
/* @ENUM:  mode */
typedef enum {
    ADC_ALL_INDEPENDENT_MODE            = ADC_SYNC_MODE_INDEPENDENT,                                               /*!< All ADCs work in standalone mode.*/
    ADC0_AND_ADC1_ROUTINE_PARALLEL_MODE = ADC_DAUL_REGULAL_PARALLEL,                                               /*!< ADC work in routine parallel mode.*/
    ADC0_AND_ADC1_ROUTINE_FOLLOW_MODE   = ADC_DAUL_REGULAL_FOLLOW_UP                                               /*!< ADC work in routine follow mode.*/
} hal_adc_mode_enum;

/* @STRUCT_MEMBER: delay_between_2_sampling_phases */
/* @ENUM:  delay between 2 sampling phases */
typedef enum {
    ADC_DELAY_5_CYCLE  = ADC_SYNC_DELAY_5CYCLE,                                                                    /*!< the delay between 2 sampling phases in ADC synchronization modes to 5 ADC clock cycles. */
    ADC_DELAY_6_CYCLE  = ADC_SYNC_DELAY_6CYCLE,                                                                    /*!< the delay between 2 sampling phases in ADC synchronization modes to 6 ADC clock cycles. */
    ADC_DELAY_7_CYCLE  = ADC_SYNC_DELAY_7CYCLE,                                                                    /*!< the delay between 2 sampling phases in ADC synchronization modes to 7 ADC clock cycles. */
    ADC_DELAY_8_CYCLE  = ADC_SYNC_DELAY_8CYCLE,                                                                    /*!< the delay between 2 sampling phases in ADC synchronization modes to 8 ADC clock cycles. */
    ADC_DELAY_9_CYCLE  = ADC_SYNC_DELAY_9CYCLE,                                                                    /*!< the delay between 2 sampling phases in ADC synchronization modes to 9 ADC clock cycles. */
    ADC_DELAY_10_CYCLE = ADC_SYNC_DELAY_10CYCLE,                                                                   /*!< the delay between 2 sampling phases in ADC synchronization modes to 10 ADC clock cycles. */
    ADC_DELAY_11_CYCLE = ADC_SYNC_DELAY_11CYCLE,                                                                   /*!< the delay between 2 sampling phases in ADC synchronization modes to 11 ADC clock cycles. */
    ADC_DELAY_12_CYCLE = ADC_SYNC_DELAY_12CYCLE,                                                                   /*!< the delay between 2 sampling phases in ADC synchronization modes to 12 ADC clock cycles. */
    ADC_DELAY_13_CYCLE = ADC_SYNC_DELAY_13CYCLE,                                                                   /*!< the delay between 2 sampling phases in ADC synchronization modes to 13 ADC clock cycles. */
    ADC_DELAY_14_CYCLE = ADC_SYNC_DELAY_14CYCLE,                                                                   /*!< the delay between 2 sampling phases in ADC synchronization modes to 14 ADC clock cycles. */
    ADC_DELAY_15_CYCLE = ADC_SYNC_DELAY_15CYCLE,                                                                   /*!< the delay between 2 sampling phases in ADC synchronization modes to 15 ADC clock cycles. */
    ADC_DELAY_16_CYCLE = ADC_SYNC_DELAY_16CYCLE,                                                                   /*!< the delay between 2 sampling phases in ADC synchronization modes to 16 ADC clock cycles. */
    ADC_DELAY_17_CYCLE = ADC_SYNC_DELAY_17CYCLE,                                                                   /*!< the delay between 2 sampling phases in ADC synchronization modes to 17 ADC clock cycles. */
    ADC_DELAY_18_CYCLE = ADC_SYNC_DELAY_18CYCLE,                                                                   /*!< the delay between 2 sampling phases in ADC synchronization modes to 18 ADC clock cycles. */
    ADC_DELAY_19_CYCLE = ADC_SYNC_DELAY_19CYCLE,                                                                   /*!< the delay between 2 sampling phases in ADC synchronization modes to 19 ADC clock cycles. */
    ADC_DELAY_20_CYCLE = ADC_SYNC_DELAY_20CYCLE                                                                    /*!< the delay between 2 sampling phases in ADC synchronization modes to 20 ADC clock cycles. */
} hal_adc_delay_between_2_sampling_phases_enum;

/* @STRUCT_MEMBER: calibration_number */
/* @ENUM:  calibration number */
typedef enum {
    ADC_CALIBRATION_NUMBER_1  = ADC_CALIBRATION_NUM1,                                                              /*!< ADC calibration 1 time */
    ADC_CALIBRATION_NUMBER_2  = ADC_CALIBRATION_NUM2,                                                              /*!< ADC calibration 2 times */
    ADC_CALIBRATION_NUMBER_4  = ADC_CALIBRATION_NUM4,                                                              /*!< ADC calibration 4 times */
    ADC_CALIBRATION_NUMBER_8  = ADC_CALIBRATION_NUM8,                                                              /*!< ADC calibration 8 times */
    ADC_CALIBRATION_NUMBER_16 = ADC_CALIBRATION_NUM16,                                                             /*!< ADC calibration 16 times */
    ADC_CALIBRATION_NUMBER_32 = ADC_CALIBRATION_NUM32                                                              /*!< ADC calibration 32 times */
} hal_adc_calibration_number_enum;

/* @STRUCT_MEMBER: clock_prescaler */
/* @ENUM:  clock prescaler */
typedef enum {
    ADC_ASYNCHRONOUS_CLOCK_DIVIDED_BY_1   = ADC_CLK_ASYNC_DIV1,                                                    /*!< ADC async clock mode div1 */
    ADC_ASYNCHRONOUS_CLOCK_DIVIDED_BY_2   = ADC_CLK_ASYNC_DIV2,                                                    /*!< ADC async clock mode div2 */
    ADC_ASYNCHRONOUS_CLOCK_DIVIDED_BY_4   = ADC_CLK_ASYNC_DIV4,                                                    /*!< ADC async clock mode div4 */
    ADC_ASYNCHRONOUS_CLOCK_DIVIDED_BY_6   = ADC_CLK_ASYNC_DIV6,                                                    /*!< ADC async clock mode div6 */
    ADC_ASYNCHRONOUS_CLOCK_DIVIDED_BY_8   = ADC_CLK_ASYNC_DIV8,                                                    /*!< ADC async clock mode div8 */
    ADC_ASYNCHRONOUS_CLOCK_DIVIDED_BY_10  = ADC_CLK_ASYNC_DIV10,                                                   /*!< ADC async clock mode div10 */
    ADC_ASYNCHRONOUS_CLOCK_DIVIDED_BY_12  = ADC_CLK_ASYNC_DIV12,                                                   /*!< ADC async clock mode div12 */
    ADC_ASYNCHRONOUS_CLOCK_DIVIDED_BY_16  = ADC_CLK_ASYNC_DIV16,                                                   /*!< ADC async clock mode div16 */
    ADC_ASYNCHRONOUS_CLOCK_DIVIDED_BY_32  = ADC_CLK_ASYNC_DIV32,                                                   /*!< ADC async clock mode div32 */
    ADC_ASYNCHRONOUS_CLOCK_DIVIDED_BY_64  = ADC_CLK_ASYNC_DIV64,                                                   /*!< ADC async clock mode div64 */
    ADC_ASYNCHRONOUS_CLOCK_DIVIDED_BY_128 = ADC_CLK_ASYNC_DIV128,                                                  /*!< ADC async clock mode div128 */
    ADC_ASYNCHRONOUS_CLOCK_DIVIDED_BY_256 = ADC_CLK_ASYNC_DIV256,                                                  /*!< ADC async clock mode div256 */
    ADC_SYNCHRONOUS_CLOCK_DIVIDED_BY_2    = ADC_CLK_SYNC_HCLK_DIV2,                                                /*!< ADC sync clock mode HCLK div2 */
    ADC_SYNCHRONOUS_CLOCK_DIVIDED_BY_4    = ADC_CLK_SYNC_HCLK_DIV4,                                                /*!< ADC sync clock mode HCLK div4 */
    ADC_SYNCHRONOUS_CLOCK_DIVIDED_BY_6    = ADC_CLK_SYNC_HCLK_DIV6,                                                /*!< ADC sync clock mode HCLK div6 */
    ADC_SYNCHRONOUS_CLOCK_DIVIDED_BY_8    = ADC_CLK_SYNC_HCLK_DIV8,                                                /*!< ADC sync clock mode HCLK div8 */
    ADC_SYNCHRONOUS_CLOCK_DIVIDED_BY_10   = ADC_CLK_SYNC_HCLK_DIV10,                                               /*!< ADC sync clock mode HCLK div10 */
    ADC_SYNCHRONOUS_CLOCK_DIVIDED_BY_12   = ADC_CLK_SYNC_HCLK_DIV12,                                               /*!< ADC sync clock mode HCLK div12 */
    ADC_SYNCHRONOUS_CLOCK_DIVIDED_BY_14   = ADC_CLK_SYNC_HCLK_DIV14,                                               /*!< ADC sync clock mode HCLK div14 */
    ADC_SYNCHRONOUS_CLOCK_DIVIDED_BY_16   = ADC_CLK_SYNC_HCLK_DIV16                                                /*!< ADC sync clock mode HCLK div16 */
} hal_adc_clock_prescaler_enum;

/* @STRUCT_MEMBER: oversampling_right_shift */
/* @ENUM:  oversampling right shift */
typedef enum {
    ADC_OVERSAMPLE_SHIFT_NONE = (uint16_t)(ADC_OVERSAMPLING_SHIFT_NONE),                                           /*!< no oversampling shift */
    ADC_OVERSAMPLE_SHIFT_1B   = (uint16_t)(ADC_OVERSAMPLING_SHIFT_1B),                                             /*!< 1-bit oversampling shift */
    ADC_OVERSAMPLE_SHIFT_2B   = (uint16_t)(ADC_OVERSAMPLING_SHIFT_2B),                                             /*!< 2-bit oversampling shift */
    ADC_OVERSAMPLE_SHIFT_3B   = (uint16_t)(ADC_OVERSAMPLING_SHIFT_3B),                                             /*!< 3-bit oversampling shift */
    ADC_OVERSAMPLE_SHIFT_4B   = (uint16_t)(ADC_OVERSAMPLING_SHIFT_4B),                                             /*!< 4-bit oversampling shift */
    ADC_OVERSAMPLE_SHIFT_5B   = (uint16_t)(ADC_OVERSAMPLING_SHIFT_5B),                                             /*!< 5-bit oversampling shift */
    ADC_OVERSAMPLE_SHIFT_6B   = (uint16_t)(ADC_OVERSAMPLING_SHIFT_6B),                                             /*!< 6-bit oversampling shift */
    ADC_OVERSAMPLE_SHIFT_7B   = (uint16_t)(ADC_OVERSAMPLING_SHIFT_7B),                                             /*!< 7-bit oversampling shift */
    ADC_OVERSAMPLE_SHIFT_8B   = (uint16_t)(ADC_OVERSAMPLING_SHIFT_8B),                                             /*!< 8-bit oversampling shift */
    ADC_OVERSAMPLE_SHIFT_9B   = (uint16_t)(ADC_OVERSAMPLING_SHIFT_9B),                                             /*!< 8-bit oversampling shift */
    ADC_OVERSAMPLE_SHIFT_10B  = (uint16_t)(ADC_OVERSAMPLING_SHIFT_10B),                                            /*!< 8-bit oversampling shift */
    ADC_OVERSAMPLE_SHIFT_11B  = (uint16_t)(ADC_OVERSAMPLING_SHIFT_11B)                                             /*!< 8-bit oversampling shift */
} hal_adc_oversampling_right_shift_enum;

/* @STRUCT_MEMBER: number_of_conversions */
/* @ENUM:  number of conversions */
typedef enum {
    ADC_NUMBER_OF_CONVERSIONS_0  = ((uint32_t)0x00000000U),                                                        /*!< ADC routine channel sequence 0 */
    ADC_NUMBER_OF_CONVERSIONS_1  = ((uint32_t)0x00000001U),                                                        /*!< ADC routine channel sequence 1 */
    ADC_NUMBER_OF_CONVERSIONS_2  = ((uint32_t)0x00000002U),                                                        /*!< ADC routine channel sequence 2 */
    ADC_NUMBER_OF_CONVERSIONS_3  = ((uint32_t)0x00000003U),                                                        /*!< ADC routine channel sequence 3 */
    ADC_NUMBER_OF_CONVERSIONS_4  = ((uint32_t)0x00000004U),                                                        /*!< ADC routine channel sequence 4 */
    ADC_NUMBER_OF_CONVERSIONS_5  = ((uint32_t)0x00000005U),                                                        /*!< ADC routine channel sequence 5 */
    ADC_NUMBER_OF_CONVERSIONS_6  = ((uint32_t)0x00000006U),                                                        /*!< ADC routine channel sequence 6 */
    ADC_NUMBER_OF_CONVERSIONS_7  = ((uint32_t)0x00000007U),                                                        /*!< ADC routine channel sequence 7 */
    ADC_NUMBER_OF_CONVERSIONS_8  = ((uint32_t)0x00000008U),                                                        /*!< ADC routine channel sequence 8 */
    ADC_NUMBER_OF_CONVERSIONS_9  = ((uint32_t)0x00000009U),                                                        /*!< ADC routine channel sequence 9 */
    ADC_NUMBER_OF_CONVERSIONS_10 = ((uint32_t)0x0000000AU),                                                        /*!< ADC routine channel sequence 10 */
    ADC_NUMBER_OF_CONVERSIONS_11 = ((uint32_t)0x0000000BU),                                                        /*!< ADC routine channel sequence 11 */
    ADC_NUMBER_OF_CONVERSIONS_12 = ((uint32_t)0x0000000CU),                                                        /*!< ADC routine channel sequence 12 */
    ADC_NUMBER_OF_CONVERSIONS_13 = ((uint32_t)0x0000000DU),                                                        /*!< ADC routine channel sequence 13 */
    ADC_NUMBER_OF_CONVERSIONS_14 = ((uint32_t)0x0000000EU),                                                        /*!< ADC routine channel sequence 14 */
    ADC_NUMBER_OF_CONVERSIONS_15 = ((uint32_t)0x0000000FU),                                                        /*!< ADC routine channel sequence 15 */
    ADC_NUMBER_OF_CONVERSIONS_16 = ((uint32_t)0x00000010U)                                                         /*!< ADC routine channel sequence 16 */
} hal_adc_number_of_conversions_enum;

/* @STRUCT_MEMBER: routine_external_trigger_edge */
/* @ENUM:  routine external trigger edge */
typedef enum {
    ADC_ROUTINE_EXTERNAL_TRIGGER_EDGE_NONE           = EXTERNAL_TRIGGER_DISABLE,                                   /*!< no external trigger */
    ADC_ROUTINE_EXTERNAL_TRIGGER_EDGE_RISING         = EXTERNAL_TRIGGER_RISING,                                    /*!< rising edge of external trigger */
    ADC_ROUTINE_EXTERNAL_TRIGGER_EDGE_FALLING        = EXTERNAL_TRIGGER_FALLING,                                   /*!< falling edge of external trigger */
    ADC_ROUTINE_EXTERNAL_TRIGGER_EDGE_RISING_FALLING = EXTERNAL_TRIGGER_RISING_FALLING                             /*!< rising and falling edge of external trigger */
} hal_adc_routine_external_trigger_edge_enum;

/* @STRUCT_MEMBER: routine_sequence */
/* @ENUM:  routine rank sequence */
typedef enum {
    ADC_ROUTINE_SEQUENCE_0  = ((uint8_t)0x00U),                                                                    /*!< ADC routine channel sequence 1 */
    ADC_ROUTINE_SEQUENCE_1  = ((uint8_t)0x01U),                                                                    /*!< ADC routine channel sequence 2 */
    ADC_ROUTINE_SEQUENCE_2  = ((uint8_t)0x02U),                                                                    /*!< ADC routine channel sequence 3 */
    ADC_ROUTINE_SEQUENCE_3  = ((uint8_t)0x03U),                                                                    /*!< ADC routine channel sequence 4 */
    ADC_ROUTINE_SEQUENCE_4  = ((uint8_t)0x04U),                                                                    /*!< ADC routine channel sequence 5 */
    ADC_ROUTINE_SEQUENCE_5  = ((uint8_t)0x05U),                                                                    /*!< ADC routine channel sequence 6 */
    ADC_ROUTINE_SEQUENCE_6  = ((uint8_t)0x06U),                                                                    /*!< ADC routine channel sequence 7 */
    ADC_ROUTINE_SEQUENCE_7  = ((uint8_t)0x07U),                                                                    /*!< ADC routine channel sequence 8 */
    ADC_ROUTINE_SEQUENCE_8  = ((uint8_t)0x08U),                                                                    /*!< ADC routine channel sequence 9 */
    ADC_ROUTINE_SEQUENCE_9  = ((uint8_t)0x09U),                                                                    /*!< ADC routine channel sequence 10 */
    ADC_ROUTINE_SEQUENCE_10 = ((uint8_t)0x0AU),                                                                    /*!< ADC routine channel sequence 11 */
    ADC_ROUTINE_SEQUENCE_11 = ((uint8_t)0x0BU),                                                                    /*!< ADC routine channel sequence 12 */
    ADC_ROUTINE_SEQUENCE_12 = ((uint8_t)0x0CU),                                                                    /*!< ADC routine channel sequence 13 */
    ADC_ROUTINE_SEQUENCE_13 = ((uint8_t)0x0DU),                                                                    /*!< ADC routine channel sequence 14 */
    ADC_ROUTINE_SEQUENCE_14 = ((uint8_t)0x0EU),                                                                    /*!< ADC routine channel sequence 15 */
    ADC_ROUTINE_SEQUENCE_15 = ((uint8_t)0x0FU)                                                                     /*!< ADC routine channel sequence 16 */
} hal_adc_routine_sequence_enum;

/* ADC common settings */
/* @PARA: p_init */
/* @STRUCT: ADC settings config struct */
typedef struct {
    hal_adc_mode_enum mode;                                                                                        /*!< ADC work mode */
    uint32_t dma_sync_mode;                                                                                        /*!< ADC dma sync mode */
    uint32_t dma_sync_cont_req;                                                                                    /*!< ADC dma sync continuous mode */
    hal_adc_delay_between_2_sampling_phases_enum delay_between_2_sampling_phases;                                  /*!< the delay between 2 sampling phases in ADC synchronization modes */
    uint32_t difctl_in0;                                                                                           /*!< ADC channel 0 differential mode */
    uint32_t difctl_in1;                                                                                           /*!< ADC channel 1 differential mode */
    uint32_t difctl_in2;                                                                                           /*!< ADC channel 2 differential mode */
    uint32_t difctl_in3;                                                                                           /*!< ADC channel 3 differential mode */
    uint32_t difctl_in4;                                                                                           /*!< ADC channel 4 differential mode */
    uint32_t difctl_in5;                                                                                           /*!< ADC channel 5 differential mode */
    uint32_t difctl_in6;                                                                                           /*!< ADC channel 6 differential mode */
    uint32_t difctl_in7;                                                                                           /*!< ADC channel 7 differential mode */
    uint32_t difctl_in8;                                                                                           /*!< ADC channel 8 differential mode */
    uint32_t difctl_in9;                                                                                           /*!< ADC channel 9 differential mode */
    uint32_t difctl_in10;                                                                                          /*!< ADC channel 10 differential mode */
    uint32_t difctl_in11;                                                                                          /*!< ADC channel 11 differential mode */
    uint32_t difctl_in12;                                                                                          /*!< ADC channel 12 differential mode */
    uint32_t difctl_in13;                                                                                          /*!< ADC channel 13 differential mode */
    uint32_t difctl_in14;                                                                                          /*!< ADC channel 14 differential mode */
    uint32_t difctl_in15;                                                                                          /*!< ADC channel 15 differential mode */
    uint32_t difctl_in16;                                                                                          /*!< ADC channel 16 differential mode */
    uint32_t difctl_in17;                                                                                          /*!< ADC channel 17 differential mode */
    uint32_t difctl_in18;                                                                                          /*!< ADC channel 18 differential mode */
    uint32_t difctl_in19;                                                                                          /*!< ADC channel 19 differential mode */
    uint32_t difctl_in20;                                                                                          /*!< ADC channel 20 differential mode */
    uint32_t difctl_in21;                                                                                          /*!< ADC channel 21 differential mode */
    uint32_t data_alignment;                                                                                       /*!< ADC data alignment */
    uint32_t calibration;                                                                                          /*!< ADC calibration disable or enable */
    uint32_t calibration_mode;                                                                                     /*!< ADC calibration mode */
    hal_adc_calibration_number_enum calibration_number;                                                            /*!< ADC calibration times */
    hal_adc_clock_prescaler_enum clock_prescaler;                                                                  /*!< ADC clock prescaler */
    uint32_t resolution;                                                                                           /*!< ADC data resolution select */
    uint32_t scan_mode;                                                                                            /*!< scan mode */
    uint32_t hardware_oversampling;                                                                                /*!< hardware oversampling */
    hal_adc_oversampling_right_shift_enum oversampling_right_shift;                                                /*!< oversampling right shift */
    uint16_t oversampling_ratio;                                                                                   /*!< oversampling ratio */
    uint32_t triggered_oversampling;                                                                               /*!< triggered oversampling */
    uint32_t hpdf_mode;                                                                                            /*!< HPDF mode disable or enable */
} hal_adc_init_struct;

/* ADC routine conversion config */
/* @PARA: p_routine */
/* @STRUCT: ADC routine conversion config struct */
typedef struct {
    ControlStatus routine_conversions;                                                                             /*!< ADC routine conversion disable or enable */
    hal_adc_number_of_conversions_enum number_of_conversions;                                                      /*!< ADC routine channel sequence */
    uint32_t routine_external_trigger_source;                                                                      /*!< ADC routine external trigger source */
    hal_adc_routine_external_trigger_edge_enum routine_external_trigger_edge;                                      /*!< ADC routine external trigger edge */
    uint32_t continuous_mode;                                                                                      /*!< continuous mode */
    uint32_t discontinuous_mode;                                                                                   /*!< discontinuous mode */
    uint32_t number_of_conversions_in_discontinuous_mode;                                                          /*!< number of conversions in discontinuous mode */
    uint32_t dma_cont_req;                                                                                         /*!< ADC routine channel dma continuous requests */
} hal_adc_routine_config_struct;

/* ADC routine rank config */
/* @PARA: p_routine_rank */
/* @STRUCT: ADC routine rank config struct */
typedef struct {
    uint8_t channel;                                                                                               /*!< ADC channel select */
    uint32_t sampling_time;                                                                                        /*!< ADC sample time */
    hal_adc_routine_sequence_enum routine_sequence;                                                                /*!< ADC routine sequence */
} hal_adc_routine_rank_config_struct;

/* analog watchdog0 settings */
/* @PARA: p_watchdog0 */
/* @STRUCT: analog watchdog0 settings config struct */
typedef struct {
    uint32_t routine_seq_watchdog0;                                                                                /*!< ADC analog watchdog 0 monitor routine sequence disable or enable */
    uint32_t watchdog0_mode;                                                                                       /*!< ADC analog watchdog 0 monitor all channels or single channel */
    uint32_t watchdog0_channel;                                                                                    /*!< ADC analog0 monitor channel number */
    uint32_t watchdog0_high_threshold;                                                                             /*!< ADC analog watchdog high threshold */
    uint32_t watchdog0_low_threshold;                                                                              /*!< ADC analog watchdog low threshold */
} hal_adc_watchdog0_config_struct;

/* @STRUCT_MEMBER: watchdog0_high_threshold */
/* @=NULL */

/* @STRUCT_MEMBER: watchdog0_low_threshold */
/* @=NULL */

/* Analog Watchdog1 Settings */
/* @PARA: p_watchdog1 */
/* @STRUCT: analog watchdog1 settings config struct */
typedef struct {
    uint32_t watchdog1_channel0;                                                                                   /*!< ADC analog watchdog1 monitor channel0 */
    uint32_t watchdog1_channel1;                                                                                   /*!< ADC analog watchdog1 monitor channel1 */
    uint32_t watchdog1_channel2;                                                                                   /*!< ADC analog watchdog1 monitor channel2 */
    uint32_t watchdog1_channel3;                                                                                   /*!< ADC analog watchdog1 monitor channel3 */
    uint32_t watchdog1_channel4;                                                                                   /*!< ADC analog watchdog1 monitor channel4 */
    uint32_t watchdog1_channel5;                                                                                   /*!< ADC analog watchdog1 monitor channel5 */
    uint32_t watchdog1_channel6;                                                                                   /*!< ADC analog watchdog1 monitor channel6 */
    uint32_t watchdog1_channel7;                                                                                   /*!< ADC analog watchdog1 monitor channel7 */
    uint32_t watchdog1_channel8;                                                                                   /*!< ADC analog watchdog1 monitor channel8 */
    uint32_t watchdog1_channel9;                                                                                   /*!< ADC analog watchdog1 monitor channel9 */
    uint32_t watchdog1_channel10;                                                                                  /*!< ADC analog watchdog1 monitor channel10 */
    uint32_t watchdog1_channel11;                                                                                  /*!< ADC analog watchdog1 monitor channel11 */
    uint32_t watchdog1_channel12;                                                                                  /*!< ADC analog watchdog1 monitor channel12 */
    uint32_t watchdog1_channel13;                                                                                  /*!< ADC analog watchdog1 monitor channel13 */
    uint32_t watchdog1_channel14;                                                                                  /*!< ADC analog watchdog1 monitor channel14 */
    uint32_t watchdog1_channel15;                                                                                  /*!< ADC analog watchdog1 monitor channel15 */
    uint32_t watchdog1_channel16;                                                                                  /*!< ADC analog watchdog1 monitor channel16 */
    uint32_t watchdog1_channel17;                                                                                  /*!< ADC analog watchdog1 monitor channel17 */
    uint32_t watchdog1_channel18;                                                                                  /*!< ADC analog watchdog1 monitor channel18 */
    uint32_t watchdog1_channel19;                                                                                  /*!< ADC analog watchdog1 monitor channel19 */
    uint32_t watchdog1_channel20;                                                                                  /*!< ADC analog watchdog1 monitor channel20 */
    uint32_t watchdog1_high_threshold;                                                                             /*!< ADC analog watchdog1 high threshold */
    uint32_t watchdog1_low_threshold;                                                                              /*!< ADC analog watchdog1 low threshold */
} hal_adc_watchdog1_config_struct;

/* @STRUCT_MEMBER: watchdog1_high_threshold */
/* @=NULL */

/* @STRUCT_MEMBER: watchdog1_low_threshold */
/* @=NULL */

/* analog watchdog1 settings */
/* @PARA: p_watchdog2 */
/* @STRUCT: analog watchdog2 settings config struct */
typedef struct {
    uint32_t watchdog2_channel0;                                                                                   /*!< ADC analog watchdog2 monitor channel0 */
    uint32_t watchdog2_channel1;                                                                                   /*!< ADC analog watchdog2 monitor channel1 */
    uint32_t watchdog2_channel2;                                                                                   /*!< ADC analog watchdog2 monitor channel2 */
    uint32_t watchdog2_channel3;                                                                                   /*!< ADC analog watchdog2 monitor channel3 */
    uint32_t watchdog2_channel4;                                                                                   /*!< ADC analog watchdog2 monitor channel4 */
    uint32_t watchdog2_channel5;                                                                                   /*!< ADC analog watchdog2 monitor channel5 */
    uint32_t watchdog2_channel6;                                                                                   /*!< ADC analog watchdog2 monitor channel6 */
    uint32_t watchdog2_channel7;                                                                                   /*!< ADC analog watchdog2 monitor channel7 */
    uint32_t watchdog2_channel8;                                                                                   /*!< ADC analog watchdog2 monitor channel8 */
    uint32_t watchdog2_channel9;                                                                                   /*!< ADC analog watchdog2 monitor channel9 */
    uint32_t watchdog2_channel10;                                                                                  /*!< ADC analog watchdog2 monitor channel10 */
    uint32_t watchdog2_channel11;                                                                                  /*!< ADC analog watchdog2 monitor channel11 */
    uint32_t watchdog2_channel12;                                                                                  /*!< ADC analog watchdog2 monitor channel12 */
    uint32_t watchdog2_channel13;                                                                                  /*!< ADC analog watchdog2 monitor channel13 */
    uint32_t watchdog2_channel14;                                                                                  /*!< ADC analog watchdog2 monitor channel14 */
    uint32_t watchdog2_channel15;                                                                                  /*!< ADC analog watchdog2 monitor channel15 */
    uint32_t watchdog2_channel16;                                                                                  /*!< ADC analog watchdog2 monitor channel16 */
    uint32_t watchdog2_channel17;                                                                                  /*!< ADC analog watchdog2 monitor channel17 */
    uint32_t watchdog2_channel18;                                                                                  /*!< ADC analog watchdog2 monitor channel18 */
    uint32_t watchdog2_channel19;                                                                                  /*!< ADC analog watchdog2 monitor channel19 */
    uint32_t watchdog2_channel20;                                                                                  /*!< ADC analog watchdog2 monitor channel20 */
    uint32_t watchdog2_high_threshold;                                                                             /*!< ADC analog watchdog2 high threshold */
    uint32_t watchdog2_low_threshold;                                                                              /*!< ADC analog watchdog2 low threshold */
} hal_adc_watchdog2_config_struct;

/* ADC device interrupt callback function pointer structure */
typedef struct {
    hal_irq_handle_cb adc_eoc_handle;                                                                              /*!< EOC interrupt handler function */
    hal_irq_handle_cb adc_rovf_handle;                                                                             /*!< ROVF interrupt handler function */
    hal_irq_handle_cb adc_watchdog0_handle;                                                                        /*!< watchdog0 event interrupt handler function */
    hal_irq_handle_cb adc_watchdog1_handle;                                                                        /*!< watchdog1 event interrupt handler function */
    hal_irq_handle_cb adc_watchdog2_handle;                                                                        /*!< watchdog2 event interrupt handler function */
    hal_irq_handle_cb adc_dma_full_complete_handle;                                                                /*!< dma full transfer complete handler function */
    hal_irq_handle_cb adc_dma_half_complete_handle;                                                                /*!< dma half transfer complete handler function */
    hal_irq_handle_cb adc_dma_error_handle;                                                                        /*!< dma transfer error handler function */
} hal_adc_irq_struct;

/* ADC device information structure */
typedef struct {
    uint32_t periph;                                                                                               /*!< ADC device */
    hal_adc_irq_struct adc_irq;                                                                                    /*!< ADC device interrupt callback function pointer structure */
    hal_dma_dev_struct *p_dma_adc;                                                                                 /*!< DMA device information structure */
    __IO hal_adc_error_enum error_state;                                                                           /*!< ADC error state */
    __IO hal_mutex_enum mutex;                                                                                     /*!< ADC mutex set */
    __IO hal_adc_state_enum state;                                                                                 /*!< ADC state */
    void *adc_eoc_callback;                                                                                        /*!< adc eoc callback */
    void *adc_rovf_callback;                                                                                       /*!< adc rovf callback */
    void *adc_watchdog0_callback;                                                                                  /*!< adc watchdog0 callback */
    void *adc_watchdog1_callback;                                                                                  /*!< adc watchdog1 callback */
    void *adc_watchdog2_callback;                                                                                  /*!< adc watchdog2 callback */
    void *adc_dma_full_complete_callback;                                                                          /*!< dma full transfer complete handler function */
    void *adc_dma_half_complete_callback;                                                                          /*!< dma half transfer complete handler function */
    void *adc_dma_error_callback;                                                                                  /*!< dma transfer error handler function */
} hal_adc_dev_struct;

/* ADC device interrupt callback function pointer structure */
typedef struct {
    hal_irq_handle_cb adc_eoc_callback;                                                                            /*!< EOC interrupt handler function */
    hal_irq_handle_cb adc_rovf_callback;                                                                           /*!< ROVF interrupt handler function */
    hal_irq_handle_cb adc_watchdog0_callback;                                                                      /*!< watchdog0 event interrupt handler function */
    hal_irq_handle_cb adc_watchdog1_callback;                                                                      /*!< watchdog1 event interrupt handler function */
    hal_irq_handle_cb adc_watchdog2_callback;                                                                      /*!< watchdog2 event interrupt handler function */
    hal_irq_handle_cb adc_dma_full_complete_callback;                                                              /*!< dma full transfer complete handler function */
    hal_irq_handle_cb adc_dma_half_complete_callback;                                                              /*!< dma half transfer complete handler function */
    hal_irq_handle_cb adc_dma_error_callback;                                                                      /*!< dma transfer error handler function */
} hal_adc_irq_user_callback_struct;

/* @STRUCT_MEMBER: watchdog2_high_threshold */
/* @=NULL */

/* @STRUCT_MEMBER: watchdog2_low_threshold */
/* @=NULL */

/* function declarations */
/* initialize the ADC structure with the default values */
int32_t hal_adc_struct_init(hal_adc_struct_type_enum hal_struct_type, void *p_struct);
/* deinitialize ADC */
int32_t hal_adc_deinit(hal_adc_dev_struct *adc_dev);
/* @FUNCTION: initialize ADC */
int32_t hal_adc_init(hal_adc_dev_struct *adc_dev, uint32_t adc_periph, hal_adc_init_struct *p_init);
/* start ADC calibration function */
int32_t hal_adc_calibration_start(hal_adc_dev_struct *adc_dev);

/* ADC interrupt handle functions */
/* ADC interrupt handler content function, which is merely used in ADC_IRQHandler */
void hal_adc_irq(hal_adc_dev_struct *adc_dev);
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_adc_irq_handle_set(hal_adc_dev_struct *adc_dev, hal_adc_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_adc_irq_handle_all_reset(hal_adc_dev_struct *adc_dev);

/* ADC routine channel */
/* @FUNCTION: routine config */
int32_t hal_adc_routine_config(hal_adc_dev_struct *adc_dev, uint32_t adc_periph, \
                               hal_adc_routine_config_struct *p_routine);
/* @FUNCTION: routine rank config */
int32_t hal_adc_routine_rank_config(hal_adc_dev_struct *adc_dev, uint32_t adc_periph, \
                                    hal_adc_routine_rank_config_struct *p_routine_rank);
/* enable ADC and start the conversion of routine sequence */
int32_t hal_adc_routine_start(hal_adc_dev_struct *adc_dev);
/* stop the conversion of routine sequence and disable ADC */
int32_t hal_adc_routine_stop(hal_adc_dev_struct *adc_dev);
/* enable ADC and start the conversion of routine sequence */
int32_t hal_adc_routine_start_interrupt(hal_adc_dev_struct *adc_dev, hal_adc_irq_user_callback_struct *p_user_func);
/* stop the conversion of routine sequence and disable ADC */
int32_t hal_adc_routine_stop_interrupt(hal_adc_dev_struct *adc_dev);
/* enable ADC and start the conversion of routine sequence with DMA */
int32_t hal_adc_routine_start_dma(hal_adc_dev_struct *adc_dev, hal_adc_irq_user_callback_struct *p_user_func, \
                                  uint16_t length, uint32_t *pdata);
/* stop the conversion of routine sequence, disable ADC DMA mode and disable ADC */
int32_t hal_adc_routine_stop_dma(hal_adc_dev_struct *adc_dev);
/* enable ADC and start the conversion of routine sequence with sync DMA */
int32_t hal_adc_routine_sync_start_dma(hal_adc_dev_struct *adc_dev, hal_adc_irq_user_callback_struct *p_user_func, \
                                       uint16_t length, uint32_t *pdata);
/* stop the conversion of routine sequence, disable ADC DMA mode and disable ADC */
int32_t hal_adc_routine_sync_stop_dma(hal_adc_dev_struct *adc_dev);
/* polling for ADC routine sequence conversion */
int32_t hal_adc_routine_conversion_poll(hal_adc_dev_struct *adc_dev, uint32_t timeout);
/* enable ADC routine sequence software trigger */
int32_t hal_adc_routine_software_trigger_enable(hal_adc_dev_struct *adc_dev);

/* ADC watchdog*/
/* polling for analog watchdog event */
int32_t hal_adc_watchdog_event_poll(hal_adc_dev_struct *adc_dev, uint32_t event_type, uint32_t timeout);
/* ADC watchdog0 */
/* @FUNCTION: configure watchdog0 */
int32_t hal_adc_watchdog0_config(hal_adc_dev_struct *adc_dev, uint32_t adc_periph, \
                                 hal_adc_watchdog0_config_struct *p_watchdog0);
/* enable ADC watchdog0 interrupt */
int32_t hal_adc_watchdog0_interrupt_enable(hal_adc_dev_struct *adc_dev);
/* disable ADC watchdog0 interrupt */
int32_t hal_adc_watchdog0_interrupt_disable(hal_adc_dev_struct *adc_dev);

/* ADC watchdog1 */
/* @FUNCTION: configure watchdog1 */
int32_t hal_adc_watchdog1_config(hal_adc_dev_struct *adc_dev, uint32_t adc_periph, \
                                 hal_adc_watchdog1_config_struct *p_watchdog1);
/* enable ADC watchdog1 interrupt */
int32_t hal_adc_watchdog1_interrupt_enable(hal_adc_dev_struct *adc_dev);
/* disable ADC watchdog1 interrupt */
int32_t hal_adc_watchdog1_interrupt_disable(hal_adc_dev_struct *adc_dev);

/* ADC watchdog2 */
/* @FUNCTION: configure watchdog2 */
int32_t hal_adc_watchdog2_config(hal_adc_dev_struct *adc_dev, uint32_t adc_periph, \
                                 hal_adc_watchdog2_config_struct *p_watchdog2);
/* @END */
/* enable ADC watchdog2 interrupt */
int32_t hal_adc_watchdog2_interrupt_enable(hal_adc_dev_struct *adc_dev);
/* disable ADC watchdog2 interrupt */
int32_t hal_adc_watchdog2_interrupt_disable(hal_adc_dev_struct *adc_dev);

/* get ADC value and state */
/* get routine sequence conversion result */
uint32_t hal_adc_routine_value_get(hal_adc_dev_struct *adc_dev);
/* get ADC error */
hal_adc_error_enum hal_adc_error_get(hal_adc_dev_struct *adc_dev);
/* get ADC state */
hal_adc_state_enum hal_adc_state_get(hal_adc_dev_struct *adc_dev);

/* function declarations */
/* enable ADC interface */
void hals_adc_enable(uint32_t adc_periph);
/* disable ADC interface */
void hals_adc_disable(uint32_t adc_periph);

/* ADC calibration and reset calibration */
void hals_adc_calibration_enable(uint32_t adc_periph);
/* configure ADC calibration mode */
int32_t hals_adc_calibration_mode_config(uint32_t adc_periph, uint32_t clb_mode);
/* configure ADC calibration number */
void hals_adc_calibration_number(uint32_t adc_periph, hal_adc_calibration_number_enum clb_num);
/* enable DMA request */
void hals_adc_dma_mode_enable(uint32_t adc_periph);
/* disable DMA request */
void hals_adc_dma_mode_disable(uint32_t adc_periph);
/* configure the ADC sync mode */
void hals_adc_sync_mode_config(hal_adc_mode_enum sync_mode);
/* configure ADC sync DMA mode selection */
void hals_adc_sync_dma_config(uint32_t dma_mode);
/* configure the delay between 2 sampling phases in ADC sync modes */
void hals_adc_sync_delay_config(hal_adc_delay_between_2_sampling_phases_enum sample_delay);
/* configure ADC data alignment */
int32_t hals_adc_data_alignment_config(uint32_t adc_periph, uint32_t data_alignment);
/* configure differential mode for ADC channel */
void hals_adc_channel_differential_mode_config(uint32_t adc_periph, uint32_t adc_channel, ControlStatus newvalue);
/* configure the ADC clock */
void hals_adc_clock_config(uint32_t adc_periph, hal_adc_clock_prescaler_enum prescaler);
/* configure ADC resolution */
int32_t hals_adc_resolution_config(uint32_t adc_periph, uint32_t resolution);
/* enable or disable ADC special function */
void hals_adc_special_function_config(uint32_t adc_periph, uint32_t function, uint32_t newvalue);
/* configure ADC oversample mode */
int32_t hals_adc_oversample_mode_config(uint32_t adc_periph, uint32_t mode, hal_adc_oversampling_right_shift_enum shift, uint16_t ratio);
/* enable ADC oversample mode */
void hals_adc_oversample_mode_enable(uint32_t adc_periph);
/* disable ADC oversample mode */
void hals_adc_oversample_mode_disable(uint32_t adc_periph);
/* enable hpdf mode */
void hals_adc_hpdf_mode_enable(uint32_t adc_periph);
/* disable hpdf mode */
void hals_adc_hpdf_mode_disable(uint32_t adc_periph);
/* configure the length of routine channel sequence */
void hals_adc_channel_length_config(uint32_t adc_periph, uint8_t adc_sequence, hal_adc_number_of_conversions_enum length);
/* configure ADC routine channel */
int32_t hals_adc_routine_channel_config(uint32_t adc_periph, uint8_t rank, uint8_t adc_channel, uint32_t sample_time);
/* enable ADC external trigger */
void hals_adc_routine_external_trigger_config(uint32_t adc_periph, hal_adc_routine_external_trigger_edge_enum trigger_mode);
/* enable ADC software trigger */
void hals_adc_software_trigger_enable(uint32_t adc_periph, uint8_t adc_sequence);

/* read ADC routine sequence data register */
uint32_t hals_adc_routine_data_read(uint32_t adc_periph);
/* configure ADC internal channel */
int32_t hals_adc_internal_channel_config(uint32_t adc_periph, uint8_t adc_channel);
/* configure ADC analog watchdog 0 single channel */
void hals_adc_watchdog0_single_channel_enable(uint32_t adc_periph, uint8_t adc_channel);
/* configure ADC analog watchdog 0 sequence channel */
void hals_adc_watchdog0_sequence_channel_enable(uint32_t adc_periph, uint8_t adc_sequence);
/* configure ADC analog watchdog 0 threshold */
void hals_adc_watchdog0_threshold_config(uint32_t adc_periph, uint32_t low_threshold, uint32_t high_threshold);
/* configure ADC analog watchdog 1 threshold */
void hals_adc_watchdog1_threshold_config(uint32_t adc_periph, uint32_t low_threshold, uint32_t high_threshold);
/* configure ADC analog watchdog 2 threshold */
void hals_adc_watchdog2_threshold_config(uint32_t adc_periph, uint32_t low_threshold, uint32_t high_threshold);
/* get the ADC flag bits */
FlagStatus hals_adc_flag_get(uint32_t adc_periph, uint32_t flag);
/* clear the ADC flag bits */
void hals_adc_flag_clear(uint32_t adc_periph, uint32_t flag);
/* enable ADC interrupt */
void hals_adc_interrupt_enable(uint32_t adc_periph, uint32_t interrupt);
/* disable ADC interrupt */
void hals_adc_interrupt_disable(uint32_t adc_periph, uint32_t interrupt);
/* clear the ADC flag */
void hals_adc_interrupt_flag_clear(uint32_t adc_periph, uint32_t int_flag);
/* get the ADC interrupt bits */
FlagStatus hals_adc_interrupt_flag_get(uint32_t adc_periph, uint32_t int_flag);

#endif /* GD32H7XX_HAL_ADC_H */
