/*
    \file  main.c
*/
/*
    Copyright (c) 2023, GigaDevice Semiconductor Inc.

    All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this 
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice, 
       this list of conditions and the following disclaimer in the documentation 
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors 
       may be used to endorse or promote products derived from this software without 
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
*/
#include "gd32f3x0_hal.h"
#include "gd32f3x0_hal_init.h"
/* user code [global 0] begin */
__IO uint8_t g_wakeup = 0x00;
__IO uint8_t g_counter = 0x00;
hal_uart_irq_struct uart_irq;

void uart_wakeup_cb(void *uart);
void systick_cb(void);
void led_spark(void);
/* user code [global 0] end */

/*!
    \brief      main function
    \param[in]  none
    \param[out] none
    \retval     none
*/
int main(void)
{
    /* user code [local 0] begin */
    uint8_t rx_buf[1];
    hal_uart_struct_init(HAL_UART_IRQ_INIT_STRUCT, &uart_irq);
    /* user code [local 0] end */

    msd_system_init();
    msd_clock_init();
    /* user code [local 1] begin */

    /* user code [local 1] end */
    msd_gpio_init();
    msd_usart0_init();

    /* user code [local 2] begin */
    hal_sys_basetick_delay_ms(2000);
    /* use start bit wakeup MCU */
    hals_uart_wakeup_mode_config(USART0, USART_WUM_STARTB);

    /* enable USART */
    hals_uart_enable(USART0);

    /* ensure USART is enabled */
    while(RESET == hals_uart_flag_get(uart0_info.periph, USART_FLAG_REA));
    /* check USART is not transmitting */
    while(SET == hals_uart_flag_get(uart0_info.periph, USART_FLAG_BSY));

    /* set the wakeup handle function */
    uart_irq.wakeup_handle = uart_wakeup_cb;
    hal_uart_irq_handle_set(&uart0_info, &uart_irq);
    /* enable USART wakeup mode */
    hals_uart_wakeup_enable(USART0);

    /* enable PWU APB clock */
    hal_rcu_periph_clk_enable(RCU_PMU);
    /* enter deep-sleep mode */
    hals_pmu_to_deepsleepmode(PMU_LDO_LOWPOWER, PMU_LOWDRIVER_ENABLE, WFI_CMD);

    /* wait a WUIE interrupt event */
    while(0x00 == g_wakeup);

    /* disable USART peripheral in deepsleep mode */
    hals_uart_wakeup_disable(USART0);

    hal_uart_receive_poll(&uart0_info, rx_buf, 1, 0x1FFFFF);

    /* disable the USART */
    hals_uart_disable(USART0);

    /* reconfigure systemclock */
    msd_system_init();
    msd_clock_init();
    /* user code [local 2] end */

    while(1){
    /* user code [local 3] begin */

    /* user code [local 3] end */
    }
}
/* user code [global 1] begin */
/*!
    \brief      uart wakeup user callback function
    \param[in]  uart: uart device information structure
    \param[out] none
    \retval     none
*/
void uart_wakeup_cb(void *uart)
{
    g_wakeup = 1;
}

/*!
    \brief      LED spark
    \param[in]  none
    \param[out] none
    \retval     none
*/
void led_spark(void)
{
    static __IO uint32_t time_delay = 0;

    if(0x00 != time_delay) {
        if(time_delay < 500) {
            /* light on */
            hal_gpio_bit_set(LED2_GPIO_PORT, LED2_PIN);;
        } else {
            /* light off */
            hal_gpio_bit_reset(LED2_GPIO_PORT, LED2_PIN);;
        }
        time_delay--;
    } else {
        time_delay = 1000;
    }
}
/* user code [global 1] end */	
