/*
    \file  main.c
*/
/*
    Copyright (c) 2023, GigaDevice Semiconductor Inc.

    All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this 
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice, 
       this list of conditions and the following disclaimer in the documentation 
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors 
       may be used to endorse or promote products derived from this software without 
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
*/
#include "gd32f3x0_hal.h"
#include "gd32f3x0_hal_init.h"
/* user code [global 0] begin */
#include "smartcard.h"

sc_APDU_commands sc_APDU;
sc_APDU_response sc_response;
sc_state_enum sc_state = SC_POWER_OFF;
__IO uint32_t card_inserted = 0;
const uint8_t master_root[2] = {0x3F, 0x00};

ErrStatus test_T0(void);
void smartcard_wait_ATR(void);
uint8_t smartcard_creat_MF(void);
void smartcard_error_handler(hal_smartcard_dev_struct *smartcard);

hal_smartcard_irq_struct smartcard0_irq;
/* user code [global 0] end */

/*!
    \brief      main function
    \param[in]  none
    \param[out] none
    \retval     none
*/
int main(void)
{
    /* user code [local 0] begin */
    uint32_t count = 0, state = 0, pre_state = 0;
    /* user code [local 0] end */

    msd_system_init();
    msd_clock_init();
    /* user code [local 1] begin */

    /* user code [local 1] end */
    msd_gpio_init();
    msd_dma_init();
    msd_usart0_init();

    /* user code [local 2] begin */
    /* wait for stable pin state */
    while(1000U != count){
        state = sc_detect();
        if(pre_state == state){
            ++count;
        }else{
            count = 0;
        }
        pre_state = state;
    }
    if(state){
        /* the card has already been inserted */
        card_inserted = 1;
        sc_power_cmd(ENABLE);
    }else{
        card_inserted = 0;
        sc_power_cmd(DISABLE);
    }
    while(0U == card_inserted);
    
    hal_smartcard_struct_init(HAL_SMARTCARD_IRQ_INIT_STRUCT, &smartcard0_irq);
    smartcard0_irq.error_handle = (hal_irq_handle_cb)smartcard_error_handler;
    hal_smartcard_irq_handle_set(&smartcard0_info, &smartcard0_irq);

    /* wait for smartcard ATR information */
    smartcard_wait_ATR();

    /* inserts delay(400ms) for smartard clock resynchronisation */
    hal_sys_basetick_delay_ms(400);
    
    /* test T0 PSAM card */
    test_T0();
    /* user code [local 2] end */

    while(1){
    /* user code [local 3] begin */

    /* user code [local 3] end */
    }
}
/* user code [global 1] begin */

/*!
    \brief      test of t0 card protocol interface
    \param[in]  none
    \param[out] none
    \retval     ErrStatus: ERROR or SUCCESS
*/
ErrStatus test_T0(void)
{
    uint8_t i = 0;

    /* select MF */
    sc_APDU.header.cla = 0x00;
    sc_APDU.header.ins = 0xA4;
    sc_APDU.header.p1 = 0x00;
    sc_APDU.header.p2 = 0x00;
    sc_APDU.body.lc = 0x02;
    sc_APDU.body.data[0] = master_root[0];
    sc_APDU.body.data[1] = master_root[1];
    sc_APDU.body.le = 0;

    sc_handler((sc_state_enum*)&sc_state, &sc_APDU, &sc_response);

    /* get response on MF */
    if(0x61 == sc_response.sw1){
        sc_APDU.header.cla = 0x00;
        sc_APDU.header.ins = SC_GET_RESPONSE;
        sc_APDU.header.p1 = 0x00;
        sc_APDU.header.p2 = 0x00;
        sc_APDU.body.lc = 0x00;
        sc_APDU.body.le = sc_response.sw2;

        sc_handler((sc_state_enum*)&sc_state, &sc_APDU, &sc_response);
    }

    /* create EF file of <GD> */
    if(((sc_response.sw1 << 8) | (sc_response.sw2)) == SC_OP_TERMINATED){
        /* create file */
        sc_APDU.header.cla = 0x80;
        sc_APDU.header.ins = 0xE0;
        sc_APDU.header.p1 = 0x02;
        sc_APDU.header.p2 = 0x00;
        sc_APDU.body.lc = 0x07;

        sc_APDU.body.data[0] = 'G';
        sc_APDU.body.data[1] = 'D';
        /* EF type: binary */
        sc_APDU.body.data[2] = 0x00;
        /* read access: 0~F */
        sc_APDU.body.data[3] = 0x0F;
        /* write access: 0~F */
        sc_APDU.body.data[4] = 0x0F;
        sc_APDU.body.data[5] = 0x00;
        sc_APDU.body.data[6] = 0xFF;

        sc_APDU.body.le = 0x00;

        sc_handler((sc_state_enum*)&sc_state, &sc_APDU, &sc_response);
    }

    /* select 'GD' EF file, if SW is 6A80, indicates the file already exists */
    if(((sc_response.sw1 << 8) | (sc_response.sw2)) == SC_OP_TERMINATED || 
        (0x6A80 == ((sc_response.sw1 << 8) |sc_response.sw2))){

        /* select EF */
        sc_APDU.header.cla = 0x00;
        sc_APDU.header.ins = 0xA4;
        sc_APDU.header.p1 = 0x02;
        sc_APDU.header.p2 = 0x00;
        sc_APDU.body.lc = 0x02;
        sc_APDU.body.data[0] = 'G';
        sc_APDU.body.data[1] = 'D';
        sc_APDU.body.le = 0;

        sc_handler((sc_state_enum*)&sc_state, &sc_APDU, &sc_response);
    }

    /* update binary */
    if(0x61 == sc_response.sw1){
        sc_APDU.header.cla = 00;
        sc_APDU.header.ins = 0xD6;
        sc_APDU.header.p1 = 0x00;
        sc_APDU.header.p2 = 0x00;
        sc_APDU.body.lc = 0x0B;
        sc_APDU.body.data[0] = 'g';
        sc_APDU.body.data[1] = 'i';
        sc_APDU.body.data[2] = 'g';
        sc_APDU.body.data[3] = 'a';
        sc_APDU.body.data[4] = ' ';
        sc_APDU.body.data[5] = 'd';
        sc_APDU.body.data[6] = 'e';
        sc_APDU.body.data[7] = 'v';
        sc_APDU.body.data[8] = 'i';
        sc_APDU.body.data[9] = 'c';
        sc_APDU.body.data[10] = 'e';
        sc_APDU.body.le = 0;

        sc_handler((sc_state_enum*)&sc_state, &sc_APDU, &sc_response);
    }

    /* read binary */
    if(((sc_response.sw1 << 8) | (sc_response.sw2)) == SC_OP_TERMINATED){
        sc_APDU.header.cla = 00;
        sc_APDU.header.ins = 0xB0;
        sc_APDU.header.p1 = 0x00;
        sc_APDU.header.p2 = 0x00;
        sc_APDU.body.lc = 0x00;
        sc_APDU.body.le = 0x0B;

        sc_handler((sc_state_enum*)&sc_state, &sc_APDU, &sc_response);
    }

    if(((sc_response.sw1 << 8) | (sc_response.sw2)) == SC_OP_TERMINATED){
        /* comparison data */
        for(i = 0; i < sc_APDU.body.le; i++){
            if(sc_response.data[i] != sc_APDU.body.data[i]){
                break;
            }
        }

        /* disable the smartcard interface */
        sc_state = SC_POWER_OFF;
        sc_handler(&sc_state, &sc_APDU, &sc_response);
        card_inserted = 0;
        
        if(i == sc_APDU.body.le){
            /* test success */
            hal_gpio_bit_set(LED3_GPIO_PORT, LED3_PIN);
            return SUCCESS;

        }else{
            /* test fail */
            hal_gpio_bit_set(LED4_GPIO_PORT, LED4_PIN);
            return ERROR;
        }
    }

    return ERROR;
}

/*!
    \brief      wait for smartcard ATR information
    \param[in]  none
    \param[out] none
    \retval     none
*/
void smartcard_wait_ATR(void)
{
    sc_state = SC_POWER_ON;
    sc_APDU.header.cla = 0x00;
    sc_APDU.header.ins = SC_GET_A2R;
    sc_APDU.header.p1 = 0x00;
    sc_APDU.header.p2 = 0x00;
    sc_APDU.body.lc = 0x00;

    while(sc_state != SC_ACTIVE_ON_T0){
        sc_handler((sc_state_enum*)&sc_state, &sc_APDU, &sc_response);
    }

    /* apply the procedure type selection (PTS) */
    sc_PTS_config();
}

/*!
    \brief      if the card is empty, user need to create MF file
    \param[in]  none
    \param[out] none
    \retval     operate state: 1(success), 2(exist), or 0(fail)
*/
uint8_t smartCard_creat_MF(void)
{
    /* create file */
    sc_APDU.header.cla = 0x80;
    sc_APDU.header.ins = 0xE0;
    sc_APDU.header.p1 = 0x00;
    sc_APDU.header.p2 = 0x00;
    sc_APDU.body.lc = 0x0F;

    sc_APDU.body.data[0] = 0xFF;
    sc_APDU.body.data[1] = 0xFF;
    sc_APDU.body.data[2] = 0xFF;
    sc_APDU.body.data[3] = 0xFF;
    sc_APDU.body.data[4] = 0xFF;
    sc_APDU.body.data[5] = 0xFF;
    sc_APDU.body.data[6] = 0xFF;
    sc_APDU.body.data[7] = 0xFF;
    /* create file access: 0~F */
    sc_APDU.body.data[8] = 0x0F;
    /* short file identifier */
    sc_APDU.body.data[9] = 0x00;
    sc_APDU.body.data[10] = 'M';
    sc_APDU.body.data[11] = 'C';
    sc_APDU.body.data[12] = 'U';
    sc_APDU.body.data[13] = 'G';
    sc_APDU.body.data[14] = 'D';
    
    sc_APDU.body.le = 0x00;

    sc_handler((sc_state_enum*)&sc_state, &sc_APDU, &sc_response);

    if(((sc_response.sw1 << 8) | (sc_response.sw2)) == SC_OP_TERMINATED){
        /* create end MF file */
        sc_APDU.header.cla = 0x80;
        sc_APDU.header.ins = 0xE0;
        sc_APDU.header.p1 = 0x00;
        sc_APDU.header.p2 = 0x01;
        sc_APDU.body.lc = 0x02;
        sc_APDU.body.data[0] = 0x3F;
        sc_APDU.body.data[1] = 0x00;
        sc_handler((sc_state_enum*)&sc_state, &sc_APDU, &sc_response);

        if(((sc_response.sw1 << 8) | (sc_response.sw2)) == SC_OP_TERMINATED){
            /* MF file create success */
            return 1;
        }
    }else if(((sc_response.sw1 << 8) | (sc_response.sw2)) == 0x6A80){
        /* MF file exist */
        return 2;
    }

    return 0;
}

/*!
    \brief      smartcard error handler
    \param[in]  smartcard: smartcard device information structure
                    the structure is not necessary to be reconfigured after structrue initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
void smartcard_error_handler(hal_smartcard_dev_struct *smartcard)
{
    if(SET == (smartcard->error_state & HAL_SMARTCARD_ERROR_FERR)){
        /* flush the smartcard receive data register */
        hals_smartcard_command_enable(smartcard->periph, USART_CMD_RXFCMD);

        /* resend the byte that failed to be received (by the smartcard) correctly */
        hal_smartcard_transmit_poll(smartcard, (uint8_t *)&smartcard->txbuffer.buffer[smartcard->txbuffer.length - 1], \
                                    1, SC_RECEIVE_TIMEOUT);
    }

    if(SET == (smartcard->error_state & HAL_SMARTCARD_ERROR_PERR)){
        /* enable smartcard RXNE interrupt (until receiving the corrupted byte) */
        hals_smartcard_interrupt_disable(smartcard->periph, USART_INT_RBNE);

        /* flush the smartcard receive data register */
        hals_smartcard_command_enable(smartcard->periph, USART_CMD_RXFCMD);
    }

    if(SET == (smartcard->error_state & HAL_SMARTCARD_ERROR_NERR)){
        /* flush the smartcard receive data register */
        hals_smartcard_command_enable(smartcard->periph, USART_CMD_RXFCMD);
    }

    if(SET == (smartcard->error_state & HAL_SMARTCARD_ERROR_ORERR)){
        /* flush the smartcard receive data register */
        hals_smartcard_command_enable(smartcard->periph, USART_CMD_RXFCMD);
    }
}
/* user code [global 1] end */	
