/*!
    \file  smartcard.h
*/

/*
    Copyright (c) 2019, GigaDevice Semiconductor Inc.

    All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this 
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice, 
       this list of conditions and the following disclaimer in the documentation 
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors 
       may be used to endorse or promote products derived from this software without 
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
*/

#ifndef SMARTCARD_H
#define SMARTCARD_H

#include "gd32f3x0_hal.h"

/* if the PSAM card is used and there is no card detection function, the macro should be set to 1 */
#define USE_PSAMCARD                       1U

/* configure transfer mode */
#define SC_TRANSFER_POLL                   1U
#define SC_TRANSFER_INT                    0U
#define SC_TRANSFER_DMA                    0U

/* USART communication interface */
#define SC_USART                           USART0

/* smartcard interface GPIO pins */
#define SC_3_5V_PIN                        GPIO_PIN_13
#define SC_3_5V_GPIO_PORT                  GPIOB
#define SC_3_5V_CLK_ENABLE()               hal_rcu_periph_clk_enable(RCU_GPIOB)

#define SC_RESET_PIN                       GPIO_PIN_11
#define SC_RESET_GPIO_PORT                 GPIOB
#define SC_RESET_CLK_ENABLE()              hal_rcu_periph_clk_enable(RCU_GPIOB)

#define SC_CMDVCC_PIN                      GPIO_PIN_15
#define SC_CMDVCC_GPIO_PORT                GPIOB
#define SC_CMDVCC_CLK_ENABLE()             hal_rcu_periph_clk_enable(RCU_GPIOB)

#define SC_DETECT_PIN                      GPIO_PIN_6
#define SC_DETECT_GPIO_PORT                GPIOB
#define SC_DETECT_CLK_ENABLE()             hal_rcu_periph_clk_enable(RCU_GPIOB)

#define SC_DETECT_EXTI                     EXTI_PIN_6
#define SC_DETECT_EXTI_IRQn                EXTI4_15_IRQn 
#define SC_DETECT_EXTI_IRQHandler          EXTI4_15_IRQHandler

/* exported constants */
#define T0_PROTOCOL                        0x00   /* T0 protocol */
#define DIRECT                             0x3B   /* direct bit convention */
#define INDIRECT                           0x3F   /* indirect bit convention */
#define SETUP_LENGTH                       20
#define HIST_LENGTH                        20
#define LC_MAX                             20
#define SC_RECEIVE_TIMEOUT                 65    /* direction to reader */
#define SC_TRANSMIT_TIMEOUT                65    /* direction to transmit */

/* sc APDU command: operation code */
/* command class */
#define SC_CLA_PSAM_INTER_NOTSAFE          0x00   /* internal no safe tranfer */
#define SC_CLA_PSAM_INTER_ADDSAFE          0x04   /* internal add safe tranfer */
#define SC_CLA_PSAM_EXTER_NOTSAFE          0x80   /* external add safe tranfer */
#define SC_CLA_PSAM_EXTER_ADDSAFE          0x84   /* external no safe tranfer */

/* data area management commands */
#define SC_WRITE_KEY                       0xD4
#define SC_READ_BINARY                     0xB0
#define SC_UPDATE_BINARY                   0xD6
#define SC_READ_RECORD                     0xB2
#define SC_APPEND_RECORD                   0xE2
#define SC_UPDATE_RECORD                   0xDC

/* administrative commands */ 
#define SC_CREATE_FILE                     0xE0
#define SC_SELECT_FILE                     0xA4

/* encryption algorithm commands */ 
#define SC_INT_FOR_DESCRYPT                0x1A   /* general DES calculation initialization */
#define SC_DES_CRYPT                       0xFA   /* general DES calculation */
#define SC_INT_SAM_FOR_PURCHASE            0x70   /* MAC1 calculation */
#define SC_CREDIT_SAM_FOR_PURCHASE         0x72   /* MAC2 verify */

/* safety management commands */
#define SC_UNBLOCK_PIN                     0x2C   /* unlock */
#define SC_APPLY_BLOCK                     0x1E   /* application lock */
#define SC_APPLY_UNBLOCK                   0x18   /* application unlock */
#define SC_CARD_BLOCK                      0x16   /* card lock */
#define SC_EXTERNAL_AUTH                   0x82   /* external authentication */
#define SC_GET_CHALLENGE                   0x84   /* generate random numbers */
#define SC_GET_RESPONSE                    0xC0   /* get response */
#define SC_INTERNAL_AUTH                   0x88   /* internal authentication */
#define SC_VERIFY                          0x20   /* PIN verify */
#define SC_CHANGE                          0x24   /* PIN unlock */
#define SC_CHANGE_PIN                      0x5E   /* PIN change */
#define SC_RELOAD_PIN                      0x5E   /* PIN reload */
#define SC_OUT_KEY                         0xF6   /* export KEY */

/* answer to reset commands */
#define SC_GET_A2R                         0x00

/* sc status: status code */
#define SC_EF_SELECTED                     0x9F
#define SC_DF_SELECTED                     0x9F
#define SC_OP_TERMINATED                   0x9000 /* correct execution of command */

/* file type */
#define SC_FILE_TYPE_BINARY                0x00
#define SC_FILE_TYPE_FIXED_REC             0x01
#define SC_FILE_TYPE_CHANGE_REC            0x02
#define SC_FILE_TYPE_ROUND_REC             0x03
#define SC_FILE_TYPE_ATR                   0x04
#define SC_FILE_TYPE_SECRET_KEY            0x05

/* secret key type */
#define SC_KEY_TYPE_KEY_MAC                0x00
#define SC_KEY_TYPE_PLAIN_MAC              0x40
#define SC_KEY_TYPE_CRYPTO                 0x80
#define SC_KEY_TYPE_CRYPTO_MAC             0xC0

/* smartcard voltage */
#define SC_VOLTAGE_5V                      0
#define SC_VOLTAGE_3V                      1

/* smartcard state type */
typedef enum {
    SC_POWER_ON = 0x00,
    SC_RESET_LOW = 0x01,
    SC_RESET_HIGH = 0x02,
    SC_ACTIVE = 0x03,
    SC_ACTIVE_ON_T0 = 0x04,
    SC_POWER_OFF = 0x05
} sc_state_enum;

/* ATR structure - answer To reset */
typedef struct {
  uint8_t ts;               /* bit convention */
  uint8_t t0;               /* high nibble = number of setup byte; low nibble = number of historical byte */
  uint8_t t[SETUP_LENGTH];  /* setup array */
  uint8_t h[HIST_LENGTH];   /* historical array */
  uint8_t tlength;          /* setup array dimension */
  uint8_t hlength;          /* historical array dimension */
} sc_ATR;

/* APDU-Header command structure */
typedef struct {
  uint8_t cla;    /* command class */
  uint8_t ins;    /* operation code */
  uint8_t p1;     /* selection mode */
  uint8_t p2;     /* selection option */
} sc_header;

/* APDU-Body command structure */
typedef struct {
  uint8_t lc;            /* data field length */
  uint8_t data[LC_MAX];  /* command parameters */
  uint8_t le;            /* expected length of data to be returned */
} sc_body;

/* APDU command structure */
typedef struct {
    sc_header header;
    sc_body body;
} sc_APDU_commands;

/* sc response structure */
typedef struct {
    uint8_t data[LC_MAX];  /* data returned from the card */
    uint8_t sw1;           /* command processing status */
    uint8_t sw2;           /* command processing qualification */
} sc_APDU_response;

/* handles all smartcard states and serves to send and receive all
communication data between smartcard and reader */
void sc_handler(sc_state_enum *scstate, sc_APDU_commands *sc_APDU, sc_APDU_response *sc_response);
/* enable or disable the power to the smartcard communication data between smartcard and reader */
void sc_power_cmd(EventStatus new_state);
/* set or clear the smartcard reset pin */
void sc_reset(FlagStatus reset_state);
/* configures the communication baudrate */
void sc_PTS_config(void);
/* configures GPIO used for samrtcard */
void sc_io_config(void);
/* detect whether the smartcard is present */
uint8_t sc_detect(void);

extern sc_ATR sc_A2R;
extern __IO uint32_t card_inserted;

#endif /* SMARTCARD_H */
