/*!
    \file    gd32h77x_trng.c
    \brief   TRNG driver

    \version 2025-12-10, V0.3.0, firmware for GD32H77x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h77x_trng.h"

/*!
    \brief      deinitialize the TRNG (API_ID(0x0001U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_deinit(void)
{
    rcu_periph_reset_enable(RCU_TRNGRST);
    rcu_periph_reset_disable(RCU_TRNGRST);
}

/*!
    \brief      enable the TRNG interface (API_ID(0x0002U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_enable(void)
{
    TRNG_CTL |= TRNG_CTL_TRNGEN;
}

/*!
    \brief      disable the TRNG interface (API_ID(0x0003U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_disable(void)
{
    TRNG_CTL &= ~TRNG_CTL_TRNGEN;
}

/*!
    \brief      lock the TRNG control bits (API_ID(0x0004U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_lock(void)
{
    TRNG_CTL |= TRNG_CTL_LK;
}

/*!
    \brief      configure TRNG working mode (API_ID(0x0005U))
    \param[in]  mode_select: the TRNG working mode
                only one parameter can be selected which is shown as below:
      \arg        TRNG_MODSEL_LFSR: TRNG working in LFSR mode
      \arg        TRNG_MODSEL_NIST: TRNG working in NIST mode
    \param[out] none
    \retval     none
*/
void trng_mode_config(trng_modsel_enum mode_select)
{
    uint32_t trng_config = TRNG_CTL;

    trng_config &= ~TRNG_CTL_MODSEL;
    trng_config |= (uint32_t)mode_select;
    TRNG_CTL = trng_config;
}

/*!
    \brief      enable the TRNG post-processing module (API_ID(0x0006U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_postprocessing_enable(void)
{
    TRNG_CTL |= TRNG_CTL_PPEN;
}

/*!
    \brief      disable the TRNG post-processing module (API_ID(0x0007U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_postprocessing_disable(void)
{
    TRNG_CTL &= ~TRNG_CTL_PPEN;
}

/*!
    \brief      enable the TRNG conditioning module (API_ID(0x0008U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_conditioning_enable(void)
{
    TRNG_CTL |= TRNG_CTL_CONDEN;
}

/*!
    \brief      disable the TRNG conditioning module (API_ID(0x0009U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_conditioning_disable(void)
{
    TRNG_CTL &= ~TRNG_CTL_CONDEN;
}

/*!
    \brief      configure TRNG conditioning module input bitwidth (API_ID(0x000AU))
    \param[in]  input_bitwidth: the input bit width
                only one parameter can be selected which is shown as below:
      \arg        TRNG_INMOD_256BIT: conditioning module input bitwidth 256bits
      \arg        TRNG_INMOD_440BIT: conditioning module input bitwidth 440bits
    \param[out] none
    \retval     none
*/
void trng_conditioning_input_bitwidth(trng_inmod_enum input_bitwidth)
{
    uint32_t trng_config = TRNG_CTL;

    trng_config &= ~TRNG_CTL_INMOD;
    trng_config |= (uint32_t)input_bitwidth;
    TRNG_CTL = trng_config;
}

/*!
    \brief      configure TRNG conditioning module output bitwidth (API_ID(0x000BU))
    \param[in]  output_bitwidth: the output bit width
                only one parameter can be selected which is shown as below:
      \arg        TRNG_OUTMOD_128BIT: conditioning module output bitwidth 128bits
      \arg        TRNG_OUTMOD_256BIT: conditioning module output bitwidth 256bits
    \param[out] none
    \retval     none
*/
void trng_conditioning_output_bitwidth(trng_outmod_enum output_bitwidth)
{
    uint32_t trng_config = TRNG_CTL;

    trng_config &= ~TRNG_CTL_OUTMOD;
    trng_config |= (uint32_t)output_bitwidth;
    TRNG_CTL = trng_config;
}

/*!
    \brief      enable TRNG replace test (API_ID(0x000CU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_replace_test_enable(void)
{
    TRNG_CTL |= TRNG_CTL_RTEN;
}

/*!
    \brief      disable TRNG replace test (API_ID(0x000DU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_replace_test_disable(void)
{
    TRNG_CTL &= ~TRNG_CTL_RTEN;
}

/*!
    \brief      enable hash algorithm init when conditioning module enable (API_ID(0x000EU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_hash_init_enable(void)
{
    TRNG_CTL |= TRNG_CTL_INIT;
}

/*!
    \brief      disable hash algorithm init when conditioning module enable (API_ID(0x000FU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_hash_init_disable(void)
{
    TRNG_CTL &= ~TRNG_CTL_INIT;
}

/*!
    \brief      configure TRNG analog power mode (API_ID(0x0010U))
    \param[in]  powermode: the power mode selection
                only one parameter can be selected which is shown as below:
      \arg        TRNG_NR_ULTRALOW: TRNG analog power mode ultralow
      \arg        TRNG_NR_LOW:      TRNG analog power mode low
      \arg        TRNG_NR_MEDIUM:   TRNG analog power mode medium
      \arg        TRNG_NR_HIGH:     TRNG analog power mode high
    \param[out] none
    \retval     none
*/
void trng_powermode_config(trng_pwrmod_enum powermode)
{
    uint32_t trng_config = TRNG_CTL;

    trng_config &= ~TRNG_CTL_NR;
    trng_config |= (uint32_t)powermode;
    TRNG_CTL = trng_config;
}

/*!
    \brief      configure TRNG clock divider (API_ID(0x0011U))
    \param[in]  clkdiv: TRNG clock divider
                only one parameter can be selected which is shown as below:
      \arg        TRNG_CLK_DIVx (x=1, 2, 4, 8, 16, 32, 64, 128, 256, 512, 1024, 2048, 4096, 8192, 16384, 32768): TRNG clock divided by x
    \param[out] none
    \retval     none
*/
void trng_clockdiv_config(trng_clkdiv_enum clkdiv)
{
    uint32_t trng_config = TRNG_CTL;

    trng_config &= ~TRNG_CTL_CLKDIV;
    trng_config |= (uint32_t)clkdiv;
    TRNG_CTL = trng_config;
}

/*!
    \brief      enable the TRNG clock error detection (API_ID(0x0012U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_clockerror_detection_enable(void)
{
    TRNG_CTL |= TRNG_CTL_CED;
}

/*!
    \brief      disable the TRNG clock error detection (API_ID(0x0013U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_clockerror_detection_disable(void)
{
    TRNG_CTL &= ~TRNG_CTL_CED;
}

/*!
    \brief      get the true random data (API_ID(0x0014U))
    \param[in]  none
    \param[out] none
    \retval     the generated random data
*/
uint32_t trng_get_true_random_data(void)
{
    return (TRNG_DATA);
}

/*!
    \brief      enable the conditioning logic reset (API_ID(0x0015U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_conditioning_reset_enable(void)
{
    TRNG_CTL |= TRNG_CTL_CONDRST;
}

/*!
    \brief      disable the conditioning logic reset (API_ID(0x0016U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_conditioning_reset_disable(void)
{
    TRNG_CTL &= ~TRNG_CTL_CONDRST;
}

/*!
    \brief      configure the conditioning module hash algorithm (API_ID(0x0017U))
    \param[in]  module_algo: module hash algorithm
                only one parameter can be selected which is shown as below:
      \arg        TRNG_ALGO_SHA1:   TRNG conditioning module hash SHA1
      \arg        TRNG_ALGO_MD5:    TRNG conditioning module hash MD5
      \arg        TRNG_ALGO_SHA224: TRNG conditioning module hash SHA224
      \arg        TRNG_ALGO_SHA256: TRNG conditioning module hash SHA256
    \param[out] none
    \retval     none
*/
void trng_conditioning_algo_config(trng_algo_enum module_algo)
{
    uint32_t tmp = TRNG_CTL;

    tmp &= ~(TRNG_CTL_ALGO);
    tmp |= (uint32_t)module_algo;
    TRNG_CTL = tmp;
}

/*!
    \brief      configure health tests default value (API_ID(0x0018U))
    \param[in]  apt_threshold: 0x00000000-0x000003FF, adaptive proportion test threshold value
    \param[in]  rct_threshold: 0x00-0x7F, repetitive (00/11) count test threshold value
    \param[out] none
    \retval     none
*/
void trng_health_tests_config(uint32_t apt_threshold, uint8_t rct_threshold)
{
    uint32_t tmp = TRNG_HTCFG;

    tmp &= ~(TRNG_HTCFG_APTTH | TRNG_HTCFG_RCTTH);
    tmp |= (((uint32_t)(apt_threshold << 16U) & TRNG_HTCFG_APTTH) | (rct_threshold & TRNG_HTCFG_RCTTH));
    TRNG_HTCFG = tmp;
}

/*!
    \brief      get the TRNG status flags (API_ID(0x0019U))
    \param[in]  flag: TRNG status flag, refer to trng_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        TRNG_FLAG_DRDY: random data ready status
      \arg        TRNG_FLAG_CECS: clock error current status
      \arg        TRNG_FLAG_SECS: seed error current status
      \arg        TRNG_FLAG_ERRSTA: NIST mode error flag
      \arg        TRNG_FLAG_GSSTA: condition stage is in generating random number sample state flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus trng_flag_get(trng_flag_enum flag)
{
    FlagStatus status = RESET;
    if(RESET != (TRNG_STAT & flag)) {
        status = SET;
    } else {
        status = RESET;
    }
    return status;
}

/*!
    \brief      enable the TRNG interrupt (API_ID(0x001AU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_interrupt_enable(void)
{
    TRNG_CTL |= TRNG_CTL_IE;
}

/*!
    \brief      disable the TRNG interrupt (API_ID(0x001BU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_interrupt_disable(void)
{
    TRNG_CTL &= ~TRNG_CTL_IE;
}

/*!
    \brief      enable the TRNG DRDY interrupt (API_ID(0x001CU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_drdy_interrupt_enable(void)
{
    TRNG_CTL &= ~TRNG_CTL_DRDYIMSK;
}

/*!
    \brief      disable the TRNG DRDY interrupt (API_ID(0x001DU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void trng_drdy_interrupt_disable(void)
{
    TRNG_CTL |= TRNG_CTL_DRDYIMSK;
}


/*!
    \brief      get the TRNG interrupt flags (API_ID(0x001EU))
    \param[in]  int_flag: TRNG interrupt flag, refer to trng_int_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        TRNG_INT_FLAG_CEIF: clock error interrupt flag
      \arg        TRNG_INT_FLAG_SEIF: seed error interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus trng_interrupt_flag_get(trng_int_flag_enum int_flag)
{
    FlagStatus status = RESET;
    if(RESET != (TRNG_STAT & int_flag)) {
        status = SET;
    } else {
        status = RESET;
    }
    return status;
}

/*!
    \brief      clear the TRNG interrupt flags (API_ID(0x001FU))
    \param[in]  int_flag: TRNG interrupt flag, refer to trng_int_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        TRNG_INT_FLAG_CEIF: clock error interrupt flag
      \arg        TRNG_INT_FLAG_SEIF: seed error interrupt flag
    \param[out] none
    \retval     none
*/
void trng_interrupt_flag_clear(trng_int_flag_enum int_flag)
{
    TRNG_STAT &= ~(uint32_t)int_flag;
}
