/*!
    \file    gd32e25x_syscfg.c
    \brief   SYSCFG driver

    \version 2025-07-25, V0.2.0, firmware for GD32E25x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32e25x_syscfg.h"

#define SYSCFG_I2C_FMPEN_MASK                    ((uint32_t)0x000B0000U)
#define SYSCFG_PIN_REMAP_MASK                    ((uint32_t)0x00000050U)
#define SYSCFG_DMA_REMAP_MASK                    ((uint32_t)0x00001F00U)
#define SYSCFG_CFG1_LOCK_MASK                    ((uint32_t)0x00000007U)
#define SYSCFG_STAT_ECC_MASK                     ((uint32_t)0x00000003U)

/*!
    \brief      reset the SYSCFG registers (API_ID(0x0001U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void syscfg_deinit(void)
{
    rcu_periph_reset_enable(RCU_CFGCMPRST);
    rcu_periph_reset_disable(RCU_CFGCMPRST);
}

/*!
    \brief      enable I2C Fm+ mode (API_ID(0x0002U))
    \param[in]  syscfg_gpio
                only one parameters can be selected which are shown as below:
      \arg        SYSCFG_PB6_FMPEN: PB6 pin I2C Fm+ mode
      \arg        SYSCFG_PB7_FMPEN: PB7 pin I2C Fm+ mode
      \arg        SYSCFG_PB9_FMPEN: PB9 pin I2C Fm+ mode
    \param[out] none
    \retval     none
*/
void syscfg_i2c_fast_mode_plus_enable(uint32_t syscfg_gpio)
{
    SYSCFG_CFG0 |= syscfg_gpio & SYSCFG_I2C_FMPEN_MASK;
}

/*!
    \brief      disable I2C Fm+ mode (API_ID(0x0003U))
    \param[in]  syscfg_gpio
                only one parameters can be selected which are shown as below:
      \arg        SYSCFG_PB6_FMPEN: PB6 pin I2C Fm+ mode
      \arg        SYSCFG_PB7_FMPEN: PB7 pin I2C Fm+ mode
      \arg        SYSCFG_PB9_FMPEN: PB9 pin I2C Fm+ mode
    \param[out] none
    \retval     none
*/
void syscfg_i2c_fast_mode_plus_disable(uint32_t syscfg_gpio)
{
    SYSCFG_CFG0 &= (uint32_t)(~(syscfg_gpio & SYSCFG_I2C_FMPEN_MASK));
}

/*!
    \brief      get the boot mode (API_ID(0x0004U))
    \param[in]  none
    \param[out] none
    \retval     the boot mode
      \arg      SYSCFG_BOOTMODE_FLASH: boot from the main flash
      \arg      SYSCFG_BOOTMODE_SYSTEM: boot from the system flash memory
      \arg      SYSCFG_BOOTMODE_SRAM: boot from the embedded SRAM
*/
uint8_t syscfg_bootmode_get(void)
{
    uint8_t mode;
    mode = (uint8_t)(SYSCFG_CFG0 & SYSCFG_CFG0_BOOT_MODE);
    if(0x01U == mode) {
        return SYSCFG_BOOTMODE_SYSTEM;
    } else if(0x02U == mode) {
        return SYSCFG_BOOTMODE_SRAM;
    } else {
        return SYSCFG_BOOTMODE_FLASH;
    }
}

/*!
    \brief      enable remap pin function (API_ID(0x0005U))
    \param[in]  remap_pin: specify the pin to remap
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_CFG0_BOOT0_PB9_RMP: remap BOOT0 PB9
      \arg        SYSCFG_CFG0_PA11_PA12_RMP: remap PA11 PA12
    \param[out] none
    \retval     none
*/
void syscfg_pin_remap_enable(uint32_t remap_pin)
{
    SYSCFG_CFG0 |= remap_pin & SYSCFG_PIN_REMAP_MASK;
}

/*!
    \brief      disable remap pin function (API_ID(0x0006U))
    \param[in]  remap_pin: specify the pin to remap
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_CFG0_BOOT0_PB9_RMP: remap BOOT0 PB9
      \arg        SYSCFG_CFG0_PA11_PA12_RMP: remap PA11 PA12
    \param[out] none
    \retval     none
*/
void syscfg_pin_remap_disable(uint32_t remap_pin)
{
    SYSCFG_CFG0 &= (~(remap_pin & SYSCFG_PIN_REMAP_MASK));
}

/*!
    \brief      enable the DMA channels remapping (API_ID(0x0007U))
    \param[in]  syscfg_dma_remap: specify the DMA channels
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_CFG0_TIMER16_DMA_RMP: remap TIMER16 channel0 and UP DMA requests to channel1(default channel0)
      \arg        SYSCFG_CFG0_TIMER15_DMA_RMP: remap TIMER15 channel2 and UP DMA requests to channel3(default channel2)
      \arg        SYSCFG_CFG0_USART0_RX_DMA_RMP: remap USART0 Rx DMA request to channel4(default channel2)
      \arg        SYSCFG_CFG0_USART0_TX_DMA_RMP: remap USART0 Tx DMA request to channel3(default channel1)
      \arg        SYSCFG_CFG0_ADC_DMA_RMP: remap ADC DMA requests from channel0 to channel1
    \param[out] none
    \retval     none
*/
void syscfg_dma_remap_enable(uint32_t syscfg_dma_remap)
{
    SYSCFG_CFG0 |= syscfg_dma_remap & SYSCFG_DMA_REMAP_MASK;
}

/*!
    \brief      disable the DMA channels remapping (API_ID(0x0008U))
    \param[in]  syscfg_dma_remap: specify the DMA channels
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_CFG0_TIMER16_DMA_RMP: remap TIMER16 channel0 and UP DMA requests to channel1(default channel0)
      \arg        SYSCFG_CFG0_TIMER15_DMA_RMP: remap TIMER15 channel2 and UP DMA requests to channel3(default channel2)
      \arg        SYSCFG_CFG0_USART0_RX_DMA_RMP: remap USART0 Rx DMA request to channel4(default channel2)
      \arg        SYSCFG_CFG0_USART0_TX_DMA_RMP: remap USART0 Tx DMA request to channel3(default channel1)
      \arg        SYSCFG_CFG0_ADC_DMA_RMP: remap ADC DMA requests from channel0 to channel1
    \param[out] none
    \retval     none
*/
void syscfg_dma_remap_disable(uint32_t syscfg_dma_remap)
{
    SYSCFG_CFG0 &= (~(syscfg_dma_remap & SYSCFG_DMA_REMAP_MASK));
}


/*!
    \brief      enable VREF function (API_ID(0x0009U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void syscfg_vref_enable(void)
{
    SYSCFG_CFG1 |= SYSCFG_CFG1_VREF_EN;
}

/*!
    \brief      disable VREF function (API_ID(0x000AU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void syscfg_vref_disable(void)
{
    SYSCFG_CFG1 &= ~SYSCFG_CFG1_VREF_EN;
}

/*!
    \brief      configure the GPIO pin reset mode (API_ID(0x000BU))
    \param[in]  syscfg_pin_reset_mode: specifies the GPIO pin reset mode.
                only one parameter can be selected which is shown as below:
      \arg        SYSCFG_PIN_NRST: GPIO pin configuration will retain state across any reset event except for the POR event
      \arg        SYSCFG_PIN_RST: GPIO pin configuration is reset when any reset event occurs
    \param[out] none
    \retval     none
*/
void syscfg_pin_reset_mode_config(uint32_t syscfg_pin_reset_mode)
{
    uint32_t reg;

    reg = SYSCFG_CFG1;
    /* reset the ENET_PHY_SEL bit and set according to syscfg_enet_phy_interface */
    reg &= ~SYSCFG_CFG1_PIN_RSTMD;
    SYSCFG_CFG1 = (reg | (syscfg_pin_reset_mode & SYSCFG_CFG1_PIN_RSTMD));
}

/*!
    \brief      configure the GPIO pin as EXTI Line (API_ID(0x000CU))
    \param[in]  exti_port: specify the GPIO port used in EXTI
                only one parameter can be selected which is shown as below:
      \arg        EXTI_SOURCE_GPIOx(x = A,B,F): EXTI GPIO port
    \param[in]  exti_pin: specify the EXTI line
                only one parameter can be selected which is shown as below:
      \arg        EXTI_SOURCE_PINx(GPIOA x = 0..15,GPIOB x = 0..11,GPIOF x = 0..3): EXTI GPIO pin
    \param[out] none
    \retval     none
*/
void syscfg_exti_line_config(uint8_t exti_port, uint8_t exti_pin)
{
    uint32_t clear_exti_mask = ~((uint32_t)EXTI_SS_MASK << (EXTI_SS_MSTEP(exti_pin)));
    uint32_t config_exti_mask = ((uint32_t)exti_port) << (EXTI_SS_MSTEP(exti_pin));

    switch(exti_pin / EXTI_SS_JSTEP) {
    case EXTISS0:
        /* clear EXTI source line(0..3) */
        SYSCFG_EXTISS0 &= clear_exti_mask;
        /* configure EXTI source line(0..3) */
        SYSCFG_EXTISS0 |= config_exti_mask;
        break;
    case EXTISS1:
        /* clear EXTI source line(4..7) */
        SYSCFG_EXTISS1 &= clear_exti_mask;
        /* configure EXTI source line(4..7) */
        SYSCFG_EXTISS1 |= config_exti_mask;
        break;
    case EXTISS2:
        /* clear EXTI source line(8..11) */
        SYSCFG_EXTISS2 &= clear_exti_mask;
        /* configure EXTI source line(8..11) */
        SYSCFG_EXTISS2 |= config_exti_mask;
        break;
    case EXTISS3:
        /* clear EXTI source line(12..15) */
        SYSCFG_EXTISS3 &= clear_exti_mask;
        /* configure EXTI source line(12..15) */
        SYSCFG_EXTISS3 |= config_exti_mask;
        break;
    default:
        break;
    }
}

/*!
    \brief      configure the PB3 as EXTI Line11 and Line15 (API_ID(0x000DU))
    \param[in]  line: EXTI line number, refer to syscfg_exti_line
                only one parameter can be selected which is shown as below:
      \arg        SYSCFG_EXTI_LINE_11 or SYSCFG_EXTI_LINE_15
    \param[out] none
    \retval     none
*/
void syscfg_exti_line11_line15_pb3(uint8_t line)
{
    uint32_t clear_exti_mask = ~((uint32_t)EXTI_SS_MASK << 12);
    uint32_t config_exti_mask = (uint32_t)(0x001 << 12);

    if(line == SYSCFG_EXTI_LINE_11) {
        SYSCFG_EXTISS2 &= clear_exti_mask;
        SYSCFG_EXTISS2 |= config_exti_mask;
    }
    if(line == SYSCFG_EXTI_LINE_15) {
        SYSCFG_EXTISS3 &= clear_exti_mask;
        SYSCFG_EXTISS3 |= config_exti_mask;
    }
}

/*!
    \brief      connect TIMER0/14/15/16 break input to the selected parameter (API_ID(0x000EU))
    \param[in]  syscfg_lock: Specify the parameter to be connected
                only one parameter can be selected which is shown as below:
      \arg        SYSCFG_LOCK_LOCKUP: Cortex-M23 lockup output connected to the break input
      \arg        SYSCFG_LOCK_SRAM_PARITY_ERROR: SRAM_PARITY check error connected to the break input
      \arg        SYSCFG_LOCK_LVD: LVD interrupt connected to the break input
    \param[out] none
    \retval     none
*/
void syscfg_lock_config(uint32_t syscfg_lock)
{
    SYSCFG_CFG1 |= syscfg_lock & SYSCFG_CFG1_LOCK_MASK;
}

/*!
    \brief      set the wait state counter value (API_ID(0x000FU))
    \param[in]  irq_latency: IRQ_LATENCY value (0x00 - 0xFF)
    \param[out] none
    \retval     none
*/
void irq_latency_set(uint8_t irq_latency)
{
    uint32_t reg;
    reg = SYSCFG_CPU_IRQ_LAT & (~(uint32_t)SYSCFG_CPU_IRQ_LAT_IRQ_LATENCY);
    reg |= (uint32_t)(IRQ_LATENCY(irq_latency));

    SYSCFG_CPU_IRQ_LAT = (uint32_t)reg;
}

/*!
    \brief      get SRAM error address in SYSCFG_CFG0 or SYSCFG_SRAM_ECC (API_ID(0x0010U))
    \param[in]  errtype: error type
                only one parameter can be selected which is shown as below:
      \arg        SYSCFG_ERR_PC: SRAM parity check error
      \arg        SYSCFG_ERR_ECC: SRAM ECC check error
    \param[out] none
    \retval     the error address returned.
*/
uint32_t syscfg_sram_error_address_get(syscfg_error_enum errtype)
{
    uint32_t addr = 0U;

    if(SYSCFG_ERR_PC == errtype) {
        addr = (SYSCFG_CFG0 & SYSCFG_CFG0_SRAM_ERRADDR) >> 21;
    } else if(SYSCFG_ERR_ECC == errtype) {
        addr = (SYSCFG_SRAM_ECC & SYSCFG_SRAM_ECC_ECCEADDR0) >> 21;
    } else {
        /* no operation */
    }
    return addr;
}

/*!
    \brief      get SRAM ECC single-bit correctable error bit in SYSCFG_SRAM_ECC (API_ID(0x0011U))
    \param[in]  none
    \param[out] none
    \retval     the error bit returned.
*/
uint32_t syscfg_sram_error_bit_get(void)
{
    uint32_t errbit;
    errbit = (SYSCFG_SRAM_ECC & SYSCFG_SRAM_ECC_ECCSERRBITS0) >> 10;
    return errbit;
}

/*!
    \brief      check if the specified flag in SYSCFG_CFG1 is set or not (API_ID(0x0012U))
    \param[in]  syscfg_flag: specify the flag in SYSCFG_CFG1 to check.
      \arg        SYSCFG_SRAM_PCEF: SRAM parity check error flag.
    \param[out] none
    \retval     the syscfg_flag state returned (SET or RESET).
  */
FlagStatus syscfg_flag_get(uint32_t syscfg_flag)
{
    FlagStatus flag = RESET;

    if((SYSCFG_CFG1 & syscfg_flag) != (uint32_t)RESET) {
        flag = SET;
    } else {
        flag = RESET;
    }

    return flag;
}

/*!
    \brief      clear the flag in SYSCFG_CFG1 by writing 1 (API_ID(0x0013U))
    \param[in]  syscfg_flag: Specify the flag in SYSCFG_CFG1 to clear.
      \arg        SYSCFG_SRAM_PCEF: SRAM parity check error flag.
    \param[out] none
    \retval     none
*/
void syscfg_flag_clear(uint32_t syscfg_flag)
{
    SYSCFG_CFG1 |= (uint32_t) syscfg_flag & SYSCFG_CFG1_SRAM_PCEF;
}

/*!
    \brief      enable SRAM ECC interrupt (API_ID(0x0014U))
    \param[in]  inttype: interrupt type
                only one parameter can be selected which is shown as below:
      \arg        SYSCFG_INT_ECCSE: SRAM single bit correction interrupt
      \arg        SYSCFG_INT_ECCME: SRAM multi bits non-correction interrupt
    \param[out] none
    \retval     none
*/
void syscfg_interrupt_enable(syscfg_interrupt_enum inttype)
{
    SYSCFG_SRAM_ECC |= inttype;
}

/*!
    \brief      disable SRAM ECC interrupt (API_ID(0x0015U))
    \param[in]  inttype: interrupt type
                only one parameter can be selected which is shown as below:
      \arg        SYSCFG_INT_ECCSE: SRAM single bit correction interrupt
      \arg        SYSCFG_INT_ECCME: SRAM multi bits non-correction interrupt
    \param[out] none
    \retval     none
*/
void syscfg_interrupt_disable(syscfg_interrupt_enum inttype)
{
    SYSCFG_SRAM_ECC &= ~(uint32_t)inttype;
}

/*!
    \brief      check if the specified interrupt flag in SYSCFG_STAT is set or not (API_ID(0x0016U))
    \param[in]  int_flag: specify the interrupt flag in SYSCFG_STAT to check.
      \arg        SYSCFG_SRAM_ECCMEIF: SRAM multi bits non-correction event flag.
                  SYSCFG_SRAM_ECCSEIF: SRAM single bit correction event flag.
    \param[out] none
    \retval     the int_flag state returned (SET or RESET).
  */
FlagStatus syscfg_interrupt_flag_get(uint32_t int_flag)
{
    FlagStatus flag = RESET;

    if((SYSCFG_STAT & int_flag) != (uint32_t)RESET) {
        flag = SET;
    } else {
        flag = RESET;
    }

    return flag;
}

/*!
    \brief      clear the interrupt flag in SYSCFG_STAT by writing 1 (API_ID(0x0017U))
    \param[in]  int_flag: specify the interrupt flag in SYSCFG_STAT to clear.
      \arg        SYSCFG_SRAM_ECCMEIF: SRAM multi bits non-correction event flag.
                  SYSCFG_SRAM_ECCSEIF: SRAM single bit correction event flag.
    \param[out] none
    \retval     none
*/
void syscfg_interrupt_flag_clear(uint32_t int_flag)
{
    SYSCFG_STAT |= (uint32_t) int_flag & SYSCFG_STAT_ECC_MASK;
}
