/*!
    \file    gd32e25x_gpio.c
    \brief   GPIO driver

    \version 2025-07-25, V0.2.0, firmware for GD32E25x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32e25x_gpio.h"

#define GPIO_MODE_PMASK           ((uint32_t)0x00000003U)
#define GPIO_PULL_MODE_PMASK      ((uint32_t)0x00000003U)
#define GPIO_OSPEED_PMASK         ((uint32_t)0x00000003U)
#define GPIO_AF_PMASK             ((uint32_t)0x0000000FU)
#if defined(GD32E251)
#define GPIO_DFEN_PMASK           ((uint32_t)0x00000007U)
#define GPIO_INT0_SEL_PMASK       ((uint32_t)0x00000007U)
#define GPIO_INT1_SEL_PMASK       ((uint32_t)0x000000F0U)
#endif /* GD32E251 */

/*!
    \brief      reset GPIO port (API_ID(0x0001U))
    \param[in]  gpio_periph: GPIOx(x = A,B,F,SP)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,F,SP)
    \param[out] none
    \retval     none
*/
void gpio_deinit(uint32_t gpio_periph)
{

    switch(gpio_periph) {
    case GPIOA:
        /* reset GPIOA */
        rcu_periph_reset_enable(RCU_GPIOARST);
        rcu_periph_reset_disable(RCU_GPIOARST);
        break;
    case GPIOB:
        /* reset GPIOB */
        rcu_periph_reset_enable(RCU_GPIOBRST);
        rcu_periph_reset_disable(RCU_GPIOBRST);
        break;
    case GPIOF:
        /* reset GPIOF */
        rcu_periph_reset_enable(RCU_GPIOFRST);
        rcu_periph_reset_disable(RCU_GPIOFRST);
        break;
#if defined(GD32E251)
    case GPIOSP:
        /* reset GPIOSP */
        rcu_periph_reset_enable(RCU_NEW_FEATRST);
        rcu_periph_reset_disable(RCU_NEW_FEATRST);
        break;
#endif /* GD32E251 */
    default:
        break;
    }
}

/*!
    \brief      set GPIO mode (API_ID(0x0002U))
    \param[in]  gpio_periph: GPIOx(x = A,B,F)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,F)
    \param[in]  mode: gpio pin mode
                only one parameter can be selected which is shown as below:
      \arg        GPIO_MODE_INPUT: input mode
      \arg        GPIO_MODE_OUTPUT: output mode
      \arg        GPIO_MODE_AF: alternate function mode
      \arg        GPIO_MODE_ANALOG: analog mode
    \param[in]  pull_up_down: gpio pin with pull-up or pull-down resistor
                only one parameter can be selected which is shown as below:
      \arg        GPIO_PUPD_NONE: floating mode, no pull-up and pull-down resistors
      \arg        GPIO_PUPD_PULLUP: with pull-up resistor
      \arg        GPIO_PUPD_PULLDOWN:with pull-down resistor
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[out] none
    \retval     none
*/
void gpio_mode_set(uint32_t gpio_periph, uint32_t mode, uint32_t pull_up_down, uint32_t pin)
{
    uint16_t i;
    uint32_t ctl, pupd;

    ctl = GPIO_CTL(gpio_periph);
    pupd = GPIO_PUD(gpio_periph);

    for(i = 0U; i < 16U; i++) {
        if((1U << i) & pin) {
            /* clear the specified pin mode bits */
            ctl &= ~GPIO_MODE_MASK(i);
            /* set the specified pin mode bits */
            ctl |= GPIO_MODE_SET(i, (mode & GPIO_MODE_PMASK));

            /* clear the specified pin pupd bits */
            pupd &= ~GPIO_PUPD_MASK(i);
            /* set the specified pin pupd bits */
            pupd |= GPIO_PUPD_SET(i, (pull_up_down & GPIO_PULL_MODE_PMASK));
        }
    }

    GPIO_CTL(gpio_periph) = ctl;
    GPIO_PUD(gpio_periph) = pupd;
}

/*!
    \brief      set GPIO output type and speed (API_ID(0x0003U))
    \param[in]  gpio_periph: GPIOx(x = A,B,F)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,F)
    \param[in]  otype: gpio pin output mode
                only one parameter can be selected which is shown as below:
      \arg        GPIO_OTYPE_PP: push pull mode
      \arg        GPIO_OTYPE_OD: open drain mode
    \param[in]  speed: gpio pin output max speed
                only one parameter can be selected which is shown as below:
      \arg        GPIO_OSPEED_2MHZ: output max speed 2MHz
      \arg        GPIO_OSPEED_10MHZ: output max speed 10MHz
      \arg        GPIO_OSPEED_50MHZ: output max speed 50MHz
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[out] none
    \retval     none
*/
void gpio_output_options_set(uint32_t gpio_periph, uint8_t otype, uint32_t speed, uint32_t pin)
{
    uint16_t i;
    uint32_t ospeed;

    if(GPIO_OTYPE_OD == otype) {
        GPIO_OMODE(gpio_periph) |= (uint32_t)(pin & GPIO_PIN_ALL);
    } else {
        GPIO_OMODE(gpio_periph) &= (uint32_t)(~(pin & GPIO_PIN_ALL));
    }

    /* get the specified pin output speed bits value */
    ospeed = GPIO_OSPD(gpio_periph);

    for(i = 0U; i < 16U; i++) {
        if((1U << i) & pin) {
            /* clear the specified pin output speed bits */
            ospeed &= ~GPIO_OSPEED_MASK(i);
            /* set the specified pin output speed bits */
            ospeed |= GPIO_OSPEED_SET(i, (speed & GPIO_OSPEED_PMASK));
        }
    }
    GPIO_OSPD(gpio_periph) = ospeed;
}

/*!
    \brief      set GPIO pin bit (API_ID(0x0004U))
    \param[in]  gpio_periph: GPIOx(x = A,B,F)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,F)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[out] none
    \retval     none
*/
void gpio_bit_set(uint32_t gpio_periph, uint32_t pin)
{
    GPIO_BOP(gpio_periph) = (uint32_t)(pin & GPIO_PIN_ALL);
}

/*!
    \brief      reset GPIO pin bit (API_ID(0x0005U))
    \param[in]  gpio_periph: GPIOx(x = A,B,F)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,F)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[out] none
    \retval     none
*/
void gpio_bit_reset(uint32_t gpio_periph, uint32_t pin)
{
    GPIO_BC(gpio_periph) = (uint32_t)(pin & GPIO_PIN_ALL);
}

/*!
    \brief      write data to the specified GPIO pin (API_ID(0x0006U))
    \param[in]  gpio_periph: GPIOx(x = A,B,F)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,F)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[in]  bit_value: SET or RESET
                only one parameter can be selected which is shown as below:
      \arg        RESET: clear the port pin
      \arg        SET: set the port pin
    \param[out] none
    \retval     none
*/
void gpio_bit_write(uint32_t gpio_periph, uint32_t pin, bit_status bit_value)
{
    if(RESET != bit_value) {
        GPIO_BOP(gpio_periph) = (uint32_t)(pin & GPIO_PIN_ALL);
    } else {
        GPIO_BC(gpio_periph) = (uint32_t)(pin & GPIO_PIN_ALL);
    }
}

/*!
    \brief      write data to the specified GPIO port (API_ID(0x0007U))
    \param[in]  gpio_periph: GPIOx(x = A,B,F)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,F)
    \param[in]  data: specify the value to be written to the port output control register
    \param[out] none
    \retval     none
*/
void gpio_port_write(uint32_t gpio_periph, uint16_t data)
{
    GPIO_OCTL(gpio_periph) = (uint32_t)data;
}

/*!
    \brief      get GPIO pin input status (API_ID(0x0008U))
    \param[in]  gpio_periph: GPIOx(x = A,B,F)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,F)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[out] none
    \retval     SET or RESET
*/
FlagStatus gpio_input_bit_get(uint32_t gpio_periph, uint32_t pin)
{
    FlagStatus reval = RESET;

    if((uint32_t)RESET != (GPIO_ISTAT(gpio_periph) & (pin))) {
        reval = SET;
    } else {
        reval = RESET;
    }

    return reval;
}

/*!
    \brief      get GPIO all pins input status (API_ID(0x0009U))
    \param[in]  gpio_periph: GPIOx(x = A,B,F)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,F)
    \param[out] none
    \retval     state of GPIO all pins
*/
uint16_t gpio_input_port_get(uint32_t gpio_periph)
{
    uint16_t reval = 0U;

    reval = ((uint16_t)GPIO_ISTAT(gpio_periph));

    return reval;
}

/*!
    \brief      get GPIO pin output status (API_ID(0x000AU))
    \param[in]  gpio_periph: GPIOx(x = A,B,F)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,F)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[out] none
    \retval     SET or RESET
*/
FlagStatus gpio_output_bit_get(uint32_t gpio_periph, uint32_t pin)
{
    FlagStatus reval = RESET;

    if((uint32_t)RESET != (GPIO_OCTL(gpio_periph) & (pin))) {
        reval = SET;
    } else {
        reval = RESET;
    }

    return reval;
}

/*!
    \brief      get GPIO all pins output status (API_ID(0x000BU))
    \param[in]  gpio_periph: GPIOx(x = A,B,F)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,F)
    \param[out] none
    \retval     state of GPIO all pins
*/
uint16_t gpio_output_port_get(uint32_t gpio_periph)
{
    uint16_t reval = 0U;

    reval = ((uint16_t)GPIO_OCTL(gpio_periph));

    return reval;
}

/*!
    \brief      set GPIO alternate function (API_ID(0x000CU))
    \param[in]  gpio_periph: GPIOx(x = A,B,F)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,F)
    \param[in]  alt_func_num: GPIO pin af function, please refer to specific device datasheet
                only one parameter can be selected which is shown as below:
      \arg        GPIO_AF_0: EVENTOUT, SPI0, CK_OUT, TIMER14, I2S0, TIMER16, SWDIO, SWCLk, TIMER13,
                             USART0,IFRP_OUT
      \arg        GPIO_AF_1: USART0, UART1, TIMER2, IFRP_OUT, EVENTOUT, I2C0, I2C1
      \arg        GPIO_AF_2: TIMER1, TIMER0, EVENTOUT, TIMER15, TIMER16, TIMER2,CMP0
      \arg        GPIO_AF_3: CLA0OUT, CLA1OUT, CLA2OUT, CLA3OUT, I2C0, EVENTOUT,TIMER14, UART1, CK_OUT
      \arg        GPIO_AF_4(port A,B only): I2C1, TIMER0, TIMER13, CMP0, UART1, I2C0, I2C2, CLA2OUT
      \arg        GPIO_AF_5(port A,B only): I2C0, I2C2, USART0, CMP0, I2C1,TIMER0,
                                            TIMER14,TIMER2,TIMER15,TIMER16,CK_OUT,I2S0
      \arg        GPIO_AF_6(port A,B only): SPI1, EVENTOUT, TIMER1, TIMER2, TIMER0, TIMER16
      \arg        GPIO_AF_7(port A,B only): CMP0, I2C1, TIMER13, TIMER0, TIMER1, SPI1, CLA3OUT, CLA2OUT
      \arg        GPIO_AF_8(port A,B only): TIMER14, UART1, TIMER0
      \arg        GPIO_AF_9(port A,B only): USART0
      \arg        GPIO_AF_10(port A,B only): SPI1
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[out] none
    \retval     none
*/
void gpio_af_set(uint32_t gpio_periph, uint32_t alt_func_num, uint32_t pin)
{
    uint16_t i;
    uint32_t afrl, afrh;

    afrl = GPIO_AFSEL0(gpio_periph);
    afrh = GPIO_AFSEL1(gpio_periph);

    for(i = 0U; i < 8U; i++) {
        if((1U << i) & pin) {
            /* clear the specified pin alternate function bits */
            afrl &= ~GPIO_AFR_MASK(i);
            afrl |= GPIO_AFR_SET(i, (alt_func_num & GPIO_AF_PMASK));
        }
    }

    for(i = 8U; i < 16U; i++) {
        if((1U << i) & pin) {
            /* clear the specified pin alternate function bits */
            afrh &= ~GPIO_AFR_MASK(i - 8U);
            afrh |= GPIO_AFR_SET(i - 8U, (alt_func_num & GPIO_AF_PMASK));
        }
    }

    GPIO_AFSEL0(gpio_periph) = afrl;
    GPIO_AFSEL1(gpio_periph) = afrh;
}

/*!
    \brief      lock GPIO pin bit (API_ID(0x000DU))
    \param[in]  gpio_periph: GPIOx(x = A,B,F)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,F)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[out] none
    \retval     none
*/
void gpio_pin_lock(uint32_t gpio_periph, uint32_t pin)
{
    uint32_t lock = 0x00010000U;
    lock |= (pin & GPIO_PIN_ALL);

    /* lock key writing sequence: write 1->write 0->write 1->read 0->read 1 */
    GPIO_LOCK(gpio_periph) = (uint32_t)lock;
    GPIO_LOCK(gpio_periph) = (uint32_t)(pin & GPIO_PIN_ALL);
    GPIO_LOCK(gpio_periph) = (uint32_t)lock;
    lock = GPIO_LOCK(gpio_periph);
    lock = GPIO_LOCK(gpio_periph);
}

/*!
    \brief      toggle GPIO pin status (API_ID(0x000EU))
    \param[in]  gpio_periph: GPIOx(x = A,B,F)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,F)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[out] none
    \retval     none
*/
void gpio_bit_toggle(uint32_t gpio_periph, uint32_t pin)
{
    GPIO_TG(gpio_periph) = (uint32_t)(pin & GPIO_PIN_ALL);
}

/*!
    \brief      toggle GPIO port status (API_ID(0x000FU))
    \param[in]  gpio_periph: GPIOx(x = A,B,F)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,F)
    \param[out] none
    \retval     none
*/
void gpio_port_toggle(uint32_t gpio_periph)
{
    GPIO_TG(gpio_periph) = 0x0000FFFFU;
}
#if defined(GD32E251)
/*!
    \brief      enable the priority crossbar decoder (API_ID(0x0010U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void gpio_crossbar_enable(void)
{
    GPIO_CBREN &= ~GPIO_CBREN_CBAREN;
    GPIO_CBREN |= GPIO_CBREN_CBAREN;
}

/*!
    \brief      disable the priority crossbar decoder (API_ID(0x0011U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void gpio_crossbar_disable(void)
{
    GPIO_CBREN &= ~GPIO_CBREN_CBAREN;
}

/*!
    \brief      enable the periph routed to port pins (API_ID(0x0012U))
    \param[in]  gpio_periph_routed: please refer to specific device datasheet
                only one parameter can be selected which are shown as below:
      \arg        CROSSBAR_USART0: USART0 TX1, RX1 routed to Port pins PB5 and PB4.
      \arg        CROSSBAR_SPI0: SPI0 I/O routed to Port pins. The SPI can be assigned either 3 or 4 GPIO pins.
      \arg        CROSSBAR_SMB0: SMBus0 1 I/O routed to Port pins.
      \arg        CROSSBAR_CP0E: CP0 routed to Port pin.
      \arg        CROSSBAR_CP0AE: Asynchronous CP0 routed to Port pin.
      \arg        CROSSBAR_SYSCKE: SYSCLK output routed to Port pin.
      \arg        CROSSBAR_TIMER0_NONE: TIMER0 I/O unavailable at Port pins.
      \arg        CROSSBAR_TIMER0_CH0: TIMER0 CH0 routed to Port pin.
      \arg        CROSSBAR_TIMER0_CH0_2_CH1: TIMER0 CH0, CH1 routed to Port pins.
      \arg        CROSSBAR_TIMER0_CH0_2_CH2: TIMER0 CH0, CH1, CH2 routed to Port pins.
      \arg        CROSSBAR_TIMER0_CH0_2_CH3: TIMER0 CH0, CH1, CH2, CH3 routed to Port pins.
      \arg        CROSSBAR_TIMER0_CH0_2_CH4: TIMER0 CH0, CH1, CH2, CH3, CH4 routed to Port pins.
      \arg        CROSSBAR_TIMER0_CH0_2_CH5: TIMER0 CH0, CH1, CH2, CH3, CH4, CH5 routed to Port pins.
      \arg        CROSSBAR_TIMER0_ETR_EN: TIMER0_ETR routed to Port pin.
      \arg        CROSSBAR_TIMER1_ETR_EN: TIMER1_ETR routed to Port pin.
      \arg        CROSSBAR_TIMER2_ETR_EN: TIMER2_ETR routed to Port pin.
      \arg        CROSSBAR_TIMER13_EN: TIMER13 CH0 routed to Port pin.
      \arg        CROSSBAR_URT1E: UART1 TX RX routed to Port pins.
      \arg        CROSSBAR_URT1RTSE: UART1 RTS routed to Port pin.
      \arg        CROSSBAR_URT1CTSE: UART1 CTS routed to Port pin.
    \param[out] none
    \retval     none
*/
void gpio_crossbar_periph_routed_enable(gpio_crossbar_periph_enable_enum gpio_periph_routed)
{
    uint32_t mask = 0U;

    if(((uint32_t)gpio_periph_routed & (0x700U)) != 0x0U) {
        mask |= CROSSBAR_TIMER0ME_MASK;
    } else {
        mask |= (uint32_t)gpio_periph_routed;
    }
    GPIO_CBREN &= ~mask;
    GPIO_CBREN |= (uint32_t)gpio_periph_routed;
}

/*!
    \brief      disable the periph routed to port pins (API_ID(0x0013U))
    \param[in]  gpio_periph_routed: please refer to specific device datasheet
                only one parameter can be selected which are shown as below:
      \arg        CROSSBAR_USART0: USART0 TX1, RX1 routed to Port pins PB5 and PB4.
      \arg        CROSSBAR_SPI0: SPI0 I/O routed to Port pins. The SPI can be assigned either 3 or 4 GPIO pins.
      \arg        CROSSBAR_SMB0: SMBus0 1 I/O routed to Port pins.
      \arg        CROSSBAR_CP0E: CP0 routed to Port pin.
      \arg        CROSSBAR_CP0AE: Asynchronous CP0 routed to Port pin.
      \arg        CROSSBAR_SYSCKE: SYSCLK output routed to Port pin.
      \arg        CROSSBAR_TIMER0_NONE: TIMER0 I/O unavailable at Port pins.
      \arg        CROSSBAR_TIMER0_CH0: TIMER0 CH0 routed to Port pin.
      \arg        CROSSBAR_TIMER0_CH0_2_CH1: TIMER0 CH0, CH1 routed to Port pins.
      \arg        CROSSBAR_TIMER0_CH0_2_CH2: TIMER0 CH0, CH1, CH2 routed to Port pins.
      \arg        CROSSBAR_TIMER0_CH0_2_CH3: TIMER0 CH0, CH1, CH2, CH3 routed to Port pins.
      \arg        CROSSBAR_TIMER0_CH0_2_CH4: TIMER0 CH0, CH1, CH2, CH3, CH4 routed to Port pins.
      \arg        CROSSBAR_TIMER0_CH0_2_CH5: TIMER0 CH0, CH1, CH2, CH3, CH4, CH5 routed to Port pins.
      \arg        CROSSBAR_TIMER0_ETR_EN: TIMER0_ETR routed to Port pin.
      \arg        CROSSBAR_TIMER1_ETR_EN: TIMER1_ETR routed to Port pin.
      \arg        CROSSBAR_TIMER2_ETR_EN: TIMER2_ETR routed to Port pin.
      \arg        CROSSBAR_TIMER13_EN: TIMER13 CH0 routed to Port pin.
      \arg        CROSSBAR_URT1E: UART1 TX RX routed to Port pins.
      \arg        CROSSBAR_URT1RTSE: UART1 RTS routed to Port pin.
      \arg        CROSSBAR_URT1CTSE: UART1 CTS routed to Port pin.
    \param[out] none
    \retval     none
*/
void gpio_crossbar_periph_routed_disable(gpio_crossbar_periph_enable_enum gpio_periph_routed)
{
    uint32_t mask = 0U;

    if(((uint32_t)gpio_periph_routed & (0x700U)) != 0x0U) {
        mask |= CROSSBAR_TIMER0ME_MASK;
    } else {
        mask |= (uint32_t)gpio_periph_routed;
    }
    GPIO_CBREN &= ~mask;
}

/*!
    \brief      enable the digital function that not assign by crossbar (API_ID(0x0014U))
    \param[in]  periph_digital:please refer to specific device datasheet
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_DFEN_I2C0PINEN: Enable I2C0 to the selected GPIO pin
      \arg        GPIO_DFEN_CLA0PINEN: Enable CLA0 asynchronous output to the selected GPIO pin
      \arg        GPIO_DFEN_CLA1PINEN: Enable CLA1 asynchronous output to the selected GPIO pin
      \arg        GPIO_DFEN_CLA2PINEN: Enable CLA2 asynchronous output to the selected GPIO pin
      \arg        GPIO_DFEN_CLA3PINEN: Enable CLA3 asynchronous output to the selected GPIO pin
    \param[out] none
    \retval     none
*/
void gpio_digital_function_enable(uint32_t periph_digital)
{
    GPIO_DFEN &= ~(periph_digital & GPIO_DFEN_PMASK);
    GPIO_DFEN |= (periph_digital & GPIO_DFEN_PMASK);
}

/*!
    \brief      disable the digital function that not assign by crossbar (API_ID(0x0015U))
    \param[in]  periph_digital:please refer to specific device datasheet
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_DFEN_I2C0PINEN: Disable I2C0 to the selected GPIO pin
      \arg        GPIO_DFEN_CLA0PINEN: Disable CLA0 asynchronous output to the selected GPIO pin
      \arg        GPIO_DFEN_CLA1PINEN: Disable CLA1 asynchronous output to the selected GPIO pin
      \arg        GPIO_DFEN_CLA2PINEN: Disable CLA2 asynchronous output to the selected GPIO pin
      \arg        GPIO_DFEN_CLA3PINEN: Disable CLA3 asynchronous output to the selected GPIO pin
    \param[out] none
    \retval     none
*/
void gpio_digital_function_disable(uint32_t periph_digital)
{
    GPIO_DFEN &= ~(periph_digital & GPIO_DFEN_PMASK);
}

/*!
    \brief      set the port pin is skipped by the crossbar (API_ID(0x0016U))
    \param[in]  gpio_skip: skip the pin that assign by crossbar, please refer to crossbar decoder datasheet
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_SKIP_SKIPx(x = 0 - 21):  please refer to user manual
    \param[out] none
    \retval     none
*/
void gpio_skip_pin(uint32_t gpio_skip)
{
    GPIO_SKIP &= ~(gpio_skip & GPIO_SKIP_ALL);
    GPIO_SKIP |= (gpio_skip & GPIO_SKIP_ALL);
}

/*!
    \brief      set the port pin is not skipped by the crossbar (API_ID(0x0017U))
    \param[in]  gpio_nskip: nskip the pin that assign by crossbar, please refer to crossbar decoder datasheet
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_SKIP_SKIPx(x = 0 - 18): please refer to crossbar decoder datasheet.
    \param[out] none
    \retval     none
*/
void gpio_nskip_pin(uint32_t gpio_nskip)
{
    GPIO_SKIP &= ~(gpio_nskip & GPIO_SKIP_ALL);
}

/*!
    \brief      config int0 (API_ID(0x0018U))
    \param[in]  polarity: config int0 polarity
                only one parameter can be selected which are shown as below:
      \arg        INT_POLARITY_LOW: INT0 input is active low.
      \arg        INT_POLARITY_HIGH:INT0 input is active high.
    \param[in]  int_sel: INT0 Port Pin Selection
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_INT0_SEL_PB9: Select PB9
      \arg        GPIO_INT0_SEL_PB8: Select PB8
      \arg        GPIO_INT0_SEL_PF0: Select PF0
      \arg        GPIO_INT0_SEL_PB5: Select PB5
      \arg        GPIO_INT0_SEL_PB4: Select PB4
      \arg        GPIO_INT0_SEL_PB3: Select PB3
      \arg        GPIO_INT0_SEL_PA15: Select PA15
    \param[out] none
    \retval     none
*/
void gpio_int0_config(uint8_t polarity, uint32_t int_sel)
{
    if(INT_POLARITY_LOW == polarity) {
        GPIO_INTCTL &= ~GPIO_INTCTL_INT0PL;
    } else {
        GPIO_INTCTL |= GPIO_INTCTL_INT0PL;
    }
    GPIO_INTCTL &= ~GPIO_INTCTL_INT0SL;
    GPIO_INTCTL |= (int_sel & GPIO_INT0_SEL_PMASK);
}

/*!
    \brief      config int1 (API_ID(0x0019U))
    \param[in]  polarity: config int1 polarity
                only one parameter can be selected which are shown as below:
      \arg        INT_POLARITY_LOW: INT1 input is active low.
      \arg        INT_POLARITY_HIGH:INT1 input is active high.
    \param[in]  int_sel: INT1 Port Pin Selection
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_INT1_SEL_PB9: Select PB9
      \arg        GPIO_INT1_SEL_PB8: Select PB8
      \arg        GPIO_INT1_SEL_PF0: Select PF0
      \arg        GPIO_INT1_SEL_PB5: Select PB5
      \arg        GPIO_INT1_SEL_PB4: Select PB4
      \arg        GPIO_INT1_SEL_PB3: Select PB3
      \arg        GPIO_INT1_SEL_PA15: Select PA15
    \param[out] none
    \retval     none
*/
void gpio_int1_config(uint8_t polarity, uint32_t int_sel)
{
    if(INT_POLARITY_LOW == polarity) {
        GPIO_INTCTL &= ~GPIO_INTCTL_INT1PL;
    } else {
        GPIO_INTCTL |= GPIO_INTCTL_INT1PL;
    }
    GPIO_INTCTL &= ~GPIO_INTCTL_INT1SL;
    GPIO_INTCTL |= (int_sel & GPIO_INT1_SEL_PMASK);
}

/*!
    \brief      set port pin mask value (API_ID(0x001AU))
    \param[in]  gpio_pin_mask: please refer to user manual
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_MASK_MSKx(x = 0 - 21)
      \arg        GPIO_MASK_ALL
    \param[in]  mask
                only one parameter can be selected which are shown as below:
      \arg        GPIO_PIN_MASK_TRUE: Pxx logic value is compared to register MSKx
      \arg        GPIO_PIN_MASK_FALSE: Pxx logic value is ignored and will not cause a port mismatch event.
    \param[out] none
    \retval     none
*/
void gpio_pin_mask_set(uint32_t gpio_pin_mask, uint8_t mask)
{
    if(GPIO_PIN_MASK_TRUE == mask) {
        GPIO_MASK |= gpio_pin_mask & GPIO_MASK_ALL;
    } else {
        GPIO_MASK &= ~(gpio_pin_mask & GPIO_MASK_ALL);
    }
}

/*!
    \brief      set port pin match value (API_ID(0x001BU))
    \param[in]  gpio_pin_match
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_MATCH_MATCHx(x = 0 - 21)
    \param[in]  match
                only one parameter can be selected which are shown as below:
      \arg        GPIO_PIN_MATCH_HIGH: Pxx logic value is compared to register MATCHx
      \arg        GPIO_PIN_MATCH_LOW: Pxx logic value is ignored and will not cause a port mismatch event.
    \param[out] none
    \retval     none
*/
void gpio_pin_match_set(uint32_t gpio_pin_match, uint8_t match)
{
    if(GPIO_PIN_MATCH_HIGH == match) {
        GPIO_MATCH |= (gpio_pin_match & GPIO_MATCH_ALL);
    } else {
        GPIO_MATCH &= ~(gpio_pin_match & GPIO_MATCH_ALL);
    }
}

/*!
    \brief      clear pin not match flag (API_ID(0x001CU))
    \param[in]  match_nflag:
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_MATCHF_MATCHFx (x = 0 - 21)
    \param[out] none
    \retval     none
*/
void gpio_pin_not_match_flag_clear(uint32_t match_nflag)
{
    GPIO_MATCHF |= (match_nflag & GPIO_MATCHF_ALL);
}

/*!
    \brief      get pin not match flag (API_ID(0x001DU))
    \param[in]  match_flag:
                only one parameter can be selected which are shown as below:
      \arg        GPIO_MATCHF_MATCHFx (x = 0 - 21)
    \param[out] none
    \retval     SET or RESET
*/
FlagStatus gpio_pin_not_match_flag_get(uint32_t match_flag)
{
    FlagStatus reval = RESET;

    if(RESET != (GPIO_MATCHF & match_flag)) {
        reval = SET;
    } else {
        reval = RESET;
    }
    return reval;
}

/*!
  \brief      pin not match enable (API_ID(0x001EU))
  \param[in]  none
  \param[out] none
  \retval     none
*/
void gpio_pin_not_match_enable(void)
{
    GPIO_MATCHF |= GPIO_MATCHF_MATCHIE;
}

/*!
  \brief      pin not match disable (API_ID(0x001FU))
  \param[in]  none
  \param[out] none
  \retval     none
*/
void gpio_pin_not_match_disable(void)
{
    GPIO_MATCHF &= (~GPIO_MATCHF_MATCHIE);
}
#endif /* GD32E251 */