/*!
    \file    gd32e25x_fmc.c
    \brief   FMC driver

    \version 2025-07-25, V0.2.0, firmware for GD32E25x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32e25x_fmc.h"

/* FMC register bit offset */
#define FMC_OBSTAT_USER_OFFSET            ((uint32_t)8U)
#define FMC_OBSTAT_DATA_OFFSET            ((uint32_t)16U)

/* write option byte */
static void ob_write(option_byte_struct *ob_struct, option_byte_enum ob_num);

/*!
    \brief      unlock the main FMC operation (API_ID(0x0001U))
                it is better to use in pairs with fmc_lock
    \param[in]  none
    \param[out] none
    \retval     none
*/
void fmc_unlock(void)
{
    if((RESET != (FMC_CTL & FMC_CTL_LK))) {
        /* write the FMC key */
        FMC_KEY = UNLOCK_KEY0;
        FMC_KEY = UNLOCK_KEY1;
    }
}

/*!
    \brief      lock the main FMC operation (API_ID(0x0002U))
                it is better to use in pairs with fmc_unlock after an operation
    \param[in]  none
    \param[out] none
    \retval     none
*/
void fmc_lock(void)
{
    /* set the LK bit*/
    FMC_CTL |= FMC_CTL_LK;
}

/*!
    \brief      set the wait state counter value (API_ID(0x0003U))
    \param[in]  wscnt: wait state counter value
      \arg        WS_WSCNT_0: 0 wait state added
      \arg        WS_WSCNT_1: 1 wait state added
      \arg        WS_WSCNT_2: 2 wait state added
    \param[out] none
    \retval     none
*/
void fmc_wscnt_set(uint8_t wscnt)
{
    uint32_t reg;

    reg = FMC_WS;
    /* set the wait state counter value */
    reg &= ~FMC_WS_WSCNT;
    FMC_WS = (reg | ((uint32_t)wscnt & FMC_WS_WSCNT));
}

/*!
    \brief      pre-fetch enable  (API_ID(0x0004U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void fmc_prefetch_enable(void)
{
    FMC_WS |= FMC_WS_PFEN;
}

/*!
    \brief      pre-fetch disable (API_ID(0x0005U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void fmc_prefetch_disable(void)
{
    FMC_WS &= ~FMC_WS_PFEN;
}

/*!
    \brief      erase page (API_ID(0x0006U))
    \param[in]  page_number: page offset
                only one parameter can be selected which is shown as below:
      \arg        0 ~ (max count of pages)-1
    \param[out] none
    \retval     fmc_state: state of FMC
      \arg        FMC_READY: the operation has been completed
      \arg        FMC_BUSY: the operation is in progress
      \arg        FMC_PGERR: program error
      \arg        FMC_PGAERR: program alignment error
      \arg        FMC_WPERR: erase/program protection error
      \arg        FMC_TOERR: timeout error
      \arg        FMC_OB_HSPC: option byte security protection code high
      \arg        FMC_UNDEFINEDERR: undefined error for function input parameter checking
*/
fmc_state_enum fmc_page_erase(uint32_t page_number)
{
    fmc_state_enum fmc_state;
#ifdef FW_DEBUG_ERR_REPORT
    if(page_number >= MAIN_FLASH_PAGE_TOTAL_NUM) {
        fw_debug_report_err(FMC_MODULE_ID, API_ID(0x0006U), ERR_PARAM_OUT_OF_RANGE);
        fmc_state = FMC_UNDEFINEDERR;
    } else
#endif /* FW_DEBUG_ERR_REPORT */
    {
        fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);

        if(FMC_READY == fmc_state) {
            /* start page erase */
            FMC_CTL |= FMC_CTL_PER;
            FMC_ADDR = MAIN_FLASH_BASE_ADDRESS + page_number * MAIN_FLASH_PAGE_SIZE;
            FMC_CTL |= FMC_CTL_START;

            /* wait for the FMC ready */
            fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);

            /* reset the PER bit */
            FMC_CTL &= ~FMC_CTL_PER;
        }
    }
    /* return the FMC state  */
    return fmc_state;
}

/*!
    \brief      erase whole chip (API_ID(0x0007U))
    \param[in]  none
    \param[out] none
    \retval     fmc_state: state of FMC
      \arg        FMC_READY: the operation has been completed
      \arg        FMC_BUSY: the operation is in progress
      \arg        FMC_PGERR: program error
      \arg        FMC_PGAERR: program alignment error
      \arg        FMC_WPERR: erase/program protection error
      \arg        FMC_TOERR: timeout error
      \arg        FMC_OB_HSPC: option byte security protection code high
*/
fmc_state_enum fmc_mass_erase(void)
{
    fmc_state_enum fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);

    if(FMC_READY == fmc_state) {
        /* start chip erase */
        FMC_CTL |= FMC_CTL_MER;
        FMC_CTL |= FMC_CTL_START;

        /* wait for the FMC ready */
        fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);

        /* reset the MER bit */
        FMC_CTL &= ~FMC_CTL_MER;
    }

    /* return the fmc state  */
    return fmc_state;
}

/*!
    \brief      program a double word at the corresponding address in main flash, this
                function also applies to OTP(address 0x1FFF_7000~0x1FFF_73FF) programming (API_ID(0x0008U))
    \param[in]  address: address to program
    \param[in]  data: double word to program
    \param[out] none
    \retval     fmc_state: state of FMC
      \arg        FMC_READY: the operation has been completed
      \arg        FMC_BUSY: the operation is in progress
      \arg        FMC_PGERR: program error
      \arg        FMC_PGAERR: program alignment error
      \arg        FMC_WPERR: erase/program protection error
      \arg        FMC_TOERR: timeout error
      \arg        FMC_OB_HSPC: option byte security protection code high
*/
fmc_state_enum fmc_doubleword_program(uint32_t address, uint64_t data)
{
    uint32_t data0, data1;
    fmc_state_enum fmc_state;
#ifdef FW_DEBUG_ERR_REPORT
    if((NOT_FMC_VALID_ADDRESS(address)) || (0U != (address % 8U))) {
        fw_debug_report_err(FMC_MODULE_ID, API_ID(0x0008U), ERR_PARAM_OUT_OF_RANGE);
        fmc_state = FMC_UNDEFINEDERR;
    } else
#endif /* FW_DEBUG_ERR_REPORT */
    {
        fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
        
        data0 = (uint32_t)(data & 0xFFFFFFFFU);
        data1 = (uint32_t)((data >> 32U) & 0xFFFFFFFFU);
        
        /* configure program width */
        if(FMC_READY == fmc_state) {
            /* set the PG bit to start program */
            FMC_CTL |= FMC_CTL_PG;
            REG32(address) = data0;
            REG32(address + 4U) = data1;
            /* wait for the FMC ready */
            fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
            /* reset the PG bit */
            FMC_CTL &= ~FMC_CTL_PG;
        }
    }
    /* return the FMC state */
    return fmc_state;
}

/*!
    \brief      SRAM parity check enable, SRAM ECC disable (API_ID(0x0009U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void fmc_sramparity_enable(void)
{
    FMC_CTL |= FMC_CTL_RPCS;
    FMC_CTL &= ~FMC_CTL_RPCD;
}

/*!
    \brief      SRAM parity check disable, SRAM ECC enable (API_ID(0x000AU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void fmc_sramparity_disable(void)
{
    FMC_CTL |= FMC_CTL_RPCS;
    FMC_CTL |= FMC_CTL_RPCD;
}

/*!
    \brief      PB11 pin as NRST (API_ID(0x000BU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void fmc_nrstmdvalue_enable(void)
{
    FMC_CTL |= FMC_CTL_NRSTMD_REG;
    FMC_CTL |= FMC_CTL_NRSTMD_VALUE;
}

/*!
    \brief      PB11 pin as GPIO (API_ID(0x000CU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void fmc_nrstmdvalue_disable(void)
{
    FMC_CTL |= FMC_CTL_NRSTMD_REG;
    FMC_CTL &= ~FMC_CTL_NRSTMD_VALUE;
}

/*!
    \brief      unlock the option byte operation (API_ID(0x000DU))
                it is better to use in pairs with ob_lock
    \param[in]  none
    \param[out] none
    \retval     none
*/
void ob_unlock(void)
{
    if(RESET == (FMC_CTL & FMC_CTL_OBWEN)) {
        /* write the FMC key */
        FMC_OBKEY = UNLOCK_KEY0;
        FMC_OBKEY = UNLOCK_KEY1;
    }
    /* wait until OBWEN bit is set by hardware */
    while(RESET == (FMC_CTL & FMC_CTL_OBWEN)) {
    }
}

/*!
    \brief      lock the option byte operation (API_ID(0x000EU))
                it is better to use in pairs with ob_unlock after an operation
    \param[in]  none
    \param[out] none
    \retval     none
*/
void ob_lock(void)
{
    /* reset the OBWE bit */
    FMC_CTL &= ~FMC_CTL_OBWEN;
}

/*!
    \brief      reload the option byte and generate a system reset (API_ID(0x000FU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void ob_reset(void)
{
    /* set the OBRLD bit */
    FMC_CTL |= FMC_CTL_OBRLD;
}

/*!
    \brief      erase the option byte (API_ID(0x0010U))
                programmer must ensure FMC & option byte are both unlocked before calling this function
    \param[in]  none
    \param[out] none
    \retval     fmc state
      \arg        FMC_READY: the operation has been completed
      \arg        FMC_BUSY: the operation is in progress
      \arg        FMC_PGERR: program error
      \arg        FMC_PGAERR: program alignment error
      \arg        FMC_WPERR: erase/program protection error
      \arg        FMC_TOERR: timeout error
      \arg        FMC_OB_HSPC: option byte security protection code high
*/
fmc_state_enum ob_erase(void)
{
    option_byte_struct ob_struct;

    fmc_state_enum fmc_state = FMC_UNDEFINEDERR;
    
    /* check the option bytes security protection value */
    if(OB_OBSTAT_PLEVEL_HIGH == ob_obstat_plevel_get())
    {
        fmc_state = FMC_OB_HSPC;
    } else {

        ob_struct.ob_spc = OB_SPC;
        ob_struct.ob_user = 0xFFFFU;
        ob_struct.ob_data = 0xFFFFFFFFU;
            
        fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
        if(FMC_READY == fmc_state) {
            /* start erase the option byte */
            FMC_CTL |= FMC_CTL_OBER;
            FMC_CTL |= FMC_CTL_START;
        
            /* wait for the FMC ready */
            fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
        
            if(FMC_READY == fmc_state) {
                /* reset the OBER bit */
                FMC_CTL &= ~FMC_CTL_OBER;
        
                /* set the OBPG bit */
                FMC_CTL |= FMC_CTL_OBPG;
        
                /* restore the last get option byte security protection code */
                ob_write(&ob_struct, DOUBLEWORD_SPC_USER_DATA);
                /* wait for the FMC ready */
                fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
        
                if(FMC_TOERR != fmc_state) {
                    /* reset the OBPG bit */
                    FMC_CTL &= ~FMC_CTL_OBPG;
                }
            } else {
                if(FMC_TOERR != fmc_state) {
                    /* reset the OBER bit */
                    FMC_CTL &= ~FMC_CTL_OBER;
                }
            }
        }
    }
    /* return the FMC state */
    return fmc_state;
}

/*!
    \brief      enable option byte write protection (OB_WP) (API_ID(0x0011U))
    \param[in]  ob_wp: write protection configuration data
    \param[out] none
    \retval     fmc_state: state of FMC
      \arg        FMC_READY: the operation has been completed
      \arg        FMC_BUSY: the operation is in progress
      \arg        FMC_PGERR: program error
      \arg        FMC_PGAERR: program alignment error
      \arg        FMC_WPERR: erase/program protection error
      \arg        FMC_TOERR: timeout error
      \arg        FMC_OB_HSPC: option byte security protection code high
*/
fmc_state_enum ob_write_protection_enable(uint16_t ob_wp)
{
    option_byte_struct ob_struct;
    uint32_t temp;
    uint32_t ob_wrp_val = 0U;
    fmc_state_enum fmc_state = FMC_UNDEFINEDERR;
    
    /* check the option bytes security protection value */
    if(OB_OBSTAT_PLEVEL_HIGH == ob_obstat_plevel_get())
    {
        fmc_state = FMC_OB_HSPC;
    } else {
        
        ob_struct.ob_spc = OB_SPC;
        ob_struct.ob_user = OB_USER;
        ob_struct.ob_data = OB_DATA;
        ob_struct.ob_wp = OB_WP;
        
        temp = ~((uint32_t)ob_wp);
        
        ob_wrp_val |= temp & 0x00FFU;
        ob_wrp_val |= ((temp & 0xFF00U) >> 8U) << 16U;
        
        ob_struct.ob_wp = ob_struct.ob_wp & ob_wrp_val;
        
        fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
        
        if(FMC_READY == fmc_state) {
            /* start erase the option byte */
            FMC_CTL |= FMC_CTL_OBER;
            FMC_CTL |= FMC_CTL_START;
        
            /* wait for the FMC ready */
            fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
        
            if(FMC_READY == fmc_state) {
        
                /* reset the OBER bit */
                FMC_CTL &= ~FMC_CTL_OBER;
        
                /* enable the option bytes programming */
                FMC_CTL |= FMC_CTL_OBPG;
                ob_write(&ob_struct, DOUBLEWORD_WP);
                ob_write(&ob_struct, DOUBLEWORD_SPC_USER_DATA);
                
                /* wait for the FMC ready */
                fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
        
                if(FMC_TOERR != fmc_state) {
                    /* reset the OBPG bit */
                    FMC_CTL &= ~FMC_CTL_OBPG;
                }
            } else {
                if(FMC_TOERR != fmc_state) {
                    /* reset the OBER bit */
                    FMC_CTL &= ~FMC_CTL_OBER;
                }
            }
        }
    }
    /* return the FMC state */
    return fmc_state;
}

/*!
    \brief      configure security protection (API_ID(0x0012U))
    \param[in]  ob_spc: specify security protection code
      \arg        FMC_NSPC: no security protection
      \arg        FMC_LSPC: low security protection
      \arg        FMC_HSPC: high security protection
    \param[out] none
    \retval     fmc_state: state of FMC
      \arg        FMC_READY: the operation has been completed
      \arg        FMC_BUSY: the operation is in progress
      \arg        FMC_PGERR: program error
      \arg        FMC_PGAERR: program alignment error
      \arg        FMC_WPERR: erase/program protection error
      \arg        FMC_TOERR: timeout error
      \arg        FMC_OB_HSPC: option byte security protection code high
*/
fmc_state_enum ob_security_protection_config(uint16_t ob_spc)
{
    option_byte_struct ob_struct;
    fmc_state_enum fmc_state = FMC_UNDEFINEDERR;

    /* check the option bytes security protection value */
    if(OB_OBSTAT_PLEVEL_HIGH == ob_obstat_plevel_get())
    {
        fmc_state = FMC_OB_HSPC;
    } else {
        
        ob_struct.ob_spc = ob_spc;
        ob_struct.ob_user = OB_USER;
        ob_struct.ob_data = OB_DATA;
        ob_struct.ob_wp = OB_WP;
        fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
        
        if(FMC_READY == fmc_state) {
            /* start erase the option byte */
            FMC_CTL |= FMC_CTL_OBER;
            FMC_CTL |= FMC_CTL_START;
        
            /* wait for the FMC ready */
            fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
        
            if(FMC_READY == fmc_state) {
        
                /* reset the OBER bit */
                FMC_CTL &= ~FMC_CTL_OBER;
        
                /* enable the option bytes programming */
                FMC_CTL |= FMC_CTL_OBPG;
        
                ob_write(&ob_struct, DOUBLEWORD_SPC_USER_DATA);
                ob_write(&ob_struct, DOUBLEWORD_WP);
                /* wait for the FMC ready */
                fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
        
                if(FMC_TOERR != fmc_state) {
                    /* reset the OBPG bit */
                    FMC_CTL &= ~FMC_CTL_OBPG;
                }
            } else {
                if(FMC_TOERR != fmc_state) {
                    /* reset the OBER bit */
                    FMC_CTL &= ~FMC_CTL_OBER;
                }
            }
        }
    }
    /* return the FMC state */
    return fmc_state;
}

/*!
    \brief      program the FMC user option byte (API_ID(0x0013U))
                this function can only clear the corresponding bits to be 0 rather than 1.
                the function ob_erase is used to set all the bits to be 1.
    \param[in]  ob_user: user option byte
                one or more parameters (bitwise AND) can be selected which are shown as below:
      \arg        OB_FWDGT_HW: hardware free watchdog timer
      \arg        OB_DEEPSLEEP_RST: generate a reset instead of entering deepsleep mode
      \arg        OB_STDBY_RST: generate a reset instead of entering standby mode
      \arg        OB_BOOT0_SET_1: BOOT0 bit is 1
      \arg        OB_BOOT1_SET_1: BOOT1 bit is 1
      \arg        OB_VDDA_DISABLE: disable VDDA monitor
      \arg        OB_SRAM_PARITY_ENABLE: enable sram parity check
      \arg        OB_NRSTMD_PB11: PB11 is GPIO
    \param[out] none
    \retval     fmc_state: state of FMC
      \arg        FMC_READY: the operation has been completed
      \arg        FMC_BUSY: the operation is in progress
      \arg        FMC_PGERR: program error
      \arg        FMC_PGAERR: program alignment error
      \arg        FMC_WPERR: erase/program protection error
      \arg        FMC_TOERR: timeout error
      \arg        FMC_OB_HSPC: option byte security protection code high
*/
fmc_state_enum ob_user_write(uint8_t ob_user)
{
    option_byte_struct ob_struct;
    fmc_state_enum fmc_state = FMC_UNDEFINEDERR;
    /* check the option bytes security protection value */
    if(OB_OBSTAT_PLEVEL_HIGH == ob_obstat_plevel_get())
    {
        fmc_state = FMC_OB_HSPC;
    } else {
        ob_struct.ob_user = (uint16_t)ob_user;
        ob_struct.ob_spc = OB_SPC;
        ob_struct.ob_data = OB_DATA;
        ob_struct.ob_wp = OB_WP;
         /* check whether FMC is ready or not */
        fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
        if(FMC_READY == fmc_state) {
            /* start erase the option byte */
            FMC_CTL |= FMC_CTL_OBER;
            FMC_CTL |= FMC_CTL_START;
        
            /* wait for the FMC ready */
            fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
        
            if(FMC_READY == fmc_state) {
        
                /* reset the OBER bit */
                FMC_CTL &= ~FMC_CTL_OBER;
        
                /* enable the option bytes programming */
                FMC_CTL |= FMC_CTL_OBPG;
        
                ob_write(&ob_struct, DOUBLEWORD_SPC_USER_DATA);
                ob_write(&ob_struct, DOUBLEWORD_WP);
                /* wait for the FMC ready */
                fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
        
                if(FMC_TOERR != fmc_state) {
                    /* reset the OBPG bit */
                    FMC_CTL &= ~FMC_CTL_OBPG;
                }
            } else {
                if(FMC_TOERR != fmc_state) {
                    /* reset the OBER bit */
                    FMC_CTL &= ~FMC_CTL_OBER;
                }
            }
        }
    }
    /* return the FMC state */
    return fmc_state;
}

/*!
    \brief      program the FMC data option byte (API_ID(0x0014U))
    \param[in]  data: the data to be programmed, OB_DATA[0:15]
    \param[out] none
    \retval     fmc_state: state of FMC
      \arg        FMC_READY: the operation has been completed
      \arg        FMC_BUSY: the operation is in progress
      \arg        FMC_PGERR: program error
      \arg        FMC_PGAERR: program alignment error
      \arg        FMC_WPERR: erase/program protection error
      \arg        FMC_TOERR: timeout error
      \arg        FMC_OB_HSPC: high security protection
      \arg        FMC_UNDEFINEDERR: undefined error for function input parameter checking
*/
fmc_state_enum ob_data_program(uint16_t data)
{
    option_byte_struct ob_struct;
    fmc_state_enum fmc_state = FMC_UNDEFINEDERR;
    
    /* check the option bytes security protection value */
    if(OB_OBSTAT_PLEVEL_HIGH == ob_obstat_plevel_get())
    {
        fmc_state = FMC_OB_HSPC;
    } else {
        
        fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
        uint32_t val = 0U;
        
        ob_struct.ob_spc = OB_SPC;
        ob_struct.ob_user = OB_USER;
        ob_struct.ob_wp = OB_WP;
        val |= (uint32_t)data & 0x00FFU;
        val |= (((uint32_t)data & 0xFF00U) >> 8U) << 16U;
        ob_struct.ob_data = val;
        
        if(FMC_READY == fmc_state) {
            /* start erase the option byte */
            FMC_CTL |= FMC_CTL_OBER;
            FMC_CTL |= FMC_CTL_START;
        
            /* wait for the FMC ready */
            fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
            if(FMC_READY == fmc_state) {
                /* reset the OBER bit */
                FMC_CTL &= ~FMC_CTL_OBER;
                /* set the OBPG bit */
                FMC_CTL |= FMC_CTL_OBPG;
                ob_write(&ob_struct, DOUBLEWORD_SPC_USER_DATA);
                ob_write(&ob_struct, DOUBLEWORD_WP);
                /* wait for the FMC ready */
                fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);
        
                if(FMC_TOERR != fmc_state) {
                    /* reset the OBPG bit */
                    FMC_CTL &= ~FMC_CTL_OBPG;
                }
            } else {
                if(FMC_TOERR != fmc_state) {
                    /* reset the OBER bit */
                    FMC_CTL &= ~FMC_CTL_OBER;
                }
            }
        }
    }
    /* return the FMC state */
    return fmc_state;
}

/*!
    \brief      get OB_USER in register FMC_OBSTAT (API_ID(0x0015U))
    \param[in]  none
    \param[out] none
    \retval     ob_user
*/
uint8_t ob_user_get(void)
{
    return (uint8_t)(FMC_OBSTAT >> FMC_OBSTAT_USER_OFFSET);
}

/*!
    \brief      get OB_DATA in register FMC_OBSTAT (API_ID(0x0016U))
    \param[in]  none
    \param[out] none
    \retval     ob_data
*/
uint16_t ob_data_get(void)
{
    return (uint16_t)(FMC_OBSTAT >> FMC_OBSTAT_DATA_OFFSET);
}

/*!
    \brief      get the FMC option byte write protection (OB_WP) in register FMC_WP (API_ID(0x0017U))
    \param[in]  none
    \param[out] none
    \retval     OB_WP
*/
uint16_t ob_write_protection_get(void)
{
    return (uint16_t)(FMC_WP);
}

/*!
    \brief      get the value of FMC option byte security protection level (PLEVEL) in FMC_OBSTAT register (API_ID(0x0018U))
    \param[in]  none
      \arg        OB_OBSTAT_PLEVEL_NO: the operation has been completed
      \arg        OB_OBSTAT_PLEVEL_LOW: the operation has been completed
      \arg        OB_OBSTAT_PLEVEL_HIGH: the operation has been completed
    \param[out] none
    \retval     the value of PLEVEL
*/
uint32_t ob_obstat_plevel_get(void)
{
    return ((FMC_OBSTAT & FMC_OBSTAT_PLEVEL) >> 1U);
}

/*!
    \brief      get the FMC ECC error address in register FMC_ECCADDR (API_ID(0x0019U))
    \param[in]  none
    \param[out] none
    \retval     uint32_t, the FMC ECC error address
*/
uint32_t fmc_ecc_error_address_get(void)
{
    /* return the FMC write protection option bytes value */
    return FMC_ECCADDR;
}

/*!
    \brief      FMC ECC error injection enable (API_ID(0x001AU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void fmc_ecc_error_injection_enable(void)
{
    FMC_ECCCS |= FMC_ECCCS_ECCINJEN;
}

/*!
    \brief      FMC ECC error injection disable (API_ID(0x001BU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void fmc_ecc_error_injection_disable(void)
{
    FMC_ECCCS &= ~FMC_ECCCS_ECCINJEN;
}

/*!
    \brief      FMC ECC error injection (API_ID(0x001CU))
    \param[in]  error_typ: flash ECC error injection type
                only one parameter can be selected which is shown as below:
      \arg        FMC_NO_ECC_ERROR: no flash ECC error fault injection
      \arg        FMC_ONE_BIT_ECC_ERROR_1: flash one bit ECC error fault injection
      \arg        FMC_ONE_BIT_ECC_ERROR_2: flash one bit ECC error fault injection
      \arg        FMC_TWO_BITS_ECC_ERROR: flash two bits ECC error fault injection
    \param[out] none
    \retval     none
*/
void fmc_ecc_error_injection(uint8_t error_typ)
{
    FMC_ECCCS |= ((uint32_t)error_typ << 24U) & FMC_ECCCS_ECCINJDATA;
}

/* FMC interrupts and flags management functions */
/*!
    \brief      enable FMC interrupt (API_ID(0x001DU))
    \param[in]  interrupt: the FMC interrupt source
      \arg        FMC_INT_END: FMC flash end of operation interrupt
      \arg        FMC_INT_ERR: FMC flash error interrupt
      \arg        FMC_INT_ECCDET: FMC flash double bits ECC error interrupt
      \arg        FMC_INT_ECCCOR: FMC flash single bit ECC error interrupt
    \param[out] none
    \retval     none
*/
void fmc_interrupt_enable(fmc_interrupt_enum interrupt)
{
    /* enable interrupt in CTL0 register */
    FMC_REG_VAL(interrupt) |= BIT(FMC_BIT_POS(interrupt));
}

/*!
    \brief      disable FMC interrupt (API_ID(0x001EU))
    \param[in]  interrupt: the FMC interrupt source
      \arg        FMC_INT_END: FMC flash end of operation interrupt
      \arg        FMC_INT_ERR: FMC flash error interrupt
      \arg        FMC_INT_ECCDET: FMC flash double bits ECC error interrupt
      \arg        FMC_INT_ECCCOR: FMC flash single bit ECC error interrupt
    \param[out] none
    \retval     none
*/
void fmc_interrupt_disable(fmc_interrupt_enum interrupt)
{
    /* disable interrupt in CTL0 register */
    FMC_REG_VAL(interrupt) &= ~BIT(FMC_BIT_POS(interrupt));
}

/*!
    \brief      get flag set or reset (API_ID(0x001FU))
    \param[in]  flag: check FMC flag
                only one parameter can be selected which is shown as below:
      \arg        FMC_FLAG_BUSY: flash busy flag
      \arg        FMC_FLAG_PGERR: flash programming error flag
      \arg        FMC_FLAG_PGAERR: flash program alignment error flag
      \arg        FMC_FLAG_WPERR: flash write protection error flag
      \arg        FMC_FLAG_ENDF: flash end of operation flag
      \arg        FMC_FLAG_PARITYEN: SRAM parity check enable flag
      \arg        FMC_FLAG_ECCEN: SRAM ECC check enable flag
      \arg        FMC_FLAG_ECCCOR: flash single bit ECC error flag
      \arg        FMC_FLAG_ECCDET: flash double bits ECC error flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus fmc_flag_get(fmc_flag_enum flag)
{
    FlagStatus ret;

    /* get flag */
    if(0U != (FMC_REG_VAL(flag) & BIT(FMC_BIT_POS(flag)))) {
        ret = SET;
    } else {
        ret = RESET;
    }

    return ret;
}

/*!
    \brief      clear the FMC pending flag by writing 1 (API_ID(0x0020U))
    \param[in]  flag: clear FMC flag
                only one parameter can be selected which is shown as below:
      \arg        FMC_FLAG_PGERR: flash programming error flag
      \arg        FMC_FLAG_PGAERR: flash program alignment error flag
      \arg        FMC_FLAG_WPERR: flash write protection error flag
      \arg        FMC_FLAG_ENDF: flash end of programming flag
      \arg        FMC_FLAG_ECCCOR: flash single bit ECC error flag
      \arg        FMC_FLAG_ECCDET: flash double bits ECC error flag
    \param[out] none
    \retval     none
*/
void fmc_flag_clear(fmc_flag_enum flag)
{
    /* clear the flags in STAT/ECCCS register */
    FMC_REG_VAL(flag) = BIT(FMC_BIT_POS(flag));
}

/*!
    \brief      get interrupt flag set or reset (API_ID(0x0021U))
    \param[in]  flag: check FMC flag
                only one parameter can be selected which is shown as below:
      \arg        FMC_INT_FLAG_PGERR: flash programming error flag
      \arg        FMC_INT_FLAG_PGAERR: flash program alignment error flag
      \arg        FMC_INT_FLAG_WPERR: flash write protection error flag
      \arg        FMC_INT_FLAG_END: flash end of programming flag
      \arg        FMC_INT_FLAG_ECCCOR: flash single bit ECC error interrupt flag
      \arg        FMC_INT_FLAG_ECCDET: flash double bits ECC error interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus fmc_interrupt_flag_get(fmc_interrupt_flag_enum int_flag)
{
    uint32_t intenable = 0U, flagstatus = 0U;
    FlagStatus ret;

    /* get the interrupt enable bit status */
    intenable = (FMC_REG_VAL(int_flag) & BIT(FMC_BIT_POS(int_flag)));
    /* get the corresponding flag bit status */
    flagstatus = (FMC_REG_VAL2(int_flag) & BIT(FMC_BIT_POS2(int_flag)));

    if(flagstatus && intenable) {
        ret = SET;
    } else {
        ret = RESET;
    }

    return ret;
}

/*!
    \brief      clear the FMC interrupt pending flag by writing 1 (API_ID(0x0022U))
    \param[in]  flag: clear FMC flag
                only one parameter can be selected which is shown as below:
      \arg        FMC_INT_FLAG_PGERR: flash programming error flag
      \arg        FMC_INT_FLAG_PGAERR: flash program alignment error flag
      \arg        FMC_INT_FLAG_WPERR: flash write protection error flag
      \arg        FMC_INT_FLAG_END: flash end of programming flag
      \arg        FMC_INT_FLAG_ECCCOR: flash single bit ECC error interrupt flag
      \arg        FMC_INT_FLAG_ECCDET: flash double bits ECC error interrupt flag
    \param[out] none
    \retval     none
*/
void fmc_interrupt_flag_clear(fmc_interrupt_flag_enum int_flag)
{
    /* clear the interrupt flag in FMC_STAT/FMC_ECCCS register */
    FMC_REG_VAL2(int_flag) = BIT(FMC_BIT_POS2(int_flag));
}

/*!
    \brief      get the FMC state (API_ID(0x0023U))
    \param[in]  none
    \param[out] none
    \retval     fmc_state: state of FMC
      \arg        FMC_READY: the operation has been completed
      \arg        FMC_BUSY: the operation is in progress
      \arg        FMC_PGERR: program error
      \arg        FMC_PGAERR: program alignment error
      \arg        FMC_WPERR: erase/program protection error
*/
fmc_state_enum fmc_state_get(void)
{
    fmc_state_enum fmc_state = FMC_READY;

    if((uint32_t)0x00U != (FMC_STAT & FMC_STAT_BUSY)) {
        fmc_state = FMC_BUSY;
    } else {
        if((uint32_t)0x00U != (FMC_STAT & FMC_STAT_WPERR)) {
            fmc_state = FMC_WPERR;
        } else if((uint32_t)0x00U != (FMC_STAT & FMC_STAT_PGERR)) {
            fmc_state = FMC_PGERR;
        } else if((uint32_t)0x00U != (FMC_STAT & FMC_STAT_PGAERR)) {
            fmc_state = FMC_PGAERR;
        } else {
            /* illegal parameters */
        }
    }
    /* return the FMC state */
    return fmc_state;
}

/*!
    \brief      check whether FMC is ready or not (API_ID(0x0024U))
    \param[in]  timeout: timeout count
    \param[out] none
    \retval     state of FMC
      \arg        FMC_READY: the operation has been completed
      \arg        FMC_BUSY: the operation is in progress
      \arg        FMC_PGERR: program error
      \arg        FMC_PGAERR: program alignment error
      \arg        FMC_WPERR: erase/program protection error
      \arg        FMC_TOERR: timeout error
    \note       This function includes timeout exit scenarios.
                Modify according to the user's actual usage scenarios.
*/
fmc_state_enum fmc_ready_wait(uint32_t timeout)
{
    fmc_state_enum fmc_state = FMC_BUSY;

    /* wait for FMC ready */
    do {
        /* get FMC state */
        fmc_state = fmc_state_get();
        timeout--;
    } while((FMC_BUSY == fmc_state) && (0U != timeout));

    if(FMC_BUSY == fmc_state) {
        fmc_state = FMC_TOERR;
    }
    /* return the FMC state */
    return fmc_state;
}

/*!
    \brief      write option byte (API_ID(0x0025U))
    \param[in]  ob_struct: option byte structure
                           and the member values are shown as below:
                  ob_spc: value of spc in option byte
                  ob_user: value of user in option byte
                  ob_data: value of data in option byte
                  ob_wp: value of wp in option byte
    \param[in]  ob_num: refer to rcu_periph_enum
      \arg        DOUBLEWORD_SPC_USER_DATA: first 64 bits of option byte, include SPC, USER, DATA
      \arg        DOUBLEWORD_WP: second 64 bits of option byte, include WP
    \param[out] none
    \retval     none
*/
static void ob_write(option_byte_struct *ob_struct, option_byte_enum ob_num)
{
    uint32_t ob_val0 = 0U;
    uint32_t ob_val1 = 0U;

    /* write first 64 bits of option byte */
    if(DOUBLEWORD_SPC_USER_DATA == ob_num) {
        ob_val0 = (uint32_t)(ob_struct->ob_spc);
        ob_val0 |= (((uint32_t)ob_struct->ob_user) << 16U);
        ob_val1 |= ob_struct->ob_data;
        REG32(0x1ffff800U) = ob_val0;
        REG32(0x1ffff804U) = ob_val1;
    } else {
        /* write second 64 bits of option byte */
        ob_val0 = (uint32_t)(ob_struct->ob_wp);
        ob_val1 = 0xffffffffU;
        REG32(0x1ffff808U) = ob_val0;
        REG32(0x1ffff80cU) = ob_val1;
    }
}
