/*!
    \file    gd32e25x_cmp.c
    \brief   CMP driver

    \version 2025-07-25, V0.2.0, firmware for GD32E25x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32e25x_cmp.h"

/*!
    \brief      CMP deinit (API_ID(0x0001U))
    \param[in]  cmp_periph
      \arg        CMP0: comparator 0
    \param[out] none
    \retval     none
*/
void cmp_deinit(cmp_enum cmp_periph)
{
    if(CMP0 == cmp_periph){
        CMP_CS &= ((uint32_t)0x00000000U);
    }else{
        /* illegal parameters */
    }
}

/*!
    \brief      CMP mode init (API_ID(0x0002U))
    \param[in]  cmp_periph
      \arg        CMP0: comparator 0
    \param[in]  operating_mode
      \arg        CMP_MODE_HIGHSPEED: high speed mode
      \arg        CMP_MODE_MIDDLESPEED: medium speed mode
      \arg        CMP_MODE_LOWSPEED: low speed mode
      \arg        CMP_MODE_VERYLOWSPEED: very-low speed mode
    \param[in]  inverting_input
      \arg        CMP_INVERTING_INPUT_1_4VREFINT: CMP inverting input 1/4 Vrefint
      \arg        CMP_INVERTING_INPUT_1_2VREFINT: CMP inverting input 1/2 Vrefint
      \arg        CMP_INVERTING_INPUT_3_4VREFINT: CMP inverting input 3/4 Vrefint
      \arg        CMP_INVERTING_INPUT_VREFINT: CMP inverting input Vrefint
      \arg        CMP_INVERTING_INPUT_PA4: CMP inverting input PA4
      \arg        CMP_INVERTING_INPUT_PA5: CMP inverting input PA5
      \arg        CMP_INVERTING_INPUT_PA0: CMP inverting input PA0
      \arg        CMP_INVERTING_INPUT_PA2: CMP inverting input PA2
      \arg        CMP_INVERTING_INPUT_DAC0_OUT0: CMP inverting input DAC0_OUT0
      \arg        CMP_INVERTING_INPUT_DAC0_OUT1: CMP inverting input DAC0_OUT1
      \arg        CMP_INVERTING_INPUT_PA1: CMP inverting input PA1
      \arg        CMP_INVERTING_INPUT_PA3: CMP inverting input PA3 
      \arg        CMP_INVERTING_INPUT_PB0: CMP inverting input PB0
      \arg        CMP_INVERTING_INPUT_PB1: CMP inverting input PB1
      \arg        CMP_INVERTING_INPUT_PB3: CMP inverting input PB3
      \arg        CMP_INVERTING_INPUT_PB10: CMP inverting input PB10
      \arg        CMP_INVERTING_INPUT_PA14: CMP inverting input PA14
      \arg        CMP_INVERTING_INPUT_PA15: CMP inverting input PA15
    \param[in]  output_hysteresis
      \arg        CMP_HYSTERESIS_NO: output no hysteresis
      \arg        CMP_HYSTERESIS_LOW: output low hysteresis
      \arg        CMP_HYSTERESIS_MIDDLE: output middle hysteresis
      \arg        CMP_HYSTERESIS_HIGH: output high hysteresis
    \param[out] none
    \retval     none
*/
void cmp_mode_init(cmp_enum cmp_periph, uint32_t operating_mode, uint32_t inverting_input, uint32_t output_hysteresis)
{
    uint32_t temp = 0U;
    if(CMP0 == cmp_periph){
        /* initialize comparator 0 mode */
        temp = CMP_CS;
        temp &= ~(uint32_t)(CMP_CS_CMP0M | CMP_CS_CMP0MSEL | CMP_CS_CMP0HST);
        temp |= (uint32_t)((operating_mode & CMP_CS_CMP0M) | (output_hysteresis & CMP_CS_CMP0HST) \
                        | ((uint32_t)(inverting_input) & CMP_CS_CMP0MSEL));
        CMP_CS = temp;
    }else{
        /* illegal parameters */
    }
}

/*!
    \brief      initialize comparator input  (API_ID(0x0003U))
    \param[in]  cmp_periph
      \arg        CMP0: comparator 0
    \param[in]  noninverting_input
      \arg        CMP_NONINVERTING_INPUT_PA1: CMP noninverting input PA1
      \arg        CMP_NONINVERTING_INPUT_DAC0_OUT0: CMP noninverting input DAC0_OUT0
      \arg        CMP_NONINVERTING_INPUT_PA4: CMP noninverting input PA4
      \arg        CMP_NONINVERTING_INPUT_PA0: CMP noninverting input PA0
      \arg        CMP_NONINVERTING_INPUT_PA2: CMP noninverting input PA2
      \arg        CMP_NONINVERTING_INPUT_PA3: CMP noninverting input PA3
      \arg        CMP_NONINVERTING_INPUT_PB0: CMP noninverting input PB0
      \arg        CMP_NONINVERTING_INPUT_PB1: CMP noninverting input PB1
      \arg        CMP_NONINVERTING_INPUT_PB3: CMP noninverting input PB3
      \arg        CMP_NONINVERTING_INPUT_PB10: CMP noninverting input PB10
      \arg        CMP_NONINVERTING_INPUT_PA14: CMP noninverting input PA14
      \arg        CMP_NONINVERTING_INPUT_PA15: CMP noninverting input PA15
    \param[out] none
    \retval     none
*/
void cmp_noninverting_input_select(cmp_enum cmp_periph, uint32_t noninverting_input)
{
    uint32_t temp;
    if(CMP0 == cmp_periph){
        temp = CMP_CS;
        /* initialize comparator mode */
        temp &= ~(uint32_t)CMP_CS_CMP0PSEL;
        temp |= (uint32_t)(noninverting_input & CMP_CS_CMP0PSEL);
        CMP_CS = temp;
    }else{
        /* illegal parameters */
    }
}

/*!
    \brief      initialize comparator output (API_ID(0x0004U))
    \param[in]  cmp_periph
      \arg        CMP0: comparator 0
    \param[in]  output_selection
      \arg        CMP_OUTPUT_NONE: CMP output none
      \arg        CMP_OUTPUT_TIMER0_BKIN: CMP output TIMER0 break input
      \arg        CMP_OUTPUT_TIMER0_IC0: CMP output TIMER0_CH0 input capture
      \arg        CMP_OUTPUT_TIMER0_OCPRECLR: CMP output TIMER0 OCPRE_CLR input
      \arg        CMP_OUTPUT_TIMER1_IC3: CMP output TIMER1_CH3 input capture
      \arg        CMP_OUTPUT_TIMER1_OCPRECLR: CMP output TIMER1 OCPRE_CLR input
      \arg        CMP_OUTPUT_TIMER2_IC0: CMP output TIMER2_CH0 input capture
      \arg        CMP_OUTPUT_TIMER2_OCPRECLR: CMP output TIMER2 OCPRE_CLR input
    \param[in]  output_polarity
      \arg        CMP_OUTPUT_POLARITY_INVERTED: output is inverted
      \arg        CMP_OUTPUT_POLARITY_NONINVERTED: output is not inverted
    \param[out] none
    \retval     none
*/
void cmp_output_init(cmp_enum cmp_periph, uint32_t output_selection, uint32_t output_polarity)
{
    uint32_t temp = 0U;
    if(CMP0 == cmp_periph){
        /* initialize comparator 0 output */
        temp = CMP_CS;
        temp &= ~(uint32_t)CMP_CS_CMP0OSEL;
        temp |= (uint32_t)(output_selection & CMP_CS_CMP0OSEL);
        /* output polarity */
        if(CMP_OUTPUT_POLARITY_INVERTED == output_polarity){
            temp |= (uint32_t)CMP_CS_CMP0PL;
        }else{
            temp &= ~(uint32_t)CMP_CS_CMP0PL;
        }
        CMP_CS = temp;
    }else{
        /* illegal parameters */
    }
}

/*!
    \brief      enable CMP (API_ID(0x0005U))
    \param[in]  cmp_periph
      \arg        CMP0: comparator 0
    \param[out] none
    \retval     none
*/
void cmp_enable(cmp_enum cmp_periph)
{
    if(CMP0 == cmp_periph){
        CMP_CS |= (uint32_t)CMP_CS_CMP0EN;
    }else{
        /* illegal parameters */
    }
}

/*!
    \brief      disable CMP (API_ID(0x0006U))
    \param[in]  cmp_periph
      \arg        CMP0: comparator 0
    \param[out] none
    \retval     none
*/
void cmp_disable(cmp_enum cmp_periph)
{
    if(CMP0 == cmp_periph){
        CMP_CS &= ~(uint32_t)CMP_CS_CMP0EN;
    }else{
        /* illegal parameters */
    }
}

/*!
    \brief      lock the CMP (API_ID(0x0007U))
    \param[in]  cmp_periph
      \arg        CMP0: comparator 0
    \param[out] none
    \retval     none
*/
void cmp_lock_enable(cmp_enum cmp_periph)
{
    if(CMP0 == cmp_periph){
        /* lock CMP0 */
        CMP_CS |= (uint32_t)CMP_CS_CMP0LK;
    }else{
        /* illegal parameters */
    }
}

/*!
    \brief      get output level (API_ID(0x0008U))
    \param[in]  cmp_periph
      \arg        CMP0: comparator 0
    \param[out] none
    \retval     the output level
*/
uint32_t cmp_output_level_get(cmp_enum cmp_periph)
{
    uint32_t reval = CMP_OUTPUTLEVEL_LOW;
    /* get output level of CMP0 */
    if((uint32_t)RESET != (CMP_CS & CMP_CS_CMP0O)) {
        reval = CMP_OUTPUTLEVEL_HIGH;
    }else{
        reval = CMP_OUTPUTLEVEL_LOW;
    }
    return reval;
}
