/*!
    \file    gd32e25x_cla.c
    \brief   CLA driver

    \version 2025-07-25, V0.2.0, firmware for GD32E25x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32e25x_cla.h"

/*!
    \brief      reset CLA (API_ID(0x0001U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void cla_deinit(void)
{
    rcu_periph_reset_enable(RCU_CLARST);
    rcu_periph_reset_disable(RCU_CLARST);
}

/*!
    \brief      enable CLA (API_ID(0x0002U))
    \param[in]  cla_periph: CLAx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void cla_enable(cla_enum cla_periph)
{
    switch(cla_periph){
        case CLA0:
             CLA_GCTL |= CLA_GCTL_CLA0EN;
             break;
        case CLA1:
             CLA_GCTL |= CLA_GCTL_CLA1EN;
             break;
        case CLA2:
             CLA_GCTL |= CLA_GCTL_CLA2EN;
             break;
        case CLA3:
             CLA_GCTL |= CLA_GCTL_CLA3EN;
             break;
        default:
             break;
    }
}

/*!
    \brief      disable CLA (API_ID(0x0003U))
    \param[in]  cla_periph: CLAx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void cla_disable(cla_enum cla_periph)
{
    switch(cla_periph){
        case CLA0:
             CLA_GCTL &= (~CLA_GCTL_CLA0EN);
             break;
        case CLA1:
             CLA_GCTL &= (~CLA_GCTL_CLA1EN);
             break;
        case CLA2:
             CLA_GCTL &= (~CLA_GCTL_CLA2EN);
             break;
        case CLA3:
             CLA_GCTL &= (~CLA_GCTL_CLA3EN);
             break;
        default:
             break;
    }
}

/*!
    \brief      get CLA output state (API_ID(0x0004U))
    \param[in]  cla_periph: CLAx(x=0,1,2,3)
    \param[out] none
    \retval     cla_outputstatus_enum: CLA_OUTPUT_HIGH or CLA_OUTPUT_LOW
*/
cla_outputstatus_enum cla_output_state_get(cla_enum cla_periph)
{
    uint32_t status;
    cla_outputstatus_enum output_state;

    switch(cla_periph){
        case CLA0:
             status = (CLA_STAT & CLA_STAT_CLA0OUT);
             break;
        case CLA1:
             status = ((CLA_STAT & CLA_STAT_CLA1OUT)>>1U);
             break;
        case CLA2:
             status = ((CLA_STAT & CLA_STAT_CLA2OUT)>>2U);
             break;
        case CLA3:
             status = ((CLA_STAT & CLA_STAT_CLA3OUT)>>3U);
             break;
        default:
             break;
    }
    if(SET == status){
        output_state = CLA_OUTPUT_HIGH;
    }else{
        output_state = CLA_OUTPUT_LOW;
    }

    return output_state;
}

/*!
    \brief      configure multiplexer input (API_ID(0x0005U))
    \param[in]  cla_periph: CLAx(x=0,1,2,3)
    \param[in]  sigs: signal selector
                only one parameter can be selected which is shown as below:
      \arg        SIGS0: multiplexer 0
      \arg        SIGS1: multiplexer 1
    \param[in]  input: input of signal selector

     SIGS0 input selection:
     only one parameter can be selected which is shown as below:
     --------------------------------------------------------------------------------------------------------------------------
     |SIGS0[3:0]|      CLA0SIGS0             |      CLA1SIGS0             |      CLA2SIGS0             |     CLA3SIGS0            |
     --------------------------------------------------------------------------------------------------------------------------
     |   0000   | CLA0SIGS0_CLA0_ASYNC_OUT   | CLA0SIGS0_CLA3_ASYNC_OUT   | CLA0SIGS0_CLA3_ASYNC_OUT   | CLA0SIGS0_CLA3_ASYNC_OUT |
     --------------------------------------------------------------------------------------------------------
     |   0001   | CLA1SIGS0_CLA1_ASYNC_OUT   | CLA1SIGS0_CLA1_ASYNC_OUT   | CLA1SIGS0_CLA1_ASYNC_OUT   | CLA1SIGS0_CLA1_ASYNC_OUT |
     --------------------------------------------------------------------------------------------------------
     |   0010   | CLA2SIGS0_CLA2_ASYNC_OUT   | CLA2SIGS0_CLA2_ASYNC_OUT   | CLA2SIGS0_CLA2_ASYNC_OUT   | CLA2SIGS0_CLA2_ASYNC_OUT |
     --------------------------------------------------------------------------------------------------------
     |   0011   | CLA3SIGS0_CLA3_ASYNC_OUT   | CLA3SIGS0_CLA3_ASYNC_OUT   | CLA3SIGS0_CLA3_ASYNC_OUT   | CLA3SIGS0_CLA3_ASYNC_OUT |
     --------------------------------------------------------------------------------------------------------------------------
     |   0100   | CLA0SIGS0_TIMER1_TRGO      | CLA1SIGS0_TIMER1_TRGO      | CLA2SIGS0_TIMER1_TRGO      | CLA3SIGS0_TIMER1_TRGO    |
     --------------------------------------------------------------------------------------------------------------------------
     |   0101   | CLA0SIGS0_TIMER0_CH0       | CLA1SIGS0_TIMER0_CH0       | CLA2SIGS0_TIMER0_CH1       | CLA3SIGS0_TIMER0_CH2     |
     --------------------------------------------------------------------------------------------------------------------------
     |   0110   | CLA0SIGS0_TIMER0_CH1       | CLA1SIGS0_TIMER0_CH3       | CLA2SIGS0_TIMER0_CH3       | CLA3SIGS0_TIMER1_CH0     |
     --------------------------------------------------------------------------------------------------------------------------
     |   0111   | CLA0SIGS0_TIMER0_CH2       | CLA1SIGS0_TIMER1_CH0       | CLA2SIGS0_TIMER1_CH1       | CLA3SIGS0_TIMER1_CH1     |
     --------------------------------------------------------------------------------------------------------------------------
     |   1000   | CLA0SIGS0_CLAIN0           | CLA1SIGS0_CLAIN4           | CLA2SIGS0_CLAIN0           | CLA3SIGS0_CLAIN2         |
     --------------------------------------------------------------------------------------------------------------------------
     |   1001   | CLA0SIGS0_CLAIN2           | CLA1SIGS0_CLAIN5           | CLA2SIGS0_CLAIN1           | CLA3SIGS0_CLAIN3         |
     --------------------------------------------------------------------------------------------------------------------------
     |   1010   | CLA0SIGS0_CLAIN4           | CLA1SIGS0_CLAIN8           | CLA2SIGS0_CLAIN8           | CLA3SIGS0_CLAIN6         |
     --------------------------------------------------------------------------------------------------------------------------
     |   1011   | CLA0SIGS0_CLAIN6           | CLA1SIGS0_CLAIN10          | CLA2SIGS0_CLAIN9           | CLA3SIGS0_CLAIN7         |
     --------------------------------------------------------------------------------------------------------------------------
     |   1100   | CLA0SIGS0_CLAIN8           | CLA1SIGS0_CLAIN12          | CLA2SIGS0_CLAIN14          | CLA3SIGS0_CLAIN10        |
     --------------------------------------------------------------------------------------------------------------------------
     |   1101   | CLA0SIGS0_CLAIN10          | CLA1SIGS0_CLAIN13          | CLA2SIGS0_CLAIN15          | CLA3SIGS0_CLAIN11        |
     --------------------------------------------------------------------------------------------------------------------------
     |   1110   | CLA0SIGS0_CLAIN12          | CLA1SIGS0_CLAIN16          | CLA2SIGS0_CLAIN16          | CLA3SIGS0_CLAIN18        |
     --------------------------------------------------------------------------------------------------------------------------
     |   1111   | CLA0SIGS0_CLAIN14          | CLA1SIGS0_CLAIN18          | CLA2SIGS0_CLAIN17          | CLA3SIGS0_CLAIN19        |
     --------------------------------------------------------------------------------------------------------------------------
     |  SWBEN0  | CLAxSIGS0_SOFTWARE
     --------------------------------------------------------------------------------------------------------------------------
     SIGS1 input selection:
     only one parameter can be selected which is shown as below:
     --------------------------------------------------------------------------------------------------------------------------
     | SIGS1[3:0]|      CLA0SIGS1             |      CLA1SIGS1             |       CLA2SIGS1            |     CLA3SIGS1            |
     --------------------------------------------------------------------------------------------------------------------------
     |   0000   | CLA0SIGS1_CLA0_ASYNC_OUT   | CLA0SIGS1_CLA0_ASYNC_OUT   | CLA0SIGS1_CLA0_ASYNC_OUT   | CLA0SIGS1_CLA0_ASYNC_OUT |
     --------------------------------------------------------------------------------------------------------------------------
     |   0001   | CLA1SIGS1_CLA1_ASYNC_OUT   | CLA1SIGS1_CLA1_ASYNC_OUT   | CLA1SIGS1_CLA1_ASYNC_OUT   | CLA1SIGS1_CLA1_ASYNC_OUT |
     --------------------------------------------------------------------------------------------------------------------------
     |   0010   | CLA2SIGS1_CLA2_ASYNC_OUT   | CLA2SIGS1_CLA2_ASYNC_OUT   | CLA2SIGS1_CLA2_ASYNC_OUT   | CLA2SIGS1_CLA2_ASYNC_OUT |
     --------------------------------------------------------------------------------------------------------------------------
     |   0011   | CLA3SIGS1_CLA3_ASYNC_OUT   | CLA3SIGS1_CLA3_ASYNC_OUT   | CLA3SIGS1_CLA3_ASYNC_OUT   | CLA3SIGS1_CLA3_ASYNC_OUT |
     --------------------------------------------------------------------------------------------------------------------------
     |   0100   | CLA0SIGS1_ADC_CONV         | CLA1SIGS1_ADC_CONV         | CLA2SIGS1_ADC_CONV         | CLA3SIGS1_ADC_CONV       |
     --------------------------------------------------------------------------------------------------------------------------
     |   0101   | CLA0SIGS1_TIMER0_CH3       | CLA1SIGS1_TIMER0_CH1       | CLA2SIGS1_TIMER0_CH0       | CLA3SIGS1_TIMER0_CH0     |
     --------------------------------------------------------------------------------------------------------------------------
     |   0110   | CLA0SIGS1_TIMER1_CH0       | CLA1SIGS1_TIMER0_CH2       | CLA2SIGS1_TIMER0_CH2       | CLA3SIGS1_TIMER0_CH1     |
     --------------------------------------------------------------------------------------------------------------------------
     |   0111   | CLA0SIGS1_TIMER1_CH1       | CLA1SIGS1_TIMER1_CH1       | CLA2SIGS1_TIMER1_CH0       | CLA3SIGS1_TIMER0_CH3     |
     --------------------------------------------------------------------------------------------------------------------------
     |   1000   | CLA0SIGS1_CLAIN1           | CLA1SIGS1_CLAIN6           | CLA2SIGS1_CLAIN2           | CLA3SIGS1_CLAIN0         |
     --------------------------------------------------------------------------------------------------------------------------
     |   1001   | CLA0SIGS1_CLAIN3           | CLA1SIGS1_CLAIN7           | CLA2SIGS1_CLAIN3           | CLA3SIGS1_CLAIN1         |
     --------------------------------------------------------------------------------------------------------------------------
     |   1010   | CLA0SIGS1_CLAIN5           | CLA1SIGS1_CLAIN9           | CLA2SIGS1_CLAIN10          | CLA3SIGS1_CLAIN4         |
     --------------------------------------------------------------------------------------------------------------------------
     |   1011   | CLA0SIGS1_CLAIN6           | CLA1SIGS1_CLAIN11          | CLA2SIGS1_CLAIN11          | CLA3SIGS1_CLAIN5         |
     --------------------------------------------------------------------------------------------------------------------------
     |   1100   | CLA0SIGS1_CLAIN9           | CLA1SIGS1_CLAIN14          | CLA2SIGS1_CLAIN12          | CLA3SIGS1_CLAIN8         |
     --------------------------------------------------------------------------------------------------------------------------
     |   1101   | CLA0SIGS1_CLAIN11          | CLA1SIGS1_CLAIN15          | CLA2SIGS1_CLAIN13          | CLA3SIGS1_CLAIN9         |
     --------------------------------------------------------------------------------------------------------------------------
     |   1110   | CLA0SIGS1_CLAIN11          | CLA1SIGS1_CLAIN15          | CLA2SIGS1_CLAIN13          | CLA3SIGS1_CLAIN9         |
     --------------------------------------------------------------------------------------------------------------------------
     |   1111   | CLA0SIGS1_CLAIN13          | CLA1SIGS1_CLAIN17          | CLA2SIGS1_CLAIN18          | CLA3SIGS1_CLAIN16        |
     --------------------------------------------------------------------------------------------------------------------------
     |  SWBEN1  | CLAxSIGS1_SOFTWARE
     --------------------------------------------------------------------------------------------------------------------------
    \param[out] none
    \retval     none
*/
void cla_sigs_input_config(cla_enum cla_periph, cla_sigs_enum sigs, sigs_input_enum input)
{
    if(SIGS0 == sigs){
        if(CLAxSIGS0_SOFTWARE == input){
            CLA_CTL(cla_periph) |= CLA_CTL_SWBEN0;
        }else{
            CLA_CTL(cla_periph) &= (~CLA_CTL_SWBEN0);
            CLA_SIGS(cla_periph) &= (~CLA_SIGS_SIGS0);
            CLA_SIGS(cla_periph) |= (uint32_t)input;
        }
    }else{
        if(CLAxSIGS1_SOFTWARE == input){
            CLA_CTL(cla_periph) |= CLA_CTL_SWBEN1;
        }else{
            CLA_CTL(cla_periph) &= (~CLA_CTL_SWBEN1);
            CLA_SIGS(cla_periph) &= (~CLA_SIGS_SIGS1);
            CLA_SIGS(cla_periph) |= (uint32_t)input;
        }
    }
}

/*!
    \brief      configure CLA LCU control register value (API_ID(0x0006U))
    \param[in]  cla_periph: CLAx(x=0,1,2,3)
    \param[in]  lcuctl_value: the value need to set
    \param[out] none
    \retval     none
*/
void cla_lcu_control_config(cla_enum cla_periph, uint8_t lcuctl_value)
{
    CLA_LCUCTL(cla_periph) &= (~CLA_LCU);
    CLA_LCUCTL(cla_periph) |= LCUCTL(lcuctl_value);
}

/*!
    \brief      configure multiplexer software output (API_ID(0x0007U))
    \param[in]  cla_periph: CLAx(x=0,1,2,3)
    \param[in]  sigs: signal selector
                only one parameter can be selected which is shown as below:
                SIGS0: multiplexer 0
                SIGS1: multiplexer 1
    \param[in]  output: multiplexer output value
     only one parameter can be selected which is shown as below:
                SIGS_OUTPUT_SWB: multiplexer output 1
                SIGS_OUTPUT_SELECT_INPUT: multiplexer output 0
    \param[out] none
    \retval     none
*/
void cla_sigs_software_output_set(cla_enum cla_periph, cla_sigs_enum sigs, cla_sigs_output_enum sigs_output)
{
    if(SIGS0 == sigs){
        if(SIGS_OUTPUT_SWB == sigs_output){
            CLA_CTL(cla_periph) &= (~CLA_CTL_SWB0);
        }else{
            CLA_CTL(cla_periph) |= CLA_CTL_SWB0;
        }
    }else{
        if(SIGS_OUTPUT_SWB == sigs_output){
            CLA_CTL(cla_periph) &= (~CLA_CTL_SWB1);
        }else{
            CLA_CTL(cla_periph) |= CLA_CTL_SWB1;
        }
    }
}

/*!
    \brief      configure CLA output (API_ID(0x0008U))
    \param[in]  cla_periph: CLAx(x=0,1,2,3)
    \param[in]  output: output of CLA
                only one parameter can be selected which is shown as below:
      \arg        FLIP_FLOP_OUTPUT: flip-flop output is selected as CLA output
      \arg        LCU_RESULT: LCU result is selected as CLA output
    \param[out] none
    \retval     none
*/
void cla_output_config(cla_enum cla_periph, uint32_t output)
{
    CLA_CTL(cla_periph) &= (~CLA_CTL_OSEL);
    CLA_CTL(cla_periph) |= (output & CLA_CTL_OSEL);
}

/*!
    \brief      enable CLA output (API_ID(0x0009U))
    \param[in]  cla_periph: CLAx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void cla_output_enable(cla_enum cla_periph)
{
    CLA_CTL(cla_periph) |= CLA_CTL_OEN;
}

/*!
    \brief      disable CLA output (API_ID(0x000AU))
    \param[in]  cla_periph: CLAx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void cla_output_disable(cla_enum cla_periph)
{
    CLA_CTL(cla_periph) &= (~CLA_CTL_OEN);
}

/*!
    \brief      reset the flip-flop output asynchronously (API_ID(0x000BU))
    \param[in]  cla_periph: CLAx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void cla_flip_flop_output_reset(cla_enum cla_periph)
{
    CLA_CTL(cla_periph) |= CLA_CTL_FFRST;
}

/*!
    \brief      configure clock polarity of flip-flop (API_ID(0x000CU))
    \param[in]  cla_periph: CLAx(x=0,1,2,3)
    \param[in]  polarity: clock polarity of flip-flop
                only one parameter can be selected which is shown as below:
      \arg        CLA_CLOCKPOLARITY_POSEDGE: clock posedge is valid
      \arg        CLA_CLOCKPOLARITY_NEGEDGE: clock negedge is valid
    \param[out] none
    \retval     none
*/
void cla_flip_flop_clockpolarity_config(cla_enum cla_periph, uint32_t polarity)
{
    CLA_CTL(cla_periph) &= (~CLA_CTL_CPOL);
    CLA_CTL(cla_periph) |= (polarity & CLA_CTL_CPOL);
}

/*!
    \brief      configure clock source of flip-flop (API_ID(0x000DU))
    \param[in]  cla_periph: CLAx(x=0,1,2,3)
    \param[in]  clock_source: clock source of flip-flop
                only one parameter can be selected which is shown as below:
      \arg        PRE_CLA_LCU_RESULT: the result of the previous CLA units
      \arg        SIGS0_OUTPUT: the signal selector output of SIGS0
      \arg        HCLK: HCLK
      \arg        TIMER_TRGO: TIMER_TRGO
    \param[out] none
    \retval     none
*/
void cla_flip_flop_clocksource_config(cla_enum cla_periph, uint32_t clock_source)
{
    CLA_CTL(cla_periph) &= (~CLA_CTL_CSEL);
    CLA_CTL(cla_periph) |= (clock_source & CLA_CTL_CSEL);
}

/*!
    \brief      check CLA flag is set or not (API_ID(0x000EU))
    \param[in]  flag: CLA flags,refer to cla_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        CLA_FLAG_CLA0NF: CLA0 unit negedge flag
      \arg        CLA_FLAG_CLA0PF: CLA0 unit posedge flag
      \arg        CLA_FLAG_CLA1NF: CLA1 unit negedge flag
      \arg        CLA_FLAG_CLA1PF: CLA1 unit posedge flag
      \arg        CLA_FLAG_CLA2NF: CLA2 unit negedge flag
      \arg        CLA_FLAG_CLA2PF: CLA2 unit posedge flag
      \arg        CLA_FLAG_CLA3NF: CLA3 unit negedge flag
      \arg        CLA_FLAG_CLA3PF: CLA3 unit posedge flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus cla_flag_get(cla_flag_enum flag)
{
    uint32_t status;
    FlagStatus returnval;

    switch(flag){
        case CLA_FLAG_CLA0NF:
             status = (CLA_INTF & CLA_INTF_CLA0NF);
             break;
        case CLA_FLAG_CLA0PF:
             status = (CLA_INTF & CLA_INTF_CLA0PF)>>1;
             break;
        case CLA_FLAG_CLA1NF:
             status = (CLA_INTF & CLA_INTF_CLA1NF)>>2;
             break;
        case CLA_FLAG_CLA1PF:
             status = (CLA_INTF & CLA_INTF_CLA1PF)>>3;
             break;
        case CLA_FLAG_CLA2NF:
             status = (CLA_INTF & CLA_INTF_CLA2NF)>>4;
             break;
        case CLA_FLAG_CLA2PF:
             status = (CLA_INTF & CLA_INTF_CLA2PF)>>5;
             break;
        case CLA_FLAG_CLA3NF:
             status = (CLA_INTF & CLA_INTF_CLA3NF)>>6;
             break;
        case CLA_FLAG_CLA3PF:
             status = (CLA_INTF & CLA_INTF_CLA3PF)>>7;
             break;
        default:
             break;
    }
    if(SET == status){
        returnval = SET;
    }else{
        returnval = RESET;
    }

    return returnval;
}

/*!
    \brief      clear CLA flag (API_ID(0x000FU))
    \param[in]  flag: CLA flags,refer to cla_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        CLA_FLAG_CLA0NF: CLA0 unit negedge flag
      \arg        CLA_FLAG_CLA0PF: CLA0 unit posedge flag
      \arg        CLA_FLAG_CLA1NF: CLA1 unit negedge flag
      \arg        CLA_FLAG_CLA1PF: CLA1 unit posedge flag
      \arg        CLA_FLAG_CLA2NF: CLA2 unit negedge flag
      \arg        CLA_FLAG_CLA2PF: CLA2 unit posedge flag
      \arg        CLA_FLAG_CLA3NF: CLA3 unit negedge flag
      \arg        CLA_FLAG_CLA3PF: CLA3 unit posedge flag
    \param[out] none
    \retval     none
*/
void cla_flag_clear(cla_flag_enum flag)
{
    switch(flag){
        case CLA_FLAG_CLA0NF:
             CLA_INTF &= (~CLA_INTF_CLA0NF);
             break;
        case CLA_FLAG_CLA0PF:
             CLA_INTF &= (~CLA_INTF_CLA0PF);
             break;
        case CLA_FLAG_CLA1NF:
             CLA_INTF &= (~CLA_INTF_CLA1NF);
             break;
        case CLA_FLAG_CLA1PF:
             CLA_INTF &= (~CLA_INTF_CLA1PF);
             break;
        case CLA_FLAG_CLA2NF:
             CLA_INTF &= (~CLA_INTF_CLA2NF);
             break;
        case CLA_FLAG_CLA2PF:
             CLA_INTF &= (~CLA_INTF_CLA2PF);
             break;
        case CLA_FLAG_CLA3NF:
             CLA_INTF &= (~CLA_INTF_CLA3NF);
             break;
        case CLA_FLAG_CLA3PF:
             CLA_INTF &= (~CLA_INTF_CLA3PF);
             break;
        default:
            break;
    }
}

/*!
    \brief      enable CLA negedge interrupt (API_ID(0x0010U))
    \param[in]  clanie: CLA negedge interrupt enable
                one or more parameter can be selected which is shown as below:
      \arg        CLA0_INT_NEGEDGE: CLA0 negedge interrupt
      \arg        CLA1_INT_NEGEDGE: CLA1 negedge interrupt
      \arg        CLA2_INT_NEGEDGE: CLA2 negedge interrupt
      \arg        CLA3_INT_NEGEDGE: CLA3 negedge interrupt
    \param[out] none
    \retval     none
*/
void cla_negedge_interrupt_enable(uint32_t interrupt)
{
    CLA_INTE |= interrupt;
}

/*!
    \brief      disable CLA negedge interrupt (API_ID(0x0011U))
    \param[in]  clanidis: CLA negedge interrupt
                one or more parameter can be selected which is shown as below:
      \arg        CLA0_INT_NEGEDGE: CLA0 negedge interrupt
      \arg        CLA1_INT_NEGEDGE: CLA1 negedge interrupt
      \arg        CLA2_INT_NEGEDGE: CLA2 negedge interrupt
      \arg        CLA3_INT_NEGEDGE: CLA3 negedge interrupt
    \param[out] none
    \retval     none
*/
void cla_negedge_interrupt_disable(uint32_t interrupt)
{
    CLA_INTE &= (~interrupt);
}

/*!
    \brief      enable CLA posedge interrupt (API_ID(0x0012U))
    \param[in]  clapie: CLA posedge interrupt enable
                one or more parameter can be selected which is shown as below:
      \arg        CLA0_INT_POSEDGE: CLA0 posedge interrupt
      \arg        CLA1_INT_POSEDGE: CLA1 posedge interrupt
      \arg        CLA2_INT_POSEDGE: CLA2 posedge interrupt
      \arg        CLA3_INT_POSEDGE: CLA3 posedge interrupt
    \param[out] none
    \retval     none
*/
void cla_posedge_interrupt_enable(uint32_t interrupt)
{
    CLA_INTE |= interrupt;
}

/*!
    \brief      disable CLA posedge interrupt (API_ID(0x0013U))
    \param[in]  clanidis: CLA posedge interrupt disable
                one or more parameter can be selected which is shown as below:
      \arg        CLA0_INT_POSEDGE: CLA0 posedge interrupt
      \arg        CLA1_INT_POSEDGE: CLA1 posedge interrupt
      \arg        CLA2_INT_POSEDGE: CLA2 posedge interrupt
      \arg        CLA3_INT_POSEDGE: CLA3 posedge interrupt
    \param[out] none
    \retval     none
*/
void cla_posedge_interrupt_disable(uint32_t interrupt)
{
    CLA_INTE &= (~interrupt);
}

/*!
    \brief      check CLA interrupt flag is set or not (API_ID(0x0014U))
    \param[in]  int_flag: CLA interrupt flags,refer to cla_interrupt_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        CLA_INT_FLAG_CLA0NF: CLA0 unit negedge interrupt flag
      \arg        CLA_INT_FLAG_CLA0PF: CLA0 unit posedge interrupt flag
      \arg        CLA_INT_FLAG_CLA1NF: CLA1 unit negedge interrupt flag
      \arg        CLA_INT_FLAG_CLA1PF: CLA1 unit posedge interrupt flag
      \arg        CLA_INT_FLAG_CLA2NF: CLA2 unit negedge interrupt flag
      \arg        CLA_INT_FLAG_CLA2PF: CLA2 unit posedge interrupt flag
      \arg        CLA_INT_FLAG_CLA3NF: CLA3 unit negedge interrupt flag
      \arg        CLA_INT_FLAG_CLA3PF: CLA3 unit posedge interrupt flag
    \param[out] none
    \retval     SET or RESET
*/
FlagStatus cla_interrupt_flag_get(cla_interrupt_flag_enum int_flag)
{
    uint32_t cla_inte;
    uint32_t cla_intf;
    FlagStatus status;

    switch(int_flag){
        case CLA_INT_FLAG_CLA0NF:
             cla_inte = CLA_INTE & CLA_INTE_CLA0NIE;
             cla_intf = CLA_INTF & CLA_INTF_CLA0NF;
             break;
        case CLA_INT_FLAG_CLA0PF:
             cla_inte = CLA_INTE & CLA_INTE_CLA0PIE;
             cla_intf = CLA_INTF & CLA_INTF_CLA0PF;
             break;
        case CLA_INT_FLAG_CLA1NF:
             cla_inte = CLA_INTE & CLA_INTE_CLA1NIE;
             cla_intf = CLA_INTF & CLA_INTF_CLA1NF;
             break;
        case CLA_INT_FLAG_CLA1PF:
             cla_inte = CLA_INTE & CLA_INTE_CLA1PIE;
             cla_intf = CLA_INTF & CLA_INTF_CLA1PF;
             break;
        case CLA_INT_FLAG_CLA2NF:
             cla_inte = CLA_INTE & CLA_INTE_CLA2NIE;
             cla_intf = CLA_INTF & CLA_INTF_CLA2NF;
             break;
        case CLA_INT_FLAG_CLA2PF:
             cla_inte = CLA_INTE & CLA_INTE_CLA2PIE;
             cla_intf = CLA_INTF & CLA_INTF_CLA2PF;
             break;
        case CLA_INT_FLAG_CLA3NF:
             cla_inte = CLA_INTE & CLA_INTE_CLA3NIE;
             cla_intf = CLA_INTF & CLA_INTF_CLA3NF;
             break;
        case CLA_INT_FLAG_CLA3PF:
             cla_inte = CLA_INTE & CLA_INTE_CLA3PIE;
             cla_intf = CLA_INTF & CLA_INTF_CLA3PF;
             break;
        default:
             break;
    }
    if((0U != cla_inte) && (0U != cla_intf)){
        status = SET;
    }else{
        status = RESET;
    }

    return status;
}

/*!
    \brief      clear CLA interrupt flag (API_ID(0x0015U))
    \param[in]  flag: CLA interrupt flags,refer to cla_interrupt_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        CLA_INT_FLAG_CLA0NF: CLA0 unit negedge interrupt flag
      \arg        CLA_INT_FLAG_CLA0PF: CLA0 unit posedge interrupt flag
      \arg        CLA_INT_FLAG_CLA1NF: CLA1 unit negedge interrupt flag
      \arg        CLA_INT_FLAG_CLA1PF: CLA1 unit posedge interrupt flag
      \arg        CLA_INT_FLAG_CLA2NF: CLA2 unit negedge interrupt flag
      \arg        CLA_INT_FLAG_CLA2PF: CLA2 unit posedge interrupt flag
      \arg        CLA_INT_FLAG_CLA3NF: CLA3 unit negedge interrupt flag
      \arg        CLA_INT_FLAG_CLA3PF: CLA3 unit posedge interrupt flag
    \param[out] none
    \retval     none
*/
void cla_interrupt_flag_clear(cla_interrupt_flag_enum int_flag)
{
    switch(int_flag){
        case CLA_INT_FLAG_CLA0NF:
             CLA_INTF &= (~CLA_INTF_CLA0NF);
             break;
        case CLA_INT_FLAG_CLA0PF:
             CLA_INTF &= (~CLA_INTF_CLA0PF);
             break;
        case CLA_INT_FLAG_CLA1NF:
             CLA_INTF &= (~CLA_INTF_CLA1NF);
             break;
        case CLA_INT_FLAG_CLA1PF:
             CLA_INTF &= (~CLA_INTF_CLA1PF);
             break;
        case CLA_INT_FLAG_CLA2NF:
             CLA_INTF &= (~CLA_INTF_CLA2NF);
             break;
        case CLA_INT_FLAG_CLA2PF:
             CLA_INTF &= (~CLA_INTF_CLA2PF);
             break;
        case CLA_INT_FLAG_CLA3NF:
             CLA_INTF &= (~CLA_INTF_CLA3NF);
             break;
        case CLA_INT_FLAG_CLA3PF:
             CLA_INTF &= (~CLA_INTF_CLA3PF);
             break;
        default:
             break;
    }
}
