/*!
    \file    gd32e25x_bkp.c
    \brief   bkp driver

    \version 2025-07-25, V0.2.0, firmware for GD32E25x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32e25x_bkp.h"

/*!
    \brief      reset BKP registers (API_ID (0x0001U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void bkp_deinit(void)
{
    /* enable BKP reset */
    rcu_bkp_reset_enable();
    /* disable BKP reset */
    rcu_bkp_reset_disable();
}

/*!
    \brief      enable tamper detection (API_ID (0x0002U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void bkp_tamper_detection_enable(void)
{
    /* enable tamper detection */
    BKP_TAMP |= BKP_TAMP_TP1EN;
}

/*!
    \brief      disable tamper detection (API_ID (0x0003U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void bkp_tamper_detection_disable(void)
{
    /* disable tamper detection */
    BKP_TAMP &= ~BKP_TAMP_TP1EN;
}

/*!
    \brief      configure tamper detection sampling frequency (API_ID(0x0004U))
    \param[in]  frequency: specify the sampling frequency
    \arg          BKP_FREQ_DIV32768: Sampling once every 32768 clock cycles
    \arg          BKP_FREQ_DIV16384: Sampling once every 16384 clock cycles
    \arg          BKP_FREQ_DIV8192:  Sampling once every 8192 clock cycles
    \arg          BKP_FREQ_DIV4096:  Sampling once every 4096 clock cycles
    \arg          BKP_FREQ_DIV2048:  Sampling once every 2048 clock cycles
    \arg          BKP_FREQ_DIV1024:  Sampling once every 1024 clock cycles
    \arg          BKP_FREQ_DIV512:   Sampling once every 512 clock cycles
    \arg          BKP_FREQ_DIV256:   Sampling once every 256 clock cycles
    \param[out] none
    \retval     none
*/
void bkp_tamper_sampling_frequency_config(uint32_t frequency)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_BKP_SAMPLE_FREQ(frequency)) {
        fw_debug_report_err(BKP_MODULE_ID, API_ID(0x0004U), ERR_PARAM_OUT_OF_RANGE);
        return;
    }
#endif /* FW_DEBUG_ERR_REPORT */
    {
        uint32_t ctl = 0U;

        ctl = BKP_TAMP;
        /* clear previous frequency settings */
        ctl &= ~(uint32_t)BKP_TAMP_FREQ;
        /* set new frequency */
        ctl |= (frequency & SAMP_FREQ_MASK);
        BKP_TAMP = ctl;
    }
}

/*!
    \brief      configure pre-charge duration time (API_ID(0x0005U))
    \param[in]  time: duration time
    \arg          BKP_PRCH_1C: 1 BKP clock precharge time before each sampling
    \arg          BKP_PRCH_2C: 2 BKP clock precharge time before each sampling
    \arg          BKP_PRCH_4C: 4 BKP clock precharge time before each sampling
    \arg          BKP_PRCH_8C: 8 BKP clock precharge time before each sampling
    \param[out] none
    \retval     none
*/
void bkp_sampling_duration_time_config(uint32_t time)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_BKP_DURATION_TIME(time)) {
        fw_debug_report_err(BKP_MODULE_ID, API_ID(0x0005U), ERR_PARAM_OUT_OF_RANGE);
        return;
    }
#endif /* FW_DEBUG_ERR_REPORT */
    {
        uint32_t ctl = 0U;

        ctl = BKP_TAMP;
        /* clear previous frequency settings */
        ctl &= ~(uint32_t)BKP_TAMP_PRCH;
        /* set new frequency */
        ctl |= (time & DURATION_TIME_MASK) ;
        BKP_TAMP = ctl;
    }
}

/*!
    \brief      enable pull up pre-charge duration(API_ID(0x0006U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void bkp_pull_up_enable(void)
{
    /* enable pull up */
    BKP_TAMP &= ~BKP_TAMP_DISPU;
}

/*!
    \brief      disable pull up pre-charge duration(API_ID(0x0007U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void bkp_pull_up_disable(void)
{
    /* disable pull up */
    BKP_TAMP |= BKP_TAMP_DISPU;
}

/*!
    \brief      configure filter count (API_ID(0x0008U))
    \param[in]  count: number of consecutive valid level
    \arg          BKP_FLT_EDGE: detecting tamper event using edge mode. precharge duration is disabled automatically
    \arg          BKP_FLT_2S: detecting tamper event using level mode.2 consecutive valid level samples will make an effective tamper event
    \arg          BKP_FLT_4S: detecting tamper event using level mode.4 consecutive valid level samples will make an effective tamper event
    \arg          BKP_FLT_8S: detecting tamper event using level mode.8 consecutive valid level samples will make an effective tamper event
    \param[out] none
    \retval     none
*/
void bkp_filter_count_config(uint32_t count)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_BKP_FILTER_COUNT(count)) {
        fw_debug_report_err(BKP_MODULE_ID, API_ID(0x0008U), ERR_PARAM_OUT_OF_RANGE);
        return;
    }
#endif /* FW_DEBUG_ERR_REPORT */
    {
        uint32_t ctl = 0U;

        ctl =  BKP_TAMP;
        /* clear previous frequency settings */
        ctl &= ~(uint32_t)BKP_TAMP_FLT;
        /* set new value */
        ctl |= (count & FLT_COUNT_MASK);
        BKP_TAMP = ctl;
    }
}

/*!
    \brief      configure event trigger edge (API_ID(0x0009U))
    \param[in]  edge: trigger edge
    \arg          BKP_TRIG_RISING_LOW: tamper 1 event trigger edge rising or low level
    \arg          BKP_TRIG_FALLING_HIGH: tamper 1 event trigger edge falling or high level
    \param[out] none
    \retval     none
*/
void bkp_tamper_trigger_edge_config(uint32_t edge)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_BKP_TRIG_EDGE(edge)) {
        fw_debug_report_err(BKP_MODULE_ID, API_ID(0x0009U), ERR_PARAM_OUT_OF_RANGE);
        return;
    }
#endif /* FW_DEBUG_ERR_REPORT */
    {
        uint32_t ctl = 0U;

        ctl =  BKP_TAMP;
        /* clear previous frequency settings */
        ctl &= ~BKP_TAMP_TP1EG;
        /* set new frequency */
        ctl |= (edge & TRIG_EDGE_MASK);
        BKP_TAMP = ctl;
    }
}

/*!
    \brief      write BKP data register (API_ID(0x000AU))
    \param[in]  register_number: refer to bkp_data_register_enum, only one parameter can be selected
      \arg        BKP_DATA_x(x = 0..4): bkp data register number x
    \param[in]  data: the data to be write in BKP data register
    \param[out] none
    \retval     none
*/
void bkp_write_data(bkp_data_register_enum register_number, uint32_t data)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_BKP_REGISTER(register_number)) {
        fw_debug_report_err(BKP_MODULE_ID, API_ID(0x000AU), ERR_PARAM_OUT_OF_RANGE);
        return;
    }
#endif /* FW_DEBUG_ERR_REPORT */
    {
        BKP_DATA0_4(register_number) = data;
    }
}

/*!
    \brief      read BKP data register (API_ID(0x000BU))
    \param[in]  register_number: refer to bkp_data_register_enum, only one parameter can be selected
      \arg        BKP_DATA_x(x = 0..4): bkp data register number x
    \param[out] none
    \retval     data of BKP data register
*/
uint32_t bkp_read_data(bkp_data_register_enum register_number)
{
    uint32_t data = 0U;
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_BKP_REGISTER(register_number)) {
        fw_debug_report_err(BKP_MODULE_ID, API_ID(0x000BU), ERR_PARAM_OUT_OF_RANGE);
        return 0U;
    }
#endif /* FW_DEBUG_ERR_REPORT */
    {
        data = BKP_DATA0_4(register_number);
        return data;
    }
}

/*!
    \brief      enable tamper interrupt (API_ID(0x000CU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void bkp_tamper_interrupt_enable(void)
{
    /* enable tamper interrupt */
    BKP_TAMP |= BKP_TAMP_TPIE;
}

/*!
    \brief      disable tamper interrupt (API_ID(0x000DU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void bkp_tamper_interrupt_disable(void)
{
    /* disable tamper interrupt */
    BKP_TAMP &= ~BKP_TAMP_TPIE;
}

/*!
    \brief      get BKP flag status (API_ID(0x000EU))
    \param[in]  flag: specify which flag to get
      \arg        BKP_FLAG_TAMP1: Tamper 1 event flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus bkp_flag_get(uint32_t flag)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_BKP_FLAG(flag)) {
        fw_debug_report_err(BKP_MODULE_ID, API_ID(0x000EU), ERR_PARAM_OUT_OF_RANGE);
        return RESET;
    }
#endif /* FW_DEBUG_ERR_REPORT */
    {
        FlagStatus reval = RESET;
        if(RESET != (BKP_STAT & flag)) {
            reval = SET;
        } else {
            /* do nothing */
        }
        return reval;
    }
}

/*!
    \brief      clear BKP flag(API_ID (0x000FU))
    \param[in]  flag: specify which flag to clear
      \arg        BKP_FLAG_TAMP1: Tamper 1 event flag
    \param[out] none
    \retval     none
*/
void bkp_flag_clear(uint32_t flag)
{
#ifdef FW_DEBUG_ERR_REPORT
    if(NOT_BKP_FLAG(flag)) {
        fw_debug_report_err(BKP_MODULE_ID, API_ID(0x000FU), ERR_PARAM_OUT_OF_RANGE);
        return;
    }
#endif /* FW_DEBUG_ERR_REPORT */
    {
        /* clear the specified flag */
        BKP_STAT &= (uint32_t)(~flag);
    }
}
