/*!
    \file    gd32e25x_adc.c
    \brief   ADC driver

    \version 2025-07-25, V0.2.0, firmware for GD32E25x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32e25x_adc.h"

#define ADC_FLAG_MASK                              ((uint32_t)0xC000001FU)
#define ADC_INT_FLAG_MASK                          ((uint32_t)0xC0000007U)
#define ADC_LATCH_DATA_MASK                        ((uint32_t)0xC0000003U)

/*!
    \brief      reset ADC (API_ID(0x0001U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void adc_deinit(void)
{
    rcu_periph_reset_enable(RCU_ADCRST);
    rcu_periph_reset_disable(RCU_ADCRST);
}

/*!
    \brief      enable ADC interface (API_ID(0x0002U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void adc_enable(void)
{
    if(RESET == (ADC_CTL1 & ADC_CTL1_ADCON)){
        ADC_CTL1 |= (uint32_t)ADC_CTL1_ADCON;
    }
}

/*!
    \brief      disable ADC interface (API_ID(0x0003U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void adc_disable(void)
{
    ADC_CTL1 &= ~((uint32_t)ADC_CTL1_ADCON);
}

/*!
    \brief      ADC calibration and reset calibration (API_ID(0x0004U))
    \param[in]  none
    \param[out] none
    \retval     none
    \note       This function contains scenarios leading to an infinite loop.
                Modify according to the user's actual usage scenarios.
*/
void adc_calibration_enable(void)
{
    /* reset the selected ADC calibration register */
    ADC_CTL1 |= (uint32_t) ADC_CTL1_RSTCLB;
    /* check the RSTCLB bit state */
    while((ADC_CTL1 & ADC_CTL1_RSTCLB)){
    }

    /* enable ADC calibration process */
    ADC_CTL1 |= ADC_CTL1_CLB;
    /* check the CLB bit state */
    while((ADC_CTL1 & ADC_CTL1_CLB)){
    }
}

/*!
    \brief      enable DMA request (API_ID(0x0005U))
    \param[in]  adc_sequence: select the sequence
                only one parameter can be selected which is shown as below:
      \arg        ADC_ROUTINE_CHANNEL: routine sequence
      \arg        ADC_INSERTED_CHANNEL: inserted sequence
    \param[out] none
    \retval     none
*/
void adc_dma_mode_enable(uint8_t adc_sequence)
{
    switch(adc_sequence){
    case ADC_ROUTINE_CHANNEL:
        /* DMA request enable for routine channel */
        ADC_CTL1 |= (uint32_t)(ADC_CTL1_RDMA);
        break;
    case ADC_INSERTED_CHANNEL:
        /* DMA request enable for inserted channel */
        ADC_CTL2 |= (uint32_t)(ADC_CTL2_IDMA);
        break;
    default:
        break;
    }
}

/*!
    \brief      disable DMA request (API_ID(0x0006U))
    \param[in]  adc_sequence: select the sequence
                only one parameter can be selected which is shown as below:
      \arg        ADC_ROUTINE_CHANNEL: routine sequence
      \arg        ADC_INSERTED_CHANNEL: inserted sequence
    \param[out] none
    \retval     none
*/
void adc_dma_mode_disable(uint8_t adc_sequence)
{
    switch(adc_sequence){
    case ADC_ROUTINE_CHANNEL:
        /* DMA request disable for routine channel */
        ADC_CTL1 &= ~((uint32_t)ADC_CTL1_RDMA);
        break;
    case ADC_INSERTED_CHANNEL:
        /* DMA request disable for inserted channel */
        ADC_CTL2 &= ~((uint32_t)ADC_CTL2_IDMA);
        break;
    default:
        break;
    }
}

/*!
    \brief      enable the high precision temperature sensor and vrefint channel (API_ID(0x0007U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void adc_tempsensor_vrefint_enable(void)
{
    /* enable the high precision temperature sensor and vrefint channel  */
    ADC_CTL1 |= ADC_CTL1_TSVREN;
}

/*!
    \brief      disable the high precision temperature sensor and vrefint channel (API_ID(0x0008U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void adc_tempsensor_vrefint_disable(void)
{
    /* disable the high precision temperature sensor and vrefint channel */
    ADC_CTL1 &= ~ADC_CTL1_TSVREN;
}

/*!
    \brief      enable 1/2 analog supply voltage channel (API_ID(0x0009U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void adc_vdda_enable(void)
{
    /* enable 1/2 analog supply voltage channel */
    ADC_CTL1 |= ADC_CTL1_VDDADIV2EN;
}

/*!
    \brief      disable 1/2 analog supply voltage channel (API_ID(0x000AU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void adc_vdda_disable(void)
{
    /* disable 1/2 analog supply voltage channel */
    ADC_CTL1 &= ~ADC_CTL1_VDDADIV2EN;
}

/*!
    \brief      enable high precision temperature sensor channel (API_ID(0x000BU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void adc_hpts_enable(void)
{
    /* enable high precision temperature sensor channel */
    ADC_CTL1 |= ADC_CTL1_HPTSEN;
}

/*!
    \brief      disable high precision temperature sensor channel (API_ID(0x000CU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void adc_hpts_disable(void)
{
    /* disable high precision temperature sensor channel */
    ADC_CTL1 &= ~ADC_CTL1_HPTSEN;
}

/*!
    \brief      high precision temperature sensor enable through the HPTSEN bit (API_ID(0x000DU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void adc_hpts_en_sel_through_hptsen(void)
{
    /* high precision temperature sensor enable through the HPTSEN bit */
    ADC_CTL1 |= ADC_CTL1_HPTSENSEL;
}

/*!
    \brief      high precision temperature sensor enable through the TSVREN bit (API_ID(0x000EU))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void adc_hpts_en_sel_through_tsvren(void)
{
    /* high precision temperature sensor enable through the TSVREN bit */
    ADC_CTL1 &= ~ADC_CTL1_HPTSENSEL;
}

/*!
    \brief      configure ADC discontinuous mode (API_ID(0x000FU))
    \param[in]  adc_sequence: select the sequence
                only one parameter can be selected which is shown as below:
      \arg        ADC_ROUTINE_CHANNEL: routine sequence
      \arg        ADC_INSERTED_CHANNEL: inserted sequence
      \arg        ADC_CHANNEL_DISCON_DISABLE: disable discontinuous mode of routine and inserted channel
    \param[in]  length: number of conversions in discontinuous mode, the number can be 1..8
                        for routine channel, the number has no effect for inserted channel
    \param[out] none
    \retval     none
*/
void adc_discontinuous_mode_config(uint8_t adc_sequence, uint8_t length)
{
    ADC_CTL0 &= ~((uint32_t)(ADC_CTL0_DISRC | ADC_CTL0_DISIC));

    switch(adc_sequence){
    case ADC_ROUTINE_CHANNEL:
        /* configure the number of conversions in discontinuous mode */
        ADC_CTL0 &= ~((uint32_t)ADC_CTL0_DISNUM);
        ADC_CTL0 |= CTL0_DISNUM(((uint32_t)length - 1U));
        ADC_CTL0 |= (uint32_t)ADC_CTL0_DISRC;
        break;
    case ADC_INSERTED_CHANNEL:
        ADC_CTL0 |= (uint32_t)ADC_CTL0_DISIC;
        break;
    case ADC_CHANNEL_DISCON_DISABLE:
    default:
        break;
    }
}

/*!
    \brief      configure ADC special function (API_ID(0x0010U))
    \param[in]  function: the function to configure
                one or more parameters can be selected which is shown as below:
      \arg        ADC_SCAN_MODE: scan mode select
      \arg        ADC_INSERTED_CHANNEL_AUTO: inserted sequence convert automatically
      \arg        ADC_CONTINUOUS_MODE: continuous mode select
    \param[in]  newvalue: ENABLE or DISABLE
    \param[out] none
    \retval     none
*/
void adc_special_function_config(uint32_t function, ControlStatus newvalue)
{
    if(newvalue){
        /* enable ADC scan mode */
        if(RESET != (function & ADC_SCAN_MODE)){
            ADC_CTL0 |= ADC_SCAN_MODE;
        }
        /* enable ADC inserted sequence convert automatically */
        if(RESET != (function & ADC_INSERTED_CHANNEL_AUTO)){
            ADC_CTL0 |= ADC_INSERTED_CHANNEL_AUTO;
        }
        /* enable ADC continuous mode */
        if(RESET != (function & ADC_CONTINUOUS_MODE)){
            ADC_CTL1 |= ADC_CONTINUOUS_MODE;
        }
    }else{
        /* disable ADC scan mode */
        if(RESET != (function & ADC_SCAN_MODE)){
            ADC_CTL0 &= ~ADC_SCAN_MODE;
        }
        /* disable ADC inserted sequence convert automatically */
        if(RESET != (function & ADC_INSERTED_CHANNEL_AUTO)){
            ADC_CTL0 &= ~ADC_INSERTED_CHANNEL_AUTO;
        }
        /* disable ADC continuous mode */
        if(RESET != (function & ADC_CONTINUOUS_MODE)){
            ADC_CTL1 &= ~ADC_CONTINUOUS_MODE;
        }
    }
}

/*!
    \brief      configure ADC data alignment (API_ID(0x0011U))
    \param[in]  data_alignment: data alignment select
                only one parameter can be selected which is shown as below:
      \arg        ADC_DATAALIGN_RIGHT: right alignment
      \arg        ADC_DATAALIGN_LEFT: left alignment
    \param[out] none
    \retval     none
*/
void adc_data_alignment_config(uint32_t data_alignment)
{
    if(ADC_DATAALIGN_RIGHT != data_alignment){
        ADC_CTL1 |= ADC_CTL1_DAL;
    }else{
        ADC_CTL1 &= ~((uint32_t)ADC_CTL1_DAL);
    }
}

/*!
    \brief      configure the length of routine sequence or inserted sequence (API_ID(0x0012U))
    \param[in]  adc_sequence: select the sequence
                only one parameter can be selected which is shown as below:
      \arg        ADC_ROUTINE_CHANNEL: routine sequence
      \arg        ADC_INSERTED_CHANNEL: inserted sequence
    \param[in]  length: the length of the channel
                        routine channel 1-23
                        inserted channel 1-4
    \param[out] none
    \retval     none
*/
void adc_channel_length_config(uint8_t adc_sequence, uint32_t length)
{
    switch(adc_sequence){
    case ADC_ROUTINE_CHANNEL:
        /* configure the length of routine sequence */
        ADC_RSQ0 &= ~((uint32_t)ADC_RSQ0_RL);
        ADC_RSQ0 |= RSQ0_RL((uint32_t)(length-1U));
        break;
    case ADC_INSERTED_CHANNEL:
        /* configure the length of inserted sequence */
        ADC_ISQ &= ~((uint32_t)ADC_ISQ_IL);
        ADC_ISQ |= ISQ_IL((uint32_t)(length-1U));
        break;
    default:
        break;
    }
}

/*!
    \brief      configure ADC routine channel (API_ID(0x0013U))
    \param[in]  rank: the routine sequence rank, this parameter must be between 0 to 22
    \param[in]  channel: the selected ADC channel
                only one parameter can be selected which is shown as below:
      \arg        ADC_CHANNEL_x(x=0..22): ADC Channelx
    \param[in]  sample_time: the sample time value
                only one parameter can be selected which is shown as below:
      \arg        ADC_SAMPLETIME_1POINT5: 1.5 cycles
      \arg        ADC_SAMPLETIME_7POINT5: 7.5 cycles
      \arg        ADC_SAMPLETIME_13POINT5: 13.5 cycles
      \arg        ADC_SAMPLETIME_28POINT5: 28.5 cycles
      \arg        ADC_SAMPLETIME_41POINT5: 41.5 cycles
      \arg        ADC_SAMPLETIME_55POINT5: 55.5 cycles
      \arg        ADC_SAMPLETIME_71POINT5: 71.5 cycles
      \arg        ADC_SAMPLETIME_239POINT5: 239.5 cycles
    \param[out] none
    \retval     none
*/
void adc_routine_channel_config(uint8_t rank, uint8_t channel, uint32_t sample_time)
{
    uint32_t rsq,sampt;

    /* configure ADC routine sequence */
    if(rank < 6U){
        rsq = ADC_RSQ2;
        rsq &=  ~((uint32_t)(ADC_RSQX_RSQN << (5U*rank)));
        rsq |= ((uint32_t)(channel & ADC_RSQX_RSQN) << (5U*rank));
        ADC_RSQ2 = rsq;
    }else if(rank < 12U){
        rsq = ADC_RSQ1;
        rsq &= ~((uint32_t)(ADC_RSQX_RSQN << (5U*(rank-6U))));
        rsq |= ((uint32_t)(channel & ADC_RSQX_RSQN) << (5U*(rank-6U)));
        ADC_RSQ1 = rsq;
    }else if(rank < 17U){
        rsq = ADC_RSQ0;
        if(rank == 16U)
        {
            rsq &= ~((uint32_t)(ADC_RSQX_RSQN << (5U*(rank-11U))));
            rsq |= ((uint32_t)(channel & ADC_RSQX_RSQN) << (5U*(rank-11U)));
        }
        else
        {
            rsq &= ~((uint32_t)(ADC_RSQX_RSQN << (5U*(rank-12U))));
            rsq |= ((uint32_t)(channel & ADC_RSQX_RSQN) << (5U*(rank-12U)));
        }
        ADC_RSQ0 = rsq;
    }else{
        rsq = ADC_RSQ3;
        rsq &= ~((uint32_t)(ADC_RSQX_RSQN << (5U*(rank-17U))));
        rsq |= ((uint32_t)(channel & ADC_RSQX_RSQN) << (5U*(rank-17U)));
        ADC_RSQ3 = rsq;
    }

    /* configure ADC sampling time */
    if(channel < 10U){
        sampt = ADC_SAMPT1;
        sampt &= ~((uint32_t)(ADC_SAMPTX_SPTN << (3U*channel)));
        sampt |= (uint32_t)((sample_time & ADC_SAMPTX_SPTN) << (3U*channel));
        ADC_SAMPT1 = sampt;
    }else if(channel < 19U){
        sampt = ADC_SAMPT0;
        sampt &= ~((uint32_t)(ADC_SAMPTX_SPTN << (3U*(channel-10U))));
        sampt |= (uint32_t)((sample_time & ADC_SAMPTX_SPTN) << (3U*(channel-10U)));
        ADC_SAMPT0 = sampt;
    }else{
        sampt = ADC_SAMPT2;
        sampt &= ~((uint32_t)(ADC_SAMPTX_SPTN << (3U*(channel-19U))));
        sampt |= (uint32_t)((sample_time & ADC_SAMPTX_SPTN) << (3U*(channel-19U)));
        ADC_SAMPT2 = sampt;
    }
}

/*!
    \brief      configure ADC inserted channel (API_ID(0x0014U))
    \param[in]  rank: the inserted sequence rank,this parameter must be between 0 to 3
    \param[in]  channel: the selected ADC channel
                only one parameter can be selected which is shown as below:
      \arg        ADC_CHANNEL_x(x=0..22): ADC Channelx
    \param[in]  sample_time: The sample time value
                only one parameter can be selected which is shown as below:
      \arg        ADC_SAMPLETIME_1POINT5: 1.5 cycles
      \arg        ADC_SAMPLETIME_7POINT5: 7.5 cycles
      \arg        ADC_SAMPLETIME_13POINT5: 13.5 cycles
      \arg        ADC_SAMPLETIME_28POINT5: 28.5 cycles
      \arg        ADC_SAMPLETIME_41POINT5: 41.5 cycles
      \arg        ADC_SAMPLETIME_55POINT5: 55.5 cycles
      \arg        ADC_SAMPLETIME_71POINT5: 71.5 cycles
      \arg        ADC_SAMPLETIME_239POINT5: 239.5 cycles
    \param[out] none
    \retval     none
*/
void adc_inserted_channel_config(uint8_t rank, uint8_t channel, uint32_t sample_time)
{
    uint8_t inserted_length;
    uint32_t isq,sampt;

    inserted_length = (uint8_t)GET_BITS(ADC_ISQ , 20U , 21U);
    /* the channel number is written to these bits to select a channel as the nth conversion in the inserted sequence */
    if(rank < 4U){
        isq = ADC_ISQ;
        isq &= ~((uint32_t)(ADC_ISQ_ISQN << (15U - (inserted_length - rank)*5U)));
        isq |= ((uint32_t)channel << (15U - (inserted_length - rank)*5U));
        ADC_ISQ = isq;
    }
    /* configure ADC sampling time */
    if(channel < 10U){
        sampt = ADC_SAMPT1;
        sampt &= ~((uint32_t)(ADC_SAMPTX_SPTN << (3U*channel)));
        sampt |= ((uint32_t) (sample_time & ADC_SAMPTX_SPTN) << (3U*channel));
        ADC_SAMPT1 = sampt;
    }else if(channel < 20U){
        sampt = ADC_SAMPT0;
        sampt &= ~((uint32_t)(ADC_SAMPTX_SPTN << (3U*(channel - 10U))));
        sampt |= ((uint32_t)(sample_time & ADC_SAMPTX_SPTN) << (3U*(channel - 10U)));
        ADC_SAMPT0 = sampt;
    }else{
        sampt = ADC_SAMPT2;
        sampt &= ~((uint32_t)(ADC_SAMPTX_SPTN << (3U*(channel-19U))));
        sampt |= ((uint32_t)(sample_time & ADC_SAMPTX_SPTN) << (3U*(channel-19U)));
        ADC_SAMPT2 = sampt;
    }
}

/*!
    \brief      configure ADC inserted channel offset (API_ID(0x0015U))
    \param[in]  inserted_channel: insert channel select
                only one parameter can be selected which is shown as below:
      \arg        ADC_INSERTED_CHANNEL_0: ADC inserted channel 0
      \arg        ADC_INSERTED_CHANNEL_1: ADC inserted channel 1
      \arg        ADC_INSERTED_CHANNEL_2: ADC inserted channel 2
      \arg        ADC_INSERTED_CHANNEL_3: ADC inserted channel 3
    \param[in]  offset: the offset data
    \param[out] none
    \retval     none
*/
void adc_inserted_channel_offset_config(uint8_t inserted_channel, uint16_t offset)
{
    uint8_t inserted_length;
    uint32_t num = 0U;

    inserted_length = (uint8_t)GET_BITS(ADC_ISQ, 20U, 21U);
    num = 3U - (inserted_length - inserted_channel);

    if(num <= 3U){
        /* calculate the offset of the register */
        num = num * 4U;
        /* configure the offset of the selected channels */
        REG32((ADC) + 0x14U + num) = IOFFX_IOFF((uint32_t)offset);
    }
}

/*!
    \brief      configure ADC inserted sequence repeat times (API_ID(0x0016U))
    \param[in]  time: a value between 0 and 63
    \param[out] none
    \retval     none
*/
void adc_inserted_channel_repeat_time_config(uint8_t time)
{
    ADC_CTL2 &= (~(uint32_t)ADC_CTL2_IREP);
    ADC_CTL2 |= (uint32_t)(((uint32_t)time << 2U)&ADC_CTL2_IREP);
}

/*!
    \brief      enable or disable ADC external trigger (API_ID(0x0017U))
    \param[in]  adc_sequence: select the sequence
                one or more parameters can be selected which is shown as below:
      \arg        ADC_ROUTINE_CHANNEL: routine sequence
      \arg        ADC_INSERTED_CHANNEL: inserted sequence
    \param[in]  newvalue: ENABLE or DISABLE
    \param[out] none
    \retval     none
*/
void adc_external_trigger_config(uint8_t adc_sequence, ControlStatus newvalue)
{
    if(newvalue){
        /* external trigger enable for routine channel */
        if(RESET != (adc_sequence & ADC_ROUTINE_CHANNEL)){
            ADC_CTL1 |= ADC_CTL1_ETERC;
        }
        /* external trigger enable for inserted channel */
        if(RESET != (adc_sequence & ADC_INSERTED_CHANNEL)){
            ADC_CTL1 |= ADC_CTL1_ETEIC;
        }
    }else{
        /* external trigger disable for routine channel */
        if(RESET != (adc_sequence & ADC_ROUTINE_CHANNEL)){
            ADC_CTL1 &= ~ADC_CTL1_ETERC;
        }
        /* external trigger disable for inserted channel */
        if(RESET != (adc_sequence & ADC_INSERTED_CHANNEL)){
            ADC_CTL1 &= ~ADC_CTL1_ETEIC;
        }
    }
}

/*!
    \brief      configure ADC external trigger source (API_ID(0x0018U))
    \param[in]  adc_sequence: select the sequence
                only one parameter can be selected which is shown as below:
      \arg        ADC_ROUTINE_CHANNEL: routine sequence
      \arg        ADC_INSERTED_CHANNEL: inserted sequence
    \param[in]  external_trigger_source: routine or inserted sequence trigger source
                only one parameter can be selected which is shown as below:
                for routine channel:
      \arg        ADC_EXTTRIG_ROUTINE_T0_CH0: TIMER0 CH0 event select
      \arg        ADC_EXTTRIG_ROUTINE_T0_CH1: TIMER0 CH1 event select
      \arg        ADC_EXTTRIG_ROUTINE_T0_CH2: TIMER0 CH2 event select
      \arg        ADC_EXTTRIG_ROUTINE_T1_CH1: TIMER1 CH1 event select
      \arg        ADC_EXTTRIG_ROUTINE_T2_TRGO: TIMER2 TRGO event select
      \arg        ADC_EXTTRIG_ROUTINE_T14_CH0: TIMER14 CH0 event select
      \arg        ADC_EXTTRIG_ROUTINE_EXTI_11: external interrupt line 11
      \arg        ADC_EXTTRIG_ROUTINE_NONE: software trigger
      \arg        ADC_EXTTRIG_ROUTINE_CLA0: CLA0 output
      \arg        ADC_EXTTRIG_ROUTINE_CLA2: CLA2 output
                for inserted channel:
      \arg        ADC_EXTTRIG_INSERTED_T0_TRGO: TIMER0 TRGO event select
      \arg        ADC_EXTTRIG_INSERTED_T0_CH3: TIMER0 CH3 event select
      \arg        ADC_EXTTRIG_INSERTED_T1_TRGO: TIMER1 TRGO event select
      \arg        ADC_EXTTRIG_INSERTED_T1_CH0: TIMER1 CH0 event select
      \arg        ADC_EXTTRIG_INSERTED_T2_CH3: TIMER2 CH3 event select
      \arg        ADC_EXTTRIG_INSERTED_T14_TRGO: TIMER14 TRGO event select
      \arg        ADC_EXTTRIG_INSERTED_EXTI_15: external interrupt line 15
      \arg        ADC_EXTTRIG_INSERTED_NONE: software trigger
      \arg        ADC_EXTTRIG_INSERTED_CLA1: CLA1 output
      \arg        ADC_EXTTRIG_INSERTED_CLA3: CLA3 output
    \param[out] none
    \retval     none
*/
void adc_external_trigger_source_config(uint8_t adc_sequence, uint32_t external_trigger_source)
{
    switch(adc_sequence){
    case ADC_ROUTINE_CHANNEL:
        /* external trigger select for routine channel */
        ADC_CTL1 &= ~((uint32_t)ADC_CTL1_ETSRC);
        ADC_CTL1 |= (uint32_t)(external_trigger_source & ADC_CTL1_ETSRC);
        break;
    case ADC_INSERTED_CHANNEL:
        /* external trigger select for inserted channel */
        ADC_CTL1 &= ~((uint32_t)ADC_CTL1_ETSIC);
        ADC_CTL1 |= (uint32_t)(external_trigger_source & ADC_CTL1_ETSIC);
        break;
    default:
        break;
    }
}

/*!
    \brief      enable ADC software trigger (API_ID(0x0019U))
    \param[in]  adc_sequence: select the sequence
                one or more parameters can be selected which is shown as below:
      \arg        ADC_ROUTINE_CHANNEL: routine sequence
      \arg        ADC_INSERTED_CHANNEL: inserted sequence
    \param[out] none
    \retval     none
*/
void adc_software_trigger_enable(uint8_t adc_sequence)
{
    /* enable routine sequence channel software trigger */
    if(RESET != (adc_sequence & ADC_ROUTINE_CHANNEL)){
        ADC_CTL1 |= ADC_CTL1_SWRCST;
    }
    /* enable inserted sequence software trigger */
    if(RESET != (adc_sequence & ADC_INSERTED_CHANNEL)){
        ADC_CTL1 |= ADC_CTL1_SWICST;
    }
}

/*!
    \brief      read ADC routine sequence data register (API_ID(0x001AU))
    \param[in]  none
    \param[out] none
    \retval     the conversion value
*/
uint16_t adc_routine_data_read(void)
{
    return ((uint16_t)ADC_RDATA);
}

/*!
    \brief      read ADC inserted sequence data register (API_ID(0x001BU))
    \param[in]  adc_periph: ADCx, x=0,1
    \param[out] none
    \retval     the conversion value:
*/
uint16_t adc_inserted_data_read(void)
{
    return (uint16_t)(ADC_IDATA);
}

/*!
    \brief      read ADC latch data register (API_ID(0x001CU))
    \param[in]  latch_data: latch data select
                only one parameter can be selected which is shown as below:
      \arg        ADC_LATCH_DATA_0: ADC latch data 0
      \arg        ADC_LATCH_DATA_1: ADC latch data 1
      \arg        ADC_LATCH_DATA_2: ADC latch data 2
      \arg        ADC_LATCH_DATA_3: ADC latch data 3
    \param[out] none
    \retval     the conversion value
*/
uint16_t adc_latch_data_read(uint8_t latch_data)
{
    uint32_t ldata = 0U;
    
    /* read the data of the selected channel */
    switch(latch_data){
        case ADC_LATCH_DATA_0:
            ldata = ADC_LDATA0;
            break;
        case ADC_LATCH_DATA_1:
            ldata = ADC_LDATA1;
            break;
        case ADC_LATCH_DATA_2:
            ldata = ADC_LDATA2;
            break;
        case ADC_LATCH_DATA_3:
            ldata = ADC_LDATA3;
            break;
        default:
            break;
    }
    return (uint16_t)ldata;
}

/*!
    \brief      configure ADC latch data source (API_ID(0x001DU))
    \param[in]  latch_data: latch data select
                only one parameter can be selected which is shown as below:
      \arg        ADC_LATCH_DATA_0: ADC latch data 0
      \arg        ADC_LATCH_DATA_1: ADC latch data 1
      \arg        ADC_LATCH_DATA_2: ADC latch data 2
      \arg        ADC_LATCH_DATA_3: ADC latch data 3
    \param[in]  adc_sequence: select the sequence
                only one parameter can be selected which is shown as below:
      \arg        ADC_ROUTINE_CHANNEL: routine sequence
      \arg        ADC_INSERTED_CHANNEL: inserted sequence
    \param[in]  rank: the routine sequence rank, this parameter must be between 0 to 22
                      the inserted sequencer rank,this parameter must be between 0 to 3
    \param[out] none
    \retval     none
*/
void adc_latch_data_source_config(uint8_t latch_data, uint8_t adc_sequence, uint8_t rank)
{
    uint32_t ldctl;
    ldctl = ADC_LDCTL;
    ldctl &= ~((ADC_LDCTL_COVSELx | ADC_LDCTL_SEQSELx) << (24U - (latch_data & ADC_LATCH_DATA_3)*8U));

    switch(adc_sequence){
    case ADC_ROUTINE_CHANNEL:
        /* ADC latch data source selection for routine channel */
        ldctl |= (ADC_LDCTL_SEQSELx << (24U - (latch_data & ADC_LATCH_DATA_MASK)*8U));
        ldctl |= ((uint32_t)rank << (24U - (latch_data & ADC_LATCH_DATA_MASK)*8U));
        break;
    case ADC_INSERTED_CHANNEL:
        /* ADC latch data source selection for inserted channel */
        if(rank < 4U){
            ldctl |= ((uint32_t)rank << (24U - (latch_data & ADC_LATCH_DATA_MASK)*8U));
        }
        break;
    default:
        break;
    }

    ADC_LDCTL = ldctl;
}

/*!
    \brief      configure ADC analog watchdog single channel (API_ID(0x001EU))
    \param[in]  channel: the selected ADC channel
                only one parameter can be selected which is shown as below:
      \arg        ADC_CHANNEL_x(x=0..22): ADC Channelx
    \param[out] none
    \retval     none
*/
void adc_watchdog_single_channel_enable(uint8_t channel)
{
    ADC_CTL0 &= (uint32_t)~(ADC_CTL0_RWDEN | ADC_CTL0_IWDEN | ADC_CTL0_WDSC | ADC_CTL0_WDCHSEL);

    ADC_CTL0 |= (uint32_t)(channel & ADC_CTL0_WDCHSEL);
    ADC_CTL0 |= (uint32_t)(ADC_CTL0_RWDEN | ADC_CTL0_IWDEN | ADC_CTL0_WDSC);
}

/*!
    \brief      configure ADC analog watchdog sequence channel (API_ID(0x001FU))
    \param[in]  adc_sequence: the sequence use analog watchdog
                only one parameter can be selected which is shown as below:
      \arg        ADC_ROUTINE_CHANNEL: routine sequence
      \arg        ADC_INSERTED_CHANNEL: inserted sequence
      \arg        ADC_ROUTINE_INSERTED_CHANNEL: both routine and inserted sequence
    \param[out] none
    \retval     none
*/
void adc_watchdog_sequence_channel_enable(uint8_t adc_sequence)
{
    ADC_CTL0 &= (uint32_t)~(ADC_CTL0_RWDEN | ADC_CTL0_IWDEN | ADC_CTL0_WDSC);

    /* select the sequence */
    switch(adc_sequence){
    case ADC_ROUTINE_CHANNEL:
        ADC_CTL0 |= (uint32_t)ADC_CTL0_RWDEN;
        break;
    case ADC_INSERTED_CHANNEL:
        ADC_CTL0 |= (uint32_t)ADC_CTL0_IWDEN;
        break;
    case ADC_ROUTINE_INSERTED_CHANNEL:
        ADC_CTL0 |= (uint32_t)(ADC_CTL0_RWDEN | ADC_CTL0_IWDEN);
        break;
    default:
        break;
    }
}

/*!
    \brief      disable ADC analog watchdog (API_ID(0x0020U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void adc_watchdog_disable(void)
{
    ADC_CTL0 &= (uint32_t)~(ADC_CTL0_RWDEN | ADC_CTL0_IWDEN | ADC_CTL0_WDSC | ADC_CTL0_WDCHSEL);
}

/*!
    \brief      configure ADC analog watchdog threshold (API_ID(0x0021U))
    \param[in]  low_threshold: analog watchdog low threshold, 0..4095
    \param[in]  high_threshold: analog watchdog high threshold, 0..4095
    \param[out] none
    \retval     none
*/
void adc_watchdog_threshold_config(uint16_t low_threshold, uint16_t high_threshold)
{
    ADC_WDLT = (uint32_t)WDLT_WDLT(low_threshold);
    ADC_WDHT = (uint32_t)WDHT_WDHT(high_threshold);
}

/*!
    \brief      configure ADC resolution (API_ID(0x0022U))
    \param[in]  resolution: ADC resolution
                only one parameter can be selected which is shown as below:
      \arg        ADC_RESOLUTION_12B: 12-bit ADC resolution
      \arg        ADC_RESOLUTION_10B: 10-bit ADC resolution
      \arg        ADC_RESOLUTION_8B: 8-bit ADC resolution
      \arg        ADC_RESOLUTION_6B: 6-bit ADC resolution
    \param[out] none
    \retval     none
*/
void adc_resolution_config(uint32_t resolution)
{
    ADC_CTL0 &= ~((uint32_t)ADC_CTL0_DRES);
    ADC_CTL0 |= (uint32_t)(resolution & ADC_CTL0_DRES);
}

/*!
    \brief      configure ADC oversample mode (API_ID(0x0023U))
    \param[in]  mode: ADC oversampling mode
                only one parameter can be selected which is shown as below:
      \arg        ADC_OVERSAMPLING_ALL_CONVERT: all oversampled conversions for a channel are done consecutively after a trigger
      \arg        ADC_OVERSAMPLING_ONE_CONVERT: each oversampled conversion for a channel needs a trigger
    \param[in]  shift: ADC oversampling shift
                only one parameter can be selected which is shown as below:
      \arg        ADC_OVERSAMPLING_SHIFT_NONE: no oversampling shift
      \arg        ADC_OVERSAMPLING_SHIFT_1B: 1-bit oversampling shift
      \arg        ADC_OVERSAMPLING_SHIFT_2B: 2-bit oversampling shift
      \arg        ADC_OVERSAMPLING_SHIFT_3B: 3-bit oversampling shift
      \arg        ADC_OVERSAMPLING_SHIFT_4B: 3-bit oversampling shift
      \arg        ADC_OVERSAMPLING_SHIFT_5B: 5-bit oversampling shift
      \arg        ADC_OVERSAMPLING_SHIFT_6B: 6-bit oversampling shift
      \arg        ADC_OVERSAMPLING_SHIFT_7B: 7-bit oversampling shift
      \arg        ADC_OVERSAMPLING_SHIFT_8B: 8-bit oversampling shift
    \param[in]  ratio: ADC oversampling ratio
                only one parameter can be selected which is shown as below:
      \arg        ADC_OVERSAMPLING_RATIO_MUL2: oversampling ratio multiple 2
      \arg        ADC_OVERSAMPLING_RATIO_MUL4: oversampling ratio multiple 4
      \arg        ADC_OVERSAMPLING_RATIO_MUL8: oversampling ratio multiple 8
      \arg        ADC_OVERSAMPLING_RATIO_MUL16: oversampling ratio multiple 16
      \arg        ADC_OVERSAMPLING_RATIO_MUL32: oversampling ratio multiple 32
      \arg        ADC_OVERSAMPLING_RATIO_MUL64: oversampling ratio multiple 64
      \arg        ADC_OVERSAMPLING_RATIO_MUL128: oversampling ratio multiple 128
      \arg        ADC_OVERSAMPLING_RATIO_MUL256: oversampling ratio multiple 256
    \param[out] none
    \retval     none
*/
void adc_oversample_mode_config(uint8_t mode, uint16_t shift, uint8_t ratio)
{
    /* configure ADC oversampling mode */
    if(ADC_OVERSAMPLING_ONE_CONVERT == mode){
        ADC_OVSAMPCTL |= (uint32_t)ADC_OVSAMPCTL_TOVS;
    }else{
        ADC_OVSAMPCTL &= ~((uint32_t)ADC_OVSAMPCTL_TOVS);
    }

    /* configure the shift and ratio */
    ADC_OVSAMPCTL &= ~((uint32_t)(ADC_OVSAMPCTL_OVSR | ADC_OVSAMPCTL_OVSS));
    ADC_OVSAMPCTL |= ((uint32_t)(shift & ADC_OVSAMPCTL_OVSS) | (uint32_t)(ratio & ADC_OVSAMPCTL_OVSR));
}

/*!
    \brief      enable ADC oversample mode (API_ID(0x0024U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void adc_oversample_mode_enable(void)
{
    ADC_OVSAMPCTL |= ADC_OVSAMPCTL_OVSEN;
}

/*!
    \brief      disable ADC oversample mode (API_ID(0x0025U))
    \param[in]  none
    \param[out] none
    \retval     none
*/
void adc_oversample_mode_disable(void)
{
    ADC_OVSAMPCTL &= ~((uint32_t)ADC_OVSAMPCTL_OVSEN);
}

/*!
    \brief      get the ADC flag bits (API_ID(0x0026U))
    \param[in]  flag: the adc flag bits
                only one parameter can be selected which is shown as below:
      \arg        ADC_FLAG_WDE: analog watchdog event flag
      \arg        ADC_FLAG_EORC: end of routine sequence conversion flag
      \arg        ADC_FLAG_EOIC: end of inserted sequence conversion flag
      \arg        ADC_FLAG_STIC: start flag of inserted sequence
      \arg        ADC_FLAG_STRC: start flag of routine sequence
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus adc_flag_get(uint32_t flag)
{
    FlagStatus reval = RESET;

    if(ADC_STAT & flag){
        reval = SET;
    }
    return reval;
}

/*!
    \brief      clear the ADC flag (API_ID(0x0027U))
    \param[in]  flag: the adc flag
                one or more parameters can be selected which is shown as below:
      \arg        ADC_FLAG_WDE: analog watchdog event flag
      \arg        ADC_FLAG_EORC: end of routine sequence conversion flag
      \arg        ADC_FLAG_EOIC: end of inserted sequence conversion flag
      \arg        ADC_FLAG_STIC: start flag of inserted sequence
      \arg        ADC_FLAG_STRC: start flag of routine sequence
    \param[out] none
    \retval     none
*/
void adc_flag_clear(uint32_t flag)
{
    ADC_STAT = ~((uint32_t)flag & ADC_FLAG_MASK);
}

/*!
    \brief      enable ADC interrupt (API_ID(0x0028U))
    \param[in]  interrupt: the adc interrupt
                one or more parameters can be selected which is shown as below:
      \arg        ADC_INT_WDE: analog watchdog interrupt
      \arg        ADC_INT_EORC: end of routine sequence conversion interrupt
      \arg        ADC_INT_EOIC: end of inserted sequence conversion interrupt
    \param[out] none
    \retval     none
*/
void adc_interrupt_enable(uint32_t interrupt)
{
    /* enable analog watchdog interrupt */
    if(RESET != (interrupt & ADC_INT_WDE)){
        ADC_CTL0 |= (uint32_t)ADC_CTL0_WDEIE;
    }

    /* enable end of routine sequence conversion interrupt */
    if(RESET != (interrupt & ADC_INT_EORC)){
        ADC_CTL0 |= (uint32_t)ADC_CTL0_EORCIE;
    }

    /* enable end of inserted sequence conversion interrupt */
    if(RESET != (interrupt & ADC_INT_EOIC)){
        ADC_CTL0 |= (uint32_t)ADC_CTL0_EOICIE;
    }
}

/*!
    \brief      disable ADC interrupt (API_ID(0x0029U))
    \param[in]  interrupt: the adc interrupt flag
                one or more parameters can be selected which is shown as below:
      \arg        ADC_INT_WDE: analog watchdog interrupt
      \arg        ADC_INT_EORC: end of routine sequence conversion interrupt
      \arg        ADC_INT_EOIC: end of inserted sequence conversion interrupt
    \param[out] none
    \retval     none
*/
void adc_interrupt_disable(uint32_t interrupt)
{
    /* disable analog watchdog interrupt */
    if(RESET != (interrupt & ADC_INT_WDE)){
        ADC_CTL0 &= ~(uint32_t)ADC_CTL0_WDEIE;
    }

    /* disable end of routine sequence conversion interrupt */
    if(RESET != (interrupt & ADC_INT_EORC)){
        ADC_CTL0 &= ~(uint32_t)ADC_CTL0_EORCIE;
    }

    /* disable end of inserted sequence conversion interrupt */
    if(RESET != (interrupt & ADC_INT_EOIC)){
        ADC_CTL0 &= ~(uint32_t)ADC_CTL0_EOICIE;
    }
}

/*!
    \brief      get the ADC interrupt flag (API_ID(0x002AU))
    \param[in]  flag: the adc interrupt flag
                only one parameter can be selected which is shown as below:
      \arg        ADC_INT_FLAG_WDE: analog watchdog interrupt flag
      \arg        ADC_INT_FLAG_EORC: end of routine sequence conversion interrupt flag
      \arg        ADC_INT_FLAG_EOIC: end of inserted sequence conversion interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus adc_interrupt_flag_get(uint32_t flag)
{
    FlagStatus interrupt_flag = RESET;
    uint32_t state;

    /* check the interrupt bits */
    switch(flag){
    case ADC_INT_FLAG_WDE:
        state = ADC_STAT & ADC_STAT_WDE;
        if((ADC_CTL0 & ADC_CTL0_WDEIE) && state){
          interrupt_flag = SET;
        }
        break;
    case ADC_INT_FLAG_EORC:
        state = ADC_STAT & ADC_STAT_EORC;
        if((ADC_CTL0 & ADC_CTL0_EORCIE) && state){
            interrupt_flag = SET;
          }
        break;
    case ADC_INT_FLAG_EOIC:
        state = ADC_STAT & ADC_STAT_EOIC;
        if((ADC_CTL0 & ADC_CTL0_EOICIE) && state){
            interrupt_flag = SET;
        }
        break;
    default:
        break;
    }
    return interrupt_flag;
}

/*!
    \brief      clear ADC interrupt flag (API_ID(0x002BU))
    \param[in]  flag: the adc interrupt flag
                only one parameter can be selected which is shown as below:
                one or more parameters can be selected which is shown as below:
      \arg        ADC_INT_FLAG_WDE: analog watchdog interrupt flag
      \arg        ADC_INT_FLAG_EORC: end of routine sequence conversion interrupt flag
      \arg        ADC_INT_FLAG_EOIC: end of inserted sequence conversion interrupt flag
    \param[out] none
    \retval     none
*/
void adc_interrupt_flag_clear(uint32_t flag)
{
    ADC_STAT = ~((uint32_t)flag & ADC_INT_FLAG_MASK);
}
