/*!
    \file    gd32e25x_syscfg.h
    \brief   definitions for the SYSCFG

    \version 2025-07-25, V0.2.0, firmware for GD32E25x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32E25X_SYSCFG_H
#define GD32E25X_SYSCFG_H

#include "gd32e25x.h"

/* SYSCFG definitions */
#define SYSCFG                              SYSCFG_BASE                         /*!< SYSCFG base address */

/* registers definitions */
#define SYSCFG_CFG0                         REG32(SYSCFG + 0x00U)               /*!< system configuration register 0 */
#define SYSCFG_EXTISS0                      REG32(SYSCFG + 0x08U)               /*!< EXTI sources selection register 0 */
#define SYSCFG_EXTISS1                      REG32(SYSCFG + 0x0CU)               /*!< EXTI sources selection register 1 */
#define SYSCFG_EXTISS2                      REG32(SYSCFG + 0x10U)               /*!< EXTI sources selection register 2 */
#define SYSCFG_EXTISS3                      REG32(SYSCFG + 0x14U)               /*!< EXTI sources selection register 3 */
#define SYSCFG_CFG1                         REG32(SYSCFG + 0x18U)               /*!< system configuration register 2 */
#define SYSCFG_STAT                         REG32(SYSCFG + 0x24U)               /*!< SRAM ECC status register */
#define SYSCFG_SRAM_ECC                     REG32(SYSCFG + 0x28U)               /*!< SRAM ECC register */
#define SYSCFG_CPU_IRQ_LAT                  REG32(SYSCFG + 0x100U)              /*!< IRQ Latency register */

/* SYSCFG_CFG0 bits definitions */
#define SYSCFG_CFG0_BOOT_MODE               BITS(0,1)                           /*!< Boot mode */
#define SYSCFG_CFG0_PA11_PA12_RMP           BIT(4)                              /*!< PA11 and PA12 remapping bit for small packages (28 and 20 pins) */
#if !defined(GD32E251)
#define SYSCFG_CFG0_BOOT0_PB9_RMP           BIT(6)                              /*!< BOOT0 and PB9 remapping bit */
#endif /* !defined(GD32E251) */
#define SYSCFG_CFG0_ADC_DMA_RMP             BIT(8)                              /*!< ADC DMA request remapping */
#define SYSCFG_CFG0_USART0_TX_DMA_RMP       BIT(9)                              /*!< USART0 Tx DMA remapping */
#define SYSCFG_CFG0_USART0_RX_DMA_RMP       BIT(10)                             /*!< USART0 Rx DMA remapping */
#define SYSCFG_CFG0_TIMER15_DMA_RMP         BIT(11)                             /*!< TIMER 15 DMA remapping */
#define SYSCFG_CFG0_TIMER16_DMA_RMP         BIT(12)                             /*!< TIMER 16 DMA remapping */
#define SYSCFG_CFG0_PB6_FMPEN               BIT(16)                             /*!< I2C Fm+ mode on PB6 pin */
#define SYSCFG_CFG0_PB7_FMPEN               BIT(17)                             /*!< I2C Fm+ mode on PB7 pin */
#define SYSCFG_CFG0_PB9_FMPEN               BIT(19)                             /*!< I2C Fm+ mode on PB9 pin */
#define SYSCFG_CFG0_SRAM_ERRADDR            BITS(21,31)                         /*!< SRAM parity check error address(read only) */

/* SYSCFG_EXTISS0 bits definitions */
#define SYSCFG_EXTISS0_EXTI0_SS             BITS(0,3)                           /*!< EXTI 0 configuration */
#define SYSCFG_EXTISS0_EXTI1_SS             BITS(4,7)                           /*!< EXTI 1 configuration */
#define SYSCFG_EXTISS0_EXTI2_SS             BITS(8,11)                          /*!< EXTI 2 configuration */
#define SYSCFG_EXTISS0_EXTI3_SS             BITS(12,15)                         /*!< EXTI 3 configuration */

/* SYSCFG_EXTISS1 bits definitions */
#define SYSCFG_EXTISS1_EXTI4_SS             BITS(0,3)                           /*!< EXTI 4 configuration */
#define SYSCFG_EXTISS1_EXTI5_SS             BITS(4,7)                           /*!< EXTI 5 configuration */
#define SYSCFG_EXTISS1_EXTI6_SS             BITS(8,11)                          /*!< EXTI 6 configuration */
#define SYSCFG_EXTISS1_EXTI7_SS             BITS(12,15)                         /*!< EXTI 7 configuration */

/* SYSCFG_EXTISS2 bits definitions */
#define SYSCFG_EXTISS2_EXTI8_SS             BITS(0,3)                           /*!< EXTI 8 configuration */
#define SYSCFG_EXTISS2_EXTI9_SS             BITS(4,7)                           /*!< EXTI 9 configuration */
#define SYSCFG_EXTISS2_EXTI10_SS            BITS(8,11)                          /*!< EXTI 10 configuration */
#define SYSCFG_EXTISS2_EXTI11_SS            BITS(12,15)                         /*!< EXTI 11 configuration */

/* SYSCFG_EXTISS3 bits definitions */
#define SYSCFG_EXTISS3_EXTI12_SS            BITS(0,3)                           /*!< EXTI 12 configuration */
#define SYSCFG_EXTISS3_EXTI13_SS            BITS(4,7)                           /*!< EXTI 13 configuration */
#define SYSCFG_EXTISS3_EXTI14_SS            BITS(8,11)                          /*!< EXTI 14 configuration */
#define SYSCFG_EXTISS3_EXTI15_SS            BITS(12,15)                         /*!< EXTI 15 configuration */

/* SYSCFG_CFG1 bits definitions */
#define SYSCFG_CFG1_LOCKUP_LOCK             BIT(0)                              /*!< enable and lock the LOCKUP (Hardfault) output of Cortex-M23 with break input of TIMER0/14/15/16 */
#define SYSCFG_CFG1_SRAM_PARITY_ERROR_LOCK  BIT(1)                              /*!< enable and lock the SRAM_PARITY error signal with break input of TIMER0/14/15/16 */
#define SYSCFG_CFG1_LVD_LOCK                BIT(2)                              /*!< enable and lock the LVD connection with TIMER0/14/15/16 break input and also the LVDEN and LVDT[2:0] bits of the power control interface */
#define SYSCFG_CFG1_SRAM_PCEF               BIT(8)                              /*!< SRAM parity check error flag */
#define SYSCFG_CFG1_VREF_EN                 BIT(16)                             /*!< VREF enable */
#define SYSCFG_CFG1_PIN_RSTMD               BIT(24)                             /*!< pin reset mode */

/* SYSCFG_STAT bits definitions */
#define SYSCFG_STAT_ECCMEIF0                BIT(0)                              /*!< SRAM multi bits non-correction event flag */
#define SYSCFG_STAT_ECCSEIF0                BIT(1)                              /*!< SRAM single bit correction event flag. */

/* SYSCFG_SRAM_ECC bits definitions */
#define SYSCFG_SRAM_ECC_ECCMEIE0            BIT(0)                             /*!< SRAM multi bits non-correction interrupt enable */
#define SYSCFG_SRAM_ECC_ECCSEIE0            BIT(1)                             /*!< SRAM single bit correction interrupt enable */
#define SYSCFG_SRAM_ECC_ECCSERRBITS0        BITS(10,15)                        /*!< indicates which one bit has an ECC single-bit correctable error */
#define SYSCFG_SRAM_ECC_ECCEADDR0           BITS(21,31)                        /*!< indicates the last address of ECC event on SRAM occurred. */

/* SYSCFG_CPU_IRQ_LAT bits definitions */
#define SYSCFG_CPU_IRQ_LAT_IRQ_LATENCY      BITS(0,7)                           /*!< IRQ_LATENCY specifies the minimum number of cycles between an interrupt */

/* constants definitions */
/* boot mode definitions */
#define SYSCFG_BOOTMODE_FLASH               ((uint8_t)0x00U)                    /*!< main flash memory remap */
#define SYSCFG_BOOTMODE_SYSTEM              ((uint8_t)0x01U)                    /*!< boot loader remap */
#define SYSCFG_BOOTMODE_SRAM                ((uint8_t)0x02U)                    /*!< SRAM/NOR 0 and 1 of EXMC remap */

/* Pin remap definitions */
#define SYSCFG_PA11_REMAP_PA12              SYSCFG_CFG0_PA11_PA12_RMP           /*!< PA11 PA12 remap */
#if !defined(GD32E251)
#define SYSCFG_BOOT0_REMAP_PB9              SYSCFG_CFG0_BOOT0_PB9_RMP           /*!< BOOT0 PB9 remap */
#endif /* !defined(GD32E251) */

/* DMA remap definitions */
#define SYSCFG_DMA_REMAP_ADC                SYSCFG_CFG0_ADC_DMA_RMP             /*!< ADC DMA remap */
#define SYSCFG_DMA_REMAP_USART0TX           SYSCFG_CFG0_USART0_TX_DMA_RMP       /*!< USART0_TX DMA remap */
#define SYSCFG_DMA_REMAP_USART0RX           SYSCFG_CFG0_USART0_RX_DMA_RMP       /*!< USART0_RX DMA remap */
#define SYSCFG_DMA_REMAP_TIMER15            SYSCFG_CFG0_TIMER15_DMA_RMP         /*!< TIMER15 DMA remap */
#define SYSCFG_DMA_REMAP_TIMER16            SYSCFG_CFG0_TIMER16_DMA_RMP         /*!< TIMER16 DMA remap */

/* I2C Fm+ mode definitions */
#define SYSCFG_PB6_FMPEN                    SYSCFG_CFG0_PB6_FMPEN                /*!< PB6 pin high current capability enable */
#define SYSCFG_PB7_FMPEN                    SYSCFG_CFG0_PB7_FMPEN                /*!< PB7 pin high current capability enable */
#define SYSCFG_PB9_FMPEN                    SYSCFG_CFG0_PB9_FMPEN                /*!< PB9 pin high current capability enable */

/* EXTI source select definition */
#define EXTISS0                             ((uint8_t)0x00U)                    /*!< EXTI source select register 0 */
#define EXTISS1                             ((uint8_t)0x01U)                    /*!< EXTI source select register 1 */
#define EXTISS2                             ((uint8_t)0x02U)                    /*!< EXTI source select register 2 */
#define EXTISS3                             ((uint8_t)0x03U)                    /*!< EXTI source select register 3 */

/* EXTI source select mask bits definition */
#define EXTI_SS_MASK                        BITS(0,3)                           /*!< EXTI source select mask */

/* EXTI source select jumping step definition */
#define EXTI_SS_JSTEP                       ((uint8_t)(0x04U))                  /*!< EXTI source select jumping step */

/* EXTI source select moving step definition */
#define EXTI_SS_MSTEP(pin)                  (EXTI_SS_JSTEP * ((pin) % EXTI_SS_JSTEP))   /*!< EXTI source select moving step */

/* EXTI source port definitions */
#define EXTI_SOURCE_GPIOA                   ((uint8_t)0x00U)                    /*!< EXTI GPIOA configuration */
#define EXTI_SOURCE_GPIOB                   ((uint8_t)0x01U)                    /*!< EXTI GPIOB configuration */
#define EXTI_SOURCE_GPIOF                   ((uint8_t)0x05U)                    /*!< EXTI GPIOF configuration */

/* EXTI source pin definitions */
#define EXTI_SOURCE_PIN0                    ((uint8_t)0x00U)                    /*!< EXTI GPIO pin0 configuration */
#define EXTI_SOURCE_PIN1                    ((uint8_t)0x01U)                    /*!< EXTI GPIO pin1 configuration */
#define EXTI_SOURCE_PIN2                    ((uint8_t)0x02U)                    /*!< EXTI GPIO pin2 configuration */
#define EXTI_SOURCE_PIN3                    ((uint8_t)0x03U)                    /*!< EXTI GPIO pin3 configuration */
#define EXTI_SOURCE_PIN4                    ((uint8_t)0x04U)                    /*!< EXTI GPIO pin4 configuration */
#define EXTI_SOURCE_PIN5                    ((uint8_t)0x05U)                    /*!< EXTI GPIO pin5 configuration */
#define EXTI_SOURCE_PIN6                    ((uint8_t)0x06U)                    /*!< EXTI GPIO pin6 configuration */
#define EXTI_SOURCE_PIN7                    ((uint8_t)0x07U)                    /*!< EXTI GPIO pin7 configuration */
#define EXTI_SOURCE_PIN8                    ((uint8_t)0x08U)                    /*!< EXTI GPIO pin8 configuration */
#define EXTI_SOURCE_PIN9                    ((uint8_t)0x09U)                    /*!< EXTI GPIO pin9 configuration */
#define EXTI_SOURCE_PIN10                   ((uint8_t)0x0AU)                    /*!< EXTI GPIO pin10 configuration */
#define EXTI_SOURCE_PIN11                   ((uint8_t)0x0BU)                    /*!< EXTI GPIO pin11 configuration */
#define EXTI_SOURCE_PIN12                   ((uint8_t)0x0CU)                    /*!< EXTI GPIO pin12 configuration */
#define EXTI_SOURCE_PIN13                   ((uint8_t)0x0DU)                    /*!< EXTI GPIO pin13 configuration */
#define EXTI_SOURCE_PIN14                   ((uint8_t)0x0EU)                    /*!< EXTI GPIO pin14 configuration */
#define EXTI_SOURCE_PIN15                   ((uint8_t)0x0FU)                    /*!< EXTI GPIO pin15 configuration */

/* constants definitions */
/* EXTI source pin definitions */
typedef enum {
    SYSCFG_EXTI_LINE_11 = 0,                                                    /*!< EXTI line 11 */
    SYSCFG_EXTI_LINE_15                                                         /*!< EXTI line 15 */
} syscfg_exti_line;

/* EXTI source pin definitions */
typedef enum {
    SYSCFG_INT_ECCME = SYSCFG_SRAM_ECC_ECCMEIE0,                                /*!< SRAM multi bits non-correction interrupt */
    SYSCFG_INT_ECCSE = SYSCFG_SRAM_ECC_ECCSEIE0                                 /*!< SRAM single bit correction interrupt */
} syscfg_interrupt_enum;

/* EXTI source pin definitions */
typedef enum {
    SYSCFG_ERR_PC = 0,                                                          /*!< SRAM parity check error */
    SYSCFG_ERR_ECC                                                              /*!< SRAM ECC check error */
} syscfg_error_enum;

/* lock definitions */
#define SYSCFG_LOCK_LOCKUP                  SYSCFG_CFG1_LOCKUP_LOCK             /*!< LOCKUP output lock */
#define SYSCFG_LOCK_SRAM_PARITY_ERROR       SYSCFG_CFG1_SRAM_PARITY_ERROR_LOCK  /*!< SRAM parity error lock */
#define SYSCFG_LOCK_LVD                     SYSCFG_CFG1_LVD_LOCK                /*!< LVD lock */

/* SRAM parity check error flag definitions */
#define SYSCFG_SRAM_PCEF                    SYSCFG_CFG1_SRAM_PCEF               /*!< SRAM parity check error flag */

/* SRAM ECC bit correction event flag definitions */
#define SYSCFG_SRAM_ECCMEIF                 SYSCFG_STAT_ECCMEIF0                /*!< SRAM multi bits non-correction event flag */
#define SYSCFG_SRAM_ECCSEIF                 SYSCFG_STAT_ECCSEIF0                /*!< SRAM single bit correction event flag */

/* pin reset mode definitions */
#define SYSCFG_PIN_NRST                     ((uint32_t)0x01000000U)             /*!< GPIO pin configuration will retain state across any reset event except for the POR event */
#define SYSCFG_PIN_RST                      ((uint32_t)0x00000000U)             /*!< GPIO pin configuration is reset when any reset event occurs */

/* SYSCFG_CPU_IRQ_LAT register IRQ_LATENCY value */
#define IRQ_LATENCY(regval)                 (BITS(0,7) & ((uint32_t)(regval) << 0U))     /*!< write value to IRQ_LATENCY bits field */

/* function declarations */
/* initialization functions */
/* reset the SYSCFG registers */
void syscfg_deinit(void);

/* function configuration */
/* enable I2C Fm+ mode */
void syscfg_i2c_fast_mode_plus_enable(uint32_t syscfg_gpio);
/* disable I2C Fm+ mode */
void syscfg_i2c_fast_mode_plus_disable(uint32_t syscfg_gpio);

/* get the boot mode */
uint8_t syscfg_bootmode_get(void);

/* enable remap pin function */
void syscfg_pin_remap_enable(uint32_t remap_pin);
/* disable remap pin function */
void syscfg_pin_remap_disable(uint32_t remap_pin);

/* enable the DMA channels remapping */
void syscfg_dma_remap_enable(uint32_t syscfg_dma_remap);
/* disable the DMA channels remapping */
void syscfg_dma_remap_disable(uint32_t syscfg_dma_remap);

/* enable VREF function */
void syscfg_vref_enable(void);
/* disable VREF function */
void syscfg_vref_disable(void);

/* configure the GPIO pin reset mode */
void syscfg_pin_reset_mode_config(uint32_t syscfg_pin_reset_mode);

/* configure the GPIO pin as EXTI Line */
void syscfg_exti_line_config(uint8_t exti_port, uint8_t exti_pin);
/* configure the PB3 as EXTI Line11 and Line15 */
void syscfg_exti_line11_line15_pb3(uint8_t line);
/* connect TIMER0/14/15/16 break input to the selected parameter */
void syscfg_lock_config(uint32_t syscfg_lock);

/* set the IRQ_LATENCY value */
void irq_latency_set(uint8_t irq_latency);

/* get SRAM error address in SYSCFG_CFG0 or SYSCFG_SRAM_ECC */
uint32_t syscfg_sram_error_address_get(syscfg_error_enum errtype);
/* get SRAM ECC single-bit correctable error bit in SYSCFG_SRAM_ECC */
uint32_t syscfg_sram_error_bit_get(void);

/* check if the specified flag in SYSCFG_CFG1 is set or not */
FlagStatus syscfg_flag_get(uint32_t syscfg_flag);
/* clear the flag in SYSCFG_CFG1 by writing 1 */
void syscfg_flag_clear(uint32_t syscfg_flag);
/* enable SRAM ECC interrupt */
void syscfg_interrupt_enable(syscfg_interrupt_enum inttype);
/* disable SRAM ECC interrupt */
void syscfg_interrupt_disable(syscfg_interrupt_enum inttype);
/* check if the specified interrupt flag in SYSCFG_STAT is set or not */
FlagStatus syscfg_interrupt_flag_get(uint32_t int_flag);
/* clear the interrupt flag in SYSCFG_STAT by writing 1 */
void syscfg_interrupt_flag_clear(uint32_t int_flag);

#endif /* GD32E25X_SYSCFG_H */
