/*!
    \file    gd32e25x_gpio.h
    \brief   definitions for the GPIO

    \version 2025-07-25, V0.2.0, firmware for GD32E25x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32E25X_GPIO_H
#define GD32E25X_GPIO_H

#include "gd32e25x.h"
/* GPIOx(x=A,B,F,SP) definitions */
#define GPIOA                      (GPIO_BASE + 0x00000000U)     /*!< GPIOA base address */
#define GPIOB                      (GPIO_BASE + 0x00000400U)     /*!< GPIOB base address */
#define GPIOF                      (GPIO_BASE + 0x00001400U)     /*!< GPIOF base address */
#if defined(GD32E251)
#define GPIOSP                     (GPIO_BASE + 0x00001000U)     /*!< GPIOSP base address */
#endif /* GD32E251 */
/* registers definitions */
#define GPIO_CTL(gpiox)            REG32((gpiox) + 0x00U)        /*!< GPIO port control register 1 */
#define GPIO_OMODE(gpiox)          REG32((gpiox) + 0x04U)        /*!< GPIO port output mode register */
#define GPIO_OSPD(gpiox)           REG32((gpiox) + 0x08U)        /*!< GPIO port output speed register */
#define GPIO_PUD(gpiox)            REG32((gpiox) + 0x0CU)        /*!< GPIO port pull-up/pull-down register */
#define GPIO_ISTAT(gpiox)          REG32((gpiox) + 0x10U)        /*!< GPIO port input status register */
#define GPIO_OCTL(gpiox)           REG32((gpiox) + 0x14U)        /*!< GPIO port output control register */
#define GPIO_BOP(gpiox)            REG32((gpiox) + 0x18U)        /*!< GPIO port bit operation register */
#define GPIO_LOCK(gpiox)           REG32((gpiox) + 0x1CU)        /*!< GPIO port configuration lock register */
#define GPIO_AFSEL0(gpiox)         REG32((gpiox) + 0x20U)        /*!< GPIO alternate function selected register 0 */
#define GPIO_AFSEL1(gpiox)         REG32((gpiox) + 0x24U)        /*!< GPIO alternate function selected register 1 */
#define GPIO_BC(gpiox)             REG32((gpiox) + 0x28U)        /*!< GPIO bit clear register */
#define GPIO_TG(gpiox)             REG32((gpiox) + 0x2CU)        /*!< GPIO port bit toggle register */
#if defined(GD32E251)
#define GPIO_CBREN                 REG32((GPIOSP) + 0x34U)       /*!< GPIO crossbar enable register */
#define GPIO_DFEN                  REG32((GPIOSP) + 0x38U)       /*!< GPIO Digital function enable register */
#define GPIO_SKIP                  REG32((GPIOSP) + 0x3CU)       /*!< GPIO crossbar skip register */
#define GPIO_INTCTL                REG32((GPIOSP) + 0x40U)       /*!< INT0 and INT1 control register */
#define GPIO_MASK                  REG32((GPIOSP) + 0x44U)       /*!< GPIO pin mask register */
#define GPIO_MATCH                 REG32((GPIOSP) + 0x48U)       /*!< GPIO pin match register */
#define GPIO_MATCHF                REG32((GPIOSP) + 0x4CU)       /*!< GPIO pin match flag register */
#endif /* GD32E251 */
/* bits definitions */
/* GPIO_CTL */
#define GPIO_CTL_CTL0              BITS(0, 1)                    /*!< pin 0 configuration bits */
#define GPIO_CTL_CTL1              BITS(2, 3)                    /*!< pin 1 configuration bits */
#define GPIO_CTL_CTL2              BITS(4, 5)                    /*!< pin 2 configuration bits */
#define GPIO_CTL_CTL3              BITS(6, 7)                    /*!< pin 3 configuration bits */
#define GPIO_CTL_CTL4              BITS(8, 9)                    /*!< pin 4 configuration bits */
#define GPIO_CTL_CTL5              BITS(10, 11)                  /*!< pin 5 configuration bits */
#define GPIO_CTL_CTL6              BITS(12, 13)                  /*!< pin 6 configuration bits */
#define GPIO_CTL_CTL7              BITS(14, 15)                  /*!< pin 7 configuration bits */
#define GPIO_CTL_CTL8              BITS(16, 17)                  /*!< pin 8 configuration bits */
#define GPIO_CTL_CTL9              BITS(18, 19)                  /*!< pin 9 configuration bits */
#define GPIO_CTL_CTL10             BITS(20, 21)                  /*!< pin 10 configuration bits */
#define GPIO_CTL_CTL11             BITS(22, 23)                  /*!< pin 11 configuration bits */
#define GPIO_CTL_CTL12             BITS(24, 25)                  /*!< pin 12 configuration bits */
#define GPIO_CTL_CTL13             BITS(26, 27)                  /*!< pin 13 configuration bits */
#define GPIO_CTL_CTL14             BITS(28, 29)                  /*!< pin 14 configuration bits */
#define GPIO_CTL_CTL15             BITS(30, 31)                  /*!< pin 15 configuration bits */

/* GPIO_OMODE */
#define GPIO_OMODE_OM0             BIT(0)                        /*!< pin 0 output mode bit */
#define GPIO_OMODE_OM1             BIT(1)                        /*!< pin 1 output mode bit */
#define GPIO_OMODE_OM2             BIT(2)                        /*!< pin 2 output mode bit */
#define GPIO_OMODE_OM3             BIT(3)                        /*!< pin 3 output mode bit */
#define GPIO_OMODE_OM4             BIT(4)                        /*!< pin 4 output mode bit */
#define GPIO_OMODE_OM5             BIT(5)                        /*!< pin 5 output mode bit */
#define GPIO_OMODE_OM6             BIT(6)                        /*!< pin 6 output mode bit */
#define GPIO_OMODE_OM7             BIT(7)                        /*!< pin 7 output mode bit */
#define GPIO_OMODE_OM8             BIT(8)                        /*!< pin 8 output mode bit */
#define GPIO_OMODE_OM9             BIT(9)                        /*!< pin 9 output mode bit */
#define GPIO_OMODE_OM10            BIT(10)                       /*!< pin 10 output mode bit */
#define GPIO_OMODE_OM11            BIT(11)                       /*!< pin 11 output mode bit */
#define GPIO_OMODE_OM12            BIT(12)                       /*!< pin 12 output mode bit */
#define GPIO_OMODE_OM13            BIT(13)                       /*!< pin 13 output mode bit */
#define GPIO_OMODE_OM14            BIT(14)                       /*!< pin 14 output mode bit */
#define GPIO_OMODE_OM15            BIT(15)                       /*!< pin 15 output mode bit */

/* GPIO_OSPD */
#define GPIO_OSPD_OSPD0            BITS(0, 1)                    /*!< pin 0 output max speed bits */
#define GPIO_OSPD_OSPD1            BITS(2, 3)                    /*!< pin 1 output max speed bits */
#define GPIO_OSPD_OSPD2            BITS(4, 5)                    /*!< pin 2 output max speed bits */
#define GPIO_OSPD_OSPD3            BITS(6, 7)                    /*!< pin 3 output max speed bits */
#define GPIO_OSPD_OSPD4            BITS(8, 9)                    /*!< pin 4 output max speed bits */
#define GPIO_OSPD_OSPD5            BITS(10, 11)                  /*!< pin 5 output max speed bits */
#define GPIO_OSPD_OSPD6            BITS(12, 13)                  /*!< pin 6 output max speed bits */
#define GPIO_OSPD_OSPD7            BITS(14, 15)                  /*!< pin 7 output max speed bits */
#define GPIO_OSPD_OSPD8            BITS(16, 17)                  /*!< pin 8 output max speed bits */
#define GPIO_OSPD_OSPD9            BITS(18, 19)                  /*!< pin 9 output max speed bits */
#define GPIO_OSPD_OSPD10           BITS(20, 21)                  /*!< pin 10 output max speed bits */
#define GPIO_OSPD_OSPD11           BITS(22, 23)                  /*!< pin 11 output max speed bits */
#define GPIO_OSPD_OSPD12           BITS(24, 25)                  /*!< pin 12 output max speed bits */
#define GPIO_OSPD_OSPD13           BITS(26, 27)                  /*!< pin 13 output max speed bits */
#define GPIO_OSPD_OSPD14           BITS(28, 29)                  /*!< pin 14 output max speed bits */
#define GPIO_OSPD_OSPD15           BITS(30, 31)                  /*!< pin 15 output max speed bits */

/* GPIO_PUD */
#define GPIO_PUD_PUD0              BITS(0, 1)                    /*!< pin 0 pull-up or pull-down bits */
#define GPIO_PUD_PUD1              BITS(2, 3)                    /*!< pin 1 pull-up or pull-down bits */
#define GPIO_PUD_PUD2              BITS(4, 5)                    /*!< pin 2 pull-up or pull-down bits */
#define GPIO_PUD_PUD3              BITS(6, 7)                    /*!< pin 3 pull-up or pull-down bits */
#define GPIO_PUD_PUD4              BITS(8, 9)                    /*!< pin 4 pull-up or pull-down bits */
#define GPIO_PUD_PUD5              BITS(10, 11)                  /*!< pin 5 pull-up or pull-down bits */
#define GPIO_PUD_PUD6              BITS(12, 13)                  /*!< pin 6 pull-up or pull-down bits */
#define GPIO_PUD_PUD7              BITS(14, 15)                  /*!< pin 7 pull-up or pull-down bits */
#define GPIO_PUD_PUD8              BITS(16, 17)                  /*!< pin 8 pull-up or pull-down bits */
#define GPIO_PUD_PUD9              BITS(18, 19)                  /*!< pin 9 pull-up or pull-down bits */
#define GPIO_PUD_PUD10             BITS(20, 21)                  /*!< pin 10 pull-up or pull-down bits */
#define GPIO_PUD_PUD11             BITS(22, 23)                  /*!< pin 11 pull-up or pull-down bits */
#define GPIO_PUD_PUD12             BITS(24, 25)                  /*!< pin 12 pull-up or pull-down bits */
#define GPIO_PUD_PUD13             BITS(26, 27)                  /*!< pin 13 pull-up or pull-down bits */
#define GPIO_PUD_PUD14             BITS(28, 29)                  /*!< pin 14 pull-up or pull-down bits */
#define GPIO_PUD_PUD15             BITS(30, 31)                  /*!< pin 15 pull-up or pull-down bits */

/* GPIO_ISTAT */
#define GPIO_ISTAT_ISTAT0          BIT(0)                        /*!< pin 0 input status */
#define GPIO_ISTAT_ISTAT1          BIT(1)                        /*!< pin 1 input status */
#define GPIO_ISTAT_ISTAT2          BIT(2)                        /*!< pin 2 input status */
#define GPIO_ISTAT_ISTAT3          BIT(3)                        /*!< pin 3 input status */
#define GPIO_ISTAT_ISTAT4          BIT(4)                        /*!< pin 4 input status */
#define GPIO_ISTAT_ISTAT5          BIT(5)                        /*!< pin 5 input status */
#define GPIO_ISTAT_ISTAT6          BIT(6)                        /*!< pin 6 input status */
#define GPIO_ISTAT_ISTAT7          BIT(7)                        /*!< pin 7 input status */
#define GPIO_ISTAT_ISTAT8          BIT(8)                        /*!< pin 8 input status */
#define GPIO_ISTAT_ISTAT9          BIT(9)                        /*!< pin 9 input status */
#define GPIO_ISTAT_ISTAT10         BIT(10)                       /*!< pin 10 input status */
#define GPIO_ISTAT_ISTAT11         BIT(11)                       /*!< pin 11 input status */
#define GPIO_ISTAT_ISTAT12         BIT(12)                       /*!< pin 12 input status */
#define GPIO_ISTAT_ISTAT13         BIT(13)                       /*!< pin 13 input status */
#define GPIO_ISTAT_ISTAT14         BIT(14)                       /*!< pin 14 input status */
#define GPIO_ISTAT_ISTAT15         BIT(15)                       /*!< pin 15 input status */

/* GPIO_OCTL */
#define GPIO_OCTL_OCTL0            BIT(0)                        /*!< pin 0 output bit */
#define GPIO_OCTL_OCTL1            BIT(1)                        /*!< pin 1 output bit */
#define GPIO_OCTL_OCTL2            BIT(2)                        /*!< pin 2 output bit */
#define GPIO_OCTL_OCTL3            BIT(3)                        /*!< pin 3 output bit */
#define GPIO_OCTL_OCTL4            BIT(4)                        /*!< pin 4 output bit */
#define GPIO_OCTL_OCTL5            BIT(5)                        /*!< pin 5 output bit */
#define GPIO_OCTL_OCTL6            BIT(6)                        /*!< pin 6 output bit */
#define GPIO_OCTL_OCTL7            BIT(7)                        /*!< pin 7 output bit */
#define GPIO_OCTL_OCTL8            BIT(8)                        /*!< pin 8 output bit */
#define GPIO_OCTL_OCTL9            BIT(9)                        /*!< pin 9 output bit */
#define GPIO_OCTL_OCTL10           BIT(10)                       /*!< pin 10 output bit */
#define GPIO_OCTL_OCTL11           BIT(11)                       /*!< pin 11 output bit */
#define GPIO_OCTL_OCTL12           BIT(12)                       /*!< pin 12 output bit */
#define GPIO_OCTL_OCTL13           BIT(13)                       /*!< pin 13 output bit */
#define GPIO_OCTL_OCTL14           BIT(14)                       /*!< pin 14 output bit */
#define GPIO_OCTL_OCTL15           BIT(15)                       /*!< pin 15 output bit */

/* GPIO_BOP */
#define GPIO_BOP_BOP0              BIT(0)                        /*!< pin 0 set bit */
#define GPIO_BOP_BOP1              BIT(1)                        /*!< pin 1 set bit */
#define GPIO_BOP_BOP2              BIT(2)                        /*!< pin 2 set bit */
#define GPIO_BOP_BOP3              BIT(3)                        /*!< pin 3 set bit */
#define GPIO_BOP_BOP4              BIT(4)                        /*!< pin 4 set bit */
#define GPIO_BOP_BOP5              BIT(5)                        /*!< pin 5 set bit */
#define GPIO_BOP_BOP6              BIT(6)                        /*!< pin 6 set bit */
#define GPIO_BOP_BOP7              BIT(7)                        /*!< pin 7 set bit */
#define GPIO_BOP_BOP8              BIT(8)                        /*!< pin 8 set bit */
#define GPIO_BOP_BOP9              BIT(9)                        /*!< pin 9 set bit */
#define GPIO_BOP_BOP10             BIT(10)                       /*!< pin 10 set bit */
#define GPIO_BOP_BOP11             BIT(11)                       /*!< pin 11 set bit */
#define GPIO_BOP_BOP12             BIT(12)                       /*!< pin 12 set bit */
#define GPIO_BOP_BOP13             BIT(13)                       /*!< pin 13 set bit */
#define GPIO_BOP_BOP14             BIT(14)                       /*!< pin 14 set bit */
#define GPIO_BOP_BOP15             BIT(15)                       /*!< pin 15 set bit */
#define GPIO_BOP_CR0               BIT(16)                       /*!< pin 0 clear bit */
#define GPIO_BOP_CR1               BIT(17)                       /*!< pin 1 clear bit */
#define GPIO_BOP_CR2               BIT(18)                       /*!< pin 2 clear bit */
#define GPIO_BOP_CR3               BIT(19)                       /*!< pin 3 clear bit */
#define GPIO_BOP_CR4               BIT(20)                       /*!< pin 4 clear bit */
#define GPIO_BOP_CR5               BIT(21)                       /*!< pin 5 clear bit */
#define GPIO_BOP_CR6               BIT(22)                       /*!< pin 6 clear bit */
#define GPIO_BOP_CR7               BIT(23)                       /*!< pin 7 clear bit */
#define GPIO_BOP_CR8               BIT(24)                       /*!< pin 8 clear bit */
#define GPIO_BOP_CR9               BIT(25)                       /*!< pin 9 clear bit */
#define GPIO_BOP_CR10              BIT(26)                       /*!< pin 10 clear bit */
#define GPIO_BOP_CR11              BIT(27)                       /*!< pin 11 clear bit */
#define GPIO_BOP_CR12              BIT(28)                       /*!< pin 12 clear bit */
#define GPIO_BOP_CR13              BIT(29)                       /*!< pin 13 clear bit */
#define GPIO_BOP_CR14              BIT(30)                       /*!< pin 14 clear bit */
#define GPIO_BOP_CR15              BIT(31)                       /*!< pin 15 clear bit */

/* GPIO_LOCK */
#define GPIO_LOCK_LK0              BIT(0)                        /*!< pin 0 lock bit */
#define GPIO_LOCK_LK1              BIT(1)                        /*!< pin 1 lock bit */
#define GPIO_LOCK_LK2              BIT(2)                        /*!< pin 2 lock bit */
#define GPIO_LOCK_LK3              BIT(3)                        /*!< pin 3 lock bit */
#define GPIO_LOCK_LK4              BIT(4)                        /*!< pin 4 lock bit */
#define GPIO_LOCK_LK5              BIT(5)                        /*!< pin 5 lock bit */
#define GPIO_LOCK_LK6              BIT(6)                        /*!< pin 6 lock bit */
#define GPIO_LOCK_LK7              BIT(7)                        /*!< pin 7 lock bit */
#define GPIO_LOCK_LK8              BIT(8)                        /*!< pin 8 lock bit */
#define GPIO_LOCK_LK9              BIT(9)                        /*!< pin 9 lock bit */
#define GPIO_LOCK_LK10             BIT(10)                       /*!< pin 10 lock bit */
#define GPIO_LOCK_LK11             BIT(11)                       /*!< pin 11 lock bit */
#define GPIO_LOCK_LK12             BIT(12)                       /*!< pin 12 lock bit */
#define GPIO_LOCK_LK13             BIT(13)                       /*!< pin 13 lock bit */
#define GPIO_LOCK_LK14             BIT(14)                       /*!< pin 14 lock bit */
#define GPIO_LOCK_LK15             BIT(15)                       /*!< pin 15 lock bit */
#define GPIO_LOCK_LKK              BIT(16)                       /*!< pin sequence lock key */

/* GPIO_AFSEL0 */
#define GPIO_AFSEL0_SEL0           BITS(0, 3)                    /*!< pin 0 alternate function selected */
#define GPIO_AFSEL0_SEL1           BITS(4, 7)                    /*!< pin 1 alternate function selected */
#define GPIO_AFSEL0_SEL2           BITS(8, 11)                   /*!< pin 2 alternate function selected */
#define GPIO_AFSEL0_SEL3           BITS(12, 15)                  /*!< pin 3 alternate function selected */
#define GPIO_AFSEL0_SEL4           BITS(16, 19)                  /*!< pin 4 alternate function selected */
#define GPIO_AFSEL0_SEL5           BITS(20, 23)                  /*!< pin 5 alternate function selected */
#define GPIO_AFSEL0_SEL6           BITS(24, 27)                  /*!< pin 6 alternate function selected */
#define GPIO_AFSEL0_SEL7           BITS(28, 31)                  /*!< pin 7 alternate function selected */

/* GPIO_AFSEL1 */
#define GPIO_AFSEL1_SEL8           BITS(0, 3)                    /*!< pin 8 alternate function selected */
#define GPIO_AFSEL1_SEL9           BITS(4, 7)                    /*!< pin 9 alternate function selected */
#define GPIO_AFSEL1_SEL10          BITS(8, 11)                   /*!< pin 10 alternate function selected */
#define GPIO_AFSEL1_SEL11          BITS(12, 15)                  /*!< pin 11 alternate function selected */
#define GPIO_AFSEL1_SEL12          BITS(16, 19)                  /*!< pin 12 alternate function selected */
#define GPIO_AFSEL1_SEL13          BITS(20, 23)                  /*!< pin 13 alternate function selected */
#define GPIO_AFSEL1_SEL14          BITS(24, 27)                  /*!< pin 14 alternate function selected */
#define GPIO_AFSEL1_SEL15          BITS(28, 31)                  /*!< pin 15 alternate function selected */

/* GPIO_BC */
#define GPIO_BC_CR0                BIT(0)                        /*!< pin 0 clear bit */
#define GPIO_BC_CR1                BIT(1)                        /*!< pin 1 clear bit */
#define GPIO_BC_CR2                BIT(2)                        /*!< pin 2 clear bit */
#define GPIO_BC_CR3                BIT(3)                        /*!< pin 3 clear bit */
#define GPIO_BC_CR4                BIT(4)                        /*!< pin 4 clear bit */
#define GPIO_BC_CR5                BIT(5)                        /*!< pin 5 clear bit */
#define GPIO_BC_CR6                BIT(6)                        /*!< pin 6 clear bit */
#define GPIO_BC_CR7                BIT(7)                        /*!< pin 7 clear bit */
#define GPIO_BC_CR8                BIT(8)                        /*!< pin 8 clear bit */
#define GPIO_BC_CR9                BIT(9)                        /*!< pin 9 clear bit */
#define GPIO_BC_CR10               BIT(10)                       /*!< pin 10 clear bit */
#define GPIO_BC_CR11               BIT(11)                       /*!< pin 11 clear bit */
#define GPIO_BC_CR12               BIT(12)                       /*!< pin 12 clear bit */
#define GPIO_BC_CR13               BIT(13)                       /*!< pin 13 clear bit */
#define GPIO_BC_CR14               BIT(14)                       /*!< pin 14 clear bit */
#define GPIO_BC_CR15               BIT(15)                       /*!< pin 15 clear bit */

/* GPIO_TG */
#define GPIO_TG_TG0                BIT(0)                        /*!< pin 0 toggle bit */
#define GPIO_TG_TG1                BIT(1)                        /*!< pin 1 toggle bit */
#define GPIO_TG_TG2                BIT(2)                        /*!< pin 2 toggle bit */
#define GPIO_TG_TG3                BIT(3)                        /*!< pin 3 toggle bit */
#define GPIO_TG_TG4                BIT(4)                        /*!< pin 4 toggle bit */
#define GPIO_TG_TG5                BIT(5)                        /*!< pin 5 toggle bit */
#define GPIO_TG_TG6                BIT(6)                        /*!< pin 6 toggle bit */
#define GPIO_TG_TG7                BIT(7)                        /*!< pin 7 toggle bit */
#define GPIO_TG_TG8                BIT(8)                        /*!< pin 8 toggle bit */
#define GPIO_TG_TG9                BIT(9)                        /*!< pin 9 toggle bit */
#define GPIO_TG_TG10               BIT(10)                       /*!< pin 10 toggle bit */
#define GPIO_TG_TG11               BIT(11)                       /*!< pin 11 toggle bit */
#define GPIO_TG_TG12               BIT(12)                       /*!< pin 12 toggle bit */
#define GPIO_TG_TG13               BIT(13)                       /*!< pin 13 toggle bit */
#define GPIO_TG_TG14               BIT(14)                       /*!< pin 14 toggle bit */
#define GPIO_TG_TG15               BIT(15)                       /*!< pin 15 toggle bit */

#if defined(GD32E251)
/* GPIO_CBREN */
#define GPIO_CBREN_USART0EN        BIT(0)                        /*!< Crossbar USART0 I/O Enable bit */
#define GPIO_CBREN_SPI0EN          BIT(1)                        /*!< Crossbar SPI0 I / O Enable bit */
#define GPIO_CBREN_SMBEN           BIT(2)                        /*!< Crossbar SMBus I / O Enable bit */
#define GPIO_CBREN_CCMP0EN         BIT(3)                        /*!< Crossbar Comparator0 Output Enable bit */
#define GPIO_CBREN_CCMP0AEN        BIT(4)                        /*!< Crossbar Comparator0 Asynchronous Output Enable bit */
#define GPIO_CBREN_CSYSCKEN        BIT(7)                        /*!< Crossbar SYSCLK Output Enable bit */
#define GPIO_CBREN_CTIMER0MOD      BITS(8,10)                    /*!< Crossbar TIMER0 Module I/O Enable bit */
#define GPIO_CBREN_CTIMER0ETIEN    BIT(11)                       /*!< Crossbar TIMER0 External Counter Input Enable bit */
#define GPIO_CBREN_CTIMER1ETIEN    BIT(12)                       /*!< Crossbar TIMER1 External Channel0 trigger enable bit */
#define GPIO_CBREN_CTIMER2ETIEN    BIT(13)                       /*!< Crossbar TIMER2 External trigger enable bit */
#define GPIO_CBREN_CTIMER13EN      BIT(14)                       /*!< Crossbar TIMER13 Channel0 enable bit */
#define GPIO_CBREN_CUART1EN        BIT(16)                       /*!< Crossbar UART1 I/O Enable bit */
#define GPIO_CBREN_CUART1RTSEN     BIT(17)                       /*!< Crossbar UART1 RTS Output Enable bit */
#define GPIO_CBREN_CUART1CTSEN     BIT(18)                       /*!< Crossbar UART1 CTS Output Enable bit */
#define GPIO_CBREN_CBAREN          BIT(19)                       /*!< Crossbar Enable bit */

/* GPIO_DFEN */
#define GPIO_DFEN_I2C0PINEN        BIT(0)                        /*!< I2C0 to the selected GPIO pin bit */
#define GPIO_DFEN_CLA0PINEN        BIT(1)                        /*!< CLA0 asynchronous output to the selected GPIO pin bit */
#define GPIO_DFEN_CLA1PINEN        BIT(2)                        /*!< CLA1 asynchronous output to the selected GPIO pin bit */
#define GPIO_DFEN_CLA2PINEN        BIT(3)                        /*!< CLA2 asynchronous output to the selected GPIO pin bit */
#define GPIO_DFEN_CLA3PINEN        BIT(4)                        /*!< CLA3 asynchronous output to the selected GPIO pin bit */

/* GPIO_SKIP */
#define GPIO_SKIP_SKIP0            BIT(0)                        /*!< SKIP0 value bit */
#define GPIO_SKIP_SKIP1            BIT(1)                        /*!< SKIP1 value bit */
#define GPIO_SKIP_SKIP2            BIT(2)                        /*!< SKIP2 value bit */
#define GPIO_SKIP_SKIP3            BIT(3)                        /*!< SKIP3 value bit */
#define GPIO_SKIP_SKIP4            BIT(4)                        /*!< SKIP4 value bit */
#define GPIO_SKIP_SKIP5            BIT(5)                        /*!< SKIP5 value bit */
#define GPIO_SKIP_SKIP6            BIT(6)                        /*!< SKIP6 value bit */
#define GPIO_SKIP_SKIP7            BIT(7)                        /*!< SKIP7 value bit */
#define GPIO_SKIP_SKIP8            BIT(8)                        /*!< SKIP8 value bit */
#define GPIO_SKIP_SKIP9            BIT(9)                        /*!< SKIP9 value bit */
#define GPIO_SKIP_SKIP10           BIT(10)                       /*!< SKIP10 value bit */
#define GPIO_SKIP_SKIP11           BIT(11)                       /*!< SKIP11 value bit */
#define GPIO_SKIP_SKIP12           BIT(12)                       /*!< SKIP12 value bit */
#define GPIO_SKIP_SKIP13           BIT(13)                       /*!< SKIP13 value bit */
#define GPIO_SKIP_SKIP14           BIT(14)                       /*!< SKIP14 value bit */
#define GPIO_SKIP_SKIP15           BIT(15)                       /*!< SKIP15 value bit */
#define GPIO_SKIP_SKIP16           BIT(16)                       /*!< SKIP16 value bit */
#define GPIO_SKIP_SKIP17           BIT(17)                       /*!< SKIP17 value bit */
#define GPIO_SKIP_SKIP18           BIT(18)                       /*!< SKIP18 value bit */
#define GPIO_SKIP_ALL              BITS(0,18)                    /*!< ALL SKIP value bit */

/*GPIO_INTCTL */
#define GPIO_INTCTL_INT0SL          BITS(0,2)                     /*!< INT0 Port Pin Selection bit */
#define GPIO_INTCTL_INT0PL          BIT(3)                        /*!< INT0 Polarity bit */
#define GPIO_INTCTL_INT1SL          BITS(4,6)                     /*!< INT1 Port Pin Selection bit */
#define GPIO_INTCTL_INT1PL          BIT(7)                        /*!< INT1 Polarity bit */

/* GPIO_MASK */
#define GPIO_MASK_MSK0             BIT(0)                        /*!< MSK0 value bit */
#define GPIO_MASK_MSK1             BIT(1)                        /*!< MSK1 value bit */
#define GPIO_MASK_MSK2             BIT(2)                        /*!< MSK2 value bit */
#define GPIO_MASK_MSK3             BIT(3)                        /*!< MSK3 value bit */
#define GPIO_MASK_MSK4             BIT(4)                        /*!< MSK4 value bit */
#define GPIO_MASK_MSK5             BIT(5)                        /*!< MSK5 value bit */
#define GPIO_MASK_MSK6             BIT(6)                        /*!< MSK6 value bit */
#define GPIO_MASK_MSK7             BIT(7)                        /*!< MSK7 value bit */
#define GPIO_MASK_MSK8             BIT(8)                        /*!< MSK8 value bit */
#define GPIO_MASK_MSK9             BIT(9)                        /*!< MSK9 value bit */
#define GPIO_MASK_MSK10            BIT(10)                       /*!< MSK10 value bit */
#define GPIO_MASK_MSK11            BIT(11)                       /*!< MSK11 value bit */
#define GPIO_MASK_MSK12            BIT(12)                       /*!< MSK12 value bit */
#define GPIO_MASK_MSK13            BIT(13)                       /*!< MSK13 value bit */
#define GPIO_MASK_MSK14            BIT(14)                       /*!< MSK14 value bit */
#define GPIO_MASK_MSK15            BIT(15)                       /*!< MSK15 value bit */
#define GPIO_MASK_MSK16            BIT(16)                       /*!< MSK16 value bit */
#define GPIO_MASK_MSK17            BIT(17)                       /*!< MSK17 value bit */
#define GPIO_MASK_MSK18            BIT(18)                       /*!< MSK18 value bit */
#define GPIO_MASK_MSK19            BIT(19)                       /*!< MSK19 value bit */
#define GPIO_MASK_MSK20            BIT(20)                       /*!< MSK20 value bit */
#define GPIO_MASK_MSK21            BIT(21)                       /*!< MSK21 value bit */
#define GPIO_MASK_ALL              BITS(0,21)                    /*!< ALL MSK value bit */

/* GPIO_MATCH */
#define GPIO_MATCH_MATCH0          BIT(0)                        /*!< MATCH0 value bit */
#define GPIO_MATCH_MATCH1          BIT(1)                        /*!< MATCH1 value bit */
#define GPIO_MATCH_MATCH2          BIT(2)                        /*!< MATCH2 value bit */
#define GPIO_MATCH_MATCH3          BIT(3)                        /*!< MATCH3 value bit */
#define GPIO_MATCH_MATCH4          BIT(4)                        /*!< MATCH4 value bit */
#define GPIO_MATCH_MATCH5          BIT(5)                        /*!< MATCH5 value bit */
#define GPIO_MATCH_MATCH6          BIT(6)                        /*!< MATCH6 value bit */
#define GPIO_MATCH_MATCH7          BIT(7)                        /*!< MATCH7 value bit */
#define GPIO_MATCH_MATCH8          BIT(8)                        /*!< MATCH8 value bit */
#define GPIO_MATCH_MATCH9          BIT(9)                        /*!< MATCH9 value bit */
#define GPIO_MATCH_MATCH10         BIT(10)                       /*!< MATCH10 value bit */
#define GPIO_MATCH_MATCH11         BIT(11)                       /*!< MATCH11 value bit */
#define GPIO_MATCH_MATCH12         BIT(12)                       /*!< MATCH12 value bit */
#define GPIO_MATCH_MATCH13         BIT(13)                       /*!< MATCH13 value bit */
#define GPIO_MATCH_MATCH14         BIT(14)                       /*!< MATCH14 value bit */
#define GPIO_MATCH_MATCH15         BIT(15)                       /*!< MATCH15 value bit */
#define GPIO_MATCH_MATCH16         BIT(16)                       /*!< MATCH16 value bit */
#define GPIO_MATCH_MATCH17         BIT(17)                       /*!< MATCH17 value bit */
#define GPIO_MATCH_MATCH18         BIT(18)                       /*!< MATCH18 value bit */
#define GPIO_MATCH_MATCH19         BIT(19)                       /*!< MATCH19 value bit */
#define GPIO_MATCH_MATCH20         BIT(20)                       /*!< MATCH20 value bit */
#define GPIO_MATCH_MATCH21         BIT(21)                       /*!< MATCH21 value bit */
#define GPIO_MATCH_ALL             BITS(0,21)                    /*!< ALL MATCH Value bit */

/* GPIO_MATCH_FLAG */
#define GPIO_MATCHF_MATCHF0        BIT(0)                        /*!< MATCHF0 Value bit */
#define GPIO_MATCHF_MATCHF1        BIT(1)                        /*!< MATCHF1 Value bit */
#define GPIO_MATCHF_MATCHF2        BIT(2)                        /*!< MATCHF2 Value bit */
#define GPIO_MATCHF_MATCHF3        BIT(3)                        /*!< MATCHF3 Value bit */
#define GPIO_MATCHF_MATCHF4        BIT(4)                        /*!< MATCHF4 Value bit */
#define GPIO_MATCHF_MATCHF5        BIT(5)                        /*!< MATCHF5 Value bit */
#define GPIO_MATCHF_MATCHF6        BIT(6)                        /*!< MATCHF6 Value bit */
#define GPIO_MATCHF_MATCHF7        BIT(7)                        /*!< MATCHF7 Value bit */
#define GPIO_MATCHF_MATCHF8        BIT(8)                        /*!< MATCHF8 Value bit */
#define GPIO_MATCHF_MATCHF9        BIT(9)                        /*!< MATCHF9 Value bit */
#define GPIO_MATCHF_MATCHF10       BIT(10)                       /*!< MATCHF10 Value bit */
#define GPIO_MATCHF_MATCHF11       BIT(11)                       /*!< MATCHF11 Value bit */
#define GPIO_MATCHF_MATCHF12       BIT(12)                       /*!< MATCHF12 Value bit */
#define GPIO_MATCHF_MATCHF13       BIT(13)                       /*!< MATCHF13 Value bit */
#define GPIO_MATCHF_MATCHF14       BIT(14)                       /*!< MATCHF14 Value bit */
#define GPIO_MATCHF_MATCHF15       BIT(15)                       /*!< MATCHF15 Value bit */
#define GPIO_MATCHF_MATCHF16       BIT(16)                       /*!< MATCHF16 Value bit */
#define GPIO_MATCHF_MATCHF17       BIT(17)                       /*!< MATCHF17 Value bit */
#define GPIO_MATCHF_MATCHF18       BIT(18)                       /*!< MATCHF18 Value bit */
#define GPIO_MATCHF_MATCHF19       BIT(19)                       /*!< MATCHF19 Value bit */
#define GPIO_MATCHF_MATCHF20       BIT(20)                       /*!< MATCHF20 Value bit */
#define GPIO_MATCHF_MATCHF21       BIT(21)                       /*!< MATCHF21 Value bit */
#define GPIO_MATCHF_ALL            BITS(0,21)                    /*!< ALL MATCHF Value bit */
#define GPIO_MATCHF_MATCHIE        BIT(22)                       /*!< enable port not match interrupt bit */
#endif /* GD32E251 */

/* input and output mode definitions */
#define CTL_CLTR(regval)           (BITS(0,2) & ((uint32_t)(regval) << 0))
#define GPIO_MODE_INPUT            CTL_CLTR(0)                   /*!< input mode */
#define GPIO_MODE_OUTPUT           CTL_CLTR(1)                   /*!< output mode */
#define GPIO_MODE_AF               CTL_CLTR(2)                   /*!< alternate function mode */
#define GPIO_MODE_ANALOG           CTL_CLTR(3)                   /*!< analog mode */

/* pull-up/pull-down definitions */
#define PUD_PUPD(regval)           (BITS(0, 1) & ((uint32_t)(regval) << 0))
#define GPIO_PUPD_NONE             PUD_PUPD(0)                   /*!< floating mode, no pull-up and pull-down resistors */
#define GPIO_PUPD_PULLUP           PUD_PUPD(1)                   /*!< with pull-up resistor */
#define GPIO_PUPD_PULLDOWN         PUD_PUPD(2)                   /*!< with pull-down resistor */

/* GPIO pin definitions */
#define GPIO_PIN_0                 BIT(0)                        /*!< GPIO pin 0 */
#define GPIO_PIN_1                 BIT(1)                        /*!< GPIO pin 1 */
#define GPIO_PIN_2                 BIT(2)                        /*!< GPIO pin 2 */
#define GPIO_PIN_3                 BIT(3)                        /*!< GPIO pin 3 */
#define GPIO_PIN_4                 BIT(4)                        /*!< GPIO pin 4 */
#define GPIO_PIN_5                 BIT(5)                        /*!< GPIO pin 5 */
#define GPIO_PIN_6                 BIT(6)                        /*!< GPIO pin 6 */
#define GPIO_PIN_7                 BIT(7)                        /*!< GPIO pin 7 */
#define GPIO_PIN_8                 BIT(8)                        /*!< GPIO pin 8 */
#define GPIO_PIN_9                 BIT(9)                        /*!< GPIO pin 9 */
#define GPIO_PIN_10                BIT(10)                       /*!< GPIO pin 10 */
#define GPIO_PIN_11                BIT(11)                       /*!< GPIO pin 11 */
#define GPIO_PIN_12                BIT(12)                       /*!< GPIO pin 12 */
#define GPIO_PIN_13                BIT(13)                       /*!< GPIO pin 13 */
#define GPIO_PIN_14                BIT(14)                       /*!< GPIO pin 14 */
#define GPIO_PIN_15                BIT(15)                       /*!< GPIO pin 15 */
#define GPIO_PIN_ALL               BITS(0, 15)                   /*!< GPIO pin all */

/* GPIO mode configuration values */
#define GPIO_MODE_SET(n, mode)     ((uint32_t)((uint32_t)(mode) << (2U * (n))))
#define GPIO_MODE_MASK(n)          (0x3U << (2U * (n)))

/* GPIO pull-up/pull-down values */
#define GPIO_PUPD_SET(n, pupd)     ((uint32_t)((uint32_t)(pupd) << (2U * (n))))
#define GPIO_PUPD_MASK(n)          (0x3U << (2U * (n)))

/* GPIO output speed values */
#define GPIO_OSPEED_SET(n, speed)  ((uint32_t)((uint32_t)(speed) << (2U * (n))))
#define GPIO_OSPEED_MASK(n)        (0x3U << (2U * (n)))

/* GPIO output type */
#define GPIO_OTYPE_PP              ((uint8_t)(0x00U))          /*!< push pull mode */
#define GPIO_OTYPE_OD              ((uint8_t)(0x01U))          /*!< open drain mode */

/* GPIO output max speed value */
#define OSPD_OSPD0(regval)         (BITS(0, 1) & ((uint32_t)(regval) << 0))
#define GPIO_OSPEED_2MHZ           OSPD_OSPD0(0)               /*!< output max speed 2MHz */
#define GPIO_OSPEED_10MHZ          OSPD_OSPD0(1)               /*!< output max speed 10MHz */
#define GPIO_OSPEED_50MHZ          OSPD_OSPD0(3)               /*!< output max speed 50MHz */

/* GPIO alternate function values */
#define GPIO_AFR_SET(n, af)        ((uint32_t)((uint32_t)(af) << (4U * (n))))
#define GPIO_AFR_MASK(n)           (0xFU << (4U * (n)))

/* GPIO alternate function */
#define AF(regval)                 (BITS(0, 3) & ((uint32_t)(regval) << 0))
#define GPIO_AF_0                  AF(0)                       /*!< alternate function 0 selected */
#define GPIO_AF_1                  AF(1)                       /*!< alternate function 1 selected */
#define GPIO_AF_2                  AF(2)                       /*!< alternate function 2 selected */
#define GPIO_AF_3                  AF(3)                       /*!< alternate function 3 selected */
#define GPIO_AF_4                  AF(4)                       /*!< alternate function 4 selected (port A,B only) */
#define GPIO_AF_5                  AF(5)                       /*!< alternate function 5 selected (port A,B only) */
#define GPIO_AF_6                  AF(6)                       /*!< alternate function 6 selected (port A,B only) */
#define GPIO_AF_7                  AF(7)                       /*!< alternate function 7 selected (port A,B only) */
#define GPIO_AF_8                  AF(8)                       /*!< alternate function 8 selected (port A,B only) */
#define GPIO_AF_9                  AF(9)                       /*!< alternate function 9 selected (port A,B only) */
#define GPIO_AF_10                 AF(10)                      /*!< alternate function 10 selected (port A,B only) */

/* GPIO alternate function values */
#define GPIO_AFR_SET(n, af)        ((uint32_t)((uint32_t)(af) << (4U * (n))))
#define GPIO_AFR_MASK(n)           (0xFU << (4U * (n)))

#if defined(GD32E251)
/* GPIO crossbar mode configuration values */
#define GPIO_CROSS_MODE_SET(n)     ((uint32_t)((uint32_t)(0x1U) << (n)))
#define GPIO_CROSS_MODE_MASK(n)    (0x1U << (n))
/* crossbar resource timer0 module enable */
#define TIMER0_MOD(regval)          (BITS(8, 10) & ((uint32_t)(regval) << 8))
#define GPIO_CBREN_CTIMER0_MOD      TIMER0_MOD(0)              /*!< all timer0 i/o unavailable at port pins */
#define GPIO_CBREN_CH0              TIMER0_MOD(1)              /*!< CH0 routed at port pins */
#define GPIO_CBREN_CH0_2_CH1        TIMER0_MOD(2)              /*!< CH0 CH1 routed at port pins */
#define GPIO_CBREN_CH0_2_CH2        TIMER0_MOD(3)              /*!< CH0 CH1 CH2 routed at port pins */
#define GPIO_CBREN_CH0_2_CH3        TIMER0_MOD(4)              /*!< CH0 CH1 CH2 CH3 routed at port pins */
#define GPIO_CBREN_CH0_2_CH4        TIMER0_MOD(5)              /*!< CH0 CH1 CH2 CH3 CH4 routed at port pins */
#define GPIO_CBREN_CH0_2_CH5        TIMER0_MOD(6)              /*!< CH0 CH1 CH2 CH3 CH4 CH5 routed at port pins */
#define CROSSBAR_TIMER0ME_MASK      (TIMER0_MOD(0x7U))

typedef enum {
    CROSSBAR_USART0                 = GPIO_CBREN_USART0EN,     /*!Crossbar USART0 I/O Enable */
    CROSSBAR_SPI0                   = GPIO_CBREN_SPI0EN,       /*!< Crossbar SPI0 I / O Enable */
    CROSSBAR_SMB0                   = GPIO_CBREN_SMBEN,        /*!< Crossbar SMBus I / O Enable */
    CROSSBAR_CP0E                   = GPIO_CBREN_CCMP0EN,      /*!< Crossbar Comparator0 Output Enable */
    CROSSBAR_CP0AE                  = GPIO_CBREN_CCMP0AEN,     /*!< Crossbar Comparator0 Asynchronous Output Enable */
    CROSSBAR_SYSCKE                 = GPIO_CBREN_CSYSCKEN,     /*!< Crossbar SYSCLK Output Enable */
    CROSSBAR_TIMER0_NONE            = GPIO_CBREN_CTIMER0_MOD,  /*!< Crossbar TIMER0 Module I/O Enable */
    CROSSBAR_TIMER0_CH0             = GPIO_CBREN_CH0,          /*!< Crossbar TIMER0 CH0 Enable */
    CROSSBAR_TIMER0_CH0_2_CH1       = GPIO_CBREN_CH0_2_CH1,    /*!< Crossbar TIMER0 CH0 CH1 Enable */
    CROSSBAR_TIMER0_CH0_2_CH2       = GPIO_CBREN_CH0_2_CH2,    /*!< Crossbar TIMER0 CH0 CH1 CH2 Enable */
    CROSSBAR_TIMER0_CH0_2_CH3       = GPIO_CBREN_CH0_2_CH3,    /*!< Crossbar TIMER0 CH0 CH1 CH2 CH3 Enable */
    CROSSBAR_TIMER0_CH0_2_CH4       = GPIO_CBREN_CH0_2_CH4,    /*!< Crossbar TIMER0 CH0 CH1 CH2 CH3 CH4 Enable */
    CROSSBAR_TIMER0_CH0_2_CH5       = GPIO_CBREN_CH0_2_CH5,    /*!< Crossbar TIMER0 CH0 CH1 CH2 CH3 CH4 CH5 Enable */
    CROSSBAR_TIMER0_ETR_EN          = GPIO_CBREN_CTIMER0ETIEN, /*!< Crossbar TIMER0 External Counter Input Enable */
    CROSSBAR_TIMER1_ETR_EN          = GPIO_CBREN_CTIMER1ETIEN, /*!< Crossbar TIMER1 External Channel0 trigger Enable */
    CROSSBAR_TIMER2_ETR_EN          = GPIO_CBREN_CTIMER2ETIEN, /*!< Crossbar TIMER2 External trigger Enable */
    CROSSBAR_TIMER13_EN             = GPIO_CBREN_CTIMER13EN,   /*!< Crossbar TIMER13 Channel0 Enable */
    CROSSBAR_USRT1E                 = GPIO_CBREN_CUART1EN,     /*!< Crossbar UART1 I/O Enable */
    CROSSBAR_USRT1RTSE              = GPIO_CBREN_CUART1RTSEN,  /*!< Crossbar UART1 RTS Output Enable */
    CROSSBAR_USRT1CTSE              = GPIO_CBREN_CUART1CTSEN   /*!< Crossbar UART1 CTS Output Enable */
} gpio_crossbar_periph_enable_enum;

#define GPIO_SKIP_ENABLE            (uint8_t)(1U)              /*!< gpio skip enable value */
#define GPIO_SKIP_DISABLE           (uint8_t)(0U)              /*!< gpio skip disable value */

/* int0 port pin selection */
#define INT0SL(regval)              (BITS(0, 2) & ((uint32_t)(regval) << 0))
#define GPIO_INT0_SEL_PB9           INT0SL(0)                  /*!< int0 port select PB9 */
#define GPIO_INT0_SEL_PB8           INT0SL(1)                  /*!< int0 port select PB8 */
#define GPIO_INT0_SEL_PF0           INT0SL(2)                  /*!< int0 port select PF0 */
#define GPIO_INT0_SEL_PB5           INT0SL(3)                  /*!< int0 port select PB5 */
#define GPIO_INT0_SEL_PB4           INT0SL(4)                  /*!< int0 port select PB4 */
#define GPIO_INT0_SEL_PB3           INT0SL(5)                  /*!< int0 port select PB3 */
#define GPIO_INT0_SEL_PA15          INT0SL(6)                  /*!< int0 port select PA15 */

/* int1 port pin selection */
#define INT1SL(regval)              (BITS(4, 6) & ((uint32_t)(regval) << 4))
#define GPIO_INT1_SEL_PB9           INT1SL(0)                  /*!< int1 port select PB9 */
#define GPIO_INT1_SEL_PB8           INT1SL(1)                  /*!< int1 port select PB8 */
#define GPIO_INT1_SEL_PF0           INT1SL(2)                  /*!< int1 port select PF0 */
#define GPIO_INT1_SEL_PB5           INT1SL(3)                  /*!< int1 port select PB5 */
#define GPIO_INT1_SEL_PB4           INT1SL(4)                  /*!< int1 port select PB4 */
#define GPIO_INT1_SEL_PB3           INT1SL(5)                  /*!< int1 port select PB3 */
#define GPIO_INT1_SEL_PA15          INT1SL(6)                  /*!< int1 port select PA15 */

/* INT0 or INT1 polarity */
#define INT_POLARITY_LOW            (uint8_t)(0U)              /*!< the int polarity low value */
#define INT_POLARITY_HIGH           (uint8_t)(1U)              /*!< the int polarity high value */

/* gpio pin mask a mismatch event */
#define GPIO_PIN_MASK_FALSE         (uint8_t)(0U)              /*!< the gpio pin mask false value */
#define GPIO_PIN_MASK_TRUE          (uint8_t)(1U)              /*!< the gpio pin mask true value */

/* the level of gpio match */
#define GPIO_PIN_MATCH_HIGH         (uint8_t)(1U)              /*!< the gpio pin match high value */
#define GPIO_PIN_MATCH_LOW          (uint8_t)(0U)              /*!< the gpio pin match low value */
#endif /* GD32E251 */

/* constants definitions */
typedef FlagStatus bit_status;

#if defined(GD32E251)
typedef FlagStatus crossbar_enable;
#endif /* GD32E251 */

/* function declarations */
/* reset GPIO port */
void gpio_deinit(uint32_t gpio_periph);
/* set GPIO mode */
void gpio_mode_set(uint32_t gpio_periph, uint32_t mode, uint32_t pull_up_down, uint32_t pin);
/* set GPIO output type and speed */
void gpio_output_options_set(uint32_t gpio_periph, uint8_t otype, uint32_t speed, uint32_t pin);
/* set GPIO pin bit */
void gpio_bit_set(uint32_t gpio_periph, uint32_t pin);
/* reset GPIO pin bit */
void gpio_bit_reset(uint32_t gpio_periph, uint32_t pin);
/* write data to the specified GPIO pin */
void gpio_bit_write(uint32_t gpio_periph, uint32_t pin, bit_status bit_value);
/* write data to the specified GPIO port */
void gpio_port_write(uint32_t gpio_periph, uint16_t data);
/* get GPIO pin input status */
FlagStatus gpio_input_bit_get(uint32_t gpio_periph, uint32_t pin);
/* get GPIO port input status */
uint16_t gpio_input_port_get(uint32_t gpio_periph);
/* get GPIO pin output status */
FlagStatus gpio_output_bit_get(uint32_t gpio_periph, uint32_t pin);
/* get GPIO port output status */
uint16_t gpio_output_port_get(uint32_t gpio_periph);
/* set GPIO alternate function */
void gpio_af_set(uint32_t gpio_periph, uint32_t alt_func_num, uint32_t pin);
/* lock GPIO pin bit */
void gpio_pin_lock(uint32_t gpio_periph, uint32_t pin);
/* toggle GPIO pin status */
void gpio_bit_toggle(uint32_t gpio_periph, uint32_t pin);
/* toggle GPIO port status */
void gpio_port_toggle(uint32_t gpio_periph);
#if defined(GD32E251)
/* crossbar enable */
void gpio_crossbar_enable(void);
/* crossbar disable*/
void gpio_crossbar_disable(void);
/* enable the periph routed to port pins */
void gpio_crossbar_periph_routed_enable(gpio_crossbar_periph_enable_enum gpio_periph_routed);
/* disable the periph routed to port pins */
void gpio_crossbar_periph_routed_disable(gpio_crossbar_periph_enable_enum gpio_periph_routed);
/* enable the digital function that not assign by crossbar */
void gpio_digital_function_enable(uint32_t periph_digital);
/* disable the digital function that not assign by crossbar */
void gpio_digital_function_disable(uint32_t periph_digital);
/* pin is skipped by the crossbar */
void gpio_skip_pin(uint32_t gpio_skip);
/* pin is not skipped by the crossbar */
void gpio_nskip_pin(uint32_t gpio_nskip);
/* config int0 */
void gpio_int0_config(uint8_t polarity, uint32_t int_sel);
/* config int1 */
void gpio_int1_config(uint8_t polarity, uint32_t int_sel);
/* set port pin mask value */
void gpio_pin_mask_set(uint32_t gpio_pin_mask, uint8_t mask);
/* set port pin match value */
void gpio_pin_match_set(uint32_t gpio_pin_match, uint8_t match);
/* clear pin not match flag */
void gpio_pin_not_match_flag_clear(uint32_t match_nflag);
/* get pin not match flag */
FlagStatus gpio_pin_not_match_flag_get(uint32_t match_flag);
/* pin not match enable */
void gpio_pin_not_match_enable(void);
/* pin not match disable */
void gpio_pin_not_match_disable(void);
#endif /* GD32E251 */
#endif /* GD32E25X_GPIO_H */
